/*
			Copyright (c) 2002 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/hdf5/xp_mods/h5_utils.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <stdio.h>
#include <stdlib.h>

#include <avs/port.h>
#include <avs/dtype.h>

#include "h5_xp.h"

#define METHOD_SUCCESS 1
#define METHOD_FAILURE 0

hid_t
HDF5probe_attribute(hid_t parent_id, const char *name, int close_flag)
{
    hid_t h5_attr_id;

    H5E_auto_t auto_func;
    void *auto_data;

    H5Eget_auto( &auto_func, &auto_data );  /* get old error reporting */
    H5Eset_auto( (H5E_auto_t)0, NULL );     /* disable error reporting */

    h5_attr_id = H5Aopen_name( parent_id, name );
    if( close_flag && (h5_attr_id > 0) )
        H5Aclose( h5_attr_id );

    H5Eset_auto( auto_func, auto_data );   /* restore old */
    return h5_attr_id;
}

hid_t
HDF5probe_dataset(hid_t parent_id, const char *name, int close_flag)
{
    hid_t h5_dset_id;

    H5E_auto_t auto_func;
    void *auto_data;

    H5Eget_auto( &auto_func, &auto_data );  /* get old error reporting */
    H5Eset_auto( (H5E_auto_t)0, NULL );     /* disable error reporting */

    h5_dset_id = H5Dopen( parent_id, name );
    if( close_flag && (h5_dset_id > 0) )
        H5Dclose( h5_dset_id );

    H5Eset_auto( auto_func, auto_data );   /* restore old */
    return h5_dset_id;
}

hid_t
HDF5probe_group(hid_t parent_id, const char *name, int close_flag)
{
    hid_t h5_grp_id;

    H5E_auto_t auto_func;
    void *auto_data;

    H5Eget_auto( &auto_func, &auto_data );  /* get old error reporting */
    H5Eset_auto( (H5E_auto_t)0, NULL );     /* disable error reporting */

    h5_grp_id = H5Gopen( parent_id, name );
    if( close_flag && (h5_grp_id > 0) )
        H5Gclose( h5_grp_id );

    H5Eset_auto( auto_func, auto_data );   /* restore old */
    return h5_grp_id;
}


/* 64-bit porting. Only Modified Internally */
void
HDF5map_dtypes_rd(
    hid_t h5_dtype_file,	/* in: how its stored in HDF5 file */
    hid_t *h5_dtype_app,	/* out: how its stored by OM, expressed in
                                   types that HDF knows about. */
    int *data_type )		/* out: OM data type */
{
    *h5_dtype_app = 0;
    *data_type = DTYPE_UNSET;

#if 0
    fprintf( stderr, "INT Types:\t%d %d %d %d %d\n",
             H5T_NATIVE_INT_g,
             H5T_NATIVE_INT32_g,
             H5T_NATIVE_INT_LEAST32_g,
             H5T_NATIVE_INT_FAST32_g,
             H5T_STD_I32LE_g
             );

    fprintf( stderr, "TypesTest:\t%d %d %d %d\n",
             H5Tequal(H5T_STD_I32LE_g, H5T_NATIVE_INT_g ),
             H5Tequal(H5T_STD_I32LE_g, H5T_NATIVE_INT32_g ),
             H5Tequal(H5T_STD_I32LE_g, H5T_NATIVE_INT_LEAST32_g ),
             H5Tequal(H5T_STD_I32LE_g, H5T_NATIVE_INT_FAST32_g )
             );
#endif

    if( H5Tequal(h5_dtype_file, H5T_STD_I8LE_g) ||   /* signed */
        H5Tequal(h5_dtype_file, H5T_STD_I8BE_g) ) {
        *data_type    = DTYPE_CHAR;
        *h5_dtype_app = H5T_NATIVE_SCHAR_g;
    }
    else if( H5Tequal(h5_dtype_file, H5T_STD_U8LE_g) ||  /* unsigned */
             H5Tequal(h5_dtype_file, H5T_STD_U8BE_g) ) {
        *data_type    = DTYPE_BYTE;
        *h5_dtype_app = H5T_NATIVE_UCHAR_g;
    }
    else if( H5Tequal(h5_dtype_file, H5T_STD_I16LE_g) ||
             H5Tequal(h5_dtype_file, H5T_STD_I16BE_g) ) {
        *data_type    = DTYPE_SHORT;
        *h5_dtype_app = H5T_NATIVE_SHORT_g;
    }
    else if( H5Tequal(h5_dtype_file, H5T_STD_I32LE_g) ||
             H5Tequal(h5_dtype_file, H5T_STD_I32BE_g) ) {
        /* This case is now ambigious, it could be a xp_long
         * written on a 32-bit platform.
         */
        *data_type    = DTYPE_INT;
        *h5_dtype_app = H5T_NATIVE_INT_g;
    }
    else if( H5Tequal(h5_dtype_file, H5T_STD_I64LE_g) ||
             H5Tequal(h5_dtype_file, H5T_STD_I64BE_g) ) {
        /* Might overflow on a 32-bit platform, but you might
         * as well try it out and hope for the best.
         */
        *data_type    = DTYPE_LONG;
        *h5_dtype_app = H5T_NATIVE_XP_LONG_g;
    }
    else if( H5Tequal(h5_dtype_file, H5T_IEEE_F32LE_g) ||
             H5Tequal(h5_dtype_file, H5T_IEEE_F32BE_g) ) {
        *data_type    = DTYPE_FLOAT;
        *h5_dtype_app = H5T_NATIVE_FLOAT_g;
    }
    else if( H5Tequal(h5_dtype_file, H5T_IEEE_F64LE_g) ||
             H5Tequal(h5_dtype_file, H5T_IEEE_F64BE_g) ) {
        *data_type    = DTYPE_DOUBLE;
        *h5_dtype_app = H5T_NATIVE_DOUBLE_g;
    }
    else {
        fprintf( stderr, "HDF5map_dtypes_rd: unrecognized HDF5 type!\n" );
    }
}

/* 64-bit porting. Only Modified Internally */
void
HDF5map_dtypes_wr(
    int data_type,		/* in: OM data type */
    hid_t *h5_dtype_file,	/* out: how to store in HD5 file */
    hid_t *h5_dtype_app )	/* out: how its stored by OM, expressed in
                                   types that HDF knows about. */
{
    /* When you write out a native type to file, what really happens
       is that the appropriate "IEEE" or "STD" type appears in the
       file.  So using NATIVE_INT, on a little-endian machine STD_I32LE
       would be written out and on a big-endian machine STD_I32BE would
       be written out.  At least thats what 'h5dump' shows.
    */

    switch( data_type ) {
    case DTYPE_CHAR:
        *h5_dtype_file = H5T_NATIVE_INT8_g; /* signed */
        *h5_dtype_app  = H5T_NATIVE_SCHAR_g;
        break;
    case DTYPE_BYTE:
        *h5_dtype_file = H5T_NATIVE_UINT8_g; /* unsigned */
        *h5_dtype_app  = H5T_NATIVE_UCHAR_g;
        break;
    case DTYPE_SHORT:
        *h5_dtype_file = H5T_NATIVE_INT16_g;
        *h5_dtype_app  = H5T_NATIVE_SHORT_g;
        break;
    case DTYPE_INT:
        *h5_dtype_file = H5T_NATIVE_INT32_g;
        *h5_dtype_app  = H5T_NATIVE_INT_g;
        break;
    case DTYPE_LONG:
        *h5_dtype_file = H5T_NATIVE_XP_LONG_F_g;
        *h5_dtype_app  = H5T_NATIVE_XP_LONG_g;
        break;
    case DTYPE_FLOAT:
        *h5_dtype_file = H5T_NATIVE_FLOAT_g;
        *h5_dtype_app  = H5T_NATIVE_FLOAT_g;
        break;
    case DTYPE_DOUBLE:
        *h5_dtype_file = H5T_NATIVE_DOUBLE_g;
        *h5_dtype_app  = H5T_NATIVE_DOUBLE_g;
        break;
    default:
        *h5_dtype_file = -1;
        *h5_dtype_app  = -1;
    }
    return;
}

int
HDF5read_int_attr( hid_t parent_id, const char *name, int *value )
{
    hid_t h5_attr_id, h5_dtype_id, h5_dspace_id;
    int h5_ndim;

    h5_attr_id = H5Aopen_name(parent_id, name);
    if (h5_attr_id > 0) {
        h5_dtype_id  = H5Aget_type( h5_attr_id );
        if( h5_dtype_id > 0 ) {
            /* Hmmm, any int type that could be read into a native int
               would be OK.  How to test?
            */
            if( H5Tequal( h5_dtype_id, H5T_STD_I32LE_g) ||
                H5Tequal( h5_dtype_id, H5T_STD_I32BE_g) ||
                H5Tequal( h5_dtype_id, H5T_STD_I16LE_g) ||
                H5Tequal( h5_dtype_id, H5T_STD_I16BE_g) ) {

                h5_dspace_id = H5Aget_space( h5_attr_id );
                if( h5_dspace_id > 0 ) {

                    h5_ndim = H5Sget_simple_extent_ndims( h5_dspace_id );
                    if( h5_ndim == 0 ) { /* scalar has ndim == 0 */
                        H5Aread(h5_attr_id, H5T_NATIVE_INT_g, value);
                    }
                    else if( h5_ndim == 1 ) {
                        hsize_t h5_dims[1];
                        H5Sget_simple_extent_dims( h5_dspace_id, h5_dims,
                                                   NULL );
                        if( h5_dims[0] == 1 )
                            H5Aread(h5_attr_id, H5T_NATIVE_INT_g, value);
                    }
                    /* else not a scalar */

                    H5Sclose(h5_dspace_id);	/* close dataspace */
                }
            }
            H5Tclose(h5_dtype_id);	/* close datatype */
        }
        H5Aclose(h5_attr_id);	/* close attribute */
    }
    /* Could not find attribute */
    else return METHOD_FAILURE;

    return METHOD_SUCCESS;
}

void
HDF5write_int_attr( hid_t parent_id, const char *name, int value )
{
    hid_t h5_dspace_id, h5_attr_id;

    h5_dspace_id = H5Screate( H5S_SCALAR );
    if (h5_dspace_id > 0) {
        h5_attr_id = H5Acreate( parent_id, name, H5T_NATIVE_INT32_g,
                                h5_dspace_id, H5P_DEFAULT );
        if( h5_attr_id > 0 ) {
            H5Awrite( h5_attr_id, H5T_NATIVE_INT_g, &value );
            H5Aclose( h5_attr_id );	/* close attribute */
        }
        H5Sclose( h5_dspace_id );	/* close dataspace */
    }
    return;
}

#ifdef WORDLENGTH_64
/* 64-bit porting. Newly Introduced */
int
HDF5read_int64_attr( hid_t parent_id, const char *name, xp_long *value )
{
    hid_t h5_attr_id, h5_dtype_id, h5_dspace_id;
    int h5_ndim;

    h5_attr_id = H5Aopen_name(parent_id, name);
    if (h5_attr_id > 0) {
        h5_dtype_id  = H5Aget_type( h5_attr_id );
        if( h5_dtype_id > 0 ) {
            /* Hmmm, any int type that could be read into a native int64
               would be OK.  How to test?
            */
            if( H5Tequal( h5_dtype_id, H5T_STD_I64LE_g) ||
                H5Tequal( h5_dtype_id, H5T_STD_I64BE_g) ||
                H5Tequal( h5_dtype_id, H5T_STD_I32LE_g) ||
                H5Tequal( h5_dtype_id, H5T_STD_I32BE_g) ||
                H5Tequal( h5_dtype_id, H5T_STD_I16LE_g) ||
                H5Tequal( h5_dtype_id, H5T_STD_I16BE_g) ) {

                h5_dspace_id = H5Aget_space( h5_attr_id );
                if( h5_dspace_id > 0 ) {

                    h5_ndim = H5Sget_simple_extent_ndims( h5_dspace_id );
                    if( h5_ndim == 0 ) { /* scalar has ndim == 0 */
                        H5Aread(h5_attr_id, H5T_NATIVE_LONG_g, value);
                    }
                    else if( h5_ndim == 1 ) {
                        hsize_t h5_dims[1];
                        H5Sget_simple_extent_dims( h5_dspace_id, h5_dims,
                                                   NULL );
                        if( h5_dims[0] == 1 )
                            H5Aread(h5_attr_id, H5T_NATIVE_LONG_g, value);
                    }
                    /* else not a scalar */

                    H5Sclose(h5_dspace_id);	/* close dataspace */
                }
            }
            H5Tclose(h5_dtype_id);	/* close datatype */
        }
        H5Aclose(h5_attr_id);	/* close attribute */
    }
    /* Could not find attribute */
    else return METHOD_FAILURE;

    return METHOD_SUCCESS;
}

/* 64-bit porting. Newly Introduced */
void
HDF5write_int64_attr( hid_t parent_id, const char *name, xp_long value )
{
    hid_t h5_dspace_id, h5_attr_id;

    h5_dspace_id = H5Screate( H5S_SCALAR );
    if (h5_dspace_id > 0) {
        h5_attr_id = H5Acreate( parent_id, name, H5T_NATIVE_INT64_g,
                                h5_dspace_id, H5P_DEFAULT );
        if( h5_attr_id > 0 ) {
            H5Awrite( h5_attr_id, H5T_NATIVE_LONG_g, &value );
            H5Aclose( h5_attr_id );	/* close attribute */
        }
        H5Sclose( h5_dspace_id );	/* close dataspace */
    }
    return;
}
#endif

/* Caller must free string */
/* 64-bit porting. Only Modified Internally */
int
HDF5read_string_attr( hid_t parent_id, const char *name, char **value )
{
    hid_t h5_dtype_id, h5_attr_id;
    char *str;

    *value = NULL;

    h5_attr_id = H5Aopen_name(parent_id, name);
    if (h5_attr_id > 0) {
        xp_long len;
        h5_dtype_id = H5Aget_type( h5_attr_id );
        len = H5Tget_size( h5_dtype_id );

        if( len > 0 ) {
            if( H5Tget_strpad( h5_dtype_id ) == H5T_STR_NULLTERM ) {
                str = malloc( len );
                H5Aread(h5_attr_id, h5_dtype_id, str);

                *value = str;
            }
        }
        H5Tclose(h5_dtype_id);	/* close datatype */
        H5Aclose(h5_attr_id);	/* close attribute */
    }
    /* Could not find attribute */
    else return METHOD_FAILURE;

    return METHOD_SUCCESS;
}

void
HDF5write_string_attr( hid_t parent_id, const char *name,
                       const char *value )
{
    hid_t h5_dspace_id, h5_dtype_id, h5_attr_id;

    h5_dspace_id = H5Screate( H5S_SCALAR );
    if (h5_dspace_id > 0) {
        h5_dtype_id   = H5Tcopy( H5T_C_S1_g );
        if( h5_dtype_id > 0 ) {
            H5Tset_size( h5_dtype_id, strlen(value)+1 );
            h5_attr_id = H5Acreate( parent_id, name,
                                    h5_dtype_id, h5_dspace_id, H5P_DEFAULT );
            if( h5_attr_id > 0 ) {
                H5Awrite( h5_attr_id, h5_dtype_id, (char *)value );
                H5Aclose( h5_attr_id );	/* close attribute */
            }
            H5Tclose( h5_dtype_id );	/* close datatype */
        }
        H5Sclose( h5_dspace_id );	/* close dataspace */
    }
    return;
}

/* 64-bit porting. Directly Modified */
int
HDF5read_int_array_attr( hid_t h5_root_id, const char *name,
                         xp_long *n, int **arr_ptr )
{
    hid_t h5_attr_id, h5_dtype_id, h5_dspace_id;
    hsize_t h5_dims[H5S_MAX_RANK];

    int *arr = NULL;

    *n = 0;
    *arr_ptr = NULL;

    if( (h5_attr_id = probe_attribute( h5_root_id, name, 0 )) > 0 ) {

        h5_dtype_id  = H5Aget_type( h5_attr_id );
        if( h5_dtype_id > 0 ) {
            h5_dspace_id = H5Aget_space( h5_attr_id );
            if( h5_dspace_id > 0 ) {

                int ndims = H5Sget_simple_extent_ndims( h5_dspace_id );

                if( ndims == 1 ) {
                    H5Sget_simple_extent_dims( h5_dspace_id, h5_dims, NULL );
                    arr = malloc( h5_dims[0] * sizeof(int) );
                    H5Aread( h5_attr_id, H5T_NATIVE_INT_g, arr );
                }
                H5Sclose( h5_dspace_id );
            }
            H5Tclose( h5_dtype_id );
        }
        H5Aclose( h5_attr_id );
    }

    *n = h5_dims[0];
    *arr_ptr = arr;

    return METHOD_SUCCESS;
}

/* 64-bit porting. Directly Modified */
void
HDF5write_int_array_attr( hid_t parent_id, const char *name,
                     int *array, xp_long size )
{
    hsize_t h5_dims = size;
    hid_t h5_dspace_id, h5_dset_id;

    h5_dspace_id = H5Screate_simple(1, &h5_dims, NULL);
    if( h5_dspace_id > 0 ) {
        h5_dset_id   = H5Dcreate(parent_id, name,
                                 H5T_NATIVE_INT32_g, h5_dspace_id,
                                 H5P_DEFAULT);
        if( h5_dset_id > 0 ) {
            H5Dwrite(h5_dset_id, H5T_NATIVE_INT_g,
                     h5_dspace_id, h5_dspace_id, H5P_DEFAULT,
                     array);
            H5Dclose(h5_dset_id);
        }
        H5Sclose(h5_dspace_id);
    }
}

#ifdef WORDLENGTH_64
/* 64-bit porting. Newly Introduced */
int
HDF5read_int64_array_attr( hid_t h5_root_id, const char *name,
                         xp_long *n, xp_long **arr_ptr )
{
    hid_t h5_attr_id, h5_dtype_id, h5_dspace_id;
    hsize_t h5_dims[H5S_MAX_RANK];

    xp_long *arr = NULL;

    *n = 0;
    *arr_ptr = NULL;

    if( (h5_attr_id = probe_attribute( h5_root_id, name, 0 )) > 0 ) {

        h5_dtype_id  = H5Aget_type( h5_attr_id );
        if( h5_dtype_id > 0 ) {
            h5_dspace_id = H5Aget_space( h5_attr_id );
            if( h5_dspace_id > 0 ) {

                int ndims = H5Sget_simple_extent_ndims( h5_dspace_id );

                if( ndims == 1 ) {
                    H5Sget_simple_extent_dims( h5_dspace_id, h5_dims, NULL );
                    arr = malloc( h5_dims[0] * sizeof(xp_long) );
                    H5Aread( h5_attr_id, H5T_NATIVE_LONG_g, arr );
                }
                H5Sclose( h5_dspace_id );
            }
            H5Tclose( h5_dtype_id );
        }
        H5Aclose( h5_attr_id );
    }

    *n = h5_dims[0];
    *arr_ptr = arr;

    return METHOD_SUCCESS;
}

/* 64-bit porting. Newly Introduced */
void
HDF5write_int64_array_attr( hid_t parent_id, const char *name,
                     xp_long *array, xp_long size )
{
    hsize_t h5_dims = size;
    hid_t h5_dspace_id, h5_dset_id;

    h5_dspace_id = H5Screate_simple(1, &h5_dims, NULL);
    if( h5_dspace_id > 0 ) {
        h5_dset_id   = H5Dcreate(parent_id, name,
                                 H5T_NATIVE_INT64_g, h5_dspace_id,
                                 H5P_DEFAULT);
        if( h5_dset_id > 0 ) {
            H5Dwrite(h5_dset_id, H5T_NATIVE_LONG_g,
                     h5_dspace_id, h5_dspace_id, H5P_DEFAULT,
                     array);
            H5Dclose(h5_dset_id);
        }
        H5Sclose(h5_dspace_id);
    }
}
#endif

/* 64-bit porting. Directly Modified */
int
HDF5read_dims( hid_t h5_root_id, xp_long **fld_dims_ptr )
{
    xp_long fld_ndim;
    return HDF5read_long_array_attr( h5_root_id, "dims",
                                    &fld_ndim, fld_dims_ptr );
}
