/*
			Copyright (c) 2002 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/hdf5/xp_mods/h5_wr_fld.c#1 $
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>  /* time_t */

#define XP_WIDE_API	/* Use Wide APIs */
#include <avs/f_utils.h>
#include <avs/err.h>
#include <avs/om.h>
#include <avs/om_type.h> /* OM_name_value */
#include <avs/fld.h>
#include <avs/mat.h>

#include "h5_xp.h"

#define METHOD_SUCCESS 1
#define METHOD_FAILURE 0

#define ERR_RETURN(A) {ERRerror("HDF5 write field", 0, ERR_ORIG, A); \
                       return(METHOD_FAILURE);}

typedef enum { FLD_Unspecified = 0,
               FLD_UNIF,
               FLD_RECT,
               FLD_STRUCT,
               FLD_UNSTRUCT
} FLD_Type;

int HDF5write_field_update( OMobj_id, OMevent_mask, int );
int HDF5write_time_field_update( OMobj_id, OMevent_mask, int );
int HDF5_ARRwrite_field_update( OMobj_id, OMevent_mask, int );

static int HDF5write_field( hid_t h5_root_id, OMobj_id fld_id, int compress );

static int HDF5write_time_field( hid_t h5_root_id, OMobj_id fld_id, int compress );

static void HDF5write_data_array( hid_t h5_grp_id, OMobj_id data_array_id,
                                  int ndim, xp_long *dims, OMobj_id tmpl_id,
                                  int compress );

static int HDF5write_xform( hid_t h5_root_id, OMobj_id xform_id );

static int HDF5write_dims( hid_t h5_grp_id, OMobj_id fld_id,
                           int fld_type, xp_long **fld_dims_ptr );

static int HDF5write_coords( hid_t h5_grp_id, OMobj_id fld_id,
                             OMobj_id grd_tmpl_id,
                             int fld_type, int fld_ndim, xp_long *fld_dims,
                             int compress );

static int HDF5write_points( hid_t h5_grp_id, OMobj_id fld_id,
                             int fld_type );

static int HDF5write_cells ( hid_t h5_root_id, OMobj_id fld_id );

static int HDF5write_node_data( hid_t h5_root_id, OMobj_id fld_id,
                                OMobj_id ndata_id,
                                int fld_type, int fld_ndim, xp_long *fld_dims,
                                int compress );

static int HDF5write_cell_data (hid_t h5_root_id, OMobj_id fld_id,
                                OMobj_id cdata_id );

static int HDF5write_time_cell_data ( hid_t h5_root_id,	 OMobj_id fld_id,
                                      OMobj_id cdata_tmpl_id );


/* 64-bit porting. Only Modified Internally */
int
HDF5write_field_update(OMobj_id mod_id, OMevent_mask mask, int seq_num)
{
    OMobj_id file_id, trigger_id, tmp_in_id, fld_id;

    hid_t h5_file_id, h5_root_id;
    herr_t h5_stat;

    char *temp_str = NULL, *filename = NULL;
    char file_buf[AVS_PATH_MAX];
    /* char *root_str = NULL; */
    int newfile, tmp_int, compress;
    xp_long tmp_long;

    /* in */
    file_id = OMfind_subobj( mod_id, OMstr_to_name("filename"), OM_OBJ_RD );
    tmp_in_id  = OMfind_subobj( mod_id, OMstr_to_name("inFld"), OM_OBJ_RD );

    trigger_id  = OMfind_subobj( mod_id, OMstr_to_name("trigger"),
                                 OM_OBJ_RD );

    /* OK if missing entirely, means we are running in non-UI mode. */
    if( !OMis_null_obj( trigger_id ) ) {
        if( !(OM_EVENT_INST & mask) && !OMchanged( trigger_id, seq_num ) )
            return METHOD_SUCCESS;
    }

    if( OMget_str_val( file_id, &temp_str, 0 ) != OM_STAT_SUCCESS )
        ERR_RETURN( "Can't get filename." );

    filename = FILEmap_variables( temp_str, file_buf );

    if (temp_str) { free (temp_str); temp_str = NULL; }

    if (!filename) {
        ERR_RETURN( "Bad filename" );
    }

    if( OMget_name_int_val( mod_id,  OMstr_to_name("compress"), &compress )
        != OM_STAT_SUCCESS )
        /* If there is a problem, just turn off compression */
        compress = 0;

    /* Skip past links */
    if( OMget_obj_val( tmp_in_id, &fld_id ) != OM_STAT_SUCCESS ) {
       ERR_RETURN( "Can't access input field" );
    }

    /* Field's in pretty bad shape if we can't even get nnodes! */
    if( FLDget_nnodes( fld_id, &tmp_long ) != OM_STAT_SUCCESS ) {
        ERR_RETURN( "Could not get field nnodes" );
    }
    /* Likewise for nspace. */
    if( FLDget_nspace( fld_id, &tmp_int ) != OM_STAT_SUCCESS ) {
        ERR_RETURN( "Could not get field nspace" );
    }
    /* Initialize HDF5 library.  OK to call multiple times. */
    if( H5open() < 0 ) {
        ERR_RETURN( "Could not initialize HDF5 library." );
    }
    /* Detect a problem with global variables and DLL settings. */
    if( H5T_NATIVE_INT_g == 0 || H5T_NATIVE_INT_g == -1 ) {
        ERR_RETURN( "Could not initialize HDF5 library." );
    }

    if( OMget_name_int_val( mod_id,  OMstr_to_name("newfile"), &newfile )
        != OM_STAT_SUCCESS )
      newfile = 1;

    if( newfile == 0 ) {
        time_t mod_time = 0;
        if( !FILEexists(filename, FILE_READ|FILE_WRITE, &mod_time) )
          newfile = 1;
    }

    if( newfile ) {
        /*
         * Create a new HDF5 file using H5_ACC_TRUNC access, default
         * file creation properties, and default file access properties.
         */
        h5_file_id = H5Fcreate( filename,
                                H5F_ACC_TRUNC,  /* overwrite existing files */
                                H5P_DEFAULT,    /* default file creation    */
                                H5P_DEFAULT );  /* default file access      */
        if( h5_file_id < 0 ) {
            ERR_RETURN( "Couldn't create output file" );
        }
    }
    else {
        /*
         * Open a existing HDF5 file.
         */
        h5_file_id = H5Fopen( filename,
                              H5F_ACC_RDWR, /* read-write access to file */
                              H5P_DEFAULT );  /* default file access      */
        if( h5_file_id < 0 ) {
            ERR_RETURN( "Couldn't open output file" );
        }
    }


    /* Grab root group */
#if 0
    if( OMget_name_str_val( mod_id, OMstr_to_name("h5root"), &root_str, 0 )
        == OM_STAT_SUCCESS ) {
        h5_root_id = H5Gopen( h5_file_id, root_str );
        free( root_str );
    }
    else
#endif
        h5_root_id = H5Gopen(h5_file_id, "/");

    if( HDF5write_field( h5_root_id, fld_id, compress ) != METHOD_SUCCESS ) {
        ERR_RETURN( "Error while writing file" );
    }

    H5Gclose( h5_root_id );

    h5_stat = H5Fclose( h5_file_id );

    if( h5_stat < 0 ) {
        ERRerror( "HDF5 write_field", 1, ERR_ORIG,
                  "Error %d while closing HDF5 file", h5_stat );
      return METHOD_FAILURE;
    }

    return METHOD_SUCCESS;
}

/* 64-bit porting. Only Modified Internally */
int
HDF5write_time_field_update(OMobj_id mod_id, OMevent_mask mask, int seq_num)
{
    OMobj_id file_id, trigger_id, tmp_in_id, fld_id;

    hid_t h5_file_id, h5_root_id;
    herr_t h5_stat;

    char *temp_str = NULL, *filename = NULL;
    char file_buf[AVS_PATH_MAX];
    int tmp_int, compress =0;
    xp_long tmp_long;

    /* in */
    file_id = OMfind_subobj( mod_id, OMstr_to_name("filename"), OM_OBJ_RD );
    tmp_in_id  = OMfind_subobj( mod_id, OMstr_to_name("inFld"), OM_OBJ_RD );

    trigger_id  = OMfind_subobj( mod_id, OMstr_to_name("trigger"),
                                 OM_OBJ_RD );

    /* OK if missing entirely, means we are running in non-UI mode. */
    if( !OMis_null_obj( trigger_id ) ) {
        if( !(OM_EVENT_INST & mask) && !OMchanged( trigger_id, seq_num ) )
            return METHOD_SUCCESS;
    }

    if( OMget_str_val( file_id, &temp_str, 0 ) != OM_STAT_SUCCESS )
        ERR_RETURN( "Can't get filename." );

    filename = FILEmap_variables( temp_str, file_buf );

    if (temp_str) { free (temp_str); temp_str = NULL; }

    if (!filename) {
        ERR_RETURN( "Bad filename" );
    }

    if( OMget_name_int_val( mod_id,  OMstr_to_name("compress"), &compress )
        != OM_STAT_SUCCESS )
        /* If there is a problem, just turn off compression */
        compress = 0;

    /* Skip past links */
    if( OMget_obj_val( tmp_in_id, &fld_id ) != OM_STAT_SUCCESS ) {
       ERR_RETURN( "Can't access input field" );
    }

    /* Field's in pretty bad shape if we can't even get nnodes! */
    if( FLDget_nnodes( fld_id, &tmp_long ) != OM_STAT_SUCCESS ) {
        ERR_RETURN( "Could not get field nnodes" );
    }
    /* Likewise for nspace. */
    if( FLDget_nspace( fld_id, &tmp_int ) != OM_STAT_SUCCESS ) {
        ERR_RETURN( "Could not get field nspace" );
    }
    /* Initialize HDF5 library.  OK to call multiple times. */
    if( H5open() < 0 ) {
        ERR_RETURN( "Could not initialize HDF5 library." );
    }
    /* Detect a problem with global variables and DLL settings. */
    if( H5T_STD_I32LE_g == 0 ) {
        ERR_RETURN( "Could not initialize HDF5 library." );
    }

    /*
     * Create a new HDF5 file using H5_ACC_TRUNC access, default
     * file creation properties, and default file access properties.
     */
    h5_file_id = H5Fcreate( filename,
                            H5F_ACC_TRUNC,  /* overwrite existing files */
                            H5P_DEFAULT,    /* default file creation    */
                            H5P_DEFAULT     /* default file access      */
                            );

    if( h5_file_id < 0 ) {
        ERR_RETURN( "Couldn't create output file" );
    }

    /* Grab root group */
    h5_root_id = H5Gopen( h5_file_id, "/" );

    /* This is the only difference from HDF5write_field_update, combine? */
    if( HDF5write_time_field( h5_root_id, fld_id, compress ) != METHOD_SUCCESS ) {
        ERR_RETURN( "Error while writing file" );
    }

    H5Gclose( h5_root_id );

    h5_stat = H5Fclose( h5_file_id );

    if( h5_stat < 0 ) {
        ERRerror( "HDF5 write_field", 1, ERR_ORIG,
                  "Error %d while closing HDF5 file", h5_stat );
      return METHOD_FAILURE;
    }

    return METHOD_SUCCESS;
}

#if 0
int
HDF5_ARRwrite_field_update(OMobj_id mod_id, OMevent_mask mask, int seq_num)
{
    OMobj_id file_id, trigger_id, tmp_in_id, fldarr_id, fld_id;

    hid_t h5_file_id, h5_root_id, h5_grp_id;
    herr_t h5_stat;

    char *temp_str = NULL, *filename = NULL;
    char file_buf[AVS_PATH_MAX], buffer[64];
    int i, num_fields, stat;

    /* in */
    file_id = OMfind_subobj( mod_id, OMstr_to_name("filename"), OM_OBJ_RD );
    tmp_in_id  = OMfind_subobj( mod_id, OMstr_to_name("inFld"), OM_OBJ_RD );

    trigger_id  = OMfind_subobj( mod_id, OMstr_to_name("trigger"),
                                 OM_OBJ_RD );

    /* OK if missing entirely, means we are running in non-UI mode. */
    if( !OMis_null_obj( trigger_id ) ) {
        if( !(OM_EVENT_INST & mask) && !OMchanged( trigger_id, seq_num ) )
            return METHOD_SUCCESS;
    }

    if( OMget_str_val( file_id, &temp_str, 0 ) != OM_STAT_SUCCESS )
        ERR_RETURN( "Can't get filename." );

    filename = FILEmap_variables( temp_str, file_buf );

    if (temp_str) { free (temp_str); temp_str = NULL; }

    if (!filename) {
        ERR_RETURN( "Bad filename" );
    }

    /* Skip past links */
    if( OMget_obj_val( tmp_in_id, &fldarr_id ) != OM_STAT_SUCCESS ) {
       ERR_RETURN( "Can't access input field" );
    }

    stat = OMget_array_size( fldarr_id, &num_fields );
    if( stat != OM_STAT_SUCCESS ) return METHOD_FAILURE;
    if( num_fields == 0 ) return METHOD_SUCCESS;

    /* Initialize HDF5 library.  OK to call multiple times. */
    if( H5open() < 0 ) {
        ERR_RETURN( "Could not initialize HDF5 library." );
    }
    /* Detect a problem with global variables and DLL settings. */
    if( H5T_STD_I32LE_g == 0 ) {
        ERR_RETURN( "Could not initialize HDF5 library." );
    }

    /*
     * Create a new HDF5 file using H5_ACC_TRUNC access, default
     * file creation properties, and default file access properties.
     */
    h5_file_id = H5Fcreate( filename,
                            H5F_ACC_TRUNC,  /* overwrite existing files */
                            H5P_DEFAULT,    /* default file creation    */
                            H5P_DEFAULT     /* default file access      */
                            );

    if( h5_file_id < 0 ) {
        ERR_RETURN( "Couldn't create output file" );
    }

    /* Grab root group */
    h5_root_id = H5Gopen(h5_file_id, "/");

    HDF5write_int_attr( h5_root_id, "nblocks", num_fields );

    for( i = 0; i < num_fields; ++i ) {
        stat = OMget_array_val( fldarr_id, i, &fld_id, OM_OBJ_RD );
        if( stat != OM_STAT_SUCCESS ) continue;

        sprintf( buffer, "field[%d]", i );
        h5_grp_id = H5Gcreate(h5_root_id, buffer, 0);

        if( HDF5write_field( h5_grp_id, fld_id ) != METHOD_SUCCESS ) {
            ERR_RETURN( "Error while writing file" );
        }

        H5Gclose( h5_grp_id );
    }

    H5Gclose( h5_root_id );

    h5_stat = H5Fclose( h5_file_id );

    if( h5_stat < 0 ) {
        ERRerror( "HDF5 write_field", 1, ERR_ORIG,
                  "Error %d while closing HDF5 file", h5_stat );
      return METHOD_FAILURE;
    }

    return METHOD_SUCCESS;
}
#endif

/* 64-bit porting. Only Modified Internally */
static void
HDF5write_prim_attr( hid_t h5_parent_id, OMobj_id prim_id )
{
    int type;
    const char *name = NULL;
    OMobj_name obj_name;

    /* Assuming we want to use the name of the prim as
       the name of the HDF5 attribute.
    */
    if( OMget_obj_name( prim_id, &obj_name ) != OM_STAT_SUCCESS )
        return;
    name = OMname_to_str( obj_name );

    if (OMget_data_type( prim_id, &type ) == OM_STAT_SUCCESS) {
        hid_t h5_dtype_file, h5_dtype_app;
        hid_t h5_dspace_id, h5_attr_id;

        HDF5map_dtypes_wr( type, &h5_dtype_file, &h5_dtype_app );
        switch( type ) {
        case OM_TYPE_CHAR:
        case OM_TYPE_BYTE:
        case OM_TYPE_SHORT:
        case OM_TYPE_INT:
        {
            int value;
            if( OMget_int_val( prim_id, &value ) == OM_STAT_SUCCESS ) {
                h5_dspace_id = H5Screate( H5S_SCALAR );
                if( h5_dspace_id > 0 ) {
                    h5_attr_id = H5Acreate( h5_parent_id, name,
                                            h5_dtype_file, h5_dspace_id,
                                            H5P_DEFAULT );
                    if( h5_attr_id > 0 ) {
                        H5Awrite( h5_attr_id, H5T_NATIVE_INT_g, &value );
                        H5Aclose( h5_attr_id );	/* close attribute */
                    }
                    H5Sclose( h5_dspace_id );	/* close dataspace */
                }
            }
            break;
        }
        case OM_TYPE_LONG:
        {
            xp_long value;
            if( OMget_long_val( prim_id, &value ) == OM_STAT_SUCCESS ) {
                h5_dspace_id = H5Screate( H5S_SCALAR );
                if( h5_dspace_id > 0 ) {
                    h5_attr_id = H5Acreate( h5_parent_id, name,
                                            h5_dtype_file, h5_dspace_id,
                                            H5P_DEFAULT );
                    if( h5_attr_id > 0 ) {
                        H5Awrite( h5_attr_id, H5T_NATIVE_XP_LONG_g, &value );
                        H5Aclose( h5_attr_id );	/* close attribute */
                    }
                    H5Sclose( h5_dspace_id );	/* close dataspace */
                }
            }
            break;
        }
        case OM_TYPE_FLOAT:
        case OM_TYPE_DOUBLE:
        {
            double value;
            if( OMget_real_val( prim_id, &value ) == OM_STAT_SUCCESS ) {
                h5_dspace_id = H5Screate( H5S_SCALAR );
                if( h5_dspace_id > 0 ) {
                    h5_attr_id = H5Acreate( h5_parent_id, name,
                                            h5_dtype_file, h5_dspace_id,
                                            H5P_DEFAULT );
                    if (h5_attr_id > 0) {
                        H5Awrite( h5_attr_id, H5T_NATIVE_DOUBLE_g, &value );
                        H5Aclose( h5_attr_id );	/* close attribute */
                    }
                    H5Sclose( h5_dspace_id );	/* close dataspace */
                }
            }
        }
        }
    }
    return;
}

/* 64-bit porting. Only Modified Internally */
static void
HDF5write_prim_array_attr( hid_t h5_parent_id, OMobj_id prim_id )
{
    void *array = NULL;
    int type;
    xp_long size;

    const char *name = NULL;
    OMobj_name obj_name;

    /* Assuming we want to use the name of the prim as
       the name of the H5 attribute.
    */
    if( OMget_obj_name( prim_id, &obj_name ) != OM_STAT_SUCCESS )
        return;
    name = OMname_to_str( obj_name );

    if( OMget_data_type( prim_id, &type ) == OM_STAT_SUCCESS )
        array = OMret_array_ptr( prim_id, OM_GET_ARRAY_RD, &size, &type );

    if( array != NULL ) {
        hsize_t h5_dims[1];
        hid_t h5_dtype_file, h5_dtype_app;
        hid_t h5_dspace_id, h5_attr_id;

        HDF5map_dtypes_wr( type, &h5_dtype_file, &h5_dtype_app );

        h5_dims[0] = size;
        h5_dspace_id = H5Screate_simple( 1, h5_dims, NULL );
        if( h5_dspace_id > 0 ) {
            h5_attr_id   = H5Acreate( h5_parent_id, name,
                                      h5_dtype_file, h5_dspace_id,
                                      H5P_DEFAULT );
            if( h5_attr_id > 0 ) {
                H5Awrite( h5_attr_id, h5_dtype_app, array );
                H5Aclose( h5_attr_id );	/* close attribute */
            }
            H5Sclose( h5_dspace_id );	/* close dataspace */
        }

        ARRfree( array );
    }
}

static void
get_class_str(OMobj_id obj_id, char *buffer, int length)
{
#define MAX_CLASS 64
    OMobj_id classes[MAX_CLASS];
    OMobj_name class_name;
    int i, num_classes;

    buffer[0] = 0;

    if (OMget_user_classes(obj_id, classes, &num_classes,
                           MAX_CLASS, 0,
                           OM_CLASS_FOLLOW_REFS) == OM_STAT_SUCCESS) {
        for (i = 0; i < num_classes; ++i) {
            if( OMget_obj_name( classes[i], &class_name) == OM_STAT_SUCCESS ) {
                char *name_str = OMname_to_str( class_name );
                if((int)(strlen(buffer) + strlen(name_str) + 2) < length ) {
                    if( buffer[0] == 0 )
                        strcpy( buffer, name_str );
                    else {
                        strcat( buffer, "+" );
                        strcat( buffer, name_str );
                    }
                }
            }
        }
    }
#undef MAX_CLASS
}

/* 64-bit porting. Only Modified Internally */
static void
discover_grid( OMobj_id fld_id,
               int *grid_type, OMobj_id *grid_id  )
{
    static OMobj_id grd_tmpl, grd_unif_tmpl, grd_rect_tmpl, grd_struct_tmpl;
    static OMobj_id grd_rect_cyl_tmpl, grd_rect_sphr_tmpl;
    static OMobj_id grd_unif_cyl_tmpl, grd_unif_sphr_tmpl;
    static int init = 0;

    int ndim, g_type;
    xp_long npoints;
    char class_str[256];

    get_class_str( fld_id, class_str, sizeof(class_str) );

#if 0
    fprintf( stderr, "CLASS_ST: %s\n", class_str );
#endif

#undef FIND
#define FIND(tmpl, str) OMfind_subobj( tmpl, OMstr_to_name( str ), OM_OBJ_RD )

    if( !init ) {
        static OMobj_id FLD_tmpl;

        FLD_tmpl = FIND(OMtempl_obj, "FLD");
        grd_tmpl =           FIND(FLD_tmpl, "Grid");
        grd_struct_tmpl =    FIND(FLD_tmpl, "Grid_Struct");
        grd_unif_tmpl =      FIND(FLD_tmpl, "Grid_Unif");
        grd_rect_tmpl =      FIND(FLD_tmpl, "Grid_Rect");
        grd_unif_cyl_tmpl =  FIND(FLD_tmpl, "Grid_Unif_Rect");
        grd_rect_cyl_tmpl =  FIND(FLD_tmpl, "Grid_Cyl_Rect");
        grd_unif_sphr_tmpl = FIND(FLD_tmpl, "Grid_Spher_Unif");
        grd_rect_sphr_tmpl = FIND(FLD_tmpl, "Grid_Spher_Rect");
    }

#undef FIND

    /* I've decided I don't trust the OMmatch approach.  Matching
       prims works pretty well, but matching groups doesn't always
       work correctly.  I think the group matching code doesn't handle
       the combination of OM_MATCH_STRUCT and OM_MATCH_EXACT correctly;
       but you need OM_MATCH_STRUCT passed down to the prim matching
       code to correctly mach the prims that are sub-elements of the groups.
    */

#define match_flags (OM_MATCH_EXACT | OM_MATCH_NO_ERROR)

    /* Some obscure ones first. Note that the Sphr/Cyl guys
       don't have grid_type set, which makes OMmatch (even more) unreliable.
    */
    if     ( strstr( class_str, "_Cyl_Unif" ) != NULL ) {
        *grid_type = FLD_UNIF;
        if( grid_id != NULL ) *grid_id   = grd_unif_cyl_tmpl;
        return;
    }
    else if( strstr( class_str, "_Cyl_Rect" ) != NULL ) {
        *grid_type = FLD_RECT;
        if( grid_id != NULL ) *grid_id   = grd_rect_cyl_tmpl;
        return;
    }
    else if( strstr( class_str, "_Spher_Unif" ) != NULL ) {
        *grid_type = FLD_UNIF;
        if( grid_id != NULL ) *grid_id   = grd_unif_sphr_tmpl;
        return;
    }
    else if( strstr( class_str, "_Spher_Rect" ) != NULL ) {
        *grid_type = FLD_RECT;
        if( grid_id != NULL ) *grid_id   = grd_rect_sphr_tmpl;
        return;
    }

    /* The more general ones */
    *grid_type = FLD_Unspecified;

    /* if grid_type is present, lets use it first */
    if( FLDget_grid_type( fld_id, &g_type ) == OM_STAT_SUCCESS ) {
        if( g_type == 1 ) *grid_type = FLD_UNIF;
        if( g_type == 2 ) *grid_type = FLD_RECT;
    }
    else if( FLDget_ndim( fld_id, &ndim ) != OM_STAT_SUCCESS )
        /* No "ndim": probably unstructured */
        *grid_type = FLD_UNSTRUCT;
    else if( FLDget_npoints( fld_id, &npoints ) != OM_STAT_SUCCESS )
        /* Unset "npoints": probably structured. */
        *grid_type = FLD_STRUCT;
    else if( npoints == 2 )
        /* "npoints" is 2: probably uniform. */
        *grid_type = FLD_UNIF;

    if( *grid_type == FLD_Unspecified ) {
        xp_long *dims = NULL, dims_sum = 0;
        int dims_size, i;
        /* test "npoints => sum(dims)" */
        if( FLDget_dims( fld_id, &dims, &dims_size ) == OM_STAT_SUCCESS ) {
            for( i = 0; i < ndim; ++i ) dims_sum += dims[i];
            ARRfree( dims );
            if( dims_sum == npoints ) *grid_type = FLD_RECT;
        }
    }

    if( *grid_type == FLD_Unspecified ) *grid_type = FLD_STRUCT;

    if( grid_id != NULL ) {
        switch( *grid_type ) {
            case FLD_UNIF:     *grid_id = grd_unif_tmpl; break;
            case FLD_RECT:     *grid_id = grd_rect_tmpl; break;
            case FLD_STRUCT:   *grid_id = grd_struct_tmpl; break;
            case FLD_UNSTRUCT: *grid_id = grd_tmpl;
        }
    }

#undef match_flags

}

/* Main routine for regular (non-time-dependent) fields. */
/* 64-bit porting. Only Modified Internally */
static int
HDF5write_field( hid_t h5_root_id, OMobj_id fld_id, int compress )
{
    int fld_type, fld_ndim, fld_nspace;
    xp_long fld_nnodes, *fld_dims = NULL;

    OMobj_id grid_tmpl;  /* for looking up coordinates.min_vec, etc. */
    OMobj_id ndat_tmpl;  /* for looking up node_data.min_vec, etc. */
    OMobj_id cdat_tmpl;  /* for looking up cell_set.cell_data.min_vec, etc. */

    /* Field's in pretty bad shape if we can't even get nnodes! */
    if( FLDget_nnodes( fld_id, &fld_nnodes ) != OM_STAT_SUCCESS ) {
        ERR_RETURN( "Could not get field nnodes" );
    }
    /* Likewise for nspace. */
    if( FLDget_nspace( fld_id, &fld_nspace ) != OM_STAT_SUCCESS ) {
        ERR_RETURN( "Could not get field nspace" );
    }

    /*
     * First things first.  What kind of input field do we have?
     */
    discover_grid( fld_id, &fld_type, &grid_tmpl );

    ndat_tmpl = OMfind_subobj( OMtempl_obj, OMstr_to_name( "Node_Data" ),
                               OM_OBJ_RD );
    cdat_tmpl = OMfind_subobj( OMtempl_obj, OMstr_to_name( "Cell_Data" ),
                               OM_OBJ_RD );

    /* Write XP_CLASS as a global string attribute */
    {
        char xp_class_name[256];
        int cells_flag = 0, xform_flag = 0;
        int node_data_flag = 0, cell_data_flag = 0;
        OMobj_id tmp_id;

        /* It might be better to match templates, but in practice
           this code won't be fooled very often.  A gray area is
           when e.g. the Node_Data type is present, but "nnode_data"
           is either unset or 0.  It gets messier when you have to look
           into arrays of groups.  Consider "cell_set[0].ncell_data" having
           a valid value, but "cell_set[1].ncell_data" being unset.
        */

        tmp_id =  OMfind_subobj( fld_id, OMstr_to_name("ncell_sets"),
                                 OM_OBJ_RD );
        cells_flag = !OMis_null_obj( tmp_id );
        if( cells_flag ) {
            tmp_id =  OMfind_str_subobj( fld_id, "cell_set.ncell_data",
                                         OM_OBJ_RD );
            cell_data_flag = !OMis_null_obj( tmp_id );
        }

        tmp_id =  OMfind_subobj( fld_id, OMstr_to_name("xform"),
                                 OM_OBJ_RD );
        xform_flag = !OMis_null_obj( tmp_id );

        tmp_id =  OMfind_subobj( fld_id, OMstr_to_name("nnode_data"),
                                 OM_OBJ_RD );
        node_data_flag = !OMis_null_obj( tmp_id );

        if( !xform_flag && !cells_flag ) {
            const char *tmp_str;
            switch( fld_type ) {
            case FLD_UNIF:     tmp_str = "Grid_Unif"; break;
            case FLD_RECT:     tmp_str = "Grid_Rect"; break;
            case FLD_STRUCT:   tmp_str = "Grid_Struct"; break;
            case FLD_UNSTRUCT: tmp_str = "Grid";
            }
            strcpy( xp_class_name, tmp_str );
        }
        else {
            const char *tmp_str;
            switch( fld_type ) {
            case FLD_UNIF:     tmp_str = "Mesh_Unif"; break;
            case FLD_RECT:     tmp_str = "Mesh_Rect"; break;
            case FLD_STRUCT:   tmp_str = "Mesh_Struct"; break;
            case FLD_UNSTRUCT: tmp_str = "Mesh";
            }
            strcpy( xp_class_name, tmp_str );
        }
        if( node_data_flag ) {
            strcat( xp_class_name, "+Node_Data" );
        }
        if( cell_data_flag ) {
            strcat( xp_class_name, "+Cell_Data" );
        }

#if 0
        fprintf( stderr, "XP_CLASS: %s\n", xp_class_name );
#endif
        HDF5write_string_attr( h5_root_id, "XP_CLASS", xp_class_name );
    }

    /* Write field.nnodes as a global scalar int attribute. */
    /* Only needed for an unstructured field. */
    if( fld_type == FLD_UNSTRUCT ) {
        HDF5write_long_attr( h5_root_id, "nnodes", fld_nnodes );
    }

    /* Write field.nspace as a global scalar int attribute. */
    HDF5write_int_attr( h5_root_id, "nspace", fld_nspace );

    if( fld_type != FLD_UNSTRUCT ) {
        if( FLDget_ndim( fld_id, &fld_ndim ) != OM_STAT_SUCCESS ) {
            ERR_RETURN( "Could not get field ndim" );
        }
    }
    else {
        /* little trick for unstruct fields to help the node_data code. */
        fld_ndim = 1;
    }

    /* Grid */

    {
        /* Only expect to see this in Unif and Rect; somewhat redundant
         * with the class string and ndim/dim information.
         */
        int grid_type;
        if( FLDget_grid_type( fld_id, &grid_type ) == OM_STAT_SUCCESS ) {
            HDF5write_int_attr( h5_root_id, "grid_type", grid_type );
        }
    }

    if( HDF5write_dims( h5_root_id, fld_id,
                        fld_type, &fld_dims ) != METHOD_SUCCESS )
        return METHOD_FAILURE;

    if( HDF5write_coords( h5_root_id, fld_id, grid_tmpl,
                          fld_type, fld_ndim, fld_dims, compress ) != METHOD_SUCCESS )
        return METHOD_FAILURE;

    if( HDF5write_points( h5_root_id, fld_id, fld_type ) != METHOD_SUCCESS )
        return METHOD_FAILURE;

    /* Xform */
    if( HDF5write_xform( h5_root_id, fld_id ) != METHOD_SUCCESS )
        return METHOD_FAILURE;

    /* For Unstructured fields, we need to worry about cell set information.
     * Struct/Rect/Unif fields have cell set info as well, but its
     * almost always a single default cell set.
     */
    if( fld_type == FLD_UNSTRUCT ) {
        HDF5write_cells( h5_root_id, fld_id );
    }

    /* Node_Data */
    HDF5write_node_data( h5_root_id, fld_id,
                         ndat_tmpl, fld_type, fld_ndim, fld_dims, compress );

    /* Cell_Data */
    HDF5write_cell_data( h5_root_id, fld_id, cdat_tmpl );

    if( fld_dims ) ARRfree( fld_dims );

    return METHOD_SUCCESS;
}

/* Main routine for time-dependent fields. */
/* 64-bit porting. Only Modified Internally */
static int
HDF5write_time_field( hid_t h5_root_id, OMobj_id fld_id, int compress )
{
    char buffer[64];
    xp_long fld_nnodes;
    int fld_nspace, fld_nsteps;
    int fld_type, fld_ndim;
    xp_long *fld_dims = NULL;
    int t;

    OMobj_id grid_tmpl;  /* for looking up coordinates.min_vec, etc. */
    OMobj_id ndat_tmpl;  /* for looking up node_data.min_vec, etc. */
    OMobj_id cdat_tmpl;  /* for looking up cell_set.cell_data.min_vec, etc. */

    OMobj_id elem_id;

    /* Field is in pretty bad shape if we can't even get nnodes! */
    if( FLDget_nnodes( fld_id, &fld_nnodes ) != OM_STAT_SUCCESS ) {
        ERR_RETURN( "Could not get field nnodes" );
    }
    /* Likewise for nspace. */
    if( FLDget_nspace( fld_id, &fld_nspace ) != OM_STAT_SUCCESS ) {
        ERR_RETURN( "Could not get field nspace" );
    }
    /* Likewise for nsteps in a time-dependent field. */
    if( FLDget_time_nsteps( fld_id, &fld_nsteps ) != OM_STAT_SUCCESS ) {
        ERR_RETURN( "Could not get field nsteps" );
    }

    /*
     * First things first.  What kind of input field do we have?
     */
    discover_grid( fld_id, &fld_type, &grid_tmpl );

    /* These could be put in static variables as they won't change. */
    ndat_tmpl = OMfind_subobj( OMtempl_obj, OMstr_to_name( "Node_Data" ),
                               OM_OBJ_RD );
    cdat_tmpl = OMfind_subobj( OMtempl_obj, OMstr_to_name( "Cell_Data" ),
                               OM_OBJ_RD );

    /* Write XP_CLASS as a global string attribute */
    {
        char xp_class_name[256];
        int cells_flag = 0, xform_flag = 0;
        int node_data_flag = 0, cell_data_flag = 0;
        OMobj_id tmp_id;

        /* A gray area is
           when e.g. the Node_Data type is present, but "nnode_data"
           is either unset or 0.  It gets messier when you have to look
           into arrays of groups.  Consider "cell_set[0].ncell_data" having
           a valid value, but "cell_set[1].ncell_data" being unset.
        */

        tmp_id =  OMfind_subobj( fld_id, OMstr_to_name("ncell_sets"),
                                 OM_OBJ_RD );
        cells_flag = !OMis_null_obj( tmp_id );
        if( cells_flag ) {
            tmp_id =  OMfind_str_subobj( fld_id, "cell_set.ncell_data",
                                         OM_OBJ_RD );
            cell_data_flag = !OMis_null_obj( tmp_id );
        }

        tmp_id =  OMfind_subobj( fld_id, OMstr_to_name("xform"),
                                 OM_OBJ_RD );
        xform_flag = !OMis_null_obj( tmp_id );

        tmp_id =  OMfind_subobj( fld_id, OMstr_to_name("nnode_data"),
                                 OM_OBJ_RD );
        node_data_flag = !OMis_null_obj( tmp_id );

        /* After running extract_time_step you get a field that has
         * BOTH, i.e. "time_coordinates" and "coordinates".  So, rather
         * then testing for "time_xx" to determine if its time-dependent,
         * we need to test for "xx" to determine if its not time-dependent.
         */

        tmp_id =  OMfind_subobj( fld_id, OMstr_to_name("coordinates"),
                                 OM_OBJ_RD );

        if( !OMis_null_obj( tmp_id )
            || fld_type == FLD_UNIF || fld_type == FLD_RECT )
            /* Unif/Rect fields cannot have time-dependent meshes/grids */
            xp_class_name[0] = 0;
        else {
            tmp_id =  OMfind_subobj( fld_id, OMstr_to_name("time_coordinates"),
                                     OM_OBJ_RD );
            if( !OMis_null_obj( tmp_id ) )
                strcpy( xp_class_name, "Time_" );
            /* else inconsistent, a Struct/Unstruct field without
               coordinates ??
            */
        }

        if( !xform_flag && !cells_flag ) {
            const char *tmp_str;
            switch( fld_type ) {
            case FLD_UNIF:     tmp_str = "Grid_Unif"; break;
            case FLD_RECT:     tmp_str = "Grid_Rect"; break;
            case FLD_STRUCT:   tmp_str = "Grid_Struct"; break;
            case FLD_UNSTRUCT: tmp_str = "Grid";
            }
            strcat( xp_class_name, tmp_str );
        }
        else {
            const char *tmp_str;
            switch( fld_type ) {
            case FLD_UNIF:     tmp_str = "Mesh_Unif"; break;
            case FLD_RECT:     tmp_str = "Mesh_Rect"; break;
            case FLD_STRUCT:   tmp_str = "Mesh_Struct"; break;
            case FLD_UNSTRUCT: tmp_str = "Mesh";
            }
            strcat( xp_class_name, tmp_str );
        }

        if( node_data_flag ) {
            tmp_id =  OMfind_subobj( fld_id, OMstr_to_name("node_data"),
                                     OM_OBJ_RD );
            if( OMis_null_obj( tmp_id ) ) {
                tmp_id =  OMfind_subobj( fld_id,
                                         OMstr_to_name("time_node_data"),
                                         OM_OBJ_RD );
                if( !OMis_null_obj( tmp_id ) )
                    strcat( xp_class_name, "+Time_Node_Data" );
                /* else inconsistent, "nnode_data" (from above), but not
                   either "node_data" or "time_node_data" ??
                */
            }
            else
                strcat( xp_class_name, "+Node_Data" );
        }

        if( cell_data_flag ) {
            tmp_id = OMfind_str_subobj( fld_id, "cell_set.cell_data",
                                        OM_OBJ_RD );
            if( OMis_null_obj( tmp_id ) ) {
                tmp_id = OMfind_str_subobj( fld_id, "cell_set.time_cell_data",
                                            OM_OBJ_RD );
                if( !OMis_null_obj( tmp_id ) )
                    strcat( xp_class_name, "+Time_Cell_Data" );
                /* else inconsistent */
            }
            else
                strcat( xp_class_name, "+Cell_Data" );
        }

#if 0
        fprintf( stderr, "XP_CLASS: %s\n", xp_class_name );
#endif
        HDF5write_string_attr( h5_root_id, "XP_CLASS", xp_class_name );
    }

    /* Write field.nnodes as a global scalar int attribute. */
    /* Only needed for an unstructured field. */
    if( fld_type == FLD_UNSTRUCT ) {
        HDF5write_long_attr( h5_root_id, "nnodes", fld_nnodes );
    }

    /* Write field.nspace as a global scalar int attribute. */
    HDF5write_int_attr( h5_root_id, "nspace", fld_nspace );

    /* "double time[nsteps]" */
    elem_id = OMfind_subobj( fld_id, OMstr_to_name( "time" ), OM_OBJ_RD );
    if( !OMis_null_obj( elem_id ) ) {
        HDF5write_prim_array_attr( h5_root_id, elem_id );
    }
    else {
        ERR_RETURN( "Could not get field time" );
    }

    /* Write field.nsteps as a global scalar int attribute. */
    HDF5write_int_attr( h5_root_id, "nsteps", fld_nsteps );

    if( fld_type != FLD_UNSTRUCT ) {
        if( FLDget_ndim( fld_id, &fld_ndim ) != OM_STAT_SUCCESS ) {
            ERR_RETURN( "Could not get field ndim" );
        }
    }
    else {
        /* little trick for unstruct fields to help the node_data code. */
        fld_ndim = 1;
    }

    /* Grid ... in three parts */

    {
        /* Only expect to see this in Unif and Rect; somewhat redundant
         * with the class string and ndim/dim information.
         */
        int grid_type;
        if( FLDget_grid_type( fld_id, &grid_type ) == OM_STAT_SUCCESS ) {
            HDF5write_int_attr( h5_root_id, "grid_type", grid_type );
        }
    }

    /* dims for all struct fields, never time-dependent */
    if( HDF5write_dims( h5_root_id, fld_id,
                        fld_type, &fld_dims ) != METHOD_SUCCESS )
        return METHOD_FAILURE;

    /* coordinates for struct and unstruct fields, can be time-dependent */
    elem_id = OMfind_subobj( fld_id, OMstr_to_name( "coordinates" ),
                             OM_OBJ_RD );

    if( !OMis_null_obj( elem_id ) ) {
        /* coordinates not time-dependent */
        if( HDF5write_coords( h5_root_id, fld_id, grid_tmpl, fld_type,
                              fld_ndim, fld_dims, compress ) != METHOD_SUCCESS )
            return METHOD_FAILURE;
    }
    else {
        elem_id = OMfind_subobj( fld_id, OMstr_to_name( "time_coordinates" ),
                                 OM_OBJ_RD );
        /* coordinates are time-dependent */
        for( t = 0; t < fld_nsteps; ++t ) {
            OMobj_id c_id;
            hid_t h5_grp_id;

            sprintf( buffer, "time_coordinates[%d]", t );
            h5_grp_id = H5Gcreate(h5_root_id, buffer, 0);
            if( h5_grp_id > 0 ) {
                if( OMget_array_val( elem_id, t, &c_id,
                                     OM_OBJ_RD ) == OM_STAT_SUCCESS ) {
                    if( HDF5write_coords( h5_grp_id, c_id, grid_tmpl,
                                          fld_type, fld_ndim, fld_dims,
                                          compress ) != METHOD_SUCCESS )
                        return METHOD_FAILURE;
                }
                H5Gclose( h5_grp_id );
            }
        }
    }

    /* points for rect and unif fields, never time-dependent */
    if( HDF5write_points( h5_root_id, fld_id, fld_type )
        != METHOD_SUCCESS )
        return METHOD_FAILURE;

    /* Xform is never time-dependent */
    if( HDF5write_xform( h5_root_id, fld_id ) != METHOD_SUCCESS )
        return METHOD_FAILURE;

    /* Cells are never time-dependent, although cell_data might be. */
    if( fld_type == FLD_UNSTRUCT ) {
        HDF5write_cells( h5_root_id, fld_id );
    }

    /* Node_Data */
    elem_id = OMfind_subobj( fld_id, OMstr_to_name( "node_data" ),
                             OM_OBJ_RD );
    if( !OMis_null_obj( elem_id ) ) {
        /* Node data is not time-dependent */
        HDF5write_node_data( h5_root_id, fld_id,
                             ndat_tmpl, fld_type, fld_ndim, fld_dims, compress );
    }
    else {
        /* Node data is time-dependent */
        int fld_ncomp = 0;
        if( (FLDget_node_data_ncomp( fld_id, &fld_ncomp )
             == OM_STAT_SUCCESS) &&
             (fld_ncomp != 0) ) {
            HDF5write_int_attr( h5_root_id, "nnode_data", fld_ncomp );
        }

        elem_id = OMfind_subobj( fld_id, OMstr_to_name( "time_node_data" ),
                                 OM_OBJ_RD );
        if( (fld_ncomp != 0) && !OMis_null_obj(elem_id) ) {
            for( t = 0; t < fld_nsteps; ++t ) {
                OMobj_id n_id;
                if( OMget_array_val( elem_id, t, &n_id,
                                     OM_OBJ_RD ) == OM_STAT_SUCCESS ) {
                    hid_t h5_grp_id;
                    sprintf( buffer, "time_node_data[%d]", t );
                    h5_grp_id = H5Gcreate(h5_root_id, buffer, 0);
                    if( h5_grp_id > 0 ) {
                        HDF5write_node_data( h5_grp_id, n_id,
                                             ndat_tmpl, fld_type,
                                             fld_ndim, fld_dims, compress );
                    }
                    H5Gclose( h5_grp_id );
                }
            }
        }
    }

    /* Cell_Data */
    elem_id = OMfind_str_subobj( fld_id, "cell_set.cell_data",
                                 OM_OBJ_RD );
    if( !OMis_null_obj( elem_id ) ) {
        /* Cell data not time-dependent */
        HDF5write_cell_data( h5_root_id, fld_id, cdat_tmpl );
    }
    else {
        /* Cell data is time-dependent.
           Need to do time looping in lower-level function due to
           the way time-dependent cell data is nested.
        */
        HDF5write_time_cell_data( h5_root_id, fld_id, cdat_tmpl );
    }

    if( fld_dims ) ARRfree( fld_dims );

    return METHOD_SUCCESS;
}


static int
HDF5are_the_same(OMobj_id obj_id, OMobj_id tmpl_id)
{
    OMobj_id dval_id;
    int type, mode;

    if (OMget_data_type(obj_id, &type) == OM_STAT_SUCCESS) {
        /* prim */
        mode =  OM_MATCH_STRUCT | OM_MATCH_EXACT | OM_MATCH_NO_ERROR;
    }
    else {
        /* group */
        mode =  OM_MATCH_EXACT | OM_MATCH_NO_ERROR;
    }

    dval_id = OMfind_subobj( tmpl_id, OM_name_value, OM_OBJ_RD );
#if 0
    {
        char *p1, buf1[256];
        char *p2, buf2[256];
        char *p3, buf3[256];
        p1 = OMret_obj_path( obj_id, buf1, sizeof(buf1) );
        p2 = OMret_obj_path( tmpl_id, buf2, sizeof(buf2) );
        p3 = OMret_obj_path( dval_id, buf3, sizeof(buf3) );
    }
#endif

    return(OMmatch_obj(tmpl_id, obj_id, mode));
}

/*
 * Write a Data_Array.  Could be part of Node_Data, Cell_Data
 * or Grid.coordinates.
 */
/* 64-bit porting. Directly Modified */
static void
HDF5write_data_array(
    hid_t h5_grp_id,		/* Group in which to write the data */
    OMobj_id data_array_id, 	/* OM id of an object of type 'Data_Array' */
    int ndim, xp_long *dims,	/* dimensionality of data. */
    OMobj_id tmpl_id,
    int compress
    )
{
    OMobj_id elem_id;
    char *p, buff[64];
    int veclen, id, null_flag;
    xp_long nvals;

    if( h5_grp_id < 0 ) return;

    p = buff;

    /* nvals is always attached to something else; but use it for
     * the dims if nothing was passed in for the dimensions.
     */
    if( ndim == 0 || dims == NULL ) {
        elem_id = OMfind_subobj (data_array_id, OMstr_to_name("nvals"),
                                 OM_OBJ_RD);
        ndim = 1;
        OMget_long_val( elem_id, &nvals );
        dims = &nvals;
    }

    /* veclen */
    elem_id = OMfind_subobj (data_array_id, OMstr_to_name("veclen"),
                             OM_OBJ_RD);
    if ((!OMis_null_obj(elem_id)) &&
        (OMget_int_val (elem_id, &veclen) == OM_STAT_SUCCESS)) {

        OMobj_id tmp_id;

        /* veclen is necessary for Node_Data and Cell_Data, (although
           1 is a reasonable default), but is always connected to
           nspace with coordinates.
        */
        tmp_id = OMfind_subobj( tmpl_id, OMstr_to_name("veclen"), OM_OBJ_RD );
        if( !OMis_null_obj(tmp_id) ) {
            if( HDF5are_the_same( elem_id, tmp_id ) != OM_STAT_SUCCESS ) {
                HDF5write_int_attr( h5_grp_id, "veclen", veclen );
            }
        }
        else
            HDF5write_int_attr( h5_grp_id, "veclen", veclen );
    }

    /* id */
    elem_id = OMfind_subobj (data_array_id, OMstr_to_name("id"),
                             OM_OBJ_RD);
    if ((!OMis_null_obj(elem_id)) &&
        (OMget_int_val (elem_id, &id) == OM_STAT_SUCCESS)) {
        HDF5write_int_attr( h5_grp_id, "id", id );
    }

    /* values */
    elem_id = OMfind_subobj( data_array_id, OMstr_to_name("values"),
                             OM_OBJ_RD );
    if( !OMis_null_obj(elem_id) ) {
        void *array = NULL;
        int type;

        if( OMget_data_type(elem_id, &type) == OM_STAT_SUCCESS )
            array = OMret_array_ptr( elem_id, OM_GET_ARRAY_RD, NULL, &type );

        if( array != NULL ) {
            hsize_t h5_dims[16];
            int i, h5_ndim = ndim;
            hid_t h5_dtype_file, h5_dtype_app;
            hid_t h5_dspace_id, h5_dset_id;

            /*
             * Flip the dimensions because the OM stores them
             * column-major (Fortran-style), not row-major (C-style)
             * and HDF uses row-major for the dataspace dimensions.
             * When we read the dataset in one swell foop, we don't
             * need to care too much about the dataspace dimensions,
             * other than getting the total size correct, but we
             * need the dataspace dimensions to do subsetting
             * (crop, downsize) correctly.
             */
            for( i = 0; i < ndim; ++i ) {
                h5_dims[ndim - (i + 1)] = dims[i];
            }
            /* Make veclen another HDF5 dimension. */
            if( veclen != 1 ) {
                h5_dims[i] = veclen;
                ++h5_ndim;
            }

            HDF5map_dtypes_wr( type, &h5_dtype_file, &h5_dtype_app );
            h5_dspace_id = H5Screate_simple( h5_ndim, h5_dims, NULL );
            if( h5_dspace_id > 0 ) {

                hid_t h5_pl_id = H5P_DEFAULT;

                if( compress > 0 ) {
                    /* Turn on compression. */
                    h5_pl_id = H5Pcreate( H5P_DATASET_CREATE );
                    H5Pset_layout ( h5_pl_id, H5D_CHUNKED );
                    H5Pset_chunk  ( h5_pl_id, h5_ndim, h5_dims );
                    /* Allow control over compression value? (1 - 9) */
                    H5Pset_deflate( h5_pl_id, 9 );
                }

                h5_dset_id   = H5Dcreate( h5_grp_id, "values",
                                          h5_dtype_file, h5_dspace_id,
                                          h5_pl_id );
                if( h5_dset_id > 0 ) {
                    H5Dwrite( h5_dset_id, h5_dtype_app,
                              h5_dspace_id, h5_dspace_id, H5P_DEFAULT,
                              array );
                    H5Dclose( h5_dset_id );	/* close dataset */
                }
                if( compress > 0 ) {
                    H5Pclose( h5_pl_id );	/* close property list */
                }

                H5Sclose( h5_dspace_id );	/* close dataspace */
            }

            ARRfree( array );
        }
    }

    /* null_flag */
    elem_id = OMfind_subobj (data_array_id, OMstr_to_name("null_flag"),
                             OM_OBJ_RD);
    if ((!OMis_null_obj(elem_id)) &&
        (OMget_int_val (elem_id, &null_flag) == OM_STAT_SUCCESS) &&
        (null_flag != 0)) {

        HDF5write_int_attr( h5_grp_id, "null_flag", null_flag );

        /* null_value */

        elem_id = OMfind_subobj (data_array_id, OMstr_to_name("null_value"),
                                 OM_OBJ_RD);
        if (!OMis_null_obj(elem_id))
            HDF5write_prim_attr( h5_grp_id, elem_id );
    }

    if( !OMis_null_obj( tmpl_id ) ) {
        OMobj_id tmp_id;

    /* min */
        elem_id = OMfind_subobj (data_array_id, OMstr_to_name("min"),
                                 OM_OBJ_RD);
        tmp_id = OMfind_subobj( tmpl_id, OMstr_to_name("min"), OM_OBJ_RD );

#if 0
        /* debugging code */
        {
            OMobj_id val_id;
            char *p1, buff1[256];
            char *p2, buff2[256];
            char *p3, buff3[256];
            val_id = OMfind_subobj( tmp_id, OM_name_value, OM_OBJ_RD );
            p1 = OMret_obj_path( tmpl_id, buff1, sizeof(buff1) );
            p2 = OMret_obj_path( tmp_id,  buff2, sizeof(buff2) );
            p3 = OMret_obj_path( val_id,  buff3, sizeof(buff3) );
        }
#endif

        if( HDF5are_the_same( elem_id, tmp_id ) != OM_STAT_SUCCESS ) {
            HDF5write_prim_attr( h5_grp_id, elem_id );
        }

    /* max */
        elem_id = OMfind_subobj (data_array_id, OMstr_to_name("max"),
                                 OM_OBJ_RD);
        tmp_id = OMfind_subobj( tmpl_id, OMstr_to_name("max"), OM_OBJ_RD );
        if( HDF5are_the_same( elem_id, tmp_id ) != OM_STAT_SUCCESS ) {
            HDF5write_prim_attr( h5_grp_id, elem_id );
        }

    /* min_vec */
        elem_id = OMfind_subobj (data_array_id, OMstr_to_name("min_vec"),
                                 OM_OBJ_RD);
        tmp_id = OMfind_subobj( tmpl_id, OMstr_to_name("min_vec"), OM_OBJ_RD );
        if( HDF5are_the_same( elem_id, tmp_id ) != OM_STAT_SUCCESS ) {
            HDF5write_prim_array_attr( h5_grp_id, elem_id );
        }

    /* max_vec */
        elem_id = OMfind_subobj (data_array_id, OMstr_to_name("max_vec"),
                                 OM_OBJ_RD);
        tmp_id = OMfind_subobj( tmpl_id, OMstr_to_name("max_vec"), OM_OBJ_RD );
        if( HDF5are_the_same( elem_id, tmp_id ) != OM_STAT_SUCCESS ) {
            HDF5write_prim_array_attr( h5_grp_id, elem_id );
        }
    }

    /* units */
    elem_id = OMfind_subobj (data_array_id, OMstr_to_name("units"),
                             OM_OBJ_RD);
    if ((!OMis_null_obj(elem_id)) &&
        (OMget_str_val (elem_id, &p, sizeof(buff)) == OM_STAT_SUCCESS) &&
        (buff[0] != 0)) {
        HDF5write_string_attr( h5_grp_id, "units", buff );
    }

    /* labels */
    elem_id = OMfind_subobj (data_array_id, OMstr_to_name("labels"),
                             OM_OBJ_RD);
    if ((!OMis_null_obj(elem_id)) &&
        (OMget_str_val (elem_id, &p, sizeof(buff)) == OM_STAT_SUCCESS) &&
        (buff[0] != 0)) {
        HDF5write_string_attr( h5_grp_id, "labels", buff );
    }
}


/* Xform */
static int
HDF5write_xform( hid_t h5_root_id, OMobj_id xform_id )
{
    float xform[16];
    if( FLDget_xform( xform_id, xform ) == OM_STAT_SUCCESS ) {
        /* Only when it is not the identity matrix. */
        if( MATmat_is_identity( xform, 4 ) == 0 ) {
            hid_t h5_grp_id;
            hid_t h5_dspace_id, h5_dset_id;
            hsize_t h5_dims[2] = { 4, 4 };

            h5_grp_id = H5Gcreate( h5_root_id, "xform", 0 );

            h5_dspace_id = H5Screate_simple(2, h5_dims, NULL);
            if( h5_dspace_id > 0 ) {
                h5_dset_id  = H5Dcreate(h5_grp_id, "mat",
                                        H5T_NATIVE_FLOAT_g,
                                        h5_dspace_id, H5P_DEFAULT);
                if( h5_dspace_id > 0 ) {
                    H5Dwrite(h5_dset_id, H5T_NATIVE_FLOAT,
                             h5_dspace_id, h5_dspace_id, H5P_DEFAULT,
                             xform);
                    H5Dclose(h5_dset_id);	/* close dataset */
                }
                H5Sclose(h5_dspace_id);
            }
            H5Gclose(h5_grp_id);    /* close group */
        }
    }
    return METHOD_SUCCESS;
}

/* Grid_Struct */
/* 64-bit porting. Directly Modified */
static int
HDF5write_dims(
    hid_t h5_grp_id,	/* Parent H5 group, usually root of field */
    OMobj_id fld_id,	/* OM id of an object of type 'Grid_Struct' */
    int fld_type,
    xp_long **fld_dims_ptr )	/* Output parameter */
{
    *fld_dims_ptr = NULL;

    if( fld_type != FLD_UNSTRUCT ) {
        int fld_ndim, dims_size;
        xp_long *fld_dims;
        if( FLDget_ndim( fld_id, &fld_ndim ) != OM_STAT_SUCCESS ) {
            ERR_RETURN( "Could not get field ndim" );
        }
        /* Write field.ndim as a global scalar int attribute. */
        HDF5write_int_attr( h5_grp_id, "ndim", fld_ndim );

        /* Caller must ARRfree fld_dims, passed out as a output argument. */
        if( FLDget_dims( fld_id, &fld_dims, &dims_size ) != OM_STAT_SUCCESS ) {
            ERR_RETURN( "Could not get field dims" );
        }
        else {
            hid_t h5_dspace_id, h5_attr_id;
            hsize_t h5_dims = fld_ndim;

            /* Write fld_dims as a global int array attribute */

            h5_dspace_id = H5Screate_simple(1, &h5_dims, NULL);
            if( h5_dspace_id > 0 ) {
                h5_attr_id = H5Acreate(h5_grp_id, "dims",
                                       H5T_NATIVE_XP_LONG_F_g, h5_dspace_id,
                                       H5P_DEFAULT);
                if (h5_attr_id > 0) {
                    H5Awrite(h5_attr_id, H5T_NATIVE_XP_LONG_g, fld_dims);
                    H5Aclose(h5_attr_id);	/* close attribute */
                }
                H5Sclose(h5_dspace_id);	/* close dataspace */
            }

            /* Set output parameter */
            *fld_dims_ptr = fld_dims;
        }
    }


    return METHOD_SUCCESS;
}

/* Grid */
/* 64-bit porting. Directly Modified */
static int
HDF5write_coords(
    hid_t h5_pgrp_id,	/* Parent H5 group, usually root of field */
    OMobj_id fld_id,	/* OM id of an object of type 'Grid' */
    OMobj_id grid_tmpl_id,	/* OM id of Grid template */
    int fld_type,
    int fld_ndim, xp_long *fld_dims,
    int compress )
{
    xp_long fld_nnodes;
    int fld_nspace;

    FLDget_nnodes( fld_id, &fld_nnodes );
    FLDget_nspace( fld_id, &fld_nspace );

    /* Struct and Unstructured fields: write coordinates group
       and coordinates.values
    */
    if( fld_type == FLD_STRUCT || fld_type == FLD_UNSTRUCT ) {
        hid_t h5_grp_id;

        OMobj_id coord_id;

        coord_id = OMfind_subobj (fld_id, OMstr_to_name("coordinates"),
                                  OM_OBJ_RD);

        h5_grp_id = H5Gcreate(h5_pgrp_id, "coordinates", 0);

        if( (!OMis_null_obj(coord_id)) && (h5_grp_id > 0) ) {
            /* We should end up with either
               'FLD.Grid.coordinates' or 'FLD.Grid_Struct.coordinates'
            */
            OMobj_id coord_tmpl_id =
                OMfind_subobj( grid_tmpl_id, OMstr_to_name("coordinates"),
                               OM_OBJ_RD );
            if( fld_type == FLD_STRUCT )
                HDF5write_data_array( h5_grp_id, coord_id, fld_ndim, fld_dims,
                                      coord_tmpl_id, compress );
            else
                HDF5write_data_array( h5_grp_id, coord_id, 0, NULL,
                                      coord_tmpl_id, compress );
        }
        H5Gclose( h5_grp_id );
    }

    return METHOD_SUCCESS;
}

/* Grid_Struct */
/* 64-bit porting. Only Modified Internally */
static int
HDF5write_points(
    hid_t h5_grp_id,	/* Parent H5 group, usually root of field. */
    OMobj_id fld_id,	/* OM id of an object of type 'Grid' */
    int fld_type )
{
    xp_long fld_nnodes;
    int fld_nspace;

    FLDget_nnodes( fld_id, &fld_nnodes );
    FLDget_nspace( fld_id, &fld_nspace );

    /* Unif and Rect fields: write the points variable */
    /* Struct fields have the points variable, but its generally not used. */
    if( fld_type == FLD_UNIF || fld_type == FLD_RECT ) {
        /* "point" (axis) information */
        xp_long fld_npoints;
        float *fld_points = NULL;
        xp_long pts_size = 0;

        if( FLDget_npoints(fld_id, &fld_npoints) == OM_STAT_SUCCESS ) {
            HDF5write_long_attr( h5_grp_id, "npoints", fld_npoints );
        }

        if (FLDget_points(fld_id, &fld_points, &pts_size,
                          OM_GET_ARRAY_RD) != OM_STAT_SUCCESS) {
            fld_points = NULL;
            pts_size = 0;
        }

        if( fld_points != NULL ) {
            /* 2d float array */
            hid_t h5_dspace_id, h5_dset_id;
            hsize_t h5_dims[2];

            h5_dims[0] = fld_npoints;
            h5_dims[1] = fld_nspace;
            h5_dspace_id = H5Screate_simple(2, h5_dims, NULL);
            if( h5_dspace_id > 0 ) {
                h5_dset_id   = H5Dcreate(h5_grp_id, "points",
                                         H5T_NATIVE_FLOAT_g,
                                         h5_dspace_id,  H5P_DEFAULT);
                if( h5_dset_id > 0 ) {
                    H5Dwrite(h5_dset_id, H5T_NATIVE_FLOAT_g,
                             h5_dspace_id, h5_dspace_id, H5P_DEFAULT,
                             fld_points);
                    H5Dclose(h5_dset_id);	/* close dataset */
                }
                H5Sclose(h5_dspace_id);	/* close dataspace */
            }
            ARRfree( fld_points );
        }
    }
    return METHOD_SUCCESS;
}

/* 64-bit porting. Only Modified Internally */
static int
HDF5write_cells (
    hid_t h5_root_id,	/* Parent H5 group, usually root of field. */
    OMobj_id fld_id	/* OM id of an object of type 'Cells' */
    )
{
    OMobj_id cset_id;
    int fld_nsets;
    xp_long cset_ncells, cset_npolys;
    int cset_nnodes, cset_type;
    int cset_nprops, poly_flag, props_size, *conn_iarray;
    xp_long conn_size, *conn_larray = NULL;
    float *props = NULL;
    int j, om_stat;
    char buffer[256];

    hid_t h5_grp_id, h5_dspace_id, h5_dset_id;
    hsize_t dims[2];

    /* Can't have a unstructured field without valid cell sets */
    if( FLDget_ncell_sets( fld_id, &fld_nsets ) != OM_STAT_SUCCESS ) {
        ERR_RETURN( "Could not get field ncell_sets" );
    }

    HDF5write_int_attr( h5_root_id, "ncell_sets", fld_nsets );

    for( j=0; j<fld_nsets; ++j ) {

        if( FLDget_cell_set( fld_id, j, &cset_id ) != OM_STAT_SUCCESS )
            continue;

        /* For each cell set, create a group called cell_set[n] */
        sprintf( buffer, "cell_set[%d]", j );
        h5_grp_id = H5Gcreate(h5_root_id, buffer, 0);

        FLDget_poly_flag( cset_id, &poly_flag );

        if( poly_flag == 0 ) {
            /* ncells */
            FLDget_ncells( cset_id, &cset_ncells );
            HDF5write_long_attr( h5_grp_id, "ncells", cset_ncells );
        }
        else {
            /* npolys */
            FLDget_npolys( cset_id, &cset_npolys );
            HDF5write_long_attr( h5_grp_id, "npolys", cset_npolys );
        }

        /* cell_type */
        FLDget_cell_type( cset_id, &cset_type );
#if 0   /* Not as useful as cell_name, especially since not all the
           ids are unique! (e.g. cell_type=19) Whats up with that?
        */
        HDF5write_int_attr( h5_grp_id, "cell_type", cset_type );
#endif

        /* cell_name */
        FLDget_cell_set_name( cset_id, buffer, sizeof(buffer) );
        HDF5write_string_attr( h5_grp_id, "cell_name", buffer );

        /* in polys, this is computed from poly_connect_list */

        if( poly_flag == 0 ) {
            /* node_connect_list[ncells*cell_nnodes] */
            FLDget_cell_set_nnodes( cset_id, &cset_nnodes );
            FLDget_node_connect( cset_id, &conn_larray, &conn_size,
                                 OM_GET_ARRAY_RD );
            /* Or perhaps just a 1D list */
            dims[0] = cset_ncells;
            dims[1] = cset_nnodes;
            h5_dspace_id = H5Screate_simple(2, dims, NULL);
            if( h5_dspace_id > 0 ) {
                h5_dset_id   = H5Dcreate(h5_grp_id, "node_connect_list",
                                         H5T_NATIVE_XP_LONG_F_g, h5_dspace_id,
                                         H5P_DEFAULT);
                if( h5_dset_id > 0 ) {
                    H5Dwrite(h5_dset_id, H5T_NATIVE_XP_LONG_g,
                             h5_dspace_id, h5_dspace_id, H5P_DEFAULT,
                             conn_larray);
                    H5Dclose(h5_dset_id);
                }
                H5Sclose(h5_dspace_id);
            }
            ARRfree( conn_larray );
            conn_larray = NULL;
        }

        /* name (optional) */
        if( (FLDget_cell_set_user_name( cset_id, buffer, sizeof(buffer) )
             == OM_STAT_SUCCESS) &&
            (buffer[0] != 0) ) {
            HDF5write_string_attr( h5_grp_id, "name", buffer );
        }

        /* nprops (optional) */
        if( (FLDget_cell_nprops( cset_id, &cset_nprops )
             == OM_STAT_SUCCESS) &&
            (cset_nprops != 0) ) {
            HDF5write_int_attr( h5_grp_id, "nprops", cset_nprops );
        }
        /* props (optional) */
        if( FLDget_cell_props( cset_id, &props, &props_size,
                               OM_GET_ARRAY_RD ) == OM_STAT_SUCCESS ) {
            /* I think HDF5write_prim_array_attr would also work */
            dims[0] = props_size; /* should be same as cset_nprops */
            h5_dspace_id = H5Screate_simple(1, dims, NULL);
            if( h5_dspace_id > 0 ) {
                hid_t h5_attr_id;
                h5_attr_id = H5Acreate(h5_grp_id, "props",
                                    H5T_NATIVE_FLOAT_g, h5_dspace_id,
                                    H5P_DEFAULT);
                if (h5_attr_id > 0) {
                    H5Awrite(h5_attr_id, H5T_NATIVE_FLOAT_g, props);
                    H5Aclose(h5_attr_id);	/* close attribute */
                }
                H5Sclose(h5_dspace_id);	/* close dataspace */
            }
            ARRfree( props );
            props = NULL;
        }

        /* We'll assume that these guys are redundant with the cell_name.
           We could get fancy and check against the template, like with
           node_data.min, etc.
        */

        /* cell_ndim */
        /* cell_nnodes */
        /* cell_corner_nnodes */
        /* cell_order */

        /* poly_flag         FLDget_poly_flag */
        /* poly_type         <no routine> */

        /* poly_connect_list FLDget_poly_connect */
        if( poly_flag == 1 ) {
            FLDget_poly_connect( cset_id, &conn_larray, &conn_size,
                                 OM_GET_ARRAY_RD );
            HDF5write_long_array_attr( h5_grp_id, "poly_connect_list",
                                 conn_larray, conn_size );
            ARRfree( conn_larray );
            conn_larray = NULL;

            /* Yet more work if its a polyhedron */
            if (cset_type == 19) {
                int concave_flag;
                /* poly_nnodes (just for polyhedron) */
                FLDget_poly_nnodes(cset_id, &conn_iarray, &conn_size,
                                   OM_GET_ARRAY_RD);
                HDF5write_int_array_attr( h5_grp_id, "poly_nnodes",
                                     conn_iarray, conn_size );
                ARRfree( conn_iarray );
                conn_iarray = NULL;

                /* concave_flag (only used with polyhedron) */
                om_stat = OMget_name_int_val(cset_id,
                                             OMstr_to_name("concave_flag"),
                                             &concave_flag);
                if( om_stat == OM_STAT_SUCCESS ) {
                    HDF5write_int_attr( h5_grp_id, "concave_flag",
                                        concave_flag );
                }
            }
        }

        H5Gclose( h5_grp_id );
    }
    return METHOD_SUCCESS;
}


static int
HDF5write_cell_data_set (
    hid_t h5_cset_id,	/* H5 group that holds cell set. */
    OMobj_id cset_id,	/* OM id of an object of type 'Cell_Data_Set' */
    OMobj_id cdata_tmpl_id
    )
{
    OMobj_id cdata_id, comp_id;
    int cset_ncomp;
    int cdata_veclen, cdata_type;
    int j;
    char buffer[64];
    int compress = 0;

    hid_t h5_cdata_id;

    if( (FLDget_cell_data_ncomp( cset_id, &cset_ncomp ) == OM_STAT_SUCCESS) &&
        (cset_ncomp != 0) ) {

        HDF5write_int_attr( h5_cset_id, "ncell_data", cset_ncomp );

        cdata_id = OMfind_subobj( cset_id, OMstr_to_name("cell_data"),
                                  OM_OBJ_RD);

        /* Loop over the cell data components. */
        for( j=0; j<cset_ncomp; ++j ) {

            OMget_array_val(cdata_id, j, &comp_id, OM_OBJ_RD);

            if( OMis_null_obj( comp_id ) )
                continue;
            if( FLDget_cell_data_veclen( cset_id, j, &cdata_veclen ) != 1 )
                continue;
            if( FLDget_cell_data_type( cset_id, j, &cdata_type ) != 1 )
                continue;

            /* For each cell data component, create a group called
             * cell_data[n]
             */
            sprintf( buffer, "cell_data[%d]", j );
            h5_cdata_id = H5Gcreate(h5_cset_id, buffer, 0);
            if( h5_cdata_id > 0 ) {
                HDF5write_data_array( h5_cdata_id, comp_id, 0, NULL,
                                      cdata_tmpl_id, compress );
                H5Gclose( h5_cdata_id );
            }
        } /* loop over components */
    } /* if components */

    return METHOD_SUCCESS;
}


static int
HDF5write_cell_data (
    hid_t h5_root_id,	/* Parent H5 group, usually root of field. */
    OMobj_id fld_id,	/* OM id of an object of type 'Cell_Data' */
    OMobj_id cdata_tmpl_id
    )
{
    OMobj_id cset_id;
    int fld_nsets, cset_ncomp;
    int i, ncell_sets_flag;
    char buffer[64];

    hid_t h5_cset_id;

    if( FLDget_ncell_sets( fld_id, &fld_nsets ) != OM_STAT_SUCCESS ) {
        return METHOD_SUCCESS;
        /* ERR_RETURN( "Could not get field ncell_sets" ); */
    }
    else if( fld_nsets == 0 )
        return METHOD_SUCCESS;

    {
        OMobj_id tmp_id;
        tmp_id = OMfind_subobj( cdata_tmpl_id, OMstr_to_name("cell_set"),
                                OM_OBJ_RD);
        /* We should end up with 'FLD.Cell_Data.cell_set.cell_data' */
        cdata_tmpl_id = OMfind_subobj( tmp_id, OMstr_to_name("cell_data"),
                                       OM_OBJ_RD);
    }

    /* Good chance it already exists */
    if( probe_attribute( h5_root_id, "ncell_sets", 1 ) < 0 )
        ncell_sets_flag = 0;
    else
        ncell_sets_flag = 1; /* it exists, don't write it again */

    for( i=0; i<fld_nsets; ++i ) {

        if( FLDget_cell_set( fld_id, i, &cset_id ) != OM_STAT_SUCCESS )
            continue;

        if( (FLDget_cell_data_ncomp( cset_id, &cset_ncomp ) ==
             OM_STAT_SUCCESS) &&
            (cset_ncomp != 0) ) {

            if( ncell_sets_flag == 0 ) {
                /* Only write it once, and don't write it until we've
                   found some cell data.
                */
                HDF5write_int_attr( h5_root_id, "ncell_sets", fld_nsets );
                ncell_sets_flag = 1;
            }

            /* For each cell set that has cell data,
               create a group called cell_set[n]
            */
            sprintf( buffer, "cell_set[%d]", i );
            if( (h5_cset_id = probe_group( h5_root_id, buffer, 0 )) < 0)
                /* Only create if it doesn't exist */
                h5_cset_id = H5Gcreate(h5_root_id, buffer, 0);

            if( h5_cset_id > 0 ) {
                /* recheck the templ */
                HDF5write_cell_data_set( h5_cset_id, cset_id, cdata_tmpl_id );
                H5Gclose( h5_cset_id );
            }
        } /* if components */
    } /* loop over cell sets */
    return METHOD_SUCCESS;
}


static int
HDF5write_time_cell_data (
    hid_t h5_root_id,	/* Parent H5 group, usually root of field. */
    OMobj_id fld_id,	/* OM id of an object of type 'Cell_Data' */
    OMobj_id cdata_tmpl_id
    )
{
    OMobj_id cset_id, cdata_id, elem_id;
    int fld_nsteps, fld_nsets, cset_ncomp;
    int i, j;
    char buffer[64];

    hid_t h5_cset_id, h5_cdata_id;

    if( FLDget_time_nsteps( fld_id, &fld_nsteps ) != OM_STAT_SUCCESS ) {
        ERR_RETURN( "Could not get field nsteps" );
    }

    if( FLDget_ncell_sets( fld_id, &fld_nsets ) != OM_STAT_SUCCESS ) {
        return METHOD_SUCCESS;
        /* ERR_RETURN( "Could not get field ncell_sets" ); */
    }
    else if( fld_nsets == 0 )
        return METHOD_SUCCESS;

    {
        OMobj_id tmp_id;
        tmp_id = OMfind_subobj( cdata_tmpl_id, OMstr_to_name("cell_set"),
                                OM_OBJ_RD );
        /* We should end up with 'FLD.Cell_Data.cell_set.cell_data' */
        cdata_tmpl_id = OMfind_subobj( tmp_id, OMstr_to_name("cell_data"),
                                       OM_OBJ_RD);
    }

    /* Good chance it already exists */
    if( probe_attribute( h5_root_id, "ncell_sets", 1 ) < 0 )
        HDF5write_int_attr( h5_root_id, "ncell_sets", fld_nsets );

    for( i=0; i<fld_nsets; ++i ) {

        if( FLDget_cell_set( fld_id, i, &cset_id ) != OM_STAT_SUCCESS )
            continue;

        if( (FLDget_cell_data_ncomp( cset_id, &cset_ncomp ) !=
             OM_STAT_SUCCESS) ||
            (cset_ncomp == 0) )
            continue;

        /* For each cell set that has cell data,
           create a group called cell_set[n]
        */
        sprintf( buffer, "cell_set[%d]", i );
        if( (h5_cset_id = probe_group( h5_root_id, buffer, 0 )) < 0)
            /* Only create if it doesn't exist */
            h5_cset_id = H5Gcreate( h5_root_id, buffer, 0 );

        elem_id = OMfind_subobj( cset_id, OMstr_to_name( "time_cell_data" ),
                                 OM_OBJ_RD );

        if( (h5_cset_id <= 0) || OMis_null_obj(elem_id) )
            continue;

        for( j=0; j<fld_nsteps; ++j ) {

            sprintf( buffer, "time_cell_data[%d]", j );
            if( (h5_cdata_id = probe_group( h5_cset_id, buffer, 0 )) < 0)
                /* Only create if it doesn't exist */
                h5_cdata_id = H5Gcreate( h5_cset_id, buffer, 0 );

            if( h5_cdata_id > 0 ) {
                if( OMget_array_val( elem_id, j, &cdata_id, OM_OBJ_RD )
                    == OM_STAT_SUCCESS ) {
                    /* recheck the templ */
                    HDF5write_cell_data_set( h5_cdata_id, cdata_id,
                                             cdata_tmpl_id );
                }
                H5Gclose( h5_cdata_id );
            }
        } /* loop over time steps */
        H5Gclose( h5_cset_id );
    } /* loop over cell sets */
    return METHOD_SUCCESS;
}


/* Node_Data */
/* 64-bit porting. Directly Modified */
static int
HDF5write_node_data (
    hid_t h5_root_id,	/* Parent H5 group, usually root of field. */
    OMobj_id fld_id,	/* OM id of an object of type 'Node_Data' */
    OMobj_id ndata_tmpl_id,	/* OM id of template of 'FLD.Node_Data' */
    int fld_type,
    int fld_ndim, xp_long *fld_dims,
    int compress
    )
{
    int j, fld_ncomps = 0;
    xp_long fld_nnodes;
    char buffer[64];

    /* Move to calling code? */
    if( fld_type == FLD_UNSTRUCT ) {
        FLDget_nnodes( fld_id, &fld_nnodes );
        fld_ndim = 1;
        fld_dims  = &fld_nnodes;
    }

    if( (FLDget_node_data_ncomp( fld_id, &fld_ncomps ) == OM_STAT_SUCCESS) &&
        (fld_ncomps != 0) ) {
        OMobj_id node_data_id, nd_id, comp_id;

        /* Move to calling code? */
        HDF5write_int_attr( h5_root_id, "nnode_data", fld_ncomps );

        node_data_id = OMfind_subobj( fld_id, OMstr_to_name("node_data"),
                                      OM_OBJ_RD );
        if( OMis_null_obj(node_data_id) )
            return METHOD_FAILURE;  /* Don't expect this error */

        /* We should end up with 'FLD.Node_Data.node_data' */
        nd_id = OMfind_subobj( ndata_tmpl_id, OMstr_to_name( "node_data"),
                               OM_OBJ_RD);

        for( j=0; j<fld_ncomps; ++j ) {
            hid_t h5_grp_id;
            int data_type, veclen;

            if( OMget_array_val(node_data_id, j, &comp_id, OM_OBJ_RD) !=
                OM_STAT_SUCCESS )
                return METHOD_FAILURE;  /* Don't expect this error */

            if (FLDget_node_data_type(fld_id, j, &data_type) !=
                OM_STAT_SUCCESS)
                /* We won't be able to do much useful without the data type,
                   it probably means the node data itself is missing.
                */
                continue;

            if( FLDget_node_data_veclen( fld_id, j, &veclen ) !=
                OM_STAT_SUCCESS )
                /* Node data won't be valid without a veclen. */
                continue;

            /* For each node data component, create a group called
             * node_data[n]
             */
            sprintf( buffer, "node_data[%d]", j );
            h5_grp_id = H5Gcreate(h5_root_id, buffer, 0);

            if( h5_grp_id > 0 ) {
                HDF5write_data_array( h5_grp_id, comp_id, fld_ndim, fld_dims,
                                      nd_id, compress );
                H5Gclose( h5_grp_id );
            }

        } /* loop over components */
    }  /* if node data */

    return METHOD_SUCCESS;
}
