/*
        This file is under Perforce control
        $Id: //depot/express/fcs70/netcdf/examp/write_struct.c#1 $

Example Unix command line to compile (if netCDF is installed in /usr/local):
cc write_struct.c -I/usr/local/include /usr/local/lib/libnetcdf.a
*/

#include <stdio.h>
#include <stdlib.h>
#include <netcdf.h>

static void check_err( int stat, int line, const char *file )
{
    if( stat != NC_NOERR ) {
        fprintf( stderr, "error in line %d of %s: %s\n",
                 line, file, nc_strerror(stat) );
        exit -1;
    }
}

static int write_string_attr( int nc_id, int nc_var_id, const char *name,
                              const char *value )
{
    int stat = -1;
    if( name != NULL && value != NULL ) {
        /* The +1 is not necessary, strictly speaking, but it
         * makes reading the string a bit less foolproof.
         */
        int len = strlen( value ) + 1;
        stat = nc_put_att_text( nc_id, nc_var_id, name, len, value );
    }
    return stat;
}

int main( int argc, char *argv[] )
{
    int nc_id;			/* netCDF id */
    int nc_dim_ids[NC_MAX_DIMS];	/* nC dimension ids */
    int nc_cv_dim_ids[2];	/* nC dimension ids */
    int nc_var_id;		/* nC variable id */
    int nc_coord_var_id;	/* nC variable id */

    /* Variables that define a sample 3 x 4 Uniform field */
    int fld_nspace = 2;
    int fld_ndim   = 2;
    int fld_dims[2] = { 4, 3 };

    /* fld_coords[fld_nnodes][fld_nspace] */
    int fld_npoints = 12;	/* Product of the dimensions. */
    float fld_coords[12][2] = { { 0.0, 0.0 },
                                { 1.0, 0.0 },
                                { 2.0, 0.0 },
                                { 3.5, 0.0 },
                                { 0.0, 1.0 }, 
                                { 1.0, 1.0 }, 
                                { 2.0, 1.0 }, 
                                { 3.0, 1.0 }, 
                                { 0.0, 2.0 },
                                { 1.0, 2.0 },
                                { 2.0, 2.0 },
                                { 3.0, 2.5 } };

    /* fld_node_data[fld_dims[1]][fld_dims[0]][veclen] */
    /* The AVS/Express netCDF reader only handles veclen == 1 */
    int fld_node_data[3][4] = { {0,  1,  2,  3},
                                {4,  5,  6,  7},
                                {8,  9, 10, 11} };

    char *filename = "examp_struct.nc";

    if( argc > 1 )
        filename = argv[1];

    /* Create file and enter define mode */
    int stat = nc_create( filename, NC_CLOBBER, &nc_id );
    check_err( stat, __LINE__, __FILE__ );

    /* Define dimensions */

    /* NOTE ORDER OF THE DIMS. fld_dims is in column-major
     * order and netCDF uses row-major order, so we need to
     * reverse the order of the dims.
     */
    stat = nc_def_dim( nc_id, "dim0", fld_dims[1], &nc_dim_ids[0] );
    stat = nc_def_dim( nc_id, "dim1", fld_dims[0], &nc_dim_ids[1] );

    /* Dimensions of the field's coordinate information. The need to
     * explicitly represent the field's coordinates is not covered by
     * standard netCDFconventions, so we have to invent our own.
     * Dimensions named "xp_npoints" and "xp_nspace" have special meaning
     * to the AVS/Express reader.  Since these conventions are specific
     * to AVS/Express, netCDf files containing struct fields are
     * less portable than netCDf files containing unif or rect fields.
     */
    stat = nc_def_dim( nc_id, "xp_npoints", fld_npoints, &nc_cv_dim_ids[0] );
    stat = nc_def_dim( nc_id, "xp_nspace",  fld_nspace,  &nc_cv_dim_ids[1] );

    /* Define the variable for the field coordinate information.
     * A variable name "XP_COORDINATES" has special meaning to
     * to the AVS/Express reader.
     */
    stat = nc_def_var( nc_id, "XP_COORDINATES", NC_FLOAT, 
                       2, nc_cv_dim_ids, &nc_coord_var_id );

    /* Define the variable for the node data component */

    stat = nc_def_var( nc_id, "velocity", NC_INT, 
                       fld_ndim, nc_dim_ids, &nc_var_id );

    /* Write global attributes */
    stat = write_string_attr(nc_id, NC_GLOBAL, "XP_CLASS",
                             "Mesh_Struct+Node_Data");

    /* Write variable attributes */
    stat = write_string_attr(nc_id, nc_var_id, "units",
                             "furlongs per fortnight");

    /* Leave define mode */
    stat = nc_enddef( nc_id );

    /* Write coordinate data */
    stat = nc_put_var_float( nc_id, nc_coord_var_id,
                             (float *)fld_coords );

    /* Write node data */
    stat = nc_put_var_int( nc_id, nc_var_id,
                           (int *)fld_node_data );

    /* Close file */
    stat = nc_close( nc_id );

    return 0;
}
