/*
			Copyright (c) 2001 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/netcdf/xp_mods/nc_rd_arr.c#1 $
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>  /* time_t */

#define	XP_WIDE_API	/* Use Wide APIs */
#include <avs/f_utils.h>
#include <avs/err.h>
#include <avs/om.h>
/* Someday rename to unidata/netcdf.h */
#include <avs/netcdf.h>

#define METHOD_SUCCESS 1
#define METHOD_FAILURE 0

typedef struct _ncvar {                  /* variable */
  int id;
  nc_type type;
  int ndims;
  int dim_ids[NC_MAX_VAR_DIMS];
} ncvar_t;


typedef struct _ncdim {                  /* dimension */
  size_t len;
} ncdim_t;


#define ERR_RETURN(A) {ERRerror("netcdf read_array", 0, ERR_ORIG, A); \
                       return(METHOD_FAILURE);}

static int NCread_array( const char *, const char *, OMobj_id );

/* 64-bit porting. Only Modified Internally */
int NCread_array_update(OMobj_id mod_id, OMevent_mask mask, int seq_num)
{
    OMobj_id file_id, var_id, arr_id;

    char *temp_str = NULL, *filename = NULL;
    char file_buf[AVS_PATH_MAX];
    time_t mod_time = 0;

    char *var_str = NULL;

    /* in */
    file_id = OMfind_subobj( mod_id, OMstr_to_name("filename"), OM_OBJ_RD );
    var_id  = OMfind_subobj( mod_id, OMstr_to_name("variable"), OM_OBJ_RD );
    /* out */
    arr_id  = OMfind_subobj( mod_id, OMstr_to_name("outArr"), OM_OBJ_RW );

    if( OMget_str_val(file_id, &temp_str, 0) != OM_STAT_SUCCESS )
        ERR_RETURN( "Can't get filename." );

    filename = FILEmap_variables( temp_str, file_buf );

    if (temp_str) { free (temp_str); temp_str = NULL; }

    if ( !filename || !FILEexists(filename, FILE_READ, &mod_time) ) {
        ERRerror( "netcdf read_array", 1, ERR_ORIG, "Can't open data file: %s",
                  filename );
        return METHOD_FAILURE;
    }

    if( OMget_str_val(var_id, &var_str, 0) != OM_STAT_SUCCESS )
        ERR_RETURN( "Can't get variable." );

    NCread_array( filename, var_str, arr_id );

    if (var_str) { free (var_str); var_str = NULL; }

    return METHOD_SUCCESS;
}

/* 64-bit porting. Only Modified Internally */
static int NCread_array( const char *filename, const char *var,
                         OMobj_id arr_id )
{
    int i, j, k;
    int nc_stat, nc_id;

    ncvar_t nc_var;
    ncdim_t *nc_dims = NULL;
    int dim_ids_temp[NC_MAX_VAR_DIMS];

    int om_stat;
    int arr_type;
    char *arr = NULL;
    int arr_ndims;
    xp_long arr_dims[OM_ARRAY_MAXDIM];

    ncopts = NC_VERBOSE;
    nc_stat = nc_open(filename, NC_NOWRITE, &nc_id);
    if( nc_stat != NC_NOERR ) {
        return METHOD_FAILURE;
    }

    /* Get information about the variable */

    /* Given a variable name, get the variable ncid. */
    nc_stat = nc_inq_varid( nc_id, var, &nc_var.id );
    if( nc_stat != NC_NOERR )
        return METHOD_FAILURE;
    nc_stat = nc_inq_vartype(  nc_id, nc_var.id, &nc_var.type );
    nc_stat = nc_inq_varndims( nc_id, nc_var.id, &nc_var.ndims );
    /* dimensions */
    nc_stat = nc_inq_vardimid( nc_id, nc_var.id, dim_ids_temp );
    /* flip dimensions */
    for( k = 0; k < nc_var.ndims; ++k ) {
        nc_var.dim_ids[nc_var.ndims - (k+1)] = dim_ids_temp[k];
    }

    /* Hold dimension information */
    nc_dims = malloc( nc_var.ndims * sizeof(ncdim_t) );

    /* Get information about each dimension */
    for( i = 0; i < nc_var.ndims; ++i ) {
        int nc_dim_id = nc_var.dim_ids[i];
        size_t dim_len;
        nc_stat = nc_inq_dimlen( nc_id, nc_dim_id, &dim_len );
        if( nc_stat == NC_NOERR ) {
            nc_dims[i].len = dim_len;
        }
    }

    /* Completely specify the array information */

    switch( nc_var.type ) {
    case NC_BYTE:   arr_type = DTYPE_BYTE;  break;
    case NC_SHORT:  arr_type = DTYPE_SHORT; break;
    case NC_INT:    arr_type = DTYPE_INT;   break;
    case NC_FLOAT:  arr_type = DTYPE_FLOAT; break;
    case NC_DOUBLE: arr_type = DTYPE_DOUBLE;
    }

    arr_ndims = nc_var.ndims;
    for( k = 0; k < nc_var.ndims; ++k ) {
        arr_dims[k] = nc_dims[k].len;
    }

    /* Get the array for writing */

    om_stat = OMget_array( arr_id, &arr_type, &arr,
                           &arr_ndims, arr_dims, OM_GET_ARRAY_WR );

    if( arr && (om_stat == OM_STAT_SUCCESS) ) {
        switch( nc_var.type ) {
        case NC_BYTE:
            nc_get_var_uchar( nc_id, nc_var.id, (unsigned char *)arr );
            break;
        case NC_SHORT:
            nc_get_var_short( nc_id, nc_var.id, (short *)arr );
            break;
        case NC_INT:
            nc_get_var_int( nc_id, nc_var.id, (int *)arr );
            break;
        case NC_FLOAT:
            nc_get_var_float( nc_id, nc_var.id, (float *)arr );
            break;
        case NC_DOUBLE:
            nc_get_var_double( nc_id, nc_var.id, (double *)arr );
        }
    }
    else {
        if( arr ) ARRfree( arr );
        if( nc_dims ) free( nc_dims );
        ERR_RETURN( "Can't get out array for writing." );
    }

    if( arr ) ARRfree( arr );
    if( nc_dims ) free( nc_dims );

    return METHOD_SUCCESS;
}

