/*
			Copyright (c) 2001 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/netcdf/xp_mods/nc_rd_fld.c#1 $
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>  /* time_t */

#define	XP_WIDE_API	/* Use Wide APIs */
#include <avs/f_utils.h>
#include <avs/err.h>
#include <avs/om.h>
#include <avs/fld.h>
/* Someday rename to unidata/netcdf.h */
#include <avs/netcdf.h>

#define METHOD_SUCCESS 1
#define METHOD_FAILURE 0

#define INVALID_NC_DIM -1
#define INVALID_NC_VAR -1
#define INVALID_TIME_STEP -1

typedef struct _ncvar {                  /* variable */
  char name[NC_MAX_NAME];
  int id;
  nc_type type;
  int ndims;
  int dim_ids[NC_MAX_VAR_DIMS];
} ncvar_t;


typedef struct _ncdim {                  /* dimension */
  char name[NC_MAX_NAME];
  size_t len;
  int coord_var;       /* coordinate variable flag */
  int unlim;           /* unlimited (i.e. time) dimension flag */
  int var_id;
  float *points;
} ncdim_t;


#define ERR_RETURN(A) {ERRerror("netcdf read_field", 0, ERR_ORIG, A); \
                       return(METHOD_FAILURE);}

typedef enum { FLD_Unspecified = 0, FLD_UNIF, FLD_RECT, FLD_STRUCT} FLD_Type;

static int NClist_vars( const char *filename,
                        int *nvars, char *vars[], char *attrs[],
                        int *time_steps );

static int NCread_field( OMobj_id, const char *, int,
                         const char *vars[], int, FLD_Type,
                         OMobj_id * );

/* 64-bit porting. Only Modified Internally */
int NClist_vars_update(OMobj_id mod_id, OMevent_mask mask, int seq_num)
{
    OMobj_id file_id, vars_id, ts_id, title_id, hist_id;

    char *temp_str = NULL, *filename = NULL;
    char file_buf[AVS_PATH_MAX];
    time_t mod_time = 0;

    int i, nvars, time_steps;
    char *vars[NC_MAX_VARS];
    char *attrs[2] = { NULL, NULL }; /* to hold "title" and "history" */

    /* in */
    file_id = OMfind_subobj( mod_id, OMstr_to_name("filename"), OM_OBJ_RD );
    /* out */
    vars_id  = OMfind_subobj( mod_id, OMstr_to_name("variables"), OM_OBJ_RW );
    ts_id    = OMfind_subobj( mod_id, OMstr_to_name("timeSteps"), OM_OBJ_RW );
    title_id = OMfind_subobj( mod_id, OMstr_to_name("title"), OM_OBJ_RW );
    hist_id  = OMfind_subobj( mod_id, OMstr_to_name("history"), OM_OBJ_RW );

    if( OMget_str_val(file_id, &temp_str, 0) != OM_STAT_SUCCESS )
        ERR_RETURN( "Can't get filename." );

    filename = FILEmap_variables( temp_str, file_buf );

    if( temp_str ) { free (temp_str); temp_str = NULL; }

    if( !filename || !FILEexists(filename, FILE_READ, &mod_time) ) {
        /*
          ERRerror("netcdf list_vars", 1, ERR_ORIG, "Can't open data file: %s",
                   filename);
        */
        return METHOD_FAILURE;
    }

    if( NClist_vars( filename, &nvars, vars, attrs, &time_steps ) 
        != METHOD_SUCCESS ) {
        OMset_array_size( vars_id, 0 );
        for( i = 0; i < nvars; ++i ) {
            if( vars[i] ) free( vars[i] );
        }
        return METHOD_FAILURE;
    }

    OMset_array_size( vars_id, nvars );

    for( i = 0; i < nvars; ++i ) {
        OMset_str_array_val( vars_id, i, vars[i] );
        if( vars[i] ) free( vars[i] );
    }

    OMset_int_val( ts_id, time_steps );

    if( attrs[0] ) {
      OMset_str_val( title_id, attrs[0] );
      free( attrs[0] );
    }
    else OMset_str_val( title_id, NULL );

    if( attrs[1] ) {
      OMset_str_val( hist_id, attrs[1] );
      free( attrs[1] );
    }
    else OMset_str_val( hist_id, NULL );

    return METHOD_SUCCESS;
}

/* 64-bit porting. Only Modified Internally */
int NCread_field_update(OMobj_id mod_id, OMevent_mask mask, int seq_num)
{
    OMobj_id file_id, vars_id, ts_id, type_id;
    OMobj_id vars_sel_id, trigger_id, out_id, out_tmp_id;
    int om_stat;

    char *temp_str = NULL, *filename = NULL;
    char file_buf[AVS_PATH_MAX];
    time_t mod_time = 0;

    int time_step;
    xp_long i, vars_size;
    char **vars = NULL;

    FLD_Type fld_type = FLD_Unspecified;

    /* in */
    file_id = OMfind_subobj( mod_id, OMstr_to_name("filename"), OM_OBJ_RD );
    vars_id = OMfind_subobj( mod_id, OMstr_to_name("variables"), OM_OBJ_RD );
    type_id = OMfind_subobj( mod_id, OMstr_to_name("fldType"),  OM_OBJ_RD );
    ts_id   = OMfind_subobj( mod_id, OMstr_to_name("timeStep"), OM_OBJ_RD );
    /* UI in */
    vars_sel_id  = OMfind_subobj( mod_id, OMstr_to_name("selectedVariables"),
                                  OM_OBJ_RD );
    trigger_id  = OMfind_subobj( mod_id, OMstr_to_name("trigger"),
                                 OM_OBJ_RD );
    /* out */
    out_tmp_id = OMfind_subobj( mod_id, OMstr_to_name("outFld"), OM_OBJ_RD );

    /* OK if missing entirely, means we are running in non-UI mode. */
    if( !OMis_null_obj(trigger_id) ) {
        if( !(OM_EVENT_INST & mask) && !OMchanged( trigger_id, seq_num ) ) {
            int in_seq, out_seq, ts_seq, tmp_seq;

            /* All this mess is to support the notion that
             * clicking on the time step slider should produce
             * a update, but only when the file has already been
             * read at least once.
             */
#define GET_SEQ(id) OMget_obj_seq( id, OMnull_obj, OM_SEQ_VAL )

            ts_seq  = GET_SEQ( ts_id );
            out_seq = GET_SEQ( out_tmp_id );

            if( !(ts_seq > out_seq) ) return METHOD_SUCCESS;

            /* in_seq represents all inputs except time_step */
            in_seq  = GET_SEQ( file_id );
            if( in_seq < (tmp_seq = GET_SEQ(vars_id) ) )
                in_seq = tmp_seq;
            if( in_seq < (tmp_seq = GET_SEQ(type_id) ) )
                in_seq = tmp_seq;
            if( in_seq < (tmp_seq = GET_SEQ(vars_sel_id) ) )
                in_seq = tmp_seq;
            if( in_seq < (tmp_seq = GET_SEQ(trigger_id) ) )
                in_seq = tmp_seq;

#undef GET_SEQ

            if( !(out_seq > in_seq) ) return METHOD_SUCCESS;
        }
    }

    if( !OMis_null_obj(type_id) ) {
        int tmp;
        om_stat = OMget_int_val( type_id, &tmp );
        if( om_stat == OM_STAT_SUCCESS )
            /* the V enum and the C enum better match up! */
            fld_type = (FLD_Type)tmp;
    }

    if( OMget_str_val(file_id, &temp_str, 0) != OM_STAT_SUCCESS )
        ERR_RETURN( "Can't get filename." );

    filename = FILEmap_variables( temp_str, file_buf );

    if( temp_str ) { free (temp_str); temp_str = NULL; }

    if( !filename || !FILEexists(filename, FILE_READ, &mod_time) ) {
        ERRerror( "netcdf read_field", 1, ERR_ORIG, "Can't open data file: %s",
                  filename );
        return METHOD_FAILURE;
    }

    if( OMget_array_size(vars_id, &vars_size) != OM_STAT_SUCCESS )
        return METHOD_FAILURE;
    if( vars_size == 0 )
        /* TODO: Clean up output file */
        return METHOD_SUCCESS;

    /*
     * Scenerio #1: 'variables' contains a list of all the netCDF variables
     * and 'selectedVariables' contains a list of integer indicies of which
     * variable to read in.  The reason for this is so that it plays smoothly
     * with UIoptionBoxLabel which outputs a list of integer indicies.
     *
     * Note thats its OK for 'selectedVariables' to be missing or otherwise
     * invalid.
     */

    if( !OMis_null_obj(vars_sel_id) ) {
        int ndims = 0, type = OM_TYPE_INT;
        int *vars_sel_arr = NULL;
        xp_long vars_sel_dims[OM_ARRAY_MAXDIM];

        om_stat = OMget_array( vars_sel_id, &type, (char **)&vars_sel_arr,
                               &ndims, vars_sel_dims, OM_GET_ARRAY_RD );
        vars_size = vars_sel_dims[0];

        if (om_stat == OM_STAT_SUCCESS && vars_size != 0) {
            vars = malloc( vars_size * sizeof(char *) );
            for( i = 0; i < vars_size; ++i ) {
                int var_sel = vars_sel_arr[i];
                if( OMget_str_array_val( vars_id, var_sel, &vars[i], 0 )
                    != OM_STAT_SUCCESS ) {
                    free( vars );
                    vars = NULL;
                    break;
                }
            }
        }
        if( vars_sel_arr ) ARRfree( vars_sel_arr );
    }

    if (vars_size == 0)
        /* TODO: Clean up output file */
        return METHOD_SUCCESS;

    /*
     * Scenerio #2: 'variables' contains a list of all the netCDF variables
     * to be read in.  This is more convenient when this module is being used
     * directly, without the UI wrapper.
     */

    if( !vars ) {
        vars = malloc( vars_size * sizeof(char *) );

        for( i = 0; i < vars_size; ++i ) {
            if( OMget_str_array_val(vars_id, i, &vars[i], 0)
                != OM_STAT_SUCCESS ) {
                free( vars );
                vars = NULL;
                ERR_RETURN( "Cannot get variable list" );
            }
        }
    }

    /* Strip off the extra junk added by list_vars */
    for( i = 0; i < vars_size; ++i ) {
        char *p = NULL;
        if( p = strstr( vars[i], "  (" ) )   /* note two spaces */
            *p = 0;
    }

    if( !OMis_null_obj(ts_id) ) {
        om_stat = OMget_int_val( ts_id, &time_step );
        if( om_stat != OM_STAT_SUCCESS ) time_step = INVALID_TIME_STEP;
    }
    else time_step = INVALID_TIME_STEP;

    if( NCread_field( mod_id, filename, (int)vars_size, (const char **)vars,
                      time_step, fld_type, &out_id )
        != METHOD_SUCCESS ) {
        for( i = 0; i < vars_size; ++i ) free( vars[i] );
        free( vars );
        ERR_RETURN( "Error while reading file" );
    }

    for( i = 0; i < vars_size; ++i ) free( vars[i] );
    free( vars );

    out_tmp_id = OMfind_subobj( mod_id, OMstr_to_name("outFld"), OM_OBJ_RD );
    if( !OMis_null_obj(out_tmp_id) ) {
        om_stat = OMset_obj_ref_mode( out_tmp_id, OM_OBJ_REF );
        if( om_stat != OM_STAT_SUCCESS )
            return METHOD_FAILURE;

        om_stat = OMset_obj_ref( out_tmp_id, out_id, 0 );
        if( om_stat != OM_STAT_SUCCESS ) {
            ERR_RETURN( "Could not set outFld reference to Output_Field" );
        }
    }

    return METHOD_SUCCESS;
}

static int NClist_vars( const char *filename,
                        int *nvars, char *vars[], char *attrs[],
                        int *time_steps )
{
    int i, j;
    int nc_stat, nc_id;
    int nc_num_vars;
    size_t nc_att_len;
    nc_type nc_att_type;

    char buffer[256];

    ncvar_t *nc_vars = NULL;

    ncopts = NC_VERBOSE;
    nc_stat = nc_open( filename, NC_NOWRITE, &nc_id );
    if( nc_stat != NC_NOERR ) {
        *nvars = 0;
        return METHOD_FAILURE;
    }

    nc_stat = nc_inq_nvars( nc_id, &nc_num_vars );
    if( nc_stat != NC_NOERR ) {
        ncclose( nc_id );
        *nvars = 0;
        return METHOD_FAILURE;
    }

    /* Read off a few standard global attributes */

    nc_stat = nc_inq_att( nc_id, NC_GLOBAL, "title",
                          &nc_att_type, &nc_att_len );
    if( nc_stat == NC_NOERR && nc_att_type == NC_CHAR &&
        nc_att_len < sizeof(buffer) ) {
        buffer[0] = 0;
        nc_stat = nc_get_att_text(nc_id, NC_GLOBAL, "title", buffer );
        if( nc_stat == NC_NOERR && buffer[0] != 0 ) {
            if( buffer[nc_att_len-1] != 0 )
                buffer[nc_att_len] = 0;
            attrs[0] = strdup( buffer );
        }
    }

    nc_stat = nc_inq_att( nc_id, NC_GLOBAL, "history",
                          &nc_att_type, &nc_att_len );
    if( nc_stat == NC_NOERR && nc_att_type == NC_CHAR &&
        nc_att_len < sizeof(buffer) ) {
        buffer[0] = 0;
        nc_stat = nc_get_att_text(nc_id, NC_GLOBAL, "history", buffer );
        if( nc_stat == NC_NOERR && buffer[0] != 0 ) {
            if( buffer[nc_att_len-1] != 0 )
                buffer[nc_att_len] = 0;
            attrs[1] = strdup( buffer );
        }
    }

    /* Hold variable information */
    nc_vars = malloc( nc_num_vars * sizeof(ncvar_t) );

    /* Get information about each variable */
    j = 0;
    for( i = 0; i < nc_num_vars; ++i ) {
        int nc_var_id, nc_dim_id;

        /* They just count up from 0 ... this is a defined part of the
           netcdf API. */
        nc_vars[j].id = nc_var_id = i;

        /* variable name */
        nc_stat = nc_inq_varname( nc_id, nc_var_id, nc_vars[j].name );
        if( nc_stat != NC_NOERR )
            continue;

        if( !strncmp( nc_vars[j].name, "XP_COORD", 8 ) )
            continue;

        /* data type (float, int, etc.) */
        nc_stat = nc_inq_vartype( nc_id, nc_var_id, &nc_vars[j].type );
        if( nc_stat != NC_NOERR || nc_vars[j].type == NC_CHAR )
            continue;

        /* number of dimensions */
        nc_stat = nc_inq_varndims( nc_id, nc_var_id, &nc_vars[j].ndims );
        if( nc_stat != NC_NOERR || nc_vars[j].ndims < 2 )
            continue;

        /* dimensions */
        nc_stat = nc_inq_vardimid( nc_id, nc_var_id, nc_vars[j].dim_ids );
        if( nc_stat != NC_NOERR )
            continue;

        nc_stat = nc_inq_dimid(nc_id, nc_vars[j].name, &nc_dim_id);
        if( nc_stat != NC_EBADDIM ) {
          /* Its a "coordinate variable", do *not* want to consider
             as a data variable */
            continue;
        }

        ++j;
    }

    *nvars = j;
    for( i = 0; i < *nvars; ++i ) {
#if 0
        /* The simple way, just the variable names */
        vars[i] = strdup( nc_vars[i].name );
#endif
#if 0
        /* Encode more information into the variable name */
        sprintf( buffer, "%s  (%d D)",  /* note two spaces */
                 nc_vars[i].name, nc_vars[i].ndims );
        vars[i] = strdup( buffer );
#endif
#if 1
        /* Encode (much) more information into the variable name */
        strcpy( buffer, nc_vars[i].name );
        strcat( buffer, "  (" ); /* note two spaces */
        for( j = 0; j < nc_vars[i].ndims; ++j ) {
            char dim_name[128];
            nc_stat = nc_inq_dimname( nc_id, nc_vars[i].dim_ids[j], dim_name );
            if( j != 0 ) strcat( buffer, ", " );
            if( nc_stat == NC_NOERR ) {
                strcat( buffer, dim_name );
            }
        }
        strcat( buffer, ")" );
        vars[i] = strdup( buffer );
#endif
    }

    {
        /* nC unlimited (time) dimension */
        int nc_unlim_dim_id = INVALID_NC_DIM;
        size_t nc_unlim_len;

        /* nc_unlim_dim_id gets a value of -1 if there is no unlimited dim. */
        nc_stat = nc_inq_unlimdim( nc_id, &nc_unlim_dim_id );
        if( nc_stat == NC_NOERR && nc_unlim_dim_id != INVALID_NC_DIM ) {
            nc_stat = nc_inq_dimlen( nc_id, nc_unlim_dim_id, &nc_unlim_len );
            if( nc_stat == NC_NOERR ) *time_steps = (int)nc_unlim_len;
            else                      *time_steps = 0;
        }
        else *time_steps = 0;
    }

    free( nc_vars );

    ncclose( nc_id );

    return METHOD_SUCCESS;
}


static OMobj_id
NCfind_or_create( OMobj_id parent, const char *name,
                  const char *create_command )
{
    const int mode = OM_OBJ_RW;
    OMobj_name obj_name = OMstr_to_name( name );
    OMobj_id   sub_id   = OMfind_subobj(parent, obj_name, mode);
    if( OMis_null_obj(sub_id) && create_command ) {
        OMparse_buffer( parent, create_command, 0 );
        sub_id = OMfind_subobj(parent, obj_name, mode);
    }
    return sub_id;
}

static void
NCtext_attr_nc2xp( int nc_id, int nc_var_id, const char *nc_att_name,
                   OMobj_id parent_id, const char *xp_name,
                   const char *xp_create_command )
{
  int nc_stat;
  size_t nc_att_len;
  nc_type nc_att_type;

  char buffer[256];

  /* Check for the NC attribute */
  nc_stat = nc_inq_att( nc_id, nc_var_id, nc_att_name,
                        &nc_att_type, &nc_att_len );

  if( nc_stat == NC_NOERR && nc_att_type == NC_CHAR &&
      nc_att_len < sizeof(buffer) ) {
      /* Get the NC atttribute */
      nc_stat = nc_get_att_text( nc_id, nc_var_id, nc_att_name, buffer );

      if( nc_stat == NC_NOERR ) {
          /* Get the XP string object */
          OMobj_id str_id = NCfind_or_create( parent_id, xp_name,
                                              xp_create_command );
          if( !OMis_null_obj(str_id) ) {
              /* Check for missing trailing zero */
              if( buffer[nc_att_len-1] != 0 )
                  buffer[nc_att_len] = 0;
              /* Set the XP string value */
              OMset_str_val( str_id, buffer );
          }
      }
  }
}

/* 64-bit porting. Only Modified Internally */
static int NCread_field( OMobj_id parent_id, const char *filename,
                         int nvars, const char *vars[], int time_step,
                         FLD_Type fld_type, OMobj_id *out_id )
{
    int i, j, k;
    size_t n;
    int nc_stat, nc_id;
    size_t nc_att_len;
    nc_type nc_att_type;

    /* variable that holds special Express coordinate information */
    int nc_coord_id;
    /* nC unlimited (time) dimension */
    int nc_unlim_dim_id;

    ncvar_t *nc_vars = NULL;
    ncdim_t *nc_dims = NULL;

    OMobj_id outFld_id;

    xp_long fld_nnodes;
    int fld_ncomps, fld_ndims, fld_ndims_total, fld_nspace = 0;
    xp_long  fld_dims[MAX_VAR_DIMS];
    float *fld_points;
    xp_long fld_pts_size;
    char buffer[256];
    int fld_data_type;
    char *fld_node_data;
    xp_long fld_node_data_size;

    ncopts = NC_VERBOSE;
    nc_stat = nc_open(filename, NC_NOWRITE, &nc_id);
    if( nc_stat != NC_NOERR ) {
        *out_id = OMnull_obj;
        return METHOD_FAILURE;
    }

    /* Check for a non-standard Express attribute */

    nc_stat = nc_inq_att( nc_id, NC_GLOBAL, "XP_CLASS",
                          &nc_att_type, &nc_att_len );
    if( nc_stat == NC_NOERR && nc_att_type == NC_CHAR ) {
        /* Get the NC atttribute */
        nc_stat = nc_get_att_text(nc_id, NC_GLOBAL, "XP_CLASS", buffer );
        if( nc_stat == NC_NOERR ) {
            if( buffer[nc_att_len-1] != 0 )
                buffer[nc_att_len] = 0;
#if 0
            fprintf( stderr, "XP_CLASS: %s\n", buffer );
#endif

            if( strstr( buffer, "Unif" ) )
                fld_type = FLD_UNIF;
            else if( strstr( buffer, "Rect" ) )
                fld_type = FLD_RECT;
            else if( strstr( buffer, "Struct" ) )
                fld_type = FLD_STRUCT;
#if 0
            fprintf( stderr, "fld_type: %d\n", fld_type );
#endif
        }
    }

    /* If its a structured field, then there needs to be coordinate
       information somewhere, and I don't think there is a universal
       netCDF convention on how to do this.
    */

    if( fld_type == FLD_STRUCT ) {
        /* Names to check for coordinate information */
        const char *list[] = { "XP_COORDINATES", "XP_COORDS",
                               "coordinates_values", "coordinates", NULL };
        const char *name;

        for( i = 0; (name = list[i]); ++i ) {
            nc_stat = nc_inq_varid( nc_id, name, &nc_coord_id );
            if( nc_stat == NC_NOERR ) {
                int temp_ndims;
                /* Found a XP coordinates variable.  NOT to be confused with
                   a netCDF "coordinate variable".  Ideally, I should be
                   checking the size of the dimensions of this variable,
                   but I need to look at the variable information first. */
                nc_stat = nc_inq_varndims( nc_id, nc_coord_id, &temp_ndims );
                /* check to see that it has two dimensions */
                if( nc_stat == NC_NOERR && temp_ndims == 2 ) {
                    int temp_dim_ids[2];
                    size_t temp_nsp;
                    /* we want to look at the dims to recover nspace */
                    nc_stat = nc_inq_vardimid( nc_id, nc_coord_id,
                                               temp_dim_ids );
                    if( nc_stat != NC_NOERR ) continue;
                    /* The first dimension will be npoints */
                    /* The second dimension will be nspace */
                    nc_stat = nc_inq_dimlen( nc_id, temp_dim_ids[1],
                                             &temp_nsp );
                    if( nc_stat != NC_NOERR ) continue;
                    fld_nspace = (int)temp_nsp;
                    break;
                }
            }
        }
        if( !name ) {
            /* Oops, reached end of list without finding the coord var */
            fld_type = FLD_Unspecified;
        }
    }

    /* nc_unlim_dim_id gets a value of -1 if there is no unlimited dim. */
    nc_stat = nc_inq_unlimdim( nc_id, &nc_unlim_dim_id );
    if( nc_unlim_dim_id == INVALID_NC_DIM ) time_step = INVALID_TIME_STEP;

    /* Hold variable information */
    nc_vars = malloc( nvars * sizeof(ncvar_t) );

    /* Get information about each variable */
    j = 0;
    for( i = 0; i < nvars; ++i ) {
        int nc_var_id;
        int dim_ids_temp[NC_MAX_VAR_DIMS];

        /* Given a variable name, get the variable ncid. */
        nc_stat = nc_inq_varid( nc_id, vars[i], &nc_var_id );
        if( nc_stat != NC_NOERR ) {
            nc_vars[j].id = INVALID_NC_VAR; /* invalid nc id */
            nc_vars[j].name[0] = 0;
            continue;
        }
        nc_vars[j].id = nc_var_id;
        /* Copy over the name */
        memcpy( nc_vars[j].name, vars[i], strlen(vars[i])+1 );
        /* data type (float, int, etc.) */
        nc_stat = nc_inq_vartype( nc_id, nc_var_id, &nc_vars[j].type );
        /* number of dimensions */
        nc_stat = nc_inq_varndims( nc_id, nc_var_id, &nc_vars[j].ndims );
        /* dimensions */
        nc_stat = nc_inq_vardimid( nc_id, nc_var_id, dim_ids_temp );

        /* Do a little error checking on the time step variable.
         * The UI makes it possible to enable a time step even if
         * the variable(s selected does not have a time dimension.
         */
        if( i == 0 &&
            time_step != INVALID_TIME_STEP && 
            nc_unlim_dim_id != INVALID_NC_DIM ) {
            if( nc_unlim_dim_id != dim_ids_temp[0] )
                time_step = INVALID_TIME_STEP;
        }

        /* Reverse order of dimensions.  This is due to a quirk in
         * the way the OM C API represents array dimensions.  Doing it
         * this way allows the internal representation of the nC array
         * to match the OM array.  Really.  Trust me on this one, OK?
         */
        for( k = 0; k < nc_vars[j].ndims; ++k ) {
            nc_vars[j].dim_ids[nc_vars[j].ndims - (k+1)] = dim_ids_temp[k];
        }

        /* Dimension info must match for all the variables we read! */
        /* In Express terms, all the node data components must have */
        /* the same grid/mesh. */

        if( j != 0 ) {
            int err = 0;
            if( nc_vars[j].ndims != nc_vars[0].ndims ) {
                err = 1;
            }
            if( !err ) {
                for( k = 0; k < nc_vars[j].ndims; ++k ) {
                    if( nc_vars[j].dim_ids[k] != nc_vars[0].dim_ids[k] ) {
                        err = 1;
                        continue;
                    }
                }
            }
            if( err ) {
                ERRerror( "netcdf read_field", 2, ERR_WARNING,
           "Dimensions of variable %s do not match dimensions of variable %s",
                          nc_vars[j].name, nc_vars[0].name );
                continue;
            }
        }

        ++j;
    }

    /* Number of node data components, usually the same as nvars */
    fld_ncomps = j;
    /* Number of dimensions */
    fld_ndims_total = fld_ndims = nc_vars[0].ndims;

    if( fld_ncomps == 0 || fld_ndims == 0 ) {
        free( nc_vars );
        ncclose( nc_id );
        *out_id = OMnull_obj;
        /* TODO: attempt to find old "Output_Field" and clean it up. */
        return METHOD_SUCCESS;
    }

    /* Hold dimension information */
    nc_dims = malloc( fld_ndims * sizeof(ncdim_t) );

    /* Get information about each dimension */
    fld_nnodes = 1;
    for( i = 0; i < fld_ndims; ++i ) {
        int nc_dim_id = nc_vars[0].dim_ids[i];
        int nc_var_id;
        int temp_ndims = 0, temp_dim_id = INVALID_NC_DIM;
        size_t dim_len;
        char dim_name[128];

        nc_stat = nc_inq_dimlen( nc_id, nc_dim_id, &dim_len );
        if( nc_stat == NC_NOERR ) {
            fld_dims[i] = dim_len;
            nc_dims[i].len = dim_len;
            fld_nnodes *= dim_len;
        }

        nc_stat = nc_inq_dimname( nc_id, nc_dim_id, dim_name );
        memcpy( nc_dims[i].name, dim_name, strlen(dim_name)+1 );

        nc_dims[i].unlim = (nc_dim_id == nc_unlim_dim_id);

#if 0
        fprintf( stderr, "Dimension: %s [%d]\n",
                 nc_dims[i].name,  nc_dims[i].len );
#endif

        nc_stat = nc_inq_varid( nc_id,  dim_name, &nc_var_id );

        if( nc_stat == NC_NOERR )
            nc_stat = nc_inq_varndims( nc_id, nc_var_id, &temp_ndims );

        if( nc_stat == NC_NOERR && temp_ndims == 1 )
            nc_stat = nc_inq_vardimid( nc_id, nc_var_id, &temp_dim_id );

        if( nc_stat == NC_NOERR &&
            temp_ndims == 1 &&
            temp_dim_id == nc_dim_id ) {

            /* Found a "coordinate variable" */
            nc_dims[i].coord_var = 1;
            nc_dims[i].var_id = nc_var_id;
            nc_dims[i].points = malloc( nc_dims[i].len * sizeof(float) );
            /* Get the coordinate data */
            nc_get_var_float( nc_id, nc_var_id, nc_dims[i].points );

#if 0
            for( j = 0; j < nc_dims[i].len; ++j ) {
                fprintf( stderr, "%f ", nc_dims[i].points[j] );
            }
            fprintf( stderr, "\n" );
#endif

            /* TODO: an evenly spaced "coordinate variable" can still
               be represented as a Uniform Field */
            if( fld_type == FLD_Unspecified &&
                (!nc_dims[i].unlim || time_step == INVALID_TIME_STEP) )
                fld_type = FLD_RECT;
        }
        else {
            nc_dims[i].coord_var = 0;
            nc_dims[i].var_id = INVALID_NC_VAR;
            nc_dims[i].points = NULL;
        }
    }

    /* Create the output field */

    if( fld_type == FLD_STRUCT )
        outFld_id = FLDcreate( parent_id, "Mesh_Struct Node_Data",
                               "Output_Field" );
    else if( fld_type == FLD_RECT )
        outFld_id = FLDcreate( parent_id, "Mesh_Rect Node_Data",
                               "Output_Field" );
    else {
        if( fld_type == FLD_Unspecified ) fld_type = FLD_UNIF;
        outFld_id = FLDcreate( parent_id, "Mesh_Unif Node_Data",
                               "Output_Field" );
    }

    if( fld_type == FLD_STRUCT ) {
        /* Not needed for Unif and Rect */
        if( FLDset_nnodes( outFld_id, fld_nnodes ) != OM_STAT_SUCCESS )
            ERR_RETURN( "Cannot set out fld nnodes" );
    }

    /* We are only picking a single time step from the file */
    if( time_step != INVALID_TIME_STEP ) --fld_ndims;

    if( fld_nspace != 0 )
        FLDset_nspace( outFld_id, fld_nspace );
    else
        FLDset_nspace( outFld_id, fld_ndims );

    FLDset_ndim( outFld_id, fld_ndims );

    if( FLDset_dims( outFld_id, fld_dims ) != OM_STAT_SUCCESS )
        ERR_RETURN( "Cannot set out fld dims" );

    if (fld_type == FLD_STRUCT) {
        if( FLDget_coord( outFld_id, &fld_points, &fld_pts_size,
                          OM_GET_ARRAY_WR ) != OM_STAT_SUCCESS )
            ERR_RETURN( "Cannot get out fld coordinates" );
    }
    else {
        if( FLDget_points( outFld_id, &fld_points, &fld_pts_size,
                           OM_GET_ARRAY_WR ) != OM_STAT_SUCCESS )
            ERR_RETURN( "Cannot get out fld points" );
    }

    /* Generate the field's points/coordinates information */

    if( fld_type == FLD_STRUCT ) {
        /* Struct field ... need to get the coord information */
        nc_stat = nc_get_var_float( nc_id, nc_coord_id, fld_points );
    }
    else if( fld_type == FLD_RECT ) {
        /* Rect field ... need to generate the points information */
        int tmp_npoints;

        for( i = 0; i < fld_ndims; ++i ) {
            if( !nc_dims[i].coord_var ) {
                /* Its not a "coordinate variable",
                 * generate a simple linear ramp.
                 */
                if( !nc_dims[i].points )
                    nc_dims[i].points =
                        malloc( nc_dims[i].len * sizeof(float) );
                for( n = 0; n < nc_dims[i].len; ++n ) {
                    nc_dims[i].points[n] = (float)n;
                }
            }
        }

        tmp_npoints = 0;
        for( i = 0; i < fld_ndims; ++i ) {
            for( n = 0; n < nc_dims[i].len; ++n ) {
                for( k = 0; k < fld_ndims; ++k ) {
                    if( i != k ) {
                        /* starting point of each axis */
                        fld_points[tmp_npoints*fld_ndims + k] =
                            nc_dims[k].points[0];
                    }
                    else {
                        fld_points[tmp_npoints*fld_ndims + k] =
                            nc_dims[k].points[n];
                    }
                }
                tmp_npoints += 1;
            }
        }
    }
    else {
        /* Simple Uniform field ... easy */
        for( i = 0; i < fld_ndims; ++i ) {
            if( nc_dims[i].points ) {
                fld_points[i] = nc_dims[i].points[0];
                fld_points[i+fld_ndims] = nc_dims[i].points[nc_dims[i].len-1];
            }
            else {
                fld_points[i] = 0;
                fld_points[i+fld_ndims] = fld_dims[i] - 1;
            }
        }
    }

    /* All done with the points information ... whew! */
    ARRfree( fld_points );
    fld_points = NULL;

    /* Some extra netCDF information that will help us when
       we write out the field to a netcdf file.
    */
    {
        OMobj_id dims_names_id = NCfind_or_create( outFld_id, "NC_dim_names",
                                                   "string NC_dim_names[];" );
        if( !OMis_null_obj( dims_names_id ) ) {
            OMset_array_size( dims_names_id, fld_ndims );
            for( i = 0; i < fld_ndims; ++i ) {
                OMset_str_array_val( dims_names_id, i, nc_dims[i].name );
            }
        }
    }

    /* Look for nC global attributes "history" and "title" */
    /* Could search for all global text attributes (?)  */
    NCtext_attr_nc2xp( nc_id, NC_GLOBAL, "title",
                       outFld_id, "NC_title", "string NC_title;" );

    NCtext_attr_nc2xp( nc_id, NC_GLOBAL, "history",
                       outFld_id, "NC_history", "string NC_history;" );

    /*
     * Node Data
     */

    if (FLDset_node_data_ncomp( outFld_id, fld_ncomps ) == 1) {
        OMobj_id node_data_id, comp_id;
        size_t *nc_start = NULL, *nc_count = NULL;

        if( time_step != INVALID_TIME_STEP ) {
            int nc_dim_index;
            nc_start = malloc( fld_ndims_total * sizeof(size_t) );
            nc_count = malloc( fld_ndims_total * sizeof(size_t) );

            for( i = 0, nc_dim_index = fld_ndims_total-1; 
                 i < fld_ndims_total;
                 ++i, --nc_dim_index ) {

                if( nc_dims[i].unlim ) {
                    nc_start[nc_dim_index] = time_step;
                    nc_count[nc_dim_index] = 1;
                }
                else {
                    nc_start[nc_dim_index] = 0;
                    nc_count[nc_dim_index] = nc_dims[i].len;
                }
            }
        }

        node_data_id = OMfind_subobj( outFld_id, OMstr_to_name("node_data"),
                                      OM_OBJ_RW );

        if( OMis_null_obj(node_data_id) )
            return METHOD_FAILURE;  /* Don't expect this error */

        for( j = 0; j < fld_ncomps; ++j ) {

            if( OMget_array_val(node_data_id, j, &comp_id, OM_OBJ_RW) !=
                OM_STAT_SUCCESS )
                return METHOD_FAILURE;  /* Don't expect this error */

            /* netCDF does not directly support the veclen concept; the app
               has to know, for example, that u_wind and v_wind go together.
               TODO: check for a dimension called "veclen" and then
               Do The Right Thing.
            */
            FLDset_node_data_veclen( outFld_id, j, 1 );

            /* Check for null flags (netCDF standard attributes) */

            nc_stat = nc_inq_att( nc_id, nc_vars[j].id, "_FillValue",
                                  &nc_att_type, &nc_att_len );
            if( nc_stat != NC_NOERR )
                FLDset_node_null_flag( outFld_id, j, 0 );
            else {
                switch( nc_att_type ) {
                case NC_BYTE: {
                    unsigned char nv_byte;
                    nc_stat = nc_get_att_uchar( nc_id, nc_vars[j].id,
                                                "_FillValue", &nv_byte );
                    if( nc_stat == NC_NOERR ) {
                        FLDset_node_null_flag( outFld_id, j, 1 );
                        FLDset_node_null_data( outFld_id, j,
                                               (char *)&nv_byte,
                                               DTYPE_BYTE );
                    }
                    else FLDset_node_null_flag( outFld_id, j, 0 );
                }
                break;
                case NC_SHORT: {
                    short nv_short;
                    nc_stat = nc_get_att_short (nc_id, nc_vars[j].id,
                                                "_FillValue", &nv_short );
                    if( nc_stat == NC_NOERR ) {
                        FLDset_node_null_flag( outFld_id, j, 1 );
                        FLDset_node_null_data( outFld_id, j,
                                               (char *)&nv_short,
                                               DTYPE_SHORT );
                    }
                    else FLDset_node_null_flag( outFld_id, j, 0 );
                }
                break;
                case NC_INT: {
                    int nv_int;
                    nc_stat = nc_get_att_int( nc_id, nc_vars[j].id,
                                              "_FillValue", &nv_int );
                    if( nc_stat == NC_NOERR ) {
                        FLDset_node_null_flag( outFld_id, j, 1 );
                        FLDset_node_null_data( outFld_id, j,
                                               (char *)&nv_int,
                                               DTYPE_INT );
                    }
                    else FLDset_node_null_flag( outFld_id, j, 0 );
                }
                break;
                case NC_FLOAT: {
                    float nv_float;
                    nc_stat = nc_get_att_float( nc_id, nc_vars[j].id,
                                                "_FillValue", &nv_float );
                    if( nc_stat == NC_NOERR ) {
                        FLDset_node_null_flag( outFld_id, j, 1 );
                        FLDset_node_null_data( outFld_id, j,
                                               (char *)&nv_float,
                                               DTYPE_FLOAT );
                    }
                    else FLDset_node_null_flag( outFld_id, j, 0 );
                }
                break;
                case NC_DOUBLE: {
                    double nv_double;
                    nc_stat = nc_get_att_double( nc_id, nc_vars[j].id,
                                                 "_FillValue", &nv_double );
                    if( nc_stat == NC_NOERR ) {
                        FLDset_node_null_flag( outFld_id, j, 1);
                        FLDset_node_null_data( outFld_id, j,
                                               (char *)&nv_double,
                                               DTYPE_DOUBLE );
                    }
                    else FLDset_node_null_flag( outFld_id, j, 0 );
                }
                break;
                }
            }

            /* Put the nC variable name in the field's label slot */
            {
                OMobj_id label_id;
                label_id = OMfind_subobj( comp_id, OMstr_to_name("labels"),
                                          OM_OBJ_RW );
                if( !OMis_null_obj( label_id ) )
                    OMset_str_val( label_id, nc_vars[j].name );
            }

            /* Check for units (netCDF standard attributes) */
            NCtext_attr_nc2xp( nc_id, nc_vars[j].id, "units",
                               comp_id, "units", NULL );

            /* Check for long_name (netCDF standard attributes) */
            NCtext_attr_nc2xp( nc_id, nc_vars[j].id, "long_name",
                               comp_id, "NC_long_name", "string NC_long_name;" );

            switch( nc_vars[j].type ) {
            case NC_BYTE:   fld_data_type = DTYPE_BYTE;  break;
            case NC_SHORT:  fld_data_type = DTYPE_SHORT; break;
            case NC_INT:    fld_data_type = DTYPE_INT;   break;
            case NC_FLOAT:  fld_data_type = DTYPE_FLOAT; break;
            case NC_DOUBLE: fld_data_type = DTYPE_DOUBLE;
            }

            if (FLDget_node_data(outFld_id, j, &fld_data_type,
                                 &fld_node_data, 
                                 &fld_node_data_size, OM_GET_ARRAY_WR) == 1) {
                if( time_step == INVALID_TIME_STEP ) {
                    switch( nc_vars[j].type ) {
                    case NC_BYTE:
                        nc_get_var_uchar( nc_id, nc_vars[j].id,
                                          (unsigned char *)fld_node_data );
                        break;
                    case NC_SHORT:
                        nc_get_var_short( nc_id, nc_vars[j].id,
                                          (short *)fld_node_data );
                        break;
                    case NC_INT:
                        nc_get_var_int( nc_id, nc_vars[j].id,
                                        (int *)fld_node_data );
                        break;
                    case NC_FLOAT:
                        nc_get_var_float( nc_id, nc_vars[j].id,
                                          (float *)fld_node_data );
                        break;
                    case NC_DOUBLE:
                        nc_get_var_double( nc_id, nc_vars[j].id,
                                           (double *)fld_node_data );
                    }
                }
                else {
                    switch( nc_vars[j].type ) {
                    case NC_BYTE:
                        nc_get_vara_uchar( nc_id, nc_vars[j].id,
                                           nc_start, nc_count,
                                           (unsigned char *)fld_node_data );
                        break;
                    case NC_SHORT:
                        nc_get_vara_short( nc_id, nc_vars[j].id,
                                           nc_start, nc_count,
                                           (short *)fld_node_data );
                        break;
                    case NC_INT:
                        nc_get_vara_int( nc_id, nc_vars[j].id,
                                         nc_start, nc_count,
                                         (int *)fld_node_data );
                        break;
                    case NC_FLOAT:
                        nc_get_vara_float( nc_id, nc_vars[j].id,
                                           nc_start, nc_count,
                                           (float *)fld_node_data );
                        break;
                    case NC_DOUBLE:
                        nc_get_vara_double( nc_id, nc_vars[j].id,
                                            nc_start, nc_count,
                                            (double *)fld_node_data );
                    }
                }

                ARRfree( fld_node_data );
            } /* if successfully got node data */
        } /* loop over components */
        if( nc_start ) free( nc_start );
        if( nc_count ) free( nc_count );
    } /* if successfully set ncomp */

    free( nc_vars );
    for( i = 0; i < fld_ndims_total; ++i ) {
        if( nc_dims[i].points )
            free( nc_dims[i].points );
    }
    free( nc_dims );

    ncclose( nc_id );

    *out_id = outFld_id;

    return METHOD_SUCCESS;
}
