/*
			Copyright (c) 1993 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/netcdf/xp_mods/nc_rd_obj.c#1 $
*/

#define	XP_WIDE_API	/* Use Wide APIs */

#include <stdio.h>
#include <sys/types.h>  /* time_t */

#include <avs/util.h>
#include <avs/f_utils.h>
#include <avs/err.h>
#include <avs/dtype.h>
#include <avs/om.h>
#include <avs/om_att.h>
#include <avs/om_type.h>
#include <avs/netcdf.h>

#define ERR_RETURN(A)  {ERRerror("NCread_object", 0, ERR_ORIG, A);return(0);}
#define MAX_NAME_LEN  1024
#define MAX_NDIM  64

/* 64-bit changes. This is for compat with netCDF files written with
 * Express pre-7.0. Added to map pre-7.0 Express types to 7.0 (and later)
 * types. The real issue is the insertion of DTYPE_LONG into the middle
 * of the DTYPE range.
 */
int rd_get_mapped_type[DTYPE_STRING+1] = { 
  DTYPE_CHAR, DTYPE_BYTE, DTYPE_SHORT, DTYPE_INT, 
  DTYPE_FLOAT, DTYPE_DOUBLE, DTYPE_PTR, DTYPE_STRING, 
  DTYPE_LONG };

static char buf[MAX_NAME_LEN];

/* 64-bit changes. taken from 'nc.h' */
extern int nc_get_att(int ncid, int varid, const char *name, void *value); 

extern int  nCDFread(OMobj_id parent, const char *filename);	/* MicroAVS needs this exported */
static int  nCDFread_class(int ncid, char *class_name, int max_len);
static int  nCDFread_obj(int ncid, OMobj_id obj_id);
static int  nCDFcreate_obj(OMobj_id parent, char *name, int type, size_t size,
			   OMobj_id *obj_id);
static int nCDFget_type(nc_type nctype, size_t size, int *type);
static int nCDFread_scalar(int ncid, OMobj_id val_id, int var_id,
			   int type, char *name);
static int nCDFread_array(int ncid, OMobj_id val_id, int var_id, int type,
			  int nvdims, size_t *dims, size_t size, char *name);
static int nCDFget_scalar_type(int type, int *rtype);
static int nCDFget_class(int ncid, int var_id, int nvdims, int *dims, char **class);
static int nCDFset_class(int ncid, OMobj_id obj_id);
static int nCDFcreate_group (OMobj_id parent, char *path_from_parent, int vector, int dim, char *class);


int NCread_object_update(OMobj_id obj_id, OMevent_mask mask, int seq_num)
{
    OMobj_id filename;
    char *str;
    int Result;

    str = NULL;
    filename = OMfind_subobj(obj_id, OMstr_to_name("filename"), OM_OBJ_RD);
    OMget_str_val(filename, &str, 0);

    Result = nCDFread(obj_id, str);

    if (str) free(str);

    return(Result);
}

/* 64-bit porting. Only Modified Internally */
int nCDFread(OMobj_id parent, const char *filename)
{
    OMobj_id out, nout;
    int ncid, stat, nc_opt;
    char class_name[MAX_NAME_LEN];

    char file_buf[AVS_PATH_MAX];
    time_t modTime = 0;

    filename = FILEmap_variables(filename, file_buf);

    if (!filename || !FILEexists(filename, FILE_READ, &modTime)) {
        ERRerror("Read netCDF", 1, ERR_ORIG, "Can't open data file: %s",
                 filename);
        return(0);
    }

    ncopts = NC_VERBOSE;
    stat = nc_open(filename, NC_NOWRITE, &ncid);
    if (stat != NC_NOERR) {
        ERR_RETURN("can't open file for reading");
    }
    ncopts = 0;
    stat = nCDFread_class(ncid, class_name, MAX_NAME_LEN);
    if (stat < 0) {
        ERR_RETURN("error reading object class");
    }
    if (stat == 1)   /* has XP_CLASS, should have all atts set */
        ncopts = NC_VERBOSE;

    nout = OMfind_subobj(parent, OMstr_to_name("net_cdf_out"), OM_OBJ_RW);
    if (!OMis_null_obj(nout)) {
        OMclr_obj_atts(parent, OM_atts_locked);
        OMuser_destroy_obj(nout);
    }
    if (OMparse_buffer(parent, class_name,0) != 1) {
        ERR_RETURN("error creating out object");
    }
    nout = OMfind_subobj(parent, OMstr_to_name("net_cdf_out"), OM_OBJ_RW);
    if (OMis_null_obj(nout)) {
        ERR_RETURN("cannot find out_net_cdf object");
    }

    /* set classes on sub objects */
    nc_opt = ncopts;
    ncopts = 0;
    stat = nCDFset_class(ncid, nout);
    ncopts = nc_opt;

    /* read variables */
    stat = nCDFread_obj(ncid, nout);
    if (stat != 1) {
        ERR_RETURN("error reading object");
    }
    nc_close(ncid);

    out = OMfind_subobj(parent, OMstr_to_name("out"), OM_OBJ_RW);
    if (OMis_null_obj(out)) {
        ERR_RETURN("cannot find out object");
    }

    stat = OMset_obj_ref(out, nout, 0);
    if (stat < 1) {
        ERR_RETURN("error setting reference");
    }

    return(stat);
}

/* 64-bit porting. Only Modified Internally */
static int  nCDFread_class(int ncid, char *class_name, int max_len)
{
    int stat;
    size_t len;
    nc_type type;

    /* initialize  string since netcdf may not put \0 */
    memset(class_name, 0, max_len);
    stat = nc_inq_att(ncid, NC_GLOBAL, "XP_CLASS", &type, &len);
    if (stat != NC_NOERR) {
        /* class does not exist assume group */
        strcpy(class_name, "group+write net_cdf_out;");
        return(0);
    }
    else if ((int)len+12 > max_len) {
        ERRerror("read_netCDF", 0, ERR_ORIG, "class name is too long");
        return(-1);
    }
    else {
        stat = nc_get_att_text(ncid, NC_GLOBAL, "XP_CLASS", class_name);
        if (stat != NC_NOERR)
            return(-1);
        strcat(class_name,"+write net_cdf_out;");
        return(1);
    }
}

/* 64-bit porting. Only Modified Internally */
static int  nCDFread_obj(int ncid, OMobj_id obj_id)
{
    OMobj_id val_id;
    int ndims, nvars;
    int nc_opt, i, j, stat, dtype, rtype;
    nc_type	nctype;
    char  name[MAX_NC_NAME], path_name[MAX_NC_NAME];
    int var_dims[MAX_VAR_DIMS], nvdims, nvatts;
    size_t size, dimsize[MAX_NDIM];
    int type;

    nc_inq(ncid, &ndims, &nvars, NULL, NULL);
    for(i=0; i<nvars; i++) {
        /* initialize  string since netcdf may not put \0 */
        memset(name, 0, MAX_NC_NAME);
        memset(path_name, 0, MAX_NC_NAME);

        stat = nc_inq_var(ncid, i, name, &nctype, &nvdims, var_dims, &nvatts);
        if(stat != NC_NOERR) {
            ERR_RETURN("error on inquire netcdf variable");
        }

        nc_opt = ncopts;
        ncopts = 0;

        strcpy(buf, "XPobject_classes");
        stat = nc_get_att(ncid, i, buf, (void *)name);
        ncopts = nc_opt;
        if(stat == NC_NOERR) /*  class attribute does  exist */
            continue;


        if (nvdims == 0)   /* scalar */
            size = 0;
        else 	    /*  array, get size */
            for (size=1,j=0; j<nvdims; j++) {
                stat = nc_inq_dim(ncid, var_dims[j], NULL, &dimsize[j]);
                if(stat != NC_NOERR) {
                    ERR_RETURN("error on inquire netcdf dimension");
                }
                size *= dimsize[j];
            }

        /* get XP name and type */
        strcpy(buf, name);
        strcat(buf, "_path");
        stat = nc_get_att_text(ncid, i, buf, path_name);
        if(stat != NC_NOERR) /*  path attribute does not exist */
            strcpy(path_name, name);

        strcpy(buf, name);
        strcat(buf, "_type");
        stat = nc_get_att_int(ncid, i, buf, &type);
        dtype = rd_get_mapped_type[type]; /* 64-bit changes. map old-style dtype to proper dtype */
        if(stat != NC_NOERR) {/*  type attribute does not exist */
            if (nCDFget_type(nctype, size, &dtype) != 1) {
                sprintf(buf, "cannot convert type for %s variable", name);
                ERRerror("NCread_object", 0, ERR_ORIG, buf);
                continue;
            }
        }
        nCDFget_scalar_type(dtype, &rtype);

        val_id = OMfind_str_subobj(obj_id, path_name, OM_OBJ_RW);

        if (OMis_null_obj(val_id)) {
            /* does not exist we need to create it */
            /*
              fprintf(stderr, "create object: %s\n",path_name);
            */
            stat = nCDFcreate_obj(obj_id, path_name, dtype, size, &val_id);
            if (stat != 1) {
                sprintf(buf, "cannot create %s object", name);
                ERRerror("NCread_object", 0, ERR_ORIG, buf);
                continue;
            }
        }
        else {
            /*
             * If a pre-7.0 Express stored, for example, FLD.nnodes
             * as an int, then don't overrride the current type.
             *
             * There is a potential for screwups here, but the most
             * common use of write_netcdf was to write out fields
             * in the pre-HDF5 days.
             */
            int dtype_curr;
            int dont_set_type = 0;
            stat = OMget_data_type(val_id, &dtype_curr);
            if (stat == 1) {
                if( dtype_curr == DTYPE_LONG &&
                    dtype == DTYPE_INT ) {
                    dont_set_type = 1;
                }
            }

            if( !dont_set_type ) {
                stat = OMset_data_type(val_id, dtype);
                if (stat != 1) {
                    sprintf(buf, "cannot set type for %s object", name);
                    ERRerror("NCread_object", 0, ERR_ORIG, buf);
                    continue;
                }
            }
        }

        if (size) {
            /*
              fprintf(stderr, "read array: %s\n",path_name);
            */
            stat = nCDFread_array(ncid, val_id,  i, dtype, 
                                  nvdims, dimsize, size, name);
        }
        else {
            /*
              fprintf(stderr, "read scalar: %s\n",path_name);
            */
            stat = nCDFread_scalar(ncid, val_id, i, rtype, name);
        }

        if (stat != 1) {
            sprintf(buf, "cannot read variable %s", name);
            ERRerror("NCread_object", 0, ERR_ORIG, buf);
            continue;
        }
    }
    return(1);
}

/* 64-bit porting. Directly Modified */
static int nCDFcreate_obj(OMobj_id parent, char *path_from_parent, int type, size_t size,
			  OMobj_id *obj_id)
{
    char  ctype[64];
    OMobj_id par_id, delem, dims_elem;
    int stat;
    const int c_dot = '.';
    char *name, path[MAX_NAME_LEN];

    name = strrchr(path_from_parent, c_dot);
    if (name == NULL) {
        name = path_from_parent;
        strcpy(path, "");
    }
    else {
        name += 1;
        strncpy(path, path_from_parent, strlen(path_from_parent)-strlen(name)-1);
        path[strlen(path_from_parent)-strlen(name)-1] = '\0';
    }

    if (strcmp(path,"") == 0)
        par_id = parent;
    else {
        par_id = OMfind_str_subobj(parent, path, OM_OBJ_RW);
        if (OMis_null_obj(par_id)) {
            ERR_RETURN("cannot find parent object");
        }
    }
    switch (type) {
    case DTYPE_CHAR:
        strcpy(ctype, "byte");
        break;
    case DTYPE_BYTE:
        strcpy(ctype, "char");
        break;
    case DTYPE_SHORT:
        strcpy(ctype, "short");
        break;
    case DTYPE_INT:
        strcpy(ctype, "int");
        break;
    case DTYPE_LONG:
        strcpy(ctype, "long");
        break;
    case DTYPE_FLOAT:
        strcpy(ctype, "float");
        break;
    case DTYPE_DOUBLE:
        strcpy(ctype, "double");
        break;
    case DTYPE_STRING:
        strcpy(ctype, "string");
        break;
    default:
        sprintf(buf, "unsupported type %d for object %s", type, name);
        ERR_RETURN(buf);
    }
    *obj_id = OMcreate_obj(OMstr_to_name(name), OMstr_to_type(ctype), par_id, 
                           OMnull_obj, OMlocal_proc_id);
    if (OMis_null_obj(*obj_id))
        return(0);

    if (size && type != DTYPE_STRING) {   /* array */
        delem = OMcreate_obj(OM_NULL_NAME, OM_type_raw_long, OMnull_obj,
                             OMnull_obj, OMlocal_proc_id);
        if (OMis_null_obj(delem))
            return(0);
        OMset_long_val(delem, size);
        dims_elem = OMcreate_obj(OM_name_dims, OM_type_val_list, *obj_id, 
                                 OMnull_obj, OMlocal_proc_id);
        if (OMis_null_obj(dims_elem))
            return(0);
        stat = OMadd_subobj(dims_elem, delem);
        if (stat != 1)
            return(0);
    }
    return(1);
}

/* 64-bit porting. Directly Modified */
static int nCDFget_type(nc_type nctype, size_t size, int *type)
{
    switch (nctype) {
    case NC_CHAR:
        *type = DTYPE_CHAR;
        break;
    case NC_BYTE:
        *type = DTYPE_BYTE;
        break;
    case NC_SHORT:
        *type = DTYPE_SHORT;
        break;
    case NC_INT:
        *type = DTYPE_INT;
        break;
    case NC_FLOAT:
        *type = DTYPE_FLOAT;
        break;
    case NC_DOUBLE:
        *type = DTYPE_DOUBLE;
        break;
    default:
        return(0);
    }
    if (nctypelen(nctype) != DTYPEtype_size[*type]) {
        return(0);
    }
    if ((nctype == NC_CHAR || nctype == NC_BYTE) && size)
        *type = DTYPE_STRING;
    return(1);
}

static int nCDFget_scalar_type(int type, int *rtype)
{
    switch (type) {
    case DTYPE_CHAR:
    case DTYPE_BYTE:
    case DTYPE_SHORT:
    case DTYPE_INT:
        *rtype = DTYPE_INT;
        break;

    case DTYPE_FLOAT:
    case DTYPE_DOUBLE:
        *rtype = DTYPE_DOUBLE;
        break;

    case DTYPE_STRING:
        *rtype = DTYPE_STRING;
        break;

    default:
        break;
    }
    return(1);
}

/* 64-bit porting. Only Modified Internally */
static int nCDFread_scalar(int ncid, OMobj_id val_id, int var_id, int type, char *name)
{
    int stat;
    size_t	slab_start[1], slab_size[1];

    slab_start[0]=0;
    slab_size[0]=1;

    switch (type) {
    case DTYPE_CHAR: {
        char cval;
        stat = nc_get_vara_text(ncid, var_id, slab_start, slab_size, &cval);
        if(stat != NC_NOERR) {
            sprintf(buf, "error getting netcdf variable %s", name);
            ERR_RETURN(buf);
        }
        stat = OMset_int_val(val_id, (int)cval);
        if (stat != 1) {
            sprintf(buf, "cannot set value for %s object", name);
            ERR_RETURN(buf);
        }
        break;
    }
    case DTYPE_BYTE: {
        unsigned char uval;
        stat = nc_get_vara_uchar(ncid, var_id, slab_start, slab_size, &uval);
        if(stat != NC_NOERR) {
            sprintf(buf, "error getting netcdf variable %s", name);
            ERR_RETURN(buf);
        }
        stat = OMset_int_val(val_id, (int)uval);
        if (stat != 1) {
            sprintf(buf, "cannot set value for %s object", name);
            ERR_RETURN(buf);
        }
        break;
    }
    case DTYPE_SHORT: {
        short sval;
        stat = nc_get_vara_short(ncid, var_id, slab_start, slab_size, &sval);
        if(stat != NC_NOERR) {
            sprintf(buf, "error getting netcdf variable %s", name);
            ERR_RETURN(buf);
        }
        stat = OMset_int_val(val_id, (int)sval);
        if (stat != 1) {
            sprintf(buf, "cannot set value for %s object", name);
            ERR_RETURN(buf);
        }
        break;
    }
    /* netCDF does not have 64-bit ints.  Putting aside strategies involving
     * using doubles or pairs of 32-bit ints as not worth the complexity,
     * we just settle for trying to put a DTYPE_LONG into a NC_INT.
     */
    case DTYPE_INT:
    case DTYPE_LONG:
    {
        int ival;
        stat = nc_get_vara_int(ncid, var_id, slab_start, slab_size, &ival);
        if(stat != NC_NOERR) {
            sprintf(buf, "error getting netcdf variable %s", name);
            ERR_RETURN(buf);
        }
        stat = OMset_int_val(val_id, ival);
        if (stat != 1) {
            sprintf(buf, "cannot set value for %s object", name);
            ERR_RETURN(buf);
        }
        break;
    } 
    case DTYPE_FLOAT: {
        float fval;
        stat = nc_get_vara_float(ncid, var_id, slab_start, slab_size, &fval);
        if(stat != NC_NOERR) {
            sprintf(buf, "error getting netcdf variable %s", name);
            ERR_RETURN(buf);
        }
        stat = OMset_real_val(val_id, (double)fval);
        if (stat != 1) {
            sprintf(buf, "cannot set value for %s object", name);
            ERR_RETURN(buf);
        }
        break;
    }
    case DTYPE_DOUBLE: {
        double dval;
        stat = nc_get_vara_double(ncid, var_id, slab_start, slab_size, &dval);
        if(stat != NC_NOERR) {
            sprintf(buf, "error getting netcdf variable %s", name);
            ERR_RETURN(buf);
        }
        stat = OMset_real_val(val_id, dval);
        if (stat != 1) {
            sprintf(buf, "cannot set value for %s object", name);
            ERR_RETURN(buf);
        }
        break;
    }
    default:
        sprintf(buf, "unsupported type %d for object %s", type, name);
        ERR_RETURN(buf);
    }
    return(1);
}

/* 64-bit porting. Directly Modified */
static int nCDFread_array(int ncid, OMobj_id val_id, int var_id, int type,
			  int nvdims, size_t *dims, size_t size, char *name)
{
    int     i, stat, dtype;
    xp_long ndims, ldims[MAX_NDIM], tdim, tsize;
    size_t	slab_start[MAX_NDIM], slab_size[MAX_NDIM];
    void    *val;
    OMobj_id dims_id, sub_id;

    for (i=0; i<nvdims; i++) {
        ldims[i] = dims[i];
        slab_start[i]=0;
        slab_size[i]=dims[i];
    }
    if (type == DTYPE_STRING)
        dtype = DTYPE_CHAR;
    else
        dtype = type;

    val = ARRalloc(NULL, dtype, size, NULL);
    if (val == NULL) {
        sprintf(buf, "allocate array for %s", name);
        ERR_RETURN(buf);
    }

    switch (dtype) {
    case DTYPE_CHAR:
        stat = nc_get_vara_text(ncid, var_id, slab_start, slab_size, (char *)val);
        break;
    case DTYPE_BYTE:
        stat = nc_get_vara_uchar(ncid, var_id, slab_start, slab_size, (unsigned char *)val);
        break;
    case DTYPE_SHORT:
        stat = nc_get_vara_short(ncid, var_id, slab_start, slab_size, (short *)val);
        break;
    case DTYPE_LONG:
    case DTYPE_INT:
        stat = nc_get_vara_int(ncid, var_id, slab_start, slab_size, (int *)val);
        break;
    case DTYPE_FLOAT:
        stat = nc_get_vara_float(ncid, var_id, slab_start, slab_size, (float *)val);
        break;
    case DTYPE_DOUBLE:
        stat = nc_get_vara_double(ncid, var_id, slab_start, slab_size, (double *)val);
        break;
    default:
        sprintf(buf, "unsupported type %d for object %s", dtype, name);
        ERR_RETURN(buf);
    }		

    if(stat != NC_NOERR) {
        sprintf(buf, "error getting netcdf variable %s", name);
        ERR_RETURN(buf);
    }
    if (type == DTYPE_STRING) {
        stat = OMset_str_val(val_id, (char *)val);
        ARRfree(val);
    }
    else {
        ERRsquash_start();
        dims_id = OMfind_subobj(val_id, OM_name_dims, OM_OBJ_RW);
        if (OMis_null_obj(dims_id)) {
            stat = OMset_array_dims(val_id, nvdims, ldims);
            if (stat != 1) {
                sprintf(buf, "cannot set dimensions for variable %s", name);
                ERR_RETURN(buf);
            }
        }
        else {
            if (OMget_array_size(dims_id, &ndims) != 1 ) {
                stat = OMset_array_dims(val_id, nvdims, ldims);
                if (stat != 1) {
                    sprintf(buf, "cannot set dimensions for variable %s", name);
                    ERR_RETURN(buf);
                }
            }
            else if ((int)ndims != nvdims) {
                /* make sure that dimensions are not set
                   in the template, we dont want to overwrite them */
                for (stat=1, tsize=1, i=0; i<(int)ndims; i++) {
                    if (OMget_array_val(dims_id, i, &sub_id, OM_OBJ_RW) != 1) {
                        stat = 0;
                        break;
                    }
                    stat = OMget_long_val(sub_id, &tdim);
                    if (stat != 1)
                        break;
                    tsize *= tdim;
                }
                if (stat != 1 || tsize != size) {
                    stat = OMset_array_dims(val_id, nvdims, ldims);
                    if (stat != 1) {
                        sprintf(buf, "cannot set dimensions for variable %s", name);
                        ERR_RETURN(buf);
                    }
                }
            }
            else {
                for (stat=1, i=0; i<nvdims; i++) {
                    if (OMget_array_val(dims_id, i, &sub_id, OM_OBJ_RW) != 1) {
                        stat = 0;
                        break;
                    }
                    stat = OMset_long_val(sub_id, ldims[i]);
                    if (stat != 1)
                        break;
                }
                if (stat != 1) {
                    stat = OMset_array_dims(val_id, nvdims, ldims);
                    if (stat != 1) {
                        sprintf(buf, "cannot set dimensions for variable %s", name);
                        ERR_RETURN(buf);
                    }
                }
            }
        }
        ERRsquash_end();
        stat = OMset_array(val_id, type, (char *)val, size, OM_SET_ARRAY_FREE);
    }
    if (stat != 1) {
        sprintf(buf, "cannot set variable %s", name);
        ERR_RETURN(buf);
    }
    return(1);
}

/* 64-bit porting. Only Modified Internally */
static int nCDFset_class(int ncid, OMobj_id obj_id)
{
    int ndims, nvars;
    int i, stat;
    int dim, vector;
    char  name[MAX_NC_NAME], path_name[MAX_NC_NAME], *class;
    int var_dims[MAX_VAR_DIMS], nvdims, nvatts;
    nc_type	nctype;

    nc_inq(ncid, &ndims, &nvars, NULL, NULL);
    for(i=0; i<nvars; i++) {
        /* initialize  string since netcdf may not put \0 */
        memset(name, 0, MAX_NC_NAME);
        memset(path_name, 0, MAX_NC_NAME);

        stat = nc_inq_var(ncid, i, name, &nctype, &nvdims, var_dims, &nvatts);
        if (stat != NC_NOERR) {
            ERR_RETURN("error on inquire netcdf variable");
        }

        strcpy(buf, "XPobject_classes");
        stat = nc_get_att_int(ncid, i, buf, &vector);
        if (stat != NC_NOERR) /*  class attribute does not exist */
            continue;
        if (vector < 0 || vector > 1000) /* sanity check */
            continue;

        stat = nCDFget_class(ncid, i, nvdims, var_dims, &class);
        if (stat != 1) {
            ERR_RETURN("cannot get class");
        }

        /* get XP path and dim */

        strcpy(buf, name);
        strcat(buf, "_path");
        stat = nc_get_att_text(ncid, i, buf, path_name);
        if (stat != NC_NOERR) {/*  path attribute does not exist */
            ERR_RETURN("path does not exist for array of groups");
        }

        strcpy(buf, name);
        strcat(buf, "_group_dim");
        stat = nc_get_att_int(ncid, i, buf, &dim);
        if (stat != NC_NOERR) {/*  type attribute does not exist */
            ERR_RETURN("dim does not exist for array of groups");
        }
        if (dim < 0 || dim > 1000) /* sanity check */
            dim = 0;
        if (dim == 0)
            continue;
        stat = nCDFcreate_group (obj_id, path_name, (int)vector, (int)dim, class);

        ARRfree(class);
    }
    return(1);
}

/* 64-bit porting. Only Modified Internally */
static int nCDFget_class(int ncid, int var_id, int nvdims, int *dims, char **class)
{
    int     i, stat;
    size_t	slab_start[MAX_NDIM], slab_size[MAX_NDIM];
    size_t  size, dimsize[MAX_NDIM];

    for (size=1,i=0; i<nvdims; i++) {
        stat = nc_inq_dim(ncid, dims[i], NULL, &dimsize[i]);
        if (stat != NC_NOERR) {
            ERR_RETURN("error on inquire netcdf dimension");
        }
        size *= dimsize[i];
    }
    slab_start[0]=0;
    slab_size[0]=size;

    *class = (char *)ARRalloc(NULL, DTYPE_CHAR, size, NULL);
    if (*class == NULL) {
        ERR_RETURN("cannot allocate value for classes");
    }
    memset(*class, 0 , size);

    stat = nc_get_vara_text(ncid, var_id, slab_start, slab_size, *class);
    if (stat != NC_NOERR) {
        ERR_RETURN("error on getting classes variable");
    }
    return(1);
}

static int nCDFcreate_group (OMobj_id parent, char *path_from_parent, int vector, int ndim, char *class)
{
    OMobj_id par_id, val_id, dims_id, class_id, mrg_id, obj_id;
    char *name, *ps, *pe, *token, path[MAX_NAME_LEN], ref_name[MAX_NAME_LEN];
    int i, stat;
    const int c_dot = '.', c_space= ' ';

    name = strrchr(path_from_parent, c_dot);
    if (name == NULL) {
        name = path_from_parent;
        strcpy(path, "");
    }
    else {
        name += 1;
        strncpy(path, path_from_parent, strlen(path_from_parent)-strlen(name)-1);
        path[strlen(path_from_parent)-strlen(name)-1] = '\0';
    }

    if (strcmp(path,"") == 0)
        par_id = parent;
    else {
        par_id = OMfind_str_subobj(parent, path, OM_OBJ_RW);
        if (OMis_null_obj(par_id)) {
            ERR_RETURN("cannot find parent object");
        }
    }
    val_id = OMfind_str_subobj(par_id, name, OM_OBJ_RW);
    if (OMis_null_obj(val_id)) {
        /* does not exist, create group */
        val_id = OMcreate_obj(OMstr_to_name(name), OMstr_to_type("group"), par_id, 
                              OMnull_obj, OMlocal_proc_id);
        if (OMis_null_obj(val_id)) {
            ERR_RETURN("cannot create group object");
        }
    }
    if (vector) {
        dims_id = OMparse_dims_to_obj(val_id, "[]");
        if (OMis_null_obj(dims_id)) {
            ERR_RETURN("cannot parse dims object");
        }
        OMadd_subobj(val_id, dims_id);

        stat =  OMset_obj_ref_mode(val_id, OM_OBJ_REF);
        if (stat != 1) {
            ERR_RETURN("cannot set reference mode");
        }
    }
    OMset_obj_atts(val_id, OM_atts_nres);

    ps = class;
    for (i=0; i<ndim; i++) {
        pe = strchr(ps, c_space);
        if (pe == NULL)
            strcpy(path, ps);
        else {
            strncpy(path, ps, strlen(ps)-strlen(pe));
            path[strlen(ps)-strlen(pe)] = '\0';
        }
        token = strtok(path, "+");

        class_id = OMfind_subobj (OMtempl_obj, OMstr_to_name(token),
                                  OM_OBJ_RD);
        if (OMis_null_obj(class_id)) {
            ERR_RETURN("cannot find class template");
        }
        if (vector) {
            sprintf(ref_name, "%s_ref_%d", name, i);
            obj_id = OMcopy_obj(class_id, OMstr_to_name(ref_name), par_id, OMlocal_proc_id, 
                                OM_COPY_CLOSE);
        }
        else {
            obj_id = val_id;
            OMmerge_obj(class_id, obj_id, 0);
        }

        token = strtok(NULL, "+");
        while (token) {
            mrg_id = OMfind_subobj (OMtempl_obj, OMstr_to_name(token),
                                    OM_OBJ_RD);
            if (OMis_null_obj(mrg_id)) {
                ERR_RETURN("cannot find class template for merge");
            }
            OMmerge_obj(mrg_id, obj_id, 0);
            token = strtok(NULL, " ");
        }
        if (vector) {
            stat = OMadd_obj_ref(val_id, obj_id, 0);
            if (stat !=1) {
                ERR_RETURN("cannot add object reference");
            }
        }

        ps = pe+1;
    }

    return(1);
}
