/*
			Copyright (c) 1993 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/netcdf/xp_mods/nc_wr_obj.c#1 $
*/

#include <stdio.h>
#include <sys/types.h>  /* time_t */

#define	XP_WIDE_API	/* Use Wide APIs */
#include <avs/util.h>
#include <avs/f_utils.h>
#include <avs/err.h>
#include <avs/dtype.h>
#include <avs/om.h>
#include <avs/om_att.h>
#include <avs/netcdf.h>

#define ERR_RETURN(A)  {ERRerror("NCwrite_object", 0, ERR_ORIG, A);return(0);}
#define MAX_CLASS 64
#define MAX_NAME_LEN  1024

/* 64-bit changes. This is for compat with netCDF files written with
 * Express pre-7.0. Added to map Express 7.0 (and later) types to pre-7.0
 * types. The real issue is the insertion of DTYPE_LONG into the middle
 * of the DTYPE range.
 */
int wr_get_mapped_type[DTYPE_STRING+1] = {0, 1, 2, 3, 4, 5, 8, 6, 7};

static char buf[MAX_NAME_LEN];

static int  nCDFwrite(OMobj_id in, char *file);
static int  nCDFwrite_class(int ncid, OMobj_id obj_id);
static int  nCDFwrite_prim(int ncid, OMobj_id obj_id, char *path_from_parent);
static int  nCDFget_variable(OMobj_id obj_id, nc_type *nctype, xp_long *size, int *ndim, xp_long *dims,
			  void **val, int *vector, char *name);
static int  nCDFwrite_obj(int ncid, OMobj_id ctx_id, OMobj_id obj_id,
			char *path_from_parent, OMobj_id par_id, int do_group);
static int  nCDFare_the_same(OMobj_id ctx_id, OMobj_id obj_id, char *path_from_parent);
static int  nCDFparse_name(char *name, char *new_name);
static int  nCDFget_groups(OMobj_id obj_id, int *size, char *name, int *vector);
static int  nCDFwrite_groups(int ncid, OMobj_id obj_id, char *path_from_parent, OMobj_id par_id);


int NCwrite_object_update(OMobj_id obj_id, OMevent_mask mask, int seq_num)
{
    OMobj_id filename, in, in_ref;
    char *str;
    int Result;

    in = OMfind_subobj(obj_id, OMstr_to_name("in"), OM_OBJ_RW);
    in_ref = OMfind_subobj(obj_id, OMstr_to_name("in"), OM_OBJ_RW);
    if (Result = OMget_obj_val(in_ref, &in) != OM_STAT_SUCCESS) {
        ERR_RETURN("can't access in subobj");
    }

    str = NULL;
    filename = OMfind_subobj(obj_id, OMstr_to_name("filename"),
                             OM_OBJ_RD);
    OMget_str_val(filename, &str, 0);

    Result = nCDFwrite(in, str);

    if (str) free(str);

    return(Result);
}

/* 64-bit porting. Only Modified Internally */
static int  nCDFwrite(OMobj_id in, char *filename)
{
    int  ncid, stat;
    char path_from_parent[MAX_NAME_LEN];
    char file_buf[AVS_PATH_MAX];

    ncopts = NC_VERBOSE;

    filename = FILEmap_variables(filename, file_buf);

    if (filename == NULL) {
        ERR_RETURN("bad filename");
    }

    stat = nc_create(filename, NC_CLOBBER, &ncid);

    if (stat != NC_NOERR) {
        ERR_RETURN("can't open file for writing");
    }
    stat = nCDFwrite_class(ncid, in);
    if (stat != 1) {
        ERR_RETURN("error writting object class");
    }

    strcpy(path_from_parent, "");

    stat = nCDFwrite_obj(ncid, in, in, path_from_parent, in, 0);
    if (stat != 1) {
        ERR_RETURN(" error writing netCDF file");
    }

    ncclose(ncid);
    return(stat);
}

/* 64-bit porting. Only Modified Internally */
static int  nCDFwrite_class(int ncid, OMobj_id obj_id)
{
    OMobj_id classes[MAX_CLASS];
    OMobj_name obj_name;
    int i, stat, num_classes;
    char  name[MAX_NAME_LEN];
/**
   if (OMget_obj_val(obj_id, &val_id) != 1) {
       ERR_RETURN("cannot get object value");
   }
**/
    stat = OMget_user_classes(obj_id, classes, &num_classes, MAX_CLASS, 0, OM_CLASS_FOLLOW_REFS);
    if (stat != 1 || num_classes == 0) {
        strcpy(name, "group");
    }
    else {
        stat = OMget_obj_name(classes[0], &obj_name);
        if (stat != 1) {
            ERR_RETURN(" cannot get class name");
        }
        strcpy(name, OMname_to_str(obj_name));
        for (i=1; i<num_classes; i++) {
            stat = OMget_obj_name(classes[i], &obj_name);
            if (stat != 1) {
                ERR_RETURN(" cannot get class name");
            }
            strcat(name, "+");
            strcat(name, OMname_to_str(obj_name));
        }
    }

    stat = nc_put_att_text(ncid, NC_GLOBAL, "XP_CLASS", strlen(name)+1, name);
    if (stat != NC_NOERR) {
        ERR_RETURN("cannot create global netcdf attribute");
    }
    nc_enddef(ncid);
    return(1);
}

/* 64-bit porting. Only Modified Internally */
static int nCDFwrite_prim(int ncid, OMobj_id obj_id, char *path_from_parent)
{
    int  dim_id[MAX_VAR_DIMS], var_id, dtype;
    int  i, stat, ndim, vector;
    xp_long size, dims[OM_ARRAY_MAXDIM];
    nc_type nctype;
    size_t dim_size, start[MAX_VAR_DIMS], count[MAX_VAR_DIMS];
    char  name[MAX_NAME_LEN];
    void *val;
    int type;

    stat = nCDFget_variable(obj_id, &nctype, &size, &ndim, dims, &val, &vector, name);
    if (stat != 1) {
        /*
          sprintf(buf, "cannot  write object in netCDF %s", path_from_parent);
          ERR_RETURN(buf);
        */
        return(0);
    }

    strcpy(buf, path_from_parent);

    nCDFparse_name(buf, name);
    strcpy(buf, name);

    stat = nc_redef(ncid);
    if (stat != NC_NOERR) {
        nc_enddef(ncid);
        ERR_RETURN(" cannot open netcdf define mode");
    }
    if (vector) {
        /*  create dimension */
        for (i=0; i<ndim; i++) {
            sprintf(buf, "%s_dim_%d", name, i);
            dim_size = dims[i];
            /**
               fprintf(stderr, "create dimension %s for %s\n", buf, name);
            **/
            stat = nc_def_dim(ncid, buf, dim_size, &dim_id[i]);
            if (stat != NC_NOERR) {
                nc_enddef(ncid);
                ERR_RETURN(" cannot create netcdf dimension");
            }
            start[i]=0;
            count[i]=dim_size;
        }
    }
    else {
        start[0]=0;
        count[0]=size;
    }

    /* create variable */
    stat = nc_def_var(ncid, name, nctype, ndim, dim_id, &var_id);
    if (stat != NC_NOERR) {
        nc_enddef(ncid);
        ERR_RETURN(" cannot create netcdf variable");
    }

    /* create attributes */
    strcpy(buf, name);
    strcat(buf, "_path");
    stat = nc_put_att_text(ncid, var_id, buf, strlen(path_from_parent)+1, path_from_parent);
    if (stat != NC_NOERR) {
        nc_enddef(ncid);
        ERR_RETURN(" cannot create netcdf path attribute");
    }

    strcpy(buf, name);
    strcat(buf, "_type");
    stat = OMget_data_type(obj_id, &dtype);
    type = wr_get_mapped_type[dtype];
    stat = nc_put_att_int(ncid, var_id, buf, NC_INT, 1, &type);
    if (stat != NC_NOERR) {
        nc_enddef(ncid);
        ERR_RETURN(" cannot create netcdf type attribute");
    }

    nc_enddef(ncid);

    /* put data */
	switch(nctype){
	case NC_CHAR:
		stat = nc_put_vara_text(ncid, var_id, start, count, (const char *) val);
		break;
	case NC_BYTE:
		stat = nc_put_vara_uchar(ncid, var_id, start, count, (const unsigned char *) val);
		break;		
	case NC_SHORT:
		stat = nc_put_vara_short(ncid, var_id, start, count, (const short *) val);
		break;
	case NC_INT:
		stat = nc_put_vara_int(ncid, var_id, start, count, (const int *) val);
		break;
	case NC_FLOAT:
		stat = nc_put_vara_float(ncid, var_id, start, count, (const float *) val);
		break;
	case NC_DOUBLE: 
		stat = nc_put_vara_double(ncid, var_id, start, count, (const double *) val);
		break;
	}
	
    if (stat != NC_NOERR) {
        ERR_RETURN(" cannot set values for netcdf variable");
    }

    ARRfree(val);
    return(1);
}

/* 64-bit porting. Directly Modified */
static int nCDFget_variable(OMobj_id obj_id, nc_type *nctype, xp_long *size, int *ndim, xp_long *dims,
			  void **val, int *vector, char *name)
{
    int type, stat;
    OMobj_name obj_name;
    char  *str;

    stat = OMget_obj_name(obj_id, &obj_name);
    if (stat != 1) {
        ERR_RETURN(" cannot get object name");
    }
    strcpy(name, OMname_to_str(obj_name));

    if (OMget_data_type(obj_id, &type) != 1) {
        sprintf(buf, "cannot get type for object %s", name);
        ERR_RETURN(buf);
    }
    if (OMget_array_dims(obj_id, ndim, dims) != 1 || *ndim == 0) {
        /* scalar */
        *ndim = 0;
        *size = 1;
        *vector = 0;
        switch (type) {
        case DTYPE_LONG:
#ifdef WORDLENGTH_64
        {
            /* It might overflow ... lets check */
            int s1, s2;
            int itemp;
            xp_long ltemp;
            s1 = OMget_int_val(obj_id, &itemp );
            s2 = OMget_long_val(obj_id, &ltemp );
            if( (s1 == 1) && (s2 == 1) && (itemp != ltemp) ) {
                ERR_RETURN("Overflow - cannot store 64-bit values");
            }
          /* FALLTHROUGH */
        }
#endif
        case DTYPE_CHAR:
        case DTYPE_BYTE:
        case DTYPE_SHORT:
        case DTYPE_INT:
            *nctype = NC_INT;
            *val = (int *)ARRalloc(NULL, DTYPE_INT, *size, NULL);
            stat = OMget_int_val(obj_id, (int *)(*val));
            if (stat != 1) {
                /*
                  sprintf(buf, "cannot get value for object %s", name);
                  ERR_RETURN(buf);
                */
                return(0);
            }
            break;

        case DTYPE_FLOAT:
        case DTYPE_DOUBLE:
            *nctype = NC_DOUBLE;
            *val = (double *)ARRalloc(NULL, DTYPE_DOUBLE, *size, NULL);
            stat = OMget_real_val(obj_id, (double *)(*val));
            if (stat != 1) {
                /*
                  sprintf(buf, "cannot get value for object %s", name);
                  ERR_RETURN(buf);
                */
                return(0);
            }
            break;

        case DTYPE_STRING:
            *ndim = 1;
            *vector = 1;
            *nctype = NC_CHAR;
            str = NULL;
            stat = OMget_str_val(obj_id, &str, 0);
            if (stat != 1) {
                /*
                  sprintf(buf, "cannot get value for object %s", name);
                  ERR_RETURN(buf);
                */
                return(0);
            }
            *size = strlen(str)+1;
            dims[0] = *size;
            *val = (char *)ARRalloc(NULL, DTYPE_CHAR, *size, NULL);
            strcpy(*val, str);
            free(str);
            break;

        default:
            sprintf(buf, "unsupported type %d for object %s", type, name);
            ERR_RETURN(buf);
        }
    }
    else {
        /* array */
        switch (type) {
        case DTYPE_CHAR:
            *nctype = NC_CHAR;
            break;
        case DTYPE_BYTE:
            *nctype = NC_BYTE;
            break;
        case DTYPE_SHORT:
            *nctype = NC_SHORT;
            break;
    /* netCDF does not have 64-bit ints.  Putting aside strategies involving
     * using doubles or pairs of 32-bit ints as not worth the complexity,
     * we just settle for trying to put a DTYPE_LONG into a NC_INT.
     */
        case DTYPE_LONG:
        case DTYPE_INT:
            *nctype = NC_INT;
            break;
        case DTYPE_FLOAT:
            *nctype = NC_FLOAT;
            break;
        case DTYPE_DOUBLE:
            *nctype = NC_DOUBLE;
            break;
        default:
            sprintf(buf, "unsupported type %d for object %s", type, name);
            ERR_RETURN(buf);
        }
        if (nctypelen(*nctype) != DTYPEtype_size[type]) {
            sprintf(buf, "object %s: netcdf type %d and express type %d have different sizes", 
                    name, (int)(*nctype), type);
            ERR_RETURN(buf);
        }
        *vector = 1;
        *val = NULL;
        *size = 0;
        if( type == DTYPE_LONG ) {
            /* Force conversion to int.  netCDF doesn't handle 64-bit ints.
             * Sometimes we'll overflow and sometimes we won't :-).
             */
            type = DTYPE_INT;
        }
        else { 
            type = DTYPE_UNSET;
        }
        stat = OMget_array_sz(obj_id, &type, (char **)val, size, OM_GET_ARRAY_RD);
        if (stat != 1) {
            /*
              sprintf(buf, "cannot get value for object %s", name);
              ERR_RETURN(buf);
            */
            return(0);
        }
    }
    return(1);
}

/* 64-bit porting. Only Modified Internally */
static int nCDFwrite_obj(int ncid, OMobj_id ctx_id, OMobj_id obj_id,
		char *path_from_parent, OMobj_id par_id, int do_group)

    /* obj_id = find_str_subobj(ctx_id,path_from_parent); */
{
    OMobj_id val_id;
    OMobj_name obj_name;
    int type, ndim;
    int stat=1;
    xp_long i, ndim_w, dims[OM_ARRAY_MAXDIM];
    char  new_path_name[MAX_NAME_LEN];

    if (OMget_data_type(obj_id, &type) == 1) {
        if (nCDFare_the_same(ctx_id, obj_id, path_from_parent))
            return(1);
        stat = OMget_obj_name(obj_id, &obj_name);
        if (stat != 1) {
            ERR_RETURN(" cannot get obj name");
        }
        if (nCDFare_the_same(par_id, obj_id, OMname_to_str(obj_name)))
            return(1);
        /* else write it out */
        if(OMget_obj_atts(obj_id, OM_atts_virtual) == 1 ||
           OMget_obj_atts(obj_id, OM_atts_wasvirtual) == 1 ||
           OMget_obj_atts(obj_id, OM_atts_nosave) == 1)
            return(1);
        stat = nCDFwrite_prim(ncid, obj_id, path_from_parent);
    }
    else {	/* we are a group */
        if (OMget_array_dims(obj_id,&ndim,dims) == 1 && ndim != 0) {
            /* We are an array */
            OMget_array_size(obj_id, &ndim_w);
            nCDFwrite_groups(ncid, obj_id, path_from_parent, par_id);
            for (i = 0; i < ndim_w; i++) {
                if (OMget_array_val(obj_id,i,&val_id,OM_OBJ_RD) == 1 &&
                    OMget_obj_val(val_id,&val_id) == 1) {
                    sprintf(new_path_name,"%s[%ld]",path_from_parent,i);
                    nCDFwrite_obj(ncid, ctx_id, val_id, new_path_name, obj_id, 0);
                }
            }
        }
        else {  /* just group */
            if (do_group) {
                nCDFwrite_groups(ncid, obj_id, path_from_parent, par_id);
            }
            OMget_num_subobjs_mode(obj_id,&ndim_w,OM_SUB_SYMBOLIC);
            for (i = 0; i < ndim_w; i++) {
                /* process primitives first */
                if (OMget_array_subobj(obj_id,i,&val_id,OM_OBJ_RD | OM_SUB_SYMBOLIC) == 1) {
                    stat = OMget_obj_name(val_id, &obj_name);
                    if (stat != 1)
                        return(stat);
                    val_id = OMfind_subobj(obj_id, obj_name,OM_OBJ_RD);
                    if(OMis_null_obj(val_id)) {
                        sprintf(buf, "cannot find object %s", path_from_parent);
                        ERR_RETURN(buf);
                    }
                    if (OMget_data_type(val_id, &type) == 1) {
                        if (strlen(OMname_to_str(obj_name)) != 0) {
                            if (strlen(path_from_parent) == 0)
                                sprintf(new_path_name,"%s",
                                        OMname_to_str(obj_name));
                            else
                                sprintf(new_path_name,"%s.%s",path_from_parent,
                                        OMname_to_str(obj_name));
                        }
                        else
                            strcpy(new_path_name, path_from_parent);
                        nCDFwrite_obj(ncid, ctx_id, val_id, new_path_name, obj_id, 0);
                    }
                }
            }
            for (i = 0; i < ndim_w; i++) {
                /* process groups second */
                if (OMget_array_subobj(obj_id,i,&val_id,OM_OBJ_RD | OM_SUB_SYMBOLIC) == 1) {
                    stat = OMget_obj_name(val_id, &obj_name);
                    if (stat != 1)
                        return(stat);
                    val_id = OMfind_subobj(obj_id, obj_name,OM_OBJ_RD);
                    if(OMis_null_obj(val_id)) {
                        sprintf(buf, "cannot find object %s", path_from_parent);
                        ERR_RETURN(buf);
                    }
                    if (OMget_data_type(val_id, &type) != 1) {
                        if (strlen(OMname_to_str(obj_name)) != 0) {
                            if (strlen(path_from_parent) == 0)
                                sprintf(new_path_name,"%s",
                                        OMname_to_str(obj_name));
                            else
                                sprintf(new_path_name,"%s.%s",path_from_parent,
                                        OMname_to_str(obj_name));
                        }
                        else
                            strcpy(new_path_name, path_from_parent);
                        nCDFwrite_obj(ncid, ctx_id, val_id, new_path_name, obj_id, 1);
                    }
                }
            }
        }
    }
    return(stat);
}


static int nCDFare_the_same(OMobj_id ctx_id, OMobj_id obj_id, char *path_from_parent)
{
    OMobj_id classes[MAX_CLASS], class_sub_id;
    int i, num_classes, type, mode;

/**
   if (OMget_obj_val(ctx_id, &val_id) != 1) {
       ERR_RETURN("cannot get object value");
   }
**/

    if (OMget_data_type(obj_id, &type) == 1)
        mode =  OM_MATCH_STRUCT |OM_MATCH_EXACT | OM_MATCH_NO_ERROR;
    else
        mode =  OM_MATCH_EXACT | OM_MATCH_NO_ERROR;

    if (OMget_user_classes(ctx_id, classes, &num_classes,MAX_CLASS,0,OM_CLASS_FOLLOW_REFS) == 1) {
        for (i = num_classes-1; i >=0; i--) {
            class_sub_id = OMfind_str_subobj(classes[i],path_from_parent,
                                             OM_OBJ_RD);
            if (!OMis_null_obj(class_sub_id)) {
                return(OMmatch_obj(class_sub_id, obj_id, mode));
            }
        }
    }
    return(0);
}

static int nCDFparse_name(char *name, char *new_name)
{
    int i, j, l;

    l = strlen(name);
    for(j=0,i=0; i<l; i++) {
        if (name[i] == '.')
            new_name[j++] = '_';
        else if (name[i] != '[' && (name[i] != ']'))
            new_name[j++] = name[i];
    }
    new_name[j] = '\0';
    return(1);
}

/* 64-bit porting. Only Modified Internally */
static int  nCDFwrite_groups(int ncid, OMobj_id obj_id, char *path_from_parent, OMobj_id par_id)
{
    OMobj_name obj_name;
    int  dim_id, var_id, nclass;
    int  stat, ndim, vector;
    size_t dim_size, start[1], count[1];
    char  name[MAX_NAME_LEN], class[MAX_NAME_LEN];
    int rndim;
    xp_long rdims[OM_ARRAY_MAXDIM];

    stat = OMget_obj_name(obj_id, &obj_name);
    if (stat != 1) {
        ERR_RETURN(" cannot get obj name");
    }
    strcpy(name, OMname_to_str(obj_name));
    if (strcmp(name, "")==0)
        return(0);

    if (OMget_array_dims(obj_id, &rndim, rdims) != 1 || rndim == 0) {
        if (nCDFare_the_same(par_id, obj_id, name))
            return(1);
    }

    stat = nCDFget_groups(obj_id, &nclass, class, &vector);
    if (stat != 1) {
        sprintf(name, "cannot write classes for %s object", path_from_parent);
        ERR_RETURN(name);
    }
    if (nclass == 0)
        return(0);

    strcpy(buf, path_from_parent);

    nCDFparse_name(buf, name);
    strcpy(buf, name);

    stat = nc_redef(ncid);
    if (stat != NC_NOERR) {
        nc_enddef(ncid);
        ERR_RETURN(" cannot open netcdf define mode");
    }

    strcpy(buf, name);
    strcat(buf, "_dim");
    dim_size = strlen(class)+1;
    /**
       fprintf(stderr, "create dimension %s for %s\n", buf, name);
    **/
    stat = nc_def_dim(ncid, buf, dim_size, &dim_id);
    if (stat != NC_NOERR) {
        nc_enddef(ncid);
        ERR_RETURN(" cannot create netcdf dimension");
    }
    ndim = 1;

    /* create variable */
    stat = nc_def_var(ncid, name, NC_CHAR, ndim, &dim_id, &var_id);
    if (stat != NC_NOERR) {
        nc_enddef(ncid);
        ERR_RETURN(" cannot create netcdf variable");
    }

    /* create attributes */
    strcpy(buf, name);
    strcat(buf, "_path");
    stat = nc_put_att_text(ncid, var_id, buf, strlen(path_from_parent)+1, path_from_parent);
    if (stat != NC_NOERR) {
        nc_enddef(ncid);
        ERR_RETURN(" cannot create netcdf path attribute");
    }

    strcpy(buf, name);
    strcat(buf, "_group_dim");
    stat = nc_put_att_int(ncid, var_id, buf, NC_INT, 1, &nclass);
    if (stat != NC_NOERR) {
        nc_enddef(ncid);
        ERR_RETURN(" cannot create netcdf type attribute");
    }

    strcpy(buf, "XPobject_classes");
    stat = nc_put_att_int(ncid, var_id, buf, NC_INT, 1, &vector);
    if (stat != NC_NOERR) {
        nc_enddef(ncid);
        ERR_RETURN(" cannot create netcdf type attribute");
    }

    nc_enddef(ncid);

    /* put data */
    start[0]=0;
    count[0]=strlen(class)+1;

    stat = nc_put_vara_text(ncid, var_id, start, count, class);
    if (stat == -1) {
        ERR_RETURN(" cannot set values for netcdf variable");
    }

    return(1);
}

/* 64-bit porting. Only Modified Internally */
static int  nCDFget_groups(OMobj_id obj_id, int *size, char *name, int *vector)
{
	OMobj_id val_id, classes[MAX_CLASS];
	OMobj_name obj_name;
	int i, n, stat, num_classes;
	int ndim;
	xp_long arr_sz, dims[OM_ARRAY_MAXDIM];

	*size = 0;

	if (OMget_array_dims(obj_id, &ndim, dims) != 1 || ndim == 0) {
		/* Not an array, just make sure the loop runs once. */
		*vector = 0;
		arr_sz = 1;
		val_id = obj_id;
	}
	else {
		*vector = 1;
		OMget_array_size(obj_id, &arr_sz);
	}

	for (n = 0; n < (int)arr_sz; n++) {
		if (*vector)
			if (OMget_array_val(obj_id,n,&val_id,OM_OBJ_RD) != 1) {
				ERR_RETURN(" cannot get array value");
			}
		if (OMget_obj_val(val_id,&val_id) != 1) {
			ERR_RETURN(" cannot get object value");
		}
		stat = OMget_user_classes(val_id, classes, &num_classes, MAX_CLASS, 0, OM_CLASS_FOLLOW_REFS);
		if (stat != 1 || num_classes == 0) {
			if (*size == 0)
				strcpy(name, "group");
			else {
				if (strlen("group")+2 >= MAX_NAME_LEN) {
					*size = 0;
					ERR_RETURN("too many classes");
				}
				strcat(name, " ");
				strcat(name, "group");
			}
		}
		else {
			stat = OMget_obj_name(classes[0], &obj_name);
			if (stat != 1) {
				ERR_RETURN(" cannot get class name");
			}
			if (*size == 0)
				strcpy(name, OMname_to_str(obj_name));
			else {
				if (strlen(OMname_to_str(obj_name))+2 >= MAX_NAME_LEN) {
					*size = 0;
					ERR_RETURN("too many classes");
				}
				strcat(name, " ");
				strcat(name, OMname_to_str(obj_name));
			}
			for (i=1; i<num_classes; i++) {
				stat = OMget_obj_name(classes[i], &obj_name);
				if (stat != 1) {
					ERR_RETURN(" cannot get class name");
				}
				if (strlen(OMname_to_str(obj_name))+2 >= MAX_NAME_LEN) {
					*size = 0;
					ERR_RETURN("too many classes");
				}
				strcat(name, "+");
				strcat(name, OMname_to_str(obj_name));
			}
		}
		*size += 1;
	}

	return(1);
}
