/*
			Copyright (c) 1994 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/tool/arr_base.c#1 $
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define XP_WIDE_API	/* Use Wide APIs */
#include <avs/port.h>
#include <avs/dll_in.h>
#include <avs/dtype.h>
#include <avs/util.h>
#include <avs/arr.h>
#include <avs/err.h>

/* 
 * This package allows inter-process manipulation of arrays.
 *   An abstraction is provided that allows a module to allocate an array,
 * send an array to a remote process, and properly free that array without
 * the caller knowing exactly how the array itself is managed.  Array types
 * are "malloc'd", "shared memory", etc.
 */

extern ARRtype *ARR_type_unmanaged;
static ARRtype *ARR_types = NULL;
static ARRtype *ARR_default_type = NULL;
static int ARR_trace = 0;

static void ARR_add_array (ARRtype *, int, void *, size_t, char *, int);
static ARRarray *ARR_find_array (void *);
static void ARR_del_array (void *);

/* First arg is really a COMMconn *   */
int (*ARR_read_prim_rtn) (void *, int, char *);
/* First arg is really a RPpacket *   */
void (*ARR_write_int_rtn) (void *, int);
void (*ARR_write_long_rtn) (void *, xp_long);
/* First arg is really a COMMconn *   */
int (*ARR_read_array_rtn) (void *, int, char **, xp_long);
/* First arg is really a RPpacket *   */
void (*ARR_write_array_rtn) (void *, int, char *, xp_long);

void ARRinit_types(void);	/* arr_type.c */

void ARRinit(void)
{
   if (getenv("ARR_TRACE")) {
      fprintf(stderr,"ARR tracing turned on\n");
      ARR_trace = 1;
   }
   ARRinit_types();
}

int
ARRset_trace(void *ptr, int value)
{
   if (ptr == NULL) ARR_trace = value;
   else {
      ARRarray *tblent = ARR_find_array(ptr);
      if (ptr == NULL) {
	 ERRverror("ARRset_trace",ERR_ORIG,
		   "can't find entry for ptr: 0x%p",ptr);
	 return(-1);
      }
      tblent->trace = value;

      if (value) {
	 fprintf(stderr,
	    "ARR: begin tracing: refcnt=%d, type=%s, size=%lu, ptr=0x%p\n",
	     tblent->refcnt, DTYPEtype_id_to_str(tblent->d_type),
	     tblent->d_len, ptr);
      }
   }
   return(1);
}

int
ARRget_trace(void *ptr)
{
   if (ptr == NULL) return ARR_trace;
   else {
      ARRarray *tblent = ARR_find_array(ptr);
      if (ptr == NULL) {
	 ERRverror("ARRget_trace",ERR_ORIG,
		   "can't find entry for ptr: 0x%p",ptr);
	 return(-1);
      }
      return ARR_trace || tblent->trace;
   }
}

void
ARRset_default_type(ARRtype *ntype)
{
   ARR_default_type = ntype;
}

/*
 * Use pointers back into the RP routines to prevent a recursive depedendency
 * from ARR to RP.
 */
/* 64-bit porting. Directly Modified */
void
ARRset_comm_funcs(
    int (*read_prim)(void *, int, char *),
    void (*write_int)(void *, int),
    void (*write_long)(void *, xp_long),
    int (*read_arr)(void *, int, char **, xp_long),
    void (*write_arr)(void *, int, char *, xp_long) )
{
   ARR_read_prim_rtn = read_prim;
   ARR_write_int_rtn = write_int;
   ARR_write_long_rtn = write_long;
   ARR_read_array_rtn = read_arr;
   ARR_write_array_rtn = write_arr;
}

/* 64-bit porting. Directly Modified */
ARRtype *
ARRnew_type(
     const char *name,
     void *(*palloc) (size_t, char *),
     void *(*prealloc) (void *, size_t, char *),
     void (*pfree) (void *, char *),
     int (*pread) (void *, int *, char **, xp_long *, int),
     int (*pwrite) (void *, int, char *, xp_long, int ) )
{
   ARRtype *ntype;

   ALLOCN(ntype,ARRtype,1,"can't allocate new array type");

   strcpy(ntype->name,name);
   ntype->palloc = palloc;
   ntype->prealloc = prealloc;
   ntype->pfree = pfree;
   ntype->pread = pread;
   ntype->pwrite = pwrite;
   /* First 4 characters of name must be unique... */
   ntype->code = (name[0] << 24) | (name[1] << 16) | (name[2] << 8) | name[3];

   ntype->next = ARR_types;
   ARR_types = ntype;

   return(ntype);
}

ARRtype *
ARRname_to_type(const char *name)
{
   ARRtype *ntype;
   for (ntype = ARR_types; ntype != NULL; ntype = ntype->next) 
      if (!strcmp(ntype->name,name)) return(ntype);
   return(NULL);
}

ARRtype *
ARRcode_to_type(unsigned int code)
{
   ARRtype *ntype;
   for (ntype = ARR_types; ntype != NULL; ntype = ntype->next) 
      if (code == ntype->code) return(ntype);
   return(NULL);
}


static ARRarray **hash_table;
static int table_size = 0;

#define ARR_HASH_SIZE   127 /* pick a prime number (using factor(1)) */
/* hash a pointer by tossing potentially zero bits and moding by hash size */
#define ARR_HASH(PTR)	((((xp_ulong)PTR) >> 3) % table_size)

static void
ARR_add_array(
     ARRtype *type,
     int d_type,
     void *d_ptr,
     size_t d_len,
     char *info,
     int trace )
{
   ARRarray **tblent, *nent;
   if (table_size == 0) {
      ALLOCN(hash_table,ARRarray *,ARR_HASH_SIZE,"can't alloc array hash"); 
      table_size = ARR_HASH_SIZE;
   }

   ALLOCN(nent,ARRarray,1,"can't allocate array entry");
   nent->arr_type = type;
   nent->d_type = d_type;
   nent->d_ptr = d_ptr;
   nent->d_len = d_len;
   nent->info = info;
   nent->refcnt = 1;
   nent->update = NULL;
   nent->upd_arg = NULL;
   nent->trace = trace;
   /* Now add it to the list for this hash value */
   tblent = &hash_table[ARR_HASH(d_ptr)];
   nent->next = *tblent;
   *tblent = nent;
}

static ARRarray *
ARR_find_array(void *ptr)
{
   ARRarray *tblent;

   /* Avoid divide by zero if this hasn't been opened up yet...*/
   if (table_size == 0) return(NULL);

   for (tblent = hash_table[ARR_HASH(ptr)]; 
	tblent != NULL && tblent->d_ptr != ptr; tblent = tblent->next)
      ;

   return(tblent);
}

static void
ARR_del_array(void *ptr)
{
   ARRarray **tblent, *tmp;

   for (tblent = &hash_table[ARR_HASH(ptr)]; 
	*tblent != NULL && (*tblent)->d_ptr != ptr; tblent = &(*tblent)->next)
      ;
   if (*tblent == NULL) {
      ERRerror("ARR_del_array", 1, ERR_ORIG,
	       "unable to delete array table entry for pointer: 0x%p", ptr);
      return;
   }
   tmp = *tblent;
   *tblent = (*tblent)->next;
   FREE(tmp);
}

void *
ARRalloc_f(
     ARRtype *arr_type,
     int  d_type,
     size_t d_len,
     char *arr_info,
     const char *file,
     int  line )
{
   void *d_ptr;
   if (arr_type == NULL) {
      arr_type = ARR_default_type; 
      if (arr_type == NULL) {
	 ERRerror("ARRalloc", 0, ERR_ORIG, "array package not initialized!");
	 return(NULL);
      }
   }
   d_ptr = (*arr_type->palloc)(d_len*DTYPEtype_size[d_type], arr_info);
   if (d_ptr != NULL) ARR_add_array(arr_type,d_type,d_ptr,d_len,arr_info,0);

   if (ARR_trace) {
	fprintf(stderr,"ARR: alloced: refcnt=1, type=%s, size=%lu, ptr=0x%p\n",
		DTYPEtype_id_to_str(d_type), d_len, d_ptr);
	fprintf(stderr,"     in %s at line %d\n", file, line);
   }

   return(d_ptr);
}

int
ARRnew_managed(
     ARRtype *type,
     void *ptr,
     int  d_type,
     size_t d_len,
     char *arr_info )
{
   if (type == NULL) type = ARR_default_type;
   if (ptr != NULL) ARR_add_array(type, d_type, ptr, d_len, arr_info, 0);

   if (ARR_trace) {
	fprintf(stderr,
		"ARR: new managed: refcnt=1, type=%s, size=%lu, ptr=0x%p\n",
		DTYPEtype_id_to_str(d_type), d_len, ptr);
   }

   return(1);
}

int
ARRnew_unmanaged(
     void *ptr,
     int  d_type,
     size_t d_len,
     char *arr_info )
{
   ARRtype *arr_type;

   arr_type = ARR_type_unmanaged; 
   if (arr_type == NULL) {
      ERRerror("ARRnew_unmanaged", 0,
               ERR_ORIG, "array package not initialized!");
      return(-1);
   }

   if (ptr != NULL) ARR_add_array(arr_type,d_type,ptr,d_len,arr_info,0);

   if (ARR_trace) {
	fprintf(stderr,
		"ARR: new unmanaged: refcnt=1, type=%s, size=%lu, ptr=0x%p\n",
		DTYPEtype_id_to_str(d_type), d_len, ptr);
   }

   return(1);
}

void *
ARRrealloc_f(
     void *ptr,
     int  d_type,
     size_t d_len,
     char *arr_info,
     const char *file,
     int line )
{
   ARRarray *tblent = ARR_find_array(ptr);
   void *d_ptr;
   ARRtype *arr_type;
   int old_trace;

   if (tblent == NULL) {
      ERRerror("ARRrealloc", 1, ERR_ORIG,
	       "unable to find entry for array pointer: 0x%p", ptr);
      return(NULL);
   }
   arr_type = tblent->arr_type;

   d_ptr = (*tblent->arr_type->prealloc)(ptr, d_len*DTYPEtype_size[d_type],
					 arr_info);

   if (ARR_trace || tblent->trace) {
      fprintf(stderr,"ARR: realloced: refcnt=%d, type=%s, size=%lu, ptr=0x%p\n",
              tblent->refcnt, DTYPEtype_id_to_str(d_type), d_len, d_ptr);
      fprintf(stderr,"     in %s at line %d\n", file, line);
      old_trace = 1;
   }
   else old_trace = 0;

   ARR_del_array(ptr);
   if (d_ptr != NULL) ARR_add_array(arr_type, d_type, d_ptr, d_len, arr_info,
				    old_trace);

   return(d_ptr);
}

void
ARRfree_f(void *ptr, const char *file, int line)
{
   ARRarray *tblent = ARR_find_array(ptr);
   int (*upd_func) (void *, void *, int, size_t, int);

   if (tblent == NULL) {
      ERRerror("ARRfree", 1, ERR_ORIG,
	       "unable to find entry for array pointer: 0x%p",ptr);
      return;
   }

   /* 
    * We want to allows call the update function
    * even if this
    */
   if (tblent->update != NULL && tblent->upd_arg != NULL) {
      /*
       * Need to set update func to NULL first so that we don't recurse
       * around in case someone tries to free this pointer again.
       */
      upd_func = tblent->update;
      tblent->update = NULL;
      (*upd_func)(tblent->upd_arg, ptr, tblent->d_type, tblent->d_len,
		  ARR_LAST_UPDATE);
      tblent->upd_arg = NULL;
   }

   if (--tblent->refcnt == 0) {
      void (*pfree) (void *,char *);
      char *info;
      if (ARR_trace || tblent->trace) {
	  fprintf(stderr,"ARR: free: refcnt=%d, type=%s, size=%lu, ptr=0x%p\n",
		tblent->refcnt, DTYPEtype_id_to_str(tblent->d_type),
		tblent->d_len, ptr);
	  fprintf(stderr,"     in %s at line %d\n", file, line);
      }
      /* 
       * Save these guys, delete the pointer from the table and then free
       * it.  Mostly just to prevent tools from complaining about using
       * a free'd pointer.
       */
      pfree = tblent->arr_type->pfree;
      info = tblent->info;
      ARR_del_array(ptr);
      (*pfree)(ptr,info);
   }
   else {
      if (ARR_trace || tblent->trace) {
	  fprintf(stderr,
	    "ARR: decrement: refcnt=%d, type=%s, size=%lu, ptr=0x%p\n",
	    tblent->refcnt, DTYPEtype_id_to_str(tblent->d_type),
	    tblent->d_len, ptr);
	  fprintf(stderr,"     in %s at line %d\n", file, line);
      }
   }
}

void
ARRfree_no_update_f(void *ptr, const char *file, int line)
{
   ARRarray *tblent = ARR_find_array(ptr);

   if (tblent == NULL) {
      ERRerror("ARRfree", 1, ERR_ORIG,
	       "unable to find entry for array pointer: 0x%p", ptr);
      return;
   }

   if (--tblent->refcnt == 0) {
      void (*pfree) (void *, char *);
      char *info;
      if (ARR_trace || tblent->trace) {
	  fprintf(stderr,"ARR: free: refcnt=%d, type=%s, size=%lu, ptr=0x%p\n",
		tblent->refcnt, DTYPEtype_id_to_str(tblent->d_type),
		tblent->d_len, ptr);
	  fprintf(stderr,"     in %s at line %d\n", file, line);
      }
      /* 
       * Save these guys, delete the pointer from the table and then free
       * it.  Mostly just to prevent tools from complaining about using
       * a free'd pointer.
       */
      pfree = tblent->arr_type->pfree;
      info = tblent->info;
      ARR_del_array(ptr);
      (*pfree)(ptr,info);
   }
   else {
      if (ARR_trace || tblent->trace) {
	  fprintf(stderr,
	    "ARR: decrement: refcnt=%d, type=%s, size=%lu, ptr=0x%p\n",
	    tblent->refcnt, DTYPEtype_id_to_str(tblent->d_type),
	    tblent->d_len, ptr);
	  fprintf(stderr,"     in %s at line %d\n", file, line);
      }
   }
}

int
ARRhas_update_func(void *ptr)
{
   ARRarray *tblent = ARR_find_array(ptr);
   if (tblent == NULL) return(-1);

   if (tblent->update == NULL) return(0);

   return(1);
}

void
ARRset_update_func(
     void *ptr,
     int (*upd_func) (void *, void *, int, size_t, int),
     void *upd_arg )
{
   ARRarray *tblent = ARR_find_array(ptr);
   if (tblent == NULL) return;

   if ((ARR_trace || tblent->trace) && tblent->update != NULL) {
      fprintf(stderr,"ARR: set_update_func: replacing existing update\n");
   }

   tblent->update = upd_func;
   tblent->upd_arg = upd_arg;
}

void
ARRupdate(void *ptr, int final)
                     /* ARR_LAST_UPDATE OR'd with ARR_NOT_MODIFIED */
{
   ARRarray *tblent = ARR_find_array(ptr);

   if (tblent->update != NULL) {
      (*tblent->update)(tblent->upd_arg, ptr, tblent->d_type, tblent->d_len,
			final);
      if (final & ARR_LAST_UPDATE) {
	 tblent->update = NULL;
	 tblent->upd_arg = NULL;
      }
   }
}

void
ARRincr_refcnt_f(void *ptr, const char *file, int line)
{
   ARRarray *tblent = ARR_find_array(ptr);

   if (tblent == NULL) {
      ERRerror("ARRincr_refcnt", 1, ERR_ORIG,
	       "unable to find entry for array pointer: 0x%p", ptr);
      return;
   }
   tblent->refcnt++;
   if (ARR_trace || tblent->trace) {
       fprintf(stderr,
	     "ARR: increment: refcnt=%d, type=%s, size=%lu, ptr=0x%p\n",
	     tblent->refcnt, DTYPEtype_id_to_str(tblent->d_type),
	     tblent->d_len, ptr);
       fprintf(stderr,"     in %s at line %d\n", file, line);
       if (tblent->refcnt > 6)
	   fprintf(stderr,"      ******refcnt is getting big!\n");
   }
}

int
ARRvalid(void *ptr)
{
   if (ptr == NULL) return(0);
   if (ARR_find_array(ptr) != NULL) return(1);
   else return(0);
}

/* 64-bit porting. Modified with Parallel APIs */
int
ARRarray_read(void *conn, int *typep, char **pptr, xp_long *lenp, int mode)
{
   int stat;
   unsigned int arr_code;
   ARRtype *arr_type;

   stat = (*ARR_read_prim_rtn)(conn, DTYPE_INT, (char *)&arr_code);
   if (stat != 1) {
      ERRerror("ARRarray_read", 1, ERR_ORIG,"failure in read primitive");
      return(-1);
   }
   arr_type = ARRcode_to_type(arr_code);

   if (arr_type == NULL) {
      ERRerror("ARRarray_read", 1, ERR_ORIG,"can't find array type with code: %d",
		arr_code);
      return(-1);
   }
   return((*arr_type->pread)(conn, typep, pptr, lenp, mode));
}

#ifdef WORDLENGTH_64
int
ARRarray_read_n(void *conn, int *typep, char **pptr, int *lenp, int mode)
{
   int status;
   xp_long lenp_w;

   status = ARRarray_read(conn, typep, pptr, &lenp_w, mode);
   *lenp = (int)lenp_w; /* Copy 64-bit int to a 32-bit int */

   return(status);
}
#endif

/* 64-bit porting. Directly Modified */
int
ARRarray_write(void *conn, int type, char *ptr, xp_long len, int mode)
{
   ARRtype *arr_type;
   ARRarray *tblent = ARR_find_array(ptr);

   if (tblent == NULL) arr_type = ARR_default_type;
   else {
      arr_type = tblent->arr_type;
      /* 
       * Update the type of the array in case someone changed it since last
       * time...
       */
      tblent->d_type = type;
   }

   (*ARR_write_int_rtn)(conn, arr_type->code);

   return((*arr_type->pwrite)(conn, type, ptr, len, mode));
}

int
ARRrefcnt(void *ptr)
{
   ARRarray *tblent = ARR_find_array(ptr);

   if (tblent == NULL) {
      ERRerror("ARRrefcnt", 0, ERR_ORIG,
		"array ptr has not been entered into ARR package");
      return(-1);
   }
   return(tblent->refcnt);
}

/*
 * We want to allow systems like purify to do their "dangling" pointer
 * test.  Thus we free up all of our references to any memory that
 * was allocated through this package (without freeing the memory).
 */
void
ARRexit(void)
{
   int i;
   ARRarray *ent, *tmp;
   for (i = 0; i < table_size; i++) {
      for (ent = hash_table[i]; ent != NULL; ) { 

         hash_table[i] = NULL;

         if (ARR_trace || ent->trace) {
            fprintf(stderr,
                    "ARR: exit: refcnt=%d, type=%s, size=%lu, ptr=0x%p\n",
                    ent->refcnt, DTYPEtype_id_to_str(ent->d_type),
                    ent->d_len, ent->d_ptr);
         }

	 tmp = ent; 
	 ent = ent->next;
         tmp->d_ptr = NULL;  /* Note that we are not freeing the array */
	 FREE(tmp);          /* Free internal data structures */
      }
   }
   if (hash_table) FREE(hash_table);
   hash_table = NULL;
}

size_t
ARRget_alloc_size(int d_type, size_t lower, size_t upper)
{
   int i;
   size_t total = 0, arrsize;
   ARRarray *ent;

   for (i = 0; i < table_size; i++) {
      for (ent = hash_table[i]; ent != NULL; ) { 
         if (d_type == DTYPE_UNSET || d_type == ent->d_type) {
            arrsize = ent->d_len * DTYPEtype_size[ent->d_type];
            if (((lower <= arrsize) || (lower == 0)) && 
                ((arrsize < upper)  || (upper == 0)))
               total += arrsize;
         }
         ent = ent->next;
      }
   }
   return total;
}

size_t
ARRget_alloc_size_total(void)
{
   return ARRget_alloc_size(DTYPE_UNSET, 0, 0);
}


/* The following will take care of those functions that did not get
 * recompiled with the new arr.h (or never included it to begin with!).
 */

#undef ARRalloc
#undef ARRrealloc
#undef ARRfree
#undef ARRincr_refcnt

/* 64-bit porting. Directly Modified */
void *
ARRalloc(ARRtype *arr_type, int d_type, size_t d_len, char *arr_info)
{
    return(ARRalloc_f(arr_type, d_type, d_len, arr_info, "undefined", -1));
}

/* 64-bit porting. Directly Modified */
void *
ARRrealloc(void *ptr, int d_type, size_t d_len, char *arr_info)
{
    return(ARRrealloc_f(ptr, d_type, d_len, arr_info, "undefined", -1));
}

void
ARRfree(void *ptr)
{
    ARRfree_f(ptr, "undefined", -1);
}

void
ARRincr_refcnt(void *ptr)
{
    ARRincr_refcnt_f(ptr, "undefined", -1);
}
