/*
			Copyright (c) 1994 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/tool/error.c#1 $
*/
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <avs/port.h>
#include <avs/util.h>

extern void db_crash_handler( int sig );  /* om/crash.c */

#define ERR_EXTERN
#include <avs/err.h>

#define ERR_MAX_ERRORS 16  /* maximum errors			 */
#define ERR_MAX_SIZE   512 /* maximum size in char's of an error */
static char ERRstack[ERR_MAX_ERRORS][ERR_MAX_SIZE];

#define ERR_PREFIX_SIZE 128
#define ERR_CTX_CHUNK_SIZE 5

typedef struct _ERRctx {
   int (*func) (void *, char *, int);
   char str[ERR_PREFIX_SIZE];
   void *datap;
} ERRctx;

static ERRctx *dlg_list;
static int dlg_head = -1;
static int dlg_alloced = 0;

static ERRctx *formatter_list;
static int formatter_head = -1;
static int formatter_alloced = 0;

static ERRctx *ctx_list;
static int ctx_head = -1;
static int ctx_alloced = 0;

static FILE *ERR_stream;

static int ERR_split_string(char *str, int start, char *buf);
static char * ERRformat_errors(const char *package, int mode);

void
ERRinit(void)
{
   const char * file_name = getenv("XP_ERROR_LOG");

   if (file_name && file_name[0] != 0) {
      FILE *fp = fopen(file_name, "w");
      if (fp) ERRset_error_stream(fp);
   }
   ERRinit_reserve_memory();
}

void
ERRset_error_stream(FILE *fp)
{
   if (fp == NULL) ERR_stream = stderr;
   else ERR_stream = fp;
}

void
ERRpush_error_dlg(
   int (*func) (void *, char *, int),
   const char *str,
   void *data )
{
   dlg_head++;
   if (dlg_head >= dlg_alloced) {
      dlg_alloced += ERR_CTX_CHUNK_SIZE;
      if (dlg_alloced == ERR_CTX_CHUNK_SIZE) {
	 ALLOCN(dlg_list,ERRctx,dlg_alloced,"can't alloc ERRCTX buffer");
      }
      else {
	 REALLOC(dlg_list,ERRctx,dlg_alloced,
		 "can't realloc ERRCTX buffer");
      }
   }
   dlg_list[dlg_head].func = func;
   if (str == NULL) {
      *dlg_list[dlg_head].str = '\0';
   }
   else {
      /* copy string, up to max size that will fit */
      register int i = 0;
      while (str[i] != 0 &&
	     i < sizeof(dlg_list[dlg_head].str) - 1) {
	 dlg_list[dlg_head].str[i] = str[i];
	 i++;
      }
      dlg_list[dlg_head].str[i] = 0;
   }
   dlg_list[dlg_head].datap = data;
}

void
ERRpop_error_dlg(
   int (*func) (void *, char *, int) )
{
   int i, j;
   if (dlg_head == -1) {
      fprintf(stderr,"ERRpop_error_dlg pop off the top of error stack!\n");
      return;
   }
   /*
    * Find the guy that is being popped.  the push's and pop's are not
    * always simply nested (unfortunately...)
    */
   for (i = dlg_head; i >= 0; i--) {
      if (dlg_list[i].func == func) {
	 break;
      }
   }
   if (i == -1) {
      fprintf(stderr,"ERRpop_error_dlg mismatch from push!\n");
   }
   else {
      /*
       * Skrunch up the list
       */
      for (j = i; j < dlg_head; j++) {
	 dlg_list[j] = dlg_list[j+1];
      }
      dlg_head--;
   }
}

void
ERRpush_error_formatter(
   int (*func) (void *, char *, int),
   const char *str,
   void *data )
{
   formatter_head++;
   if (formatter_head >= formatter_alloced) {
      formatter_alloced += ERR_CTX_CHUNK_SIZE;
      if (formatter_alloced == ERR_CTX_CHUNK_SIZE) {
	 ALLOCN(formatter_list,ERRctx,formatter_alloced,"can't alloc ERRCTX buffer");
      }
      else {
	 REALLOC(formatter_list,ERRctx,formatter_alloced,
		 "can't realloc ERRCTX buffer");
      }
   }
   formatter_list[formatter_head].func = func;
   if (str == NULL) {
      *formatter_list[formatter_head].str = '\0';
   }
   else {
      /* copy string, up to max size that will fit */
      register int i = 0;
      while (str[i] != 0 &&
	     i < sizeof(formatter_list[formatter_head].str) - 1) {
	 formatter_list[formatter_head].str[i] = str[i];
	 i++;
      }
      formatter_list[formatter_head].str[i] = 0;
   }
   formatter_list[formatter_head].datap = data;
}

void
ERRpop_error_formatter(
   int (*func) (void *, char *, int) )
{
   if (formatter_head == -1) {
      fprintf(stderr,"ERRpop_error_formatter pop off the top of error stack!\n");
      return;
   }
   if (formatter_list[formatter_head].func != func) {
      fprintf(stderr,"ERRpop_error_formatter mismatch from push!\n");
   }
   formatter_head--;
}

void
ERRpush_error_ctx(
   int (*func) (void *, char *, int),
   const char *str,
   void *data )
{
   ctx_head++;
   if (ctx_head >= ctx_alloced) {
      ctx_alloced += ERR_CTX_CHUNK_SIZE;
      if (ctx_alloced == ERR_CTX_CHUNK_SIZE) {
	 ALLOCN(ctx_list,ERRctx,ctx_alloced,"can't alloc ERRCTX buffer");
      }
      else {
	 REALLOC(ctx_list,ERRctx,ctx_alloced,
		 "can't realloc ERRCTX buffer");
      }
   }
   ctx_list[ctx_head].func = func;
   if (str == NULL) {
      *ctx_list[ctx_head].str = '\0';
   }
   else {
      /* copy string, up to max size that will fit */
      register int i = 0;
      while (str[i] != '\0' &&
	     i < sizeof(ctx_list[ctx_head].str) - 1) {
	 ctx_list[ctx_head].str[i] = str[i];
	 i++;
      }
      /*
       * Null terminate this sucker
       */
      ctx_list[ctx_head].str[i] = 0;
   }
   ctx_list[ctx_head].datap = data;
}

void
ERRpop_error_ctx(
   int (*func) (void *, char *, int),
   void *data )
{
   if (ctx_head == -1) {
      fprintf(stderr,"ERRpop_error_ctx pop off the top of error stack!\n");
      return;
   }
   if (ctx_list[ctx_head].func != func || ctx_list[ctx_head].datap != data) {
      fprintf(stderr,"ERRpop_error_ctx mismatch from push!\n");
   }
   ctx_head--;
}

int ERR_error_mask = (ERR_ERROR | ERR_DIALOG | ERR_WARNING | ERR_INFO 
			| ERR_INTERNAL);

void
ERRset_error_mask(int mask)
{
   ERR_error_mask = mask;
}

/* A dummy function for debugging purposes */
int
ERR_stop_here(void)
{
   return 0;
}

/*
 * This calls the installed error handler, if there is one.
 * Otherwise it just prints it out on the current error stream.
 */
void
ERRhandle_formatted_error(int mode, char *formatted_buf)
{
   static int handling_error = 0;
   /*
    * If no dialog handler is installed, just send the messages to the
    * print routine.
    *
    * If we are batching this error message, don't do anything right now.
    */
   if ((mode & ERR_error_mask) && !ERRerror_batched) {
      if (dlg_head == -1 || dlg_list[dlg_head].func == NULL) {
	 ERRprint(formatted_buf);
      }
      else {
	 if (handling_error) {
	    /* Prevent infinite recursion!
	     * Don't call the error handler if we're already within it.
	     * This would indicate a faulty error handler, but we still
	     * want to protect against that.
	     * Just punt, and print the error out.
	     */
	    ERRprint("WARNING: preventing recursion in error handler!\n");
	    ERRprint(formatted_buf);
	    ERRerror_index = 0;
	    return;
	 }
	 handling_error = 1;
	 (*dlg_list[dlg_head].func)(dlg_list[dlg_head].datap,
				    formatted_buf,0);
	 ERRerror_index = 0;
	 handling_error = 0;
      }
      ERR_stop_here();
   }
}

static void
ERRput_error_on_stack(int mode, const char *buf)
{
   /* Flush out any previously queued error */
   if (mode & (ERR_FLUSH | ERR_DIALOG)) ERRerror_index = 0;

   if (ERRerror_index == ERR_MAX_ERRORS - 1) {
      strcpy(&ERRstack[ERRerror_index][0],"too many errors...");
      ERRerror_index++;
   }
   else if (ERRerror_index < ERR_MAX_ERRORS) {
      /* Copy the error into the buffer */
      strncpy(&ERRstack[ERRerror_index][0],buf,ERR_MAX_SIZE-4);
      ERRstack[ERRerror_index][ERR_MAX_SIZE-4] = '.';
      ERRstack[ERRerror_index][ERR_MAX_SIZE-3] = '.';
      ERRstack[ERRerror_index][ERR_MAX_SIZE-2] = '.';
      ERRstack[ERRerror_index][ERR_MAX_SIZE-1] = '\0';
      ERRerror_index++;
   }
}

void
ERRhandle_error(const char *package, int mode, const char *buf)
{
   char *formatted_buf;

   ERRput_error_on_stack(mode, buf);
   /*
    *  If we are batching errors, just leave them on the stack
    */
   if (ERRerror_batched == 0) {
      formatted_buf = ERRformat_errors(package,mode);

      ERRhandle_formatted_error(mode, formatted_buf);
      FREE(formatted_buf);
   }
}

void
ERR_batch_end(const char *package, int mode)
{
   if (ERRerror_batched < 0) {
      ERRerror_batched = 0;
      ERRprint("mismatching ERRbatch_end calls");
   }
   /*
    * If we are no longer batched and we have some errors, dump them out
    * here
    */
   else if (!ERRerror_batched && ERRerror_index) {
      char *formatted_buf = ERRformat_errors(package,mode);

      ERRhandle_formatted_error(mode, formatted_buf);
      FREE(formatted_buf);
   }
}

/* Dump all errors to a formatted buffer.
 * Returns the allocated buffer (null-terminated), which the caller
 * should free.
 */
static char *
ERRformat_errors(const char *package,int mode)
{
   char buf[256];
   char *err_buf;
   char *bufp;
   int i;

   /* No errors... */
   if (!ERRerror_index) {
      ALLOCN(err_buf, char, 1, "Can't alloc dummy err buf");
      err_buf[0] = '\0';
      return err_buf;
   }

   /* Allocate enough space for error strings plus some extra verbiage */
   ALLOCN(err_buf, char, sizeof(ERRstack) * 2 + 128,
	  "Can't allocate err buf");
   bufp = err_buf;

   /*
    * Using the return value of sprintf does not work on traditional 
    * berkeley systems... they return bufp, not # of characters 
    */
   if (!(mode & ERR_NO_HEADER)) {
      sprintf(bufp,"--- %s: ", 
	      (mode & ERR_WARNING ? "Warning from" : 
		 ((mode & ERR_INFO ? "Information from" : 
		   (mode & ERR_INTERNAL ? "Internal error from" :
		    "Error detected in")))));
      bufp += strlen(bufp);

      if (ctx_head >= 0) {
	 if (ctx_list[ctx_head].func != NULL) {
	    if ((*ctx_list[ctx_head].func)(ctx_list[ctx_head].datap,
					   buf,sizeof(buf)) == 1) {
	       sprintf(bufp,"%s, ",buf);
	       bufp += strlen(bufp);
	    }
	 }
	 else {
	    sprintf(bufp,"%s, ",ctx_list[ctx_head].str);
	    bufp += strlen(bufp);
	 }
      }

      sprintf(bufp,"module: %s ---\n",package);
      bufp += strlen(bufp);
   }
   if (ERRerror_index == 1) {
      bufp += ERR_split_string(&ERRstack[0][0],0, bufp);
   }
   else {
      for (i = 0; i < ERRerror_index; i++) {
	 sprintf(bufp,"%2d-> ",i);
	 bufp += strlen(bufp);
	 bufp += ERR_split_string(&ERRstack[i][0],5, bufp);
      }
   }
   if (!(mode & ERR_NO_HEADER)) {
      sprintf(bufp,"--------------------------------------------------\n");
      bufp += strlen(bufp);
   }
   *bufp = '\0';
   ERRerror_index = 0;
   return err_buf;
}

/*
 * This just dumps any errors to the error stream, bypassing
 * any dialog or error handler.
 */
void
ERRdump(const char *package)
{
   char *buf;

   buf = ERRformat_errors(package,ERR_ORIG);
   ERRprint(buf);
   FREE(buf);
}

/*
 * Print an already-formatted error message in buf onto the current error
 * stream.
 */
void
ERRprint(const char *buf)
{
   if (ERR_stream == NULL) ERR_stream = stderr;
   fputs(buf, ERR_stream);
}

#define LINE_LEN 79		/* output line length */

/*
 * Inputs: str, a string to split at whitespace;
 *         start, number of spaces to add at start of each line
 *
 * Returns in buf the newly split strings (terminated with \n), and
 * function return value is total count of chars added to buf.
 * Make sure to allocate enough storage for buf!  A good (very)
 * conservative estimate is twice the string size, as long as start is
 * relatively small (less than LINE_LEN/3 or so).
 */
static int
ERR_split_string(char *str, int start, char *buf)
{
   char *c, *c1;
   int i;
   char *bufp;

   bufp = buf;

   while (strlen(str)+start >= (unsigned)LINE_LEN) {

      /* scan forwards looking for first embedded CR */
      for (c1 = str;
           *c1 != '\n' && c1 != &str[LINE_LEN-start];
           ++c1)
         ;

      /* scan backwards looking for first good place to break line */
      for (c = &str[LINE_LEN-start];
           *c != '\n' && *c != '\t' && *c != ' ' && c != str;
           --c)
         ;

      /* If the caller has embedded a CR in the string,
       * go ahead and use it.  CFS PR 24355.
       */
      if( c1 < c ) c = c1;

      /* hit start of string */
      if (c == str) break;

      *c = '\0';
      while ((*bufp++ = *str++) != '\0')
	 ;			/* copy string, increment both ptrs */
      *(bufp-1) = '\n';
      *bufp = '\0';		/* make sure it's null-terminated */

      for (i = 0; i < start; i++)
	 *bufp++ = ' ';		/* leading spaces for next line */
   }
   /* copy remaining string part, including final null */
   while (*str != '\0')
      *bufp++ = *str++;
   if (bufp[-1] != '\n')	/* terminate with only 1 newline */
      *bufp++ = '\n';
   *bufp++ = '\0';
   return bufp - buf - 1;
}

void
ERRverror_v(const char *package, int mode,
	    const char *format, va_list ap)
{
   char buf[2048];

   /* If someone is squashing errors, ignore them... */
   if (ERRerror_squashed && !ERRerror_squash_override) return;

   vsprintf(buf, format, ap);
   ERRhandle_error(package, mode, buf);
}

void
ERRverror(const char *package, int mode,
	  const char *format, ...)
{
   va_list ap;
   va_start(ap, format);
   ERRverror_v(package, mode, format, ap);
   va_end(ap);
}


/*
 * Cannot accept "float" arguments
 */
void
ERRerror(const char *package, int count, int mode, const char *format, ...)
{
   va_list ap;
   char buf[2048];
   char *arg1, *arg2, *arg3, *arg4, *arg5;

   /* If someone is squashing errors, ignore them... */
   if (ERRerror_squashed && !ERRerror_squash_override) return;

   va_start(ap,format);
   switch (count) {
      case 0:
	 strcpy(buf,format);
	 break;
      case 1:
	 arg1 = va_arg(ap, char *);
	 sprintf(buf,format,arg1);
	 break;
      case 2:
	 arg1 = va_arg(ap, char *);
	 arg2 = va_arg(ap, char *);
	 sprintf(buf,format,arg1,arg2);
	 break;
      case 3:
	 arg1 = va_arg(ap, char *);
	 arg2 = va_arg(ap, char *);
	 arg3 = va_arg(ap, char *);
	 sprintf(buf,format,arg1,arg2,arg3);
	 break;
      case 4:
	 arg1 = va_arg(ap, char *);
	 arg2 = va_arg(ap, char *);
	 arg3 = va_arg(ap, char *);
	 arg4 = va_arg(ap, char *);
	 sprintf(buf,format,arg1,arg2,arg3,arg4);
	 break;
      case 5:
	 arg1 = va_arg(ap, char *);
	 arg2 = va_arg(ap, char *);
	 arg3 = va_arg(ap, char *);
	 arg4 = va_arg(ap, char *);
	 arg5 = va_arg(ap, char *);
	 sprintf(buf,format,arg1,arg2,arg3,arg4,arg5);
	 break;
      default:
	 fprintf(stderr,"Warning: error package has maximum of 5 args!\n");
	 break;
   }
   va_end(ap);
   ERRhandle_error(package, mode, buf);
}

/*
 * the following is to allow clients to request a yes/no dialog answer
 * from the UI without calling the UI directly (and thus having a direct
 * dependency on it)
 */
static int (*yesno_dialog_func)(const char *, const char *, ...) = NULL;

void
ERRset_yesno_dialog_handler(int (*func)(const char *, const char *, ...))
{
   yesno_dialog_func = func;
}

int
ERRsync_yesno_dialog_v(int default_answer, const char *title, const char *format, 
			va_list ap)
{
   char buf[2048];

   vsprintf(buf, format, ap);

   if (yesno_dialog_func)
      return((*yesno_dialog_func)(title, buf));
   else
      return(default_answer);
}

int
ERRsync_yesno_dialog(int default_answer, const char *title, const char *format, ...)
{
   int status;

   va_list ap;
   va_start(ap, format);
   status = ERRsync_yesno_dialog_v(default_answer, title, format, ap);
   va_end(ap);

   return(status);
}

/*----------------------------------------------------------------------*/

static int (*outofmem_dialog_func)(const char *, ...) = NULL;

void
ERRset_outofmem_dialog_handler(int (*func)(const char *, ...))
{
   outofmem_dialog_func = func;
}

int
ERRsync_outofmem_dialog_v(int default_answer, const char *format, 
                          va_list ap)
{
   char buf[2048];

   vsprintf(buf, format, ap);

   if (outofmem_dialog_func)
      return((*outofmem_dialog_func)(buf));
   else
      return(default_answer);
}

int
ERRsync_outofmem_dialog(int default_answer, const char *format, ...)
{
   int status;

   va_list ap;
   va_start(ap, format);
   status = ERRsync_outofmem_dialog_v(default_answer, format, ap);
   va_end(ap);

   return(status);
}

/*----------------------------------------------------------------------*/

/*--------------------------------------------------------------------*/
/* 
 * We need to allocate a block of memory which
 * will be freed to display a warning pop-up 
 * when we have seriously exhausted memory.
 */

/* size of the rainy day memory block */

#define UTIL_SZ_RAINY_DAY 10000UL

static char *UTIL_rainy_day_mem;

void
ERRinit_reserve_memory(void)
{
   UTIL_rainy_day_mem = (char *) malloc ( UTIL_SZ_RAINY_DAY );
   if (UTIL_rainy_day_mem == NULL) {
      fprintf(stderr,"Cannot allocate reserve memory buffer");
      exit(-1);
   }
}

/* options are Save state and exit or Retry previous operation   */
/* need to check these are the right way around for Motif and NT */

#define Button_Exit  0
#define Button_Retry 1

void
ERRout_of_memory( const char *message )
{
   int answer;

   if (UTIL_rainy_day_mem != NULL) {
      free(UTIL_rainy_day_mem);
      UTIL_rainy_day_mem = NULL;
   }

   answer = ERRsync_outofmem_dialog( Button_Exit, message );

   if (answer == Button_Exit) {
      db_crash_handler(999);
      exit(-1);
   }

   UTIL_rainy_day_mem = (char *) malloc ( UTIL_SZ_RAINY_DAY );

   /* if malloc fails, ignore it      */
   /* don't start recursive dialogs ! */
}

/*----------------------------------------------------------------------*/
