/*
			Copyright (c) 1994 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/tool/image.c#1 $
*/
#include <stdio.h>
#include <string.h>
#include <sys/types.h>	/* time_t */

#include <avs/dll_in.h>
#include <avs/err.h>
#include <avs/f_utils.h>
#include <avs/util.h>

/* BMP defines */
#ifndef BI_RGB
#define BI_RGB	0
#endif
#ifndef BI_RLE8
#define BI_RLE8	1     
#endif
#ifndef BI_RLE4
#define BI_RLE4	2     
#endif
#ifndef BI_BITFIELDS
#define BI_BITFIELDS 3
#endif

#define ERR_RETURN(A) {ERRerror("write image", 0, ERR_ORIG, A); return(0);}

/* Write a 32-bit integer, big-endian. */
static int write_int_big( int x, FILE *fp )
{
   unsigned char buf[4];
   buf[0] = (x>>24) & 0xff;
   buf[1] = (x>>16) & 0xff;
   buf[2] = (x>> 8) & 0xff;
   buf[3] =  x      & 0xff;
   return fwrite(buf, 4, 1, fp);
}

/* Write a 32-bit integer, little-endian. */
static int write_int_little( int x, FILE *fp )
{
   unsigned char buf[4];
   buf[3] = (x>>24) & 0xff;
   buf[2] = (x>>16) & 0xff;
   buf[1] = (x>> 8) & 0xff;
   buf[0] =  x      & 0xff;
   return fwrite(buf, 4, 1, fp);
}

/* Write a 16-bit integer, little-endian. */
static int write_short_little( short x, FILE *fp )
{
   unsigned char buf[2];
   buf[1] = (x>> 8) & 0xff;
   buf[0] =  x      & 0xff;
   return fwrite(buf, 2, 1, fp);
}

int UTILwrite_image_AVSX(int *dims, void *img_data,
                         const char *filename, int flip)
{
   FILE *fp;
   unsigned char *idata_ptr;
   int i;
   int w, h;

   if (dims == NULL || img_data == NULL || filename == NULL) {
      ERR_RETURN("Invalid argument");
   }

   w = dims[0];
   h = dims[1];

   fp = (FILE *)FILEfopen(filename, SIO_W_BIN);
   if (fp == NULL) {
      ERR_RETURN("Could not open output file");
   }

   /* Write the dimensions of image. This should be 8 bytes, 4 each
      for the x and y size.
   */
   if (write_int_big(w, fp) != 1 ||
       write_int_big(h, fp) != 1 ) {
      fclose(fp);
      ERR_RETURN("Could not write image header");
   }

   /* See if the user has requested that we flip the data. */
   if (!flip) {
      /* Write the image data. The size is x * y * 4 (veclen = 4). */
      if (fwrite(img_data, w*h*4, 1, fp) != 1) {
         fclose(fp);
         ERR_RETURN("Could not write image data");
      }
   }
   else {
      /* We need to flip the buffer as the user requested. So
         scan the buffer from the bottom to the top a line at a
         time. 
      */
      idata_ptr = img_data;
      idata_ptr += w * (h - 1) * 4;	/* point at start of last row */
      for (i=0; i<dims[1]; i++) {
         if (fwrite(idata_ptr, w*4, 1, fp) != 1) {
            fclose(fp);
            ERR_RETURN("Could not write line of image data");
         }
         idata_ptr -= w * 4;
      }
   }
   fclose(fp);
   return(1);
}

int UTILwrite_image_PPM(int *dims, void *img_data,
                        const char *filename, int flip)
{
   FILE *fp;
   int i, j;
   int w, h;
   unsigned char *line_src, *line_dst;

   if (dims == NULL || img_data == NULL || filename == NULL) {
      ERR_RETURN("Invalid argument");
   }

   w = dims[0];
   h = dims[1];

   line_dst = malloc(w * 3);

   fp = (FILE *)FILEfopen(filename, SIO_W_BIN);
   if (fp == NULL) {
      ERR_RETURN("Could not open output file");
   }

   /* PPM header */
   fprintf( fp, "P6\n" );
   fprintf( fp, "%d %d\n", w, h );
   fprintf( fp, "255\n" );

   /* See if the user has requested that we flip the data. */
   if (!flip) {
      for (i=0; i<h; ++i) {
         line_src = (unsigned char *)img_data + (w * i * 4);
         for (j=0; j<w; ++j) {
            /* line_src[j*4] is A, discard */
            line_dst[j*3]   = line_src[j*4+1]; /* R */
            line_dst[j*3+1] = line_src[j*4+2]; /* G */
            line_dst[j*3+2] = line_src[j*4+3]; /* B */
         }
         /* Write a scan line. The size of each line is w * 3. */
         if (fwrite(line_dst, w * 3, 1, fp) != 1) {
            fclose(fp);
            free(line_dst);
            ERR_RETURN("Could not write image data");
         }
      }
   }
   else {
      /* We need to flip the buffer as the user requested. So
         scan the buffer from the bottom to the top a line at a
         time. 
      */
      for (i=h-1; i>=0; --i) {
         line_src = (unsigned char *)img_data + (w * i * 4);
         for (j=0; j<w; ++j) {
            /* line_src[j*4] is A, discard */
            line_dst[j*3]   = line_src[j*4+1]; /* R */
            line_dst[j*3+1] = line_src[j*4+2]; /* G */
            line_dst[j*3+2] = line_src[j*4+3]; /* B */
         }
         /* Write a scan line. The size of each line is w * 3 bytes. */
         if (fwrite(line_dst, w * 3, 1, fp) != 1) {
            free(line_dst);
            fclose(fp);
            ERR_RETURN("Could not write image data");
         }
      }
   }

   free(line_dst);
   fclose(fp);
   return(1);
}

int UTILwrite_image_BMP(int *dims, void *img_data,
                        const char *filename, int flip)
{
   FILE *fp;
   int i, j;
   int w, h;
   int bytes_per_line_raw, bytes_per_line_pad, pad;
   int img_size, file_size;
   unsigned char *line_src, *line_dst;

   if (dims == NULL || img_data == NULL || filename == NULL) {
      ERR_RETURN("Invalid argument");
   }

   w = dims[0];
   h = dims[1];

   bytes_per_line_raw = w * 3;
   bytes_per_line_pad = bytes_per_line_raw + 3 & ~3;
   pad = bytes_per_line_pad - bytes_per_line_raw;

   line_dst = malloc(bytes_per_line_pad);

   fp = (FILE *)FILEfopen(filename, SIO_W_BIN);
   if (fp == NULL) {
      ERR_RETURN("Could not open output file");
   }

   /* BITMAPFILEHEADER */

   fputc( 'B', fp );
   fputc( 'M', fp );

   img_size = bytes_per_line_pad * h;
   file_size = 14 +	/* size of bitmap file header */
               40 +	/* size of bitmap info header */
	        0 +	/* size of colormap */
	       img_size;	/* size of image data */

   write_int_little( file_size, fp );
   write_int_little( 0, fp );		/* reserved1, reserved2 */
   write_int_little( 14 + 40, fp );	/* offset from start of file to data */

   /* BITMAPINFOHEADER */

   write_int_little( 40, fp );	/* biSize (sizeof(BITMAPINFOHEADER)) */
   write_int_little( w, fp );	/* biWidth */
   write_int_little( h, fp );	/* biHeight */
   write_short_little(  1, fp );	/* biPlanes */
   write_short_little( 24, fp );	/* biBitCount */
   write_int_little( BI_RGB, fp );	/* biCompression:  BI_RGB == uncompr */
   write_int_little( img_size, fp );	/* biSizeImage */
   write_int_little( 0, fp );	/* biXPelsPerMeter */
   write_int_little( 0, fp );	/* biYPelsPerMeter */
   write_int_little( 0, fp );	/* biClrUsed */
   write_int_little( 0, fp );	/* biClrImportant */

   /* BMP data */

   /* BMP normally is bottom up.  A negative h means top down. */

   /* See if the user has requested that we flip the data. */
   if (flip) {
      for (i = 0; i < h; ++i) {
         line_src = (unsigned char *)img_data + (w * i * 4);
         for (j = 0; j < w; ++j) {
            /* line_src[j*4] is A, discard */
            line_dst[j*3]   = line_src[j*4+3]; /* B */
            line_dst[j*3+1] = line_src[j*4+2]; /* G */
            line_dst[j*3+2] = line_src[j*4+1]; /* R */
         }
         for (j = 0; j < pad; ++j) {
            line_dst[w*3+j] = 0;
         }

         /* Write a scan line. The size of each line is w * 3 + pad. */
         if (fwrite(line_dst, bytes_per_line_pad, 1, fp) != 1) {
            fclose(fp);
            free(line_dst);
            ERR_RETURN("Could not write image data");
         }
      }
   }
   else {
      /* We need to flip the buffer as the user requested. So
         scan the buffer from the bottom to the top a line at a
         time. 
      */
      for (i = h-1; i >= 0; --i) {
         line_src = (unsigned char *)img_data + (w * i * 4);
         for (j=0; j < w; ++j) {
            /* line_src[j*4] is A, discard */
            line_dst[j*3]   = line_src[j*4+3]; /* B */
            line_dst[j*3+1] = line_src[j*4+2]; /* G */
            line_dst[j*3+2] = line_src[j*4+1]; /* R */
         }
         for (j = 0; j < pad; ++j) {
            line_dst[w*3+j] = 0;
         }

         /* Write a scan line. The size of each line is w * 3 + pad. */
         if (fwrite(line_dst, bytes_per_line_pad, 1, fp) != 1) {
            fclose(fp);
            free(line_dst);
            ERR_RETURN("Could not write image data");
         }
      }
   }

   free(line_dst);
   fclose(fp);
   return(1);
}


int UTILwrite_image(int *dims, void *img_data, const char *filename, int flip)
{
   const char *ext;

   if (filename == NULL) {
      ERR_RETURN("Invalid argument");
   }

   ext = FILEget_file_extension(filename);
   if( ext[0] == '.' ) ext += 1;	/* skip past dot */

   if (strcmp(ext, "BMP") == 0 || strcmp(ext, "bmp") == 0)
      return UTILwrite_image_BMP(dims, img_data, filename, flip);
   else if(strcmp(ext, "PPM") == 0 || strcmp(ext, "ppm") == 0)
      return UTILwrite_image_PPM(dims, img_data, filename, flip);
   else
      return UTILwrite_image_AVSX(dims, img_data, filename, flip);
}

/*******************************************************************/

#undef ERR_RETURN
#define ERR_RETURN(A) {ERRerror("read image", 0, ERR_ORIG, A); return(0);}

/* Read a 32-bit integer, big-endian. */
static int read_int_big( int *x, FILE *fp )
{
   unsigned char buf[4];
   int ret = fread(buf, 4, 1, fp);
   if( ret != 1 ) return ret;
   *x =  (buf[0]<<24) | (buf[1]<<16) | (buf[2]<<8) | buf[3];
   return ret;
}

/* Read a 32-bit integer, little-endian. */
static int read_int_little( int *x, FILE *fp )
{
   unsigned char buf[4];
   int ret = fread(buf, 4, 1, fp);
   if( ret != 1 ) return ret;
   *x =  (buf[3]<<24) | (buf[2]<<16) | (buf[1]<<8) | buf[0];
   return ret;
}

/* Read a 16-bit integer, little-endian. */
static int read_short_little( short *x, FILE *fp )
{
   unsigned char buf[2];
   int ret = fread(buf, 2, 1, fp);
   if( ret != 1 ) return ret;
   *x = (buf[1]<<8) | buf[0];
   return ret;
}

int UTILread_image_AVSX(int *dims, void **img_data_ptr,
                        const char *filename, int flip)
{
   FILE *fp;
   unsigned char *img_data;
   /* int i; */

   int w, h;

   if (dims == NULL || img_data_ptr == NULL || filename == NULL) {
      ERR_RETURN("Invalid argument");
   }

   fp = (FILE *)FILEfopen(filename, SIO_R_BIN);
   if (fp == NULL) {
      ERR_RETURN("Could not open output file");
   }

   if (read_int_big(&w, fp) != 1 ||
       read_int_big(&h, fp) != 1 ) {
      fclose(fp);
      ERR_RETURN("Could not read image header");
   }

   img_data = malloc(w * h * 4);
   if (!img_data) {
      fclose(fp);
      ERR_RETURN("Could not allocate image data buffer");
   }

   if (fread(img_data, w * h * 4, 1, fp) != 1) {
      fclose(fp);
      ERR_RETURN("Could not read image data");
   }

   fclose(fp);

   dims[0] = w;
   dims[1] = h;
   /* Caller frees image data */
   *img_data_ptr = img_data;

   return(1);
}

int UTILread_image_PPM(int *dims, void **img_data_ptr,
                       const char *filename, int flip)
{
   FILE *fp;
   unsigned char *img_data;
   unsigned char *line_src, *line_dst;
   char buffer[64];
   int i, j;
   int w, h;
   int src_w, gray = 0;

   if (dims == NULL || img_data_ptr == NULL || filename == NULL) {
      ERR_RETURN("Invalid argument");
   }

   fp = (FILE *)FILEfopen(filename, SIO_R_BIN);
   if (fp == NULL) {
      ERR_RETURN("Could not open output file");
   }

   /* magic number - should be "P6" (PPM) or "P5" (PGM)
    *
    * The doc I've seen seems to imply that the file could start with
    * a comment, but that would mess up the magic number so I'm
    * not allowing it here.
    */
   fgets( buffer, sizeof(buffer), fp );
   if( buffer[0] != 'P' ) {
      ERR_RETURN("Bad magic number - not PPM or PGM file");
   }
   if     ( buffer[1] == '5' ) gray = 1;
   else if( buffer[1] == '6' ) gray = 0;
   else {
      ERR_RETURN("Bad magic number - not PPM or PGM file");
   }

   /* width, heigth */
   while( fgets( buffer, sizeof(buffer), fp ) != NULL ) {
     if( buffer[0] != '#' ) break;	/* break unless start of comment */
   }
   sscanf( buffer, "%d %d", &w, &h );

   /* maxval - assuming 255 */
   while( fgets( buffer, sizeof(buffer), fp ) != NULL ) {
     if( buffer[0] != '#' ) break;	/* break unless start of comment */
   }

   /* No comments allowed *after* maxval */

   if( gray ) src_w = w;	/* Gray map, 1 byte per pixel */
   else       src_w = w * 3;	/* Color map, 3 bytes per pixel */

   line_src = malloc(src_w);
   img_data = malloc(w * h * 4);
   if (line_src == NULL || img_data == NULL) {
      fclose(fp);
      ERR_RETURN("Could not allocate image data buffer");
   }

   for(i = 0; i < h; ++i ) {
      line_dst = img_data + (w * i * 4);
      if (fread(line_src, src_w, 1, fp) == 1) {
         if( gray ) {
            for(j = 0; j < w; ++j ) {
               line_dst[j*4]   = 255; /* A */
               line_dst[j*4+1] = line_src[j]; /* R */
               line_dst[j*4+2] = line_src[j]; /* G */
               line_dst[j*4+3] = line_src[j]; /* B */
            }
         }
         else {
            for(j = 0; j < w; ++j ) {
               line_dst[j*4]   = 255; /* A */
               line_dst[j*4+1] = line_src[j*3];   /* R */
               line_dst[j*4+2] = line_src[j*3+1]; /* G */
               line_dst[j*4+3] = line_src[j*3+2]; /* B */
            }
         }
      }
      /* else error while reading image data */
   }

   free(line_src);
   fclose(fp);

   dims[0] = w;
   dims[1] = h;
   /* Caller frees image data */
   *img_data_ptr = img_data;

   return(1);
}

int UTILread_image_BMP(int *dims, void **img_data_ptr,
                       const char *filename, int flip)
{
   FILE *fp;
   unsigned char *img_data, *img_buffer, *cmap;
   unsigned char *line_src, *line_dst;
   short dummy_s, planes, bit_count;
   int dummy_i, i, j;
   int w, h, info_size, offset, compression;
   int bytes_per_line_raw, bytes_per_line_pad;

   if (dims == NULL || img_data_ptr == NULL || filename == NULL) {
      ERR_RETURN("Invalid argument");
   }

   fp = (FILE *)FILEfopen(filename, SIO_R_BIN);
   if (fp == NULL) {
      ERR_RETURN("Could not open output file");
   }

   /* BITMAPFILEHEADER */

   read_short_little( &dummy_s, fp );	/* "BM" */
   if ((dummy_s & 0xFF) != 'B' ||
       ((dummy_s >> 8) & 0xFF) != 'M') {
      ERR_RETURN("Bad magic number - not BMP");
   }
   read_int_little( &dummy_i, fp );	/* file size */
   read_int_little( &dummy_i, fp );	/* reserved1, reserved2 */
   read_int_little( &offset, fp );	/* offset from start of file to data */

   /* BITMAPINFOHEADER */

   read_int_little( &info_size, fp );	/* biSize (header size) */

   /* biSize indirectly tells us what kind of BMP file we have.
    * biSize == 40 means BITMAPINFOHEADER (Win31), but there is
    * also BITMAPV4HEADER and BITMAPV5HEADER.
    */

   if (info_size != 40) {
      fclose(fp);
      ERR_RETURN("Cannot read this version of BMP file");
   }

   offset -= 14;		/* size of bitmap file header */
   offset -= info_size;		/* size of bitmap info header */

   read_int_little( &w, fp );	/* biWidth (image width) */
   read_int_little( &h, fp );	/* biHeight (image height) */

   if (w <= 0 || h == 0) {  /* negative h means flip */
      fclose(fp);
      ERR_RETURN("Invalid BMP width or height");
   }

   read_short_little( &planes, fp );	/* biPlanes (assume 1) */
   read_short_little( &bit_count, fp );	/* biBitCount */
   read_int_little( &compression, fp );	/* biCompression */

   if (planes != 1 || (compression != BI_RGB && 
                       compression != BI_RLE8 && compression != BI_RLE4)) {
      fclose(fp);
      ERR_RETURN("Cannot read this kind of BMP file");
   }
   if (bit_count != 24 && bit_count != 8 && 
       bit_count != 4  && bit_count != 1) {
      fclose(fp);
      ERR_RETURN("Cannot read this kind of BMP file");
   }

   read_int_little( &dummy_i, fp );	/* biSizeImage (image size in bytes) */
   read_int_little( &dummy_i, fp );	/* biXPelsPerMeter (don't care) */
   read_int_little( &dummy_i, fp );	/* biXPelsPerMeter (don't care) */
   read_int_little( &dummy_i, fp );	/* biClrUsed (assume 0) */
   read_int_little( &dummy_i, fp );	/* biClrImportant (assume 0) */

   if( bit_count == 24 ) {
      /* TrueColor means no color map */
      cmap = NULL;
      bytes_per_line_raw = w * 3;
   }
   else {
      int ncolors = (1 << bit_count);
      cmap = malloc( ncolors * 4 );
      if (cmap == NULL) {
         fclose(fp);
         ERR_RETURN("Could not allocate colormap buffer");
      }
      /* Read the color map */
      fread(cmap, ncolors*4, 1, fp);

      offset -= (ncolors*4);	/* size of color map */

      if( bit_count == 8 )
         bytes_per_line_raw = w;
      else if( bit_count == 4 )
         bytes_per_line_raw = (w + 1)/2;
      else if( bit_count == 1 )
         bytes_per_line_raw = (w + 7)/8;
   }

   if( offset > 0 ) {
      /* Its legal (although rare) for there to be empty space between
       * the end of the headers and the start of the data section.
       */
      for( i = 0; i < offset; ++i ) getc(fp);
   }

   bytes_per_line_pad = bytes_per_line_raw + 3 & ~3;

   /* biSizeImage == bytes_per_line_pad * h, although some true color
    * BMPs get written with biSizeImage == 0.
    */

   /* BMP data */

   line_src = img_buffer = NULL;

   if (compression == BI_RGB) {
      /* Use this when uncompressed */
      if ((line_src = malloc(bytes_per_line_pad)) == NULL) {
         if (cmap) free(cmap);
         fclose(fp);
         ERR_RETURN("Could not allocate scan line buffer");
      }
   }
   else {
      /* Use this with RLE compression */
      if ((img_buffer = malloc( w * h )) == NULL) {
         if (cmap) free(cmap);
         fclose(fp);
         ERR_RETURN("Could not allocate image buffer");
      }
   }
   /* This is the ARGB image that is returned to the caller. */
   if ((img_data = malloc(w * h * 4)) == NULL) {
      if (line_src) free(line_src);
      if (cmap) free(cmap);
      fclose(fp);
      ERR_RETURN("Could not allocate ARGB image buffer");
   }

   /* BMP normally is bottom up.  A negative h means top down. */

   if( bit_count == 24 ) {
      for (i = h-1; i >= 0; --i) {
         line_dst = img_data + (w * i * 4);
         if (fread(line_src, bytes_per_line_pad, 1, fp) == 1) {
            for (j = 0; j < w; ++j ) {
               line_dst[j*4]   = 255; /* A */
               line_dst[j*4+1] = line_src[j*3+2]; /* R */
               line_dst[j*4+2] = line_src[j*3+1]; /* G */
               line_dst[j*4+3] = line_src[j*3];   /* B */
            }
         }
         /* else error while reading image data */
      }
   }
   else if( bit_count == 8 && compression == BI_RGB ) {
      unsigned char src_byte;
      for (i = h-1; i >= 0; --i) {
         line_dst = img_data + (w * i * 4);
         if (fread(line_src, bytes_per_line_pad, 1, fp) == 1) {
            for (j = 0; j < w; ++j ) {
               src_byte = line_src[j];
               line_dst[j*4]   = 255; /* A */
               line_dst[j*4+1] = cmap[(4*src_byte)+2]; /* R */
               line_dst[j*4+2] = cmap[(4*src_byte)+1]; /* G */
               line_dst[j*4+3] = cmap[(4*src_byte)+0]; /* B */
            }
         }
         /* else error while reading image data */
      }
   }
   else if( bit_count == 8 && compression == BI_RLE8 ) {
      int c, v, k;
      unsigned char *end = img_buffer + ( w * h );
      unsigned char *cur = img_buffer;

      memset( img_buffer, 0, w * h );
      i = j = 0;

      while( cur < end && j < h) {
         c = getc( fp );
         if( c == 0 ) {	/* next byte is a special code */
            c = getc( fp );
            if( c == 0 ) {	/* end of line */
               ++j;		/* up to next scan line */
               i = 0;
               cur = img_buffer + (j * w );
               continue;	/* no chance of wraparound */
            }
            else if( c == 1 ) {	/* end of bitmap */
               cur = end;
               break;
            }
            else if( c == 2 ) {	/* 2 byte delta */
               /* Vert delta goes UP when reading in standard (for BMPs)
                * bottom-up mode.  It would go DOWN if reading flipped.
                */
               c = getc( fp );	/* horiz delta */
               i += c;
               c = getc( fp );	/* vert delta */
               j += c;		/* UP */
               cur = img_buffer + i + (j * w);
            }
            else if ( c >=3 && c <= 255 ) {
               /* "Absolute mode" */
               for (k = 0; k < c; ++k, ++i, ++cur) {
                  *cur = getc( fp );
               }
               /* Discard odd byte */
               if( c%2 ) getc( fp );
            }
         }
         else { /* c != 0, c is a simple repeat count */
            v = getc( fp );
            for (k = 0; k < c; ++k, ++i, ++cur) {
               *cur = v;
            }
         }
         /* What if i overflows? .... */
         while( i > w ) {	/* Undo the effects of any wraparound */
            i -= w;
            ++j;
         }
      }
      /* Expansion into ARGB is done below */
   }
   else if( bit_count == 4 && compression == BI_RGB ) {
      unsigned char src_byte;
      for (i = h-1; i >= 0; --i) {
         line_dst = img_data + (w * i * 4);
         if (fread(line_src, bytes_per_line_pad, 1, fp) == 1) {
            for (j = 0; j < w; ++j ) {
               if (j%2 == 0)
                  src_byte = (line_src[j/2] >> 4) & 0x0F;
               else
                  src_byte = line_src[j/2]        & 0x0F;
               line_dst[j*4]   = 255; /* A */
               line_dst[j*4+1] = cmap[(4*src_byte)+2]; /* R */
               line_dst[j*4+2] = cmap[(4*src_byte)+1]; /* G */
               line_dst[j*4+3] = cmap[(4*src_byte)+0]; /* B */
            }
         }
         /* else error while reading image data */
      }
   }
   else if( bit_count == 4 && compression == BI_RLE4 ) {
      int c, v, k;
      unsigned char *end = img_buffer + ( w * h );
      unsigned char *cur = img_buffer;

      memset( img_buffer, 0, w * h );
      i = j = 0;

      while( cur < end && j < h ) {
         c = getc( fp );
         if( c == 0 ) {       /* next byte is a special code */
            c = getc( fp );
            if( c == 0 ) {   /* end of line */
               ++j;         /* up to next scan line */
               i = 0;
               cur = img_buffer + (j * w );
               continue;    /* no chance of wraparound */
            }
            else if( c == 1 ) {      /* end of bitmap */
               cur = end;
               break;
            }
            else if( c == 2 ) {      /* 2 byte delta */
               /* Vert delta goes UP when reading in standard (for BMPs)
                * bottom-up mode.  It would go DOWN if reading flipped.
                */
               c = getc( fp );      /* horiz delta */
               i += c;
               c = getc( fp );      /* vert delta */
               j += c;              /* UP */
               cur = img_buffer + i + (j * w);
            }
            else if ( c >=3 && c <= 255 ) {
               /* "Absolute mode" */
               for (k = 0; k < c; ++k, ++i, ++cur) {
                  if( k%2 == 0 ) {
                     v = getc( fp );
                     *cur = (v >> 4) & 0x0F;
                  }
                  else
                     *cur = v & 0x0F;
               }
               /* Discard odd byte */
               if( c%4 == 1 || c%4 == 2 ) getc( fp );
            }
         }
         else { /* c != 0, c is a simple repeat count */
            v = getc( fp );
            for (k = 0; k < c; ++k, ++i, ++cur) {
               if ( k%2 == 0 )
                  *cur = (v >> 4) & 0x0F;
               else
                  *cur = v & 0x0F;
            }
         }
         /* What if i overflows? .... */
         while( i > w ) {     /* Undo the effects of any wraparound */
            i -= w;
            ++j;
         }
      }
      /* Expansion into ARGB is done below */
   }
   else if( bit_count == 1 ) {
      unsigned char src_byte;
      unsigned char src_bit;
      for (i = h-1; i >= 0; --i) {
         line_dst = img_data + (w * i * 4);
         if (fread(line_src, bytes_per_line_pad, 1, fp) == 1) {
            for (j = 0; j < w; ++j ) {
               if (j%8 == 0)
                  src_byte = line_src[j/8];
               if (src_byte & 0x80) /* Test high bit */
                  src_bit = 1;
               else
                  src_bit = 0;
               src_byte <<= 1;
               line_dst[j*4]   = 255; /* A */
               line_dst[j*4+1] = cmap[(4*src_bit)+2]; /* R */
               line_dst[j*4+2] = cmap[(4*src_bit)+1]; /* G */
               line_dst[j*4+3] = cmap[(4*src_bit)+0]; /* B */
            }
         }
         /* else error while reading image data */
      }
   }
   else {
      if (line_src) free(line_src);
      if (cmap) free(cmap);
      fclose(fp);
      ERR_RETURN("Cannot read this kind of BMP file");
   }

   if( img_buffer ) {
      /* If we unpacked into a 8-bit buffer above, now perform the
       * color mapping into full ARGB and also flip to top-down.
       */
      unsigned char *cur = img_buffer;
      unsigned char src_byte;
      for (i = h-1; i >= 0; --i) {
         line_dst = img_data + (w * i * 4);
         for (j = 0; j < w; ++j ) {
            src_byte = *cur;
            line_dst[j*4]   = 255; /* A */
            line_dst[j*4+1] = cmap[(4*src_byte)+2]; /* R */
            line_dst[j*4+2] = cmap[(4*src_byte)+1]; /* G */
            line_dst[j*4+3] = cmap[(4*src_byte)+0]; /* B */
            ++cur;
         }
      }
      free( img_buffer );
   }

   if (line_src) free(line_src);
   if (cmap) free(cmap);
   fclose(fp);

   dims[0] = w;
   dims[1] = h;
   /* Caller frees image data */
   *img_data_ptr = img_data;

   return(1);
}


int UTILread_image(int *dims, void **img_data_ptr,
                   const char *filename, int flip)
{
   FILE *fp;
   int c0, c1;

   if (filename == NULL) {
      ERR_RETURN("Invalid argument");
   }

   fp = (FILE *)FILEfopen(filename, SIO_R_BIN);
   if (fp == NULL) {
      ERR_RETURN("Could not open output file");
   }

   c0 = fgetc( fp );
   c1 = fgetc( fp );
   fclose( fp );

   if( c0 == 'B' && c1 == 'M' )
      return UTILread_image_BMP(dims, img_data_ptr, filename, flip);
   else if( c0 == 'P' && (c1 == '5' || c1 == '6') )
      return UTILread_image_PPM(dims, img_data_ptr, filename, flip);
   else
      return UTILread_image_AVSX(dims, img_data_ptr, filename, flip);
}
