//			Copyright (c) 2003 by
//			Advanced Visual Systems Inc.
//			All Rights Reserved
//
//	This software comprises unpublished confidential information of
//	Advanced Visual Systems Inc. and may not be used, copied or made
//	available to anyone, except in accordance with the license
//	under which it is furnished.
//
//	This file is under Perforce control
//	$Id: //depot/express/fcs70/v/wizard.v#1 $

#define DICTIONARY

flibrary WIZARD <
    libdeps="MODS",
    dictionary="wizard.dct"
    >
{

library+buffered Wiz_Modules <
    use_src_file=0,
    link_files="-lmods"
   >
{

// mode ...
//  0 == file not found
//  1 == recognized
//  2 == generic text
//  3 == generic binary
//  4 == unknown
//  5 == multiple possibilities

group reader_templ {
    string+opt extensions[];	// extensions this reader applies to
    string+opt description;	// description of reader (for wizard UI)
    string+opt template;	// reader template
    string+opt inst_name;	// name to use for instanced reader
    string+opt fname_sub;	// name of input filename subobject
    string+opt fld_sub;		// name of output field parameter
    string+opt obj_sub;		// name of output field parameter
    string+opt uip_sub;		// name of UIpanel (so we can raise it)
    int mode;			// see above
};

//
// Module to add reader information to the reader database.
// Allows incremental addition of IAC readers and customer readers.
//
module register_reader {
    reader_templ+Iparam reader_table[];
    int+Iparam reset = 0;	// Clear out old information
    int+Oparam do = 0;	// Set after it runs once (help with deps).

    // Generally runs just once at instance time
    omethod+notify_val+notify_inst rr_update  = "WIZregister_reader_update";
};

//
// Module to process filenames and return information needed by the
// wizard.  It checks to see if the file exists and then looks the
// the fname extensions to determine file type.
//
module examine_file {
    string+Iparam filename;
    int+read+IPort2 trigger;

    // Instead make it based on an internally maintained database
    // of reader information ... either/or?
    //reader_templ+read+IPort2 reader_table[];	// not required

    int+Oparam    mode;
    string+Oparam description;
    int+Oparam    nreaders;
    reader_templ+Oparam readers[];

    omethod+notify_val+notify_inst ef_update  = "WIZexamine_file_update";
};

//
// When the filename extension does not uniquely determine a reader,
// we have the user pick from a list of reader descriptions.  So this
// modules picks out a reader based on matching the description string.
//
module examine_descr {
    string+Iparam in_description;

    // Instead make it based on an internally maintained database
    // of reader information ... either/or?
    //reader_templ+read+IPort2 reader_table[];	// not required

    int+Oparam    mode;
    string+Oparam description;
    int+Oparam    nreaders;
    reader_templ+Oparam readers[];

    // Yes, this uses the same update function as examine_file
    omethod+notify_val+notify_inst ed_update  = "WIZexamine_file_update";
};

//
// Actually instance a reader.  Why isn't this just a simple parse_v?
// So that we can do things like delete a previous reader with the
// same name, so we can make sure to bring the reader to the top of
// the module stack menu, and so we can search for a viewer and
// automatically connect to it.
//
module instance_reader {
    reader_templ+read+nonotify+req+IPort2 reader;
    string+read+nonotify+IPort2  wizard_name;
    group+read+nonotify+IPort2   *relative;
    int+read+nonotify+IPort2     active;
    int+read+notify+IPort2       trigger;

    string+read+nonotify+IPort2  before_commands;
    string+read+nonotify+IPort2  after_commands;

    omethod+notify_val ir_update  = "WIZinstance_reader_update";
};

module view_connect {
    // Parent macro ... assumed to have a subobj called "View"
    group+read+IPort2 *parent               => <-.<-;
    // Indicates top level of application.  We search for
    // subobjs called "Uviewer3D/Uviewer2D/Uviewer"
    group+read+nonotify+IPort2 *application => <-.<-.<-;

    omethod+notify_val+notify_inst vc_update  = "WIZview_connect_update";
};


// Produces a text description of the field,
// e.g., "Uniform 3D field with node data"
module examine_field {
    Mesh+IPort2+read+notify     &in_field;
    string+Oparam       description;

    omethod+notify_val+notify_inst ef_update = "WIZexamine_field_update";
};


// Takes the selection - a string probably chosen from a menu -
// and instances a bounding box module based on the selection and the type
// of the input field.  Then it connects the input of the new module to
// the field and finally connects the module's output renderable object
// to the viewer. Currently, the only valid input strings are "none",
// "bounds", "bounding box" and "bounding_box" (there's no difference
// between the last three).
// This module assumes the presence of the DV wizard and/or
// the DI wizard, and uses them to locate the input field.
//
module viz_bounding_select {
    string+IPort2+read+notify   user_selection;
    omethod+notify_val update = "WIZviz_bounding";
};

// Takes the selection - a string probably chosen from a menu -
// and instances a axis module based on the selection and the type of
// the input field.  Then it connects the input of the new module to
// the field, and finally connects the module's output renderable object
// to the viewer.  Currently, the only valid input strings are "none"
// and "axis".
// This module assumes the presence of the DV wizard and/or
// the DI wizard, and uses them to locate the input field.
//
module viz_axis_select {
    string+IPort2+read+notify   user_selection;
    omethod+notify_val update = "WIZviz_axis";
};

// Takes the selection - a string probably chosen from a menu -
// and instances a mesh preprocess module based on the selection and
// the type of the input field.  Then it connects the input of the new
// module to the field, and finally connects the module's output renderable
// object to the viewer.   Currently, the valid input strings are "none",
// "crop", and "downsize".
// This module assumes the presence of the DV wizard and/or the
// DI wizard, and uses them to locate the input field.
//
module viz_preproc_mesh_select {
    string+IPort2+read+notify   user_selection;
    omethod+notify_val update = "WIZviz_preproc_mesh";
};

// Takes the selection - a string probably chosen from a menu -
// and instances a node/cell data preprocess module based on the selection
// and the type of the input field.  Then it connects the input of the new
// module to the field, and finally connects the module's output renderable
// object to the viewer.   Currently, the valid input strings are "none",
// "clamp" and "threshold".
// This module assumes the presence of the DV wizard and/or the
// DI wizard, and uses them to locate the input field.  It also
// is aware of the preprocess mesh module and first checks for
// the presence of either crop or downsize as a source of the
// input field.  Additionally, if it finds crop or downsize, it
// disconnects their output from the viewer.
//
module viz_preproc_data_select {
    string+IPort2+read+notify   user_selection;
    omethod+notify_val update = "WIZviz_preproc_data";
};

// Looks at the input field and generates a list of visualization
// modules that make sense for that field.  E.g. volume_render
// only works on 3D uniform fields, etc.  The output is just an
// array of strings that is then fed into other menus.
module viz_visualization_choices {
    Mesh+IPort2+read+notify     &in_field;
    string+OPort2+write   choices[];
    string+OPort2+write   icons[];
    omethod+notify_val update = "WIZviz_visualization_choices";
};

// Support routine for the core visualization panel.
// Takes a list of icon names generated by the 'choices' module
// and sets the UIlabel.labelPixmap.filename of each window.
// The window list will have enough windows for the maximum 
// possible number of icons. The number of icons vary by field type,
// so ff there are too many windows, some will have their visibility
// flag turned off. The last_visible window variable is to help with
// GUI layout issues.
module viz_visualization_set_icons {
    string+IPort2+read   &icons[];
    UIlabel+IPort2+read  &windows[];
    string+IPort2+read   prefix = "$XP_PATH<0>/runtime/pixmaps/wizard";
    UIlabel+OPort2+write &last_visible;
    omethod+notify_val update = "WIZviz_visualization_set_icons";
};

// Takes the selection - a string probably chosen from a menu -
// and instances a visualization module based on the selection
// and the type of the input field.  Then it connects the input of the new
// module to the field, and finally connects the module's output renderable
// object to the viewer.
// This module assumes the presence of the DV wizard and/or the
// DI wizard, and uses them to locate the input field. It is aware
// of the upstream modules and first checks for threshold/clamp/downsize/crop
// (in that order) as a source of the input field.
module viz_visualization_select {
    string+IPort2+read+notify   user_selection;
    omethod+notify_val update = "WIZviz_visualization";
};

// Takes the selection - a string probably chosen from a menu -
// and instances a legend based on the selection.   Then it connects
// the input of the new module the relevant renderable object, and finally
// connects the module's output renderable object to the viewer.
// This module assumes the presence of the DV wizard and/or the
// DI wizard, and uses them to locate the input fieldrenderable object.
// It is aware the upstream modules and first checks for a visualization
// module as a source of the input field.
module viz_legend_select {
    string+IPort2+read+notify   user_selection;
    omethod+notify_val update = "WIZviz_legend";
};

// Takes the selection - a string probably chosen from a menu -
// and instances a module based on the selection.   Currently the
// only supported module is the viewer background color module,
// which is instanced and connected to the viewer.
module viz_misc_select {
    string+IPort2+read+notify   user_selection;
    omethod+notify_val update = "WIZviz_misc";
};

}; // library Wiz_Modules

library+buffered Wiz_Macros<compile_subs=0> {

#ifdef MSDOS
UIlabel instructions<NEvisible=0> {
    width  => parent.clientWidth;
    string text;
    &label => .text;
    int rows = 1;
    height => (.rows * (UIdata.UIfonts[0].lineHeight + 0));
    alignment = "left";
};
#else
UItext instructions<NEvisible=0> {
    width  => parent.clientWidth;
    multiLine = 1;
    outputOnly = 1;
    updateMode = 0;
    // If used on Windows
    //color {
    //    backgroundColor = "#bbbbbb";
    //};
};
#endif

macro data_import_wizard<NEpixmapMode="large", NElargePixmapName="wizard.bmx"> {

    int visible = 1;

    // Hook up to reader output ...
    olink out_fld;
    olink out_obj<NEcolor0=0xff0000>;

    register_reader register_reader {
        reader_table[] = {
            { extensions={"fld"},
              description="AVS Field File",
              template="MODS.Read_Field",
              fname_sub="read_field_ui.filename",
              fld_sub="field",
              obj_sub="out_obj",
              uip_sub="read_field_ui.panel",
              mode=1
            },
            { extensions={"inp", "ucd"},
              description="AVS UCD File",
              template="MODS.Read_UCD",
              fname_sub="read_ucd_ui.filename",
              fld_sub="field",
              obj_sub="out_obj",
              uip_sub="read_ucd_ui.panel",
              mode=1
             },
            { extensions={"x"},
              description="AVS Image File",
              template="MODS.Read_Image",
              fname_sub="read_image_ui.filename",
              fld_sub="field",
              obj_sub="image",
              uip_sub="read_image_ui.panel",
              mode=1
            },
            { extensions={"bmp", "gif", "jpg", "jpeg",
                          "tif", "tiff", "ppm", "pgm", "pbm"},
              description="Image File",
              template="MODS.Read_Image",
              fname_sub="read_image_ui.filename",
              fld_sub="field",
              obj_sub="image",
              uip_sub="read_image_ui.panel",
              mode=1
            },
            { extensions={"dat"},
              description="AVS Volume File",
              template="MODS.Read_Volume",
              fname_sub="read_volume_ui.filename",
              fld_sub="field",
              obj_sub="geom",
              uip_sub="read_volume_ui.panel",
              mode=1
            },
            { extensions={"geo", "geom"},
              description="AVS Geometry File",
              template="MODS.Read_Geom",
              fname_sub="read_geom_ui.filename",
              fld_sub="field",
              obj_sub="geom",
              uip_sub="read_geom_ui.panel",
              mode=1
            },
            { extensions={"x", "p3d"},
              description="PLOT3D File",
              template="CFD.Read_PLOT3D",
              // There is often a second file, called a solution or q
              // file, but we won't try to handle that.
              fname_sub="ReadPlot3dParam.filename_xyz",
              // Non-standard outputs, hard to connect
              mode=1
            },
            { extensions={"nc", "cdf"},
              description="netCDF File (standard)",
              template="NETCDF.Read_Field",
              inst_name="Read_netCDF",
              fname_sub="Read_NetCDF_Param.filename",
              fld_sub="field",
              obj_sub="out_obj",
              uip_sub="read_netcdf_ui.panel",
              mode=1
            },
            { extensions={"nc", "cdf"},
              description="netCDF File (using AVS conventions)",
              template="NETCDF.Read_Object",
              inst_name="Read_netCDF_Obj",
              fname_sub="read_ui.filename",
              // Not always a field!  Oh well, what can you do?
              fld_sub="field",
              obj_sub="obj",
              uip_sub="read_ui.panel",
              mode=1
            },
            { extensions={"cgns"},
              description="CGNS File",
              template="CFD.Read_CGNS",
              fname_sub="CgnsParams.filename",
              // Multiple outputs, hard to connect
              mode=1
            },
            { extensions={"h5"},
              description="HDF5 File (using AVS conventions)",
              template="HDF5.Rd_HDF5_Field",
              fname_sub="Rd_HDF5_Param.filename",
              fld_sub="out_fld",
              obj_sub="out_obj",
              uip_sub="panel",
              mode=1
            },
            { extensions={"raw"},
              description="RAW (triangle) File",
              template="MODS.Read_Triangle",
              fname_sub="read_tri_ui.filename",
              fld_sub="out_fld",
              obj_sub="out_obj",
              uip_sub="read_tri_ui.panel",
              mode=1
            },
            { extensions={"stl", "sta", "stla"},
              description="Stereo Lithography (triangle) File",
              template="MODS.Read_Triangle",
              fname_sub="read_tri_ui.filename",
              fld_sub="out_fld",
              obj_sub="out_obj",
              uip_sub="read_tri_ui.panel",
              mode=1
            },
            { extensions={"slp"},
              description="Pro/Engineer Render (triangle) File",
              template="MODS.Read_Triangle",
              fname_sub="read_tri_ui.filename",
              fld_sub="out_fld",
              obj_sub="out_obj",
              uip_sub="read_tri_ui.panel",
              mode=1
            },
            { extensions={"tin"},
              description="Triangulated Irregular Network File",
              template="MODS.Read_Triangle",
              fname_sub="read_tri_ui.filename",
              fld_sub="out_fld",
              obj_sub="out_obj",
              uip_sub="read_tri_ui.panel",
              mode=1
            },
            { extensions={"obj"},
              description="Advanced Visualizer Object (polygon) File",
              template="MODS.Read_Polygon",
              fname_sub="read_poly_ui.filename",
              fld_sub="out_fld",
              obj_sub="out_obj",
              uip_sub="read_poly_ui.panel",
              mode=1
            },
            { extensions={"ply"},
              description="Stanford Polygon File",
              template="MODS.Read_Polygon",
              fname_sub="read_poly_ui.filename",
              fld_sub="out_fld",
              obj_sub="out_obj",
              uip_sub="read_poly_ui.panel",
              mode=1
            },
            { extensions={"vtk"},
              description="VTK (polygon) File",
              template="MODS.Read_Polygon",
              fname_sub="read_poly_ui.filename",
              fld_sub="out_fld",
              obj_sub="out_obj",
              uip_sub="read_poly_ui.panel",
              mode=1
            },
            { extensions={"dxf"},
              description="AutoCAD Drawing Interchange File",
              template="MODS.Read_DXF",
              fname_sub="read_dxf_ui.filename",
              fld_sub="out_fld",
              obj_sub="out_obj",
              uip_sub="read_dxf_ui.panel",
              mode=1
            },

            // IAC readers

            // Generic text/binary files.  Goes to 2-nd level
            // wizards instead of directly to a reader.

            { extensions={"txt", "text", "asc", "ascii"},
              description="Generic Text File",
              mode=2 },
            { extensions={"csv"},
              description="Comma-Separated-Value Text File",
              mode=2 },
            { extensions={"bin", "dat", "data"},
              description="Generic Binary File",
              mode=3 }
         };
    };

    group Wiz_Param<NEportLevels={0,1}> {
        string    filename<NEportLevels={1,2}>;

        int       mode<NEportLevels={1,2}>;
        // description of recognized filename
        string    description<NEportLevels={1,2}>;
        int       nreaders<NEportLevels={1,2}>;
        reader_templ readers<NEportLevels={1,2}>[];

        // Information about Reader that reads format
        string reader<NEportLevels={1,2}> => .readers[0].template;
    };

    examine_file examine_file {
        // In parameters
        filename    => <-.Wiz_Param.filename;
        mode        => <-.Wiz_Param.mode;
        description => <-.Wiz_Param.description;
        // Out parameters
        &readers  => <-.Wiz_Param.readers;
        &nreaders => <-.Wiz_Param.nreaders;
    };

    // The module stack is quite handy as the initial entry
    // point to the wizard.   It gives the user a reasonable way
    // to get back to the starting point after doing something else.

    UImod_panel panel {
        parent<NEportLevels={3,0}>;
        //title => name_of(<-.<-);
        // Might there be more than one at a time ...
        title = "Data Import Wizard";
        message = "Data Import Wizard control panel.";
    };

    GMOD.instancer instancer_1 {
        Value => panel.visible;
        active = 2; // don't de-instance when visible = 0
        Group => Wiz_UI;
    };

    int next_step_trigger;
    link param => .Wiz_Param;

    // Directly instance known reader.
    instance_reader trigger_known {
        &reader     => <-.Wiz_Param.readers[0];
        wizard_name => name_of(<-.<-);
        relative => <-.<-.<-;	// Application level
        trigger     => <-.Wiz_UI.continue.do;
        active       => is_valid(<-.param.mode) &&
                        (<-.param.mode == 1);
    };

    // Pass control to unknown-file second level wizard.
    GMOD.parse_v trigger_unknown {
        trigger => <-.next_step_trigger;
        active => is_valid(<-.param.mode) &&
                          ((<-.param.mode == 2)||
                           (<-.param.mode == 3)||
                           (<-.param.mode == 4));
        on_inst = 0;
        relative => <-;
        v_commands = "unknown_shell.visible = 1; ";
    };

    // Pass control to multiple-file second level wizard.
    GMOD.parse_v trigger_multiple {
        trigger => <-.next_step_trigger;
        active => is_valid(<-.param.mode) &&
                          (<-.param.mode == 5);
        on_inst = 0;
        relative => <-;
        v_commands = "multiple_shell.visible = 1; ";
    };

    macro Wiz_UI<instanced=0> {
        ilink param => <-.Wiz_Param;
        ilink panel => <-.panel;

        instructions fname_instruc<NEvisible=1> {
            parent => <-.panel;
            // The height in lines is a weak spot, depends on font width
            // and window width.
#ifdef MSDOS
            rows => 10;
#else
            rows => 12;

#endif
            text => "WIZ_DI_INSTRUCT_1";
            y    => 0;
        };

        UIlabel fname_label {
            parent => <-.panel;
            label  => "filename:";
            y      => fname_instruc.y + fname_instruc.height + 6;
            width  => 200;
            alignment = 0;
        };
        UItext file_name {
            parent => <-.panel;
            y => fname_label.y + fname_label.height + 5;
            text => <-.param.filename;
            width = 170;
            showLastPosition = 1;
        };
        UIbutton browser_visible {
            parent => <-.panel;
            x => file_name.x + file_name.width + 5;
            y => file_name.y;
            width = 75;
            parent => <-.panel;
            visible => <-.panel.visible;
            height => <-.file_name.height;
            label = "Browse...";
        };

        UIfileSB file_browser {
            GMOD.copy_on_change copy_on_change {
               trigger => <-.<-.browser_visible.do;
               input   => <-.<-.browser_visible.do;
               output  =>    <-.visible;
            };
            title = "Data Import Filename";
            searchPattern = "$XP_PATH<0>/data/*.*";
            filename => <-.param.filename;
        };

        instructions fname_instruc2<NEvisible=1> {
            parent  => <-.panel;
            visible => <-.panel.visible;
            y => <-.file_name.y + <-.file_name.height + 12;
#ifdef MSDOS
            rows => 7;
#else
            rows => 8;
#endif
            text => "WIZ_DI_INSTRUCT_2";            
        };

        UIlabel file_desc {
            parent  => <-.panel;
            visible => <-.panel.visible;
            y => <-.fname_instruc2.y + <-.fname_instruc2.height + 4;
            width => <-.panel.width;
            label => "file description:";
            alignment = "left";
        };

        UIlabel file_info {
            parent  => <-.panel;
            visible => <-.panel.visible;
            y => <-.file_desc.y + <-.file_desc.height + 4;
            width => <-.panel.width;
            color {
               backgroundColor = "#aaaabb";
            };
            // <NEportLevels={2,0}>
            label => switch( is_valid(<-.param.description)+1,
                                      "No filename",
                                      <-.param.description );
        };

        // Connect to HTML launcher
        UIbutton button_help {
            parent => <-.panel;
            y => file_info.y + file_info.height + 8;
            width = 100;
            label = "More Help";
        };
        UIlaunchBrowser launchBrowser {
            trigger  => <-.button_help.do;         
            filename = "wizards/import.htm";
            use_helppath = 1;
        };
        UIbutton continue {
            parent => <-.panel;
            x => button_help.x + button_help.width + 16;
            y => file_info.y   + file_info.height + 8;
            width = 100;
            parent => <-.panel;
            visible => <-.panel.visible;
            active  => is_valid(<-.param.mode) && (<-.param.mode != 0);
            label = "Continue >>";
            do => <-.<-.next_step_trigger;
        };
    };

    //==================================================

    // Unknown file second level wizard.

    UIshell unknown_shell {
        //parent<NEportLevels={3,0}>;
        title => "Unknown file import wizard";
        width  = 600;
        height = 720;
        visible = 0;
        message = "Unknown file import wizard";
    };

    GMOD.instancer instancer_2 {
        Value => unknown_shell.visible;
        active = 2; // don't de-instance when visible = 0
        Group => Wiz_UI_2;
    };

    macro Wiz_UI_2<instanced=0> {
        ilink param => <-.Wiz_Param;
        ilink panel => <-.unknown_shell;

        UIlabel file_name {
            parent => <-.panel;
            label  => str_format( "filename:  %s", <-.param.filename );
            x      = 4;
            y      = 4;
            width  => <-.panel.clientWidth;
            alignment = "left";
        };

        UIlabel question {
            parent => <-.panel;
            label  => "Which of the following best describes your data:";
            x      = 4;
            y      => file_name.y + file_name.height + 10;
            width  => <-.panel.clientWidth;
            alignment = 0;
        };

        //======================================

        UIframe frame_1 {
            parent => <-.panel;
            x      = 4;
            y => <-.question.y + <-.question.height + 8;
            height = 110;
            width  => .parent.clientWidth - (2*.x);
        };

        UItoggle choice_1 {
            parent => <-.frame_1;
            x = 4;
            y = 4;
            width = 300;
            label = "Column-oriented text data.";
            active => (<-.param.mode == 2) || (<-.param.mode == 4);
        };

        // Connect to HTML launcher
        UIbutton button_1 {
            parent => <-.frame_1;
            y = 0;
            //x => <-.choice_1.width + 4;
            width = 100;
            x => .parent.clientWidth - ( .width + 8);
            height => <-.choice_1.height;
            label = "More Help";
        };
        UIlaunchBrowser launchBrowser_1 {
            trigger  => <-.button_1.do;         
            filename = "wizards/cotd.htm";
            use_helppath = 1;
        };

        instance_reader instancer_1 {
            wizard_name     => name_of(<-.<-.<-);
            before_commands =
                 wizard_name + ".Wiz_UI_2.choice_1.set = 0; " +
                 wizard_name + ".unknown_shell.visible = 0;";
            reader {
                template  = "WIZARD.text_columns_to_scat";
                inst_name = "text_columns_to_scat";
                fname_sub = "Read_Text_Columns.param.filename";
                fld_sub   = "table_to_scatter_field.out_fld";
                obj_sub   = "table_to_scatter_field.out_obj";
                uip_sub   = "Read_Text_Columns.UIpanel";
            };
            trigger => <-.choice_1.do;
            relative => <-.<-.<-.<-;	// Application level
        };

        instructions label_1<NEvisible=1> {
            parent => <-.frame_1;
            x = 4;
            y => <-.choice_1.y + <-.choice_1.height + 4;
            width  => .parent.clientWidth - 8;
            rows => 4;
            text => "WIZ_DI_UNKNOWN_TEXT_1";
            active => (<-.param.mode == 2) || (<-.param.mode == 4);
        };

        //======================================

        UIframe frame_2 {
            parent => <-.panel;
            x      = 4;
            y => <-.frame_1.y + <-.frame_1.height + 4;
            height = 110;
            width  => .parent.clientWidth - (2*.x);
        };

        UItoggle choice_2 {
            parent => <-.frame_2;
            x = 4;
            //y => <-.label_1.y + <-.label_1.height + 4;
            y = 4;
            //width  => .parent.clientWidth;
            width = 300;
            label = "Text data organized in a grid.";
            active => (<-.param.mode == 2) || (<-.param.mode == 4);
        };

        // Connect to HTML launcher
        UIbutton button_2 {
            parent => <-.frame_2;
            y = 0;
            //x => <-.choice_2.width + 4;
            width = 100;
            x => .parent.clientWidth - ( .width + 8);
            height => <-.choice_2.height;
            label = "More Help";
        };
        UIlaunchBrowser launchBrowser_2 {
            trigger  => <-.button_2.do;         
            filename = "wizards/rgtd.htm";
            use_helppath = 1;
        };

        instance_reader instancer_2 {
            wizard_name     => name_of(<-.<-.<-);
            before_commands =
                 wizard_name + ".Wiz_UI_2.choice_2.set = 0; " +
                 wizard_name + ".unknown_shell.visible = 0;";
            reader {
                template  = "MODS.Read_Text_Grid";
                inst_name = "Read_Text_Grid";
                fname_sub = "param.filename";
                fld_sub   = "field";
                obj_sub   = "out_obj";
                uip_sub   = "UIpanel";
            };
            trigger => <-.choice_2.do;
            relative => <-.<-.<-.<-;	// Application level
        };

        instructions label_2<NEvisible=1> {
            parent => <-.frame_2;
            x = 4;
            y => <-.choice_2.y + <-.choice_2.height;
            width => .parent.clientWidth - 8;
            rows => 4;
            text => "WIZ_DI_UNKNOWN_TEXT_2";
            active => (<-.param.mode == 2) || (<-.param.mode == 4);
        };

        //======================================

        UIframe frame_3 {
            parent => <-.panel;
            x      = 4;
            y => <-.frame_2.y + <-.frame_2.height + 4;
            height = 110;
            width  => .parent.clientWidth - (2*.x);
        };

        UItoggle choice_3 {
            parent => <-.frame_3;
            x = 4;
            //y => <-.label_2.y + <-.label_2.height + 4;
            y = 4;
            //width  => .parent.clientWidth;
#ifdef MSDOS
            width = 300;
#else
            width = 350;
#endif
            label = "Text data organized as a continuous sequence of numbers.";
            active => (<-.param.mode == 2) || (<-.param.mode == 4);
        };

        // Connect to HTML launcher
        UIbutton button_3 {
            parent => <-.frame_3;
            y = 0;
            width = 100;
            x => .parent.clientWidth - ( .width + 8);
            height => <-.choice_3.height;
            label = "More Help";
        };
        UIlaunchBrowser launchBrowser_3 {
            trigger  => <-.button_3.do;         
            filename = "wizards/rstd.htm";
            use_helppath = 1;
        };

        instance_reader instancer_3 {
            wizard_name     => name_of(<-.<-.<-);
            before_commands =
                 wizard_name + ".Wiz_UI_2.choice_3.set = 0; " +
                 wizard_name + ".unknown_shell.visible = 0;";
            reader {
                template  = "MODS.Read_Text_Sequence";
                inst_name = "Read_Text_Sequence";
                fname_sub = "param.filename";
                fld_sub   = "field";
                obj_sub   = "out_obj";
                uip_sub   = "UIpanel";
            };
            trigger => <-.choice_3.do;
            relative => <-.<-.<-.<-;	// Application level
        };

        instructions label_3<NEvisible=1> {
            parent => <-.frame_3;
            x = 2;
            y => <-.choice_3.y + <-.choice_3.height;
            width => .parent.clientWidth - 4;
            rows => 4;
            text => "WIZ_DI_UNKNOWN_TEXT_3";
            active => (<-.param.mode == 2) || (<-.param.mode == 4);
        };

        //======================================

        UIframe frame_4 {
            parent => <-.panel;
            x      = 4;
            y => <-.frame_3.y + <-.frame_3.height + 4;
            height = 110;
            width  => .parent.clientWidth - (2*.x);
        };

        UItoggle choice_4 {
            parent => <-.frame_4;
            x = 4;
            //y => <-.label_3.y + <-.label_3.height + 4;
            y = 4;
            //width  => .parent.clientWidth;
            width = 350;
            label = "Binary data organized as a sequence of numbers.";
            // Not if text,csv, etc format ...
            active => (<-.param.mode == 3) || (<-.param.mode == 4);
        };

        // Connect to HTML launcher
        UIbutton button_4 {
            parent => <-.frame_4;
            y = 0;
            width = 100;
            x => .parent.clientWidth - ( .width + 8);
            height => <-.choice_4.height;
            label = "More Help";
        };
        UIlaunchBrowser launchBrowser_4 {
            trigger  => <-.button_4.do;         
            filename = "wizards/rbtd.htm";
            use_helppath = 1;
        };

        instance_reader instancer_4 {
            wizard_name     => name_of(<-.<-.<-);
            before_commands =
                 wizard_name + ".Wiz_UI_2.choice_4.set = 0; " +
                 wizard_name + ".unknown_shell.visible = 0;";
            reader {
                template  = "MODS.Read_Binary_Sequence";
                inst_name = "Read_Binary_Sequence";
                fname_sub = "param.filename";
                fld_sub   = "field";
                obj_sub   = "out_obj";
                uip_sub   = "UIpanel";
            };
            trigger => <-.choice_4.do;
            relative => <-.<-.<-.<-;	// Application level
        };

        instructions label_4<NEvisible=1> {
            parent => <-.frame_4;
            x = 2;
            y => <-.choice_4.y + <-.choice_4.height;
            width => .parent.clientWidth - 4;
            rows => 4;
            text => "WIZ_DI_UNKNOWN_BIN_1";
            // Not if text, csv, etc format ...
            active => (<-.param.mode == 3) || (<-.param.mode == 4);
        };

        //======================================

        instructions cancel_info<NEvisible=1> {
            parent => <-.panel;
            x = 20;
            y => <-.frame_4.y + <-.frame_4.height + 6;
            width  => .parent.clientWidth - (2*.x);
            rows => 5;
            text => "WIZ_DI_UNKNOWN_NONE";
        };

        UIbutton cancel {
            parent => <-.panel;
            x => <-.cancel_info.x;
            y => <-.cancel_info.y + <-.cancel_info.height + 8;
            width = 75;
            visible => <-.panel.visible;
            label = "Cancel";
        };

        GMOD.parse_v cancel_shell {
            trigger => <-.cancel.do;
            relative => <-.<-;
            on_inst = 0;
            v_commands = "unknown_shell.visible = 0;";
        };
    }; // Wiz_UI_2

    //==================================================

    UIshell multiple_shell {
        //parent<NEportLevels={3,0}>;
        title => "Multiple options file import wizard";
        width = 600;
        height = 350;
        visible = 0;
        message = "Multiple options file import wizard";
    };

    GMOD.instancer instancer_3 {
        Value => multiple_shell.visible;
        active = 2; // don't de-instance when visible = 0
        Group => Wiz_UI_3;
    };

    macro Wiz_UI_3<instanced=0> {
        ilink param => <-.Wiz_Param;
        ilink panel => <-.multiple_shell;

        UIlabel file_name {
            parent => <-.panel;
            label  => str_format( "filename:  %s", <-.param.filename );
            x      = 4;
            y      = 4;
            width  => <-.panel.clientWidth;
            alignment = "left";
        };

        instructions question<NEvisible=1> {
            parent => <-.panel;
            text => "WIZ_DI_MULTIPLE";
            x      = 4;
            y      => file_name.y + file_name.height + 10;
            height => (2 * (UIdata.UIfonts[0].lineHeight + 0));
        };

        //======================================

        imlink labels => param.readers.description;

        int+IPort2                selectedItem;
        string+nres+IPort2        selectedString => labels[selectedItem];

        // I want this to trigger from selectedString/selectedItem changing
        examine_descr examine_descr {
            // In parameters
            in_description => selectedString;
            mode        => param.mode;
            description => param.description;
            // Out parameters
            // This should always result in 1 reader.
            &readers  => param.readers;
            &nreaders => param.nreaders;
        };

        // I want this to trigger after examine_descr does its thing.
        // I want selectedItem to be unset so that the radiobox
        // comes back up with no choices selected.
        GMOD.parse_v next_step_parse_v {
            trigger  => <-.selectedItem;
            relative => <-.<-;
            on_inst = 0;
            sync = 0;
            active   => is_valid(<-.selectedItem) &&
                        is_valid(param.nreaders) &&
                        (param.nreaders == 1) &&
                        (multiple_shell.visible == 1);
            v_commands =
                "multiple_shell.visible = 0; " +
                "Wiz_UI_3.selectedItem => ; " +
                "next_step_trigger = 1; " ;
        };

        label_cmd label_cmd {
            labels => <-.labels;
            inItem => <-.selectedItem;
            doSet = 0;
        };
        UIradioBox rb_choices {
            parent	=> <-.panel;
            cmdList	=> <-.label_cmd.cmd;
            &selectedItem => <-.label_cmd.outItem;
            x = 20;
            y		=> <-.question.y + <-.question.height + 5;
            width	=> <-.panel.clientWidth;
        };

        instructions cancel_info<NEvisible=1> {
            parent => <-.panel;
            x = 20;
            y => <-.rb_choices.y + <-.rb_choices.height + 16;
            width => .parent.clientWidth - (2*.x);
            rows  => 5;
            text => "WIZ_DI_UNKNOWN_NONE";
        };

        UIbutton cancel {
            parent => <-.panel;
            x => <-.cancel_info.x;
            y => <-.cancel_info.y + <-.cancel_info.height + 8;
            width = 75;
            visible => <-.panel.visible;
            label = "Cancel";
        };

        GMOD.parse_v cancel_shell {
            trigger => <-.cancel.do;
            relative => <-.<-;
            on_inst = 0;
            v_commands = "multiple_shell.visible = 0;";
        };
    };
};

macro data_visualization_wizard<NEpixmapMode="large", NElargePixmapName="wizard.bmx"> {

    // Hook up to reader output (or any field) when the
    // DV wizard is being used without the data import wizard.
    ilink in_field;

    olink out_fld;

    // We actually don't want to encourage the user connecting the output
    // of the DV wiz to the viewer, because the wizard logic also tries
    // to connect to the viewer.
    link out_obj<NEportLevels={1,1},NEcolor0=0xff0000>;

    group Wiz_Param<NEportLevels={0,1}> {
        // text description of recognized field
        string description<NEportLevels={1,2}>;
        // Parameters to help drive the rest of the Wizard.
        //int type;	// unif, rect, struct, unstruct
        //int nspace;	// basic dimensionality
        //int node_data;	// yes/no flag
        //int cell_data;	// yes/no flag
    };

    examine_field examine_field {
        in_field    => <-.in_field;
        description => <-.Wiz_Param.description;
    };

    viz_bounding_select viz_bounding {
        user_selection => <-.Wiz_UI_Framing.bounding.selectedString;
    };
    viz_axis_select viz_axis {
        user_selection => <-.Wiz_UI_Framing.axis.selectedString;
    };
    viz_preproc_mesh_select viz_preproc_mesh {
        user_selection => <-.Wiz_UI_Preproc.mesh.selectedString;
    };
    viz_preproc_data_select viz_preproc_data {
        user_selection => <-.Wiz_UI_Preproc.data.selectedString;
    };
    viz_visualization_choices viz_choices {
        in_field      => <-.in_field;
        choices       => <-.Wiz_UI_Visualization.strings_viz;
    };
    viz_visualization_select viz_visualization {
        user_selection => <-.Wiz_UI_Visualization.viz.selectedString;
    };
    viz_legend_select viz_legend {
        user_selection => <-.Wiz_UI_Legend.legend.selectedString;
    };
    viz_misc_select viz_misc {
        user_selection => <-.Wiz_UI_Legend.misc.selectedString;
    };

    // The module stack is quite handy as the initial entry
    // point to the wizard.  It gives the user a reasonable way
    // to get back to the starting point after doing something else.

    UImod_panel panel {
        parent<NEportLevels={3,0}>;
        //title => name_of(<-.<-);
        // Might there be more than one at a time ...
        title = "Data Visualization Wizard";
        message = "Data Visualization Wizard control panel.";
    };

    // <instanced=0>
    macro Wiz_UI_Intro {
        ilink param => <-.Wiz_Param;
        ilink panel => <-.panel;

       instructions greeting<NEvisible=1> {
            parent => <-.panel;
#ifdef MSDOS
            rows => 13;
#else
            rows => 15;
#endif
            text => "WIZ_DV_INSTRUCT_1";
            y      => 0;
        };

        UIlabel file_desc {
            parent  => <-.panel;
            visible => <-.panel.visible;
            y => <-.greeting.y + <-.greeting.height + 8;
            width => <-.panel.width;
            label => "field description:";
            alignment = "left";
        };

        UIlabel file_info {
            parent  => <-.panel;
            visible => <-.panel.visible;
            y => <-.file_desc.y + <-.file_desc.height + 4;
            width => <-.panel.width;
            height => 2 * UIdata.UIfonts[0].lineHeight;
            color {
               backgroundColor = "#aaaabb";
            };
            // <NEportLevels={2,0}>
           label => switch( is_valid(<-.param.description)+1,
                                      "No field",
                                      <-.param.description );
        };

        // Connect to HTML launcher
        UIbutton button_help {
            parent => <-.panel;
            y => file_info.y + file_info.height + 12;
            width = 100;
            label = "More Help";
        };
        UIlaunchBrowser launchBrowser {
            trigger  => <-.button_help.do;         
            filename = "wizards/dvwiz.htm";
            use_helppath = 1;
        };
        UIbutton continue {
            parent => <-.panel;
            x => button_help.x + button_help.width + 16;
            y => file_info.y   + file_info.height + 12;
            width = 100;
            parent => <-.panel;
            visible => <-.panel.visible;
            active  => is_valid(<-.param.description) &&
                       strcmp(<-.param.description, "Invalid field") &&
                       strcmp(<-.param.description, "No field");
            label = "Continue >>";
        };

        GMOD.parse_v parse_v {
            trigger => <-.continue.do;
            on_inst = 0;
            relative => <-.<-;
            v_commands = "Wiz_UI_Framing.visible = 1;" ;
        };
    };

    macro Wiz_UI_Framing {
        ilink param => <-.Wiz_Param;

        string strings_bounds<NEportLevels=1>[] = {"none", "bounding box"};

        string strings_axis<NEportLevels=1>[] = {"none", "axis"};

        UIshell panel {
            visible = 0;
            title   = "Data Viz Wizard - Framing";
            message = "Data Visualization Wizard - Framing";
            height = 400;
            width  = 350;
        };

        int visible<NEportLevels={2,0}> => panel.visible;

        instructions instruct<NEvisible=1> {
            parent => <-.panel;
            rows   => 4;
            text   => "WIZ_DV_FRAME";
            x      => 6;
            y      => 4;
            width  => parent.clientWidth - (2 * .x);
        };

        UIradioBoxLabel bounding {
            parent => <-.panel;
            x = 4;
            y => <-.instruct.y + <-.instruct.height + 12;
            title = "Select bounding option:";
            labels => <-.strings_bounds;
            selectedItem = 0;
            UIlabel {
                alignment = "left";
                x = 4;
             };
        };

        UIlabel bounding_icon {
            parent => <-.panel;
            height => 100;
            width  => 100;
            label => ;
            x => <-.bounding.x + <-.bounding.width;
            y => <-.bounding.y;
            labelPixmap {
                filename = "$XP_PATH<0>/runtime/pixmaps/wizard/bounds.ppm";
           };
        };

        UIradioBoxLabel axis {
            parent => <-.panel;
            x = 4;
            y => bounding_icon.y + bounding_icon.height + 16;
            title = "Select axis option:";
            labels => <-.strings_axis;
            selectedItem = 0;
            UIlabel {
                alignment = "left";
                x = 4;
            };
        };

        UIlabel axis_icon {
            parent => <-.panel;
            height => 100;
            width  => 100;
            label => ;
            x => <-.axis.x + <-.axis.width;
            y => <-.axis.y;
            labelPixmap {
                filename = "$XP_PATH<0>/runtime/pixmaps/wizard/axis.ppm";
           };
        };

        UIbutton back {
            parent => <-.panel;
            x = 12;
            y => <-.axis.y + <-.axis_icon.height + 12;
            width = 100;
            parent => <-.panel;
            visible => <-.panel.visible;
            label = "<< Back";
        };
        UIbutton continue {
            parent => <-.panel;
            x = <-.back.x + <-.back.width + 12;
            y => axis.y + axis_icon.height + 12;
            width = 100;
            parent => <-.panel;
            visible => <-.panel.visible;
            label = "Continue >>";
        };

        GMOD.parse_v back_cmds {
            trigger => <-.back.do;
            on_inst = 0;
            relative => <-.<-;
            v_commands =
            "Wiz_UI_Framing.visible = 0;";
        };
        GMOD.parse_v continue_cmds {
            trigger => <-.continue.do;
            on_inst = 0;
            relative => <-.<-;
            v_commands =
            "Wiz_UI_Framing.visible = 0;Wiz_UI_Preproc.visible = 1;";
        };
    };

    macro Wiz_UI_Preproc {
        ilink param => <-.Wiz_Param;

        UIshell panel {
            visible = 0;
            title   = "Data Viz Wizard - Preprocessing";
            message = "Data Visualization Wizard - Preprocessing";
#ifdef MSDOS
            // Need to adjust height to adjust for the title bar, which
            // can vary in size depending on the user's desktop settings.
            height => <-.back.y + <-.back.height + 55;
#else
            height => <-.back.y + <-.back.height + 20;
#endif
            width  = 350;
        };

        int visible<NEportLevels={2,0}> => panel.visible;

        string strings_mesh<NEportLevels=1>[] = {"none", "crop", "downsize"};

        string strings_data<NEportLevels=1>[] = {"none", "clamp", "threshold"};

        instructions instruct<NEvisible=1> {
            parent => <-.panel;
#ifdef MSDOS
            rows   => 5;
#else
            rows   => 6;
#endif
            text   => "WIZ_DV_PREPROC_1";
            x      => 6;
            y      => 4;
            width  => parent.clientWidth - (2 * .x);
        };

        UIradioBoxLabel mesh {
            parent => <-.panel;
            x = 4;
            //y = 8;
            y => <-.instruct.y + <-.instruct.height + 4;
            title = "Select preprocess mesh option:";
            labels => <-.strings_mesh;
            selectedItem = 0;
        };

        UIlabel crop_icon {
            parent => <-.panel;
            height => 100;
            width  => 100;
            label => ;
            x => <-.mesh.x + <-.mesh.width + 12;
            y => <-.mesh.y;
            labelPixmap {
                filename = "$XP_PATH<0>/runtime/pixmaps/wizard/crop.ppm";
           };
        };

        UIlabel down_icon {
            parent => <-.panel;
            height => 100;
            width  => 100;
            label => ;
            x => <-.crop_icon.x;
            y => <-.crop_icon.y + <-.crop_icon.height + 2;
            labelPixmap {
                filename = "$XP_PATH<0>/runtime/pixmaps/wizard/down.ppm";
           };
        };

        instructions instruct2 {
            parent => <-.panel;
#ifdef MSDOS
            rows   => 4;
#else
            rows   => 5;
#endif
            text   => "WIZ_DV_PREPROC_2";
            x      => 6;
            y      => down_icon.y + down_icon.height + 16;
            width  => parent.clientWidth - (2 * .x);
        };

        UIradioBoxLabel data {
            parent => <-.panel;
            x = 4;
            y => instruct2.y + instruct2.height + 4;
            title = "Select preprocess data option:";
            labels => <-.strings_data;
            selectedItem = 0;
        };

        UIlabel clamp_icon {
            parent => <-.panel;
            height => 100;
            width  => 100;
            label => ;
            x => <-.data.x + <-.data.width + 12;
            y => <-.data.y;
            labelPixmap {
                filename = "$XP_PATH<0>/runtime/pixmaps/wizard/clamp.ppm";
           };
        };
        UIlabel thresh_icon {
            parent => <-.panel;
            height => 100;
            width  => 100;
            label => ;
            x => <-.clamp_icon.x;
            y => <-.clamp_icon.y + <-.clamp_icon.height + 2;
            labelPixmap {
                filename = "$XP_PATH<0>/runtime/pixmaps/wizard/thresh.ppm";
           };
        };

        UIbutton back {
            parent => <-.panel;
            x = 12;
            y => <-.thresh_icon.y + <-.thresh_icon.height + 12;
            width = 100;
            parent => <-.panel;
            visible => <-.panel.visible;
            label = "<< Back";
        };
        UIbutton continue {
            parent => <-.panel;
            x = <-.back.x + <-.back.width + 12;
            y => back.y;
            width = 100;
            parent => <-.panel;
            visible => <-.panel.visible;
            label = "Continue >>";
        };

        GMOD.parse_v back_cmds {
            trigger => <-.back.do;
            on_inst = 0;
            relative => <-.<-;
            v_commands =
            "Wiz_UI_Preproc.visible = 0;Wiz_UI_Framing.visible = 1;";
        };
        GMOD.parse_v continue_cmds {
            trigger => <-.continue.do;
            on_inst = 0;
            relative => <-.<-;
            v_commands =
            "Wiz_UI_Preproc.visible = 0;Wiz_UI_Visualization.visible = 1;";
        };

    };

// Visualization

    macro Wiz_UI_Visualization {
        ilink param => <-.Wiz_Param;

        UIshell panel {
            visible = 0;
            title   = "Data Viz Wizard - Visualization";
            message = "Data Visualization Wizard - Visualization";
            //height = 500;
#ifdef MSDOS
            // There can be a large difference between the
            // height and clientHeight.
            height => <-.continue.y + <-.continue.height + 60;
#else
            height => <-.continue.y + <-.continue.height + 20;
#endif
            width  = 350;
        };

        int visible<NEportLevels={2,0}> => panel.visible;

#ifdef XXXXXXXX
        // Now filled in dynamically based on field type
        string strings_viz<NEportLevels=1>[] = {"none",
            "volume_render", "isosurface", "isovolume", "contour",
            "orthoslice", "city_plot", "ribbons", "isoline",
            "solid_contour", "surf_plot", "extrude_cells"
        };
#else
        string strings_viz<NEportLevels=1>[] = {"none" };
#endif

        viz_visualization_set_icons viz_set_icons {
            icons         => <-.<-.viz_choices.icons;
            windows       => { <-.icon_1, <-.icon_2,
                               <-.icon_3, <-.icon_4,
                               <-.icon_5 };
            last_visible => <-.icon_5;
        };

        instructions instruct {
            parent => <-.panel;
#ifdef MSDOS
            rows => 4;
#else
            rows => 5;
#endif
            text => "WIZ_DV_COREVIZ";
            x      => 6;
            y      => 4;
            width  => parent.clientWidth - (2 * .x);
        };

        UIradioBoxLabel viz {
            parent => <-.panel;
            x = 4;
            y => <-.instruct.y + <-.instruct.height + 12;
            title = "Select visualization option:";
            labels => <-.strings_viz;
            selectedItem = 0;
        };

        UIlabel icon_1 {
            parent => <-.panel;
            height => 100;
            width  => 100;
            label  => ;
            x => <-.viz.x + <-.viz.width;
            y => <-.viz.y;
        };
        UIlabel icon_2 {
            parent => <-.panel;
            height => 100;
            width  => 100;
            label  => ;
            x => <-.icon_1.x;
            y => <-.icon_1.y + <-.icon_1.height + 2;
        };
        UIlabel icon_3 {
            parent => <-.panel;
            height => 100;
            width  => 100;
            label  => ;
            x => <-.icon_1.x;
            y => <-.icon_2.y + <-.icon_2.height + 2;
        };
        UIlabel icon_4 {
            parent => <-.panel;
            height => 100;
            width  => 100;
            label  => ;
            x => <-.icon_1.x;
            y => <-.icon_3.y + <-.icon_3.height + 2;
        };
        UIlabel icon_5 {
            parent => <-.panel;
            height => 100;
            width  => 100;
            label  => ;
            x => <-.icon_1.x;
            y => <-.icon_4.y + <-.icon_4.height + 2;
        };

        UIbutton back {
            parent => <-.panel;
            x = 12;
            //y => <-.viz.y + <-.viz.height + 12;
            //y => <-.isoline_icon.y + <-.isoline_icon.height + 12;
            y => <-.viz_set_icons.last_visible.y + 
                 <-.viz_set_icons.last_visible.height + 12;
            width = 100;
            parent => <-.panel;
            visible => <-.panel.visible;
            label = "<< Back";
        };
        UIbutton continue {
            parent => <-.panel;
            x => <-.back.x + <-.back.width + 12;
            y => <-.back.y;
            width = 100;
            parent => <-.panel;
            visible => <-.panel.visible;
            label = "Continue >>";
        };

        GMOD.parse_v back_cmds {
            trigger => <-.back.do;
            on_inst = 0;
            relative => <-.<-;
            v_commands =
            "Wiz_UI_Visualization.visible = 0;Wiz_UI_Preproc.visible = 1;";
        };
        GMOD.parse_v continue_cmds {
            trigger => <-.continue.do;
            on_inst = 0;
            relative => <-.<-;
            v_commands =
            "Wiz_UI_Visualization.visible = 0;Wiz_UI_Legend.visible = 1;";
        };

    };

// Legend & Misc

    macro Wiz_UI_Legend {
        ilink param => <-.Wiz_Param;

        string strings_legend<NEportLevels=1>[] =
                { "none", "horizontal", "vertical" };

        string strings_misc<NEportLevels=1>[] =
                { "none", "set background color" };

        UIshell panel {
            visible = 0;
            title   = "Data Viz Wizard - Legend";
            message = "Data Visualization Wizard - Legend";
#ifdef MSDOS
            height => <-.back.y + <-.back.height + 55;
#else
            height => <-.back.y + <-.back.height + 20;
#endif
            width  = 350;
        };

        int visible<NEportLevels={2,0}> => panel.visible;

        instructions instruct {
            parent => <-.panel;
#ifdef MSDOS
            rows   => 4;
#else
            rows   => 5;
#endif
            text   => "WIZ_DV_LEGEND";
            x      => 6;
            y      => 4;
            width  => parent.clientWidth - (2 * .x);
        };

        UIradioBoxLabel legend {
            parent => <-.panel;
            x = 4;
            y = 8;
            y => <-.instruct.y + <-.instruct.height + 12;
            title = "Select legend option:";
            labels => <-.strings_legend;
            selectedItem = 0;
            UIlabel {
                alignment = "left";
                x = 4;
             };
        };
        UIlabel legend_icon {
            parent => <-.panel;
            height => 100;
            width  => 100;
            label => ;
            x => <-.legend.x + <-.legend.width;
            y => <-.legend.y;
            labelPixmap {
                filename = "$XP_PATH<0>/runtime/pixmaps/wizard/legend.ppm";
           };
        };
        UIradioBoxLabel misc {
            parent => <-.panel;
            x = 4;
            y => legend.y + legend.height + 16;
            title = "Select misc options:";
            labels => <-.strings_misc;
            selectedItem = 0;
            UIlabel {
                alignment = "left";
                x = 4;
            };
        };

        UIbutton back {
            parent => <-.panel;
            x = 12;
            y => misc.y + misc.height + 12;
            width = 100;
            parent => <-.panel;
            visible => <-.panel.visible;
            label = "<< Back";
        };
        UIbutton finish {
            parent => <-.panel;
            x = <-.back.x + <-.back.width + 12;
            y => misc.y + misc.height + 12;
            width = 100;
            parent => <-.panel;
            visible => <-.panel.visible;
            label = "Finish";
        };
        GMOD.parse_v back_cmds {
            trigger => <-.back.do;
            on_inst = 0;
            relative => <-.<-;
            v_commands =
            "Wiz_UI_Legend.visible = 0;Wiz_UI_Visualization.visible = 1;";
        };
        GMOD.parse_v finish_cmds {
            trigger => <-.finish.do;
            on_inst = 0;
            relative => <-.<-;
            v_commands = "Wiz_UI_Legend.visible = 0;";
        };
    };

};

//===============================================================

// Support macros used by the Wizards

macro text_columns_to_scat {
    MODS.Read_Text_Columns Read_Text_Columns;
    MODS.table_to_scatter_field table_to_scatter_field {
        columns => <-.Read_Text_Columns.table;
        out_fld<NEportLevels={1,3}>;
        out_obj<NEportLevels={1,3}>;
        x_name<NEportLevels={1,3}>;
        y_name<NEportLevels={1,3}>;
        z_name<NEportLevels={1,3}>;
    };
};

macro view_set_color {

    //GD.GDview_notify_templ &View<NEportLevels={2,0},NEcolor0=0xff00ff>;
    group &View<NEportLevels={2,0},NEcolor0=0xff00ff> {
        float back_col[3];
    };

    // Automatically search for and connect to a Uviewer
    view_connect view_connect {
        parent      => <-.<-;
        application => <-.<-.<-;	// point at application level
        int back_col = 1;		// special flag
    };

    GMOD.rgb_or_hsv rgb_or_hsv<NEx=561.,NEy=374.> {
        r => <-.View.back_col[0];
        g => <-.View.back_col[1];
        b => <-.View.back_col[2];
    };

    UImod_panel panel {
        title => name_of(<-.<-,1);
        message = "Select background color control panel.";
        parent<NEportLevels={3,0}>;
    };

    string string<NEportLevels=1>[] = {"RGB", "HSV"};

    macro UIradioBoxLabel {
        ilink parent => <-.panel;
        imlink labels => <-.string;
        int selectedItem<NEportLevels={2,0}> = 0;
        string+nres selectedString<NEportLevels={2,0}> => labels[selectedItem];
        string title => name_of(<-,1);
        int visible = 1;
        int active = 1;
        int x = 4;
        int y = 4;
        int width  => .parent.clientWidth;
        int+nres height => UIdata.UIfonts[0].lineHeight;

        UI.label_cmd label_cmd {
            labels => <-.labels;
            cmd[] {
                active => <-.<-.active;
            } = {{set=1,},};
            inItem => <-.selectedItem;
        };

        UIradioBox UIradioBox {
            x => <-.x;
            y => <-.y;
            width => <-.width;
            height => <-.height;
            &visible => <-.visible;
            parent => <-.parent;
            cmdList => <-.label_cmd.cmd;
            selectedItem => <-.label_cmd.outItem;
            orientation = "horizontal";
        };
   };

   group RGB {
      UIpanel UIpanel {
         y => ((<-.<-.UIradioBoxLabel.y + <-.<-.UIradioBoxLabel.height) + 4);
         visible => (UIradioBoxLabel.selectedItem == 0);
         parent<NEportLevels={3,0}> => <-.<-.panel;
         width => .parent.clientWidth;
      };
      VUISlider r_slider {
         parent => <-.UIpanel;
         title = "red";
         width => .parent.clientWidth;
         mode = 2;
         max = 1;
         value => <-.<-.rgb_or_hsv.r;
         Slider.showValue = 0; // Cleaner appearance, given SliderField
      };
      VUISlider g_slider {
         parent => <-.UIpanel;
         y => (<-.r_slider.y + <-.r_slider.height);
         title = "green";
         width => .parent.clientWidth;
         mode = 2;
         max = 1;
         value => <-.<-.rgb_or_hsv.g;
         Slider.showValue = 0;
      };
      VUISlider b_slider {
         parent => <-.UIpanel;
         y => (<-.g_slider.y + <-.g_slider.height);
         title = "blue";
         width => .parent.clientWidth;
         mode = 2;
         max = 1;
         value => <-.<-.rgb_or_hsv.b;
         Slider.showValue = 0;
      };
   };

   group HSV {
      UIpanel UIpanel {
         y => ((<-.<-.UIradioBoxLabel.y + <-.<-.UIradioBoxLabel.height) + 4);
         visible => (UIradioBoxLabel.selectedItem == 1);
         parent<NEportLevels={3,0}> => <-.<-.panel;
         width => .parent.clientWidth;
      };
      VUISlider h_slider {
         parent => <-.UIpanel;
         title = "hue";
         width => .parent.clientWidth;
         value => <-.<-.rgb_or_hsv.h;
         Slider.showValue = 0;
      };
      VUISlider s_slider {
         parent => <-.UIpanel;
         y => (<-.h_slider.y + <-.h_slider.height);
         title = "saturation";
         width => .parent.clientWidth;
         value => <-.<-.rgb_or_hsv.s;
         Slider.showValue = 0;
      };
      VUISlider v_slider {
         parent => <-.UIpanel;
         y => (<-.s_slider.y + <-.s_slider.height);
         title = "value";
         width => .parent.clientWidth;
         value => <-.<-.rgb_or_hsv.v;
         Slider.showValue = 0;
      };
   };
};

MODS.orthoslice orthoslice {
    OrthoSliceParam {
        // If you start with the third axis, then there is less of
        // a chance you will start off viewing the slice edge on.
	axis+nres => <-.in_field.ndim-1;
    };
};

MODS.slice_orthoplane slice_orthoplane {
    SliceParam {
        // If you start with the third axis, then there is less of
        // a chance you will start off viewing the slice edge on.
        axis+nres => <-.in_field.nspace-1;
    };
};

}; // library Wiz_Macros

};
