/*
** pngtopnm.c -
** read a Portable Network Graphics file and produce a portable anymap
**
** Copyright (C) 1995 by Alexander Lehmann <alex@hal.rhein-main.de>
**                   and Willem van Schaik <gwillem@ntuvax.ntu.ac.sg>
**
** version 2.0 - August 1995
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.
**
** modeled after giftopnm by David Koblas and
** with lots of bits pasted from pnglib.txt by Guy Eric Schalnat
*/

/* pnglib forgot gamma correction for palette images, so we do it ourselves */
#define HOMEMADE_GAMMA

#include <math.h>
#include "pnm.h"
#include "png.h"

#ifndef TRUE
#define TRUE 1
#endif
#ifndef FALSE
#define FALSE 0
#endif
#ifndef NONE
#define NONE 0
#endif

/* function prototypes */
#ifdef __STDC__
png_uint_16 _get_png_val (png_byte **pp, int bit_depth);
void store_pixel (xel *pix, png_uint_16 r, png_uint_16 g, png_uint_16 b, png_uint_16 a);
void print_text (png_info *info_ptr);
int iscolor (png_color c);
void convertpng (FILE *ifp);
int main (int argc, char *argv[]);
#endif

enum alpha_handling
  {none, alpha_only, mix};

static int verbose = FALSE;
static float displaygamma = -1.0; /* display gamma */
#ifdef HOMEMADE_GAMMA
static double gamma_correct = 1.0;
static png_uint_16 *gamma_table;
#endif
static enum alpha_handling alpha = none;
static int text = FALSE;
pixval maxval;
png_uint_16 bgr, bgg, bgb; /* background colors */


#define get_png_val(p) _get_png_val (&(p), info_ptr->bit_depth)

#ifdef __STDC__
png_uint_16 _get_png_val (png_byte **pp, int bit_depth)
#else
png_uint_16 _get_png_val (pp, bit_depth)
png_byte **pp;
int bit_depth;
#endif
{
  png_uint_16 c = 0;

  if (bit_depth == 16) {
    c = (*((*pp)++)) << 8;
  }
  c |= (*((*pp)++));

  return c;
}

#ifdef __STDC__
void store_pixel (xel *pix, png_uint_16 r, png_uint_16 g, png_uint_16 b, png_uint_16 a)
#else
void store_pixel (pix, r, g, b, a)
xel *pix;
png_uint_16 r, g, b, a;
#endif
{
  if (alpha == alpha_only) {
    PNM_ASSIGN1 (*pix, a);
  } else {
#ifdef HOMEMADE_GAMMA
    if (displaygamma != -1.0) {
      r = gamma_table[r];
      g = gamma_table[g];
      b = gamma_table[b];
    }
#endif
    if (alpha == mix) {
      r = r * (double)a / maxval + ((1.0 - (double)a / maxval) * bgr);
      g = g * (double)a / maxval + ((1.0 - (double)a / maxval) * bgg);
      b = b * (double)a / maxval + ((1.0 - (double)a / maxval) * bgb);
    }
    PPM_ASSIGN (*pix, r, g, b);
  }
}

#ifdef __STDC__
void print_text (png_info *info_ptr)
#else
void print_text (info_ptr)
png_info *info_ptr;
#endif
{
  int i;
  static char *month[] =
    {"January", "February", "March", "April", "May", "June",
     "July", "August", "September", "October", "November", "December"};

  for (i = 0 ; i < info_ptr->num_text ; i++) {
    pm_message ("\n%s: %.*s", info_ptr->text[i].key,
                (int)info_ptr->text[i].text_length,
                info_ptr->text[i].text);
  }

  if (info_ptr->valid & PNG_INFO_tIME) {
    pm_message ("modification time: %02d %s %d %02d:%02d:%02d",
                info_ptr->mod_time.day, month[info_ptr->mod_time.month],
                info_ptr->mod_time.year, info_ptr->mod_time.hour,
                info_ptr->mod_time.minute, info_ptr->mod_time.second);
  }
}

#ifdef __STDC__
int iscolor (png_color c)
#else
int iscolor (c)
png_color c;
#endif
{
  return c.red != c.green || c.green != c.blue;
}

#ifdef __STDC__
void convertpng (FILE *ifp)
#else
void convertpng (ifp)
FILE *ifp;
#endif
{
  png_struct *png_ptr;
  png_info *info_ptr;
  pixel *row;
  png_byte **png_image;
  png_byte *png_pixel;
  pixel *pnm_pixel;
  int x, y;
  int linesize;
  png_uint_16 c, c2, c3, a;
  int pnm_type;
  int i;
  char *type_string;
  char *alpha_string;
  char gamma_string[80];
  int trans_mix;

  png_ptr = (png_struct *)malloc (sizeof (png_struct));
  info_ptr = (png_info *)malloc (sizeof (png_info));

  if (png_ptr == NULL || info_ptr == NULL)
    pm_error ("Cannot allocate PNGLIB structures");

  if (setjmp (png_ptr->jmpbuf)) {
    png_read_destroy (png_ptr, info_ptr, (png_info *)0);
    free (png_ptr);
    free (info_ptr);
    pm_error ("setjmp returns error condition");
  }

  png_read_init (png_ptr);
  png_info_init (info_ptr);
  png_init_io (png_ptr, ifp);
  png_read_info (png_ptr, info_ptr);

  if (verbose) {
    switch (info_ptr->color_type) {
      case PNG_COLOR_TYPE_GRAY:
        type_string = "gray";
        alpha_string = "";
        break;

      case PNG_COLOR_TYPE_GRAY_ALPHA:
        type_string = "gray";
        alpha_string = "+alpha";
        break;

      case PNG_COLOR_TYPE_PALETTE:
        type_string = "palette";
        alpha_string = "";
        break;

      case PNG_COLOR_TYPE_RGB:
        type_string = "truecolor";
        alpha_string = "";
        break;

      case PNG_COLOR_TYPE_RGB_ALPHA:
        type_string = "truecolor";
        alpha_string = "+alpha";
        break;
    }
    if (info_ptr->valid & PNG_INFO_tRNS) {
      alpha_string = "+transparency";
    }

    if (info_ptr->valid & PNG_INFO_gAMA) {
      sprintf (gamma_string, ", image gamma = %4.2f", info_ptr->gamma);
    } else {
      strcpy (gamma_string, "");
    }

    pm_message ("reading a %d x %d image, %d bit%s %s%s%s%s",
                info_ptr->width, info_ptr->height,
                info_ptr->bit_depth, info_ptr->bit_depth>1 ? "s" : "",
                type_string, alpha_string, gamma_string,
                info_ptr->interlace_type ? ", Adam7 interlaced" : "");
  }

  png_image = (png_byte **)malloc (info_ptr->height * sizeof (png_byte*));
  if (png_image == NULL)
    pm_error ("couldn't alloc space for image");

  if (info_ptr->bit_depth == 16)
    linesize = 2 * info_ptr->width;
  else
    linesize = info_ptr->width;

  if (info_ptr->color_type == PNG_COLOR_TYPE_GRAY_ALPHA)
    linesize *= 2;
  else
  if (info_ptr->color_type == PNG_COLOR_TYPE_RGB)
    linesize *= 3;
  else
  if (info_ptr->color_type == PNG_COLOR_TYPE_RGB_ALPHA)
    linesize *= 4;

  for (y = 0 ; y < info_ptr->height ; y++) {
    png_image[y] = malloc (linesize);
    if (png_image[y] == NULL)
      pm_error ("couldn't alloc space for image");
  }

  if (info_ptr->bit_depth < 8)
    png_set_packing (png_ptr);

  if (info_ptr->color_type == PNG_COLOR_TYPE_PALETTE) {
    maxval = 255;
  } else {
    maxval = (1l << info_ptr->bit_depth) - 1;
  }

  /* sBIT handling is a bit tricky. If we are extracting only the image, we
     can use the sBIT info for grayscale and color images, if the three
     values agree. If we extract the transparency/alpha mask, sBIT is
     irrelevant for trans and valid for alpha. If we mix both, the
     multiplication may result in values that require the normal bit depth,
     so we will use the sBIT info only for transparency, if we know that only
     solid and fully transparent is used */

  if (info_ptr->valid & PNG_INFO_sBIT) {
    switch (alpha) {
      case mix:
        if (info_ptr->color_type == PNG_COLOR_TYPE_RGB_ALPHA ||
            info_ptr->color_type == PNG_COLOR_TYPE_GRAY_ALPHA)
          break;
        if (info_ptr->color_type == PNG_COLOR_TYPE_PALETTE &&
            (info_ptr->valid & PNG_INFO_tRNS)) {
          trans_mix = TRUE;
          for (i = 0 ; i < info_ptr->num_trans ; i++)
            if (info_ptr->trans[i] != 0 && info_ptr->trans[i] != 255) {
              trans_mix = FALSE;
              break;
            }
          if (!trans_mix)
            break;
        }

        /* else fall though to normal case */

      case none:
        if ((info_ptr->color_type == PNG_COLOR_TYPE_PALETTE ||
             info_ptr->color_type == PNG_COLOR_TYPE_RGB ||
             info_ptr->color_type == PNG_COLOR_TYPE_RGB_ALPHA) &&
            (info_ptr->sig_bit.red != info_ptr->sig_bit.green ||
             info_ptr->sig_bit.green != info_ptr->sig_bit.blue) &&
            alpha == none) {
          pm_message ("different bit depths for color channels not supported");
          pm_message ("writing file with %dbit resolution", info_ptr->bit_depth);
        } else {
          if (info_ptr->color_type == PNG_COLOR_TYPE_PALETTE) {
            for (i = 0 ; i < info_ptr->num_palette ; i++) {
              info_ptr->palette[i].red   >>= 8-info_ptr->sig_bit.red;
              info_ptr->palette[i].green >>= 8-info_ptr->sig_bit.red;
              info_ptr->palette[i].blue  >>= 8-info_ptr->sig_bit.red;
            }
          } else {
            png_set_shift (png_ptr, &(info_ptr->sig_bit));
          }
          if (info_ptr->color_type == PNG_COLOR_TYPE_PALETTE ||
              info_ptr->color_type == PNG_COLOR_TYPE_RGB ||
              info_ptr->color_type == PNG_COLOR_TYPE_RGB_ALPHA) {
            pm_message ("image has fewer significant bits, writing file with %d bits per channel", info_ptr->sig_bit.red);
            maxval = (1l << info_ptr->sig_bit.red) - 1;
          } else {
            pm_message ("image has fewer significant bits, writing file with %d bits", info_ptr->sig_bit.gray);
            maxval = (1l << info_ptr->sig_bit.gray) - 1;
          }
        }
        break;

      case alpha_only:
        if (info_ptr->color_type == PNG_COLOR_TYPE_RGB_ALPHA ||
            info_ptr->color_type == PNG_COLOR_TYPE_GRAY_ALPHA) {
          png_set_shift (png_ptr, &(info_ptr->sig_bit));
          pm_message ("image has fewer significant bits, writing file with %d bits", info_ptr->sig_bit.alpha);
          maxval = (1l << info_ptr->sig_bit.alpha) - 1;
        }
        break;

      }
  }

  /* didn't manage to get pnglib to work (bugs?) concerning background */
  /* processing, therefore we do our own using bgr, bgg and bgb        */
  if (info_ptr->valid & PNG_INFO_bKGD)
    switch (info_ptr->color_type) {
      case PNG_COLOR_TYPE_GRAY:
      case PNG_COLOR_TYPE_GRAY_ALPHA:
        bgr = bgg = bgb = info_ptr->background.gray;
        break;
      case PNG_COLOR_TYPE_PALETTE:
        bgr = info_ptr->palette[info_ptr->background.index].red;
        bgg = info_ptr->palette[info_ptr->background.index].green;
        bgb = info_ptr->palette[info_ptr->background.index].blue;
        break;
      case PNG_COLOR_TYPE_RGB:
      case PNG_COLOR_TYPE_RGB_ALPHA:
        bgr = info_ptr->background.red;
        bgg = info_ptr->background.green;
        bgb = info_ptr->background.blue;
        break;
    }
  else
    bgr = bgg = bgb = 0;

  if (displaygamma != -1.0) {
#ifdef HOMEMADE_GAMMA
    if (info_ptr->valid & PNG_INFO_gAMA) {
      gamma_correct = 1.0 / ((double)info_ptr->gamma * (double)displaygamma);
      if (verbose)
        pm_message ("image gamma is %4.2f, converted for display gamma of %4.2f",
                    info_ptr->gamma, displaygamma);
    } else {
      gamma_correct = 1.0 / (1.0 * (double)displaygamma);
      if (verbose)
        pm_message ("image gamma assumed 1.0, converted for display gamma of %4.2f",
                    displaygamma);
    }
    gamma_table = (png_uint_16 *)malloc (maxval * sizeof (png_uint_16));
    for (i = 0 ; i <= maxval ; i++) {
      gamma_table[i] = (png_uint_16) (pow ((double)i / maxval,
                                           gamma_correct) * maxval);
    }
#else
    if (info_ptr->valid & PNG_INFO_gAMA) {
      png_set_gamma (png_ptr, displaygamma, info_ptr->gamma);
      if (verbose)
        pm_message ("image gamma is %4.2f, converted for display gamma of %4.2f",
                    info_ptr->gamma, displaygamma);
    } else {
      png_set_gamma (png_ptr, displaygamma, 1.0);
      if (verbose)
        pm_message ("image gamma assumed 1.0, converted for display gamma of %4.2f",
                    displaygamma);
    }
#endif
  }

  png_read_image (png_ptr, png_image);
  png_read_end (png_ptr, info_ptr);

  if (text)
    print_text (info_ptr);

  if (info_ptr->valid & PNG_INFO_pHYs) {
    float r;
    r = (float)info_ptr->x_pixels_per_unit / info_ptr->y_pixels_per_unit;
    if (r != 1.0) {
      pm_message ("warning - non-square pixels; to fix do a 'pnmscale -%cscale %g'",
                  r < 1.0 ? 'x' : 'y',
                  r < 1.0 ? 1.0 / r : r );
    }
  }

  if ((row = pnm_allocrow (info_ptr->width)) == NULL)
    pm_error ("couldn't alloc space for image");

  if (alpha == alpha_only) {
    if (info_ptr->color_type == PNG_COLOR_TYPE_GRAY ||
        info_ptr->color_type == PNG_COLOR_TYPE_RGB) {
      pnm_type = PBM_TYPE;
    } else
      if (info_ptr->color_type == PNG_COLOR_TYPE_PALETTE) {
        pnm_type = PBM_TYPE;
        if (info_ptr->valid & PNG_INFO_tRNS) {
          for (i = 0 ; i < info_ptr->num_trans ; i++) {
            if (info_ptr->trans[i] != 0 && info_ptr->trans[i] != maxval) {
              pnm_type = PGM_TYPE;
              break;
            }
          }
        }
      } else {
        if (maxval == 1)
          pnm_type = PBM_TYPE;
        else
          pnm_type = PGM_TYPE;
      }
  } else {
    if (info_ptr->color_type == PNG_COLOR_TYPE_GRAY ||
        info_ptr->color_type == PNG_COLOR_TYPE_GRAY_ALPHA) {
      if (info_ptr->bit_depth == 1) {
        pnm_type = PBM_TYPE;
      } else {
        pnm_type = PGM_TYPE;
      }
    } else
    if (info_ptr->color_type == PNG_COLOR_TYPE_PALETTE) {
      pnm_type = PGM_TYPE;
      for (i = 0 ; i < info_ptr->num_palette ; i++) {
        if (iscolor (info_ptr->palette[i])) {
          pnm_type = PPM_TYPE;
          break;
        }
      }
    } else {
      pnm_type = PPM_TYPE;
    }
  }

  if (verbose)
    pm_message ("writing a %s file",
                pnm_type == PBM_TYPE ? "PBM" :
                pnm_type == PGM_TYPE ? "PGM" :
                pnm_type == PPM_TYPE ? "PPM" :
                                               "UNKNOWN!");

  pnm_writepnminit (stdout, info_ptr->width, info_ptr->height, maxval,
                    pnm_type, FALSE);

  for (y = 0 ; y < info_ptr->height ; y++) {
    png_pixel = png_image[y];
    pnm_pixel = row;
    for (x = 0 ; x < info_ptr->width ; x++) {
      c = get_png_val (png_pixel);
      switch (info_ptr->color_type) {
        case PNG_COLOR_TYPE_GRAY:
          store_pixel (pnm_pixel, c, c, c,
                       (info_ptr->valid & PNG_INFO_tRNS) &&
                       c == info_ptr->trans_values.gray ?
                       0 : maxval);
          break;

        case PNG_COLOR_TYPE_GRAY_ALPHA:
          a = get_png_val (png_pixel);
          store_pixel (pnm_pixel, c, c, c, a);
          break;

        case PNG_COLOR_TYPE_PALETTE:
          store_pixel (pnm_pixel, info_ptr->palette[c].red,
                       info_ptr->palette[c].green, info_ptr->palette[c].blue,
                       (info_ptr->valid & PNG_INFO_tRNS) &&
                        c<info_ptr->num_trans ?
                        info_ptr->trans[c] : maxval);
          break;

        case PNG_COLOR_TYPE_RGB:
          c2 = get_png_val (png_pixel);
          c3 = get_png_val (png_pixel);
          store_pixel (pnm_pixel, c, c2, c3,
                       (info_ptr->valid & PNG_INFO_tRNS) &&
                        c  == info_ptr->trans_values.red &&
                        c2 == info_ptr->trans_values.green &&
                        c3 == info_ptr->trans_values.blue ?
                        0 : maxval);
          break;

        case PNG_COLOR_TYPE_RGB_ALPHA:
          c2 = get_png_val (png_pixel);
          c3 = get_png_val (png_pixel);
          a = get_png_val (png_pixel);
          store_pixel (pnm_pixel, c, c2, c3, a);
          break;

        default:
          pm_error ("unknown PNG color type");
      }
      pnm_pixel++;
    }
    pnm_writepnmrow (stdout, row, info_ptr->width, maxval, pnm_type, FALSE);
  }
  png_read_destroy (png_ptr, info_ptr, (png_info *)0);

  free (png_ptr);
  free (info_ptr);
}

#ifdef __STDC__
int main (int argc, char *argv[])
#else
int main (argc, argv)
int argc;
char *argv[];
#endif
{
  FILE *ifp;
  int argn;

  char *usage = "[-verbose] [-gamma value] [-alpha | -mix] [-text] [pngfile]";

  pnm_init (&argc, argv);
  argn = 1;

  while (argn < argc && argv[argn][0] == '-' && argv[argn][1] != '\0') {
    if (pm_keymatch (argv[argn], "-verbose", 2)) {
      verbose = TRUE;
    } else
    if (pm_keymatch (argv[argn], "-gamma", 2)) {
      if (++argn < argc)
        sscanf (argv[argn], "%f", &displaygamma);
      else
        pm_usage (usage);
    } else
    if (pm_keymatch (argv[argn], "-alpha", 2)) {
      alpha = alpha_only;
    } else
    if (pm_keymatch (argv[argn], "-mix", 2)) {
      alpha = mix;
    } else
    if (pm_keymatch (argv[argn], "-text", 2)) {
      text = TRUE;
    } else {
      pm_usage (usage);
    }
    argn++;
  }

  if (argn != argc) {
    ifp = pm_openr (argv[argn]);
    ++argn;
  } else {
    ifp = stdin;
  }
  if (argn != argc)
    pm_usage (usage);

  convertpng (ifp);

  pm_close (ifp);
  pm_close (stdout);
  exit (0);
}
