/****************************************************************************
                          INTERNATIONAL AVS CENTRE
           (This disclaimer must remain at the top of all files)

WARRANTY DISCLAIMER

This module and the files associated with it are distributed free of charge.
It is placed in the public domain and permission is granted for anyone to use,
duplicate, modify, and redistribute it unless otherwise noted.  Some modules
may be copyrighted.  You agree to abide by the conditions also included in
the AVS Licensing Agreement, version 1.0, located in the main module
directory located at the International AVS Center ftp site and to include
the AVS Licensing Agreement when you distribute any files downloaded from 
that site.

The International AVS Centre, University of Manchester, the AVS Consortium and
the individual  submitting the module and files associated with said module
provide absolutely NO WARRANTY OF ANY KIND with respect to this software.  The
entire risk as to the quality and performance of this software is with the
user.  IN NO EVENT WILL The International AVS Centre, University of Manchester,
the AVS Consortium and the individual submitting the module and files
associated with said module BE LIABLE TO ANYONE FOR ANY DAMAGES ARISING FROM
THE USE OF THIS SOFTWARE, INCLUDING, WITHOUT LIMITATION, DAMAGES RESULTING FROM
LOST DATA OR LOST PROFITS, OR ANY SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
DAMAGES.

This AVS module and associated files are public domain software unless
otherwise noted.  Permission is hereby granted to do whatever you like with
it, subject to the conditions that may exist in copyrighted materials. Should
you wish to make a contribution toward the improvement, modification, or
general performance of this module, please send us your comments:  why you
liked or disliked it, how you use it, and most important, how it helps your
work. We will receive your comments at avs@iavsc.org.

Please send AVS module bug reports to avs@iavsc.org.

***************************************************************************/
/*------------------------------------------------------------------------------
 *  pic.c:  Bio-Rad PIC file i/o routines.  Based on PIC file information
 *          from Lance Ladic's confocal microscopy web page:
 *          http://www.cs.ubc.ca/spider/ladic/confocal.html
 * 
 *  AUTHOR: Chris Pudney,
 *			Biomedical Confocal Microscopy Research Centre
 *          Dept. Pharmacology, Univ. of Western Australia, Nedlands, 6907.
 *          cpudney@alphapharm.uwa.edu.au
 *
 *  VERSION: DEC Alpha, OSF/1, cc
 *
 *  STARTED: 27/02/96
 *
 *  UPDATED: 12/03/96
 */


#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "pic.h"

/* Error string */
char	PICerrStr[128];


/*-----------------------------------------------------------------------------
 * Read the header of the PIC file.  Returns 1/0 on success/failure.
 */

int
read_PIC_header(char	*filename,
				PICheader	*hdr)
{
	int	fd;


/*	Open file for reading.
 */
	fd = open(filename, O_RDONLY);
	if (fd == -1)
	{
		(void)sprintf(PICerrStr, "Could not open %s.", filename);
		return 0;

	} /* end if (fd ... */

/*  Read header values.
 */
	hdr->nx = 0;
	if (read(fd, (char *)&hdr->nx, 2) != 2)
	{
		(void)close(fd);
		(void)sprintf(PICerrStr, "Error reading %s.", filename);
		return 0;

	} /* end if (read ... */

	hdr->ny = 0;
	if (read(fd, (char *)&hdr->ny, 2) != 2)
	{
		(void)close(fd);
		(void)sprintf(PICerrStr, "Error reading %s.", filename);
		return 0;

	} /* end if (read ... */

	hdr->nz = 0;
	if (read(fd, (char *)&hdr->nz, 2) != 2)
	{
		(void)close(fd);
		(void)sprintf(PICerrStr, "Error reading %s.", filename);
		return 0;

	} /* end if (read ... */

	hdr->ramp1Min = 0;
	if (read(fd, (char *)&hdr->ramp1Min, 2) != 2)
	{
		(void)close(fd);
		(void)sprintf(PICerrStr, "Error reading %s.", filename);
		return 0;

	} /* end if (read ... */

	hdr->ramp1Max = 0;
	if (read(fd, (char *)&hdr->ramp1Max, 2) != 2)
	{
		(void)close(fd);
		(void)sprintf(PICerrStr, "Error reading %s.", filename);
		return 0;

	} /* end if (read ... */

	hdr->notes = 0;
	if (read(fd, (char *)&hdr->notes, 4) != 4)
	{
		(void)close(fd);
		(void)sprintf(PICerrStr, "Error reading %s.", filename);
		return 0;

	} /* end if (read ... */

	hdr->byteFormat = 0;
	if (read(fd, (char *)&hdr->byteFormat, 2) != 2)
	{
		(void)close(fd);
		(void)sprintf(PICerrStr, "Error reading %s.", filename);
		return 0;

	} /* end if (read ... */

	hdr->n = 0;
	if (read(fd, (char *)&hdr->n, 2) != 2)
	{
		(void)close(fd);
		(void)sprintf(PICerrStr, "Error reading %s.", filename);
		return 0;

	} /* end if (read ... */

	if (read(fd, (char *)hdr->name, 32) != 32)
	{
		(void)close(fd);
		(void)sprintf(PICerrStr, "Error reading %s.", filename);
		return 0;

	} /* end if (read ... */

	hdr->merged = 0;
	if (read(fd, (char *)&hdr->merged, 2) != 2)
	{
		(void)close(fd);
		(void)sprintf(PICerrStr, "Error reading %s.", filename);
		return 0;

	} /* end if (read ... */

	hdr->colour1 = 0;
	if (read(fd, (char *)&hdr->colour1, 2) != 2)
	{
		(void)close(fd);
		(void)sprintf(PICerrStr, "Error reading %s.", filename);
		return 0;

	} /* end if (read ... */

	hdr->fileID = 0;
	if (read(fd, (char *)&hdr->fileID, 2) != 2)
	{
		(void)close(fd);
		(void)sprintf(PICerrStr, "Error reading %s.", filename);
		return 0;

	} /* end if (read ... */

	if (hdr->fileID != FILE_ID)
	{
		(void)close(fd);
		(void)sprintf(PICerrStr, "%s not a Bio-Rad PIC file.", filename);
		return 0;

	} /* end if (hdr ... */

	hdr->ramp2Min = 0;
	if (read(fd, (char *)&hdr->ramp2Min, 2) != 2)
	{
		(void)close(fd);
		(void)sprintf(PICerrStr, "Error reading %s.", filename);
		return 0;

	} /* end if (read ... */

	hdr->ramp2Max = 0;
	if (read(fd, (char *)&hdr->ramp2Max, 2) != 2)
	{
		(void)close(fd);
		(void)sprintf(PICerrStr, "Error reading %s.", filename);
		return 0;

	} /* end if (read ... */

	hdr->colour2 = 0;
	if (read(fd, (char *)&hdr->colour2, 2) != 2)
	{
		(void)close(fd);
		(void)sprintf(PICerrStr, "Error reading %s.", filename);
		return 0;

	} /* end if (read ... */

	hdr->edited = 0;
	if (read(fd, (char *)&hdr->edited, 2) != 2)
	{
		(void)close(fd);
		(void)sprintf(PICerrStr, "Error reading %s.", filename);
		return 0;

	} /* end if (read ... */

	hdr->lens = 0;
	if (read(fd, (char *)&hdr->lens, 2) != 2)
	{
		(void)close(fd);
		(void)sprintf(PICerrStr, "Error reading %s.", filename);
		return 0;

	} /* end if (read ... */

	hdr->mag = 0;
	if (read(fd, (char *)&hdr->mag, 4) != 4)
	{
		(void)close(fd);
		(void)sprintf(PICerrStr, "Error reading %s.", filename);
		return 0;

	} /* end if (read ... */

	hdr->dummy[0] = hdr->dummy[1] = hdr->dummy[2] = 0;
	if (read(fd, (char *)hdr->dummy, 6) != 6)
	{
		(void)close(fd);
		(void)sprintf(PICerrStr, "Error reading %s.", filename);
		return 0;

	} /* end if (read ... */


/*	Close and return.
 */
	(void)close(fd);
	return 1;

} /* end read_PIC_header() */


/*-----------------------------------------------------------------------------
 * Read the data of the PIC file.  Returns 1/0 on success/failure.
 */

int
read_PIC_data(char	*filename,
				PICheader	*hdr,
				unsigned char	*data)
{
	int	fd;
	unsigned long	size;


/*	Open file for reading.
 */
	fd = open(filename, O_RDONLY);
	if (fd == -1)
	{
		(void)sprintf(PICerrStr, "Could not open %s.", filename);
		return 0;

	} /* end if (fd ... */


	/* Check header values */
	size = (unsigned long)hdr->nx * (unsigned long)hdr->ny *
		(unsigned long)hdr->nz;
	if (!hdr->byteFormat)
	{
		(void)close(fd);
		(void)sprintf(PICerrStr, "Can only read byte files.", filename);
		return 0;

	} /* end if (hdr ... */

	if (hdr->merged)
	{
		(void)close(fd);
		(void)sprintf(PICerrStr, "Cannot read merged files.", filename);
		return 0;

	} /* end if (hdr ... */


	/* Skip header */
	if (lseek(fd, PIC_HEADER_LEN, SEEK_SET) == -1L)
	{
		(void)close(fd);
		(void)sprintf(PICerrStr, "Error reading %s.", filename);
		return 0;

	} /* end if (lseek ... */

	
	/* Read data */
	if (read(fd, (char *)data, size) != size)
	{
		(void)close(fd);
		(void)sprintf(PICerrStr, "Error reading %s.", filename);
		return 0;

	} /* end if (read ... */

/*	Close and return.
 */
	(void)close(fd);
	return 1;

} /* end read_PIC_data() */


/*------------------------------------------------------------------------------
 * Write header (hdr) of Bio-Rad PIC file (filename).  Returns 1/0 on
 * success/failure.
 */

int
write_PIC_header(
	char	*filename,
	PICheader	*hdr)
{
	int fd;

/*	Open file for writing.
 */
	fd = open(filename, O_WRONLY | O_CREAT | O_TRUNC , S_IRUSR ^ S_IWUSR);
	if (fd == -1)
	{
		(void)sprintf(PICerrStr, "Could not open %s.", filename);
		return 0;

	} /* end if (fd ... */

/*  Write hdr values.
 */
	if (write(fd, (char *)&hdr->nx, 2) != 2)
	{
		(void)close(fd);
		(void)sprintf(PICerrStr, "Error writing %s.", filename);
		return 0;

	} /* end if (write ... */

	if (write(fd, (char *)&hdr->ny, 2) != 2)
	{
		(void)close(fd);
		(void)sprintf(PICerrStr, "Error writing %s.", filename);
		return 0;

	} /* end if (write ... */

	if (write(fd, (char *)&hdr->nz, 2) != 2)
	{
		(void)close(fd);
		(void)sprintf(PICerrStr, "Error writing %s.", filename);
		return 0;

	} /* end if (write ... */

	if (write(fd, (char *)&hdr->ramp1Min, 2) != 2)
	{
		(void)close(fd);
		(void)sprintf(PICerrStr, "Error writing %s.", filename);
		return 0;

	} /* end if (write ... */

	if (write(fd, (char *)&hdr->ramp2Max, 2) != 2)
	{
		(void)close(fd);
		(void)sprintf(PICerrStr, "Error writing %s.", filename);
		return 0;

	} /* end if (write ... */

	if (write(fd, (char *)&hdr->notes, 4) != 4)
	{
		(void)close(fd);
		(void)sprintf(PICerrStr, "Error writing %s.", filename);
		return 0;

	} /* end if (write ... */

	if (write(fd, (char *)&hdr->byteFormat, 2) != 2)
	{
		(void)close(fd);
		(void)sprintf(PICerrStr, "Error writing %s.", filename);
		return 0;

	} /* end if (write ... */

	if (write(fd, (char *)&hdr->n, 2) != 2)
	{
		(void)close(fd);
		(void)sprintf(PICerrStr, "Error writing %s.", filename);
		return 0;

	} /* end if (write ... */

	if (write(fd, (char *)hdr->name, 32) != 32)
	{
		(void)close(fd);
		(void)sprintf(PICerrStr, "Error writing %s.", filename);
		return 0;

	} /* end if (write ... */

	if (write(fd, (char *)&hdr->merged, 2) != 2)
	{
		(void)close(fd);
		(void)sprintf(PICerrStr, "Error writing %s.", filename);
		return 0;

	} /* end if (write ... */

	if (write(fd, (char *)&hdr->colour1, 2) != 2)
	{
		(void)close(fd);
		(void)sprintf(PICerrStr, "Error writing %s.", filename);
		return 0;

	} /* end if (write ... */

	if (write(fd, (char *)&hdr->fileID, 2) != 2)
	{
		(void)close(fd);
		(void)sprintf(PICerrStr, "Error writing %s.", filename);
		return 0;

	} /* end if (write ... */

	if (write(fd, (char *)&hdr->ramp2Min, 2) != 2)
	{
		(void)close(fd);
		(void)sprintf(PICerrStr, "Error writing %s.", filename);
		return 0;

	} /* end if (write ... */

	if (write(fd, (char *)&hdr->ramp2Max, 2) != 2)
	{
		(void)close(fd);
		(void)sprintf(PICerrStr, "Error writing %s.", filename);
		return 0;

	} /* end if (write ... */

	if (write(fd, (char *)&hdr->colour2, 2) != 2)
	{
		(void)close(fd);
		(void)sprintf(PICerrStr, "Error writing %s.", filename);
		return 0;

	} /* end if (write ... */

	if (write(fd, (char *)&hdr->edited, 2) != 2)
	{
		(void)close(fd);
		(void)sprintf(PICerrStr, "Error writing %s.", filename);
		return 0;

	} /* end if (write ... */

	if (write(fd, (char *)&hdr->lens, 2) != 2)
	{
		(void)close(fd);
		(void)sprintf(PICerrStr, "Error writing %s.", filename);
		return 0;

	} /* end if (write ... */

	if (write(fd, (char *)&hdr->mag, 4) != 4)
	{
		(void)close(fd);
		(void)sprintf(PICerrStr, "Error writing %s.", filename);
		return 0;

	} /* end if (write ... */

	if (write(fd, (char *)hdr->dummy, 6) != 6)
	{
		(void)close(fd);
		(void)sprintf(PICerrStr, "Error writing %s.", filename);
		return 0;

	} /* end if (write ... */


/*	Close and return.
 */
	(void)close(fd);
	return 1;


} /* end write_PIC_header() */


/*------------------------------------------------------------------------------
 * Write data (data) of Bio-Rad PIC file (filename).  Header (hdr)
 * information is used.  Returns 1/0 on success/failure.
 */

int
write_PIC_data(
	char	*filename,
	PICheader	*hdr,
	unsigned char	*data)
{
	int	fd;
	unsigned	size;


/*	Open file and move file pointer to image slice and write slice.
 */
	fd = open(filename, O_WRONLY);
	if (fd == -1)
	{
		(void)sprintf(PICerrStr, "Could not open %s.", filename);
		return 0;

	} /* end if (fd ... */

	if (lseek(fd, PIC_HEADER_LEN, SEEK_SET) == -1L)
	{
		(void)close(fd);
		(void)sprintf(PICerrStr, "Error writing %s.", filename);
		return 0;

	} /* end if (lseek ... */

	size = (unsigned long)hdr->nx * (unsigned long)hdr->ny *
		(unsigned long)hdr->nz;
	if (write(fd, (char *)data, size) != size)
	{
		(void)close(fd);
		(void)sprintf(PICerrStr, "Error writing %s.", filename);
		return 0;

	} /* end if (write ... */

/*	Close file and return.
 */
	(void)close(fd);
	return 1;

} /* end write_PIC_data() */
