/****************************************************************************
                  INTERNATIONAL AVS CENTER
	(This disclaimer must remain at the top of all files)

WARRANTY DISCLAIMER

This module and the files associated with it are distributed free of charge.
It is placed in the public domain and permission is granted for anyone to use,
duplicate, modify, and redistribute it unless otherwise noted.  Some modules
may be copyrighted.  You agree to abide by the conditions also included in
the AVS Licensing Agreement, version 1.0, located in the main module
directory located at the International AVS Center ftp site and to include
the AVS Licensing Agreement when you distribute any files downloaded from 
that site.

The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module provide absolutely
NO WARRANTY OF ANY KIND with respect to this software.  The entire risk as to
the quality and performance of this software is with the user.  IN NO EVENT
WILL The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module BE LIABLE TO
ANYONE FOR ANY DAMAGES ARISING FROM THE USE OF THIS SOFTWARE, INCLUDING,
WITHOUT LIMITATION, DAMAGES RESULTING FROM LOST DATA OR LOST PROFITS, OR ANY
SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES.

This AVS module and associated files are public domain software unless
otherwise noted.  Permission is hereby granted to do whatever you like with
it, subject to the conditions that may exist in copyrighted materials. Should
you wish to make a contribution toward the improvement, modification, or
general performance of this module, please send us your comments:  why you
liked or disliked it, how you use it, and most important, how it helps your
work. We will receive your comments at avs@ncsc.org.

Please send AVS module bug reports to avs@ncsc.org.

******************************************************************************/
/* 
 * Module Name   read DICOM (Input) (Subroutine) 
 * Author:       Zachary DelProposto / zsd@umich.edu
 *				 inspired by ACR NEMA SPI, by MARCHETTI Andrea
 *
 * Date Created  March 2002
 *                       
 *                                                          
 * output 0 "image" field 2D scalar short 
 * param  0 "filename" browser "" "" ":"                     
 * End of Module Description Comments                       
 */

 
 
/*


	NOTE: 	no attempt has been made to make this biendian. This currently is only 
			guaranteed to work on little-endian architectures.
			
			If you have a big-endian architectured machine, implementing the appropriate 
			byte-swapping macros (check them first!) in the get_VRxxx() functions will
			allow them to return the correct values.
			
			Additionally, the DICOM elements are currently in little-endian format; these
			will have to be byte-swapped on a big-endian machine.
			
			Only bit-allocations of 8 and 16 are supported. Bit stored should not matter,
			as long as in an allocation of 8 or 16. 
			
			If slope/intercepts are present, the data will be rescaled using the linear equation
			y = mx+b, where x is the recorded voxel value, and y is the (newly) rescaled voxel value.
			The slope/intercepts are in Phillips MR DICOM files, and possibly others.
			
			The fields needed are extracted and put in the IMAGE_DATA struct. Fields here can be 
			output as ports as needed.
			
			This seems to work pretty well, but if you encounter any problems don't hesitate to ask.


VR types explained
	
	DS : floating-point, as string
	DA : Date yyyymmdd, as string
	TM : Time, ???, as string
	IS : Integer, as string
	US : unsigned 2 byte integer, little endian
	SS : signed 2 byte integer, little endian
	UI : Unique Identifier, as string
	
	
*/

 
 
 
#include <stdio.h>
#include <avs/avs.h>
#include <avs/port.h>
#include <avs/field.h>

#include <stdio.h>
#include <string.h>
#include <strings.h>
#include <stdlib.h>

/* Endian byte-swapping macros */
/*
#define LOWBYTE(w)  (  (w)    &0x00FF )
#define HIGHBYTE(w) ( ((w)>>8)&0x00FF )
#define SWAPWORD(w) ( (LOWBYTE(w)<<8) + HIGHBYTE(w) )
#define FIRST(dw)   ( (dw)     &0x000000FF)
#define SECOND(dw)  (((dw)>> 8)&0x000000FF)
#define THIRD(dw)   (((dw)>>16)&0x000000FF)
#define FORTH(dw)   (((dw)>>24)&0x000000FF)
#define SWAPDOUBLEWORD(dw) ( FORTH(dw) + (THIRD(dw)<<8) + (SECOND(dw)<<16) + (FIRST(dw)<<24) )
*/


// little-endian DICOM types
#define OB 0x424F
#define OW 0x574F
#define SQ 0x5153
#define UN 0x4E55
#define UT 0x5455 


// to determine if file is valid DICOM file
#define DICM_POS 128
char DICM_TEXT[] = {'D','I','C','M'};
#define DICM_TEXT_LENGTH 4


// where to start reading (128 = DICM, +4)
#define DICM_START_POS  132


// Data Structures
typedef struct {
	short element;			// element ID
	short group;			// group ID (subtype of element)
	short type;				// data type
	short reserved;			// some data types have a reserved field
	unsigned int length;	// length of data
	long dataPos;			// absolute position in file where *data starts
	unsigned char *data;	// data (raw bytes); if NOT NULL, it is NULL TERMINATED and length is length+1
	char *label;			// label (if assigned)
	char *error;			// error message (NULL if none)
} DICOM_DATA_ELEMENT;


// data required to read / convert an image.
typedef struct {
	unsigned short rows;
	unsigned short cols;
	unsigned short bits_allocated;
	unsigned short bits_stored;
	unsigned short high_bit;
	double slope;
	double intercept;
	short toRescale;			// 1 if we should, 0 if we shouldn't
	int pixel_start;
	double pixel_spacing_x;
	double pixel_spacing_y;
	double slice_thickness;
	double flip_angle;
	FILE *file;
	char *filename;
} IMAGE_DATA;


// function declarations
DICOM_DATA_ELEMENT getNextDataElement(FILE*, int);
int isDICOMFile(FILE*);
void clearImageData(IMAGE_DATA*);
void clearDicomDataElement(DICOM_DATA_ELEMENT*);
unsigned short getVR_US(unsigned char*, unsigned int);
short getVR_SS(unsigned char*, unsigned int);
double getVR_DS(unsigned char*, unsigned int);
void setImageData(IMAGE_DATA*, DICOM_DATA_ELEMENT*);
int readImage(IMAGE_DATA, AVSfield_short**);
void getVR_DS_x2(double*, double*, unsigned char*, unsigned int);


// big-endian DICOM types [unused]
// to make these little-endian, the bytes need to be swapped.
/*
#define AE 0x4145
#define AS 0x4153
#define AT 0x4154
#define CS 0x4353
#define DA 0x4441
#define DS 0x4453
#define DT 0x4454
#define FD 0x4644
#define FL 0x464C
#define IS 0x4953
#define LO 0x4C4F
#define LT 0x4C54
#define PN 0x504E
#define SH 0x5348
#define SL 0x534C
#define SS 0x5353
#define ST 0x5354
#define TM 0x544D
#define UI 0x5549
#define UL 0x554C
#define US 0x5553
#define UT 0x5554
#define OB 0x4F42
#define OW 0x4F57
#define SQ 0x5351
#define UN 0x554E
#define QQ 0x3F3F
*/







/* *****************************************/
/*  Module Description                     */
/* *****************************************/
int read_DICOM_desc()
{
	int param;
	extern int read_DICOM_compute();
	
	AVSset_module_name("read DICOM", MODULE_DATA);
	
	/* Output Port Specifications              */
	AVScreate_output_port("image", "field 2D scalar uniform short");
	AVScreate_output_port("pixel_spacing_X", "field 1D scalar real");
	AVScreate_output_port("pixel_spacing_Y", "field 1D scalar real");
	AVScreate_output_port("slice_thickness", "field 1D scalar real");
	
	
	
	/* Parameter Specifications                */
	param = AVSadd_parameter("DICOM file", "string", 0, 0, ":");
	AVSconnect_widget(param, "browser");
	
	AVSset_compute_proc(read_DICOM_compute);
	
	return(1);
}// read_DICOM_desc()





/* *****************************************/
/* Module Compute Routine                  */
/* *****************************************/
int read_DICOM_compute(	/*output ports*/ AVSfield_short **image, AVSfield_float **psX, AVSfield_float **psY, AVSfield_float **slice_thickness, 
						/* input */char *filename)
{
	long pos;
	int dims[2];
	AVSfield_float *my_psx, *my_psy, *my_st;
	IMAGE_DATA imageData;
	DICOM_DATA_ELEMENT dde;
	
	/* No data file name */
	if(filename == NULL)
	{
		return(1);
	}
	
	// clear ImageData struct
	clearImageData(&imageData);
	
	// set filename
	imageData.filename = (char *) malloc(strlen(filename) * sizeof(char));
	strcpy(imageData.filename, filename);
	
	// attempt to open file
	imageData.file = fopen(filename,"r");
	if(imageData.file == NULL)
	{ 
		AVSerror("Can't open data file %s",filename);
		return(0);
	}
	
	// is file dicom?
	if( isDICOMFile(imageData.file) != 1 )
	{
		AVSerror("Not a DICOM file (missing DICM @ offset 128)");
		fclose(imageData.file);
		return(0);
	}
	
	// read in tags, until we've read them all
	pos = DICM_START_POS;
	while(!feof(imageData.file) && imageData.pixel_start <= 0)
	{
		clearDicomDataElement(&dde);
		dde = getNextDataElement(imageData.file, pos);
		
		if(dde.error != NULL)
		{
			AVSerror("Error reading DICOM group: %s", dde.error);
			fclose(imageData.file);
			return(0);
		}
		
		// transfer needed DDE data to ImageData
		setImageData(&imageData, &dde);
		
		// print group info
		printf("(%.4x-%.4x) type: %.4x length: %u ", dde.element, dde.group, dde.type, dde.length);
		
		// don't print if too long.
		if(dde.length < 80)
		{
			printf(" \"%s\"\n", dde.data);
		}
		else
		{
			printf(" [too long: %d bytes; not printed.]\n", dde.length);
		}
		
		// free memory
		if(dde.data != NULL)
		{
			free(dde.data);
		}
		
		if(dde.error != NULL)
		{
			free(dde.error);
		}
		
		if(dde.label != NULL)
		{
			free(dde.label);
		}
		
		// get next position
		pos = ftell(imageData.file);		
	}
	
	
	if(checkImageStruct(imageData) == 0)
	{
		return(0);
	}
	
	// read the image
	readImage(imageData, image);
 	
	
	// free other output ports
	if(*psX != NULL)
	{
		AVSfield_free( (AVSfield*) psX); 
	}
 
	if(*psY != NULL)
	{
		AVSfield_free( (AVSfield*) psY); 
	}
 	
	if(*slice_thickness != NULL)
	{
		AVSfield_free( (AVSfield*) slice_thickness); 
	}
 	
	// allocate other output port data
	dims[0] = 1;
	dims[1] = 1;
	my_psx = (AVSfield_float *) AVSdata_alloc("field 1D scalar real", dims);
	my_psy = (AVSfield_float *) AVSdata_alloc("field 1D scalar real", dims);
	my_st = (AVSfield_float *) AVSdata_alloc("field 1D scalar real", dims);
	
	// set other output ports
	*my_psx->data = (float) imageData.pixel_spacing_x;
	*my_psy->data = (float) imageData.pixel_spacing_y;
	*my_st->data = (float)  imageData.slice_thickness;
	
	*psX = my_psx;
	*psY = my_psy;
	*slice_thickness = my_st;
	
	// cleanup, and exit
	fclose(imageData.file);
	return(1);
}// read_DICOM_compute()




/* ***********************************************************************/
/* Initialization for modules contained in this file.                    */
/* ***********************************************************************/
static int ((*mod_list[])()) = {
   read_DICOM_desc
};
#define NMODS (sizeof(mod_list) / sizeof(char *))

AVSinit_modules()
{
   AVSinit_from_module_list(mod_list, NMODS);
}







// returns 0 if fail, 1 if ok (AVS consistent)
int checkImageStruct(IMAGE_DATA imageData)
{
	// for debugging
	/*
	printf("imageData ---------------\n");
	printf("   rows = %d\n", imageData.rows);
	printf("   cols = %d\n", imageData.cols);
	printf("   bits_allocated = %d\n", imageData.bits_allocated);
	printf("   bits_stored = %d\n", imageData.bits_stored);
	printf("   high_bit = %d\n", imageData.high_bit);
	printf("   pixel_start = %d\n", imageData.pixel_start);
	printf("   slope = %f\n", imageData.slope);
	printf("   intercept = %f\n", imageData.intercept);
	printf("   toRescale = %d\n", imageData.toRescale);
	printf("   filename: %s\n",imageData.filename);
	printf("   pixel_spacing_x = %f\n", imageData.pixel_spacing_x);
	printf("   pixel_spacing_y = %f\n", imageData.pixel_spacing_y);
	printf("   slice_thickness = %f\n", imageData.slice_thickness);
	printf("   flip_angle = %f\n", imageData.flip_angle);
	*/
	
	
	if(imageData.rows <= 0 || imageData.cols <= 0)
	{
		AVSerror("DICOM: rows/cols <= 0 or never defined\n");
		return 0;
	}
	
	
	if(imageData.bits_allocated <= 0 || imageData.bits_stored <= 0 || imageData.high_bit <= 0)
	{
		AVSerror("DICOM: bit storage/allocation/endianness undefined\n");
		return 0;
	}
	
	if(imageData.pixel_start <= 0)
	{
		AVSerror("DICOM: pixel start location undefined\n");
		return 0;
	}
	
	return 1;
}// checkImageStruct()



int readImage(IMAGE_DATA imageData, AVSfield_short **image)
{
   int offset;
   int n, size;
   unsigned char *charBuffer;
   unsigned short *shortBuffer;
   int   dims0[2];
   AVSfield_short *avsShortImage;
    
 	if(*image != NULL)
	{
		AVSfield_free( (AVSfield*) image); 
	}
  
  
   /* Allocate space for new field output */
   dims0[0] = imageData.cols;
   dims0[1] = imageData.rows;
   avsShortImage = (AVSfield_short *) AVSdata_alloc("field 2D scalar uniform short", dims0);
   if(avsShortImage == NULL) 
   {
	  AVSerror("Allocation of image output field failed.");
	  return(0);
   }
   
   
   // set file position to beginning of pixel data
   n = fseek(imageData.file, imageData.pixel_start, SEEK_SET);
   if(n != 0)
   {
	   AVSerror("Can't seek to position %d in DICOM file!", imageData.pixel_start);
	   return(0);
   }
   
   
   
   if(imageData.bits_allocated == 8 && imageData.high_bit != 0)
   {
	   size = imageData.rows * imageData.cols;
	   charBuffer = (unsigned char *) malloc(size * sizeof(unsigned char));

	   if(charBuffer==NULL)
	   {
		   AVSerror("Allocation of image field failed"); 
		   return(0);
	   }
	   
	   
	   fread(charBuffer, sizeof(unsigned char), size, imageData.file);
	   
	   
	   if(n = ferror(imageData.file))
	   {
		   clearerr(imageData.file);
		   AVSerror("File error: %d", n);
		   return(0);
	   }
	   
	   for(offset = 0; offset<size; offset++)
	   {
		   if(imageData.toRescale == 1)
		   {
			   avsShortImage->data[offset] = (short) ((imageData.slope * charBuffer[offset]) + imageData.intercept);
		   }
		   else
		   {
			   avsShortImage->data[offset] = (short) charBuffer[offset];
		   }
	   }
	   
	   free(charBuffer);
   }
   else if(imageData.bits_allocated == 16 && imageData.high_bit != 0)
   {
	   size = imageData.rows * imageData.cols;
	   shortBuffer = (short *) malloc(size * sizeof(short));
	   
	   if(shortBuffer==NULL)
	   {
		   AVSerror("Allocation of image field failed"); 
		   return(0);
	   }
	   
	   fread(shortBuffer, sizeof(short), size, imageData.file);
	   
	   if(n = ferror(imageData.file))
	   {
		   clearerr(imageData.file);
		   AVSerror("File error: %d", n);
		   return(0);
	   }

	   
	   for(offset = 0; offset<size; offset++)
	   {
		   if(imageData.toRescale == 1)
		   {
			   avsShortImage->data[offset] = (short) ((imageData.slope * shortBuffer[offset]) + imageData.intercept);
		   }
		   else
		   {
			   avsShortImage->data[offset] = shortBuffer[offset];
	   	   }
	   }
	   free(shortBuffer);
   }  
   else
   {
	   AVSerror("Bit allocation / Hibit error (%hd / %hd)", imageData.bits_allocated, imageData.high_bit);
	   return(0);
   }
   
   // assign image pointer
   *image = avsShortImage;
   
   return(1);
}// readImage()


/* checks if file is a valid DICOM file. 
* returns 1 if true, 0 if false or error
*/
int isDICOMFile(FILE *file)
{
	int i, c, err;
	long oldPos = ftell(file);
	if(oldPos < 0)
	{
		return 0;
	}
	
	
	err = fseek(file, DICM_POS, SEEK_SET);
	if(err !=  0)
	{
		return 0;
	}
	
	for(i=0; i<DICM_TEXT_LENGTH; i++)
	{
		if( fgetc(file) != (int) DICM_TEXT[i] )
		{
			return 0;
		}
	}
	
	fseek(file, oldPos, SEEK_SET);
	
	return 1;
}// isDICOMFile
	


/*

	gets the next data element from the file
	starts at position 'p' (if p>=0);
	returns NULL if p<0
	NOTE: position is absolute
*/
DICOM_DATA_ELEMENT getNextDataElement(FILE *file, int pos)
{
	// variables
	int err;
	short slen;
	char *errorMsg = "Dicom Group read failure.";
	DICOM_DATA_ELEMENT elem;
	
	// preconditions
	if(pos < 0 || file == NULL)
	{
		elem.error = errorMsg;
		return elem;
	}
	
	// setup DDE
	clearDicomDataElement(&elem);
	
	// seek to position
	err = fseek(file, pos, SEEK_SET);
	if(err != 0)
	{
		elem.error = errorMsg;
			  printf("1 >> err = %d\n",err);
		return elem;
	}
	
	
	err = fread(&elem.element, sizeof(elem.element), 1, file);
	err += fread(&elem.group, sizeof(elem.group), 1, file);
	err += fread(&elem.type, sizeof(elem.type), 1, file);
	if(err != 3)
	{
		elem.error = errorMsg;
		return elem;
	}
	
	
	// EXPLICIT or IMPLICIT type? If EXPLICIT, there is a 16 bit reserved field
	// followed by a 32 bit length field, instead of no reserved field and 16 bit
	// length field.
	if(elem.type == OB || elem.type == OW || elem.type == SQ || elem.type == UN || elem.type == UT)
	{
		  // next 16 bits is reserved, and should be 0x0000
		  err = fread(&elem.reserved, sizeof(elem.reserved), 1, file);
		  
		  // but the length is 32 bits, unsigned
		  err += fread(&elem.length, sizeof(unsigned int), 1, file);
		  
		  if(err != 2)
		  {
			  elem.error = errorMsg;
			  return elem;
		  }
	}
	else
	{
		elem.reserved = (short) 0x0000;
		
		// read 16-bit length
		slen = 0;
		elem.length = 0;
		err = fread(&slen, sizeof(slen), 1, file);
		elem.length = (unsigned int) slen;
		
		if(err != 1)
		{
			  elem.error = errorMsg;
			  return elem;
		}
	}
	
	// set data position
	elem.dataPos = ftell(file);
	
	// allocate. Note that we null-terminate, so 1 more byte is allocated.
	elem.data = (unsigned char *) malloc(elem.length + 1);
	if(elem.data == NULL)
	{
		elem.error = errorMsg;
		fprintf(stderr, "Can't allocate %d bytes for element.\n", (elem.length+1));
		return elem;
	}
	*(elem.data + elem.length) = '\0';
	
	
	// read data. NOTE that we only read in 'length' bytes. last byte is a null, for termination of strings.
	if(elem.length > 0)
	{
		err = fread(elem.data, elem.length, 1, file);
		if(err != 1)
		{
			elem.error = errorMsg;
			return elem;
		}
	}
	
	return elem;
}// getNextDataElement()



/*

	Given an IMAGE_DATA struct, we look at the Dicom Data Element
	to see if has 'important' fields (contained in the ImageData
	struct). If so, we set the image data appropriately.


*/
void setImageData(IMAGE_DATA *imageData, DICOM_DATA_ELEMENT *dde)
{
	switch(dde->element)
	{
		case 0x0018:
			switch(dde->group)
			{
				case 0x0050:	// slice thickness
					imageData->slice_thickness = getVR_DS(dde->data, dde->length);
					break;
					
				case 0x1314:	// flip angle
					imageData->flip_angle = getVR_DS(dde->data, dde->length);
					break;
			}
			break;
		
		case 0x0028:
			switch(dde->group)
			{
				case 0x0010:	// rows
					imageData->rows = getVR_US(dde->data, dde->length);
					break;
					
				case 0x0011:	// cols
					imageData->cols = getVR_US(dde->data, dde->length);
					break;
					
				case 0x0030:	// pixel spacing [double \ double]
					getVR_DS_x2(&imageData->pixel_spacing_x, &imageData->pixel_spacing_y,
								dde->data, dde->length);
					break;
					
				case 0x0100: 	// bits allocated
					imageData->bits_allocated = getVR_US(dde->data, dde->length);
					break;
				
				case 0x0101:	// bits stored
					imageData->bits_stored = getVR_US(dde->data, dde->length);
					break;
				
				case 0x0102: 	// high bit
					imageData->high_bit = getVR_US(dde->data, dde->length);
					break;
				
				
				case 0x1052:	// rescale intercept
					imageData->intercept = getVR_DS(dde->data, dde->length);
					imageData->toRescale = 1;
					break;
				
				case 0x1053:	// rescale slope
					imageData->slope = getVR_DS(dde->data, dde->length);
					imageData->toRescale = 1;
					break;
			}
			break;
		
		
		case 0x7FE0:
			switch(dde->group)
			{
				case 0x0010:	// pixel start	
					// this is really a marker in the file for position; so the current byte-offset
					// in the file is the value we want here. 
					if(imageData->pixel_start <= 0)
					{
						imageData->pixel_start = dde->dataPos;
					}
					break;
			}
			break;
			
		default:
			break;
	}
}// setImageData()




// convert data (2 bytes + null) into unsigned short; little-endian format
unsigned short getVR_US(unsigned char *data, unsigned int length)
{
	unsigned short uShort = 0;
	
	if(length == 2)
	{
		uShort = ((*(data+1)) << 8) + *(data+0);
	}
	
	return uShort;
}// getVR_US()



// convert data (2 bytes + null) into signed short; little-endian format
short getVR_SS(unsigned char *data, unsigned int length)
{
	short sShort = -1;
	
	if(length == 2)
	{
		sShort = ((*(data+1)) << 8) + *(data+0);
	}
	
	return sShort;
}// getVR_SS()



// a floating point value, from a string.
double getVR_DS(unsigned char *data, unsigned int length)
{
	if(length == 0)
	{
		return 0.0;
	}
	
	return atof(data);
}// getVR_DS()


// for string values in float\float format (seperated by a '\'), return first
// value in d1, and second in d2.
void getVR_DS_x2(double *d1, double *d2,unsigned char *data, unsigned int length)
{
	char *pEnd;
	
	if(length == 0)
	{
		*d1 = 0.0;
		*d2 = 0.0;
	}
	
	// I hate C
	*d1 = strtod(data, &pEnd);
	pEnd++;
	*d2 = atof(pEnd);
}// getVR_DS_x2() 



// reset image data struct
// DOES NOT free any memory
void clearImageData(IMAGE_DATA *imageData)
{
	imageData->rows = 0;
	imageData->cols = 0;
	imageData->bits_allocated = 0;
	imageData->bits_stored = 0;
	imageData->high_bit = 0;
	imageData->pixel_start = -1;
	
	imageData->pixel_spacing_x = -1.0;
	imageData->pixel_spacing_y = -1.0;
	imageData->slice_thickness = -1.0;
	imageData->flip_angle = 0.0;
	
	imageData->toRescale = 0;
	imageData->slope = 0.0;
	imageData->intercept = 0.0;
	
	imageData->file = NULL;
	imageData->filename = NULL;
}// clearImageData


void clearDicomDataElement(DICOM_DATA_ELEMENT *dde)
{
	dde->element = 0;
	dde->group = 0;
	dde->type = 0;
	dde->reserved = 0;
	dde->length = 0;
	dde->dataPos = 0L;
	dde->data = NULL;
	dde->label = NULL;
	dde->error = NULL;
}// clearDicomDataElement


