/****************************************************************************
                  INTERNATIONAL AVS CENTER
	(This disclaimer must remain at the top of all files)

WARRANTY DISCLAIMER

This module and the files associated with it are distributed free of charge.
It is placed in the public domain and permission is granted for anyone to use,
duplicate, modify, and redistribute it unless otherwise noted.  Some modules
may be copyrighted.  You agree to abide by the conditions also included in
the AVS Licensing Agreement, version 1.0, located in the main module
directory located at the International AVS Center ftp site and to include
the AVS Licensing Agreement when you distribute any files downloaded from 
that site.

The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module provide absolutely
NO WARRANTY OF ANY KIND with respect to this software.  The entire risk as to
the quality and performance of this software is with the user.  IN NO EVENT
WILL The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module BE LIABLE TO
ANYONE FOR ANY DAMAGES ARISING FROM THE USE OF THIS SOFTWARE, INCLUDING,
WITHOUT LIMITATION, DAMAGES RESULTING FROM LOST DATA OR LOST PROFITS, OR ANY
SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES.

This AVS module and associated files are public domain software unless
otherwise noted.  Permission is hereby granted to do whatever you like with
it, subject to the conditions that may exist in copyrighted materials. Should
you wish to make a contribution toward the improvement, modification, or
general performance of this module, please send us your comments:  why you
liked or disliked it, how you use it, and most important, how it helps your
work. We will receive your comments at avs@ncsc.org.

Please send AVS module bug reports to avs@ncsc.org.

******************************************************************************/
/* 
 * AVS Module: track2d = Track 2D cursor motion over displayed image.
 * " Display Tracker 2D "
 *
 * Author:  I. Curington
 *          Stardent Computer Ltd.
 *          Guildford, Surrey, UK
 * 
 * Inputs an image and allows the user create mouse clicks and moves
 * in the image window, with event information passed to other
 * AVS modules using the "upstream data" concept.
 * This module acts similarly to "display image" or
 * "display tracker" in that it accepts an input image and
 * displays this in an X-window. It also then accepts
 * asynchronous mouse events in the window, and passes these
 * mouse events ( in an upstream transform structure ) to
 * an output port. This allows applications to be built using
 * this module to do all the X dependent code, leaving the
 * actual operation (such as pan, draw, etc.) to an upstream
 * module, with no concern of X events.
 * Track 2d will either work silently, or draw an optional
 * cursor in the window at each event. Drawing the cursor
 * is significantly slower than the silent mode.
 *
 * Input  Port: field 2D 4-vector byte (image)
 * Output Port: upstream transform
 * Parameters:  toggle: visible cursor (on/off)
 *
 * revision:  20 February 91  Original  (ianc)
 *             4 March    91  added cursor  (ianc)
 *             5 March    91  auto resize on new image (ianc)
 *            13 July     92  psuedo col visuals, dithering
 */

#include <stdio.h>
/* IAC CODE CHANGE : #include <avs/avs_math.h> */
#include <avs/avs_math.h>
#include <avs/flow.h>
#include <avs/field.h>
#include "X11/Xlib.h"
#include "X11/Xutil.h"
#include <avs/lui.h>
#include <avs/udata.h>

/******************************************************************************/
#define TRUE 1

typedef struct trackwin {
    Window         window;
    int            width, height;
    XImage     *image;
    GC		   gc;
    int		   box_drawn;
    upstream_transform *output;
    unsigned long  *zoom_buf;
    int		   zoom_size;
    int        curviz;
} TRACKWIN;

TRACKWIN trackwin;

static AVSfield_char *curr_input;

/******************************************************************************/
main(argc, argv)
    int argc;
    char *argv[];
{
    int trackwin_desc();
    XEvent event;
    static AVSfield_char *input = NULL;
    TRACKWIN *tp = &trackwin;
    int mask;
    int x, tmpviz;
    char *dither_technique;

    AVScorout_init(argc, argv, trackwin_desc);

    trackwin_init();

    /* Make this module run synchronously with the flow exec */
    AVScorout_set_sync(1);

    while (TRUE) {
	  mask = COROUT_WAIT;
	  x = AVScorout_X_wait(LUI_Display,NULL,&mask);

	  if (mask)
      {
	    AVScorout_input(&input,&tmpviz, &dither_technique);
	    curr_input = input;
	    if (AVSinput_changed("Image Input", 0)) 
		{
		  trackwin_compute(input, dither_technique);
	    }
        if ( tp->curviz == 1 && tmpviz == 0 ) remove_cross();
        tp->curviz = tmpviz;
	  }
	    
	  if (x)
      {
	    /*  Process LUI events */
	    while (XPending(LUI_Display))
        {
		  XNextEvent(LUI_Display, &event);
		  LUI_EventDispatch(&event);
	    }
	  }
    }
}

/******************************************************************************/
trackwin_desc()
{
    int port;

    AVSset_module_name("display tracker 2d", MODULE_RENDER);
    AVScreate_input_port("Image Input", "field 2D 4-vector byte", REQUIRED);

    port = AVScreate_output_port("Upstream Transform",
                                 "struct upstream_transform");

    AVSadd_parameter("Visible Cursor", "boolean", 0,0,0);

    AVSadd_parameter("DitherTechnique", "choice", "dither",
            "dither!floyd steinberg!random!monochrome!none", "!");

    AVSset_output_class(port,"track:upstream_transform");
    AVSload_user_data_types("/usr/avs/include/udata.h");
}

/******************************************************************************/
 static
trackwin_init()
{
    LUI_SetDefaultPreferences();
    LUI_Initialize("AVS display tracker 2d", NULL);

    trackwin.gc = XCreateGC(LUI_Display,LUI_RootWindow,0,0);
    XSetFunction(LUI_Display,trackwin.gc,GXxor);
    trackwin.box_drawn = 0;
    trackwin.output = NULL;
    trackwin.zoom_size = 0;
    trackwin.zoom_buf = (unsigned long *)malloc(1);
}

/******************************************************************************/

trackwin_process(window,event) 
Window window;
XEvent *event;
{
   int d, mask, b_state=BUTTON_UP;
   int x, y;

   TRACKWIN *tp = &trackwin;
   switch (event->type) {
     case Expose:
	   if (event->xexpose.count == 0)
	         trackwin_compute(curr_input);
       b_state=BUTTON_UP;
       break;
     case ButtonPress:
       XQueryPointer(LUI_Display,
             window, &d, &d, &d, &d, &x, &y, &mask);
       b_state=BUTTON_DOWN;
       send_matrix(tp,x,y,mask,b_state);
       break;
     case ButtonRelease:
       XQueryPointer(LUI_Display,
             window, &d, &d, &d, &d, &x, &y, &mask);
       b_state=BUTTON_DOWN;
	   send_matrix(tp,x,y,mask,b_state);
       break;
     case MotionNotify:
       XQueryPointer(LUI_Display,
             window, &d, &d, &d, &d, &x, &y, &mask);
       b_state=BUTTON_DOWN | BUTTON_MOVING;
       send_matrix(tp,x,y,mask,b_state);
       break;
     default:
       b_state=0;
	 break;
   }   
}

/* ===== send a new transformation to the parameter output port ==== */
 static
send_matrix(tp,x,y,mask,b_state)
    TRACKWIN *tp;
    int x,y;
    int mask,b_state;
{
    int i, j, dims[2], button_no;
    float *dp;

    if (!tp->output) {
	     tp->output=(upstream_transform *)
			AVSdata_alloc("struct upstream_transform",0);
    }
    
    if (mask == Button1Mask) button_no = 1;
    if (mask == Button2Mask) button_no = 2;
    if (mask == Button3Mask) button_no = 3;
    mat_identity(tp->output->msxform);
    tp->output->msxform[3][0] = (float) x; /* fake a translate matrix */
    tp->output->msxform[3][1] = (float) y;
    strcpy (tp->output->object_name, "track window 2d transform");
    tp->output->flags  = b_state;
    tp->output->button = button_no;
    tp->output->x      = x;
    tp->output->y      = y;
    tp->output->width  = tp->width;
    tp->output->height = tp->height;

    AVScorout_output(tp->output);

/* debug print:
    printf(" mask=%d state=%d x=%d y=%d\n",button_no,b_state,x,y);
 */
    if ( tp->curviz != 0 ) draw_cross ( x, y );
}

/* ======= draw a small cursor into the window ====== */
 static
draw_cross( x, y )
          int x,y;
{
    TRACKWIN *tp = &trackwin;

    XPutImage(LUI_Display, tp->window, LUI_GC_black,
          tp->image, 0, 0, 0, 0, tp->width, tp->height);
    XDrawLine(LUI_Display, tp->window, LUI_GC_black, x-20, y, x-10, y);
    XDrawLine(LUI_Display, tp->window, LUI_GC_black, x+10, y, x+20, y);
    XDrawLine(LUI_Display, tp->window, LUI_GC_black, x, y-20, x, y-10);
    XDrawLine(LUI_Display, tp->window, LUI_GC_black, x, y+10, x, y+20);
    XDrawLine(LUI_Display, tp->window, LUI_GC_white, x-10, y, x+10, y);
    XDrawLine(LUI_Display, tp->window, LUI_GC_white, x, y-10, x, y+10);
    XFlush(LUI_Display);
}

/* ======= remove small cursor from the window ====== */
 static
remove_cross()
{
   TRACKWIN *tp = &trackwin;
   XPutImage(LUI_Display, tp->window, LUI_GC_black,
          tp->image, 0, 0, 0, 0, tp->width, tp->height);
   XFlush(LUI_Display);
}

/* ======= display a new image in the window ====== */
trackwin_compute(input, dither_technique)
AVSfield_char *input;
char *dither_technique;
{
    TRACKWIN *tp = &trackwin;
    int width, height, i;
    int dummy, x, y;
    XEvent event;
    unsigned char *src;

    width  = MAXX(input); 
    height = MAXY(input);

    if (tp->window == 0) {
        x = ((LUI_VirtWidth - LUI_MapX(260))/2 - LUI_MapX(256)) + LUI_MapX(260);
        y =  (LUI_VirtHeight / 2 - LUI_MapX(256));

        tp->window = LUI_CreateWindowAt(LUI_RootWindow,x,y,width,height);
        LUI_EventAdd(tp->window,ExposureMask|ButtonPressMask|
		ButtonReleaseMask|PointerMotionHintMask|ButtonMotionMask,
		trackwin_process);
	tp->width = width;
	tp->height = height;
	XMapWindow(LUI_Display,tp->window);
	XNextEvent(LUI_Display,&event);
    }

    if (tp->width != width || tp->height != height)
    {
        XResizeWindow (LUI_Display, tp->window, width, height );
	tp->width  = width;
	tp->height = height;
    }

    if ( LUI_Depth != 4 )
    {
      if ( tp->image != 0 && tp->image->data != 0 ) XFree( tp->image->data );
    }
    if ( tp->image != 0 ) XFree( tp->image );

    tp->image = XCreateImage(LUI_Display, LUI_Visual, LUI_Depth, ZPixmap, 0, 
			 NULL, width, height, 32, 0);

    if ( LUI_Depth != 4 )
    {
        tp->image->data = (char *) malloc ( sizeof(char)*width*height*LUI_Depth/8 );

        pf_dither_image (input->data, width, height,
                         tp->image->data, dither_technique);
    }
    else
        tp->image->data = (char *)input->data;




    /* Put the image into the pixmap */
    XPutImage(LUI_Display, tp->window, LUI_GC_black,
	      tp->image, 0, 0, 0, 0, width, height);

    XFlush(LUI_Display);
}


