/****************************************************************************
                  INTERNATIONAL AVS CENTER
	(This disclaimer must remain at the top of all files)

WARRANTY DISCLAIMER

This module and the files associated with it are distributed free of charge.
It is placed in the public domain and permission is granted for anyone to use,
duplicate, modify, and redistribute it unless otherwise noted.  Some modules
may be copyrighted.  You agree to abide by the conditions also included in
the AVS Licensing Agreement, version 1.0, located in the main module
directory located at the International AVS Center ftp site and to include
the AVS Licensing Agreement when you distribute any files downloaded from 
that site.

The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module provide absolutely
NO WARRANTY OF ANY KIND with respect to this software.  The entire risk as to
the quality and performance of this software is with the user.  IN NO EVENT
WILL The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module BE LIABLE TO
ANYONE FOR ANY DAMAGES ARISING FROM THE USE OF THIS SOFTWARE, INCLUDING,
WITHOUT LIMITATION, DAMAGES RESULTING FROM LOST DATA OR LOST PROFITS, OR ANY
SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES.

This AVS module and associated files are public domain software unless
otherwise noted.  Permission is hereby granted to do whatever you like with
it, subject to the conditions that may exist in copyrighted materials. Should
you wish to make a contribution toward the improvement, modification, or
general performance of this module, please send us your comments:  why you
liked or disliked it, how you use it, and most important, how it helps your
work. We will receive your comments at avs@ncsc.org.

Please send AVS module bug reports to avs@ncsc.org.

******************************************************************************/
/*************************************************************************/
/*                                                                       */
/*  Program ID:     ucd statistics                                       */
/*                                                                       */
/*  Description:    this module computes basic statistics for UCD data   */
/*                                                                       */
/*  Entry Points:   stats_spec() - description function                  */
/*                  stats_body() - module compute function               */
/*                                                                       */
/*  Input Parms:    None                                                 */
/*                                                                       */
/*  Exit Value:     EXIT_SUCCESS - normal termination                    */
/*                  EXIT_FAILURE - abnormal termination due to open,     */
/*                                 read, allocation, or input errors.    */
/*                                                                       */
/*  Input Files:    None                                                 */
/*                                                                       */
/*  Output Files:   None                                                 */
/*                                                                       */
/*  Link Procedures: None                                                */
/*                                                                       */
/*  Special Logic Notes:                                                 */
/*                        Note that ucd data is stored in weird format.  */
/*                        For 131 components, data is in aaabcdbcdbcdeee */
/*                        order.                                         */
/*                                                                       */
/*                        Loop through calculating the stats for each    */
/*                        data value, and then return and do it for      */
/*                        all data points.                               */
/*                                                                       */
/*                                                                       */
/*************************************************************************/
/*                        MODIFICATION LOG                               */
/*                                                                       */
/*     DATE          AUTHOR                  DESCRIPTION                 */
/*   --------   ------------------  ---------------------------------    */
/*   10/05/93   Anonymous g69d      Initial mod of statistics module     */
/*   10/14/94   Anonymous g69d      added comment blocks, increased      */
/*                                  precision to double because of       */
/*                                  roundoff errors                      */
/*   01/31/95   Anonymous g69d      fixed vector calculations            */
/*   03/28/95   Anonymous g69d      Another attempt to fix vector calcs  */
/*                                  changed formatting to allow for      */
/*                                  printing of very small and large     */
/*                                  numbers.                             */
/*                                  Reorder functions and convert to ANSI*/
/*                                  Add toggle to print to stdout.       */
/*   04/05/95   Anonymous g69d      Add support for cell data            */
/*                                                                       */
/*                                                                       */
/*************************************************************************/



/*************************************************************************/
/*                                                                       */
/*  Module Name:  UCDstatistics.c                                        */
/*                                                                       */
/*  Program ID:   ucd statistics                                         */
/*                                                                       */
/*  Functions:    compute_stats()                                        */
/*                                                                       */
/*************************************************************************/

/* Includes and Defines */
#include <stdio.h>
#include <string.h>
/* IAC CODE CHANGE : #include <math.h> */
#include <avs/avs_math.h>
#include <sys/types.h>
#include <sys/time.h>
/* IAC CODE CHANGE : #include <avs_math.h> */
#include <avs/avs_math.h>
#include <avs/flow.h>
#include <avs/field.h>
#include <ucd_defs.h>
#include "my_std.h"

#define TEXT_BLK_CHARS   ( 7000 )
#define TEXT_LIN_LEN     ( 80 )
#define NODE             ( 1 )
#define CELL             ( 2 )
/* #define DEBUG       ( 1 ) */

/* Internal Function Declarations */
int stats_body( UCD_structure *input, int send_to_stdout, char *statistics );
int compute_stats( float *data, int n, int off, int vec_size, int k,
		   double *ave, double *adev, double *sdev, double *svar,
		   double *skew, double *curt, double *min, double *max );
int DisplayStats( UCD_structure *input, int data_type, char *output );


/**** Module Description Function ****/
stats_spec()
{
    int ii;
    AVSset_module_name("ucd statistics", MODULE_RENDER);
    AVScreate_input_port("ucd", "ucd", REQUIRED);

    ii = AVSadd_parameter("Send to stdout", "boolean", 0 , 0, 1 );
    ii = AVSadd_parameter("Statistics","string_block",NULL,NULL,NULL);
    AVSconnect_widget(ii,"text_block_browser");
    AVSadd_parameter_prop(ii,"width","integer",4);
    AVSadd_parameter_prop(ii,"height","integer",20);

    AVSset_compute_proc(stats_body);

    return ( 1 );
}

int ((*module_list[])()) = { stats_spec };

AVSinit_modules()
{
	AVSinit_from_module_list(module_list,
				 sizeof(module_list)/sizeof(module_list[0]));
}

int
stats_body( UCD_structure *input, int send_to_stdout, char *statistics )
{   
    char    outstring[ TEXT_BLK_CHARS ];
    char    tmp[ TEXT_LIN_LEN ]; 
    int     j;
    int     num = 0;
    int     num_components = 0;
    bool    stats_succ = TRUE;
   

    strcpy(outstring,"UCD Statistics\n" );
    sprintf(tmp,"===================\n");
    strcat(outstring,tmp);
    sprintf(tmp,"Number of Nodes: %i\n", input->nnodes );
    strcat(outstring,tmp);
    sprintf(tmp,"Number of Cells: %i\n", input->ncells );
    strcat(outstring,tmp);
    for( j = 0 ; num < input->node_veclen ; j++ )
    {
      num_components++; 
      num += input->node_components[j];
    }
    sprintf(tmp,"Number of Node Components:  %i\n", num_components );
    strcat(outstring,tmp);
    sprintf(tmp,"Total Data Values per Node:  %i\n", input->node_veclen );
    strcat(outstring,tmp);
    if ( input->cell_veclen > 0 )
    {
        for( j = 0, num_components = 0, num = 0 ; num < input->cell_veclen ; j++ )
        {
          num_components++; 
          num += input->cell_components[j];
        }
        sprintf(tmp,"Number of Cell Components:  %i\n", num_components );
        strcat(outstring,tmp);
        sprintf(tmp,"Total Data Values per Cell:  %i\n \n", input->cell_veclen );
        strcat(outstring,tmp);
    }
    else 
	strcat(outstring, " \n" );
    

    if ( input->node_veclen > 0 )
        stats_succ = DisplayStats( input, NODE, outstring );

    if ( ( input->cell_veclen > 0 ) && stats_succ )
    {
       strcat(outstring, " \n===================\n  \n" );
       stats_succ =  DisplayStats( input, CELL, outstring );
    }

    if ( ( input->node_veclen < 1 ) && ( input->node_veclen < 1 ) ) 
	strcpy( outstring, "No data exists." );

    if ( !stats_succ )
	strcpy( outstring, "Error occured.\nNo stats generated" );

    AVSmodify_parameter( "Statistics", AVS_VALUE, outstring, NULL, NULL);

    if ( send_to_stdout )
	puts( outstring );

    return( stats_succ );
}


/*************************************************************************/
/*                                                                       */
/*  Function Name:  DisplayStats                                         */
/*                                                                       */
/*  Program ID:     ucd statistics                                       */
/*                                                                       */
/*  Description:    Determines whether to process cell or node data.     */
/*                  Then calls ststa calculating func, and displays      */
/*                  results.                                             */
/*                                                                       */
/*  Parameters:     UCD_structure *input (I) - UCD structure for stats   */
/*                  int data_type (I) - flag for NODE or CELL            */
/*                  char *output (I/O) - text output for statistics      */
/*                                                                       */
/*  System Inputs:  None                                                 */
/*                                                                       */
/*  System Outputs: None                                                 */
/*                                                                       */
/*  Globals:                                                             */
/*     Used:        None                                                 */
/*     Modified:    None                                                 */
/*                                                                       */
/*  External Fxns:  None                                                 */
/*                                                                       */
/*  Return Value:   void                                                 */
/*                                                                       */
/*  Special Logic Notes:  None                                           */
/*                                                                       */
/*************************************************************************/
/*                        MODIFICATION LOG                               */
/*                                                                       */
/*     DATE          AUTHOR                  DESCRIPTION                 */
/*   --------   ------------------  ---------------------------------    */
/*   04/05/95   Anonymous g69d      Initial Release                      */
/*                                                                       */
/*************************************************************************/

int DisplayStats( UCD_structure *input, int data_type, char *output )
{   
    char    tmp[ TEXT_LIN_LEN ], label[ TEXT_LIN_LEN ];
    float   *data;
    int     points, veclen, *components;
    int     size;
    int     j,k,psize;
    int     offset = 0;
    double  avg,adeviation,sdeviation,variance,skewness,kurtosis,min,max;
    int     num = 0;
    int     num_components = 0;
    int     vecpos = 0;
    bool    stats_succ = TRUE;
   
    if ( data_type == NODE )
    {
	data = input->node_data;
	points = input->nnodes;
	veclen = input->node_veclen;
	components = input->node_components;
    }
    else
    {
	data = input->cell_data;
	points = input->ncells;
	veclen = input->cell_veclen;
	components = input->cell_components;
    }

    size = points * veclen;

    for( j = 0 ; num < veclen ; j++ )
    {
      num_components++; 
      num += components[j];
    }

#ifdef DEBUG
    printf( "Debug: data address = %ld\n", data );
    printf( "Debug: data length =  %ld\n", size );
    printf( "Debug: totals =       %ld\n", data + size );
    printf( "Debug: num_components = %d\n\n", num_components );
#endif

    for( j = 0 ; ( ( j < num_components) && stats_succ ) ; j++ )
    {
      psize = points * components[j];
      for( k = 0 ; ( ( k < components[j] ) && stats_succ ) ; k++)
      {
        stats_succ = compute_stats(data,psize,offset,components[j],
				   k, &avg, &adeviation, &sdeviation, &variance,
		                   &skewness, &kurtosis, &min, &max);
        vecpos++;
        sprintf( tmp, "%s Data Value #%i\n",
		 ( data_type==NODE ? "Node" : "Cell" ), vecpos );
        strcat( output, tmp );
	if ( data_type == NODE )
	    UCDstructure_get_node_label( input, j, label );
	else
	    UCDstructure_get_cell_label( input, j, label );

        if ( components[j] == 1 )
          sprintf( tmp,"%s\n", label );
        else
          sprintf( tmp,"%s, vector component #%i\n", label, k+1 );
        
        strcat(output,tmp);
        sprintf(tmp,"Min/Max:       % #7g % #7g\n", min, max);
        strcat(output,tmp);
        sprintf(tmp,"Average:       % #7g\n",avg);
        strcat(output,tmp);

        sprintf(tmp,"Standard Dev.: % #7g\n",sdeviation);
        strcat(output,tmp);
        sprintf(tmp,"Skewness:      % #7g\n",skewness);
        strcat(output,tmp);
        sprintf(tmp,"Kurtosis:      % #7g\n   \n",kurtosis);
        strcat(output,tmp);
      } /*end loop for vectors*/
      offset += psize;
    } /* end the for loop for partial stats*/

    if ( ( veclen > 1 ) && stats_succ )
    {
      psize = size;
      offset = 0;
      stats_succ = compute_stats(data, psize, offset, 1, 0, &avg, &adeviation,
                                 &sdeviation, &variance, &skewness, &kurtosis,
                                 &min, &max);

      sprintf(tmp,"   \n");
      strcat(output,tmp);
      sprintf(tmp,"All %s Data Values\n", ( data_type==NODE ? "Node" : "Cell" ) );
      strcat(output,tmp);
      sprintf(tmp,"Min/Max:       % #7g % #7g\n", min, max);
      strcat(output,tmp);
      sprintf(tmp,"Average:       % #7g\n",avg);
      strcat(output,tmp);

      sprintf(tmp,"Standard Dev.: % #7g\n",sdeviation);
      strcat(output,tmp);
      sprintf(tmp,"Skewness:      % #7g\n",skewness);
      strcat(output,tmp);
      sprintf(tmp,"Kurtosis:      % #7g\n",kurtosis);
      strcat(output,tmp);
    }

    return( stats_succ );
}


/*************************************************************************/
/*                                                                       */
/*  Function Name:  compute_stats()                                      */
/*                                                                       */
/*  Program ID:     UCDstatistics                                        */
/*                                                                       */
/*  Description:    this function computes all of the statistics         */
/*                                                                       */
/*  Parameters:     float data (I) - pointer to start of data array      */
/*                  int n (I) - size of pertinent data array             */
/*                  int off (I) - offset from start of data to start of  */
/*                                pertinent data                         */
/*                  int vec_size (I) - vector length of current component*/
/*                  int k (I) - position in vector of current component  */
/*                  double ave, adev, sdev, svar, skew, curt, min, max   */
/*                        (O) - statistics                               */
/*                                                                       */
/*  System Inputs:  None                                                 */
/*                                                                       */
/*  System Outputs: None                                                 */
/*                                                                       */
/*  Globals:                                                             */
/*     Used:        None                                                 */
/*     Modified:    None                                                 */
/*                                                                       */
/*  External Fxns:  None                                                 */
/*                                                                       */
/*  Return Value:   void                                                 */
/*                                                                       */
/*  Special Logic Notes:  None                                           */
/*                                                                       */
/*************************************************************************/
/*                        MODIFICATION LOG                               */
/*                                                                       */
/*     DATE          AUTHOR                  DESCRIPTION                 */
/*   --------   ------------------  ---------------------------------    */
/*   01/31/95   Anonymous g69d      Initial documentation and bug fixes  */
/*                                  of existing function.                */
/*                                                                       */
/*************************************************************************/
 
int compute_stats( float *data, int n, int off, int vec_size, int k,
		   double *ave, double *adev, double *sdev, double *svar,
		   double *skew, double *curt, double *min, double *max )
{
    int	i;
    double s = 0.0;
    double p;
    int numpts;

    numpts = n / vec_size;

#ifdef DEBUG
    printf( "Debug: data address = %ld\n", data );
    printf( "Debug: current array =  %ld\n", n );
    printf( "Debug: offset =       %ld\n", off );
    printf( "Debug: current veclen = %d\n", vec_size );
    printf( "Debug: pos in vector = %d\n", k );
    printf( "Debug: number of pts = %d\n", numpts );
    printf( "Debug: n+off+k = %d\n", n+off+k );
    printf( "Debug: &data[n+off+k] = %ld\n\n", &data[n+off+k] );
#endif

    if( n <= 1 || vec_size < 1 )
      return ( FALSE );

    *min = *max = data[ off + k ];
    for( i = (off+k) ; i < (n+off+k) ; i += vec_size )
    {
      s += data[i];
      if (data[i] < *min) 
        *min = data[i];
      else if (data[i] > *max)
        *max = data[i];
    }
    *ave = s/(double)numpts;
    *adev = (*svar) = (*skew) = (*curt) = 0.0;

    for(i=(off+k);i<(n+off+k);i+=vec_size)
    {
      *adev += fabs(s=data[i]-(*ave));
      *svar += (p = s*s);
      *skew += (p *= s);
      *curt += (p *= s);
    }
    *adev /= (double)numpts;
    *svar /= (double)(numpts-1);
    *sdev  = sqrt(*svar);

    if(*svar != 0.0)
    {
	*skew /= (double)numpts*(*svar)*(*sdev);
        *curt  = (*curt)/((double)numpts*(*svar)*(*svar)) - 3.0;
    }

    return ( TRUE );
}	

void __ansi_fflush()
{ 
  return;
}

