/****************************************************************************
                  INTERNATIONAL AVS CENTER
	(This disclaimer must remain at the top of all files)

WARRANTY DISCLAIMER

This module and the files associated with it are distributed free of charge.
It is placed in the public domain and permission is granted for anyone to use,
duplicate, modify, and redistribute it unless otherwise noted.  Some modules
may be copyrighted.  You agree to abide by the conditions also included in
the AVS Licensing Agreement, version 1.0, located in the main module
directory located at the International AVS Center ftp site and to include
the AVS Licensing Agreement when you distribute any files downloaded from 
that site.

The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module provide absolutely
NO WARRANTY OF ANY KIND with respect to this software.  The entire risk as to
the quality and performance of this software is with the user.  IN NO EVENT
WILL The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module BE LIABLE TO
ANYONE FOR ANY DAMAGES ARISING FROM THE USE OF THIS SOFTWARE, INCLUDING,
WITHOUT LIMITATION, DAMAGES RESULTING FROM LOST DATA OR LOST PROFITS, OR ANY
SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES.

This AVS module and associated files are public domain software unless
otherwise noted.  Permission is hereby granted to do whatever you like with
it, subject to the conditions that may exist in copyrighted materials. Should
you wish to make a contribution toward the improvement, modification, or
general performance of this module, please send us your comments:  why you
liked or disliked it, how you use it, and most important, how it helps your
work. We will receive your comments at avs@ncsc.org.

Please send AVS module bug reports to avs@ncsc.org.

******************************************************************************/
/*
 * write_geom module
 *
 *
 * Author: Georg Thallinger, 
 *         JOANNEUM RESEARCH, Institute for Information Systems
 *
 *
 */

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <avs/avs.h>
#include <avs/field.h>
#include <avs/geom.h>
#include <avs/geomdata.h>

#define FMT_TXT  0
#define FMT_BIN  1
#define FMT_DXF  2

#define EPSILON 1e-6

double SQR();


typedef struct {
  int ((*WriteHeader)());
  int ((*WriteObject)());
  int ((*WriteTrailer)());
} IOFunctions;

/* a GeomWriteObject which "uses" streams */
int GeomWriteObj();      /* SRT GEOMobj *obj, FILE *fp, int fummy); */

extern int DoNothing();  /* SRT */
extern int DXFHeader (); /* SRT FILE *fp, int objCnt); */
extern int DXFTrailer(); /* SRT FILE *fp); */

int WriteDXF (); /* GEOMobj *object, FILE *fp, int dummy); SRT */

/* table holding output functions for the various formats */
IOFunctions WriteFuncs[] = {
  { DoNothing, GEOMwrite_text, DoNothing},
  { DoNothing, GeomWriteObj, DoNothing},
  { DXFHeader, WriteDXF, DXFTrailer}
};

/* defines for the geom-objects start from 1 !! */
char *ObjNames[] = { "", "GEOM_MESH", "GEOM_POLYHEDRON", "GEOM_POLYTRI",
		       "GEOM_SPHERE", "GEOM_LABEL"};
	
static int FormatChoice;

static FILE *OutFp;
static int  OutFD;

static int NumObjects, ActObject;

/* function prototypes */
extern int write_geom_compute();

/******************************************************************************/
/* Description Routine, set up module attributes                              */
/******************************************************************************/
write_geom_desc()
{
    int in_port, param;

    AVSset_module_name("write geom", MODULE_RENDER);

    in_port  = AVScreate_input_port("Geometry", "geom", REQUIRED);

    param = AVSadd_parameter("Filename", "string", NULL, NULL, NULL);
    AVSconnect_widget(param, "browser");

    param = AVSadd_parameter("Format", "choice", "text",
			     "text:binary:dxf", ":");
    AVSconnect_widget(param, "radio_buttons");

    AVSset_compute_proc(write_geom_compute);
}

/*************************************************************************/
/* Initialization for modules contained in this file.                    */
/*************************************************************************/
int ((*mod_list[])()) = {
  write_geom_desc,
};

#define NMODS (sizeof(mod_list) / sizeof(char *))

void AVSinit_modules()
{
  AVSinit_from_module_list(mod_list, NMODS);
}


int write_geom_compute( 	input_geom, 
				FileName, 
				WriteFmt)
GEOMedit_list input_geom;
char *FileName; 
char *WriteFmt;
{
    int write_it();
    int CountObjects();

    NumObjects = ActObject = 0;


    if (AVSparameter_changed ("Filename"))
    {
      FormatChoice = AVSchoice_number("Format", WriteFmt) - 1;
      if (FormatChoice < 0)
      {
	AVSerror( "The choice %s is not valid !\n", WriteFmt);
	return 0;
      }

      /* the file descriptor is needed by GEOMwrite_obj, 
      ** the FILE-pointer is used by GEOMwrite_txt and WriteDXF
      */
      OutFD = open (FileName, O_WRONLY|O_CREAT|O_TRUNC, 0644);
      if (OutFD<0)
      {
	AVSerror ("Unable to open the file\n%s !\n", FileName);
	return 0;
      }

      OutFp = fdopen(OutFD, "wb");
      if (OutFp==NULL)
      {
	AVSerror ("Unable to open the file\n%s !\n", FileName);
	return 0;
      }

      /* count the number of objects in the geometry */
      GEOMset_new_geometry_func(CountObjects);
      GEOMprocess_edit_list(input_geom);

      /* write the header */
      WriteFuncs[FormatChoice].WriteHeader(OutFp, NumObjects);

      /* process all objects in the geometry */
      GEOMset_new_geometry_func(write_it);
      GEOMprocess_edit_list(input_geom);

      /* write the trailer */
      WriteFuncs[FormatChoice].WriteTrailer(OutFp);

      fclose (OutFp);

      return 1;
    }
    return 0;
}

/*****************************************************************************/
int write_it(name, object)
char *name; 
GEOMobj *object;
{ 
  printf ("%s\n", name);

  WriteFuncs[FormatChoice].WriteObject(object, OutFp, 0);
}

int WriteDXF(object, fp, dummy)
GEOMobj *object;
FILE *fp;
int dummy;
{
  int cnt, triCnt, ptCnt;
  float *pts;

  char layer[30];

  switch (object->type)
  {
    case GEOM_POLYTRI:
    {
      /* 3 possibilities : line-segments, polylines or triangle strips */
      /* process the polytriangles */
      if (PT(object).npts > 0)
	sprintf (layer, "Objectlayer%04d",ActObject++);

      for (cnt=0; cnt<(PT(object).npts); cnt++)
      {
	float area;
	float v[3],w[3];

	pts = PT(object).ptverts[cnt].l;

	for (triCnt=0; triCnt < PT(object).ptverts[cnt].n-2; triCnt++)
	{ 
	  /* compute the area of the triangle via the exprod */
	  v[0] = pts[3*(triCnt+1)] - pts[3*triCnt];
	  v[1] = pts[3*(triCnt+1)+1] - pts[3*triCnt+1];
	  v[2] = pts[3*(triCnt+1)+2] - pts[3*triCnt+2];

	  w[0] = pts[3*(triCnt+2)] - pts[3*triCnt];
	  w[1] = pts[3*(triCnt+2)+1] - pts[3*triCnt+1];
	  w[2] = pts[3*(triCnt+2)+2] - pts[3*triCnt+2];

	  area = SQR((double)v[1]*w[2]-v[2]*w[1])
	       + SQR((double)v[2]*w[0]-v[0]*w[2]) 
	       + SQR((double)v[0]*w[1]-v[1]*w[0]);
	  
	  if (area > EPSILON) 
	  {
	    fprintf (fp, "  0\n3DFACE\n  8\n%s\n", layer);
	    fprintf 
	      (fp, " 10\n%f\n 20\n%f\n 30\n%f\n",
	       pts[3*triCnt], pts[3*triCnt+1], pts[3*triCnt+2]);
	    fprintf 
	      (fp, " 11\n%f\n 21\n%f\n 31\n%f\n",
	       pts[3*(triCnt+1)], pts[3*(triCnt+1)+1], pts[3*(triCnt+1)+2]);
	    fprintf 
	      (fp, " 12\n%f\n 22\n%f\n 32\n%f\n",
	       pts[3*(triCnt+2)], pts[3*(triCnt+2)+1], pts[3*(triCnt+2)+2]);
	    fprintf 
	      (fp, " 13\n%f\n 23\n%f\n 33\n%f\n",
	       pts[3*(triCnt+2)], pts[3*(triCnt+2)+1], pts[3*(triCnt+2)+2]);
	  }
	}
      }
	
      /* process the polylines */
      for (cnt=0; cnt<PT(object).npls; cnt++)
      {
	pts = PT(object).plverts[cnt].l;

	sprintf (layer, "Objectlayer%04d",ActObject++);

	fprintf (fp, "  0\nPOLYLINE\n  8\n%s\n 66\n1\n 70\n8\n", layer);

	for (triCnt=0; triCnt<PT(object).plverts[cnt].n; triCnt++)
	{
	  fprintf (fp, "  0\nVERTEX\n  8\n%s\n 70\n32\n", layer);
	  fprintf (fp, " 10\n%f\n 20\n%f\n 30\n%f\n",
		   pts[3*triCnt], pts[3*triCnt+1], pts[3*triCnt+2]);
	}
	fprintf (fp, "  0\nSEQEND\n");
      }

      /* process the lines */
      pts = PT(object).dlverts.l;

      if (PT(object).dlverts.n > 0)
	sprintf (layer, "Objectlayer%04d",ActObject++);

      for (cnt=0; cnt<PT(object).dlverts.n; cnt++)
      {
	fprintf (fp, "  0\nLINE\n  8\n%s\n",layer);
	fprintf (fp, " 10\n%f\n 20\n%f\n 30\n%f\n",
		 pts[3*cnt], pts[3*cnt+1], pts[3*cnt+2]);
	fprintf (fp, " 11\n%f\n 21\n%f\n 31\n%f\n",
		 pts[3*(cnt+1)], pts[3*(cnt+1)+1], pts[3*(cnt+1)+2]);
	
	cnt++;
      }
      
      break;
    }
    case GEOM_POLYHEDRON:
    {
      int *polygons = PH(object).ptlist.l;
      float area;
      float v[3],w[3];
      
      pts = PH(object).verts.l;

      sprintf (layer, "Objectlayer%04d",ActObject++);
      
      cnt = 0; /* serves as index into polygons array */
      
      while (polygons[cnt]!=0)
      {
	/* polygons with more than 3 vertices are split into triangles */
	for (ptCnt=1; ptCnt<polygons[cnt]-1; ptCnt++)
	{
	  /* compute the area of the triangle via the exprod */
	  v[0] = pts[3*(polygons[cnt+ptCnt+1]-1)] - pts[3*(polygons[cnt+1]-1)];
	  v[1] = pts[3*(polygons[cnt+ptCnt+1]-1)+1] 
	    - pts[3*(polygons[cnt+1]-1)+1];
	  v[2] = pts[3*(polygons[cnt+ptCnt+1]-1)+2] 
	    - pts[3*(polygons[cnt+1]-1)+2];
	  
	  w[0] = pts[3*(polygons[cnt+ptCnt+2]-1)] - pts[3*(polygons[cnt+1]-1)];
	  w[1] = pts[3*(polygons[cnt+ptCnt+2]-1)] 
	    - pts[3*(polygons[cnt+1]-1)+1];
	  w[2] = pts[3*(polygons[cnt+ptCnt+2]-1)] 
	    - pts[3*(polygons[cnt+1]-1)+2];
	  
	  area = SQR((double)v[1]*w[2]-v[2]*w[1]) 
               + SQR((double)v[2]*w[0]-v[0]*w[2]) 
	       + SQR((double)v[0]*w[1]-v[1]*w[0]);

	  if (area > EPSILON) 
	  {
	    fprintf (fp, "  0\nSOLID\n  8\n%s\n", layer);
	    fprintf 
	      (fp, " 10\n%f\n 20\n%f\n 30\n%f\n", 
	       pts[3*(polygons[cnt+1]-1)], 
	       pts[3*(polygons[cnt+1]-1)+1], 
	       pts[3*(polygons[cnt+1]-1)+2]);
	    fprintf 
	      (fp, " 11\n%f\n 21\n%f\n 31\n%f\n", 
	       pts[3*(polygons[cnt+ptCnt+1]-1)], 
	       pts[3*(polygons[cnt+ptCnt+1]-1)+1], 
	       pts[3*(polygons[cnt+ptCnt+1]-1)+2]);
	    fprintf 
	      (fp, " 12\n%f\n 22\n%f\n 32\n%f\n", 
	       pts[3*(polygons[cnt+ptCnt+2]-1)], 
	       pts[3*(polygons[cnt+ptCnt+2]-1)+1], 
	       pts[3*(polygons[cnt+ptCnt+2]-1)+2]);
	    fprintf 
	      (fp, " 13\n%f\n 23\n%f\n 33\n%f\n", 
	       pts[3*(polygons[cnt+ptCnt+2]-1)], 
	       pts[3*(polygons[cnt+ptCnt+2]-1)+1], 
	       pts[3*(polygons[cnt+ptCnt+2]-1)+2]);
	  }
	}
	cnt += polygons[cnt] + 1;
      }
      
      break;
    }
    case GEOM_SPHERE:
    case GEOM_MESH:
    case GEOM_LABEL:
    {
      /* this  are not processed now */
      fprintf (stderr, "WriteDXF: Object type %s not supported !", 
	       ObjNames[object->type]); 
      break;
    }
  }
  return 0;
}

int DXFHeader (fp, ObjCount)
FILE *fp;
int ObjCount;
{
  int i;

  /* Tables section, define a linetype and a layerers for the objects */
  fprintf (fp, "  0\nSECTION\n  2\nTABLES\n");

  fprintf (fp, "  0\nTABLE\n  2\nLTYPE\n 70\n1\n");
  fprintf (fp, "  0\nLTYPE\n  2\nCONTINUOUS\n 70\n64\n  3\nSolid line\n");
  fprintf (fp, " 72\n65\n 73\n0\n 40\n0.0\n  0\nENDTAB\n");

  fprintf (fp, "  0\nTABLE\n  2\nLAYER\n 70\n%d\n", ObjCount);
  for (i=0; i<ObjCount; i++)
  {
    fprintf (fp, "  0\nLAYER\n  2\nObjectlayer%04d\n 70\n64\n 62\n1\n", i);
    fprintf (fp, "  6\nCONTINUOUS\n");
  }
  fprintf (fp,"  0\nENDTAB\n");

  fprintf (fp, "  0\nENDSEC\n");

  /* begin of entity section, here all generated objects are stored */
  fprintf (fp, "  0\nSECTION\n  2\nENTITIES\n");

  return 0;
}

int DXFTrailer (fp)
FILE *fp;
{
  fprintf (fp, "  0\nENDSEC\n");
  fprintf (fp, "  0\nEOF\n");

  return 0;
}

/* dummy routine used for formats without header/trailer ! */
int DoNothing()
{
  return 0;
}

int GeomWriteObj(obj, fp, dummy)
GEOMobj *obj;
FILE *fp;
int dummy;
{
  return GEOMwrite_obj(obj, OutFD, dummy);
}

/* Geometry function to count all the geometric entities in the geometry */
int CountObjects(name, object)
char *name;
GEOMobj *object;
{
  switch (object->type)
  {
    case GEOM_POLYTRI:
    {
      /* 3 possibilities : line-segments, polylines or triangle strips */
      /* process the polytriangles */
      if ( PT(object).npts > 0 ) 
	NumObjects ++;

      /* process the polylines */
      NumObjects += PT(object).npls;

      /* process the lines */
      if (PT(object).dlverts.n >0)
	NumObjects ++;

      break;
    }
    case GEOM_POLYHEDRON:
    {
      NumObjects++;

      break;
    }
    case GEOM_SPHERE:
    case GEOM_MESH:
    case GEOM_LABEL:
    {
      /* these are not processed now */

      break;
    }
  }
  return 0;
}

/***************************************************************************/
double SQR(x)
double x;
{
  return x*x;
}

/* end of write_geom.c */
