/*   
     This software is copyright (C) 1991,  Regents  of  the
University  of  California.   Anyone may reproduce 3d_axis.c,
the software in this distribution, in whole or in part, pro-
vided that:

(1)  Any copy  or  redistribution  of  3d_axis.c  must  show  the
     Regents  of  the  University of California, through its
     Lawrence Berkeley Laboratory, as the source,  and  must
     include this notice;

(2)  Any use of this software must reference this  distribu-
     tion,  state that the software copyright is held by the
     Regents of the University of California, and  that  the
     software is used by their permission.

     It is acknowledged that the U.S. Government has  rights
in  3d_axis.c  under  Contract DE-AC03-765F00098 between the U.S.
Department of Energy and the University of California.

     3d_axis.c is provided as a professional  academic  contribu-
tion  for  joint exchange.  Thus it is experimental, is pro-
vided ``as is'', with no warranties of any kind  whatsoever,
no  support,  promise  of updates, or printed documentation.
The Regents of the University of California  shall  have  no
liability  with respect to the infringement of copyrights by
3d_axis.c, or any part thereof.     

*/

/*
 *  filename 3D_axis.c
 *  Wayne Hurlbert , LBL.
 *  May 1990
 *
 *  Generate 3D coordinate axis.
 *
 *  UNIFORM data sets are mapped into a 3D space of dimensions -1 to 1, so
 *  a line drawn from say, (-1,-1,-1) to (1,1,1), will go from the far lower
 *  left to the near upper right of the data set.  Here, (-1,-1,-1) is used
 *  as the origin of the 3D axis, but ticks and labels are based on the
 *  dimensions of the data set (0 - MAXX, 0 - MAXY, 0 - MAXZ).  These min and
 *  max values may be changed by the user, but such changes will only change
 *  ticks and labels, not the drawn axis.
 **
     UNIFORM fields are now mapped into a space corresponding to
     0..size-1 for each of the data dimensions.  This mapping corresponds
     to the AVS3 world of uniform fields. 

     Access to the world of AVS3 is enabled by using the compile
     flag -DAVS3.

     Significant reworking of the loops for tick marks, labels and
     meshes have been redone to reflect this change, and also to
     support having the "min" value greater than the "max" value.

     22 May 1991 w.bethel
 *
 *  RECTILINEAR and IRREGULAR data sets, and the axes generated here, are
 *  drawn in the world space of the coordinate info stored in the data set.
 *  The origin of the axis system is placed at the minimum coordinate value
 *  of each dimension.  Note that this will probably not coincide with the
 *  world origin (0,0,0), but it is usually the far lower left corner of the
 *  data set.  This means that changing the axis minimum and maximum values
 *  will change the positions of the drawn axis, but will not change label
 *  values or the screen location of any particular point.

     19 June 92

     Changed the 6 float parameters defining the axis extents into two
     typeins, each of which must consist of 3 floats.  Improved the
     initial entry into this module so that parm_set's in a saved network
     file will not be overwritten when the module executes for the first
     time.

     26 June 92

     Misc bugs fixed.


     1 Aug 92

     1. changes to accomodate 2D rectilinear and uniform fields. x/y
     coords are either supplied or computed.  z coordinates are computed
     from the data.

     2. changed the locator used to position the orthogonal planes so
     that it is now in the coordinate system of the data rather than some
     mysterious normalized space.

     3.  changed the tick size parameter so that its range gets reset based
     upon some function of the spatial extents of the data.

 */

#include <stdio.h>
#include <math.h>

#include <avs/vex.h>
#include <avs/flow.h>
#include <avs/field.h>
#include <avs/geom.h>
#include <avs/geomdata.h>

#define max(a, b) ((a) > (b) ? (a) : (b))
#define min(a, b) ((a) < (b) ? (a) : (b))

#define MAX_TICK_PROPORTION 0.5
int first_time = TRUE;

#define AMNX "Axis Min"
#define AMXX "Axis Max"
#define INIT_STRING "null"

/********** AVS initialization ************************************************/

AVSinit_modules()
{
    int axis_desc();
    AVSmodule_from_desc(axis_desc);
}


/******************************************************************************/

int axis_desc()
{
    int axis_compute();
    int parm;

    AVSset_module_name("3D axis", MODULE_MAPPER);

    AVScreate_input_port("Field Input", "field", REQUIRED);
    AVScreate_output_port("Geometry", "geom");

    /*
     *  Set number of ticks per axis, and tick size.
     */
    parm = AVSadd_parameter("message1","string","Set ticks:", "","");
    AVSconnect_widget(parm,"text");
    AVSadd_parameter_prop(parm,"width","integer",4);
    parm = AVSadd_parameter("x axis ticks", "integer", 0, 0, 32);
    AVSconnect_widget(parm,"islider");
    parm = AVSadd_parameter("y axis ticks", "integer", 0, 0, 32);
    AVSconnect_widget(parm,"islider");
    parm = AVSadd_parameter("z axis ticks", "integer", 0, 0, 32);
    AVSconnect_widget(parm,"islider");
    parm = AVSadd_float_parameter("tick size", 0.01, 0.0, 1.0);
    AVSconnect_widget(parm,"slider");


    /*
     *  Turn planes on and off, and set the position of the plane in the
     *  3rd dimension. The plane mesh is determined by the tick spacing.
     */
    parm = AVSadd_parameter("message2","string","Set mesh planes:", "","");
    AVSconnect_widget(parm,"text");
    AVSadd_parameter_prop(parm,"width","integer",4);
    parm = AVSadd_float_parameter("xz plane position", 0.0, 0.0, 1.0);
    AVSconnect_widget(parm,"slider");
    parm = AVSadd_float_parameter("xy plane position", 0.0, 0.0, 1.0);
    AVSconnect_widget(parm,"slider");
    parm = AVSadd_float_parameter("yz plane position", 0.0, 0.0, 1.0);
    AVSconnect_widget(parm,"slider");
    AVSadd_parameter("Show xz plane", "boolean", 0, 0, 1);
    AVSadd_parameter("Show xy plane", "boolean", 0, 0, 1);
    AVSadd_parameter("Show yz plane", "boolean", 0, 0, 1);


    /*
     *  Turn tick labels on and off, set font size, set precision, set
     *  number of ticks per label, and set which axis labels the origin.
     */
    parm = AVSadd_parameter("message3","string","Set labels:", "","");
    AVSconnect_widget(parm,"text");
    AVSadd_parameter_prop(parm,"width","integer",4);
    parm = AVSadd_float_parameter("font size", 0.07, 0.01, 0.1);
    AVSconnect_widget(parm,"slider");
    parm = AVSadd_parameter("label precision", "integer", 1, 0, 9);
    AVSconnect_widget(parm,"islider");
    parm = AVSadd_parameter("font number", "integer", 1, 0, 21);
    AVSconnect_widget(parm,"islider");
    parm = AVSadd_parameter("x label density", "integer", 1, 1, 100);
    AVSconnect_widget(parm,"islider");
    parm = AVSadd_parameter("y label density", "integer", 1, 1, 100);
    AVSconnect_widget(parm,"islider");
    parm = AVSadd_parameter("z label density", "integer", 1, 1, 100);
    AVSconnect_widget(parm,"islider");

    parm = AVSadd_parameter("origin labels","string","","","");
    AVSconnect_widget(parm,"typein");
    AVSadd_parameter("Show labels","boolean", 0, 0, 1);


    /*
     *  Show the min and max values of the axis, as defined by the data
     *  set.  Then allow the user to change those values.  For a UNIFORM
     *  data set this simply changes the values associated with the axis
     *  ticks.  For RECTILINEAR and IRREGULAR data sets, this changes the
     *  actual positions of the axis endpoints.
     */
    parm = AVSadd_parameter("message4","string","Axis extents in R3", "","");
    AVSconnect_widget(parm,"text");
    AVSadd_parameter_prop(parm,"width","integer",4);

    
    parm = AVSadd_parameter("message5","string","Axis Min (3 floats)", "","");
    AVSconnect_widget(parm,"text");
    AVSadd_parameter_prop(parm,"width","integer",4);
    
    parm = AVSadd_parameter(AMNX,"string",INIT_STRING,NULL,NULL);
    AVSadd_parameter_prop(parm,"width","integer",4);
    
    parm = AVSadd_parameter("message6","string","Axis Max (3 floats)", "","");
    AVSconnect_widget(parm,"text");
    AVSadd_parameter_prop(parm,"width","integer",4);
    
    parm = AVSadd_parameter(AMXX,"string",INIT_STRING,NULL,NULL);
    AVSadd_parameter_prop(parm,"width","integer",4);
    
    AVSset_compute_proc(axis_compute);
}


/******************************************************************************/

axis_compute(inf, outg, mess1, x_ticks, y_ticks, z_ticks,t_size,
                         mess2, xz_pos, xy_pos, yz_pos, xz_pl, xy_pl, yz_pl,
                         mess3, font_size, precision, font_no,
	                        x_lab_den, y_lab_den,
                                z_lab_den, origin_labels, show_labels,
                         mess4, mess5,axis_min, mess6,axis_max)
AVSfield *inf;
GEOMedit_list **outg;
char *mess1,*mess2,*mess3,*mess4;    /* Dummy messages for titles.  */
char *mess5,*mess6;
int x_ticks,y_ticks,z_ticks;         /* Number of ticks per axis.   */
float *t_size;                       /* Tick size.                  */
float *xz_pos,*xy_pos,*yz_pos;       /* Positioning of planes.      */
int xz_pl,xy_pl,yz_pl;               /* Plane on/off switchs.       */
float *font_size;                    /* Size of label font.         */
int precision;                       /* Label precision. */
int font_no;          /* mysterious avs font numbers 0..21 */
int show_labels;                     /* Turn labels on and off. */
char *origin_labels;                 /* Axis whose labels are at origin, */
int x_lab_den,y_lab_den,z_lab_den;   /* Number of ticks per label.  */
char *axis_min,*axis_max;
{
    GEOMobj *axis_obj;
    GEOMobj *label_x_obj;
    GEOMobj *label_y_obj;
    GEOMobj *label_z_obj;
    int label_flg;
    float axis_pts[6];
    int i, size, num_ticks[3], which_planes[3], label_density[3];
    float where[3];
    float *t;
    char buffer[128],*c1,*c2,*c3,*c4;
    double foo;
    extern double strtod();
    float min_extents[3],max_extents[3];
    int npoints;
    int two_d_field=0;
    double fake_zlimits[2];
    float computed_tick_size,temp;

    /**
      * the following is a somewhat hokey way to verify that only certain
      * types of input fields get processed.
      *
      * we will permit all fields with 3 coord dimensions, or 3 data
      * dimensions.  additionally, we will permit 2D fields which are not
      * irregular.  for uniform fields, either the extents or points
      * are used for the x/y extents.  for rectilinear fields, the coords
      * provide the x/y extents.  the Z values will be derived from the
      * data itself.
    **/

    i = 1;  
    if (inf->ndim != 3 && inf->nspace != 3)
	i = 0;
    if ((inf->ndim == 2) && (inf->uniform != IRREGULAR))
    {
	compute_fake_zlimits(fake_zlimits,inf);
	two_d_field = 1;
	i = 1;
    }
    
    *outg     = (GEOMedit_list *)GEOMinit_edit_list(*outg);
    axis_obj    = GEOMcreate_obj(GEOM_POLYTRI, GEOM_NULL);
    label_flg   = GEOMcreate_label_flags(font_no,0,0,0,GEOM_LABEL_CENTER,0);
    label_x_obj = GEOMcreate_label(GEOM_NULL,label_flg);
    label_z_obj = GEOMcreate_label(GEOM_NULL,label_flg);
    label_flg   = GEOMcreate_label_flags(font_no,0,0,0,GEOM_LABEL_RIGHT,0);
    label_y_obj = GEOMcreate_label(GEOM_NULL,label_flg);

    /* Put this stuff in vectors for ease of passing. */
    num_ticks[0] = x_ticks;
    num_ticks[1] = y_ticks;
    num_ticks[2] = z_ticks;
    where[0] = *xz_pos;
    where[1] = *xy_pos;
    where[2] = *yz_pos;
    which_planes[0] = xz_pl;
    which_planes[1] = xy_pl;
    which_planes[2] = yz_pl;
    label_density[0] = x_lab_den;
    label_density[1] = y_lab_den;
    label_density[2] = z_lab_den;

    switch(inf->uniform)
   {
        case UNIFORM:

	/* first, look to see if the parms have already been set.  if
	   not, then look to see if there's any info in the points array
	   which define the extents in space for the uniform field.  if
	   there's something there, then use that info, else use the
	   size of the data as the spatial extent. */
	
	if (strcmp(axis_min,INIT_STRING) == 0)  /* no parameters.
						   need to compute them. */
	{
	    if (inf->points) /* there's data. assume that there's
				  nspace*2 points. watch out for segv's here.*/
	    {
		/* here, we assume that this maps to "having extents."
		   apparently, avs ignores the extents portion of uniform
		   fields, but treats the values in the coords arrays as
		   if it were extents. */
		
		t = inf->points;
		
		axis_pts[0] = *t;
		axis_pts[1] = *(t+1);
		axis_pts[2] = *(t+2);
		axis_pts[3] = *(t+3);
		if (two_d_field)
		{
		    axis_pts[4] = fake_zlimits[0];
		    axis_pts[5] = fake_zlimits[1];
		}
		else
		{
		    axis_pts[4] = *(t+4);
		    axis_pts[5] = *(t+5);
		}
	    }
	    else /* make something up. */
	    {
#ifdef AVS3
		axis_pts[0] = axis_pts[2] = axis_pts[4] = 0;
		axis_pts[1] = inf->dimensions[0]-1;
		axis_pts[3] = inf->dimensions[1]-1;
		axis_pts[5] = inf->dimensions[2]-1;
#else
		axis_pts[0] = axis_pts[2] = axis_pts[4] = -1.0;
		axis_pts[1] = 1.0;
		axis_pts[3] = 1.0;
		axis_pts[5] = 1.0;
#endif
	    }
	    
	    /* now, sprintf the floats into a char buffer for use
	       on setting the module widgets to reflect wtf is going on */
	    
	    sprintf(buffer,"%g %g %g",axis_pts[0],axis_pts[2],axis_pts[4]);
	    AVSmodify_parameter(AMNX,AVS_VALUE,buffer,"","");
		
	    sprintf(buffer,"%g %g %g",axis_pts[1],axis_pts[3],axis_pts[5]);
	    AVSmodify_parameter(AMXX,AVS_VALUE,buffer,"","");

	    AVSmodify_parameter("xy plane position",(AVS_VALUE | AVS_MINVAL | AVS_MAXVAL),axis_pts+4,axis_pts+4,axis_pts+5);
	    AVSmodify_parameter("xz plane position",(AVS_VALUE | AVS_MINVAL | AVS_MAXVAL),axis_pts+2,axis_pts+2,axis_pts+3);
	    AVSmodify_parameter("yz plane position",(AVS_VALUE | AVS_MINVAL | AVS_MAXVAL),axis_pts+0,axis_pts+0,axis_pts+1);

	    temp = axis_pts[0] - axis_pts[1];
	    temp = (temp < 0) ? (-1. * temp) : temp;
	    computed_tick_size = temp;
	    temp = axis_pts[2] - axis_pts[3];
	    temp = (temp < 0) ? (-1. * temp) : temp;
	    computed_tick_size = (computed_tick_size >= temp) ?
		computed_tick_size : temp;
	    temp = axis_pts[4] - axis_pts[5];
	    temp = (temp < 0) ? (-1. * temp) : temp;
	    computed_tick_size = (computed_tick_size >= temp) ?
		computed_tick_size : temp;
	    computed_tick_size *= MAX_TICK_PROPORTION;
	    temp = computed_tick_size;

	    AVSmodify_parameter("tick size",AVS_MAXVAL,&temp,&temp,&temp);
	    
	}
	else
	{
	    /* use the parameters as specified. do the min values first */
	    c1 = c2 = axis_min;
	    
	    for (i=0;i<6;i+=2)
	    {
		foo = strtod(c1,&c2);
		if (c1 == c2)
		{
		    AVSwarning(" badly formed numeric values: %s",c1);
		    return(0);
		}
		axis_pts[i] = foo;
		c1 = c2;
	    }
	    /* now do the max points. */
	    c1 = c2 = axis_max;
	    for (i=1;i<6;i+=2)
	    {
		foo = strtod(c1,&c2);
		if (c1 == c2)
		{
		    AVSwarning(" badly formed numeric values: %s",c1);
		    return(0);
		}
		axis_pts[i] = foo;
		c1 = c2;
	    }

	}
	break;
	
        case RECTILINEAR:
	/* here, we will check to see if the parameters have been
	   defined.  if not, then we will compute them based upon the values
	   supplied in the points array, but only if the extents for the
	   field have not already been set */
	
	if (strcmp(axis_min,INIT_STRING) == 0)  /* no parameters.
						   need to compute them. */
	{
	    if (AVSfield_get_extent(inf,min_extents,max_extents))
	    {
		for (i=0;i<3;i++)
		    axis_pts[((i==0)?(0):(1)) <<i] = min_extents[i];
		for (i=0;i<3;i++)
		    axis_pts[(((i==0)?(0):(1)) <<i) + 1] = max_extents[i];
		if (two_d_field)
		{
		    axis_pts[4] = fake_zlimits[0];
		    axis_pts[5] = fake_zlimits[1];
		}
	    }
	    else
	    {
		t = inf->points;
		axis_pts[0] = *t;
		t += inf->dimensions[0]-1;
		axis_pts[1] = *t;
		t++;
		axis_pts[2] = *t;
		t += inf->dimensions[1]-1;
		axis_pts[3] = *t;
		t++;
		if (two_d_field)
	        {
		    axis_pts[4] = fake_zlimits[0];
		    axis_pts[5] = fake_zlimits[1];
		}
		else
		{
		    axis_pts[4] = *t;
		    t+= inf->dimensions[2]-1;
		    axis_pts[5] = *(t+5);
		}
	    }
	    /* now, sprintf the floats into a char buffer for use
	       on setting the module widgets to reflect wtf is going on */
	    
	    sprintf(buffer,"%g %g %g",axis_pts[0],axis_pts[2],axis_pts[4]);
	    AVSmodify_parameter(AMNX,AVS_VALUE,buffer,"","");
		
	    sprintf(buffer,"%g %g %g",axis_pts[1],axis_pts[3],axis_pts[5]);
	    AVSmodify_parameter(AMXX,AVS_VALUE,buffer,"","");

	    AVSmodify_parameter("xy plane position",(AVS_VALUE | AVS_MINVAL | AVS_MAXVAL),axis_pts+4,axis_pts+4,axis_pts+5);
	    AVSmodify_parameter("xz plane position",(AVS_VALUE | AVS_MINVAL | AVS_MAXVAL),axis_pts+2,axis_pts+2,axis_pts+3);
	    AVSmodify_parameter("yz plane position",(AVS_VALUE | AVS_MINVAL | AVS_MAXVAL),axis_pts+0,axis_pts+0,axis_pts+1);
	    
	    temp = axis_pts[0] - axis_pts[1];
	    temp = (temp < 0) ? (-1. * temp) : temp;
	    computed_tick_size = temp;
	    temp = axis_pts[2] - axis_pts[3];
	    temp = (temp < 0) ? (-1. * temp) : temp;
	    computed_tick_size = (computed_tick_size >= temp) ?
		computed_tick_size : temp;
	    temp = axis_pts[4] - axis_pts[5];
	    temp = (temp < 0) ? (-1. * temp) : temp;
	    computed_tick_size = (computed_tick_size >= temp) ?
		computed_tick_size : temp;
	    computed_tick_size *= MAX_TICK_PROPORTION;
	    temp = computed_tick_size;

	    AVSmodify_parameter("tick size",AVS_MAXVAL,&temp,&temp,&temp);
	}
	else
	{
	    /* use the parameters. */
	    /* use the parameters as specified. do the min values first */
	    c1 = c2 = axis_min;
	    
	    for (i=0;i<6;i+=2)
	    {
		foo = strtod(c1,&c2);
		if (c1 == c2)
		{
		    AVSwarning(" badly formed numeric values: %s",c1);
		    return(0);
		}
		axis_pts[i] = foo;
		c1 = c2;
	    }
	    /* now do the max points. */
	    c1 = c2 = axis_max;
	    for (i=1;i<6;i+=2)
	    {
		foo = strtod(c1,&c2);
		if (c1 == c2)
		{
		    AVSwarning(" badly formed numeric values: %s",c1);
		    return(0);
		}
		axis_pts[i] = foo;
		c1 = c2;
	    }
	}

#if 0
	make_axis(axis_obj,axis_pts);
	make_ticks(axis_obj,axis_pts,num_ticks,t_size,inf->uniform);
	make_planes(axis_obj,axis_pts,num_ticks,which_planes,where,
		    inf->uniform);
	make_labels(axis_pts,num_ticks,inf->uniform, label_x_obj,
		    label_y_obj,label_z_obj, show_labels,precision,
		    label_density,font_size,origin_labels);
#endif
	break;
	
        case IRREGULAR:

	/* again, check to see if the parameters have been set.  if so,
	   use them.  if not, compute them. */
	if (strcmp(axis_min,INIT_STRING) == 0)
	{
	    /* no parameters. need to compute them. */
	    if (AVSfield_get_extent(inf,min_extents,max_extents))
	    {
		for (i=0;i<3;i++)
		    axis_pts[((i==0)?(0):(1)) <<i] = min_extents[i];
		for (i=0;i<3;i++)
		    axis_pts[(((i==0)?(0):(1)) <<i) + 1] = max_extents[i];
	    }
	    else
	    {
		for(i=0,npoints=1;i<inf->ndim;i++)
		    npoints *= inf->dimensions[i];
	    
		/* search x coordinates */
	    
		t = inf->points;
		axis_pts[0] = 1e10; /* some big number */
		axis_pts[1] = -1e10; /* some big negative number */
		for (i=0;i<npoints;i++,t++)
		{
		    axis_pts[0] = min(axis_pts[0],*t);
		    axis_pts[1] = max(axis_pts[1],*t);
		}
	    
		/* search y coordinates */
		
		axis_pts[2] = 1e10; /* some big number */
		axis_pts[3] = -1e10; /* some big negative number */
		for (i=0;i<npoints;i++,t++)
		{
		    axis_pts[2] = min(axis_pts[2],*t);
		    axis_pts[3] = max(axis_pts[3],*t);
		}
	    
		/* search z coordinates */
	    
		axis_pts[4] = 1e10; /* some big number */
		axis_pts[5] = -1e10; /* some big negative number */
		for (i=0;i<npoints;i++,t++)
		{
		    axis_pts[4] = min(axis_pts[4],*t);
		    axis_pts[5] = max(axis_pts[5],*t);
		}
	    }
	    /* now, sprintf the floats into a char buffer for use
	       on setting the module widgets to reflect wtf is going on */
	    
	    sprintf(buffer,"%g %g %g",axis_pts[0],axis_pts[2],axis_pts[4]);
	    AVSmodify_parameter(AMNX,AVS_VALUE,buffer,"","");
		
	    sprintf(buffer,"%g %g %g",axis_pts[1],axis_pts[3],axis_pts[5]);
	    AVSmodify_parameter(AMXX,AVS_VALUE,buffer,"","");
	    
	    AVSmodify_parameter("xy plane position",(AVS_VALUE | AVS_MINVAL | AVS_MAXVAL),axis_pts+4,axis_pts+4,axis_pts+5);
	    AVSmodify_parameter("xz plane position",(AVS_VALUE | AVS_MINVAL | AVS_MAXVAL),axis_pts+2,axis_pts+2,axis_pts+3);
	    AVSmodify_parameter("yz plane position",(AVS_VALUE | AVS_MINVAL | AVS_MAXVAL),axis_pts+0,axis_pts+0,axis_pts+1);
	    
	    temp = axis_pts[0] - axis_pts[1];
	    temp = (temp < 0) ? (-1. * temp) : temp;
	    computed_tick_size = temp;
	    temp = axis_pts[2] - axis_pts[3];
	    temp = (temp < 0) ? (-1. * temp) : temp;
	    computed_tick_size = (computed_tick_size >= temp) ?
		computed_tick_size : temp;
	    temp = axis_pts[4] - axis_pts[5];
	    temp = (temp < 0) ? (-1. * temp) : temp;
	    computed_tick_size = (computed_tick_size >= temp) ?
		computed_tick_size : temp;
	    computed_tick_size *= MAX_TICK_PROPORTION;
	    temp = computed_tick_size;

	    AVSmodify_parameter("tick size",AVS_MAXVAL,&temp,&temp,&temp);
	}
	else /* use the parameters */
	{
	    /* use the parameters. */
	    /* use the parameters as specified. do the min values first */
	    c1 = c2 = axis_min;
	    
	    for (i=0;i<6;i+=2)
	    {
		foo = strtod(c1,&c2);
		if (c1 == c2)
		{
		    AVSwarning(" badly formed numeric values: %s",c1);
		    return(0);
		}
		axis_pts[i] = foo;
		c1 = c2;
	    }
	    /* now do the max points. */
	    c1 = c2 = axis_max;
	    for (i=1;i<6;i+=2)
	    {
		foo = strtod(c1,&c2);
		if (c1 == c2)
		{
		    AVSwarning(" badly formed numeric values: %s",c1);
		    return(0);
		}
		axis_pts[i] = foo;
		c1 = c2;
	    }
	}

	break;
	
        default : break;
    } /* of switch */

    make_axis(axis_obj,axis_pts);
    make_ticks(axis_obj,axis_pts,num_ticks,t_size,inf->uniform);
    make_planes(axis_obj,axis_pts,num_ticks,which_planes,where,
		inf->uniform);
    make_labels(axis_pts,num_ticks,inf->uniform, label_x_obj,
		label_y_obj,label_z_obj, show_labels,precision,
		label_density,font_size,origin_labels);
    
    /* Add into the edit list */
    GEOMedit_geometry(*outg, "3D_axis", axis_obj);
    GEOMedit_geometry(*outg, "3D_axis", label_x_obj);
    GEOMedit_geometry(*outg, "3D_axis", label_y_obj);
    GEOMedit_geometry(*outg, "3D_axis", label_z_obj);
    /* Now kill it all. */
    GEOMdestroy_obj(axis_obj);
    GEOMdestroy_obj(label_x_obj);
    GEOMdestroy_obj(label_y_obj);
    GEOMdestroy_obj(label_z_obj);
    return(1);
}


/******************************************************************************/

make_axis(obj,axis_pts)
GEOMobj *obj;
float axis_pts[6];
/*
 *  This function loads 3D coordinates for 3 lines into axis_lines[].
 *  These lines will then be added to a list of ines for display
 *  by GEOMadd_disjoint_line().  axis_lines[] stores the lines as
 *  x, y, and z axis, where each axis is specified by the three
 *  coordinate values for its 2 endpoints (axis_lines[0][?] and
 *  axis_lines[1][?] make up the x axis).  Colors are stored in
 *  axis_color by rgb values in a similar way.
 */
{
    float axis_color[6][3],
          axis_lines[6][3];

    axis_lines[0][0] = axis_pts[0];    /* x axis */
    axis_lines[0][1] = axis_pts[2];
    axis_lines[0][2] = axis_pts[4];

    axis_lines[1][0] = axis_pts[1];
    axis_lines[1][1] = axis_pts[2];
    axis_lines[1][2] = axis_pts[4];

    axis_lines[2][0] = axis_pts[0];    /* y axis */
    axis_lines[2][1] = axis_pts[2];
    axis_lines[2][2] = axis_pts[4];

    axis_lines[3][0] = axis_pts[0];
    axis_lines[3][1] = axis_pts[3];
    axis_lines[3][2] = axis_pts[4];

    axis_lines[4][0] = axis_pts[0];    /* z axis */
    axis_lines[4][1] = axis_pts[2];
    axis_lines[4][2] = axis_pts[4];

    axis_lines[5][0] = axis_pts[0];
    axis_lines[5][1] = axis_pts[2];
    axis_lines[5][2] = axis_pts[5];

    axis_color[0][0] = 1.0;
    axis_color[0][1] = 0.0;
    axis_color[0][2] = 0.0;

    axis_color[1][0] = 1.0;
    axis_color[1][1] = 0.0;
    axis_color[1][2] = 0.0;

    axis_color[2][0] = 0.0;
    axis_color[2][1] = 1.0;
    axis_color[2][2] = 0.0;

    axis_color[3][0] = 0.0;
    axis_color[3][1] = 1.0;
    axis_color[3][2] = 0.0;

    axis_color[4][0] = 0.0;
    axis_color[4][1] = 1.0;
    axis_color[4][2] = 1.0;

    axis_color[5][0] = 0.0;
    axis_color[5][1] = 1.0;
    axis_color[5][2] = 1.0;

    GEOMadd_disjoint_line(obj,axis_lines,axis_color,6,GEOM_COPY_DATA);
}

make_ticks(axis_obj,axis_pts,num_ticks,tick_size,data_type)
GEOMobj *axis_obj;
float axis_pts[6];
int num_ticks[3];
float *tick_size;
int data_type;
/*
 *  This function places ticks along each axis.  There are actually 2
 *  marks, one parallel to each of the other 2 axis.  We get the
 *  increment value and work our way along the axis making the 2 marks.
 *
 *  In the case of a UNIFORM data set, axis_pts[] contains the dimensions
 *  of the data set, but the actual plotting is normalized to the interval
 *  -1 to 1.  So we first save the dimensions and then for each axis get
 *  an increment for the dimensions, normalize the increment, and change
 *  axis_pts[] to normalized dimensions (-1 to 1).  (After each axis is
 *  done we restore axis_pts[] to the original dimensions so we can do
 *  the next one.)
 */
{
    float get_increment();
    float increment;
    float loc;
    float t_size = 2 * *tick_size;
    float axis_tick[4][3];
    float tick_color[4][3];
    int i;
#ifndef AVS3
    float temp_axis_pts[6];
#endif

#ifndef AVS3
    if (data_type == UNIFORM)
        for (i = 0; i < 6; ++i)
            temp_axis_pts[i] = axis_pts[i];
#endif

    /* Ticks on x axis. */
    if (num_ticks[0]) {
        increment = get_increment(num_ticks[0],axis_pts[0],axis_pts[1]);
#ifndef AVS3
        if (data_type == UNIFORM)
            normalize(&increment,axis_pts,axis_pts[0],axis_pts[1]);
#endif
        set_tick_color(tick_color, 1.0, 0.0, 0.0);
        loc = axis_pts[0] + increment;
	for (i=0;i<=num_ticks[0];i++)
	{
            axis_tick[0][0] = loc;
            axis_tick[0][1] = axis_pts[2];
            axis_tick[0][2] = axis_pts[4] + t_size;

            axis_tick[1][0] = loc;
            axis_tick[1][1] = axis_pts[2];
            axis_tick[1][2] = axis_pts[4];

            axis_tick[2][0] = loc;
            axis_tick[2][1] = axis_pts[2] + t_size;
            axis_tick[2][2] = axis_pts[4];

            axis_tick[3][0] = loc;
            axis_tick[3][1] = axis_pts[2];
            axis_tick[3][2] = axis_pts[4];

            GEOMadd_disjoint_line(axis_obj,axis_tick,tick_color,4,GEOM_COPY_DATA);
            loc += increment;
        }
#ifndef AVS3
        if (data_type == UNIFORM)
            for (i = 0; i < 6; ++i)
                axis_pts[i] = temp_axis_pts[i];
#endif
    }

    /* Ticks on y axis. */
    if (num_ticks[1]) {
        increment = get_increment(num_ticks[1],axis_pts[2],axis_pts[3]);
#ifndef AVS3
        if (data_type == UNIFORM)
            normalize(&increment,axis_pts,axis_pts[2],axis_pts[3]);
#endif
        set_tick_color(tick_color, 0.0, 1.0, 0.0);
        loc = axis_pts[2] + increment;
	for (i=0;i<=num_ticks[1];i++)
	{
            axis_tick[0][0] = axis_pts[0];
            axis_tick[0][1] = loc;
            axis_tick[0][2] = axis_pts[4] + t_size;

            axis_tick[1][0] = axis_pts[0];
            axis_tick[1][1] = loc;
            axis_tick[1][2] = axis_pts[4];

            axis_tick[2][0] = axis_pts[0] + t_size;
            axis_tick[2][1] = loc;
            axis_tick[2][2] = axis_pts[4];

            axis_tick[3][0] = axis_pts[0];
            axis_tick[3][1] = loc;
            axis_tick[3][2] = axis_pts[4];

            GEOMadd_disjoint_line(axis_obj,axis_tick,tick_color,4,GEOM_COPY_DATA);
            loc += increment;
        }
#ifndef AVS3
        if (data_type == UNIFORM)
            for (i = 0; i < 6; ++i)
                axis_pts[i] = temp_axis_pts[i];
#endif
    }

    /* Ticks on z axis. */
    if (num_ticks[2]) {
        increment = get_increment(num_ticks[2],axis_pts[4],axis_pts[5]);
#ifndef AVS3
        if (data_type == UNIFORM)
            normalize(&increment,axis_pts,axis_pts[4],axis_pts[5]);
#endif
        set_tick_color(tick_color, 0.0, 1.0, 1.0);
        loc = axis_pts[4] + increment;
	for(i=0;i<=num_ticks[2];i++)
	{
            axis_tick[0][0] = axis_pts[0] + t_size;
            axis_tick[0][1] = axis_pts[2];
            axis_tick[0][2] = loc;

            axis_tick[1][0] = axis_pts[0];
            axis_tick[1][1] = axis_pts[2];
            axis_tick[1][2] = loc;

            axis_tick[2][0] = axis_pts[0];
            axis_tick[2][1] = axis_pts[2] + t_size;
            axis_tick[2][2] = loc;

            axis_tick[3][0] = axis_pts[0];
            axis_tick[3][1] = axis_pts[2];
            axis_tick[3][2] = loc;

            GEOMadd_disjoint_line(axis_obj,axis_tick,tick_color,4,GEOM_COPY_DATA);
            loc += increment;
        }
#ifndef AVS3
        if (data_type == UNIFORM)
            for (i = 0; i < 6; ++i)
                axis_pts[i] = temp_axis_pts[i];
#endif
    }
}

make_planes(obj,axis_pts,mesh_size,which_planes,where,data_type)
GEOMobj *obj;
float axis_pts[6];
int mesh_size[3];       /* this is the number of ticks */
int which_planes[3];
float where[3];
int data_type;
/*
 *  This routine creates mesh planes by extending perpendiculars to the
 *  appropriate axis at tick mark locations.  This is all very similar
 *  to the make_ticks() function.
 */
{
    float get_increment();
    float increment;
    float loc;
    float mesh_line[2][3];
    float plane_color[2][3];
    int i;
    int ct;
#ifndef AVS3
    float temp_axis_pts[6];

    if (data_type == UNIFORM)
        for (i = 0; i < 6; ++i)
            temp_axis_pts[i] = axis_pts[i];
#endif

    /* xz plane */
    if (which_planes[0]) {
        increment = get_increment(mesh_size[0],axis_pts[0],axis_pts[1]);
#ifndef AVS3
        if (data_type == UNIFORM)
            normalize(&increment,axis_pts,axis_pts[0],axis_pts[1]);
#endif
        set_plane_color(plane_color, 0.0, 1.0, 0.0);
        loc = axis_pts[0];
	for(ct=0;ct<=mesh_size[0]+1;ct++)
	{
            mesh_line[0][0] = loc;
	    mesh_line[0][1] = where[0]; /*not mesh_line[0][0] SRT*/
            mesh_line[0][2] = axis_pts[4];
            mesh_line[1][0] = loc;
	    mesh_line[1][1] = where[0];
            mesh_line[1][2] = axis_pts[5];
            GEOMadd_disjoint_line(obj,mesh_line,plane_color,2,GEOM_COPY_DATA);
            loc += increment;
        }
#ifndef AVS3
        if (data_type == UNIFORM)
            for (i = 0; i < 6; ++i)
                axis_pts[i] = temp_axis_pts[i];
#endif

        increment = get_increment(mesh_size[2],axis_pts[4],axis_pts[5]);
#ifndef AVS3
        if (data_type == UNIFORM)
            normalize(&increment,axis_pts,axis_pts[4],axis_pts[5]);
#endif
        set_plane_color(plane_color, 0.0, 1.0, 0.0);
        loc = axis_pts[4];
	for (ct=0;ct<=mesh_size[2]+1;ct++)
        {
            mesh_line[0][0] = axis_pts[0];
	    mesh_line[0][1] = where[0];
            mesh_line[0][2] = loc;
            mesh_line[1][0] = axis_pts[1];
	    mesh_line[1][1] = where[0];
            mesh_line[1][2] = loc;
            GEOMadd_disjoint_line(obj,mesh_line,plane_color,2,GEOM_COPY_DATA);
            loc += increment;
        }
#ifndef AVS3
        if (data_type == UNIFORM)
            for (i = 0; i < 6; ++i)
                axis_pts[i] = temp_axis_pts[i];
#endif
    }

    /* xy plane */
    if (which_planes[1]) {
        increment = get_increment(mesh_size[0],axis_pts[0],axis_pts[1]);
#ifndef AVS3
        if (data_type == UNIFORM)
            normalize(&increment,axis_pts,axis_pts[0],axis_pts[1]);
#endif
        set_plane_color(plane_color, 0.0, 1.0, 1.0);
        loc = axis_pts[0];
	for(ct=0;ct<=mesh_size[0]+1;ct++)
	{
            mesh_line[0][0] = loc;
            mesh_line[0][1] = axis_pts[2];
	    mesh_line[0][2] = where[1];
            mesh_line[1][0] = loc;
            mesh_line[1][1] = axis_pts[3];
	    mesh_line[1][2] = where[1];
            GEOMadd_disjoint_line(obj,mesh_line,plane_color,2,GEOM_COPY_DATA);
            loc += increment;
        }
#ifndef AVS3
        if (data_type == UNIFORM)
            for (i = 0; i < 6; ++i)
                axis_pts[i] = temp_axis_pts[i];
#endif

        increment = get_increment(mesh_size[1],axis_pts[2],axis_pts[3]);
#ifndef AVS3
        if (data_type == UNIFORM)
            normalize(&increment,axis_pts,axis_pts[2],axis_pts[3]);
#endif
        set_plane_color(plane_color, 0.0, 1.0, 1.0);
        loc = axis_pts[2];
	for (ct=0;ct<=mesh_size[1]+1;ct++)
	{
            mesh_line[0][0] = axis_pts[0];
            mesh_line[0][1] = loc;
	    mesh_line[0][2] = where[1];
            mesh_line[1][0] = axis_pts[1];
            mesh_line[1][1] = loc;
	    mesh_line[1][2] = where[1];
            GEOMadd_disjoint_line(obj,mesh_line,plane_color,2,GEOM_COPY_DATA);
            loc += increment;
        }
#ifndef AVS3
        if (data_type == UNIFORM)
            for (i = 0; i < 6; ++i)
                axis_pts[i] = temp_axis_pts[i];
#endif
    }

    /* yz plane */
    if (which_planes[2]) {
        increment = get_increment(mesh_size[1],axis_pts[2],axis_pts[3]);
#ifndef AVS3
        if (data_type == UNIFORM)
            normalize(&increment,axis_pts,axis_pts[2],axis_pts[3]);
#endif
        set_plane_color(plane_color, 1.0, 0.0, 0.0);
        loc = axis_pts[2];
	for (ct=0;ct<=mesh_size[1]+1;ct++)
	{
	    mesh_line[0][0] = where[2];
            mesh_line[0][1] = loc;
            mesh_line[0][2] = axis_pts[4];
	    mesh_line[1][0] = where[2];
            mesh_line[1][1] = loc;
            mesh_line[1][2] = axis_pts[5];
            GEOMadd_disjoint_line(obj,mesh_line,plane_color,2,GEOM_COPY_DATA);
            loc += increment;
        }
#ifndef AVS3
        if (data_type == UNIFORM)
            for (i = 0; i < 6; ++i)
                axis_pts[i] = temp_axis_pts[i];
#endif

        increment = get_increment(mesh_size[2],axis_pts[4],axis_pts[5]);
#ifndef AVS3
        if (data_type == UNIFORM)
            normalize(&increment,axis_pts,axis_pts[4],axis_pts[5]);
#endif
        set_plane_color(plane_color, 1.0, 0.0, 0.0);
        loc = axis_pts[4];
	for (ct=0;ct<=mesh_size[2]+1;ct++)
	{
	    mesh_line[0][0] = where[2];
            mesh_line[0][1] = axis_pts[2];
            mesh_line[0][2] = loc;
	    mesh_line[1][0] = where[2];
            mesh_line[1][1] = axis_pts[3];
            mesh_line[1][2] = loc;
            GEOMadd_disjoint_line(obj,mesh_line,plane_color,2,GEOM_COPY_DATA);
            loc += increment;
        }
#ifndef AVS3
        if (data_type == UNIFORM)
            for (i = 0; i < 6; ++i)
                axis_pts[i] = temp_axis_pts[i];
#endif
    }
}

make_labels(axis_pts,num_ticks,data_type,label_x_obj,label_y_obj,
            label_z_obj,show_labels,precision,label_density,
            font_size,origin_labels)
float axis_pts[6];
int num_ticks[3];
int data_type;
GEOMobj *label_x_obj,*label_y_obj,*label_z_obj;
int precision;
int label_density[3];  /* Number of ticks per label. */
float *font_size;
int show_labels;       /* Turn labels on and off. */
char *origin_labels;   /* Axis whose label is at the origin. */
/*
 *  This routine puts labels on the various axis, at tick points,
 *  if requested.  Labels may not occur at every tick, as determined
 *  by the label_density parameter.  This is all very similar to the
 *  make_ticks() function.
 *
 *  Note that for a UNIFORM data set we label the axis according to
 *  the dimensions of the data set, or the user supplied dimensions,
 *  as opposed to the (normalized) screen size of the data.  This
 *  amounts to using the local variables u_inc and u_loc to track the
 *  the values we need for the labels.
 *
 *  origin_labels is a string which we scan for the characters x,y,
 *  and z.  For each one found we will label the origin according
 *  to the min value of that axis.  So the origin can get pretty
 *  busy.  If a particular label is called for we start labeling
 *  at axis_pts[min], otherwise we start at axis_points[min] +
 *  increment.
 */
{
    float get_increment();
    float increment,u_inc;
    float loc,u_loc;
#ifndef AVS3
    float temp_axis_pts[6];
#endif
    float ref_pt[3];
    float offset[3];
    float color[3];
    char num[30];
    char arg[5];
    int i;
    int x_label,y_label,z_label;
    int ct;

    /* Find out which axis labels the origin. */
    x_label = y_label = z_label = FALSE;
    for (i = 0; i < strlen(origin_labels); ++i) {
        if (origin_labels[i] == 'x')
            x_label = TRUE;
        if (origin_labels[i] == 'y')
            y_label = TRUE;
        if (origin_labels[i] == 'z')
            z_label = TRUE;
    }

    /* But show the x origin if everyone is the same. */
    if (axis_pts[0] == axis_pts[2] && axis_pts[0] == axis_pts[4]
                                   && y_label == FALSE
                                   && z_label == FALSE)
        x_label = TRUE;

    /* Set the number of digits to the right of the decimal via precision. */
    arg[0] = '%';
    arg[1] = '.';
    arg[2] = precision + 0x30;
    arg[3] = 'f';
    arg[4] = '\0';

    color[0] = color[1] = color[2] = 1.0;
#ifndef AVS3
    if (data_type == UNIFORM)
        for (i = 0; i < 6; ++i)
            temp_axis_pts[i] = axis_pts[i];
#endif

    /* Labels on x axis. */
    if (num_ticks[0] && show_labels) {
        increment = u_inc = get_increment(num_ticks[0],axis_pts[0],axis_pts[1]);
#ifndef AVS3
        if (data_type == UNIFORM)
            normalize(&increment,axis_pts,axis_pts[0],axis_pts[1]);
#endif
        loc = axis_pts[0] + ((x_label) ? 0.0 : increment);
#ifndef AVS3
        if (data_type == UNIFORM) /* Get labels from data dimensions or user. */
            u_loc = temp_axis_pts[0] + ((x_label) ? 0.0 : u_inc);
#endif
	for (ct=0;ct<=(num_ticks[0]/label_density[0]+(x_label?1:0));ct++)
	{
            ref_pt[0] = loc;
            ref_pt[1] = axis_pts[2];
            ref_pt[2] = axis_pts[4];
            offset[0] = 0.0;
            offset[1] = -0.06;
            offset[2] = 0.0;
#ifdef AVS3
            sprintf(num,arg,loc);
#else
            sprintf(num,arg,(data_type == UNIFORM) ? u_loc : loc);
#endif
            GEOMadd_label(label_x_obj,num,ref_pt,offset,*font_size,color,-1);
            loc += increment * label_density[0];
#ifndef AVS3
            if (data_type == UNIFORM)
                u_loc += u_inc * label_density[0];
#endif
        }
	/* do the last label */
	if (x_label)
	{
#ifdef AVS3
	    loc = axis_pts[1];
	    sprintf(num,arg,loc);
#else
	    u_loc = axis_pts[1];  /* NOT TESTED FOR AVS 2 !! 22may1991 w.bethel */
	    sprintf(num,arg,loc);
#endif
	    GEOMadd_label(label_x_obj,num,ref_pt,offset,*font_size,color,-1);
	}
		
#ifndef AVS3
        if (data_type == UNIFORM)
            for (i = 0; i < 6; ++i)
                axis_pts[i] = temp_axis_pts[i];
#endif
    }

    /* Labels on y axis. */
    if (num_ticks[1] && show_labels) {
        increment = u_inc = get_increment(num_ticks[1],axis_pts[2],axis_pts[3]);
#ifndef AVS3
        if (data_type == UNIFORM)
            normalize(&increment,axis_pts,axis_pts[2],axis_pts[3]);
#endif
        loc = axis_pts[2] + ((y_label) ? 0.0 : increment);
#ifndef AVS3
        if (data_type == UNIFORM)
            u_loc = temp_axis_pts[2] + ((y_label) ? 0.0 : u_inc);
#endif
	for (ct=0;ct<=(num_ticks[1]/label_density[1] + (y_label?1:0));ct++)
	{
            ref_pt[0] = axis_pts[0];
            ref_pt[1] = loc;
            ref_pt[2] = axis_pts[4];
            offset[0] = -0.02;
            offset[1] = -0.02;
            offset[2] = 0.0;
#ifdef AVS3
            sprintf(num,arg,loc);
#else
            sprintf(num,arg,(data_type == UNIFORM) ? u_loc : loc);
#endif
            GEOMadd_label(label_y_obj,num,ref_pt,offset,*font_size,color,-1);
            loc += increment * label_density[1];
#ifndef AVS3
            if (data_type == UNIFORM)
                u_loc += u_inc * label_density[1];
#endif
        }
	/* do the last label */
	if (y_label)
	{
#ifdef AVS3
	    loc = axis_pts[3];
	    sprintf(num,arg,loc);
#else
	    u_loc = axis_pts[3];  /* NOT TESTED FOR AVS 2 !! 22may1991 w.bethel */
	    sprintf(num,arg,loc);
#endif
	    GEOMadd_label(label_x_obj,num,ref_pt,offset,*font_size,color,-1);
	}
#ifndef AVS3
        if (data_type == UNIFORM)
            for (i = 0; i < 6; ++i)
                axis_pts[i] = temp_axis_pts[i];
#endif
    }

    /* Labels on z axis. */
    if (num_ticks[2] && show_labels) {
        increment = u_inc = get_increment(num_ticks[2],axis_pts[4],axis_pts[5]);
#ifndef AVS3
        if (data_type == UNIFORM)
            normalize(&increment,axis_pts,axis_pts[4],axis_pts[5]);
#endif
        loc = axis_pts[4] + ((z_label) ? 0.0 : increment);
#ifndef AVS3
        if (data_type == UNIFORM)
            u_loc = temp_axis_pts[4] + ((z_label) ? 0.0 : u_inc);
#endif
        for (ct=0;ct<=(num_ticks[2]/label_density[2]+(z_label?1:0));ct++)
	{
            ref_pt[0] = axis_pts[0];
            ref_pt[1] = axis_pts[2];
            ref_pt[2] = loc;
            offset[0] = 0.0;
            offset[1] = -0.06;
            offset[2] = 0.0;
#ifdef AVS3
            sprintf(num,arg,loc);
#else
            sprintf(num,arg,(data_type == UNIFORM) ? u_loc : loc);
#endif
            GEOMadd_label(label_z_obj,num,ref_pt,offset,*font_size,color,-1);
            loc += increment * label_density[2];
#ifndef AVS3
            if (data_type == UNIFORM)
                u_loc += u_inc * label_density[2];
#endif
        }
	/* do the last label */
	if (z_label)
	{
#ifdef AVS3
	    loc = axis_pts[5];
	    sprintf(num,arg,loc);
#else
	    u_loc = axis_pts[5];  /* NOT TESTED FOR AVS 2 !! 22may1991 w.bethel */
	    sprintf(num,arg,loc);
#endif
	    GEOMadd_label(label_x_obj,num,ref_pt,offset,*font_size,color,-1);
	}
#ifndef AVS3
        if (data_type == UNIFORM)
            for (i = 0; i < 6; ++i)
                axis_pts[i] = temp_axis_pts[i];
#endif
    }
}


/************** Support stuff. **********************************************/

set_tick_color(tc,r,g,b)
float tc[4][3];
float r,g,b;
{
    tc[0][0] = tc [1][0] = tc[2][0] = tc[3][0] = r;
    tc[0][1] = tc [1][1] = tc[2][1] = tc[3][1] = g;
    tc[0][2] = tc [1][2] = tc[2][2] = tc[3][2] = b;
}

set_plane_color(pc,r,g,b)
float pc[2][3];
float r,g,b;
{
    pc[0][0] = pc [1][0] = r;
    pc[0][1] = pc [1][1] = g;
    pc[0][2] = pc [1][2] = b;
}

normalize(increment,axis_pts,min_pt,max_pt)
float *increment;
float axis_pts[6];
float min_pt,max_pt;
/*
 *  This is a uniform data set so we want to scale increment
 *  and set the axis dimensions back to -1,1.
 **** undone 22 may 1991
 */
{
    *increment = 2 * *increment / (max_pt - min_pt);
    axis_pts[0] = axis_pts[2] = axis_pts[4] = -1.0;
    axis_pts[1] = axis_pts[3] = axis_pts[5] = 1.0;
}


/************** Increment calculation. **************************************/

double mypow(),nicenum();

float get_increment(ticks,axis_min,axis_max)
int ticks;
float axis_min,axis_max;
{
    double range,d;
#if 0
    if (ticks <= (axis_max - axis_min)) {
        d = (double)((int)((axis_max - axis_min)/ticks + 0.5));
    }
    else {
        range = nicenum(axis_max - axis_min, 0);
        d = nicenum(range / (ticks - 1), 1);
    }
#endif
    d = (axis_max - axis_min)/(ticks+1);
    return (float)d;
}

double nicenum(x, round)
double x;
int round;
/*
 * nicenum: find a "nice" number approximately equal to x
 * round if round=1, ceil if round=0
 */
{
    long exp;
    double f, y;

    exp = floor(log10(x));
    f = x/mypow(10., exp);   /* fraction between 1 and 10 */
    if (round)
        if (f<1.5) y = 1.;
        else if (f<3.) y = 2.;
        else if (f<7.) y = 5.;
        else y = 10.;
    else
        if (f<=1.) y = 1.;
        else if (f<=2.) y = 2.;
        else if (f<=5.) y = 5.;
        else y = 10.;
    return y*mypow(10., exp);
}


double mypow(a, n)
double a;
register int n;
/*
 * mypow(a,n)=a^n for integer n
 * roundoff errors in pow were causing problems, so I wrote my own
 */
{
    double x;

    x = 1.;
    if (n>0)
        for (; n>0; n--)
            x *= a;
    else
        for (; n<0; n++)
            x /= a;
    return x;
}

int
compute_fake_zlimits(fz,f)
double *fz;
AVSfield *f;
{
    double minval,maxval;
    int i,num_pts;
    unsigned char *bp;
    int *bi;
    float *bf;
    double *bd;

    minval = 1e12;
    maxval = -1e12;

    num_pts = f->dimensions[0] * f->dimensions[1] * f->veclen;

    switch (f->type)
    {
    case AVS_TYPE_BYTE:
	bp = f->field_union.field_data_char;
	for (i=0;i<num_pts;i++,bp++)
	{
	    if (*bp < minval)
		minval = *bp;
	    if (*bp > maxval)
		maxval = *bp;
	}
	break;
    case AVS_TYPE_INTEGER:
	bi = f->field_union.field_data_int_u;
	for (i=0;i<num_pts;i++,bi++)
	{
	    if (*bi < minval)
		minval = *bi;
	    if (*bi > maxval)
		maxval = *bi;
	}
	break;
    case AVS_TYPE_REAL:
	bf = f->field_union.field_data_float_u;
	for (i=0;i<num_pts;i++,bf++)
	{
	    if (*bf < minval)
		minval = *bf;
	    if (*bf > maxval)
		maxval = *bf;
	}
	break;
    case AVS_TYPE_DOUBLE:
	bd = f->field_union.field_data_double_u;
	for (i=0;i<num_pts;i++,bd++)
	{
	    if (*bd < minval)
		minval = *bd;
	    if (*bd > maxval)
		maxval = *bd;
	}
	break;
    }
    *fz = minval;
    *(fz+1) = maxval;
}
