/****************************************************************************
                  INTERNATIONAL AVS CENTER
	(This disclaimer must remain at the top of all files)

WARRANTY DISCLAIMER

This module and the files associated with it are distributed free of charge.
It is placed in the public domain and permission is granted for anyone to use,
duplicate, modify, and redistribute it unless otherwise noted.  Some modules
may be copyrighted.  You agree to abide by the conditions also included in
the AVS Licensing Agreement, version 1.0, located in the main module
directory located at the International AVS Center ftp site and to include
the AVS Licensing Agreement when you distribute any files downloaded from 
that site.

The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module provide absolutely
NO WARRANTY OF ANY KIND with respect to this software.  The entire risk as to
the quality and performance of this software is with the user.  IN NO EVENT
WILL The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module BE LIABLE TO
ANYONE FOR ANY DAMAGES ARISING FROM THE USE OF THIS SOFTWARE, INCLUDING,
WITHOUT LIMITATION, DAMAGES RESULTING FROM LOST DATA OR LOST PROFITS, OR ANY
SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES.

This AVS module and associated files are public domain software unless
otherwise noted.  Permission is hereby granted to do whatever you like with
it, subject to the conditions that may exist in copyrighted materials. Should
you wish to make a contribution toward the improvement, modification, or
general performance of this module, please send us your comments:  why you
liked or disliked it, how you use it, and most important, how it helps your
work. We will receive your comments at avs@ncsc.org.

Please send AVS module bug reports to avs@ncsc.org.

******************************************************************************/
/* 
 * KEYFRAME ANIMATOR:
 * perform keyframe animation and interpolation of objects in a network.
 * interpolation can be either linear or splines, and rotations are
 * interpolated using quaternions.
 *
 * created by Brian Kaplan
 *      kaplan@cica.indiana.edu
 *      Center for Innovative Computer Applications (CICA)
 *      Indiana University
 *
 * thanks to Andrew Hanson and Robert Cross for help with their quaternion
 * interpolation code.
 *
 * Copyright (c) 1993 Brian Kaplan, CICA, Indiana University
 * All rights reserved.
 *
 * This material was developed at the Center for Innovative Computer
 * Applications (CICA), Indiana University.  Permission to copy this
 * software or any portion of it, to redistribute it, and to use it for
 * any purpose and without fee is granted, subject to the following
 * restrictions and understandings.
 *
 * 1. Any copy made of this software must include this copyright notice
 * in full.  Any materials containing this software or any part of this
 * software must include this copyright notice in full.
 *
 * 2. Users of this software agree to make their best efforts (a) to
 * return to the Center for Innovative Computer Applications any improvements
 * or extensions that they make, so that these may be included in future
 * releases; and (b) to inform Indiana University of noteworthy uses of this
 * software.
 *
 * 3. All materials developed as a consequence of the use of this
 * software shall duly acknowledge such use, in accordance with the usual
 * standards of acknowledging credit in academic research.
 *
 * 4. Indiana University has made no warrantee or representation that the
 * operation of this software will be error-free, and Indiana University
 * is under no obligation to provide any services, by way of maintenance,
 * update, or otherwise.
 *
 * 5. In conjunction with products arising from the use of this material,
 * there shall be no use of the name of Indiana University or the Center
 * for Innovative Computer Applications nor of any adaptation thereof in
 * any advertising, promotional, or sales literature without prior written
 * consent from Indiana University in each case.
 *
 */

#include <stdio.h>
#include <avs/flow.h>
#include <avs/avs_data.h>
#include <avs/field.h>
#include <avs/geom.h>
#include <avs/udata.h>
#include <math.h>
#include "Keyframe.h"

/**********************
 * AVS SETUP ROUTINES *
 **********************/

keyframe_desc()
/*****************************************************************************/
/* The description function is used to define the inputs, outputs, and       */
/* parameters to the module, the init, compute, and destroy routines, and    */
/* other module information.                                                 */
/*****************************************************************************/
{
  int keyframe_compute();                 /* the name of the compute routine */
  int keyframe_init();                       /* the name of the init routine */
  int keyframe_destroy();                 /* the name of the destroy routine */
  int inpt, outpt, parm;
  
  /* Set the module  name 	          type */
  AVSset_module_name("Keyframe Animator", MODULE_MAPPER);
  
  /* Create input ports */
  AVScreate_input_port("Frame","integer",OPTIONAL|INVISIBLE);

  inpt = AVScreate_input_port("Pick Info","struct upstream_geom",
			      OPTIONAL|INVISIBLE);
  AVSset_input_class(inpt,"upstream_geom");

  inpt = AVScreate_input_port("Trans Info","struct upstream_transform",
			      OPTIONAL|INVISIBLE);
  AVSset_input_class(inpt,"upstream_transform");

  /* Create output ports */
  AVScreate_output_port( "Geometry", "geom");

  outpt = AVScreate_output_port("Frameout", "integer");
  AVSset_output_flags(outpt,INVISIBLE);

  /* Add parameters to the module and place them where they belong */
  parm = AVSadd_parameter("title","string","Animation:  Untitled",NULL,NULL);
    AVSadd_parameter_prop(parm,"layout","string",
      "manipulator \"$Module:title\" -w text -xy 10,10");
    AVSadd_parameter_prop(parm,"width","integer",3);
  parm = AVSadd_parameter("exit","oneshot",NULL,NULL,NULL);
    AVSadd_parameter_prop(parm,"layout","string",
      "manipulator \"$Module:exit\" -w oneshot -xy 187,10");
    AVSadd_parameter_prop(parm,"width","integer",1);

  parm = AVSadd_parameter(">>>","string","EMPTY-1",NULL,NULL);
    AVSadd_parameter_prop(parm,"layout","string",
      "manipulator \"$Module:>>>\" -w typein -xy 10,36");
    AVSadd_parameter_prop(parm,"width","integer",4);
  parm = AVSadd_parameter("object attributes","choice","name",
			  "name|parent","|");
    AVSadd_parameter_prop(parm,"columns","integer",2);
    AVSadd_parameter_prop(parm,"layout","string",
      "manipulator \"$Module:object attributes\" -w radio_buttons -xy 10,56");

  parm = AVSadd_parameter("new obj","oneshot",NULL,NULL,NULL);
    AVSadd_parameter_prop(parm,"layout","string",
      "manipulator \"$Module:new obj\" -w oneshot -xy 10,86");
    AVSadd_parameter_prop(parm,"width","integer",1);
  parm = AVSadd_parameter("del obj", "oneshot", NULL, NULL, NULL);
    AVSadd_parameter_prop(parm,"layout","string",
      "manipulator \"$Module:del obj\" -w oneshot -xy 69,86");
    AVSadd_parameter_prop(parm,"width","integer",1);
  parm = AVSadd_parameter("add obj", "boolean", 0, NULL, NULL);
    AVSadd_parameter_prop(parm,"layout","string",
      "manipulator \"$Module:add obj\" -w toggle -xy 128,86");
    AVSadd_parameter_prop(parm,"width","integer",1);
  parm = AVSadd_parameter("next obj","oneshot",NULL,NULL,NULL);
    AVSadd_parameter_prop(parm,"layout","string",
      "manipulator \"$Module:next obj\" -w oneshot -xy 187,86");
    AVSadd_parameter_prop(parm,"width","integer",1);

  parm = AVSadd_parameter("ROTATE", "boolean",0,NULL,NULL);
    AVSadd_parameter_prop(parm,"layout","string",
      "manipulator \"$Module:ROTATE\" -w toggle -xy 10,116");
    AVSadd_parameter_prop(parm,"width","integer",1);
  parm = AVSadd_parameter("TRANS", "boolean",0,NULL,NULL);
    AVSadd_parameter_prop(parm,"layout","string",
      "manipulator \"$Module:TRANS\" -w toggle -xy 69,116");
    AVSadd_parameter_prop(parm,"width","integer",1);
  parm = AVSadd_parameter("SCALE", "boolean",0,NULL,NULL);
    AVSadd_parameter_prop(parm,"layout","string",
      "manipulator \"$Module:SCALE\" -w toggle -xy 128,116");
    AVSadd_parameter_prop(parm,"width","integer",1);
  parm = AVSadd_parameter("ACTIVATE ALL", "oneshot",NULL,NULL,NULL);
    AVSadd_parameter_prop(parm,"layout","string",
      "manipulator \"$Module:ACTIVATE ALL\" -w oneshot -xy 10,136");
    AVSadd_parameter_prop(parm,"width","integer",2);
  parm = AVSadd_parameter("DEACTIVATE ALL", "oneshot",NULL,NULL,NULL);
    AVSadd_parameter_prop(parm,"layout","string",
      "manipulator \"$Module:DEACTIVATE ALL\" -w oneshot -xy 128,136");
    AVSadd_parameter_prop(parm,"width","integer",2);

  parm = AVSadd_parameter("initialize transformations", "oneshot",NULL,NULL,NULL);
    AVSadd_parameter_prop(parm,"layout","string",
      "manipulator \"$Module:initialize transformations\" -w oneshot -xy 10,166");
    AVSadd_parameter_prop(parm,"width","integer",4);

  parm = AVSadd_parameter("update frame", "boolean",0,NULL,NULL);
    AVSadd_parameter_prop(parm,"layout","string",
      "manipulator \"$Module:update frame\" -w toggle -xy 10,196");
    AVSadd_parameter_prop(parm,"width","integer",4);

  parm = AVSadd_parameter("animation frame","integer",1,1,150);
    AVSadd_parameter_prop(parm,"layout","string",
      "manipulator \"$Module:animation frame\" -w islider -xy 10,226");
    AVSadd_parameter_prop(parm,"immediate","boolean",1);
  parm = AVSadd_parameter("   <--", "oneshot",NULL,NULL,NULL);
    AVSadd_parameter_prop(parm,"layout","string",
      "manipulator \"$Module:   <--\" -w oneshot -xy 10,266");
    AVSadd_parameter_prop(parm,"width","integer",1);
  parm = AVSadd_parameter("   -->", "oneshot",NULL,NULL,NULL);
    AVSadd_parameter_prop(parm,"layout","string",
      "manipulator \"$Module:   -->\" -w oneshot -xy 187,266");
    AVSadd_parameter_prop(parm,"width","integer",1);
  parm = AVSadd_parameter("prev key", "oneshot",NULL,NULL,NULL);
    AVSadd_parameter_prop(parm,"layout","string",
      "manipulator \"$Module:prev key\" -w oneshot -xy 69,266");
    AVSadd_parameter_prop(parm,"width","integer",1);
  parm = AVSadd_parameter("next key", "oneshot",NULL,NULL,NULL);
    AVSadd_parameter_prop(parm,"layout","string",
      "manipulator \"$Module:next key\" -w oneshot -xy 128,266");
    AVSadd_parameter_prop(parm,"width","integer",1);

  parm = AVSadd_parameter("animate", "boolean",0,NULL,NULL);
    AVSadd_parameter_prop(parm,"layout","string",
      "manipulator \"$Module:animate\" -w toggle -xy 10,286");
    AVSadd_parameter_prop(parm,"width","integer",4); 
  parm = AVSadd_parameter("anim mode","choice","loop",
			  "loop|swing|once|off","|");
    AVSadd_parameter_prop(parm,"columns","integer",4);
    AVSadd_parameter_prop(parm,"width","integer",1);
    AVSadd_parameter_prop(parm,"layout","string",
      "manipulator \"$Module:anim mode\" -w radio_buttons -xy 10,316");

  parm = AVSadd_parameter("animation speed","integer",1,-50,50);
    AVSadd_parameter_prop(parm,"layout","string",
      "manipulator \"$Module:animation speed\" -w islider -xy 10,336");
    AVSadd_parameter_prop(parm,"immediate","boolean",1);
  parm = AVSadd_parameter("anim start", "integer",1,1,10000);
    AVSadd_parameter_prop(parm,"layout","string",
      "manipulator \"$Module:anim start\" -w typein_integer -xy 10,376");
    AVSadd_parameter_prop(parm,"width","integer",1);
  parm = AVSadd_parameter("anim end", "integer",NUMFRAMES,1,10000);
    AVSadd_parameter_prop(parm,"layout","string",
      "manipulator \"$Module:anim end\" -w typein_integer -xy 128,376");
    AVSadd_parameter_prop(parm,"width","integer",1);

  parm = AVSadd_parameter("save keyframe","oneshot",NULL,NULL,NULL);
    AVSadd_parameter_prop(parm,"layout","string",
      "manipulator \"$Module:save keyframe\" -w oneshot -xy 10,406");
    AVSadd_parameter_prop(parm,"width","integer",4);

  parm = AVSadd_parameter("ease in", "boolean",0,NULL,NULL);
    AVSadd_parameter_prop(parm,"layout","string",
      "manipulator \"$Module:ease in\" -w toggle -xy 10,436");
    AVSadd_parameter_prop(parm,"width","integer",1);
  parm = AVSadd_parameter("ease out", "boolean",0,NULL,NULL);
    AVSadd_parameter_prop(parm,"layout","string",
      "manipulator \"$Module:ease out\" -w toggle -xy 187,436");
    AVSadd_parameter_prop(parm,"width","integer",1);
  parm = AVSadd_parameter("create linear interpolation",
			  "oneshot",NULL,NULL,NULL);
    AVSadd_parameter_prop(parm,"layout","string",
      "manipulator \"$Module:create linear interpolation\" -xy 10,456");
    AVSadd_parameter_prop(parm,"width","integer",4);
  parm = AVSadd_parameter("create spline interpolation",
			  "oneshot",NULL,NULL,NULL);
    AVSadd_parameter_prop(parm,"layout","string",
      "manipulator \"$Module:create spline interpolation\" -xy 10,476");
    AVSadd_parameter_prop(parm,"width","integer",4);
  parm = AVSadd_parameter("create hold", "oneshot",NULL,NULL,NULL);
    AVSadd_parameter_prop(parm,"layout","string",
      "manipulator \"$Module:create hold\" -w oneshot -xy 10,496");
    AVSadd_parameter_prop(parm,"width","integer",4);

  parm = AVSadd_parameter("trace object path", "boolean",0,NULL,NULL);
    AVSadd_parameter_prop(parm,"layout","string",
      "manipulator \"$Module:trace object path\" -w toggle -xy 10,576");
    AVSadd_parameter_prop(parm,"width","integer",4);

  parm = AVSadd_parameter("clear keyframe", "oneshot",NULL,NULL,NULL);
    AVSadd_parameter_prop(parm,"layout","string",
      "manipulator \"$Module:clear keyframe\" -w oneshot -xy 10,526");
    AVSadd_parameter_prop(parm,"width","integer",2);
  parm = AVSadd_parameter("clear all keyfms", "oneshot",NULL,NULL,NULL);
    AVSadd_parameter_prop(parm,"layout","string",
      "manipulator \"$Module:clear all keyfms\" -w oneshot -xy 128,526");
    AVSadd_parameter_prop(parm,"width","integer",2);
  parm = AVSadd_parameter("clear keys and frame information","oneshot",NULL,NULL,NULL);
    AVSadd_parameter_prop(parm,"layout","string",
      "manipulator \"$Module:clear keys and frame information\" -w oneshot -xy 10,546");
    AVSadd_parameter_prop(parm,"width","integer",4);

  parm = AVSadd_parameter("wait for sync", "boolean",0,NULL,NULL);
    AVSadd_parameter_prop(parm,"layout","string",
      "manipulator \"$Module:wait for sync\" -w toggle -xy 10,606");
    AVSadd_parameter_prop(parm,"width","integer",2);
  parm = AVSadd_parameter("sync frme", "integer",0,0,10000);
    AVSadd_parameter_prop(parm,"layout","string",
      "manipulator \"$Module:sync frme\" -w typein_integer -xy 128,606");
    AVSadd_parameter_prop(parm,"width","integer",1);

  parm = AVSadd_parameter("load anim", "boolean",0,NULL,NULL);
    AVSadd_parameter_prop(parm,"layout","string",
      "manipulator \"$Module:load anim\" -w toggle -xy 10,636");
    AVSadd_parameter_prop(parm,"width","integer",2);
  parm = AVSadd_parameter("save anim", "boolean",0,NULL,NULL);
    AVSadd_parameter_prop(parm,"layout","string",
      "manipulator \"$Module:save anim\" -w toggle -xy 128,636");
    AVSadd_parameter_prop(parm,"width","integer",2);
  parm = AVSadd_parameter("edit settings", "boolean",0,NULL,NULL);
    AVSadd_parameter_prop(parm,"layout","string",
      "manipulator \"$Module:edit settings\" -w toggle -xy 10,656");
    AVSadd_parameter_prop(parm,"width","integer",4);

  /* load animation popup menu */
  parm = AVSadd_parameter("load animation","string",NULL,NULL,ANIMEXT);
    AVSadd_parameter_prop(parm,"layout","string",
      "panel \"loadbrow.$Module\" -w panel -p ui -hide -xy 300,600 -wh 250,243\n \
	manipulator \"$Module:load animation\" -w browser -p \"loadbrow.$Module\" -xy 8,11");
  parm = AVSadd_parameter("cancel load", "oneshot",NULL,NULL,NULL);
    AVSadd_parameter_prop(parm,"layout","string",
	"manipulator \"$Module:cancel load\" -w oneshot -p \"loadbrow.$Module\" -xy 8,211");
    AVSadd_parameter_prop(parm,"width","integer",4);

  /* save animation popup menu */
  parm = AVSadd_parameter("save animation","string",NULL,NULL,ANIMEXT);
    AVSadd_parameter_prop(parm,"layout","string",
      "panel \"savebrow.$Module\" -w panel -p ui -hide -xy 320,620 -wh 250,243\n \
	manipulator \"$Module:save animation\" -w browser -p \"savebrow.$Module\" -xy 8,11");
  parm = AVSadd_parameter("cancel save", "oneshot",NULL,NULL,NULL);
    AVSadd_parameter_prop(parm,"layout","string",
	"manipulator \"$Module:cancel save\" -w oneshot -p \"savebrow.$Module\" -xy 8,211");
    AVSadd_parameter_prop(parm,"width","integer",4);

  /* settings popup menu */
  /* Add parameters to the module and place them where they belong */
  parm = AVSadd_parameter("settings title","string","Keyframe Animation Settings",NULL,NULL);
    AVSadd_parameter_prop(parm,"layout","string",
      "panel \"settingsbrow.$Module\" -w panel -p ui -hide -xy 300,300 -wh 250,243\n \
	manipulator \"$Module:settings title\" -w text -p \"settingsbrow.$Module\" -xy 8,11");
    AVSadd_parameter_prop(parm,"width","integer",4);
  parm = AVSadd_parameter("key type","choice","numeric key markers",
	"numeric key markers|spherical keys markers|keyframe markers off","|");
    AVSadd_parameter_prop(parm,"layout","string",
	"manipulator \"$Module:key type\" -w radio_buttons -p \"settingsbrow.$Module\" -xy 8,41");
    AVSadd_parameter_prop(parm,"width","integer",4);
  parm = AVSadd_float_parameter("key marker size","float",KEYSIZE, .01, 1.0);
    AVSadd_parameter_prop(parm,"layout","string",
	"manipulator \"$Module:key marker size\" -w slider -p \"settingsbrow.$Module\" -xy 8,111");
    AVSadd_parameter_prop(parm,"immediate","boolean",1);
  parm = AVSadd_parameter("close settings", "oneshot",NULL,NULL,NULL);
    AVSadd_parameter_prop(parm,"layout","string",
	"manipulator \"$Module:close settings\" -w oneshot -p \"settingsbrow.$Module\" -xy 8,211");
    AVSadd_parameter_prop(parm,"width","integer",4);

  /* Tell AVS what subroutine to call to initialize the module */
  AVSset_init_proc(keyframe_init);

  /* Tell AVS what subroutine to call before destroying the module */
  AVSset_destroy_proc(keyframe_destroy);
}

  
/********************
 * GLOBAL VARIABLES *
 ********************/

double trans[3] = {0.0, 0.0, 0.0};   /* the current obj's translation */
double scale[3] = {1.0, 1.0, 1.0};         /* the current obj's scale */
double qrot[4] = {0.0, 0.0, 0.0, 1.0};  /* the current obj's rotation */

/* global settings */
int keytype = NUMERICKEYS;
float *keysize;

int write_animfile(char *, int, int);
int obj_exists(char *);
int read_animfile(char *, int);

/******************************
 * THE MAIN COMPUTE PROCEDURE *
 ******************************/

main(argc, argv)
    int argc;
    char *argv[];
{

  /*******************
   * LOCAL VARIABLES *
   *******************/

  /* some temporary variables */
  int i,j,s,e;                                    /* temporary integers */
  FILE *fd;
  char *mess;                           /* return string from an AVSmessage */
  char tempstring[256];
  char *tempstr;
  char cmdin[512];
  double ttrans[3];
  double tscale[3];      
  double tqrot[4];
  int animmode = LOOP;

  int Obj = 1;                                 /* the current object number */
  int notransf = 1;                      /* used for object transformations */
  
  /* inputs and parameters */
  int done, cframe;
  upstream_geom *pickinfo;
  upstream_transform *transinfo;
  char *title, *stitle, *value, *attrib;
  int newobj, delobj, pick, nextobj;
  int qroton=0, transon=0, sizeon=0, allon, alloff, init;
  int update=0, frame=1, backward, forward, prevkey, nextkey;
  char *animatemode;
  int speed=1, start=1, end=NUMFRAMES, save;
  int easein=0, easeout=0, linterp, sinterp, hold, trace=0;
  int clear, clearall, clearfr;
  char *loadfile, *savefile;
  int loadsets, savesets, animate=0, waitsync, syncval, cancelload, cancelsave;
  char *tracekeytype;
  int settings, cancelsets;

  AVScorout_init(argc,argv,keyframe_desc);
  
  AVScorout_wait();

  while(1) {
  
      /******************
       * ANIMATION LOOP *
       ******************/
  
      while (animate) {
	  /* enter animation loop */
	  frame+=speed;
	  if (frame > end) {
	     switch (animmode) {
		 case LOOP:
		    frame = start;
		    break;
		 case SWING:
		    frame = end-1;
		    speed = -speed;
		    AVSmodify_parameter("animation speed",AVS_VALUE,speed,NULL,NULL);
		    break;
		 case ONCE:
		    frame = end;
		    break;
		 default:
		    frame = end;
		    animate = 0;
		    AVSmodify_parameter("animate",AVS_VALUE,animate,NULL,NULL);
		    break;
	     }
	  }
	  else if (frame < start) {
	     switch (animmode) {
		 case LOOP:
		    frame = end;
		    break;
		 case SWING:
		    frame = start+1;
		    speed = -speed;
		    AVSmodify_parameter("animation speed",AVS_VALUE,speed,NULL,NULL);
		    break;
		 case ONCE:
		    frame = start;
		    break;
		 default:
		    frame = start;
		    animate = 0;
		    AVSmodify_parameter("animate",AVS_VALUE,animate,NULL,NULL);
		    break;
	     }
	  }
	     
	  AVSmodify_parameter("animation frame",AVS_VALUE,frame,NULL,NULL);
    
	  transform_objects(frame, Obj, update);
    
	  /* see if the animation mode has been turned off */
	  AVScorout_input(&cframe, &pickinfo, &transinfo, &title, &done, 
			 &value, &attrib, &newobj, &delobj, &pick, &nextobj,
			 &qroton, &transon, &sizeon, &allon, &alloff,
			 &init, &update, &frame, &backward, &forward,
			 &prevkey, &nextkey, &animate, &animatemode,
			 &speed, &start, &end, &save, 
			 &easein, &easeout, &linterp, &sinterp,
			 &hold, &trace, &clear, &clearall, &clearfr,
			 &waitsync, &syncval, &loadsets, &savesets, &settings, 
			 &loadfile, &cancelload, &savefile, &cancelsave,
			 &stitle, &tracekeytype, &keysize, &cancelsets);

	 animmode = AVSchoice_number("anim mode", animatemode);
      }
      
      /**********************************************
       * CHECK FOR CHANGES IN INPUTS AND PARAMETERS *
       **********************************************/

      AVScorout_wait();
      AVScorout_input(&cframe, &pickinfo, &transinfo, &title, &done, 
		     &value, &attrib, &newobj, &delobj, &pick, &nextobj,
		     &qroton, &transon, &sizeon, &allon, &alloff,
		     &init, &update, &frame, &backward, &forward,
		     &prevkey, &nextkey, &animate, &animatemode,
		     &speed, &start, &end, &save, 
		     &easein, &easeout, &linterp, &sinterp,
		     &hold, &trace, &clear, &clearall, &clearfr,
		     &waitsync, &syncval, &loadsets, &savesets, &settings, 
		     &loadfile, &cancelload, &savefile, &cancelsave, 
		     &stitle, &tracekeytype, &keysize, &cancelsets);

      /******************************************
       * CHECK FOR CHANGES IN OBJECT TRANSFORMS *
       ******************************************/
    
      /* check for the user manipulating the object */
      if (AVSinput_changed("Trans Info",0)) {
	if (animate) {
	    notransf = 1;
	    continue;
	}
    
	if (notransf) {
	    notransf = 0;
	    continue;
	}
    
	/*** need to add code to make sure that the transformed object is ***/
	/*** the one we are currently animating                           ***/
    
	for (i=0; i<3; i++) {
	   ttrans[i] = trans[i];
	   tscale[i] = scale[i];
	   tqrot[i] = qrot[i];
	}
	tqrot[3] = qrot[3];
    
	if (mat_decode(transinfo->msxform, trans, scale, qrot) < 0) continue;
    
	if ((ttrans[0] != trans[0]) || (ttrans[1] != trans[1]) ||
	    (ttrans[2] != trans[2])) {
	  transon = 1;
	  AVSmodify_parameter("TRANS",AVS_VALUE,1,NULL,NULL);
	}
    
	if ((tscale[0] != scale[0]) || (tscale[1] != scale[1]) ||
	    (tscale[2] != scale[2])) {
	  sizeon = 1;
	  AVSmodify_parameter("SCALE",AVS_VALUE,1,NULL,NULL);
	}
    
	if ((tqrot[0] != qrot[0]) || (tqrot[1] != qrot[1]) ||
	    (tqrot[2] != qrot[2]) || (tqrot[3] != qrot[3])) {
	  qroton = 1;
	  AVSmodify_parameter("ROTATE",AVS_VALUE,1,NULL,NULL);
	}
    
	update = 0;
	AVSmodify_parameter("update frame",AVS_VALUE,0,NULL,NULL);
    
	continue;
      }

      /* this variable is used to make ensure that the user has moved an */
      /* object by making sure the object is transformed twice in a row */
      else notransf = 1;
    
      /****************************************************
       * CHECK FOR CHANGES IN OTHER PARAMETERS AND INPUTS *
       ****************************************************/
    
      /* go to a frame of the animation */
      if (AVSparameter_changed("animation frame")) {
	transform_objects(frame, Obj, update);
	continue;
      }

      /* step backward or forward to the next keyframe */
      else if (AVSparameter_changed("prev key")) {
	if (frame > start) frame = PrevKey(start,frame);
	else frame = end;
	AVSmodify_parameter("animation frame",AVS_VALUE,frame,NULL,NULL);

        transform_objects(frame, Obj, update);
	continue;
      }
      else if (AVSparameter_changed("next key")) {
	if (frame < end) frame = NextKey(frame,end);
	else frame = start;
	AVSmodify_parameter("animation frame",AVS_VALUE,frame,NULL,NULL);

        transform_objects(frame, Obj, update);
	continue;
      }
      /* step forward of backward some frames */
      else if (AVSparameter_changed("   -->")) {
	 frame+=speed;
	 if (frame > end) {
	     switch (animmode) {
		 case LOOP:
		    frame = start;
		    break;
		 case SWING:
		    frame = end-1;
		    speed = -speed;
		    AVSmodify_parameter("animation speed",AVS_VALUE,speed,NULL,NULL);
		    break;
		 default:
		    frame = end;
		    break;
	     }
	 }
	 else if (frame < start) {
	     switch (animmode) {
		 case LOOP:
		    frame = end;
		    break;
		 case SWING:
		    frame = start+1;
		    speed = -speed;
		    AVSmodify_parameter("animation speed",AVS_VALUE,speed,NULL,NULL);
		    break;
		 default:
		    frame = start;
		    break;
	     }
	 }
	 AVSmodify_parameter("animation frame",AVS_VALUE,frame,NULL,NULL);

         transform_objects(frame, Obj, update);
	 continue;
      }
      else if (AVSparameter_changed("   <--")) {
	 frame-=speed;
	 if (frame > end) {
	     switch (animmode) {
		 case LOOP:
		    frame = start;
		    break;
		 case SWING:
		    frame = end-1;
		    speed = -speed;
		    AVSmodify_parameter("animation speed",AVS_VALUE,speed,NULL,NULL);
		    break;
		 default:
		    frame = end;
		    break;
	     }
	 }
	 else if (frame < start) {
	     switch (animmode) {
		 case LOOP:
		    frame = end;
		    break;
		 case SWING:
		    frame = start+1;
		    speed = -speed;
		    AVSmodify_parameter("animation speed",AVS_VALUE,speed,NULL,NULL);
		    break;
		 default:
		    frame = start;
		    break;
	     }
	 }
	 AVSmodify_parameter("animation frame",AVS_VALUE,frame,NULL,NULL);

         transform_objects(frame, Obj, update);
	 continue;
      }
      
      /* incriment current frame after sync frame is reached */
      else if (AVSinput_changed("Frame",0)) {
	if (!update) {
	    update = 1;
	    AVSmodify_parameter("update frame",AVS_VALUE,update,NULL,NULL);
	}

	/* if we are not currently in sync check to see if we should be */
	if (waitsync && syncval) {
	  if (cframe == syncval) {
	    waitsync = 0;
	    AVSmodify_parameter("wait for sync",AVS_VALUE,0,NULL,NULL);
	  }
	}

        if (waitsync) continue;

	/* if we are in sync, incriment the current frame */
	frame+=speed;
	if (frame > end) {
	     switch (animmode) {
		 case LOOP:
		    frame = start;
		    break;
		 case SWING:
		    frame = end-1;
		    speed = -speed;
		    AVSmodify_parameter("animation speed",AVS_VALUE,speed,NULL,NULL);
		    break;
		 default:
		    frame = end;
		    break;
	     }
	}
	else if (frame < start) {
	     switch (animmode) {
		 case LOOP:
		    frame = end;
		    break;
		 case SWING:
		    frame = start+1;
		    speed = -speed;
		    AVSmodify_parameter("animation speed",AVS_VALUE,speed,NULL,NULL);
		    break;
		 default:
		    frame = start;
		    break;
	     }
	}
	     
	AVSmodify_parameter("animation frame",AVS_VALUE,frame,NULL,NULL);

        transform_objects(frame, Obj, update);
	continue;
      }

      /* handle picking of an object with the left mouse button */
      else if (AVSinput_changed("Pick Info",0)) { 
	if (pick) {
	  pick = 0;
	  AVSmodify_parameter("add obj",AVS_VALUE,pick,NULL,NULL);
	  set_notify(0);
	}
    
	/* find out the name of the object picked */
	if (strlen(pickinfo->picked_obj) < 2)
	  sprintf(tempstring,"%%%s","top");
	else
	  sprintf(tempstring,"%%%s",pickinfo->picked_obj);
    
	/* make the picked object the current object */
	sprintf(cmdin,"geom_set_cur_obj -object %s",tempstring);
	send_command(cmdin);
    
	/* see if that object already exists in the database */
	for (i=1; i<=GetNumobjs(); i++) {
	  if (!strcmp(GetName(i),tempstring)) {
    	    /* the object exists -- make it the current object */
	    Obj = i;
    
	    AVSmodify_parameter("object attributes",AVS_VALUE,"name",NULL,NULL);
	    AVSmodify_parameter(">>>",AVS_VALUE,GetName(Obj),NULL,NULL);
    
	    /* turn off all active dials */
	    qroton = 0;
	    AVSmodify_parameter("ROTATE",AVS_VALUE,0,NULL,NULL);
	    transon = 0;
	    AVSmodify_parameter("TRANS",AVS_VALUE,0,NULL,NULL);
	    sizeon = 0;
	    AVSmodify_parameter("SCALE",AVS_VALUE,0,NULL,NULL);
    
	    update = 1;
	    AVSmodify_parameter("update frame",AVS_VALUE,update,NULL,NULL);
    
	    break;
	  }
	}
    
	if (i > GetNumobjs()) {
	  /* if it doesn't exist, ask to add it to the database, */
	  /* or replace the current object with it */
	  mess = AVSmessage(VERSION,AVS_Warning,NULL,"PICK OBJ",
	    "ADD OBJ!REPLACE OBJ!ABORT",
	    "The object %%%s is not currently being animated.\n%s%s%s",
	    tempstring,
	    "Press ADD OBJ to add the object.\n",
	    "Press REPLACE OBJ to replace the current object with that object.\n",
	    "Press ABORT to do nothing.\n");
	  if (!strcmp(mess,"ABORT")) {
	     continue;
	  }
	  else if (!strcmp(mess,"REPLACE OBJ")) {
	    /* replace the current object with the new one */
    
	    /* reset the old object */
	    reset_object(GetName(Obj));
    
	    /* set up the new object */
	    SetName(Obj, tempstring);
	    set_object(tempstring, GetParent(Obj));

	    AVSmodify_parameter("object attributes",AVS_VALUE,"name",NULL,NULL);
	    AVSmodify_parameter(">>>",AVS_VALUE,tempstring,NULL,NULL);
	  }
	  else {
	    /* add the new object to the database */
	    if (Anim_addobjects(1) < 0) {
	      AVSfatal("\n\nUnable to allocate memory for object information!\n");
	      continue;
	    }
	    Obj = GetNumobjs();
	    
	    SetName(Obj, tempstring);
	    set_object(tempstring, "%top");

	    AVSmodify_parameter("object attributes",AVS_VALUE,"name",NULL,NULL);
	    AVSmodify_parameter(">>>",AVS_VALUE,tempstring,NULL,NULL);
	  }
	}

        transform_objects(frame, Obj, update);
    	if (trace) trace_all(Obj, start, end);

	continue;
      }
      
      else if (AVSparameter_changed("update frame")) {
	/* don't allow user to turn off update in animate mode */
	if (!update && animate) {
	  update = 1;
	  AVSmodify_parameter("update frame",AVS_VALUE,update,NULL,NULL);
	  continue;
	}

        transform_objects(frame, Obj, update);
	continue;
      }

      /* to add an object, the top object must be in notify mode */    
      else if (AVSparameter_changed("add obj")) {
	/* set the top object to allow picking */
	set_notify(pick);

	continue;
      }
    
      /* turn on animation mode */
      else if (AVSparameter_changed("animate")) {
	if (animate) {
	    /* make sure update comes on */
	    update = 1;
	    AVSmodify_parameter("update frame",AVS_VALUE,update,NULL,NULL);
	}

	continue;
      }
    
      /* make sure end is not less than start and that the animation window */
      /* reflects the start and end */
      else if (AVSparameter_changed("anim start")) {
	if (start > end) {
	  end = start;
	  AVSmodify_parameter("anim end",AVS_VALUE,start,NULL,NULL);
	}
	AVSmodify_parameter("anim end",AVS_MINVAL,NULL,start,NULL);
	AVSmodify_parameter("animation frame",AVS_MINVAL|AVS_MAXVAL,
			    NULL,start,end);
	if (start > frame) {
	  frame = start;
	  AVSmodify_parameter("animation frame",AVS_VALUE,frame,NULL,NULL);
	}
    
	/* allocate more memory for frame arrays if needed */
	if (end > GetNumframes()) {
	    if (Anim_addframes(end-GetNumframes()) < 0) {
		AVSfatal("\n\nUnable to allocate memory for frame information!\n");
		start = 1;
		AVSmodify_parameter("anim start",AVS_VALUE,start,NULL,NULL);
		end = GetNumframes();
		AVSmodify_parameter("anim end",AVS_VALUE,start,NULL,NULL);
		frame = end;
		AVSmodify_parameter("animation frame",AVS_VALUE,frame,NULL,NULL);
		AVSmodify_parameter("animation frame",AVS_MINVAL|AVS_MAXVAL,
			    NULL,start,end);
		continue;
	    }
	}

        transform_objects(frame, Obj, update);
	if (trace)  trace_all(Obj, start, end);

	continue;
      }
      else if (AVSparameter_changed("anim end")) {
	if (start > end) {
	  end = start;
	  AVSmodify_parameter("anim end",AVS_VALUE,start,NULL,NULL);
	}
	AVSmodify_parameter("animation frame",AVS_MAXVAL,NULL,NULL,end);
	if (end < frame) {
	  frame = end;
	  AVSmodify_parameter("animation frame",AVS_VALUE,frame,NULL,NULL);
	}
    
	/* allocate more memory for frame arrays if needed */
	if (end > GetNumframes()) {
	    if (Anim_addframes(end-GetNumframes()) < 0) {
		AVSfatal("\n\nUnable to allocate memory for frame information!\n");
		start = 1;
		AVSmodify_parameter("anim start",AVS_VALUE,start,NULL,NULL);
		end = GetNumframes();
		AVSmodify_parameter("anim end",AVS_VALUE,start,NULL,NULL);
		frame = end;
		AVSmodify_parameter("animation frame",AVS_VALUE,frame,NULL,NULL);
		AVSmodify_parameter("animation frame",AVS_MINVAL|AVS_MAXVAL,
			    NULL,start,end);
		continue;
	    }
	}

        transform_objects(frame, Obj, update);
	if (trace)  trace_all(Obj, start, end);

	continue;
      }
    
      /* change the current attribute */
      else if (AVSparameter_changed("object attributes")) {
	if (!strcmp(attrib,"name")) 
	  AVSmodify_parameter(">>>", AVS_VALUE, GetName(Obj), NULL,NULL);
	else if (!strcmp(attrib,"parent")) 
	  AVSmodify_parameter(">>>", AVS_VALUE, GetParent(Obj), NULL,NULL);
	else
	  AVSmodify_parameter(">>>",AVS_VALUE,"no-attribute", NULL,NULL);

	continue;
      }    
      /* change the value of the current attribute */
      else if (AVSparameter_changed(">>>")) {
	if (!strcmp(attrib,"name")) {
	
	  /* make sure the name has a % in front */
	  if ((strncmp(value,"EMPTY",5)) && (value[0] != '%') ) {
	    strcpy(tempstring, "%");
	    strcat(tempstring, value);
	    AVSmodify_parameter(">>>",AVS_VALUE,tempstring,NULL,NULL);
	  }
	  else strcpy(tempstring, value);
	  
	  if (!obj_exists(tempstring)) {
	      fprintf(stderr, "ANIM:  Object %s doesn't exist.\n", tempstring);
	      AVSmodify_parameter(">>>",AVS_VALUE,GetName(Obj),NULL,NULL);
	      continue;
	  }

	  /* reset the old object */
	  reset_object(GetName(Obj));

          SetName(Obj, tempstring);
          set_object(tempstring, "%top");

          /* make the picked object the current object */
	  if (strncmp(tempstring,"EMPTY",5)) {
	    sprintf(cmdin,"geom_set_cur_obj -object %s",value);
	    send_command(cmdin);
	  }

          transform_objects(frame, Obj, update);
	  if (trace) trace_all(Obj, start, end);

	  continue;
	}
	else if (!strcmp(attrib,"parent")) {
	  /* make sure the name has a % in front */
	  if ((strncmp(value,"EMPTY",5)) && (value[0] != '%') ) {
	    strcpy(tempstring, "%");
	    strcat(tempstring, value);
	    AVSmodify_parameter(">>>",AVS_VALUE,tempstring,NULL,NULL);
	  }
	  else strcpy(tempstring, value);

	  if (!obj_exists(tempstring)) {
	      fprintf(stderr, "ANIM:  Parent object %s doesn't exist.\n", tempstring);
	      AVSmodify_parameter(">>>",AVS_VALUE,GetParent(Obj),NULL,NULL);
	      continue;
	  }

	  SetParent(Obj, tempstring);
	  set_parent(GetName(Obj), tempstring);
	  if (trace) trace_all(Obj, start, end);

	  continue;
	}
      }
    
      /* add a new object and allocate more memory for it */
      else if (AVSparameter_changed("new obj")) {
	if (Anim_addobjects(1) < 0) {
	  AVSfatal("\n\nUnable to allocate memory for object information!\n");
	  continue;
	}
	Obj = GetNumobjs();
	AVSmodify_parameter("object attributes",AVS_VALUE,"name",NULL,NULL);
	AVSmodify_parameter(">>>",AVS_VALUE,GetName(Obj),NULL,NULL);

	/* redraw the trace object */
	if (trace) trace_all(Obj, start, end);

	continue;
      }
      /* change the current object to an empty object */
      else if (AVSparameter_changed("del obj")) {
	/* reset the old object */
        reset_object(GetName(Obj));
    
	/* set the names for the objects */
	sprintf(tempstring,"%s-%d","EMPTY",Obj);
	SetName(Obj, tempstring);
    
	/* set the objects parents */
	SetParent(Obj, "%top");
    
	AVSmodify_parameter("object attributes",AVS_VALUE,"name",NULL,NULL);
	AVSmodify_parameter(">>>",AVS_VALUE,GetName(Obj),NULL,NULL);

	continue;
      }
      /* go to the next object */
      else if (AVSparameter_changed("next obj")) {
	Obj++;
	if (Obj > GetNumobjs()) Obj = 1;
    
	AVSmodify_parameter("object attributes",AVS_VALUE,"name",NULL,NULL);
	AVSmodify_parameter(">>>",AVS_VALUE,GetName(Obj),NULL,NULL);
    
	/* make the picked object the current object */
	sprintf(cmdin,"geom_set_cur_obj -object %s\n",value);
	send_command(cmdin);
    
	/* turn off all active dials */
	qroton = 0;
	AVSmodify_parameter("ROTATE",AVS_VALUE,0,NULL,NULL);
	transon = 0;
	AVSmodify_parameter("TRANS",AVS_VALUE,0,NULL,NULL);
	sizeon = 0;
	AVSmodify_parameter("SCALE",AVS_VALUE,0,NULL,NULL);
    
	update = 1;
	AVSmodify_parameter("update frame",AVS_VALUE,update,NULL,NULL);

        transform_objects(frame, Obj, update);
	if (trace) trace_all(Obj, start, end);

	continue;
      }
      
      /* set the animation mode to loop, once, or swing */
      else if (AVSparameter_changed("anim mode")) {
	 animmode = AVSchoice_number("anim mode", animatemode);
	 continue;
      }
    
      /* handle settings popup window */
      else if (AVSparameter_changed("edit settings")) {
        if (settings) send_command("panel \"settingsbrow.$Module\" -w panel -p ui -show -wh 250,243\n \
	manipulator \"$Module:settings title\" -w text -p \"settingsbrow.$Module\" -xy 8,11\n \
	manipulator \"$Module:key type\" -w radio_buttons -p \"settingsbrow.$Module\" -xy 8,41\n \
	manipulator \"$Module:key marker size\" -w slider -p \"settingsbrow.$Module\" -xy 8,111\n \
	manipulator \"$Module:close settings\" -w oneshot -p \"settingsbrow.$Module\" -xy 8,211");
        else send_command("panel \"settingsbrow.$Module\" -hide");

	continue;
      }
      else if (AVSparameter_changed("close settings")) {
        send_command("panel \"settingsbrow.$Module\" -hide");
        if (settings) {
	    settings = 0;
	    AVSmodify_parameter("edit settings", AVS_VALUE,settings,NULL,NULL);
	}
	continue;
      }

      /* set the animation mode to loop, once, or swing */
      else if (AVSparameter_changed("key type")) {
	 keytype = AVSchoice_number("key type", tracekeytype);

	 if (trace) trace_keys(Obj, start, end);
	 continue;
      }
      /* set the size of the keyframe markers (spheres only) */
      if (AVSparameter_changed("key marker size")) {
	 if (trace) trace_keys(Obj, start, end);
 	 continue;
      }
      
      /* handle popup windows for loading and saving animations */
      else if (AVSparameter_changed("load anim")) {
        if (loadsets) send_command("panel \"loadbrow.$Module\" -w panel -p ui -show -wh 250,243\n \
	manipulator \"$Module:load animation\" -w browser -p \"loadbrow.$Module\" -xy 8,11\n \
	manipulator \"$Module:cancel load\" -w oneshot -p \"loadbrow.$Module\" -xy 8,211");
        else send_command("panel \"loadbrow.$Module\" -hide");

	continue;
      }
      else if (AVSparameter_changed("save anim")) {
        if (savesets) send_command("panel \"savebrow.$Module\" -w panel -p ui -show -wh 250,243\n \
	manipulator \"$Module:save animation\" -w browser -p \"savebrow.$Module\" -xy 8,11\n \
	manipulator \"$Module:cancel save\" -w oneshot -p \"savebrow.$Module\" -xy 8,211");
	else send_command("panel \"savebrow.$Module\" -hide");

	continue;
      }

      else if (AVSparameter_changed("cancel load")) {
        send_command("panel \"loadbrow.$Module\" -hide");
        if (loadsets) {
	    loadsets = 0;
	    AVSmodify_parameter("load anim", AVS_VALUE,loadsets,NULL,NULL);
	}
	continue;
      }
      else if (AVSparameter_changed("cancel save")) {
        send_command("panel \"savebrow.$Module\" -hide");
        if (savesets) {
	    savesets = 0;
	    AVSmodify_parameter("save anim", AVS_VALUE,savesets,NULL,NULL);
	}
	continue;
      }

      /* save current objects, keyframes, and frame information */
      else if (AVSparameter_changed("save animation")) {
	if (savefile == NULL) continue;
	
	/* make sure save filename has a .anim extension */
	tempstr = strrchr(savefile, '.');
	if ((tempstr == NULL) || strcmp(tempstr, ANIMEXT)) {
	    strcat(savefile, ANIMEXT);
	    AVSmodify_parameter("save animation", AVS_VALUE, savefile, NULL, NULL);
	}

	/* see if the file already exists */
	if ((fd = fopen(savefile,"r")) != NULL) {
	  fclose(fd);
	  mess = AVSmessage(VERSION,AVS_Warning,NULL,"SAVE ANIM","REPLACE!ABORT",
			     "%s %s %s\n\n%s\n\n%s\n",
			     "The file",
			     savefile,
			     "already exists!",
			     "Press REPLACE to replace it.",
			     "Press ABORT to abort save.");
	  if (!strcmp(mess,"ABORT")) continue;
	}

	/* see if the animation window is smaller than the animation */
	s = start;
	e = end;
	if ((end < GetNumframes()) || (start > 1)) {
	  mess = AVSmessage(VERSION,AVS_Warning,NULL,"SAVE ANIM",
	    "SAVE!SAVEALL!ABORT",
	    "Press SAVE to save ONLY frames %d to %d!\n\n%s (%d frames).\n\n%s\n",
			    start,end,
			    "Press SAVEALL to save ALL frames of the animation",
			    GetNumframes(),
			    "Press ABORT to abort save.");
	  if (!strcmp(mess,"ABORT")) continue;
	  else if (!strcmp(mess,"SAVEALL")) {
		s = 1;
		e = GetNumframes();
	  }
	}

       	write_animfile(savefile, s, e);
	
    	/* hide the browser window */
	savesets = 0;
	AVSmodify_parameter("save anim", AVS_VALUE,savesets,NULL,NULL);
        send_command("panel \"savebrow.$Module\" -hide");

	/* change the current animation filename title */
	tempstr = strrchr(savefile, '/');
	if (tempstr == NULL) tempstr = savefile;
	else tempstr++;
        sprintf(title, "Animation:  %s", tempstr);
        AVSmodify_parameter("title", AVS_VALUE, title, NULL, NULL);

	continue;
      }

      /* load current objects, keyframes, and frame information */
      else if (AVSparameter_changed("load animation")) {
	if (loadfile == NULL) continue;

	j = LOADREPLACE;
	mess = AVSmessage(VERSION,AVS_Warning,NULL,"LOAD ANIM",
			  "LOAD!MERGE!APPEND!ABORT",
			  "%s\n%s %s.\n\n%s\n\n%s\n",
			  "Loading animation will destroy frame information",
			  "currently in memory!  Press LOAD to load",
			  loadfile,
			  "Press MERGE to merge, APPEND to append animations.",
			  "Press ABORT to abort load.");
	if (!strcmp(mess,"ABORT")) continue;
	else if (!strcmp(mess,"MERGE")) j = LOADMERGE;
	else if (!strcmp(mess,"APPEND")) j = LOADAPPEND;
	
    	/* reset all objects */
	for (i=1; i<=GetNumobjs(); i++) {
	    reset_object(GetName(i));
	}
    
	if ((e = read_animfile(loadfile, j)) < 1) {
	    AVSfatal("\n\nUnable to allocate memory for animation!\n");
	    exit(-1);
	}
    
	/* set up all objects for manipulation with the mouse */
	/* and set the correct parents */
	for (i=1; i<=GetNumobjs(); i++) {
	    set_object(GetName(i), GetParent(i));
	}

	/* initialize certain parameters */
	Obj = 1;
	start = 1;
	AVSmodify_parameter("anim start",AVS_VALUE,start,NULL,NULL);
	end = e;
	AVSmodify_parameter("anim end",AVS_VALUE,end,NULL,NULL);
	frame = 1;
	AVSmodify_parameter("animation frame",AVS_VALUE|AVS_MINVAL|AVS_MAXVAL,
			    frame,start,end);
    
	AVSmodify_parameter("object attributes",AVS_VALUE,"name",NULL,NULL);
	AVSmodify_parameter(">>>",AVS_VALUE,GetName(Obj),NULL,NULL);

       	update = 1;
	AVSmodify_parameter("update frame",AVS_VALUE,update,NULL,NULL);
    
	/* turn on all active dials */
	qroton = 1;
	AVSmodify_parameter("ROTATE",AVS_VALUE,1,NULL,NULL);
	transon = 1;
	AVSmodify_parameter("TRANS",AVS_VALUE,1,NULL,NULL);
	sizeon = 1;
	AVSmodify_parameter("SCALE",AVS_VALUE,1,NULL,NULL);

	/* hide the browser window */
	loadsets = 0;
	AVSmodify_parameter("load anim", AVS_VALUE,loadsets,NULL,NULL);
        send_command("panel \"loadbrow.$Module\" -hide");

	/* change the current animation filename */
	tempstr = strrchr(loadfile, '/');
	if (tempstr == NULL) tempstr = loadfile;
	else tempstr++;
        sprintf(title, "Animation:  %s", tempstr);
        AVSmodify_parameter("title", AVS_VALUE, title, NULL, NULL);

        transform_objects(frame, Obj, update);
	if (trace) trace_all(Obj, start, end);

	make_beep();
	continue;
      }
    
      /* turn all active buttons on or off */
      else if (AVSparameter_changed("ACTIVATE ALL")) {
	qroton = 1;
	AVSmodify_parameter("ROTATE",AVS_VALUE,1,NULL,NULL);
	transon = 1;
	AVSmodify_parameter("TRANS",AVS_VALUE,1,NULL,NULL);
	sizeon = 1;
	AVSmodify_parameter("SCALE",AVS_VALUE,1,NULL,NULL);

	continue;
      }
      else if (AVSparameter_changed("DEACTIVATE ALL")) {
	qroton = 0;
	AVSmodify_parameter("ROTATE",AVS_VALUE,0,NULL,NULL);
	transon = 0;
	AVSmodify_parameter("TRANS",AVS_VALUE,0,NULL,NULL);
	sizeon = 0;
	AVSmodify_parameter("SCALE",AVS_VALUE,0,NULL,NULL);

	continue;
      }
      
      /* clear the values of active dials */
      else if (AVSparameter_changed("clear keys and frame information")) {
	mess = AVSmessage(VERSION,AVS_Warning,NULL,"CLEAR FRAME INFO",
			  "ABORT!CLEAR",
	 "%s %d to %d!\n\n%s\n%s\n",
	 "This command will clear all keyframes and stored values of active dials for frames",
	 start,end,
	 "Press CLEAR to clear all keyframes and frame information.",
	 "Press ABORT to abort clear.");
	if (!strcmp(mess,"ABORT")) continue;
    
	Quat_init(qrot);
	Trans_init(trans);
	Scale_init(scale);

	for(i=start; i<=end; i++) {
	  if (qroton) SetRot(i, Obj, qrot);
	  if (transon) SetTrans(i, Obj, trans);
	  if (sizeon) SetSize(i, Obj, scale[0]);
	  SetKey(i, 0);
	}

        transform_objects(frame, Obj, update);
	if (trace) trace_all(Obj, start, end);

	make_beep();
	continue;
      }     
      
      /* clear current keyframe */
      else if (AVSparameter_changed("clear keyframe")) {
	SetKey(frame, 0);

	if (trace) trace_keys(Obj, start, end);

	make_beep();
	continue;
      }
      /* clear all keyframes */
      else if (AVSparameter_changed("clear all keyfms")) {
	for(i=start; i<=end; i++) SetKey(i, 0);

	if (trace) trace_keys(Obj, start, end);

	make_beep();
	continue;
      }
      
      /* create a linear interpolation based on the current keyframes */
      else if (AVSparameter_changed("create linear interpolation")) { 
	make_beep();

	/* make sure that the start and end frames are keyframes */
	SetKey(start, 1);
	SetKey(end, 1);

	/* create linear interpolations */
	Interp_linear(start,end,easein,easeout,Obj,
				 qroton,transon,transon,transon,sizeon);
	
	update = 1;
	AVSmodify_parameter("update frame",AVS_VALUE,update,NULL,NULL);

        transform_objects(frame, Obj, update);
	if (trace) trace_all(Obj, start, end);

	make_beep();
	continue;
      }
      /* create a spline interpolation based on the current keyframes */
      else if (AVSparameter_changed("create spline interpolation")) {
	make_beep();

	/* make sure that the start and end frames are keyframes */
	SetKey(start, 1);
	SetKey(end, 1);
    
	/* create spline interpolations */
	Interp_spline(start,end,easein,easeout,Obj,
				 qroton,transon,transon,transon,sizeon);
	
	update = 1;
	AVSmodify_parameter("update frame",AVS_VALUE,update,NULL,NULL);

        transform_objects(frame, Obj, update);
	if (trace) trace_all(Obj, start, end);

	make_beep();
	continue;
      }
      /* create a hold based on the current frame */
      else if (AVSparameter_changed("create hold")) {
	make_beep();

	Interp_hold(frame, start, end, Obj,
	    qroton,transon,transon,transon,sizeon);

	update = 1;
	AVSmodify_parameter("update frame",AVS_VALUE,update,NULL,NULL);
	
        transform_objects(frame, Obj, update);
	if (trace) trace_all(Obj, start, end);

	make_beep();
	continue;
      }
      
      /* turn on/off path traces */
      else if (AVSparameter_changed("trace object path")) {
    	if (trace) trace_all(Obj, start, end);
	else trace_all(Obj, -1, -1);

       	continue;
      }

      /* reset active dials to zero */
      else if (AVSparameter_changed("initialize transformations")) {
	if (qroton) Rot_init(qrot);
	if (transon) Trans_init(trans);
	if (sizeon) Scale_init(scale);
    
	update = 0;
	AVSmodify_parameter("update frame",AVS_VALUE,update,NULL,NULL);

        transform_objects(frame, Obj, update);
	continue;
      }
    
      /* mark the current positions for the currently active dials as */
      /* a keyframe */
      else if (AVSparameter_changed("save keyframe")) {
	/* save keyframe position of active channels */
	if (qroton) SetRot(frame, Obj, qrot);
	if (transon) SetTrans(frame, Obj, trans);
	if (sizeon) SetSize(frame, Obj, scale[0]);
    
	SetKey(frame, 1);

	/* redraw the trace object */
	if (trace) trace_all(Obj, start, end);

	make_beep();
	continue;
      }
      
      else if (AVSparameter_changed("exit")) {
	mess = AVSmessage(VERSION,AVS_Warning,NULL,"EXIT KEYFRAME ANIMATOR",
			  "Cancel!OK",
	 "%s\n%s\n",
	 "The work you have done in the Keyframe Animator will NOT BE SAVED!", 
	 "Do you really want to exit now?");

	if (!strcmp(mess,"Cancel")) continue;
        
	cleanup(trace);
	continue;
      }

   }
}
    
/*********************************
 * THE INIT AND DESTROY ROUTINES *
 *********************************/

keyframe_init()
/* set up global structures before invoking the module */
{
  fprintf(stderr,"%s (%s)\n%s\n%s\n",
	  "KEYFRAME ANIMATOR",VERSION,
	  "   Copyright (c) 1991 Brian Kaplan, CICA, Indiana University",
	  "   All rights reserved.");

  Rot_init(qrot);
  Trans_init(trans);
  Scale_init(scale);
  Anim_init(NUMFRAMES,NUMOBJS,NUMCHANNELS);
}

keyframe_destroy()
/* free global structures before destroying the module */
{
  Anim_free();
  fprintf(stderr, "ANIM:  Module exited.  Goodbye.\n");
}

/************************
 * SOME OUTPUT ROUTINES *
 ************************/

transform_objects(frame, obj, update)
    int frame, obj, update;
{
    int i, j;
    double ttrans[3];
    double tscale[3];      
    double tqrot[4];
    float fvec[3];
    float xform[4][4];
    float tmat[4][4];
    GEOMedit_list output = GEOM_NULL;
    
    /* add transformations to the editlist for each object in the database */
    output = GEOMinit_edit_list(output);

    /* do transformations on the objects */
    for (i=1; i<obj; i++) { 
	if (strncmp(GetName(i),"EMPTY",5)) {
	    GetRot(tqrot, frame, i);
	    GetTrans(ttrans, frame, i);
	    GetSize(tscale, frame, i);

	    /* do scale */
	    for (j=0;j<3;j++) fvec[j] = (float)tscale[j];
	    mat_scale(xform,fvec[0],fvec[1],fvec[2]);
    
	    /* do rotate */
	    Quat_evalmatrix(tmat, tqrot);

	    mat_multiply(xform,tmat,xform);
	    GEOMedit_set_matrix(output, GetName(i), xform);

	    /* do translate */
	    for (j=0;j<3;j++) fvec[j] = (float)ttrans[j];
	    GEOMedit_position(output, GetName(i), fvec);
	}
    }
    /* the current object only gets updated if "update" is on */
    if (strncmp(GetName(obj),"EMPTY",5)) {
        if (update) {
	    GetRot(qrot, frame, obj);
	    GetTrans(trans, frame, obj);
	    GetSize(scale, frame, obj);
	}
  
	/* do scale */
	for (j=0;j<3;j++) fvec[j] = (float)scale[j];
	mat_scale(xform,fvec[0],fvec[1],fvec[2]);

	/* do rotate */
	Quat_evalmatrix(tmat, qrot);

	mat_multiply(xform,tmat,xform);
	GEOMedit_set_matrix(output, GetName(obj), xform);

	/* do translate */
	for (j=0;j<3;j++) fvec[j] = (float)trans[j];
	GEOMedit_position(output, GetName(obj), fvec);
    }
    for (i=(obj+1); i<=GetNumobjs(); i++) {
	if (strncmp(GetName(i),"EMPTY",5)) {
	    GetRot(tqrot, frame, i);
	    GetTrans(ttrans, frame, i);
	    GetSize(tscale, frame, i);

	    /* do scale */
	    for (j=0;j<3;j++) fvec[j] = (float)tscale[j];
	    mat_scale(xform,fvec[0],fvec[1],fvec[2]);
    
	    /* do rotate */
	    Quat_evalmatrix(tmat, tqrot);

	    mat_multiply(xform,tmat,xform);
	    GEOMedit_set_matrix(output, GetName(i), xform);

	    /* do translate */
	    for (j=0;j<3;j++) fvec[j] = (float)ttrans[j];
	    GEOMedit_position(output, GetName(i), fvec);
	}
    }

    AVScorout_exec();
    AVScorout_output(output, frame);
}    

reset_object(char *name)
{
    GEOMedit_list output = GEOM_NULL;

    if (!strncmp(name,"EMPTY",5)) return;

    output = GEOMinit_edit_list(output);

    /* reset an object to normal transform and selection modes */
    GEOMedit_transform_mode(output,name,"normal", BUTTON_UP|BUTTON_MOVING);
    GEOMedit_selection_mode(output,name,"normal", BUTTON_DOWN);
    if (strncmp(name,"%top",4)) GEOMedit_parent(output,name,"%top");

    /* mark the frame output as unchanged */
    AVSmark_output_unchanged("Frameout");

    AVScorout_exec();
    AVScorout_output(output, 0);
}

set_object(char *name, char *parent)
{
    GEOMedit_list output = GEOM_NULL;

    if (!strncmp(name,"EMPTY",5)) return;

    output = GEOMinit_edit_list(output);

    /* set up object so that it will notify the keyframe animator */
    GEOMedit_transform_mode(output,name,"notify", BUTTON_UP|BUTTON_MOVING);
    GEOMedit_selection_mode(output,name,"notify", BUTTON_DOWN);
    if (strncmp(name,"%top",4)) GEOMedit_parent(output,name,parent);

    /* mark the frame output as unchanged */
    AVSmark_output_unchanged("Frameout");

    AVScorout_exec();
    AVScorout_output(output, 0);
}

set_notify(int on)
{
    GEOMedit_list output = GEOM_NULL;

    /* sets the notify mode of the top object on or off */

    output = GEOMinit_edit_list(output);

    if (on) GEOMedit_selection_mode(output,"%top","notify",BUTTON_UP);
    else GEOMedit_selection_mode(output,"%top","normal",BUTTON_UP);

    /* mark the frame output as unchanged */
    AVSmark_output_unchanged("Frameout");

    AVScorout_exec();
    AVScorout_output(output, 0);
}

set_parent(char *name, char *parent)
{
    GEOMedit_list output = GEOM_NULL;

    if (!strncmp(name,"%top",4)) return;

    output = GEOMinit_edit_list(output);

    GEOMedit_parent(output,name,parent);

    /* mark the frame output as unchanged */
    AVSmark_output_unchanged("Frameout");

    AVScorout_exec();
    AVScorout_output(output, 0);
}

trace_all(object, start, end)
    int object, start, end;
{
    GEOMedit_list output = GEOM_NULL;

    /* output traces and keyframe markers for an object */

    output = GEOMinit_edit_list(output);

    trace_trans(&output, object, start, end);
    switch(keytype) {
	case SPHEREKEYS:
	    trace_sphere_keys(&output, object, start, end);
	    break;
	case NUMERICKEYS:
	    trace_numeric_keys(&output, object, start, end);
	    break;
	default:   /* NOKEYS */
	    trace_numeric_keys(&output, object, -1, -1);
	    break;
    }

    /* mark the frame output as unchanged */
    AVSmark_output_unchanged("Frameout");

    AVScorout_exec();
    AVScorout_output(output, 0);
}

trace_keys(object, start, end)
    int object, start, end;
{
    GEOMedit_list output = GEOM_NULL;

    /* output only keyframe markers for an object */

    output = GEOMinit_edit_list(output);

    switch(keytype) {
	case SPHEREKEYS:
	    trace_sphere_keys(&output, object, start, end);
	    break;
	case NUMERICKEYS:
	    trace_numeric_keys(&output, object, start, end);
	    break;
	default:   /* NOKEYS */
	    trace_numeric_keys(&output, object, -1, -1);
	    break;
    }

    AVSmark_output_unchanged("Frameout");

    AVScorout_exec();
    AVScorout_output(output, 0);
}

trace_trans(editlist, object, start, end)
    GEOMedit_list *editlist;
    int object, start, end;
{

    /* create (or delete) the trace object path for the given object */
    /* over the given frame interval from start to end.  If start is */
    /* negative, the trace object path will be deleted.              */

    int nfrmes, j, e;
    float *Traceverts, *Tracecols;
    GEOMobj *traceobj;

    if (start < 0) {
	/* turn off the trace object */
	GEOMedit_visibility(*editlist,"ANIM-Trace",-1);
	return;
    }

    nfrmes = end-start+1;

    /* allocate memory for trace object */
    traceobj = GEOMcreate_obj(GEOM_POLYTRI,GEOM_NULL);
    Traceverts = (float *) malloc(nfrmes*3*sizeof(float));
    Tracecols = (float *) malloc(nfrmes*3*sizeof(float));
    
    /* calculate trace path */
    e = 0;
    for (j=0; j<nfrmes; j++) {
	Traceverts[e] = GetChan(j+start, object, XTRN);
	Tracecols[e++] = TRACERED;
	Traceverts[e] = GetChan(j+start, object, YTRN);
	Tracecols[e++] = TRACEGREEN;
	Traceverts[e] = GetChan(j+start, object, ZTRN);
	Tracecols[e++] = TRACEBLUE;
    }
   
    /* create the polyline and add it to the edit list */
    GEOMadd_polyline(traceobj,Traceverts,Tracecols,nfrmes,GEOM_COPY_DATA);
    GEOMedit_geometry(*editlist,"ANIM-Trace",traceobj);
    GEOMedit_selection_mode(*editlist,"ANIM-Trace","ignore",
				BUTTON_DOWN|BUTTON_MOVING|BUTTON_UP);
    GEOMedit_parent(*editlist,"ANIM-Trace",GetParent(object));

    /* free memory for trace object */
    GEOMdestroy_obj(traceobj);
    free(Traceverts);
    free(Tracecols);
    
}

trace_numeric_keys(editlist, object, start, end)
    GEOMedit_list *editlist;
    int object, start, end;
{
    int c, lflags;
    char text[16];
    float ref[3], color[3], offset[3];
    float height = *keysize;
    GEOMobj *labelobj;

    if (start < 0) {
	/* turn off the spheres */
	GEOMedit_visibility(*editlist,"ANIM-Keys",-1);
	return;
    }

    labelobj = GEOMcreate_obj(GEOM_LABEL, GEOM_NULL);
    lflags = GEOMcreate_label_flags(0, 0, 0, 0, GEOM_LABEL_CENTER, 0);
    color[0] = KEYRED;
    color[1] = KEYGREEN;
    color[2] = KEYBLUE;
    offset[0] = 0.0;
    offset[1] = 0.0;
    offset[2] = 0.0;

    c = start;
    while (c < end) {
	ref[0] = GetChan(c, object, XTRN);
	ref[1] = GetChan(c, object, YTRN);
	ref[2] = GetChan(c, object, ZTRN);
	sprintf(text, "%d", c);
        GEOMadd_label(labelobj, text, ref, offset, height, color, lflags);
	c = NextKey(c, end);
    }
    ref[0] = GetChan(c, object, XTRN);
    ref[1] = GetChan(c, object, YTRN);
    ref[2] = GetChan(c, object, ZTRN);
    sprintf(text, "%d", c);
    GEOMadd_label(labelobj, text, ref, offset, height, color, lflags);

    /* create the object */
    GEOMedit_geometry(*editlist,"ANIM-Keys",labelobj);
    GEOMedit_selection_mode(*editlist,"ANIM-Keys","ignore",
				BUTTON_DOWN|BUTTON_MOVING|BUTTON_UP);
    GEOMedit_parent(*editlist,"ANIM-Keys",GetParent(object));

    /* free memory for trace object */
    GEOMdestroy_obj(labelobj);
}


trace_sphere_keys(editlist, object, start, end)
    GEOMedit_list *editlist;
    int object, start, end;
{
    int nkeys, c, j, e;
    float *sphereverts, *spherecols, *sphererads;
    GEOMobj *sphereobj;

    if (start < 0) {
	/* turn off the spheres */
	GEOMedit_visibility(*editlist,"ANIM-Keys",-1);
	return;
    }

    /* calculate the number of keyframes */
    nkeys = 0;
    c = start;
    while (c < end) {
	nkeys++;
	c = NextKey(c, end);
    }
    nkeys++;

    sphereverts = (float *) malloc(nkeys*3*sizeof(float));
    spherecols = (float *) malloc(nkeys*3*sizeof(float));
    sphererads = (float *) malloc(nkeys*sizeof(float));

    e = 0;
    c = start;
    while (c < end) {
	sphereverts[e] = GetChan(c, object, XTRN);
	spherecols[e++] = KEYRED;
	sphereverts[e] = GetChan(c, object, YTRN);
	spherecols[e++] = KEYGREEN;
	sphereverts[e] = GetChan(c, object, ZTRN);
	spherecols[e++] = KEYBLUE;
	c = NextKey(c, end);
    }
    sphereverts[e] = GetChan(c, object, XTRN);
    spherecols[e++] = KEYRED;
    sphereverts[e] = GetChan(c, object, YTRN);
    spherecols[e++] = KEYGREEN;
    sphereverts[e] = GetChan(c, object, ZTRN);
    spherecols[e++] = KEYBLUE;

    for (j=0; j<nkeys; j++) sphererads[j] = *keysize;

    /* create the object */
    sphereobj = GEOMcreate_sphere(GEOM_NULL, sphereverts, sphererads,
		GEOM_NULL, GEOM_NULL, nkeys, GEOM_COPY_DATA);
    GEOMadd_float_colors(sphereobj, spherecols, nkeys, GEOM_COPY_DATA);
    GEOMedit_geometry(*editlist,"ANIM-Keys",sphereobj);
    GEOMedit_selection_mode(*editlist,"ANIM-Keys","ignore",
				BUTTON_DOWN|BUTTON_MOVING|BUTTON_UP);
    GEOMedit_parent(*editlist,"ANIM-Keys",GetParent(object));

    /* free memory for trace object */
    GEOMdestroy_obj(sphereobj);
    free(sphereverts);
    free(spherecols);
    free(sphererads);
}

/*************************
 * SOME UTILITY ROUTINES *
 *************************/

Rot_init(qrot)
    double qrot[4];
{
    Quat_init(qrot);
}

Trans_init(trans)
    double trans[3];
{
    trans[0] = trans[1] = trans[2] = 0.0;
}

Scale_init(scale)
    double scale[3];
{
    scale[0] = scale[1] = scale[2] = 1.0;
}

make_beep()
{
    /* make the terminal beep */
    fprintf(stderr, "\007");
    fflush(stderr);
}

send_command(char *cmdin)
{
    char *cout[512];
    char *cerr[512];

    AVScommand("kernal", cmdin, cout, cerr);
    
    return;
}

int obj_exists(char *obj)
{
    char *cout[512];
    char *cerr[512];
    char cin[256];

    /* check to see if an object exists */
    /* objects beginning with "%light or "%camera" always return(0) */
    /* this routine is a hack.  AVS doesn't seem to have a nice way of */
    /* seeing if a particular object exists */
    /* this routine fails miserably if the scene you want to check isn't */
    /* the current one.  There's no good way around this that I know of. */

    if ((!strncmp(obj, "%camera", 7)) || (!strncmp(obj, "%light", 6))) return(0);
    
    /* try to get the object's matrix and see if we get an error (yuck!) */
    sprintf(cin, "geom_get_matrix -object %s", obj);
    AVScommand("kernal", cin, cout, cerr);
    if (strcmp(*cerr, "")) return(0);
    return(1);
}    
    
cleanup(int trace)
{
  int i;

  /* reset objects and close any popups that might be open */
  /* and kills the Keyframe Animator module */
  
  /* reset all objects */
  for (i=1; i<=GetNumobjs(); i++) reset_object(GetName(i));
  
  /* turn off trace object */
  if (trace) trace_all(1, -1, -1);
    
  /* close all popups */
  send_command("panel \"settingsbrow.$Module\" -hide");
  send_command("panel \"loadbrow.$Module\" -hide");
  send_command("panel \"savebrow.$Module\" -hide");

  send_command("mod_delete $Module");
}
  
int write_animfile(char *savefile, int start, int end)
{
    int mode, empty, objs, i, j, k;
    char *mess;
    FILE *fd;
    
    if (savefile == NULL) return(-1);
    if ((fd = fopen(savefile,"w")) == NULL) return(-1);

    /* see if there are any EMPTY objects */
    empty = 0;
    for (i=1; i<=GetNumobjs(); i++) if (!strncmp(GetName(i),"EMPTY",5)) empty++;

    mode = SAVEALL;
    if (empty > 0) {
	mess = AVSmessage(VERSION,AVS_Warning,NULL,"SAVE ANIM",
	      "SAVE!SAVEALL!ABORT",
	      "Press SAVE to save ONLY the %d %s!\n\n%s (%d objects).\n\n%s\n",
			GetNumobjs()-empty,
			"object(s) that are NOT \"EMPTY\"",
			"Press SAVEALL to save ALL objects in the animation",
			GetNumobjs(),
			"Press ABORT to abort save.");
	if (!strcmp(mess,"ABORT")) return(1);
	else if (!strcmp(mess,"SAVE")) mode = SAVENONEMPTY;
    }
    
    switch(mode) {
	case SAVENONEMPTY:
	    objs = GetNumobjs()-empty;
	    break;
	default:  /* SAVEALL */
	    objs = GetNumobjs();
	    break;
    }
    
    /* start writing the file */
    fprintf(stderr,
	  "ANIM:  Writing animation file %s (frames %d-%d, %d object(s))...\n",
	  savefile,start,end,objs);

    /* write version, total frames, and total objects */
    fprintf(fd,"%s\n",VERSION);
    fprintf(fd,"%d %d\n",end-start+1,objs);

    /* write object table */
    k = 0;
    for(i=1; i<=GetNumobjs(); i++) {
      if (mode == SAVEALL || strncmp(GetName(i),"EMPTY",5)) {
	k++;
	fprintf(fd,"obj%02d: %s %s\n",k,GetName(i),GetParent(i));
      }
    }

    /* write out each frame of the animation */
    for(i=start; i<=end; i++) {
      fprintf(fd,"frame%05d: %d\n",i-start+1,GetKey(i));
      k = 0;
      for (j=1; j<=GetNumobjs(); j++) {
	if (mode == SAVEALL || strncmp(GetName(j),"EMPTY",5)) {
	  k++;
	  GetRot(qrot, i, j);
	  GetSize(scale, i, j);
	  GetTrans(trans, i, j);
	  fprintf(fd,"     obj%02d: %f\n",k,(float)scale[0]);
	  fprintf(fd,"            %f %f %f %f\n",(float)qrot[0],(float)qrot[1], (float)qrot[2], (float)qrot[3]);
	  fprintf(fd,"            %f %f %f\n",(float)trans[0], (float)trans[1], (float)trans[2]);
	}
      }
    }

    fclose(fd);
    return(0);
}

int read_animfile(loadfile, mode)
    char *loadfile;
    int mode;
{
    int frames, objs, i, j, f, o;
    char *mess;
    FILE *fd;
    char str[256];
    char tname[128];
    char tparent[128];
    double ttrans[3];
    double tqrot[4];
    double size;    

    if (loadfile == NULL) return(-1);
    if ((fd = fopen(loadfile,"r")) == NULL) return(-1);

    objs = 0;
    frames = 0;
    switch(mode) {
	case LOADMERGE:
	    objs = GetNumobjs();
	    break;
	case LOADAPPEND:
	    frames = GetNumframes();
	    break;
	default:  /* LOADREPLACE */
	    break;
    }
    
    fscanf(fd,"%s",str);
    if (strcmp(str,VERSION)) {
      mess = AVSmessage(VERSION,AVS_Warning,NULL,"LOAD ANIM",
	"ABORT!LOAD",
	"\n%s (%s)\n%s (%s)!\n\n%s\n",
	"The version number of the saved animation",
	str,
	"does not match this version of the Keyframe Animator module",
	VERSION,
	"Press ABORT to abort load, LOAD to attempt load anyway.");
      if (!strcmp(mess,"ABORT")) {
	fclose(fd);
	return(1);
      }
    }
    
    fscanf(fd,"%d %d", &f, &o);
    fprintf(stderr,
	    "ANIM:  Reading animation file %s (%d frames, %d object(s))...\n",
	    loadfile,f,o);

    f += frames;
    /* add more frames if need be */
    if (f>GetNumframes()) {
	if (Anim_addframes(f-GetNumframes()) < 0) {
	    AVSfatal("\n\nUnable to allocate memory for frame information!\n");
	    return(-1);
	}
    }

    o += objs;
    /* add more objects if need be */
    if (o>GetNumobjs()) {
	if (Anim_addobjects(o-GetNumobjs()) < 0) {
	    AVSfatal("\n\nUnable to allocate memory for object information!\n");
	    return(-1);
	}
    }

    /* read object table */
    for(i=objs+1; i<=o; i++) {
      fscanf(fd,"%*s %s %s", tname, tparent);
      SetName(i, tname);
      SetParent(i, tparent);
      if (!obj_exists(tname)) {
	  sprintf(str, "EMPTY-%d", i);
	  fprintf(stderr, "ANIM:  Object %s doesn't exist...using %s.\n",
		tname, str);
	  SetName(i, str);
      }
      if (!obj_exists(GetParent(i))) {
	  fprintf(stderr, "ANIM:  Parent object %s doesn't exist...using %top.\n",
		tparent);
	  SetParent(i, "%top");
      }
    }

    /* read frame information */
    for(i=frames+1; i<=f; i++) {
      fscanf(fd,"%*s %d", &j);
      SetKey(i, j);
      for (j=objs+1; j<=o; j++) {
	fscanf(fd,"%*s %lf", &size);
	fscanf(fd,"%lf %lf %lf %lf", &tqrot[0], &tqrot[1], &tqrot[2], &tqrot[3]);
	fscanf(fd,"%lf %lf %lf", &ttrans[0], &ttrans[1], &ttrans[2]);

	SetSize(i, j, size);
	SetRot(i, j, tqrot);
	SetTrans(i, j, ttrans);
      }
    }

    fclose(fd);
    
    return(f);
}
