/****************************************************************************
                  INTERNATIONAL AVS CENTER
	(This disclaimer must remain at the top of all files)

WARRANTY DISCLAIMER

This module and the files associated with it are distributed free of charge.
It is placed in the public domain and permission is granted for anyone to use,
duplicate, modify, and redistribute it unless otherwise noted.  Some modules
may be copyrighted.  You agree to abide by the conditions also included in
the AVS Licensing Agreement, version 1.0, located in the main module
directory located at the International AVS Center ftp site and to include
the AVS Licensing Agreement when you distribute any files downloaded from 
that site.

The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module provide absolutely
NO WARRANTY OF ANY KIND with respect to this software.  The entire risk as to
the quality and performance of this software is with the user.  IN NO EVENT
WILL The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module BE LIABLE TO
ANYONE FOR ANY DAMAGES ARISING FROM THE USE OF THIS SOFTWARE, INCLUDING,
WITHOUT LIMITATION, DAMAGES RESULTING FROM LOST DATA OR LOST PROFITS, OR ANY
SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES.

This AVS module and associated files are public domain software unless
otherwise noted.  Permission is hereby granted to do whatever you like with
it, subject to the conditions that may exist in copyrighted materials. Should
you wish to make a contribution toward the improvement, modification, or
general performance of this module, please send us your comments:  why you
liked or disliked it, how you use it, and most important, how it helps your
work. We will receive your comments at avs@ncsc.org.

Please send AVS module bug reports to avs@ncsc.org.

******************************************************************************/
/* 
 * KEYFRAME ANIMATOR:
 * perform keyframe animation and interpolation of objects in a network.
 * rotation and interpolation of rotation can be either standard X-Y-Z matrix
 * rotation, or quaternion rotation.
 *
 * created by Brian Kaplan
 *      kaplan@cica.indiana.edu
 *      Center for Innovative Computer Applications (CICA)
 *      Indiana University
 *
 * thanks to the following for their help:
 *      Gregory Travis, Eric Ost -- general C help.
 *      Andrew Hanson -- help with interpolation code.
 *
 * Copyright (c) 1991 Brian Kaplan, CICA, Indiana University
 * All rights reserved.
 *
 * This material was developed at the Center for Innovative Computer
 * Applications (CICA), Indiana University.  Permission to copy this
 * software or any portion of it, to redistribute it, and to use it for
 * any purpose and without fee is granted, subject to the following
 * restrictions and understandings.
 *
 * 1. Any copy made of this software must include this copyright notice
 * in full.  Any materials containing this software or any part of this
 * software must include this copyright notice in full.
 *
 * 2. Users of this software agree to make their best efforts (a) to
 * return to the Center for Innovative Computer Applications any improvements
 * or extensions that they make, so that these may be included in future
 * releases; and (b) to inform Indiana University of noteworthy uses of this
 * software.
 *
 * 3. All materials developed as a consequence of the use of this
 * software shall duly acknowledge such use, in accordance with the usual
 * standards of acknowledging credit in academic research.
 *
 * 4. Indiana University has made no warrantee or representation that the
 * operation of this software will be error-free, and Indiana University
 * is under no obligation to provide any services, by way of maintenance,
 * update, or otherwise.
 *
 * 5. In conjunction with products arising from the use of this material,
 * there shall be no use of the name of Indiana University or the Center
 * for Innovative Computer Applications nor of any adaptation thereof in
 * any advertising, promotional, or sales literature without prior written
 * consent from Indiana University in each case.
 *
 */

#include <stdio.h>
#include <math.h>
#include "Keyframe.h"

/*********************************************
 * ROUTINES FOR HANDLING QUATERNION ROTATION *
 *********************************************/

Quat_init(Q)
/**************************************************************/
/* Initializes quaternion Q to its default values (1,0,0,0)   */
/* which specifies NO rotation.                               */ 
/**************************************************************/
     float Q[4];
{
  int i;

  Q[0] = 1.0;
  for (i = 1; i < 4; i++) Q[i] = 0.0;
}

Quat_rotate(Q,a,i)
/**************************************************************/
/* Rotate a quaternion given a RELATIVE angle of rotation in  */
/* degrees and an axis of rotation: 1=Xaxis, 2=Yaxis, 3=Zaxis */
/* resulting Quaternion is in float Q[4].                     */
/**************************************************************/
float Q[4];
float	a;
int	i;
{
  int	j, k;
  float	E,x,y;

  /* Parts of this code modified from Graphics Gems. */

  x = cos(rad(a)/2);
  y = sin(rad(a)/2);

  /* Compute the successor axis of i [1,2,3] and  j [1,2,3] */
  if ((j = i + 1) > 3) j = 1;
  if ((k = j + 1) > 3) k = 1;
  E = Q[i];
  Q[i] = E * x + 1 * y * Q[0];
  Q[0] = Q[0] * x - 1 * y * E;
  E = Q[j];
  Q[j] = E * x + y * Q[k];
  Q[k] = Q[k] * x - y * E;
}

Quat_evalmatrix(xform,Q)
/**************************************************************/
/* Builds the transformation matrix xform[4][4] based on      */
/* the current quaternion Q[4].                               */
/**************************************************************/
float xform[4][4];
float Q[4];
{
  float M[4][4];
  float	e, f, r[4];
  int	i, j, k;

  /* Parts of this code modified from Graphics Gems. */

  /* Find square values */
  for (i = 0; i < 4; i++) r[i] = Q[i] * Q[i];

  /* Compute each element of the matrix. */
  /* j is the successor of i (in 1,2,3), while k is the successor of j. */
  for (i = 1; i < 4; i++) {
    if ((j = i + 1) > 3) j = 1;
    if ((k = j + 1) > 3) k = 1;
    e = 2. * Q[i] * Q[j];
    f = 2. * Q[k] * Q[0];
    M[j][i] = e - f;
    M[i][j] = e + f;
    M[i][i] = r[i] + r[0] - r[j] - r[k];
    M[0][i] = 0.;
    M[i][0] = 0.;
  }
  M[0][0] = 1.;

  /* copy the transformation matrix to the correct form for AVS */
  for (i=0; i<3; i++) {
    for (j=0; j<3; j++) {
      xform[j][i]=M[i+1][j+1];
    }
    xform[3][i] = M[0][i+1];
    xform[i][3] = 0.0;
  }
  xform[3][3] = 1.0;
}

quat_dot(Q,F,G)
/**************************************************************/
     float Q[4];
     float F[4];
     float G[4];
{
  float temp;
  int i;

  /* Perform a quaternion dot-product. */
  Q[0] = (F[0] * G[0]) - ((F[1] * G[1]) + (F[2] * G[2]) + (F[3] * G[3]));
  Q[1] = (F[0] * G[1]) + (G[0] * F[1]) + ((F[2] * G[3]) - (F[3] * G[2]));
  Q[2] = (F[0] * G[2]) + (G[0] * F[2]) + ((F[3] * G[1]) - (F[1] * G[3]));
  Q[3] = (F[0] * G[3]) + (G[0] * F[3]) + ((F[1] * G[2]) - (F[2] * G[1]));

  /* make sure the resulting quaternion is a unit quaternion */
  temp = sqrt((double)(Q[0]*Q[0] + Q[1]*Q[1] + Q[2]*Q[2] + Q[3]*Q[3]));
  for (i=0; i<4; i++) Q[i] = Q[i]/temp;
}


/***********************************************
 * ROUTINES FOR HANDLING CHANNEL INTERPOLATION *
 ***********************************************/

Interp_linear(start,end,ei,eo,object,qr,xr,yr,zr,xt,yt,zt,sc) 
/**************************************************************/
/* Linearly interpolate channels based on keyframes.          */
/* The interpolation will be from the frame specified by      */
/* start to the frame specified by end.  The object to be     */
/* interpolated is specified in object.  The other parameters */
/* are either 1 to interpolate that channel, or 0 to not      */
/* interpolate that channel.                                  */
/**************************************************************/
     int start,end,ei,eo,object,qr,xr,yr,zr,xt,yt,zt,sc;
{
  int s,e,ease;

  /* if the keyframes are consecutive or the same, return now */
  if (end-start < 2) return(1);

  if (ei) ease = -1;
  else ease = 0;

  s = start;
  e = 0;

  /* interpolate each animation segment between keyframes */
  while (e < end) {
    e = NextKey(s,end);

    if (eo && (e == end)) ease = 1;

    /* interp from s to e */
    if (qr) qlinterp(s,e,ease,object);
    else {
      if (xr) rlinterp(s,e,ease,object,XROT);
      if (yr) rlinterp(s,e,ease,object,YROT);
      if (zr) rlinterp(s,e,ease,object,ZROT);
    }
    if (xt) rlinterp(s,e,ease,object,XTRN);
    if (yt) rlinterp(s,e,ease,object,YTRN);
    if (zt) rlinterp(s,e,ease,object,ZTRN);
    if (sc) rlinterp(s,e,ease,object,SIZE);

    ease = 0;
    s = e;
  }  
}

rlinterp(s,e,ease,obj,channel)
/**************************************************************/
     int s,e,ease,obj,channel;
{
  int j;
  float u,deltau,temp,diff;

  /* change frame numbers and object number to array indices */
  e--;
  s--;
  obj--;

  diff = Anim[e][obj][channel] - Anim[s][obj][channel];
  deltau = 1.0/(e-s);
  temp = 0.0;

  /* fill in frames between s and e */
  for (j=1; j<(e-s); j++) {
    temp+=deltau;
    if (ease < 0) u = 1.0 - sin((1.0-temp)*1.5708);
    else if (ease > 0) u = sin(temp*1.5708);
    else u=temp;

    Anim[j+s][obj][channel] = Anim[s][obj][channel] + (u*diff);
  }
}

qlinterp(s,e,ease,obj)
/**************************************************************/
     int s,e,ease,obj;
{
  float F[4],G[4],N[4],O[4];
  float u,deltau,temp;
  int j;

  /* change frame numbers and object number to array indices */
  s--;
  e--;
  obj--;

  /* get the start and end quaternions in F and G */
  F[0] = Anim[s][obj][QTRC];
  F[1] = Anim[s][obj][QTRX];
  F[2] = Anim[s][obj][QTRY];
  F[3] = Anim[s][obj][QTRZ];
  G[0] = Anim[e][obj][QTRC];
  G[1] = Anim[e][obj][QTRX];
  G[2] = Anim[e][obj][QTRY];
  G[3] = Anim[e][obj][QTRZ];
  /* determine the interpolation constants (stored in N) */
  setup_linterp_quaternion(N,F,G);

  /* linearly interpolate u from 0 to 1 */
  deltau = 1.0/(e-s);
  temp = 0.0;

  /* do the interpolations for each frame based on u */
  for (j=1; j<(e-s); j++) {
    temp+=deltau;
    if (ease < 0) u = 1.0 - sin((1.0-temp)*1.5708);
    else if (ease > 0) u = sin(temp*1.5708);
    else u=temp;

    linterp_quaternion(O,F,N,u);
    Anim[j+s][obj][QTRC] = O[0];
    Anim[j+s][obj][QTRX] = O[1];
    Anim[j+s][obj][QTRY] = O[2];
    Anim[j+s][obj][QTRZ] = O[3];
  }
}

setup_linterp_quaternion(N,F,G)
/**************************************************************/
     float N[4];
     float F[4];
     float G[4];
{
  float T[4];
  float Finv[4];
  float sinth2, sinth2inv, sum;
  int i;

  /* Calculate the constant values (stored in N) which are then */
  /* passed to linterp_quaternion for the final interpolation.  */
  /* The constants are calculated from the original quaternions */
  /* F and G which we are interpolating between.                */

  /* inverse quaternion F */
  Finv[0] = F[0];
  for (i=1; i<4; i++) Finv[i] = -F[i];

  /* calculate the constants stored in N */
  quat_dot(T,Finv,G);
  N[0] = 2.0*acos(T[0]);

  /*  Ideally, we just want to do this:              */
  /*       sinth2inv = 1.0/sqrt(1.0 - T[0]*T[0]);    */
  /*       for (i=1; i<4; i++) N[i] = N[i] * sum;    */
  /*  instead, we have to check some special cases:  */

  sinth2 = sin(.5*N[0]);
  /* are we very close to zero? */
  if (fabs(sinth2) < .0000001) {
    N[1] = 0.0;
    N[2] = 0.0;
    N[3] = 1.0;
  }
  else {
    sinth2inv = 1.0/sinth2;
    sum = 0.0;
    for (i=1; i<4; i++) {
      N[i] = T[i]*sinth2inv;
      sum += N[i]*N[i];
    }
    sum = 1.0/sqrt((double)sum);
    for (i=1; i<4; i++) N[i] = N[i] * sum;
  }
}

linterp_quaternion(Q,F,N,u)
/**************************************************************/
     float Q[4];
     float F[4];
     float N[4];
     float u;
{
  float T[4];
  float theta2,sinth2;
  int i;

  /* Give a new quaternion Q that is linearly interpolated      */
  /* based on the parameter u which ranges from 0 to 1          */
  /* F is the initial quaternion rotation.                      */
  /* N contains constants which are calculated from             */
  /* setup_linterp_quaternion.                                  */
  /* note, to make the object rotate past the keyframe          */
  /* quaternion, u may be increased beyond 1.                   */

  theta2 = 0.5*u*N[0];

  /* adjust T for the proper timestep */
  T[0] = cos(theta2);
  sinth2 = sin(theta2);
  for (i=1; i<4; i++) T[i] = N[i]*sinth2;

  /* calculate the final interpolated rotation quaternion Q */
  quat_dot(Q,F,T);
}  


Interp_spline(start,end,ei,eo,object,qr,xr,yr,zr,xt,yt,zt,sc) 
/**************************************************************/
/* Spline interpolate channels based on keyframes.            */
/* The interpolation will be from the frame specified by      */
/* start to the frame specified by end.  ei and eo are 0 or 1 */
/* for an ease-in and/or an ease-out.  The object to be       */
/* interpolated is specified in object.  The other parameters */
/* are either 1 to interpolate that channel, or 0 to not      */
/* interpolate that channel.                                  */
/**************************************************************/
     int start,end,ei,eo,object,qr,xr,yr,zr,xt,yt,zt,sc;
{
  int p,s,e,n,ease;

  /* if the keyframes are consecutive or the same, return now */
  if (end-start < 2) return(1);

  if (ei) ease = -1;
  else ease = 0;

  s = start;
  e = start;
  n = NextKey(e,end);

  /* interpolate each animation segment between keyframes */
  
  do {
    p = s;
    s = e;
    e = n;
    n = NextKey(e,end);

    if (eo && (e == end)) ease = 1;

    /* interp from s to e using p as previous and n as next keyframes */
    if (qr) qsinterp(p,s,e,n,ease,object);
    else {
      if (xr) rsinterp(p,s,e,n,ease,object,XROT);
      if (yr) rsinterp(p,s,e,n,ease,object,YROT);
      if (zr) rsinterp(p,s,e,n,ease,object,ZROT);
    }
    if (xt) rsinterp(p,s,e,n,ease,object,XTRN);
    if (yt) rsinterp(p,s,e,n,ease,object,YTRN);
    if (zt) rsinterp(p,s,e,n,ease,object,ZTRN);
    if (sc) rsinterp(p,s,e,n,ease,object,SIZE);

    ease = 0;
  } while (e < end);
}

rsinterp(p,s,e,n,ease,obj,channel)
/**************************************************************/
     int p,s,e,n,ease,obj,channel;
{

  float P,S,E,N;
  float u2,u3,U1,U2,U3;
  float deltau,u,temp;
  int steps,i;

  /* change frame numbers and object number to array indices */
  p--;
  s--;
  e--;
  n--;
  obj--;

  /* determine the values stored in the previous, current, and next keys */
  P = Anim[p][obj][channel];
  S = Anim[s][obj][channel];
  E = Anim[e][obj][channel];
  N = Anim[n][obj][channel];

  /* Draw a Catmull Rom spline curve */
  /* Parts of this code modified from code written by Andrew Hanson */

  steps = e-s;

  /* we need only interpolate if we have a frame between keyframes */
  if (steps < 2) return(1);

  temp = 0.0;
  deltau = 1.0/steps;

  for (i=1; i<steps; i++) {
    temp+=deltau;
    /* use a sine curve for ease-ins and ease-outs */
    if (ease < 0) u = 1.0 - sin((1.0-temp)*1.5708);
    else if (ease > 0) u = sin(temp*1.5708);
    else u=temp;

    u2=u*u;
    u3=u*u2;
    U3 = (-P + 3*S - 3*E + N);
    U2 = (2*P - 5*S + 4*E - N);
    U1 = (-P + E);

    Anim[i+s][obj][channel] = 0.5*(u3*U3 + u2*U2 + u*U1) + S;
  }
}

qsinterp(p,s,e,n,ease,obj)
/**************************************************************/
     int p,s,e,n,ease,obj;
{
  int i;

  /* change frame numbers and object number to array indices */
  p--;
  s--;
  e--;
  n--;
  obj--;

  i = p + s + e + n + obj + ease;
  /* the code for spline quaternion roations will go here */
  return(i);
}
