/* 
 * GraphicsGems.h  
 * Version 1.0 - Andrew Glassner
 * from "Graphics Gems", Academic Press, 1990
 */

/*********************/
/* 2d geometry types */
/*********************/

typedef struct Point2Struct {	/* 2d point */
	double x, y;
	} Point2;
typedef Point2 Vector2;

typedef struct IntPoint2Struct {	/* 2d integer point */
	int x, y;
	} IntPoint2;

typedef struct Matrix3Struct {	/* 3-by-3 matrix */
	double element[3][3];
	} Matrix3;

typedef struct Box2dStruct {		/* 2d box */
	Point2 min, max;
	} Box2;
	

/*********************/
/* 3d geometry types */
/*********************/

typedef struct Point3Struct {	/* 3d point */
	double x, y, z;
	} Point3;
typedef Point3 Vector3;

typedef struct IntPoint3Struct {	/* 3d integer point */
	int x, y, z;
	} IntPoint3;


typedef struct Matrix4Struct {	/* 4-by-4 matrix */
	double element[4][4];
	} Matrix4;

typedef struct Box3dStruct {		/* 3d box */
	Point3 min, max;
	} Box3;

typedef struct Vector4Struct {
	double x, y, z, w;
	} Vector4;


/****************************/
/* memory allocation macros */
/****************************/


#define NEWTYPE(x)       (x *)(malloc((unsigned)sizeof(x)))

/***********************/
/* one-argument macros */
/***********************/

/* absolute value of a */
#define ABS(a)		(((a)<0) ? -(a) : (a))

/* round a to nearest integer towards 0 */
#define FLOOR(a)		((a)>0 ? (int)(a) : -(int)(-a))

/* round a to nearest integer away from 0 */
#define CEILING(a) \
((a)==(int)(a) ? (a) : (a)>0 ? 1+(int)(a) : -(1+(int)(-a)))

/* round a to nearest int */
#define ROUND(a)	((a)>0 ? (int)(a+0.5) : -(int)(0.5-a))		

/* take sign of a, either -1, 0, or 1 */
#define ZSGN(a)		(((a)<0) ? -1 : (a)>0 ? 1 : 0)	

/* take binary sign of a, either -1, or 1 if >= 0 */
#define SGN(a)		(((a)<0) ? -1 : 0)

/* shout if something that should be true isn't */
#define ASSERT(x) \
if (!(x)) fprintf(stderr," Assert failed: x\n");

/* square a */
#define SQR(a)		((a)*(a))	


/***********************/
/* two-argument macros */
/***********************/

/* find minimum of a and b */
#define MIN(a,b)	(((a)<(b))?(a):(b))	

/* find maximum of a and b */
#define MAX(a,b)	(((a)>(b))?(a):(b))	

/* swap a and b (see Gem by Wyvill) */
#define SWAP(a,b)	{ a^=b; b^=a; a^=b; }

/* linear interpolation from l (when a=0) to h (when a=1)*/
/* (equal to (a*h)+((1-a)*l) */
#define LERP(a,l,h)	((l)+(((h)-(l))*(a)))

/* clamp the input to the specified range */
#define CLAMP(v,l,h)	((v)<(l) ? (l) : (v) > (h) ? (h) : v)

/********************/
/* useful constants */
/********************/

#define PI		3.141592	/* the venerable pi */
#define PITIMES2	6.283185	/* 2 * pi */
#define PIOVER2		1.570796	/* pi / 2 */
#define E		2.718282	/* the venerable e */
#define SQRT2		1.414214	/* sqrt(2) */
#define SQRT3		1.732051	/* sqrt(3) */
#define GOLDEN		1.618034	/* the golden ratio */
#define DTOR		0.017453	/* convert degrees to radians */
#define RTOD		57.29578	/* convert radians to degrees */
#define deg(x) (((x)*180.0)/M_PI)
#define rad(x) ((x)*M_PI/180.0)

/***********************************************************************
*  The following macros handle most vector operations, the exceptions
*  usually being complex equations with four or more vectors.
*
*  An alternate form for the multiple-statement macros is the
*  "if (1) <macro_body> else" form.  This allows for temporary variable
*  declaration and control-flow constructs, but cannot be used
*  everywhere a function call could, as with the form used below.
*
*  Note that since the vector arguments are not enclosed in parentheses
*  in the macro body, you can scale the vector arguments in the macro
*  calls, e.g. Vec2Op(vec1,=,scalar*vec2).
*
*  Here are some example uses of the following macros:
*
*      printf ("Vector = <%lg %lg %lg>\n", VecList(vector))
*      vector_dot = VecDot (vec1, vec2)
*      norm = VecNorm (vector)
*      VecScalar (vector, /=, norm)
*      VecScalar (vector, *=, scale)
*      Vec3Scalar (Xaxis, =, 1.0, 0.0, 0.0)
*      Vec3Scalar (vector, *=, Xshear, Yshear, Zshear)
*      Vec2Op (vector, =, Xaxis)
*      Vec2Op (vector, +=, norm * Xaxis)
*      Vec3Op (vec1, =, vec2, =, Xaxis)
*      Vec3Op (vec1, =, vec2, -, vec3)
*      Vec3Op (vec1, +=, scale2 * vec2, -, scale3 * vec3)
*      VecCross (vec1, -=, vec2, X, vec3)
*      VecCrossSafe (vec1, =, vec1, X, Xaxis)
***********************************************************************/

#include <math.h>       /* Needed for sqrt() definition. */

    /* Vector type definition.  If you define colors in the same manner,
    ** you can also use these macros for color vector operations.  */

typedef double  Vector[3];
typedef Vector      Point;              /* For readability. */

    /* VecList enumerates the vector fields for function calls. */

#define VecList(V)      V[0], V[1], V[2]

    /* This macro computes the dot product of two vectors. */

#define VecDot(A,B)     ((A[0]*B[0]) + (A[1]*B[1]) + (A[2]*B[2]))

    /* The VecNorm macro computes the norm of the vector. */

#define VecNorm(V)      sqrt(VecDot(V,V))

    /* VecScalar provides for scalar operations on a vector. */

#define VecScalar(V,assign_op,k)        \
(   V[0] assign_op k,   \
    V[1] assign_op k,   \
    V[2] assign_op k    \
)

    /* Vec3Scalar provides for vector operations that involve three
    ** distinct scalar factors. */

#define Vec3Scalar(V,assign_op,a,b,c)   \
(   V[0] assign_op a,   \
    V[1] assign_op b,   \
    V[2] assign_op c    \
)

    /* Vec2Op provides for operations with two vectors. */

#define Vec2Op(A,assign_op,B)   \
(   A[0] assign_op B[0],        \
    A[1] assign_op B[1],        \
    A[2] assign_op B[2]         \
)

    /* Vec3op handles vector operations with three vectors. */

#define Vec3Op(A,assign_op,B,op,C)      \
(   A[0] assign_op B[0] op C[0],        \
    A[1] assign_op B[1] op C[1],        \
    A[2] assign_op B[2] op C[2]         \
)

    /* The cross product macros come in two flavors.  VecCross() requires
    ** that all three vectors are distinct.  With the VecCrossSafe()
    ** macro, it's OK to do A <- A X B, but this requires a temporary
    ** vector for storage, which in turn requires the "if (1) ... else"
    ** form.  As an alternative, a global temporary vector could be used.
    */

#define VecCross(A,assign_op,B,dummy_op,C)              \
(   A[0] assign_op (B[1] * C[2]) - (B[2] * C[1]),       \
    A[1] assign_op (B[2] * C[0]) - (B[0] * C[2]),       \
    A[2] assign_op (B[0] * C[1]) - (B[1] * C[0])        \
)

#define VecCrossSafe(A,assign_op,B,dummy_op,C)  \
if (1)                          \
{   auto Vector result;         \
    VecCross (result,=,B,X,C);  \
    Vec2Op (A,=,result);        \
} else

/********** routines **************/

extern double V2SquaredLength(), V2Length();
extern double V2Dot(), V2DistanceBetween2Points(); 
extern Vector2 *V2Negate(), *V2Normalize(), *V2Scale(), *V2Add(), *V2Sub();
extern Vector2 *V2Lerp(), *V2Combine(), *V2Mul(), *V2MakePerpendicular();
extern Vector2 *V2New(), *V2Duplicate();
extern Point2 *V2MulPointByProjMatrix();
extern Matrix3 *V2MatMul(), *TransposeMatrix3();

extern double V3SquaredLength(), V3Length();
extern double V3Dot(), V3DistanceBetween2Points();
extern Vector3 *V3Normalize(), *V3Scale(), *V3Add(), *V3Sub();
extern Vector3 *V3Lerp(), *V3Combine(), *V3Mul(), *V3Cross();
extern Vector3 *V3New(), *V3Duplicate();
extern Point3 *V3MulPointByMatrix(), *V3MulPointByProjMatrix();
extern Matrix4 *V3MatMul();
extern Vector4 *V4MulPointByMatrix();

extern double RegulaFalsi(), NewtonRaphson(), findroot();
