/****************************************************************************
                  INTERNATIONAL AVS CENTER
	(This disclaimer must remain at the top of all files)

WARRANTY DISCLAIMER

This module and the files associated with it are distributed free of charge.
It is placed in the public domain and permission is granted for anyone to use,
duplicate, modify, and redistribute it unless otherwise noted.  Some modules
may be copyrighted.  You agree to abide by the conditions also included in
the AVS Licensing Agreement, version 1.0, located in the main module
directory located at the International AVS Center ftp site and to include
the AVS Licensing Agreement when you distribute any files downloaded from 
that site.

The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module provide absolutely
NO WARRANTY OF ANY KIND with respect to this software.  The entire risk as to
the quality and performance of this software is with the user.  IN NO EVENT
WILL The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module BE LIABLE TO
ANYONE FOR ANY DAMAGES ARISING FROM THE USE OF THIS SOFTWARE, INCLUDING,
WITHOUT LIMITATION, DAMAGES RESULTING FROM LOST DATA OR LOST PROFITS, OR ANY
SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES.

This AVS module and associated files are public domain software unless
otherwise noted.  Permission is hereby granted to do whatever you like with
it, subject to the conditions that may exist in copyrighted materials. Should
you wish to make a contribution toward the improvement, modification, or
general performance of this module, please send us your comments:  why you
liked or disliked it, how you use it, and most important, how it helps your
work. We will receive your comments at avs@ncsc.org.

Please send AVS module bug reports to avs@ncsc.org.

******************************************************************************/
/* 
 * KEYFRAME ANIMATOR:
 * perform keyframe animation and interpolation of objects in a network.
 * rotation and interpolation of rotation can be either standard X-Y-Z matrix
 * rotation, or quaternion rotation.
 *
 * created by Brian Kaplan
 *      kaplan@cica.indiana.edu
 *      Center for Innovative Computer Applications (CICA)
 *      Indiana University
 *
 * thanks to the following for their help:
 *      Gregory Travis, Eric Ost -- general C help.
 *      Andrew Hanson -- help with interpolation code.
 *
 * Copyright (c) 1991 Brian Kaplan, CICA, Indiana University
 * All rights reserved.
 *
 * This material was developed at the Center for Innovative Computer
 * Applications (CICA), Indiana University.  Permission to copy this
 * software or any portion of it, to redistribute it, and to use it for
 * any purpose and without fee is granted, subject to the following
 * restrictions and understandings.
 *
 * 1. Any copy made of this software must include this copyright notice
 * in full.  Any materials containing this software or any part of this
 * software must include this copyright notice in full.
 *
 * 2. Users of this software agree to make their best efforts (a) to
 * return to the Center for Innovative Computer Applications any improvements
 * or extensions that they make, so that these may be included in future
 * releases; and (b) to inform Indiana University of noteworthy uses of this
 * software.
 *
 * 3. All materials developed as a consequence of the use of this
 * software shall duly acknowledge such use, in accordance with the usual
 * standards of acknowledging credit in academic research.
 *
 * 4. Indiana University has made no warrantee or representation that the
 * operation of this software will be error-free, and Indiana University
 * is under no obligation to provide any services, by way of maintenance,
 * update, or otherwise.
 *
 * 5. In conjunction with products arising from the use of this material,
 * there shall be no use of the name of Indiana University or the Center
 * for Innovative Computer Applications nor of any adaptation thereof in
 * any advertising, promotional, or sales literature without prior written
 * consent from Indiana University in each case.
 *
 */

#include <stdio.h>
#include "Keyframe.h"

/*******************************
 * GLOBAL DATABASE DEFINITIONS *
 *******************************/

struct Objinfo *Objlist;          /* stores information about objects */
struct Frameinfo *Framelist;       /* stores information about frames */
float ***Anim;         /* stores animation values for transformations */
int Totobjs;              /* the total number of objects in animation */
int Totframes;         /* the total number of frames in the animation */
int Totchannels;     /* the total number of channels in the animation */

float ***Alloc3D();

/**************************************************************
 *  ROUTINES FOR HANDLING THE ANIMATION DATABASE              *
 **************************************************************/

Anim_init(numframes,numobjs,numchannels)
/**************************************************************/
/* initializes the animation database                         */
/* numframes   = number of frames in the animation            */
/* numobjs     = number of objects                            */
/* numchannels = number of channels (animation parameters)    */
/**************************************************************/
     int numframes,numobjs,numchannels;
{
  int x,y;

  /* set the global variables */
  Totframes = numframes;
  Totobjs = numobjs;
  Totchannels = numchannels;

  fprintf(stderr,"ANIM:  Creating %d frame(s), %d object(s), %d channels...\n",
	  Totframes,Totobjs,Totchannels);

  /* initialize the Frame List (which contains information about frames) */
  if (initframelist(Totframes) < 0) return(-1);
  /* initialize the Object List (containing information about objects) */
  if (initobjlist(Totobjs) < 0) return(-1);

  /* initialize the Animation Storage (containing channel animation info) */ 
  Anim = Alloc3D(Totframes,Totobjs,Totchannels);
  if (Anim == NULL) return(-1);

  /* initialize channel info to default values */
  for (x=0; x<Totframes; x++)
    for (y=0; y<Totobjs; y++) {
      Anim[x][y][XROT] = 0.0;
      Anim[x][y][YROT] = 0.0;
      Anim[x][y][ZROT] = 0.0;
      Anim[x][y][XTRN] = 0.0;
      Anim[x][y][YTRN] = 0.0;
      Anim[x][y][ZTRN] = 0.0;
      Anim[x][y][SIZE] = 1.0;
      Anim[x][y][QTRC] = 1.0;
      Anim[x][y][QTRX] = 0.0;
      Anim[x][y][QTRY] = 0.0;
      Anim[x][y][QTRZ] = 0.0;
    }

  return(0);
}

initframelist(numframes)
/**************************************************************/
     int numframes;
{
  int i;

  Framelist = (struct Frameinfo *) calloc(numframes,sizeof(struct Frameinfo));
  if (Framelist == NULL) return(-1);

  for (i=0; i<numframes; i++) {
    /* set the keyframe info for the frame */
    Framelist[i].keyframe = 0;
  }

  return(0);
}

initobjlist(numobjs)
/**************************************************************/
     int numobjs;
{
  int i;
  
  Objlist = (struct Objinfo *) calloc(numobjs,sizeof(struct Objinfo));
  if (Objlist == NULL) return(-1);

  for (i=0; i<numobjs; i++) {
    /* set the names for the objects */
    sprintf(Objlist[i].name,"%s-%d","EMPTY",i+1);

    /* set the objects parents */
    strcpy(Objlist[i].parent,"%top");

    /* set the default quaternion rotations for the objects */
    strcpy(Objlist[i].rotation,"quaternion");

    /* set the default color type for the objects */
    strcpy(Objlist[i].colortype,"RGB");
  }

  return(0);
}

Anim_free()
/**************************************************************/
/* Frees the memory associated with the animation database.   */
/**************************************************************/
{
  free(Objlist);
  free(Framelist);
  free(Anim);
}

Anim_addframes(n)
/**************************************************************/
/* add frames to the animation database                       */
/* n = number of frames to add to the database                */
/**************************************************************/
     int n;
{
  float ***temp;
  int x,y,z;

  fprintf(stderr,"ANIM:  Adding %d new frame(s)...\n",n);

  temp = Alloc3D(Totframes+n,Totobjs,Totchannels);
  if (temp == NULL) return(-1);

  for (x=0; x<Totframes; x++)
    for (y=0; y<Totobjs; y++)
      for(z=0; z<Totchannels; z++) temp[x][y][z] = Anim[x][y][z];
  free(Anim);
  Anim = temp;

  for (x=Totframes; x<(Totframes+n); x++)
    for (y=0; y<Totobjs; y++) {
      Anim[x][y][XROT] = 0.0;
      Anim[x][y][YROT] = 0.0;
      Anim[x][y][ZROT] = 0.0;
      Anim[x][y][XTRN] = 0.0;
      Anim[x][y][YTRN] = 0.0;
      Anim[x][y][ZTRN] = 0.0;
      Anim[x][y][SIZE] = 1.0;
      Anim[x][y][QTRC] = 1.0;
      Anim[x][y][QTRX] = 0.0;
      Anim[x][y][QTRY] = 0.0;
      Anim[x][y][QTRZ] = 0.0;
    }
  

  /* note that addframelist depends on Totframes being equal to the */
  /* number of frames BEFORE the new frames were added */
  if (addframelist(n) < 0) return(-1);

  Totframes += n;

  /*
  fprintf(stderr,"done\n");
  */

  return(0);
}

addframelist(newframes)
/**************************************************************/
     int newframes;
{
  int i,j;
  
  j = Totframes + newframes;

  Framelist = (struct Frameinfo *) realloc(Framelist,
					  j*sizeof(struct Frameinfo));
  if (Framelist == NULL) return(-1);

  for (i=Totframes; i<j; i++) {
    /* set the keyframe info for the frame */
    Framelist[i].keyframe = 0;
  }

  return(0);
}

Anim_addobjects(n)
/**************************************************************/
/* add objects to the animation database                      */
/* n = number of objects to add to the database               */
/**************************************************************/
     int n;
{
  float ***temp;
  int x,y,z;

  fprintf(stderr,"ANIM:  Adding %d new object(s)...\n",n);

  temp = Alloc3D(Totframes,Totobjs+n,Totchannels);
  if (temp == NULL) return(-1);

  for (x=0; x<Totframes; x++)
    for (y=0; y<Totobjs; y++)
      for(z=0; z<Totchannels; z++) temp[x][y][z] = Anim[x][y][z];
  free(Anim);
  Anim = temp;

  for (x=0; x<Totframes; x++)
    for (y=Totobjs; y<(Totobjs+n); y++) {
      Anim[x][y][XROT] = 0.0;
      Anim[x][y][YROT] = 0.0;
      Anim[x][y][ZROT] = 0.0;
      Anim[x][y][XTRN] = 0.0;
      Anim[x][y][YTRN] = 0.0;
      Anim[x][y][ZTRN] = 0.0;
      Anim[x][y][SIZE] = 1.0;
      Anim[x][y][QTRC] = 1.0;
      Anim[x][y][QTRX] = 0.0;
      Anim[x][y][QTRY] = 0.0;
      Anim[x][y][QTRZ] = 0.0;
    }
  /* note that addobjlist depends on Totframes being equal to the */
  /* number of frames BEFORE the new frames were added */
  if (addobjlist(n) < 0) return(-1);

  Totobjs += n;

  /*
  fprintf(stderr,"done\n");
  */

  return(0);
}


addobjlist(newobjs)
/**************************************************************/
     int newobjs;
{
  int i,j;
  
  j = Totobjs + newobjs;

  Objlist = (struct Objinfo *) realloc(Objlist,j*sizeof(struct Objinfo));
  if (Objlist == NULL) return(-1);

  for (i=Totobjs; i<j; i++) {
    /* set the names for the objects */
    sprintf(Objlist[i].name,"%s-%d","EMPTY",i+1);

    /* set the objects parents */
    strcpy(Objlist[i].parent,"%top");

    /* set the default quaternion rotations for the objects */
    strcpy(Objlist[i].rotation,"quaternion");

    /* set the default color type for the objects */
    strcpy(Objlist[i].colortype,"RGB");
  }

  return(0);
}

float ***Alloc3D(x,y,z)
/**************************************************************/
     int x,y,z;
{
  float ***arr,*p,**q;
  int i,j;

  /* allocate and return pointer to 3D float data,              */
  /* with header pointers.                                      */
  /* Declare your variable to point to the data with:           */
  /*         float ***arr;                                      */
  /* Call this function with the dimensions of the array:       */
  /*         arr = Alloc3D(10, 10, 10);                         */
  /* Use C array indexing to set and access elements, as in:    */
  /*         arr[xi][yi][zi]                                    */
  /* The data itself starts at address &arr[0][0][0]            */

  arr = (float ***)malloc((unsigned)(x*sizeof(float **)+
					     x*y*sizeof(float *)+
					     x*y*z*sizeof(float)));
  if (arr == NULL) return(arr);

  if (!arr) return(0);
  
  q = (float **)(arr+x);
  p = (float *)(q+x*y);
  for (i=0; i<x; i++) {
    arr[i] = q;
    q += y;
    for (j=0;j<y;j++)	{
      arr[i][j] = p;
      p += z;
    }
  }

  return(arr);
}

/***********************************************
 * FUNCTIONS FOR HANDLING THE KEYFRAME DATABASE *
 ***********************************************/

int NextKey(cframe,end)
     int cframe,end;
/**************************************************************/
/* Does a search for a keyframe FOLLOWING the specified frame */
/* up to the end of the animation window (specified by end).  */
/* Frame end is always considered a keyframe.                 */
/* If end > the number of frames in the animation             */
/* it will be reset to the last frame in the animation.       */
/* Note:  specifying cframe as 0 will return the first        */
/*        keyframe in the animation window.                   */
/* returns -1 if cframe is less than 0 or greater than the    */
/*        end window.                                         */
/* returns the end window if cframe equals the end window.    */
/**************************************************************/
{
  if (end > Totframes) end = Totframes;
  if (cframe < 0 || cframe > end) return(-1);

  cframe++;
  if (cframe > end) return(end);
  
  while(Framelist[cframe-1].keyframe == 0) {
    cframe++;
    if (cframe > end) return(end);
  }

  return(cframe);
}

int PrevKey(start,cframe)
     int start,cframe;
/**************************************************************/
/* Does a backward search for a keyframe PRECEEDING the       */
/* specified frame to the start of the animation window       */
/* (specified by start).  start is always seen as a keyframe. */
/* If start < 1, it will be reset to the first frame          */
/* in the animation.                                          */
/* Note:  specifying cframe as Totframes+1 will return the    */
/*        last keyframe in the animation window.              */
/* returns -1 if cframe is greater than the number of frames  */
/*        in the animation or less than the start window.     */
/* returns the start window if cframe = the start window.     */
/**************************************************************/
{
  if (start < 1) start = 1;
  if (cframe > (Totframes+1) || cframe < start) return(-1);

  cframe--;
  if (cframe < start) return(start);
  
  while(Framelist[cframe-1].keyframe == 0) {
    cframe--;
    if (cframe < start) return(start);
  }

  return(cframe);
}


/*************
 * UTILITIES *
 *************/


HSVtoRGB(H,S,V,R,G,B)
/**************************************************************/
/* Convert Hue-Saturation-Value to Red-Green-Blue             */
/* All values range from 0 to 1                               */
/**************************************************************/
     float H,S,V;
     float *R, *G, *B;
{
  float p1, p2, p3;
  int i;
  float f;

  H = H * 360;
  if (H == 360) H = 0;
  H = H / 60;
  i = H;
  f = H - i;
  p1 = V * (1 - S);
  p2 = V * (1 - (S * f));
  p3 = V * (1 - (S * (1 - f)));
  if (i == 0) { *R = V; *G = p3; *B = p1; }
  else if (i == 1) { *R = p2; *G = V; *B = p1; }
  else if (i == 2) { *R = p1; *G = V; *B = p3; }
  else if (i == 3) { *R = p1; *G = p2; *B = V; }
  else if (i == 4) { *R = p3; *G = p1; *B = V; }
  else { *R = V; *G = p1; *B = p2; }
}    
