/****************************************************************************
                  INTERNATIONAL AVS CENTER
	(This disclaimer must remain at the top of all files)

WARRANTY DISCLAIMER

This module and the files associated with it are distributed free of charge.
It is placed in the public domain and permission is granted for anyone to use,
duplicate, modify, and redistribute it unless otherwise noted.  Some modules
may be copyrighted.  You agree to abide by the conditions also included in
the AVS Licensing Agreement, version 1.0, located in the main module
directory located at the International AVS Center ftp site and to include
the AVS Licensing Agreement when you distribute any files downloaded from 
that site.

The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module provide absolutely
NO WARRANTY OF ANY KIND with respect to this software.  The entire risk as to
the quality and performance of this software is with the user.  IN NO EVENT
WILL The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module BE LIABLE TO
ANYONE FOR ANY DAMAGES ARISING FROM THE USE OF THIS SOFTWARE, INCLUDING,
WITHOUT LIMITATION, DAMAGES RESULTING FROM LOST DATA OR LOST PROFITS, OR ANY
SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES.

This AVS module and associated files are public domain software unless
otherwise noted.  Permission is hereby granted to do whatever you like with
it, subject to the conditions that may exist in copyrighted materials. Should
you wish to make a contribution toward the improvement, modification, or
general performance of this module, please send us your comments:  why you
liked or disliked it, how you use it, and most important, how it helps your
work. We will receive your comments at avs@ncsc.org.

Please send AVS module bug reports to avs@ncsc.org.

******************************************************************************/
#include <stdio.h>
#include <avs/avs.h>
#include <avs/field.h>
#include "voxel.h"

#define IABS(a) ((a) < 0) ? -1*(a) : (a)

static float vw[4]=
{
    1.0,0.6666,0.333,0.0
};

int
setup_cvoxels(vlist,inf,num_voxels)
CVoxel **vlist;
AVSfield *inf;
int *num_voxels;
{
    CVoxel *v=NULL;
    int nvoxels,i,j,k,index=0,xsize,xysize,l;
    int npts;
    int iz,jz,kz;
    float z,*dz;
    int offset[7];
    float z_diffs[7];
    int min_l;
    int order[3];
    int istart,iend,iinc,jstart,jend,jinc,kstart,kend,kinc;

    /* dz is float * to z coords in original data. */
    npts = inf->dimensions[0] * inf->dimensions[1] * inf->dimensions[2];
    dz = inf->points + npts*2;
    xsize = inf->dimensions[0];
    xysize = inf->dimensions[1] * xsize;
    nvoxels = (inf->dimensions[0]-1) * (inf->dimensions[1]-1) * (inf->dimensions[2]-1);
    *vlist = NULL;
    v = (CVoxel *)malloc(sizeof(CVoxel)*nvoxels);
    if (v == NULL)
	return(0);

    index = 0;
    
    for (k=0;k<inf->dimensions[2]-1;k++)
	for (j=0;j<inf->dimensions[1]-1;j++)
	    for (i=0;i<inf->dimensions[0]-1;i++,index++)
		    compute_czmins(v,index,i,j,k,xsize,xysize,dz);
		
    *vlist = v;
    *num_voxels = nvoxels;
    return(1);
}

compute_czmins(v,index,i,j,k,xsize,xysize,dz)
CVoxel *v;
int index,i,j,k,xsize,xysize;
float *dz;
{
    int offset[7];
    float z_diffs[7];
    int min_l;
    int l,base,iz,jz,kz;
    
    offset[0] = 1;
    offset[1] = xsize;
    offset[2] = xsize+1;
    offset[3] = xysize;
    offset[4] = xysize+1;
    offset[5] = xysize+xsize;
    offset[6] = xysize+xsize+1;
    
    base= i+j*xsize+k*xysize;
    v[index].i = i;
    v[index].j = j;
    v[index].k = k;
    for (l=0;l<7;l++)
	z_diffs[l] = dz[base+offset[l]] - dz[base];
    min_l = -1;
    for (l=0;l<7;l++)
	if (z_diffs[l] < 0)
	{
	    if (min_l == -1)
		min_l = l;
	    else if (z_diffs[l] < z_diffs[min_l])
		min_l = l;
	}
    switch(min_l)
    {
    case -1:
	iz = i; jz = j; kz = k;
	v[index].min_z = dz[base];
	break;
    case 0:
	iz = i+1; jz = j; kz = k;
	v[index].min_z = dz[base+offset[min_l]];
	break;
    case 1:
	iz = i; jz = j+1; kz = k;
	v[index].min_z = dz[base+offset[min_l]];
	break;
    case 2 :
	iz = i+1; jz = j+1; kz = k;
	v[index].min_z = dz[base+offset[min_l]];
	break;
    case 3:
	iz = i; jz = j; kz = k+1;
	v[index].min_z = dz[base+offset[min_l]];
	break;
    case 4:
	iz = i+1; jz = j; kz = k+1;
	v[index].min_z = dz[base+offset[min_l]];
	break;
    case 5:
	iz = i; jz = j+1; kz = k+1;
	v[index].min_z = dz[base+offset[min_l]];
	break;
    case 6:
	iz = i+1; jz = j+1; kz = k+1;
	v[index].min_z = dz[base+offset[min_l]];
	break;
    }
    v[index].iz = iz;
    v[index].jz = jz;
    v[index].kz = kz;
}

int
setup_voxels(vlist,inf,number_voxels)
Voxel **vlist;
AVSfield *inf;
int *number_voxels;
{
    int ivoxels,jvoxels,kvoxels,num_voxels;
    int i,j,k,ct=0,iz,jz,kz;
    float barf;
    Voxel *v=NULL;
    
    ivoxels = (inf->dimensions[0]+2)/3 - 1;
    jvoxels = (inf->dimensions[1]+2)/3 - 1;
    kvoxels = (inf->dimensions[2]+2)/3 - 1;
    num_voxels = ivoxels * jvoxels * kvoxels;
    v = (Voxel *)malloc(sizeof(Voxel)*num_voxels);
    if (v==NULL)
	return(0);

    for (k=0;k<kvoxels;k++)
    {
	for (j=0;j<jvoxels;j++)
	{
	    for (i=0;i<ivoxels;i++,ct++)
	    {
		copy_control_points(inf,(v+ct)->x,(v+ct)->y,(v+ct)->z,
				    (v+ct)->d,i,j,k);

		/* find min Z value. */
#ifdef STARDENT
		F_PILE_MIN(&((v+ct)->z[0].m[0][0]),&barf);
#elif SUN
		f_pile_min_(&((v+ct)->z[0].m[0][0]),&barf);
#endif
		(v+ct)->min_z = barf;

		/* now, orient the voxel so that min_z is at
		   u=v=w=0 */

		compute_zmins(v+ct,&iz,&jz,&kz);

		if (iz != 0)
		    flip_about_u(v+ct);

		if (jz != 0)
		    flip_about_v(v+ct);

		if (kz != 0)
		    flip_about_w(v+ct);
		
		/* see which is smaller, the z at v=1 or w=1 */
		if ((v+ct)->z[0].m[0][3] < (v+ct)->z[3].m[0][0])
		{
		    /* the z at v=1 is smaller than the one at w=1. check to
		       see if this one is smaller than the one at u=1.  if so,
		       transpose the voxel about the uv axis. */
		    
		    if ((v+ct)->z[0].m[0][3] < (v+ct)->z[0].m[3][0])
			transpose_uv(v+ct);
		    
		    /* else the z at u=1 is the next smallest (to min_z), and
		       we don't have to do anything. */
		}
		else
		{
		    /* the z at w=1 is smaller than the one at v=1.  check to see
		       if this one is smaller than the one at u=1.  if so, transpose
		       the voxel about the uw axis. */
		    
		    if ((v+ct)->z[3].m[0][0] < (v+ct)->z[0].m[3][0])
			transpose_uw(v+ct);
		    
		    /* else, we want the 2nd smallest zmin at v=1 rather than
		       w=1.  so, transpose the voxel about the vw axis. */
		    else
			transpose_vw(v+ct);
		}
	    }
	}
    }
    *number_voxels = num_voxels;
    *vlist = v;
    return(1);
}

compute_zmins(v,iz,jz,kz)
Voxel *v;
int *iz,*jz,*kz;
{
    double z[8],t;
    int index[8];
    int i,j,no_change,temp;

    for (i=0;i<8;i++)
	index[i] = i;
    
    z[0] = v->z[0].m[0][0];
    z[1] = v->z[0].m[3][0];
    z[2] = v->z[0].m[0][3];
    z[3] = v->z[0].m[3][3];
    z[4] = v->z[3].m[0][0];
    z[5] = v->z[3].m[3][0];
    z[6] = v->z[3].m[0][3];
    z[7] = v->z[3].m[3][3];

    j=8;
    no_change = 1;
    while (no_change == 1) 
    {
	no_change = 0;
	for (i=0;i<j-1;i++)
	{
	    if (z[i] > z[i+1])
	    {
		no_change = 1;
		t = z[i];
		z[i] = z[i+1];
		z[i+1] = t;
		temp = index[i];
		index[i] = index[i+1];
		index[i+1] = temp;
	    }
	}
	j--;
    }
    
    switch (index[0])
    {
    case 0:
	*iz = *jz = *kz = 0;
	break;
    case 1:
        *iz = 1;
	*jz = *kz = 0;
	break;
    case 2:
	*jz = 1;
	*iz = *kz = 0;
	break;
    case 3:
	*iz = *jz = 1;
	*kz = 0;
	break;
    case 4:
	*iz = *jz = 0;
	*kz = 1;
	break;
    case 5:
	*iz = *kz = 1;
	*jz = 0;
	break;
    case 6:
	*jz = *kz = 1;
	*iz = 0;
	break;
    case 7:
	*iz = *jz = *kz = 1;
	break;
    }
}

sort_voxels(vlist,vi,n,main_flag)
Voxel *vlist;
int n,*vi;
int main_flag; /* indicates whether or not to show sort completion status */
{
    /**
      * this routine will bubble sort the list of voxels.  actually,
      * rather than copying the voxels around (expensive), an array
      * containing indeces is sorted.  thus, the array index is
      * sorted, and the voxels are only copied once.
      *
      * note that this routine may be called prior to voxel rendering
      * is started, according to whether or not the user indicates that
      * the voxels should be sorted.  it is also called from within the
      * rendering code so that all subvoxels generated from a voxel
      * subdivision are composited from front-to-back.  for this reason,
      * we add the "main_flag" parameter to indicate whether or not to
      * show the "voxel sort completion status".
    **/
    int no_change;
    int count;
    int i,t;
    
#ifdef AVS3
    float percent=0.,delta;
    int ipercent=0;
    
    delta = 1./(float)n;
    if (main_flag)
	AVSmodule_status("Sorting Voxels",ipercent);
#endif
    
    no_change = 0;
    count = n;
    while (no_change == 0)
    {
	no_change = 1;
	for (i=0;i<count-1;i++)
            if ((vlist+(*(vi+i)))->min_z >
		(vlist+(*(vi+i+1)))->min_z)
	    {
		no_change = 0;
		t = *(vi+i);
		*(vi+i) = *(vi+i+1);
		*(vi+i+1) = t;
	    }
	count--;
#ifdef AVS3
	ipercent = (int)(percent * 100.0);
	percent += delta;
	if (main_flag)
	    AVSmodule_status("Sorting Voxels",ipercent);
#endif
    }
}

sort_cvoxels(vlist,vi,n,main_flag)
CVoxel *vlist;
int n,*vi;
int main_flag; /* indicates whether or not to show sort completion status */
{
    int i,j,h,v;
    int loops;
    
    float percent=0.,delta;
    int ipercent=0;

    for (h=1,loops=0;h<=n;h=3*h+1,loops++)
	;

    /**
      * the following is a "shellsort" algorithm described in
      * "Algorithms" by Sedgewick.
    **/
    
#ifdef AVS3
    
    delta = 1./(float)loops;
    if (main_flag)
	AVSmodule_status("Sorting Voxels",ipercent);
#endif
    


    
    for (h=h/3;;h=h/3)
    {
	for (i=h+1;i<=n;i++)
	{
	    v = vi[i-1];
	    j = i;
	    while ((j >= 1) && ((vlist+vi[j-h-1])->min_z > (vlist+v)->min_z))
	    {
		vi[j-1] = vi[j-h-1];
		j = j - h;
		if (j <= h)
		    break;
	    }
	    vi[j-1] = v;
	}
	if (h == 1)
	    break;
#ifdef AVS3
	ipercent = (int)(percent * 100.0);
	percent += delta;
	if (main_flag)
	    AVSmodule_status("Sorting Voxels",ipercent);
#endif
    }
    
#if 0
    no_change = 0;
    count = n;
    while (no_change == 0)
    {
	no_change = 1;
	for (i=0;i<count-1;i++)
            if ((vlist+vi[i])->min_z >
		(vlist+vi[i+1])->min_z)
	    {
		no_change = 0;
		t = vi[i];
		vi[i] = vi[i+1];
		vi[i+1] = t;
	    }
	count--; 
#ifdef AVS3
	ipercent = (int)(percent * 100.0);
	percent += delta;
	if (main_flag)
	    AVSmodule_status("Sorting Voxels",ipercent);
#endif
    }
#endif
}

sort_trivoxels(tv,ind,n)
trivox *tv;
int *ind;
int n;
{
    int t,no_change,count,i;
    
    no_change = 0;
    count = n;
    while (no_change == 0)
    {
	no_change = 1;
	for (i=0;i<count-1;i++)
            if ((tv+(*(ind+i)))->min_z >
		(tv+(*(ind+i+1)))->min_z)
	    {
		no_change = 0;
		t = *(ind+i);
		*(ind+i) = *(ind+i+1);
		*(ind+i+1) = t;
	    }
	count--;
    }    
}

static  int p1[64][3],p2[64][3],p3[64][3];
static  int p4[64][3],p5[64][3],p6[64][3],p0[64][3];

static  int p21[64][3],p22[64][3],p23[64][3];
static    int pa[64][3],pb[64][3],pc[64][3],pd[64][3],pe[64][3],pf[64][3],
       pg[64][3],ph[64][3];
    
static     int mask[64],mask1[64];
static   float sc[64],s1[64],s2[64],s3[64];

make_tricubic_voxel(tvoxel,cvoxel,inf,np,beta)
Voxel *tvoxel;
CVoxel *cvoxel;
AVSfield *inf;
int np; /* total number of grid points */
float *beta;
{
    /* given some curvilinear voxel (cell bounded by 8 points in
       space), we will compute the additional control points required
       in order to appropriately represent a tricubic solid from
       within the curvilinear space from which the cell comes.
       all of this computation is done using arrays of indeces for speed,
       rather than point-at-a-time (slow). after
       we compute all this stuff, then we will orient the voxel so
       that z_min corresponds to u=v=w=0 in parametric space.
    **/

    int i,xlimit,ylimit,zlimit,j,k,offset,index;
    int xsize,xysize;
    float *xp,*yp,*zp,*dp,*f;

    xlimit = inf->dimensions[0] - 1;
    ylimit = inf->dimensions[1] - 1;
    zlimit = inf->dimensions[2] - 1;

    /* load the index arrays. */

    /* do the x pass first */

    /** the mask is used to cause processing only on a given set
      of points.  the first pass is used in computing only the first
      order partial derivatives.  This corresponds to those interior
      points which are on the boundary edges of the voxel. **/

    for (i=0;i<64;i++)
	mask[i] = 0;

    mask[0] = mask[1] = mask[2] = mask[3] = mask[4] = mask[7] = 1;
    mask[8] = mask[11] = mask[12] = mask[13] = mask[14] = mask[15] = 1;

    mask[16] = mask[19] = mask[28] = mask[31] = 1;

    mask[32] = mask[35] = mask[44] = mask[47] = 1;

    mask[48] = mask[49] = mask[50] = mask[51] = mask[52] = mask[55] = 1;
    mask[56] = mask[59] = mask[60] = mask[61] = mask[61] = mask[62] = mask[63] = 1;
    
    for (i=0;i<64;i++)
    {
	if (mask[i])
	{
	    switch (i)
	    {
	    case 0:
	    case 4:
	    case 8:
	    case 12:
	    case 16:
	    case 20:
	    case 24:
	    case 28:
	    case 32:
	    case 36:
	    case 40:
	    case 44:
	    case 48:
	    case 52:
	    case 56:
	    case 60:
		p1[i][0] = p2[i][0] = p3[i][0] = cvoxel->i;
		break;
	    case 3:
	    case 7:
	    case 11:
	    case 15:
	    case 19:
	    case 23:
	    case 27:
	    case 31:
	    case 35:
	    case 39:
	    case 43:
	    case 47:
	    case 51:
	    case 55:
	    case 59:
	    case 63:
		p1[i][0] = p2[i][0] = p3[i][0] = cvoxel->i + 1;
		break;
	    case 1:
	    case 5:
	    case 9:
	    case 13:
	    case 17:
	    case 21:
	    case 25:
	    case 29:
	    case 33:
	    case 37:
	    case 41:
	    case 45:
	    case 49:
	    case 53:
	    case 57:
	    case 61:
		p2[i][0] = cvoxel->i;
		p3[i][0] = cvoxel->i + 1;
		p1[i][0] = (cvoxel->i == 0) ? 0 : cvoxel->i - 1;
		break;
	    case 2:
	    case 6:
	    case 10:
	    case 14:
	    case 18:
	    case 22:
	    case 26:
	    case 30:
	    case 34:
	    case 38:
	    case 42:
	    case 46:
	    case 50:
	    case 54:
	    case 58:
	    case 62:
		p2[i][0] = cvoxel->i + 1;
		p3[i][0] = cvoxel->i;
		p1[i][0] = (p2[i][0] == xlimit) ? xlimit : p2[i][0] + 1;
		break;
	    default:
		fprintf(stderr," i ordinate case not handled !! %d\n",i);
		break;
	    }
	}
    }

    /* next, do the y pass. */
    for (i=0;i<64;i++)
    {
	if (mask[i])
	{
	    switch (i)
	    {
	    case 0:
	    case 1:
	    case 2:
	    case 3:
	    case 16:
	    case 17:
	    case 18:
	    case 19:
	    case 32:
	    case 33:
	    case 34:
	    case 35:
	    case 48:
	    case 49:
	    case 50:
	    case 51:
		p1[i][1] = p2[i][1] = p3[i][1] = cvoxel->j;
		break;
	    case 12:
	    case 13:
	    case 14:
	    case 15:
	    case 28:
	    case 29:
	    case 30:
	    case 31:
	    case 44:
	    case 45:
	    case 46:
	    case 47:
	    case 60:
	    case 61:
	    case 62:
	    case 63:
		p1[i][1] = p2[i][1] = p3[i][1] = cvoxel->j + 1;
		break;
	    case 4:
	    case 5:
	    case 6:
	    case 7:
	    case 20:
	    case 21:
	    case 22:
	    case 23:
	    case 36:
	    case 37:
	    case 38:
	    case 39:
	    case 52:
	    case 53:
	    case 54:
	    case 55:
		p2[i][1] = cvoxel->j;
		p3[i][1] = cvoxel->j+1;
		p1[i][1] = (cvoxel->j == 0) ? 0 : cvoxel->j - 1;
		break;
	    case 8:
	    case 9:
	    case 10:
	    case 11:
	    case 24:
	    case 25:
	    case 26:
	    case 27:
	    case 40:
	    case 41:
	    case 42:
	    case 43:
	    case 56:
	    case 57:
	    case 58:
	    case 59:
		p2[i][1] = cvoxel->j+1;
		p3[i][1] = cvoxel->j;
		p1[i][1] = (p2[i][1] == ylimit) ? ylimit : p2[i][1] + 1;
		break;

	    default:
		fprintf(stderr," j ordinate case not handled !! %d\n",i);
		break;
	    }
	    
	}
	
    }

    /* finally, do the z pass. */
    for (i=0;i<64;i++)
    {
	if (mask[i])
	{
	    switch (i)
	    {
	    case 0:
	    case 1:
	    case 2:
	    case 3:
	    case 4:
	    case 5:
	    case 6:
	    case 7:
	    case 8:
	    case 9:
	    case 10:
	    case 11:
	    case 12:
	    case 13:
	    case 14:
	    case 15:
		p1[i][2] = p2[i][2] = p3[i][2] = cvoxel->k;
		break;
	    case 48:
	    case 49:
	    case 50:
	    case 51:
	    case 52:
	    case 53:
	    case 54:
	    case 55:
	    case 56:
	    case 57:
	    case 58:
	    case 59:
	    case 60:
	    case 61:
	    case 62:
	    case 63:
		p1[i][2] = p2[i][2] = p3[i][2] = cvoxel->k + 1;
		break;
	    case 16:
	    case 17:
	case 18:
	    case 19:
	    case 20:
	    case 21:
	    case 22:
	    case 23:
	    case 24:
	    case 25:
	    case 26:
	    case 27:
	    case 28:
	    case 29:
	    case 30:
	    case 31:
		p2[i][2] = cvoxel->k;
		p3[i][2] = cvoxel->k+1;
		p1[i][2] = (cvoxel->k == 0) ? 0 : cvoxel->k - 1;
		break;
	    case 32:
	    case 33:
	    case 34:
	    case 35:
	    case 36:
	    case 37:
	    case 38:
	    case 39:
	    case 40:
	    case 41:
	    case 42:
	    case 43:
	    case 44:
	    case 45:
	    case 46:
	    case 47:
		p2[i][2] = cvoxel->k + 1;
		p3[i][2] = cvoxel->k;
		p1[i][2] = (p2[i][2] == zlimit) ? zlimit : p2[i][2] + 1;
		break;
	    default:
		fprintf(stderr," k ordinate case not handled !! %d \n",i);
		break;
	    }
	}
    }

    /* now do the catmull rom for each of the 64 control points. the
       indeces in the p1,p2,p3 arrays indicate the location of the
       points to use. */

    xp = inf->points;
    yp = xp + np;
    zp = yp + np;
    dp = inf->field_union.field_data_float_u;

    xsize = inf->dimensions[0];
    xysize = xsize * inf->dimensions[1];

    /* process x coords first */

    for (i=0;i<64;i++)
    {
	if (mask[i])
	{
	    s2[i] = xp[p2[i][0] + p2[i][1]*xsize + p2[i][2]*xysize];
	    s1[i] = xp[p1[i][0] + p1[i][1]*xsize + p1[i][2]*xysize];
	    s3[i] = xp[p3[i][0] + p3[i][1]*xsize + p3[i][2]*xysize];
	}
    }
    for (i=0;i<64;i++)
	catmull_rom(sc,s1,s2,s3,64,mask,beta);
#if 0
    f = &(tvoxel->x[0].m[0][0]);
    for (i=0;i<64;i++)
	*f++ = sc[i];
#endif
    
    index = 0;
    for (k=0;k<4;k++)
	for (j=0;j<4;j++)
	    for (i=0;i<4;i++)
		tvoxel->x[k].m[i][j] = sc[index++];
    
    /* process y coords */
    for (i=0;i<64;i++)
    {
	if (mask[i])
	{
	    s2[i] = yp[p2[i][0] + p2[i][1]*xsize + p2[i][2]*xysize];
	    s1[i] = yp[p1[i][0] + p1[i][1]*xsize + p1[i][2]*xysize];
	    s3[i] = yp[p3[i][0] + p3[i][1]*xsize + p3[i][2]*xysize];
	}
    }
    for (i=0;i<64;i++)
	catmull_rom(sc,s1,s2,s3,64,mask,beta);

    index = 0;
    for (k=0;k<4;k++)
	for (j=0;j<4;j++)
	    for (i=0;i<4;i++)
		tvoxel->y[k].m[i][j] = sc[index++];
#if 0    
    f = &(tvoxel->y[0].m[0][0]);
    for (i=0;i<64;i++)
	*f++ = sc[i];
#endif    
    /* process z coords */
    for (i=0;i<64;i++)
    {
	if (mask[i])
	{
	    s2[i] = zp[p2[i][0] + p2[i][1]*xsize + p2[i][2]*xysize];
	    s1[i] = zp[p1[i][0] + p1[i][1]*xsize + p1[i][2]*xysize];
	    s3[i] = zp[p3[i][0] + p3[i][1]*xsize + p3[i][2]*xysize];
	}
    }
    for (i=0;i<64;i++)
	catmull_rom(sc,s1,s2,s3,64,mask,beta);

    index = 0;
    for (k=0;k<4;k++)
	for (j=0;j<4;j++)
	    for (i=0;i<4;i++)
		tvoxel->z[k].m[i][j] = sc[index++];
#if 0
    f = &(tvoxel->z[0].m[0][0]);
    for (i=0;i<64;i++)
	*f++ = sc[i];
#endif    

    /* process data */
    for (i=0;i<64;i++)
    {
	if (mask[i])
	{
	    s2[i] = dp[p2[i][0] + p2[i][1]*xsize + p2[i][2]*xysize];
	    s1[i] = dp[p1[i][0] + p1[i][1]*xsize + p1[i][2]*xysize];
	    s3[i] = dp[p3[i][0] + p3[i][1]*xsize + p3[i][2]*xysize];
	}
    }
    for (i=0;i<64;i++)
	catmull_rom(sc,s1,s2,s3,64,mask,beta);

    f = &(tvoxel->d[0].m[0][0]);
    index = 0;
    for (k=0;k<4;k++)
	for (j=0;j<4;j++)
	    for (i=0;i<4;i++)
		tvoxel->d[k].m[i][j] = sc[index++];
/*    
    for (i=0;i<64;i++)
	*f++ = sc[i];
*/

    /** now compute the second order partials.  these points correspond
      to the non-boundary interior points which are on each of the surfaces
      of the voxel.  there are a total of 24 of these. **/

    for (i=0;i<64;i++)
	mask1[i] = 0;

    mask1[5] = mask1[6] = mask1[9] = mask1[10] = 1;
    
    mask1[53] = mask1[54] = mask1[57] = mask1[58] = 1;

    mask1[20] = mask1[24] = mask1[23] = mask1[27] = 1;
    mask1[17] = mask1[18] = mask1[29] = mask1[30] = 1;

    mask1[33] = mask1[34] = mask1[36] = mask1[39] = 1;
    mask1[40] = mask1[43] = mask1[45] = mask1[46] = 1;

    /** the int arrays now have the following meanings:
      p0 gives the indeces in the ordinate or data arrays of
         the source point.
      p0 gives indeces into tvoxel array (4x4x4) for corner points 
      p1,p2 give the indeces into the ordinate or data of the
         already computed first order partials.  these indeces
	 consist of a tuple which ranges from 0..3, which is
	 the extents of the 3d control/data point arrays.
      p3,p4,p5,p6 give the indeces into the cvoxel list of the
         points to use in computing the second order partials. **/

    for (i=0;i<64;i++)
    {
	if (mask1[i])
	{
	    switch (i)
	    {
	    case 5:
	    case 9:
	    case 17:
	    case 29:
	    case 33:

	    case 53:
	    case 57:
		p0[i][0] = 0;
		p1[i][0] = 1;
		p2[i][0] = 0;
		p3[i][0] = cvoxel->i + 1;
		p4[i][0] = (cvoxel->i == 0) ? 0 : cvoxel->i - 1;
		p5[i][0] = p3[i][0];
		p6[i][0] = p4[i][0];
		break;
	    case 45:
		p0[i][0] = 0;
		p1[i][0] = 0;
		p2[i][0] = 1;
		p3[i][0] = cvoxel->i + 1;
		p4[i][0] = (cvoxel->i == 0) ? 0 : cvoxel->i - 1;
		p5[i][0] = p3[i][0];
		p6[i][0] = p4[i][0];
		break;
		
	    case 6:
	    case 10:
	    case 18:
	    case 30:
	    case 34:
	    case 46:
	    case 54:
	    case 58:
		p0[i][0] = 3;
		p1[i][0] = 2;
		p2[i][0] = 3;
		p3[i][0] = p5[i][0] = cvoxel->i;
		p4[i][0] = p6[i][0] = (cvoxel->i +2 >= xlimit) ? xlimit : cvoxel->i + 2;
		break;
	    case 20:
	    case 24:
	    case 36:
	    case 40:
		p0[i][0] = 0;
		p1[i][0] = p2[i][0] = 0;
		p3[i][0] = p4[i][0] = p5[i][0] = p6[i][0] = cvoxel->i;
		break;
	    case 23:
	    case 27:
	    case 39:
	    case 43:
		p0[i][0] = 3;
		p1[i][0] = p2[i][0] = 3;
		p3[i][0] = p4[i][0] = p5[i][0] = p6[i][0] = cvoxel->i + 1;
		break;
	    default:
		break;
	    }
	}
    }
    for (i=0;i<64;i++)
    {
	if (mask1[i])
	{
	    switch (i)
	    {
	    case 5:
	    case 6:
	    case 20:
	    case 23:
	    case 39:
	    case 36:
	    case 53:
	    case 54:
		p0[i][1] = 0;
		p1[i][1] = 0;
		p2[i][1] = 1;
		p3[i][1] = p5[i][1] = cvoxel->j + 1;
		p4[i][1] = p6[i][1] = (cvoxel->j == 0) ? 0 : cvoxel->j - 1;
		break;
	    case 9:
	    case 10:
	    case 24:
	    case 27:
	    case 40:
	    case 43:
	    case 57:
	    case 58:
		p0[i][1] = 3;
		p1[i][1] = 3;
		p2[i][1] = 2;
		p3[i][1] = p5[i][1] = cvoxel->j;
		p4[i][1] = p6[i][1] = (cvoxel->j + 2 >= ylimit) ? ylimit : cvoxel->j + 2;
		break;
	    case 17:
	    case 18:
	    case 33:
	    case 34:
		p0[i][1] = 0;
		p1[i][1] = p2[i][1] = 0;
		p3[i][1] = p4[i][1] = p5[i][1] = p6[i][1] = cvoxel->j;
		break;
	    case 29:
	    case 30:
	    case 45:
	    case 46:
		p0[i][1] = 3;
		p1[i][1] = p2[i][1] = 3;
		p3[i][1] = p4[i][1] = p5[i][1] = p6[i][1] = cvoxel->j + 1;
		break;
	    default:
		break;
	    }
	}
    }
    for (i=0;i<64;i++)
    {
	if (mask1[i])
	{
	    switch (i)
	     {
	     case 5:
	     case 6:
	     case 9:
	     case 10:
		 p0[i][2] = 0;
		 p1[i][2] = 0;
		 p2[i][2] = 0;
		 p3[i][2] = p4[i][2] = p5[i][2] = p6[i][2] = cvoxel->k;
		 break;
	     case 53:
	     case 54:
	     case 57:
	     case 58:
		 p0[i][2] = 3;
		 p1[i][2] = 3;
		 p2[i][2] = 3;
		 p3[i][2] = p4[i][2] = p5[i][2] = p6[i][2] = cvoxel->k + 1;
		 break;
	     case 17:
	     case 18:
	     case 29:
	     case 30:
		 p0[i][2] = 0;
		 p1[i][2] = 0;
		 p2[i][2] = 1;
		 p3[i][2] = p5[i][2] = cvoxel->k + 1;
		 p4[i][2] = p6[i][2] = (cvoxel->k == 0) ? 0 : cvoxel->k - 1;
		 break;
	     case 20:
	     case 23:
	     case 24:
	     case 27:
		 p0[i][2] = 0;
		 p1[i][2] = 1;
		 p2[i][2] = 0;
		 p3[i][2] = p5[i][2] = cvoxel->k + 1;
		 p4[i][2] = p6[i][2] = (cvoxel->k == 0) ? 0 : cvoxel->k - 1;
		 break;
	     case 36:
	     case 39:
	     case 40:
	     case 43:
	     case 45:
		 p0[i][2] = 3;
		 p1[i][2] = 2;
		 p2[i][2] = 3;
		 p3[i][2] = p5[i][2] = cvoxel->k;
		 p4[i][2] = p6[i][2] = (cvoxel->k + 2 >= zlimit) ? zlimit : cvoxel->k + 2;
		 break;
	     case 33:
	     case 34:
	     case 46:
		 p0[i][2] = 3;
		 p1[i][2] = 3;
		 p2[i][2] = 2;
		 p3[i][2] = p5[i][2] = cvoxel->k;
		 p4[i][2] = p6[i][2] = (cvoxel->k + 2 >= zlimit) ? zlimit : cvoxel->k + 2;
		 break;
	     default:
		 break;
	     }
	}
    }

    /** need loops to compute the values for the second order partials. */

    for (i=0;i<64;i++)
	catmull_rom_2partial(p0,p1,p2,p3,p4,p5,p6,64,mask1,tvoxel->x,xp,xsize,xysize,beta);

    for (i=0;i<64;i++)
	catmull_rom_2partial(p0,p1,p2,p3,p4,p5,p6,64,mask1,tvoxel->y,xp,xsize,xysize,beta);

    for (i=0;i<64;i++)
	catmull_rom_2partial(p0,p1,p2,p3,p4,p5,p6,64,mask1,tvoxel->z,zp,xsize,xysize,beta);

    for (i=0;i<64;i++)
	catmull_rom_2partial(p0,p1,p2,p3,p4,p5,p6,64,mask1,tvoxel->d,dp,xsize,xysize,beta);

    /* now, need to compute the eight 3rd order partials. **/

    /** the int arrays now have the following meanings:
      p0 gives the indeces in the ordinate or data arrays of
         the source point.
      p1,p2,p3 give the indeces into the ordinate or data of the
         already computed first order partials.  these indeces
	 consist of a tuple which ranges from 0..3, which is
	 the extents of the 3d control/data point arrays.

      p21,p22,p23 give the indeces of the already computed second
         order partials.

      pa,pb,pc,pd,pe,pf,pg,ph give the indeces into the cvoxel list
         of the points used in computing the third order partials.
	 
      **/

    for (i=0;i<64;i++)
	mask[i] = 0;

    mask[21] = mask[22] = mask[25] = mask[26] = 1;
    mask[37] = mask[38] = mask[41] = mask[42] = 1;

    for (i=0;i<64;i++)
    {
	if (mask[i])
	{
	    switch (i)
	    {
	    case 21:
	    case 25:
	    case 37:
	    case 41:
		p0[i][0] = 0;
		p1[i][0] = 1;
		p2[i][0] = 0;
		p3[i][0] = 0;
		p21[i][0] = 1;
		p22[i][0] = 0;
		p23[i][0] = 1;
		pa[i][0] = pc[i][0] = pe[i][0] = pg[i][0] = cvoxel->i + 1;
		pb[i][0] = pd[i][0] = pf[i][0] = ph[i][0] = (cvoxel->i == 0) ? 0 : cvoxel->i - 1;
		break;
	    case 22:
	    case 26:
	    case 38:
	    case 42:
		p0[i][0] = 3;
		p1[i][0] = 2;
		p2[i][0] = 3;
		p3[i][0] = 3;
		p21[i][0] = 2;
		p22[i][0] = 3;
		p23[i][0] = 2;
		pa[i][0] = pc[i][0] = pe[i][0] = pg[i][0] = cvoxel->i;
		pb[i][0] = pd[i][0] = pf[i][0] = ph[i][0] = (cvoxel->i == xlimit) ? xlimit : (cvoxel->i + 1 == xlimit) ? xlimit : cvoxel->i + 2;
		break;
	    default:
		break;
	    }
	}
    }
    for (i=0;i<64;i++)
    {
	if (mask[i])
	{
	    switch (i)
	    {
	    case 21:
	    case 22:
	    case 37:
	    case 38:
		p0[i][1] = 0;
		p1[i][1] = 0;
		p2[i][1] = 1;
		p3[i][1] = 0;
		p21[i][1] = 1;
		p22[i][1] = 1;
		p23[i][1] = 0;
		pa[i][1] = pb[i][1] = pe[i][1] = pf[i][1] = cvoxel->j+1;
		pc[i][1] = pd[i][1] = pg[i][1] = ph[i][1] = (cvoxel->j == 0) ? 0 : cvoxel->j - 1;
		break;
	    case 25:
	    case 26:
	    case 41:
	    case 42:
		p0[i][1] = 3;
		p1[i][1] = 3;
		p2[i][1] = 2;
		p3[i][1] = 3;
		p21[i][1] = 2;
		p22[i][1] = 2;
		p23[i][1] = 3;
		pa[i][1] = pb[i][1] = pe[i][1] = pf[i][1] = cvoxel->j;
		pc[i][1] = pd[i][1] = pg[i][1] = ph[i][1] = (cvoxel->j == ylimit) ? ylimit : (cvoxel->j+1 == ylimit) ?  ylimit : cvoxel->j + 2;
		break;
	    default:
		break;
	    }
	}
    }
    for (i=0;i<64;i++)
    {
	if (mask[i])
	{
	    switch (i)
	    {
	    case 21:
	    case 25:
	    case 22:
	    case 26:
		p0[i][2] = 0;
		p1[i][2] = 0;
		p2[i][2] = 0;
		p3[i][2] = 1;
		p21[i][2] = 0;
		p22[i][2] = 1;
		p23[i][2] = 1;
		pa[i][2] = pb[i][2] = pc[i][2] = pd[i][2] = cvoxel->k + 1;
		pe[i][2] = pf[i][2] = pg[i][2] = ph[i][2] = (cvoxel->k == 0) ? 0 : cvoxel->k - 1;
		break;
	    case 37:
	    case 38:
	    case 41:
	    case 42:
		p0[i][2] = 3;
		p1[i][2] = 3;
		p2[i][2] = 3;
		p3[i][2] = 2;
		p21[i][2] = 3;
		p22[i][2] = 2;
		p23[i][2] = 2;
		pa[i][2] = pb[i][2] = pc[i][2] = pd[i][2] = cvoxel->k;
		pe[i][2] = pf[i][2] = pg[i][2] = ph[i][2] = (cvoxel->k == zlimit) ? zlimit : (cvoxel->k + 1 == zlimit) ? zlimit : cvoxel->k + 2;
		break;
	    default:
		break;
	    }
	}
    }

    for (i=0;i<64;i++)
    {
	if (mask[i])
	    catmull_rom_3partial(p0,p1,p2,p3,p21,p22,p23,pa,pb,pc,pd,
				 pe,pf,pg,ph,tvoxel->x,xp,xsize,xysize,i,beta);
    }
    for (i=0;i<64;i++)
    {
	if (mask[i])
	    catmull_rom_3partial(p0,p1,p2,p3,p21,p22,p23,pa,pb,pc,pd,
				 pe,pf,pg,ph,tvoxel->y,yp,xsize,xysize,i,beta);
    }
    for (i=0;i<64;i++)
    {
	if (mask[i])
	    catmull_rom_3partial(p0,p1,p2,p3,p21,p22,p23,pa,pb,pc,pd,
				 pe,pf,pg,ph,tvoxel->z,zp,xsize,xysize,i,beta);
    }
    for (i=0;i<64;i++)
    {
	if (mask[i])
	    catmull_rom_3partial(p0,p1,p2,p3,p21,p22,p23,pa,pb,pc,pd,
				 pe,pf,pg,ph,tvoxel->d,dp,xsize,xysize,i,beta);
    }

#if 0
    index = cvoxel->i + cvoxel->j*xsize + cvoxel->k*xysize;
    
    f = &(tvoxel->d[0].m[0][0]);
    for (k=0;k<4;k++)
	for (j=0;j<4;j++)
	    for (i=0;i<4;i++)
	    {
		tvoxel->d[k].m[i][j] = dp[index]*vw[i]*vw[j]*vw[k] +
		    dp[index+1]*vw[3-i]*vw[j]*vw[k] +
			
		    dp[index+xsize]*vw[i]*vw[3-j]*vw[k] +
		    dp[index+xsize+1]*vw[3-i]*vw[3-j]*vw[k] +
			
		    dp[index+xysize]*vw[i]*vw[j]*vw[3-k] +
		    dp[index+xysize+1]*vw[3-i]*vw[j]*vw[3-k] +
		
		    dp[index+xysize+xsize]*vw[i]*vw[3-j]*vw[3-k] +
		    dp[index+xysize+xsize+1]*vw[3-i]*vw[3-j]*vw[3-k];
	    }
#endif
    /* now, orient the tvoxel so that min_z is at u=v=w=0 */

    if (cvoxel->i != cvoxel->iz)
	flip_about_u(tvoxel);

    if (cvoxel->j != cvoxel->jz)
	flip_about_v(tvoxel);

    if (cvoxel->k != cvoxel->kz)
	flip_about_w(tvoxel);

    /* compute min/max extents. */

#ifdef SUN
    f_pile_min_(tvoxel->x,&(tvoxel->min_extent[0]));
    f_pile_min_(tvoxel->y,&(tvoxel->min_extent[1]));
    f_pile_min_(tvoxel->z,&(tvoxel->min_extent[2]));
    f_pile_max_(tvoxel->x,&(tvoxel->max_extent[0]));
    f_pile_max_(tvoxel->y,&(tvoxel->max_extent[1]));
    f_pile_max_(tvoxel->z,&(tvoxel->max_extent[2]));
#elif STARDENT
    F_PILE_MIN(tvoxel->x,&(tvoxel->min_extent[0]));
    F_PILE_MIN(tvoxel->y,&(tvoxel->min_extent[1]));
    F_PILE_MIN(tvoxel->z,&(tvoxel->min_extent[2]));
    F_PILE_MAX(tvoxel->x,&(tvoxel->max_extent[0]));
    F_PILE_MAX(tvoxel->y,&(tvoxel->max_extent[1]));
    F_PILE_MAX(tvoxel->z,&(tvoxel->max_extent[2]));
#endif

    /* now, we want to orient the voxel to that the next
       smallest z is at u=1. */

    /* see which is smaller, the z at v=1 or w=1 */
    if (tvoxel->z[0].m[0][3] < tvoxel->z[3].m[0][0])
    {
	/* the z at v=1 is smaller than the one at w=1. check to
	   see if this one is smaller than the one at u=1.  if so,
	   transpose the voxel about the uv axis. */
	if (tvoxel->z[0].m[0][3] < tvoxel->z[0].m[3][0])
	    transpose_uv(tvoxel);
	/* else the z at u=1 is the next smallest (to min_z), and
	   we don't have to do anything. */
	
    }
    else
    {
	/* the z at w=1 is smaller than the one at v=1.  check to see
	   if this one is smaller than the one at u=1.  if so, transpose
	   the voxel about the uw axis. */
	if (tvoxel->z[3].m[0][0] < tvoxel->z[0].m[3][0])
	    transpose_uw(tvoxel);
	/* else, we want the 2nd smallest zmin at v=1 rather than
	   w=1.  so, transpose the voxel about the vw axis. */
	else
	    transpose_vw(tvoxel);
    }

    tvoxel->min_z = cvoxel->min_z;
}

transpose_uv(v)
Voxel *v;
{
    Voxel t;
    int i,j,k;
    
    memcpy((char *)&t,(char *)v,sizeof(Voxel));

    for (k=0;k<4;k++)
	for (j=0;j<4;j++)
	    for (i=0;i<4;i++)
	    {
		v->x[k].m[i][j] = t.x[k].m[j][i];
		v->y[k].m[i][j] = t.y[k].m[j][i];
		v->z[k].m[i][j] = t.z[k].m[j][i];
		v->d[k].m[i][j] = t.d[k].m[j][i];
	    }

}

transpose_uw(v)
Voxel *v;
{
    Voxel t;
    int i,j,k;
    
    memcpy((char *)&t,(char *)v,sizeof(Voxel));

    for (k=0;k<4;k++)
	for (j=0;j<4;j++)
	    for (i=0;i<4;i++)
	    {
		v->x[k].m[i][j] = t.x[i].m[k][j];
		v->y[k].m[i][j] = t.y[i].m[k][j];
		v->z[k].m[i][j] = t.z[i].m[k][j];
		v->d[k].m[i][j] = t.d[i].m[k][j];
	    }

}

transpose_vw(v)
Voxel *v;
{
    Voxel t;
    int i,j,k;
    
    memcpy((char *)&t,(char *)v,sizeof(Voxel));

    for (k=0;k<4;k++)
	for (j=0;j<4;j++)
	    for (i=0;i<4;i++)
	    {
		v->x[k].m[i][j] = t.x[j].m[i][k];
		v->y[k].m[i][j] = t.y[j].m[i][k];
		v->z[k].m[i][j] = t.z[j].m[i][k];
		v->d[k].m[i][j] = t.d[j].m[i][k];
	    }

}

flip_about_u(v)
Voxel *v;
{
    Voxel t;
    int i,j,k;

    memcpy((char *)&t,(char *)v,sizeof(Voxel));
	   
    for (k=0;k<4;k++)
	for (j=0;j<4;j++)
	    for (i=0;i<4;i++)
	    {
		v->x[k].m[i][j] = t.x[k].m[3-i][j];
		v->y[k].m[i][j] = t.y[k].m[3-i][j];
		v->z[k].m[i][j] = t.z[k].m[3-i][j];
		v->d[k].m[i][j] = t.d[k].m[3-i][j];
	    }
}

flip_about_v(v)
Voxel *v;
{
    Voxel t;
    int i,j,k;

    memcpy((char *)&t,(char *)v,sizeof(Voxel));
	   
    for (k=0;k<4;k++)
	for (j=0;j<4;j++)
	    for (i=0;i<4;i++)
	    {
		v->x[k].m[i][j] = t.x[k].m[i][3-j];
		v->y[k].m[i][j] = t.y[k].m[i][3-j];
		v->z[k].m[i][j] = t.z[k].m[i][3-j];
		v->d[k].m[i][j] = t.d[k].m[i][3-j];
	    }
}

flip_about_w(v)
Voxel *v;
{
    Voxel t;
    int i,j,k;

    memcpy((char *)&t,(char *)v,sizeof(Voxel));
	   
    for (k=0;k<4;k++)
	for (j=0;j<4;j++)
	    for (i=0;i<4;i++)
	    {
		v->x[k].m[i][j] = t.x[3-k].m[i][j];
		v->y[k].m[i][j] = t.y[3-k].m[i][j];
		v->z[k].m[i][j] = t.z[3-k].m[i][j];
		v->d[k].m[i][j] = t.d[3-k].m[i][j];
	    }
}

compute_order(inf,order,istart,iinc,iend,jstart,jinc,jend,
	      kstart,kinc,kend)
AVSfield *inf;
int *order;
int *istart,*jstart,*kstart,*iinc,*jinc,*kinc,*iend,*jend,*kend;
{
    /**
      * the purpose of this routine is to determine the ordering
      * of the i,j,k indeces in how the curvilinear field will
      * be processed.  the idea is that by intelligently processing
      * the voxels, initially, that the cost of sorting them will
      * be reduced.  this is the case because a bubble sort is used,
      * and this algorithm works well on "nearly sorted" lists.
      *
      * the assumption used below is that the volume is more or
      * less "regular" to begin with -- somewhat of a contradiction
      * to the premise that this code is used to render curvilinear
      * volumes.  however, if the volume is so irregular that the
      * assumption doesn't hold, then trying to determine an
      * appropriate order is pretty useless.
      *
    **/
    
    double z[8],t;
    int index[8];
    int i,j,no_change,temp;
    int xsize,xysize,npts;
    int ustart,uend,vstart,vend,wstart,wend;
    int b1,b2;
    float *dz;
    static int bitcount();

    xsize = inf->dimensions[0];
    xysize = inf->dimensions[1] * xsize;
    npts = inf->dimensions[2] * xysize;

    dz = inf->points + 2*npts;  /* pointer to array of z-coords */

    for (i=0;i<8;i++)
	index[i] = i;
    
    z[0] = dz[0];
    z[1] = dz[xsize];
    z[2] = dz[xysize];
    z[3] = dz[xysize+xsize];
    z[4] = dz[npts - xysize - 1];
    z[5] = dz[npts - xysize + xsize - 1];
    z[6] = dz[npts - xsize - 1];
    z[7] = dz[npts - 1];

    j=8;
    no_change = 1;
    while (no_change == 1) 
    {
	no_change = 0;
	for (i=0;i<j-1;i++)
	{
	    if (z[i] > z[i+1])
	    {
		no_change = 1;
		t = z[i];
		z[i] = z[i+1];
		z[i+1] = t;
		temp = index[i];
		index[i] = index[i+1];
		index[i+1] = temp;
	    }
	}
	j--;
    }

    ustart = vstart = wstart = index[0];  /* found the origin */

/*    b1 = bitcount(ustart); */
    b1 = index[0];
    i = 1;
    for (;i<8;i++)
    {
	b2 = index[i];
	temp = bitcount(b1 ^ b2);
        if (temp == 1)
	{
	    uend = index[i];
	    i++;
	    break;
	}
    }
    if (i==8) /* bad news. */
    {
	fprintf(stderr," can't find uend.\n");
    }

    for (;i<8;i++)
    {
	b2 = index[i];
	temp = bitcount(b1 ^ b2);
/*	if (IABS(temp) == 1) /* positions differ by only one bit */
        if (temp == 1)
	{
	    vend = index[i];
	    i++;
	    break;
	}
    }
    if (i==8) /* bad news. */
    {
	fprintf(stderr," can't find vend. \n");
    }
    
    for (;i<8;i++)
    {
	b2 = index[i];
	temp = bitcount(b1^b2);
/*	if (IABS(temp) == 1) /* positions differ by only one bit */
        if (temp == 1)
	{
	    wend = index[i];
	    i++;
	    break;
	}
    }
    if (i==8) /* bad news. */
    {
	fprintf(stderr," can't find wend. \n");
    }

    switch (ustart ^ uend)
    {
    case 01:
	if (ustart & 01)
	{
	    *istart = inf->dimensions[0]-2;
	    *iinc = -1;
	    *iend = 0;
	}
	else
	{
	    *istart = 0;
	    *iinc = 1;
	    *iend = inf->dimensions[0]-2;
	}
	order[0] = 0;
	break;
    case 02:
	if (ustart & 02)
	{
	    *jstart = inf->dimensions[1]-2;
	    *jinc = -1;
	    *jend = 0;
	}
	else
	{
	    *jstart = 0;
	    *jinc = 1;
	    *jend = inf->dimensions[1]-2;
	}
	order[0] = 1;
	break;
    case 04:
	if (ustart & 04)
	{
	    *kstart = inf->dimensions[2]-2;
	    *kinc = -1;
	    *kend = 0;
	}
	else
	{
	    *kstart = 0;
	    *kinc = 1;
	    *kend = inf->dimensions[2]-2;
	}
	order[0] = 2;
	break;
    }
    switch (vstart ^ vend)
    {
    case 01:
	if (vstart & 01)
	{
	    *istart = inf->dimensions[0]-2;
	    *iinc = -1;
	    *iend = 0;
	}
	else
	{
	    *istart = 0;
	    *iinc = 1;
	    *iend = inf->dimensions[0]-2;
	}
	order[1] = 0;
	break;
    case 02:
	if (vstart & 02)
	{
	    *jstart = inf->dimensions[1]-2;
	    *jinc = -1;
	    *jend = 0;
	}
	else
	{
	    *jstart = 0;
	    *jinc = 1;
	    *jend = inf->dimensions[1]-2;
	}
	order[1] = 1;
	break;
    case 04:
	if (vstart & 04)
	{
	    *kstart = inf->dimensions[2]-2;
	    *kinc = -1;
	    *kend = 0;
	}
	else
	{
	    *kstart = 0;
	    *kinc = 1;
	    *kend = inf->dimensions[2]-2;
	}
	order[1] = 2;
	break;
    }
    switch(wend ^ wstart)
    {
    case 01:
	if (wstart & 01)
	{
	    *istart = inf->dimensions[0]-2;
	    *iinc = -1;
	    *iend = 0;
	}
	else
	{
	    *istart = 0;
	    *iinc = 1;
	    *iend = inf->dimensions[0]-2;
	}
	order[2] = 0;
	break;
    case 02:
	if (wstart & 02)
	{
	    *jstart = inf->dimensions[1]-2;
	    *jinc = -1;
	    *jend = 0;
	}
	else
	{
	    *jstart = 0;
	    *jinc = 1;
	    *jend = inf->dimensions[1]-2;
	}
	order[2] = 1;
	break;
    case 04:
	if (wstart & 04)
	{
	    *kstart = inf->dimensions[2]-2;
	    *kinc = -1;
	    *kend = 0;
	}
	else
	{
	    *kstart = 0;
	    *kinc = 1;
	    *kend = inf->dimensions[2]-2;
	}
	order[2] = 2;
	break;
    }
}

static int
bitcount(n)
unsigned int n;
{
    int i,b;
    for (b=0,i=0;i<8;i++)
    {
	if (n & 01)
	    b++;
	n = (n >> 1);
    }
    return(b);
}
