/****************************************************************************
                  INTERNATIONAL AVS CENTER
	(This disclaimer must remain at the top of all files)

WARRANTY DISCLAIMER

This module and the files associated with it are distributed free of charge.
It is placed in the public domain and permission is granted for anyone to use,
duplicate, modify, and redistribute it unless otherwise noted.  Some modules
may be copyrighted.  You agree to abide by the conditions also included in
the AVS Licensing Agreement, version 1.0, located in the main module
directory located at the International AVS Center ftp site and to include
the AVS Licensing Agreement when you distribute any files downloaded from 
that site.

The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module provide absolutely
NO WARRANTY OF ANY KIND with respect to this software.  The entire risk as to
the quality and performance of this software is with the user.  IN NO EVENT
WILL The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module BE LIABLE TO
ANYONE FOR ANY DAMAGES ARISING FROM THE USE OF THIS SOFTWARE, INCLUDING,
WITHOUT LIMITATION, DAMAGES RESULTING FROM LOST DATA OR LOST PROFITS, OR ANY
SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES.

This AVS module and associated files are public domain software unless
otherwise noted.  Permission is hereby granted to do whatever you like with
it, subject to the conditions that may exist in copyrighted materials. Should
you wish to make a contribution toward the improvement, modification, or
general performance of this module, please send us your comments:  why you
liked or disliked it, how you use it, and most important, how it helps your
work. We will receive your comments at avs@ncsc.org.

Please send AVS module bug reports to avs@ncsc.org.

******************************************************************************/
/*
 * Marker_Pick - Pick, Edit, Create, and Move Markers on Images
 *
 *    ======================================================
 *         A Test Bed for marker picking applications
 *               with the AVS5 Image Viewer
 *    ======================================================
 *
 *
 * Type: Mapper, Image viewer upstream control module
 *
 * Language:  K&R C
 *
 * Author:    Ian Curington, AVS/UNIRAS Ltd. UK
 *
 * Description:
 *
 * Allow marker pick identification and display 
 * from an input marker list and the image viewer
 * system. Takes as input an AVS field with
 * a list of coordinates for markers, displays them
 * over an image (map) in the image viewer, and
 * reacts to mouse picks on these markers, sending
 * the pick selection to the output port.
 * Markers are defined by geometric polyline vector lists,
 * not ROI pixel masks.
 *
 * Markers can be created by picking and dragging, where
 * the distance of the drag determines marker size,
 * centered on the initial pick.
 *
 * Undo, Clear, options allow list editing.
 * A series of predefined shapes are available from a choice list.
 *
 * The input marker field is fully specified.
 * You can externally control position, shape, size, etc.
 * Any marker field sent out on the output port can
 * be read back in to the input field port. This way
 * state save and restore are possible. New marker
 * lists can be made using the "marker_pick" module.
 *
 * Inputs:
 *   Input Image: Reference image, same as that sent for
 *                display in the image viewer, used
 *                for feedback path.
 *   Mouse Info:  Part of upstream pick feedback communication
 *                with the image viewer
 *   IV Struct:   Part of upstream communications with the image viewer.
 *   Poly_Marker: Input field,
 *                  for marker pick list definition.
 *                This field is 1D uniform integer N-vector.
 *                The marker field is assumed 
 *                to be a full marker spec as defined
 *                in "marker.h", usually 7. This can fully specifiy
 *                variable marker shape, size, position, and hidden
 *                stroke vector used in the shape creation.
 *
 * Outputs:
 *   Poly_Marker_Output:
 *                This contains a full marker list, as a mirror to
 *                to the input list. It is always full 7-vector.
 *                This port can be saved to a file with the
 *                "write field" module, and later restored with
 *                "read_field" on the input port during a later session.
 *
 *  IV Draw Struct:
 *                Part of the image viewer upstream communication
 *                system.
 *
 * Parameters:
 *  Set Pick Mode: A one-shot button that registers the pick operation
 *                mode with the image viewer, in the case that multiple
 *                modules are also requesting pick communication with the
 *                image viewer.
 *
 *  Pick Status Text Block:
 *                This is a square read-only text block used to
 *                show annotation about the picked marker.
 *
 *  Clear List    Oneshot to clear out the marker list.
 *
 *  Undo Last     backup the list one entry.
 *
 * -----------------------------------------------
 * Adapted from ip_read_line,
 * by Ian Curington
 * -----------------------------------------------
 * Revision: 
 * 18 March 93 ianc - Original (Ensign Demo)
 * 22 March 93 ianc - inverted output port Y
 *                  - fixed buffer update bug
 *  7 Feb   94 ianc - adapted from Polyline
 *  8 Feb   94 ianc - updated output firing guard, fill style
 * 20 June  94 ianc - update of search pick algorithm
 */

/*
                  Copyright (c) 1992 by
                  Advanced Visual Systems Inc.
                  All Rights Reserved
      
      This software comprises unpublished confidential information of
      Advanced Visual Systems Inc. and may not be used, copied or made
      available to anyone, except in accordance with the license
      under which it is furnished.
      
*/

/**********************************************************/
#include <stdio.h>
/* IAC CODE CHANGE : #include <math.h> */
#include <avs/avs_math.h>

#include <avs/avs.h>
#include <avs/field.h>
#include <avs/udata.h>
#include <avs/image_draw.h>
#include <avs/image_upstrm.h>

#include "marker.h"

#define BUFSIZE 1024
#define CLI_BUFFER_SIZE BUFSIZE*32
#define MAX_POLYLINE 500
#define CMD_STR \
 "%s -scene_id %d -image_name \"%s\" -module $Module %f %f %f %f"


/**********************************************************/

typedef struct
{
  char buffer[CLI_BUFFER_SIZE], *bptr;
} CLIbuffer;

static CLIbuffer commandBuffer;
static int       bufferInited = 0;
static char      imageName[80];

/**********************************************************/
/* Module Description Section */
/**********************************************************/
marker_pick_desc()
{
  int port, param;
  char spath[128];
  extern char *AVSpath;
  int marker_pick_compute();
    
  AVSset_module_name("Marker Pick", MODULE_MAPPER);

  /* can't be REENTRANT? (compute function has statics) */
  AVSset_module_flags(COOPERATIVE);

  AVScreate_input_port("input", "field 2D", REQUIRED);

  port = AVScreate_input_port("Image Viewer Id", "struct image_viewer_id",
                        INVISIBLE | OPTIONAL);
  AVSset_input_class(port, "Image Draw:image_viewer_id");

  port = AVScreate_input_port("Mouse Info","struct mouse_info",
                        INVISIBLE | OPTIONAL);
  AVSset_input_class(port, "Image Draw:mouse_info");

  AVScreate_input_port("Polyline Input", MARKER_FIELD_SPEC, OPTIONAL);

  AVScreate_output_port("Polyline Output", MARKER_FIELD_SPEC);

  AVScreate_output_port("Image Draw", "struct image_draw");

  param = AVSadd_parameter("set pick mode", "oneshot", 0, 0, 1);
  AVSconnect_widget(param, "oneshot");

  param = AVSadd_parameter("clear marker_pick", "oneshot", 0, 0, 1);
  AVSconnect_widget(param, "oneshot");

  param = AVSadd_parameter("undo last pick", "oneshot", 0, 0, 1);
  AVSconnect_widget(param, "oneshot");

  param = AVSadd_parameter("marker_type", "choice",
          "Square",
          "Square#Diamond#Circle#X#Fleur de Lis#Line","#");

  param = AVSadd_parameter("marker_fill", "choice",
          "Filled",
          "Filled#Outlined","#");

   param=
    AVSadd_parameter("Selected Marker","string_block",
                     " === Selected Marker Information ===\n","","");
    AVSadd_parameter_prop(param,"height","integer",8);
    AVSadd_parameter_prop(param,"width","integer",4);

  AVSset_compute_proc(marker_pick_compute);

  /* collect runtime configurable data type */
  if (AVSpath != NULL)
    strcpy(spath, AVSpath);
  else
    strcpy(spath, "/usr/avs");
  strcat(spath, "/include/image_draw.h");

  AVSload_user_data_types(spath);
}

/**********************************************************/
/* Module Computational Section */
/**********************************************************/
int marker_pick_compute(input,
                     iv_id,
                     mouseInfo,
                     poly_in,
                     poly,
                     im_draw,
                     mode,
                     clear,
                     undo,
                     c_marker_type,
                     c_marker_fill,
                     markerInfo )


    AVSfield        *input;      /* input image */
    image_viewer_id *iv_id;      /* image viewer feedback */
    mouse_info      *mouseInfo;  /* mouse status feedback */
    AVSfield_int    *poly_in;    /* input marker_pick field */
    AVSfield_int    **poly;      /* output marker_pick field */
    image_draw      **im_draw;   /* output draw commands (unused) */
    int             mode;        /* set mouse active mode for this module */
    int             clear;       /* clear line list out */
    int             undo;        /* undo the last pick */
    char            *c_marker_type;  /* choice list of markers */
    char            *c_marker_fill;  /* choice list of markers */
    char            *markerInfo;     /* marker information */
{
  char      command[BUFSIZE], label[256];
  CLIbuffer *clibuf;
  int       i, j, data_type, mask, end, npixels, status;
  float     length, min, max;
  char      *in, *out, *line_buffer;
  short     *in_short, *out_short;
  float     *in_float, *out_float;
  int       dims[1];
  int       marker_type;
  int       marker_fill;
  int       x, y, sz;
  int       search_dist[ MAX_POLYLINE ];
  int       nearest;

  /* statics for remembering last line, and marker_pick */
  static int       begin = 1;
  static int       startX, startY, endX, endY;
  static int       marker_pick_buf[MARKER_VECLEN][MAX_POLYLINE];
  static int       num_marker_picks;
  static int       picked_marker;

  /*
   * Quick Get Out if only the image input port has changed
   */
  if ( AVSinput_changed("input", 0))
  {
    AVSmark_output_unchanged("Polyline Output");
    return(1);
  }

  /*
   * Resolve string choice param
   */
  marker_type = AVSchoice_number("marker_type",c_marker_type);
  marker_fill = AVSchoice_number("marker_fill",c_marker_fill);

  /*
   * Build Output drawing command list
   * not used, but causes upstream events to be registered.
   */
  if (*im_draw == NULL)
    *im_draw = (image_draw *) AVSdata_alloc("struct image_draw", 0);

  if (input->uniform != UNIFORM)
  {
    AVSerror("Incompatible data type: can only process 2D uniform fields");
    return(0);
  }

  /*
   * Initialize CLI command buffer
   * requests are blocked up for efficiency
   */
  clibuf = &commandBuffer;

  if (!bufferInited)
  {
    clibuf->bptr = clibuf->buffer;
    bufferInited = 1;
  }

  /*
   * This module will do nothing unless
   * it has upstream feedback
   */
  if (iv_id == NULL)
  {
    /* not connected to image viewer */
    imageName[0] = '\0';
    AVSmark_output_unchanged("Polyline Output");
    return(1);
  }

  /*
   * Pre-initialize marker buffer if Optional input field
   */
  if ( poly_in && AVSinput_changed("Polyline Input",0))
  {
     for (i=0; i<MAXX(poly_in); i++)
     for (j=0; j<MARKER_VECLEN; j++)
     {
          marker_pick_buf[j][i] = poly_in->data[i*MARKER_VECLEN+j];
     }
     num_marker_picks = MAXX(poly_in);
     fill_field ( poly, marker_pick_buf, num_marker_picks, MAXY(input) );
     return(1);
  }


  /*
   * check buffer size
   */
  if ( num_marker_picks+1 >= MAX_POLYLINE )
  {
      AVSerror("marker_pick buffer is full, no more room.");
      return(0);
  }

  /*
   * clear all previous lines from image,
   * erase buffer
   */
  if (clear)
  {  
    num_marker_picks = 0;
    fill_field ( poly, marker_pick_buf, num_marker_picks, MAXY(input) );
    return(1);
  }

  /*
   * Undo the last pick, back up the line 
   */
  if ( undo && num_marker_picks > 0 )
  {
     /* remove one from the list */
     num_marker_picks--;

     /* send the revised marker_pick list out */
     dims[0] = num_marker_picks;
     fill_field ( poly, marker_pick_buf, num_marker_picks, MAXY(input) );
     return(1);
  }

  if (AVSinput_changed("Image Viewer Id", 0))
  {
    sprintf(command,
    "image_wakeup_when_defined -scene_id %d -mesh_id %d -module \"$Module\"",
       iv_id->scene_id, input->mesh_id);
    add_command(command, clibuf, 1);
    /**
    num_marker_picks = 0;
    **/

    /* don't fire field port for new ref image */
    AVSmark_output_unchanged("Polyline Output");
  }

  if (mouseInfo && AVSinput_changed("Mouse Info", 0))
  {
    if (mouseInfo->nEvents == 0)
    {
      if (imageName[0] != '\0')
      {
          /* a new image - stop events for old */
          sprintf(command,
      "%s %s %d %s \"%s\" -module $Module -func_id \"%s\"",
              "image_stop_events",
              "-scene_id",
              iv_id->scene_id,
              "-image_name",
              imageName,
              "marker_pick");
          add_command(command, clibuf, 1);
          num_marker_picks = 0;
      }

      /*
       * request pick events from a new image
       * register request for events with the image viewer
       */
      strcpy(imageName, mouseInfo->image_name);
      mask = IMAGE_Button1Mask | IMAGE_PressMask | IMAGE_ReleaseMask |
                                 IMAGE_MotionMask;
      sprintf(command,
      "%s %d %s \"%s\" %s \"marker_pick\" -mask %d",
          "image_select_events -scene_id",
          iv_id->scene_id,
          "-image_name",
          imageName,
          "-module $Module -func_id",
          mask);
      add_command(command, clibuf, 1);

      sprintf(command, "image_set_pick_mode \"marker_pick\"");
      add_command(command, clibuf, 1);
    }
    else  /* mouse events in the queue */
    {
      /* got mouse events to deal with */
      end = 0;

      /* loop over all events */
      for (i = 0; i < mouseInfo->nEvents; i++)
      {
        if (((mouseInfo->buttonMask[i] & IMAGE_Button1Mask) &&
           (mouseInfo->buttonMask[i] & IMAGE_PressMask)))
        {
            /* button down defines the start point */
            startX = (int) mouseInfo->image_x[i];
            if (startX < 0)
              startX = 0;
            else if (startX > MAXX(input))
              startX = MAXX(input);

            startY = (int) mouseInfo->image_y[i];
            if (startY < 0)
              startY = 0;
            else if (startY > MAXY(input))
              startY = MAXY(input);

            begin = 1;
            AVSmark_output_unchanged("Polyline Output");
        }
        else if ((mouseInfo->buttonMask[i] & IMAGE_Button1Mask) &&
                 (mouseInfo->buttonMask[i] & IMAGE_ReleaseMask))
        end = 1;
      }

      /* process picking up of existing marker */
      /* search list for match */
      if ( begin )
      {

          /*
           * find the (squared euclidean)
           * distance from pick point to each marker
           * and save in a list.
           */
          for ( i=0; i < num_marker_picks ; i++ )
          {
              x    = marker_pick_buf[0][i];
              y    = marker_pick_buf[1][i];
              search_dist[i] =  ( startX - x ) * ( startX - x ) +
                                ( startY - y ) * ( startY - y );
          }

          /*
           * search the list for the nearest marker,
           * and throw out if too far away
           */
          nearest = 80000; /* start in outer space */
          picked_marker = -1;
          for ( i=0; i < num_marker_picks ; i++ )
          {
              if ( search_dist[i] < nearest )
              {
                  nearest = search_dist[i];
                  picked_marker = i;
              }
          }
          /* 20 pixel pick radius */
          if ( picked_marker >= 0 &&
               nearest > 400 ) picked_marker = -1;



          display_info (marker_pick_buf, picked_marker);
      }

      /* draw lines and build structures based on the event */
      /* Not at the beginning, rubber band mode */
      if ( !end && !begin)
      {
          /* undraw previous line */
          sprintf(command, CMD_STR,
              "image_add_xor_line",
               iv_id->scene_id,
               imageName,
               (float) startX,
               (float) startY,
               (float) endX,
               (float) endY  );
          add_command(command, clibuf, 0);
          flush_command_buffer(clibuf);
      }

      /* Get NEW end point */
      /* clamp pick to be inside image */
      endX = (int) mouseInfo->image_x[mouseInfo->nEvents-1];
      if (endX < 0)
          endX = 0;
      else if (endX > MAXX(input))
          endX = MAXX(input);

      endY = (int) mouseInfo->image_y[mouseInfo->nEvents-1];
      if (endY < 0)
          endY = 0;
      else if (endY > MAXY(input))
          endY = MAXY(input);


      /* rubber band mode */
      /* draw new end-point line if no button up */
      if ( !end &&  !begin)
      {
          sprintf(command, CMD_STR,
                  "image_add_xor_line",
                  iv_id->scene_id,
                  imageName,
                  (float) startX,
                  (float) startY,
                  (float) endX,
                  (float) endY);
          add_command(command, clibuf, 0);
          flush_command_buffer(clibuf);

          /* don't fire field port for every rubber-band move */
          AVSmark_output_unchanged("Polyline Output");
      }


      /* process a completed segment */
      if ( end )
      {

          /* undraw previous line */
          sprintf(command, CMD_STR,
              "image_add_xor_line",
               iv_id->scene_id,
               imageName,
               (float) startX,
               (float) startY,
               (float) endX,
               (float) endY  );
          add_command(command, clibuf, 0);
          flush_command_buffer(clibuf);

          /* Measure the Line */
          length = ((startX - endX) * (startX - endX)) +
                   ((startY - endY) * (startY - endY));
          if (length > 0.00001)
            length = sqrt(length);
          else
            length = 1.0;
          npixels = (int) length + 0.5;

          /*
           * add marker point to list
           */

          /* update marker new center, leave everything else the same */
          if ( picked_marker >= 0 )
          {
              marker_pick_buf[0][picked_marker] = endX;
              marker_pick_buf[1][picked_marker] = endY;
              picked_marker = -1;
          }
          else
          /* Assemble new data structures */
          {
              marker_pick_buf[0][num_marker_picks] = startX;
              marker_pick_buf[1][num_marker_picks] = startY;
              marker_pick_buf[2][num_marker_picks] = marker_type;
              marker_pick_buf[3][num_marker_picks] = npixels;
              marker_pick_buf[4][num_marker_picks] = marker_fill;
              marker_pick_buf[5][num_marker_picks] = endX;
              marker_pick_buf[6][num_marker_picks] = endY;
              num_marker_picks++;
          }

          /*
           * process output scatter field for marker_pick
           */
          fill_field ( poly, marker_pick_buf, num_marker_picks, MAXY(input) );

          end = 0;
          return(1);

      } /* end case */

      if ( begin ) begin = 0; /* toggle begin mode off for next time! */
    }
  }


  /*
   * Process the attention command button
   */
  if (AVSparameter_changed("set pick mode"))
  {
    if (imageName[0] != '\0')
    {
      sprintf(command, "image_set_pick_mode \"marker_pick\"");
      add_command(command, clibuf, 1);
    }
  }

  return(1);
}

/**********************************************************/
add_command(command, clibuf, flush)
char        *command;
CLIbuffer *clibuf;
int       flush;
{
  int len;

  len = strlen(command) + 1;
  if ((CLI_BUFFER_SIZE - (clibuf->bptr - clibuf->buffer)) < len)
    flush_command_buffer(clibuf);

  strcpy(clibuf->bptr, command);
  clibuf->bptr += len;
  *(clibuf->bptr - 1) = '\n';

  if (flush)
    flush_command_buffer(clibuf);
} 

/**********************************************************/
flush_command_buffer(clibuf)
CLIbuffer *clibuf;
{
  char *dummy;

  if (clibuf->bptr != clibuf->buffer)
  {
    *(clibuf->bptr) = '\0';
    AVScommand("kernel", clibuf->buffer, &dummy, &dummy);
    clibuf->bptr = clibuf->buffer;
  }
}
/**********************************************************/
fill_field( poly, marker_pick_buf, num_marker_picks, y_size )

    AVSfield_int  **poly;      /* output marker_pick field */
    int       marker_pick_buf[2][MAX_POLYLINE];
    int       num_marker_picks;
    int       y_size;
{
    int dims[3];
    int i, veclen;
    int x, y, type, sz, fill, endx, endy;

    dims[0] = num_marker_picks;
    if ( *poly ) AVSfield_free( *poly );
    *poly = (AVSfield_int *) AVSdata_alloc( MARKER_FIELD_SPEC, dims);

    veclen = (*poly)->veclen;

    for ( i=0; i< dims[0]; i++ )
    {
        x    = marker_pick_buf[0][i];
        y    = marker_pick_buf[1][i];
        type = marker_pick_buf[2][i];
        sz   = marker_pick_buf[3][i];
        fill = marker_pick_buf[4][i];
        endx = marker_pick_buf[5][i];
        endy = marker_pick_buf[6][i];

        (*poly)->data[i*veclen+0] = x;
        (*poly)->data[i*veclen+1] = y;
        (*poly)->data[i*veclen+2] = type;
        (*poly)->data[i*veclen+3] = sz;
        (*poly)->data[i*veclen+4] = fill;
        (*poly)->data[i*veclen+5] = endx;
        (*poly)->data[i*veclen+6] = endy;
    }

    AVSfield_set_labels (*poly,
     "X#Y#TYP#SZ#FILL#ENDX#ENDY", "#");
}

#define SCAT  strcat ( str, strt );

display_info (marker_pick_buf, picked_marker)
int       marker_pick_buf[MARKER_VECLEN][MAX_POLYLINE];
int       picked_marker;
{
  int i;
  char str[500], strt[80];

  if (picked_marker >= 0)
  {
     sprintf(str, " === Selected Marker Information ===\n");
     sprintf(strt," Start X = %d\n", marker_pick_buf[0][picked_marker]); SCAT
     sprintf(strt," Start Y = %d\n", marker_pick_buf[1][picked_marker]); SCAT
     sprintf(strt," Type    = %d\n", marker_pick_buf[2][picked_marker]); SCAT
     sprintf(strt," Size    = %d\n", marker_pick_buf[3][picked_marker]); SCAT
     sprintf(strt," Fill    = %d\n", marker_pick_buf[4][picked_marker]); SCAT
     sprintf(strt," End X   = %d\n", marker_pick_buf[5][picked_marker]); SCAT
     sprintf(strt," End Y   = %d\n", marker_pick_buf[6][picked_marker]); SCAT
  }
  else
  {
     sprintf(str, " === Selected Marker Information ===\n");
     sprintf(strt,"No marker selected\n");         SCAT
  }
  AVSmodify_parameter("Selected Marker",AVS_VALUE,str,NULL,NULL);

}

/**********************************************************/

#ifdef sep_exe
AVSinit_modules()
{
  AVSmodule_from_desc(marker_pick_desc);
}
#endif


