/****************************************************************************
                  INTERNATIONAL AVS CENTER
	(This disclaimer must remain at the top of all files)

WARRANTY DISCLAIMER

This module and the files associated with it are distributed free of charge.
It is placed in the public domain and permission is granted for anyone to use,
duplicate, modify, and redistribute it unless otherwise noted.  Some modules
may be copyrighted.  You agree to abide by the conditions also included in
the AVS Licensing Agreement, version 1.0, located in the main module
directory located at the International AVS Center ftp site and to include
the AVS Licensing Agreement when you distribute any files downloaded from 
that site.

The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module provide absolutely
NO WARRANTY OF ANY KIND with respect to this software.  The entire risk as to
the quality and performance of this software is with the user.  IN NO EVENT
WILL The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module BE LIABLE TO
ANYONE FOR ANY DAMAGES ARISING FROM THE USE OF THIS SOFTWARE, INCLUDING,
WITHOUT LIMITATION, DAMAGES RESULTING FROM LOST DATA OR LOST PROFITS, OR ANY
SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES.

This AVS module and associated files are public domain software unless
otherwise noted.  Permission is hereby granted to do whatever you like with
it, subject to the conditions that may exist in copyrighted materials. Should
you wish to make a contribution toward the improvement, modification, or
general performance of this module, please send us your comments:  why you
liked or disliked it, how you use it, and most important, how it helps your
work. We will receive your comments at avs@ncsc.org.

Please send AVS module bug reports to avs@ncsc.org.

******************************************************************************/
/*
 * scatter_rectangles
 *
 * draw geometric representation of scatter input
 *
 * If your vector data is 1D 1-vector or 4-vector irregular,
 * this module will draw cubes from the
 * specified positions in the scatter field coord
 * array.
 *
 * Author: I. Curington, AVS Inc, UK
 *
 * Revision:
 * 31 March 92  Original (from ucd_cell_cubes, cell data)
 * 14 July  92  Conversion to scatter fields
 * 15 July  92  bug fix in positioning
 * 15 Aug   92  conversion of scatter_arrows to scatter_cubes
 *    Jan   94  converted to scatter_rect by
 *              Scott Goodyear, for Arc/Info project
 *
 */

/*-----------------------------------------------------*
 *                                                     *
 *     ****  scatter_rect      module  ****            *
 *                                                     *
 *                                                     *
 *-----------------------------------------------------*/

#include <stdio.h>


#include <avs/avs.h>
/* IAC CODE CHANGE : #include <avs/avs_math.h> */
#include <avs/avs_math.h>
#include <avs/geom.h>
#include <avs/field.h>

#define min(a,b) ((a) < (b) ? (a) : (b))
#define max(a,b) ((a) > (b) ? (a) : (b))

/* =============== compute routine ============== */

static scatter_rect (input, output, r_scale, h_scale, zjitter)

  AVSfield_float *input;
  GEOMedit_list **output;
  float         *r_scale;
  float         *h_scale;
  float         *zjitter;
{
    char model_name[80];
    GEOMobj *obj, *lobj, *obj_wire;
    float *xc, *yc, *zc, *cell_data;
    int cell, i, name_flag, util_flag, data_len, cell_len, node_len,
        n, num_nodes, num_cells;
    int num_cubes;

    num_cubes = input->dimensions[0];  /* number of rectangles to draw */


    /* establish the geometry output object */
    *output = (GEOMedit_list *)GEOMinit_edit_list(*output);
    obj    =  GEOMcreate_obj ( GEOM_POLYHEDRON, NULL);
    obj_wire =  GEOMcreate_obj ( GEOM_POLYHEDRON, NULL);

    /* draw the geometry from the scatter list */
    display_cubes(input, num_cubes, obj,obj_wire, *r_scale, *h_scale, *zjitter);


    /* complete the geometry to output port process */
    GEOMgen_normals(obj,0);
    GEOMcvt_polyh_to_polytri(obj,
        GEOM_NO_CONNECTIVITY | GEOM_SURFACE | GEOM_WIREFRAME);

    GEOMcvt_polyh_to_polytri(obj_wire, GEOM_NO_CONNECTIVITY | GEOM_WIREFRAME);

    GEOMedit_geometry (*output, "ElemBars", obj);
    GEOMedit_geometry (*output, "ElemBarsBounds", obj_wire);
    GEOMdestroy_obj (obj);
    GEOMdestroy_obj (obj_wire);
    GEOMedit_transform_mode (*output, "ElemBarsBounds", "parent");
    GEOMedit_transform_mode (*output, "ElemBars", "parent");

    return(1);
}


/*-----------------------------------------------------*
 *                                                     *
 *          ****  display_cubes  ****                 *
 *                                                     *
 *-----------------------------------------------------*/

static display_cubes (input, num_cells, obj, obj_wire, 
                      r_scale, h_scale, zjitter )

  AVSfield_float *input;  /* scatter input port */
  int num_cells;          /* how many cells */
  GEOMobj *obj;           /* add polygons to this object */
  GEOMobj *obj_wire;           /* add polygons to this object */
  float r_scale;            /* */
  float h_scale;            /* */
  float zjitter;             /* */

{
      float colour[3], cx, cy, cz, czb, czt;
      int cell, cell_len;
      float *xc, *yc, *zc;    /* node position arrays */
      float radius;           /* size of cube */
      float height;
      float ldl, bgval, sal, msrval;

      /*********************************************/
      /**** loop over ALL cell  surfaces        ****/
      /*********************************************/
    
      cell_len = input->veclen;
      radius    = r_scale;

      for (cell = 0; cell < num_cells; cell++)
      {

        cx = input->points[ 0 * num_cells + cell ]; 
        cy = input->points[ 1 * num_cells + cell ];
        cz = zjitter + input->points[ 2 * num_cells + cell ];

        ldl = input->data[ cell_len * cell +0 ];
        bgval = input->data[ cell_len * cell +1 ];
        sal = input->data[ cell_len * cell +2 ];
        msrval = input->data[ cell_len * cell +3 ];
/*
        fprintf(stderr,"Scatter Rects: LDL = %f BACK = %f SAL = %f MSR = %f\n",
                        ldl, bgval, sal, msrval);
*/

        if (msrval == -1 ) {
         fprintf(stderr,"Scatter Rects: No measured data for node #%d\n",cell);
        } else {
        /* Draw the Green Box */
        /* Set the Colors for the Cube */
        colour[0] = 0.0;
        colour[1] = 1.0;
        colour[2] = 0.0;
        czb = 0.0;
        czt = min(msrval, bgval);
/* fprintf(stderr,"Scatter Rects: Height for green (min) = %f\n", height); */
        create_cubes (obj, cx-radius, cx+radius,
                           cy-radius, cy+radius,
                           cz + h_scale*czb, cz + h_scale*czt, colour );

        /* Draw the Yellow Box */
        /* Set the Colors for the Cube */
        colour[0] = 1.0;
        colour[1] = 1.0;
        colour[2] = 0.0;
        czb = bgval;
        czt = min(max(msrval,bgval), sal);
/* fprintf(stderr,"Scatter Rects: Height for green (min) = %f\n", height); */
        create_cubes (obj, cx-radius, cx+radius,
                           cy-radius, cy+radius,
                           cz + h_scale*czb, cz + h_scale*czt, colour );

        /* Draw the Red Box */
        /* Set the Colors for the Cube */
        colour[0] = 1.0;
        colour[1] = 0.0;
        colour[2] = 0.0;
        czb = sal;
        czt = max(msrval,sal);
/* fprintf(stderr,"Scatter Rects: Height for green (min) = %f\n", height); */
        create_cubes (obj, cx-radius, cx+radius,
                           cy-radius, cy+radius,
                           cz + h_scale*czb, cz + h_scale*czt, colour );

        /* Draw Box */
        /* Bottom of the box should be at cz */
        /* Top of the box should be at max(msrval, sal) */
        colour[0] = 1.0;
        colour[1] = 1.0;
        colour[2] = 1.0;
        czb = cz;
        czt = max(msrval,sal);
        create_wire (obj_wire, cx-(1.25*radius), cx+(1.25*radius),
                           cy-(1.25*radius), cy+(1.25*radius),
                           czb, cz + h_scale*czt, colour );

        /* Draw Box */

        } /* End if */
      }  /* end of cell loop */
    
}

/*-----------------------------------------------------*
 *                                                     *
 *        ****  create_cubes  ****                     *
 *                                                     *
 *-----------------------------------------------------*/
/* routine to draw one box of certain size and position */

create_cubes (obj, xs, xe, ys, ye, zs, ze, color )
    GEOMobj *obj;
    float xs, xe, ys, ye, zs, ze;
    float *color;
{
         float verts[ 4 * 3 ];
         float cols [ 4 ][ 3 ];  
         int   i;

         /* copy the same colour to all sides */
         for ( i=0; i<4; i++ )
         {
	     cols [ i ][ 0 ]         = color[ 0 ];
	     cols [ i ][ 1 ]         = color[ 1 ];
	     cols [ i ][ 2 ]         = color[ 2 ];
         }

         /* draw top quad */
         verts[ 0 * 3 + 0 ] = xs;
	 verts[ 0 * 3 + 1 ] = ys;
	 verts[ 0 * 3 + 2 ] = ze;

         verts[ 1 * 3 + 0 ] = xe;
	 verts[ 1 * 3 + 1 ] = ys;
	 verts[ 1 * 3 + 2 ] = ze;

         verts[ 2 * 3 + 0 ] = xe;
	 verts[ 2 * 3 + 1 ] = ye;
	 verts[ 2 * 3 + 2 ] = ze;

         verts[ 3 * 3 + 0 ] = xs;
	 verts[ 3 * 3 + 1 ] = ye;
	 verts[ 3 * 3 + 2 ] = ze;
         GEOMadd_disjoint_polygon ( obj, verts, NULL, NULL, 4,
				   GEOM_NOT_SHARED, GEOM_COPY_DATA);
         GEOMadd_float_colors(obj,cols,4,GEOM_COPY_DATA);

         /* draw bottom quad */
         verts[ 0 * 3 + 0 ] = xs;
	 verts[ 0 * 3 + 1 ] = ys;
	 verts[ 0 * 3 + 2 ] = zs;

         verts[ 1 * 3 + 0 ] = xe;
	 verts[ 1 * 3 + 1 ] = ys;
	 verts[ 1 * 3 + 2 ] = zs;

         verts[ 2 * 3 + 0 ] = xe;
	 verts[ 2 * 3 + 1 ] = ye;
	 verts[ 2 * 3 + 2 ] = zs;

         verts[ 3 * 3 + 0 ] = xs;
	 verts[ 3 * 3 + 1 ] = ye;
	 verts[ 3 * 3 + 2 ] = zs;
         GEOMadd_disjoint_polygon ( obj, verts, NULL, NULL, 4,
				   GEOM_NOT_SHARED, GEOM_COPY_DATA);
         GEOMadd_float_colors(obj,cols,4,GEOM_COPY_DATA);

         /* draw back quad */
         verts[ 0 * 3 + 0 ] = xs;
	 verts[ 0 * 3 + 1 ] = ys;
	 verts[ 0 * 3 + 2 ] = zs;

         verts[ 1 * 3 + 0 ] = xs;
	 verts[ 1 * 3 + 1 ] = ys;
	 verts[ 1 * 3 + 2 ] = ze;

         verts[ 2 * 3 + 0 ] = xs;
	 verts[ 2 * 3 + 1 ] = ye;
	 verts[ 2 * 3 + 2 ] = ze;

         verts[ 3 * 3 + 0 ] = xs;
	 verts[ 3 * 3 + 1 ] = ye;
	 verts[ 3 * 3 + 2 ] = zs;
         GEOMadd_disjoint_polygon ( obj, verts, NULL, NULL, 4,
				   GEOM_NOT_SHARED, GEOM_COPY_DATA);
         GEOMadd_float_colors(obj,cols,4,GEOM_COPY_DATA);

         /* draw front quad  SDG - OK */
         verts[ 0 * 3 + 0 ] = xs;
	 verts[ 0 * 3 + 1 ] = ye;
	 verts[ 0 * 3 + 2 ] = zs;

         verts[ 1 * 3 + 0 ] = xs;
	 verts[ 1 * 3 + 1 ] = ye;
	 verts[ 1 * 3 + 2 ] = ze;

         verts[ 2 * 3 + 0 ] = xe;
	 verts[ 2 * 3 + 1 ] = ye;
	 verts[ 2 * 3 + 2 ] = ze;

         verts[ 3 * 3 + 0 ] = xe;
	 verts[ 3 * 3 + 1 ] = ye;
	 verts[ 3 * 3 + 2 ] = zs;
         GEOMadd_disjoint_polygon ( obj, verts, NULL, NULL, 4,
				   GEOM_NOT_SHARED, GEOM_COPY_DATA);
         GEOMadd_float_colors(obj,cols,4,GEOM_COPY_DATA);

         /* draw left side quad - SDG - OK*/
         verts[ 0 * 3 + 0 ] = xs;
	 verts[ 0 * 3 + 1 ] = ys;
	 verts[ 0 * 3 + 2 ] = zs;

         verts[ 1 * 3 + 0 ] = xe;
	 verts[ 1 * 3 + 1 ] = ys;
	 verts[ 1 * 3 + 2 ] = zs;

         verts[ 2 * 3 + 0 ] = xe;
	 verts[ 2 * 3 + 1 ] = ys;
	 verts[ 2 * 3 + 2 ] = ze;

         verts[ 3 * 3 + 0 ] = xs;
	 verts[ 3 * 3 + 1 ] = ys;
	 verts[ 3 * 3 + 2 ] = ze;
         GEOMadd_disjoint_polygon ( obj, verts, NULL, NULL, 4,
				   GEOM_NOT_SHARED, GEOM_COPY_DATA);
         GEOMadd_float_colors(obj,cols,4,GEOM_COPY_DATA);

         /* draw right side  quad  - SDG - OK */
         verts[ 0 * 3 + 0 ] = xe;
	 verts[ 0 * 3 + 1 ] = ys;
	 verts[ 0 * 3 + 2 ] = zs;

         verts[ 1 * 3 + 0 ] = xe;
	 verts[ 1 * 3 + 1 ] = ye;
	 verts[ 1 * 3 + 2 ] = zs;

         verts[ 2 * 3 + 0 ] = xe;
	 verts[ 2 * 3 + 1 ] = ye;
	 verts[ 2 * 3 + 2 ] = ze;

         verts[ 3 * 3 + 0 ] = xe;
	 verts[ 3 * 3 + 1 ] = ys;
	 verts[ 3 * 3 + 2 ] = ze;
         GEOMadd_disjoint_polygon ( obj, verts, NULL, NULL, 4,
				   GEOM_NOT_SHARED, GEOM_COPY_DATA);
         GEOMadd_float_colors(obj,cols,4,GEOM_COPY_DATA);
} 

/*-----------------------------------------------------*
 *                                                     *
 *        ****  create_wireframe  ****                 *
 *                                                     *
 *-----------------------------------------------------*/
/* routine to draw one box of certain size and position */

create_wire (obj_wire, xs, xe, ys, ye, zs, ze, color )
    GEOMobj *obj_wire;
    float xs, xe, ys, ye, zs, ze;
    float *color;
{
         float verts[ 4 * 3 ];
         float cols [ 4 ][ 3 ];  
         int   i;

         /* copy the same colour to all sides */
         for ( i=0; i<4; i++ )
         {
	     cols [ i ][ 0 ]         = color[ 0 ];
	     cols [ i ][ 1 ]         = color[ 1 ];
	     cols [ i ][ 2 ]         = color[ 2 ];
         }

         /* draw top quad */
         verts[ 0 * 3 + 0 ] = xs;
	 verts[ 0 * 3 + 1 ] = ys;
	 verts[ 0 * 3 + 2 ] = ze;

         verts[ 1 * 3 + 0 ] = xs;
	 verts[ 1 * 3 + 1 ] = ye;
	 verts[ 1 * 3 + 2 ] = ze;

         verts[ 2 * 3 + 0 ] = xe;
	 verts[ 2 * 3 + 1 ] = ye;
	 verts[ 2 * 3 + 2 ] = ze;

         verts[ 3 * 3 + 0 ] = xe;
	 verts[ 3 * 3 + 1 ] = ys;
	 verts[ 3 * 3 + 2 ] = ze;
         GEOMadd_disjoint_polygon ( obj_wire, verts, NULL, NULL, 4,
				   GEOM_NOT_SHARED, GEOM_COPY_DATA);
         GEOMadd_float_colors(obj_wire,cols,4,GEOM_COPY_DATA);

         /* draw bottom quad */
         verts[ 0 * 3 + 0 ] = xs;
	 verts[ 0 * 3 + 1 ] = ys;
	 verts[ 0 * 3 + 2 ] = zs;

         verts[ 1 * 3 + 0 ] = xe;
	 verts[ 1 * 3 + 1 ] = ys;
	 verts[ 1 * 3 + 2 ] = zs;

         verts[ 2 * 3 + 0 ] = xe;
	 verts[ 2 * 3 + 1 ] = ye;
	 verts[ 2 * 3 + 2 ] = zs;

         verts[ 3 * 3 + 0 ] = xs;
	 verts[ 3 * 3 + 1 ] = ye;
	 verts[ 3 * 3 + 2 ] = zs;
         GEOMadd_disjoint_polygon ( obj_wire, verts, NULL, NULL, 4,
				   GEOM_NOT_SHARED, GEOM_COPY_DATA);
         GEOMadd_float_colors(obj_wire,cols,4,GEOM_COPY_DATA);

         /* draw back quad */
         verts[ 0 * 3 + 0 ] = xs;
	 verts[ 0 * 3 + 1 ] = ys;
	 verts[ 0 * 3 + 2 ] = zs;

         verts[ 1 * 3 + 0 ] = xs;
	 verts[ 1 * 3 + 1 ] = ys;
	 verts[ 1 * 3 + 2 ] = ze;

         verts[ 2 * 3 + 0 ] = xs;
	 verts[ 2 * 3 + 1 ] = ye;
	 verts[ 2 * 3 + 2 ] = ze;

         verts[ 3 * 3 + 0 ] = xs;
	 verts[ 3 * 3 + 1 ] = ye;
	 verts[ 3 * 3 + 2 ] = zs;
         GEOMadd_disjoint_polygon ( obj_wire, verts, NULL, NULL, 4,
				   GEOM_NOT_SHARED, GEOM_COPY_DATA);
         GEOMadd_float_colors(obj_wire,cols,4,GEOM_COPY_DATA);

         /* draw front quad */
         verts[ 0 * 3 + 0 ] = xs;
	 verts[ 0 * 3 + 1 ] = ye;
	 verts[ 0 * 3 + 2 ] = zs;

         verts[ 1 * 3 + 0 ] = xs;
	 verts[ 1 * 3 + 1 ] = ye;
	 verts[ 1 * 3 + 2 ] = ze;

         verts[ 2 * 3 + 0 ] = xe;
	 verts[ 2 * 3 + 1 ] = ye;
	 verts[ 2 * 3 + 2 ] = ze;

         verts[ 3 * 3 + 0 ] = xe;
	 verts[ 3 * 3 + 1 ] = ye;
	 verts[ 3 * 3 + 2 ] = zs;
         GEOMadd_disjoint_polygon ( obj_wire, verts, NULL, NULL, 4,
				   GEOM_NOT_SHARED, GEOM_COPY_DATA);
         GEOMadd_float_colors(obj_wire,cols,4,GEOM_COPY_DATA);

         /* draw left side quad */
         verts[ 0 * 3 + 0 ] = xs;
	 verts[ 0 * 3 + 1 ] = ys;
	 verts[ 0 * 3 + 2 ] = zs;

         verts[ 1 * 3 + 0 ] = xs;
	 verts[ 1 * 3 + 1 ] = ys;
	 verts[ 1 * 3 + 2 ] = ze;

         verts[ 2 * 3 + 0 ] = xe;
	 verts[ 2 * 3 + 1 ] = ys;
	 verts[ 2 * 3 + 2 ] = ze;

         verts[ 3 * 3 + 0 ] = xe;
	 verts[ 3 * 3 + 1 ] = ys;
	 verts[ 3 * 3 + 2 ] = zs;
         GEOMadd_disjoint_polygon ( obj_wire, verts, NULL, NULL, 4,
				   GEOM_NOT_SHARED, GEOM_COPY_DATA);
         GEOMadd_float_colors(obj_wire,cols,4,GEOM_COPY_DATA);

         /* draw right side  quad */
         verts[ 0 * 3 + 0 ] = xe;
	 verts[ 0 * 3 + 1 ] = ys;
	 verts[ 0 * 3 + 2 ] = zs;

         verts[ 1 * 3 + 0 ] = xe;
	 verts[ 1 * 3 + 1 ] = ys;
	 verts[ 1 * 3 + 2 ] = ze;

         verts[ 2 * 3 + 0 ] = xe;
	 verts[ 2 * 3 + 1 ] = ye;
	 verts[ 2 * 3 + 2 ] = ze;

         verts[ 3 * 3 + 0 ] = xe;
	 verts[ 3 * 3 + 1 ] = ye;
	 verts[ 3 * 3 + 2 ] = zs;
         GEOMadd_disjoint_polygon ( obj_wire, verts, NULL, NULL, 4,
				   GEOM_NOT_SHARED, GEOM_COPY_DATA);
         GEOMadd_float_colors(obj_wire,cols,4,GEOM_COPY_DATA);
} 

/*-----------------------------------------------------*
 *                                                     *
 *        ****  scatter_rect_desc  ****               *
 *                                                     *
 *-----------------------------------------------------*/
scatter_rect_desc()
{
  int scatter_rect(), param;

  /* name on the box */
  AVSset_module_name ("Scatter Rect", MODULE_MAPPER);

  AVScreate_input_port ("Input",
			"field 1D 3-space 4-vector irregular float", 
                        REQUIRED);

  /* output geomerty to draw */
  AVScreate_output_port ("scatter_cubes", "geom");

  /* Radius */
  param = AVSadd_float_parameter("Radius", 1.0, 0.0, 200.0);

  /* Height */
  param = AVSadd_float_parameter("Height", 1.0, 0.0, 200.0);

  /* Z Jitter */
  param = AVSadd_float_parameter("ZJitter", 0.0, 0.0, 50.0);

  /* routine pointers */
  AVSset_compute_proc (scatter_rect);

}  /* end of description routine */

AVSinit_modules()
{
        AVSmodule_from_desc(scatter_rect_desc);
}

