/*****************************************************************************
*
*                         NCSA DTM version 2.0
*                               June 10, 1991
*
* NCSA DTM Version 2.0 source code and documentation are in the public
* domain.  Specifically, we give to the public domain all rights for future
* licensing of the source code, all resale rights, and all publishing rights.
*
* We ask, but do not require, that the following message be included in all
* derived works:
*
* Portions developed at the National Center for Supercomputing Applications at
* the University of Illinois at Urbana-Champaign.
*
* THE UNIVERSITY OF ILLINOIS GIVES NO WARRANTY, EXPRESSED OR IMPLIED, FOR THE
* SOFTWARE AND/OR DOCUMENTATION PROVIDED, INCLUDING, WITHOUT LIMITATION,
* WARRANTY OF MERCHANTABILITY AND WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE
*
*****************************************************************************/

/*********************************************************************
**
**  $Header: /netdev/dtm/libsrc/RCS/socket.c,v 1.13 1991/09/18 15:33:08 jplevyak Exp $
**
**********************************************************************/

/*
 * $Log: socket.c,v $
 * Revision 1.13  1991/09/18  15:33:08  jplevyak
 * Added additional parameter to dtm_socket_init
 *
 * Revision 1.12  91/09/13  20:28:52  sreedhar
 * accept :9900 change
 * 
 * Revision 1.11  1991/09/13  20:13:35  sreedhar
 * take current host as default
 *
 * Revision 1.10  1991/08/19  18:53:37  jefft
 * Fixed bug with dtm_socket_init, now checks port number for absolute
 * address instead of the IP address (which isn't used anyway).
 *
 * Revision 1.9  1991/08/15  18:56:35  sreedhar
 * Changes for logical portname version
 *
 * Revision 1.7  1991/06/11  15:19:45  sreedhar
 * disclaimer added
 *
 * Revision 1.6  1991/06/07  16:07:21  sreedhar
 * Changes for sequence start message
 *
 * Revision 1.5  1991/05/30  15:52:10  sreedhar
 * Changes for readMsg/writeMsg internal release
 *
 * Revision 1.4  1990/12/11  14:11:38  jefft
 * made dtm_get_ipaddr CRAY specific to fix final portability problem.
 *
 * Revision 1.3  90/11/21  12:43:15  jefft
 * Fixed portibility problem with dtm_get_ipaddr.
 * 
 * Revision 1.2  90/11/21  10:54:18  jefft
 * Added new routine, dtm_get_ipaddr.  It returns an ascii string of the
 * current hosts IP address.
 * 
 * Revision 1.1  90/11/08  16:39:40  jefft
 * Initial revision
 * 
 */

/*
	+++++ System call - merge dtm_connect, dtm_quick_connect +++++

	Check on whether dtm_get_ipaddr and init_sockaddr can be merged.
*/

#include <stdio.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <sys/uio.h>
#include <sys/socket.h>
#include <string.h>

/*	Machine specific header file(s)	*/

#ifdef	RS6000
#include <sys/select.h>
#endif

#include <netinet/in.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <errno.h>


#ifdef  macintosh
#  include "DTMMac.h"
#  include <MacSockDefs.h>
#  include <socket.ext.h>
#endif



#include "dtmint.h"
#include "debug.h"

extern int	errno;



#define		REFUSELIMIT	120


static int	buf_size = 32768;

/*
	Function to check whether given address string is in dotted
	decimal notation and if so, to return the address in network byte 
	order.

	Return	values	:	TRUE, if in dotted decimal notation.
				      ( network order address returned thru *addr )
				FALSE, if not.
*/

int	ipaddr( s, addr )
char	*s;		/* address string */
long	*addr;		/* location to return network byte order address */
{
int	b1, b2, b3, b4;

	if( sscanf(s, "%d.%d.%d.%d", &b1, &b2, &b3, &b4) != 4 )
    		return FALSE;

  	*addr = htonl(b1 << 24 | b2 << 16 | b3 << 8 | b4);
  	STDINT(*addr);
  	return TRUE;
}

int dtm_quick_select(s, count)
  int	s, *count;
{
  fd_set		filedes;
  static struct timeval	timeout = {0L, 0L};

  DBGFLOW("# dtm_quick_select called.\n");

  FD_ZERO(&filedes);
  FD_SET(s, &filedes);

  if (select(32, &filedes, (fd_set *)NULL, (fd_set *)NULL, &timeout))  {
    ioctl(s, FIONREAD, count);
    return TRUE;
    }
  else  {
    *count = 0;
    return FALSE;
    }
}


int dtm_select( s, count, time )
int	s ;
int	*count ;
int	time ;
{
fd_set	filedes;
static 	struct timeval	timeout = { 0L, 0L };

  	DBGFLOW("# dtm_select called.\n");

	timeout.tv_sec = time ;

  	FD_ZERO( &filedes );
  	FD_SET( s, &filedes );

  	if( (*count = select( 32, &filedes, (fd_set *)NULL, (fd_set *)NULL, &timeout ) ))
	{
    		ioctl( s, FIONREAD, count );
    		return TRUE;
    	}
  	else
	{
    		return FALSE;
    	}
}

/*
	Function to accept connection request on specified socket.
*/

int dtm_accept( s, sn, timeout )
int	s;
S_ADDR	*sn;
struct	timeval	*timeout ;
{
int	snsize = sizeof (S_ADDR);

  	DBGFLOW( "dtm_accept called.\n");
	DBGINT( "dtm_accept: sockfd = %d\n", s );

	/*	
		Await connect for specified time period only.	

		if timeout == NULL, it means just goahead and accept,
		else wait for specified period and accept only if
		connection request arrives in that period.
	*/

	if( timeout )
	{
	fd_set	readmask ;		
	fd_set	*fchk = &readmask ;
	int	nf ;

		FD_ZERO( fchk );
		FD_SET( s, fchk ); 

		nf = select( FD_SETSIZE, fchk, (fd_set *)0, (fd_set *)0, timeout );
		if( nf < 0 )
		{
		extern	int	errno ;

			DBGINT( "dtm_accept: select errno %d\n",				errno );
			DTMerrno = DTMSELECT ;
			return DTMERROR ;
		} 

		if( nf == 0 )
		{
			/* No connect request in specified time	*/

			DBGFLOW( "dtm_accept: timed out\n" );
			return	DTMERROR ;
		}
	}

  	/* accept connections on socket */

  	if( (s = accept(s, sn, &snsize)) < 0 )
	{
     		DTMerrno = DTMSOCK;
     		DBGINT("dtm_accept: error %d accepting connection.", errno );
    		return DTMERROR ;
     	}

  	return s;
}

int dtm_connect(sn, s)
  S_ADDR	*sn;
  int		*s;
{
  int	d;
  int	refusedcount = 0;

  	DBGFLOW("dtm_connect called.\n");
	DBGINT( "dtm_connect: s_addr = %x\n", 
		ntohl( sn -> sin_addr.s_addr ) );
	DBGINT( "dtm_connect: sin_port = %d\n", 
			ntohs( sn -> sin_port ));

  while (1)  {

    /* create socket */
    if ((d = socket(AF_INET, SOCK_STREAM, 0)) < 0)  {
      DTMerrno = DTMSOCK;
      DTMERR("dtm_connect: could not create socket.");
      return -1;
      }

    /* attempt to connect to receiver */
    if (connect(d, sn, sizeof (S_ADDR)) < 0)  {

      /* if connection refused, try again in 2 second */
      if (errno == ECONNREFUSED)  {
        close(d);
        sleep(2);
        if ((refusedcount += 1) > REFUSELIMIT)  {
          DTMerrno = DTMTIMEOUT;
          return -1;
          }
        else
          continue;
        }

      /* system error, can not connect, quit */
      else {
        DTMerrno = DTMSOCK;
        DTMERR("dtm_connect: could not connect.");
        return -1;
        }
      }

    /* connect complete, set working socket to original socket */
    else  {
      *s = d;
      setsockopt(*s, IPPROTO_TCP, TCP_NODELAY, (char *)&d, sizeof d);
      setsockopt(*s, SOL_SOCKET, SO_SNDBUF, (char *)&buf_size, sizeof(int));
      return 0;
      }

    }  /* end while */
}


int dtm_quick_connect(sn, s)
  S_ADDR	*sn;
  int		*s;
{
  int	d;

  DBGFLOW("# dtm_quick_connect called.\n");

  /* create socket */
  if ((d = socket(AF_INET, SOCK_STREAM, 0)) < 0)  {
   DTMerrno = DTMSOCK;
   DBGFLOW("dtm_quick_connect: could not create socket.");
   return -1;
   }

  /* attempt to connect to receiver */
  if (connect(d, sn, sizeof (S_ADDR)) < 0)  {

    /* if connection refused */
    if (errno == ECONNREFUSED)  {
      close(d);
      DTMerrno = DTMTIMEOUT;
      return -1;
      }

    /* system error, can not connect, quit */
    else {
      DTMerrno = DTMSOCK;
      DBGFLOW("dtm_quick_connect: could not connect.");
      return -1;
      }
    }

  /* else connection has been made */
  else  {
    *s = d;
    setsockopt(*s, IPPROTO_TCP, TCP_NODELAY, (char *)&d, sizeof d);
    setsockopt(*s, SOL_SOCKET, SO_SNDBUF, (char *)&buf_size, sizeof (int));
    return 0;
    }
}

int dtm_end_connect(s)
  int		s;
{

struct	linger	lbuf ;

  	DBGFLOW("# dtm_end_connect called.\n");
	DBGINT( "dtm_end_connect: sockfd %d\n", s );

/*
	lbuf.l_onoff = 0 ;
	setsockopt( s, SOL_SOCKET, SO_LINGER, &lbuf, sizeof( struct linger ) );
*/
	return close( s );
}


/*
	Return	values	:	
				On success,
				Direct   - host address in network byte order.
				Indirect - *ipaddr has host address in dotted 
					   decimal notation.   

				On error, 0.
	Notes:
		  Error is returned as 0, since an internet address
		  of 0 is not possible for any host ( 0 refers to 'this' host
		  in internet context ).
*/

unsigned long	dtm_get_ipaddr( ipaddrstr )
char	*ipaddrstr ;
{
char	hostname[MAXHOSTNAMELEN];
struct 	hostent	*hp;

	DBGFLOW( "dtm_get_ipaddr called\n" );

	/* get hostname */

  	gethostname( hostname, sizeof hostname );

#ifdef macintosh

  	/* check if hostname is in dotted decimal notation - this is a Mac-Hack */
  	if( ipaddr( hostname, &tmp ) )
	{
    		strcpy( ipaddrstr , hostname );
    		return 0;
    	}
#endif

  	/* lookup IP address */

  	if( (hp = gethostbyname(hostname)) == NULL )
	{
    		DTMerrno = DTMHOST;
    		return 0;
    	}


/*
#if defined(CONVEX) || defined(SGI)

	strcpy( ipaddrstr , inet_ntoa( *(int *)(hp -> h_addr_list[0]))) ;
#else
#ifdef	CRAY
  	memcpy(&addr, hp->h_addr_list[0], hp->h_length);
  	strcpy(ipaddrstr , inet_ntoa(addr));
#else
	strcpy( ipaddrstr , inet_ntoa( hp -> h_addr_list[ 0 ] ));
#endif
#endif
*/
  	/* extract dotted decimal address */

	{
	struct	in_addr	inaddr ;

		inaddr = *((struct in_addr *)( hp -> h_addr_list[ 0 ])) ;
  		strcpy( ipaddrstr , inet_ntoa( inaddr ));
	}

	DBGINT( "dtm_get_ipaddr: dotted decimal address = '%s'\n", ipaddrstr  );
  	return	inet_addr( ipaddrstr  ) ; 
}

/*
	Function to acquire and bind a UDP or TCP port.
*/

int dtm_socket_init( sockaddr, porttype, fLogicalName )
	S_ADDR	*sockaddr ;
	int		porttype ;
	int		fLogicalName;
{
	int	sockfd ;
	int	type ;
	int	protocol ;
	int	opt = 1 ;

	DBGFLOW( "dtm_socket_init called \n" );
	DBGINT( "dtm_socket_init: sockaddr -> s_addr = %x\n", ntohl( sockaddr -> sin_addr.s_addr) );
	DBGINT( "dtm_socket_init: sockaddr -> sin_port = %d\n", ntohs( sockaddr -> sin_port) );

	sockaddr -> sin_family = AF_INET ;	
	if ( fLogicalName ) 
	{
		/* 
			Logical name had been supplied for makeport. 
			Assign port from system ( sin_port = 0 ), and accept
			from all network interfaces for multi-homed host 
			( INADDR_ANY ).
		*/
		sockaddr -> sin_addr.s_addr = htonl( INADDR_ANY );
		sockaddr -> sin_port = htons( 0 ) ;
	}


	/*	Acquire appropriate socket ( UDP or TCP )	*/

	if( porttype == INPORTTYPE )
	{
		sockaddr -> sin_addr.s_addr = htonl( INADDR_ANY );
		type = SOCK_STREAM ;
		protocol = IPPROTO_TCP ;
	}
	else
	{
		type = SOCK_DGRAM ;
		protocol = IPPROTO_UDP ;
	}

	if( (sockfd = socket( sockaddr -> sin_family, type, protocol )) < 0 )
	{
		DTMerrno = DTMSOCK ; 	
		DBGINT( "dtm_socket_init: socket create error %d", errno );
		return -1 ;
	}

	/*	Set socket options.		*/

	setsockopt( sockfd, SOL_SOCKET, SO_REUSEADDR, (char *)&opt, sizeof opt );
	setsockopt( sockfd, SOL_SOCKET, SO_RCVBUF, (char *)&buf_size, sizeof(int) );
	if( porttype == INPORTTYPE )
	{
		setsockopt( sockfd, IPPROTO_TCP, TCP_NODELAY, (char *)&opt, sizeof opt );
	}

	/*	Bind name to socket	*/

	DBGFLOW( "dtm_socket_init: Before bind\n" );
	DBGINT( "dtm_socket_init: sockfd = %d\n", sockfd );
	DBGINT( "dtm_socket_init: sockaddr -> family = %d\n", sockaddr -> sin_family );
	DBGINT( "dtm_socket_init: sockaddr -> s_addr = %x\n", ntohl( sockaddr -> sin_addr.s_addr) );
	DBGINT( "dtm_socket_init: sockaddr -> sin_port = %d\n", ntohs( sockaddr -> sin_port) );

	if( bind( sockfd, sockaddr, sizeof( struct sockaddr_in ) ) < 0 )
	{
		DTMerrno = DTMSOCK ;
		DBGINT( "dtm_socket_init: could not bind to sockaddr, errno = %d\n", errno );
		return -1 ;
	}

	/* 	Listen at socket for TCP port, buffer for 5 pending connections */

	if( porttype == INPORTTYPE )
	{
		listen( sockfd, 5 );
	}	

	/*	
		Get the actual assigned (port) address ( netid/hostid/portid )
		- netid/hostid from dtm_get_ipaddr(),portid from getsockname().

		Netid/hostid and portid is in network byte order.
		Assumption - host is not multi-homed.
	*/

	if( fLogicalName )
	{
	int	sockaddrsize = sizeof( struct sockaddr_in );
	char	buf[ 128 ] ;

		if( getsockname( sockfd, sockaddr, &sockaddrsize ) < 0 )
		{
			DBGINT( "dtm_socket_init: Unable to get sin_port, errno %d\n", errno );
			DTMerrno = DTMSOCK ;
			return -1 ;
		}
		if( (sockaddr -> sin_addr.s_addr = dtm_get_ipaddr( buf )) == 0)
		{
			DBGFLOW( "dtm_socket_init: Unable to get s_addr\n" );
			DTMerrno = DTMSOCK ;
			return -1 ;
		}

		DBGFLOW( "dtm_socket_init: Verify nethostid/portid\n" );
		DBGINT( "dtm_socket_init: Nethostid = %x\n", 
				ntohl( sockaddr -> sin_addr.s_addr ) ); 
		DBGINT( "dtm_socket_init: Portid = %d \n", 
				ntohs( sockaddr -> sin_port ) );
	}

	DBGINT( "dtm_socket_init: exit sockfd = %d\n", sockfd );
	return sockfd ;
}

/*
	Function to get sockaddr if portname is specified in
	physical portname format ( e.g. "kankakee:9900" )

	Return	value	:	0 on success,
				-1 on error

	Notes	:  Algorithm -

		   1. Check portname format.
		   2. If logical format, sockaddr.sin_addr.s_addr = 0
		   3. If physical format, fill in sockaddr.sin_port and
		      sockaddr.sin_addr.s_addr.

		It returns:
			sockaddr in network byte order.
			*pfLogicalName = TRUE if the port is logical.

*/

int init_sockaddr( sockaddr, portname, pfLogicalName )
	struct	sockaddr_in	*sockaddr ;
	char	*portname ;			/* read-only */
	int		*pfLogicalName;
{
	char	*host ;
	char	*port ;
	char	lportname[ PNAMELEN ] ;
	char	hostname[ MAXHOSTNAMELEN ] ;

	strncpy( lportname, portname, PNAMELEN - 1 );
	lportname[ PNAMELEN - 1 ] = '\0' ;

	DBGFLOW( "init_sockaddr called\n" );

	if( lportname[0] == ':' )
	{
		host = NULL ; 
		port = strtok( lportname, COLON );
	}
	else
	{
		host = strtok( lportname, COLON );
		if( (port = strtok( NULL,  COLON )) == NULL )
		{
			/* Logical format */
			DBGSTR( "init_sockaddr: logical portname %s\n", host );
			sockaddr -> sin_port = 0;
			sockaddr -> sin_addr.s_addr = 0;
			*pfLogicalName = TRUE;
			DBGINT( "init_sockaddr: sin_port = %d\n", sockaddr->sin_port );
			return 0;
		}
	}
	*pfLogicalName = FALSE;

	/* 
		Physical format - hostname is either in dotted decimal 
			          notation ( call ipaddr() ) or direct or missing.
	*/

	if( host == NULL )
	{
		gethostname( hostname, sizeof hostname );
		host = hostname ;
	}	
	DBGINT( "init_sockaddr: host %s\n", host );
	DBGINT( "init_sockaddr: port %s\n", port );

	if( !ipaddr( host, &sockaddr -> sin_addr.s_addr) )
	{
	struct	hostent	*hp ;

		if( (hp = gethostbyname( host )) == NULL )
		{
			DBGFLOW("init_sockaddr: gethostbyname returns error\n");
			DTMerrno = DTMHOST ;
			return -1 ;
		}
		else
		{
			memcpy( (char *)&sockaddr -> sin_addr.s_addr, 
				hp -> h_addr_list[ 0 ], hp -> h_length);
		}
	}

	/* Fill in port id */
	sockaddr -> sin_port = htons((unsigned short)atol( port ));

	DBGINT( "init_sockaddr: nethostid = %x\n", 
			ntohl( sockaddr -> sin_addr.s_addr ));
	DBGINT( "init_sockaddr: portid = %d\n", ntohs( sockaddr -> sin_port) );

	return 0 ;
}
