/*****************************************************************************
*
*                         NCSA DTM version 2.0
*                               June 10, 1991
*
* NCSA DTM Version 2.0 source code and documentation are in the public
* domain.  Specifically, we give to the public domain all rights for future
* licensing of the source code, all resale rights, and all publishing rights.
*
* We ask, but do not require, that the following message be included in all
* derived works:
*
* Portions developed at the National Center for Supercomputing Applications at
* the University of Illinois at Urbana-Champaign.
*
* THE UNIVERSITY OF ILLINOIS GIVES NO WARRANTY, EXPRESSED OR IMPLIED, FOR THE
* SOFTWARE AND/OR DOCUMENTATION PROVIDED, INCLUDING, WITHOUT LIMITATION,
* WARRANTY OF MERCHANTABILITY AND WARRANTY OF FITNESS FOR A PARTICULAR PURPOSE
*
*****************************************************************************/

/***************************************************************************
**
** rwrtns.c - provides very low level routines for reading and writing
**		buffer.  This code should be independent of communication
**		channel as long as the descriptor will work with system
**		read and write routines.
**
***************************************************************************/

/*********************************************************************
**
**  $Header: /netdev/dtm/libsrc/RCS/rwrtns.c,v 1.7 1991/08/20 15:56:06 sreedhar Exp $
**
**********************************************************************/

#ifdef RCSLOG

 $Log: rwrtns.c,v $
 * Revision 1.7  1991/08/20  15:56:06  sreedhar
 * Removed unused functions - dtm_write_buffer, dtm_send, dtm_recv
 *
 * Revision 1.6  1991/08/15  18:56:52  sreedhar
 * Changes for logical portname version
 *
 * Revision 1.4  1991/06/11  15:19:51  sreedhar
 * disclaimer added
 *
 * Revision 1.3  1991/06/07  16:06:29  sreedhar
 * sizeof( int ) replaced by 4 for message to be sent out
 *
 * Revision 1.2  1991/05/30  15:51:50  sreedhar
 * Changes for readMsg/writeMsg internal release
 *
 * Revision 1.1  1990/11/08  16:38:13  jefft
 * Initial revision
 *

#endif

#include	<stdio.h>
#include	<sys/types.h>
#include	<sys/socket.h>
#include	<sys/uio.h>
#include	<netinet/in.h>
#include	<fcntl.h>
#include	<errno.h>

#ifdef	macintosh
#  include	<MacSockDefs.h>
#endif


#include	"dtmint.h"
#include	"debug.h"



extern char	*dtm_discard;

static int	padding[] = {0, 3, 2, 1};


/*
 * dtm_read_buffer() - attempts to fill the next dtm buffer.  The 
 *	blocklen variable must be set to -1 after each surface
 *	to force recv_buffer to move the next surface.
 */

int dtm_read_buffer(d, blocklen, buffer, length)
  int	d, *blocklen, length;
  char	*buffer;
{
  register int	tmp, readcnt, count = 0;

  DBGFLOW("# dtm_read_buffer called.\n");
  DBGINT("recv_buffer: attempting to read %d bytes.\n", length);
  DBGINT("recv_buffer: initial blocklen = %d\n", *blocklen);

  /* if block length is -1 this is a new surface */
  /* get initial block count */
  if (*blocklen == -1)  {
    recv(d, blocklen, 4, 0);
    LOCALINT(*blocklen);
    DBGINT("initial blocklen = %d\n", *blocklen);
    }

  /* attempt to get a full buffer */
  while (1)  {

    /* if block length is 0, because last call to fill_buffer hit */
    /* the EOS or because this surface is zero length, return 0   */
    /* to indicate the end of surface.				  */
    if (*blocklen == 0)
      return 0;


    /* if block length is greater than buffer size then... */
    if (*blocklen >= length - count)  {

      /* fill next buffer */
      readcnt = length - count;
      while (readcnt)
        if ((tmp = recv(d, buffer + length - readcnt, readcnt, 0)) > 0)
          readcnt -= tmp;
        else  {
          DTMerrno = DTMREAD;
          DTMERR("dtm_read_buffer: error on read.");
          return -1;
          }

      /* decrement block length, if 0 get next block length */
      *blocklen -= (length - count);
      if (*blocklen == 0)  {
        recv(d, blocklen, 4, 0);
        LOCALINT(*blocklen);
        DBGINT("blocklen = %d\n", *blocklen);
        }

      /* if block length is 0 now, the EOS will be returned on */
      /* the next call to fill_buffer */

      /* return full buffer count */
      DBGINT("recv_buffer: buffer full, returning %d\n", length);
      return length;
      }

    /* else block length is less than buffer size */
    else  {

      /* read in end of block */
      readcnt = *blocklen;
      while (readcnt)
        if ((tmp=recv(d, buffer + count + *blocklen - readcnt, readcnt,0)) > 0)
          readcnt -= tmp;
        else  {
          DTMerrno = DTMREAD;
          DTMERR("dtm_read_buffer: error on read.");
          return -1;
          }

      /* increment count */
      count += *blocklen;

      /* get next block length */
      recv(d, blocklen, 4, 0);
      LOCALINT(*blocklen);
      DBGINT("blocklen = %d\n", *blocklen);

      /* if block length is 0 now, the correct count will be */
      /* returned now, and EOS on the next call to fill_buffer */ 
      if (*blocklen == 0)
        return count;

      }
    } /* end while */
}

/*
	Function to read header and return size.  

	Notes	: If buffer is too small, dump remainder of header 
		  and return error.
		  Actually, this is function to read length of data and
		  then to receive that much data - the data is called header
		  everywhere since that was the first usage of the function.
*/

int dtm_recv_header( d, header, length )
int	d, length;
char	*header;
{
int	readcnt, headerlen, tmp;
struct	sockaddr_in from ;
int	fromlen = sizeof( struct sockaddr_in ) ;

	DBGFLOW("# dtm_recv_header called.\n");

  	/* get header length */
	
  	if( (readcnt = recvfrom(d, &headerlen, 4, 0, ( struct sockaddr *)&from, 			( int *)&fromlen)) != 4)
	{
    		/* somehow hit EOF, return DTMEOF instead */

    		if( readcnt == 0 )
		{
      			DTMerrno = DTMEOF;
      			return -1;
      		}
		else
		{
    			if( errno == ECONNRESET )
			{
    				/* connection closed by writer, return EOF */

      				DTMerrno = DTMEOF;
      				return -1;
      			}
    			else
			{
    				/* don't know what the problem is, punt... */

      				DTMerrno = DTMREAD;
      				return -1;
      			}
		}
  	}    

  	LOCALINT(headerlen);

	/*  read the header */ 

  	readcnt = (length > headerlen) ? headerlen : length ;
  	header += readcnt;

  	while(readcnt)
	{
    		if( (tmp = recvfrom(d, header - readcnt, readcnt, 0, 
			( struct sockaddr *)&from, ( int *)&fromlen)) > 0)
		{
      			readcnt -= tmp;
		}
    		else
		{
      			DTMerrno = DTMREAD;
      			return -1;
      		}
	}

   	/* check for header greater than buffer size provided */ 

  	if( length >= headerlen ) 
	{
		return headerlen;
	}
  	else
	{
  		/* discard remaining header */

    		readcnt = headerlen - length;
		while (readcnt)
		{
    			if ((tmp = recvfrom(d, dtm_discard, readcnt, 0, 
				(struct sockaddr *)&from, (int *)&fromlen)) > 0)
			{
      				readcnt -= tmp;
			}
    			else
			{
      				DTMerrno = DTMREAD;
      				return -1;
      			}
		}
    
		DTMerrno = DTMHEADER;
    		return -1;
    	}
}

/*
 	dtm_recv_ack() - receive message ackowledgement

	Notes	: Berkeley implementation returns 0 from recv
		  if socket connection breaks while waiting in
		  recv system call.  System V returns -1 and 
		  ECONNRESET in errno for same error.

		  For historical reasons, DTMEOF is returned when
		  socket connection breaks in middle instead of
		  say DTMFCONN ( DTM connection failed error )
*/

int	dtm_recv_ack( d, ack )
int	d ;
int	*ack ;
{
int	tmp ;

	DBGFLOW("# dtm_recv_ack called.\n");

  	if( (tmp = recv( d, ack, 4, 0 )) != 4 )
	{
		DBGINT( "Recv_ack errno = %d\n", errno ) ;
		if( tmp == 0 )
		{
			/* Courtesy Berkeley */

			DTMerrno = DTMEOF ;
		}
		else
		{
			if( errno == ECONNRESET )
			{
				/* Courtesy system V */

				DTMerrno = DTMEOF ;
			}
			else
			{
    				DTMerrno = DTMREAD;
			}
		}
    		return -1;
    	}

	DBGINT( "ack received, tmp = %d\n", tmp );
  	LOCALINT(*ack);
  	return 0;
}


/*
 * dtm_send_ack() - send message acknowledgement
 */
int	dtm_send_ack(d, ack)
int	d, ack;
{
int	tmp ;

	DBGFLOW("# dtm_send_ack called.\n");

  	STDINT(ack);

  	if( (tmp = send(d, &ack, 4, 0)) != 4 )
	{
		if( (tmp < 0) && (errno == EPIPE) )
		{
			/* socket connection broke in middle */

			DTMerrno = DTMEOF ;
		}
		else
		{
			DTMerrno = DTMWRITE ;
		}

    		return -1;
	}

	return 0;
}

/*
	dtm_writev_buffer() - sends the buffers to receiving process.
*/

int	dtm_writev_buffer( fd, iov, iovlen, iovsize, addr, addrlen )
int	fd ;
struct	iovec	*iov ;
int	iovlen ;
int	iovsize ;
struct	sockaddr *addr ;
int	addrlen ;
{
int	tmp;
struct	msghdr	msgbuf ;

  	DBGINT("# dtm_writev_buffer called, fd %d.\n", fd );
	
	msgbuf.msg_name = (caddr_t)addr ; 
	msgbuf.msg_namelen = addrlen ;
	msgbuf.msg_iov = iov ;
	msgbuf.msg_iovlen = iovlen ;
	msgbuf.msg_accrights = 0 ;

	if( (tmp = sendmsg( fd, &msgbuf, 0 )) != iovsize )
	{ 
		DBGINT( "dtm_writev_buffer: sendmsg errno = %d\n", errno );
		if( (tmp < 0) && (errno == EPIPE) )
		{
			/* socket connection broke in middle */

			DTMerrno = DTMEOF ;
		}
		else
		{
			DTMerrno = DTMWRITE ;
		}
		return -1 ;	
	}	

	DBGINT( "dtm_writev_buffer tmp = %d\n", tmp );
	
	return	0 ;
}
