/*******************************************************************************
 *
 *  These are a set of utility field file routines that can be used by 
 *  user-written AVS modules.
 *
 *	29 May 91  Phil McDonald, NOAA/ERL/FSL	Original version.
 *
 ******************************************************************************/



#include	"avs_utils.h"



/*******************************************************************************
 *
 *  Read a field file.
 *
 */

AVSfield	*UTILS_fld_file_read (fld_file_name)

char	*fld_file_name;
{
    AVSfield		*field;
    FILE		*fld_file;


    if (fld_file_name == NULL) return (NULL);

    fld_file = fopen (fld_file_name, "r");

    if (fld_file == NULL)
    {
        AVSerror ("%s%s%s", "An error occurred while opening ",
                  "the field file\n\n", fld_file_name);
        return (NULL);
    }

    field = UTILS_fld_file_read_hdr (fld_file, fld_file_name);

    if (field != NULL)
    {
        field->field_data = UTILS_fld_file_read_dat (fld_file, fld_file_name,
                                                     field);
        if (field->field_data == NULL)
            field = UTILS_field_destroy (field);
        else
        {
            if (field->uniform != UNIFORM)
            {
                field->points = UTILS_fld_file_read_pts (fld_file,
                                                         fld_file_name,
                                                         field);
                if (field->points == NULL)
                    field = UTILS_field_destroy (field);
                else
                    UTILS_field_extents_find (field);
            }
        }
    }
    fclose (fld_file);

    return (field);
}



/*******************************************************************************
 *
 *  Read the specified number bytes (n_items * size) from the binary
 *  section of a field file.
 *
 */

unsigned char	*UTILS_fld_file_read_bin (fld_file, n_items, size)

FILE		*fld_file;
int		n_items, size;
{
    unsigned char	*ptr;

    if (fld_file == NULL) return (NULL);
    if ((n_items * size) <= 0) return (NULL);

    ptr = (unsigned char *) malloc (n_items * size);

    if (ptr != NULL)
    {
        if (fread (ptr, (unsigned int) size, n_items, fld_file) != n_items)
        {
            free (ptr);
            ptr = NULL;
        }
    }

    return (ptr);
}



/*******************************************************************************
 *
 *  Read the binary data section of a field file.
 *
 */

unsigned char	*UTILS_fld_file_read_dat (fld_file, fld_file_name, field)

FILE		*fld_file;
char		*fld_file_name;
AVSfield	*field;
{
    register int	ndat;

    if (fld_file == NULL) return (NULL);
    if (field    == NULL) return (NULL);

    ndat = UTILS_field_ndat (field);

    field->field_data = UTILS_fld_file_read_bin (fld_file, ndat, field->size);

    if (field->field_data == NULL)
    {
        AVSerror ("%s%s%s", "An error occurred while reading the data ",
                  "section of the field file\n\n", fld_file_name);
    }

    return (field->field_data);
}



/*******************************************************************************
 *
 *  Read the header of a field file.  Reading proceeds through the two
 *  form feeds.
 *
 */

AVSfield	*UTILS_fld_file_read_hdr (fld_file, fld_file_name)

FILE	*fld_file;
char	*fld_file_name;
{
    int			ndim, nspace, veclen, type, grid;
    int			dims[FLD_FILE_MAX_DIMS];
    int			i_str, i_val, i_hdr_stat, i;
    char		*parm_name, *parm_valu, pts_name[256], str[256];
    AVSfield		*field;

    if (fld_file == NULL) return (NULL);

    field       = NULL;
    pts_name[0] = '\0';

    ndim = nspace = veclen = type = grid = -1;
    for (i = 0; i < FLD_FILE_MAX_DIMS; i++) dims[i] = -1;

    i_hdr_stat = FLD_FILE_HDR_OK;
    while (i_hdr_stat == FLD_FILE_HDR_OK)
    {
        i_hdr_stat = UTILS_fld_file_read_hdr_line (fld_file, str);
        i_str = 0;

        if (i_hdr_stat == FLD_FILE_HDR_OK)
        {
            i_hdr_stat = UTILS_fld_file_read_hdr_atom_get (str, &i_str,
                                                           &parm_name);

            if (i_hdr_stat == FLD_FILE_HDR_OK)
            {
                i_hdr_stat = UTILS_fld_file_read_hdr_atom_get (str, &i_str,
                                                               &parm_valu);

                if (i_hdr_stat == FLD_FILE_HDR_OK)
                {
                    sscanf (parm_valu, "%d", &i_val);

                    switch (UTILS_fld_file_read_hdr_match (parm_name,
                                                           FLD_FILE_HDR_PARM_N,
                                                           FLD_FILE_HDR_PARM_L,
                                                           fld_file_hdr_parm))
                    {

                        case FLD_FILE_HDR_PARM_NDIM :

                            ndim = i_val;
                            break;

                        case FLD_FILE_HDR_PARM_DIM :

                            sscanf (&parm_name[3], "%d", &i);
                            if ((i > 0) && (i <= ndim)) dims[i-1] = i_val;
                            break;

                        case FLD_FILE_HDR_PARM_NSPACE :

                            nspace = i_val;
                            break;

                        case FLD_FILE_HDR_PARM_VECLEN :

                            veclen = i_val;
                            break;

                        case FLD_FILE_HDR_PARM_DATA :

                            i = UTILS_fld_file_read_hdr_match (parm_valu,
                                    FLD_FILE_HDR_DATA_N, FLD_FILE_HDR_DATA_L,
                                    fld_file_hdr_data);
                            if (i >= 0) type = fld_file_hdr_type[i];
                            break;

                        case FLD_FILE_HDR_PARM_FIELD :

                            i = UTILS_fld_file_read_hdr_match (parm_valu,
                                    FLD_FILE_HDR_GRID_N, FLD_FILE_HDR_GRID_L,
                                    fld_file_hdr_grid);
                            if (i >= 0) grid = i;
                            break;


                        case FLD_FILE_HDR_PARM_POINTS :

                            strcpy (pts_name, parm_valu);
                            break;

                    }

                }

            }

        }

        if (i_hdr_stat == FLD_FILE_HDR_COMMENT) i_hdr_stat = FLD_FILE_HDR_OK;
    }
    if (i_hdr_stat == EOF)
    {
        AVSerror ("%s%s%s", "An error occurred while reading the header ",
                  "section of the field file\n\n", fld_file_name);
        return (NULL);
    }


    field = UTILS_field_create (ndim, dims, nspace, veclen, type, grid);

    if (field == NULL) return (NULL);

    if ((field->uniform != UNIFORM) && (pts_name[0] != '\0'))
    {
        field->points = (float *) malloc (strlen (pts_name));
        strcpy ((char *) field->points, pts_name);
    }

    return (field);
}



/*******************************************************************************
 *
 *  Get the next atom ("word") from the current line that has been
 *  read from the field file header.  Atoms are separated by white space
 *  and "=".  If a comment ("#") is encountered, the rest of the line is
 *  ignored.  If the end of the line is encountered, EOF is returned.
 *
 */

int	UTILS_fld_file_read_hdr_atom_get (str, i_str, atom)

char	*str;
int	*i_str;
char	**atom;
{
    register int	i;

    *atom = NULL;

    i = *i_str;
    while ((isspace (str[i])) || (str[i] == '=')) i++;
    if (str[i] == '\0') return (EOF);
    if (str[i] == '#')
    {
        while (str[++i] != '\0');
        return (FLD_FILE_HDR_COMMENT);
    }
    *atom = &str[i];

    while (str[i] != '\0')
    {
        if ((isspace (str[i])) || (str[i] == '='))
        {
            str[i] = '\0', i++;
            break;
        }
        if (str[i] == '#')
        {
            while (str[i] != '\0') str[i++] = '\0';
            break;
        }
        i++;
    }

    *i_str = i;

    return (FLD_FILE_HDR_OK);
}



/*******************************************************************************
 *
 *  Read the next line from the field file header.  Lines are 
 *  terminated by newlines.  Two consecutive form feeds mark the end
 *  of the header.
 *
 */

int	UTILS_fld_file_read_hdr_line (fld_file, str)

FILE	*fld_file;
char	*str;
{
    register int	i;

    i = 0;
    while ((str[i] = fgetc (fld_file)) != '\n')
    {
        if (str[i] == '\f')
            if ((str[++i] = fgetc (fld_file)) == '\f') return (FLD_FILE_HDR_END);
        if (str[i] == EOF) return (EOF);
        i++;
    }
    str[i] = '\0';

    return (FLD_FILE_HDR_OK);
}



/*******************************************************************************
 *
 *  Test a string (str) against a list of strings (strs) for a match.
 *  If a match is found, the index of the matching member of strs is
 *  returned, otherwise -1 is returned.  A match will be made if all
 *  of str is found as the beginning of a strs member.  Therefore, 
 *  str need only be an unambiguous initial subset of a strs member.
 *  If str is an ambiguous initial subset of more than one strs member,
 *  the matching member will be the earliest one in the list.  If str is
 *  longer than a member and the member is an initial subset of str, the
 *  remaining characters of str will be ignored and a match will be
 *  indicated.
 *
 */

int	UTILS_fld_file_read_hdr_match (str, nstrs, lstrs, strs)

char	*str;
int	nstrs, lstrs;
char	*strs;
{
    register int	i, istrs;
    register char	*p_strs;

    for (istrs = 0, p_strs = strs; istrs < nstrs; istrs++, p_strs += lstrs)
    {
        i = 0;
        while (i >= 0)
        {
            if (str[i]    == '\0') return (istrs);
            if (p_strs[i] == '\0') return (istrs);
            if (str[i] == p_strs[i])
                i++;
            else
                i = -1;
        }
    }

    return (-1);
}



/*******************************************************************************
 *
 *  Read the binary coordinate data of a field file.
 *
 */

float		*UTILS_fld_file_read_pts (fld_file, fld_file_name, field)

FILE		*fld_file;
char		*fld_file_name;
AVSfield	*field;
{
    register int	npts;
    char		file_dir[256], *pts_file_name;

    if (field == NULL) return (NULL);

    if (field->uniform == UNIFORM) return (field->points);

    if (fld_file == NULL) return (NULL);

    if (field->points == NULL)
    {
        npts = UTILS_field_npts (field);

        field->points = (float *) UTILS_fld_file_read_bin (fld_file, npts,
                                                           sizeof (float));
        field->points = UTILS_fld_file_read_pts_check (field->points);
        if (field->points == NULL)
            AVSerror ("%s%s%s", "An error occurred while reading the ",
                      "coordinate section of the field file\n\n",
                      fld_file_name);
    }

    else	/* field->points better point to a coordinate file name */
    {
            pts_file_name = (char *) field->points;
            file_dir[0]   = '\0';
            if (pts_file_name[0] != '/')
                UTILS_file_name_dir_get (fld_file_name, file_dir);
            UTILS_file_name_dir_add (pts_file_name, file_dir);
            free (field->points);

            field->points = UTILS_fld_file_read_pts_file (file_dir, field);
        }

    return (field->points);
}



/*******************************************************************************
 *
 *  Check coordinate data area to see if it contains binary coordinate data.
 *
 */

float		*UTILS_fld_file_read_pts_check (p_points)

float		*p_points;
{
    register int		i;
    register unsigned char	*p_pts;

    if (p_points != NULL)
    {
        p_pts = (unsigned char *) p_points;
        for (i = 0; i < 4; i++)
            if (!isprint (p_pts[i])) return (p_points);
        free (p_points);
        p_points = NULL;
    }

    return (p_points);
}



/*******************************************************************************
 *
 *  Read binary coordinate data from a coordinate file.
 *
 */

float		*UTILS_fld_file_read_pts_file (pts_file_name, field)

char		*pts_file_name;
AVSfield	*field;
{
    register int	n;
    FILE		*pts_file;

    if (field == NULL) return (NULL);

    if (field->uniform == UNIFORM) return (field->points);

    if (pts_file_name == NULL) return (NULL);

    pts_file = fopen (pts_file_name, "r");
    n        = UTILS_field_npts (field);

    field->points = (float *) UTILS_fld_file_read_bin (pts_file, n, 4);
    field->points = UTILS_fld_file_read_pts_check (field->points);
    if (pts_file != NULL) fclose (pts_file);

    if ((pts_file == NULL) || (field->points == NULL))
        AVSerror ("%s%s%s", "An error occurred while reading the ",
                  "coordinate file\n\n", pts_file_name);

    return (field->points);
}
