/*******************************************************************************
 *
 *  These are a set of utility color routines that can be used by 
 *  user-written AVS modules.
 *
 *	29 Mar 91  Phil McDonald, NOAA/ERL/FSL	Original version.
 *
 *	19 Aug 91  Phil McDonald		Correct rgb_get.
 *
 *	13 Sep 91  Phil McDonald		Add rgb_from_field.
 *
 *	17 Sep 91  Phil McDonald		Add map_index.
 *
 *	29 Oct 91  Phil McDonald		Change map_rgb_get to
 *						rgb_from_value.  Add
 *						rgb_from_index.
 *
 ******************************************************************************/



#include	"avs_utils.h"



/*******************************************************************************
 *
 *  Convert hue, saturation, and value values to normalized red, green,
 *  and blue values.
 *
 */

void	UTILS_color_hsv_to_rgb (hsv, rgb)

FLOAT3		hsv, rgb;
{
    register float	sat, val, fac;

    sat = hsv[1];
    val = hsv[2];
    UTILS_color_hue_to_rgb (hsv[0], rgb);

    fac = (rgb[0] >= rgb[1]) ? rgb[0] : rgb[1];
    fac = (rgb[2] > fac) ? rgb[2] : fac;
    fac = 1.0 / fac;
    rgb[0] *= fac, rgb[1] *= fac, rgb[2] *= fac;

    fac = 1.0 - sat;
    rgb[0] = (rgb[0] * sat) + fac;
    rgb[1] = (rgb[1] * sat) + fac;
    rgb[2] = (rgb[2] * sat) + fac;

    rgb[0] *= val, rgb[1] *= val, rgb[2] *= val;

    return;
}



/*******************************************************************************
 *
 *  Convert hue to normalized red, green, and blue values.
 *
 */

void	UTILS_color_hue_to_rgb (hue, rgb)

float		hue;
FLOAT3		rgb;
{
    static float	RED = 0.0;
    static float	GRN = 1.0 / 3.0;
    static float	BLU = 2.0 / 3.0;

    rgb[0] = (hue < 0.5) ? hue - RED : 1.0 - hue;
    rgb[0] = 1.0 - (3.0 * rgb[0]);
    if (rgb[0] < 0.0) rgb[0] = 0.0;

    rgb[1] = (hue > GRN) ? hue - GRN : GRN - hue;
    rgb[1] = 1.0 - (3.0 * rgb[1]);
    if (rgb[1] < 0.0) rgb[1] = 0.0;

    rgb[2] = (hue > BLU) ? hue - BLU : BLU - hue;
    rgb[2] = 1.0 - (3.0 * rgb[2]);
    if (rgb[2] < 0.0) rgb[2] = 0.0;

    return;
}



/*******************************************************************************
 *
 *  Return the index of a value in the specified colormap.
 *
 */

int	UTILS_color_map_index (p_cmap, val)

AVScolormap	*p_cmap;
float		val;
{

    if (p_cmap == NULL) return (-1);

    if (val  < p_cmap->lower) return (0);
    if (val >= p_cmap->upper) return (p_cmap->size - 1);

    return ((int) (((val - p_cmap->lower) /
                    (p_cmap->upper - p_cmap->lower)) *
                    (p_cmap->size - 1)));
}



/*******************************************************************************
 *
 *  Return RGB values for the data values of a field.
 *
 */

void	UTILS_color_rgb_from_field (p_cmap, p_field, p_p_rgb)

AVScolormap	*p_cmap;
AVSfield	*p_field;
FLOAT3		**p_p_rgb;
{
    register int	i, j, nx, ny;
    register float	val;
    register FLOAT3	*p_rgb;

    if (p_cmap == GEOM_NULL)
    {
        if (*p_p_rgb != GEOM_NULL)
        {
            free (*p_p_rgb);
            *p_p_rgb = GEOM_NULL;
        }
        return;
    }

    if (p_field == NULL) return;

    if (p_field->ndim < 1) return;
    if (p_field->ndim > 2) return;

    nx = (p_field->ndim > 0) ? p_field->dimensions[0] : 0;
    ny = (p_field->ndim > 1) ? p_field->dimensions[1] : 0;
    if (nx < 1) return;
    if (ny < 1) ny = 1;

    if (*p_p_rgb == GEOM_NULL)
    {
        *p_p_rgb = (FLOAT3 *) malloc (nx * ny * sizeof (FLOAT3));
        if (*p_p_rgb == NULL) return;
    }

    UTILS_field_pt_init (p_field);

    p_rgb = *p_p_rgb;
    for (j = 0; j < ny; j++)
    {
        for (i = 0; i < nx; i++)
        {
            val = UTILS_field_pt_val (i, j, 0);
            UTILS_color_rgb_from_value (p_cmap, val, p_rgb++);
        }
    }

    return;
}



/*******************************************************************************
 *
 *  Return the normalized red, green, and blue values for a colormap value.
 *
 */

void	UTILS_color_rgb_from_index (p_cmap, index, rgb)

AVScolormap	*p_cmap;
int		index;
FLOAT3		rgb;
{
    FLOAT3		hsv;

    if (p_cmap == NULL)		/* Use gray scale	*/
    {
        index = (index < 0) ? 0 : (index > 255) ? 255 : index;
        rgb[0] = rgb[1] = rgb[2] = ((float) index) / 255.0;
    }
    else
    {
        hsv[0] = p_cmap->hue[index];
        hsv[1] = p_cmap->saturation[index];
        hsv[2] = p_cmap->value[index];

        UTILS_color_hsv_to_rgb (hsv, rgb);

    }

    return;
}



/*******************************************************************************
 *
 *  Return the normalized red, green, and blue values for a colormap value.
 *
 */

void	UTILS_color_rgb_from_value (p_cmap, val, rgb)

AVScolormap	*p_cmap;
float		val;
FLOAT3		rgb;
{
    register int	imap;
    FLOAT3		hsv;

    if (p_cmap == NULL)		/* Use gray scale	*/
    {
        val = (val < 0.0) ? 0.0 : (val > 255.0) ? 255.0 : val;
        rgb[0] = rgb[1] = rgb[2] = val / 255.0;
    }
    else
    {
        imap = UTILS_color_map_index (p_cmap, val);

        hsv[0] = p_cmap->hue[imap];
        hsv[1] = p_cmap->saturation[imap];
        hsv[2] = p_cmap->value[imap];

        UTILS_color_hsv_to_rgb (hsv, rgb);

    }

    return;
}
