/*******************************************************************************
 *
 *  These are a set of utility draw routines that can be used by 
 *  user-written AVS modules.
 *
 *	15 Sep 92  Phil McDonald, NOAA/ERL/FSL	Original version.
 *
 *	03 Mar 93  Phil McDonald		Clean up things.  Change
 *						rotation funcs.
 *
 ******************************************************************************/



#include	"avs_utils.h"



/*******************************************************************************
 *
 *  Draw an arrow to depict flow (UVW).
 *
 */

void	UTILS_draw_arrow (uvw, xyz, size, scale, rot, rgb, p_obj)

FLOAT3		uvw, xyz;
float		size;
FLOAT3		scale, rot, rgb;
GEOMobj		*p_obj;
{
    register float	len, wid;
    FLOAT3		pts[2], *colors, rgb_dat[2];

    if (size <= 0.0) return;

    UTILS_draw_barrow_init (uvw, xyz, -size, scale, rot, pts);

    colors = NULL;
    if (rgb != NULL)
    {
        colors = rgb_dat;
        UTILS_float3_copy (rgb, (rgb_dat));
        UTILS_float3_copy (rgb, (rgb_dat+1));
    }

    GEOMadd_disjoint_line (p_obj, pts, colors, 2, GEOM_COPY_DATA);

    len = DRAW_ARROW_HEAD_LEN * size;
    wid = DRAW_ARROW_HEAD_WID * size;

    UTILS_float3_sub ((pts+1), pts);
    if (UTILS_float3_sqrt ((pts+1)) <= len) return;

    pts[1][0] = -len;
    pts[1][1] =  wid;
    pts[1][2] =  wid;
    UTILS_draw_arrow_line (pts, uvw, rot, colors, p_obj);

    pts[1][0] = -len;
    pts[1][1] = -wid;
    pts[1][2] =  wid;
    UTILS_draw_arrow_line (pts, uvw, rot, colors, p_obj);

    pts[1][0] = -len;
    pts[1][1] = -wid;
    pts[1][2] = -wid;
    UTILS_draw_arrow_line (pts, uvw, rot, colors, p_obj);

    pts[1][0] = -len;
    pts[1][1] =  wid;
    pts[1][2] = -wid;
    UTILS_draw_arrow_line (pts, uvw, rot, colors, p_obj);

    return;
}



/*******************************************************************************
 *
 *  Draw one line of an arrow depicting flow (UVW).
 *
 */

void	UTILS_draw_arrow_line (pts, uvw, rot, colors, p_obj)

FLOAT3		*pts, uvw, rot, colors;
GEOMobj		*p_obj;
{

    UTILS_draw_rot_by_vec ((pts+1), uvw);

    UTILS_draw_rot_by_deg ((pts+1), rot);

    UTILS_float3_add ((pts+1), pts);

    GEOMadd_disjoint_line (p_obj, pts, colors, 2, GEOM_COPY_DATA);

    return;
}



/*******************************************************************************
 *
 *  Draw an wind barb to depict flow (UVW).
 *
 */

void	UTILS_draw_barb (uvw, xyz, size, scale, rot, rgb, p_obj)

FLOAT3		uvw, xyz;
float		size;
FLOAT3		scale, rot, rgb;
GEOMobj		*p_obj;
{
    register float	x, xx, siz, len, spc;
    float		spd;
    FLOAT3		pts[2], *colors, rgb_dat[2];

    if (size <= 0.0) return;

    uvw[0] = -uvw[0], uvw[1] = -uvw[1], uvw[2] = 0.0;

    spd = UTILS_draw_barrow_init (uvw, xyz, size, scale, rot, pts);

    if (spd < DRAW_BARB_SPD_MIN)
    {
        size *= DRAW_BARB_CIRCLE_RAD;
        UTILS_draw_circle (xyz, size, scale, rot, rgb, p_obj);
        return;
    }

    colors = NULL;
    if (rgb != NULL)
    {
        colors = rgb_dat;
        UTILS_float3_copy (rgb, (rgb_dat));
        UTILS_float3_copy (rgb, (rgb_dat+1));
    }

    GEOMadd_disjoint_line (p_obj, pts, colors, 2, GEOM_COPY_DATA);

    siz = DRAW_BARB_SYM_SIZ * size;
    len = DRAW_BARB_SYM_LEN * size;
    spc = DRAW_BARB_SYM_SPC * size;

    x   = siz;
    xx  = x + spc;

    if (spd > DRAW_BARB_SPD_MAX) spd = DRAW_BARB_SPD_MAX;
    spd += 2.5;

    while (spd >= 50.0)
    {
        UTILS_float3_set (x, 0.0, 0.0, pts);
        UTILS_float3_set (xx, len, 0.0, (pts+1));
        UTILS_draw_barb_line (xyz, pts, uvw, rot, colors, p_obj);
        x -= spc;
        UTILS_float3_set (x, 0.0, 0.0, pts);
        UTILS_float3_set (xx, len, 0.0, (pts+1));
        UTILS_draw_barb_line (xyz, pts, uvw, rot, colors, p_obj);
        xx   = x, x -= spc;
        spd -= 50.0;
    }

    while (spd >= 10.0)
    {
        UTILS_float3_set (x, 0.0, 0.0, pts);
        UTILS_float3_set (xx, len, 0.0, (pts+1));
        UTILS_draw_barb_line (xyz, pts, uvw, rot, colors, p_obj);
        xx   = x, x -= spc;
        spd -= 10.0;
    }

    if (spd >= 5.0)
    {
        if (x == siz)
            x -= spc, xx -= spc;
        UTILS_float3_set (x, 0.0, 0.0, pts);
        xx -= spc * 0.5;
        UTILS_float3_set (xx, (len*0.5), 0.0, (pts+1));
        UTILS_draw_barb_line (xyz, pts, uvw, rot, colors, p_obj);
    }

    return;
}



/*******************************************************************************
 *
 *  Draw one line of an wind barb to depicting flow (UVW).
 *
 */

void	UTILS_draw_barb_line (xyz, pts, uvw, rot, colors, p_obj)

FLOAT3		xyz, *pts, uvw, rot, colors;
GEOMobj		*p_obj;
{

    UTILS_draw_rot_by_vec (pts, uvw);
    UTILS_draw_rot_by_vec ((pts+1), uvw);

    UTILS_draw_rot_by_deg (pts, rot);
    UTILS_draw_rot_by_deg ((pts+1), rot);

    UTILS_float3_add (pts, xyz);
    UTILS_float3_add ((pts+1), xyz);
    GEOMadd_disjoint_line (p_obj, pts, colors, 2, GEOM_COPY_DATA);

    return;
}



/*******************************************************************************
 *
 *  Initalize parameters for a wind barb or an arrow depicting flow (UVW).
 *
 */

float	UTILS_draw_barrow_init (uvw, xyz, size, scale, rot, pts)

FLOAT3		uvw, xyz;
float		size;
FLOAT3		scale, rot, *pts;
{
    float		u, v, w, spd, fac, factors[3];

    UTILS_float3_get (uvw, &u, &v, &w);

    u *= u, v *= v, w *= w;
    if (u < DRAW_VERY_SMALL) uvw[0] = 0.0;
    if (v < DRAW_VERY_SMALL) uvw[1] = 0.0;
    if (w < DRAW_VERY_SMALL) uvw[2] = 0.0;

    spd = UTILS_float3_sqrt (uvw);

    UTILS_float3_set (0.0, 0.0, 0.0, pts);
    UTILS_float3_set (0.0, 0.0, 0.0, (pts+1));

    if (spd > 0.0)
    {
        fac = (size > 0.0) ? size / spd : -size;
        UTILS_float3_set (fac, fac, fac, factors);
        UTILS_float3_mul (factors, scale);

        pts[0][0] += uvw[0] * factors[0];
        pts[0][1] += uvw[1] * factors[1];
        pts[0][2] += uvw[2] * factors[2];

        UTILS_draw_rot_by_deg (pts, rot);

        UTILS_float3_add (pts, xyz);
        UTILS_float3_add ((pts+1), xyz);
    }

    return (spd);
}



/*******************************************************************************
 *
 *  Draw a character at the specified location (XYZ).
 *
 */

void	UTILS_draw_char (chr, xyz, size, p_obj)

char		chr;
FLOAT3		xyz;
float		size;
GEOMobj		*p_obj;
{
    register int	i, plot;
    register FLOAT3	vec[2];

    if (draw_char_vec_ix == NULL) return;

    size *= draw_char_vec_factor;

    UTILS_float3_copy (xyz, vec);
    UTILS_float3_copy (xyz, (vec+1));

    plot = TRUE;

    i    = draw_char_vec_index[chr];
    while (i >= 0)
    {
        if (draw_char_vec_ix[i] > draw_char_vec_ix_max)
        {
            plot = FALSE;
            if (draw_char_vec_iy[i] == draw_char_vec_iy_max) i = -999;
        }
        else
        {
            vec[1][0] = xyz[0] + (((float) draw_char_vec_ix[i]) * size);
            vec[1][1] = xyz[1] + (((float) draw_char_vec_iy[i]) * size);

            if (plot)
            {
                GEOMadd_disjoint_line (p_obj, vec, GEOM_NULL, 2,
                                       GEOM_COPY_DATA);
            }

            vec[0][0] = vec[1][0], vec[0][1] = vec[1][1];
            plot = TRUE;
        }

        i++;
    }

    UTILS_float3_copy (vec, xyz);


    return;
}



/*******************************************************************************
 *
 *  Initialize the character vectors to those found in the specified file.
 *
 */

int	UTILS_draw_char_init (file_name)

char	*file_name;
{

    register int	i, j;
    char		*p_name, *p_tmp;
    FILE		*file;

/*
 *  The use of p_name and p_tmp is necessary because UTILS_file_name_check
 *  will wipe out the file name if the file cannot be opened.
 */
    i      = strlen (file_name);
    p_name = (char *) calloc ((i+1), sizeof (char));
    strcpy (p_name, file_name);
    p_tmp = p_name;

    file = UTILS_file_name_check (&p_name, "r", "The character vector file");

    free (p_tmp);

    if (file == NULL) return (FALSE);

    fscanf (file, "%d%d%d", &draw_char_vec_num,
            &draw_char_vec_ix_max, &draw_char_vec_iy_max);

    i = (draw_char_vec_ix_max > draw_char_vec_iy_max) ?
         draw_char_vec_ix_max : draw_char_vec_iy_max;
    draw_char_vec_factor = (i > 0) ? 1.0 / ((float) i) : 1.0;

    for (i = 0; i < 128; i++)
    {
        while (fscanf (file, "%d", (draw_char_vec_index+i)) == 0)
        {
            while (fgetc (file) != '\n');
        }
    }
    while (fgetc (file) != '\n');

    draw_char_vec_ix = (int *) malloc (draw_char_vec_num * sizeof (int));
    draw_char_vec_iy = (int *) malloc (draw_char_vec_num * sizeof (int));

    for (i = 0; i < draw_char_vec_num; )
    {
        j = i;
        while (fscanf (file, "%d", (draw_char_vec_ix+i)) != 0) i++;
        while (fgetc (file) != '\n');
        i = j;
        while (fscanf (file, "%d", (draw_char_vec_iy+i)) != 0) i++;
        while (fgetc (file) != '\n');
    }

    fclose (file);


    return (TRUE);
}



/*******************************************************************************
 *
 *  Draw a character string at the specified location (XYZ).
 *
 */

void	UTILS_draw_char_str (str, xyz, size, p_obj)

char		*str;
FLOAT3		xyz;
float		size;
GEOMobj		*p_obj;
{
    register int	i;

    if (str != NULL)
    {
        i = 0;
        while (str[i] != '\0') UTILS_draw_char (str[i++], xyz, size, p_obj);
    }


    return;
}



/*******************************************************************************
 *
 *  Draw the character string for a value at the specified location (XYZ).
 *
 */

void	UTILS_draw_char_val (val, xyz, size, fmt, p_obj)

float		val;
FLOAT3		xyz;
float		size;
char		*fmt;
GEOMobj		*p_obj;
{
    register char	str[11];

    sprintf (str, fmt, val);

    UTILS_draw_char_str (str, xyz, size, p_obj);

    return;
}



/*******************************************************************************
 *
 *  Draw a circle at the specified location (XYZ).
 *
 */

void	UTILS_draw_circle (xyz, size, scale, rot, rgb, p_obj)

FLOAT3		xyz;
float		size;
FLOAT3		scale, rot, rgb;
GEOMobj		*p_obj;
{

    static float	circle_xy[DRAW_CIRCLE_NPTS][2];

    register int	i;
    register float	ang, dang;
    FLOAT3		pts[DRAW_CIRCLE_NPTS];
    FLOAT3		*colors, facs, rgb_dat[DRAW_CIRCLE_NPTS];

    if (size <= 0.0) return;

    if (circle_xy[0][0] == 0.0)
    {
        dang = (atan (1.0) * 8.0) / (float) (DRAW_CIRCLE_NPTS - 1);
        ang  = 0.0;
        for (i = 0; i < DRAW_CIRCLE_NPTS; i++)
        {
            circle_xy[i][0] = sin (ang);
            circle_xy[i][1] = cos (ang);
            ang += dang;
        }
    }

    UTILS_float3_set (1.0, 1.0, 1.0, facs);
    UTILS_float3_mul (facs, scale);
    for (i = 0; i < DRAW_CIRCLE_NPTS; i++)
    {
        pts[i][0] = circle_xy[i][0] * size * facs[0];
        pts[i][1] = circle_xy[i][1] * size * facs[1];
        pts[i][2] = 0.0;

        UTILS_draw_rot_by_deg ((pts+i), rot);

        UTILS_float3_add ((pts+i), xyz);
    }

    colors = NULL;
    if (rgb != NULL)
    {
        colors = rgb_dat;
        for (i = 0; i < DRAW_CIRCLE_NPTS; i++)
            UTILS_float3_copy (rgb, (colors+i));
    }

    GEOMadd_polyline (p_obj, pts, colors, DRAW_CIRCLE_NPTS, GEOM_COPY_DATA);


    return;
}



/*******************************************************************************
 *
 *  Rotate a point (XYZ) by 3D angles in degrees (ROT).
 *
 */

void	UTILS_draw_rot_by_deg (xyz, rot)

FLOAT3		xyz, rot;
{
    register float	mat[4][4], m[4][4];

    if ((xyz == NULL) || (rot == NULL)) return;

    mat_identity (mat);

    if (rot[0] != 0.0) mat_x_rotate (rot[0], m), mat_multiply (mat, m, mat);
    if (rot[1] != 0.0) mat_y_rotate (rot[1], m), mat_multiply (mat, m, mat);
    if (rot[2] != 0.0) mat_z_rotate (-rot[2], m), mat_multiply (mat, m, mat);

    mat_vecmul (xyz, mat);


    return;
}



/*******************************************************************************
 *
 *  Rotate a point (XYZ) by a 3D vector (UVW).
 *
 */

void	UTILS_draw_rot_by_vec (xyz, uvw)

FLOAT3		xyz, uvw;
{
    register float	u, v, r, c, s;
    register float	mat[4][4], m[4][4];

    if ((xyz == NULL) || (uvw == NULL)) return;

    mat_identity (mat);

/*  Rotate about W axis */

    r = 0.0;

    if ((uvw[0] != 0.0) || (uvw[1] != 0.0))
    {
        u = uvw[0], v = uvw[1];

        r = sqrt ((u * u) + (v * v));

        c = u / r, s = v / r;

        mat_identity (m);
        m[0][0] = c, m[0][1] = s, m[1][0] = -s, m[1][1] = c;

        mat_multiply (mat, m, mat);
    }

/*  Rotate about V' axis */

    if (uvw[2] != 0.0)
    {
        u = r, v = uvw[2];

        r = sqrt ((u * u) + (v * v));

        c = u / r, s = v / r;

        mat_identity (m);
        m[1][1] = c, m[1][2] = s, m[2][1] = -s, m[2][2] = c;

        mat_multiply (mat, m, mat);
    }

    mat_vecmul (xyz, mat);


    return;
}
