/*******************************************************************************
 *
 *  These are a set of utility geometry destroy routines that can be used by 
 *  user-written AVS modules.
 *
 *	19 Mar 92  Phil McDonald, NOAA/ERL/FSL	Original version.
 *
 ******************************************************************************/



#include	"avs_utils.h"



/*******************************************************************************
 *
 *  Destroy a geometry edit list.
 *
 */

void	UTILS_geom_destroy_edlist (edlist)

GEOMedit_list	edlist;

{

    GEOMedit	*list, *next;

    if (edlist == GEOM_NULL) return;

    list = edlist->l;
    while (list != GEOM_NULL)
    {

        if (list->type == 0)
            UTILS_geom_destroy_obj (list->data);
        else
            if (list->data != NULL) free (list->data);

        if (list->name != NULL) free (list->name);

        next = list->next;

        free (list);

        list = next;
    }
    free (edlist);

    return;
}



/*******************************************************************************
 *
 *  Destroy a geometry object.
 *
 */

void	UTILS_geom_destroy_obj (geom_obj)

GEOMobj		*geom_obj;
{
    if (geom_obj == NULL) return;

    switch (geom_obj->type)
    {
        case GEOM_MESH :
            UTILS_geom_destroy_mesh (&(geom_obj->d.m));
            break;

        case GEOM_POLYHEDRON :
            UTILS_geom_destroy_polyh (&(geom_obj->d.ph));
            break;

        case GEOM_POLYTRI :
            UTILS_geom_destroy_polytri (&(geom_obj->d.pt));
            break;

        case GEOM_SPHERE :
            UTILS_geom_destroy_sphere (&(geom_obj->d.sp));
            break;

        case GEOM_LABEL :
            UTILS_geom_destroy_label (&(geom_obj->d.la));
            break;

        default :
            ;
    }

    free (geom_obj);


    return;
}



/*******************************************************************************
 *
 *  Destroy a geometry label object.
 *
 */

void	UTILS_geom_destroy_label (label)

GEOMlabel	*label;
{

    if (label == NULL) return;

    UTILS_geom_destroy_chlist (&(label->labels),  0);
    UTILS_geom_destroy_vtlist (&(label->verts),   0);
    UTILS_geom_destroy_vtlist (&(label->offsets), 0);
    UTILS_geom_destroy_vtlist (&(label->colors),  0);
    UTILS_geom_destroy_sclist (&(label->heights), 0);
    UTILS_geom_destroy_inlist (&(label->vflags),  0);

    free (label);


    return;
}



/*******************************************************************************
 *
 *  Destroy a geometry mesh object.
 *
 */

void	UTILS_geom_destroy_mesh (mesh)

GEOMmesh	*mesh;
{

    if (mesh == NULL) return;

    UTILS_geom_destroy_vtlist (&(mesh->verts),   0);
    UTILS_geom_destroy_vtlist (&(mesh->normals), 0);
    UTILS_geom_destroy_vtlist (&(mesh->colors),  0);
    UTILS_geom_destroy_sclist (&(mesh->uvs),     0);
    UTILS_geom_destroy_inlist (&(mesh->vdata),   0);
    UTILS_geom_destroy_inlist (&(mesh->pdata),   0);

    free (mesh);


    return;
}



/*******************************************************************************
 *
 *  Destroy a geometry polyhedron object.
 *
 */

void	UTILS_geom_destroy_polyh (polyh)

GEOMpolyh	*polyh;
{

    if (polyh == NULL) return;

    UTILS_geom_destroy_vtlist (&(polyh->verts),   0);
    UTILS_geom_destroy_vtlist (&(polyh->normals), 0);
    UTILS_geom_destroy_vtlist (&(polyh->colors),  0);
    UTILS_geom_destroy_sclist (&(polyh->uvs),     0);
    UTILS_geom_destroy_ptlist (&(polyh->ptlist),  0);
    UTILS_geom_destroy_inlist (&(polyh->vdata),   0);
    UTILS_geom_destroy_inlist (&(polyh->pdata),   0);

    free (polyh);


    return;
}



/*******************************************************************************
 *
 *  Destroy a geometry polytriangle object.
 *
 */

void	UTILS_geom_destroy_polytri (polytri)

GEOMpolytri	*polytri;
{
    int			i;

    if (polytri == NULL) return;

    for (i = 0; i < polytri->npts; i++)
    {
        UTILS_geom_destroy_vtlist (polytri->ptverts,   i);
        UTILS_geom_destroy_vtlist (polytri->ptcolors,  i);
        UTILS_geom_destroy_vtlist (polytri->ptnormals, i);
        UTILS_geom_destroy_sclist (polytri->ptuvs,     i);
        UTILS_geom_destroy_inlist (polytri->ptvdata,   i);
        UTILS_geom_destroy_inlist (polytri->ptpdata,   i);
    }
    for (i = 0; i < polytri->npls; i++)
    {
        UTILS_geom_destroy_vtlist (polytri->plverts,  i);
        UTILS_geom_destroy_vtlist (polytri->plcolors, i);
        UTILS_geom_destroy_inlist (polytri->plvdata,  i);
        UTILS_geom_destroy_inlist (polytri->plpdata,  i);
    }
    UTILS_geom_destroy_vtlist (&(polytri->dlverts),  0);
    UTILS_geom_destroy_vtlist (&(polytri->dlcolors), 0);
    UTILS_geom_destroy_inlist (&(polytri->dlvdata),  0);
    UTILS_geom_destroy_inlist (&(polytri->dlpdata),  0);

    free (polytri);


    return;
}



/*******************************************************************************
 *
 *  Destroy a geometry sphere object.
 *
 */

void	UTILS_geom_destroy_sphere (sphere)

GEOMsphere	*sphere;
{

    if (sphere == NULL) return;

    UTILS_geom_destroy_vtlist (&(sphere->verts),   0);
    UTILS_geom_destroy_sclist (&(sphere->radii),   0);
    UTILS_geom_destroy_vtlist (&(sphere->colors),  0);
    UTILS_geom_destroy_vtlist (&(sphere->normals), 0);
    UTILS_geom_destroy_inlist (&(sphere->vdata),   0);

    free (sphere);


    return;
}


/*******************************************************************************
 *
 *  Destroy a geometry char list.
 *
 */

void	UTILS_geom_destroy_chlist (chlist, index)

GEOMchar_list	*chlist;
int		index;
{
    register int	i;
    GEOMchar_list	*pc;

    if (chlist == NULL) return;

    pc = chlist + index;

    if (pc->l != NULL)
    {
        for (i = 0; i < pc->n; i++)
            if (pc->l[i] != NULL) free (pc->l[i]);
        free (pc->l);
    }
    free (pc);


    return;
}



/*******************************************************************************
 *
 *  Destroy a geometry int list.
 *
 */

void	UTILS_geom_destroy_inlist (inlist, index)

GEOMint_list	*inlist;
int		index;
{
    GEOMint_list		*pi;

    if (inlist == NULL) return;

    pi = inlist + index;

    if (pi->l != NULL) free (pi->l);
    free (pi);


    return;
}



/*******************************************************************************
 *
 *  Destroy a geometry point index list.
 *
 */

void	UTILS_geom_destroy_ptlist (ptlist, index)

GEOMp_list	*ptlist;
int		index;
{
    GEOMp_list		*pp;

    if (ptlist == NULL) return;

    pp = ptlist + index;

    if (pp->l != NULL) free (pp->l);
    free (pp);


    return;
}


/*******************************************************************************
 *
 *  Destroy a geometry scalar list.
 *
 */

void	UTILS_geom_destroy_sclist (sclist, index)

GEOMvert_list	*sclist;
int		index;
{
    GEOMvert_list	*ps;

    if (sclist == NULL) return;

    ps = sclist + index;

    if (ps->l != NULL) free (ps->l);
    free (ps);


    return;
}


/*******************************************************************************
 *
 *  Destroy a geometry vertex list.
 *
 */

void	UTILS_geom_destroy_vtlist (vtlist, index)

GEOMvert_list	*vtlist;
int		index;
{
    GEOMvert_list	*pv;

    if (vtlist == NULL) return;

    pv = vtlist + index;

    if (pv->l != NULL) free (pv->l);
    free (pv);


    return;
}
