/****************************************************************************
                  INTERNATIONAL AVS CENTER
	(This disclaimer must remain at the top of all files)

WARRANTY DISCLAIMER

This module and the files associated with it are distributed free of charge.
It is placed in the public domain and permission is granted for anyone to use,
duplicate, modify, and redistribute it unless otherwise noted.  Some modules
may be copyrighted.  You agree to abide by the conditions also included in
the AVS Licensing Agreement, version 1.0, located in the main module
directory located at the International AVS Center ftp site and to include
the AVS Licensing Agreement when you distribute any files downloaded from 
that site.

The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module provide absolutely
NO WARRANTY OF ANY KIND with respect to this software.  The entire risk as to
the quality and performance of this software is with the user.  IN NO EVENT
WILL The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module BE LIABLE TO
ANYONE FOR ANY DAMAGES ARISING FROM THE USE OF THIS SOFTWARE, INCLUDING,
WITHOUT LIMITATION, DAMAGES RESULTING FROM LOST DATA OR LOST PROFITS, OR ANY
SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES.

This AVS module and associated files are public domain software unless
otherwise noted.  Permission is hereby granted to do whatever you like with
it, subject to the conditions that may exist in copyrighted materials. Should
you wish to make a contribution toward the improvement, modification, or
general performance of this module, please send us your comments:  why you
liked or disliked it, how you use it, and most important, how it helps your
work. We will receive your comments at avs@ncsc.org.

Please send AVS module bug reports to avs@ncsc.org.

******************************************************************************/
/*   
     This software is copyright (C) 1991,  Regents  of  the
University  of  California.   Anyone may reproduce trivar.c,
the software in this distribution, in whole or in part, pro-
vided that:

(1)  Any copy  or  redistribution  of  trivar.c  must  show  the
     Regents  of  the  University of California, through its
     Lawrence Berkeley Laboratory, as the source,  and  must
     include this notice;

(2)  Any use of this software must reference this  distribu-
     tion,  state that the software copyright is held by the
     Regents of the University of California, and  that  the
     software is used by their permission.

     It is acknowledged that the U.S. Government has  rights
in  trivar.c  under  Contract DE-AC03-765F00098 between the U.S.
Department of Energy and the University of California.

     trivar.c is provided as a professional  academic  contribu-
tion  for  joint exchange.  Thus it is experimental, is pro-
vided ``as is'', with no warranties of any kind  whatsoever,
no  support,  promise  of updates, or printed documentation.
The Regents of the University of California  shall  have  no
liability  with respect to the infringement of copyrights by
trivar.c, or any part thereof.     

Author:
	Wes Bethel   
	Lawrence Berkeley Laboratory
	1 Cyclotron Rd.   Mail Stop 50-F
	Berkeley CA 94720
	510-486-6626
	ewbethel@lbl.gov
    
    19 June 1991

    This AVS module will do 3 variable resampling of scattered
    data.

    The algorithm is essentially to weight the value at each output
    grid point as a function of any number of input points which
    lie within a user-specified "search radius".

    11 Dec 1991

    Changed output field from irregular to rectilinear field.

    23 April 1992

    Rewrote computation function so as to be more or less vectorized.
*/

#include <stdio.h>
/* IAC CODE CHANGE : #include <math.h> */
#include <avs/avs_math.h>
#include <avs/avs.h>
#include <avs/field.h>

AVSinit_modules()
{
    int trivar();
    AVSmodule_from_desc(trivar);
}

trivar()
{
    int p,trivar_p();

    AVSset_module_name("trivar",MODULE_MAPPER);
    AVScreate_input_port("input_field","field 3-coordinate irregular scalar float",REQUIRED);
    AVScreate_output_port("output_field","field 3D rectilinear scalar float");

    p=AVSadd_parameter("dummy1","string","Interpolation Range Control","","");
    AVSconnect_widget(p,"text");
    AVSadd_parameter_prop(p,"width","integer",4);
    
    p=AVSadd_float_parameter("Xmin",-1.0,FLOAT_UNBOUND,FLOAT_UNBOUND);
    AVSconnect_widget(p,"typein_real");
    p=AVSadd_float_parameter("Xmax",1.0,FLOAT_UNBOUND,FLOAT_UNBOUND);
    AVSconnect_widget(p,"typein_real");
    
    p=AVSadd_float_parameter("Ymin",-1.0,FLOAT_UNBOUND,FLOAT_UNBOUND);
    AVSconnect_widget(p,"typein_real");
    p=AVSadd_float_parameter("Ymax",1.0,FLOAT_UNBOUND,FLOAT_UNBOUND);
    AVSconnect_widget(p,"typein_real");

    p=AVSadd_float_parameter("Zmin",-1.0,FLOAT_UNBOUND,FLOAT_UNBOUND);
    AVSconnect_widget(p,"typein_real");
    p=AVSadd_float_parameter("Zmax",1.0,FLOAT_UNBOUND,FLOAT_UNBOUND);
    AVSconnect_widget(p,"typein_real");

    p=AVSadd_parameter("dummy2","string","Output Grid Resolution","","");
    AVSconnect_widget(p,"text");
    AVSadd_parameter_prop(p,"width","integer",4);
    
    p=AVSadd_parameter("Xsamples","integer",20,2,1000);
    AVSconnect_widget(p,"idial");
    p=AVSadd_parameter("Ysamples","integer",20,2,1000);
    AVSconnect_widget(p,"idial");
    p=AVSadd_parameter("Zsamples","integer",20,2,1000);
    AVSconnect_widget(p,"idial");

    p=AVSadd_parameter("dummy3","string","Algorithmic Parameters","","");
    AVSconnect_widget(p,"text");
    AVSadd_parameter_prop(p,"width","integer",4);
    
    p=AVSadd_float_parameter("Search Radius",1.0,0.0,100000.0);
    AVSconnect_widget(p,"dial");
    p=AVSadd_float_parameter("Weight Function",1.0,0.0,100.0);
    AVSconnect_widget(p,"dial");
    p=AVSadd_float_parameter("Undefined Value",-1.0,FLOAT_UNBOUND,FLOAT_UNBOUND);
    AVSconnect_widget(p,"typein_real");

    AVSset_compute_proc(trivar_p);
}

trivar_p(inf,outf,dummy1,xmin,xmax,ymin,ymax,zmin,zmax,
	dummy2,xs,ys,zs,dummy3,sr,weight_function,undef_value)
AVSfield_float *inf,**outf;
char *dummy1,*dummy2,*dummy3;
float *xmin,*xmax,*ymin,*ymax,*sr,*weight_function,*zmin,*zmax,*undef_value;
int xs,ys,zs;
{
    AVSfield_float template;
    int dims[3];
    int trivar_work_routine();

    /**
      * check for following error conditions:
      * 1. field data dimensions greater than one
      *
    **/

    if (inf->ndim > 1)
    {
	AVSwarning(" Only 1-d scatter fields are presently allowed.");
	return(0);
    }

    memset((char *)&template,0,sizeof(AVSfield_float));
    template.ndim = 3;  /* take 1-d scatter data, interpolate in 3-d */
    template.nspace = 3;
    template.veclen = 1;
    template.type = AVS_TYPE_REAL;
    template.uniform = RECTILINEAR;
    template.size = sizeof(float);
    dims[0] = xs;
    dims[1] = ys;
    dims[2] = zs;
    if (*outf)
	AVSfield_free(*outf);
    *outf = (AVSfield_float *)AVSfield_alloc(&template,dims);
    if (*outf == NULL)
    {
	AVSwarning(" Output Field malloc error.");
	return(0);
    }
    
    if (trivar_work_routine(inf,*outf,xmin,xmax,ymin,ymax,zmin,zmax,
		   xs,ys,zs,sr,weight_function,undef_value))
	return(1);
    else
	return(0);
}

int
trivar_work_routine(inf,outf,xmin,xmax,ymin,ymax,zmin,zmax,
		    xs,ys,zs,search_radius,weight_function,undef_value)
AVSfield_float *inf,*outf;
float *xmin,*xmax,*ymin,*ymax,*search_radius,*weight_function,*zmin,*zmax,*undef_value;
int xs,ys,zs;
{
    extern double sqrt(),pow();
    int i,j,l,k,d,m;
    int inpoints;
    float Z,dZ,X,dX,Y,dY,*outdata,*indata;
    float *x,*xg,*y,*yg,*z,*zg;
    float *xp,*yp,*zp;
    float *diffs_array;
    int ip;
    double percent,dp;

    inpoints = inf->dimensions[0];
    
    /**
      * now, we will evaluate the function at the desired number
      * of locations.
    **/
    X = *xmin;
    dX = (*xmax - *xmin)/(double)(xs - 1);
    Y = *ymin;
    dY = (*ymax - *ymin)/(double)(ys - 1);
    Z = *zmin;
    dZ = (*zmax - *zmin)/(double)(zs - 1);

    xp = outf->points;
    yp = xp + xs;
    zp = yp + ys;

    /* load the coordinate values for the output grid. */
    Z = *zmin;
    for (l=0;l<zs;l++,Z+=dZ)
	*zp++ = Z;

    Y = *ymin;
    for (j=0;j<ys;j++,Y += dY)
	*yp++ = Y;
    
    X = *xmin;
    for (i=0;i<xs;i++,X += dX)
	*xp++ = X;

    inpoints = inf->dimensions[0];
    xg = outf->points;
    yg = xg + xs;
    zg = yg + ys;
    x = inf->points;
    y = x + inpoints;
    z = y + inpoints;

    indata = inf->data;
    outdata = outf->data;

    /** the following is for the status widget in AVS 3 or greater **/
    percent = 0.;
    dp = 100.0/(double)zs;
    ip = 0;

    d=0; 

    diffs_array = (float *)malloc(sizeof(float)*inpoints*3);
    if (diffs_array == NULL)
    {
	AVSerror(" Can't malloc enough space for work area. ");
	return(0);
    }

    for (m=0;m<zs;m++)
    {
	for (k=0;k<ys;k++)  /* loop on logical Y values in output field. */
	{
	    for (j=0;j<xs;j++,d++) /* loop on logical X values in output field.*/
	    {
		compute_value(j,k,m,d,x,y,z,xg,yg,zg,indata,outdata,inpoints,search_radius,undef_value,weight_function,diffs_array);
	    }
	}
	/** the following is for the status widget in AVS 3 or greater **/
	AVSmodule_status("Trivar: processing...",ip);
	ip = (int)percent;
	percent += dp;
    }


/* IAC CODE CHANGE :     free((char *)diffs_array); */
     free(diffs_array);
    return(1);
}

compute_value(j,k,m,d,x,y,z,xg,yg,zg,indata,outdata,inpoints,sr,undef_value,
	      weight_function,diffs_array)
float *indata,*outdata,*xg,*yg,*zg,*x,*y,*z,*weight_function,*sr;
float *undef_value,*diffs_array;
int j,k,m,inpoints,d;
{
    /**
      * in this routine, we compute the value at the output grid location
      * which has the coordinates of (xg[j],yg[k],zg[m]) given a list of
      * input coordintes x[],y[],z[] along with the various parameters.
      *
      * the value at an output grid location is computed thus:
      *   output coord = sum(1/(distances[i]**weight_function)*data[i]) /
      *                  sum(1/(distances[i]**weight_function)
      * for all distances[i] less than the user defined search radius.
      *
      * the distances[i] are computed as:
      *    sqrt((xg[j]-x[i])**2+(yg[k]-y[i])**2+(zg[m]-z[i])**2)
      *
      * this entire process has been converted to a series of vector
      * operations where the vector ops have been written in fortran
      * so as to really sing on hardware which has vector units, or
      * to take advantage of "super scalar" modes on CPU's.
      *
      * the meaning of the diffs_array changes throughout this routine,
      * but it is basically a work area.
    **/
    float denom,numer;
    int imin;
    float sr2;
    int count;
    float threshold = 1e-5;
    int find_min();
    int outcount;

    sr2 = *sr * *sr;

    count = inpoints;

#ifdef SUN
    vector_scalar_diff_(xg+j,x,diffs_array,&count);
    vector_square_(diffs_array,&count);
    
    vector_scalar_diff_(yg+k,y,diffs_array+count,&count);
    vector_square_(diffs_array+count,&count);
    
    vector_scalar_diff_(zg+m,z,diffs_array+(count<<1),&count);
    vector_square_(diffs_array+(count<<1),&count);
    
    vector_3add_(diffs_array,diffs_array+count,diffs_array+(count<<1),&count);
    
#elif TITAN
    VECTOR_SCALAR_DIFF(xg+j,x,diffs_array,&count);
    VECTOR_SQUARE(diffs_array,&count);
    
    VECTOR_SCALAR_DIFF(yg+k,y,diffs_array+count,&count);
    VECTOR_SQUARE(diffs_array+count,&count);
    
    VECTOR_SCALAR_DIFF(zg+m,z,diffs_array+(count<<1),&count);
    VECTOR_SQUARE(diffs_array+(count<<1),&count);
    
    VECTOR_3ADD(diffs_array,diffs_array+count,diffs_array+(count<<1),&count);
#endif    

    imin = find_min(diffs_array,&count,&threshold);
    if (imin != -1)
    {
	outdata[d] = indata[imin];
	return;
    }

#ifdef SUN
    vector_threshold_(diffs_array,&count,&sr2);
#elif TITAN
    VECTOR_THRESHOLD(diffs_array,&count,&sr2);
#endif

    pick_out_nonzeros(diffs_array,diffs_array+count,inpoints,&outcount,diffs_array+(count<<1),indata);
    if (outcount == 0)
    {
	outdata[d] = *undef_value;
	return;
    }

#ifdef SUN
    vector_power_inverse_(diffs_array+count,&outcount,weight_function);
    vector_scale_sum_(diffs_array+count,&outcount,diffs_array+(count<<1),&numer);
    vector_sum_(diffs_array+count,&outcount,&denom);
#elif TITAN
    VECTOR_POWER_INVERSE(diffs_array+count,&outcount,weight_function);
    VECTOR_SCALE_SUM(diffs_array+count,&outcount,diffs_array+(count<<1),&numer);
    VECTOR_SUM(diffs_array+count,&outcount,&denom);
#endif
    outdata[d] = numer/denom;
}

int
find_min(d,n,t)
float *d,*t;
int *n;
{
    register float *f;
    int i,nn,rvalue= -1;

    f = d;
    nn = *n;
    for (i=0;i<nn;i++,f++)
	if (*f < *t)
	{
	    rvalue = i;
	    break;
	}
    return(rvalue);
}

pick_out_nonzeros(src,dst,incount,outcount,dst_data,indata)
float *src,*dst,*dst_data,*indata;
int incount,*outcount;
{
    register float *s,*d,zero=0.;
    int i,out=0;

    s = src;
    d = dst;
    
    for (i=0;i<incount;i++,s++)
    {
	if (*s != zero)
	{
	    *d++ = *s;
	    *(dst_data+out) = *(indata+i);
	    out++;

	}
    }
    *outcount = out;
}
