/****************************************************************************
                  INTERNATIONAL AVS CENTER
	(This disclaimer must remain at the top of all files)

WARRANTY DISCLAIMER

This module and the files associated with it are distributed free of charge.
It is placed in the public domain and permission is granted for anyone to use,
duplicate, modify, and redistribute it unless otherwise noted.  Some modules
may be copyrighted.  You agree to abide by the conditions also included in
the AVS Licensing Agreement, version 1.0, located in the main module
directory located at the International AVS Center ftp site and to include
the AVS Licensing Agreement when you distribute any files downloaded from 
that site.

The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module provide absolutely
NO WARRANTY OF ANY KIND with respect to this software.  The entire risk as to
the quality and performance of this software is with the user.  IN NO EVENT
WILL The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module BE LIABLE TO
ANYONE FOR ANY DAMAGES ARISING FROM THE USE OF THIS SOFTWARE, INCLUDING,
WITHOUT LIMITATION, DAMAGES RESULTING FROM LOST DATA OR LOST PROFITS, OR ANY
SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES.

This AVS module and associated files are public domain software unless
otherwise noted.  Permission is hereby granted to do whatever you like with
it, subject to the conditions that may exist in copyrighted materials. Should
you wish to make a contribution toward the improvement, modification, or
general performance of this module, please send us your comments:  why you
liked or disliked it, how you use it, and most important, how it helps your
work. We will receive your comments at avs@ncsc.org.

Please send AVS module bug reports to avs@ncsc.org.

******************************************************************************/
/*
 * ucd_cell_arrows
 *
 * draw geometric representation of UCD input
 * showing CELL data, CELL Material ID, or Cell Number Sequence.
 *
 * This is a special ucd mapper module designed to make
 * representations based on CELL based values rather than
 * the more typical NODE based values. If you have CELL based
 * values, you should consider the ucd_cell_to_node module.
 *
 * If your cell data is 2 or 3-vector,
 * this module will draw arrows from the cell centroid.
 *
 * Author: I. Curington, AVS Inc, UK
 *
 * Revision:
 * 31 March 92  Original (cell data)
 *
 */

/*-----------------------------------------------------*
 *                                                     *
 *     ****  ucd_cell_arrows   module  ****            *
 *                                                     *
 *                                                     *
 *-----------------------------------------------------*/

#include <stdio.h>


#include <avs/avs.h>
/* IAC CODE CHANGE : #include <avs/avs_math.h> */
#include <avs_math.h>
#include <avs/geom.h>
#include <avs/ucd_defs.h>


/* =============== compute routine ============== */

static ucd_cell_arrows (input, output, scale )

  UCD_structure *input;
  GEOMedit_list **output;
  float         *scale;
{
    char model_name[80];
    GEOMobj *obj, *lobj;
    float *xc, *yc, *zc, *cell_data;
    int cell, i, name_flag, util_flag, data_len, cell_len, node_len,
        n, num_nodes, num_cells;

    /* get general sizes */
    if (!UCDstructure_get_header (input, model_name, &data_len, &name_flag, 
			        &num_cells, &cell_len, &num_nodes, &node_len, 
			        &util_flag))
    {
      AVSerror ("Error in ucd_cell_arrows: can't get header.\n"); 
      return (0);
    }

    if ( cell_len <= 1 )
    {
      AVSerror ("Error ucd_cell_arrows: no 3-vector or 2-vector cell data\n"); 
      return (0);
    }

    /* find out where the nodes are */
    if (!UCDstructure_get_node_positions (input, &xc, &yc, &zc))
    {
      AVSerror ("Error in ucd_cell_arrows: can't get node coordinates.\n"); 
      return (0);
    }

    /* get a pointer to where the cell data is stored */
    if ( cell_len )
    {
        if (!UCDstructure_get_cell_data (input, &cell_data))
        {
          AVSerror ("Error in ucd_cell_arrows: can't get cell data.\n"); 
          return (0);
        }
    }

    /* establish the geometry output object */
    *output = (GEOMedit_list *)GEOMinit_edit_list(*output);
    obj    =  GEOMcreate_obj (GEOM_POLYTRI, NULL);

    /* draw the geometry from the cell list */
    display_arrows (input, num_cells, obj, 
                     xc, yc, zc, cell_len, cell_data, *scale );


    /* complete the geometry to output port process */
    GEOMedit_geometry (*output, "ucd_cell_arrows", obj);
    GEOMdestroy_obj (obj);
    GEOMedit_transform_mode (*output, "ucd_cell_arrows", "parent");

    return(1);
}


/*-----------------------------------------------------*
 *                                                     *
 *          ****  display_arrows  ****                 *
 *                                                     *
 *-----------------------------------------------------*/

static display_arrows (input, num_cells, obj, 
		 xc, yc, zc, cell_len, cell_data, scale )

  UCD_structure *input;   /* ucd input port */
  int num_cells;          /* how many cells */
  GEOMobj *obj;           /* add polygons to this object */
  float *xc, *yc, *zc;    /* node position arrays */
  int cell_len;           /* cell vector length */
  float *cell_data;       /* data to use for arrows */
  float scale;            /* length of arrows */

{
      char ctype[40];
      float vector[3], cx, cy, cz;

      int cell, cell_id, mat_id, cell_type, me_flags, *node_list,
          node, face, v, numv, num_fv, 
          num_nodes, i, k, j;

      /*********************************************/
      /**** loop over ALL cell  surfaces        ****/
      /*********************************************/
    
      for (cell = 0; cell < num_cells; cell++)
      {
        /* query cell */
        UCDcell_get_information (input, cell, &cell_id, ctype, &mat_id,
    			     &cell_type, &me_flags, &node_list);
    
        /* identify cell topology */
        num_nodes = UCD_num_nodes[cell_type];
        cx = cy = cz = 0.0;
    
        /* loop over cell vertex list */
        for (j = 0; j < num_nodes; j++)
        {
                v = node_list[j];
                cx += xc[v];
                cy += yc[v];
                cz += zc[v];
        }                          /* end of vertex loop */

        cx /= (float) num_nodes;
        cy /= (float) num_nodes;
        cz /= (float) num_nodes;

        vector[0] = cell_data[            cell ];
        vector[1] = cell_data[ num_cells + cell ];
        if ( cell_len <= 2 )
            vector[2] = 0.0;
        else
            vector[2] = cell_data[ 2*num_cells + cell ];

/*
printf("cell_len %d\n",cell_len);
printf("vector %f %f %f\n",vector[0],vector[1],vector[2]);
printf("scale %f\n",scale);
*/

        create_arrow (obj, cx, cy, cz, vector, scale);

      }  /* end of cell loop */
    
}

/*-----------------------------------------------------*
 *                                                     *
 *        ****  create_arrow  ****                     *
 *                                                     *
 *-----------------------------------------------------*/

static create_arrow (obj, x, y, z, v1, scale)
            GEOMobj *obj;            /* output geom object */
            float         x, y, z;   /* arrow position */
            float         v1[3];     /* vector data */
            float         scale;     /* length scale */
{
  double sqrt();

  float mag, s1, x1, y1, z1, nx, ny, nz, s, vt[3], v2[3], v3[3], vx, vy, vz;
  float verts[20][3], vcolors[20][3], r, g, b;
  int i, n, m, index;
  static float arrow_scale = 0.4;

  index = -1;
  nx = scale * v1[0];
  ny = scale * v1[1];
  nz = scale * v1[2];
  mag = sqrt(nx * nx + ny * ny + nz * nz);

  if (mag == 0.0) {
    n = 0;
    verts[n][0] = x;
    verts[n][1] = y;
    verts[n++][2] = z;
    verts[n][0] = x;
    verts[n][1] = y;
    verts[n++][2] = z;
    }
  else {
    s = 0.9;
    s1 = arrow_scale * (1.0 - s) * mag;
    vx = x + s * nx;
    vy = y + s * ny;
    vz = z + s * nz;

    /*  compute v2[] and v3[], which will be 
        the sides of the arrow head.          */

    vt[0] = -v1[1];
    vt[1] = v1[0];
    vt[2] = 0.0;

    if ((vt[0] == 0.0) && (vt[1] == 0.0))
      vt[0] = 1.0;

    vcross_prod (v1, vt, v2);
    vcross_prod (v1, v2, v3);

    vnorm (v2);
    vnorm (v3);

    n = 0;
    verts[n][0] = x, verts[n][1] = y, verts[n++][2] = z;
    verts[n][0] = x1 = x + nx; 
    verts[n][1] = y1 = y + ny;
    verts[n++][2] = z1 = z + nz;

    verts[n][0] = x1,  verts[n][1] = y1,  verts[n++][2] = z1;  
    verts[n][0] = vx + s1 * v2[0];
    verts[n][1] = vy + s1 * v2[1]; 
    verts[n++][2] = vz + s1 * v2[2];

    verts[n][0] = x1,  verts[n][1] = y1,  verts[n++][2] = z1;  
    verts[n][0] = vx + s1 * v3[0]; 
    verts[n][1] = vy + s1 * v3[1];
    verts[n++][2] = vz + s1 * v3[2];

    verts[n][0] = x1,  verts[n][1] = y1,  verts[n++][2] = z1;  
    verts[n][0] = vx - s1 * v2[0];  
    verts[n][1] = vy - s1 * v2[1];
    verts[n++][2] = vz - s1 * v2[2]; 

    verts[n][0] = x1,  verts[n][1] = y1,  verts[n++][2] = z1;  
    verts[n][0] = vx - s1 * v3[0]; 
    verts[n][1] = vy - s1 * v3[1];
    verts[n++][2] = vz - s1 * v3[2]; 


    /*  form the base of the arrow head.  */

    for (m = n, i = 3; i < m; i += 2) {
      verts[n][0] = verts[i][0];  
      verts[n][1] = verts[i][1];  
      verts[n++][2] = verts[i][2];  

      if (i == 9) {
        verts[n][0] = verts[3][0];  
        verts[n][1] = verts[3][1];  
        verts[n++][2] = verts[3][2];  
        }
      else {
        verts[n][0] = verts[i + 2][0];  
        verts[n][1] = verts[i + 2][1];  
        verts[n++][2] = verts[i + 2][2];  
        }
      }
    }

    GEOMadd_disjoint_line (obj, verts, NULL, n, GEOM_COPY_DATA);
}

/* normalize the length of a 3-vector */
static vnorm (v)
        float v[];
{
  double mag, sqrt();

  mag = sqrt(v[0] * v[0] + v[1] * v[1] + v[2] * v[2]);

  if (mag == 0.0) {
    v[0] = 0.0;
    v[1] = 0.0;
    v[2] = 0.0;
    return (0);
    }
  else {
    v[0] = v[0] / mag;
    v[1] = v[1] / mag;
    v[2] = v[2] / mag;
    }

  return (1);
}

/* 3-vector cross product */
static vcross_prod (v1,  v2,  v3)
             float *v1, *v2, *v3;
{
  v3[0] = v1[1] * v2[2] - v1[2] * v2[1];
  v3[1] = v1[2] * v2[0] - v1[0] * v2[2];
  v3[2] = v1[0] * v2[1] - v1[1] * v2[0];
}

/*-----------------------------------------------------*
 *                                                     *
 *        ****  ucd_cell_arrows_desc  ****               *
 *                                                     *
 *-----------------------------------------------------*/
ucd_cell_arrows_desc()
{
  int ucd_cell_arrows(), param;

  /* name on the box */
  AVSset_module_name ("ucd cell arrows", MODULE_MAPPER);

  /* input ucd structure to draw */
  AVScreate_input_port ("Input", "ucd", REQUIRED);

  /* output geomerty to draw */
  AVScreate_output_port ("ucd_cell_arrows", "geom");

  param = AVSadd_float_parameter("Scale", 1.0, FLOAT_UNBOUND, FLOAT_UNBOUND);

  /* routine pointers */
  AVSset_compute_proc (ucd_cell_arrows);

}  /* end of description routine */

AVSinit_modules()
{
        AVSmodule_from_desc(ucd_cell_arrows_desc);
}

