/****************************************************************************
                  INTERNATIONAL AVS CENTER
	(This disclaimer must remain at the top of all files)

WARRANTY DISCLAIMER

This module and the files associated with it are distributed free of charge.
It is placed in the public domain and permission is granted for anyone to use,
duplicate, modify, and redistribute it unless otherwise noted.  Some modules
may be copyrighted.  You agree to abide by the conditions also included in
the AVS Licensing Agreement, version 1.0, located in the main module
directory located at the International AVS Center ftp site and to include
the AVS Licensing Agreement when you distribute any files downloaded from 
that site.

The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module provide absolutely
NO WARRANTY OF ANY KIND with respect to this software.  The entire risk as to
the quality and performance of this software is with the user.  IN NO EVENT
WILL The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module BE LIABLE TO
ANYONE FOR ANY DAMAGES ARISING FROM THE USE OF THIS SOFTWARE, INCLUDING,
WITHOUT LIMITATION, DAMAGES RESULTING FROM LOST DATA OR LOST PROFITS, OR ANY
SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES.

This AVS module and associated files are public domain software unless
otherwise noted.  Permission is hereby granted to do whatever you like with
it, subject to the conditions that may exist in copyrighted materials. Should
you wish to make a contribution toward the improvement, modification, or
general performance of this module, please send us your comments:  why you
liked or disliked it, how you use it, and most important, how it helps your
work. We will receive your comments at avs@ncsc.org.

Please send AVS module bug reports to avs@ncsc.org.

******************************************************************************/
/*
 * ucd_cell_cones
 *
 * draw geometric representation of UCD input
 * showing CELL data
 *
 * This is a special ucd mapper module designed to make
 * representations based on CELL based values rather than
 * the more typical NODE based values. If you have CELL based
 * values, you should consider the ucd_cell_to_node module.
 *
 * If your cell data is 2 or 3-vector,
 * this module will draw cones from the cell centroid.
 *
 * Author: I. Curington, AVS Inc, UK
 *
 * Note: Part of the cone code was lifted from another
 * module in the NCSC AVS library, "cone", from
 *         Cone.c adapted from Stardent stick (tube) module
 *         by S.H. Lamson at GE CR&D 2/90
 *
 * Revision:
 * 3 April 92  Original (cell data)
 * 6 April 92  added normalize mode
 *
 */

/*-----------------------------------------------------*
 *                                                     *
 *     ****  ucd_cell_cones module  ****               *
 *                                                     *
 *                                                     *
 *-----------------------------------------------------*/

#include <stdio.h>
/* IAC CODE CHANGE : #include <math.h> */
#include <avs_math.h>

#include <avs/avs.h>
#include <avs/geom.h>
#include <avs/ucd_defs.h>


/* =============== compute routine ============== */

static ucd_cell_cones (input, output, scale, width, num_steps, norm )

  UCD_structure *input;
  GEOMedit_list **output;
  float         *scale;
  float         *width;
  int           num_steps;
  int           norm;
{
    char model_name[80];
    GEOMobj *obj, *lobj;
    float *xc, *yc, *zc, *cell_data;
    int cell, i, name_flag, util_flag, data_len, cell_len, node_len,
        n, num_nodes, num_cells;

    /* get general sizes */
    if (!UCDstructure_get_header (input, model_name, &data_len, &name_flag, 
			        &num_cells, &cell_len, &num_nodes, &node_len, 
			        &util_flag))
    {
      AVSerror ("Error in ucd_cell_cones: can't get header.\n"); 
      return (0);
    }

    if ( cell_len <= 1 )
    {
      AVSerror ("Error ucd_cell_cones: no 3-vector or 2-vector cell data\n"); 
      return (0);
    }

    /* find out where the nodes are */
    if (!UCDstructure_get_node_positions (input, &xc, &yc, &zc))
    {
      AVSerror ("Error in ucd_cell_cones: can't get node coordinates.\n"); 
      return (0);
    }

    /* get a pointer to where the cell data is stored */
    if ( cell_len )
    {
        if (!UCDstructure_get_cell_data (input, &cell_data))
        {
          AVSerror ("Error in ucd_cell_cones: can't get cell data.\n"); 
          return (0);
        }
    }

    /* establish the geometry output object */
    *output = (GEOMedit_list *)GEOMinit_edit_list(*output);
    obj    =  GEOMcreate_obj (GEOM_POLYTRI, NULL);

    /* draw the geometry from the cell list */
    display_cones (input, num_cells, obj, 
                     xc, yc, zc, cell_len, cell_data,
                     *scale, *width, num_steps+1, norm );


    /* complete the geometry to output port process */
    GEOMedit_geometry (*output, "ucd_cell_cones", obj);
    GEOMdestroy_obj (obj);
    GEOMedit_transform_mode (*output, "ucd_cell_cones", "parent");

    return(1);
}


/*-----------------------------------------------------*
 *                                                     *
 *          ****  display_cones  ****                  *
 *                                                     *
 *-----------------------------------------------------*/

static display_cones (input, num_cells, obj, 
		 xc, yc, zc, cell_len, cell_data,
                 scale, width, num_steps, norm )

  UCD_structure *input;   /* ucd input port */
  int num_cells;          /* how many cells */
  GEOMobj *obj;           /* add polygons to this object */
  float *xc, *yc, *zc;    /* node position arrays */
  int cell_len;           /* cell vector length */
  float *cell_data;       /* data to use for cones */
  float scale;            /* length of cones */
  float width;            /* base width of cones */
  int   num_steps;        /* cone polygon res */
  int   norm;             /* normalize cone length */

{
      char ctype[40];
      float vector[3], cx, cy, cz;

      int cell, cell_id, mat_id, cell_type, me_flags, *node_list,
          node, face, v, numv, num_fv, 
          num_nodes, i, k, j;

      /*********************************************/
      /**** loop over ALL cell  surfaces        ****/
      /*********************************************/
    
      for (cell = 0; cell < num_cells; cell++)
      {
        /* query cell */
        UCDcell_get_information (input, cell, &cell_id, ctype, &mat_id,
    			     &cell_type, &me_flags, &node_list);
    
        /* identify cell topology */
        num_nodes = UCD_num_nodes[cell_type];
        cx = cy = cz = 0.0;
    
        /* loop over cell vertex list to find centroid */
        for (j = 0; j < num_nodes; j++)
        {
                v = node_list[j];
                cx += xc[v];
                cy += yc[v];
                cz += zc[v];
        }                          /* end of vertex loop */

        cx /= (float) num_nodes;
        cy /= (float) num_nodes;
        cz /= (float) num_nodes;

        vector[0] = cell_data[            cell ];
        vector[1] = cell_data[ num_cells + cell ];
        if ( cell_len <= 2 )
            vector[2] = 0.0;
        else
            vector[2] = cell_data[ 2*num_cells + cell ];

        create_cone (obj, cx, cy, cz, vector,
                     scale, width, num_steps, norm);

      }  /* end of cell loop */
    
}

/*-----------------------------------------------------*
 *                                                     *
 *        ****  create_cone   ****                     *
 *                                                     *
 *-----------------------------------------------------*/

static create_cone (obj, x, y, z, v1, scale, width, num_steps, norm)
            GEOMobj *obj;            /* output geom object */
            float         x, y, z;   /* arrow position */
            float         v1[3];     /* vector data */
            float         scale;     /* length scale */
            float         width;     /* width scale */
            int           num_steps; /* cone polygon res */
            int           norm;      /* cone normalize length */
{
  float mag, s1, x1, y1, z1, nx, ny, nz, s, vt[3], v2[3], v3[3], vx, vy, vz;
  float verts[20][3], vcolors[20][3], r, g, b;
  int i, n, m, index;
  float	*pt_list, *nl_list;
  int   num_vertex;
  float    cos_array[64];
  float    sin_array[64];

  /* set up the cone offset array */
  for(i=0;i<num_steps;i++)
  {   cos_array[i] = cos(2.0*M_PI*(float)i/(float)(num_steps-1));
      sin_array[i] = sin(2.0*M_PI*(float)i/(float)(num_steps-1));
  }

   pt_list = nl_list = NULL;

   num_vertex = 2 * num_steps;

   pt_list = (float *)malloc(3*num_vertex*sizeof(float));
   nl_list = (float *)malloc(3*num_vertex*sizeof(float));

   if ( norm )
   {
	mag = sqrt(v1[0]*v1[0]+v1[1]*v1[1]+v1[2]*v1[2]);
	if (mag > 0.0 )
        { 
	    v1[0] /= mag;
	    v1[1] /= mag;
 	    v1[2] /= mag;
        }
   }

   verts[0][0] = x + v1[0] * scale;
   verts[0][1] = y + v1[1] * scale;
   verts[0][2] = z + v1[2] * scale;
   verts[1][0] = x;
   verts[1][1] = y;
   verts[1][2] = z;

   cone_disjline(pt_list, nl_list, verts, 1, scale, width,
                 num_steps, cos_array, sin_array );

   GEOMadd_polytriangle(obj, pt_list, nl_list,
				NULL, num_vertex,
				GEOM_COPY_DATA);

/* IAC CODE CHANGE :    if (pt_list) free(pt_list); */
   if (pt_list)  FREE_LOCAL(pt_list) ;
/* IAC CODE CHANGE :    if (nl_list) free(nl_list); */
   if (nl_list)  FREE_LOCAL(nl_list) ;
}
/*-----------------------------------------------------*/

static float    xp[3] = { 1.0, 0.0, 0.0 };
static float    yp[3] = { 0.0, 1.0, 0.0 };
static float    zp[3] = { 0.0, 0.0, 1.0 };

/*-----------------------------------------------------*/

cone_disjline(pt_list,nl_list,src,n,scale,width,num_steps,
                 cos_array, sin_array )
float	pt_list[][3];
float	nl_list[][3];
float	src[][3];
int	n;
float   scale;
float   width;
int     num_steps;
float   *cos_array, *sin_array;

{   int		i, j;
    float       x[3], y[3], z[3];
    float       l, ct, st;
    int		index, save_index;
    float local_offset;
    float thresh;

    index = save_index = 0;
    thresh = 0.000001;

    for(i=0;i<n;i+=2)
    {   x[0] = src[i+1][0] - src[i][0];
	x[1] = src[i+1][1] - src[i][1];
	x[2] = src[i+1][2] - src[i][2];

	local_offset = width;
	l = sqrt(x[0]*x[0]+x[1]*x[1]+x[2]*x[2]);
	if (l < thresh) local_offset = 0.0;
	x[0] /= l;
	x[1] /= l;
	x[2] /= l;

	z[0] = x[1]*xp[2] - x[2]*xp[1];
	z[1] = x[2]*xp[0] - x[0]*xp[2];
	z[2] = x[0]*xp[1] - x[1]*xp[0];

	st = sqrt(z[0]*z[0]+z[1]*z[1]+z[2]*z[2]);

	if(st==0.0)
	{   y[0] = yp[0];
	    y[1] = yp[1];
	    y[2] = yp[2];
	    
	    z[0] = zp[0];
	    z[1] = zp[1];
	    z[2] = zp[2];
	}
	else
	{   
	    ct = x[0]*xp[0]+x[1]*xp[1]+x[2]*xp[2];

	    z[0] /= st;
	    z[1] /= st;
	    z[2] /= st;
 
	    y[0] = (xp[0] - x[0]*ct)/st;
	    y[1] = (xp[1] - x[1]*ct)/st;
	    y[2] = (xp[2] - x[2]*ct)/st;

	}

	if(index != 0)
	{   save_index = index;
	    index += 2;
	}
	for(j=0;j<num_steps;j++)
	{   nl_list[index][0] = nl_list[index+1][0] = cos_array[j]*y[0] + sin_array[j]*z[0];
	    nl_list[index][1] = nl_list[index+1][1] = cos_array[j]*y[1] + sin_array[j]*z[1];
	    nl_list[index][2] = nl_list[index+1][2] = cos_array[j]*y[2] + sin_array[j]*z[2];
	    
	    pt_list[index][0] = src[i][0] ;
	    pt_list[index][1] = src[i][1] ;
	    pt_list[index][2] = src[i][2] ;
	    
	    pt_list[index+1][0] = src[i+1][0] + local_offset*nl_list[index][0];
	    pt_list[index+1][1] = src[i+1][1] + local_offset*nl_list[index][1];
	    pt_list[index+1][2] = src[i+1][2] + local_offset*nl_list[index][2];
	    
	    
	    index += 2;
	}
	if(save_index != 0)
	{
	    nl_list[save_index][0] = nl_list[save_index-1][0];
	    nl_list[save_index][1] = nl_list[save_index-1][1];
	    nl_list[save_index][2] = nl_list[save_index-1][2];
	    pt_list[save_index][0] = pt_list[save_index-1][0];
	    pt_list[save_index][1] = pt_list[save_index-1][1];
	    pt_list[save_index][2] = pt_list[save_index-1][2];

	    nl_list[save_index+1][0] = nl_list[save_index+2][0];
	    nl_list[save_index+1][1] = nl_list[save_index+2][1];
	    nl_list[save_index+1][2] = nl_list[save_index+2][2];
	    pt_list[save_index+1][0] = pt_list[save_index+2][0];
	    pt_list[save_index+1][1] = pt_list[save_index+2][1];
	    pt_list[save_index+1][2] = pt_list[save_index+2][2];

	}
    }

}


/*-----------------------------------------------------*
 *                                                     *
 *       ****  ucd_cell_cones_desc  ****               *
 *                                                     *
 *-----------------------------------------------------*/
ucd_cell_cones_desc()
{
  int ucd_cell_cones(), param;

  /* name on the box */
  AVSset_module_name ("ucd cell cones", MODULE_MAPPER);

  /* input ucd structure to draw */
  AVScreate_input_port ("Input", "ucd", REQUIRED);

  /* output geomerty to draw */
  AVScreate_output_port ("ucd_cell_cones", "geom");

  /* controls */
  param = AVSadd_float_parameter("Scale", 1.0, FLOAT_UNBOUND, FLOAT_UNBOUND);
  param = AVSadd_float_parameter("Width", 0.2, FLOAT_UNBOUND, FLOAT_UNBOUND);
  param = AVSadd_parameter("Subdivision","integer", 6, 2, 16);
  AVSconnect_widget(param,"islider");
  param = AVSadd_parameter("Normalize","boolean", 0, 0, 1);

  /* routine pointers */
  AVSset_compute_proc (ucd_cell_cones);

}  /* end of description routine */

AVSinit_modules()
{
        AVSmodule_from_desc(ucd_cell_cones_desc);
}

