/* NOTE:  THIS MODULE AND SOURCE CODE IS FOR USE 
   WITH THE AVS SOFTWARE ENVIRONMENT ONLY */
/*
 * ucd_cell_geom
 *
 * draw geometric representation of UCD input
 * showing CELL data, CELL Material ID, or Cell Number Sequence.
 *
 * This is a special ucd mapper module designed to make
 * representations based on CELL based values rather than
 * the more typical NODE based values. If you have CELL based
 * values, you should consider the ucd_cell_to_node module.
 *
 * This module colours each cell ONE color, without color interpolation,
 * showing a single scalar quantity. If your cell data is n-vector,
 * this module will only color the first one, so use ucd_extract to
 * get the one you want. It will color the cells based on
 * [1] the linear cell number, or cell_id tag. It maps this range onto the
 *     color input map.
 * [2] cell scalar data. Use the "upper" and "lower" colormap dials
 *     to set the color to value mapping.
 * [3] material id numbers. It maps the max material id to cover the 
 *     input color map range.
 *
 * This module also has a direct SHADED/WIRE_FRAME toggle control,
 * just to save time going to the viewer panel.
 *
 * This module is not like the standard ucd_to_geom,
 * in that it:
 * - does not cache information about the ucd structure
 *   in between invocations,
 * - does not have a color array input from ucd_legend,
 *   rather it colors directly with a colormap input,
 * - does not have "explode materials" mode
 * - does not have "shrink" function
 * - does not cache geometry for fast animation
 *
 *
 * Author: I. Curington, AVS Inc, UK
 *
 * Revision:
 * 15 March 92  Original (cell data)
 * 16 March 92  Added Material Id and Cell Id and Wire Frame
 * 15 March 93  Converted to cell interaction mode, C. Farmer, I. Curington
 *              Removed function statics
 *              Added includes
 *              Added upstream xform and picking ports
 *              Added distance dial, xform icon
 * 19 March 93  Added pick id code
 * 31 March 93  Worked on node colours, new mode added
 *              Selected Face Hi-lite
 *              hue controls
 *              drag model coords mode
 *  2 April 93  added remove interior mode
 *  6 April 93  choice list for node data
 * 29 April 93  added downsize dial
 *
 * 18 May   93  removed wireframe option,
 *              added xyz coord color option
 *              added face dip color option
 *              added dual edit list for speed
 *              with fast_update mode
 *              added smooth normal shading option
 * 26 Aug   93  fixed pointer indirection prob
 *
 */

/*-----------------------------------------------------*
 *                                                     *
 *       ****  ucd_cell_geom   module  ****            *
 *                                                     *
 *                                                     *
 *-----------------------------------------------------*/

#include <stdio.h>
#include <avs/avs.h>
#include <avs/geom.h>
#include <avs/ucd_defs.h>
#include <avs/colormap.h>
#include <avs/field.h>
#include <avs/udata.h>

/* allow, or comment out, as needed: */
/*****
#define DEBUG 1
*****/

/* =============== local declarations  ============== */
/* in AVS5, this should really be pulled from ucd_topo.h */

extern char *AVSstatic;

/* ucd cell type topology */
static int   tri_topo[]  = {3, 0, 1, 2}, 
  
             quad_topo[] = {4, 0, 1, 2, 3},

             tet_topo[]  = {3, 1, 2, 0, 
                            3, 2, 3, 0, 
                            3, 3, 1, 0, 
                            3, 1, 3, 2},

             pyr_topo[]  = {3, 0, 1, 2,
                            3, 0, 2, 3,
                            3, 0, 3, 4,
                            3, 4, 1, 0,
                            4, 1, 4, 3, 2},

             prism_topo[]= {3, 5, 4, 3,
                            3, 0, 1, 2,
                            4, 1, 4, 5, 2,
                            4, 1, 0, 3, 4,
                            4, 0, 2, 5, 3},

             hex_topo[]  = {4, 0, 1, 2, 3, 
                            4, 1, 5, 6, 2, 
                            4, 3, 2, 6, 7,
                            4, 0, 3, 7, 4, 
                            4, 0, 4, 5, 1, 
                            4, 4, 7, 6, 5};
 
static int *cell_topos[] = {0, 0, tri_topo, quad_topo, tet_topo, pyr_topo, 
                           prism_topo, hex_topo}, me_node_mask = 0xffffff;

static float *global_colors;   /* colour index table */
static int   global_cmap_size;

typedef struct   _UCD_Face_List {
        unsigned char         num_vtx;
        int                   vtx_index[4];
        int                   mat_id;
        int                   cell_id;
} UCD_Face_List;

typedef struct   _UCD_Line_List {
        int                   vtx_index[2];
        int                   mat_id;
        int                   cell_id;
} UCD_Line_List;


/* =============== compute routine ============== */

ucd_cell_geom (input,
               cmap_input,
               xform,
               xgeom,
               output,
               pick_result,
               method,
               distance,
               sphere_size,
               apply,
               mark_face,
               drag_line,
               drag_model,
               face_hue,
               sphere_hue,
               show_icon,
               remove_interior,
               /*** comp_select, ***/
               downsize,
               opt_normals )

  UCD_structure *input;       /* ucd input */
  AVScolormap   *cmap_input;  /* color for node-cell data */
  upstream_transform *xform;  /* transformations for moved geometry */
  upstream_geom *xgeom;       /* picking info, back from viewer */
  GEOMedit_list *output;      /* output geometry picture - icons */
  AVSfield_int  **pick_result;/* field with pick info */
  char          *method;      /* color meaning method */
  float         *distance;    /* normal distance dial */
  float         *sphere_size; /* marker size */
  int           apply;        /* signal to send data to output port */
  int           mark_face;    /* show selected - picked face in new colour */
  int           drag_line;    /* show drag line from pick to target */
  int           drag_model;   /* drag lines from model from pick to target */
  float         *face_hue;    /* marked face colour */
  float         *sphere_hue;  /* target sphere colour */
  int           show_icon;    /* show xform construction guide lines */
  int           remove_interior; /* control flag */
/*  char          *comp_select; */ /* node data component label choice */
  int           downsize;     /* display only a subset of model */
  int           opt_normals;  /* optimize surface normals */

{
    char model_name[80];
    GEOMobj *obj, *lobj, *sobj1, *sobj2;
    float *xc, *yc, *zc, *cell_data, *node_data;
    int cell, i, name_flag, util_flag, data_len, cell_len, node_len,
        n, num_nodes, num_cells, method_number;
    float         newmat[4][4], identity[4][4];  /* transformations */
    float         tmp_vec[3], extent[6], min_extent[3], max_extent[3];
    float         width, height, depth, maxdim;
    float         cx, cy, cz;
    float         select_icon[5][3]; /* local (moved) copy */
    float         target[3];
    int           dims[2];
    int           reset_xform;
    char          *outbuf, *errbuf;
    char          string[80];
    float         target_hue[3];
    float         target_sph_hue[3];
    int           select;
    int           fast_update;

    /************
     * Statics! *
     ************/
    static  float     model_icon[5][3];  /* picked point on object */
    static  float     picked_point[3];   /* picked vertex */
    static  int       picked_cell;       /* picked cell */
    static  int       picked_node;       /* picked node */
    static  int       have_reset_xform=0;/* history of reset event */
 
    static  int       faces_count;
    static  int       lines_count;
    static  int       edges_count;
    static  int       verts_count;
    static    UCD_Face_List      *face_list;
    static    UCD_Line_List      *line_list;
    static    UCD_Line_List      *edge_list;


    /*************
     * BODY      *
     ************/
 

    /* get method choice */
    method_number = AVSchoice_number("Method", method);

    /* set up the colour mapping table */
    global_colors = (float *)malloc(3 * sizeof(float) * cmap_input->size);
    global_cmap_size = cmap_input->size;

    for (i = 0; i < cmap_input->size; i++)
    FILTERhsv_to_rgb (&(global_colors[i*3+0]),
		      &(global_colors[i*3+1]),
		      &(global_colors[i*3+2]),
                      cmap_input->hue[i], cmap_input->saturation[i],
		      cmap_input->value[i]);

    FILTERhsv_to_rgb (&(target_hue[0]),&(target_hue[1]),&(target_hue[2]),
                      *face_hue, 1.0, 1.0 );

    FILTERhsv_to_rgb (&(target_sph_hue[0]),
                      &(target_sph_hue[1]),
                      &(target_sph_hue[2]),
                      *sphere_hue, 1.0, 1.0 );


    /* get general sizes */
    if (!UCDstructure_get_header (input, model_name, &data_len, &name_flag, 
			        &num_cells, &cell_len, &num_nodes, &node_len, 
			        &util_flag))
    {
      AVSerror ("Error in ucd_cell_geom: can't get header.\n"); 
      return (0);
    }

    /* check if material ids are present */
    if ( method_number==3 && !(name_flag & UCD_MATERIAL_IDS) )
         method_number=0; /* else set to zero, default */

    /* find out where the nodes are */
    if (!UCDstructure_get_node_positions (input, &xc, &yc, &zc))
    {
      AVSerror ("Error in ucd_cell_geom: can't get node coordinates.\n"); 
      return (0);
    }

    /* update node data choice list */
/*****
    select = AVSchoice_number("node data",comp_select)-1;
    if ( select < 0 ) select=0;
    if ( AVSinput_changed("Input", 0) ||
         AVSparameter_changed("Method")  )
    {
        select = update_menu( input,
                 comp_select, node_len, cell_len, method_number );
        if ( select < 0 ) select=0;
    }
******/

    /* figure out if we are in fast update mode or not */
    if ( !AVSinput_changed("Input", 0) &&              /* old ucd */
         !AVSinput_changed("Input Colourmap", 0) &&    /* old colors */
         !AVSparameter_changed("Method") &&            /* same method */
         !AVSparameter_changed("Remove Interior") &&   /* same face list */
         !AVSparameter_changed("Optimize Surface Normals") && /* same norms */
         !AVSparameter_changed("Downsize")             /* same num faces */
        )
        fast_update = 1;
    else
        fast_update = 0;

    /* get a pointer to where the cell data is stored */
    if ( cell_len )
    {
        if (!UCDstructure_get_cell_data (input, &cell_data))
        {
          AVSerror ("Error in ucd_cell_geom: can't get cell data.\n"); 
          return (0);
        }
    }

    /* get a pointer to where the node data is stored */
    if ( node_len )
    {
        if (!UCDstructure_get_node_data (input, &node_data))
        {
          AVSerror ("Error in ucd_cell_geom: can't get node data.\n"); 
          return (0);
        }
    }

    /* find extent of whole object */
    UCDstructure_get_extent (input, min_extent, max_extent);

    extent[0] = min_extent[0], extent[1] = max_extent[0];
    extent[2] = min_extent[1], extent[3] = max_extent[1];
    extent[4] = min_extent[2], extent[5] = max_extent[2];

    cx = (extent[0] + extent[1]) / 2.0;
    cy = (extent[2] + extent[3]) / 2.0;
    cz = (extent[4] + extent[5]) / 2.0;

    width  = extent[1] - extent[0];
    height = extent[3] - extent[2];
    depth  = extent[5] - extent[4];

    maxdim = (width > height ? width : height);
    maxdim = (depth > maxdim ? depth : maxdim);

    if ( AVSinput_changed("Input", 0) )
    {
#ifdef DEBUG
        printf("ucd_cell_geom: new ucd struct arrived, reset xform\n");
#endif
        /* make dial show actual model coordinate distance */
        AVSmodify_float_parameter("Distance",
                      AVS_MINVAL | AVS_MAXVAL,
                      0.0, -maxdim , maxdim );

        /* build the icon table */
        define_profile( model_icon, cx, cy, cz, maxdim );

        /* transform icon to reference position */
        picked_point[0] = cx;
        picked_point[1] = cy;
        picked_point[2] = cz;
        AVScommand("kernel","geom_lookup_obj_names $Module",
            &outbuf, &errbuf );
        sprintf(string,"geom_reset -object %s",outbuf);
        AVScommand("kernel",string, &outbuf, &errbuf );

        /* reset state */
        reset_xform = 1;
        have_reset_xform=1;
        picked_cell = -1;
        picked_node = -1;

        /* interior removal mode */
        if (    face_list)
           free(face_list);
        if (    line_list)
           free(line_list);
        if (    edge_list)
           free(edge_list);
        face_list = NULL;
        line_list = NULL;
        edge_list = NULL;
        faces_count = 0;
        edges_count = 0;
        verts_count = 0;

        /* Now Process New Face List */
        /*
         * this scans the ucd model, builds a map
         * of the exterior faces of cells,
         * by analysis of material id, and
         * the node connectivity, if nodes are shared
         * by adjacent cells, then a homogeneous
         * region is defined, if not, then the face
         * will be returned as a boundary face.
         * 2D cells, such as tri's or quads are always
         * considered external faces.
         * This is an internal undocumented AVS utility
         * used by several modules including ucd_to_geom.
         */
         if ( remove_interior )
         {
            if ( ucd_rm_interior_polys
              (input,                   /* input structure */
               num_cells,               /* number of cells */
               num_nodes,               /* number of nodes */
               &faces_count, /* RTN: num ext. faces */
               &verts_count, /* RTN: num verts */
               &face_list,   /* RTN: face list struct */
               &lines_count, /* RTN: num ext. lines */
               &line_list,   /* RTN: line list struct */
               NULL) ==0 )              /* cell subset list */
            {
             /* don't call us, we'll call you */
             AVSerror( "ucd_cell_g: ucd_rm_interior_polys failed.\n");
             return(0);
            }
        }

    }
    else
    {
        reset_xform = 0;
    }

    /************************/
    /* process a pick event */
    /************************/
    if ( xgeom != NULL &&
         AVSinput_changed( "pick", 0)  )
    {
#ifdef DEBUG
        printf("ucd_cell_geom: new picked face, reset xforms\n");
#endif
        picked_point[0] = xgeom->vertex[0];
        picked_point[1] = xgeom->vertex[1];
        picked_point[2] = xgeom->vertex[2];
        reset_xform = 1;
        have_reset_xform=1;
        AVScommand("kernel","geom_lookup_obj_names $Module",
            &outbuf, &errbuf );
        sprintf(string,"geom_reset -object %s",outbuf);
        AVScommand("kernel",string, &outbuf, &errbuf );
        picked_node = xgeom->vdata;  /* Picked Node index */
        if ( mark_face )
        {
            picked_cell = xgeom->odata;  /* Picked Cell index */
        }
        else
        {
            picked_cell = -1;
        }
    }
 
    if ( xform != NULL &&
         AVSinput_changed( "upstream", 0)  )
    {
#ifdef DEBUG
        printf("ucd_cell_geom: new xform target move request\n");
#endif
        if ( have_reset_xform )
        {
           have_reset_xform=0;
           return(0);
        }
        have_reset_xform=0;
    }

    /* Transformation Request Control */
    mat_identity( identity );
    if (xform != NULL &&
        !strncmp(xform->object_name,
        "ucd_cell_geom", strlen("ucd_cell_geom")) &&
        !reset_xform )
    {
        track_box_transform(newmat, min_extent, max_extent,
            xform->msxform, *distance);
    }
    else
    {
        track_box_transform(newmat, min_extent, max_extent,
            identity, *distance);
    }

    for ( i=0; i<5; i++ )
    {
        /* get original reference model coords to transform: */
        select_icon[i][0] = model_icon[i][0];
        select_icon[i][1] = model_icon[i][1];
        select_icon[i][2] = model_icon[i][2];
    }
    target[0] = picked_point[0];
    target[1] = picked_point[1];
    target[2] = picked_point[2];

    /* move the local copy: */
    for ( i=0; i<5; i++ )
        mat_vecmul( &(select_icon[i][0]) ,newmat);

    mat_vecmul( target ,newmat);

    /* establish the geometry output object */
    obj     =  GEOMcreate_obj (GEOM_POLYHEDRON, NULL);
    lobj    =  GEOMcreate_obj (GEOM_POLYTRI, NULL);


    /* draw the marked and moved point */
    if ( show_icon )
    {
        sobj1 = GEOMcreate_sphere(extent, picked_point,
                    sphere_size, GEOM_NULL, GEOM_NULL, 1, GEOM_COPY_DATA );
        GEOMadd_float_colors(sobj1,target_sph_hue, 1, GEOM_COPY_DATA );
        sobj2 = GEOMcreate_sphere(extent, target,
                    sphere_size, GEOM_NULL, GEOM_NULL, 1, GEOM_COPY_DATA );
        GEOMadd_float_colors(sobj2,target_sph_hue, 1, GEOM_COPY_DATA );

        /* draw drag line between picked vertex and target move point */
        if ( drag_line )
             draw_drag_line ( lobj, picked_point, target );

        /* create the visual icon representation: before and after */
        draw_profile_icon ( lobj, model_icon,  0 );
        draw_profile_icon ( lobj, select_icon, 1 );
    }

    /* draw the geometry from the cell list */
    if ( !fast_update || drag_model )
    {
        display_surface (input, num_cells, obj, 
                     xc, yc, zc, cmap_input,
                     node_len, node_data,
                     cell_len, cell_data,
                     method_number, picked_cell,
                     picked_node, drag_model,
                     target_hue, target,
                     num_nodes, remove_interior,
                     faces_count, face_list, downsize,
                     fast_update, opt_normals );
    }


    /************************************************/
    /* complete the geometry to output port process */
    /************************************************/
    GEOMgen_normals (obj, 0);
    if ( opt_normals )
        GEOMcvt_polyh_to_polytri(obj,
        GEOM_SURFACE );
    else
        GEOMcvt_polyh_to_polytri(obj,
        GEOM_WIREFRAME | GEOM_SURFACE | GEOM_NO_CONNECTIVITY );

    /************************************************/
    *output = GEOMinit_edit_list(*output);
    if ( fast_update )
        GEOMedit_geometry (*output, "ucd_cell_geom_anno", obj);
    else
        GEOMedit_geometry (*output, "ucd_cell_geom", obj);
    GEOMdestroy_obj (obj);
    if ( show_icon )
    {
        GEOMedit_geometry (*output, "ucd_cell_geom_anno", lobj);
        GEOMedit_geometry (*output, "ucd_cell_geom_anno", sobj1);
        GEOMedit_geometry (*output, "ucd_cell_geom_anno", sobj2);
        GEOMdestroy_obj (lobj);
        GEOMdestroy_obj (sobj1);
        GEOMdestroy_obj (sobj2);
    }

 /* if no xform allowed, then
  *  GEOMedit_transform_mode (*output, "ucd_cell_geom", "parent");
  * else:  */

    GEOMedit_transform_mode(*output, "ucd_cell_geom", "redirect",
                          BUTTON_MOVING | BUTTON_UP);
    GEOMedit_selection_mode(*output,"ucd_cell_geom",
                   "notify", BUTTON_MOVING | BUTTON_DOWN );
    if ( show_icon )
    {
    GEOMedit_transform_mode(*output, "ucd_cell_geom_anno", "redirect",
                          BUTTON_MOVING | BUTTON_UP);
    GEOMedit_selection_mode(*output,"ucd_cell_geom_anno",
                   "notify", BUTTON_MOVING | BUTTON_DOWN );
    }



    /* clean up memory */
    if ( global_colors )  free ( global_colors );


    /* Process the "Apply" button, send to output port */
    if ( xgeom != NULL && apply )
    {
#ifdef DEBUG
        printf("ucd_cell_geom: apply move, send data to field output port\n");
#endif
        dims[0] = 1;
        if (*pick_result) AVSfield_free( *pick_result);
        *pick_result = (AVSfield_int *) AVSdata_alloc(
            "field 1D 2-vector integer 3-space irregular", dims);

        (*pick_result)->data[0] = xgeom->vdata;  /* Picked Node index */
        (*pick_result)->data[1] = xgeom->odata;  /* Picked Cell index */
        (*pick_result)->points[0] = target[0];   /* new position request */
        (*pick_result)->points[1] = target[1];
        (*pick_result)->points[2] = target[2];
    }
    else
    {
        AVSmark_output_unchanged("pick_result");
    }


    return(1);
}


/*-----------------------------------------------------*
 *                                                     *
 *          ****  display_surface  ****                *
 *                                                     *
 *-----------------------------------------------------*/

display_surface (input, num_cells, obj, 
		 xc, yc, zc, cmap,
                 node_len, node_data,
                 cell_len, cell_data,
                 method, picked_cell,
                 picked_node, drag_model,
                 face_hue, target,
                 n_nodes, rem_int,
                 faces_count, face_list,
                 downsize, fast_update,
                 opt_normals )

  UCD_structure *input;   /* ucd input port */
  int num_cells;          /* how many cells */
  GEOMobj *obj;           /* add polygons to this object */
  float *xc, *yc, *zc;    /* node position arrays */
  AVScolormap *cmap;      /* color table */
  int node_len;           /* cell vector length */
  int cell_len;           /* cell vector length */
  float *node_data;       /* data to use for colors */
  float *cell_data;       /* data to use for colors */
  int method;             /* choice mode on parameters */
  int picked_cell;        /* cell to hi-light */
  float face_hue[3];      /* picked face colour */
  int   picked_node;      /* picked node id */
  int   drag_model;       /* drag model or not */
  float target[3];        /* new node position */
  int   n_nodes;          /* number of nodes on model */
  int   rem_int;          /* remove interior mode control */
  int   faces_count;      /* number of external faces */
  UCD_Face_List *face_list;  /* list of external faces */
  int   downsize;         /* selective display ratio */
  int   fast_update;      /* only update picked cell */
  int   opt_normals;      /* fancy surface shading */
{
      char ctype[40];
      float colors[8][3], verts[8][3], cx, cy, cz, x, y, z;
      int   vdata[8];

      int cell, cell_id, mat_id, cell_type, me_flags, *node_list,
          count, num_faces, *topo, node, face, v, numv, num_fv, 
          num_nodes, i, k, j, node_vector, cell_vector, top_mat_id;
      int ii, loop_count;
      float dip;
      float face_normal[4];
      float ref_vector[3];
      float vdot_prod ();
      int   touches_pick;

      /* hard code to first one on list */
      node_vector=0;
      cell_vector=0;

      /* search for top material id */
      if ( method == 3 )
      {
          for (cell = top_mat_id = 0; cell < num_cells; cell++)
          {
            UCDcell_get_information (input, cell, &cell_id, ctype, &mat_id,
    			     &cell_type, &me_flags, &node_list);
            if ( mat_id > top_mat_id ) top_mat_id = mat_id;
          }
      }

      /*********************************************/
      /**** loop over ALL cell  surfaces        ****/
      /*********************************************/
    
      if ( rem_int )
          loop_count = faces_count;
      else
          loop_count = num_cells;

      for (ii = 0; ii < loop_count; ii+=downsize)
      {
        if ( rem_int )
        {
            num_faces = 1;
            mat_id = face_list[ ii ].mat_id;
            numv   = face_list[ ii ].num_vtx;
        }
        else
        {
            /* query cell */
            cell = ii;
            UCDcell_get_information (input, cell,
                             &cell_id, ctype, &mat_id,
    			     &cell_type, &me_flags, &node_list);
    
            /* identify cell topology */
            num_nodes = UCD_num_nodes[cell_type];
            num_faces = UCD_num_faces[cell_type];
            topo      = cell_topos[cell_type];
        } 

        /* scan current cell for a pick reference */
        if ( fast_update )
        {
            if ( !rem_int )
                numv = num_nodes; 
            for ( j=touches_pick=0; j < numv; j++ )
            {
                if ( rem_int )
                    v = face_list[ ii ].vtx_index[ j ];
                else
                    v = node_list[j];
                if ( v == picked_node )
                    touches_pick = 1;
            }
            if ( touches_pick == 0 ) num_faces = 0; /* inhibit loop */
        }

        /* show each vertex of each face */
	for (face = 0, count = 0; face < num_faces; face++ )
	{
            if ( !rem_int )
                numv = topo [ count++ ];
    
            /* vertex loop */
            for (j = 0; j < numv; j++)
            {
                if ( rem_int )
                    v = face_list[ ii ].vtx_index[ j ];
                else
                    v = node_list[topo[count++]];
                verts[j][0] = xc[v];
                verts[j][1] = yc[v];
                verts[j][2] = zc[v];
                vdata[j]    = v;

                /* dynamic drag mode */
                if ( drag_model  && v == picked_node  )
                {
#ifdef DEBUG
                    printf("ucd_cell_geom: drag mode lines with target\n");
#endif

                    verts[j][0] = target[0];
                    verts[j][1] = target[1];
                    verts[j][2] = target[2];
                }
    
                /* turn cell user selected hue if picked */
                if ( cell == picked_cell )
                {
                        colors[j][0] = face_hue[0];
                        colors[j][1] = face_hue[1];
                        colors[j][2] = face_hue[2];
                }
                /* color based on cell number in list */
                else if ( method == 1 )
                {
                        map_data_to_colours ( 1.0, (float)loop_count,
                        (float) ii,
                        &colors[j][0], 
                        &colors[j][1],
                        &colors[j][2] );
                }
	        /* blast cell based data to all corners */
                else if ( cell_len && method==2 )
                {
                    map_data_to_colours ( cmap->lower, cmap->upper,
                        cell_data[ cell_vector*cell_len + cell],
                        &colors[j][0], 
                        &colors[j][1],
                        &colors[j][2] );
                }
                /* color the material id code for each cell */
                else if ( method==3 )
                {
                    map_data_to_colours ( 1.0, (float)top_mat_id,
                        (float)mat_id,
                        &colors[j][0], 
                        &colors[j][1],
                        &colors[j][2] );
                }
                /* color based on node number in list */
                else if ( method == 4 )
                {
                    map_data_to_colours ( 1.0, (float)n_nodes,
                        (float)v,
                        &colors[j][0], 
                        &colors[j][1],
                        &colors[j][2] );
                }
	        /* interpolate node data to  corners */
                else if ( node_len && method==5 )
                {
                    map_data_to_colours ( cmap->lower, cmap->upper,
                        node_data[ node_vector*node_len + v],
                        &colors[j][0], 
                        &colors[j][1],
                        &colors[j][2] );
                }
	        /* cell type, topology of cell */
                /* pt, line, tri, quad, tet, pyr, prism, hex */
                else if ( method==6 )
                {
                    map_data_to_colours ( 0.0, 7.0,
                        (float) cell_type,
                        &colors[j][0], 
                        &colors[j][1],
                        &colors[j][2] );
                }
	        /* face order, topology of cell */
                else if ( method==7 )
                {
                    map_data_to_colours ( 0.0, 5.0,
                        (float) face,
                        &colors[j][0], 
                        &colors[j][1],
                        &colors[j][2] );
                }
	        /* X-coord node position value */
                else if ( method==8 )
                {
                    map_data_to_colours (  cmap->lower, cmap->upper,
                        xc[v],
                        &colors[j][0], 
                        &colors[j][1],
                        &colors[j][2] );
                }
	        /* Y-coord node position value */
                else if ( method==9 )
                {
                    map_data_to_colours (  cmap->lower, cmap->upper,
                        yc[v],
                        &colors[j][0], 
                        &colors[j][1],
                        &colors[j][2] );
                }
	        /* Z-coord node position value */
                else if ( method==10 )
                {
                    map_data_to_colours (  cmap->lower, cmap->upper,
                        zc[v],
                        &colors[j][0], 
                        &colors[j][1],
                        &colors[j][2] );
                }
	        /* Dip, angle of surface away from flat */
                else if ( method==11 )
                { /* Nothing, defer till all verts known, see below */ }
                else if ( method==12 )
                { /* Nothing, defer till all verts known, see below */ }
                else if ( method==13 )
                { /* Nothing, defer till all verts known, see below */ }

                /* no valid choice given input data, so default */
                else
                {
                        colors[j][0] = .95;  /* nearly white */
                        colors[j][1] = .95;
                        colors[j][2] = .95;
                }
            }  /* end of vertex loop */

            /* Dip, angle of surface away from flat */
            if ( method==11 || method==12 || method==13 )
            {
                /* find face normal vector */
                get_planeq ( verts, face_normal );

                /* user axis as reference vector */
                ref_vector[0] = ref_vector[1] = ref_vector[2] = 0.0;
                if ( method == 11 ) ref_vector[0] = 1.0;
                if ( method == 12 ) ref_vector[1] = 1.0;
                if ( method == 13 ) ref_vector[2] = 1.0;
                dip = vdot_prod (face_normal, ref_vector);

                for (j = 0; j < numv; j++)
                {
                    map_data_to_colours (  cmap->lower, cmap->upper,
                        dip,
                        &colors[j][0], 
                        &colors[j][1],
                        &colors[j][2] );
                }
            }



            GEOMadd_disjoint_polygon (obj, verts, NULL, colors, numv,
                     (opt_normals ? GEOM_SHARED:GEOM_NOT_SHARED), 0 );

            GEOMadd_prim_data(obj,&cell,1,GEOM_COPY_DATA);
            GEOMadd_vertex_data(obj,vdata,numv,GEOM_COPY_DATA);


	}    /* end of face loop */
      }  /* end of cell loop */
    
}

/*-----------------------------------------------------*
 *                                                     *
 *             ****  map_data_to_colours  ****         *
 *                                                     *
 *-----------------------------------------------------*/

map_data_to_colours ( vmin, vmax, val, red, green, blue )

float vmin;   /* range min */
float vmax;   /* range max */
float val;    /* input value to color */
float *red;   /* output color for sample */
float *green; /* output color for sample */
float *blue;  /* output color for sample */

{

  int index;
  float scale, offset;

  if (vmax != vmin)
    scale =  (float)(global_cmap_size - 1) / (vmax - vmin);
  else
    scale = 0.0;

  index = (int)((val - vmin) * scale);

  if ( index < 0 ) index = 0;
  else if ( index >= global_cmap_size ) index = global_cmap_size -1;

  *red      = global_colors[index * 3];
  *green    = global_colors[index * 3 + 1];
  *blue     = global_colors[index * 3 + 2];

}


/*-----------------------------------------------------*
 *                                                     *
 *    ****  track_box_transform  ****                  *
 *                                                     *
 *-----------------------------------------------------*/

track_box_transform(out_matrix, ll, ur, addtl, distance)
    float out_matrix[4][4], addtl[4][4];
    float ll[3], ur[3], distance;
{
    float tx, ty, tz, tmpmat[4][4], hw, hh, hd;

    hw = (ll[0] + ur[0]) / 2.0;
    hh = (ll[1] + ur[1]) / 2.0;
    hd = (ll[2] + ur[2]) / 2.0;

    mat_identity(out_matrix);

    tx = out_matrix[3][0] * (ur[0] - ll[0]);
    ty = out_matrix[3][1] * (ur[1] - ll[1]);
    tz = out_matrix[3][2] * (ur[2] - ll[2]);
    out_matrix[0][3] = out_matrix[1][3] = out_matrix[2][3] = 0.0;

    mat_translate(tmpmat, -hw, -hh, -hd+distance);
    mat_multiply(tmpmat, out_matrix, out_matrix);

    mat_multiply(out_matrix, addtl, out_matrix);

    mat_translate(tmpmat,  tx+hw,  ty+hh,  tz+hd);
    mat_multiply(out_matrix, tmpmat, out_matrix);
}



/*-----------------------------------------------------*
 *                                                     *
 *       ****  define_profile      ****                *
 *                                                     *
 *-----------------------------------------------------*/
define_profile( profile_icon, cx, cy, cz, maxdim )
        float profile_icon[5][3];
        float cx, cy, cz, maxdim;
{
    int i;

    /* define the starting shape for icon */
    for ( i=0; i<5; i++ )
    {
            profile_icon[i][2] = cz;
    }

    profile_icon[0][0] = cx - maxdim *0.5;
    profile_icon[0][1] = cy - maxdim *0.5;

    profile_icon[1][0] = cx + maxdim *0.5;
    profile_icon[1][1] = cy - maxdim *0.5;

    profile_icon[2][0] = cx + maxdim *0.5;
    profile_icon[2][1] = cy + maxdim *0.5;

    profile_icon[3][0] = cx - maxdim *0.5;
    profile_icon[3][1] = cy + maxdim *0.5;

    profile_icon[4][0] = cx - maxdim *0.5;
    profile_icon[4][1] = cy - maxdim *0.5;

}



/*-----------------------------------------------------*
 *                                                     *
 *       ****  draw_profile_icon   ****                *
 *                                                     *
 *-----------------------------------------------------*/
#define SCW  0.05
#define RCW  0.95


draw_profile_icon ( obj, pts, code )
    GEOMobj       *obj;
    float         pts[5][3];
    int code;
{
    int i, j;
    float verts[2][3];
    static int vdata[ 4 ] = { -1, -1, -1, -1 };
    /* define light green */
    static float co1[2][3] = { .3, .9, .3, .3, .9, .3 };
    /* define light yellow */
    static float co2[2][3] = { .9, .9, .3, .9, .9, .3 };

    /* primary outline of icon */
    for (i=0; i<4; i++)
    {
        GEOMadd_disjoint_line (obj, &pts[i][0], (code?co2:co1),
                               2, GEOM_COPY_DATA);
        GEOMadd_disjoint_vertex_data (obj, vdata,
                               2, GEOM_COPY_DATA);
    }
    /* secondary lines */
    for (i=1; i<=3; i+=2)
    {
        for (j=0; j<3; j++)
        {
            verts[0][j] = (RCW * pts[i][j] + SCW * pts[i-1][j]);
            verts[1][j] = ((RCW * pts[i][j] + SCW * pts[i-1][j]) - pts[i][j]) +
                          ((RCW * pts[i][j] + SCW * pts[i+1][j]) );
        }
        GEOMadd_disjoint_line (obj, verts, (code?co2:co1), 2, GEOM_COPY_DATA);
        GEOMadd_disjoint_vertex_data (obj, vdata,
                               2, GEOM_COPY_DATA);
        for (j=0; j<3; j++)
        {
            verts[0][j] = (RCW * pts[i][j] + SCW * pts[i+1][j]);
            verts[1][j] = ((RCW * pts[i][j] + SCW * pts[i-1][j]) - pts[i][j]) +
                          ((RCW * pts[i][j] + SCW * pts[i+1][j]) );
        }
        GEOMadd_disjoint_line (obj, verts, (code?co2:co1), 2, GEOM_COPY_DATA);
        GEOMadd_disjoint_vertex_data (obj, vdata,
                               2, GEOM_COPY_DATA);
    }
}

/*-----------------------------------------------------*
 *                                                     *
 *       ****  draw_drag_line  ****                    *
 *                                                     *
 *-----------------------------------------------------*/
draw_drag_line (  obj, picked_point, target )
    GEOMobj       *obj;
    float         *picked_point, *target;
{
    int i;
    float verts[2][3];
    static int vdata[ 4 ] = { -1, -1, -1, -1 };
    /* define light green */
    static float co[2][3] = { .3, .9, .3, .3, .9, .3 };

    for (i=0; i<3; i++)
    {
            verts[0][i] = picked_point[i];
            verts[1][i] = target[i];
    }
    GEOMadd_disjoint_line (obj, verts, co, 2, GEOM_COPY_DATA);
    GEOMadd_disjoint_vertex_data (obj, vdata,
                           2, GEOM_COPY_DATA);

}

/*-----------------------------------------------------*
 *                                                     *
 *       ****  Update Menu         ****                *
 *       re-paints the menu choices                    *
 *       returns node/cell vector offset number        *
 *                                                     *
 *-----------------------------------------------------*/

int
update_menu( ucd_input, data_type, node_len, cell_len, method )

  UCD_structure *ucd_input;  /* ucd input structure pointer */ 
  char       *data_type;     /* current choice selection string */
  int        node_len;       /* number of values per node */
  int        cell_len;       /* number of cell values */
  int        method;         /* which method to use */
{
    /* storage for scalar component names */
    char  scalar_comps[50][80];
    char  temp[100];
    int   i,j,k,t;
    char  string     [ UCD_LABEL_LEN ];
    char  data_labels[ UCD_LABEL_LEN ];
    char  delim[3];
    int   select;         /* selected scalar to return */
    int   num_comp;       /* number of components */
    int   *nd_comp_list;  /* component list */

    /* If UCD changed */

        /* if ucd input changes, redo the menu */
        if ( method==2 && cell_len )
        {
            UCDstructure_get_cell_labels (ucd_input, data_labels, delim);
            UCDstructure_get_cell_components (ucd_input, &nd_comp_list);
            node_len = cell_len;
        }
        else if (node_len)
        {
            UCDstructure_get_node_labels (ucd_input, data_labels, delim);
            UCDstructure_get_node_components (ucd_input, &nd_comp_list);
        }
        else
        {
            node_len = 0;
            strcpy( string, "<none>");
        }
        
        /* compute (i), the number of components, and names */
        for(i=k=t=num_comp=0; i < node_len; num_comp++)
        {
            /* accumulate component size */
            i += nd_comp_list[num_comp];
            /* copy out component name */
            for ( j=0;
                  ( j < 100 ) &&
                  ( k < 600 ) &&
                  (data_labels[k] != delim[0]) &&
                  (data_labels[k] != '\0'    );
                  j++ )
            {
                temp[j] = data_labels[k++]; 
            }
            temp[j] = '\0';
            k++;             /* skip delimiter */
            for ( j=0; j<nd_comp_list[num_comp]; j++ )
            {
                if ( nd_comp_list[num_comp] == 1 )  /* copy scalar name */
                    sprintf( &scalar_comps[t][0],
                            "%s",temp); 
                else
                    sprintf( &scalar_comps[t][0],   /* add sub-numbers */
                            "%s-%d",temp,j+1); 

                t++;
            }
        }

        /* assemble menu choice string */
        if ( t != node_len )
            printf(" node_len = %d, num strings = %d\n",node_len, t);
        strcpy( string, &scalar_comps[0][0] );
        for( j=1; j < node_len; j++ )
        {
            strcat( string, delim );
            strcat( string, &scalar_comps[j][0] );
        }

        /* re-paint the menu list */
        AVSmodify_parameter ("node data",  AVS_VALUE | AVS_MINVAL | AVS_MAXVAL,
                             data_type,
                             string, delim);
        
        select = AVSchoice_number ("node data", data_type ) - 1;

        return( select ); 
    
}



/*-----------------------------------------------------*/
/* normalize the length of a 3-vector */
/*-----------------------------------------------------*/
int
vnorm (v)
        float v[];
{
  double mag, sqrt();

  mag = sqrt(v[0] * v[0] + v[1] * v[1] + v[2] * v[2]);

  if (mag == 0.0) {
    v[0] = 0.0;
    v[1] = 0.0;
    v[2] = 0.0;
    return (0);
    }
  else {
    v[0] = v[0] / mag;
    v[1] = v[1] / mag;
    v[2] = v[2] / mag;
    }
    return (1);
}

/*-----------------------------------------------------*/
/* 2-vector dot product */
/*-----------------------------------------------------*/
float
vdot_prod (v1,  v2)
    float *v1, *v2;
{
  float    dot;

  dot = ( v1[0]*v2[0] + v1[1]*v2[1] + v1[2]*v2[2] );
  return ( dot );
}


/*-----------------------------------------------------*/
/* get plane equation from two edges */
/*-----------------------------------------------------*/
get_planeq ( v, eq )
    float v[5][3], eq[4];
{

/* page 114 in Bowyer and Woodwark */

    float xkj, ykj, zkj, xlj, ylj, zlj;

    /* get difference vectors */
    xkj = v[1][0] - v[0][0];
    ykj = v[1][1] - v[0][1];
    zkj = v[1][2] - v[0][2];

    xlj = v[2][0] - v[0][0];
    ylj = v[2][1] - v[0][1];
    zlj = v[2][2] - v[0][2];

    /* cross product for plane normal */
    eq[0] = ykj * zlj - zkj * ylj;
    eq[1] = zkj * xlj - xkj * zlj;
    eq[2] = xkj * ylj - ykj * xlj;

    /* normalize vector */
    if ( vnorm( eq ) == 0 )
    {
        eq[0] = eq[1] = 0.0;
        eq[2] = 1.0;
    }

    /* use plane equation for distance offset */
    eq[3] = -1.0 * (v[1][0] * eq[0] +
                    v[1][1] * eq[1] +
                    v[1][2] * eq[2] );

}



/*-----------------------------------------------------*
 *                                                     *
 *       ****  ucd_cell_geom_init  ****                *
 *                                                     *
 *-----------------------------------------------------*/
ucd_cell_geom_init()
{
  AVSstatic = (char *)NULL;
}

/*-----------------------------------------------------*
 *                                                     *
 *       ****  ucd_cell_geom_finis  ****               *
 *                                                     *
 *-----------------------------------------------------*/
ucd_cell_geom_finis()
{
  AVSstatic = (char *)NULL;
}

/*-----------------------------------------------------*
 *                                                     *
 *        ****  ucd_cell_geom_desc  ****               *
 *                                                     *
 *-----------------------------------------------------*/
ucd_cell_geom_desc()
{
  int ucd_cell_geom(), param;
  int flags;
  char string[300];

/*
  flags = INVISIBLE | OPTIONAL ;
  flags = REQUIRED ;
 */
  flags = OPTIONAL ;

  /* name on the box */
  AVSset_module_name ("ucd cell geometry", MODULE_MAPPER);

  /* input ucd structure to draw */
  AVScreate_input_port ("Input", "ucd", REQUIRED);

  /* colors for the faces */
  AVScreate_input_port ("Input Colourmap", "colormap", REQUIRED);


  /* interactive transformation and picking ports */
  param = AVScreate_input_port("upstream",
              "struct upstream_transform", flags);
  AVSset_input_class(param, "Output:upstream_transform");

  param = AVScreate_input_port("pick",
              "struct upstream_geom", flags);
  AVSset_input_class(param,"upstream_geom");


  /* output geomerty to draw */
  AVScreate_output_port ("ucd_cell_geom", "geom");

  /* output port for upstream flow */
  AVScreate_output_port ("pick_result",
      "field 1D 2-vector integer 3-space irregular");
  

  /* parameter controls */
  /* color meaning choices: */
  strcpy ( string, "Cell Number#" );
  strcat ( string, "Cell Data#"   );
  strcat ( string, "Material Id#" );
  strcat ( string, "Node Number#" );
  strcat ( string, "Node Data#"   );
  strcat ( string, "Cell Type#"   );
  strcat ( string, "Face Order#"  );
  strcat ( string, "X-Coord#"     );
  strcat ( string, "Y-Coord#"     );
  strcat ( string, "Z-Coord#"     );
  strcat ( string, "cos(X Dip)#"  );
  strcat ( string, "cos(Y Dip)#"  );
  strcat ( string, "cos(Z Dip)"   );

  param = AVSadd_parameter("Method","choice", 
  "Node Data", string, "#");
  AVSadd_parameter_prop(param, "width", "integer", 2);

  param = AVSadd_float_parameter("Distance",0.0,
                                  FLOAT_UNBOUND, FLOAT_UNBOUND);

  param = AVSadd_float_parameter("Sphere Size",1.0,
                                  FLOAT_UNBOUND, FLOAT_UNBOUND);

  param = AVSadd_parameter("Apply Change","oneshot",0,0,1);

  param = AVSadd_parameter("Mark Picked Face","boolean",0,0,1);
  AVSadd_parameter_prop(param, "width", "integer", 2);

  param = AVSadd_parameter("Show Drag Line","boolean",0,0,1);
  AVSadd_parameter_prop(param, "width", "integer", 2);

  param = AVSadd_parameter("Drag Model Lines","boolean",0,0,1);
  AVSadd_parameter_prop(param, "width", "integer", 2);

  param = AVSadd_float_parameter("Marked Face Hue",1.0, 0.0, 1.0);
  AVSconnect_widget(param, "slider" );

  param = AVSadd_float_parameter("Target Hue",0.3, 0.0, 1.0);
  AVSconnect_widget(param, "slider" );

  param = AVSadd_parameter("Show XFM Icon","boolean",0,0,1);
  AVSadd_parameter_prop(param, "width", "integer", 2);

  param = AVSadd_parameter("Remove Interior","boolean",1,0,1);
  AVSadd_parameter_prop(param, "width", "integer", 2);

/*****
  param = AVSadd_parameter("node data",
           "choice", "data_1","data_1#data_2#data3","#");
 *****/

  param = AVSadd_parameter("Downsize", "integer",1,1,20);

  param = AVSadd_parameter("Optimize Surface Normals",
                           "boolean",0,0,1);
  AVSadd_parameter_prop(param, "width", "integer", 3);

  /* routine pointers */
  AVSset_init_proc    (ucd_cell_geom_init);
  AVSset_destroy_proc (ucd_cell_geom_finis);
  AVSset_compute_proc (ucd_cell_geom);

}  /* end of description routine */

AVSinit_modules()
{
        AVSmodule_from_desc(ucd_cell_geom_desc);
}

