/****************************************************************************
                  INTERNATIONAL AVS CENTER
	(This disclaimer must remain at the top of all files)

WARRANTY DISCLAIMER

This module and the files associated with it are distributed free of charge.
It is placed in the public domain and permission is granted for anyone to use,
duplicate, modify, and redistribute it unless otherwise noted.  Some modules
may be copyrighted.  You agree to abide by the conditions also included in
the AVS Licensing Agreement, version 1.0, located in the main module
directory located at the International AVS Center ftp site and to include
the AVS Licensing Agreement when you distribute any files downloaded from 
that site.

The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module provide absolutely
NO WARRANTY OF ANY KIND with respect to this software.  The entire risk as to
the quality and performance of this software is with the user.  IN NO EVENT
WILL The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module BE LIABLE TO
ANYONE FOR ANY DAMAGES ARISING FROM THE USE OF THIS SOFTWARE, INCLUDING,
WITHOUT LIMITATION, DAMAGES RESULTING FROM LOST DATA OR LOST PROFITS, OR ANY
SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES.

This AVS module and associated files are public domain software unless
otherwise noted.  Permission is hereby granted to do whatever you like with
it, subject to the conditions that may exist in copyrighted materials. Should
you wish to make a contribution toward the improvement, modification, or
general performance of this module, please send us your comments:  why you
liked or disliked it, how you use it, and most important, how it helps your
work. We will receive your comments at avs@ncsc.org.

Please send AVS module bug reports to avs@ncsc.org.

******************************************************************************/
#ifdef SCCS
static char sccsid[]="@(#)ucd_line.c 6.0  Jonathan Cox 29/10/93";
#endif
/*
 * FILE: ucd_line.c
 * 
 * MODULE
 * 
 * ucd line - interactive sampling of a line of data from a UCD structure.
 * 
 * DESCRIPTION
 *
 * This module produces an arbitrary line of sampled node data from a 3D UCD
 * mesh and outputs it as a field.  This can be a conventional AVS 1D field
 * with irregular 1D coordinates or a 2D field suitable for connection to the
 * graph viewer module.  It is also possible to output a uniformly spaced 1D
 * field.
 * 
 * The position and orientation of the line is arbitrary and is controllable
 * either by the direct manipulation of a probe object or by manipulating a
 * collection of dial widgets.  The probe object consists of two transparent
 * orthogonal planes and a line of intersection which is the sample line.  When
 * using the dial widgets on the module's control panel, the XRotation,
 * YRotation and ZRotation parameters control the orientation of the line as
 * three rotations about the major axes applied in the order ZYX.  The two
 * Distance parameters are positional offsets along the normals of the two
 * guide planes of the probe.  When controlling the orientation and position of
 * the probe by direct manipulation within the geometry viewer, the dial
 * parameters are updated to maintain a consistent state.
 * 
 * The sample is obtained by bi-linear interpolation of the node data values
 * along the two transparent planes within the probe object.  There is one
 * point on the sample line for every intersected face of the UCD structure.
 * Therefore, by default, the spacing along the line is irregular.  If uniform
 * spacing is required, the data is interpolated again to a regularly spaced
 * line.  In general, the uniformly spaced data will be less accurate and have
 * more grid points than the irregularly spaced data.
 *  
 * NOTICE
 *
 * (c) University of Manchester 1991, 1992.
 * Author:  Jonathan Cox.
 * Address: IT301, Department of Computer Science, University of Manchester,
 *          Oxford Road, Manchester.  M13 9PL.  England.
 * Email:   coxjp@cs.man.ac.uk
 *
 * No part of this software may be altered, incorporated in other programs or
 * given to third parties, except under the conditions of the license, if any,
 * with which it was distributed.  This notice may not be removed from the
 * source code with which it is associated.
 *
 * VERSION HISTORY
 *
 * 5.0 - Jonathan Cox.  First version for general release.
 * 5.1 - Jonathan Cox.  Fixed problem with dissapearing input ports.
 * 5.2 - Jonathan Cox.  Made modules cooperative and rentrant.
 * 5.3 - Jonathan Cox.  ANSI C.
 * 6.0 - Jonathan Cox.  Second version for general release.
 * 
 */

/*
 * Include Files.
 */

#include <stdio.h>
#include <avs.h>
#include <port.h>
/* IAC CODE CHANGE : #include <avs_math.h> */
/* IAC CODE CHANGE : #include <avs/avs_math.h> */
#include <avs/avs_math.h>
#include <field.h>
#include <geom.h>
#include <ucd_defs.h>
#include <udata.h>

#include "cut_ucd.h"
#include "utils.h"
#include "mat.h"

/*****************************************************************************
 *
 * Global Defaults.
 */

/*
 * CONTINUAL_UPDATE will cause the dial parameters to be continually updated
 * while the probe object is being transformed.  If, in addition,
 * CONTINUAL_SAMPLE is defined the sample line will be continually regenerated
 * while the probe is being transformed.
 */
#define CONTINUAL_UPDATE
#define CONTINUAL_SAMPLE

/*
 * PRECISION is a small floating point value used to ensure that comparisons
 * for equality are interpreted correctly even if the values are slightly
 * inaccurate.
 */
#define PRECISION 1.0e-4

/*
 * Minimum distance between points on the sampled line as a fraction of the
 * domain size.
 */
#define MIN_DIST 1.0e-2

/*
 * Default transparency alpha coefficient for the two planes in the probe
 * object.
 */
#define ALPHA 0.3


/*****************************************************************************
 *
 * Global read write static data.
 */

typedef struct _static_data {
  /*
   * Description of the Domain - the 3D region covered by the UCD structure.
   */
  float Domain_Size;		/* Max dimension of the domain. */
  float Domain_Delta;		/* Minimum distance between line points */
  float Domain_Extent[6];	/* Size of the domain. */
  float Domain_Centre[3];	/* Centre of the domain. */

  /*
   * The untransformed sample line is defined as the intersection of the two
   * planes z=0 and y=0.  Under transformation by matrix M the equations of
   * these planes become:
   *
   *     Nz.P=0 and Ny.P=0,
   * 
   * where Nz and Ny are the transformations of the unit vectors in z and y
   * respectively (ie. M.k and M.j) and P is an arbitrary position vector on
   * the planes.  Clearly Nz and Ny are the normals of the transformed planes
   * z=0 and y=0.  Additionally, the parametric equation of the intersection of
   * these two planes, the sample line, is then:
   *
   *     C + lambda.Nx,
   * 
   * where C is any point on the line, Nx is the transformation of the unit
   * vector in x (= M.i) and lambda is a free ranging scalar variable.
   *
   * For convenience, the probes position and orientation is held globally as
   * the three normals, Normal, Ny and Nz, and two offsets, Dy and Dz in the
   * direction of Nz and Ny relative to the origin.  Hence, a suitable value
   * for C is:
   *
   *     Dz.Nz + Dy.Ny,
   *
   * although C is actually chosen so that lambda=0 is the point on the line
   * closest to the centre of the UCD structure.
   */
  
  float Nx[3];
  float Ny[3];
  float Nz[3];
  float Dy;
  float Dz;

  /*
   * Global descriptions of the Range - this is the data to be interpolated to
   * the sample line.
   */
  float *Range;			/* A pointer to a component of node data. */
  float Range_Min;		/* Bracketing values for the data. */
  float Range_Max;		/*  */
  float Range_Scale;		/* Makes range same size as domain. */

} static_data;


extern char *AVSstatic;


/*****************************************************************************
 *
 * Type definitions.
 */

/*
 * Structure to hold information about nodes in the structure necessary for the
 * calculation of a line through the UCD structure. (The line is derived as the
 * intersection of two cutting planes Nz.P=0 and Ny.P=0).
 */
typedef struct _node_info 
{
  float s;			/* Distance between node and plane Nz.P=0 */
  float t;			/* Distance between node and plane Ny.P=0 */
  char sflag;			/* Status flags */
  char tflag;
} node_info;


typedef struct _line_store_entry
{
  float lambda;			/* Distance of the point along the line */
  float value;			/* Data value at the point */
  struct _line_store_entry *left; /* Points of lower lambda */
  struct _line_store_entry *right; /* Points of higher lambda */
} line_store_entry;


typedef struct _line_store
{
  node_info *ntable;		/* Table of information on the UCD nodes */
  line_store_entry *root;	/* The points in the sample line */
  int points;			/* The number of points in the sample line */
} line_store;


/**************************************************************************
 *
 * Code to generate a transparent probe, in the correct position and
 * orientation on the first geometry output port.  The probe object is two
 * crossed planes, the line of intersection being the sample line.
 *
 */

get_line_probe_trans(static_data *S, upstream_transform *trans_info, float
                     xrot, float yrot, float zrot, float disty, float distz,
                     float matrix[4][4], float position[3])
/*
 * Derives the global definition of the sample line either from the upstream
 * transformation matrix or the dial widgets.  In the case of the upstream
 * transformation matrix the widgets values are modified to reflect the new
 * transformation matrix.  Otherwise the line definition is derived from the
 * parameter values and the probe object's transformation matrix is reset
 * according to 'matrix' and 'position' which are also calculated by this
 * function.
 */
{
  int i;
  float theta, phi, psi;	/* Rotation in X, Y and Z */
  float v[3];			/* A temporary vector */
  float m1[4][4], m2[4][4];	/* Temporary matrices */
  float phi_sgn, angle;

  if (trans_info)
    {
      /*
       * calculate and update the transformation parameters from the new
       * transformation matrix.
       */

      mat_copy(matrix, trans_info->msxform);
      
      /*
       * In order to decode the transformation matrix into parameters we assume
       * that the transformation matrix was constructed from a series of
       * transformations: translate by -c, rotate in z by psi, rotate in y by
       * phi, rotate in x by theta and translate by c and dy*Ny + dz*Nz; where
       * c is the centre of rotation Domain_Centre; theta, phi and psi are
       * angles in degrees; Ny and Nz are the normalised normals to the
       * transformed planes y=0 and z=0 respectively; and dy and dz are offset
       * distances along these normals relative to the Domain_Centre (not the
       * origin as with the global variables Dy and Dz).
       *
       * Note that the order of rotations is opposite to the order of rotations
       * used in AVS's transformations options panel.  This applies the
       * separate rotations in the order X Y Z.  However, rotating about Z
       * first makes more sense in this case as, as the probe plane is z=0, the
       * xrot and yrot parameters then control the orientation of the probe and
       * the zrot parameter controls the angle of rotation of the line z=y=0
       * within the probe plane.  If the user wishes to define the plane in the
       * normal AVS way, the transformation options panel can still be used as
       * normal.  If this is done the parameters will be calculated and updated
       * to reflect an equivalent z, y, x rotation order.
       */

      /*
       * Normalise the transformation matrix to cancel out any scaling that may
       * have been introduced by direct manipulation of the probe object within
       * the geometry viewer window.
       */
      normalize_vec(matrix[0]);
      normalize_vec(matrix[1]);
      normalize_vec(matrix[2]);

      /*
       * 'disty' and 'distz' can be found by taking the difference between the
       * dot product of any point on the transformed plane (eg. the
       * transformation of the centre including any translations) with the
       * plane normal, and the dot product of the centre of rotation with the
       * normal.  To avoid trouble with floating point inaccuracies, some
       * rounding is done.
       */
      vec_copy(v, S->Domain_Centre);
      mat_vecmul(v, matrix);
      disty = 0.0;
      distz = 0.0;
      for (i = 0; i < 3; i++)
	{
	  v[i] -= S->Domain_Centre[i];
	  distz += v[i] * matrix[2][i];
	  disty += v[i] * matrix[1][i];
	}
      if (fabs(distz) < PRECISION * S->Domain_Size) distz = 0.0;
      if (fabs(disty) < PRECISION * S->Domain_Size) disty = 0.0;


#ifndef CONTINUAL_UPDATE
      /*
       * If the trans_info has been generated mid way through a mouse
       * transformation the dial parameters are not updated.
       */
      if (trans_info->flags != BUTTON_UP) return;
#endif

      /*
       * Theta (zrot), phi (yrot) and psi (zrot) are calculated below by
       * looking at the transformation various points under various
       * transformations.  However, the method falls apart in phi = 90 degrees
       * (when the plane normal is parallel to the x-axis) so this is dealt
       * with as a special case.
       */
      vec_copy(v, matrix[2]);
      if (v[1] == 0.0 && v[2] == 0.0)
	{
	  /*
	   * In the case that phi is +/- 90 degrees the net rotation is
	   * equivalent to a rotation of psi-/+theta in z followed by a +/-90
	   * degree rotation in y.  This angle can be found by taking the arc
	   * target of -y/-+z of the transformation of i.  Both y and z cannot
	   * be zero in this case as x is.
	   */
	  phi_sgn = v[0] >= 0.0 ? 1.0 : -1.0;
	  theta = 0.0;
	  phi = 90.0 * phi_sgn;
	  vec_copy(v, matrix[0]);
	  angle = atan2(- v[1], - phi_sgn * v[2]) / 3.14159265358979323846 * 180.0;

	  /*
	   * In order to prevent jumps in the dial widgets while rotating the
	   * probe plane in the geometry viewer the previous values for xrot
	   * and zrot are examined in order to make dial movement as smoothly
	   * as possible.
	   */
	  if (xrot == 0)
	    {
	      theta = 0;
	      psi = angle;
	    }
	  else if (zrot == 0)
	    {
	      psi = 0;
	      theta = - phi_sgn * angle;
	    }
	  else if (! different_quadrant(zrot - phi_sgn * xrot, angle))
	    {
	      theta = xrot;
	      psi = angle + phi_sgn * theta;
	    }
	  else
	    {
	      theta = 0;
	      psi = angle;
	    }
	}
      else
	{
	  /*
	   * Theta can be obtained by taking the arctan of -y/z where y and z
	   * are the respective coordinates of v, the transformation (ignoring
	   * translations) of the vector k.  Theta is restricted to lie in the
	   * range (-90,90] degrees.  This is to ensure that theta, phi and psi
	   * are unambiguous (see below).
	   */
	  theta = (v[2] != 0.0) ? atan( -v[1]/v[2] ) / 3.14159265358979323846 * 180.0 : 90.0;

	  /*
	   * Once theta is known we can rotate v by - theta and take the arc
	   * tangent of x/z to get phi.  Arctan2 is used to obtain the correct
	   * quadrant for theta.  Note that atan2 is total except for the case
	   * where both arguments are zero.  In this case z and x cannot both
	   * be zero as y is.
	   */
	  mat_x_rotate(-theta, m1);
	  mat3_vecmul(v, m1);
	  phi = atan2(v[0], v[2]) / 3.14159265358979323846 * 180.0;

	  /*
	   * Psi can be obtained by taking the arc tangent of -y/x where x and
	   * y are the respective components of the transformation of the
	   * vector i ([1, 0, 0]) rotated by -theta in x and -phi in y.  Note
	   * that both y and x cannot be zero as z is.
	   */
	  vec_copy(v, matrix[0]);
	  mat3_vecmul(v, m1);
	  mat_y_rotate(-phi, m2);
	  mat3_vecmul(v, m2);
	  psi = atan2(-v[1], v[0]) / 3.14159265358979323846 * 180.0;

	  /*
	   * Theta, phi and psi are not the only specification of the plane
	   * possible.  This is because for every theta, phi and psi there is
	   * also 180-theta, 180-phi and 180-psi which perform the identical
	   * transformation.  Rather than making the specification unambiguous
	   * (by restricting the range of one of them to +-90 degrees) theta,
	   * phi and psi are chosen so that at least two of them are in the
	   * same quadrant as the previous values xrot, yrot and zrot.
	   */
	  if (different_quadrant(theta, xrot) +
	      different_quadrant(phi, yrot) +
	      different_quadrant(psi, zrot) > 1)
	    {
	      theta = rot180(theta);
	      phi = rot180(phi);
	      psi = rot180(psi);
	    }

	  /*
	   * However, so that reset works correctly, if none of theta, phi and
	   * psi are in the quadrant as 0.0 then the other set of angles are
	   * taken.
	   */
	  if (different_quadrant(theta, 0.0) &&
	      different_quadrant(phi, 0.0) &&
	      different_quadrant(psi, 0.0))
	    {
	      theta = rot180(theta);
	      phi = rot180(phi);
	      psi = rot180(psi);
	    }
	}

      /*
       * In order to avoid floating point rounding errors the dial rotation
       * values are set to the nearest degree.  However, the slice plane is
       * calculated to full accuracy.
       */
      xrot = floor(theta + 0.5);
      yrot = floor(phi + 0.5);
      zrot = floor(psi + 0.5);
      
      /*
       * Update the parameters to reflect the calculated parameters.
       */
      AVSmodify_float_parameter("XRotation", AVS_VALUE, xrot, 0.0, 0.0);
      AVSmodify_float_parameter("YRotation", AVS_VALUE, yrot, 0.0, 0.0);
      AVSmodify_float_parameter("ZRotation", AVS_VALUE, zrot, 0.0, 0.0);
      AVSmodify_float_parameter("ZDistance", AVS_VALUE, distz, 0.0, 0.0);
      AVSmodify_float_parameter("YDistance", AVS_VALUE, disty, 0.0, 0.0);
    }
  else
    {
      /*
       * Calculate the probe's transformation matrix from the module
       * parameters.  Matrix is formed from a series of simple transformations.
       * First the Domain_Centre is translated to the origin.
       */
      mat_translate(matrix, - S->Domain_Centre[0], - S->Domain_Centre[1],
		    - S->Domain_Centre[2]);

      /*
       * Then the probe is rotated about the z, y and x axes.
       */
      mat_z_rotate(zrot, m1);
      mat_multiply(matrix, m1, matrix);
      mat_y_rotate(yrot, m1);
      mat_multiply(matrix, m1, matrix);
      mat_x_rotate(xrot, m1);
      mat_multiply(matrix, m1, matrix);
      /*
       * Then the origin is translated back to the Domain_Centre.
       */
      for (i = 0; i < 3; i++)
	matrix[3][i] += S->Domain_Centre[i];
      /*
       * The position of the probe is disty and distz in the directions of the
       * respective normals.
       */
      for (i = 0; i < 3; i++)
	position[i] = distz * matrix[2][i] + disty * matrix[1][i];
    }

  /*
   * Set up the global definition of the probe's orientation.
   */
  /*
   * The first three columns of matrix are the Normals to the planes x=0,
   * y=0 and z=0.
   */
  vec_copy(S->Nx, matrix[0]);
  vec_copy(S->Ny, matrix[1]);
  vec_copy(S->Nz, matrix[2]);
  
  /*
   * The translation component of the probe is set according to distx and
   * disty.
   */
  S->Dy = disty + dot_product(S->Ny, S->Domain_Centre);
  S->Dz = distz + dot_product(S->Nz, S->Domain_Centre);
}


create_line_probe(static_data *S, GEOMedit_list elist)
/*
 * Creates a probe object to replace the current probe object in elist.
 */
{
  GEOMobj *sheaf, *xeq0;
  float v[2][3], c[2][3];
  
  /*
   * Create the probe object which consists of two transparent orthogonal
   * planes (z=0 and y=0) and a line (z=y=0) of a size proportional to the size
   * of the domain of the UCD structure.
   */
  sheaf = GEOMcreate_obj(GEOM_POLYTRI, NULL);
  xeq0 = GEOMcreate_obj(GEOM_POLYTRI, NULL);

  add_square(sheaf, S->Domain_Centre, S->Domain_Size * 0.707, 1, 0);
  add_square(sheaf, S->Domain_Centre, S->Domain_Size * 0.707, 2, 0);
  vec_copy(v[0], S->Domain_Centre);
  vec_copy(v[1], S->Domain_Centre);
  v[0][0] -= S->Domain_Size * 0.866;
  v[1][0] += S->Domain_Size * 0.866;
  vec_copy(c[0], White);
  vec_copy(c[1], White);
  GEOMadd_disjoint_line(xeq0, (float *)v, (float *)c, 2, GEOM_COPY_DATA);
  GEOMunion_extents(xeq0, sheaf);

  /*
   * Add objects to the elist list and then free their storage.
   */
  GEOMedit_geometry(elist, "Guide", sheaf);
  GEOMedit_geometry(elist, "Probe", xeq0);
  GEOMedit_parent(elist, "Guide", "Probe");
  GEOMedit_selection_mode(elist, "Guide", "ignore", 0);
  GEOMedit_transform_mode(elist, "Guide", "parent", 0);
  GEOMedit_center(elist, "Probe", S->Domain_Centre);
  GEOMdestroy_obj(sheaf);
  GEOMdestroy_obj(xeq0);

  /*
   * Set the probe object's colour and rendering mode.
   */
  GEOMedit_properties(elist, "Probe", 1.0, 0.0, 0.0, 0.0, ALPHA, White);
  GEOMedit_render_mode(elist, "Probe", "flat");
}


update_line_probe(static_data *S, GEOMedit_list *elist, upstream_transform
                  *trans_info, float xrot, float yrot, float zrot, float disty,
                  float distz, int do_define_probe)
/*
 * Creates a geometry edit list to create a probe object in the correct
 * position and orientation.
 */
{
  int i;
  float extent[6];
  float matrix[4][4];
  float position[3];
  
  get_line_probe_trans(S, trans_info, xrot, yrot, zrot, disty, distz, matrix,
		       position);
  
  if (trans_info && !do_define_probe) 
    {
      AVSmark_output_unchanged("probe");
      return;
    }

  /*
   * Initialise the edit list to be passed to the geometry viewer.
   */
  *elist = GEOMinit_edit_list(*elist);

  /*
   * Create a probe the probe object.
   */
  if (do_define_probe)
    create_line_probe(S, *elist);

  /*
   * Set the probe's matrix and position vector.
   */
  GEOMedit_transform_mode(*elist, "Probe", "normal", 0);
  GEOMedit_set_matrix(*elist, "Probe", (float *)matrix);
  GEOMedit_position(*elist, "Probe", position);


  /*
   * Set the probe object's transformation mode.
   */
#ifdef CONTINUAL_UPDATE
  GEOMedit_transform_mode(*elist, "Probe", "notify", BUTTON_UP |
			  BUTTON_MOVING);
#else
  GEOMedit_transform_mode(*elist, "Probe", "notify", BUTTON_UP);
#endif

  /*
   * Set the probe's centre of rotation.
   */
  GEOMedit_center(*elist, "Probe", S->Domain_Centre);

  /*
   * Set the window to a root3/2 times the maximum extent of the domain.
   */
  for (i=0; i<3; i++)
    {
      extent[2*i] = S->Domain_Centre[i] - S->Domain_Size * 0.866;
      extent[2*i+1] = S->Domain_Centre[i] + S->Domain_Size * 0.866;
    }
  GEOMedit_window(*elist, "Probe", extent);
}


/****************************************************************************
 *
 * Code to generate a sample line through the ucd structure.
 *
 */

node_info *test_against_z0(static_data *S, UCD_structure *ucd_struct, int
                           nnodes)
/*
 * function to allocate a table of information about the nodes used to make the
 * process of calculating the sample line more computationally efficient.  The
 * table contains the distance of the node from the plane Nz.P=0 and a flag
 * 'sflag' to indicate if the point is above or below the plane.  It also
 * contains space for the distance of the node from Ny.P=0, however, this is
 * only filled in if it is required.  A flag 'tflag' records whether or not
 * this distance has been calculated.
 */
{
  int i;
  float *x, *y, *z;		/* Structure's node positions */
  node_info *ntable;		/* The created table of node information */
  node_info *table_ptr;		/* Pointer into the node table */

  /*
   * Query ucd structure.
   */
  UCDstructure_get_node_positions(ucd_struct, &x, &y, &z);

  /*
   * Allocate storage space for information about each node.
   */
  ntable = (node_info *)calloc(nnodes, sizeof(node_info));

  /*
   * Compute distance between plane Nz.P=0 and nodes and set some flags to
   * speed computation.  'Sflag' is true if the node is in front of the plane
   * Nz.P=0.
   */
  table_ptr = ntable;
  for (i = 0; i < nnodes; i++)
    {
      table_ptr->s = (S->Nz[0] * *x++ + S->Nz[1] * *y++ + S->Nz[2] * *z++ -
		      S->Dz);
      table_ptr->sflag = (table_ptr->s >= 0.0);
      table_ptr->tflag = 0;
      table_ptr++;
    }

  return ntable;
}


init_line_store(static_data *S, UCD_structure *ucd_struct, int nnodes,
                line_store *ls)
/*
 * Initialises the data structure used to calculate and store the sample line.
 */
{
  ls->ntable = test_against_z0(S, ucd_struct, nnodes);
  ls->points = 0;
  ls->root = 0;
}


int test_against_y0(static_data *S, int *nodes, int nvert, UCD_structure
                    *ucd_struct, line_store *ls)
/*
 * Function to ensure that the distance between the nodes held in 'nodes',
 * which are the nodes of a cell with 'nvert' vertices, and the plane Ny.P=0
 * has been calculated and stored in 'ls'.  The function returns true if the
 * nodes lie on both sides of the plane, indicating that the cell to which the
 * nodes belong is intersected by Ny.P=0.
 */
{
  register int i;
  int pos_verts;
  node_info *nt;
  
  pos_verts = 0;
  
  i = nvert;
  while (i--)
    {
      nt = ls->ntable + *nodes;
      if (! nt->tflag)
	{
	  nt->tflag = 1;
	  nt->t = (S->Ny[0] * ucd_struct->x[*nodes] +
		   S->Ny[1] * ucd_struct->y[*nodes] +
		   S->Ny[2] * ucd_struct->z[*nodes] - S->Dy);
	}
      if (nt->t >= 0.0) pos_verts++;
      nodes++;
    }
  return (pos_verts > 0 && pos_verts < nvert);
}


line_store_entry *insert_line_store_entry(static_data *S, line_store_entry
                                          **root, float lambda)
/*
 * Function that inserts a new line_store_entry, representing a point on the
 * sample line at C + lambda.Nx, into the binary tree of line points.  The
 * point is only inserted if it is over Domain_Delta away from any other point
 * on the line.
 */
{
  line_store_entry *leaf;

  if (! *root)
    {
      leaf = (line_store_entry *)malloc(sizeof(struct _line_store_entry));
      leaf->lambda = lambda;
      leaf->left = 0;
      leaf->right = 0;
      return *root = leaf;
    }

  if (lambda < (*root)->lambda - S->Domain_Delta)
    return insert_line_store_entry(S, &(*root)->left, lambda);
  if (lambda > (*root)->lambda + S->Domain_Delta)
    return insert_line_store_entry(S, &(*root)->right, lambda);
  return 0;
}


add_point(static_data *S, int n[4], float s0, float s1, float s2, 
          float s3, float t0, float t1, UCD_structure *ucd_struct, 
          line_store *ls)
/*
 * Adds a point to the sample line held in 'ls' derived by bi-linear
 * interpolation of the 4 nodes in 'n' along the plane Nz.P=0 and Ny.P=0.  The
 * parameters 's0' to 's3' are the distances of the nodes n[0] to n[3] from the
 * plane Nz.P=0 and 't0' and 't1' are the distances from Ny.P=0 of the
 * intersections of the lines n[0]n[1] and n[2]n[3] with Nz.P=0 respectively.
 */
{
  int i;
  float w[4], wtot;
  float lambda, value;
  line_store_entry *new_entry;
  
  /*
   * Calculate the interpolation weights for the point of intersection.  (Uses
   * straightforward bi-linear interpolation in Nz.P=0 and then Ny.P=0).
   */
  w[0] = t1*s1;
  w[1] = -t1*s0;
  w[2] = -t0*s3;
  w[3] = t0*s2;
  wtot = w[0] + w[1] + w[2] + w[3];

  /*
   * Calculate the position along the line of the point of intersection of the
   * line and the face.
   */
  lambda = 0.0;
  for (i = 0; i < 4; i++)
    {
      lambda += w[i] * S->Nx[0] * ucd_struct->x[n[i]];
      lambda += w[i] * S->Nx[1] * ucd_struct->y[n[i]];
      lambda += w[i] * S->Nx[2] * ucd_struct->z[n[i]];
    }
  lambda = lambda / wtot - dot_product(S->Domain_Centre, S->Nx);

  /*
   * Try to insert a new point to the tree of points already found.
   */
  new_entry = insert_line_store_entry(S, &ls->root, lambda);

  /*
   * If the new point did not clash with an existing point calculate and fill
   * in its associated data value and increment the points count.
   */
  if (new_entry)
    {
      value = 0.0;
      for (i = 0; i < 4; i++)
	value += w[i] * S->Range[n[i]];
      new_entry->value = value / wtot;
      ls->points++;
    }
}


add_face_intersections(static_data *S, int cell, UCD_structure *ucd_struct,
                       line_store *ls)
/*
 * Test the given cell of the UCD structure for intersection with the sample
 * line and then add the appropriate face intersections to the line store.
 * To do this the cell is first tested for intersection with the plane Nz.P=0.
 * If this is so the cell is cut to yield a polygon of 3 to six sides.  This
 * polygon is then tested for intersection with the plane Ny.P=0 to yield a
 * maximum of two points on the sample line.
 */
{
  register int i, bit;		/* Temporary variables and pointers */
  int *nptr, *nptr2;

  int nvert;			/* Number of vertices in the cutting polygon */
  int nodes[16];		/* A list of pairs of nodes defining the */
				/* cutting polygon */
  node_info *nt;
  register int s;		/* Cell's vertices bit map. One bit for each */
				/* node: 1->above, 0->below plane Nz.P=0 */
  cutting_info *stable;		/* A pointer to the current cutting case of */
				/* the cell with Nz.P=0 */

  float s0, s1, old_s0, old_s1;	/* Distances of cell nodes from Nz.P=0 */
  float t, old_t;		/* Distances of polygon vertices from Ny.P=0 */
  int tflag, old_tflag;		/* Whether polygon vertices are above Ny.P=0 */
  
  char elem_type[9];		/* Variables describing UCD cell info */
  int name, mat_type, cell_type;
  int mid_edge_flags, *node_list;

  /*
   * Find out what type of cell we are dealing with.
   */
  UCDcell_get_information(ucd_struct, cell, &name, elem_type, &mat_type,
			  &cell_type, &mid_edge_flags, &node_list);
  
  /*
   * Firstly the cell is sliced with the plane Nz.P=0, requiring that we
   * determine which cutting case we're in.  A cutting case is numbered
   * according to a bit map where each bit represents a node.  A 1 implies the
   * node is above the plane Nz.P=0, a zero that the node is below.
   */
  s = 0;
  bit = Ucd_Num_Nodes[cell_type];
  if (!bit) return;
  nptr = node_list + bit;
  nt = ls->ntable;
  while (bit--)
      s |= (nt[*--nptr].sflag & 1) << bit;

  /*
   * Use the predefined tables to find the number of vertices the cutting
   * polygon has.  If cell is not cut (nvert=0) return.
   */
  stable = Ucd_Cutting_Info[cell_type] + s;
  nvert = stable->nvert;
  if (!nvert) return;
  
  /*
   * If the cell is cut by Nz.P=0 the distance of all the cell vertices from
   * Ny.P=0 is used to determine if the cell is cut by Ny.P=0 as well.  If this
   * is so then the cell is probably (but not always) cut by the sample line.
   */

  if (! test_against_y0(S, node_list, Ucd_Num_Nodes[cell_type], ucd_struct,
			ls))
    return;
  
  /*
   * The cutting of the cell by Nz.P=0 yields a polygon that is then tested for
   * intersection against Ny.P=0.  To do this the pairs of cell vertices that
   * are interpolated between to give the polygon vertices are copied into a
   * circular array.
   */

  nptr = nodes;
  nptr2 = stable->node_triples;
  i = nvert;
  while (i--)
    {
      *nptr++=node_list[*nptr2++];
      *nptr++=node_list[*nptr2++];
      nptr2++;
    }
  *nptr++ = node_list[stable->node_triples[0]];
  *nptr++ = node_list[stable->node_triples[1]];
  
  /*
   * The polygon points, each described by a pair of cell vertices, are then
   * examined to find which, if any, of the polygon edges cut the plane Ny.P=0.
   * This is identified by a sign change in 't' (the distance of the
   * intersection point of the cell edge and Nz.P=0 from Ny.P=0) between
   * consecutive polygon vertices.
   */

  s0 = nt[nodes[0]].s;
  s1 = nt[nodes[1]].s;
  t = (s1 * nt[nodes[0]].t - s0 * nt[nodes[1]].t) / (s1 - s0);
  tflag = (t >= 0.0);

  nptr = nodes;
  i = nvert;
  while (i--)
    {
      old_s0 = s0;
      old_s1 = s1;
      old_t = t;
      old_tflag = tflag;

      s0 = nt[nptr[2]].s;
      s1 = nt[nptr[3]].s;
      t = (s1 * nt[nptr[2]].t - s0 * nt[nptr[3]].t) / (s1 - s0);
      tflag = (t >= 0.0);
      
      /*
       * If the distance between consecutive points on the polygon and Ny.P=0
       * changes sign then that edge is intersected to give a point on the
       * sample line.
       */
      if (tflag ^ old_tflag)
	add_point(S, nptr, old_s0, old_s1, s0, s1, old_t, t, ucd_struct, ls);

      nptr += 2;
    }
}


fill_in_irregular_field(float **data_ptr, float **coord_ptr, line_store_entry
                        *entry)
/*
 * Flattens out the coordinate/data pairs held in the binary tree into the two
 * linear floating point arrays given.
 */
{
  if (entry)
    {
      fill_in_irregular_field(data_ptr, coord_ptr, entry->left);
      *(*data_ptr)++ = entry->value;
      *(*coord_ptr)++ = entry->lambda;
      fill_in_irregular_field(data_ptr, coord_ptr, entry->right);
    }
}


fill_in_graph_viewer_field(float **data_ptr, line_store_entry *entry)
/*
 * Flattens out the coordinate/data pairs held in the binary tree into the
 * single linear floating point array given.
 */
{
  if (entry)
    {
      fill_in_graph_viewer_field(data_ptr, entry->left);
      *(*data_ptr)++ = entry->lambda;
      *(*data_ptr)++ = entry->value;
      fill_in_graph_viewer_field(data_ptr, entry->right);
    }
}


int get_dims(float *lambda, int n)
{
  int i, m;
  float gap, min_dist;
  
  min_dist = lambda[1] - lambda[0];
  i = 2;
  while (i < n)
    {
      gap = lambda[i] - lambda[i-1];
      if (gap < min_dist)
	min_dist = gap;
      i++;
    }
  
  m = ((int) ((lambda[n-1] - lambda[0]) / min_dist)) + 1;
  
  return (m <= n * 5) ? m : n * 5;
}


convert_to_uniform_field(float *data_ptr, int m, float *lambda, float *values,
                         int n)
/*
 * Uses linear interpolation to convert 1D irregularly spaced data to 1D
 * uniformly spaced data.  'data_ptr' is the output array of 'm' elements.
 * 'lambda' is the coordinate array for 'values' - the irregularly spaced data
 * of which there are 'n' items.
 */
{
  int i;
  float inc;
  float x;
  
  inc = (lambda[n-1] - lambda[0]) / (m - 1);
  x = lambda[0] + inc;
  *data_ptr++ = *values;
  i = 1;
  m -= 2;
  while (i < n)
    {
      while (x < lambda[i] && m)
	{
	  *data_ptr++ = ((lambda[i] - x) * values[i-1] - (lambda[i-1] - x) *
			 values[i]) / (lambda[i] - lambda[i-1]);
	  x += inc;
	  m--;
	}
      i++;
    }
  *data_ptr++ = values[n-1];  
}


convert_line_store_to_field(static_data *S, line_store *ls, AVSfield_float
                            **sample_out, int graph_viewer, int uniform)
/*
 * Converts the sample line, as held in the internal 'line_store' structure
 * into an AVS field.  If 'graph_viewer' is true the graph will be compatible
 * with AVS's graph viewer module.  If 'uniform' is true the sample line, which
 * by default has an irregular coordinate system, is interpolated to a uniform
 * spacing.
 */
{
  int dims[2];
  float min[2], max[2];
  float *lambda, *values;
  float *data_ptr, *coord_ptr;
  
  if (*sample_out)
    AVSfield_free((AVSfield *)*sample_out);

  if (uniform)
    {
      /*
       * First flatten the tree of data pairs into two temporary arrays.
       */
      values = (float *)calloc(ls->points, sizeof(float));
      lambda = (float *)calloc(ls->points, sizeof(float));

      data_ptr = values;
      coord_ptr = lambda;
      fill_in_irregular_field(&data_ptr, &coord_ptr, ls->root);

      /*
       * Find a suitable number of points to interpolate to.
       */
      dims[0] = get_dims(lambda, ls->points);
      
      /*
       * The output field is a simple list of data values without coordinates.
       */
      *sample_out = (AVSfield_float *)
	AVSdata_alloc("field 1D scalar real 1-coord uniform", dims);
      
      /*
       * Interpolate from the irregularly spaced points.
       */
      convert_to_uniform_field((*sample_out)->data, dims[0], lambda, values,
			       ls->points);

      /*
       * Establish the extents of space and data.
       */
      min[0] = lambda[0];
      max[0] = lambda[ls->points - 1];
      AVSfield_set_extent((AVSfield *)*sample_out, min, max);
      min[0] = S->Range_Min;
      max[0] = S->Range_Max;
      AVSfield_set_minmax((AVSfield *)*sample_out, min, max);
      
      /*
       * Free temporary storage.
       */

/* IAC CODE CHANGE :       free(lambda); */

/* IAC CODE CHANGE :        free(lambda); */
        free(lambda);

/* IAC CODE CHANGE :       free(values); */

/* IAC CODE CHANGE :        free(values); */
        free(values);
    }
  else if (graph_viewer)
    {
      /*
       * Output field is a 2D array of data values with no coordinates.  The
       * first dimension of the array is 2 so the array is basically a list of
       * (x,y) pairs to plot.
       */
      dims[0] = 2;
      dims[1] = ls->points;

      *sample_out = (AVSfield_float *)
	AVSdata_alloc("field 2D scalar real 1-coord uniform", dims);
      
      data_ptr = (*sample_out)->data;
      
      fill_in_graph_viewer_field(&data_ptr, ls->root);

      AVSfield_invalid_minmax((AVSfield *)*sample_out);
    }
  else
    {
      /*
       * Output field is a 1D array of floating point values with coordinates
       * in 1-space.
       */
      dims[0] = ls->points;
      
      *sample_out = (AVSfield_float *)
	AVSdata_alloc("field 1D scalar real 1-coord rectilinear", dims);
      
      data_ptr = (*sample_out)->data;
      coord_ptr = (*sample_out)->points;
      
      fill_in_irregular_field(&data_ptr, &coord_ptr, ls->root);

      /*
       * Establish the extents of space and data.
       */
      min[0] = (*sample_out)->points[0];
      max[0] = (*sample_out)->points[ls->points - 1];
      AVSfield_set_extent((AVSfield *)*sample_out, min, max);
      min[0] = S->Range_Min;
      max[0] = S->Range_Max;
      AVSfield_set_minmax((AVSfield *)*sample_out, min, max);
    }
}


free_line_store_entry(line_store_entry *entry)
/*
 * Deallocates dynamic memory associated with the line store.
 */
{
  if (entry)
    {
      free_line_store_entry(entry->left);
      free_line_store_entry(entry->right);

/* IAC CODE CHANGE :       free(entry); */

/* IAC CODE CHANGE :        free(entry); */
        free(entry);
    }
}


free_line_store(line_store *ls)
{

/* IAC CODE CHANGE :   free(ls->ntable); */

/* IAC CODE CHANGE :    free(ls->ntable); */
    free(ls->ntable);
  free_line_store_entry(ls->root);
}


get_line(static_data *S, UCD_structure *ucd_struct, AVSfield_float
         **sample_out, int graph_viewer, int uniform)
/*
 * Interpolates from the selected component of node data (defined globally) of
 * the given UCD mesh to an arbitrary line (defined globally) and outputs the
 * results as two AVS fields.
 */
{
  int i;
  line_store ls;

  char name[20];		/* Variables for reading the UCD structure */
  int data_veclen, name_flag;
  int ncells, cell_veclen, nnodes, node_veclen;
  int util_flag;

  /*
   * Get the number of cells and nodes in the structure
   */
  UCDstructure_get_header(ucd_struct, name, &data_veclen, &name_flag, &ncells,
			  &cell_veclen, &nnodes, &node_veclen, &util_flag);

  /*
   * The simple implementation adopted here is to simply test every cell in the
   * UCD structure, ensuring that cells which are not intersected are discarded
   * quickly.  A more efficient approach for larger structures would to be to
   * compute connectivity information and use it to follow the line through the
   * structure.
   */

  init_line_store(S, ucd_struct, nnodes, &ls);

  for (i=0; i<ncells; i++)
    add_face_intersections(S, i, ucd_struct, &ls);

  if (ls.points)
    convert_line_store_to_field(S, &ls, sample_out, graph_viewer, uniform);
  else
    AVSmark_output_unchanged("sample");
  
  free_line_store(&ls);
}


/******************************************************************************
 *
 * Functions to decode inputs and parameter changes.
 *
 */

int stat_ucd_line_data(static_data *S, UCD_structure *ucd_struct, char
                       *component)
/*
 * Check that the ucd structure exists and if it has changed find its extent.
 */
{
  int ok, i;
  float dom_min[3], dom_max[3];	/* Variables to read domain extent */
  float *rng_min, *rng_max;	/* Variables to read range extent */
  char str100[100];		/* Variables to read names, labels etc. */
  char label[20];
  char delimiter[2];
  int data_veclen, name_flag;	/* Values to hold info about ucd_struct */
  int ncells, cell_veclen;
  int nnodes, node_veclen;
  int util_flag, *comps;
  int comp_number, range_pos;

  /*
   * Query ucd structure
   */
  ok = UCDstructure_get_header(ucd_struct, str100, &data_veclen,
				&name_flag, &ncells, &cell_veclen,
				&nnodes, &node_veclen, &util_flag);
  if (!ok)
    {
      AVSerror("Error: Couldn't get structure information");
      return 0;
    }

  /*
   * Decode choice number in terms of current 'Node Data' selection.
   */
  comp_number = AVSchoice_number("Node Data", component)-1;
  
  if (AVSinput_changed("ucd", 0))
    {
      /*
       * Find the centre, extent and size of the domain.
       */
      ok = UCDstructure_get_extent(ucd_struct, dom_min, dom_max);
      if (!ok)
	{
	  AVSerror("Error: Couldn't find domain's extent");
	  return 0;
	}

      for (i = 0; i < 3; i++)
	S->Domain_Centre[i] = (dom_max[i] + dom_min[i])/2;
      S->Domain_Size = 0.0;
      for (i = 0; i < 3; i++)
	{
	  S->Domain_Extent[2*i] = dom_min[i];
	  S->Domain_Extent[2*i+1] = dom_max[i];
	  if (S->Domain_Size < dom_max[i] - dom_min[i])
	    S->Domain_Size = dom_max[i] - dom_min[i];
	}
      S->Domain_Delta = S->Domain_Size * MIN_DIST;

      /*
       * Set Distance parameters limits to reflect domain size.
       */
      AVSmodify_float_parameter("YDistance", AVS_MINVAL | AVS_MAXVAL, 0.0,
				- S->Domain_Size, S->Domain_Size);
      AVSmodify_float_parameter("ZDistance", AVS_MINVAL | AVS_MAXVAL, 0.0,
				- S->Domain_Size, S->Domain_Size);

      /*
       * Make the 'Node Data' parameter reflect the choice of node data within
       * the UCD structure.
       */
      if (node_veclen)
	{
          for (i=0; i<100; i++)
	    str100[i] = 0;
	  UCDstructure_get_node_labels(ucd_struct, str100, delimiter);
	  UCDstructure_get_node_label(ucd_struct, 0, label);
	  AVSmodify_parameter("Node Data", AVS_VALUE | AVS_MINVAL | AVS_MAXVAL,
			      label, str100, delimiter);
	  comp_number = 0;
	}
      else
	AVSmodify_parameter("Node Data", AVS_VALUE | AVS_MINVAL | AVS_MAXVAL,
			    "<none>", "<none>", ";");
    }

  if (node_veclen > 0 && comp_number > -1)
    {
      /*
       * Find and store details about the range - that is the data component to
       * sampled along the line.
       */
      S->Range = 0;
      rng_min = (float *)malloc(node_veclen * sizeof(float));
      rng_max = (float *)malloc(node_veclen * sizeof(float));
      
      /*
       * Get information about the node data from the UCD structure.
       */
      ok = (UCDstructure_get_node_components(ucd_struct, &comps) &&
	    UCDstructure_get_node_minmax(ucd_struct, rng_min, rng_max) &&
	    UCDstructure_get_node_data(ucd_struct, &S->Range));

      /*
       * If the selected component from the UCD node data structure pick it out
       * and fill in the global variables describing it.
       */
      if (ok && comps[comp_number] == 1)
	{
	  /*
	   * Locate the actual array of data and its bounding values.
	   */
	  i = 0;
	  range_pos = 0;
	  while (i < comp_number)
	    {
	      range_pos += comps[i];
	      i++;
	    }
	  S->Range += range_pos*nnodes;
	  S->Range_Min = rng_min[range_pos];
	  S->Range_Max = rng_max[range_pos];
	  
	  /*
	   * Work out a scaling value to make projected graphs look
	   * approximately square.
	   */
	  S->Range_Scale = S->Domain_Size / (S->Range_Max - S->Range_Min);
	}
      else
	{
	  AVSerror("Error: Selected component is not a scalar.");
	  S->Range = 0;
	}


/* IAC CODE CHANGE :       free(rng_min); */

/* IAC CODE CHANGE :        free(rng_min); */
        free(rng_min);

/* IAC CODE CHANGE :       free(rng_max); */

/* IAC CODE CHANGE :        free(rng_max); */
        free(rng_max);
    }
  return 1;
}


/*****************************************************************************
 *
 * The computation function of the module.
 */

int ucd_line_compute(UCD_structure *ucd_struct, upstream_transform *trans_info,
                     GEOMedit_list *probe_out, AVSfield_float **sample_out, int
                     do_sample, int graph_viewer, int uniform, float *xrot,
                     float *yrot, float *zrot, float *disty, float *distz, char
                     *component)
/*
 * Module's main computation function.
 */
{
  static_data *S;
  int trans_info_flag;
  int new_ucd;
  int force_regenerate_sample;
    
  S = (static_data *)AVSstatic;

  /*
   * Check the inputs and parameters are in a well defined state.
   */

  if (!ucd_struct) return 0;

  if ((new_ucd = AVSinput_changed("ucd", 0)) ||
      AVSparameter_changed("Node Data"))
    if (!stat_ucd_line_data(S, ucd_struct, component))
      return 0;

  trans_info_flag = get_trans_info_flag(trans_info);

  force_regenerate_sample = 0;
    
  /*
   * If necessary update the probe geometry.
   */

  if (new_ucd ||
      trans_info_flag ||
      AVSparameter_changed("XRotation") ||
      AVSparameter_changed("YRotation") ||
      AVSparameter_changed("ZRotation") ||
      AVSparameter_changed("YDistance") ||
      AVSparameter_changed("ZDistance"))
    {
      update_line_probe(S, probe_out, trans_info_flag ? trans_info : 0,
			*xrot, *yrot, *zrot, *disty, *distz, new_ucd);

#if defined(CONTINUAL_SAMPLE) && defined(CONTINUAL_UPDATE)
      force_regenerate_sample = 1;
#else
      force_regenerate_sample = !(trans_info_flag & BUTTON_MOVING);
#endif
    }
  else
    AVSmark_output_unchanged("probe");

  /*
   * If necessary generate the sample field.
   */
  
  if (do_sample &&
      (force_regenerate_sample ||
       AVSparameter_changed("Do Sample") ||
       AVSparameter_changed("Graph Viewer") ||
       AVSparameter_changed("Uniform") ||
       AVSparameter_changed("Node Data")))
    get_line(S, ucd_struct, sample_out, graph_viewer, uniform);
  else
    AVSmark_output_unchanged("sample");

  /*
   * Return success.
   */
  return 1;
}


ucd_line_init(void)
{
  AVSstatic = malloc(sizeof(struct _static_data));
}


ucd_line_fini(void)
{

/* IAC CODE CHANGE :   free(AVSstatic); */

/* IAC CODE CHANGE :    free(AVSstatic); */
    free(AVSstatic);
  AVSstatic = 0;
}


/*****************************************************************************
 *
 * Module description functions.
 */

ucd_line_desc(void)
{
  int param, port;

  AVSset_module_name("ucd line", MODULE_MAPPER);
  AVSset_module_flags(COOPERATIVE | REENTRANT);

  AVScreate_input_port("ucd", "ucd", OPTIONAL);
  port = AVScreate_input_port("transformation info",
			      "struct upstream_transform",
			      OPTIONAL | INVISIBLE);
  AVSset_input_class(port, "probe:upstream_transform");

  AVScreate_output_port("probe", "geom");
  AVScreate_output_port("sample", "field scalar real");

  AVSadd_parameter("Do Sample", "boolean", 0, 0, 0);
  AVSadd_parameter("Graph Viewer", "boolean", 1, 0, 0);
  AVSadd_parameter("Uniform", "boolean", 0, 0, 0);
  AVSadd_float_parameter("XRotation", 0.0, -180.0, 180.0);
  AVSadd_float_parameter("YRotation", 0.0, -180.0, 180.0);
  AVSadd_float_parameter("ZRotation", 0.0, -180.0, 180.0);
  AVSadd_float_parameter("YDistance", 0.0, -5.0, 5.0);
  AVSadd_float_parameter("ZDistance", 0.0, -5.0, 5.0);

  param = AVSadd_parameter("Node Data", "choice", "<none>", "<none>", ";");
  AVSadd_parameter_prop(param, "width", "integer", 2);

  AVSset_compute_proc(ucd_line_compute);
  AVSset_init_proc(ucd_line_init);
  AVSset_destroy_proc(ucd_line_fini);
}


#ifdef sep_exe

AVSinit_modules(void)
{
  AVSmodule_from_desc(ucd_line_desc);
}

#endif
