/****************************************************************************
                  INTERNATIONAL AVS CENTER
	(This disclaimer must remain at the top of all files)

WARRANTY DISCLAIMER

This module and the files associated with it are distributed free of charge.
It is placed in the public domain and permission is granted for anyone to use,
duplicate, modify, and redistribute it unless otherwise noted.  Some modules
may be copyrighted.  You agree to abide by the conditions also included in
the AVS Licensing Agreement, version 1.0, located in the main module
directory located at the International AVS Center ftp site and to include
the AVS Licensing Agreement when you distribute any files downloaded from 
that site.

The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module provide absolutely
NO WARRANTY OF ANY KIND with respect to this software.  The entire risk as to
the quality and performance of this software is with the user.  IN NO EVENT
WILL The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module BE LIABLE TO
ANYONE FOR ANY DAMAGES ARISING FROM THE USE OF THIS SOFTWARE, INCLUDING,
WITHOUT LIMITATION, DAMAGES RESULTING FROM LOST DATA OR LOST PROFITS, OR ANY
SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES.

This AVS module and associated files are public domain software unless
otherwise noted.  Permission is hereby granted to do whatever you like with
it, subject to the conditions that may exist in copyrighted materials. Should
you wish to make a contribution toward the improvement, modification, or
general performance of this module, please send us your comments:  why you
liked or disliked it, how you use it, and most important, how it helps your
work. We will receive your comments at avs@ncsc.org.

Please send AVS module bug reports to avs@ncsc.org.

******************************************************************************/
#ifdef SCCS
static char sccsid[]="@(#)ucd_planes.c	5.0  Jonathan Cox 21/08/92";
#endif
/*
 * FILE: ucd_planes.c
 *
 * MODULE
 * 
 * ucd planes - investigation of UCD structures with planar probes
 * 
 * DESCRIPTION
 * 
 * The UCD PLANES module supports the interactive sub-sampling of node data
 * held within a 3D UCD structure.  The region sampled is selected
 * interactively in the region of a transparent 'probe' object.  Several
 * different types of probes are supported, all based upon a collection of
 * planes.
 *
 * The selection of probe type is made from the 'Probe Type' choice widget on
 * the module's control panel.  Possible probes include a single plane
 * (SIMPLE), two and three crossed orthogonal planes (SHEAF and CROSSED) and
 * multiple parallel planes (MULTIPLE).  These probe types produce a 'sample'
 * that is the same shape as the probe object but is coloured according to a
 * scalar component of the structure's nodal data.
 *
 * In addition, a further probe type (PROJECTED) is supported which uses a
 * probe consisting of a single plane just as with the SIMPLE plane probe.
 * However, the sample produced is not a simple coloured display of the slice
 * plane.  Instead of this, another component of nodal data is used to add
 * height to the sample.  This forms a 3D graph of the data on the sampled
 * plane.  Axes, tick marks and labels can be automatically generated for the
 * graph.  The number of tick marks is controllable.  It is also possible to
 * add a flat wire-frame rendering of the sliced UCD mesh to the base of the
 * graph.  This makes the perception of surface height clearer.  An optional
 * input port is provided, that accepts a user defined 'scale_info' data type.
 * This is to allow the correct labeling of data that has been mapped onto a
 * logarithmic scale.
 *
 * For all probe types the orientation and position of the probe can be
 * controlled in two ways.  The first method is to use the dial widgets on the
 * module's control panel.  The XRotation, YRotation and ZRotation parameters
 * control the orientation of the plane as three rotations about the major
 * axes.  The rotations are applied in the order ZYX.  The Scale parameter
 * specifies a uniform scaling factor.  The Distance parameter is a positional
 * offset.  Due to the varying make up of the different types of probes, some
 * parameters are of limited use with some probe types.  This is described in
 * more detail below.
 *
 * The alternative method of controlling the orientation and position of the
 * probe is by direct manipulation of the probe object within the geometry
 * viewer.  During direct manipulation the dial parameters are calculated from
 * the new transformation matrix for the probe object.  This ensures that the
 * dial parameters and the probe's orientation are always mutually consistent.
 *
 * The correspondence between transformation by the dials and by direct
 * manipulation is not 1 to 1.  This is because, by default, it is possible to
 * translate objects in the geometry viewer in 3D where as there is only one
 * dial controlling position of the probe - the Distance dial.
 *
 * This restriction is fine for those probe types (SIMPLE, MULTIPLE and
 * PROJECTED) in which all constituent planes are parallel to each other.  As
 * translation parallel to the planes is irrelevant to the definition of the
 * probe, translation can be restricted to being in the direction of the
 * planes' normals without any loss of generality.  In fact, this is desirable
 * as it ensures that the normals always pass through the centre of the UCD
 * structure.  With this in mind, translations in the geometry viewer for these
 * probe types are restricted to the direction of the probe's normal.  This
 * feature can be turned off with a compilation option as it does not work well
 * on AVS platforms which are not highly interactive - such as an X terminal.
 *
 * For the SHEAF and CROSSED probe types translation in 3D becomes significant.
 * For these probe types 3D positioning is only possible through direct
 * manipulation of the probe object within the geometry viewer window or from
 * the geometry viewer's transformation options panel.
 * 
 * NOTICE
 *
 * (c) University of Manchester 1991, 1992.
 * Author:  Jonathan Cox.
 * Address: IT301, Department of Computer Science, University of Manchester,
 *          Oxford Road, Manchester.  M13 9PL.  England.
 * Email:   coxjp@cs.man.ac.uk
 *
 * No part of this software may be altered, incorporated in other programs or
 * given to third parties, except under the conditions of the license, if any,
 * with which it was distributed.  This notice may not be removed from the
 * source code with which it is associated.
 *
 * VERSION HISTORY
 *
 * 5.0 - Jonathan Cox.  First version for general release.
 * 
 */

/*
 * Include Files.
 */

#include <stdio.h>
/* IAC CODE CHANGE : #include <math.h> */
/* IAC CODE CHANGE : #include <avs/avs_math.h> */
#include <avs/avs_math.h>
#include <avs.h>
#include <field.h>
#include <geom.h>
#include <ucd_defs.h>
#include <udata.h>

/*
 * Matrix/vector manipulation routines.
 */
#include "mat.h"
/*
 * A user-defined AVS type detailing which, if any, of the components of data
 * for the UCD module are in a logarithmic scale.
 */
#include "scale_info.h"
/*
 * Tables of pre-calculated data detailing how to form the polygons of a slice
 * plane from intersected UCD cells given that it is known which cell vertices
 * are above and below the plane.  All 3D UCD cell types are supported.
 */
#include "cut_ucd.h"

/*****************************************************************************
 *
 * Type definitions.
 */

typedef struct _plane {
  int axis;			/* The axis which the plane cuts */
  float centre[3];		/* The plane's centre point */
} plane;

/*
 * Structure to hold information about nodes in the structure necessary for the
 * calculation of a cutting plane.
 */
typedef struct _node_info {
  float s;			/* Distance between node and plane */
  int flag;			/* Status flag */
  int targets[12];		/* Target node of edges from this node */
  int indices[12];		/* Indices of the vertices on these edges */
} node_info;

/*
 * Structure to hold the intersecting 'sample' plane as it is constructed.
 * Information contains: 'node_info' for each node - detailing which edges are
 * intersected, and a list of polygons which make up the sample - stored as a
 * list of indices indirectly referencing shared vertices, colours and normals.
 */
typedef struct _sstore {
  node_info *ntable;		/* Table of information for nodes */
  int *next_index;		/* A pointer to the next slot in 'indices' */
  int *indices;			/* A list of polygon indices for the sample */
  int next_vertex;		/* The next vertex to be allocated */
  float (*vertices)[3];		/* A list of vertices for the sample */
  float (*colours)[3];		/* A list of colours for the vertices */
  float (*normals)[3];		/* A list of normals for the vertices */
} sstore;

/*****************************************************************************
 *
 * Global Defaults.
 */

/*
 * The module usually modifies the upstream probe transformation matrix it
 * receives so that a) the plane normal always passes through the centre of the
 * UCD structure and b) the distance parameter is kept constant during rotates
 * and scales.  If NORMAL_TRANSFORMS is defined these features are turned off.
 *
 * If REDIRECT is defined the transformation of the probe object is controlled
 * by this module rather than in the geometry viewer.  This makes it physically
 * impossible to translate the probe other than along the plane normal (thus
 * ensuring that the normal always passes through the center of the structure)
 * and secondly the centre of rotation is always the centre of the UCD
 * structure (thus preserving distance parameter during rotations and scales).
 * Unfortunately it also makes the position of the bounding box appear pretty
 * arbitrary.  Therefore, it is not particularly suitable for use on a non
 * responsive AVS platform such as an X terminal.  REDIRECT cannot be defined
 * if NORMAL_TRANSFORMS is.
 *
 * If neither REDIRECT nor NORMAL_TRANSFORMS is defined it will be possible to
 * transform normally but there will be a second transformation of the probe,
 * once the mouse button is released, to ensure that the plane normal passes
 * through the centre and that the distance parameter is preserved.
 * 
 * CONTINUAL_UPDATE will cause the dial parameters to be continually updated
 * while the probe object is being transformed.  If, in addition,
 * CONTINUAL_SAMPLE is defined the sample will be continually regenerated while
 * the probe is being transformed.
 * 
 * CONTINUAL_UPDATE and CONTINUAL_SAMPLE work best with either REDIRECT or
 * NORMAL_TRANSFORMS defined as otherwise the second plane movement and time
 * lag interferes with the transformation of the probe.
 */

#if AVS_VERSION == 4
#define REDIRECT
#else
#define NORMAL_TRANSFORMS
#endif

#define CONTINUAL_UPDATE
#define CONTINUAL_SAMPLE

#if defined (NORMAL_TRANSFORMS) && defined(REDIRECT)
#undef REDIRECT
#endif

#if defined (CONTINUAL_SAMPLE)
#define CONTINUAL_UPDATE
#endif

/*
 * If MID_EDGE_NODES is defined then they will be taken into consideration,
 * otherwise they are ignored.  Support for mid edge nodes is very simple.
 * Linear interpolation is used rather than a higher order interpolation scheme
 * and the algorithm will not detect edges that are intersected twice by the
 * plane.
 */
/* #define MID_EDGE_NODES */

/*
 * PRECISION is a small floating point value used to ensure that comparisons
 * for equality are interpreted correctly even if the values are slightly
 * inaccurate.
 */
#define PRECISION 1.0e-4


/*
 * Default camera viewing position and scale for a projected plane
 */
#define PROJECTED_ZROT 80.0
#define PROJECTED_XROT -80.0
#define PROJECTED_VSCALE 0.7

/*
 * Default camera scale for viewing simple slice planes
 */
#define SLICE_VSCALE 0.9

/*
 * Default transparency alpha coefficient for the probe object
 */
#define ALPHA 0.3

/*
 * Default number of planes and ticks and maximum values
 */
#define DEFAULT_PLANES 5
#define MAX_PLANES 10
#define DEFAULT_TICKS 10
#define MAX_TICKS 20

/*
 * Offsets and sizes for various parts of the 3D graph
 */
#define Z_AXIS_OFFSET 0.0
#define XY_AXIS_OFFSET 0.0
#define AXIS_OFFSET 0.05
#define AXIS_LABEL_OFFSET 0.1
#define TICK_LABEL_OFFSET 0.03
#define AXES_LABEL_HEIGHT 0.05
#define TICK_LABEL_HEIGHT 0.04
#define TITLE_LABEL_HEIGHT 0.15
#define TICK_LENGTH 0.03

/*****************************************************************************
 *
 * Global definitions
 */

/*
 * The different types of probes.
 */
#define SIMPLE 1
#define CROSSED 2
#define SHEAF 3
#define MULTI 4
#define PROJECTED 5

static char Options[] = "Simple;Crossed;Sheaf;Multiple;Projected";

/*
 * Coordinates of the origin.
 */
static float Origin[3] = {
  0.0, 0.0, 0.0
};

/*
 * RGB definition of the colour white.
 */
static float White[3] = {
  1.0, 1.0, 1.0
};

/*
 * Array to generate points of a square as a polytriangle.
 */
static int Square_Signs[4][3] = {
  -1, -1, 0,
  1, -1, 0,
  -1, 1, 0,
  1, 1, 0
};


/*****************************************************************************
 *
 * Global variables.
 */

/*
 * Description of the Domain - the 3D region covered by the UCD structure.
 */
static float Domain_Size;	/* Max dimension of the domain. */
static float Domain_Extent[6]; /* Size of the domain. */
static float Domain_Centre[3];	/* Centre of the domain. */

/*
 * Definition of the Probe - the collection of planes used to sample the UCD
 * structure.  Although the probe's orientation can be arbitrary, at present
 * the probe can only contain planes that are orthogonal to a major axis.
 * However, these could easily be altered.
 */
static int Num_Planes;		/* The number of planes in the current probe */
static plane Planes[MAX_PLANES]; /* List of planes in the probe */

/*
 * The transformation matrix for the Probe.  (Established in get_probe_trans).
 */
static float Probe_Trans[4][4];

/*
 * Descriptions of the Range - this is the data to be projected in a projected
 * plane probe.
 */
static float *Range;		/* A pointer to a component of node data. */
static float Range_Min;		/* Bracketing values for the data. */
static float Range_Max;
static int Range_Log_Scale;	/* True if Range is in a log scale. */
static float Range_Lower_Limit;	/* Lower limit for log scale. */
static float Range_Scale;	/* Makes range same size as domain. */
static char *Range_Label;	/* Label for the data component. */
static char *Range_Units;	/* Units for the data component. */

/*
 * Description of the Sample - this is the geometric structure created by
 * sampling (ie. slicing) the UCD structure in the domain of the Probe.
 */
static float Sample_Extent[6];	/* Extent of the sample */
static float Sample_Centre[3];	/* Centre of the sample */
static int Axes_Present;	/* True if the axes are currently displayed. */
static int Base_Present;	/* True if the base is currently displayed. */
static int Axes_Label_Flags;	/* Label flags see ucd_planes_init */
static int Tick_Label_Flags;
static int Title_Label_Flags;

/*
 * Housekeeping flags.
 */
static int Initialise;		/* Module Initialisation flag. */

/****************************************************************************
 *
 * Subroutines to control the setting of viewing matrices.
 *
 */

set_no_trans(elist, name)
/*
 * Resets the given object's transformation matrix
 */
GEOMedit_list elist;
char *name;

{
  float matrix[4][4];

  mat_identity(matrix);
  GEOMedit_set_matrix(elist, name, matrix);
}


set_slice_view_trans(elist)
/*
 * Sets the camera's transformation matrix to the sensible value for the
 * viewing of a simple slice plane.
 */
GEOMedit_list elist;

{
  float matrix[4][4];

  mat_scale(matrix, SLICE_VSCALE, SLICE_VSCALE, SLICE_VSCALE);
  GEOMedit_set_matrix(elist, "camera1", matrix);
}


set_projected_view_trans(elist)
/*
 * Sets the camera's transformation matrix to a sensible value for the viewing
 * of projected planes
 */
GEOMedit_list elist;

{
  float temp[4][4], matrix[4][4];

  mat_z_rotate(PROJECTED_ZROT, matrix);
  mat_x_rotate(PROJECTED_XROT, temp);
  mat_multiply(matrix, temp, matrix);
  mat_scale(temp, PROJECTED_VSCALE, PROJECTED_VSCALE, PROJECTED_VSCALE);
  mat_multiply(matrix, temp, matrix);
  GEOMedit_set_matrix(elist, "camera1", matrix);
}


/**************************************************************************
 *
 * Code to generate a transparent probe, in the correct position and
 * orientation on the first geometry output port.
 *
 */

get_probe_trans(trans_info, xrot, yrot, zrot, scale, dist, centre, redirect,
		matrix, position)
/*
 * Calculates the Probe_Trans matrix either from the upstream transformation
 * matrix or the dial widgets.  In the case of the upstream transformation
 * matrix the widgets values are modified to reflect the new transformation
 * matrix.  Otherwise the Probe_Trans is formed according to the parameter
 * values and the transformation matrix and the probe object's transformation
 * matrix is reset according to 'matrix' and 'position' which are also
 * calculated by this function.  If 'centre' is true the translation component
 * of the upstream matrix is reset so that the plane normal always passes
 * through the centre of the ucd structure.  Depending on whether 'redirect' is
 * true this can cause a second transformation of the probe object in the
 * geometry viewer once the mouse button is released.
 */
upstream_transform *trans_info;
float xrot;
float yrot;
float zrot;
float scale;
float dist;
int centre;
int redirect;
float matrix[4][4];
float position[3];

{
  int i;
  float theta, phi, psi, d, s;
  float v[3], n[3], diff[3];
  float m1[4][4], m2[4][4];
  float phi_sgn, angle;

  if (trans_info)
    {
      /*
       * calculate and update the transformation parameters from the new
       * transformation matrix.
       */

      mat_copy(Probe_Trans, trans_info->msxform);
      mat_copy(matrix, trans_info->msxform);
      
      /*
       * In order to decode the transformation matrix into parameters we assume
       * that the transformation matrix was constructed from a series of
       * transformations: translate by -c, scale by s, rotate in z by psi,
       * rotate in y by phi, rotate in x by theta and translate by c + d.n;
       * where c is the centre of rotation; s is a scaling vector; theta, phi
       * and psi are angles in degrees; n is the normalised normal to the
       * plane; and d is an offset distance along the normal.
       *
       * Note that the order of rotations is opposite to the order of rotations
       * used in AVS's transformations options panel.  This applies the
       * separate rotations in the order X Y Z.  However, rotating about Z
       * first makes more sense in this case as, as the probe plane is z=0, the
       * xrot and yrot parameters then control the orientation of the probe and
       * the zrot parameter controls the angle of rotation of the probe about
       * the plane normal.  Hence the user can usually disregard the zrot
       * parameter as it does not effect the definition of the plane.  It's
       * only use is to control the orientation of the sample with respect to
       * the plane definition.  If the user wishes to define the plane in the
       * normal AVS way, the transformation options panel can still be used as
       * normal.  If this is done the parameters will be calculated and updated
       * to reflect an equivalent z, y, x rotation order.
       */

      /*
       * The normal is the transformation of the vector k [0, 0, 1], ie. the
       * third column of the transformation matrix.
       */
      vec_copy(n, Probe_Trans[2]);
      normalize_vec(n);

      if (! centre)
	{
	  /*
	   * 'd' can be found by taking the difference between the dot product
	   * of any point on the transformed plane (eg. the transformation of
	   * the centre including any translations) with the plane normal, and
	   * the dot product of the centre of rotation with the normal.  To
	   * avoid trouble with floating point inaccuracies, some rounding is
	   * done.
	   */

	  vec_copy(v, Domain_Centre);
	  mat_vecmul(v, Probe_Trans);
	  d = 0.0;
	  for (i = 0; i < 3; i++)
	    d += (v[i] - Domain_Centre[i]) * n[i];
	  if (fabs(d) < PRECISION * Domain_Size) d = 0.0;
	}
      else
	{
	  /*
	   * The plane in the geometry viewer can be transformed away from the
	   * centre of the domain.  To ensure that the normal of the
	   * transformed plane always passes through the centre of the domain,
	   * we reset the translation vector of the Probe_Trans to c-v+dn where
	   * v is the transformation of c ignoring translations.  (In redirect
	   * mode c is the origin otherwise c is Domain_Centre).
	   */

	  vec_copy(v, Domain_Centre);
	  mat3_vecmul(v, Probe_Trans);	

	  /*
	   * First task is to establish d.  If the transformation was just a
	   * rotation or scale it makes more sense to keep d constant.  (This
	   * case can be spotted as 'diff' the difference between the
	   * transformation of c by Probe_Trans and c itself will still be
	   * of magnitude 'dist'.
	   */

	  for (i = 0; i < 3; i++)
	    if (redirect)
	      diff[i] = Probe_Trans[3][i];
	    else
	      diff[i] = v[i] + Probe_Trans[3][i] - Domain_Centre[i];

	  if (fabs(vec_mag(diff) - dist) > PRECISION * Domain_Size)
	    {
	      d = 0.0;
	      for (i = 0; i < 3; i++)
		d += diff[i] * n[i];
	    }
	  else
	    d = dist;

	  if (fabs(d) < PRECISION * Domain_Size) d = 0.0;
      
	  for (i = 0; i < 3; i++)
	    {
	      position[i] = d * n[i];
	      matrix[3][i] = Probe_Trans[3][i] - diff[i];
	      Probe_Trans[3][i] = Domain_Centre[i] - v[i] + position[i];
	    }
	}
      
#ifndef CONTINUAL_UPDATE
      /*
       * If the trans_info has been generated mid way through a mouse
       * transformation the dial parameters are not updated.
       */
      if (trans_info->flags != BUTTON_UP) return;
#endif

      /*
       * The scale parameter can be obtained by taking any row or column vector
       * magnitude in the upper 3x3 matrix of the transformation matrix.  To
       * avoid rounding errors the value is snapped to 1.0 if it is close.
       */
      s = (float)vec_mag(Probe_Trans[0]);
      if (fabs(s - 1.0) < PRECISION) s = 1.0;
      
      /*
       * Theta (zrot), phi (yrot) and psi (zrot) are calculated below by
       * looking at the transformation various points under various
       * transformations.  However, the method falls apart in phi = 90 degrees
       * (when the plane normal is parallel to the x-axis) so this is dealt
       * with as a special case.
       */
      vec_copy(v, Probe_Trans[2]);
      if (v[1] == 0.0 && v[2] == 0.0)
	{
	  /*
	   * In the case that phi is +/- 90 degrees the net rotation is
	   * equivalent to a rotation of psi-/+theta in z followed by a +/-90
	   * degree rotation in y.  This angle can be found by taking the arc
	   * target of -y/-+z of the transformation of i.  Both y and z cannot
	   * be zero in this case as x is.
	   */
	  phi_sgn = v[0] >= 0.0 ? 1.0 : -1.0;
	  theta = 0.0;
	  phi = 90.0 * phi_sgn;
	  vec_copy(v, Probe_Trans[0]);
	  angle = atan2(- v[1], - phi_sgn * v[2]) / M_PI * 180.0;

	  /*
	   * In order to prevent jumps in the dial widgets while rotating the
	   * probe plane in the geometry viewer the previous values for xrot
	   * and zrot are examined in order to make dial movement as smoothly
	   * as possible.
	   */
	  if (xrot == 0)
	    {
	      theta = 0;
	      psi = angle;
	    }
	  else if (zrot == 0)
	    {
	      psi = 0;
	      theta = - phi_sgn * angle;
	    }
	  else if (! different_quadrant(zrot - phi_sgn * xrot, angle))
	    {
	      theta = xrot;
	      psi = angle + phi_sgn * theta;
	    }
	  else
	    {
	      theta = 0;
	      psi = angle;
	    }
	}
      else
	{
	  /*
	   * Theta can be obtained by taking the arctan of -y/z where y and z
	   * are the respective coordinates of v, the transformation (ignoring
	   * translations) of the vector k.  Theta is restricted to lie in the
	   * range (-90,90] degrees.  This is to ensure that theta, phi and psi
	   * are unambiguous (see below).
	   */
	  theta = (v[2] != 0.0) ? atan( -v[1]/v[2] ) / M_PI * 180.0 : 90.0;

	  /*
	   * Once theta is known we can rotate v by - theta and take the arc
	   * tangent of x/z to get phi.  Arctan2 is used to obtain the correct
	   * quadrant for theta.  Note that atan2 is total except for the case
	   * where both arguments are zero.  In this case z and x cannot both
	   * be zero as y is.
	   */
	  mat_x_rotate(-theta, m1);
	  mat3_vecmul(v, m1);
	  phi = atan2(v[0], v[2]) / M_PI * 180.0;

	  /*
	   * Psi can be obtained by taking the arc tangent of -y/x where x and
	   * y are the respective components of the transformation of the
	   * vector i ([1, 0, 0]) rotated by -theta in x and -phi in y.  Note
	   * that both y and x cannot be zero as z is.
	   */
	  vec_copy(v, Probe_Trans[0]);
	  mat3_vecmul(v, m1);
	  mat_y_rotate(-phi, m2);
	  mat3_vecmul(v, m2);
	  psi = atan2(-v[1], v[0]) / M_PI * 180.0;

	  /*
	   * Theta, phi and psi are not the only specification of the plane
	   * possible.  This is because for every theta, phi and psi there is
	   * also 180-theta, 180-phi and 180-psi which perform the identical
	   * transformation.  Rather than making the specification unambiguous
	   * (by restricting the range of one of them to +-90 degrees) theta,
	   * phi and psi are chosen so that at least two of them are in the
	   * same quadrant as the previous values xrot, yrot and zrot.
	   */
	  if (different_quadrant(theta, xrot) +
	      different_quadrant(phi, yrot) +
	      different_quadrant(psi, zrot) > 1)
	    {
	      theta = rot180(theta);
	      phi = rot180(phi);
	      psi = rot180(psi);
	    }

	  /*
	   * However, so that reset works correctly, if none of theta, phi and
	   * psi are in the quadrant as 0.0 then the other set of angles are
	   * taken.
	   */
	  if (different_quadrant(theta, 0.0) &&
	      different_quadrant(phi, 0.0) &&
	      different_quadrant(psi, 0.0))
	    {
	      theta = rot180(theta);
	      phi = rot180(phi);
	      psi = rot180(psi);
	    }
	}

      /*
       * In order to avoid floating point rounding errors the dial rotation
       * values are set to the nearest degree.  However, the slice plane is
       * calculated to full accuracy.
       */
      theta = floor(theta + 0.5);
      phi = floor(phi + 0.5);
      psi = floor(psi + 0.5);
      
      /*
       * Update the parameters to reflect the calculated parameters.
       */
      AVSmodify_float_parameter("XRotation", AVS_VALUE, theta, 0.0, 0.0);
      AVSmodify_float_parameter("YRotation", AVS_VALUE, phi, 0.0, 0.0);
      AVSmodify_float_parameter("ZRotation", AVS_VALUE, psi, 0.0, 0.0);
      AVSmodify_float_parameter("Scale", AVS_VALUE, s, 0.0, 0.0);
      AVSmodify_float_parameter("Distance", AVS_VALUE, d, 0.0, 0.0);
    }
  else
    {
      /*
       * Calculate the probe's transformation matrix from the module
       * parameters.  Matrix is formed from a series of simple transformations.
       * First the Domain_Centre is translated to the origin.
       */
      mat_translate(Probe_Trans, - Domain_Centre[0], - Domain_Centre[1],
		    - Domain_Centre[2]);

      /*
       * Then the probe is scaled and rotated about the z, y and x axes.
       */
      mat_scale(m1, scale, scale, scale);
      mat_multiply(Probe_Trans, m1, Probe_Trans);
      mat_z_rotate(zrot, m1);
      mat_multiply(Probe_Trans, m1, Probe_Trans);
      mat_y_rotate(yrot, m1);
      mat_multiply(Probe_Trans, m1, Probe_Trans);
      mat_x_rotate(xrot, m1);
      mat_multiply(Probe_Trans, m1, Probe_Trans);

      /*
       * Then the origin is translated to the Domain_Centre.
       */
      for (i = 0; i < 3; i++)
	Probe_Trans[3][i] += Domain_Centre[i];

      /*
       * Copy Probe_Trans into 'matrix' which is used to set the Probes
       * transformation matrix.  If 'redirect' is true the centre of rotation
       * is the origin, so the translation component of the matrix is zeroed.
       */
      mat_copy(matrix, Probe_Trans);
      if (redirect)
	vec_copy(matrix[3], Origin);

      /*
       * Calculate the last translation component of the transformation.
       */
      d = dist / scale;
      for (i = 0; i < 3; i++)
	{
	  position[i] = d * Probe_Trans[2][i];
	  Probe_Trans[3][i] += position[i];
	}      
    }
}


define_probe(probe_type, num_planes)
/*
 * Defines the position and orientation of the planes in this probe type.
 */
int probe_type;
int num_planes;

{
  int i;
  float c[3];
  
  /*
   * Fill in the global definition of the structure of the current probe
   */
  switch (probe_type)
    {
    case SIMPLE:
    case PROJECTED:
      /*
       * A single plane z=0 centred on the origin
       */
      Num_Planes = 1;
      Planes->axis = 2;
      vec_copy(Planes->centre, Domain_Centre);
      break;
    case CROSSED:
      /*
       * 3 orthogonal planes centred on the origin
       */
      Num_Planes = 3;
      for (i = 0; i < 3; i++)
	{
	  Planes[i].axis = i;
	  vec_copy(Planes[i].centre, Domain_Centre);
	}
      break;
    case SHEAF:
      /*
       * 2 orthogonal planes (x=0 and y=0) centred on the origin
       */
      Num_Planes = 2;
      for (i = 0; i < 2; i++)
	{
	  Planes[i].axis = i;
	  vec_copy(Planes[i].centre, Domain_Centre);
	}
      break;
    case MULTI:
      /*
       * Num_Planes planes parallel to z=0 centred on the origin
       */
      Num_Planes = num_planes;
      vec_copy(c, Domain_Centre);
      for (i = 0; i < Num_Planes; i++)
	{
	  Planes[i].axis = 2;
	  c[2] = Domain_Centre[2] + ((float)i / (float)Num_Planes - 0.5) * 
              Domain_Size;
	  vec_copy(Planes[i].centre, c);
	}
      break;
    }
}


add_square(obj, centre, size, axis, transform)
GEOMobj *obj;
float centre[3];
float size;
int axis;
int transform;

{
  register int i, j;
  float vertices[4][3];

  for (i=0; i<3; i++)
    for (j=0; j<4; j++)
      vertices[j][i] = centre[i] + size*Square_Signs[j][(i+2-axis)%3];

  /*
   * If the probe transformation mode is 'redirect' then we have to transform
   * the vertices manually.
   */
  if (transform)
    for (j=0; j<4; j++)
      mat_vecmul(vertices[j], Probe_Trans);
  
  GEOMadd_polytriangle(obj, vertices, NULL, NULL, 4, GEOM_COPY_DATA);
}


create_probe(elist, transform)
/*
 * Creates a probe object to replace the current probe object in elist.
 */
GEOMedit_list elist;
int transform;

{
  int i;
  GEOMobj *obj;

  /*
   * Create the polytriangle probe object from the description above
   */
  obj = GEOMcreate_obj(GEOM_POLYTRI, NULL);

  /*
   * For each plane in the probe add a square to the object
   */
  for (i = 0; i < Num_Planes; i++)
    add_square(obj, Planes[i].centre, Domain_Size * 0.707, Planes[i].axis,
	       transform);
    
  /*
   * If there is only one probe plane add a second interior square.
   */
  if (Num_Planes == 1)
    add_square(obj, Planes->centre, Domain_Size * 0.5, Planes->axis,
	       transform);

  /*
   * Add object to the elist list and then free its storage.
   */
  GEOMedit_geometry(elist, "Probe", obj);
  GEOMdestroy_obj(obj);

  /*
   * Set the probe object's colour and rendering mode.
   */
  GEOMedit_properties(elist, "Probe", 1.0, 0.0, 0.0, 0.0, ALPHA, White);
  GEOMedit_render_mode(elist, "Probe", "flat");
}


update_probe(elist, probe_type, count, trans_info, xrot, yrot, zrot, scale,
	     dist, do_define_probe)
/*
 * Creates a geometry representing the probe in the correct orientation.
 */
GEOMedit_list *elist;
int probe_type;
int count;
upstream_transform *trans_info;
float xrot;
float yrot;
float zrot;
float scale;
float dist;
int do_define_probe;

{
  int i;
  int centre;
  int redirect;
  float extent[6];
  float matrix[4][4];
  float position[3];
  
#ifdef NORMAL_TRANSFORMS
  centre = 0;
  redirect = 0;
#else
  centre = (probe_type != SHEAF && probe_type != CROSSED);
#ifdef REDIRECT
  redirect = centre;
#endif
#endif

  get_probe_trans(trans_info, xrot, yrot, zrot, scale, dist, centre, redirect,
		  matrix, position);
  
  if (!do_define_probe && trans_info && !centre) 
    {
      AVSmark_output_unchanged("probe");
      return;
    }

  /*
   * If necessary redefine the probe object.
   */
  if (do_define_probe)
    define_probe(probe_type, count);
  
  /*
   * Initialise the edit list to be passed to the geometry viewer.
   */
  *elist = GEOMinit_edit_list(*elist);

  /*
   * If necessary create the probe object.
   */
  if (do_define_probe || redirect)
    create_probe(*elist, redirect);

  /*
   * Set the probe's matrix and position vector.
   */
  if (redirect)
    GEOMedit_transform_mode(*elist, "Probe", "redirect", BUTTON_DOWN );
  else
    GEOMedit_transform_mode(*elist, "Probe", "normal", 0);
  GEOMedit_set_matrix(*elist, "Probe", matrix);
  GEOMedit_position(*elist, "Probe", position);

  /*
   * Set the probe object's transformation mode.
   */
  if (redirect)
    GEOMedit_transform_mode(*elist, "Probe", "redirect", BUTTON_MOVING |
			    BUTTON_UP);
  else
    {
#ifdef CONTINUAL_UPDATE
      GEOMedit_transform_mode(*elist, "Probe", "notify", BUTTON_UP |
			      BUTTON_MOVING);
#else
      GEOMedit_transform_mode(*elist, "Probe", "notify", BUTTON_UP);
#endif
    }
  
  /*
   * Set the probe's centre of rotation.
   */
  GEOMedit_center(*elist, "Probe", redirect ? Origin : Domain_Centre);

  /*
   * Set the window to a root3/2 times the maximum extent of the domain.
   */
  for (i=0; i<3; i++)
    {
      extent[2*i] = Domain_Centre[i] - Domain_Size * 0.866;
      extent[2*i+1] = Domain_Centre[i] + Domain_Size * 0.866;
    }
  GEOMedit_window(*elist, "Probe", extent);
}


/****************************************************************************
 *
 * Code to generate a sample of the ucd structure, such as a sliced plane, in
 * the correct position and orientation on the second geometry output port.
 *
 */

node_info *preprocess(ucd_struct, nnodes, eqn, dist)
UCD_structure *ucd_struct;
int nnodes;
float eqn[3];
float dist;
{
  int i;
  float *x, *y, *z;		/* Structure's node positions */
  node_info *ntable;		/* The created table of node information */
  node_info *table_ptr;		/* Pointer into the node table */

  /*
   * Query ucd structure.
   */
  UCDstructure_get_node_positions(ucd_struct, &x, &y, &z);

  /*
   * Allocate storage space for information about each node.
   */
  ntable = (node_info *)calloc(nnodes, sizeof(node_info));

  /*
   * Compute distance between plane and nodes and set some flags to speed
   * computation.  Flag is true if the node is in front of the plane.  List of
   * edges is initialised to empty.
   */
  table_ptr = ntable;
  for (i = 0; i < nnodes; i++)
    {
      table_ptr->s = (eqn[0]*x[i] + eqn[1]*y[i] + eqn[2]*z[i] - dist);
      table_ptr->flag = (table_ptr->s >= 0 ? 1 : 0);
      table_ptr->targets[0] = 0;
      table_ptr++;
    }

  return ntable;
}


add_vertex(n1, n2, n3, mid_edge_node, ucd_struct, colour_field, data, matrix,
	   normal, ss)
/*
 * Add the single intersection point on the edge n1 n2 to the sample - if it
 * has not already been calculated.
 */
int n1;
int n2;
int n3;
int mid_edge_node;
UCD_structure *ucd_struct;
AVSfield_float *colour_field;
float *data;
float (*matrix)[4];
float *normal;
sstore *ss;

{
  int i;
  float w1, w2;			/* Interpolation weights */
  float *x, *y, *z;		/* Node positions */
  float *col1, *col2;		/* Two colours to interpolate between */
  int *targets, *indices;	/* Pointers into the node_info structure */
  float *v;

  targets = ss->ntable[n1].targets;
  indices = ss->ntable[n1].indices;

  /*
   * Check that the particular edge intersection has not already been
   * calculated.
   */
  while (*targets)
    {
      if (*targets == n2)
	{
	  *(ss->next_index)++ = *indices;
	  return;
	}
      targets++;
      indices++;
    };

  /*
   * The target node of the newly intersected edge is n2.
   */
  *targets++ = n2;
  *targets = 0;

#ifdef MID_EDGE_NODES
  if (mid_edge_node)
    {
      /*
       * Very simple approximation for mid edge nodes.  Uses linear
       * interpolation with the mid edge node rather than the target node.
       * A better approximation for mid edge nodes would be to use a quadratic
       * and calculate the intersection of the quadratic and the plane.
       * However, even this will not work if the edges are not approximately
       * linear as the plane could intersect the edge twice.  To deal with this
       * properly would require significant alteration to the code.
       */
      if (ss->ntable[n3].flag == ss->ntable[n1].flag) n1 = n3;
      if (ss->ntable[n3].flag == ss->ntable[n2].flag) n2 = n3;
    }
#endif
      
  /*
   * Calculate the interpolation weights for calculating the point of
   * intersection.  (Uses straightforward linear interpolation).
   */
  w2 = ss->ntable[n1].s / (ss->ntable[n1].s - ss->ntable[n2].s);
  w1 = 1.0 - w2;

  /*
   * Calculate the position of the intersection with the edge.
   */
  UCDstructure_get_node_positions(ucd_struct, &x, &y, &z);
  v = ss->vertices[ss->next_vertex];
  v[0] = x[n1]*w1 + x[n2]*w2;
  v[1] = y[n1]*w1 + y[n2]*w2;
  v[2] = z[n1]*w1 + z[n2]*w2;

  /*
   * If necessary transform the point to a point on the plane parallel to z=0
   * passing through the centre of the domain.
   */
  if (matrix)
    mat2_vecmul(v, matrix);

  /*
   * If there is data to be projected, set the z coordinate according to the
   * interpolated data values of the Range.
   */

  if (data)
    v[2] = (data[n1]*w1 + data[n2]*w2 - Range_Min) * Range_Scale;

  /*
   * Set the colour of the intersection vertex by interpolating from the
   * colours of the nodes at each end of the edge.
   */
  if (colour_field)
    {
      v = ss->colours[ss->next_vertex];
      col1 = I1DV(colour_field, n1);
      col2 = I1DV(colour_field, n2);
      for (i=0; i<3; i++)
	v[i] = col1[i]*w1 + col2[i]*w2;
    }
  
  /*
   * If the normal for the point of intersection is known, copy it into the
   * normals array.
   */
  if (normal)
    vec_copy(ss->normals[ss->next_vertex], normal);

  /*
   * Set the index for the newly intersected edge (from n1) and next index in
   * the sample's polygons array to point to this vertex.
   * NB. Vertices are numbered from 0 but indices are numbered from 1.
   */
  *indices = ++(ss->next_vertex);
  *(ss->next_index)++ = *indices;
}


add_cutting_polygon(cell, ucd_struct, colour_field, data, matrix, normal, ss)
int cell;
UCD_structure *ucd_struct;
AVSfield_float *colour_field;
float *data;
float (*matrix)[4];
float *normal;
sstore *ss;

{
  register int i, bit, temp_index;
  int *node;			/* Temporary pointers */
  float *v0, *v1, *v2;
  node_info *nt;
  int nvert;			/* Number of vertices of the polygon */
  int *polygon_indices;		/* Pointer to the polygon's indices list */
  char elem_type[9];		/* Variables describing UCD cell info */
  int name, mat_type, cell_type;
  int mid_edge_flags, *node_list;
  register int s;		/* Cell's vertices bit map. One bit for each */
				/* node: 1->above, 0->below plane */
  int *polygon_nodes;		/* The current edge which is being cut */
  int n1, n2;			/* The two nodes on the edge being cut */
  cutting_info *stable;		/* A pointer to the current cutting case */

  /*
   * Find out what type of cell we are dealing with.
   */
  UCDcell_get_information(ucd_struct, cell, &name, elem_type, &mat_type,
			  &cell_type, &mid_edge_flags, &node_list);
  
  /*
   * Determine which cutting case we're in.  A cutting case is numbered
   * according to a bit map where each bit represents a node.  A 1 implies the
   * node is above the plane a zero that the node is below.
   */
  s = 0;
  bit = Ucd_Num_Nodes[cell_type];
  if (!bit) return;
  node = node_list + bit;
  nt = ss->ntable;
  while (bit--)
      s |= (nt[*--node].flag & 1) << bit;

  /*
   * Use the predefined tables to find the number of vertices the cutting
   * polygon has.  If cell is not cut (nvert=0) return.  Otherwise store the
   * number of vertices in the sample's indices array.  The actual indices
   * values are then listed immediately afterwards. 
   */
  stable = Ucd_Cutting_Info[cell_type] + s;
  nvert = stable->nvert;
  if (!nvert) return;
  i = *(ss->next_index)++ = nvert;
  polygon_indices = ss->next_index;
  polygon_nodes = stable->node_triples;

  /*
   * Each pair of nodes, as indicated in the 'cutting_info' table, are
   * interpolated between to produce one polygon vertex.
   */
  while (i--)
    {
      /*
       * Add the point of intersection of the edge and the plane to the sample
       */
      n1 = node_list[*polygon_nodes++];
      n2 = node_list[*polygon_nodes++];

#ifdef MID_EDGE_NODES
      n3 = node_list[*polygon_nodes++];
      if (n1 < n2)
	add_vertex(n1, n2, n3, (mid_edge_flags | 1), ucd_struct, colour_field,
		   data, matrix, normal, ss);
      else
	add_vertex(n2, n1, n3, (mid_edge_flags | 1), ucd_struct, colour_field,
		   data, matrix, normal, ss);
      mid_edge_flags >> 1;
#else
      polygon_nodes++;
      if (n1 < n2)
	add_vertex(n1, n2, 0, 0, ucd_struct, colour_field, data, matrix,
		   normal, ss);
      else
	add_vertex(n2, n1, 0, 0, ucd_struct, colour_field, data, matrix,
		   normal, ss);
#endif
    }

  /*
   * In order for the surface to be shaded correctly all polygons must be
   * ordered in the same direction.  This is most easily ensured by reversing
   * the order of the indices in those polygons in which the z component of the
   * normal is positive and so pointing away from the camera.
   */
  v0 = ss->vertices[polygon_indices[0]-1];
  v1 = ss->vertices[polygon_indices[1]-1];
  v2 = ss->vertices[polygon_indices[2]-1];
  if (((v0[0]-v1[0])*(v2[1]-v1[1])-(v0[1]-v1[1])*(v2[0]-v1[0])) > 0.0)
    for (i=0; i < (nvert >> 1); i++)
      {
	temp_index = polygon_indices[i];
	polygon_indices[i] = polygon_indices[nvert-1-i];
	polygon_indices[nvert-1-i] = temp_index;
      }
}


add_plane(nnodes, ncells, eqn, dist, ucd_struct, colour_field, data, matrix,
	  normal, ss)
int nnodes;
int ncells;
float eqn[3];
float dist;
UCD_structure *ucd_struct;
AVSfield_float *colour_field;
float *data;
float (*matrix)[4];
float *normal;
sstore *ss;
{
  int i;

  /*
   * The simple implementation adopted here is to simply test every cell in the
   * UCD structure.  A more efficient approach for larger structures would to
   * be to compute connectivity information and use it to follow the plane
   * through the structure.
   */
  ss->ntable = preprocess(ucd_struct, nnodes, eqn, dist);
  for (i=0; i<ncells; i++)
    add_cutting_polygon(i, ucd_struct, colour_field, data, matrix, normal, ss);

/* IAC CODE CHANGE :   free(ss->ntable); */

/* IAC CODE CHANGE :    free(ss->ntable); */
    free(ss->ntable);
}


alloc_sstore(ss, nnodes, ncells, colour_field, data)
/*
 * Allocate memory for the relevant parts of ss.
 */
sstore *ss;
int nnodes;
int ncells;
AVSfield_float *colour_field;
float *data;
{
  ss->next_vertex = 0;
  ss->indices = (int *)calloc(ncells*8*Num_Planes, sizeof(int));
  ss->next_index = ss->indices;
  ss->vertices = (float (*)[3])calloc(nnodes*Num_Planes, 3*sizeof(float));
  if (colour_field)
    ss->colours = (float (*)[3])calloc(nnodes*Num_Planes, 3*sizeof(float));
  else
    ss->colours = 0;
  if (!data)
    ss->normals = (float (*)[3])calloc(nnodes*Num_Planes, 3*sizeof(float));
  else
    ss->normals = 0;
}


free_sstore(ss)
/*
 * Deallocate memory from the sample store.
 */
sstore *ss;
{

/* IAC CODE CHANGE :   free(ss->indices); */

/* IAC CODE CHANGE :    free(ss->indices); */
    free(ss->indices);

/* IAC CODE CHANGE :   free(ss->vertices); */

/* IAC CODE CHANGE :    free(ss->vertices); */
    free(ss->vertices);

/* IAC CODE CHANGE :   if (ss->colours) free(ss->colours); */

/* IAC CODE CHANGE :   if (ss->colours)  free(ss->colours); */
  if (ss->colours)   free(ss->colours);

/* IAC CODE CHANGE :   if (ss->normals) free(ss->normals); */

/* IAC CODE CHANGE :   if (ss->normals)  free(ss->normals); */
  if (ss->normals)   free(ss->normals);
}


add_planes(ucd_struct, colour_field, data, map_to_z0, ss)
UCD_structure *ucd_struct;
AVSfield_float *colour_field;
float *data;
int map_to_z0;
sstore *ss;
{
  register int i;
  float c[3];
  float eqn[3];			/* Equation of the plane */
  float dist;			/* and its distance from the Domain Centre*/
  float matrix[4][4];
  float normal[3];		/* Normal to the plane */
  char name[20];		/* Variables for reading the UCD structure */
  int data_veclen, name_flag;
  int ncells, cell_veclen, nnodes, node_veclen;
  int util_flag;

  /*
   * Get the number of cells and nodes in the structure
   */
  UCDstructure_get_header(ucd_struct, name, &data_veclen, &name_flag, &ncells,
			  &cell_veclen, &nnodes, &node_veclen, &util_flag);

  /*
   * Initialise the sstore of the sample
   */
  alloc_sstore(ss, nnodes, ncells, colour_field, data);
  
  /*
   * If necessary calculate a matrix to map the sample plane back onto the
   * plane parallel to the z axis through the Domain_Centre.
   */
  if (map_to_z0)
    mat_inverse(matrix, Probe_Trans);
  
  /*
   * For each plane in the sample
   */
  for (i = 0; i < Num_Planes; i++)
    {
      /*
       * Calculate the plane's equation.
       */
      vec_copy(eqn, Probe_Trans[Planes[i].axis]);
      vec_copy(c, Planes[i].centre);
      mat_vecmul(c, Probe_Trans);
      dist = dot_product(c, eqn);
      
      /*
       * If the sample if flat and so has a normal calculate it.
       */
      if (!data)
	if (map_to_z0)
	  {
	    vec_copy(normal, Origin);
	    normal[Planes[i].axis] = 1.0;
	  }
	else
	  vec_copy(normal, eqn);
	  
      /*
       * Add the current plane to the sample.
       */
      add_plane(nnodes, ncells, eqn, dist, ucd_struct, colour_field, data,
		map_to_z0 ? matrix : 0, data ? 0 : normal, ss);
    }

  /*
   * The list of indices should be null terminated.
   */
  *(ss->next_index) = 0;
}


get_sample_extent(vertices, next_vertex)
/*
 * Calculate extent and centre for the sample object.
 */
float (*vertices)[3];
int next_vertex;
{
  register i;
  register float *v, *e;
  
  v = (float *)vertices;
  e = Sample_Extent;
  i = 3;
  while (i--)
    {
      *e++ = *v;
      *e++ = *v++;
    }
  i = next_vertex;
  while (--i)
    {
      e = Sample_Extent;
      if (*v < *e) *e = *v;
      e++;
      if (*v > *e) *e = *v;	    
      v++;
      e++;
      if (*v < *e) *e = *v;
      e++;
      if (*v > *e) *e = *v;
      v++;
      e++;
      if (*v < *e) *e = *v;
      e++;
      if (*v > *e) *e = *v;
      e++;
      v++;
    }
  for (i = 0; i < 3; i++)
    Sample_Centre[i] = (Sample_Extent[(i<<1)+1] + Sample_Extent[i<<1]) / 2.0;
}


add_sample_object(elist, ss)
GEOMedit_list elist;
sstore *ss;
{
  GEOMobj *obj;

  /*
   * The sample is a polyhedron object.
   */
  get_sample_extent(ss->vertices, ss->next_vertex);
  obj = GEOMcreate_obj(GEOM_POLYHEDRON, Sample_Extent);

  /*
   * Add the polygons in the sample to the sample object.
   */
  GEOMadd_polygons(obj, ss->indices, GEOM_COMPLEX, GEOM_COPY_DATA);
  GEOMadd_vertices(obj, ss->vertices, ss->next_vertex, GEOM_COPY_DATA);
  
  /*
   * If there are colours add these as well.
   */
  if (ss->colours)
    GEOMadd_float_colors(obj, ss->colours, ss->next_vertex, GEOM_COPY_DATA);
  
  /*
   * If the surface is not flat calculate the surface normals.  Otherwise
   * set the vertex normals to the normal for the plane.
   */
  if (ss->normals)
    GEOMadd_normals(obj, ss->normals, ss->next_vertex, GEOM_COPY_DATA);
  else
    GEOMgen_normals(obj, 0);
  
  /*
   * Convert the polyhedron object into a polytriangle with both a surface
   * and a wireframe description. (So that the 'Outline Gourand' rendering
   * mode works.
   */
  GEOMcvt_polyh_to_polytri(obj, GEOM_SURFACE | GEOM_WIREFRAME);
  
  /*
   * Replace the sample object in the viewer with the new object.
   */
  GEOMedit_geometry(elist, "Sample", obj);
  GEOMdestroy_obj(obj);
  GEOMedit_center(elist, "Sample", Sample_Centre);
  GEOMedit_transform_mode(elist, "Sample", "normal", BUTTON_UP);
}


add_base_object(elist, ss)
GEOMedit_list elist;
sstore *ss;
{
  register int i;
  float base_pos;
  float extent[6], centre[3];
  GEOMobj *obj;

  /*
   * Flatten the sample in z.
   */
  base_pos = - Z_AXIS_OFFSET * Domain_Size;
  for (i = 0; i < ss->next_vertex; i++)
    ss->vertices[i][2] = base_pos;
  for (i = 0; i < 4; i++)
    {
      extent[i] = Sample_Extent[i];
      centre[i>>1] = Sample_Centre[i>>1];
    }
  centre[2] = extent[4] = extent[5] = base_pos;

  /*
   * The base is a polyhedron object.
   */
  obj = GEOMcreate_obj(GEOM_POLYHEDRON, extent);
  GEOMadd_polygons(obj, ss->indices, GEOM_COMPLEX, GEOM_COPY_DATA);
  GEOMadd_vertices(obj, ss->vertices, ss->next_vertex, GEOM_COPY_DATA); 
  GEOMcvt_polyh_to_polytri(obj, GEOM_WIREFRAME);

  /*
   * Replace the sample object in the viewer with the new object.
   */
  GEOMedit_geometry(elist, "Base", obj);
  GEOMdestroy_obj(obj);
  GEOMedit_center(elist, "Base", centre);
  GEOMedit_parent(elist, "Base", "Sample");
  GEOMedit_transform_mode(elist, "Base", "parent", BUTTON_UP);
  Base_Present = 1;
}
      

add_framework_objects(elist, nticks, do_base)
/*
 * Adds a title and axes with ticks and labels to the sample geometry.
 */
GEOMedit_list elist;
int nticks;
int do_base;

{
  int i;
  int zpow;			/* A multiplier for the vertical axes labels */
  float vertices[2][3];		/* Variables for adding lines and labels */
  float colours[2][3];
  float pos[3], offset[3];
  char tlabel[16];
  float z, zstep, zscale;	/* Variables for generating tick marks */
  float zstart, zfinish;
  float z_offset, xy_offset;	/* Various graph dimensions */
  float a_offset, la_offset;
  float lt_offset, t_length;
  float t_offset;
  float ae[6];			/* The extent of the axes */
  float le[6];			/* The extent of the labels */
  GEOMobj *axes;		/* The objects to be created */
  GEOMobj *axes_labels;
  GEOMobj *tick_labels;
  GEOMobj *title;

  /*
   * Make the sample size in the vertical equal to the size of the domain.
   */
  Sample_Extent[4] = 0.0;
  Sample_Extent[5] = Domain_Size;
  Sample_Centre[2] = Domain_Size / 2.0;

  /*
   * Scale the offset relative to the Domain Size
   */
  z_offset = Z_AXIS_OFFSET * Domain_Size;
  xy_offset = XY_AXIS_OFFSET * Domain_Size;
  a_offset = AXIS_OFFSET * Domain_Size;
  la_offset = AXIS_LABEL_OFFSET * Domain_Size;
  t_length = TICK_LENGTH * Domain_Size;
  lt_offset = t_length + TICK_LABEL_OFFSET * Domain_Size;

  /*
   * Axes are offset from the sample below in xy by XY_AXIS_OFFSET and in z by
   * Z_AXIS_OFFSET.  All axes are offset above the sample by AXIS_OFFSET.
   */
  ae[0] = Sample_Extent[0] - xy_offset;
  ae[1] = Sample_Extent[1] + a_offset;
  ae[2] = Sample_Extent[2] - xy_offset;
  ae[3] = Sample_Extent[3] + a_offset;
  ae[4] = Sample_Extent[4] - z_offset;
  ae[5] = Sample_Extent[5] + a_offset;

  /*
   * Labels are offset from the axes below and above by TICK_LABEL_OFFSET and
   * AXIS_LABEL_OFFSET respectively.
   */
  le[0] = ae[0] - lt_offset;
  le[1] = ae[1] + la_offset;
  le[2] = ae[2] - lt_offset;
  le[3] = ae[3] + la_offset;
  le[4] = ae[4] - lt_offset;
  le[5] = ae[5] + la_offset;

  /*
   * All lines are white.
   */
  vec_copy(colours[0], White);
  vec_copy(colours[1], White);

  /*
   * Create the objects.
   */
  axes = GEOMcreate_obj(GEOM_POLYTRI, 0);
  axes_labels = GEOMcreate_label(0, Axes_Label_Flags);
  tick_labels = GEOMcreate_label(0, Tick_Label_Flags);
  title = GEOMcreate_label(0, Title_Label_Flags);
  
  /*
   * Create Major axes.
   */
  vertices[0][0] = ae[0];
  vertices[0][1] = ae[2];
  vertices[0][2] = ae[4];
  vertices[1][0] = ae[1];
  vertices[1][1] = ae[2];
  vertices[1][2] = ae[4];
  GEOMadd_disjoint_line(axes, vertices, colours, 2, GEOM_COPY_DATA);
  vertices[1][0] = ae[0];
  vertices[1][1] = ae[3];
  GEOMadd_disjoint_line(axes, vertices, colours, 2, GEOM_COPY_DATA);
  vertices[1][0] = ae[0];
  vertices[1][1] = ae[2];
  vertices[1][2] = ae[5];
  GEOMadd_disjoint_line(axes, vertices, colours, 2, GEOM_COPY_DATA);
  
  /*
   * Label X, Y and Z axes.
   */
  vec_copy(offset, Origin);
  offset[1] = - AXES_LABEL_HEIGHT / 2.0;
  pos[0] = le[1];
  pos[1] = ae[2];
  pos[2] = ae[4];
  GEOMadd_label(axes_labels, "X", pos, offset, AXES_LABEL_HEIGHT, White, -1);
  pos[0] = ae[0];
  pos[1] = le[3];
  pos[2] = ae[4];
  GEOMadd_label(axes_labels, "Y", pos, offset, AXES_LABEL_HEIGHT, White, -1);
  pos[0] = ae[0];
  pos[1] = ae[2];
  pos[2] = le[5];
  GEOMadd_label(axes_labels, Range_Label, pos, offset, AXES_LABEL_HEIGHT,
		White, -1);
  
  /*
   * Add ticks and labels on the vertical axis.
   */
  zpow = 0;
  if (nticks)
    {
      z = 0.0;
      zstep = Domain_Size / (float)nticks;
      if (Range_Log_Scale == 0)
	{
	  zpow = (int) log10((Range_Max - Range_Min) / (float)nticks);
	  zscale = pow(0.1, (double)zpow);
	}
      else
	zscale = 1.0;
      zstart = Range_Min * zscale;
      zfinish = Range_Max * zscale;
      zscale /= Range_Scale;
      vertices[0][0] = ae[0];
      vertices[0][1] = ae[2];
      vertices[1][0] = ae[0] - t_length;
      vertices[1][1] = ae[2] - t_length;
      t_offset = t_length + TICK_LABEL_HEIGHT * (Range_Log_Scale ? 4.0:3.0);
      pos[0] = ae[0] - t_offset;
      pos[1] = ae[2] - t_offset;
      vec_copy(offset, Origin);
      offset[1] = - TICK_LABEL_HEIGHT / 2.0;
      while (nticks--)
	{
	  vertices[0][2] = z;
	  vertices[1][2] = z;
	  pos[2] = z;
	  GEOMadd_disjoint_line(axes, vertices, colours, 2, GEOM_COPY_DATA);
	  if (Range_Log_Scale)
	    sprintf(tlabel, "%.1e", pow(10.0, zstart + z * zscale) +
		    Range_Lower_Limit - 1.0);
	  else
	    sprintf(tlabel, "%.2f", zstart + z * zscale);
	  GEOMadd_label(tick_labels, tlabel, pos, offset, TICK_LABEL_HEIGHT,
			White, -1);
	  z += zstep;
	}
      vertices[0][2] = Domain_Size;
      vertices[1][2] = Domain_Size;
      pos[2] = Domain_Size;
      GEOMadd_disjoint_line(axes, vertices, colours, 2, GEOM_COPY_DATA);
      if (Range_Log_Scale)
	sprintf(tlabel, "%.1e", pow(10.0, (double)zfinish) + Range_Lower_Limit
		- 1.0);
      else
	sprintf(tlabel, "%.2f", zfinish);
      GEOMadd_label(tick_labels, tlabel, pos, offset, TICK_LABEL_HEIGHT, White,
		    -1);
    }

  /*
   * Add ticks and labels on the horizontal axes.
   */
  if (nticks && do_base)
    {
      vertices[0][1] = ae[2];
      vertices[0][2] = ae[4];
      vertices[1][1] = ae[2] - t_length;
      vertices[1][2] = ae[4] - t_length;
      vertices[0][0] = Sample_Extent[0];
      vertices[1][0] = Sample_Extent[0];
      GEOMadd_disjoint_line(axes, vertices, colours, 2, GEOM_COPY_DATA);
      vertices[0][0] = Sample_Extent[1];
      vertices[1][0] = Sample_Extent[1];
      GEOMadd_disjoint_line(axes, vertices, colours, 2, GEOM_COPY_DATA);
      vertices[0][0] = ae[0];
      vertices[1][0] = ae[0] - t_length;
      vertices[0][1] = Sample_Extent[2];
      vertices[1][1] = Sample_Extent[2];
      GEOMadd_disjoint_line(axes, vertices, colours, 2, GEOM_COPY_DATA);
      vertices[0][1] = Sample_Extent[3];
      vertices[1][1] = Sample_Extent[3];
      GEOMadd_disjoint_line(axes, vertices, colours, 2, GEOM_COPY_DATA);

      vec_copy(offset, Origin);
      offset[1] = - TICK_LABEL_HEIGHT / 2.0;
      pos[2] = le[4];
      pos[1] = le[2];
      pos[0] = Sample_Extent[0];
      sprintf(tlabel, "%.2g", Sample_Extent[0]);
      GEOMadd_label(tick_labels, tlabel, pos, offset, TICK_LABEL_HEIGHT, White,
		    -1); 
      pos[0] = Sample_Extent[1];
      sprintf(tlabel, "%.2g", Sample_Extent[1]);
      GEOMadd_label(tick_labels, tlabel, pos, offset, TICK_LABEL_HEIGHT, White,
		    -1); 
      pos[0] = le[0];
      pos[1] = Sample_Extent[2];
      sprintf(tlabel, "%.2g", Sample_Extent[2]);
      GEOMadd_label(tick_labels, tlabel, pos, offset, TICK_LABEL_HEIGHT, White,
		    -1); 
      pos[1] = Sample_Extent[3];
      sprintf(tlabel, "%.2g", Sample_Extent[3]);
      GEOMadd_label(tick_labels, tlabel, pos, offset, TICK_LABEL_HEIGHT, White,
		    -1);
    }

  /*
   * Add a plot title.
   */
  if (zpow && Range_Units)
    sprintf(tlabel, "%s (%sx1+e%d)", Range_Label, Range_Units, zpow);
  else if (zpow)
    sprintf(tlabel, "%s (x1+e%d)", Range_Label, zpow);
  else if (Range_Units)
    sprintf(tlabel, "%s (%s)", Range_Label, Range_Units);
  else
    sprintf(tlabel, "%s", Range_Label);
  pos[0]=0.0;
  pos[1]=1.0 - TITLE_LABEL_HEIGHT * 1.0;
  pos[2]=0.0;
  GEOMadd_label(title, tlabel, pos, Origin, TITLE_LABEL_HEIGHT, White,
		Title_Label_Flags); 

  /*
   * Add the newly defined objects into the geometry.
   */
  GEOMedit_geometry(elist, "Axes", axes);
  GEOMdestroy_obj(axes);
  GEOMedit_parent(elist, "Axes", "Sample");
  GEOMedit_transform_mode(elist, "Axes", "parent", BUTTON_UP);

  GEOMedit_geometry(elist, "Axes Labels", axes_labels);
  GEOMdestroy_obj(axes_labels);
  GEOMedit_parent(elist, "Axes Labels", "Sample");
  GEOMedit_transform_mode(elist, "Axes Labels", "parent", BUTTON_UP);

  GEOMedit_geometry(elist, "Tick Labels", tick_labels);
  GEOMdestroy_obj(tick_labels);
  GEOMedit_parent(elist, "Tick Labels", "Sample");
  GEOMedit_transform_mode(elist, "Tick Labels", "parent", BUTTON_UP);

  GEOMedit_geometry(elist, "Title", title);
  GEOMdestroy_obj(title);
  GEOMedit_parent(elist, "Title", "Sample");
  GEOMedit_transform_mode(elist, "Title", "parent", BUTTON_UP);

  /*
   * Extent for the sample are now the label extent.
   */
  for (i = 0; i < 6; i++)
    Sample_Extent[i] = le[i];

  Axes_Present = 1;
}


do_slice_plane(elist, ucd_struct, colour_field, do_trans)
/*
 * Generates one or more slice planes through the structure.
 */
GEOMedit_list *elist;
UCD_structure *ucd_struct;
AVSfield_float *colour_field;
int do_trans;

{
  sstore ss;

  /*
   * Extract the 2D sample consisting of planes from the UCD structure and hold
   * it in the intermediate store ss.  This is basically a list of polygons
   * which indirectly reference the vertices, and corresponding vertex data,
   * colours and normals, that make up the sample.
   */

  add_planes(ucd_struct, colour_field, (float *)0, !do_trans, &ss);

  if (! ss.next_vertex)
    {
      free_sstore(&ss);
      return 0;
    }

  *elist = GEOMinit_edit_list(*elist);
  add_sample_object(*elist, &ss);
  
  /*
   * Delete the axes and base objects if they were present.
   */
  if (Axes_Present)
    {
      GEOMedit_visibility(*elist, "Axes", GEOM_EDIT_DELETE);
      GEOMedit_visibility(*elist, "Axes Labels", GEOM_EDIT_DELETE);
      GEOMedit_visibility(*elist, "Tick Labels", GEOM_EDIT_DELETE);
      GEOMedit_visibility(*elist, "Title", GEOM_EDIT_DELETE);
      Axes_Present = 0;
    }
  if (Base_Present)
    {
      GEOMedit_visibility(*elist, "Base", GEOM_EDIT_DELETE);
      Base_Present = 0;
    }
  
  if (!do_trans)
    {
      set_no_trans(*elist, "Sample");
/*       set_no_trans(*elist, "%top"); */
      GEOMedit_window(*elist, "Sample", Sample_Extent);
      set_slice_view_trans(*elist);
    }
  else
    GEOMedit_window(*elist, "Sample", Sample_Extent);

  return 1;
}


do_projected_plane(elist, ucd_struct, colour_field, do_base, nticks)
/*
 * Generates a data projection of a plane through the structure.
 */
GEOMedit_list *elist;
UCD_structure *ucd_struct;
AVSfield_float *colour_field;
int do_base;
int nticks;

{
  sstore ss;

  /*
   * Extract the 2D sample consisting of planes from the UCD structure and hold
   * it in the intermediate store ss.  This is basically a list of polygons
   * which indirectly reference the vertices, and corresponding vertex data,
   * colours and normals, that make up the sample.
   */
  add_planes(ucd_struct, colour_field, Range, 1, &ss);

  if (! ss.next_vertex)
    {
      free_sstore(&ss);
      return 0;
    }

  /*
   * Initialise the edit list.
   */
  *elist = GEOMinit_edit_list(*elist);
  add_sample_object(*elist, &ss);
  if (do_base)
    add_base_object(*elist, &ss);
  else
    {
      GEOMedit_visibility(*elist, "Base", GEOM_EDIT_DELETE);
      Base_Present = 0;
    }

  /*
   * Add axes and labels.
   */
  add_framework_objects(*elist, nticks, do_base);

  /*
   * Set a sensible default viewing position.
   */
  set_no_trans(*elist, "Sample");
/*   set_no_trans(*elist, "%top"); */
  GEOMedit_window(*elist, "Sample", Sample_Extent);
  set_projected_view_trans(*elist);

  return 1;
}


/******************************************************************************
 *
 * Functions to decode inputs and parameter changes.
 *
 */

int find_probe_type(choice)
/*
 * Conversion function to change a text description of the probe's type
 * to an integer one.  It also makes sure that the modules parameters which are
 * dependent on probe type are in a consistent state.
 */
char *choice;

{
  int probe_type;

  if (!choice)
    return 0;
  
  probe_type = AVSchoice_number("Probe Choice", choice);

  if (probe_type && (AVSparameter_changed("Probe Choice") || Initialise))
    {
      AVSparameter_visible("Node Data", probe_type == PROJECTED);
      if (probe_type == PROJECTED)
	{
	  AVSmodify_parameter_prop("Count", "title", "string", "#Ticks");
	  AVSmodify_parameter("Count", AVS_VALUE | AVS_MINVAL | AVS_MAXVAL,
			      DEFAULT_TICKS, 0, MAX_TICKS);
	  AVSparameter_visible("Count", 1);
	  AVSmodify_parameter_prop("Toggle", "title", "string", "Base");
	  AVSmodify_parameter("Toggle", AVS_VALUE , 0, 0, 0);
	  AVSparameter_visible("Toggle", 1);
	}
      else if (probe_type == MULTI)
	{
	  AVSmodify_parameter_prop("Count", "title", "string", "#Planes");
	  AVSmodify_parameter("Count", AVS_VALUE | AVS_MINVAL | AVS_MAXVAL,
			      DEFAULT_PLANES, 2, MAX_PLANES);
	  AVSparameter_visible("Count", 1);
	  AVSparameter_visible("Toggle", 0);
	}
      else if (probe_type == SIMPLE)
	{
	  AVSmodify_parameter_prop("Toggle", "title", "string",
				   "Transform Sample");
	  AVSmodify_parameter("Toggle", AVS_VALUE , 0, 0, 0);
	  AVSparameter_visible("Toggle", 1);
	  AVSparameter_visible("Count", 0);
	}
      else
	{
	  AVSparameter_visible("Count", 0);
	  AVSparameter_visible("Toggle", 0);
	}
      Initialise = 0;
    }
	  
  return probe_type;
}


int stat_ucd_data(ucd_struct, ucd_scales, component)
/*
 * Check that the ucd structure exists and if it has changed find its extent.
 */
UCD_structure *ucd_struct;
scale_info *ucd_scales;
char *component;

{
  int ok, i;
  float dom_min[3], dom_max[3];	/* Variables to read domain extent */
  float *rng_min, *rng_max;	/* Variables to read range extent */
  char str100[100];		/* Variables to read names, labels etc. */
  char label[20];
  char delimiter[2];
  int data_veclen, name_flag;	/* Values to hold info about ucd_struct */
  int ncells, cell_veclen;
  int nnodes, node_veclen;
  int util_flag, *comps;
  int comp_number, range_pos;

  /*
   * Query ucd structure
   */
  ok = UCDstructure_get_header(ucd_struct, str100, &data_veclen,
				&name_flag, &ncells, &cell_veclen,
				&nnodes, &node_veclen, &util_flag);
  if (!ok)
    {
      AVSerror("Error: Couldn't get structure information");
      return 0;
    }

  /*
   * Decode choice number in terms of current 'Node Data' selection.
   */
  comp_number = AVSchoice_number("Node Data", component)-1;
  
  if (AVSinput_changed("ucd", 0))
    {
      /*
       * Find the centre, extent and size of the domain.
       */
      ok = UCDstructure_get_extent(ucd_struct, dom_min, dom_max);
      if (!ok)
	{
	  AVSerror("Error: Couldn't find domain's extent");
	  return 0;
	}

      for (i = 0; i < 3; i++)
	Domain_Centre[i] = (dom_max[i] + dom_min[i])/2;
      Domain_Size = 0.0;
      for (i = 0; i < 3; i++)
	{
	  Domain_Extent[2*i] = dom_min[i];
	  Domain_Extent[2*i+1] = dom_max[i];
	  if (Domain_Size < dom_max[i] - dom_min[i])
	    Domain_Size = dom_max[i] - dom_min[i];
	}

      /*
       * Set Distance parameter limits to reflect domain size.
       */
      AVSmodify_float_parameter("Distance", AVS_MINVAL | AVS_MAXVAL, 0.0,
				- Domain_Size, Domain_Size);

      /*
       * Make the 'Node Data' parameter reflect the choice of node data within
       * the UCD structure.
       */
      if (node_veclen)
	{
          for (i=0; i<100; i++)
	    str100[i] = 0;
	  UCDstructure_get_node_labels(ucd_struct, str100, delimiter);
	  UCDstructure_get_node_label(ucd_struct, 0, label);
	  AVSmodify_parameter("Node Data", AVS_VALUE | AVS_MINVAL | AVS_MAXVAL,
			      label, str100, delimiter);
	  comp_number = 0;
	}
      else
	AVSmodify_parameter("Node Data", AVS_VALUE | AVS_MINVAL | AVS_MAXVAL,
			    "<none>", "<none>", ";");
    }

  if (node_veclen > 0 && comp_number > -1)
    {
      /*
       * Find and store details about the range - that is the data component to
       * be projected in a 'projected probe'.
       */
      Range = 0;

/* IAC CODE CHANGE :       if (Range_Label) free(Range_Label); */

/* IAC CODE CHANGE :       if (Range_Label)  free(Range_Label); */
      if (Range_Label)   free(Range_Label);

/* IAC CODE CHANGE :       if (Range_Units) free(Range_Units); */

/* IAC CODE CHANGE :       if (Range_Units)  free(Range_Units); */
      if (Range_Units)   free(Range_Units);
      rng_min = (float *)malloc(node_veclen * sizeof(float));
      rng_max = (float *)malloc(node_veclen * sizeof(float));
      
      /*
       * Get information about the node data from the UCD structure.
       */
      ok = (UCDstructure_get_node_components(ucd_struct, &comps) &&
	    UCDstructure_get_node_minmax(ucd_struct, rng_min, rng_max) &&
	    UCDstructure_get_node_data(ucd_struct, &Range));

      /*
       * If the selected component from the UCD node data structure pick it out
       * and fill in the global variables describing it.
       */
      if (ok && comps[comp_number] == 1)
	{
	  /*
	   * Locate the actual array of data and its bounding values.
	   */
	  i = 0;
	  range_pos = 0;
	  while (i < comp_number)
	    {
	      range_pos += comps[i];
	      i++;
	    }
	  Range += range_pos*nnodes;
	  Range_Min = rng_min[range_pos];
	  Range_Max = rng_max[range_pos];
	  
	  /*
	   * Note if the date is in a log scale or not.
	   */
	  if (ucd_scales)
	    {
	      Range_Log_Scale = ucd_scales->log_scale[comp_number];
	      Range_Lower_Limit = ucd_scales->lower_limit[comp_number];
	    }
	  else
	    Range_Log_Scale = 0;

	  /*
	   * Work out a scaling value to make projected graphs look
	   * approximately square.
	   */
	  Range_Scale = Domain_Size / (Range_Max - Range_Min);

	  /*
	   * Find the label and units for the data component.
	   */
	  Range_Label = (char *)malloc(256*sizeof(char));
	  *Range_Label = 0;
	  UCDstructure_get_node_label(ucd_struct, comp_number, Range_Label);
	  Range_Units = (char *)malloc(256*sizeof(char));
	  *Range_Units = 0;
	  ok = UCDstructure_get_node_unit(ucd_struct, comp_number,
					  Range_Units);
	  if (!ok || *Range_Units == 0)
	    {

/* IAC CODE CHANGE : 	      free(Range_Units); */

/* IAC CODE CHANGE : 	       free(Range_Units); */
	        free(Range_Units);
	      Range_Units = 0;
	    }
	}
      else
	{
	  AVSerror("Error: Selected component is not a scalar.");
	  Range = 0;
	}


/* IAC CODE CHANGE :       free(rng_min); */

/* IAC CODE CHANGE :        free(rng_min); */
        free(rng_min);

/* IAC CODE CHANGE :       free(rng_max); */

/* IAC CODE CHANGE :        free(rng_max); */
        free(rng_max);
    }
  return 1;
}


int get_trans_info_flag(trans_info)
upstream_transform *trans_info;
{
  if (!trans_info ||
      !AVSinput_changed("transformation info", 0) ||
      strncmp(trans_info->object_name, "Probe", 5))
    return 0;
  return trans_info->flags;
} 
  

/*****************************************************************************
 *
 * The computation function of the module.
 */

int ucd_planes_compute(ucd_struct, ucd_scales, colour_field, trans_info,
		       probe_out, sample_out, do_sample, xrot, yrot, zrot,
		       scale, dist, pctitle, probe_choice, component, toggle,
		       count)
UCD_structure *ucd_struct;
scale_info *ucd_scales;
AVSfield_float *colour_field;
upstream_transform *trans_info;
GEOMedit_list *probe_out;
GEOMedit_list *sample_out;
int do_sample;
float *xrot;
float *yrot;
float *zrot;
float *scale;
float *dist;
char *pctitle;
char *probe_choice;
char *component;
int toggle;
int count;

{
  int probe_type, trans_info_flag;
  int new_ucd, do_define_probe, force_regenerate_sample, sample_changed;
    
  /*
   * Check the inputs and parameters are in a well defined state.
   */

  if (!(probe_type = find_probe_type(probe_choice))) return 0;

  if (!ucd_struct) return 0;

  if ((new_ucd = AVSinput_changed("ucd", 0)) ||
      AVSparameter_changed("Node Data") ||
      AVSinput_changed("scale info", 0))
    if (!stat_ucd_data(ucd_struct, ucd_scales, component))
      return 0;

  trans_info_flag = get_trans_info_flag(trans_info);

  do_define_probe = (new_ucd ||
		     AVSparameter_changed("Probe Choice") ||
		     (AVSparameter_changed("Count") &&
		      probe_type == MULTI));

  force_regenerate_sample = 0;
    
  /*
   * If necessary update the probe geometry.
   */

  if (do_define_probe ||
      trans_info_flag ||
      AVSparameter_changed("XRotation") ||
      AVSparameter_changed("YRotation") ||
      AVSparameter_changed("ZRotation") ||
      AVSparameter_changed("Scale") ||
      AVSparameter_changed("Distance"))
    {
      update_probe(probe_out, probe_type, count, trans_info_flag ? trans_info :
		   0, *xrot, *yrot, *zrot, *scale, *dist, do_define_probe);

#if defined(CONTINUAL_SAMPLE) && defined(CONTINUAL_UPDATE)
      force_regenerate_sample = 1;
#else
      force_regenerate_sample = !(trans_info_flag & BUTTON_MOVING);
#endif
    }
  else
    AVSmark_output_unchanged("probe");

  /*
   * If necessary generate the sample geometry.
   */
  
  sample_changed = 0;
  
  if (do_sample &&
      (force_regenerate_sample ||
       AVSinput_changed("colour field", 0) ||
       AVSparameter_changed("Do Sample") ||
       AVSparameter_changed("Node Data") ||
       AVSparameter_changed("Toggle") ||
       AVSparameter_changed("Count")))
    switch (probe_type)
      {
      case PROJECTED:
	if (Range)
	  sample_changed = do_projected_plane(sample_out, ucd_struct,
					      colour_field, toggle, count);
	break;
      case SIMPLE:
	sample_changed = do_slice_plane(sample_out, ucd_struct, colour_field,
					toggle);
	break;
      default:
	sample_changed = do_slice_plane(sample_out, ucd_struct, colour_field,
					1);
	break;
      }

  if (! sample_changed)
    AVSmark_output_unchanged("sample");

  /*
   * Return success.
   */
  return 1;
}


/*****************************************************************************
 *
 * Module initialisation and destruction functions.
 *
 */

ucd_planes_init()
{
  int label_font;
  int title_font;

  Initialise = 1;

  /*
   * Initially there is no Range, Axes or Base present.
   */
  Range = 0;
  Range_Label = 0;
  Range_Units = 0;
  Axes_Present = 0;
  Base_Present = 0;

  /*
   * Calculate some flags for the labels used in the projected plane sample.
   */
#if AVS_VERSION == 4
  label_font = GEOMget_font_number("Helvetica", 0, 0);
  title_font = GEOMget_font_number("Times", 0, 0);
#else
  label_font = 0;
  title_font = 0;
#endif

  Axes_Label_Flags = GEOMcreate_label_flags(label_font, 0, 0, 0,
					    GEOM_LABEL_CENTER, 0);
  Tick_Label_Flags = GEOMcreate_label_flags(label_font, 0, 0, 0,
					    GEOM_LABEL_CENTER, 0);
  Title_Label_Flags = GEOMcreate_label_flags(title_font, 1, 1, 0,
					     GEOM_LABEL_CENTER, 0);
}

ucd_planes_finis()
{
  /*
   * Free static storage
   */

/* IAC CODE CHANGE :   if (Range_Label) free(Range_Label); */

/* IAC CODE CHANGE :   if (Range_Label)  free(Range_Label); */
  if (Range_Label)   free(Range_Label);

/* IAC CODE CHANGE :   if (Range_Units) free(Range_Units); */

/* IAC CODE CHANGE :   if (Range_Units)  free(Range_Units); */
  if (Range_Units)   free(Range_Units);
}

/*****************************************************************************
 *
 * Module description functions.
 */

ucd_planes_desc()
{
  int param, port;

  AVSset_module_name("ucd planes", MODULE_MAPPER);

  AVScreate_input_port("ucd", "ucd");
  AVScreate_input_port("scale info", "struct scale_info", OPTIONAL);
  AVScreate_input_port("colour field", "field 1D 3-vector real", OPTIONAL);
  port = AVScreate_input_port("transformation info",
			      "struct upstream_transform",
			      OPTIONAL | INVISIBLE);
  AVSset_input_class(port, "probe:upstream_transform");

  AVScreate_output_port("probe", "geom");
  AVScreate_output_port("sample", "geom");

  AVSadd_parameter("Do Sample", "boolean", 0, 0, 0);
  AVSadd_float_parameter("XRotation", 0.0, -180.0, 180.0);
  AVSadd_float_parameter("YRotation", 0.0, -180.0, 180.0);
  AVSadd_float_parameter("ZRotation", 0.0, -180.0, 180.0);
  AVSadd_float_parameter("Scale", 1.0, 0.0, 1.0);
  AVSadd_float_parameter("Distance", 0.0, -5.0, 5.0);

  param = AVSadd_parameter("Probe Type", "string", "Probe Type", NULL, NULL);
  AVSconnect_widget(param, "text");
  param = AVSadd_parameter("Probe Choice", "choice", "Simple", Options, ";");
  AVSadd_parameter_prop(param, "width", "integer", 2);
  param = AVSadd_parameter("Node Data", "choice", "<none>", "<none>", ";");
  AVSadd_parameter_prop(param, "width", "integer", 2);
  AVSadd_parameter("Toggle", "boolean", 0, 0, 0);
  param = AVSadd_parameter("Count", "integer", 1, 1, 10);

  AVSset_init_proc(ucd_planes_init);
  AVSset_destroy_proc(ucd_planes_finis);
  AVSset_compute_proc(ucd_planes_compute);
}


int ((*ucd_planes_mod_list[])()) = {
	ucd_planes_desc,
};

#define NMODS sizeof(ucd_planes_mod_list) / sizeof(char *) 

AVSinit_modules()
{
	AVSinit_from_module_list(ucd_planes_mod_list, NMODS);
}
