/*******************************************************************************
 *
 *  These are a set of utility field point routines that can be used by 
 *  user-written AVS modules.
 *
 *	26 Mar 92  Phil McDonald, NOAA/ERL/FSL	Original version.
 *
 ******************************************************************************/



#include	"avs_utils.h"



/*******************************************************************************
 *
 *  Return the index of a point in a field.
 *
 */

int	UTILS_field_pt_index (ix, jy, kz)

int		ix, jy, kz;
{
    if (fldpt_ptr == NULL) return (0);

    return (ix + (fldpt_numijk[0] * (jy + (fldpt_numijk[1] * kz))));
}



/*******************************************************************************
 *
 *  Initialize the field point variables.
 *
 */

void	UTILS_field_pt_init (p_field)

AVSfield	*p_field;
{

    if (p_field == NULL)

        UTILS_field_pt_clear ();

    else
    {
        fldpt_ptr       = p_field;

        fldpt_ndim      = p_field->ndim;
        fldpt_numijk[0] = fldpt_numijk[1] = fldpt_numijk[2] = 1;
        if (p_field->ndim > 0) fldpt_numijk[0] = p_field->dimensions[0];
        if (p_field->ndim > 1) fldpt_numijk[1] = p_field->dimensions[1];
        if (p_field->ndim > 2) fldpt_numijk[2] = p_field->dimensions[2];
        fldpt_maxijk[0] = fldpt_numijk[0] - 1;
        fldpt_maxijk[1] = fldpt_numijk[1] - 1;
        fldpt_maxijk[2] = fldpt_numijk[2] - 1;

        fldpt_dattyp    = p_field->type;
        fldpt_grid      = p_field->uniform;
        fldpt_nspace    = p_field->nspace;
        fldpt_veclen    = p_field->veclen;

        fldpt_pts       = p_field->points;

        switch (fldpt_grid)
        {

            case UNIFORM :

                UTILS_field_pt_limits (fldpt_minxyz, fldpt_maxxyz);
                fldpt_unifac[0] = (fldpt_numijk[0] < 1) ? 0.0 :
                                  (fldpt_maxxyz[0] - fldpt_minxyz[0]) /
                                  ((float) fldpt_maxijk[0]);
                fldpt_unifac[1] = (fldpt_numijk[1] < 1) ? 0.0 :
                                  (fldpt_maxxyz[1] - fldpt_minxyz[1]) /
                                  ((float) fldpt_maxijk[1]);
                fldpt_unifac[2] = (fldpt_numijk[2] < 1) ? 0.0 :
                                  (fldpt_maxxyz[2] - fldpt_minxyz[2]) /
                                  ((float) fldpt_maxijk[2]);
                break;

            case RECTILINEAR :

                fldpt_off1 = fldpt_numijk[0];
                fldpt_off2 = fldpt_numijk[0] + fldpt_numijk[1];
                UTILS_field_pt_limits (fldpt_minxyz, fldpt_maxxyz);
                break;

            case IRREGULAR :

                fldpt_off1 = fldpt_numijk[0] * fldpt_numijk[1] *
                             fldpt_numijk[2];
                fldpt_off2 = fldpt_off1 * 2;
                UTILS_field_pt_limits (fldpt_minxyz, fldpt_maxxyz);
                break;
        }
    }


    return;
}



/*******************************************************************************
 *
 *  Clear the field point variables.
 *
 */

void	UTILS_field_pt_clear ()

{

    fldpt_ptr = NULL;


    return;
}



/*******************************************************************************
 *
 *  Find the spatial limits of a field.
 *
 */

void	UTILS_field_pt_limits (min, max)

FLOAT3		min, max;
{
    register FLOAT3	xyz1, xyz2;

    min[0] = min[1] = min[2] = max[0] = max[1] = max[2] = 0.0;

    if (fldpt_ptr == NULL) return;

    if (fldpt_grid == UNIFORM)

        UTILS_field_pt_limits_uni (min, max);

    else
    {
        UTILS_field_pt_xyz (0, 0, 0, xyz1);

        UTILS_field_pt_xyz (fldpt_maxijk[0], fldpt_maxijk[1], fldpt_maxijk[2],
                            xyz2);

        if (fldpt_ndim > 0) min[0] = xyz1[0], max[0] = xyz2[0];
        if (fldpt_ndim > 1) min[1] = xyz1[1], max[1] = xyz2[1];
        if (fldpt_ndim > 2) min[2] = xyz1[2], max[2] = xyz2[2];
    }


    return;
}



/*******************************************************************************
 *
 *  Find the spatial limits of a uniform field.
 *
 */

void	UTILS_field_pt_limits_uni (min, max)

FLOAT3		min, max;
{
    register int	i, j, n, has_exts;
    register float	*p_min, *p_max;

    min[0] = min[1] = min[2] = max[0] = max[1] = max[2] = 0.0;

    if (fldpt_ptr == NULL) return;

    if (fldpt_grid != UNIFORM)

        UTILS_field_pt_limits (min, max);

    else
    {
        n = fldpt_ndim;

        if (fldpt_pts != NULL)
        {
            p_min = fldpt_pts, p_max = fldpt_pts + 1;
            for (i = 0, j = 0; i < n; i++, j += 2)
                min[i] = p_min[j], max[i] = p_max[j];
        }
        else
        {
            p_min    = fldpt_ptr->min_extent, p_max = fldpt_ptr->max_extent;
            has_exts = TRUE;
            for (i = 0; i < n; i++)
            {
                min[i] = p_min[i], max[i] = p_max[i];
                if (min[i] >= max[i]) has_exts = FALSE;
            }
            if (!has_exts)
            {
                min[0] = min[1] = min[2] = max[0] = max[1] = max[2] = 0.0;
                if (n > 0) max[0] = fldpt_maxijk[0];
                if (n > 1) max[1] = fldpt_maxijk[1];
                if (n > 2) max[2] = fldpt_maxijk[2];
            }
        }
    }


    return;
}



/*******************************************************************************
 *
 *  Return the value at a point in a field.
 *
 */

float	UTILS_field_pt_val (ix, jy, kz)

int		ix, jy, kz;
{
    register int	i;


    if (fldpt_ptr    == NULL) return (0.0);
    if (fldpt_veclen >     1) return (0.0);

    i = UTILS_field_pt_index (ix, jy, kz);

    switch (fldpt_dattyp)
    {
        case AVS_TYPE_BYTE :

            return ((float) fldpt_ptr->field_data[i]);

        case AVS_TYPE_INTEGER :

            return ((float) fldpt_ptr->field_data_int[i]);

        case AVS_TYPE_REAL :

            return (fldpt_ptr->field_data_float[i]);

        case AVS_TYPE_DOUBLE :

            return ((float) fldpt_ptr->field_data_double[i]);

    }

    return (0.0);
}



/*******************************************************************************
 *
 *  Return the vector at a point in a field.
 *
 */

void	UTILS_field_pt_vec (ix, jy, kz, p_vec)

int		ix, jy, kz;
float		*p_vec;
{
    register int	i, iv, nv;


    if (fldpt_ptr == NULL) return;

    nv = fldpt_veclen;

    i  = UTILS_field_pt_index (ix, jy, kz) * nv;

    for (iv = 0; iv < nv; iv++)
    {
        switch (fldpt_dattyp)
        {
            case AVS_TYPE_BYTE :

                p_vec[iv] = (float) fldpt_ptr->field_data[i+iv];
                break;

            case AVS_TYPE_INTEGER :

                p_vec[iv] = (float) fldpt_ptr->field_data_int[i+iv];
                break;

            case AVS_TYPE_REAL :

                p_vec[iv] = fldpt_ptr->field_data_float[i+iv];
                break;

            case AVS_TYPE_DOUBLE :

                p_vec[iv] = (float) fldpt_ptr->field_data_double[i+iv];
                break;

            default:

                p_vec[iv] = 0.0;
                break;
        }
    }


    return;
}



/******************************************************************************/
/*
 *  Convert XY coordinates in computational space to XYZ in physical space.
 *
 */


int	UTILS_field_pt_xy_to_xyz (xyz)

FLOAT3		xyz;
{
    register int	i, ix, iy, jx, jy;
    register float	x, y, dat[4];
    register FLOAT3	pt[4];
    int			n = 2, interp = -1;
    float		xx, yy;

    if (fldpt_ptr == NULL) return (0);

    x = xyz[0], y = xyz[1];

    if ((x < 0.0) || (x > fldpt_maxijk[0])) return (0);
    if ((y < 0.0) || (y > fldpt_maxijk[1])) return (0);

    jx = ix = x, jy = iy = y;
    if (jx < fldpt_maxijk[0]) jx++;
    if (jy < fldpt_maxijk[1]) jy++;

    UTILS_field_pt_xyz (ix, iy, 0, &pt[0][0]);
    UTILS_field_pt_xyz (jx, iy, 0, &pt[1][0]);
    UTILS_field_pt_xyz (ix, jy, 0, &pt[2][0]);
    UTILS_field_pt_xyz (jx, jy, 0, &pt[3][0]);

    xx = x - ix, yy = y - iy;

    xx++, yy++;		/* Coords for FORTRAN must be 1-based */

    for (i = 0; i < 4; i++) dat[i] = pt[i][0];
    triangular_interp_ (dat, &n, &n, &interp, &xx, &yy, &xyz[0]);

    for (i = 0; i < 4; i++) dat[i] = pt[i][1];
    triangular_interp_ (dat, &n, &n, &interp, &xx, &yy, &xyz[1]);

    for (i = 0; i < 4; i++) dat[i] = pt[i][2];
    triangular_interp_ (dat, &n, &n, &interp, &xx, &yy, &xyz[2]);


    return (1);
}



/*******************************************************************************
 *
 *  Return the 3D coordinates of a point in a field.
 *
 */

int	UTILS_field_pt_xyz (ix, jy, kz, xyz)

int		ix, jy, kz;
FLOAT3		xyz;
{
    register int	i;

    if (fldpt_ptr == NULL) return (0);

    if ((ix < 0) || (ix > fldpt_maxijk[0])) return (0);
    if ((jy < 0) || (jy > fldpt_maxijk[1])) return (0);
    if ((kz < 0) || (kz > fldpt_maxijk[2])) return (0);

    xyz[0] = xyz[1] = xyz[2] = UTILS_field_pt_val (ix, jy, kz); 

    switch (fldpt_grid)
    {

        case UNIFORM :

            if (fldpt_ndim > 0)
            {
                xyz[0] = fldpt_minxyz[0] + (((float) ix) * fldpt_unifac[0]);
                if (fldpt_ndim > 1)
                {
                    xyz[1] = fldpt_minxyz[1] + (((float) jy) * fldpt_unifac[1]);
                    if (fldpt_ndim > 2)
                    {
                        xyz[2] = fldpt_minxyz[2] +
                                 (((float) kz) * fldpt_unifac[2]);
                    }
                }
            }
            break;

        case RECTILINEAR :

            xyz[0] = fldpt_pts[ix];
            if (fldpt_ndim > 1)
            {
                xyz[1] = fldpt_pts[jy+fldpt_off1];
                if (fldpt_ndim > 2)
                {
                    xyz[2] = fldpt_pts[kz+fldpt_off2];
                }
            }
            break;

        case IRREGULAR :

            i = UTILS_field_pt_index (ix, jy, kz);

            xyz[0] = fldpt_pts[i];
            if (fldpt_nspace > 1)
            {
                xyz[1] = fldpt_pts[i+fldpt_off1];
                if (fldpt_nspace > 2)
                {
                    xyz[2] = fldpt_pts[i+fldpt_off2];
                }
            }
            break;
    }

    return;
}
