/****************************************************************************
                  INTERNATIONAL AVS CENTER
	(This disclaimer must remain at the top of all files)

WARRANTY DISCLAIMER

This module and the files associated with it are distributed free of charge.
It is placed in the public domain and permission is granted for anyone to use,
duplicate, modify, and redistribute it unless otherwise noted.  Some modules
may be copyrighted.  You agree to abide by the conditions also included in
the AVS Licensing Agreement, version 1.0, located in the main module
directory located at the International AVS Center ftp site and to include
the AVS Licensing Agreement when you distribute any files downloaded from 
that site.

The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module provide absolutely
NO WARRANTY OF ANY KIND with respect to this software.  The entire risk as to
the quality and performance of this software is with the user.  IN NO EVENT
WILL The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module BE LIABLE TO
ANYONE FOR ANY DAMAGES ARISING FROM THE USE OF THIS SOFTWARE, INCLUDING,
WITHOUT LIMITATION, DAMAGES RESULTING FROM LOST DATA OR LOST PROFITS, OR ANY
SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES.

This AVS module and associated files are public domain software unless
otherwise noted.  Permission is hereby granted to do whatever you like with
it, subject to the conditions that may exist in copyrighted materials. Should
you wish to make a contribution toward the improvement, modification, or
general performance of this module, please send us your comments:  why you
liked or disliked it, how you use it, and most important, how it helps your
work. We will receive your comments at avs@ncsc.org.

Please send AVS module bug reports to avs@ncsc.org.

******************************************************************************/

/* NOTE:  THIS MODULE AND SOURCE CODE IS FOR USE 
   WITH THE AVS SOFTWARE ENVIRONMENT ONLY */
/****************************************************************************
                  INTERNATIONAL AVS CENTER
	(This disclaimer must remain at the top of all files)

WARRANTY DISCLAIMER

This module and the files associated with it are distributed free of charge.
It is placed in the public domain and permission is granted for anyone to use,
duplicate, modify, and redistribute it unless otherwise noted.  Some modules
may be copyrighted.  You agree to abide by the conditions also included in
the AVS Licensing Agreement, version 1.0, located in the main module
directory located at the International AVS Center ftp site and to include
the AVS Licensing Agreement when you distribute any files downloaded from 
that site.

The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module provide absolutely
NO WARRANTY OF ANY KIND with respect to this software.  The entire risk as to
the quality and performance of this software is with the user.  IN NO EVENT
WILL The International AVS Center, MCNC, the AVS Consortium and the individual
submitting the module and files associated with said module BE LIABLE TO
ANYONE FOR ANY DAMAGES ARISING FROM THE USE OF THIS SOFTWARE, INCLUDING,
WITHOUT LIMITATION, DAMAGES RESULTING FROM LOST DATA OR LOST PROFITS, OR ANY
SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES.

This AVS module and associated files are public domain software unless
otherwise noted.  Permission is hereby granted to do whatever you like with
it, subject to the conditions that may exist in copyrighted materials. Should
you wish to make a contribution toward the improvement, modification, or
general performance of this module, please send us your comments:  why you
liked or disliked it, how you use it, and most important, how it helps your
work. We will receive your comments at avs@ncsc.org.

Please send AVS module bug reports to avs@ncsc.org.

******************************************************************************/
/* Module Name: "zone_picker" (Filter) (Subroutine)                        */
/* Author: Ian_Curington                                                   */
/*
 * zone_picker - display, pick and transform zone transitions on wells 
 *
 * This routine takes two 1D  inputs, one the
 * zonation table, and second, the deviation file
 * for one well. 
 * It creates a geometric representation of the zonation picks
 * along the well bore,
 * and allows picking and dragging of the positions for
 * editing.
 *
 * It looks at the labels on the well dev field to figure
 * out which well it is, then indexes the zonations by this,
 * so there is no menu selection needed here.
 *
 * the zone icons can be picked and dragged along the well,
 * the new positions recorded in the memory file structure.
 *
 * ****** !!!!!!  CRITICAL NOTE !!!!!!! *********
 * ****** !!!!!!  ADVANCED TECHNIQUE !!!!!!! *********
 * This routine contains an advanced module writer trick,
 * of modifying the input port data inplace, without
 * explicitly or implicitly copying the data out of shared memory.
 * This REQUIRES that a special flag be turned on in
 * your .avsrc file:
 * ReadOnlySharedMemory 0
 *
 * If you get it wrong, you will either get a shared memory
 * access violation, or when the module is killed a message like:
 *   zone_picker: AVSfield_free: bad 'refcnt' 0, field=0x00072e10
 *
 *
 * Author: Ian J. Curington, Advanced Visual Systems Inc.
 *
 * Revision:
 *   30 Aug 93 ianc: Original
 *                   added upstream xform
 *   13 Sep 93 ianc: added magic number invalid data detection
 *   13 Sep 93 ianc: fixed bugs, added z invert, new estimate_zinc algorithm
 *
 */

#include <stdio.h>
#include <avs/avs.h>
#include <avs/port.h>
#include <avs/field.h>
#include <avs/geom.h>
#include <avs/udata.h>
/* IAC CODE CHANGE : #include <avs/avs_math.h> */
#include <avs/avs_math.h>


#define EPS   1.0e-5
#define ABS(a) ( ( (a) >0.0) ? (a) : ( -1.0 * (a)) )
#define DELIM  "#"
#define READ_WRITE_SHARED_MEMORY 1
/**
#define INVERT_Z 1
 **/
#ifdef  INVERT_Z
#define MAGIC  999.90
#else
#define MAGIC -999.90
#endif
#define MAGIC_TEST(a) \
                  ( a >= MAGIC-1.0 && a <= MAGIC+1.0 )

static int polytri_count = 0;
 
/* *****************************************/
/*  Module Description                     */
/* *****************************************/
int zone_picker_desc()
{

	int in_port, out_port, param, iresult;
	extern int zone_picker_compute();
	extern int zone_picker_init();
	extern int zone_picker_destroy();

	AVSset_module_name("zone_picker", MODULE_MAPPER);
        AVSset_module_flags( COOPERATIVE | REENTRANT);

	/* Input Port Specifications               */
	in_port = AVScreate_input_port("Zonation", 
		"field 2D 2-space uniform float", REQUIRED);

	in_port = AVScreate_input_port("Well_Devs", 
		"field 1D 3-space irregular float", REQUIRED);

	in_port = AVScreate_input_port("WellList", "string", REQUIRED );

        /* interactive transformation and picking ports */
        in_port = AVScreate_input_port("upstream",
                    "struct upstream_transform", INVISIBLE | OPTIONAL);
        AVSset_input_class(in_port, "Output:upstream_transform");

        in_port = AVScreate_input_port("pick",
                    "struct upstream_geom", OPTIONAL | INVISIBLE);
        AVSset_input_class(in_port,"upstream_geom");

	/* Output Port Specifications              */
	out_port = AVScreate_output_port("Zone_Picks", "geom" );

	out_port = AVScreate_output_port("New_Zones", 
	            "field 1D 3-space irregular float");


	/* Parameter  Specifications              */
        AVSadd_float_parameter("Scale", 1.0, FLOAT_UNBOUND, FLOAT_UNBOUND );

        AVSadd_parameter("On/Off", "boolean", 1, 0, 1 );

        param = AVSadd_float_parameter("Depth", 0.0, FLOAT_UNBOUND, FLOAT_UNBOUND );
        AVSconnect_widget( param, "typein_real" );
        AVSadd_parameter_prop( param, "width", "integer", 4 );

        AVSadd_parameter("Snap to Meter", "boolean", 0, 0, 1 );

	AVSset_compute_proc(zone_picker_compute);
	AVSset_init_proc(zone_picker_init);
	AVSset_destroy_proc(zone_picker_destroy);

	return(1);
}
 
/* *****************************************/
/* Module Compute Routine                  */
/* *****************************************/
int zone_picker_compute( Zonation, Well_Devs, WellList,
                         xform, xgeom,
                         Zone_Picks, New_Zones,
                         scale, on_off, depth, snap )
	AVSfield_float *Zonation;    /* input zone defs */
	AVSfield_float *Well_Devs;   /* input deviations for one well */
        char           *WellList;    /* input well name list */
        upstream_transform *xform;
        upstream_geom *xgeom;
        GEOMedit_list  *Zone_Picks;  /* output geometric pick representation */
	AVSfield_float **New_Zones;  /* new output field for editing update */
	float          *scale;       /* zone pick marker size */
        int            on_off;       /* control switch for whole module */
        float          *depth;       /* type-in depth control for editing */
        int            snap;         /* snap to whole Meter depths */
{
    GEOMobj *obj;
    GEOMobj *obj_label;
    float zinc;
    float *x, *y, *z;
    int   num_zones;
    int   num_wells;
    int   well;
    char  labels[200], delim[4];
    char  zone_name[80];
    char  well_name[80];
    char  string[80];
    int   i, j;
    float         newmat[4][4], identity[4][4];
    float         tmp_vec[3];
    float         min_extent[3], max_extent[3];
    float         target[3];
    char          *outbuf, *errbuf;
    int           reset_xform;
    float         est_z_incr();

    /***********/
    /* Statics */
    /***********/
    static  float     picked_point[3];   /* picked vertex */
    static  int       picked_node = -1;  /* picked node */
    static  int       have_reset_xform=0;/* history of reset event */


    /***********/
    /* BODY    */
    /***********/
    /* Get size of zone list for this well */
    AVSfield_get_labels ( Well_Devs, well_name, delim );
    well = resolve_choice_num( WellList, well_name, DELIM ) - 1;
    if ( well < 0 ) well = 0;
    num_wells = MAXX( Zonation );
    num_zones = Zonation->veclen;
    polytri_count = 0;

    /* estimate extent */
    min_extent[0] = max_extent[0] = 0.0;
    min_extent[1] = max_extent[1] = 0.0;
    min_extent[2] = Zonation->data[ (well)*num_zones ];
    max_extent[2] = Zonation->data[ (well)*num_zones + (num_zones-1) ];


    if ( AVSinput_changed("Well_Devs", 0)  ||
         AVSinput_changed("Zonation",  0) )
    {
        picked_point[0] = 0.0;
        picked_point[1] = 0.0;
        picked_point[2] = max_extent[2]-min_extent[2];
        AVScommand("kernel","geom_lookup_obj_names $Module",
            &outbuf, &errbuf );
        sprintf(string,"geom_reset -object %s",outbuf);
        AVScommand("kernel",string, &outbuf, &errbuf );

        /* reset state */
        reset_xform = 1;
        have_reset_xform=1;
        picked_node = -1;
    }
    else
    {
        reset_xform = 0;
    }



    if ( on_off )
    {
        /************************/
        /* process a pick event */
        /************************/
        if ( xgeom != NULL &&
             AVSinput_changed( "pick", 0)  )
        {
            picked_point[0] = xgeom->vertex[0];
            picked_point[1] = xgeom->vertex[1];
            picked_point[2] = xgeom->vertex[2];
            reset_xform = 1;
            have_reset_xform=1;
            AVScommand("kernel","geom_lookup_obj_names $Module",
                &outbuf, &errbuf );
            sprintf(string,"geom_reset -object %s",outbuf);
            AVScommand("kernel",string, &outbuf, &errbuf );
            picked_node = xgeom->vdata;  /* Picked Node index */
        }

        /* Detect pick/transform mode change */
        if ( xform != NULL &&
             AVSinput_changed( "upstream", 0)  )
        {
            if ( have_reset_xform )
            {
               have_reset_xform=0;
               return(0);
            }
            have_reset_xform=0;
        }


        /* Transformation Request Control */
        mat_identity( identity );
        if (xform != NULL &&
            !strncmp(xform->object_name,
            "Zone_Picker", strlen("Zone_Picker")) &&
            !reset_xform )
        {
            track_box_xform(newmat, min_extent, max_extent,
                xform->msxform, 0.0);
        }
        else
        {
            track_box_xform(newmat, min_extent, max_extent,
                identity, 0.0);
        }

        target[0] = picked_point[0];
        target[1] = picked_point[1];
        target[2] = picked_point[2];
        mat_vecmul( target ,newmat);

        /* Snap to whole units of depth */
        if ( snap )
            target[2] = (float)((int)target[2]);

        /* Link Type-in to edit depth value */
        if ( AVSparameter_changed("Depth"))
            target[2] = *depth;

        /* Allocate space for the zone pick location list */
        x = (float *) malloc ( sizeof(float) * num_zones );
        y = (float *) malloc ( sizeof(float) * num_zones );
        z = (float *) malloc ( sizeof(float) * num_zones );

        /* Fill the Z array with Zonation Depths */
        /* "TOP" zone surface only */
        for ( i=0; i < num_zones; i++ )
        {
            z[i] = Zonation->data[ (0*num_wells+well)*num_zones+i ];
        }

        /* Alter the Picked Zone Marker */
        if ( picked_node >= 0 && picked_node < num_zones )
        {
#ifdef READ_WRITE_SHARED_MEMORY
            Zonation->data[(0*num_wells+well)*num_zones+picked_node ]=target[2];
#endif
            z[ picked_node ] = target[2];
        }

        /* Estimate Z interval in Deviation Field */
        zinc = est_z_incr ( Well_Devs );

#ifdef INVERT_Z
    zinc = -1.0 * zinc;
#endif

        /* Locate the spatial location of zone picks */
        resample_zz ( Well_Devs, x, y, z, num_zones, zinc ); 


        /* get zone names going down the well for 3D text labels */
        AVSfield_get_labels ( Zonation, labels, delim );

    }

    /* Create Geometric Representation */

    *Zone_Picks = GEOMinit_edit_list(*Zone_Picks);
    obj = GEOMcreate_obj(GEOM_POLYTRI, NULL);
    obj_label = GEOMcreate_label(GEOM_NULL, NULL);


    if ( on_off )
    {
        for ( i=0; i < num_zones; i++ )
        {
          if ( !MAGIC_TEST(z[i]) )
          {
            extract_label( labels, zone_name, i, delim );    /* in display loop */
            draw_doughtnut( obj, x[i], y[i], z[i], *scale,i); /* define ring shape */
            draw_label_group (obj_label, zone_name, x[i], y[i], z[i], *scale );
          }
        }
    }

    GEOMedit_geometry(*Zone_Picks, "Zone_Picker", obj);
    GEOMedit_geometry(*Zone_Picks, "Zone_Picker", obj_label);
    GEOMdestroy_obj( obj );
    GEOMdestroy_obj( obj_label );

    GEOMedit_transform_mode(*Zone_Picks, "Zone_Picker", "redirect",
                          BUTTON_MOVING | BUTTON_UP);
    GEOMedit_selection_mode(*Zone_Picks,"Zone_Picker",
                   "notify", BUTTON_MOVING | BUTTON_DOWN );

    /* edit update for storage, output field */
    *New_Zones = NULL;

    /* free temp module storage */
    if ( on_off )
    {

/* IAC CODE CHANGE :         if ( x ) free ( x ); */
        if ( x )  free(x );

/* IAC CODE CHANGE :         if ( y ) free ( y ); */
        if ( y )  free(y );

/* IAC CODE CHANGE :         if ( z ) free ( z ); */
        if ( z )  free(z );
    }

    return(1);
}
 
/* ***********************************************************************/
/* Initialization for modules contained in this file.                    */
/* ***********************************************************************/
#ifdef SEP_EXE
int ((*mod_list[])()) = {
	zone_picker_desc,
};
#define NMODS (sizeof(mod_list) / sizeof(char *))

AVSinit_modules()
{
	AVSinit_from_module_list(mod_list, NMODS);
}
#endif
 

/* *****************************************/
/* Module Initialization Routine           */
/* *****************************************/
int zone_picker_init()
{
}

/* *****************************************/
/* Module Destroy Routine                  */
/* *****************************************/
int zone_picker_destroy()
{
}

/* *****************************************/
/* Estimate Depth Increment Routine        */
/* *****************************************/
float
est_z_incr ( Well_Devs )
	AVSfield_float *Well_Devs;
{
    int i, n;
    float zinc;

    n = Well_Devs->dimensions[0];

    if ( n <= 1 )
    {
        zinc= 0.0;
    }
    else if ( n == 2 )
    {
        zinc = ABS(Well_Devs->points[ 2 * n + 1 ] -
                   Well_Devs->points[ 2 * n + 0 ]);
    }
    else
    if ( n > 2 )
        zinc = ABS(Well_Devs->points[ 2 * n + 2 ] -
                   Well_Devs->points[ 2 * n + 1 ]);

    return( zinc );

}

/* *****************************************/
/* Resample Depth Routine                  */
/* *****************************************/
resample_zz ( Well_Devs, X, Y, Z, num_zones, zinc )
	AVSfield_float *Well_Devs;
        float *X, *Y, *Z;
        int   num_zones;
        float zinc;
{
    int i, n_devs;

    n_devs = MAXX( Well_Devs );

    /* ==== Linear Only  ==== */
        zzznmoli ( Well_Devs->points,                /* x-axis deviations */
                 Z,                                /* z of zones */
                 X,                                /* x of zones */
                 zinc,                             /* depth spacing of dev data */
                 n_devs,                           /* number of deviations */
                 num_zones );                      /* number of zone samples */

        zzznmoli ( &Well_Devs->points[ n_devs ],     /* y-axis deviations */
                 Z,                                /* z of zones */
                 Y,                                /* y of zones */
                 zinc,                             /* depth spacing of dev data */
                 n_devs,                           /* number of deviations */
                 num_zones );                      /* number of zone samples */


}

/* *****************************************/
/* Resample Utility Routines               */
/* *****************************************/

/*
 * zzznmoli - depth normal moveout vector resampling, linear interpolation
 *
 * Author: Ian Curington, AVS Inc.
 *
 * Revision:
 * 21 Aug 93 ianc - Original, from original Fortran template
 *
 *
 *  PURPOSE:      To linearly interpolate from an input seismic trace of
 *                sample values, to produce an output trace of sample
 *                values, using an input vector of sample times at which
 *                to perform interpolation.
 *
 *  FORMULA:      The interpolation is linear, in the normal sense that
 *                a weighted average of adjacent (1 preceding, 1
 *                following) sample input values is used in computing
 *                output sample values.  If sample values outside the
 *                input range of X(I) are required by the interpolation
 *                process, the output value of Z(J) will be set to 0.0.
 *
 *                   For k = 1 to NNMO-1
 *                       n = trunc(Y(k)/SR)
 *                       F = frac(Y(k)/SR)
 *
 *                       If ( n+1 <= 0 or n+2 > N ) then
 *                          Z(k) = 0.0
 *                       Else
 *                          Z(k) = (X(n+2) - X(n+1))*F + X(n+1)
 *
 *
 *  DESCRIPTION:  ZNMOLI converts out-of-place,
 *                the intput vector Y of trace
 *                sample values into output vector Z of corresponding
 *                trace sample values, which are interpolated from input
 *                vector X of original trace sample values.
 *
 *      S0 and S1 represent the 2 values used
 *      in the interpolation formula.
 *
 */

zzznmoli ( x, y, z, sr, n, nnmo )

    float *x;  /* vector of input trace samples */
    float *y;  /* input vector of time samples to retrieve */
    float *z;  /* output vector of new trace samples at new times */
    float sr;  /* sample rate of input trace vector */
    int    n;  /* number of samples in input trace */
    int nnmo;  /* number of time samples to get */

{
    double k, s0, s1, f;
    int i, j;

    if ( n > 1 && nnmo > 0 && sr != 0.0 )
    {
        for ( j=0; j < nnmo; j ++ )
        {
            k = y[j] / sr;
            i = (int)k;
            f = k - (float)i;
            if ( i < 0 )
            {
                s0 = s1 = x[0];
            }
            else if ( i+1 > n )
            {
                s0 = s1 = x[n-1];
            }
            else
            {
                s0 = x[i];
                s1 = x[i+1];
            }
            z[j] = (s1 - s0) * f + s0;
        }
    }
}

/* ======================================================== */

/*
 * znmoqi - depth normal moveout vector resampling, quadratic interpolation
 *
 * Author: Ian Curington, AVS Inc.
 *
 * Revision:
 * 21 Aug 93 ianc - Original, from original Fortran template
 *
 *
 *  PURPOSE:      To quadratically interpolate
                  from an input seismic trace of
 *                sample values, to produce an output trace of sample
 *                values, using an input vector of sample times at which
 *                to perform interpolation.
 *
 *  FORMULA:      The interpolation is quadratic, in the normal sense that
 *                a weighted average of adjacent (2 preceding, 1
 *                following) sample input values is used in computing
 *                output sample values.  If sample values outside the
 *                input range of X(I) are required by the interpolation
 *                process, the output value of Z(J) will be set to 0.0.
 *
 *                   For k = 1 to NNMO-1
 *                       n = trunc(Y(k)/SR)
 *                       F = frac(Y(k)/SR)
 *
 *                       If ( n <= 0 or n+2 > N ) then
 *                          Z(k) = 0.0
 *                       Else
 *                          Z(k) = 0.5*((X(n)-2*X(n+1)+X(n+2)*F**2
 *                          +(X(n+2)-X(n))*F + 2*X(n+1))
 *
 *
 *
 *  DESCRIPTION:  ZNMOLI converts out-of-place,
 *                the intput vector Y of trace
 *                sample values into output vector Z of corresponding
 *                trace sample values, which are interpolated from input
 *                vector X of original trace sample values.
 *
 *      S0 and S1 represent the 2 values used
 *      in the interpolation formula.
 *
 */

zzznmoqi ( x, y, z, sr, n, nnmo )

    float *x;  /* vector of input trace samples */
    float *y;  /* input vector of time samples to retrieve */
    float *z;  /* output vector of new trace samples at new times */
    float sr;  /* sample rate of input trace vector */
    int    n;  /* number of samples in input trace */
    int nnmo;  /* number of time samples to get */

{
    float k, s0, s1, s2, f;
    int i, j;

    if ( n > 2 && nnmo > 0 && sr != 0.0 )
    {
        for ( j=0; j < nnmo; j ++ )
        {
            k = y[j] / sr;
            i = (int)k;
            f = k - (float)i;
            if ( i < 0 || i+2 > n )
                s0 = s1 = s2 = 0.0;
            else
            {
                s0 = x[i];
                s1 = x[i+1];
                s2 = x[i+2];
            }
            z[j] = (s1 - s0) * f + s0;
            z[j] = 0.5 * 
                   ( (s0 - 2.0 * s1 + s2) * f*f +
                     (s2 - s0) * f +
                     (2.0 * s1) );
        }
    }
}


/******************************************************************************/
/*
 * Resolve choices from string list, curent val, into int offset.
 */
/******************************************************************************/

int
resolve_choice_num(name, str, delim)
    char *name;
    char *str;
    char *delim;
{
   char *choices, *cp;
   int i, flag;

   choices = name;
   for (i = 1; choices; i++) {
	   cp = strchr(choices, *delim);
	   if (cp) {
		   *cp = '\0';
	   }
	   flag = strcmp(choices, str);
	   if (cp) {
		   *cp = *delim;
		   choices = cp + 1;
	   } else {
		   choices = 0;
	   }
	   if (flag == 0)
		   return(i);
   }
   return(0);
}


/******************************************************************************/
/*
 * Extract the Nth Label from a string of DELIM separated labels
 * Start Counting at Zero
 */
/******************************************************************************/

extract_label( label_list, new_label, n, delim )
    char *label_list;
    char *new_label;
    int  n;
    char *delim;
{
   int i, j, k, len;

   /* get size of list */
   len = strlen ( label_list );

   /* find the Nth-1 Delimiter */
   i=0;
   for ( j=0; j < n; j++ )
   {
       while ( label_list[i] != delim[0] && i < len ) i++;
       i++;  /* skip the delimiter */
   }

   /* copy the sub-string */
   k=0;
   while ( label_list[i] != delim[0] && i < len )
   {
        new_label[ k ] = label_list[i];
        i++, k++;
   }
   new_label[k] = '\0';

}

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/
/******************************************************************************/

/*****************************************/
/* Global Declarations                   */
/*****************************************/

static int             num_steps = 11;
static float           cos_array[128];
static float           sin_array[128];
static float    xp[3] = { 1.0, 0.0, 0.0 };
static float    yp[3] = { 0.0, 1.0, 0.0 };
static float    zp[3] = { 0.0, 0.0, 1.0 };
static float    yellow[3]={ 0.99, 0.99, 0.0 };


/******************************************************/
/* Doughnut Processing Function                           */
/******************************************************/

draw_doughtnut( obj, x, y, z, offset, tag )

       GEOMobj *obj;
       float   x, y, z;
       float   offset;
       int     tag;
{
       int        i,k;
       float      *pt_list, *nl_list, *cl_list;
       int        *tag_list;
       int        num_vertex;
       float      vector[12];

       vector[ 0 * 3 + 0 ] = x;
       vector[ 0 * 3 + 1 ] = y;
       vector[ 0 * 3 + 2 ] = z;
       vector[ 1 * 3 + 0 ] = x;
       vector[ 1 * 3 + 1 ] = y;
       vector[ 1 * 3 + 2 ] = z + offset;

       for(i=0;i<num_steps;i++)
       {
          cos_array[i] = cos(2.0*M_PI*(float)i/(float)(num_steps-1));
          sin_array[i] = sin(2.0*M_PI*(float)i/(float)(num_steps-1));
       }

       pt_list = nl_list = NULL;

       num_vertex = 12  * num_steps ;

       pt_list = (float *)ALLOC_LOCAL(3*num_vertex*sizeof(float));
       nl_list = (float *)ALLOC_LOCAL(3*num_vertex*sizeof(float));
       cl_list = (float *)ALLOC_LOCAL(3*num_vertex*sizeof(float));
       tag_list = (int *)ALLOC_LOCAL(num_vertex*sizeof(int));

       /* preset colors to yellow */
       for (k=0; k<3; k++)
           for (i=0; i<num_vertex; i++)
               cl_list[ i*3+k ] = yellow[ k ];

       /* fill tag array for zone info */
       for (i=0; i<num_vertex; i++)
           tag_list[i] = tag;

       num_vertex = cylinder_create(
                        pt_list,
                        nl_list,
                        cl_list,
                        vector,
                        2,
                        NULL,
                        offset );

       GEOMadd_polytriangle(obj,
                        pt_list,
                        nl_list,
                        cl_list,
                        num_vertex,
                        GEOM_COPY_DATA);

        GEOMadd_polytriangle_vertex_data(obj,
                        tag_list,
                        polytri_count++,
                        num_vertex,
                        GEOM_COPY_DATA);

       if (pt_list) FREE_LOCAL (pt_list);
       if (nl_list) FREE_LOCAL (nl_list);
       if (cl_list) FREE_LOCAL (cl_list);
       if (tag_list) FREE_LOCAL (tag_list);
}

/******************************************************/
/* Utility to make cylinders from disjoint line data  */
/******************************************************/
int
cylinder_create(pt_list,nl_list,cl_list,src,n,in_colors,offset)
float      pt_list[][3];
float      nl_list[][3];
float      cl_list[][3];
float      src[][3];
float      in_colors[][3];
int        n;
float      offset;
{
    int         i, j;
    float       x[3], y[3], z[3];
    float       l, ct, st;
    int         index, save_index;
    float       local_offset;
    int         if_colors;

    if_colors = 0;
    index = save_index = 0;

    for(i=0;i<n;i+=2)
    {

      x[0] = src[i+1][0] - src[i][0];
      x[1] = src[i+1][1] - src[i][1];
      x[2] = src[i+1][2] - src[i][2];

      l = 1.0/sqrt(x[0]*x[0]+x[1]*x[1]+x[2]*x[2]);
      x[0] *= l;
      x[1] *= l;
      x[2] *= l; 

      z[0] = x[1]*xp[2] - x[2]*xp[1];
      z[1] = x[2]*xp[0] - x[0]*xp[2];
      z[2] = x[0]*xp[1] - x[1]*xp[0];

      st = sqrt(z[0]*z[0]+z[1]*z[1]+z[2]*z[2]);

      if(st==0.0)
      {   y[0] = yp[0];
          y[1] = yp[1];
          y[2] = yp[2];
          
          z[0] = zp[0];
          z[1] = zp[1];
          z[2] = zp[2];
      }
      else
      {   
          ct = x[0]*xp[0]+x[1]*xp[1]+x[2]*xp[2];

          z[0] /= st;
          z[1] /= st;
          z[2] /= st;
 
          y[0] = (xp[0] - x[0]*ct)/st;
          y[1] = (xp[1] - x[1]*ct)/st;
          y[2] = (xp[2] - x[2]*ct)/st;

      }

      if(index != 0)
      {   save_index = index;
          index += 2;
      }
      for(j=0;j<num_steps;j++)
      {   nl_list[index][0] = nl_list[index+1][0] =
                              cos_array[j]*y[0] + sin_array[j]*z[0];
          nl_list[index][1] = nl_list[index+1][1] =
                              cos_array[j]*y[1] + sin_array[j]*z[1];
          nl_list[index][2] = nl_list[index+1][2] =
                              cos_array[j]*y[2] + sin_array[j]*z[2];
          
          local_offset = offset * 0.25;
          pt_list[index][0] = src[i][0] + local_offset*nl_list[index][0];
          pt_list[index][1] = src[i][1] + local_offset*nl_list[index][1];
          pt_list[index][2] = src[i][2] + local_offset*nl_list[index][2];
          
          local_offset = offset;
          /* was src[i+1 ..., changed to src[i  */
          pt_list[index+1][0] = src[i][0] + local_offset*nl_list[index][0];
          pt_list[index+1][1] = src[i][1] + local_offset*nl_list[index][1];
          pt_list[index+1][2] = src[i][2] + local_offset*nl_list[index][2];

          /* assign vertex colours from input lines */
	  if ( if_colors )
	  {
            cl_list[index][0] = in_colors[i][0];
            cl_list[index][1] = in_colors[i][1];
            cl_list[index][2] = in_colors[i][2];
          
            cl_list[index+1][0] = in_colors[i+1][0];
            cl_list[index+1][1] = in_colors[i+1][1];
            cl_list[index+1][2] = in_colors[i+1][2];
	  }
          
          index += 2;
      }
      if(save_index != 0)
      {
          nl_list[save_index][0] = nl_list[save_index-1][0];
          nl_list[save_index][1] = nl_list[save_index-1][1];
          nl_list[save_index][2] = nl_list[save_index-1][2];
          pt_list[save_index][0] = pt_list[save_index-1][0];
          pt_list[save_index][1] = pt_list[save_index-1][1];
          pt_list[save_index][2] = pt_list[save_index-1][2];
	  if ( if_colors )
	  {
            cl_list[save_index][0] = cl_list[save_index-1][0];
            cl_list[save_index][1] = cl_list[save_index-1][1];
            cl_list[save_index][2] = cl_list[save_index-1][2];
          }

          nl_list[save_index+1][0] = nl_list[save_index+2][0];
          nl_list[save_index+1][1] = nl_list[save_index+2][1];
          nl_list[save_index+1][2] = nl_list[save_index+2][2];
          pt_list[save_index+1][0] = pt_list[save_index+2][0];
          pt_list[save_index+1][1] = pt_list[save_index+2][1];
          pt_list[save_index+1][2] = pt_list[save_index+2][2];
	  if ( if_colors )
	  {
            cl_list[save_index+1][0] = cl_list[save_index+2][0];
            cl_list[save_index+1][1] = cl_list[save_index+2][1];
            cl_list[save_index+1][2] = cl_list[save_index+2][2];
          }

      }
    }
    return ( index );
}

/******************************************************************************/
/*-----------------------------------------------------*
 *                                                     *
 *    ****  draw labels at zone positions  ****        *
 *                                                     *
 *-----------------------------------------------------*/

draw_label_group (obj, text, x, y, z, offset)
    GEOMobj *obj;
    char *text;
    float x, y, z;
    float offset;
{

    int         i, k, n;
    float       p[3];

    int label_flags;
    int font, title, background, drop, align, stroke;
    float label_offset[3];
    float label_size;
    char  string[40];


    /********************
     *  B O D Y         *
     ********************/

    font = 1;                   /* font 0-21 */
    title = 0;                  /* use ref_point */
    background = 1;             /* 1=filled rectable */
    drop = 1;                   /* 0=no, 1=drop shadow */
    stroke = 0;                 /* always zero */
    align = GEOM_LABEL_RIGHT;
    label_size = 0.04;          /* height in screen space */
    label_offset[0] = -0.1;
    label_offset[1] = label_offset[2] = 0.0;
    label_flags = GEOMcreate_label_flags(
               font, title, background, drop,align, stroke);

    /* get label point */
    p[0] =  x;
    p[1] =  y;
    p[2] =  z;

#ifdef INVERT_Z
    sprintf(string,"%s = %f", text, -1.0 * z);   /* negative  depth */
#else
    sprintf(string,"%s = %f", text, z);   /*  depth */
#endif
    GEOMadd_label ( obj, string, p,
                      label_offset, label_size, yellow, label_flags );


}


/*-----------------------------------------------------*
 *                                                     *
 *    ****  track_box_xform  ****                  *
 *                                                     *
 *-----------------------------------------------------*/

track_box_xform(out_matrix, ll, ur, addtl, distance)
    float out_matrix[4][4], addtl[4][4];
    float ll[3], ur[3], distance;
{
    float tx, ty, tz, tmpmat[4][4], hw, hh, hd;

    hw = (ll[0] + ur[0]) / 2.0;
    hh = (ll[1] + ur[1]) / 2.0;
    hd = (ll[2] + ur[2]) / 2.0;

    mat_identity(out_matrix);

    tx = out_matrix[3][0] * (ur[0] - ll[0]);
    ty = out_matrix[3][1] * (ur[1] - ll[1]);
    tz = out_matrix[3][2] * (ur[2] - ll[2]);
    out_matrix[0][3] = out_matrix[1][3] = out_matrix[2][3] = 0.0;

    mat_translate(tmpmat, -hw, -hh, -hd+distance);
    mat_multiply(tmpmat, out_matrix, out_matrix);

    mat_multiply(out_matrix, addtl, out_matrix);

    mat_translate(tmpmat,  tx+hw,  ty+hh,  tz+hd);
    mat_multiply(out_matrix, tmpmat, out_matrix);
}

