
/*********************************************************************

Copyright (C) 1995, Lawrence Berkeley Laboratory.  All Rights
Reserved.  Permission to copy and modify this software and its
documentation (if any) is hereby granted, provided that this notice
is retained thereon and on all copies.  

This software is provided as a professional academic contribution
for joint exchange.   Thus it is experimental and scientific
in nature, undergoing development, and is provided "as is" with
no warranties of any kind whatsoever, no support, promise of
updates or printed documentation.

This work is supported by the U. S. Department of Energy under 
contract number DE-AC03-76SF00098 between the U. S. Department 
of Energy and the University of California.


	Author: Wes Bethel
		Lawrence Berkeley Laboratory
                Berkeley, California 

  "this software is 100% hand-crafted by a human being in the USA"

  Spring 1995

  This header file implements device-event-to-AVS-field mappings.
  
  
*********************************************************************/


#ifndef __vrutil_h_
#define __vrutil_h_

/**
  * for the spaceball, include the .h file with the necessary typedefs.
**/

#ifdef SPACEBALL
#define Sb_ENVIRONMENT_INCLUDES
#include "sblibry.h"
#endif

#ifdef DIVISION_IPU
#include "fastrak.h"
#endif

#define VR_FIELD_STRING "field float 1D 1-space uniform"

/**
  * how much data do we need in the field?
  * well, we need 3 floats for the translation vector and 3 floats
  * for the rotation vector.
  * we need some space for some flags and a button press mask.  probably
  * 4 bytes for each of the flags and the mask will be ok.
  *
  * note that the field will be malloc'd as a float field, so there will
  * be some monkey business to treat the flags and button pushes as int's.
**/
#define VR_TRANSLATIONS 3
#define VR_ROTATIONS 3
#define VR_FLAGS 1
#define VR_BUTTON_MASK_LENGTH 1

#define VR_DATA_VECTOR_LENGTH (VR_TRANSLATIONS + VR_ROTATIONS + VR_BUTTON_MASK_LENGTH + VR_FLAGS)

#define VR_FLAGS_INDEX 0
#define VR_BUTTON_MASK_INDEX 1

#define VR_X_TRANS_INDEX 2
#define VR_Y_TRANS_INDEX 3
#define VR_Z_TRANS_INDEX 4

#define VR_X_ROT_INDEX 5
#define VR_Y_ROT_INDEX 6
#define VR_Z_ROT_INDEX 7


#define VR_HAVE_ROTATIONS (1<<0)
#define VR_HAVE_TRANSLATIONS (1<<1)
#define VR_HAVE_ANY_BUTTONS (1<<2)

#define VR_BUTTON1_MASK (1<<0)
#define VR_BUTTON2_MASK (1<<1)
#define VR_BUTTON3_MASK (1<<2)
#define VR_BUTTON4_MASK (1<<3)

#define VR_BUTTON5_MASK (1<<4)
#define VR_BUTTON6_MASK (1<<5)
#define VR_BUTTON7_MASK (1<<6)
#define VR_BUTTON8_MASK (1<<7)


/** the following is a work pointer used in the defines below. **/
static unsigned int *loc_iptr;
static unsigned int loc_ival,loc_rega,loc_regb;

/**
  * internally used macros.
**/

#define VR_SET_BIT(word,bval) ((word) = (word) | (bval))
#define VR_CLEAR_BIT(word,bval) ((word) = (word) & ~(bval))

/**
  * externally called macros to get and test presence of various types
  * of events.
  *
  * these are of the form TEST_XXX_FLAG and SET_XXX_FLAG.
  *
  * TEST_XXX_FLAG(value,field) will return 1 in "value" if the flag is
  *    set, and 0 in "value" if the flag isn't set.
  *
  * SET_XXX_FLAG(flagval,field) expects a "0" or a "1" in the
  *    flagval variable, and will set or clear the appropriate bit in
  *    the data structure.
**/


/* use this to see if any rotation events are present. */
#define TEST_ROTATIONS_FLAG(outflag,field) \
{ \
     loc_iptr = (unsigned int *)(((field)->field_union.field_data_float_u)+VR_FLAGS_INDEX); \
     outflag = (*loc_iptr) & VR_HAVE_ROTATIONS; \
}

/* use this to set/clear the "have rotations" flag. */
#define SET_ROTATIONS_FLAG(val,field) \
{ \
     loc_iptr = (unsigned int *)(((field)->field_union.field_data_float_u)+VR_FLAGS_INDEX); \
     loc_ival = *loc_iptr; \
     if (val) \
         VR_SET_BIT(loc_ival,VR_HAVE_ROTATIONS); \
     else \
         VR_CLEAR_BIT(loc_ival,VR_HAVE_ROTATIONS); \
     *loc_iptr = loc_ival; \
}

/* use this to see if any translation events are present. */
#define TEST_TRANSLATIONS_FLAG(outflag,field) \
{ \
     loc_iptr = (unsigned int *)(((field)->field_union.field_data_float_u)+VR_FLAGS_INDEX); \
     outflag = (*loc_iptr) & VR_HAVE_TRANSLATIONS; \
}

/* use this to set/clear the "have translations" flag. */
#define SET_TRANSLATIONS_FLAG(val,field) \
{ \
     loc_iptr = (unsigned int *)(((field)->field_union.field_data_float_u)+VR_FLAGS_INDEX); \
     loc_ival = *loc_iptr; \
     if (val) \
         VR_SET_BIT(loc_ival,VR_HAVE_TRANSLATIONS); \
     else \
         VR_CLEAR_BIT(loc_ival,VR_HAVE_TRANSLATIONS); \
     *loc_iptr = loc_ival; \
}

/* use this to see if there are any button events. */
#define TEST_BUTTONS_FLAG(outflag,field) \
{ \
     loc_iptr = (unsigned int *)(((field)->field_union.field_data_float_u)+VR_FLAGS_INDEX); \
     loc_ival = (*loc_iptr) & VR_HAVE_ANY_BUTTONS; \
     outflag = (loc_ival != 0); \
}

/* use this to set/clear the "have any buttons" flag */
#define SET_BUTTONS_FLAG(val,field) \
{ \
     loc_iptr = (unsigned int *)(((field)->field_union.field_data_float_u)+VR_FLAGS_INDEX); \
     loc_ival = *loc_iptr; \
     loc_rega = loc_ival; \
     loc_regb = VR_HAVE_ANY_BUTTONS; \
     if (val) \
         VR_SET_BIT(loc_ival,VR_HAVE_ANY_BUTTONS); \
     else \
         VR_CLEAR_BIT(loc_ival,VR_HAVE_ANY_BUTTONS); \
     *loc_iptr = loc_ival; \
}

/**
  * set the value (typically either 1 or 0) for a button.
  * "value" should be either 1 or 0.
  * "button_mask" should be one of VR_BUTTON1_MASK..VR_BUTTON8_MASK (above).
  * "field" is the AVS field which will be touched by this operation.
**/

#define VR_SET_BUTTON(button_mask,value,field) \
{ \
     loc_iptr = (unsigned int *)(((field)->field_union.field_data_float_u)+VR_BUTTON_MASK_INDEX); \
     loc_ival = *loc_iptr; \
     if (value) \
         VR_SET_BIT(loc_ival,button_mask); \
     else \
         VR_CLEAR_BIT(loc_ival,button_mask); \
     *loc_iptr = loc_ival; \
}

/**
  * get the value (typically either 1 or 0) for button indicated by
  * "button mask"  (which MUST be one of VR_BUTTON1_MASK.VR_BUTTON8_INDEX
  * from above).   the data source is "field".
**/

#define VR_TEST_BUTTON(button_mask,value,field) \
{ \
     loc_iptr = (unsigned int *)(((field)->field_union.field_data_float_u)+VR_BUTTON_MASK_INDEX); \
     loc_ival = *loc_iptr; \
     value = loc_ival & button_mask; \
}

/**
  * set the translation vector, ie, the "position" component of 6-d input
  * data.  the x,y,z components are in the scalar arguments.  the array
  * to be used for output is "data".
**/

#define VR_SET_TRANS_VECTOR(x,y,z,field) \
{ \
      ((field)->field_union.field_data_float_u)[VR_X_TRANS_INDEX] = (x); \
      ((field)->field_union.field_data_float_u)[VR_Y_TRANS_INDEX] = (y); \
      ((field)->field_union.field_data_float_u)[VR_Z_TRANS_INDEX] = (z); \
}

/**
  * get the translation vector, ie, the "position" component of 6-d input
  * data from the array "data".  the values will be returned in the x,y,z
  * scalar arguments.
**/

#define VR_GET_TRANS_VECTOR(x,y,z,field) \
{ \
      (x) = ((field)->field_union.field_data_float_u)[VR_X_TRANS_INDEX]; \
      (y) = ((field)->field_union.field_data_float_u)[VR_Y_TRANS_INDEX]; \
      (z) = ((field)->field_union.field_data_float_u)[VR_Z_TRANS_INDEX]; \
}


/**
  * set the rotation vector, ie, the "orientation" component of 6-d input
  * data.  the x,y,z components are in the scalar arguments.  the array
  * to be used for output is "data".
**/

#define VR_SET_ROT_VECTOR(x,y,z,field) \
{ \
      ((field)->field_union.field_data_float_u)[VR_X_ROT_INDEX] = (x); \
      ((field)->field_union.field_data_float_u)[VR_Y_ROT_INDEX] = (y); \
      ((field)->field_union.field_data_float_u)[VR_Z_ROT_INDEX] = (z); \
}

/**
  * get the rotation vector, ie, the "orientation" component of 6-d input
  * data from the array "data".  the values will be returned in the x,y,z
  * scalar arguments.
**/
#define VR_GET_ROT_VECTOR(x,y,z,field) \
{ \
      (x) = ((field)->field_union.field_data_float_u)[VR_X_ROT_INDEX]; \
      (y) = ((field)->field_union.field_data_float_u)[VR_Y_ROT_INDEX]; \
      (z) = ((field)->field_union.field_data_float_u)[VR_Z_ROT_INDEX]; \
}


/**
  * macro used for setting relevant items in an AVSfield template
  * for use in AVSfield_alloc() operations.
**/

static int VR_field_dims[1] = {VR_DATA_VECTOR_LENGTH};

#define MAKE_VR_AVS_FIELD_TEMPLATE(t) \
{ \
      (t)->uniform = UNIFORM; \
      (t)->ndim = 1; \
      (t)->nspace = 1; \
      (t)->veclen = 1; \
      (t)->type = AVS_TYPE_REAL; \
      (t)->size = sizeof(float); \
}

#define VR_COPY_ALL_DATA(d,s) \
       memcpy((d->field_union.field_data_char),(s->field_union.field_data_char),((s)->size)*VR_DATA_VECTOR_LENGTH);

#endif
