/*
 * Oslo.c - Given a refined knot vector, add control points to a surface.
 *
 * John Peterson

   Adapted by CGU
   Adapted by M.J.Evers

   05/08/1996
 */

#define MAXORDER 20         /* Maximum order allowed (for local array sizes) */
#define MAX(a,b)    (((a)>(b))?(a):(b))
#define MIN(a,b)    (((a)<(b))?(a):(b))

#define CHECK( n ) \
    { if (!(n)) { fprintf( stderr, "Ran out of memory\n" ); exit(-1); } }



#include <stdlib.h>
#include "blending.h"


/*
 * Return the current knot the parameter u is less than or equal to.
 * Find this "breakpoint" allows the evaluation routines to concentrate on
 * only those control points actually effecting the curve around u.]
 *
 *	m   is the number of points on the curve (or surface direction)
 *	k   is the order of the curve (or surface direction)
 *	kv  is the knot vector ([0..m+k-1]) to find the break point in.
 */

int
FindBreakPoint( Pfloat u, Pfloat * kv, int m, int k )
{
    int i;

    if (u == kv[m+1])	/* Special case for closed interval */
	return m;

    i = m + k;
    while ((u < kv[i]) && (i > 0))
	i--;
    return( i );
}


/*
 * Given the original knot vector ukv, and a new knotvector vkv, compute
 * the "alpha matrix" used to generate the corresponding new control points.
 * This routines allocates the alpha matrix if it isn't allocated already.
 *
 * This is from Bartels, Beatty & Barsky, p. 407
 */
static void
CalcAlpha( Pfloat * ukv, Pfloat * wkv, int m, int n, int k, Pfloat *** alpha )
{
    register int i, j;
    int brkPoint, r, rm1, last, s;
    Pfloat omega;
    Pfloat aval[MAXORDER];

    if (! *alpha)	/* Must allocate alpha */
    {
	CHECK( *alpha = (Pfloat **) malloc( (int) ((k+1) * sizeof( Pfloat * ))) );
	for (i = 0; i <= k; i++)
	    CHECK( (*alpha)[i] = (Pfloat *) malloc( (int) ((m + n + 1)
						    * sizeof( Pfloat ))) );
    }

    for (j = 0; j <= m + n; j++)
    {
	brkPoint = FindBreakPoint( wkv[j], ukv, m, k );

	aval[0] = 1.0;
	for (r = 2; r <= k; r++)
	{
	    rm1 = r - 1;
	    last = MIN( rm1, brkPoint );
	    i = brkPoint - last;

	    if (last < rm1)
		aval[last] = aval[last] * (wkv[j + r - 1] - ukv[i])
			    / (ukv[i + r - 1] - ukv[i]);
	    else
		aval[last] = 0.0;

	    for (s = last - 1; s >= 0; s-- )
	    {
		i++;
		omega = (wkv[j + r - 1] - ukv[i]) / (ukv[i + r - 1] - ukv[i]);
		aval[s + 1] = aval[s+1] + (1 - omega) * aval[s];
		aval[s] = omega * aval[s];
	    }
	}
	last = MIN( k - 1, brkPoint );
	for (i = 0; i <= k; i++)
	    (*alpha)[i][j] = 0.0;
	for (s = 0; s <= last; s++)
	    (*alpha)[last - s][j] = aval[s];
    }
}

/*
 * Apply the alpha matrix computed above to the rows (or columns)
 * of the surface.  If dirflag is true do the U's (row), else do V's (col).
 */
void
RefineSurface( PR_nurb * src, PR_nurb * dest, Boolean dirflag )
{
    register int i, j, out;
    register Ppoint4 * dp, * sp;
    int i1, brkPoint, maxj, maxout;
    register Pfloat tmp;
    Pfloat ** alpha = NULL;

    /* Compute the alpha matrix and indexing variables for the requested direction */

    if (dirflag)
    {
	CalcAlpha( src->pf_u.pf_kk->knots, dest->pf_u.pf_kk->knots, src->pf_u.pf_n - 1, dest->pf_u.pf_n - src->pf_u.pf_n,
		   src->pf_u.pf_k, &alpha );
	maxj = dest->pf_u.pf_n;
	maxout = src->pf_v.pf_n;
    }
    else
    {
	CalcAlpha( src->pf_v.pf_kk->knots, dest->pf_v.pf_kk->knots, src->pf_v.pf_n - 1, dest->pf_v.pf_n - src->pf_v.pf_n,
		   src->pf_v.pf_k, &alpha );
	maxj = dest->pf_v.pf_n;
	maxout = dest->pf_u.pf_n;
    }

    /* Apply the alpha matrix to the original control points, generating new ones */

    for (out = 0; out < maxout; out++)
	for (j = 0; j < maxj; j++)
	{
	    if (dirflag)
	    {
		dp = &(dest->pf_ppp->pts[out * dest->pf_u.pf_n + j]);
		brkPoint = FindBreakPoint( dest->pf_u.pf_kk->knots[j], src->pf_u.pf_kk->knots,
					   src->pf_u.pf_n-1, src->pf_u.pf_k );

		i1 = MAX( brkPoint - src->pf_u.pf_k + 1, 0 );
		
		sp = &(src->pf_ppp->pts[out * src->pf_u.pf_n + i1]);
	    } else {
		dp = &(dest->pf_ppp->pts[j * dest->pf_u.pf_n * out]);
		brkPoint = FindBreakPoint( dest->pf_v.pf_kk->knots[j], src->pf_v.pf_kk->knots,
					   src->pf_v.pf_n-1, src->pf_v.pf_k );

		i1 = MAX( brkPoint - src->pf_v.pf_k + 1, 0 );
		sp = &(src->pf_ppp->pts[i1 * src->pf_u.pf_n + out]);
	    }
	    dp->x = 0.0;
	    dp->y = 0.0;
	    dp->z = 0.0;
	    dp->w = 0.0;

	    /* Now fiddle the break point so that it doesn't crash with our non-clamped NURBS */

	    for (i = i1; i <= brkPoint; i++)
	    {
		tmp = alpha[i - i1][j];

		sp = (dirflag ? &(src->pf_ppp->pts[out * src->pf_u.pf_n + i]) : &(src->pf_ppp->pts[i * src->pf_u.pf_n + out]) );    

		dp->x += tmp * sp->x;
		dp->y += tmp * sp->y;
		dp->z += tmp * sp->z;
		dp->w += tmp * sp->w;
	    }
	}

    /* Free up the alpha matrix */
    for (i = 0; i <= (dirflag ? src->pf_u.pf_k : src->pf_v.pf_k); i++)
	free( alpha[i] );
    free( alpha );
}


