/*******************************************************************************
 *  PROJECT: GNOME Colorscheme
 *
 *  AUTHOR: Jonathon Jongsma
 *
 *  Copyright (c) 2005 Jonathon Jongsma
 *
 *  License:
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the 
 *    Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 *    Boston, MA  02111-1307  USA
 *
 *******************************************************************************/

#ifndef __GCS_COLORSWATCH_WIDGET_H
#define __GCS_COLORSWATCH_WIDGET_H

#include <memory>   // for std::auto_ptr
#include <gtkmm/drawingarea.h>
#include <gtkmm/tooltips.h>
#include <gdkmm/colormap.h>
#include <gdkmm/window.h>
#include "core/gcs-types.h"
#include "core/gcs-color.h"
namespace gcs
{
    namespace Widgets
    {

        /** A widget which displays a colorswatch
         * 
         * The ColorSwatch is a rectangular box with its background set to the
         * color we want to display.  It also displays some text describing the
         * color being displayed.
         *
         * The ColorSwatch widget is modeled on the example found at
         * http://www.gtkmm.org/docs/gtkmm-2.4/docs/tutorial/html/ch14s03.html
         */
        class ColorSwatch : public Gtk::DrawingArea
        {
            public:
                /** Constructor with background and foreground colors specified in
                 * RGB format
                 */
                ColorSwatch(ColorPtr bg);
                /** Copy constructor
                */
                ColorSwatch(const ColorSwatch& c);
                virtual ~ColorSwatch(void);

                /** Sets the width of the black border around the colorswatch
                 */
                void set_border_width(int width);
                /** Sets the size of the swatch in pixels */
                void set_min_size(int sz) { m_minSize = sz; set_size_request(sz, sz); }
                /** Sets the background color of the colorswatch.  
                 *
                 * This would perhaps more appropriately be named setColor,
                 * since its the main color that is being displayed
                 */
                virtual void set_color(ColorPtr bg);

                ColorPtr get_color() { return m_background; }

                /** A signal will be emitted when the user clicks the "Use
                 * Color" menu item on the right-click context menu for a
                 * particular colorswatch.  Another widget can connect to this
                 * and take actions based on it.
                 */
                sigc::signal<bool>& signal_selected(void) { return m_signal_selected; }

                /** An auto_ptr to a Gdk::Color */
                typedef std::auto_ptr<Gdk::Color> ref_gdkcolor_type;

            protected:
                /** Initialize stuff when the object is realized.  Before
                 * realization (i.e. in the constructor), we can't call
                 * get_window() or anything because the Gdk::Window doesn't yet
                 * exist
                 */
                virtual void on_realize(void);

                /* Actions to take whenever the drawing area needs to be redrawn
                 * (i.e. after return from being minimized or being brought in
                 * front of another window.  This is where all of the drawing takes
                 * place.
                 */
                virtual bool on_expose_event(GdkEventExpose *e);

                /** Handles the action of pressing a mouse button on a
                 * ColorSwatch object.  Will pop up a context menu if one is
                 * defined, else will do nothing
                 */
                virtual bool on_button_press_event(GdkEventButton *e);

                virtual void on_drag_data_get(const
                        Glib::RefPtr<Gdk::DragContext>& context,
                        Gtk::SelectionData& selection_data, guint info,
                        guint time);

                void set_color_icon(const Glib::RefPtr<Gdk::DragContext>& context);

                /** The colormap of the ColorSwatch
                */
                Glib::RefPtr<Gdk::Colormap> m_colormap;

                /** The background color of the ColorSwatch
                */
                ColorPtr m_background;

                sigc::signal<bool> m_signal_selected;

                /** The minimum size that a swatch is allowed to get.  This is
                 * used for both the height and the width of the swatch
                 */
                int m_minSize;

                /** The width of the line used to draw the border around the
                 * swatch
                 */
                int m_borderWidth;

                int m_cornerRadius;

                /** The padding around the edge of the swatch.  If several
                 * swatches are next to eachother, this will add some space
                 * between them so they're not stuck right together
                 */
                int m_swatchPadding;

                /** The color white will be used to draw text when the
                 * background color is dark
                 */
                static ColorPtr m_white;

                /** The color black will be used to draw text when the
                 * background color is light
                 */
                static ColorPtr m_black;

                static Gtk::Tooltips* pTooltips;
                static int tooltip_refs;
        };

    } // namespace Widgets
} // namespace gcs

#endif // __GCS_COLORSWATCH_WIDGET_H
