/*
 *   This file is part of Dianara
 *   Copyright 2012-2014  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "configdialog.h"

ConfigDialog::ConfigDialog(GlobalObject *globalObject,
                           QString dataDirectory, int updateInterval,
                           int postsPerPageMain, int postsPerPageOther,
                           int tabsPosition, bool tabsMovable,
                           bool publicPosts, FDNotifications *notifier,
                           QWidget *parent) : QWidget(parent)
{
    this->globalObj = globalObject;
    this->fdNotifier = notifier;

    this->setWindowTitle(tr("Program Configuration") + " - Dianara");
    this->setWindowIcon(QIcon::fromTheme("configure",
                                         QIcon(":/images/button-configure.png")));
    this->setWindowFlags(Qt::Window);
    this->setWindowModality(Qt::ApplicationModal);
    this->setMinimumSize(560, 420);

    QSettings settings;
    settings.beginGroup("Configuration");

    // Standalone Proxy config window
    QByteArray proxyPasswd = QByteArray::fromBase64(settings.value("proxyPassword").toByteArray());
    proxyDialog = new ProxyDialog(settings.value("proxyType", 0).toInt(),
                                  settings.value("proxyHostname").toString(),
                                  settings.value("proxyPort").toString(),
                                  settings.value("proxyUseAuth", false).toBool(),
                                  settings.value("proxyUser").toString(),
                                  QString::fromLocal8Bit(proxyPasswd),
                                  this);


    //////////////////////////////////////////////////////////////// Upper part

    // Tab 1, general options
    this->generalOptionsLayout = new QFormLayout();

    updateIntervalSpinbox = new QSpinBox();
    updateIntervalSpinbox->setRange(2, 99); // 2-99 min
    updateIntervalSpinbox->setSuffix(" "+ tr("minutes"));
    updateIntervalSpinbox->setValue(updateInterval);
    generalOptionsLayout->addRow(tr("Timeline &update interval"),
                                 updateIntervalSpinbox);

    postsPerPageMainSpinbox = new QSpinBox();
    postsPerPageMainSpinbox->setRange(5, 100); // 5-100 ppp
    postsPerPageMainSpinbox->setSuffix(" "+ tr("posts",
                                               "Goes after a number, as: 25 posts"));
    postsPerPageMainSpinbox->setValue(postsPerPageMain);
    generalOptionsLayout->addRow(tr("&Posts per page, main timeline"),
                                 postsPerPageMainSpinbox);

    postsPerPageOtherSpinbox = new QSpinBox();
    postsPerPageOtherSpinbox->setRange(1, 100); // 1-100 ppp
    postsPerPageOtherSpinbox->setSuffix(" "+ tr("posts",
                                                "This goes after a number, like: 10 posts"));
    postsPerPageOtherSpinbox->setValue(postsPerPageOther);
    generalOptionsLayout->addRow(tr("Posts per page, &other timelines"),
                                 postsPerPageOtherSpinbox);


    tabsPositionCombobox = new QComboBox();
    tabsPositionCombobox->addItem(QIcon::fromTheme("arrow-up"),
                                  tr("Top"));
    tabsPositionCombobox->addItem(QIcon::fromTheme("arrow-down"),
                                  tr("Bottom"));
    tabsPositionCombobox->addItem(QIcon::fromTheme("arrow-left"),
                                  tr("Left side"));
    tabsPositionCombobox->addItem(QIcon::fromTheme("arrow-right"),
                                  tr("Right side"));
    tabsPositionCombobox->setCurrentIndex(tabsPosition);
    generalOptionsLayout->addRow(tr("&Tabs position"),
                                 tabsPositionCombobox);


    tabsMovableCheckbox = new QCheckBox();
    tabsMovableCheckbox->setChecked(tabsMovable);
    generalOptionsLayout->addRow(tr("&Movable tabs"),
                                 tabsMovableCheckbox);


    publicPostsCheckbox = new QCheckBox();
    publicPostsCheckbox->setChecked(publicPosts);
    generalOptionsLayout->addRow(tr("Public posts as &default"),
                                 publicPostsCheckbox);

    proxyConfigButton = new QPushButton(QIcon::fromTheme("preferences-system-network"),
                                        tr("Pro&xy Settings"));
    connect(proxyConfigButton, SIGNAL(clicked()),
            proxyDialog, SLOT(show()));
    generalOptionsLayout->addRow(tr("Network configuration"),
                                 proxyConfigButton);

    filterEditorButton = new QPushButton(QIcon::fromTheme("view-filter"),
                                         tr("Set Up F&ilters"));
    connect(filterEditorButton, SIGNAL(clicked()),
            this, SIGNAL(filterEditorRequested()));
    generalOptionsLayout->addRow(tr("Filtering rules"),
                                 filterEditorButton);

    generalOptionsWidget = new QWidget();
    generalOptionsWidget->setLayout(generalOptionsLayout);



    // Tab 2, fonts
    this->fontOptionsLayout = new QVBoxLayout();
    fontPicker1 = new FontPicker(tr("Post Titles"),
                                 globalObj->getPostTitleFont());
    fontPicker2 = new FontPicker(tr("Post Contents"),
                                 globalObj->getPostContentsFont());
    fontPicker3 = new FontPicker(tr("Comments"),
                                 globalObj->getCommentsFont());
    fontPicker4 = new FontPicker(tr("Minor Feed"),
                                 globalObj->getMinorFeedFont());
    // FIXME: more for "Timestamps" or something else?

    fontOptionsLayout->addWidget(fontPicker1);
    fontOptionsLayout->addWidget(fontPicker2);
    fontOptionsLayout->addWidget(fontPicker3);
    fontOptionsLayout->addWidget(fontPicker4);
    fontOptionsLayout->addStretch(1);

    fontOptionsWidget = new QWidget();
    fontOptionsWidget->setLayout(fontOptionsLayout);



    // Tab 3, colors
    this->colorOptionsLayout = new QVBoxLayout();

    colorPicker1 = new ColorPicker(tr("You are among the recipients "
                                      "of the activity, such as "
                                      "a comment addressed to you.")
                                   + "\n"
                                   + tr("Used also when highlighting posts "
                                        "addressed to you in the timelines."),
                                   settings.value("color1", "DISABLED#401020").toString());
    colorOptionsLayout->addWidget(colorPicker1);

    colorPicker2 = new ColorPicker(tr("The activity is in reply to something "
                                      "done by you, such as a comment posted "
                                      "in reply to one of your notes."),
                                   settings.value("color2", "DISABLED#154060").toString());
    colorOptionsLayout->addWidget(colorPicker2);

    colorPicker3 = new ColorPicker(tr("You are the object of the activity, "
                                      "such as someone adding you to a list."),
                                   settings.value("color3", "DISABLED#403510").toString());
    colorOptionsLayout->addWidget(colorPicker3);

    colorPicker4 = new ColorPicker(tr("The activity is related to one of "
                                      "your objects, such as someone "
                                      "liking one of your posts.")
                                   + "\n"
                                   + tr("Used also when highlighting your "
                                        "own posts in the timelines."),
                                   settings.value("color4", "DISABLED#103510").toString());
    colorOptionsLayout->addWidget(colorPicker4);

    colorPicker5 = new ColorPicker(tr("Item highlighted due to filtering rules."),
                                   settings.value("color5").toString());
    colorOptionsLayout->addWidget(colorPicker5);

    colorPicker6 = new ColorPicker(tr("Item is new."),
                                   settings.value("color6").toString());
    colorOptionsLayout->addWidget(colorPicker6);


    colorOptionsWidget = new QWidget();
    colorOptionsWidget->setLayout(colorOptionsLayout);



    // Tab 4, notification and systray icon options
    notificationOptionsLayout = new QFormLayout();

    notificationStyleCombobox = new QComboBox();
    notificationStyleCombobox->addItem(QIcon::fromTheme("preferences-desktop-notification"),
                                       tr("As system notifications"));
    notificationStyleCombobox->addItem(QIcon::fromTheme("view-conversation-balloon"),
                                       tr("Using own notifications"));
    notificationStyleCombobox->addItem(QIcon::fromTheme("user-busy"), // dialog-cancel
                                       tr("Don't show notifications"));
    notificationStyleCombobox->setCurrentIndex(settings.value("showNotifications", 0).toInt());
    notificationOptionsLayout->addRow(tr("Notification Style"),
                                      notificationStyleCombobox);


    notificationOptionsLayout->addRow(new QLabel("\n" + tr("Notify when receiving:")));

    notifyNewTLCheckbox = new QCheckBox();
    notifyNewTLCheckbox->setChecked(settings.value("notifyNewTL", false).toBool());
    notificationOptionsLayout->addRow(tr("New posts"),
                                      notifyNewTLCheckbox);

    notifyHLTLCheckbox = new QCheckBox();
    notifyHLTLCheckbox->setChecked(settings.value("notifyHLTL", true).toBool());
    notificationOptionsLayout->addRow(tr("Highlighted posts"),
                                      notifyHLTLCheckbox);

    notifyNewMWCheckbox = new QCheckBox();
    notifyNewMWCheckbox->setChecked(settings.value("notifyNewMW", false).toBool());
    notificationOptionsLayout->addRow(tr("New activities in minor feed"),
                                      notifyNewMWCheckbox);

    notifyHLMWCheckbox = new QCheckBox();
    notifyHLMWCheckbox->setChecked(settings.value("notifyHLMW", true).toBool());
    notificationOptionsLayout->addRow(tr("Highlighted activities in minor feed"),
                                      notifyHLMWCheckbox);

    this->syncNotifierOptions();

    QFrame *lineFrame = new QFrame();
    lineFrame->setFrameStyle(QFrame::HLine);
    lineFrame->setMinimumHeight(16);
    notificationOptionsLayout->addRow(lineFrame);


    systrayIconTypeCombobox = new QComboBox();
    systrayIconTypeCombobox->addItem(tr("Default"));
    systrayIconTypeCombobox->addItem(tr("System iconset, if available"));
    systrayIconTypeCombobox->addItem(tr("Show your current avatar"));
    systrayIconTypeCombobox->addItem(tr("Custom icon"));
    systrayIconTypeCombobox->setCurrentIndex(settings.value("systrayIconType", 0).toInt());
    notificationOptionsLayout->addRow(tr("System Tray Icon &Type"),
                                      systrayIconTypeCombobox);


    systrayCustomIconButton = new QPushButton(tr("S&elect..."));
    systrayCustomIconFN = settings.value("systrayCustomIconFN").toString();
    // FIXME: merge this with code used in pickCustomIconFile()
    // and turn the warning messageBox into a label
    if (!QPixmap(systrayCustomIconFN).isNull())
    {
        systrayCustomIconButton->setIcon(QIcon(systrayCustomIconFN));
    }
    else
    {
        systrayCustomIconButton->setIcon(QIcon(":/icon/32x32/dianara.png"));
    }
    connect(systrayCustomIconButton, SIGNAL(clicked()),
            this, SLOT(pickCustomIconFile()));
    notificationOptionsLayout->addRow(tr("Custom &Icon"),
                                      systrayCustomIconButton);


    notificationOptionsWidget = new QWidget();
    notificationOptionsWidget->setLayout(notificationOptionsLayout);

    settings.endGroup();



    //// Tab widget
    tabWidget = new QTabWidget();
    tabWidget->setSizePolicy(QSizePolicy::MinimumExpanding,
                             QSizePolicy::MinimumExpanding);
    tabWidget->addTab(generalOptionsWidget,
                      QIcon::fromTheme("preferences-other"),
                      tr("&General Options"));
    tabWidget->addTab(fontOptionsWidget,
                      QIcon::fromTheme("preferences-desktop-font"),
                      tr("&Fonts"));
    tabWidget->addTab(colorOptionsWidget,
                      QIcon::fromTheme("preferences-desktop-color"),
                      tr("Co&lors"));
    tabWidget->addTab(notificationOptionsWidget,
                      QIcon::fromTheme("preferences-desktop-notification"),
                      tr("&Notifications and System Tray"));



    /////////////////////////////////////////////////////////////// Bottom part

    // Label to show where the data directory is
    dataDirectoryLabel = new QLabel(tr("Dianara stores data in this folder:")
                                    + QString(" <a href=\"%1\">%2</a>")
                                      .arg(dataDirectory).arg(dataDirectory));
    dataDirectoryLabel->setWordWrap(true);
    dataDirectoryLabel->setOpenExternalLinks(true);
    QFont dataDirectoryFont;
    dataDirectoryFont.setPointSize(dataDirectoryFont.pointSize() - 1);
    dataDirectoryLabel->setFont(dataDirectoryFont);


    // Save / Cancel buttons
    saveConfigButton = new QPushButton(QIcon::fromTheme("document-save",
                                                        QIcon(":/images/button-save.png")),
                                 tr("&Save Configuration"));
    connect(saveConfigButton, SIGNAL(clicked()),
            this, SLOT(saveConfiguration()));
    cancelButton = new QPushButton(QIcon::fromTheme("dialog-cancel",
                                                    QIcon(":/images/button-cancel.png")),
                                   tr("&Cancel"));
    connect(cancelButton, SIGNAL(clicked()),
            this, SLOT(hide()));

    this->buttonsLayout = new QHBoxLayout();
    buttonsLayout->setAlignment(Qt::AlignRight);
    buttonsLayout->addWidget(saveConfigButton);
    buttonsLayout->addWidget(cancelButton);

    closeAction = new QAction(this);
    closeAction->setShortcut(QKeySequence(Qt::Key_Escape));
    connect(closeAction, SIGNAL(triggered()),
            this, SLOT(hide()));
    this->addAction(closeAction);


    // Set up main layout

    mainLayout = new QVBoxLayout();
    mainLayout->addWidget(tabWidget, 2);
    mainLayout->addSpacing(16);
    mainLayout->addStretch(1);
    mainLayout->addWidget(dataDirectoryLabel);
    mainLayout->addSpacing(16);
    mainLayout->addStretch(1);
    mainLayout->addLayout(buttonsLayout);
    this->setLayout(mainLayout);


    qDebug() << "Config dialog created";
}

ConfigDialog::~ConfigDialog()
{
    qDebug() << "Config dialog destroyed";
}



void ConfigDialog::syncNotifierOptions()
{
    this->fdNotifier->setNotificationOptions(notificationStyleCombobox->currentIndex(),
                                             notifyNewTLCheckbox->isChecked(),
                                             notifyHLTLCheckbox->isChecked(),
                                             notifyNewMWCheckbox->isChecked(),
                                             notifyHLMWCheckbox->isChecked());
}



//////////////////////////////////////////////////////////////////////////////
///////////////////////////////// SLOTS //////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////



void ConfigDialog::saveConfiguration()
{
    QSettings settings;
    settings.beginGroup("Configuration");

    settings.setValue("updateInterval", this->updateIntervalSpinbox->value());

    settings.setValue("postsPerPageMain",  this->postsPerPageMainSpinbox->value());
    settings.setValue("postsPerPageOther", this->postsPerPageOtherSpinbox->value());

    settings.setValue("tabsPosition", this->tabsPositionCombobox->currentIndex());
    settings.setValue("tabsMovable",  this->tabsMovableCheckbox->isChecked());

    settings.setValue("publicPosts", this->publicPostsCheckbox->isChecked());

    settings.setValue("font1", this->fontPicker1->getFontInfo());
    settings.setValue("font2", this->fontPicker2->getFontInfo());
    settings.setValue("font3", this->fontPicker3->getFontInfo());
    settings.setValue("font4", this->fontPicker4->getFontInfo());
    this->globalObj->syncFontSettings(this->fontPicker1->getFontInfo(),
                                      this->fontPicker2->getFontInfo(),
                                      this->fontPicker3->getFontInfo(),
                                      this->fontPicker4->getFontInfo());

    settings.setValue("color1", this->colorPicker1->getCurrentColor());
    settings.setValue("color2", this->colorPicker2->getCurrentColor());
    settings.setValue("color3", this->colorPicker3->getCurrentColor());
    settings.setValue("color4", this->colorPicker4->getCurrentColor());
    settings.setValue("color5", this->colorPicker5->getCurrentColor());
    settings.setValue("color6", this->colorPicker6->getCurrentColor());

    settings.setValue("showNotifications", this->notificationStyleCombobox->currentIndex());
    settings.setValue("notifyNewTL", this->notifyNewTLCheckbox->isChecked());
    settings.setValue("notifyHLTL",  this->notifyHLTLCheckbox->isChecked());
    settings.setValue("notifyNewMW", this->notifyNewMWCheckbox->isChecked());
    settings.setValue("notifyHLMW",  this->notifyHLMWCheckbox->isChecked());
    this->syncNotifierOptions();

    settings.setValue("systrayIconType", this->systrayIconTypeCombobox->currentIndex());
    settings.setValue("systrayCustomIconFN", this->systrayCustomIconFN);

    settings.endGroup();


    settings.sync();
    emit configurationChanged(); // Ask MainWindow to reload some stuff

    this->hide();   // this->close() would end the program if mainWindow was hidden

    qDebug() << "ConfigDialog: config saved";
}



void ConfigDialog::pickCustomIconFile()
{
    systrayCustomIconFN = QFileDialog::getOpenFileName(this,
                                                       tr("Select custom icon"),
                                                       QDir::homePath(),
                                                       tr("Image files")
                                                       + " (*.png *.jpg *.jpeg *.gif);;"
                                                       + tr("All files") + " (*)");

    if (!systrayCustomIconFN.isEmpty())
    {
        qDebug() << "Selected" << systrayCustomIconFN << "as new custom tray icon";

        QPixmap iconPixmap = QPixmap(systrayCustomIconFN);
        if (!iconPixmap.isNull())
        {
            this->systrayCustomIconButton->setIcon(QIcon(systrayCustomIconFN));
        }
        else
        {
            QMessageBox::warning(this,
                                 tr("Invalid image"),
                                 tr("The selected image is not valid."));
            qDebug() << "Invalid tray icon file selected";
        }
    }
}
