// SPDX-License-Identifier: GPL-2.0-or-later OR MPL-1.1 OR LGPL-2.1-or-later
/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*-
 *
 */
/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is EGE Output Action.
 *
 * The Initial Developer of the Original Code is
 * Jon A. Cruz.
 * Portions created by the Initial Developer are Copyright (C) 2007
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

/* Note: this file should be kept compilable as both .cpp and .c */

#include <cstring>

#include <gtk/gtk.h>
#include <gtkmm/container.h>
#include <gtkmm/label.h>

#include "widgets/ege-output-action.h"


static void ege_output_action_get_property( GObject* obj, guint propId, GValue* value, GParamSpec * pspec );
static void ege_output_action_set_property( GObject* obj, guint propId, const GValue *value, GParamSpec* pspec );
static void fixup_labels( GObject *gobject, GParamSpec *arg1, gpointer user_data );

/* static GtkWidget* create_menu_item( GtkAction* action ); */
static GtkWidget* create_tool_item( GtkAction* action );

typedef struct {
    gboolean useMarkup;
} EgeOutputActionPrivate;

#define EGE_OUTPUT_ACTION_GET_PRIVATE( o ) \
    reinterpret_cast<EgeOutputActionPrivate *>(ege_output_action_get_instance_private (o))

enum {
    PROP_USE_MARKUP = 1,
};

G_DEFINE_TYPE_WITH_PRIVATE(EgeOutputAction, ege_output_action, GTK_TYPE_ACTION);

void ege_output_action_class_init( EgeOutputActionClass* klass )
{
    if ( klass ) {
        GObjectClass* objClass = G_OBJECT_CLASS( klass );

        objClass->get_property = ege_output_action_get_property;
        objClass->set_property = ege_output_action_set_property;

/*         klass->parent_class.create_menu_item = create_menu_item; */
        klass->parent_class.create_tool_item = create_tool_item;

        g_object_class_install_property( objClass,
                                         PROP_USE_MARKUP,
                                         g_param_spec_boolean( "use-markup",
                                                               "UseMarkup",
                                                               "If markup should be used",
                                                               FALSE,
                                                               (GParamFlags)(G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT) ) );
    }
}


void ege_output_action_init( EgeOutputAction* action )
{
    auto priv = EGE_OUTPUT_ACTION_GET_PRIVATE( action );
    priv->useMarkup = FALSE;

    g_signal_connect( action, "notify", G_CALLBACK( fixup_labels ), NULL );
}

EgeOutputAction* ege_output_action_new( const gchar *name,
                                        const gchar *label,
                                        const gchar *tooltip,
                                        const gchar *stock_id )
{
    GObject* obj = (GObject*)g_object_new( EGE_OUTPUT_ACTION_TYPE,
                                           "name", name,
                                           "label", label,
                                           "tooltip", tooltip,
                                           "stock_id", stock_id,
                                           "use-markup", FALSE,
                                           NULL );

    EgeOutputAction* action = EGE_OUTPUT_ACTION( obj );

    return action;
}

gboolean ege_output_action_get_use_markup( EgeOutputAction* action )
{
    g_return_val_if_fail( IS_EGE_OUTPUT_ACTION(action), FALSE );
    auto priv = EGE_OUTPUT_ACTION_GET_PRIVATE( action );

    return priv->useMarkup;
}

void ege_output_action_set_use_markup( EgeOutputAction* action, gboolean setting )
{
    g_object_set( G_OBJECT(action), "use-markup", setting, NULL );
}

void ege_output_action_get_property( GObject* obj, guint propId, GValue* value, GParamSpec * pspec )
{
    EgeOutputAction* action = EGE_OUTPUT_ACTION( obj );
    auto priv = EGE_OUTPUT_ACTION_GET_PRIVATE( action );
    switch ( propId ) {
        case PROP_USE_MARKUP:
            g_value_set_boolean( value, priv->useMarkup );
            break;

        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID( obj, propId, pspec );
    }
}

void ege_output_action_set_property( GObject* obj, guint propId, const GValue *value, GParamSpec* pspec )
{
    EgeOutputAction* action = EGE_OUTPUT_ACTION( obj );
    auto priv = EGE_OUTPUT_ACTION_GET_PRIVATE( action );
    switch ( propId ) {
        case PROP_USE_MARKUP:
        {
            priv->useMarkup = g_value_get_boolean( value );
        }
        break;

        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID( obj, propId, pspec );
    }
}


/* static GtkWidget* create_menu_item( GtkAction* action ) */

GtkWidget* create_tool_item( GtkAction* action )
{
    GtkWidget* item = nullptr;

    if ( IS_EGE_OUTPUT_ACTION(action) )
    {
        auto act  = EGE_OUTPUT_ACTION (action);
        auto priv = EGE_OUTPUT_ACTION_GET_PRIVATE( act );
        GValue value;
        auto hb = gtk_box_new(GTK_ORIENTATION_HORIZONTAL, 5);
	gtk_box_set_homogeneous(GTK_BOX(hb), FALSE);
        GtkWidget* lbl = nullptr;
        memset( &value, 0, sizeof(value) );

        g_value_init( &value, G_TYPE_STRING );
        g_object_get_property( G_OBJECT(action), "short_label", &value );
        const gchar* sss = g_value_get_string( &value );

        item = GTK_WIDGET( gtk_tool_item_new() );

        lbl = gtk_label_new( " " );
        gtk_container_add( GTK_CONTAINER(hb), lbl );

        if ( priv->useMarkup ) {
            lbl = gtk_label_new(nullptr);
            gtk_label_set_markup( GTK_LABEL(lbl), sss ? sss : " " );
        } else {
            lbl = gtk_label_new( sss ? sss : " " );
        }
        gtk_container_add( GTK_CONTAINER(hb), lbl );

        lbl = gtk_label_new( " " );
        gtk_container_add( GTK_CONTAINER(hb), lbl );

        gtk_container_add( GTK_CONTAINER(item), hb );

        gtk_widget_show_all( item );

        g_value_unset( &value );
    } else {
        item = GTK_ACTION_CLASS(ege_output_action_parent_class)->create_tool_item( action );
    }

    return item;
}

void fixup_labels( GObject *gobject, GParamSpec *arg1, gpointer user_data )
{
    /* TODO: handle 'use-markup' getting changed also */

    if ( arg1 && arg1->name && (strcmp("label", arg1->name) == 0) ) {
        GSList* proxies = gtk_action_get_proxies( GTK_ACTION(gobject) );
        gchar* str = nullptr;
        g_object_get( gobject, "label", &str, NULL );
        Glib::ustring str2(str);
        (void)user_data;
        while ( proxies ) {
            if ( GTK_IS_TOOL_ITEM(proxies->data) ) {
                /* Search for the things we built up in create_tool_item() */
                std::vector<Gtk::Widget*> children = Glib::wrap(GTK_CONTAINER(proxies->data))->get_children();
                if ( !children.empty() ) {
                    if ( GTK_IS_BOX(children[0]->gobj()) ) {
                        children = dynamic_cast<Gtk::Container *>(children[0])->get_children();
                        if ( children.size()>1 ) {
                            Gtk::Widget *child = children[1];
                            if ( GTK_IS_LABEL(child->gobj()) ) {
                                Gtk::Label* lbl = dynamic_cast<Gtk::Label *>(child);
                                auto action = EGE_OUTPUT_ACTION(gobject);
                                auto priv   = EGE_OUTPUT_ACTION_GET_PRIVATE( action );
                                if ( priv->useMarkup ) {
                                    lbl->set_markup(str2);
                                } else {
                                    lbl->set_text(str2);
                                }
                            }
                        }
                    }
                }
            }
            proxies = g_slist_next( proxies );
        }
        g_free( str );
    }
}
/*
  Local Variables:
  mode:c++
  c-file-style:"stroustrup"
  c-file-offsets:((innamespace . 0)(inline-open . 0)(case-label . +))
  indent-tabs-mode:nil
  fill-column:99
  End:
*/
// vim: filetype=cpp:expandtab:shiftwidth=4:tabstop=8:softtabstop=4:fileencoding=utf-8:textwidth=99 :
