/***************************** LICENSE START ***********************************

 Copyright 2012 ECMWF and INPE. This software is distributed under the terms
 of the Apache License version 2.0. In applying this license, ECMWF does not
 waive the privileges and immunities granted to it by virtue of its status as
 an Intergovernmental Organization or submit itself to any jurisdiction.

 ***************************** LICENSE END *************************************/

#include <MvRequestUtil.hpp>
#include "Assertions.hpp"

#include "MagicsTranslator.h"
#include "ObjectList.h"
#include "PlotModConst.h"

// =========================================================
//
// Singleton objects for each class
AxisTranslator         axisTranslatorInstance    ( "PAXIS" );
BufrTranslator         bufrTranslatorInstance    ( "BUFR" );
CoastTranslator        coast1TranslatorInstance  ( "PCOAST" );
//CoastTranslator        coast2TranslatorInstance  ( "GEO_COASTLINE" );
//CoastTranslator        coast3TranslatorInstance  ( "GEO_LAND_SHADE" );
//CoastTranslator        coast4TranslatorInstance  ( "GEO_SEA_SHADE" );
//CoastTranslator        coast5TranslatorInstance  ( "GEO_BOUNDARIES" );
//CoastTranslator        coast6TranslatorInstance  ( "GEO_CITIES" );
//CoastTranslator        coast7TranslatorInstance  ( "GEO_RIVERS" );
//CoastTranslator        coast8TranslatorInstance  ( "GEO_GRID" );
ContTranslator         contTranslatorInstance    ( "PCONT" );
GeopointsTranslator    geopTranslatorInstance    ( "GEOPOINTS" );
GraphTranslator        graphTranslatorInstance   ( "PGRAPH" );
GribTranslator         gribTranslatorInstance    ( "GRIB" );
GribVectorsTranslator  iivectorsTranslatorInstance ( "GRIB_VECTORS" );
NetCDFTranslator       ncGeoPTranslatorInstance  ( "NETCDF_GEO_POINTS"  );
NetCDFTranslator       ncGeoVTranslatorInstance  ( "NETCDF_GEO_VECTORS" );
NetCDFTranslator       ncGeoMTranslatorInstance  ( "NETCDF_GEO_MATRIX"  );
NetCDFTranslator       ncMatTranslatorInstance   ( "NETCDF_XY_MATRIX"   );
NetCDFTranslator       ncXYVTTranslatorInstance  ( "NETCDF_XY_VECTORS"   );
NetCDFTranslator       ncXYPTranslatorInstance   ( "NETCDF_XY_POINTS"   );
ObsTranslator          obsTranslatorInstance     ( "MOBS" );
PageTranslator         pageTranslatorInstance    ( "PAGE" );
SymbolTranslator       symbolTranslatorInstance  ( "PSYMB" );
TextTranslator         textTranslatorInstance	 ( "PTEXT" );
TextTranslator         mtextTranslatorInstance   ( "MTEXT" );
VoidTranslator         input1TranslatorInstance  ( "INPUT_HISTOGRAM" );
VoidTranslator         input2TranslatorInstance  ( "INPUT_XY_POINTS" );
VoidTranslator         input3TranslatorInstance  ( "INPUT_GEO_POINTS" );
VoidTranslator         input4TranslatorInstance  ( "INPUT_XY_VECTORS" );
VoidTranslator         input5TranslatorInstance  ( "INPUT_GEO_VECTORS" );
VoidTranslator         input6TranslatorInstance  ( "INPUT_XY_BINNING" );
VoidTranslator         input7TranslatorInstance  ( "INPUT_GEO_BINNING" );
VoidTranslator         input8TranslatorInstance  ( "INPUT_XY_BOXES" );
VoidTranslator         input9TranslatorInstance  ( "INPUT_GEO_BOXES" );
VoidTranslator         input10TranslatorInstance ( "INPUT_HOR_BAR" );
VoidTranslator         input11TranslatorInstance ( "INPUT_VER_BAR" );
VoidTranslator         mcoastTranslatorInstance  ( "MCOAST" );
VoidTranslator         mcontTranslatorInstance   ( "MCONT" );
VoidTranslator         mgraphTranslatorInstance  ( "MGRAPH" );
VoidTranslator         mlegendTranslatorInstance ( "MLEGEND" );
VoidTranslator         msymbolTranslatorInstance ( "MSYMB" );
VoidTranslator         mwindTranslatorInstance   ( "MWIND" );
VoidTranslator         odbGeoPTranslatorInstance ( "ODB_GEO_POINTS" );
VoidTranslator         odbGeoVTranslatorInstance ( "ODB_GEO_VECTORS" );
VoidTranslator         odbXyPTranslatorInstance  ( "ODB_XY_POINTS" );
VoidTranslator         odbXyVTranslatorInstance  ( "ODB_XY_VECTORS" );
VoidTranslator         odbXyMTranslatorInstance  ( "ODB_XY_BINNING" );
ImportTranslator       mimportTranslatorInstance ( "MIMPORT" );
VoidTranslator         prasterTranslatorInstance ( "PRASTER" );
VoidTranslator         prloopTranslatorInstance  ( "PRASTERLOOP" );
VoidTranslator         pvoidTranslatorInstance   ( "PVOID" );
VoidTranslator         table1TranslatorInstance  ( "TABLE_HISTOGRAM" );
VoidTranslator         table2TranslatorInstance  ( "TABLE_XY_POINTS" );
VoidTranslator         table3TranslatorInstance  ( "TABLE_GEO_POINTS" );
VoidTranslator         table4TranslatorInstance  ( "TABLE_XY_VECTORS" );
VoidTranslator         table5TranslatorInstance  ( "TABLE_GEO_VECTORS" );
VoidTranslator         table6TranslatorInstance  ( "TABLE_XY_BINNING" );
VoidTranslator         table7TranslatorInstance  ( "TABLE_GEO_BINNING" );
VoidTranslator         table8TranslatorInstance  ( "TABLE_XY_BOXES" );
VoidTranslator         table9TranslatorInstance  ( "TABLE_GEO_BOXES" );
VoidTranslator         table10TranslatorInstance ( "TABLE_HOR_BAR" );
VoidTranslator         table11TranslatorInstance ( "TABLE_VER_BAR" );
WindTranslator         windTranslatorInstance    ( "PWIND" );

#if 0
//GeoLayersTranslator    geolTranslatorInstance    ( "GEOLAYERS" );
ImageTranslator   imageTranslatorInstance ( "IMAGE"  );
FrameTranslator   frameTranslatorInstance ( "FRAME"  );
CurveTranslator   curvTranslatorInstance  ( "CURVE"  );
PolarTranslator   polarrTranslatorInstance( "POLAR_FIELD" );
MatrixTranslator  matrixTranslatorInstance( "MATRIX" );

PImageTranslator  pimageTranslatorInstance( "PMIMAGE" );
DummyDataTranslator dummyDataTranslatorInstance("DUMMYDATA");
NoteTranslator      noteTranslatorInstance("NOTE");
StringTranslator   stringTranslatorInstance("STRING");

ImportTranslator  psFileTranslatorInstance("PSFILE");
ImportTranslator  pngTranslatorInstance("PNG");
ImportTranslator  jpegTranslatorInstance("JPEG");
#endif

//====== Helper function for generating subpage size ==
static void generateSubpageSize(const MvRequest&, MvRequest&, double,double,bool expanded = false);

//========================================================
//
// Default Object ( required by the "prototype" template )
MagicsTranslator&
MagicsTranslatorTraits::DefaultObject()
{
	return GribTranslator::Instance();
}

//==================================================
//
//  Methods for the MagicsTranslator class
//
MagicsTranslator::~MagicsTranslator()
{
	// Empty
}

//=================================================
// GRIB Translator

MagicsTranslator& 
GribTranslator::Instance()
{
	return gribTranslatorInstance;
}

MvRequest
GribTranslator::Execute ( const MvRequest& dataUnitRequest, 
			  const MvRequest& dataInfoRequest,
			  MvRequest& visdefRequest )
{
	MvRequest magicsRequest( "PGRIB" );
	MvRequest metviewRequest = dataUnitRequest;

	// Set the common parameters of all kind of grib data

	// Set the input file name and Magics internal parameters
	const char* dataPath = metviewRequest("PATH");
	magicsRequest ( "GRIB_INPUT_TYPE"        ) = "FILE";
	magicsRequest ( "GRIB_INPUT_FILE_NAME"   ) = dataPath;
	magicsRequest ( "GRIB_FILE_ADDRESS_MODE" ) = "BYTE_OFFSET";
	magicsRequest ( "GRIB_SPECIFICATION"     ) = "OFF";
	CopySomeParameters( metviewRequest,magicsRequest,"$METVIEW_" );

	// Set GRIB_* parameters selected by the user.
	// As two fields can be drawn within a 'Newpage' Magics
	// command, the request must be expanded in order to set/reset
	// all the parameters. This will avoid the use of parameters
	// that were set for the previous plot but not set for this one.
	MvRequest expandRequest = ObjectList::ExpandRequest (visdefRequest,EXPAND_DEFAULTS);
	CopySomeParameters( expandRequest,magicsRequest,"GRIB_" );
	CopySomeParameters( expandRequest,magicsRequest,"INPUT_FIELD_SUPPRESS" );

	// Set the specific parameters for each kind of data
	// If it is a single field:
	//	GRIB_DIMENSION = 1, GRIB_POSITION = pos
	//	(or) GRIB_DIMENSION = 2, GRIB_POSITION_1 = pos1, GRIB_POSITION_2 = pos2
	// If it is all fields:
	//	GRIB_DIMENSION = 1
	//	(or)GRIB_ DIMENSION = 2 
	// If it is subset of fields:
	//	GRIB_DIMENSION = 1, GRIB_POSITION = list
	//	(or) GRIB_DIMENSION = 2, GRIB_POSITION_1 = list1, GRIB_POSITION_2 = list2

	// Metview's "GRIB_SCALING_OF_RETRIEVED_FIELDS" is equivalent to Magics' "GRIB_AUTOMATIC_SCALING"
	// Metview's "GRIB_SCALING_OF_DERIVED_FIELDS"   is equivalent to Magics' "GRIB_AUTOMATIC_DERIVED_SCALING"

	const char *scalingOfRetrievedFields = (const char*) expandRequest("GRIB_SCALING_OF_RETRIEVED_FIELDS");
	if (scalingOfRetrievedFields)
		magicsRequest("GRIB_AUTOMATIC_SCALING") = scalingOfRetrievedFields;

	const char *scalingOfDerivedFields = (const char*) expandRequest("GRIB_SCALING_OF_DERIVED_FIELDS");
	if (scalingOfDerivedFields)
		magicsRequest("GRIB_AUTOMATIC_DERIVED_SCALING") = scalingOfDerivedFields;

     //ir const char *derived = dataInfoRequest("DERIVED");
     //ir //if (derived
     //ir magicsRequest ( "GRIB_AUTOMATIC_SCALING") = "OFF" ;

     // Copy GRIB_* parameters
     CopySomeParameters(dataInfoRequest,magicsRequest,"GRIB_");

     // This is an image field data
     if ( strcmp( (const char*)dataInfoRequest.getVerb(),"IMAGE") == 0)
     {
          magicsRequest ( "GRIB_AUTOMATIC_SCALING" ) = "OFF" ;
          magicsRequest ( "SUBPAGE_MAP_PROJECTION" ) = "INPUT_IMAGE";
     }

	//magicsRequest ( "GRIB_AUTOMATIC_SCALING") = "OFF";

	// Copy hidden parameters
	magicsRequest ("_NAME" ) = (const char*)dataUnitRequest ("_NAME");
	magicsRequest ("_CLASS") = (const char*)dataUnitRequest ("_CLASS");
	magicsRequest ("_ID"   ) = (const char*)dataUnitRequest ("_ID");

	return magicsRequest;
} 

#if 0
//=============================================================
//
// Image Translator
//
MvRequest
ImageTranslator::Execute ( const MvRequest& dataUnitRequest, 
			   long dataUnitIndex,
			   const MvRequest& dataInfoRequest,
			   MvRequest& )
{

	MvRequest magicsRequest;
	const char *dataIdent = dataInfoRequest ("IDENT");

	require ( dataIdent != NULL );	// this is  an IMAGE

	magicsRequest.setVerb("PGRIB");

	// Set the input file name and field position
	const char* dataPath = dataUnitRequest("PATH");
	
	magicsRequest ( "GRIB_INPUT_TYPE"        ) = "FILE";
	magicsRequest ( "GRIB_INPUT_FILE_NAME"   ) = dataPath;
	magicsRequest ( "GRIB_POSITION"    ) = (int) dataUnitIndex;
	magicsRequest ( "GRIB_FILE_ADDRESS_MODE" ) = "BYTE_OFFSET";
	magicsRequest ( "GRIB_SPECIFICATION"     ) = "OFF";
	magicsRequest ( "GRIB_SCALING"           ) = "OFF" ;

	magicsRequest ( "SUBPAGE_MAP_PROJECTION" ) = "INPUT_IMAGE";

//magicsRequest ( "MAP_COASTLINE_LAND_SHADE_COLOUR" ) = "BEIGE";
//magicsRequest ( "MAP_COASTLINE_SEA_SHADE_COLOUR" ) = "MAGENTA";

return magicsRequest;
} 
#endif

//==============================================================
//
// Bufr Translator
//
MvRequest
BufrTranslator::Execute ( const MvRequest& dataUnitRequest, 
			  const MvRequest& dataInfoRequest,
			  MvRequest& )
{
	MvRequest magicsRequest;	
	const char* dataPath;

	// The real path is normally in the dataInfoRequest,
	// where the name of the generated file for this
	// interval is. If no grouping/matching is done,
	// use the original datarequest.
	dataPath = dataInfoRequest("PATH");
	if ( !dataPath )
		dataPath = dataUnitRequest("PATH");

	magicsRequest.setVerb("BUFR");
	magicsRequest("OBS_INPUT_TYPE"      ) = "FILE";
	magicsRequest("OBS_INPUT_FORMAT"    ) = "BUFR";
	magicsRequest("OBS_INPUT_FILE_NAME" ) = dataPath;
	magicsRequest("OBS_INPUT_FILE_UNIT" ) = 23;
	magicsRequest("TEXT_LINE_1")=  "Observations";
	magicsRequest("TEXT_LINE_COUNT")=  1;

  return magicsRequest;
}

//===================================================
// Obs Translator, for setting how the obs. should be plotted.
//
MvRequest
ObsTranslator::Execute ( const MvRequest& req )
{
  return req;
}

//=================================================================
//
// Coast Translator
//
MvRequest
CoastTranslator::Execute ( const MvRequest& mapviewRequest )
{
//MvRequest magicsRequest = mapviewRequest.getSubrequest ("COASTLINES");

   // Replace verb
   const char* verb = mapviewRequest.getVerb();
   MvRequest magicsRequest = mapviewRequest;
   magicsRequest.setVerb("MCOAST");

   // Keep backwards compatibility
   if  ( strcmp ( verb, "PCOAST" ) == 0 )
      return magicsRequest;

#if 0
   // Since Coast and Grid are set to ON by default, switch them to OFF according
   // to the relevant Geo action. This will avoid them to be drawn several times.

   //IMPORTANTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTT!!!!!!!!!!!!!
   // SWITCH BACK TO OFF BEFORE SUBMITING.
   // IT IS ON BECAUSE MAGICS DOES NOT PLOT IT CORRECTTLY
   if ( strcmp ( verb, "GEO_COASTLINE" ) != 0 )
      magicsRequest("MAP_COASTLINE") = "on"; //"OFF";

   if ( strcmp ( verb, "GEO_GRID" ) != 0 )
   {
      magicsRequest("MAP_GRID")      = "OFF";
      magicsRequest("MAP_LABEL")     = "OFF";
   }
#endif
   return magicsRequest;
}

//=================================================================
//
// --- METHOD : Contour Translator::Execute
//
// --- PURPOSE: Translate a metview Contour request into
//              a MAGICS contour request
// 
MvRequest
ContTranslator::Execute ( const MvRequest& visdefRequest )
{
	// Retrieve the visdef request
	MvRequest metviewRequest = visdefRequest;

	// Check LEGEND_ENTRY subrequest 
	MvRequest magicsRequest ("PCONT");
	MvRequest tmpRequest ("PCONT");
	MvRequest legRequest = metviewRequest.getSubrequest("LEGEND_ENTRY");
	if ( legRequest )
	{
		// Strip out the "GRIB_" and the "INPUT_FIELD_SUPPRESS" parts
		MvRequest tmp1Request ("PCONT");
		CopyAndRemoveParameters ( metviewRequest, tmpRequest, "INPUT_FIELD_SUPPRESS" );
		CopyAndRemoveParameters ( tmpRequest, tmp1Request, "GRIB_" );
		// Translate the LEGEND_ENTRY subrequest
		CopyAndRemoveParameters(tmp1Request,magicsRequest,"LEGEND_ENTRY");
		CopySomeParameters(legRequest,magicsRequest,"LEGEND");
	}
	else
	{
		// Strip out the "GRIB_" the "INPUT_FIELD_SUPPRESS" parts
		CopyAndRemoveParameters ( metviewRequest, tmpRequest, "INPUT_FIELD_SUPPRESS" );
		CopyAndRemoveParameters ( tmpRequest, magicsRequest, "GRIB_" );
	}

	return magicsRequest;
}

#if 0
//=================================================================
//
// Frame Translator
//
MvRequest
FrameTranslator::Execute ( const MvRequest& metviewRequest )
{
	MvRequest magicsRequest ( "FRAME" );
	magicsRequest ("SUPER_PAGE_FRAME") = "OFF";

	MvRequest mvRequest = metviewRequest; // Get rid of the const

	CopySomeParameters ( mvRequest, magicsRequest,
			     "SUBPAGE_FRAME");	
	CopySomeParameters ( mvRequest, magicsRequest,
			     "PAGE_FRAME");	
return magicsRequest;
}
#endif

MvRequest
PageTranslator::Execute ( const MvRequest& viewRequest )
{
	// Set layout
   MvRequest metviewRequest = viewRequest;
	MvRequest magicsRequest("PAGE");
	double width  = metviewRequest ( "_WIDTH"  );
	double height = metviewRequest ( "_HEIGHT" );
	magicsRequest ( "PAGE_X_LENGTH" ) = width; 
	magicsRequest ( "PAGE_Y_LENGTH" ) = height; 

	magicsRequest ( "PAGE_X_POSITION" ) =  (double)metviewRequest("_X_ORIGIN");
	magicsRequest ( "PAGE_Y_POSITION" ) =  (double)metviewRequest("_Y_ORIGIN");

	// Set/Remove page ID line. By default, Metview does not use this parameter
	magicsRequest ( "PAGE_ID_LINE") = "OFF";
	CopySomeParameters(metviewRequest, magicsRequest, "PAGE_FRAME");
	CopySomeParameters(metviewRequest, magicsRequest, "SUBPAGE_FRAME");
	CopySomeParameters(metviewRequest, magicsRequest, "SUBPAGE_METADATA");
	if ( (const char*)metviewRequest( "SUBPAGE_VERTICAL_AXIS_WIDTH" ) )
		magicsRequest( "SUBPAGE_VERTICAL_AXIS_WIDTH" ) = (double)metviewRequest( "SUBPAGE_VERTICAL_AXIS_WIDTH" );
	if ( (const char*)metviewRequest( "SUBPAGE_HORIZONTAL_AXIS_HEIGHT" ) )
		magicsRequest( "SUBPAGE_HORIZONTAL_AXIS_HEIGHT" ) = (double)metviewRequest( "SUBPAGE_HORIZONTAL_AXIS_HEIGHT" );

    // Set zoom definition
    if ( (const char*)metviewRequest( "_ZOOM_DEFINITION" ) )
    {
        magicsRequest( "SUBPAGE_MAP_JSON_DEFINITION" ) = (const char*)metviewRequest( "_ZOOM_DEFINITION" );
        metviewRequest.unsetParam("_ZOOM_DEFINITION");
    }

	// Set subpage sizes
	generateSubpageSize(metviewRequest,magicsRequest,width,height);

	// Include the MAP information
	// Test if the projection exists - if not, expand the request to
	// search for the default projection -- if not, set projection to 
	// NONE
	const char* projection = metviewRequest( "MAP_PROJECTION" );
	MvRequest tmpReq;
	if ( !projection )
	{
		tmpReq = ObjectList::ExpandRequest(metviewRequest,EXPAND_DEFAULTS);
		projection = tmpReq( "MAP_PROJECTION" );
	}

	// Temporary solution to draw cartesian view
	if (projection != 0 && (strcmp(projection,"cartesian") == 0 || strcmp(projection,"tephigram") == 0 ||  strcmp(projection,"taylor") == 0) )
	{
		// Add a new request related to the VIEW
		magicsRequest("SUBPAGE_MAP_PROJECTION") = "NEXT";
		RemoveParameters(metviewRequest,"PAGE");
		RemoveParameters(metviewRequest,"SUBPAGE");
		magicsRequest = magicsRequest + metviewRequest;
	}
	else if ( projection )
	{
      // Translate AREA parameter
      const char* area = metviewRequest( "AREA" );
      if ( area != 0 )
      {
         magicsRequest( "SUBPAGE_LOWER_LEFT_LATITUDE" ) = (double)metviewRequest ( "AREA", 0 );
         magicsRequest( "SUBPAGE_LOWER_LEFT_LONGITUDE") = (double)metviewRequest ( "AREA", 1 );
         magicsRequest( "SUBPAGE_UPPER_RIGHT_LATITUDE") = (double)metviewRequest ( "AREA", 2 );
         magicsRequest( "SUBPAGE_UPPER_RIGHT_LONGITUDE")= (double)metviewRequest ( "AREA", 3 );
      }

      // Translate MAP to SUBPAGE_MAP parameters
      CopySomeParameters ( metviewRequest,magicsRequest,"MAP_","SUBPAGE_MAP_" );
	}
#if 0 //SatelliteView is not available at the moment
	else if ( metviewRequest.getVerb() == Cached ( "SATELLITEVIEW" ) )
	{
		magicsRequest ( "SUBPAGE_MAP_PROJECTION" ) = "SATELLITE";

//		MvRequest tmpRequest = metviewRequest;

		if ( metviewRequest ( "IMAGE_SUBAREA_SELECTION" ) == ON )
		{
			magicsRequest ( "IMAGE_SUBAREA_SELECTION" ) = ON;
			CopySomeParameters( metviewRequest, magicsRequest, "SUBPAGE_MAP_CENTRE" );
			CopySomeParameters( metviewRequest, magicsRequest, "SUBPAGE_MAP_CORNER" );
		}

		CopySomeParameters( metviewRequest, magicsRequest, "INPUT_IMAGE" );
		CopySomeParameters( metviewRequest, magicsRequest, "SUBPAGE_MAP_INITIAL" );
		CopySomeParameters( metviewRequest, magicsRequest, "SUBPAGE_MAP_SUB_SAT" );
		CopySomeParameters( metviewRequest, magicsRequest, "SUBPAGE_MAP_X_EARTH_DIAMETER" );
		CopySomeParameters( metviewRequest, magicsRequest, "SUBPAGE_MAP_Y_EARTH_DIAMETER" );
		CopySomeParameters( metviewRequest, magicsRequest, "SUBPAGE_MAP_GRID_ORIENTATION" );
		CopySomeParameters( metviewRequest, magicsRequest, "SUBPAGE_MAP_CAMERA_ALTITUDE" );
	}
#endif
	else
	{
		// In case of a non MapView this parameter should be set
		magicsRequest ( "SUBPAGE_MAP_PROJECTION" ) = "NONE";
	}

	// Add overlay information
//	if ( (const char*)metviewRequest("MAP_OVERLAY_CONTROL") )
//		magicsRequest("SUBPAGE_OVERLAY_CONTROL") = (const char*)metviewRequest("MAP_OVERLAY_CONTROL");

   return magicsRequest;
}

//=================================================================
//
// Text Translator
//
MvRequest
TextTranslator::Execute ( const MvRequest& textsRequest )
{
   MvRequest magicsRequest = textsRequest;

   // Set text box coordinates explicitely. If these coordinates are not defined,
   // Magics will compute them by using an internal scheme. To guarantee that Magics
   // will use the Metview default values, these coordinates need to be added to the
   // request.
   if ( (const char*)magicsRequest("TEXT_MODE") && strcmp((const char*)magicsRequest("TEXT_MODE"),"POSITIONAL") == 0 )
   {
      MvRequest expRequest = ObjectList::ExpandRequest ( magicsRequest,EXPAND_DEFAULTS );
      magicsRequest("TEXT_BOX_X_POSITION") = expRequest("TEXT_BOX_X_POSITION");
      magicsRequest("TEXT_BOX_Y_POSITION") = expRequest("TEXT_BOX_Y_POSITION");
      magicsRequest("TEXT_BOX_X_LENGTH")   = expRequest("TEXT_BOX_X_LENGTH");
      magicsRequest("TEXT_BOX_Y_LENGTH")   = expRequest("TEXT_BOX_Y_LENGTH");
   }

   return magicsRequest;
}

MvRequest
AxisTranslator::Execute ( const MvRequest& metviewRequest )
{
	MvRequest tmpRequest = metviewRequest;
	tmpRequest.rewind();

	MvRequest magicsRequest;

	while ( tmpRequest )
	{
		if ( tmpRequest.getVerb() == Cached ( "PAXIS") )
			magicsRequest = magicsRequest + tmpRequest.justOneRequest();
		tmpRequest.advance();
	}

	return magicsRequest;
}

#if 0
//=================================================================
//
// Curve Translator
// 
MvRequest
CurveTranslator::Execute ( const MvRequest& metviewRequest )
{
	MvRequest magicsRequest = metviewRequest;

	ensure ( magicsRequest.getVerb() == Cached ( "CURVE" ) ); 

return magicsRequest;
}

MvRequest
CurveTranslator::Execute ( const MvRequest& dataUnitRequest, 
			   long, const MvRequest&, MvRequest&)
{
	MvRequest magicsRequest ("CURVE" );
	MvRequest duRequest = dataUnitRequest; // to be allowed to advance.

	const char* graphType = duRequest( "GRAPH_TYPE" );
	const char *xAxisType = duRequest("_XAXIS_TYPE");
	bool xIsDate;
	if ( xAxisType )
	       	xIsDate = ( strcmp(xAxisType,"DATE") == 0 );
	else
		xIsDate = false;

	const char *yAxisType = duRequest("_YAXIS_TYPE");
	bool yIsDate;
	if ( yAxisType )
	       	yIsDate = ( strcmp(yAxisType,"DATE") == 0 );
	else
		yIsDate = false;

        Cached dataVectorYName, dataVectorXName;
	Cached dataVectorY2Name,dataVectorX2Name;

	if ( graphType && graphType == Cached( "BAR" ) )
	{
	  if ( yIsDate )
	    {
	      
	      dataVectorYName = Cached( "GRAPH_BAR_DATE_Y_UPPER_VALUES" );
	      dataVectorY2Name = Cached( "GRAPH_BAR_DATE_Y_LOWER_VALUES" );
	    }
	  else
	    {
	      dataVectorYName = Cached( "GRAPH_BAR_Y_UPPER_VALUES" );
	      dataVectorY2Name = Cached( "GRAPH_BAR_Y_LOWER_VALUES" );
	    }
	  if ( xIsDate )
	    dataVectorXName = Cached( "GRAPH_BAR_DATE_X_VALUES" );
	  else
	    dataVectorXName = Cached( "GRAPH_BAR_X_VALUES" );
	}
	else if ( graphType && graphType == Cached( "AREA" ) )
	  {
	    if ( xIsDate )
	      {
		dataVectorX2Name = Cached( "GRAPH_CURVE2_DATE_X_VALUES" );
		dataVectorXName = Cached( "GRAPH_CURVE_DATE_X_VALUES" );
	      }
	    else
	      {
		dataVectorX2Name = Cached( "GRAPH_CURVE2_X_VALUES" );
		dataVectorXName = Cached( "GRAPH_CURVE_X_VALUES" );
	      }
	    if ( yIsDate )
	      {

		dataVectorY2Name = Cached( "GRAPH_CURVE2_DATE_Y_VALUES" );
		dataVectorYName = Cached( "GRAPH_CURVE_DATE_Y_VALUES" );
	      }
	    else
	      {
		dataVectorY2Name = Cached( "GRAPH_CURVE2_Y_VALUES" );
		dataVectorYName = Cached( "GRAPH_CURVE_Y_VALUES" );
	      }
	  }

	else
	  {
	    if ( yIsDate )
	      dataVectorYName = Cached( "GRAPH_CURVE_DATE_Y_VALUES" );
	    else
	      dataVectorYName = Cached( "GRAPH_CURVE_Y_VALUES" );
	    if ( xIsDate )
	      dataVectorXName = Cached( "GRAPH_CURVE_DATE_X_VALUES" );
	    else
	      dataVectorXName = Cached( "GRAPH_CURVE_X_VALUES" );
	  }
	
	const char *oneValue;
	for ( int i=0; i<duRequest.countValues( "DATA_X_VALUES" ); i++ )
	{
		oneValue = duRequest ("DATA_X_VALUES", i ) ;
		magicsRequest.addValue( dataVectorXName, oneValue );
		oneValue = duRequest ("DATA_Y_VALUES", i );
		magicsRequest.addValue( dataVectorYName, oneValue );
	}
	
	if ( dataVectorX2Name )
	  {
	    for ( int i=0; i<duRequest.countValues( "DATA_X2_VALUES" ); i++ )
	      {
		oneValue = duRequest ("DATA_X2_VALUES", i );
		magicsRequest.addValue( dataVectorX2Name, oneValue );
	      }
	  }
	if ( dataVectorY2Name )
	  {
	    for ( int i=0; i<duRequest.countValues( "DATA_Y2_VALUES" ); i++ )
	      {
		oneValue = duRequest ("DATA_Y2_VALUES", i );
		magicsRequest.addValue( dataVectorY2Name, oneValue );
	      }
	  }
	CopySomeParameters( duRequest, magicsRequest, "TEXT" );

	// Check if the curve carries some axis or ptephig along, if so copy them.
	duRequest.advance();
	while ( duRequest )
	  {
	    MvRequest  request =  duRequest.justOneRequest();
	    Cached verb        =  request.getVerb();
	    if ( verb == PAXIS || !strcmp(verb,"PTEPHIG") )
		 magicsRequest = request + magicsRequest;

	    duRequest.advance();
	  }

return magicsRequest;
}
#endif

//===================================================
// Graph Translator
//
MvRequest 
GraphTranslator::Execute ( const MvRequest& metviewRequest )
{
    MvRequest tmpRequest = metviewRequest;
    ensure ( tmpRequest.getVerb() == Cached ( "PGRAPH" ) );

    MvRequest magicsRequest( "PGRAPH" );
    CopyAndRemoveParameters(tmpRequest,magicsRequest,"LEGEND_ENTRY");

    MvRequest legRequest = tmpRequest.getSubrequest("LEGEND_ENTRY");
    if ( legRequest ) 
        CopySomeParameters(legRequest,magicsRequest,"LEGEND");

    return magicsRequest;
}

//=================================================================
//
// Grib_Vectors Translator
//
MvRequest
GribVectorsTranslator::Execute ( const MvRequest& dataUnitRequest1,
                                 const MvRequest&, MvRequest& )
{
   // Copy input request (this is because functions iterInit and iterGetNextValue
   // do not accept a const variable)
   MvRequest dataUnitRequest = dataUnitRequest1;

   // Get GRIB info subrequest
   MvRequest gribReq = dataUnitRequest("GRIB");

   // Create Magics request from the input requests
   MvRequest magicsRequest("PGRIB");
   magicsRequest ("GRIB_INPUT_TYPE") = "FILE";
   magicsRequest ("GRIB_INPUT_FILE_NAME") = (const char*)gribReq("PATH");
   magicsRequest ("GRIB_FILE_ADDRESS_MODE") = "BYTE_OFFSET";
   magicsRequest ("GRIB_WIND_MODE") = (const char*)dataUnitRequest("GRIB_WIND_MODE");

   // Get grib dimension list
   int cnt = dataUnitRequest.iterInit( "GRIB_DIMENSION" );
   const char* str;
   int i;
   for( i = 0; i < cnt; ++i )
   {
      dataUnitRequest.iterGetNextValue( str );
      magicsRequest.addValue ("GRIB_DIMENSION",str);
   }

   // Get grib positions list
   cnt = gribReq.iterInit("OFFSET");
   for ( i = 0; i < cnt; ++i )
   {
       gribReq.iterGetNextValue( str );
       magicsRequest.addValue("GRIB_POSITION",str);
   }

   return magicsRequest;
}

#if 0
//=================================================================
//
// Polar_field Translator
// 


MvRequest
PolarTranslator::Execute ( const MvRequest& metviewRequest )
{
	MvRequest magicsRequest = metviewRequest;
	magicsRequest.setVerb( "PGRIB" );

return magicsRequest;
}

MvRequest
PolarTranslator::Execute ( const MvRequest& dataUnitRequest , 
			   long, const MvRequest&, MvRequest& )
{
  MvRequest int_request = dataUnitRequest("INTENSITY");
  MvRequest dir_request = dataUnitRequest("DIRECTION");

  MvRequest magicsRequest( "PGRIB" );
  magicsRequest( "GRIB_INPUT_TYPE"        ) = "FILE";
  magicsRequest( "GRIB_INPUT_FILE_NAME"   ) = int_request("PATH");
  magicsRequest( "GRIB_INPUT_FILE_NAME_2" ) = dir_request("PATH");
  magicsRequest( "GRIB_INPUT_FILE_NUMBER" ) = "TWO";
  magicsRequest( "GRIB_FILE_ADDRESS_MODE" ) = "BYTE_OFFSET";
  magicsRequest( "GRIB_DIMENSION"   ) = 2;
  magicsRequest( "GRIB_POSITION_1"   ) = (long)int_request("OFFSET");
  magicsRequest( "GRIB_POSITION_2"   ) = (long)dir_request("OFFSET");
  magicsRequest( "GRIB_SPECIFICATION"     ) = "OFF";
  magicsRequest( "GRIB_SCALING"           ) = "OFF";
  magicsRequest("GRIB_WIND_MODE"          ) = "SD";

return magicsRequest;
}
#endif

//===================================================
// Wind Translator
//
MvRequest 
WindTranslator::Execute ( const MvRequest& metviewRequest )
{
 	MvRequest magicsRequest = metviewRequest;

	ensure ( magicsRequest.getVerb() == Cached ( "PWIND" ) ); 

	return magicsRequest;
}

#if 0
// 
// PImage Translator
//
MvRequest
PImageTranslator::Execute ( const MvRequest& metviewRequest )
{
	MvRequest magicsRequest = metviewRequest;

	ensure ( magicsRequest.getVerb() == Cached ( "PMIMAGE" ) );
	magicsRequest.setVerb( "PIMAGE" );


	/* The parameter IMAGE_COLOUR_TABLE_TYPE was introduced
		solely for the Metview user interface, and is not
		a valid MAGICS parameter. Therefore, we remove it here
		before passing the request on to MAGICS. */

	magicsRequest.unsetParam ( "IMAGE_COLOUR_TABLE_TYPE" );

return magicsRequest;
}
#endif

MvRequest
SymbolTranslator::Execute ( const MvRequest& visdefRequest )
{
	ensure ( visdefRequest.getVerb() == Cached ( "PSYMB" ) );

	MvRequest metviewRequest = visdefRequest;
	
        // Copy request
 	MvRequest magicsRequest( "PSYMB" );
	CopyAndRemoveParameters(metviewRequest,magicsRequest,"LEGEND_ENTRY");

	// Check LEGEND_ENTRY subrequest
	MvRequest legRequest = metviewRequest.getSubrequest("LEGEND_ENTRY");
	if ( legRequest ) CopySomeParameters(legRequest,magicsRequest,"LEGEND");

        return magicsRequest;
}

MvRequest
GeopointsTranslator::Execute ( const MvRequest& dataUnitRequest, 
			       const MvRequest&, MvRequest& psymb)
{
	MvRequest geo("GEOPOINTS");

	MvRequest record = dataUnitRequest.getSubrequest("RECORD");

	const char *symbType =  psymb("SYMBOL_TYPE");
	const char *tableMode = psymb("SYMBOL_TABLE_MODE");
	string type = "NUMBER";

	// If SYMBOL_TYPE is set, use the value. If SYMBOL_TABLE_MODE is also
	// set, type needs to be number, whatever SYMBOL_TYPE is set to.
	if (  symbType )
		type = symbType;
	if ( tableMode && strcmp(tableMode,"ON") == 0 )
		type = "NUMBER";

	record("SYMBOL_TYPE") = type.c_str();

	geo.setValue("RECORD",record);

	// Copy hidden parameters
	geo("_NAME" ) = (const char*)dataUnitRequest ("_NAME");
	geo("_CLASS") = (const char*)dataUnitRequest ("_CLASS");
	geo("_ID"   ) = (const char*)dataUnitRequest ("_ID");

	return geo;
}

#if 0
MvRequest
MatrixTranslator::Execute ( const MvRequest& dataUnitRequest, 
			    long, const MvRequest&, MvRequest& )
{
  return dataUnitRequest;
}
#endif

MvRequest
NetCDFTranslator::Execute ( const MvRequest& dataUnitRequest,
			    const MvRequest&, MvRequest& )
{
   return dataUnitRequest;
}

MvRequest
VoidTranslator::Execute ( const MvRequest& request,
			  const MvRequest&, MvRequest& )
{
   return request;
}
MvRequest
VoidTranslator::Execute ( const MvRequest& request)
{
   return request;
}

#if 0
MvRequest
DummyDataTranslator::Execute ( const MvRequest& dataUnitRequest, 
			       long, const MvRequest&, MvRequest& )
{
  MvRequest magicsRequest;
  MvRequest duRequest = dataUnitRequest; // Original request is const

  // Skip first request, only a dummy
  duRequest.advance();

  // Copy any subsequent requests.
  while ( duRequest )
    {
      MvRequest oneRequest =  duRequest.justOneRequest();
      if ( ! magicsRequest )
	magicsRequest = oneRequest;
      else
	magicsRequest = magicsRequest + oneRequest;

      duRequest.advance(); 
    }

  if ( ! magicsRequest )
    return MvRequest("EMPTY");
  else
    return magicsRequest;
      
}

MvRequest NoteTranslator::Execute(const MvRequest& dataUnitRequest, 
				  long,
				  const MvRequest&,
			          MvRequest& )
{


  MvRequest out("PTEXT");

  const char *path = dataUnitRequest("PATH");

  int counter = 1;
  char buf[300],magicsName[30];
  if ( path )
    {
      ifstream in(path, ios::in);
      if ( ! in )
	return MvRequest("EMPTY");

      while ( in.getline(buf,299) && counter <= 10 )
	{
	  sprintf(magicsName,"TEXT_LINE_%d",counter);
	  out(magicsName) = buf;
	  counter++;
	}
      out("TEXT_LINE_COUNT") = counter -1;
      out("TEXT_BOX_X_POSITION") = (double)dataUnitRequest("_DROP_X");
      out("TEXT_BOX_Y_POSITION") = (double)dataUnitRequest("_DROP_Y");
      out("TEXT_JUSTIFICATION") = "LEFT";
      out("TEXT_ORIGIN") = "USER";
      out("TEXT_MODE")  = "POSITIONAL";
      in.close();

      MvRequest ptext = dataUnitRequest.getSubrequest("_PTEXT");
      
      CopyAndRemoveParameters(ptext,out,"TEXT_LINE");

      return out;
    }  
  return MvRequest("EMPTY");
}


MvRequest StringTranslator::Execute(const MvRequest& dataUnitRequest, 
				    long,const MvRequest&,MvRequest& )
{
  // Read value field, split into lines and add to TEXT_LINE_1 ... for one or 
  // more annotations requests.
  MvRequest out("PTEXT");
  
  const char *value = dataUnitRequest("VALUE");
  if ( ! value )
    return MvRequest("EMPTY");

  // For now just add it, with embedded newlines.
  out("TEXT_LINE_1") = value;
  out("TEXT_BOX_X_POSITION") = (double)dataUnitRequest("_DROP_X");
  out("TEXT_BOX_Y_POSITION") = (double)dataUnitRequest("_DROP_Y");
  out("TEXT_JUSTIFICATION") = "LEFT";
  out("TEXT_ORIGIN") = "USER";
  out("TEXT_MODE")  = "POSITIONAL";
  return out;
}
#endif

MvRequest ImportTranslator::Execute( const MvRequest& req )
{
   // The file is an embedded icon. Extracts the path.
   MvRequest fileReq = req.getSubrequest("IMPORT_FILE");
   if ( fileReq )
   {
      MvRequest out = req;
      out.unsetParam("IMPORT_FILE");
      out("IMPORT_FILE_NAME") = fileReq("PATH");
      return out;
   }

   // Nothing to be translated if file name is given.
   if ( (const char*)req("IMPORT_FILE_NAME") )
      return req;

   // Data file not provided
   return MvRequest("EMPTY");
}

///////////////// Utility function, not normal translator
void generateSubpageSize(const MvRequest& metviewRequest, MvRequest &magicsRequest, double width, double height, bool expanded )
{
   // Copy input request
   MvRequest tmpRequest = metviewRequest.justOneRequest();

   // Check if the request needs to be expanded
   if ( !expanded )
      tmpRequest = ObjectList::ExpandRequest ( tmpRequest,EXPAND_DEFAULTS );

   magicsRequest ( "SUBPAGE_X_LENGTH" ) = width * ( (double)tmpRequest("SUBPAGE_X_LENGTH")/100 );
   magicsRequest ( "SUBPAGE_Y_LENGTH" ) = height * ( (double)tmpRequest("SUBPAGE_Y_LENGTH")/ 100 );
   magicsRequest ( "SUBPAGE_X_POSITION" ) = width * ( (double)tmpRequest("SUBPAGE_X_POSITION") /100 );
   magicsRequest ( "SUBPAGE_Y_POSITION" ) = height * ( (double)tmpRequest("SUBPAGE_Y_POSITION") / 100 );

   return;
}
