"""Tests for match script"""

import unittest
import argparse
from io import StringIO

try:
    from unittest.mock import patch, Mock, MagicMock, mock_open, call
except ImportError:
    from mock import patch, Mock, MagicMock, mock_open, call

import scripts.match
import oa.errors

EXAMPLE_EMAIL = ()

class TestRevokeReport(unittest.TestCase):
    raw_messages = [u"Message Stub 1",
                    u"Message Stub 2",
                    u"Message Stub 3",]

    def setUp(self):
        ruleset = patch("scripts.match.oa.rules."
                        "parser.parse_pad_rules").start()
        self.ctxt = ruleset.return_value.get_ruleset.return_value.ctxt
        patch("scripts.match.MessageList").start()
        self.mock_get_configs = patch("oa.config.get_config_files").start()
        msg_class = patch("scripts.match.oa.message.Message").start()
        self.messages = [msg_class(self.ctxt, x) for x in self.raw_messages]

    def tearDown(self):
        patch.stopall()
        super(TestRevokeReport, self).tearDown()

    def test_report(self):
        options = scripts.match.parse_arguments(["--report",
                                                 "--siteconfigpath", ".",
                                                 "--configpath", "."])
        options.messages = [[StringIO(x) for x in self.raw_messages]]
        patch("scripts.match.parse_arguments",
              return_value=options).start()
        scripts.match.main()
        calls = [call(x) for x in self.messages]
        self.ctxt.hook_report.assert_has_calls(calls)
        self.ctxt.hook_revoke.assert_not_called()

    def test_report_user_prefs(self):
        options = scripts.match.parse_arguments(["--report",
                                                 "--siteconfigpath", ".",
                                                 "--configpath", ".",
                                                 "--prefspath", "/prefs/path"])
        options.messages = [[StringIO(x) for x in self.raw_messages]]
        patch("scripts.match.parse_arguments",
              return_value=options).start()
        self.mock_get_configs.return_value = ["1.pre", "2.cf"]
        scripts.match.main()
        self.assertEqual(self.mock_get_configs.call_args,
                         call('.', '.', '/prefs/path'))
        calls = [call(x) for x in self.messages]
        self.ctxt.hook_report.assert_has_calls(calls)
        self.ctxt.hook_revoke.assert_not_called()

    def test_revoke(self):
        options = scripts.match.parse_arguments(["--revoke",
                                                 "--siteconfigpath", ".",
                                                 "--configpath", ".", "-D"])
        options.messages = [[StringIO(x) for x in self.raw_messages]]
        patch("scripts.match.parse_arguments",
              return_value=options).start()
        scripts.match.main()
        calls = [call(x) for x in self.messages]
        self.ctxt.hook_revoke.assert_has_calls(calls)
        self.ctxt.hook_report.assert_not_called()

    def test_max_recursion_exception(self):
        options = scripts.match.parse_arguments(["--revoke",
                                                 "--siteconfigpath", ".",
                                                 "--configpath", "."])
        options.messages = [[StringIO(x) for x in self.raw_messages]]
        patch("scripts.match.parse_arguments",
              return_value=options).start()
        self.mock_parse = patch(
            "oa.rules.parser.parse_pad_rules",
            side_effect=oa.errors.MaxRecursionDepthExceeded).start()
        with self.assertRaises(SystemExit):
            scripts.match.main()

    def test_parsing_error_exception(self):
        options = scripts.match.parse_arguments(["--revoke",
                                                 "--siteconfigpath", ".",
                                                 "--configpath", "."])
        options.messages = [[StringIO(x) for x in self.raw_messages]]
        patch("scripts.match.parse_arguments",
              return_value=options).start()
        self.mock_parse = patch(
            "oa.rules.parser.parse_pad_rules",
            side_effect=oa.errors.ParsingError).start()
        with self.assertRaises(SystemExit):
            scripts.match.main()

    def test_parse_no_configs(self):
        """If no configs were found the script shouldn't be run"""
        options = scripts.match.parse_arguments(["--revoke",
                                                 "--siteconfigpath", ".",
                                                 "--configpath", "."])
        options.messages = [[StringIO(x) for x in self.raw_messages]]
        patch("scripts.match.parse_arguments",
              return_value=options).start()
        self.mock_get_configs.return_value = []
        with self.assertRaises(SystemExit):
            scripts.match.main()

    def test_both(self):
        with self.assertRaises(SystemExit):
            with patch("sys.stderr"):
                scripts.match.parse_arguments(["--revoke", "--report", "-"])



def suite():
    """Gather all the tests from this package in a test suite."""
    test_suite = unittest.TestSuite()
    test_suite.addTest(unittest.makeSuite(TestRevokeReport, "test"))
    return test_suite

if __name__ == '__main__':
    unittest.main(defaultTest='suite')
