#!/usr/bin/env python

"""Tests of code for pyronoise"""

__author__ = "Jens Reeder"
__copyright__ = "Copyright 2011, The QIIME Project"
# remember to add yourself if you make changes
__credits__ = ["Jens Reeder"]
__license__ = "GPL"
__version__ = "1.9.1"
__maintainer__ = "Justin Kuczynski"
__email__ = "justinak@gmail.com"

from os import remove, mkdir, close
from shutil import rmtree
from tempfile import mkstemp, mkdtemp
from unittest import TestCase, main
from numpy.testing import assert_almost_equal

from skbio.util import remove_files
from bfillings.denoiser import Flowgram, FlowgramCollection, get_header_info
from qiime.denoise_wrapper import fast_denoiser, extract_cluster_size


class DenoiseWrapperTests(TestCase):

    """Tests of the abstract DenoiseWrapper OTU Picker class"""

    def setUp(self):
        self.file_content = file_content
        self.header = get_header_info(header.split('\n'))
        self.small_flowgram_collection = FlowgramCollection(
            flowgrams,
            header_info=self.header)
        self.large_flowgram_collection = FlowgramCollection(
            large_example,
            header_info=self.header)

        fd, self.sff_path = mkstemp(
            prefix='DenoiseWrapperTest_', suffix='.sff.txt')
        close(fd)
        self.large_flowgram_collection.writeToFile(self.sff_path)

        fd, self.sff_path2 = mkstemp(
            prefix='fastDenoiserTest_', suffix='.sff.txt')
        close(fd)
        fh = open(self.sff_path2, "w")
        fh.write(fasting_subset_sff)
        fh.close()

        fd, self.seq_path = mkstemp(
            prefix='fastDenoiserTest_', suffix='.fasta')
        close(fd)
        fh = open(self.seq_path, "w")
        fh.write(fasting_seqs_subset)
        fh.close()

        self.tmp_out_dir = mkdtemp(
            prefix='fastDenoiserTest_', suffix='/')

    def tearDown(self):
        """remove tmp files"""

        remove(self.sff_path)
        remove(self.sff_path2)
        remove(self.seq_path)
        rmtree(self.tmp_out_dir)

    def test_fast_denoiser(self):
        """The fast denoiser works as expected."""

        actual_centroids, actual_otu_map = fast_denoiser(
            [self.sff_path2], self.seq_path, self.tmp_out_dir,
            num_cpus=1, primer="YATGCTGCCTCCCGTAGGAGT", verbose=False)

        actual = list(actual_centroids)
        self.assertEqual(actual, expected_centroids)
        # centroids are sorted, so first one should be unique
        #(all others have cluster size 1, so relative ordering is not guaranteed
        self.assertEqual(actual[0], expected_centroids[0])
        self.assertEqual(actual_otu_map, expected_otu_map)

    def test_extract_cluster_size(self):
        """extract_cluster_size grabs the correct cluster size"""

        header1 = ">GCC6FHY01EQVIC | cluster size: 5"
        header2 = ">blabla"

        self.assertEqual(extract_cluster_size(header1), 5)
        self.assertEqual(extract_cluster_size(header2), 0)


### Test data ###
expected_centroids = [
    ('FLP3FBN01APRWO | cluster size: 3',
     'TTGGGCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTCTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGG'),
    ('FLP3FBN01ELBSX',
     'CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTTACCCTCTCAGGCCGGCTACGCATCATCGCCTTGGTGGGCCGTTACCTCACCAACTAGCTAATGCGCCGCAGGTCCATCCATGTTCACGCCTTGATGGGCGCTTTAATATACTGAGCATGCGCTCTGTATACCTATCCGGTTTTAGCTACCGTTTCCAGCAGTTATCCCGGACACATGGGCTAGG'),
    ('FLP3FBN01EG8AX',
     'TTGGACCGTGTCTCAGTTCCAATGTGGGGGCCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGGGCCGTTACCCCGCCAACAACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGAAGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCA'),
    ('FLP3FBN01EEWKD',
     'TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAGTCTCTTAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGACATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCTTGGG'),
    ('FLP3FBN01DEHK3',
     'CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGACTTGGTGAGCCGTTACCTCACCAACTATCTAATCAGACGCGAGCCCATCTTTCAGCGGATTGCTCCTTTGGTATTCCGGCGATGCCGCCAAAATCATTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGAAAGGCAGGTTGCTCACG'),
    ('FLP3FBN01DGFYQ',
     'TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGTCCATCTCACACCGATTAATCTTTTTCCAACCAGAGCATGCGCCCCTGTTGGCTTATGCGGTATTAGCGGTCGTTTCCAACTGTTATCCCCCTGTGTGAGGCAGGTTACCCACGCGTTACTCACCCGTCCG'),
    ('FLP3FBN01A55WZ',
     'TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGAGAAAATGCTCACTCATGATACCATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCAGGTTGGATACGCGTTACTCACCCGTGCGCCGG'),
    ('FLP3FBN01D7O1S',
     'TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGATAACCGAAATTCTTTAATAGTGAAACCATGCGGAAATACTATACTATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTTATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCATC'),
    ('FLP3FBN01CV5ZU',
     'TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAACCAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCTCTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCG'),
    ('FLP3FBN01DQ783',
     'TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG'),
    ('FLP3FBN01DDPFF',
     'TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCAGGTTGGGTACGCGTTACTC'),
    ('FLP3FBN01CPD70',
     'CTGGACCGTGTCTCAGTTCCAATGTGGGGGGCCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGGGCCGTTACCCCGCCAACAACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGAAGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCG'),
    ('FLP3FBN01BBAE6',
     'CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCGGCACCGGGATGCCCCGATGCCGCGTCACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTGGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCG'),
    ('FLP3FBN01AWYZD',
     'TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGACATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCTTGGG'),
    ('FLP3FBN01AM0P3',
     'CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGCCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATCAGACGCGGGCCCATCCCGTACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTTACTCACCCG'),
    ('FLP3FBN01BO4IE',
     'CTGGGCCGTATCTCAGTCCCAATGTGGCCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGACTTGGTAGGCCGTTACCCCACCAACTATCTAATCAGACGCAAGCCCATCTATCAGCGGATTGCTCCTTTCCCATTTATATCATGTGATATTCATGGCATATGCGGTATTAGCAGTCATTTCTAACTGTTGTTCCCCTCTGATAGG'),
    ('FLP3FBN01BPX14',
     'CTGGTCCGTGTCTCAGTACCAGTGTGGGGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGTCACGCGAGCCTATCCTCATCCGACGGATCTTTAGATGGAACCAGATGCCTGATTCCATCGCCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATCGAGGGCAAGTTGCATACGCG'),
    ('FLP3FBN01DB5I7',
     'TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCCTGCACCGCCGGAGCTTCCCCCGCCGCCCCATGCGGGGCTGCGGGCATATGCGGTATTAGCAGCCGTTTCCGGCTGTTGTCCCCCAGTGCAGGGCAGGTTGCCCACGCGTTACTCACCCGTCCG'),
    ('FLP3FBN01AK9OO',
     'CTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTCGAGCCGTTACCTCACCAACTAGCTAATGCGCCGCGGGCCCATCTCATAGCGGATTACTCCTTTAATTGCTACTTCATGCGAAGCTACAATCTTATGCGGTATTAATCTCCCTTTCGGAAGGCTATTCCCCTCTATGAGGTCAGGTTG'),
    ('FLP3FBN01ANGF2',
     'CTGGACCGTGTCTCAGTTCCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGTCACGCGAGCCTATCCTCATCCGACGGATCTTTAGATGGAACCAGATGCCTGATTCCATCGCCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCAAGTTGCTCACGCG'),
    ('FLP3FBN01AF994',
     'TTGGTCCGTGTCTCAGTACCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGTCGGTTTGGTGGGCCGTTACCCCGCCAACTGCCTAATGGAACGCATGCCTATCTATCAGCGATGAATCTTTAGCAAATATCCCCATGCGGGGCCCCTGCTTCATGCGGTATTAGTCCGACTTTCGCCGGGTTATCCCCTCCTGATAGGTAAGTTGCATACGCGTTACTTCACCCGTCGCGG'),
    ('FLP3FBN01AHXU8',
     'TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTACTGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCGGGGATGCCCCCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGGCAGAGAG')
]

expected_otu_map = {0: ['FLP3FBN01AM0P3'], 1: ['FLP3FBN01DDPFF'],
                    2: ['FLP3FBN01APRWO', 'FLP3FBN01APRWO2', 'FLP3FBN01APRWO3'],
                    3: ['FLP3FBN01DQ783'], 4: ['FLP3FBN01AF994'], 5: ['FLP3FBN01BPX14'],
                    6: ['FLP3FBN01ELBSX'], 7: ['FLP3FBN01DGFYQ'], 8: ['FLP3FBN01AK9OO'],
                    9: ['FLP3FBN01ANGF2'], 10: ['FLP3FBN01A55WZ'], 11: ['FLP3FBN01DB5I7'],
                    12: ['FLP3FBN01BBAE6'], 13: ['FLP3FBN01CPD70'], 14: ['FLP3FBN01D7O1S'],
                    15: ['FLP3FBN01EG8AX'], 16: ['FLP3FBN01DEHK3'], 17: ['FLP3FBN01BO4IE'],
                    18: ['FLP3FBN01CV5ZU'], 19: ['FLP3FBN01AHXU8'], 20: ['FLP3FBN01AWYZD'],
                    21: ['FLP3FBN01EEWKD']}

file_content = """20 400
0 48 1.03 0.04 0.96 0.08 0.05 1.05 0.09 1.02 0.08 1.09 0.09 0.06 2.09 1.09 0.09 1.04 0.07 1.05 0.14 0.08 1.02 1.03 3.93 2.06 1.00 0.96 0.15 1.04 1.01 0.14 1.89 0.16 0.09 0.93 1.13 0.94 0.07 0.14 2.00 1.05 0.96 3.00 1.19 0.94 0.13 1.04 0.12 0.15
1 48 1.03 0.04 1.02 0.13 0.09 0.99 0.13 1.14 0.11 1.09 0.14 0.11 1.92 0.99 0.13 1.25 0.09 0.95 0.20 0.10 1.02 1.07 3.81 2.22 1.02 0.97 0.22 0.98 0.96 0.25 1.89 0.25 0.09 0.91 1.18 0.92 0.09 0.16 1.96 1.06 0.93 2.91 1.15 0.92 0.12 1.01 0.14 0.14
2 48 1.05 0.04 0.95 0.10 0.07 1.03 0.08 1.09 0.08 1.06 0.08 0.08 1.92 1.04 0.06 1.11 0.08 0.94 0.15 0.09 1.00 1.10 3.98 2.22 1.00 1.03 0.16 0.96 1.01 0.16 2.06 0.18 0.08 0.98 1.07 1.01 0.08 0.12 2.10 1.01 1.01 3.08 1.11 0.95 0.11 1.05 0.14 0.14
3 48 1.01 0.04 0.97 0.08 0.05 1.05 0.07 1.03 0.10 1.11 0.08 0.07 2.05 1.08 0.05 0.87 0.06 1.03 0.13 0.08 1.02 1.06 3.96 2.08 1.01 1.07 0.18 1.05 0.96 0.15 1.89 0.14 0.08 0.98 1.15 0.92 0.07 0.16 2.04 1.08 0.90 3.03 1.08 0.94 0.11 1.06 0.12 0.13
4 48 1.08 0.03 0.95 0.11 0.07 0.99 0.12 1.05 0.10 1.10 0.12 0.08 2.01 1.04 0.10 1.10 0.07 1.00 0.14 0.10 1.01 1.07 4.03 2.01 1.01 0.95 0.18 1.06 1.01 0.17 1.90 0.16 0.07 0.86 1.11 0.83 0.09 0.16 1.99 1.04 0.97 2.87 1.17 0.87 0.13 1.09 0.13 0.15
5 48 1.03 0.04 0.91 0.11 0.09 1.06 0.10 1.12 0.09 1.08 0.11 0.11 1.90 1.03 0.12 1.09 0.08 0.96 0.20 0.09 1.08 1.01 3.97 2.07 0.96 0.99 0.21 1.00 1.02 0.27 1.99 0.24 0.13 0.91 1.20 0.85 0.11 0.19 1.92 0.96 0.98 2.72 1.10 0.91 0.14 1.22 0.15 0.14
6 48 1.05 0.01 0.90 0.08 0.06 1.05 0.10 1.09 0.06 1.08 0.11 0.08 1.96 1.02 0.08 1.08 0.06 1.02 0.16 0.09 0.90 1.10 4.03 2.21 1.01 1.00 0.19 1.01 1.00 0.21 1.93 0.24 0.08 0.89 1.11 0.89 0.06 0.19 2.09 1.08 0.95 3.02 1.24 0.97 0.13 1.01 0.14 0.13
7 48 1.03 0.03 0.97 0.11 0.09 1.03 0.11 1.09 0.10 1.09 0.11 0.10 1.91 1.05 0.14 1.15 0.07 1.01 0.19 0.11 0.95 1.04 3.94 2.22 0.98 0.99 0.17 0.96 0.95 0.16 2.09 0.17 0.09 1.02 1.13 0.95 0.09 0.13 2.03 1.00 1.02 3.05 1.06 0.95 0.11 1.14 0.13 0.13
8 48 1.07 0.04 0.97 0.15 0.11 1.00 0.20 1.12 0.12 1.06 0.22 0.13 1.98 0.98 0.14 1.17 0.08 0.93 0.16 0.11 1.04 1.07 4.10 2.23 1.02 1.02 0.19 0.96 1.00 0.25 2.06 0.26 0.08 0.97 1.08 0.93 0.11 0.16 2.05 1.10 1.03 3.12 1.09 0.97 0.13 0.90 0.16 0.13
9 48 1.00 0.05 1.07 0.07 0.18 0.96 0.08 1.17 0.08 1.17 0.10 0.16 1.94 1.06 0.21 1.42 0.09 0.98 0.22 0.10 1.06 1.00 3.75 2.22 1.01 0.94 0.27 1.03 1.07 0.34 1.77 0.36 0.15 0.79 1.22 0.86 0.11 0.15 1.90 1.05 0.90 2.86 1.21 0.91 0.16 1.16 0.24 0.16
10 48 1.01 0.04 1.08 0.15 0.15 0.92 0.33 1.18 0.11 1.02 0.10 0.09 1.67 0.96 0.06 1.01 0.18 0.93 0.17 0.13 1.07 1.14 4.09 2.07 1.01 1.02 0.33 0.92 1.03 0.09 2.09 0.22 0.09 0.93 1.10 0.93 0.11 0.14 2.02 1.03 1.06 3.20 1.08 1.01 0.09 0.93 0.21 0.16
11 48 1.02 0.04 1.01 0.11 0.09 1.02 0.11 1.05 0.09 1.00 0.10 0.09 2.03 0.96 0.09 1.11 0.08 0.88 0.16 0.09 1.06 1.00 3.95 2.22 1.02 0.92 0.17 0.98 0.97 0.18 1.99 0.17 0.09 0.89 1.14 0.99 0.09 0.15 1.97 1.05 1.00 2.71 1.13 1.00 0.13 1.00 0.14 0.14
12 48 0.99 0.03 1.00 0.06 0.07 1.06 0.03 1.04 0.06 1.15 0.05 0.09 2.02 1.05 0.08 1.05 0.06 1.00 0.16 0.07 1.04 1.00 3.98 2.11 0.99 1.01 0.17 1.02 1.01 0.20 2.00 0.18 0.08 0.99 1.14 0.94 0.06 0.13 2.04 1.00 0.98 3.04 1.05 0.99 0.13 1.05 0.09 0.16
13 48 1.07 0.05 0.93 0.15 0.07 1.01 0.19 1.07 0.13 1.06 0.21 0.12 2.06 1.04 0.14 1.14 0.09 0.92 0.23 0.14 1.03 1.06 3.96 2.03 0.97 0.97 0.18 0.97 1.00 0.19 1.90 0.21 0.10 0.95 1.20 0.96 0.10 0.23 2.05 1.10 0.97 2.91 1.13 0.91 0.15 0.95 0.13 0.21
14 48 1.08 0.05 0.93 0.10 0.17 1.00 0.11 1.29 0.10 1.07 0.12 0.17 1.89 0.87 0.19 1.34 0.12 0.83 0.29 0.15 1.08 0.99 3.70 2.15 1.02 0.86 0.24 0.91 1.00 0.40 1.91 0.35 0.14 0.90 1.20 0.94 0.17 0.22 1.95 1.04 1.01 2.60 1.08 0.98 0.17 1.02 0.18 0.16
15 48 0.97 0.04 1.01 0.13 0.10 1.06 0.10 1.23 0.11 1.08 0.14 0.11 1.90 1.06 0.12 1.16 0.12 0.87 0.17 0.10 1.04 1.07 3.80 2.12 1.07 0.88 0.27 0.88 0.95 0.27 1.90 0.21 0.15 0.87 0.99 0.93 0.09 0.17 2.04 1.04 0.99 2.73 1.17 0.94 0.15 1.10 0.13 0.13
16 48 1.05 0.03 0.88 0.08 0.01 1.06 0.08 1.01 0.06 1.07 0.10 0.04 2.04 1.10 0.06 1.00 0.08 1.00 0.13 0.08 1.00 1.06 4.00 1.83 1.04 0.99 0.13 1.10 1.04 0.16 1.73 0.16 0.13 0.87 1.14 0.87 0.08 0.15 1.93 1.02 0.93 2.93 1.33 0.95 0.16 1.24 0.12 0.20
17 48 0.98 0.07 0.91 0.16 0.15 1.10 0.21 1.17 0.12 0.96 0.22 0.14 1.99 1.09 0.14 1.18 0.11 1.02 0.21 0.12 0.91 1.10 4.13 2.11 0.92 1.01 0.23 1.01 1.01 0.25 1.89 0.30 0.16 0.86 1.07 0.93 0.14 0.18 2.03 1.08 0.99 3.08 1.34 0.88 0.13 1.21 0.18 0.14
18 48 1.04 0.05 0.88 0.06 0.03 1.07 0.07 0.93 0.09 1.05 0.09 0.06 2.13 1.05 0.08 0.89 0.08 1.01 0.14 0.10 0.93 1.11 4.44 2.11 1.05 1.00 0.14 1.01 1.04 0.09 2.09 0.10 0.07 1.01 1.05 0.90 0.10 0.13 2.09 1.04 0.98 3.06 1.12 0.94 0.11 1.10 0.12 0.10
19 48 1.04 0.05 0.88 0.06 0.03 1.07 0.07 0.93 1.09 0.05 0.09 0.06 3.13 1.05 1.08 0.89 0.08 1.01 0.14 1.10 0.93 1.11 2.44 2.11 1.05 2.00 0.14 1.01 1.04 1.09 2.09 0.10 0.07 1.01 1.05 0.30 0.10 0.13 2.09 1.04 0.98 3.06 1.12 0.94 0.11 1.10 0.12 0.10
"""

flowgrams = [(
    28, "1.03 0.04 0.96 0.08 0.05 1.05 0.09 1.02 0.08 1.09 0.09 0.06 2.09 1.09 0.09 1.04 0.07 1.05 0.14 0.08 1.02 1.03 3.93 2.06 1.00 0.96 0.15 1.04 1.01 0.14 1.89 0.16 0.09 0.93 1.13 0.94 0.07 0.14 2.00 1.05 0.96 3.00 1.19 0.94 0.13 1.04 0.12 0.15"),
    (405,
     "1.03 0.04 1.02 0.13 0.09 0.99 0.13 1.14 0.11 1.09 0.14 0.11 1.92 0.99 0.13 1.25 0.09 0.95 0.20 0.10 1.02 1.07 3.81 2.22 1.02 0.97 0.22 0.98 0.96 0.25 1.89 0.25 0.09 0.91 1.18 0.92 0.09 0.16 1.96 1.06 0.93 2.91 1.15 0.92 0.12 1.01 0.14 0.14"),
    (1349,
     "1.05 0.04 0.95 0.10 0.07 1.03 0.08 1.09 0.08 1.06 0.08 0.08 1.92 1.04 0.06 1.11 0.08 0.94 0.15 0.09 1.00 1.10 3.98 2.22 1.00 1.03 0.16 0.96 1.01 0.16 2.06 0.18 0.08 0.98 1.07 1.01 0.08 0.12 2.10 1.01 1.01 3.08 1.11 0.95 0.11 1.05 0.14 0.14"),
    (2519,
     "1.01 0.04 0.97 0.08 0.05 1.05 0.07 1.03 0.10 1.11 0.08 0.07 2.05 1.08 0.05 0.87 0.06 1.03 0.13 0.08 1.02 1.06 3.96 2.08 1.01 1.07 0.18 1.05 0.96 0.15 1.89 0.14 0.08 0.98 1.15 0.92 0.07 0.16 2.04 1.08 0.90 3.03 1.08 0.94 0.11 1.06 0.12 0.13"),
    (2986,
     "1.08 0.03 0.95 0.11 0.07 0.99 0.12 1.05 0.10 1.10 0.12 0.08 2.01 1.04 0.10 1.10 0.07 1.00 0.14 0.10 1.01 1.07 4.03 2.01 1.01 0.95 0.18 1.06 1.01 0.17 1.90 0.16 0.07 0.86 1.11 0.83 0.09 0.16 1.99 1.04 0.97 2.87 1.17 0.87 0.13 1.09 0.13 0.15"),
    (7231,
     "1.03 0.04 0.91 0.11 0.09 1.06 0.10 1.12 0.09 1.08 0.11 0.11 1.90 1.03 0.12 1.09 0.08 0.96 0.20 0.09 1.08 1.01 3.97 2.07 0.96 0.99 0.21 1.00 1.02 0.27 1.99 0.24 0.13 0.91 1.20 0.85 0.11 0.19 1.92 0.96 0.98 2.72 1.10 0.91 0.14 1.22 0.15 0.14"),
    (8446,
     " 1.05 0.01 0.90 0.08 0.06 1.05 0.10 1.09 0.06 1.08 0.11 0.08 1.96 1.02 0.08 1.08 0.06 1.02 0.16 0.09 0.90 1.10 4.03 2.21 1.01 1.00 0.19 1.01 1.00 0.21 1.93 0.24 0.08 0.89 1.11 0.89 0.06 0.19 2.09 1.08 0.95 3.02 1.24 0.97 0.13 1.01 0.14 0.13"),
    (9909,
     " 1.03 0.03 0.97 0.11 0.09 1.03 0.11 1.09 0.10 1.09 0.11 0.10 1.91 1.05 0.14 1.15 0.07 1.01 0.19 0.11 0.95 1.04 3.94 2.22 0.98 0.99 0.17 0.96 0.95 0.16 2.09 0.17 0.09 1.02 1.13 0.95 0.09 0.13 2.03 1.00 1.02 3.05 1.06 0.95 0.11 1.14 0.13 0.13"),
    (11113,
     " 1.07 0.04 0.97 0.15 0.11 1.00 0.20 1.12 0.12 1.06 0.22 0.13 1.98 0.98 0.14 1.17 0.08 0.93 0.16 0.11 1.04 1.07 4.10 2.23 1.02 1.02 0.19 0.96 1.00 0.25 2.06 0.26 0.08 0.97 1.08 0.93 0.11 0.16 2.05 1.10 1.03 3.12 1.09 0.97 0.13 0.90 0.16 0.13"),
    (13119,
     " 1.00 0.05 1.07 0.07 0.18 0.96 0.08 1.17 0.08 1.17 0.10 0.16 1.94 1.06 0.21 1.42 0.09 0.98 0.22 0.10 1.06 1.00 3.75 2.22 1.01 0.94 0.27 1.03 1.07 0.34 1.77 0.36 0.15 0.79 1.22 0.86 0.11 0.15 1.90 1.05 0.90 2.86 1.21 0.91 0.16 1.16 0.24 0.16"),
    (13888,
     " 1.01 0.04 1.08 0.15 0.15 0.92 0.33 1.18 0.11 1.02 0.10 0.09 1.67 0.96 0.06 1.01 0.18 0.93 0.17 0.13 1.07 1.14 4.09 2.07 1.01 1.02 0.33 0.92 1.03 0.09 2.09 0.22 0.09 0.93 1.10 0.93 0.11 0.14 2.02 1.03 1.06 3.20 1.08 1.01 0.09 0.93 0.21 0.16"),
    (2378,
     " 1.02 0.04 1.01 0.11 0.09 1.02 0.11 1.05 0.09 1.00 0.10 0.09 2.03 0.96 0.09 1.11 0.08 0.88 0.16 0.09 1.06 1.00 3.95 2.22 1.02 0.92 0.17 0.98 0.97 0.18 1.99 0.17 0.09 0.89 1.14 0.99 0.09 0.15 1.97 1.05 1.00 2.71 1.13 1.00 0.13 1.00 0.14 0.14"),
    (3816,
     " 0.99 0.03 1.00 0.06 0.07 1.06 0.03 1.04 0.06 1.15 0.05 0.09 2.02 1.05 0.08 1.05 0.06 1.00 0.16 0.07 1.04 1.00 3.98 2.11 0.99 1.01 0.17 1.02 1.01 0.20 2.00 0.18 0.08 0.99 1.14 0.94 0.06 0.13 2.04 1.00 0.98 3.04 1.05 0.99 0.13 1.05 0.09 0.16"),
    (5984,
     " 1.07 0.05 0.93 0.15 0.07 1.01 0.19 1.07 0.13 1.06 0.21 0.12 2.06 1.04 0.14 1.14 0.09 0.92 0.23 0.14 1.03 1.06 3.96 2.03 0.97 0.97 0.18 0.97 1.00 0.19 1.90 0.21 0.10 0.95 1.20 0.96 0.10 0.23 2.05 1.10 0.97 2.91 1.13 0.91 0.15 0.95 0.13 0.21"),
    (13627,
     " 1.08 0.05 0.93 0.10 0.17 1.00 0.11 1.29 0.10 1.07 0.12 0.17 1.89 0.87 0.19 1.34 0.12 0.83 0.29 0.15 1.08 0.99 3.70 2.15 1.02 0.86 0.24 0.91 1.00 0.40 1.91 0.35 0.14 0.90 1.20 0.94 0.17 0.22 1.95 1.04 1.01 2.60 1.08 0.98 0.17 1.02 0.18 0.16"),
    (15472,
     " 0.97 0.04 1.01 0.13 0.10 1.06 0.10 1.23 0.11 1.08 0.14 0.11 1.90 1.06 0.12 1.16 0.12 0.87 0.17 0.10 1.04 1.07 3.80 2.12 1.07 0.88 0.27 0.88 0.95 0.27 1.90 0.21 0.15 0.87 0.99 0.93 0.09 0.17 2.04 1.04 0.99 2.73 1.17 0.94 0.15 1.10 0.13 0.13"),
    (3222,
     " 1.05 0.03 0.88 0.08 0.01 1.06 0.08 1.01 0.06 1.07 0.10 0.04 2.04 1.10 0.06 1.00 0.08 1.00 0.13 0.08 1.00 1.06 4.00 1.83 1.04 0.99 0.13 1.10 1.04 0.16 1.73 0.16 0.13 0.87 1.14 0.87 0.08 0.15 1.93 1.02 0.93 2.93 1.33 0.95 0.16 1.24 0.12 0.20"),
    (14843,
     " 0.98 0.07 0.91 0.16 0.15 1.10 0.21 1.17 0.12 0.96 0.22 0.14 1.99 1.09 0.14 1.18 0.11 1.02 0.21 0.12 0.91 1.10 4.13 2.11 0.92 1.01 0.23 1.01 1.01 0.25 1.89 0.30 0.16 0.86 1.07 0.93 0.14 0.18 2.03 1.08 0.99 3.08 1.34 0.88 0.13 1.21 0.18 0.14"),
    (99451,
     " 1.04 0.05 0.88 0.06 0.03 1.07 0.07 0.93 0.09 1.05 0.09 0.06 2.13 1.05 0.08 0.89 0.08 1.01 0.14 0.10 0.93 1.11 4.44 2.11 1.05 1.00 0.14 1.01 1.04 0.09 2.09 0.10 0.07 1.01 1.05 0.90 0.10 0.13 2.09 1.04 0.98 3.06 1.12 0.94 0.11 1.10 0.12 0.10"),
    (78124, "1.04 0.05 0.88 0.06 0.03 1.07 0.07 0.93 1.09 0.05 0.09 0.06 3.13 1.05 1.08 0.89 0.08 1.01 0.14 1.10 0.93 1.11 2.44 2.11 1.05 2.00 0.14 1.01 1.04 1.09 2.09 0.10 0.07 1.01 1.05 0.30 0.10 0.13 2.09 1.04 0.98 3.06 1.12 0.94 0.11 1.10 0.12 0.10")]

otu_map = {'1': ['78124'],
           '0': ['28', '405', '1349', '2519', '2986', '7231', '8446', '9909', '11113',
                 '13119', '13888', '2378', '3816', '5984', '13627', '15472', '3222',
                 '14843', '99451']}

header = """Common Header:
  Magic Number:  0x2E736666
  Version:       0001
  Index Offset:  7773224
  Index Length:  93365
  # of Reads:    69
  Header Length: 440
  Key Length:    4
  # of Flows:    48
  Flowgram Code: 1
  Flow Chars:    TACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTA
  Key Sequence:  TCAG
"""


# On small examples pyronoise crashes, so we have to take a somewhat
# larger example
large_example = [
    "	0.97	0.00	0.99	0.01	0.00	1.04	0.00	0.98	1.00	1.16	0.03	0.09	1.91	0.00	0.97	0.95	1.02	0.00	0.00	0.97	1.00	0.02	0.98	0.11	0.00	1.01	0.15	0.94	0.00	0.06	0.93	0.04	0.03	1.11	0.01	0.04	0.99	0.00	0.06	0.98	0.05	0.00	1.01	0.06	1.01	0.00	0.14	0.98	0.10	0.00	2.11	0.14	1.01	0.04	2.78	0.99	0.98	1.06	0.05	1.95	0.17	1.01	0.03	0.98	2.93	0.03	0.12	1.98	0.10	0.96	2.05	0.96	1.04	0.00	0.14	0.99	1.01	0.00	1.02	0.09	0.99	0.04	0.97	0.12	0.11	0.97	0.17	0.97	1.96	0.14	1.94	0.09	0.09	2.01	0.12	0.09	1.01	0.04	0.03	1.08	0.98	0.04	0.16	4.79	0.10	1.00	2.06	0.05	2.03	0.06	1.95	0.08	0.93	0.01	0.94	0.05	1.02	0.04	1.01	0.11	0.08	1.06	0.09	1.03	0.10	2.05	4.04	0.16	1.01	0.97	0.18	0.05	0.96	0.13	1.98	0.07	0.14	1.04	0.13	0.14	1.01	0.14	1.03	1.03	0.05	2.05	0.18	1.98	3.05	0.06	0.08	1.90	0.98	0.09	0.08	0.95	0.17	1.00	0.19	0.97	0.10	0.13	2.16	1.08	2.05	1.04	2.02	0.16	1.03	0.10	0.97	0.08	0.09	0.94	1.92	0.08	0.11	2.02	0.99	0.15	1.03	0.00	0.09	1.03	0.11	0.11	2.01	0.17	0.94	2.11	0.13	0.17	0.93	0.02	0.16	1.97	0.11	1.97	1.07	0.93	0.04	0.08	1.01	0.15	0.12	1.00	0.01	0.06	1.06	0.10	3.88	0.17	0.16	0.96	0.11	0.15	0.94	0.11	1.05	1.05	0.16	0.92	0.07	0.11	1.02	2.00	2.07	1.01	0.18	2.88	0.20	0.13	1.91	0.00	1.04	0.07	2.80	2.08	0.16	0.04	0.96	2.04	0.14	0.05	0.98	0.07	0.97	0.06	0.11	3.10	2.06	0.09	0.17	0.94	0.10	0.11	1.03	0.11	0.18	0.96	0.13	0.08	1.08	1.97	1.00	0.08	0.99	0.09	1.00	0.09	0.17	1.02	0.01	1.00	0.13	0.19	2.10	0.98	0.13	0.09	1.00	1.05	2.17	0.15	0.08	0.98	0.14	0.04	1.07	0.15	1.08	3.03	1.03	0.96	0.19	0.05	1.96	2.07	0.15	0.10	0.95	0.00	0.96	0.08	2.91	0.13	0.97	0.19	2.87	0.08	0.99	1.05	0.14	2.96	0.14	1.77	0.11	0.07	0.99	0.04	1.04	0.99	0.00	0.16	1.04	0.17	4.07	1.23	0.08	0.90	0.02	0.91	2.10	0.99	0.15	0.00	0.97	0.05	1.05	2.17	0.12	0.17	0.98	0.00	0.04	1.17	0.07	1.91	2.13	0.10	0.20	1.88	0.17	0.94	0.17	0.14	1.02	0.99	1.10	1.11	1.01	0.14	0.11	0.99	1.96	0.99	1.07	0.09	0.89	0.08	0.84	0.11	0.08	0.95	3.10	1.05	1.09	0.14	0.12	0.93	0.11	0.06	1.08	0.96	0.06	0.00	1.97	1.91	1.16	0.04	0.81	0.88	0.04	0.08	2.03	0.13	0.18	0.95	0.22	0.16",
    "	0.99	0.00	0.99	0.04	0.01	1.01	0.04	0.93	0.96	1.13	0.01	0.00	1.95	0.00	0.96	1.02	0.97	0.00	0.03	0.98	1.02	0.00	0.95	0.04	0.03	1.03	0.04	0.97	0.03	0.00	0.99	0.05	0.01	1.06	0.06	0.03	0.99	0.00	0.03	0.98	0.05	0.00	0.98	0.07	1.00	0.00	0.10	0.98	0.05	0.00	1.95	0.10	0.98	0.00	2.97	1.02	1.04	1.03	0.15	1.97	0.14	0.98	0.06	0.99	2.98	0.07	0.06	1.98	0.16	1.05	2.04	0.98	1.04	0.01	0.10	0.98	0.96	0.00	0.98	0.07	0.98	0.00	0.99	0.08	0.12	1.00	0.13	0.99	2.04	0.08	2.00	0.10	0.13	2.02	0.14	0.07	1.02	0.09	0.05	0.98	1.03	0.03	0.10	4.79	0.17	1.00	2.03	0.09	2.01	0.06	1.96	0.09	0.98	0.00	0.98	0.08	0.99	0.02	1.02	0.11	0.07	1.00	0.09	1.04	0.07	2.02	4.02	0.15	1.01	0.98	0.15	0.02	1.02	0.10	2.03	0.06	0.11	1.01	0.11	0.07	0.99	0.14	1.01	1.00	0.18	2.09	0.17	2.04	3.06	0.05	0.05	1.94	1.02	0.08	0.03	0.96	0.14	0.99	0.13	1.02	0.15	0.10	2.14	1.03	2.13	1.08	2.07	0.14	0.97	0.12	0.92	0.08	0.10	0.99	1.94	0.10	0.09	1.97	1.03	0.15	1.04	0.14	0.15	0.99	0.17	0.06	1.98	0.14	1.02	1.99	0.10	0.16	0.92	0.07	0.14	2.05	0.14	1.96	1.06	0.96	0.05	0.03	1.01	0.14	0.11	0.96	0.07	0.04	1.02	0.07	3.86	0.14	0.18	0.94	0.11	0.10	1.02	0.14	1.05	1.05	0.09	0.98	0.11	0.10	0.93	1.94	1.98	1.04	0.19	2.81	0.19	0.14	1.97	0.01	1.05	0.06	2.79	2.05	0.14	0.05	0.90	1.99	0.14	0.05	1.04	0.07	0.97	0.04	0.16	2.99	2.06	0.09	0.14	0.96	0.09	0.11	1.02	0.06	0.19	0.94	0.15	0.06	1.06	1.97	1.08	0.08	0.97	0.11	1.01	0.09	0.11	0.99	0.05	1.01	0.15	0.16	2.11	1.00	0.14	0.07	0.99	1.02	2.08	0.12	0.14	0.95	0.15	0.00	1.01	0.15	1.09	2.99	1.08	1.00	0.13	0.03	1.96	2.01	0.14	0.12	0.92	0.04	0.99	0.08	2.94	0.15	0.97	0.13	2.96	0.09	0.97	1.02	0.13	2.93	0.22	1.85	0.13	0.09	1.00	0.04	1.07	0.98	0.04	0.17	0.99	0.11	3.96	1.16	0.09	0.92	0.05	0.96	1.98	0.96	0.15	0.04	0.99	0.04	1.02	2.01	0.13	0.11	1.06	0.20	0.04	0.97	0.05	1.93	2.14	0.16	0.11	1.93	0.18	0.96	0.19	0.16	1.06	0.97	1.14	1.05	0.94	0.10	0.13	1.00	1.97	0.99	1.06	0.09	0.91	0.07	0.91	0.21	0.17	1.00	2.98	1.10	1.08	0.15	0.12	0.91	0.09	0.00	1.00	0.91	0.17	0.02	1.98	1.93	1.15	0.03	0.87	0.95	0.07	0.07	1.97	0.12	0.13	0.95	0.18	0.15",
    "	0.99	0.00	1.01	0.03	0.00	0.99	0.01	0.96	1.00	1.14	0.00	0.02	1.96	0.00	0.96	0.99	1.00	0.00	0.00	0.98	1.01	0.00	0.96	0.06	0.00	1.03	0.03	0.96	0.01	0.00	0.96	0.03	0.03	1.07	0.03	0.03	0.97	0.00	0.02	0.98	0.06	0.00	0.95	0.07	0.99	0.00	0.12	0.97	0.07	0.00	2.00	0.11	1.00	0.00	2.92	1.03	1.02	1.07	0.12	2.01	0.17	1.00	0.04	1.01	2.96	0.09	0.08	2.00	0.09	1.00	1.93	1.02	1.01	0.02	0.13	0.99	1.01	0.00	0.99	0.09	0.98	0.01	0.96	0.10	0.16	0.98	0.16	0.97	2.00	0.15	1.92	0.10	0.15	2.04	0.14	0.10	1.02	0.08	0.03	1.07	0.99	0.03	0.15	4.70	0.13	0.99	1.92	0.08	1.92	0.07	1.91	0.10	0.97	0.00	0.99	0.07	1.00	0.05	1.00	0.13	0.10	1.00	0.12	1.03	0.11	2.00	3.84	0.18	1.00	0.99	0.18	0.06	1.00	0.12	1.93	0.08	0.15	1.01	0.15	0.10	1.01	0.12	1.00	1.01	0.09	1.98	0.17	1.98	2.98	0.03	0.05	1.98	1.02	0.07	0.05	0.92	0.14	1.02	0.18	1.01	0.17	0.12	2.04	1.03	2.09	1.06	1.97	0.16	1.01	0.15	0.97	0.10	0.10	0.97	1.95	0.09	0.09	2.02	1.06	0.12	0.96	0.00	0.19	1.00	0.17	0.12	1.96	0.16	0.98	2.01	0.11	0.16	0.94	0.11	0.15	1.95	0.16	2.00	1.05	0.99	0.03	0.03	0.96	0.17	0.10	0.95	0.07	0.07	1.05	0.06	3.81	0.16	0.16	0.98	0.14	0.12	1.00	0.11	1.04	1.05	0.11	0.95	0.07	0.08	0.96	1.92	1.97	1.02	0.04	2.85	0.10	0.14	1.90	0.00	1.03	0.07	2.88	2.03	0.15	0.06	0.95	1.97	1.10	0.14	0.04	2.00	0.09	0.97	0.09	0.96	1.97	0.16	0.13	0.96	0.09	0.14	1.03	0.06	0.16	0.99	0.12	0.05	1.03	1.95	1.03	0.08	1.02	0.13	1.02	0.12	0.19	1.00	0.02	1.05	0.11	0.17	2.11	0.98	0.13	0.08	0.98	0.95	2.07	0.12	0.18	1.03	0.13	0.06	1.00	0.16	1.10	3.06	1.09	1.00	0.06	0.01	2.05	2.04	0.12	0.12	0.97	0.02	0.93	0.11	2.96	0.16	1.96	0.12	2.04	0.05	0.94	0.96	0.10	2.95	0.12	1.87	0.12	0.05	1.06	0.06	1.02	1.03	0.00	0.21	1.04	0.13	3.94	1.15	0.07	0.90	0.08	0.95	2.08	0.99	0.14	0.00	0.96	0.01	1.00	2.07	0.13	0.13	1.05	0.01	0.02	1.02	0.06	1.99	2.14	0.14	0.13	1.87	0.19	0.94	0.17	0.15	1.08	0.96	1.18	1.00	0.95	0.11	0.13	0.97	1.95	1.04	1.07	0.07	0.93	0.09	0.87	0.07	0.16	0.99	3.05	1.05	1.07	0.15	0.07	0.94	0.07	0.01	0.99	0.97	0.07	0.00	1.96	1.92	1.09	0.02	0.88	0.92	0.01	0.07	1.93	0.16	0.20	0.94	0.17	0.13",
    "	0.99	0.00	1.00	0.01	0.00	1.00	0.01	1.03	1.01	1.08	0.05	0.00	1.96	0.00	0.98	0.99	1.00	0.00	0.03	0.96	1.01	0.00	0.96	0.07	0.03	1.02	0.05	0.97	0.00	0.00	0.96	0.03	0.01	1.07	0.03	0.04	0.97	0.00	0.03	0.98	0.07	0.00	0.97	0.08	0.97	0.00	0.12	0.98	0.07	0.00	2.03	0.10	1.01	0.01	2.97	1.03	0.99	1.07	0.13	2.03	0.16	1.00	0.03	0.99	3.00	0.05	0.09	2.02	0.14	0.99	1.99	1.01	1.02	0.00	0.12	1.00	0.98	0.00	0.99	0.06	1.00	0.00	0.99	0.08	0.12	0.99	0.15	0.98	2.01	0.12	2.02	0.08	0.12	2.04	0.13	0.07	1.00	0.07	0.02	1.03	1.00	0.02	0.09	4.80	0.15	0.98	2.02	0.11	2.04	0.05	1.99	0.09	0.95	0.00	1.05	0.05	1.03	0.01	1.02	0.11	0.11	0.94	0.11	1.02	0.09	1.98	3.87	0.18	1.01	0.96	0.12	0.09	1.00	0.13	2.04	0.08	0.12	1.08	0.14	0.10	1.03	0.10	1.01	0.98	0.01	2.04	0.17	1.94	2.96	0.16	0.08	1.95	0.98	0.07	0.07	0.95	0.13	0.99	0.20	1.01	0.11	0.10	2.12	1.11	2.07	1.06	1.98	0.15	0.97	0.12	0.96	0.13	0.15	0.97	2.06	0.10	0.08	2.04	1.05	0.11	1.01	0.04	0.08	0.97	0.11	0.12	2.03	0.16	0.95	2.05	0.12	0.15	0.98	0.12	0.16	1.98	0.17	2.01	1.05	0.96	0.05	0.06	0.98	0.17	0.09	1.00	0.05	0.09	0.99	0.14	3.83	0.17	0.11	1.03	0.14	0.11	0.98	0.16	1.07	1.03	0.10	0.95	0.10	0.09	0.96	1.89	2.02	1.01	0.18	2.95	0.13	0.14	1.86	0.00	1.05	0.08	2.82	2.00	0.15	0.06	0.94	2.00	0.21	0.04	1.03	0.02	0.97	0.07	0.10	3.05	2.02	0.07	0.15	0.95	0.08	0.11	1.04	0.07	0.21	0.93	0.14	0.07	0.99	1.94	1.08	0.07	0.99	0.12	1.00	0.14	0.21	1.04	0.04	1.06	0.12	0.19	2.03	0.97	0.09	0.08	1.05	0.96	2.12	0.10	0.19	1.01	0.17	0.03	1.01	0.16	1.08	3.08	1.08	0.98	0.20	0.01	2.08	2.10	0.11	0.11	0.93	0.01	0.96	0.09	2.81	0.14	0.96	0.18	2.95	0.04	0.96	1.02	0.14	2.98	0.15	1.77	0.12	0.07	1.01	0.04	1.05	1.04	0.00	0.17	1.04	0.12	4.03	1.16	0.09	0.96	0.07	0.92	2.04	0.92	0.13	0.22	0.95	0.06	1.01	2.02	0.16	0.13	1.12	0.00	0.08	1.02	0.10	1.90	2.14	0.15	0.21	1.87	0.16	0.87	0.19	0.18	1.03	0.97	1.18	1.07	1.01	0.12	0.15	1.04	1.95	1.04	1.09	0.12	0.94	0.08	0.80	0.20	0.20	0.98	3.06	1.06	1.08	0.18	0.15	0.93	0.09	0.04	1.02	0.90	0.07	0.03	1.98	1.85	1.08	0.04	0.89	0.98	0.04	0.09	2.01	0.13	0.21	0.99	0.22	0.17",
    "	1.00	0.00	0.99	0.03	0.00	1.01	0.02	1.03	1.01	1.09	0.05	0.04	1.96	0.00	0.97	0.96	0.99	0.00	0.04	0.98	0.98	0.00	0.94	0.07	0.01	1.04	0.03	1.00	0.02	0.01	0.99	0.06	0.00	1.06	0.04	0.01	0.98	0.02	0.01	0.98	0.04	0.00	0.99	0.06	0.99	0.00	0.07	0.99	0.04	0.00	1.90	0.09	1.02	0.00	2.93	0.96	1.04	1.04	0.16	1.95	0.14	1.01	0.04	0.99	3.00	0.10	0.08	1.94	0.09	1.00	1.95	1.04	0.99	0.03	0.13	0.99	0.97	0.00	0.99	0.10	1.01	0.02	0.98	0.08	0.13	1.00	0.15	1.01	2.01	0.10	1.96	0.11	0.14	1.99	0.17	0.06	1.02	0.13	0.05	1.00	0.98	0.05	0.09	4.63	0.18	1.02	1.97	0.15	1.99	0.07	1.96	0.11	0.99	0.00	0.97	0.07	1.00	0.02	1.02	0.12	0.09	1.00	0.09	1.02	0.08	2.01	3.86	0.17	0.98	1.01	0.17	0.04	0.96	0.13	1.98	0.08	0.13	1.04	0.09	0.10	1.01	0.09	0.99	1.01	0.17	2.02	0.17	1.90	2.93	0.04	0.06	1.89	1.03	0.09	0.03	1.02	0.14	0.99	0.17	0.95	0.10	0.14	2.11	1.07	1.98	1.05	1.97	0.12	0.93	0.15	0.99	0.04	0.13	1.02	1.90	0.11	0.08	2.01	0.99	0.11	1.01	0.00	0.17	1.01	0.10	0.12	1.95	0.17	0.98	2.05	0.10	0.17	0.95	0.08	0.13	2.01	0.16	1.89	1.04	0.92	0.06	0.02	1.04	0.17	0.08	0.96	0.02	0.06	1.05	0.05	3.72	0.12	0.07	0.97	0.11	0.12	1.00	0.10	1.08	1.00	0.11	0.93	0.06	0.08	1.03	1.84	1.93	1.02	0.04	2.71	0.17	0.15	1.99	0.03	1.02	0.07	2.82	1.97	0.17	0.03	0.94	2.04	0.65	0.08	0.55	0.99	0.49	0.57	0.15	1.95	2.01	0.15	0.13	0.99	0.07	0.16	1.02	0.09	0.18	0.98	0.07	0.09	1.02	1.94	1.09	0.13	0.98	0.12	1.00	0.14	0.07	1.00	0.03	1.02	0.13	0.18	2.05	0.95	0.10	0.09	1.00	0.98	2.08	0.13	0.12	1.03	0.16	0.04	1.07	0.15	1.07	3.00	1.05	0.94	0.10	0.02	1.99	2.03	0.21	0.11	0.93	0.02	0.99	0.10	2.89	0.16	1.00	0.15	2.94	0.05	0.96	1.00	0.09	2.94	0.19	1.81	0.12	0.05	1.00	0.04	1.04	1.02	0.00	0.18	1.02	0.12	3.97	1.16	0.06	0.92	0.05	0.95	2.14	0.95	0.23	0.04	0.95	0.02	1.02	2.08	0.14	0.16	1.05	0.02	0.00	1.05	0.03	1.96	2.23	0.12	0.14	1.91	0.06	0.94	0.19	0.09	1.02	0.95	1.18	1.09	1.00	0.13	0.08	0.98	1.94	1.00	1.08	0.06	0.92	0.07	0.85	0.10	0.13	0.96	2.99	1.10	1.07	0.00	0.04	0.94	0.08	0.03	1.03	0.95	0.21	0.03	2.03	1.94	1.12	0.04	0.78	0.94	0.01	0.06	1.97	0.11	0.18	0.96	0.15	0.13",
    "	1.00	0.00	1.01	0.04	0.00	0.99	0.03	0.95	0.98	1.10	0.00	0.01	2.01	0.01	0.99	0.99	1.00	0.00	0.00	0.99	0.99	0.00	0.94	0.06	0.00	1.04	0.00	1.01	0.00	0.00	0.98	0.04	0.00	1.03	0.02	0.01	0.97	0.00	0.01	0.99	0.02	0.00	0.95	0.06	0.99	0.00	0.09	1.00	0.02	0.00	1.96	0.09	1.00	0.02	2.92	1.02	1.04	1.04	0.12	2.02	0.14	0.98	0.04	0.99	2.97	0.06	0.09	1.96	0.15	1.02	1.97	1.05	1.00	0.02	0.11	0.99	1.00	0.00	0.94	0.09	1.00	0.02	0.97	0.09	0.11	0.99	0.15	1.01	2.03	0.12	1.99	0.12	0.11	2.00	0.16	0.08	0.99	0.10	0.03	0.96	1.02	0.03	0.13	4.87	0.10	1.02	1.94	0.11	1.94	0.07	2.00	0.10	0.98	0.00	0.97	0.06	1.00	0.05	1.04	0.12	0.06	0.96	0.06	1.03	0.07	1.99	3.91	0.15	0.96	1.07	0.16	0.04	1.00	0.10	1.99	0.07	0.12	1.01	0.14	0.07	0.98	0.14	1.02	0.98	0.03	2.02	0.18	1.97	2.96	0.07	0.06	1.92	1.05	0.09	0.04	0.96	0.11	1.01	0.19	1.01	0.05	0.12	2.08	1.05	2.08	1.05	2.02	0.15	0.96	0.15	0.95	0.07	0.10	0.96	1.98	0.10	0.13	2.04	1.08	0.14	1.00	0.03	0.19	0.98	0.16	0.13	2.02	0.14	0.98	2.06	0.09	0.05	0.94	0.11	0.17	1.92	0.13	1.95	1.02	0.97	0.05	0.07	1.01	0.16	0.08	1.01	0.02	0.05	1.04	0.10	3.77	0.12	0.15	0.95	0.13	0.11	0.99	0.09	1.06	1.06	0.09	0.96	0.03	0.11	0.96	1.91	1.97	1.01	0.00	2.88	0.20	0.13	1.94	0.00	1.03	0.07	2.95	2.12	0.16	0.04	1.05	2.15	0.61	0.11	0.52	0.91	0.54	0.49	0.16	2.16	2.08	0.13	0.12	0.97	0.11	0.11	1.05	0.06	0.19	0.95	0.19	0.06	1.05	2.02	1.05	0.08	1.00	0.14	0.98	0.16	0.15	0.99	0.00	1.02	0.11	0.17	2.04	1.01	0.07	0.06	1.03	1.05	2.01	0.12	0.16	0.98	0.16	0.00	1.01	0.06	1.08	3.02	1.06	0.94	0.11	0.03	2.06	2.13	0.22	0.10	0.92	0.05	1.01	0.10	2.97	0.17	0.95	0.16	3.00	0.09	1.00	0.99	0.15	3.00	0.10	1.89	0.11	0.12	1.00	0.07	1.00	1.02	0.00	0.17	1.07	0.18	4.00	1.14	0.02	0.93	0.05	0.92	2.02	1.02	0.22	0.01	0.99	0.05	1.02	2.11	0.14	0.15	0.99	0.00	0.00	1.01	0.10	1.93	2.09	0.19	0.10	1.95	0.19	0.95	0.18	0.15	1.03	0.97	1.19	1.03	0.98	0.14	0.15	0.96	1.89	1.04	1.04	0.12	0.92	0.09	0.96	0.12	0.17	1.00	2.90	1.05	1.03	0.17	0.15	0.96	0.09	0.04	1.04	0.95	0.20	0.02	1.96	1.92	1.05	0.03	0.83	0.93	0.07	0.11	1.96	0.16	0.21	0.98	0.24	0.12",
    "	1.01	0.01	1.00	0.06	0.02	0.99	0.06	0.91	0.99	1.15	0.02	0.02	2.02	0.00	0.99	0.99	1.02	0.00	0.02	1.02	0.99	0.00	0.97	0.04	0.00	1.01	0.03	0.98	0.03	0.00	0.99	0.06	0.00	1.05	0.07	0.03	0.96	0.02	0.03	0.99	0.03	0.00	1.00	0.05	1.01	0.00	0.04	1.00	0.53	0.00	1.39	0.08	1.01	0.01	2.88	0.96	1.03	1.02	0.14	1.99	0.09	0.99	0.08	0.97	3.03	0.08	0.04	1.99	0.12	1.07	1.99	0.93	1.01	0.05	0.08	1.01	0.97	0.01	0.99	0.08	1.02	0.02	0.98	0.09	0.09	1.00	0.10	1.01	2.04	0.05	1.95	0.12	0.09	1.93	0.14	0.10	0.98	0.08	0.08	1.03	1.00	0.06	0.12	4.66	0.17	1.03	1.99	0.04	1.96	0.09	1.96	0.10	0.99	0.00	0.96	0.11	0.99	0.06	0.97	0.14	0.11	0.97	0.19	1.03	0.08	1.98	3.87	0.13	1.02	0.98	0.14	0.07	0.97	0.11	1.99	0.06	0.09	1.02	0.14	0.06	1.05	0.11	1.05	1.02	0.17	1.97	0.11	1.96	3.00	0.07	0.07	1.97	0.94	0.07	0.08	0.95	0.11	1.08	0.17	0.98	0.15	0.13	2.08	1.03	2.13	1.05	1.87	0.16	0.94	0.14	0.97	0.09	0.12	0.98	1.98	0.08	0.10	1.94	1.06	0.12	1.00	0.00	0.13	1.00	0.13	0.10	1.96	0.17	0.96	1.95	0.08	0.10	0.98	0.10	0.12	2.01	0.16	1.87	1.05	0.99	0.03	0.07	0.97	0.09	0.10	0.96	0.02	0.09	0.99	0.12	3.84	0.17	0.13	0.98	0.07	0.15	1.01	0.09	1.07	1.06	0.09	0.97	0.16	0.11	0.98	1.90	1.99	0.90	0.08	2.75	0.14	0.14	1.89	0.04	1.06	0.10	2.88	2.04	0.13	0.09	1.05	1.90	0.59	0.08	0.47	0.96	0.58	0.56	0.15	1.94	1.98	0.14	0.15	0.94	0.10	0.11	1.02	0.08	0.22	1.00	0.13	0.09	1.00	1.95	1.09	0.06	0.99	0.13	0.97	0.15	0.19	1.01	0.15	1.01	0.12	0.19	2.08	0.96	0.15	0.06	0.98	0.97	2.00	0.10	0.14	0.99	0.14	0.04	1.06	0.18	1.03	3.00	1.06	0.98	0.12	0.04	1.97	1.98	0.14	0.11	0.98	0.05	0.97	0.06	2.85	0.12	0.96	0.15	2.88	0.10	0.99	1.04	0.14	2.79	0.15	1.92	0.16	0.02	1.00	0.05	1.04	1.00	0.02	0.20	0.98	0.10	4.02	1.15	0.10	0.93	0.10	0.94	2.03	0.97	0.21	0.19	0.95	0.01	1.05	2.07	0.04	0.11	1.01	0.04	0.04	1.02	0.16	1.99	2.06	0.12	0.11	1.90	0.09	0.94	0.21	0.06	1.06	0.94	1.13	1.07	1.04	0.10	0.12	0.96	1.96	0.98	1.09	0.12	0.95	0.08	0.81	0.14	0.18	0.97	2.98	1.07	1.03	0.00	0.10	0.99	0.01	0.05	1.07	1.01	0.12	0.00	1.96	1.90	1.08	0.03	0.83	0.94	0.03	0.08	1.93	0.07	0.18	0.94	0.17	0.10",
    "	0.99	0.00	1.00	0.04	0.00	1.01	0.03	0.98	0.97	1.13	0.01	0.00	2.03	0.00	0.96	1.02	1.00	0.00	0.02	0.98	1.00	0.00	0.99	0.03	0.00	1.02	0.02	0.96	0.01	0.00	0.96	0.05	0.00	1.06	0.08	0.02	0.99	0.00	0.02	0.96	0.04	0.00	0.97	0.07	0.99	0.00	0.08	0.99	0.07	0.00	1.89	0.10	1.01	0.00	2.95	1.00	1.04	1.00	0.16	1.94	0.16	1.01	0.04	1.02	2.99	0.07	0.07	1.95	0.11	1.04	2.01	0.97	1.02	0.00	0.10	0.96	0.96	0.00	1.01	0.06	0.98	0.01	0.97	0.09	0.09	0.99	0.13	1.00	2.07	0.09	1.95	0.09	0.15	2.04	0.11	0.08	1.00	0.05	0.07	1.01	0.99	0.04	0.12	4.86	0.11	1.02	2.01	0.03	2.01	0.07	2.00	0.11	0.97	0.00	0.98	0.08	1.02	0.03	0.98	0.13	0.07	0.99	0.06	1.02	0.08	2.04	3.96	0.16	1.03	1.01	0.20	0.01	0.96	0.13	1.96	0.06	0.13	1.01	0.11	0.07	1.01	0.12	0.99	1.03	0.18	2.01	0.15	2.01	2.96	0.05	0.06	1.96	1.03	0.06	0.06	0.95	0.13	1.02	0.14	0.99	0.16	0.12	2.08	1.06	2.03	1.06	2.06	0.13	0.98	0.14	0.94	0.06	0.12	0.96	2.03	0.10	0.11	2.04	1.07	0.11	0.98	0.15	0.18	0.98	0.10	0.10	1.99	0.18	0.97	1.98	0.07	0.18	0.96	0.10	0.12	1.98	0.16	1.96	1.03	1.01	0.04	0.05	1.02	0.09	0.07	0.94	0.08	0.07	1.01	0.06	3.90	0.13	0.09	0.96	0.15	0.11	1.00	0.09	1.06	1.01	0.12	0.94	0.09	0.09	0.99	1.95	2.04	1.06	0.18	2.97	0.15	0.16	2.00	0.16	1.04	0.06	2.99	2.06	0.11	0.04	0.89	1.99	0.11	0.04	2.13	3.03	2.03	0.08	0.10	0.98	0.08	0.10	1.05	0.10	0.14	0.94	0.06	0.14	1.07	1.96	4.14	1.17	0.20	0.02	1.95	0.97	0.09	0.07	1.06	0.95	2.02	0.11	0.19	0.97	0.12	0.03	1.05	0.16	1.06	2.92	1.10	0.97	0.09	0.00	1.99	2.01	0.21	0.12	0.95	0.01	0.95	0.09	2.99	0.13	0.92	0.17	2.95	0.12	0.98	1.01	0.14	2.88	0.21	1.80	0.13	0.09	1.04	0.06	1.08	0.96	0.07	0.19	1.04	0.13	3.84	1.19	0.08	0.89	0.11	0.91	2.03	0.97	0.15	0.02	0.97	0.00	1.00	2.08	0.18	0.13	1.07	0.01	0.03	1.05	0.11	2.00	2.13	0.14	0.12	1.96	0.17	0.94	0.15	0.18	0.95	0.96	1.08	1.06	1.07	0.11	0.16	1.02	1.97	1.03	1.02	0.10	0.97	0.09	0.90	0.20	0.16	0.91	3.05	1.08	1.10	0.00	0.14	0.90	0.10	0.02	0.99	0.94	0.19	0.00	2.04	1.85	1.15	0.01	0.88	0.95	0.04	0.07	2.02	0.13	0.15	0.99	0.18	0.15	1.09	0.03	0.98	0.14	0.00	0.87	1.02	0.10	2.13	1.02	0.06	0.06",
    "	0.99	0.00	0.99	0.01	0.00	1.02	0.01	1.00	1.02	1.12	0.00	0.01	1.93	0.01	0.98	1.00	0.95	0.00	0.02	0.97	0.99	0.00	0.92	0.07	0.05	1.01	0.04	1.04	0.04	0.00	0.98	0.06	0.00	1.09	0.06	0.01	0.96	0.01	0.01	0.99	0.06	0.00	0.97	0.06	0.98	0.00	0.07	0.99	0.07	0.00	1.86	0.07	0.97	0.00	3.01	0.97	1.06	1.07	0.17	2.02	0.15	1.07	0.03	0.97	2.98	0.09	0.07	1.91	0.18	0.97	1.88	1.14	0.93	0.01	0.16	1.03	1.01	0.00	0.92	0.10	1.00	0.00	0.99	0.07	0.11	0.97	0.06	0.98	1.02	0.12	2.91	0.14	0.15	2.02	0.23	0.06	0.96	0.11	0.05	0.98	1.06	0.06	0.12	4.94	0.12	0.97	1.95	0.23	1.96	0.04	1.99	0.15	0.99	0.02	0.99	0.06	0.99	0.10	1.09	0.11	0.12	0.95	0.07	0.99	0.12	1.96	3.87	0.16	0.97	1.03	0.11	0.11	0.99	0.13	1.97	0.12	0.18	1.04	0.13	0.07	1.01	0.03	1.00	0.96	0.03	1.99	0.13	1.98	2.93	0.08	0.07	1.96	1.06	0.09	0.06	1.02	0.19	0.94	0.07	1.01	0.10	0.14	1.98	1.05	2.06	0.96	1.95	0.14	0.98	0.12	1.00	0.13	0.12	0.99	2.01	0.11	0.12	2.03	1.10	0.13	0.98	0.04	0.09	0.95	0.16	0.05	2.01	0.14	0.98	1.99	0.09	0.07	0.96	0.03	0.18	1.98	0.12	1.98	1.03	0.97	0.08	0.10	1.02	0.10	0.13	0.98	0.05	0.10	0.98	0.13	3.85	0.14	0.11	0.95	0.19	0.09	1.01	0.12	1.10	0.97	0.15	1.03	0.07	0.11	1.00	1.90	1.99	0.98	0.06	2.85	0.13	0.13	1.88	0.07	1.00	0.08	2.85	2.03	0.12	0.06	1.00	1.90	1.02	0.14	0.10	2.04	0.07	0.99	0.16	0.95	2.00	0.08	0.09	1.03	0.16	0.13	1.01	0.12	0.18	0.96	0.05	0.08	1.01	2.00	1.03	0.10	1.00	0.17	1.02	0.07	0.13	1.05	0.05	0.96	0.15	0.17	2.04	1.01	0.08	0.09	1.01	0.97	2.09	0.21	0.07	1.04	0.07	0.00	1.03	0.06	1.00	3.03	1.06	0.96	0.13	0.05	2.03	1.97	0.12	0.13	1.03	0.05	0.95	0.10	2.85	0.20	0.98	0.09	3.01	0.01	0.99	0.94	0.19	2.97	0.07	1.91	0.16	0.13	0.98	0.06	0.98	1.05	0.03	0.05	1.04	0.00	3.88	1.07	0.09	0.97	0.06	1.03	1.91	0.94	0.13	0.00	1.06	0.08	0.99	2.05	0.18	0.02	1.00	0.02	0.03	0.99	0.11	1.87	2.09	0.02	0.16	2.02	0.07	0.92	0.11	0.17	1.06	0.96	1.07	1.08	1.01	0.13	0.19	0.98	1.96	1.02	1.05	0.13	0.98	0.11	0.84	0.10	0.02	0.93	3.00	1.03	1.12	0.00	0.13	1.00	0.14	0.05	1.02	0.94	0.08	0.05	1.86	1.81	1.08	0.07	0.87	0.97	0.09	0.10	1.97	0.14	0.26	0.94	0.26	0.12",
    "	0.99	0.00	0.99	0.03	0.00	1.01	0.03	1.01	0.97	1.12	0.03	0.01	2.01	0.00	0.97	0.99	0.98	0.00	0.01	1.00	0.99	0.00	0.98	0.04	0.00	1.03	0.07	0.98	0.00	0.02	0.99	0.04	0.00	1.04	0.04	0.04	0.99	0.00	0.04	0.98	0.05	0.00	0.97	0.07	0.99	0.00	0.08	0.99	0.04	0.00	2.03	0.09	1.03	0.02	2.92	1.02	0.99	1.05	0.10	1.98	0.13	0.97	0.04	0.99	3.00	0.06	0.08	2.01	0.10	1.04	2.00	0.99	1.00	0.00	0.09	0.98	0.99	0.00	1.00	0.07	0.98	0.02	0.99	0.09	0.09	0.99	0.13	1.01	2.05	0.09	2.01	0.09	0.09	2.01	0.12	0.08	1.01	0.05	0.05	1.02	0.98	0.04	0.11	4.88	0.15	1.02	1.99	0.01	2.03	0.06	1.95	0.09	0.95	0.01	0.97	0.06	1.03	0.05	1.00	0.09	0.06	1.00	0.05	1.02	0.08	1.96	3.92	0.13	1.01	1.00	0.12	0.03	1.00	0.11	1.95	0.07	0.11	1.00	0.09	0.09	1.01	0.14	1.04	0.98	0.14	2.02	0.16	1.94	2.99	0.03	0.05	1.92	1.02	0.07	0.05	0.95	0.12	0.99	0.16	1.01	0.11	0.12	2.09	1.03	2.11	1.10	2.03	0.16	0.99	0.11	0.94	0.07	0.10	0.96	1.96	0.09	0.08	1.97	1.00	0.11	1.03	0.14	0.13	1.01	0.16	0.06	2.03	0.15	0.99	1.97	0.12	0.11	0.96	0.09	0.13	1.97	0.14	1.93	1.01	0.96	0.03	0.05	1.00	0.18	0.06	0.99	0.00	0.07	1.04	0.14	3.80	0.14	0.16	0.99	0.14	0.08	0.99	0.08	1.06	1.01	0.15	0.94	0.17	0.10	1.00	1.86	1.96	1.01	0.13	2.78	0.18	0.15	1.92	0.06	1.00	0.08	2.78	1.86	0.18	0.06	0.95	1.92	1.02	0.12	0.05	1.97	0.07	1.49	0.06	0.53	1.94	0.07	0.12	0.97	0.11	0.12	1.01	0.08	0.14	0.98	0.12	0.08	1.01	2.00	1.02	0.05	0.99	0.13	1.03	0.11	0.09	1.02	0.02	1.00	0.12	0.17	2.10	1.00	0.07	0.06	1.02	0.98	2.04	0.13	0.17	1.00	0.16	0.00	1.06	0.15	1.08	3.03	1.01	0.94	0.09	0.08	2.03	2.04	0.18	0.13	0.96	0.07	1.01	0.09	2.88	0.16	0.96	0.14	2.92	0.02	1.03	0.95	0.21	2.86	0.19	1.86	0.19	0.12	1.01	0.13	1.03	1.01	0.00	0.18	1.02	0.12	3.87	1.11	0.05	0.91	0.16	1.00	2.00	0.99	0.18	0.21	1.03	0.07	0.99	2.02	0.15	0.12	1.02	0.03	0.01	1.00	0.13	1.95	2.12	0.14	0.13	2.03	0.16	0.91	0.16	0.13	1.03	0.99	1.11	1.07	0.96	0.11	0.11	1.01	1.95	1.00	1.03	0.11	1.02	0.10	0.88	0.11	0.16	0.96	3.02	1.04	1.09	0.02	0.09	0.96	0.05	0.05	1.00	0.93	0.20	0.03	1.89	1.98	1.08	0.05	0.89	0.97	0.04	0.08	1.94	0.12	0.19	0.98	0.18	0.15",
    "	0.99	0.00	1.00	0.05	0.02	1.01	0.03	1.02	1.00	1.09	0.03	0.05	1.94	0.00	0.99	0.99	1.05	0.00	0.01	1.00	1.00	0.01	0.96	0.09	0.05	0.98	0.07	0.97	0.01	0.00	0.98	0.07	0.03	1.07	0.04	0.04	0.99	0.00	0.04	0.98	0.06	0.00	0.99	0.08	0.99	0.00	0.09	0.98	0.07	0.00	1.94	0.09	0.99	0.04	2.90	0.98	1.00	1.07	0.12	1.93	0.13	1.02	0.05	1.01	2.97	0.07	0.09	2.01	0.15	1.03	1.97	0.96	1.03	0.05	0.13	1.01	0.99	0.00	1.00	0.09	0.97	0.02	0.98	0.10	0.12	0.98	0.15	0.98	2.06	0.10	2.02	0.09	0.10	2.06	0.12	0.09	1.02	0.06	0.05	1.00	0.99	0.04	0.12	4.78	0.19	1.05	2.05	0.06	2.00	0.06	1.92	0.11	1.02	0.01	0.92	0.08	1.00	0.04	0.96	0.13	0.06	1.03	0.07	1.00	0.10	2.09	4.03	0.15	1.03	0.98	0.15	0.03	1.01	0.12	1.93	0.05	0.12	1.05	0.12	0.09	0.99	0.13	0.98	1.04	0.05	2.04	0.15	1.94	3.07	0.06	0.06	1.98	1.00	0.08	0.05	0.95	0.12	1.03	0.16	0.96	0.17	0.11	2.11	1.09	2.11	1.04	2.03	0.12	0.97	0.09	0.97	0.06	0.13	0.97	1.96	0.09	0.09	2.04	1.02	0.14	1.02	0.00	0.20	0.99	0.11	0.11	1.99	0.15	1.00	2.03	0.07	0.18	0.94	0.08	0.17	2.02	0.15	2.02	1.04	0.99	0.04	0.01	0.99	0.16	0.11	0.96	0.06	0.06	1.07	0.09	3.88	0.12	0.09	0.95	0.13	0.09	0.99	0.15	1.02	1.03	0.12	0.99	0.20	0.13	0.95	1.98	1.98	1.05	0.19	2.91	0.18	0.15	1.97	0.02	1.02	0.06	2.95	2.02	0.16	0.04	0.94	1.93	1.05	0.11	0.12	1.97	0.04	1.02	0.05	0.97	2.04	0.09	0.10	0.98	0.04	0.12	1.04	0.06	0.14	0.94	0.13	0.04	1.01	1.96	1.07	0.04	1.03	0.14	1.00	0.12	0.17	1.02	0.02	0.99	0.10	0.16	2.05	0.96	0.09	0.04	1.00	0.98	2.08	0.10	0.10	1.00	0.17	0.01	0.98	0.14	1.08	3.00	1.06	0.98	0.08	0.01	2.00	2.04	0.20	0.12	0.91	0.03	1.02	0.11	2.86	0.13	0.96	0.14	2.95	0.06	1.02	1.02	0.12	3.03	0.20	1.82	0.20	0.07	1.03	0.07	0.97	0.96	0.00	0.18	1.05	0.11	4.03	1.15	0.04	0.93	0.08	0.94	1.94	0.99	0.11	0.00	0.99	0.06	1.01	2.03	0.10	0.11	1.03	0.00	0.21	1.01	0.06	1.92	2.09	0.13	0.08	1.98	0.14	0.94	0.17	0.14	1.02	0.94	1.17	1.07	0.94	0.11	0.12	0.97	2.02	1.01	1.02	0.09	0.98	0.07	0.89	0.18	0.11	0.96	2.99	1.16	1.07	0.00	0.11	0.94	0.08	0.03	0.97	0.94	0.16	0.00	1.91	1.93	1.13	0.01	0.84	0.93	0.05	0.04	1.97	0.10	0.16	1.01	0.14	0.12",
    "	0.97	0.00	1.01	0.03	0.02	1.02	0.02	0.99	0.97	1.12	0.01	0.02	1.92	0.00	0.98	0.99	1.00	0.00	0.00	1.00	1.05	0.00	0.96	0.06	0.04	0.97	0.09	0.97	0.02	0.00	0.96	0.05	0.06	1.07	0.04	0.06	0.98	0.00	0.05	0.98	0.08	0.00	0.97	0.10	0.97	0.00	0.15	0.97	0.10	0.00	1.94	0.13	1.01	0.03	2.89	1.07	1.03	1.06	0.11	1.98	0.18	0.98	0.05	0.99	2.98	0.07	0.08	2.04	0.17	1.02	1.98	1.00	1.01	0.00	0.08	0.98	0.98	0.00	0.96	0.08	0.99	0.00	0.99	0.10	0.12	1.02	0.13	0.99	2.01	0.11	2.04	0.09	0.14	2.04	0.12	0.11	0.99	0.04	0.03	1.01	1.05	0.05	0.11	4.89	0.12	0.99	2.00	0.08	2.02	0.04	1.94	0.09	0.91	0.02	1.00	0.07	1.05	0.01	0.99	0.13	0.11	1.00	0.08	1.04	0.12	2.06	3.85	0.18	1.04	0.97	0.16	0.06	0.99	0.10	1.96	0.07	0.17	1.07	0.15	0.10	0.97	0.12	1.03	0.98	0.01	2.10	0.17	1.96	3.06	0.07	0.06	2.04	1.01	0.08	0.08	0.93	0.17	1.03	0.06	0.93	0.16	0.12	1.97	1.16	1.98	1.07	1.94	0.18	0.96	0.12	1.01	0.04	0.13	0.96	1.94	0.10	0.13	1.98	1.03	0.15	0.97	0.11	0.17	1.00	0.14	0.12	2.01	0.15	0.97	2.04	0.13	0.17	0.96	0.13	0.16	2.00	0.18	2.03	1.01	0.99	0.09	0.02	0.98	0.11	0.10	0.96	0.02	0.09	1.04	0.14	3.92	0.16	0.18	1.00	0.18	0.13	0.98	0.13	1.04	1.08	0.12	0.87	0.07	0.11	0.99	1.90	2.04	1.03	0.19	2.95	0.16	0.14	1.83	0.00	1.06	0.05	2.71	2.09	0.18	0.05	0.96	1.98	1.03	0.14	0.10	1.95	0.09	0.99	0.10	0.98	1.96	0.09	0.17	0.96	0.10	0.13	1.05	0.13	0.17	0.97	0.12	0.05	1.05	2.03	1.01	0.04	0.97	0.14	1.02	0.12	0.16	1.03	0.01	0.99	0.12	0.15	2.03	1.01	0.07	0.09	1.01	1.01	2.05	0.12	0.11	1.05	0.19	0.00	1.03	0.14	1.04	3.05	1.05	1.00	0.10	0.02	2.01	2.13	0.09	0.09	0.90	0.02	0.99	0.13	2.91	0.17	0.93	0.19	3.04	0.05	0.97	0.98	0.16	2.95	0.12	1.86	0.15	0.08	1.05	0.06	1.01	1.01	0.00	0.06	1.13	0.12	3.98	1.08	0.03	0.90	0.07	0.90	1.99	0.94	0.13	0.00	1.05	0.05	0.99	2.09	0.15	0.13	1.14	0.00	0.04	1.03	0.08	1.90	2.05	0.15	0.11	1.96	0.19	0.94	0.17	0.20	1.03	0.95	1.08	1.14	0.94	0.11	0.17	0.95	1.90	1.06	1.07	0.13	0.96	0.09	0.87	0.22	0.22	0.99	2.95	1.06	1.13	0.03	0.12	0.94	0.10	0.01	0.94	0.96	0.13	0.01	2.03	1.89	1.05	0.00	0.82	0.99	0.04	0.05	2.07	0.14	0.28	0.93	0.16	0.15",
    "	0.97	0.00	1.00	0.04	0.01	1.03	0.04	0.92	1.00	1.13	0.02	0.00	1.97	0.00	0.97	0.99	1.00	0.00	0.03	1.00	1.02	0.00	0.96	0.05	0.02	1.04	0.02	0.98	0.01	0.00	0.97	0.05	0.00	1.05	0.06	0.02	0.98	0.01	0.03	0.98	0.04	0.00	0.98	0.06	1.00	0.00	0.07	0.99	0.07	0.00	1.91	0.09	0.99	0.01	2.98	0.99	1.05	1.04	0.14	1.99	0.14	1.01	0.06	0.96	2.97	0.08	0.08	2.01	0.14	1.01	1.94	1.00	1.01	0.03	0.11	1.00	0.98	0.00	0.98	0.08	0.98	0.02	0.99	0.08	0.11	0.99	0.14	1.01	2.03	0.09	1.99	0.11	0.13	1.99	0.16	0.08	1.01	0.11	0.06	0.97	1.02	0.03	0.12	4.85	0.17	1.00	1.98	0.07	1.98	0.07	1.91	0.11	0.98	0.01	0.95	0.07	1.02	0.07	1.04	0.10	0.11	0.96	0.04	1.05	0.10	1.98	3.84	0.17	0.99	0.98	0.15	0.07	1.00	0.11	1.99	0.08	0.12	1.04	0.12	0.07	0.99	0.15	1.01	0.99	0.17	2.10	0.15	2.02	3.02	0.09	0.08	1.95	0.99	0.09	0.06	0.94	0.12	1.01	0.13	0.96	0.16	0.13	2.02	1.11	2.16	1.07	1.92	0.18	1.01	0.15	0.95	0.09	0.11	0.95	2.00	0.11	0.07	1.97	1.04	0.13	0.98	0.00	0.21	0.99	0.17	0.11	2.05	0.12	1.02	2.03	0.09	0.13	0.98	0.09	0.15	1.99	0.15	1.99	0.95	0.99	0.05	0.03	1.03	0.18	0.07	1.00	0.06	0.07	1.00	0.07	3.89	0.14	0.16	0.95	0.12	0.11	0.99	0.14	1.05	1.04	0.14	0.96	0.07	0.11	0.99	1.81	2.00	1.02	0.03	2.83	0.12	0.14	1.88	0.02	1.05	0.06	2.87	2.08	0.17	0.04	0.88	1.84	0.12	0.04	2.05	2.96	2.01	0.07	0.16	1.01	0.08	0.10	1.06	0.07	0.17	0.97	0.00	0.17	1.07	2.02	4.02	1.05	0.13	0.04	1.97	0.97	0.04	0.06	1.06	0.99	2.01	0.09	0.19	0.94	0.16	0.00	1.05	0.06	1.06	2.91	1.07	0.99	0.07	0.02	1.97	2.01	0.20	0.10	0.88	0.05	0.99	0.07	2.92	0.15	0.99	0.15	2.93	0.09	0.94	1.00	0.13	2.95	0.20	1.94	0.07	0.10	1.02	0.03	1.08	0.99	0.01	0.17	1.03	0.10	3.86	1.14	0.00	0.90	0.07	0.96	2.03	0.99	0.22	0.18	0.98	0.03	0.98	2.10	0.10	0.13	1.06	0.20	0.06	1.01	0.05	1.99	2.07	0.12	0.13	1.98	0.15	0.93	0.18	0.10	1.03	0.94	1.10	1.05	1.00	0.10	0.16	0.98	1.98	1.01	1.12	0.09	0.93	0.06	0.94	0.06	0.14	0.97	3.00	1.02	1.09	0.00	0.09	0.93	0.02	0.04	1.02	0.93	0.08	0.02	2.09	1.93	1.11	0.04	0.84	0.93	0.05	0.10	1.99	0.08	0.15	1.03	0.14	0.11	1.10	0.07	1.06	0.10	0.21	0.91	0.97	0.08	2.15	1.07	0.06	0.08",
    "	0.99	0.00	0.99	0.02	0.00	1.02	0.01	0.96	0.98	1.13	0.03	0.00	1.97	0.04	0.96	0.98	1.03	0.00	0.03	0.96	1.03	0.00	0.91	0.07	0.00	1.03	0.06	1.01	0.01	0.00	0.97	0.03	0.00	1.07	0.03	0.01	0.99	0.00	0.01	0.99	0.03	0.00	0.97	0.06	0.98	0.00	0.08	0.99	0.04	0.00	1.92	0.09	0.99	0.00	3.06	1.00	1.03	1.05	0.13	2.05	0.14	1.05	0.04	0.98	3.06	0.05	0.07	2.03	0.15	0.99	1.97	0.98	1.02	0.02	0.12	1.03	0.97	0.00	1.00	0.09	1.01	0.02	0.96	0.10	0.12	1.00	0.13	1.00	2.05	0.10	1.99	0.09	0.11	2.04	0.15	0.09	1.00	0.08	0.05	1.03	1.00	0.04	0.12	4.80	0.09	0.98	2.02	0.13	2.00	0.05	1.92	0.11	0.95	0.03	1.00	0.07	1.04	0.03	1.02	0.11	0.13	0.99	0.09	1.02	0.09	1.99	3.90	0.17	0.97	0.99	0.12	0.09	0.99	0.11	1.99	0.08	0.15	1.03	0.15	0.10	1.00	0.12	1.00	1.08	0.05	1.97	0.17	1.96	2.92	0.04	0.09	1.91	0.97	0.08	0.08	0.98	0.15	0.98	0.13	0.96	0.15	0.15	1.97	1.11	2.04	1.07	1.90	0.14	0.94	0.08	1.07	0.12	0.09	0.96	1.99	0.12	0.14	1.91	0.99	0.19	0.99	0.02	0.15	0.97	0.11	0.08	2.08	0.16	1.01	2.02	0.11	0.09	0.91	0.15	0.17	1.90	0.09	1.95	0.99	0.96	0.05	0.03	1.09	0.08	0.08	1.01	0.05	0.10	1.01	0.07	3.78	0.23	0.10	0.97	0.14	0.14	0.95	0.14	1.07	1.00	0.11	0.94	0.07	0.09	1.03	1.89	1.94	1.02	0.17	2.85	0.10	0.15	1.90	0.01	1.04	0.09	2.76	1.97	0.17	0.06	1.03	2.03	0.60	0.08	0.47	0.97	0.52	0.57	0.17	1.97	2.14	0.16	0.16	0.93	0.09	0.17	1.03	0.06	0.13	0.96	0.18	0.10	1.01	1.99	1.09	0.09	0.96	0.15	1.00	0.08	0.08	1.07	0.00	0.98	0.18	0.18	2.09	1.03	0.11	0.11	0.98	0.98	2.06	0.23	0.06	1.01	0.10	0.01	1.04	0.03	1.10	3.07	0.99	1.00	0.13	0.02	2.03	2.06	0.12	0.13	0.96	0.09	0.96	0.13	2.90	0.17	0.95	0.17	2.91	0.07	0.98	0.95	0.14	2.95	0.10	1.79	0.14	0.05	1.05	0.04	1.02	1.01	0.00	0.09	1.13	0.15	3.99	1.08	0.05	0.92	0.09	0.94	2.06	0.97	0.12	0.05	0.96	0.04	1.01	2.03	0.15	0.15	0.98	0.03	0.04	1.10	0.10	1.97	2.07	0.12	0.17	1.86	0.15	0.97	0.10	0.19	1.06	0.90	1.17	1.05	0.96	0.12	0.18	0.98	1.89	0.99	1.12	0.11	0.92	0.12	0.91	0.06	0.17	0.95	2.90	1.09	1.03	0.00	0.12	0.93	0.07	0.04	1.09	0.90	0.09	0.03	1.93	1.91	1.09	0.06	0.88	0.90	0.07	0.13	1.90	0.13	0.22	1.03	0.19	0.14",
    "	0.99	0.00	1.01	0.04	0.02	1.00	0.03	0.96	0.98	1.11	0.02	0.01	2.03	0.00	0.99	1.00	1.03	0.00	0.03	1.02	1.01	0.00	0.95	0.05	0.02	0.99	0.00	1.00	0.01	0.00	0.97	0.05	0.00	1.04	0.05	0.04	0.96	0.00	0.02	0.98	0.03	0.00	0.98	0.05	1.01	0.00	0.07	0.97	0.05	0.00	1.97	0.08	1.02	0.00	2.94	1.01	1.03	1.02	0.12	2.01	0.12	0.98	0.05	0.99	3.07	0.05	0.06	1.99	0.13	1.04	2.09	0.96	1.01	0.03	0.07	1.00	0.95	0.00	1.00	0.05	0.98	0.00	0.98	0.08	0.10	1.01	0.11	1.03	2.06	0.06	2.02	0.08	0.12	2.00	0.09	0.10	1.00	0.02	0.04	1.00	0.99	0.02	0.12	4.87	0.16	1.02	2.09	0.07	2.04	0.06	1.92	0.05	0.96	0.00	0.97	0.08	1.03	0.00	0.97	0.13	0.07	1.00	0.08	1.07	0.06	2.01	4.00	0.14	1.01	0.97	0.19	0.00	0.99	0.11	1.96	0.03	0.13	1.03	0.13	0.07	0.99	0.15	1.01	1.02	0.05	2.03	0.15	1.98	2.99	0.15	0.04	1.97	1.01	0.09	0.02	0.97	0.11	1.01	0.12	0.98	0.15	0.11	2.03	1.04	2.02	1.04	2.08	0.09	0.99	0.11	1.01	0.03	0.13	0.98	1.97	0.07	0.07	1.97	0.97	0.14	1.00	0.14	0.17	0.99	0.15	0.07	2.00	0.15	0.98	1.94	0.10	0.16	0.97	0.12	0.12	1.99	0.16	1.90	1.04	1.03	0.02	0.07	1.00	0.17	0.10	0.92	0.06	0.04	1.02	0.08	3.90	0.07	0.18	0.96	0.14	0.10	1.02	0.08	1.09	1.02	0.18	0.95	0.09	0.14	0.94	1.98	2.03	1.03	0.18	2.88	0.18	0.15	1.96	0.09	1.03	0.05	2.98	1.99	0.19	0.03	0.95	2.01	1.03	0.11	0.18	1.95	0.10	0.97	0.07	0.99	1.98	0.16	0.11	1.07	0.12	0.11	1.01	0.08	0.15	0.95	0.15	0.04	1.02	2.03	1.02	0.04	1.00	0.17	1.03	0.12	0.10	0.99	0.07	0.97	0.10	0.17	2.04	1.01	0.08	0.07	1.01	1.02	2.05	0.11	0.15	1.02	0.19	0.00	1.02	0.17	1.06	2.92	1.00	0.96	0.20	0.03	1.99	2.06	0.18	0.09	1.01	0.07	1.01	0.08	2.82	0.12	0.96	0.15	2.94	0.06	1.00	0.97	0.20	2.93	0.12	1.84	0.15	0.11	1.01	0.08	1.01	0.98	0.03	0.15	1.08	0.15	4.08	1.11	0.11	0.91	0.10	0.94	2.02	1.04	0.13	0.02	1.01	0.04	0.98	2.06	0.09	0.12	1.00	0.05	0.04	1.01	0.11	1.97	2.08	0.17	0.13	2.00	0.17	0.96	0.18	0.20	1.03	1.00	0.99	1.07	0.98	0.09	0.14	1.03	1.92	1.03	1.00	0.09	0.97	0.09	0.97	0.18	0.22	0.93	3.00	1.13	1.07	0.14	0.20	0.87	0.08	0.02	0.94	0.97	0.20	0.00	1.95	1.96	1.10	0.00	0.90	0.97	0.06	0.07	2.01	0.14	0.16	0.91	0.22	0.13",
    "	0.98	0.00	1.01	0.03	0.00	1.01	0.04	1.01	0.95	1.15	0.02	0.01	2.01	0.00	0.96	0.99	1.01	0.00	0.02	0.98	0.99	0.00	0.98	0.06	0.01	1.03	0.04	1.02	0.04	0.00	0.93	0.04	0.01	1.07	0.04	0.03	0.99	0.00	0.02	1.00	0.05	0.00	0.99	0.06	0.99	0.00	0.08	0.96	0.07	0.00	1.93	0.08	0.98	0.00	3.04	1.01	1.02	1.08	0.10	2.01	0.16	1.01	0.04	0.97	3.01	0.08	0.08	1.99	0.14	1.00	1.97	1.02	1.03	0.02	0.14	0.97	0.97	0.00	0.97	0.09	1.02	0.01	1.00	0.11	0.15	0.99	0.13	0.99	2.04	0.11	1.99	0.12	0.15	2.00	0.15	0.09	0.99	0.11	0.03	1.03	1.00	0.03	0.12	4.82	0.08	0.99	1.99	0.14	2.00	0.04	1.96	0.12	0.97	0.01	0.98	0.07	1.02	0.03	1.05	0.12	0.10	0.97	0.07	1.03	0.09	2.03	3.97	0.16	0.99	1.03	0.12	0.04	0.98	0.09	2.06	0.04	0.13	1.01	0.13	0.08	0.99	0.12	1.03	0.99	0.01	2.06	0.15	1.97	3.03	0.05	0.09	1.94	1.01	0.07	0.06	0.96	0.15	0.98	0.15	0.99	0.09	0.12	2.09	1.06	2.01	1.05	2.06	0.11	0.97	0.13	1.02	0.07	0.14	1.01	2.03	0.09	0.10	1.96	1.00	0.14	0.97	0.12	0.06	0.98	0.13	0.12	2.05	0.17	1.01	2.04	0.09	0.07	0.95	0.01	0.13	1.95	0.15	2.01	1.01	0.95	0.04	0.04	1.06	0.18	0.11	0.99	0.07	0.08	1.02	0.10	3.93	0.16	0.11	0.95	0.13	0.10	1.00	0.13	1.07	1.03	0.11	0.96	0.12	0.13	0.95	1.92	1.97	1.04	0.02	2.83	0.14	0.15	1.91	0.09	1.02	0.09	2.81	2.03	0.08	0.05	0.99	2.00	0.11	0.04	0.99	0.03	0.97	0.04	0.17	3.06	1.94	0.06	0.15	0.93	0.11	0.14	1.02	0.06	0.17	0.95	0.16	0.08	1.06	1.93	1.03	0.10	0.97	0.16	1.06	0.15	0.18	1.01	0.03	1.00	0.14	0.16	2.01	0.99	0.06	0.04	1.02	0.98	2.08	0.12	0.18	1.02	0.17	0.04	1.00	0.17	1.05	3.02	1.05	0.96	0.10	0.03	1.98	2.03	0.12	0.11	1.00	0.01	0.98	0.07	2.87	0.15	0.97	0.15	2.91	0.07	0.95	1.00	0.14	2.94	0.10	1.86	0.17	0.11	1.03	0.08	1.03	1.01	0.00	0.19	1.06	0.16	4.00	1.10	0.02	0.92	0.10	0.98	1.99	0.94	0.13	0.02	1.00	0.04	0.97	2.03	0.15	0.15	1.08	0.00	0.07	1.03	0.09	1.94	2.07	0.16	0.14	1.94	0.16	0.91	0.18	0.19	1.06	0.97	1.13	1.05	1.01	0.10	0.19	0.98	1.95	1.04	0.97	0.09	0.98	0.11	0.91	0.07	0.19	0.99	2.93	1.07	1.08	0.00	0.15	0.91	0.10	0.04	1.00	0.94	0.08	0.03	1.93	1.89	1.12	0.02	0.86	0.98	0.04	0.09	1.96	0.16	0.23	0.92	0.23	0.15",
    "	1.00	0.00	0.98	0.01	0.01	1.01	0.01	1.02	1.02	1.09	0.03	0.00	2.00	0.00	0.93	1.00	1.04	0.00	0.00	0.98	0.98	0.00	0.99	0.01	0.01	0.97	0.05	1.00	0.00	0.00	0.95	0.03	0.03	1.06	0.04	0.02	0.98	0.00	0.03	0.97	0.09	0.00	0.96	0.07	1.00	0.00	0.13	0.97	0.09	0.00	1.97	0.10	0.98	0.01	2.91	1.02	1.05	1.02	0.14	2.02	0.13	1.00	0.03	1.08	2.91	0.08	0.10	1.93	0.18	1.00	1.94	0.98	1.02	0.00	0.12	0.99	0.97	0.00	1.00	0.08	1.00	0.02	0.94	0.10	0.18	0.96	0.12	0.98	2.00	0.14	1.99	0.09	0.16	2.02	0.16	0.11	1.05	0.05	0.07	0.99	0.99	0.07	0.16	4.86	0.17	0.97	1.91	0.11	1.97	0.02	1.99	0.13	0.97	0.00	0.98	0.06	1.05	0.02	1.05	0.12	0.08	1.00	0.09	1.05	0.12	2.02	3.94	0.19	0.99	0.93	0.10	0.04	0.99	0.12	1.99	0.06	0.16	1.05	0.10	0.12	1.01	0.16	0.99	0.99	0.05	2.12	0.11	2.06	3.05	0.07	0.06	1.97	1.01	0.11	0.07	0.92	0.16	1.01	0.14	0.94	0.19	0.09	1.99	1.10	2.09	1.08	2.03	0.15	1.05	0.13	0.99	0.09	0.10	0.94	2.04	0.09	0.16	1.99	1.00	0.13	0.96	0.12	0.09	0.98	0.16	0.10	2.10	0.11	0.97	2.01	0.12	0.16	0.95	0.07	0.14	2.03	0.17	2.04	1.05	0.97	0.02	0.10	0.98	0.11	0.11	1.04	0.00	0.05	0.96	0.10	3.88	0.16	0.19	1.02	0.20	0.13	0.96	0.10	1.08	1.06	0.20	0.91	0.08	0.12	0.99	1.92	1.92	0.99	0.02	2.95	0.14	0.10	1.89	0.01	1.01	0.04	2.75	2.04	0.17	0.03	1.01	1.84	1.03	0.16	0.16	1.97	0.00	1.02	0.13	0.97	1.96	0.12	0.13	1.03	0.07	0.08	1.01	0.11	0.17	0.95	0.14	0.00	1.07	2.11	0.98	0.04	0.92	0.10	1.05	0.14	0.16	1.04	0.01	0.99	0.10	0.16	2.05	1.00	0.03	0.00	1.02	0.99	2.11	0.11	0.18	1.10	0.09	0.00	1.05	0.12	1.12	3.19	0.96	0.94	0.21	0.00	2.13	2.21	0.20	0.09	0.89	0.03	1.00	0.11	2.97	0.13	1.00	0.16	3.05	0.03	1.00	0.98	0.09	3.13	0.15	1.95	2.09	0.97	0.00	0.13	1.10	0.15	3.97	1.08	0.06	0.94	0.06	0.99	2.04	1.09	0.22	0.18	1.00	0.17	0.99	2.09	0.18	0.12	0.90	0.24	0.03	1.09	0.21	2.05	2.15	0.12	0.13	1.99	0.20	0.96	0.16	0.18	1.02	0.99	1.20	0.93	0.92	0.12	0.10	1.04	1.88	1.06	1.05	0.11	1.06	0.11	0.78	0.11	0.21	1.01	2.93	1.04	1.06	0.12	0.10	0.90	0.22	0.03	1.06	0.93	0.19	0.00	1.88	1.90	1.09	0.00	0.83	0.96	0.10	0.06	1.93	0.07	0.20	0.93	0.22	0.07	1.16	0.10	1.43	0.08",
    "	0.99	0.00	1.00	0.06	0.03	1.01	0.05	0.90	0.95	1.12	0.03	0.00	2.03	0.00	1.01	0.95	1.06	0.00	0.00	1.01	1.01	0.00	1.01	0.01	0.00	1.02	0.00	0.98	0.04	0.00	0.99	0.07	0.00	1.02	0.06	0.02	0.97	0.01	0.01	1.01	0.04	0.00	1.00	0.05	1.00	0.00	0.03	1.00	0.05	0.00	1.85	0.08	0.94	0.00	3.00	0.98	1.05	1.01	0.16	2.00	0.10	1.02	0.04	1.01	3.08	0.09	0.08	1.96	0.14	1.03	2.02	0.96	1.02	0.04	0.10	0.99	0.96	0.00	0.99	0.09	1.00	0.01	0.99	0.10	0.12	0.98	0.12	1.00	2.03	0.09	1.96	0.11	0.12	1.97	0.11	0.09	0.96	0.05	0.06	1.05	0.99	0.05	0.14	4.86	0.17	1.05	2.03	0.05	2.03	0.08	1.96	0.09	0.99	0.00	0.91	0.10	0.99	0.02	0.99	0.14	0.07	0.99	0.07	1.04	0.09	2.06	4.02	0.11	1.02	1.02	0.15	0.00	1.00	0.12	1.96	0.04	0.13	0.99	0.10	0.08	0.98	0.05	1.02	1.00	0.04	2.01	0.16	2.00	2.98	0.05	0.06	1.94	1.02	0.09	0.01	0.96	0.09	1.02	0.12	1.03	0.13	0.15	2.19	0.96	2.14	1.06	2.08	0.09	0.98	0.15	1.00	0.04	0.09	1.04	1.99	0.07	0.06	2.03	0.96	0.15	0.96	0.14	0.20	1.02	0.14	0.07	1.97	0.17	0.99	1.95	0.11	0.17	0.97	0.13	0.11	2.03	0.12	1.86	1.07	0.97	0.01	0.07	1.02	0.15	0.10	0.95	0.10	0.04	1.03	0.11	3.81	0.10	0.13	0.93	0.15	0.11	1.01	0.08	1.08	1.01	0.18	0.91	0.20	0.12	1.00	1.95	2.00	1.05	0.04	2.86	0.16	0.15	1.98	0.05	0.99	0.05	2.92	2.02	0.14	0.03	0.97	2.05	0.60	0.09	0.48	1.00	0.48	0.46	0.17	1.99	2.08	0.11	0.13	1.01	0.06	0.11	0.99	0.07	0.16	0.97	0.20	0.05	1.01	1.96	1.06	0.08	1.01	0.13	1.02	0.13	0.18	1.02	0.02	0.97	0.10	0.17	2.02	0.97	0.07	0.04	1.00	1.00	2.08	0.10	0.13	0.99	0.14	0.07	1.01	0.07	1.02	2.98	1.06	0.98	0.21	0.05	1.96	2.05	0.19	0.07	0.96	0.03	1.00	0.07	2.90	0.14	1.00	0.15	2.92	0.05	0.98	1.00	0.11	2.94	0.19	1.90	0.16	0.09	0.98	0.09	1.05	1.01	0.08	0.19	1.04	0.12	4.02	1.11	0.05	0.95	0.09	0.95	2.04	0.97	0.22	0.01	0.94	0.05	1.02	2.13	0.13	0.14	1.06	0.05	0.05	1.03	0.05	2.01	2.08	0.14	0.07	1.98	0.13	0.94	0.16	0.17	1.01	0.96	1.09	1.04	1.07	0.13	0.14	1.00	1.95	0.98	1.04	0.12	0.97	0.07	0.91	0.11	0.18	0.94	2.96	1.08	1.05	0.13	0.13	0.92	0.08	0.00	1.04	0.99	0.21	0.00	1.97	1.93	1.06	0.02	0.87	1.01	0.18	0.05	1.96	0.12	0.14	0.96	0.15	0.13",
    "	1.00	0.00	1.01	0.01	0.02	1.00	0.04	0.94	0.98	1.11	0.05	0.00	2.01	0.01	0.96	1.01	1.01	0.00	0.00	1.03	1.04	0.00	0.96	0.03	0.00	0.98	0.04	0.99	0.00	0.00	0.98	0.03	0.01	1.04	0.05	0.00	0.98	0.00	0.04	1.00	0.05	0.00	0.99	0.05	0.99	0.00	0.09	0.97	0.08	0.00	1.94	0.06	1.00	0.01	2.97	1.01	1.04	1.03	0.14	2.04	0.14	0.96	0.06	0.99	3.07	0.09	0.09	2.05	0.14	1.03	2.03	0.93	1.05	0.03	0.09	0.98	1.01	0.00	1.01	0.04	0.97	0.02	0.98	0.08	0.13	0.99	0.13	1.02	2.10	0.10	1.98	0.07	0.11	1.97	0.10	0.07	0.99	0.06	0.05	1.05	0.98	0.06	0.14	5.15	0.17	1.03	2.05	0.00	2.06	0.05	1.95	0.06	0.96	0.00	0.95	0.04	1.02	0.05	0.99	0.11	0.07	1.00	0.07	1.03	0.10	2.01	4.05	0.12	1.05	0.95	0.12	0.00	0.99	0.14	1.91	0.03	0.12	0.99	0.13	0.08	1.02	0.09	1.08	1.01	0.00	2.01	0.18	2.00	3.06	0.01	0.07	2.01	0.95	0.10	0.04	0.94	0.12	1.05	0.18	1.06	0.07	0.14	2.13	1.05	2.12	1.07	2.06	0.09	0.93	0.14	0.93	0.01	0.10	1.02	1.99	0.07	0.12	2.00	1.01	0.12	1.04	0.15	0.19	0.99	0.10	0.03	2.01	0.14	0.98	2.02	0.09	0.03	0.97	0.10	0.12	2.11	0.17	1.92	1.09	0.91	0.04	0.01	1.00	0.11	0.12	0.96	0.00	0.01	1.02	0.14	3.92	0.12	0.16	0.96	0.03	0.11	1.01	0.06	1.06	1.03	0.07	0.96	0.06	0.13	0.96	1.82	1.93	1.09	0.01	2.75	0.16	0.10	2.04	0.00	1.04	0.03	2.94	2.05	0.16	0.02	0.81	1.90	0.54	0.07	1.06	2.47	1.05	0.48	0.10	0.92	1.04	0.12	0.39	0.60	0.06	0.53	0.53	0.08	0.58	1.54	2.24	0.63	0.55	0.98	1.57	0.54	0.56	0.17	1.07	0.54	1.04	0.61	0.06	0.98	0.11	0.16	1.65	0.51	0.64	1.57	0.93	0.94	1.07	0.00	0.90	1.45	0.12	0.17	1.06	0.14	1.07	1.59	2.02	0.55	0.40	0.00	2.51	0.98	0.55	0.38	0.62	1.17	0.67	0.95	1.58	0.21	0.97	0.19	2.12	0.56	0.57	0.49	0.47	1.57	1.84	1.59	0.13	0.30	0.80	0.42	1.57	1.07	0.07	0.10	1.21	0.10	2.81	1.63	0.15	0.58	0.40	0.62	1.08	0.89	0.22	0.93	1.39	0.13	0.50	2.32	0.20	0.43	0.67	0.21	0.52	1.04	0.45	1.67	1.69	0.21	0.19	1.58	0.91	1.01	0.59	0.14	1.16	0.54	0.97	0.64	0.60	0.42	1.44	1.06	1.64	0.46	0.77	0.47	0.44	0.13	1.00	0.57	0.16	0.47	2.60	1.52	1.07	0.08	0.40	1.01	0.06	0.03	1.38	0.56	0.22	0.41	1.11	0.95	1.02	0.12	0.89	0.53	0.40	0.09	1.98	0.16	0.35	0.91	0.08	0.49",
    "	0.99	0.00	1.01	0.04	0.00	1.00	0.04	0.94	1.00	1.14	0.04	0.00	2.04	0.00	0.98	1.01	1.01	0.00	0.02	0.98	1.01	0.00	0.96	0.04	0.00	1.02	0.06	1.00	0.01	0.00	0.97	0.04	0.03	1.08	0.05	0.04	0.98	0.00	0.05	0.98	0.07	0.00	0.97	0.09	0.97	0.00	0.13	0.98	0.09	0.00	1.93	0.12	1.01	0.01	2.89	1.03	1.01	1.02	0.13	1.97	0.16	1.01	0.06	1.00	2.95	0.10	0.10	1.52	0.15	1.55	1.96	0.96	0.99	0.03	0.13	1.00	0.99	0.00	0.99	0.09	0.99	0.00	0.97	0.10	0.14	1.00	0.18	0.99	2.03	0.12	1.95	0.12	0.14	2.00	0.14	0.09	1.00	0.06	0.07	1.05	0.99	0.04	0.14	4.81	0.13	1.00	1.98	0.12	2.01	0.06	1.89	0.11	0.97	0.00	0.99	0.08	1.02	0.03	0.99	0.13	0.09	0.98	0.15	1.00	0.09	2.06	3.96	0.16	1.02	1.02	0.14	0.04	0.99	0.10	1.97	0.10	0.16	0.98	0.19	0.11	1.00	0.11	1.01	1.03	0.07	2.02	0.18	2.04	3.06	0.07	0.08	1.99	1.03	0.09	0.07	0.96	0.11	1.00	0.10	0.97	0.06	0.12	2.09	1.03	2.03	1.05	2.02	0.13	1.02	0.13	0.94	0.09	0.13	1.00	1.97	0.12	0.11	1.99	1.05	0.14	0.98	0.12	0.10	0.98	0.13	0.09	1.97	0.10	0.97	2.00	0.14	0.04	0.97	0.12	0.17	1.94	0.09	1.90	1.04	0.99	0.05	0.06	1.01	0.16	0.09	0.96	0.07	0.06	1.03	0.08	3.82	0.13	0.10	0.95	0.17	0.14	1.01	0.12	1.02	1.06	0.13	0.93	0.09	0.11	1.00	1.91	1.95	1.04	0.03	2.85	0.12	0.15	1.93	0.06	1.01	0.05	2.85	1.97	0.13	0.03	0.95	2.03	1.06	0.14	0.08	1.97	0.10	1.00	0.10	0.96	2.01	0.17	0.13	1.00	0.10	0.13	0.99	0.07	0.16	0.98	0.14	0.03	1.05	1.99	1.04	0.07	0.99	0.15	1.02	0.12	0.10	0.96	0.03	1.05	0.11	0.17	2.04	0.98	0.10	0.07	1.01	0.98	2.06	0.11	0.17	1.00	0.19	0.03	1.04	0.05	1.05	2.97	1.11	0.97	0.12	0.01	2.01	2.06	0.13	0.08	0.96	0.03	0.94	0.08	2.83	0.14	0.97	0.15	2.91	0.01	1.02	0.98	0.12	2.91	0.12	1.84	0.16	0.09	1.06	0.07	1.00	0.99	0.00	0.18	1.00	0.15	3.86	1.14	0.11	0.91	0.12	0.96	1.99	1.03	0.13	0.00	0.98	0.04	1.02	2.06	0.19	0.12	1.04	0.04	0.02	1.01	0.09	1.93	2.11	0.15	0.15	1.93	0.19	0.87	0.19	0.15	1.04	1.00	1.12	1.05	0.99	0.13	0.18	1.01	1.98	1.04	1.07	0.11	0.95	0.07	0.91	0.16	0.19	0.96	2.49	0.57	1.05	0.01	0.10	0.96	0.59	0.02	0.54	1.00	0.17	0.00	1.41	1.39	0.53	0.01	1.37	1.42	0.63	0.06	1.27	0.59	0.14	0.42	1.05	0.14",
    "	0.98	0.00	1.01	0.01	0.00	1.00	0.01	0.97	1.00	1.08	0.07	0.00	1.93	0.00	0.98	0.99	1.02	0.00	0.02	0.99	0.98	0.00	0.99	0.07	0.02	1.01	0.05	1.01	0.00	0.00	0.97	0.01	0.00	1.07	0.04	0.01	0.97	0.00	0.01	1.01	0.02	0.00	0.98	0.02	0.98	0.00	0.05	1.00	0.04	0.00	1.91	0.07	0.95	0.00	2.96	1.00	1.03	1.04	0.13	2.05	0.16	1.06	0.02	1.07	2.96	0.07	0.12	1.99	0.17	0.99	1.92	1.00	0.98	0.00	0.13	1.00	0.99	0.00	0.96	0.09	1.01	0.00	0.99	0.09	0.17	0.98	0.17	0.99	1.96	0.14	1.91	0.08	0.13	1.96	0.16	0.10	1.01	0.10	0.03	1.06	0.98	0.04	0.14	4.94	0.12	0.98	1.91	0.15	1.94	0.05	1.97	0.09	0.97	0.00	0.94	0.05	1.03	0.04	1.06	0.11	0.06	1.03	0.10	1.02	0.14	1.99	4.04	0.11	0.97	1.01	0.11	0.00	1.01	0.13	2.02	0.09	0.12	1.02	0.19	0.11	0.99	0.02	1.01	0.94	0.07	2.08	0.19	2.06	3.00	0.10	0.05	2.00	1.03	0.10	0.03	1.04	0.17	0.97	0.18	0.99	0.06	0.12	2.12	1.05	1.94	0.98	2.10	0.10	1.05	0.12	1.01	0.08	0.14	0.95	1.93	0.12	0.11	1.97	1.02	0.15	0.99	0.13	0.13	0.99	0.14	0.11	1.98	0.13	0.99	2.05	0.15	0.07	0.94	0.14	0.13	1.93	0.13	2.07	0.94	1.04	0.06	0.10	1.04	0.15	0.11	1.00	0.03	0.02	1.00	0.12	3.89	0.14	0.14	0.96	0.10	0.12	1.00	0.06	1.09	1.04	0.16	0.94	0.04	0.17	1.00	1.98	2.05	0.98	0.04	2.88	0.09	0.12	1.81	0.10	1.04	0.05	2.76	1.97	0.15	0.02	0.99	2.07	0.09	0.03	1.01	0.05	0.93	0.04	0.12	3.03	2.00	0.05	0.20	1.09	0.18	0.12	0.97	0.12	0.21	0.91	0.07	0.08	0.93	2.03	1.02	0.09	1.08	0.08	1.02	0.15	0.11	1.01	0.07	0.99	0.12	0.13	2.03	1.00	0.06	0.10	1.05	1.04	2.09	0.11	0.07	1.06	0.15	0.00	1.00	0.05	1.04	2.93	1.09	0.93	0.11	0.11	1.98	2.11	0.10	0.10	0.93	0.11	0.95	0.13	2.79	0.13	1.04	0.08	3.04	0.02	1.00	0.91	0.21	2.97	0.13	1.85	0.17	0.02	1.05	0.14	0.98	1.04	0.00	0.20	1.15	0.17	4.03	1.14	0.14	1.01	0.07	0.89	1.69	0.84	0.23	0.24	0.98	0.19	0.97	2.13	0.12	0.07	1.08	0.00	0.25	0.94	0.20	1.92	1.86	0.10	0.10	1.81	0.13	0.77	0.17	0.17	1.08	1.11	1.04	1.08	0.95	0.16	0.13	1.01	1.97	0.93	0.96	0.13	1.04	0.14	0.89	0.08	0.11	1.00	3.06	1.01	1.09	0.07	0.08	0.96	0.13	0.05	0.94	0.96	0.09	0.04	2.03	1.90	1.08	0.06	0.88	1.04	0.12	0.06	2.03	0.20	0.19	1.04	0.34	0.16",
    "	0.91	0.00	1.05	0.04	0.00	1.04	0.03	1.00	0.99	1.10	0.02	0.00	2.01	0.00	0.98	0.98	1.01	0.00	0.01	1.00	1.02	0.00	0.96	0.07	0.00	1.04	0.04	0.98	0.00	0.00	0.96	0.06	0.02	1.05	0.06	0.04	0.98	0.00	0.04	0.99	0.03	0.00	0.99	0.07	0.99	0.00	0.09	0.98	0.06	0.00	1.93	0.11	1.00	0.01	2.91	1.04	1.01	1.02	0.13	1.99	0.14	1.00	0.07	0.98	3.05	0.06	0.06	2.02	0.13	1.05	2.01	0.96	1.02	0.01	0.08	1.00	0.98	0.00	0.99	0.07	0.98	0.01	0.99	0.11	0.12	1.02	0.14	1.05	2.10	0.11	2.11	0.10	0.09	1.96	0.07	0.08	0.96	0.00	0.06	0.96	1.00	0.04	0.12	4.90	0.16	1.04	2.04	0.04	2.08	0.05	1.95	0.10	0.95	0.01	0.96	0.09	1.05	0.01	1.00	0.13	0.07	0.95	0.02	1.06	0.08	2.05	4.01	0.17	0.99	1.00	0.18	0.02	1.01	0.12	2.00	0.04	0.12	1.04	0.21	0.09	1.00	0.16	1.02	1.00	0.06	2.03	0.16	2.04	2.99	0.05	0.08	1.94	1.01	0.09	0.07	0.95	0.15	1.01	0.15	1.00	0.15	0.12	2.06	1.03	2.09	1.05	2.01	0.11	1.00	0.15	0.97	0.07	0.10	0.96	1.94	0.10	0.11	2.02	1.03	0.15	1.00	0.12	0.19	0.99	0.18	0.11	2.03	0.13	0.99	1.99	0.10	0.17	0.97	0.15	0.16	1.96	0.15	1.91	1.05	0.97	0.03	0.04	0.98	0.15	0.11	0.98	0.06	0.05	1.04	0.15	3.88	0.15	0.17	0.98	0.15	0.13	0.99	0.10	1.07	0.99	0.13	0.93	0.07	0.12	0.98	1.93	1.96	1.06	0.20	2.92	0.12	0.14	2.01	0.00	1.06	0.07	2.94	2.06	0.12	0.06	0.88	2.07	0.12	0.05	1.04	0.07	0.95	0.04	0.19	3.07	2.12	0.08	0.12	1.03	0.12	0.14	0.98	0.09	0.17	0.94	0.19	0.06	0.99	1.89	1.02	0.08	1.05	0.15	1.04	0.14	0.20	1.04	0.03	1.01	0.14	0.16	2.03	0.95	0.09	0.07	1.02	0.96	2.03	0.13	0.14	1.01	0.21	0.02	1.05	0.18	1.07	3.00	1.11	0.95	0.11	0.04	2.03	2.12	0.11	0.09	0.92	0.03	0.97	0.10	2.95	0.15	0.97	0.19	2.94	0.04	1.00	1.01	0.15	3.00	0.12	1.88	0.10	0.08	1.03	0.08	1.00	1.04	0.01	0.20	1.07	0.15	3.96	1.13	0.10	0.90	0.10	0.94	1.99	0.97	0.12	0.00	1.01	0.06	1.01	2.07	0.17	0.16	1.04	0.01	0.02	1.01	0.09	1.92	2.08	0.16	0.15	1.90	0.07	0.93	0.17	0.20	1.03	0.98	1.10	1.07	1.04	0.06	0.16	0.95	0.21	0.12	0.96	0.07	1.01	1.02	0.97	0.12	1.01	0.09	0.94	0.09	0.17	0.93	3.09	1.03	1.06	0.15	0.18	0.90	0.07	0.02	1.06	0.97	0.18	0.01	1.93	1.98	1.13	0.03	0.88	0.96	0.03	0.09	2.00	0.15",
    "	0.97	0.00	1.01	0.02	0.00	1.02	0.01	0.97	0.99	1.10	0.00	0.03	1.99	0.00	0.99	1.02	0.99	0.00	0.03	0.98	1.03	0.00	0.94	0.05	0.00	1.03	0.05	1.02	0.00	0.00	0.95	0.04	0.01	1.02	0.04	0.03	0.97	0.00	0.03	0.98	0.05	0.00	0.99	0.07	0.99	0.00	0.12	1.00	0.09	0.00	1.93	0.10	0.98	0.01	2.91	1.02	1.02	1.05	0.13	2.03	0.17	1.01	0.03	0.98	3.02	0.06	0.08	2.03	0.13	1.03	1.99	1.01	0.98	0.01	0.11	1.01	0.97	0.00	0.97	0.07	0.99	0.01	0.99	0.07	0.12	1.01	0.13	1.03	2.06	0.10	1.98	0.08	0.12	2.01	0.10	0.09	1.01	0.08	0.03	0.95	1.02	0.03	0.12	4.87	0.18	1.01	1.95	0.12	1.99	0.06	1.96	0.08	0.99	0.00	1.02	0.07	1.02	0.04	1.01	0.18	0.07	1.00	0.30	0.98	0.14	1.95	3.70	0.15	1.04	0.95	0.24	0.05	1.00	0.13	1.92	0.06	0.17	1.01	0.12	0.15	0.98	0.07	0.98	1.07	0.15	2.01	0.13	2.02	2.96	0.05	0.06	1.98	1.00	0.12	0.06	0.94	0.15	0.98	0.09	1.01	0.08	0.10	2.05	1.06	2.03	1.01	1.92	0.13	0.98	0.12	1.06	0.04	0.08	1.00	1.96	0.05	0.14	1.93	1.00	0.16	0.95	0.14	0.09	0.97	0.17	0.03	1.97	0.14	1.03	1.86	0.09	0.04	0.97	0.05	0.17	1.96	0.14	1.93	1.05	0.98	0.03	0.13	0.97	0.13	0.14	0.99	0.15	0.07	0.98	0.12	3.88	0.12	0.13	0.99	0.07	0.17	0.98	0.14	1.08	1.03	0.14	0.95	0.07	0.15	1.00	2.01	2.03	0.99	0.04	2.89	0.08	0.13	2.00	0.07	1.05	0.05	2.84	1.91	0.07	0.05	0.97	1.97	1.05	0.17	0.11	1.94	0.08	0.95	0.08	1.04	1.96	0.17	0.15	0.98	0.12	0.17	1.03	0.10	0.07	1.02	0.16	0.06	0.93	2.08	1.06	0.06	1.03	0.21	0.99	0.03	0.10	1.00	0.04	0.98	0.10	0.12	2.07	0.97	0.02	0.05	1.03	1.03	2.04	0.08	0.05	1.01	0.18	0.01	0.99	0.03	1.06	2.97	1.07	0.95	0.19	0.03	1.90	2.03	0.09	0.10	0.98	0.06	0.94	0.06	2.94	0.14	0.98	0.06	2.94	0.07	1.07	0.94	0.11	2.93	0.06	1.85	0.12	0.14	0.96	0.08	1.04	1.04	0.07	0.18	1.03	0.00	3.95	1.08	0.06	0.91	0.10	1.01	2.00	0.97	0.18	0.00	0.98	0.04	1.02	2.00	0.14	0.17	1.03	0.00	0.00	1.04	0.06	1.96	2.09	0.17	0.10	1.92	0.18	0.90	0.18	0.22	0.99	0.97	1.23	1.04	0.96	0.18	0.18	1.01	1.98	1.03	1.02	0.13	0.99	0.14	0.88	0.03	0.09	0.99	3.08	1.05	0.98	0.16	0.20	0.94	0.11	0.03	1.04	0.95	0.03	0.00	1.90	1.94	1.08	0.02	0.90	0.99	0.07	0.08	1.94	0.12	0.22	1.02	0.21	0.14",
    "	0.99	0.00	1.03	0.03	0.01	0.99	0.02	0.96	0.99	1.10	0.00	0.02	2.00	0.00	0.98	1.02	1.01	0.00	0.03	0.98	1.03	0.00	0.98	0.03	0.01	1.00	0.07	0.96	0.02	0.00	0.98	0.03	0.03	1.02	0.03	0.05	0.99	0.00	0.04	0.98	0.04	0.00	0.96	0.08	0.99	0.00	0.08	0.97	0.07	0.00	2.05	0.11	1.03	0.00	2.86	1.05	1.01	1.08	0.10	1.98	0.16	0.96	0.04	0.97	2.98	0.06	0.08	2.00	0.10	1.03	2.00	0.96	1.05	0.02	0.07	0.96	0.99	0.00	1.03	0.05	0.96	0.02	0.97	0.09	0.12	1.01	0.10	1.00	2.03	0.08	2.02	0.07	0.11	2.01	0.08	0.08	1.00	0.04	0.03	1.03	1.00	0.02	0.12	4.80	0.17	1.04	2.03	0.04	2.04	0.08	1.93	0.06	0.97	0.00	0.91	0.08	1.02	0.04	0.96	0.12	0.05	1.02	0.07	1.08	0.09	2.05	4.10	0.16	1.03	0.98	0.15	0.02	0.98	0.13	1.96	0.04	0.16	1.02	0.10	0.08	1.00	0.12	1.01	1.00	0.06	1.99	0.16	2.02	3.05	0.05	0.07	1.98	0.97	0.09	0.04	1.01	0.15	1.02	0.14	1.01	0.18	0.11	2.12	1.05	2.14	1.05	2.11	0.12	0.93	0.14	0.94	0.03	0.13	1.01	1.96	0.08	0.11	1.98	1.05	0.13	0.98	0.02	0.18	0.99	0.12	0.09	2.01	0.12	1.00	2.01	0.12	0.17	0.96	0.07	0.14	1.99	0.15	1.95	1.00	0.95	0.04	0.01	1.07	0.15	0.09	0.97	0.04	0.05	1.03	0.15	3.82	0.12	0.13	0.98	0.15	0.09	0.98	0.11	1.03	1.03	0.14	0.97	0.10	0.11	0.97	1.95	1.97	1.02	0.06	2.81	0.16	0.12	1.91	0.02	1.02	0.08	2.83	2.07	0.19	0.03	0.95	1.93	1.08	0.10	0.07	1.93	0.20	0.98	0.06	0.90	1.99	0.18	0.12	1.04	0.07	0.12	1.03	0.07	0.14	0.96	0.15	0.05	1.09	1.92	1.01	0.06	0.95	0.11	1.07	0.11	0.20	1.03	0.03	0.98	0.12	0.15	2.11	1.00	0.12	0.06	0.96	0.98	2.08	0.10	0.15	1.02	0.11	0.00	1.07	0.17	1.05	2.93	1.09	0.96	0.08	0.03	1.97	2.03	0.18	0.10	0.91	0.04	1.03	0.08	2.94	0.16	0.95	0.13	2.89	0.06	1.03	0.98	0.16	2.81	0.20	1.79	0.21	0.11	0.99	0.04	1.04	1.00	0.04	0.19	1.05	0.14	4.00	1.12	0.10	0.95	0.04	0.97	2.04	0.96	0.16	0.02	0.92	0.05	1.01	2.01	0.14	0.14	1.04	0.01	0.09	1.10	0.07	2.02	2.05	0.12	0.13	1.89	0.18	0.99	0.17	0.14	1.01	0.92	1.09	1.07	1.04	0.14	0.12	0.99	1.95	0.99	1.00	0.08	0.97	0.11	0.88	0.21	0.19	0.95	2.94	1.04	1.09	0.01	0.11	0.93	0.03	0.00	1.02	0.96	0.19	0.02	1.86	1.90	1.18	0.04	0.86	0.94	0.02	0.08	1.92	0.11	0.14	0.88	0.15	0.15",
    "	0.98	0.00	0.99	0.05	0.02	1.02	0.04	0.99	0.98	1.10	0.02	0.04	1.97	0.00	0.98	1.02	1.02	0.00	0.04	1.01	0.99	0.00	0.98	0.05	0.02	1.04	0.07	0.94	0.02	0.02	0.96	0.05	0.03	1.07	0.06	0.06	0.98	0.00	0.06	0.99	0.07	0.00	0.97	0.11	0.97	0.00	0.13	0.99	0.09	0.00	1.99	0.12	1.00	0.02	2.88	1.04	1.00	1.10	0.12	1.99	0.18	0.97	0.07	0.97	3.00	0.07	0.09	1.99	0.13	1.01	2.05	0.96	1.06	0.01	0.06	1.01	0.95	0.00	1.00	0.08	1.00	0.02	0.98	0.11	0.12	1.00	0.11	0.99	1.99	0.11	1.97	0.09	0.12	2.01	0.10	0.10	1.01	0.06	0.05	1.06	0.98	0.05	0.14	4.67	0.11	0.99	2.07	0.05	2.02	0.06	1.95	0.08	0.97	0.02	1.00	0.08	1.03	0.03	0.94	0.12	0.07	1.03	0.10	1.04	0.08	2.04	3.99	0.19	1.01	1.00	0.19	0.04	0.98	0.14	2.05	0.06	0.15	1.03	0.13	0.10	1.01	0.09	1.00	0.99	0.06	1.99	0.15	2.02	3.03	0.08	0.06	2.01	1.00	0.08	0.07	0.95	0.12	1.00	0.17	1.01	0.12	0.12	2.12	1.00	2.05	1.08	2.01	0.11	1.00	0.07	1.00	0.08	0.12	0.93	1.99	0.08	0.07	2.04	1.05	0.13	0.98	0.01	0.14	1.01	0.10	0.09	2.01	0.12	0.94	1.88	0.07	0.18	0.97	0.12	0.11	1.97	0.15	1.91	1.07	0.99	0.04	0.04	0.99	0.17	0.09	0.97	0.06	0.06	1.04	0.09	3.92	0.11	0.12	0.97	0.16	0.12	1.01	0.11	1.02	1.04	0.13	0.90	0.07	0.11	1.04	1.98	2.01	1.02	0.17	2.92	0.17	0.14	1.94	0.04	1.02	0.08	2.88	1.98	0.14	0.05	0.94	2.02	0.17	0.06	1.00	0.05	0.97	0.07	0.17	3.04	2.07	0.08	0.16	1.03	0.13	0.09	1.00	0.06	0.17	0.93	0.12	0.04	1.01	2.01	1.03	0.08	1.02	0.18	1.01	0.13	0.08	1.01	0.06	1.02	0.12	0.08	2.05	1.00	0.09	0.08	0.96	1.01	2.12	0.09	0.17	0.99	0.17	0.00	1.07	0.15	1.07	3.03	1.08	0.95	0.11	0.06	1.98	2.04	0.13	0.14	0.91	0.07	1.00	0.11	2.96	0.13	0.97	0.14	2.94	0.06	1.02	1.00	0.16	2.97	0.20	1.82	0.14	0.10	0.99	0.09	1.07	0.96	0.06	0.18	1.06	0.14	3.99	1.09	0.08	0.94	0.06	0.98	2.00	0.92	0.13	0.01	1.04	0.07	0.96	1.99	0.13	0.13	1.10	0.04	0.02	1.02	0.07	1.96	2.09	0.16	0.18	1.96	0.20	0.88	0.13	0.17	1.04	0.98	1.10	1.04	0.99	0.12	0.13	1.03	1.90	0.99	0.98	0.11	1.02	0.09	0.93	0.14	0.18	0.95	3.06	1.04	1.09	0.00	0.11	0.91	0.09	0.04	1.01	1.01	0.11	0.01	1.98	1.94	1.07	0.05	0.87	0.93	0.09	0.08	2.06	0.14	0.17	0.96	0.22	0.16",
    "	0.99	0.00	1.00	0.04	0.03	1.01	0.04	0.99	0.94	1.13	0.00	0.00	2.01	0.00	0.98	0.98	1.01	0.00	0.00	1.01	1.00	0.00	1.01	0.03	0.01	1.02	0.00	0.98	0.02	0.00	0.95	0.04	0.02	1.05	0.05	0.01	0.98	0.01	0.02	0.98	0.05	0.00	1.00	0.06	0.98	0.00	0.08	0.99	0.06	0.00	1.90	0.08	1.02	0.00	2.94	0.98	1.02	1.04	0.13	1.98	0.15	0.98	0.04	0.98	3.03	0.08	0.07	2.00	0.15	1.05	1.97	1.00	1.01	0.04	0.10	0.98	0.99	0.00	0.96	0.06	1.01	0.00	0.97	0.10	0.11	1.00	0.13	1.02	2.03	0.09	2.04	0.09	0.11	1.98	0.09	0.07	0.99	0.04	0.05	0.99	0.99	0.06	0.11	4.94	0.19	1.04	2.03	0.08	2.02	0.06	2.02	0.08	0.97	0.00	0.96	0.07	1.01	0.03	0.99	0.10	0.10	0.97	0.07	1.06	0.08	2.03	4.04	0.13	1.02	1.00	0.16	0.02	0.99	0.11	2.01	0.04	0.13	1.01	0.12	0.07	1.00	0.14	1.00	0.96	0.04	2.02	0.16	2.05	2.99	0.05	0.07	1.98	1.06	0.08	0.05	0.98	0.16	1.00	0.12	0.97	0.13	0.10	2.09	1.01	2.05	1.07	2.08	0.12	0.99	0.12	0.97	0.05	0.13	0.94	1.97	0.09	0.10	1.96	1.03	0.14	1.01	0.14	0.19	1.01	0.14	0.09	2.00	0.13	0.97	1.92	0.09	0.17	0.96	0.14	0.14	1.97	0.16	1.92	1.03	1.01	0.05	0.05	1.00	0.14	0.11	0.95	0.07	0.03	1.03	0.10	3.91	0.13	0.11	0.98	0.15	0.10	0.99	0.12	1.07	1.04	0.17	0.91	0.10	0.13	0.97	1.95	2.04	1.01	0.03	2.83	0.19	0.12	1.92	0.05	1.02	0.05	2.82	1.94	0.15	0.04	1.03	1.92	0.60	0.09	0.44	1.02	0.49	0.51	0.15	1.88	1.98	0.11	0.12	1.00	0.13	0.09	0.97	0.06	0.17	0.96	0.16	0.03	1.04	1.96	1.01	0.07	1.02	0.14	1.04	0.13	0.17	1.02	0.07	0.98	0.10	0.14	2.03	0.97	0.10	0.06	0.99	1.00	2.03	0.10	0.17	1.02	0.20	0.06	1.01	0.16	1.05	2.98	1.03	0.98	0.08	0.04	2.02	2.06	0.11	0.09	0.98	0.08	0.96	0.09	2.94	0.15	0.98	0.16	3.06	0.07	0.99	1.01	0.23	2.95	0.12	1.87	0.13	0.13	1.02	0.08	1.04	0.98	0.07	0.17	1.01	0.15	4.02	1.11	0.13	0.92	0.14	0.97	1.99	1.01	0.12	0.00	1.01	0.07	1.03	2.18	0.18	0.14	1.04	0.15	0.20	0.88	0.16	1.86	2.03	0.11	0.10	1.87	0.15	0.79	0.14	0.18	1.03	0.95	0.99	0.90	0.90	0.12	0.20	1.08	2.08	1.06	1.11	0.10	1.11	0.15	0.97	0.12	0.21	1.00	3.12	1.09	1.08	0.03	0.11	0.88	0.11	0.01	0.90	0.95	0.23	0.00	1.95	1.85	1.04	0.02	0.83	0.99	0.08	0.08	2.00	0.15	0.22	0.91	0.19	0.15", "	0.98	0.00	1.01	0.03	0.00	1.00	0.04	1.03	0.99	1.16	0.04	0.00	1.97	0.00	1.00	0.96	1.01	0.00	0.03	0.98	0.99	0.00	1.00	0.04	0.01	0.99	0.05	0.98	0.00	0.01	0.98	0.05	0.02	1.07	0.06	0.05	0.96	0.00	0.05	0.98	0.07	0.00	0.97	0.09	0.98	0.00	0.15	0.99	0.08	0.00	1.91	0.10	1.00	0.03	2.97	1.03	1.01	1.03	0.13	2.00	0.09	0.99	0.06	0.98	3.02	0.09	0.11	2.06	0.15	1.04	2.02	0.99	1.05	0.00	0.10	1.00	0.96	0.00	0.95	0.06	0.99	0.03	0.98	0.11	0.17	1.01	0.18	1.00	1.98	0.14	1.91	0.08	0.14	2.00	0.14	0.11	0.99	0.07	0.05	1.07	0.99	0.06	0.18	4.82	0.11	0.99	1.97	0.11	1.97	0.04	1.93	0.11	0.97	0.00	0.99	0.08	1.02	0.06	0.96	0.12	0.07	1.01	0.12	1.04	0.11	2.02	3.89	0.16	1.03	1.00	0.14	0.00	0.99	0.13	1.89	0.04	0.13	1.02	0.20	0.10	0.98	0.13	0.98	0.99	0.04	1.98	0.17	1.99	3.00	0.08	0.07	1.97	1.04	0.08	0.08	1.03	0.12	0.99	0.08	0.95	0.06	0.15	2.13	1.02	1.99	1.05	2.03	0.10	0.99	0.13	1.00	0.08	0.09	0.99	2.00	0.08	0.09	2.04	0.98	0.14	0.98	0.14	0.19	1.04	0.12	0.03	1.95	0.16	0.94	1.88	0.10	0.05	0.95	0.14	0.11	1.92	0.09	1.87	1.16	0.98	0.05	0.09	0.96	0.14	0.12	0.92	0.08	0.07	0.97	0.14	3.34	0.10	0.51	0.99	0.20	0.12	1.03	0.12	1.04	1.03	0.12	0.96	0.08	0.10	1.03	1.81	1.93	0.97	0.08	2.94	0.12	0.12	1.83	0.06	1.01	0.09	2.77	2.02	0.18	0.05	1.02	1.93	1.01	0.09	0.07	1.84	0.08	0.99	0.09	0.98	2.04	0.10	0.12	1.07	0.11	0.12	0.98	0.12	0.17	0.93	0.14	0.09	1.04	1.96	1.00	0.08	0.95	0.17	1.07	0.12	0.11	1.04	0.00	0.95	0.13	0.16	2.02	0.99	0.07	0.07	1.08	0.99	1.98	0.11	0.06	1.00	0.22	0.00	1.03	0.03	1.06	2.94	1.14	0.90	0.12	0.05	2.03	2.07	0.10	0.14	0.86	0.05	0.99	0.07	2.92	0.15	0.96	0.14	2.91	0.01	1.06	1.02	0.16	2.93	0.11	1.86	0.16	0.12	1.00	0.06	1.01	1.01	0.00	0.07	1.04	0.12	4.01	1.02	0.04	0.95	0.07	1.03	2.09	0.98	0.13	0.00	1.00	0.06	1.01	2.13	0.20	0.18	1.04	0.02	0.03	0.96	0.13	1.92	2.17	0.03	0.18	2.01	0.18	0.91	0.12	0.15	1.00	0.99	1.12	1.06	1.02	0.12	0.21	1.00	1.87	1.01	1.02	0.15	1.00	0.12	0.97	0.11	0.06	0.91	2.99	1.06	1.04	0.15	0.09	0.87	0.09	0.04	0.98	0.97	0.10	0.00	2.07	1.98	1.13	0.05	0.83	1.01	0.12	0.11	2.09	0.16	0.18	0.94	0.22	0.14",
    "	0.99	0.00	1.00	0.04	0.02	1.01	0.04	0.95	0.98	1.13	0.00	0.02	2.02	0.00	1.01	1.01	0.99	0.00	0.00	0.99	1.01	0.00	0.92	0.07	0.04	0.98	0.01	1.02	0.01	0.00	0.99	0.04	0.03	1.08	0.05	0.04	0.97	0.00	0.02	0.97	0.06	0.00	0.98	0.06	0.97	0.00	0.09	0.97	0.08	0.00	2.03	0.08	1.04	0.01	2.91	1.04	0.99	1.06	0.12	1.99	0.18	0.96	0.06	0.94	2.96	0.07	0.09	1.89	0.16	1.06	1.96	1.02	1.01	0.01	0.11	0.96	1.01	0.00	0.97	0.08	1.03	0.00	0.96	0.11	0.13	0.99	0.15	1.00	2.03	0.11	1.95	0.10	0.15	1.97	0.15	0.10	0.98	0.09	0.04	1.00	1.02	0.05	0.13	4.77	0.10	1.01	1.97	0.08	1.98	0.05	1.96	0.10	0.96	0.00	0.98	0.09	1.01	0.04	1.04	0.13	0.07	1.01	0.07	1.01	0.09	2.02	3.92	0.14	1.02	0.95	0.17	0.04	1.01	0.11	1.99	0.03	0.16	1.03	0.14	0.09	1.00	0.09	1.02	0.99	0.06	2.03	0.16	1.99	3.06	0.15	0.05	1.99	0.98	0.07	0.05	0.98	0.11	1.02	0.20	0.95	0.08	0.13	2.13	1.10	2.05	1.07	2.09	0.17	0.94	0.09	0.96	0.05	0.12	1.00	2.00	0.08	0.10	2.01	0.97	0.15	1.02	0.15	0.18	1.02	0.12	0.10	2.02	0.15	1.00	2.00	0.10	0.18	0.95	0.11	0.15	1.99	0.14	1.96	1.09	0.96	0.06	0.04	0.99	0.16	0.12	0.96	0.08	0.06	1.03	0.13	3.85	0.12	0.11	0.97	0.13	0.09	0.96	0.10	1.06	1.05	0.11	0.96	0.07	0.11	0.99	1.88	2.08	1.04	0.06	2.84	0.20	0.10	1.97	0.02	1.00	0.05	2.99	2.03	0.15	0.01	0.90	1.98	0.13	0.03	2.04	3.02	2.07	0.09	0.14	0.97	0.05	0.13	1.07	0.12	0.17	0.95	0.05	0.17	1.08	1.99	4.06	1.10	0.13	0.05	1.97	0.98	0.08	0.08	1.04	0.95	2.03	0.09	0.11	0.96	0.15	0.04	1.06	0.15	1.06	2.89	1.09	0.98	0.20	0.02	2.00	1.94	0.16	0.07	0.96	0.05	0.94	0.07	2.88	0.13	0.98	0.15	2.95	0.10	0.99	1.02	0.19	2.83	0.13	1.78	0.15	0.09	1.04	0.04	1.02	0.97	0.00	0.15	1.06	0.11	3.91	1.11	0.09	0.92	0.06	0.96	2.09	0.98	0.12	0.00	0.99	0.02	1.01	2.04	0.16	0.14	1.03	0.02	0.00	0.97	0.07	2.03	2.19	0.12	0.16	1.95	0.16	0.95	0.19	0.19	1.08	0.96	1.02	1.03	0.99	0.13	0.17	1.02	1.95	1.04	0.99	0.11	0.97	0.07	0.96	0.07	0.22	0.97	3.00	1.06	1.12	0.15	0.17	0.85	0.05	0.01	0.96	0.99	0.09	0.00	2.03	1.89	1.08	0.03	0.88	0.95	0.04	0.07	2.00	0.13	0.20	0.97	0.22	0.15	1.17	0.06	1.02	0.12	0.18	0.92	1.01	0.11	2.12	1.02	0.07	0.10",
    "	0.98	0.00	1.02	0.00	0.00	1.00	0.00	1.09	1.01	0.98	0.11	0.00	1.99	0.00	1.03	0.93	1.03	0.00	0.12	0.91	1.12	0.13	1.09	0.15	0.00	0.97	0.31	0.93	0.00	0.15	0.89	0.02	0.15	1.00	0.00	0.18	0.98	0.00	0.17	0.99	0.16	0.00	0.95	0.16	0.97	0.00	0.31	0.95	0.15	0.00	2.44	0.18	1.11	0.14	2.83	1.35	1.00	1.19	0.09	2.07	0.16	0.91	0.15	0.93	3.02	0.03	0.07	2.42	0.08	0.91	2.11	1.08	1.05	0.00	0.07	1.02	0.99	0.00	0.98	0.04	1.00	0.00	0.99	0.05	0.10	1.01	0.13	1.00	1.97	0.09	2.18	0.05	0.10	2.10	0.11	0.07	1.03	0.07	0.00	1.05	1.06	0.01	0.10	4.81	0.03	0.87	2.06	0.16	2.12	0.05	1.91	0.06	0.89	0.29	1.17	0.02	1.08	0.06	1.08	0.06	0.14	0.89	0.20	0.98	0.08	1.97	3.63	0.24	1.03	0.94	0.33	0.30	0.96	0.14	2.02	0.04	0.10	1.11	0.18	0.04	1.05	0.12	0.88	1.05	0.01	2.04	0.10	1.91	2.99	0.03	0.14	2.02	1.10	0.05	0.22	1.11	0.10	0.98	0.27	0.93	0.38	0.11	2.01	1.46	2.25	1.03	2.03	0.21	0.96	0.11	0.96	0.20	0.09	0.99	2.33	0.06	0.11	2.31	1.05	0.10	1.40	0.11	0.34	1.00	0.18	0.13	2.22	0.09	0.93	2.17	0.36	0.14	0.91	0.27	0.16	1.89	0.17	1.95	0.91	1.06	0.04	0.06	1.03	0.13	0.14	1.12	0.00	0.00	1.02	0.08	3.67	0.05	0.11	1.03	0.10	0.11	0.87	0.06	1.13	1.09	0.14	0.86	0.13	0.13	0.91	1.91	1.91	1.13	0.04	2.68	0.28	0.12	1.74	0.13	0.93	0.00	2.79	1.83	0.16	0.00	0.97	1.76	1.06	0.08	0.11	2.03	0.19	1.13	0.02	1.01	1.91	0.16	0.07	1.00	0.18	0.09	1.00	0.22	0.15	0.76	0.12	0.41	0.92	1.85	1.13	0.08	1.04	0.15	1.54	0.11	0.14	1.35	0.13	1.07	0.11	0.15	2.09	0.89	0.30	0.03	0.97	1.14	2.00	0.17	0.12	1.32	0.28	0.00	0.93	0.15	1.01	2.98	1.06	0.88	0.18	0.17	1.85	2.03	0.16	0.18	0.89	0.11	0.90	0.04	2.86	0.13	1.17	0.11	2.76	0.12	0.90	1.05	0.11	2.84	0.19	1.71	0.17	0.10	1.02	0.00	0.95	1.10	0.00	0.14	1.15	0.09	4.02	1.01	0.01	1.03	0.03	0.96	2.05	0.91	0.05	0.11	0.93	0.05	1.05	2.30	0.29	0.13	0.98	0.01	0.24	1.11	0.06	2.04	2.04	0.15	0.19	1.92	0.16	1.04	0.11	0.16	1.01	0.89	1.33	1.16	1.42	0.14	0.21	1.04	1.90	1.06	0.96	0.15	1.03	0.11	0.99	0.20	0.16	0.86	2.80	1.07	1.37	0.11	0.05	0.87	0.21	0.03	1.68	1.23	0.17	0.08	1.73	1.76	1.11	0.00	0.79	1.54	0.04	0.05	1.92	0.13	0.16	0.98	0.18	0.32",
    "	1.01	0.00	0.99	0.01	0.00	0.99	0.00	1.01	1.01	1.06	0.04	0.03	2.08	0.00	0.98	1.01	0.96	0.00	0.00	1.00	1.02	0.00	0.97	0.05	0.00	0.98	0.01	0.99	0.00	0.00	0.99	0.03	0.00	1.01	0.02	0.02	1.00	0.00	0.01	1.01	0.00	0.00	0.98	0.07	0.99	0.00	0.10	1.01	0.00	0.28	1.63	0.08	0.99	0.00	3.17	1.00	1.04	1.03	0.12	2.11	0.08	0.98	0.02	0.95	3.44	0.04	0.07	2.14	0.08	0.99	2.17	0.96	0.99	0.00	0.06	1.02	0.98	0.00	1.03	0.05	1.01	0.00	1.00	0.07	0.02	0.99	0.13	1.04	2.27	0.12	2.20	0.07	0.04	2.13	0.10	0.08	0.99	0.04	0.02	0.88	1.02	0.01	0.09	5.46	0.00	1.08	2.14	0.01	2.24	0.01	2.10	0.08	0.91	0.00	0.93	0.05	1.03	0.00	1.02	0.12	0.00	0.92	0.00	1.06	0.00	1.60	2.64	0.40	0.59	0.63	1.48	0.08	1.04	0.26	1.15	0.04	0.22	0.59	0.76	0.10	0.52	0.21	0.68	0.54	0.25	1.09	0.41	1.61	2.02	0.88	0.09	2.30	2.24	0.12	0.07	2.06	0.45	0.99	0.22	1.46	0.18	0.36	2.18	1.76	2.16	1.14	3.40	0.57	2.24	0.48	1.96	0.18	0.41	0.84	2.35	0.14	0.13	2.49	2.25	0.19	1.12	1.12	0.44	1.08	0.45	0.13	1.93	0.57	0.79	1.83	1.23	0.24	1.50	1.02	0.17	1.86	0.45	1.68	1.02	2.03	0.00	0.94	1.53	0.42	0.07	0.93	0.42	0.17	1.21	0.42	3.93	0.16	0.40	1.08	2.33	0.18	1.04	0.49	0.82	1.11	0.46	0.78	0.61	0.47	0.94	2.30	1.96	1.00	0.47	3.76	1.06	0.47	1.80	1.21	1.03	0.14	3.52	1.51	0.46	0.05	2.26	2.34	0.14	0.03	1.08	0.87	1.19	0.13	0.00	3.22	1.79	0.54	0.12	1.59	1.05	0.24	1.01	0.48	0.11	0.93	0.38	0.12	0.95	2.41	0.97	0.13	1.65	0.93	1.64	0.16	0.49	0.95	0.32	1.00	0.21	0.53	1.93	1.33	0.08	0.15	2.17	1.42	1.87	0.13	0.35	1.43	0.95	0.00	0.88	0.38	1.00	2.78	1.74	0.82	0.57	1.75	2.70	2.58	0.23	0.00	2.01	1.05	0.99	0.19	3.35	0.11	1.46	0.20	4.49	0.01	1.51	1.04	1.67	2.86	0.44	2.30	0.15	1.33	0.99	0.96	1.01	0.90	0.45	0.14	1.56	0.46	3.88	1.09	0.57	0.82	1.94	1.50	1.90	1.53	0.14	0.41	2.35	0.40	1.01	1.90	0.46	0.20	1.80	1.12	0.22	1.14	0.51	1.93	2.48	0.46	0.26	3.11	1.06	0.89	0.26	0.92	0.94	1.67	1.56	0.96	1.68	0.34	0.46	1.83	2.54	1.00	1.00	0.51	2.22	0.41	1.56	0.29	0.26	0.99	3.62	1.09	1.15	0.38	1.48	1.48	0.51	0.11	0.74	1.32	0.13	0.01	2.56	2.23	1.08	0.03	1.76	1.85	0.51	0.08	2.28	0.38	0.02	0.99	1.06	0.12",
    "	0.98	0.00	0.98	0.00	0.00	1.04	0.00	1.03	1.00	1.15	0.05	0.00	1.90	0.01	0.90	0.99	0.98	0.02	0.02	0.97	0.98	0.04	0.88	0.05	0.01	1.02	0.09	1.01	0.00	0.03	1.01	0.00	0.00	1.13	0.01	0.00	0.98	0.00	0.00	0.95	0.01	0.00	0.95	0.05	0.97	0.00	0.11	0.95	0.04	0.00	2.09	0.07	1.04	0.02	3.06	1.10	1.05	1.00	0.07	1.89	0.14	1.00	0.00	1.00	2.88	0.04	0.08	1.88	0.15	0.97	1.92	1.03	0.94	0.00	0.17	0.97	1.04	0.00	1.02	0.05	0.95	0.01	0.93	0.05	0.15	1.05	0.17	0.95	1.94	0.13	1.96	0.06	0.13	2.10	0.12	0.05	1.02	0.09	0.00	1.07	0.98	0.02	0.12	4.69	0.09	0.97	1.94	0.10	1.95	0.05	1.93	0.05	0.94	0.02	1.06	0.01	0.99	0.03	1.05	0.07	0.09	1.02	0.11	0.98	0.06	2.02	3.79	0.12	0.97	1.01	0.12	0.09	0.99	0.14	1.98	0.06	0.15	1.06	0.15	0.09	1.02	0.12	0.96	1.00	0.05	2.01	0.12	1.94	2.93	0.01	0.09	1.88	0.98	0.07	0.09	1.05	0.08	0.94	0.17	1.00	0.17	0.11	2.05	1.14	2.02	0.99	1.92	0.18	0.91	0.09	1.02	0.10	0.08	0.97	1.98	0.07	0.14	2.02	1.09	0.07	1.04	0.10	0.10	0.95	0.09	0.06	2.01	0.16	0.95	2.16	0.09	0.07	0.92	0.15	0.12	1.84	0.13	2.04	0.99	0.89	0.03	0.05	1.12	0.07	0.07	1.02	0.03	0.09	1.00	0.08	3.71	0.16	0.14	0.99	0.14	0.13	0.97	0.13	1.07	1.01	0.10	0.99	0.08	0.15	1.04	1.85	1.94	0.97	0.08	2.79	0.13	0.09	1.92	0.01	0.99	0.05	2.79	2.05	0.16	0.04	0.96	1.89	1.12	0.10	0.10	2.00	0.12	0.97	0.15	0.96	1.85	0.17	0.13	0.94	0.09	0.13	1.02	0.10	0.12	1.02	0.15	0.21	1.00	1.85	1.04	0.20	0.99	0.06	1.05	0.08	0.16	1.01	0.19	1.04	0.18	0.15	2.09	0.94	0.21	0.15	1.00	1.00	1.91	0.14	0.11	1.06	0.09	0.02	1.11	0.15	1.06	2.91	1.13	0.97	0.11	0.00	2.13	1.97	0.12	0.14	0.87	0.10	0.93	0.16	2.88	0.17	0.97	0.15	2.96	0.18	0.96	0.99	0.12	2.97	0.17	1.87	0.13	0.03	0.99	0.06	1.08	1.02	0.00	0.16	1.12	0.13	3.81	1.14	0.00	0.90	0.01	0.94	1.95	0.91	0.12	0.00	1.02	0.05	0.97	2.01	0.18	0.17	1.06	0.18	0.10	1.07	0.08	1.89	2.19	0.18	0.16	1.75	0.05	0.98	0.15	0.13	1.10	0.90	1.16	1.02	1.05	0.17	0.08	0.97	1.95	0.95	1.10	0.10	0.88	0.06	0.87	0.12	0.14	0.94	2.86	1.12	1.09	0.01	0.10	0.94	0.04	0.06	1.12	0.84	0.08	0.06	1.87	1.78	1.13	0.06	0.82	0.82	0.04	0.08	2.03	0.07	0.20	1.07	0.18	0.12",
    "	1.01	0.00	1.00	0.03	0.02	0.99	0.02	1.00	1.00	1.06	0.05	0.00	2.03	0.01	0.98	0.98	1.02	0.00	0.00	1.04	1.03	0.00	0.94	0.08	0.02	1.00	0.03	0.98	0.04	0.00	1.00	0.04	0.02	1.05	0.04	0.02	0.98	0.00	0.02	0.98	0.07	0.00	0.99	0.05	0.98	0.00	0.07	1.00	0.08	0.00	1.91	0.08	0.97	0.01	3.03	0.99	1.06	0.99	0.13	2.01	0.17	1.01	0.04	1.00	3.07	0.08	0.09	1.98	0.14	1.03	2.04	0.96	1.01	0.00	0.07	0.98	0.97	0.00	1.00	0.09	0.98	0.00	1.01	0.10	0.15	0.97	0.16	1.01	2.01	0.11	1.98	0.11	0.13	1.98	0.15	0.11	1.01	0.06	0.04	1.03	0.96	0.03	0.12	4.95	0.13	1.03	2.01	0.06	2.00	0.04	1.96	0.09	0.96	0.00	0.96	0.07	0.99	0.02	1.01	0.14	0.07	1.01	0.09	1.06	0.12	1.96	4.03	0.18	1.00	0.99	0.19	0.01	0.99	0.14	1.93	0.07	0.15	1.00	0.15	0.10	0.99	0.14	1.01	0.97	0.09	1.98	0.16	2.02	2.96	0.09	0.06	2.01	1.06	0.08	0.04	0.99	0.13	1.01	0.18	0.97	0.17	0.14	2.12	1.03	2.05	1.03	2.08	0.16	0.98	0.16	0.97	0.08	0.13	0.99	1.99	0.10	0.10	2.02	1.03	0.13	0.98	0.13	0.16	0.99	0.13	0.11	2.08	0.16	0.98	1.97	0.08	0.08	0.93	0.11	0.14	1.98	0.15	2.05	1.02	1.01	0.07	0.01	1.05	0.17	0.10	0.98	0.06	0.02	0.99	0.09	4.01	0.12	0.12	0.93	0.16	0.14	0.97	0.12	1.09	1.04	0.18	0.95	0.09	0.12	1.00	1.96	2.01	1.01	0.20	2.96	0.20	0.13	1.94	0.02	1.02	0.03	2.85	2.09	0.12	0.03	0.95	1.94	0.13	0.03	2.05	3.09	2.01	0.07	0.20	1.03	0.11	0.10	1.04	0.10	0.17	0.92	0.09	0.15	1.04	2.01	4.04	1.18	0.20	0.03	1.98	0.94	0.07	0.07	1.13	1.01	1.98	0.12	0.17	0.94	0.16	0.03	1.07	0.17	0.97	2.90	1.04	0.99	0.18	0.01	2.01	2.05	0.20	0.08	0.96	0.05	1.03	0.05	2.92	0.14	0.94	0.15	2.96	0.08	0.95	1.00	0.19	2.95	0.20	1.82	0.20	0.06	1.02	0.03	0.99	1.00	0.05	0.16	1.11	0.13	3.95	1.12	0.05	0.92	0.05	0.98	2.02	0.96	0.21	0.21	0.97	0.16	0.93	2.06	0.12	0.08	1.09	0.03	0.00	1.04	0.06	1.94	2.13	0.13	0.05	2.01	0.18	0.95	0.18	0.17	0.99	0.96	1.14	1.06	1.04	0.12	0.14	0.95	1.97	0.97	1.09	0.09	0.97	0.09	0.91	0.11	0.12	0.93	3.15	1.01	1.10	0.13	0.15	0.96	0.07	0.01	0.98	0.92	0.22	0.00	2.04	1.85	1.15	0.00	0.85	0.99	0.04	0.01	1.95	0.11	0.16	0.98	0.18	0.08	1.08	0.06	1.00	0.09	0.00	0.84	0.98	0.12	2.10	1.12	0.03	0.09",
    "	0.98	0.00	1.03	0.01	0.00	0.99	0.01	1.02	1.01	1.10	0.04	0.06	1.92	0.00	0.96	1.02	1.01	0.00	0.04	0.95	1.02	0.00	0.93	0.10	0.00	1.04	0.10	0.96	0.02	0.00	0.97	0.03	0.01	1.08	0.02	0.02	0.97	0.00	0.03	0.99	0.06	0.00	0.94	0.07	0.97	0.00	0.11	1.00	0.07	0.00	2.00	0.12	0.99	0.02	2.95	1.07	1.03	1.08	0.11	2.05	0.07	0.99	0.05	0.99	3.02	0.06	0.13	1.94	0.13	1.00	1.95	1.10	0.99	0.00	0.16	1.01	1.02	0.00	0.92	0.09	0.98	0.01	0.96	0.09	0.15	0.99	0.19	0.99	1.97	0.16	2.02	0.13	0.18	2.05	0.14	0.10	0.98	0.11	0.04	0.99	1.04	0.05	0.12	4.95	0.14	0.98	1.92	0.17	1.98	0.04	1.90	0.12	0.98	0.00	0.99	0.07	1.01	0.09	1.08	0.10	0.11	0.95	0.17	1.00	0.16	2.03	3.90	0.16	1.00	1.02	0.09	0.13	0.99	0.12	2.06	0.12	0.16	1.03	0.18	0.11	1.01	0.04	0.98	0.95	0.07	2.05	0.17	1.98	2.91	0.07	0.07	1.93	1.06	0.08	0.09	1.00	0.11	0.98	0.09	1.05	0.07	0.10	1.99	1.07	2.07	1.01	1.94	0.16	0.94	0.14	0.97	0.13	0.10	0.95	1.90	0.12	0.13	2.00	1.17	0.16	1.00	0.06	0.10	0.96	0.16	0.01	1.94	0.17	0.91	2.07	0.15	0.05	0.94	0.00	0.13	1.89	0.17	2.06	0.97	0.96	0.06	0.08	1.06	0.14	0.11	1.05	0.00	0.09	1.01	0.09	3.79	0.23	0.12	0.96	0.19	0.15	0.95	0.15	1.10	1.01	0.14	0.94	0.06	0.12	0.99	1.93	1.92	1.02	0.02	2.84	0.13	0.14	1.76	0.00	1.02	0.09	2.83	1.99	0.12	0.05	0.99	1.89	1.09	0.14	0.18	2.00	0.04	0.97	0.12	0.98	2.00	0.11	0.17	0.98	0.11	0.16	1.05	0.13	0.08	0.91	0.17	0.05	0.99	2.00	1.04	0.09	1.02	0.16	1.01	0.16	0.10	1.05	0.00	1.00	0.14	0.15	1.98	0.96	0.09	0.10	1.06	1.02	2.08	0.15	0.20	1.06	0.17	0.00	1.07	0.18	1.09	3.12	1.06	0.93	0.13	0.02	2.11	2.18	0.11	0.13	0.89	0.04	0.96	0.08	3.03	0.18	0.92	0.19	2.89	0.05	1.01	0.98	0.20	3.04	0.11	1.91	0.17	0.12	0.98	0.04	1.03	1.09	0.00	0.22	1.07	0.15	3.92	1.12	0.05	0.94	0.06	0.95	1.96	0.98	0.13	0.00	1.00	0.04	0.99	2.12	0.21	0.16	1.01	0.00	0.05	1.07	0.13	2.02	2.13	0.17	0.21	1.94	0.22	0.88	0.09	0.21	1.13	1.04	1.13	1.00	0.93	0.11	0.09	1.04	1.99	1.09	0.98	0.13	0.98	0.10	0.79	0.04	0.22	1.00	3.00	1.11	1.10	0.15	0.17	0.88	0.12	0.05	0.99	0.97	0.10	0.05	1.96	1.90	1.05	0.04	0.86	0.96	0.07	0.09	2.03	0.14	0.32	1.01	0.22	0.12",
    "	0.97	0.00	1.00	0.00	0.02	1.02	0.00	1.06	1.00	1.22	0.08	0.19	1.86	0.00	1.00	0.91	0.91	0.11	0.00	0.95	1.04	0.10	1.02	0.11	0.02	1.05	0.11	0.91	0.00	0.11	0.92	0.00	0.11	1.06	0.00	0.10	1.02	0.00	0.09	0.94	0.14	0.00	0.95	0.12	0.98	0.00	0.16	0.94	0.16	0.00	2.15	0.13	1.05	0.10	2.74	1.19	0.96	1.03	0.05	1.94	0.33	0.93	0.00	1.20	2.80	0.04	0.15	1.96	0.16	0.96	1.87	0.99	1.03	0.02	0.12	0.94	1.06	0.00	1.06	0.06	0.95	0.10	0.93	0.14	0.18	1.01	0.27	0.92	1.93	0.19	1.89	0.08	0.17	1.96	0.12	0.16	1.06	0.07	0.01	1.23	0.93	0.01	0.19	4.76	0.18	0.90	1.97	0.16	1.88	0.04	2.02	0.08	0.92	0.01	0.94	0.03	1.09	0.05	1.03	0.09	0.14	1.02	0.00	1.19	0.09	1.86	3.92	0.34	0.94	0.87	0.11	0.11	1.05	0.12	2.00	0.19	0.13	1.03	0.16	0.03	1.02	0.11	1.01	1.04	0.14	2.01	0.16	2.00	3.14	0.15	0.15	1.98	0.92	0.06	0.25	0.85	0.22	0.92	0.15	1.11	0.24	0.11	1.81	1.13	1.94	1.09	1.83	0.17	0.95	0.12	1.12	0.17	0.28	0.99	2.03	0.05	0.14	1.84	1.05	0.11	0.90	0.04	0.24	0.93	0.17	0.19	2.36	0.11	1.03	2.16	0.00	0.18	1.00	0.00	0.15	1.94	0.19	2.31	0.97	0.92	0.17	0.00	0.94	0.14	0.10	0.99	0.07	0.22	1.02	0.12	4.11	0.16	0.15	1.08	0.00	0.15	1.07	0.16	1.01	1.12	0.15	0.88	0.20	0.05	0.93	1.95	1.96	0.97	0.06	2.64	0.23	0.12	1.94	0.12	1.12	0.02	2.75	2.10	0.25	0.03	0.81	1.81	0.10	0.10	1.15	0.00	0.95	0.14	0.06	3.24	2.37	0.11	0.27	0.87	0.02	0.13	0.94	0.04	0.13	0.94	0.15	0.19	1.05	2.14	1.03	0.12	1.06	0.05	0.96	0.16	0.10	0.97	0.56	1.18	0.11	0.05	2.11	1.02	0.23	0.21	1.05	0.95	2.12	0.10	0.10	1.02	0.19	0.08	1.03	0.14	1.03	3.02	1.17	0.95	0.04	0.01	1.97	2.09	0.18	0.12	0.90	0.00	0.94	0.22	3.08	0.24	0.96	0.13	2.94	0.07	1.28	0.99	0.13	3.07	0.19	1.82	0.19	0.06	1.01	0.00	1.21	1.03	0.00	0.13	1.03	0.12	4.24	1.37	0.21	0.95	0.00	0.83	2.19	1.07	0.09	0.07	0.95	0.02	1.10	2.13	0.25	0.12	1.05	0.02	0.10	1.06	0.21	2.04	2.36	0.16	0.29	1.85	0.08	1.00	0.06	0.04	1.03	0.89	1.22	0.97	1.00	0.14	0.15	0.97	2.00	1.07	0.99	0.06	0.91	0.02	1.28	0.09	0.15	0.95	2.63	1.02	1.08	0.19	0.04	1.02	0.07	0.08	1.18	0.83	0.14	0.09	1.71	1.93	1.19	0.12	0.82	0.77	0.23	0.16	1.77	0.08	0.35	0.85	0.06	0.17",
    "	1.01	0.00	1.02	0.05	0.03	0.98	0.04	0.95	1.00	1.11	0.00	0.02	2.00	0.00	1.00	1.00	1.00	0.00	0.01	1.01	1.01	0.00	1.00	0.03	0.01	1.00	0.00	0.98	0.02	0.00	0.98	0.06	0.03	1.00	0.05	0.04	0.98	0.00	0.04	0.99	0.07	0.00	1.02	0.07	0.98	0.00	0.09	0.96	0.10	0.00	2.01	0.11	0.99	0.02	2.97	1.03	1.03	1.00	0.13	1.99	0.19	0.96	0.05	1.03	3.00	0.09	0.09	1.94	0.19	1.05	1.98	0.95	1.04	0.03	0.10	0.97	0.98	0.00	1.00	0.10	0.99	0.03	1.01	0.14	0.14	0.97	0.14	1.00	2.04	0.10	1.97	0.12	0.16	1.96	0.13	0.13	0.99	0.07	0.04	1.03	0.99	0.07	0.15	4.87	0.15	1.03	2.03	0.06	1.96	0.06	2.01	0.08	1.00	0.00	0.93	0.08	0.99	0.04	1.00	0.15	0.11	1.00	0.11	1.04	0.13	2.01	4.08	0.15	1.05	1.01	0.19	0.01	1.02	0.09	2.00	0.08	0.13	0.99	0.13	0.10	0.97	0.14	1.03	1.01	0.10	2.01	0.17	2.07	3.13	0.09	0.02	2.07	1.00	0.10	0.04	0.92	0.11	1.06	0.19	0.97	0.19	0.13	2.18	1.02	2.06	1.10	2.03	0.11	1.03	0.14	0.94	0.04	0.10	0.97	2.03	0.10	0.11	1.93	1.01	0.15	1.02	0.11	0.09	1.00	0.15	0.08	1.99	0.15	0.99	1.95	0.09	0.18	0.95	0.14	0.14	1.98	0.19	1.96	1.11	0.98	0.06	0.02	0.98	0.18	0.11	0.99	0.07	0.01	0.99	0.06	4.04	0.13	0.13	0.96	0.16	0.12	0.97	0.10	1.06	1.03	0.18	0.94	0.08	0.12	1.01	1.94	1.98	1.04	0.04	3.05	0.20	0.14	1.90	0.03	1.07	0.05	2.78	2.06	0.15	0.04	0.97	1.94	0.83	0.09	0.32	1.50	0.30	0.98	0.16	1.65	2.14	0.17	0.15	0.99	0.11	0.13	0.99	0.09	0.16	0.95	0.16	0.03	1.03	1.99	1.01	0.01	1.01	0.14	1.02	0.13	0.19	1.02	0.06	1.03	0.10	0.15	2.06	1.02	0.07	0.04	1.03	1.00	2.06	0.11	0.15	0.97	0.17	0.00	0.99	0.17	1.08	3.03	1.12	0.98	0.10	0.04	1.99	2.09	0.13	0.10	0.92	0.05	0.96	0.08	2.91	0.17	0.99	0.17	3.09	0.02	1.05	0.99	0.16	3.08	0.21	1.94	0.11	0.13	0.96	0.08	1.01	1.00	0.03	0.18	1.04	0.13	4.22	1.17	0.04	0.96	0.08	0.98	2.07	0.97	0.21	0.05	0.94	0.08	1.03	1.98	0.18	0.12	1.05	0.03	0.05	0.99	0.08	2.05	2.17	0.18	0.12	1.91	0.16	0.92	0.18	0.15	1.01	0.96	1.09	1.05	1.03	0.14	0.13	1.02	1.94	1.00	1.10	0.11	0.95	0.08	0.93	0.12	0.21	0.95	3.00	1.06	1.06	0.01	0.09	0.91	0.08	0.04	0.94	0.97	0.20	0.00	2.02	1.94	1.16	0.04	0.89	0.96	0.09	0.08	1.98	0.18	0.20	0.91	0.24	0.15",
    "	0.99	0.00	1.01	0.03	0.01	1.00	0.03	0.99	1.00	1.10	0.00	0.02	1.98	0.00	0.98	0.97	1.00	0.00	0.05	0.97	1.00	0.00	0.97	0.06	0.04	1.02	0.01	0.99	0.04	0.00	0.99	0.04	0.02	1.05	0.04	0.03	0.99	0.00	0.01	0.99	0.05	0.00	0.98	0.08	1.00	0.00	0.07	1.00	0.07	0.00	1.98	0.09	1.00	0.00	2.98	1.00	1.01	1.04	0.11	2.05	0.17	1.01	0.05	0.99	3.03	0.07	0.08	2.01	0.17	1.02	1.94	1.01	1.01	0.00	0.13	0.98	0.99	0.00	1.00	0.10	0.99	0.00	0.99	0.10	0.15	1.00	0.16	0.99	2.01	0.11	1.96	0.10	0.13	2.03	0.15	0.09	1.00	0.08	0.05	0.99	1.01	0.04	0.14	4.94	0.10	1.01	1.99	0.10	2.01	0.04	1.99	0.10	1.00	0.00	0.95	0.07	1.02	0.08	1.04	0.14	0.11	0.99	0.08	1.01	0.13	2.01	3.99	0.16	1.00	1.00	0.14	0.04	1.00	0.11	2.03	0.09	0.12	1.00	0.13	0.10	0.98	0.14	1.02	1.04	0.04	2.03	0.16	1.96	2.96	0.03	0.06	1.91	1.00	0.08	0.05	0.95	0.13	0.99	0.20	0.99	0.09	0.11	2.09	1.08	2.05	1.08	1.94	0.17	0.95	0.12	1.03	0.08	0.14	0.97	1.99	0.08	0.12	1.96	1.03	0.12	0.96	0.14	0.21	0.99	0.18	0.04	2.04	0.15	1.00	1.99	0.06	0.08	0.95	0.06	0.13	1.99	0.16	1.97	1.04	0.94	0.06	0.03	1.03	0.19	0.10	0.97	0.06	0.08	1.04	0.15	3.87	0.16	0.19	0.98	0.12	0.14	1.01	0.09	1.05	1.02	0.12	0.94	0.04	0.10	0.99	1.94	2.01	1.04	0.02	2.86	0.19	0.15	1.92	0.07	1.03	0.07	2.83	1.96	0.11	0.07	0.90	2.08	0.22	0.04	1.08	0.00	0.91	0.06	0.19	3.10	2.07	0.09	0.15	0.97	0.10	0.11	1.07	0.06	0.22	0.92	0.21	0.07	1.01	2.00	1.04	0.08	1.00	0.15	1.02	0.12	0.18	1.03	0.04	1.00	0.14	0.17	2.06	1.00	0.07	0.07	1.04	1.00	2.07	0.10	0.17	1.00	0.19	0.02	1.03	0.18	1.06	3.03	1.01	0.98	0.14	0.03	2.08	2.12	0.11	0.14	0.91	0.04	1.01	0.08	3.02	0.17	0.94	0.15	2.93	0.03	0.98	1.04	0.11	2.99	0.11	1.86	0.13	0.07	0.99	0.06	1.03	1.03	0.00	0.19	1.03	0.13	4.03	1.15	0.00	0.91	0.06	0.96	2.03	0.95	0.13	0.01	1.02	0.05	1.01	2.02	0.15	0.15	1.06	0.02	0.04	1.05	0.07	2.00	2.16	0.16	0.11	1.93	0.20	0.91	0.14	0.17	1.06	0.98	1.07	1.04	1.00	0.13	0.13	1.01	1.98	1.07	1.03	0.15	0.93	0.10	0.87	0.11	0.21	0.99	3.00	1.07	1.11	0.03	0.18	0.90	0.11	0.04	1.00	1.00	0.21	0.02	1.95	1.89	1.08	0.02	0.89	0.96	0.12	0.10	1.93	0.14	0.22	0.94	0.18	0.14",
    "	1.02	0.00	0.97	0.02	0.00	1.01	0.02	1.04	0.92	1.09	0.05	0.00	1.95	0.00	0.98	1.02	1.04	0.00	0.01	1.00	1.01	0.00	0.90	0.04	0.04	1.03	0.07	1.00	0.02	0.00	0.98	0.03	0.02	1.09	0.05	0.01	0.96	0.00	0.00	0.99	0.05	0.00	0.96	0.05	0.99	0.00	0.09	1.00	0.06	0.00	2.02	0.08	1.00	0.00	2.99	1.00	1.02	1.05	0.13	2.02	0.14	1.02	0.04	0.99	3.02	0.06	0.11	1.96	0.15	1.02	1.96	1.04	0.97	0.00	0.16	1.04	0.98	0.00	0.94	0.11	1.01	0.00	0.97	0.08	0.13	1.00	0.15	1.00	1.99	0.12	1.97	0.12	0.14	2.02	0.16	0.09	0.97	0.13	0.04	0.97	1.03	0.03	0.12	4.89	0.09	1.02	1.94	0.16	1.97	0.05	1.95	0.14	0.99	0.00	0.94	0.07	1.01	0.02	1.09	0.10	0.13	0.98	0.08	1.00	0.11	2.01	3.91	0.18	0.99	0.99	0.17	0.12	0.98	0.11	2.01	0.12	0.12	1.03	0.13	0.10	1.03	0.11	1.02	0.97	0.04	1.98	0.16	1.96	2.88	0.09	0.07	1.89	1.03	0.06	0.08	0.96	0.16	0.99	0.16	0.99	0.13	0.09	1.99	1.08	2.04	1.02	1.95	0.20	0.96	0.11	1.01	0.14	0.17	0.97	1.94	0.12	0.11	1.98	1.08	0.10	1.01	0.08	0.12	0.94	0.11	0.15	1.92	0.16	0.95	1.97	0.13	0.08	0.97	0.13	0.13	1.92	0.17	1.99	1.03	0.97	0.06	0.03	0.97	0.17	0.11	1.06	0.07	0.09	1.00	0.13	3.88	0.22	0.11	0.99	0.20	0.15	0.96	0.09	1.09	0.99	0.13	0.91	0.10	0.10	1.02	1.95	1.98	1.00	0.06	2.85	0.14	0.12	1.84	0.04	1.03	0.11	2.71	2.00	0.18	0.09	1.09	1.88	1.02	0.12	0.11	1.94	0.07	0.96	0.08	0.94	1.93	0.10	0.11	0.98	0.07	0.17	1.00	0.09	0.17	0.98	0.15	0.06	1.04	2.04	1.04	0.08	1.00	0.10	1.02	0.12	0.07	1.03	0.02	0.99	0.15	0.17	2.11	0.98	0.59	0.08	0.46	0.99	2.10	0.13	0.20	1.00	0.13	0.00	1.12	0.13	1.08	3.09	1.04	0.98	0.12	0.01	2.02	2.09	0.10	0.09	0.86	0.01	1.02	0.08	2.96	0.15	0.89	0.15	2.86	0.07	1.01	1.05	0.12	2.98	0.11	1.83	0.16	0.06	1.01	0.06	1.06	0.99	0.00	0.21	1.04	0.14	3.99	1.10	0.02	0.93	0.13	0.96	2.17	0.98	0.11	0.01	0.95	0.02	1.01	2.10	0.20	0.12	1.03	0.01	0.03	1.02	0.12	1.97	2.20	0.14	0.14	1.88	0.14	0.95	0.14	0.18	1.05	0.94	1.06	1.06	0.99	0.10	0.10	1.10	1.92	1.10	1.07	0.08	0.87	0.08	0.82	0.14	0.07	1.00	2.98	1.03	1.09	0.16	0.12	0.95	0.07	0.01	1.05	0.92	0.08	0.00	1.92	1.99	1.12	0.02	0.83	0.90	0.10	0.07	2.04	0.11	0.20	0.98	0.21	0.15",
    "	1.02	0.00	1.03	0.02	0.01	0.95	0.03	0.96	1.97	0.00	0.99	1.03	0.98	0.00	0.04	0.98	1.02	0.00	0.96	0.04	0.01	1.04	0.04	0.99	0.00	0.00	0.98	0.01	0.00	1.01	0.02	0.02	0.99	0.00	0.03	1.01	0.03	0.00	0.96	0.05	0.97	0.00	0.09	0.98	0.04	0.00	1.99	0.08	1.01	0.00	2.88	1.04	1.03	1.06	0.09	1.96	0.13	1.01	0.06	0.97	2.92	0.05	0.09	1.98	0.16	1.01	1.94	1.05	0.97	0.01	0.12	0.96	1.01	0.00	0.97	0.08	0.99	0.01	1.00	0.07	0.11	0.99	0.13	1.00	1.97	0.12	1.92	0.09	0.16	1.95	0.16	0.08	0.97	0.09	0.02	1.05	1.00	0.00	0.13	4.77	0.17	0.99	1.95	0.14	1.94	0.05	1.98	0.05	1.01	0.00	0.97	0.06	0.98	0.05	1.04	0.12	0.08	0.97	0.09	1.02	0.09	2.00	3.92	0.15	0.97	1.04	0.10	0.04	1.02	0.15	2.00	0.05	0.14	1.01	0.14	0.06	1.02	0.13	0.97	1.00	0.01	2.01	0.14	1.96	2.94	0.08	0.09	1.91	1.00	0.09	0.07	0.96	0.14	0.98	0.18	0.99	0.08	0.13	2.00	1.12	2.02	1.07	2.02	0.14	0.98	0.10	1.00	0.05	0.12	0.99	1.94	0.08	0.08	1.97	0.99	0.14	0.97	0.07	0.11	0.97	0.16	0.14	1.95	0.10	0.99	1.99	0.13	0.04	0.94	0.03	0.18	1.94	0.16	1.96	1.00	0.97	0.05	0.06	1.03	0.16	0.08	1.00	0.05	0.03	1.03	0.11	3.74	0.12	0.12	1.00	0.10	0.11	0.99	0.13	1.07	1.00	0.13	0.93	0.04	0.12	1.02	1.89	2.02	0.98	0.01	2.82	0.09	0.15	1.88	0.08	1.02	0.07	2.77	1.89	0.12	0.05	1.02	1.96	1.04	0.10	0.08	1.86	0.06	0.96	0.11	0.99	1.97	0.16	0.16	1.06	0.11	0.13	1.04	0.10	0.13	0.92	0.10	0.10	0.94	1.98	1.02	0.13	1.06	0.17	1.05	0.08	0.10	1.06	0.01	0.99	0.12	0.15	2.04	0.97	0.07	0.07	0.98	0.94	2.08	0.12	0.05	1.09	0.15	0.00	1.07	0.16	1.07	2.95	1.01	1.00	0.09	0.01	1.91	2.03	0.09	0.06	0.98	0.07	0.94	0.11	2.77	0.16	0.98	0.08	2.97	0.01	1.00	0.93	0.17	2.92	0.20	1.83	0.12	0.09	1.06	0.10	0.96	1.01	0.00	0.18	1.11	0.05	3.82	1.13	0.02	0.91	0.10	0.97	2.05	1.01	0.12	0.00	0.94	0.06	1.00	2.08	0.20	0.12	1.03	0.00	0.03	1.06	0.11	1.95	2.05	0.18	0.14	1.91	0.20	0.94	0.13	0.21	1.06	0.99	1.10	1.11	0.91	0.14	0.05	1.00	1.98	1.00	1.09	0.12	1.01	0.11	0.85	0.10	0.07	0.96	2.92	1.03	1.02	0.05	0.17	0.92	0.10	0.04	1.07	0.95	0.09	0.00	1.93	1.79	1.05	0.03	0.87	0.97	0.03	0.08	1.94	0.15	0.17	0.98	0.20	0.15	1.17	0.14	1.29	0.12",
    "	0.99	0.00	1.02	0.03	0.00	0.99	0.02	0.96	1.01	1.11	0.00	0.02	2.02	0.00	1.00	1.01	0.99	0.00	0.03	0.96	1.01	0.00	0.92	0.04	0.01	1.00	0.06	1.01	0.00	0.00	0.98	0.05	0.00	1.05	0.03	0.03	0.97	0.00	0.02	1.00	0.05	0.00	0.97	0.08	0.99	0.00	0.10	0.99	0.07	0.00	1.98	0.10	1.01	0.00	2.98	1.05	1.00	1.01	0.11	2.05	0.17	0.98	0.02	0.98	3.01	0.06	0.05	2.06	0.13	1.04	1.96	1.02	0.98	0.00	0.09	1.00	1.01	0.00	1.02	0.08	0.99	0.01	0.94	0.10	0.12	0.97	0.14	0.98	2.00	0.13	1.96	0.08	0.13	2.03	0.11	0.09	1.03	0.05	0.04	1.06	1.00	0.03	0.11	4.80	0.19	0.98	1.92	0.05	2.01	0.03	1.93	0.09	0.94	0.00	0.94	0.07	1.05	0.01	1.04	0.13	0.06	0.99	0.03	1.04	0.10	2.02	3.92	0.16	0.98	0.96	0.13	0.07	1.02	0.10	1.97	0.05	0.11	1.03	0.21	0.07	1.03	0.15	1.00	0.98	0.18	2.00	0.17	2.02	2.96	0.17	0.06	1.91	1.01	0.07	0.07	0.99	0.12	1.06	0.12	0.95	0.16	0.10	2.05	1.06	2.10	1.05	2.01	0.14	0.95	0.13	1.01	0.09	0.12	1.01	1.94	0.07	0.10	2.02	0.97	0.16	0.98	0.16	0.21	1.00	0.18	0.12	1.92	0.14	1.00	2.06	0.06	0.18	0.95	0.11	0.13	2.04	0.14	2.00	1.09	0.96	0.04	0.08	0.97	0.19	0.10	0.92	0.06	0.07	0.98	0.15	3.78	0.10	0.13	0.96	0.12	0.12	1.03	0.14	1.07	1.05	0.11	1.00	0.07	0.12	1.00	1.85	2.01	0.97	0.01	2.90	0.16	0.13	1.93	0.03	1.04	0.08	2.94	2.04	0.13	0.04	0.90	2.02	0.09	0.02	2.11	2.97	1.90	0.09	0.14	0.96	0.06	0.13	1.07	0.10	0.16	0.95	0.03	0.13	1.10	1.98	4.01	1.13	0.12	0.04	1.97	0.92	0.05	0.07	1.07	0.98	1.96	0.11	0.08	0.95	0.14	0.00	1.08	0.18	1.04	2.89	1.03	0.99	0.10	0.02	2.05	2.04	0.18	0.09	0.95	0.03	1.03	0.07	2.90	0.09	0.91	0.17	2.96	0.03	0.97	0.99	0.10	2.96	0.21	1.89	0.16	0.12	1.01	0.02	1.07	1.04	0.00	0.19	1.02	0.13	3.91	1.10	0.13	0.95	0.04	0.94	2.11	0.96	0.14	0.00	1.00	0.04	1.03	2.07	0.13	0.12	0.98	0.00	0.00	1.04	0.20	1.97	2.12	0.12	0.12	2.03	0.12	1.00	0.15	0.15	1.10	0.92	1.07	1.13	0.97	0.98	0.17	0.06	2.01	1.03	0.97	0.12	0.96	0.11	0.88	0.17	0.19	1.00	3.00	1.14	1.12	0.15	0.15	0.94	0.03	0.03	0.96	0.97	0.06	0.01	2.01	1.90	1.10	0.01	0.85	0.94	0.03	0.09	1.99	0.12	0.14	0.99	0.20	0.15	1.13	0.04	1.01	0.11	0.18	0.91	0.99	0.10	2.12	1.10	0.11	0.10",
    "	1.00	0.00	0.99	0.02	0.00	1.01	0.04	1.01	0.99	1.08	0.02	0.00	2.03	0.00	0.96	0.98	1.02	0.00	0.01	1.01	0.98	0.00	0.99	0.05	0.03	1.01	0.05	0.99	0.02	0.00	0.97	0.05	0.02	1.08	0.07	0.02	0.96	0.00	0.04	0.98	0.06	0.00	0.97	0.07	0.96	0.00	0.10	0.96	0.06	0.00	1.97	0.10	0.99	0.00	2.96	1.05	1.03	1.08	0.13	2.01	0.12	0.99	0.06	0.98	2.96	0.06	0.09	1.94	0.17	1.01	1.96	1.03	1.03	0.03	0.13	0.98	1.01	0.00	0.96	0.08	1.00	0.00	0.97	0.09	0.11	0.97	0.17	0.99	2.01	0.12	2.00	0.10	0.14	2.03	0.17	0.07	1.02	0.10	0.06	0.99	1.02	0.05	0.12	4.90	0.14	0.98	2.00	0.11	1.99	0.06	1.95	0.11	0.95	0.00	1.02	0.07	1.03	0.04	1.00	0.10	0.08	1.00	0.09	1.03	0.12	1.99	3.91	0.18	1.03	0.98	0.21	0.03	0.99	0.14	2.00	0.06	0.14	1.03	0.14	0.10	1.01	0.13	0.99	0.99	0.09	2.06	0.18	2.06	3.06	0.06	0.07	1.96	1.01	0.07	0.07	0.93	0.13	1.03	0.19	0.99	0.07	0.15	2.08	1.05	2.03	1.08	2.02	0.13	1.00	0.14	0.95	0.08	0.09	0.96	1.95	0.10	0.07	1.97	1.09	0.13	0.99	0.02	0.18	0.97	0.12	0.09	2.04	0.15	0.96	2.07	0.10	0.15	0.97	0.04	0.17	1.96	0.15	2.01	1.03	0.97	0.04	0.00	1.02	0.17	0.07	0.99	0.00	0.06	1.02	0.14	3.91	0.15	0.16	0.98	0.15	0.09	0.99	0.09	1.05	1.01	0.15	0.92	0.21	0.16	1.00	1.97	2.03	1.05	0.02	2.83	0.16	0.14	1.87	0.04	1.05	0.07	2.78	1.92	0.16	0.06	1.00	1.97	0.63	0.08	0.47	0.99	0.59	0.47	0.11	2.06	2.04	0.11	0.13	0.96	0.09	0.11	1.02	0.05	0.17	0.92	0.14	0.06	1.04	1.97	1.09	0.09	0.96	0.10	1.04	0.14	0.17	1.04	0.02	1.02	0.13	0.15	2.05	0.97	0.11	0.07	0.99	1.00	2.12	0.10	0.14	0.97	0.16	0.03	1.01	0.19	1.07	3.06	1.08	1.04	0.21	0.00	2.07	2.10	0.17	0.09	0.91	0.03	0.91	0.10	2.92	0.17	1.02	0.15	2.93	0.03	1.00	1.02	0.07	3.01	0.11	1.86	0.12	0.07	1.03	0.06	0.99	1.06	0.00	0.18	0.98	0.12	4.06	1.17	0.12	0.89	0.11	0.96	2.09	0.98	0.14	0.20	0.98	0.04	1.03	2.06	0.16	0.15	1.05	0.04	0.03	1.00	0.07	1.96	2.12	0.12	0.13	1.99	0.19	0.93	0.18	0.12	1.04	0.98	1.17	1.02	1.03	0.08	0.16	0.94	2.02	1.04	1.08	0.10	0.91	0.11	0.88	0.20	0.15	0.99	3.08	1.07	1.11	0.16	0.14	0.92	0.04	0.00	1.02	0.95	0.08	0.00	2.03	1.94	1.04	0.01	0.85	1.00	0.05	0.06	1.99	0.10	0.18	0.94	0.19	0.11",
    "	0.99	0.00	0.99	0.03	0.00	1.02	0.02	0.95	1.00	1.10	0.02	0.02	1.96	0.00	0.98	0.98	1.02	0.00	0.00	1.01	1.02	0.00	0.96	0.03	0.00	1.02	0.00	0.97	0.03	0.03	0.96	0.04	0.04	1.06	0.03	0.05	0.99	0.00	0.05	0.97	0.07	0.00	0.97	0.11	0.97	0.00	0.17	0.96	0.08	0.00	2.02	0.15	1.02	0.03	2.86	1.03	1.00	1.07	0.12	1.98	0.12	0.98	0.03	1.03	2.92	0.09	0.13	1.94	0.12	1.00	1.89	1.00	1.01	0.00	0.16	0.99	1.03	0.00	1.00	0.08	0.99	0.02	0.96	0.13	0.16	0.96	0.18	0.95	1.97	0.12	1.89	0.09	0.17	1.99	0.13	0.14	1.00	0.07	0.06	1.14	0.98	0.06	0.16	4.73	0.13	0.99	1.96	0.09	2.00	0.02	1.94	0.12	0.96	0.00	0.94	0.09	0.99	0.07	1.04	0.15	0.08	1.04	0.17	1.00	0.13	2.00	3.88	0.14	1.04	0.98	0.11	0.03	0.98	0.15	1.94	0.09	0.14	1.01	0.16	0.14	0.99	0.03	1.02	1.05	0.10	2.02	0.16	2.03	3.05	0.08	0.04	1.97	0.96	0.11	0.04	0.97	0.13	1.01	0.10	0.96	0.09	0.14	2.21	1.05	2.11	1.08	1.94	0.16	1.01	0.10	0.99	0.10	0.12	0.95	1.97	0.12	0.15	2.04	1.05	0.17	0.98	0.14	0.08	0.98	0.15	0.03	2.01	0.17	0.97	2.03	0.12	0.06	0.95	0.02	0.16	1.96	0.14	1.96	1.11	0.93	0.04	0.13	1.00	0.17	0.11	0.98	0.06	0.07	1.00	0.10	3.85	0.16	0.11	0.96	0.20	0.14	0.98	0.15	1.05	1.05	0.16	0.98	0.06	0.14	0.96	1.94	2.01	1.01	0.00	2.93	0.07	0.14	1.95	0.00	1.06	0.06	2.86	1.99	0.15	0.06	0.90	2.01	0.08	0.05	1.04	0.04	0.99	0.04	0.16	3.00	1.90	0.10	0.15	0.97	0.08	0.13	1.03	0.07	0.06	0.92	0.19	0.07	1.00	1.92	1.06	0.06	0.99	0.17	1.04	0.14	0.18	1.03	0.02	1.00	0.16	0.15	2.04	0.98	0.09	0.07	1.03	0.99	2.06	0.12	0.17	1.02	0.19	0.00	1.03	0.03	1.04	3.04	1.07	0.95	0.11	0.04	2.03	2.10	0.07	0.10	0.95	0.03	0.96	0.08	2.89	0.16	0.97	0.19	2.98	0.00	1.00	1.01	0.18	2.98	0.08	1.87	0.12	0.14	1.00	0.03	0.98	0.97	0.00	0.04	1.12	0.16	4.02	1.07	0.05	0.96	0.06	0.97	2.02	0.98	0.13	0.00	0.99	0.05	1.02	2.07	0.14	0.15	0.98	0.02	0.02	1.08	0.08	2.01	2.12	0.14	0.14	1.90	0.20	0.93	0.18	0.21	1.07	0.93	1.13	1.01	0.97	0.17	0.17	1.04	1.96	1.02	1.06	0.12	0.99	0.16	0.86	0.08	0.05	0.93	3.01	1.06	1.09	0.14	0.20	0.90	0.06	0.01	1.00	1.00	0.04	0.01	2.05	1.88	1.11	0.02	0.80	0.98	0.04	0.08	2.09	0.12	0.22	0.99	0.23	0.14",
    "	0.96	0.00	0.99	0.04	0.01	1.05	0.04	0.98	1.02	1.10	0.01	0.03	1.97	0.02	0.98	1.03	1.02	0.00	0.03	0.96	0.98	0.00	0.94	0.06	0.00	1.02	0.10	0.98	0.00	0.00	0.94	0.06	0.05	1.08	0.05	0.06	1.01	0.01	0.06	0.97	0.09	0.00	0.95	0.11	0.96	0.00	0.17	0.96	0.09	0.00	2.00	0.12	1.01	0.05	2.85	1.04	1.03	1.06	0.14	1.90	0.09	0.98	0.06	1.01	2.97	0.09	0.10	1.99	0.18	1.04	2.00	0.99	1.07	0.02	0.10	0.96	0.99	0.00	0.98	0.07	0.94	0.02	0.95	0.10	0.14	1.01	0.18	1.01	2.05	0.15	1.98	0.09	0.12	1.98	0.09	0.12	1.00	0.04	0.04	1.06	0.99	0.06	0.15	4.82	0.18	1.03	2.05	0.06	2.03	0.05	1.89	0.09	0.95	0.00	0.97	0.07	1.03	0.05	0.97	0.13	0.07	1.01	0.10	1.02	0.10	2.04	3.94	0.16	0.99	1.01	0.17	0.04	1.00	0.13	1.95	0.05	0.15	1.02	0.13	0.07	1.00	0.11	1.01	0.99	0.06	1.98	0.13	1.91	2.98	0.07	0.07	1.97	1.03	0.07	0.07	0.98	0.15	1.04	0.13	0.97	0.07	0.14	2.01	1.05	2.06	1.07	2.00	0.16	0.93	0.10	0.96	0.07	0.12	0.96	1.95	0.09	0.14	2.09	1.03	0.13	1.01	0.02	0.16	1.05	0.17	0.14	1.96	0.12	0.96	1.95	0.13	0.04	0.95	0.11	0.14	1.92	0.09	1.91	1.05	0.97	0.04	0.02	1.01	0.17	0.10	0.95	0.03	0.07	1.06	0.07	3.83	0.14	0.13	0.98	0.15	0.12	1.04	0.12	1.00	0.99	0.15	0.92	0.02	0.09	1.05	1.81	1.97	1.01	0.06	2.86	0.08	0.16	1.94	0.00	1.05	0.11	2.77	1.97	0.12	0.08	0.92	1.88	1.09	0.16	0.09	1.92	0.07	0.98	0.10	0.96	2.02	0.10	0.14	0.99	0.13	0.17	1.03	0.06	0.19	0.97	0.16	0.12	0.99	2.00	1.03	0.11	1.02	0.12	1.04	0.08	0.08	1.03	0.00	0.95	0.13	0.06	2.06	0.98	0.09	0.08	1.02	0.99	2.06	0.15	0.19	1.01	0.12	0.00	1.08	0.05	1.14	3.01	1.02	0.94	0.18	0.03	2.11	2.03	0.09	0.14	0.88	0.05	0.95	0.12	2.96	0.17	0.97	0.08	2.95	0.05	1.03	0.97	0.14	2.93	0.12	1.88	0.08	0.04	1.04	0.09	1.05	1.07	0.00	0.07	0.98	0.15	3.91	1.11	0.03	0.95	0.06	0.97	2.02	0.91	0.12	0.01	1.01	0.04	1.00	2.02	0.10	0.06	1.07	0.00	0.01	0.97	0.06	1.94	2.19	0.16	0.16	2.02	0.15	0.92	0.13	0.11	1.05	0.95	1.11	1.00	0.99	0.12	0.13	1.09	1.97	1.00	1.14	0.13	0.90	0.06	0.80	0.09	0.18	0.97	3.07	1.08	1.11	0.15	0.14	0.91	0.06	0.03	1.03	0.91	0.06	0.03	1.91	1.91	1.09	0.06	0.91	0.94	0.05	0.14	2.02	0.14	0.22	1.00	0.15	0.17",
    "	1.00	0.01	1.02	0.01	0.01	0.97	0.00	0.98	1.00	1.03	0.01	0.00	1.98	0.02	0.97	1.00	1.04	0.01	0.00	1.00	1.02	0.01	0.97	0.03	0.00	0.99	0.00	1.00	0.01	0.00	0.98	0.02	0.00	1.01	0.01	0.00	0.99	0.03	0.00	1.00	0.04	0.00	0.97	0.02	1.02	0.02	0.04	1.00	0.06	0.00	2.00	0.06	0.99	0.03	2.96	1.01	1.03	1.03	0.09	2.02	0.13	0.94	0.00	1.00	3.06	0.10	0.03	2.01	0.16	1.02	2.07	0.98	1.04	0.05	0.10	0.98	0.98	0.02	0.99	0.04	0.99	0.05	0.99	0.06	0.13	0.96	0.11	1.03	2.09	0.13	2.05	0.07	0.14	1.94	0.12	0.06	1.01	0.12	0.00	0.99	0.98	0.07	0.10	4.99	0.08	1.01	2.06	0.06	2.06	0.09	1.98	0.08	0.99	0.02	0.93	0.06	1.02	0.08	1.00	0.10	0.11	0.95	0.00	1.08	0.11	1.96	4.07	0.12	1.00	0.97	0.11	0.01	1.01	0.16	2.04	0.05	0.15	1.02	0.09	0.07	1.02	0.14	1.03	0.97	0.04	1.93	0.13	2.06	3.00	0.08	0.03	1.97	0.99	0.11	0.01	0.97	0.10	1.01	0.14	1.03	0.19	0.17	2.19	1.05	2.07	1.02	2.00	0.14	0.99	0.13	0.94	0.08	0.12	0.92	1.99	0.08	0.11	1.94	1.03	0.12	1.04	0.01	0.16	1.03	0.10	0.17	2.04	0.12	0.99	1.92	0.01	0.05	0.91	0.08	0.14	1.98	0.15	1.93	1.05	0.93	0.06	0.00	1.02	0.16	0.10	0.97	0.00	0.02	1.02	0.09	3.93	0.09	0.11	0.96	0.01	0.07	1.01	0.13	1.06	1.03	0.09	0.92	0.00	0.07	1.01	1.78	1.95	1.04	0.00	2.63	0.11	0.13	1.96	0.00	1.03	0.05	2.88	1.91	0.12	0.03	0.87	1.90	0.53	0.07	0.98	2.35	1.05	0.56	0.07	0.94	1.02	0.15	0.57	0.58	0.06	0.52	0.47	0.14	0.52	1.46	2.11	0.59	0.50	0.94	1.45	0.58	0.47	0.11	1.06	0.54	1.10	0.45	0.05	0.98	0.08	0.16	1.61	0.50	0.59	1.56	0.95	0.92	1.03	0.02	1.03	1.46	0.16	0.05	0.97	0.13	1.02	1.55	2.10	0.60	0.50	0.00	2.38	0.89	0.44	0.49	0.43	1.30	0.51	0.97	1.38	0.12	0.85	0.07	1.87	0.53	0.34	0.48	0.41	1.26	1.97	1.46	0.07	0.37	0.36	0.36	1.58	1.05	0.01	0.00	1.06	0.10	2.83	1.78	0.09	0.51	0.33	0.34	1.01	1.00	0.17	1.04	1.59	0.07	0.63	2.13	0.10	0.57	0.40	0.15	0.65	0.95	0.60	1.57	1.60	0.15	0.04	1.56	0.99	1.13	0.64	0.03	0.98	0.43	0.98	0.63	0.55	0.47	1.47	1.01	1.48	0.46	0.58	0.54	0.44	0.18	1.14	0.54	0.17	0.53	2.63	1.66	0.79	0.12	0.80	0.92	0.05	0.11	1.61	0.53	0.11	0.52	0.93	1.17	1.14	0.20	0.98	0.44	0.14	0.47	1.67	0.10	1.09	1.03	0.03	0.18",
    "	1.03	0.00	1.00	0.03	0.00	0.97	0.03	1.05	1.02	1.09	0.01	0.01	2.01	0.00	1.02	0.98	1.01	0.00	0.02	0.97	1.02	0.00	0.96	0.04	0.07	0.98	0.05	0.98	0.02	0.00	0.96	0.03	0.02	1.02	0.04	0.04	1.02	0.01	0.04	0.98	0.08	0.00	0.97	0.10	0.99	0.00	0.14	0.99	0.09	0.00	1.88	0.11	1.03	0.02	3.04	0.97	1.06	1.00	0.16	1.99	0.18	1.00	0.04	1.00	3.03	0.12	0.09	2.04	0.13	1.00	2.07	0.98	1.04	0.02	0.13	0.98	0.98	0.00	1.00	0.08	1.00	0.04	1.00	0.14	0.11	0.95	0.11	0.97	2.01	0.08	1.98	0.08	0.17	1.92	0.13	0.16	1.01	0.08	0.05	1.14	0.95	0.09	0.08	4.85	0.12	0.96	2.05	0.09	1.95	0.08	1.98	0.10	0.98	0.00	0.96	0.06	0.98	0.07	0.97	0.15	0.06	1.06	0.15	1.09	0.13	2.03	3.93	0.17	0.98	1.00	0.15	0.00	0.99	0.11	1.97	0.08	0.14	1.04	0.17	0.16	0.99	0.01	1.00	1.01	0.10	1.98	0.17	2.00	3.01	0.06	0.09	2.02	0.99	0.13	0.04	0.95	0.16	0.99	0.03	1.00	0.09	0.14	1.97	0.99	2.00	1.06	2.03	0.16	0.98	0.12	1.09	0.08	0.12	1.04	1.94	0.12	0.12	2.00	0.99	0.16	0.96	0.04	0.10	0.97	0.17	0.02	1.94	0.13	0.94	1.47	0.10	0.49	0.96	0.04	0.19	1.94	0.14	1.86	1.03	1.01	0.04	0.08	1.00	0.07	0.15	0.96	0.14	0.07	1.02	0.13	3.76	0.13	0.16	0.97	0.07	0.16	0.99	0.02	1.04	1.05	0.15	0.97	0.11	0.16	1.00	1.94	1.92	1.00	0.10	2.71	0.11	0.15	1.87	0.03	1.00	0.08	2.82	2.06	0.10	0.07	1.00	1.96	1.05	0.09	0.12	2.00	0.11	0.95	0.08	0.97	1.96	0.17	0.06	1.03	0.07	0.17	0.99	0.07	0.19	1.01	0.03	0.08	1.03	1.95	1.00	0.09	0.99	0.14	1.03	0.07	0.11	1.06	0.05	0.94	0.13	0.16	2.09	0.97	0.10	0.08	1.01	1.01	2.11	0.14	0.20	1.02	0.20	0.00	1.09	0.00	1.03	3.00	1.02	1.03	0.10	0.04	2.06	2.11	0.06	0.12	0.94	0.08	0.98	0.10	2.92	0.16	0.98	0.06	2.94	0.17	0.99	0.97	0.15	3.05	0.13	1.93	0.11	0.13	0.98	0.08	0.98	1.06	0.02	0.03	1.07	0.16	4.00	1.10	0.04	0.92	0.08	0.97	2.05	1.03	0.10	0.00	0.97	0.02	0.96	2.02	0.19	0.15	1.06	0.00	0.02	1.13	0.09	1.91	2.15	0.02	0.11	2.04	0.20	0.89	0.09	0.01	0.98	1.02	1.09	1.06	0.95	0.17	0.20	0.98	2.02	1.03	1.16	0.10	0.93	0.13	0.90	0.06	0.06	0.95	2.96	1.09	1.11	0.17	0.17	0.88	0.08	0.02	0.94	0.91	0.05	0.01	2.02	1.96	1.04	0.03	0.96	0.96	0.07	0.09	1.97	0.10	0.23	0.95	0.20	0.12",
    "	0.99	0.00	0.99	0.04	0.00	1.02	0.01	1.01	0.99	1.07	0.01	0.03	1.99	0.00	1.00	0.97	1.00	0.00	0.00	1.00	1.02	0.00	0.95	0.09	0.00	1.06	0.04	0.97	0.00	0.02	0.93	0.04	0.05	1.03	0.03	0.08	0.97	0.00	0.06	0.97	0.10	0.00	0.94	0.11	0.98	0.00	0.17	0.95	0.09	0.00	2.07	0.14	1.03	0.06	2.85	1.10	0.97	1.07	0.10	1.95	0.12	0.93	0.05	1.05	2.97	0.08	0.12	2.01	0.16	1.03	1.94	0.97	1.05	0.01	0.11	0.93	1.02	0.00	0.98	0.06	0.97	0.05	0.98	0.12	0.15	1.01	0.18	0.97	2.00	0.15	1.95	0.06	0.15	1.98	0.07	0.13	1.02	0.05	0.03	1.13	0.97	0.06	0.09	4.67	0.11	0.96	2.00	0.04	1.98	0.07	1.89	0.06	0.94	0.01	1.01	0.07	1.03	0.02	0.99	0.14	0.07	1.03	0.14	1.02	0.10	2.01	3.87	0.15	0.98	0.99	0.10	0.04	0.99	0.09	2.04	0.04	0.11	1.04	0.13	0.13	0.98	0.15	0.99	1.02	0.11	2.01	0.17	1.96	3.00	0.05	0.08	1.94	0.98	0.08	0.08	0.96	0.12	0.96	0.12	1.03	0.12	0.13	2.04	1.11	1.99	1.02	2.01	0.10	0.96	0.13	1.02	0.05	0.13	0.99	1.95	0.08	0.09	1.98	0.95	0.13	1.01	0.05	0.09	1.00	0.15	0.02	1.95	0.11	0.98	1.96	0.10	0.07	0.91	0.01	0.16	1.93	0.15	1.95	1.13	0.90	0.03	0.06	1.04	0.09	0.10	0.95	0.10	0.11	1.02	0.07	3.81	0.09	0.09	0.97	0.07	0.11	1.04	0.14	1.06	0.99	0.14	0.92	0.08	0.11	1.02	1.96	1.97	1.02	0.04	2.76	0.12	0.18	2.00	0.00	1.00	0.09	2.88	2.09	0.15	0.04	0.93	1.97	0.15	0.03	1.99	3.00	1.97	0.09	0.14	0.94	0.06	0.16	1.06	0.04	0.11	1.04	0.03	0.12	1.05	1.92	3.99	1.10	0.11	0.01	1.84	0.97	0.00	0.10	0.99	0.97	2.00	0.11	0.17	1.00	0.13	0.04	1.11	0.05	1.04	2.83	1.08	0.95	0.10	0.03	1.90	1.92	0.11	0.12	0.97	0.09	0.99	0.10	2.85	0.14	0.97	0.17	2.93	0.12	0.94	0.96	0.15	2.87	0.09	1.84	0.17	0.09	1.00	0.06	1.06	1.08	0.01	0.09	1.05	0.13	3.84	1.05	0.13	0.96	0.07	0.94	1.96	0.94	0.13	0.00	0.97	0.07	1.04	2.06	0.15	0.15	1.05	0.00	0.13	1.12	0.08	1.93	2.17	0.11	0.13	1.92	0.10	0.92	0.12	0.13	1.05	0.92	1.02	1.08	0.98	0.13	0.16	0.97	1.99	1.06	1.11	0.10	0.91	0.08	0.91	0.12	0.15	1.01	2.79	1.06	1.05	0.00	0.07	0.90	0.04	0.06	1.04	0.92	0.08	0.01	1.93	1.87	1.12	0.05	0.93	0.91	0.06	0.10	1.95	0.08	0.17	0.99	0.16	0.13	1.04	0.07	1.05	0.12	0.00	0.91	1.08	0.12	2.01	0.97	0.10	0.12",
    "	1.00	0.00	1.03	0.05	0.02	0.97	0.02	0.97	1.00	1.08	0.00	0.01	2.04	0.00	1.00	1.04	1.03	0.00	0.00	0.98	1.02	0.00	0.98	0.05	0.00	1.01	0.03	0.98	0.00	0.00	0.95	0.03	0.00	1.00	0.03	0.03	0.99	0.00	0.01	1.01	0.04	0.00	0.96	0.04	0.99	0.00	0.07	0.98	0.06	0.00	1.96	0.09	1.01	0.00	3.01	1.07	1.05	1.01	0.09	2.03	0.18	0.95	0.04	0.97	3.14	0.07	0.07	2.00	0.12	1.06	2.08	0.98	1.04	0.00	0.05	0.97	0.98	0.00	1.06	0.05	0.98	0.02	0.94	0.13	0.13	0.96	0.16	1.01	2.06	0.12	2.00	0.08	0.13	2.00	0.07	0.11	1.02	0.01	0.04	1.11	0.97	0.04	0.17	4.88	0.12	1.01	2.07	0.00	2.01	0.05	1.90	0.09	0.92	0.00	0.93	0.07	1.02	0.05	0.99	0.14	0.05	1.03	0.09	1.08	0.13	2.06	4.14	0.14	1.02	0.94	0.11	0.00	0.99	0.08	1.97	0.05	0.14	1.02	0.08	0.09	1.01	0.13	1.01	1.10	0.05	2.05	0.11	2.07	3.08	0.00	0.03	1.98	0.95	0.10	0.03	0.92	0.13	1.06	0.18	1.00	0.10	0.14	2.26	1.04	2.16	1.10	2.07	0.12	0.97	0.07	0.98	0.02	0.11	0.98	2.06	0.09	0.10	2.04	0.94	0.14	1.01	0.11	0.16	1.02	0.12	0.13	2.03	0.14	1.06	2.01	0.01	0.05	0.94	0.07	0.15	2.03	0.13	2.00	1.06	0.92	0.04	0.00	0.98	0.13	0.11	0.95	0.04	0.04	1.08	0.13	3.99	0.10	0.08	0.97	0.06	0.13	0.99	0.10	1.04	1.00	0.10	0.94	0.03	0.10	1.03	1.94	2.03	1.03	0.00	2.88	0.11	0.13	2.05	0.13	1.07	0.07	2.98	2.11	0.14	0.04	0.81	1.95	0.58	0.07	1.06	2.53	1.05	0.49	0.09	0.96	1.01	0.15	0.59	0.44	0.08	0.54	0.54	0.09	0.55	1.56	2.29	0.61	0.65	0.89	1.60	0.55	0.43	0.07	1.04	0.54	1.06	0.50	0.08	0.99	0.12	0.16	1.62	0.49	0.59	1.58	0.97	0.98	0.97	0.01	1.09	1.47	0.13	0.06	0.96	0.04	1.08	1.46	2.06	0.54	0.37	0.00	2.46	0.97	0.55	0.57	0.49	1.47	0.55	0.97	1.58	0.04	0.98	0.08	2.01	0.59	0.53	0.61	0.46	1.57	2.25	1.61	0.12	0.36	0.40	0.38	1.73	1.07	0.09	0.03	0.94	0.05	2.81	1.78	0.23	0.51	0.32	0.40	1.14	1.00	0.14	1.03	1.56	0.03	0.57	2.10	0.09	0.54	0.50	0.14	0.64	0.93	0.47	1.66	1.62	0.11	0.20	1.33	0.94	1.11	0.66	0.07	1.00	0.41	0.99	0.59	0.49	0.45	1.48	1.03	1.55	0.37	0.40	0.43	0.31	0.13	1.03	0.54	0.09	0.41	2.56	1.57	1.07	0.09	0.25	0.79	0.00	0.04	1.58	0.54	0.19	0.42	0.87	1.00	1.17	0.11	0.93	0.41	0.13	0.46	1.47	0.07	1.13	0.91	0.01	0.15",
    "	0.98	0.00	1.00	0.03	0.04	1.02	0.04	1.00	0.98	1.11	0.00	0.02	1.98	0.00	0.99	1.00	1.00	0.00	0.02	0.97	1.03	0.00	0.97	0.05	0.02	1.04	0.06	1.01	0.00	0.01	0.98	0.04	0.04	1.03	0.06	0.03	0.97	0.02	0.04	0.99	0.07	0.00	0.98	0.08	0.98	0.00	0.12	0.97	0.09	0.00	2.00	0.11	1.01	0.03	2.92	1.01	1.03	1.04	0.15	1.98	0.16	0.96	0.05	1.02	2.94	0.07	0.10	1.95	0.16	1.00	1.96	1.00	1.03	0.02	0.11	1.03	0.97	0.00	0.99	0.08	0.99	0.02	0.98	0.08	0.13	0.98	0.16	1.00	2.03	0.11	1.99	0.09	0.13	2.00	0.14	0.08	1.00	0.09	0.06	1.00	0.98	0.05	0.13	4.83	0.11	1.04	1.97	0.03	2.00	0.06	2.02	0.09	0.99	0.00	0.95	0.06	1.02	0.06	0.99	0.12	0.10	1.00	0.06	1.03	0.11	1.98	3.99	0.15	1.00	0.98	0.13	0.03	1.00	0.12	1.98	0.07	0.13	1.04	0.11	0.10	0.99	0.11	1.02	1.03	0.04	1.98	0.17	2.00	3.07	0.03	0.07	2.03	1.00	0.07	0.05	0.95	0.09	1.05	0.11	0.94	0.18	0.14	2.11	1.04	2.11	1.08	2.00	0.16	0.98	0.14	0.98	0.08	0.13	0.99	2.02	0.10	0.09	1.97	0.98	0.13	1.01	0.00	0.16	1.00	0.11	0.09	2.07	0.15	1.00	1.98	0.07	0.13	0.94	0.08	0.12	2.04	0.16	1.93	1.04	0.96	0.05	0.02	1.05	0.09	0.10	0.96	0.00	0.06	1.06	0.09	3.97	0.15	0.13	0.93	0.18	0.09	1.00	0.11	1.00	1.09	0.12	0.92	0.06	0.10	0.97	1.88	2.03	1.02	0.04	2.92	0.19	0.17	1.89	0.05	1.08	0.07	2.86	2.04	0.14	0.04	0.94	1.91	0.09	0.04	2.05	3.00	2.01	0.05	0.16	0.99	0.13	0.10	0.99	0.12	0.15	0.96	0.16	0.06	1.01	2.04	1.07	0.09	1.03	0.15	1.00	0.12	0.09	0.96	0.09	0.98	0.11	0.18	2.06	1.01	0.06	0.08	1.05	0.96	2.13	0.10	0.16	1.00	0.16	0.01	1.04	0.14	1.03	3.00	1.03	1.03	0.12	0.05	2.03	1.96	0.20	0.10	0.92	0.05	1.05	0.09	2.82	0.15	0.94	0.15	2.94	0.07	0.98	1.02	0.19	2.92	0.21	1.88	0.10	0.10	1.00	0.08	1.04	0.97	0.00	0.18	1.04	0.14	4.08	1.08	0.02	0.96	0.09	0.97	2.09	0.98	0.20	0.18	0.99	0.02	1.02	2.04	0.17	0.14	1.02	0.05	0.02	1.01	0.15	1.96	2.10	0.15	0.11	1.93	0.18	0.94	0.16	0.15	1.06	0.96	1.17	1.05	0.96	0.13	0.15	1.00	1.95	0.95	1.03	0.09	1.01	0.09	0.92	0.18	0.17	0.93	3.07	1.04	1.08	0.01	0.18	0.91	0.11	0.04	1.04	0.96	0.21	0.00	1.98	1.90	1.04	0.02	0.87	0.94	0.09	0.05	1.92	0.09	0.18	0.93	0.19	0.13	1.17	0.00	1.16	0.11",
    "	0.99	0.00	0.97	0.03	0.03	1.03	0.00	1.02	0.97	1.08	0.04	0.04	1.84	0.05	0.96	0.99	0.97	0.03	0.01	1.02	0.96	0.00	0.99	0.09	0.05	1.02	0.07	0.99	0.07	0.01	1.01	0.04	0.05	1.08	0.04	0.03	0.97	0.01	0.01	1.01	0.09	0.00	1.00	0.08	0.98	0.00	0.08	1.00	0.11	0.00	2.02	0.10	0.97	0.02	3.11	1.00	1.04	1.06	0.13	2.08	0.19	1.04	0.04	0.94	3.02	0.07	0.05	2.03	0.10	0.99	2.06	1.05	0.99	0.01	0.12	1.03	0.98	0.00	0.96	0.11	1.01	0.00	1.02	0.08	0.19	1.00	0.14	0.99	1.99	0.14	2.11	0.11	0.19	2.07	0.05	0.10	1.00	0.15	0.04	1.01	1.00	0.05	0.10	5.03	0.08	1.01	2.07	0.06	2.03	0.06	2.05	0.11	0.95	0.02	0.98	0.09	1.03	0.07	1.05	0.12	0.15	1.00	0.02	1.04	0.11	2.05	4.75	0.16	0.50	0.95	0.13	0.08	0.99	0.07	2.07	0.09	0.18	1.02	0.13	0.10	1.06	0.00	1.02	1.03	0.00	2.06	0.08	1.95	2.98	0.00	0.08	1.93	0.97	0.11	0.06	0.94	0.09	1.03	0.03	1.01	0.08	0.14	2.20	1.12	2.04	1.07	2.05	0.10	0.91	0.06	0.95	0.11	0.18	1.00	2.08	0.13	0.19	1.98	1.00	0.14	1.05	0.00	0.01	1.00	0.12	0.00	2.10	0.12	1.03	2.04	0.48	1.09	0.39	0.92	0.59	1.33	0.12	0.77	0.96	0.36	0.12	0.46	0.49	0.13	0.72	0.52	2.34	0.11	0.34	0.58	1.64	0.15	0.62	0.38	0.59	0.63	0.39	0.51	0.48	0.43	0.54	1.49	1.20	0.60	0.46	2.29	0.89	0.37	0.93	0.94	0.49	0.10	2.06	0.87	0.45	0.05	1.47	1.73	0.08	0.07	1.34	1.66	1.75	0.08	0.14	1.00	0.43	0.18	0.72	0.92	0.13	1.09	0.10	0.46	1.60	1.14	2.38	1.15	0.04	0.14	1.60	0.57	0.11	0.56	0.67	0.63	1.83	0.95	0.58	0.71	0.50	0.01	1.16	0.13	0.56	2.34	0.81	1.11	0.18	0.00	2.17	1.76	0.18	0.12	0.92	0.45	1.67	0.17	1.86	0.50	0.58	0.28	2.26	0.07	1.05	1.90	0.59	2.37	0.20	0.92	1.03	0.81	0.66	0.05	1.12	0.47	0.55	0.20	2.00	0.09	2.82	0.89	1.12	0.73	0.30	0.96	1.13	1.81	0.08	0.78	0.72	0.00	1.15	1.08	0.60	0.55	0.85	0.10	0.63	0.60	1.85	1.83	1.36	0.54	0.00	1.42	0.88	1.03	0.21	0.11	0.96	0.46	1.25	1.63	0.69	0.17	0.37	0.61	1.30	1.10	0.61	0.94	1.36	0.07	0.62	0.77	0.22	1.13	1.96	0.67	1.17	0.29	0.29	1.14	0.54	0.11	0.74	0.89	0.98	0.44	1.59	1.10	0.91	0.05	0.97	0.58	0.14	0.41	2.30	0.55	0.57	0.43	0.08	0.48	0.61	0.05	1.21	0.59	0.25	0.46	1.55	1.00	1.31	0.47	0.44	0.43	0.81	0.42	0.94	0.22",
    "	0.98	0.00	1.01	0.05	0.04	1.01	0.02	0.95	0.99	1.03	0.00	0.00	2.08	0.00	1.00	1.01	1.05	0.00	0.00	0.98	1.04	0.00	1.00	0.06	0.00	0.97	0.03	0.97	0.05	0.00	0.99	0.04	0.03	1.04	0.03	0.03	0.97	0.00	0.02	1.00	0.08	0.00	0.99	0.07	1.00	0.00	0.07	0.98	0.09	0.00	1.99	0.12	0.98	0.00	3.05	1.03	1.03	1.00	0.12	1.97	0.18	0.95	0.02	1.02	3.04	0.09	0.10	1.95	0.19	1.04	2.03	1.00	1.01	0.00	0.15	0.96	0.97	0.00	1.02	0.09	0.99	0.01	1.00	0.12	0.16	0.95	0.15	1.00	2.08	0.08	1.98	0.11	0.17	1.96	0.10	0.12	1.00	0.03	0.04	1.07	0.98	0.04	0.14	4.89	0.14	1.04	2.05	0.00	2.02	0.02	2.00	0.09	0.96	0.00	0.94	0.08	0.99	0.03	0.98	0.15	0.07	1.04	0.12	1.07	0.14	2.07	4.30	0.12	1.07	0.88	0.12	0.00	0.96	0.10	2.06	0.05	0.14	0.97	0.08	0.11	1.05	0.10	1.08	1.04	0.06	2.01	0.15	2.11	3.10	0.00	0.01	2.00	0.96	0.08	0.01	0.92	0.13	1.07	0.08	0.98	0.20	0.11	2.34	1.01	2.21	1.14	2.14	0.07	0.97	0.11	0.88	0.07	0.07	0.96	2.09	0.10	0.11	2.01	0.95	0.15	1.07	0.06	0.11	1.10	0.12	0.10	2.17	0.08	0.99	2.05	0.00	0.04	0.95	0.01	0.14	1.99	0.15	1.92	1.15	0.89	0.06	0.00	0.99	0.10	0.14	0.94	0.00	0.00	1.09	0.04	4.12	0.12	0.10	0.95	0.02	0.12	1.02	0.05	1.01	1.03	0.12	0.87	0.00	0.08	1.03	1.89	2.12	1.06	0.19	2.82	0.11	0.11	1.88	0.10	1.00	0.04	2.63	1.96	0.08	0.03	0.82	1.74	0.03	0.01	1.47	1.33	1.52	0.06	0.02	1.80	0.96	0.11	0.60	0.43	0.04	0.57	0.52	0.10	0.53	1.53	2.31	0.56	0.55	0.92	1.53	0.50	0.53	0.05	1.01	0.50	1.11	0.46	0.05	0.94	0.09	0.17	1.58	0.48	0.58	1.59	1.04	0.97	0.98	0.01	1.09	1.54	0.17	0.17	0.95	0.02	1.02	1.55	2.12	0.53	0.53	0.01	2.41	0.90	0.58	0.56	0.47	1.41	0.48	1.00	1.43	0.04	0.98	0.03	1.97	0.54	0.48	0.51	0.44	1.25	2.23	1.54	0.11	0.37	0.35	0.35	1.61	0.94	0.00	0.00	0.96	0.03	2.70	1.70	0.08	0.45	0.36	0.50	1.04	0.95	0.12	1.05	1.54	0.01	0.64	2.12	0.12	0.50	0.52	0.18	0.59	0.94	0.43	1.62	1.57	0.11	0.18	1.43	0.96	1.08	0.73	0.05	0.96	0.40	1.00	0.56	0.50	0.43	1.52	1.06	1.55	0.35	0.49	0.46	0.34	0.09	0.98	0.56	0.09	0.39	2.57	1.54	1.08	0.04	0.39	0.85	0.03	0.02	1.53	0.53	0.17	0.41	0.90	1.09	1.13	0.09	0.92	0.39	0.21	0.46	1.51	0.09	1.06	0.87	0.02	0.15",
    "	1.01	0.00	1.03	0.06	0.01	0.95	0.05	0.91	0.99	1.09	0.02	0.00	2.04	0.00	1.01	1.02	1.02	0.00	0.02	1.00	1.03	0.00	1.03	0.02	0.00	0.99	0.04	0.95	0.00	0.00	0.98	0.06	0.00	1.03	0.07	0.04	0.98	0.00	0.04	0.99	0.03	0.00	0.96	0.07	0.99	0.00	0.08	0.99	0.04	0.00	2.00	0.08	1.00	0.00	2.97	1.04	1.02	1.03	0.12	2.03	0.15	0.97	0.07	0.96	3.15	0.08	0.05	2.03	0.10	1.07	2.04	0.93	1.04	0.02	0.06	1.01	0.98	0.00	0.99	0.06	0.99	0.02	0.98	0.11	0.09	0.97	0.09	1.04	2.05	0.08	1.98	0.09	0.09	1.97	0.08	0.08	1.01	0.00	0.05	1.01	0.95	0.04	0.13	4.90	0.16	1.04	2.05	0.08	2.01	0.07	1.93	0.07	0.99	0.00	0.96	0.09	1.00	0.00	0.97	0.14	0.04	0.98	0.05	1.04	0.08	2.02	4.01	0.14	1.00	1.02	0.11	0.01	1.01	0.15	1.96	0.02	0.13	1.01	0.12	0.09	1.01	0.11	0.96	1.00	0.05	1.98	0.16	2.01	2.98	0.10	0.06	1.99	1.03	0.10	0.01	1.02	0.14	1.05	0.14	0.99	0.14	0.15	2.14	0.96	2.08	1.02	2.05	0.10	1.03	0.11	0.97	0.04	0.04	0.94	1.99	0.10	0.08	2.06	1.01	0.15	0.97	0.13	0.16	1.00	0.18	0.12	1.95	0.18	1.01	1.98	0.14	0.07	0.95	0.15	0.17	1.93	0.14	1.89	1.03	0.97	0.03	0.06	1.04	0.16	0.10	0.95	0.06	0.05	1.02	0.13	3.84	0.10	0.11	0.96	0.20	0.11	1.00	0.12	1.08	1.00	0.14	0.93	0.06	0.14	0.98	1.93	2.05	1.04	0.03	2.92	0.20	0.16	1.88	0.09	1.00	0.08	2.84	1.86	0.15	0.04	0.93	1.93	0.63	0.09	0.36	1.23	0.42	0.64	0.14	1.75	1.99	0.17	0.11	0.99	0.09	0.14	0.94	0.09	0.14	0.99	0.10	0.08	0.98	2.05	0.99	0.06	1.09	0.18	1.07	0.08	0.11	0.98	0.01	0.95	0.12	0.16	2.08	1.05	0.09	0.08	1.00	0.92	2.13	0.11	0.17	1.08	0.20	0.03	0.98	0.18	1.03	2.98	1.14	0.96	0.12	0.05	2.01	2.06	0.21	0.05	0.97	0.07	0.95	0.10	2.89	0.14	0.98	0.18	2.97	0.07	1.00	1.00	0.19	2.98	0.12	1.86	0.12	0.14	1.04	0.09	0.96	0.97	0.04	0.18	1.07	0.12	3.93	1.11	0.07	0.93	0.15	1.00	1.99	0.98	0.10	0.01	0.99	0.09	0.99	2.08	0.13	0.13	1.02	0.07	0.01	1.03	0.11	1.91	2.10	0.17	0.14	1.97	0.20	0.95	0.14	0.07	0.98	1.04	1.06	1.03	1.01	0.15	0.14	1.03	1.88	1.05	1.01	0.10	1.03	0.13	0.86	0.21	0.09	0.95	3.00	0.99	1.05	0.02	0.21	0.98	0.09	0.05	0.98	0.98	0.09	0.02	1.94	1.90	1.09	0.05	0.91	0.99	0.09	0.07	2.01	0.19	0.15	0.92	0.23	0.16",
    "	0.96	0.00	1.00	0.03	0.01	1.04	0.03	0.98	0.96	1.13	0.03	0.01	2.05	0.00	1.00	0.96	1.01	0.00	0.01	0.96	1.03	0.02	0.96	0.04	0.00	1.03	0.05	1.03	0.00	0.00	0.97	0.05	0.00	1.05	0.04	0.03	0.98	0.00	0.04	0.97	0.06	0.00	1.00	0.07	0.97	0.00	0.11	0.99	0.08	0.00	1.92	0.11	0.99	0.00	3.01	1.01	1.03	1.01	0.13	1.94	0.17	1.02	0.04	0.98	3.02	0.07	0.10	1.94	0.17	1.04	2.00	0.98	1.05	0.01	0.13	0.98	0.96	0.00	1.00	0.09	0.96	0.02	0.99	0.12	0.15	1.00	0.18	1.01	2.04	0.13	1.97	0.10	0.17	2.02	0.16	0.12	0.98	0.11	0.04	1.02	0.99	0.03	0.15	4.78	0.16	1.02	2.05	0.12	1.93	0.07	1.93	0.10	1.00	0.00	0.97	0.08	1.01	0.06	1.00	0.13	0.09	0.99	0.12	1.01	0.12	2.03	4.00	0.16	1.00	1.03	0.10	0.04	1.01	0.14	1.95	0.09	0.16	0.98	0.14	0.10	0.99	0.13	1.01	1.01	0.06	2.01	0.16	2.01	3.01	0.08	0.08	1.97	0.99	0.08	0.06	0.96	0.19	1.00	0.18	1.00	0.18	0.14	2.13	1.07	2.09	1.07	2.05	0.13	0.99	0.13	0.96	0.09	0.13	0.93	2.02	0.10	0.08	1.97	1.00	0.16	1.00	0.00	0.20	1.02	0.16	0.13	2.00	0.16	0.99	1.97	0.10	0.18	0.95	0.11	0.16	1.95	0.17	1.91	1.05	0.94	0.05	0.02	1.03	0.18	0.10	0.98	0.05	0.04	1.04	0.09	3.87	0.11	0.13	0.96	0.13	0.13	0.98	0.09	1.06	1.02	0.13	0.94	0.07	0.12	1.00	1.87	2.00	1.03	0.05	2.91	0.19	0.13	1.97	0.04	1.03	0.08	2.83	1.96	0.13	0.06	0.96	2.04	0.15	0.02	0.98	0.05	0.96	0.08	0.08	3.07	2.08	0.09	0.18	0.99	0.08	0.13	1.01	0.10	0.18	0.95	0.15	0.09	1.03	1.97	1.05	0.08	0.96	0.15	1.05	0.15	0.09	1.02	0.03	1.05	0.14	0.15	2.07	0.94	0.11	0.08	1.01	1.00	2.08	0.12	0.16	1.01	0.13	0.01	1.09	0.17	1.01	3.03	1.06	0.96	0.09	0.04	2.05	2.12	0.14	0.09	0.94	0.03	0.96	0.08	3.00	0.16	0.99	0.19	2.94	0.00	0.99	1.03	0.18	3.08	0.10	1.78	0.16	0.06	1.07	0.11	0.97	1.01	0.02	0.19	1.05	0.16	4.06	1.19	0.11	0.89	0.07	0.90	2.05	0.99	0.10	0.00	0.97	0.04	0.97	2.15	0.16	0.13	1.07	0.00	0.06	1.07	0.09	2.03	2.09	0.15	0.17	1.93	0.17	0.92	0.14	0.14	1.02	0.98	1.11	1.09	0.98	0.14	0.14	0.97	1.97	1.05	1.04	0.13	0.95	0.07	0.92	0.09	0.23	0.92	3.13	1.07	1.08	0.16	0.09	0.91	0.09	0.03	1.02	0.91	0.22	0.00	1.96	1.97	1.16	0.01	0.86	0.94	0.03	0.09	2.01	0.16	0.17	0.96	0.21	0.15",
    "	1.00	0.00	1.00	0.02	0.00	1.00	0.02	0.98	0.95	1.15	0.00	0.02	2.02	0.00	0.95	0.97	1.04	0.00	0.02	0.98	1.01	0.00	0.99	0.03	0.00	1.05	0.01	0.98	0.01	0.00	0.97	0.05	0.01	1.06	0.06	0.03	0.96	0.00	0.02	1.00	0.04	0.00	0.96	0.06	0.99	0.00	0.06	0.99	0.06	0.00	1.96	0.08	0.98	0.00	3.00	1.03	1.03	1.04	0.15	2.02	0.15	1.01	0.04	1.00	3.01	0.04	0.07	1.97	0.13	1.02	1.97	0.99	1.02	0.01	0.12	1.02	0.96	0.00	1.02	0.08	1.00	0.00	0.97	0.10	0.11	0.96	0.15	0.98	2.03	0.08	1.97	0.09	0.12	2.05	0.15	0.09	1.00	0.09	0.05	1.01	1.02	0.04	0.12	4.80	0.11	1.00	1.99	0.16	1.96	0.04	1.93	0.10	0.96	0.00	0.95	0.08	1.04	0.01	1.06	0.11	0.10	1.02	0.11	0.99	0.11	1.98	3.87	0.15	1.01	1.02	0.13	0.06	0.98	0.12	1.99	0.08	0.14	1.00	0.15	0.12	1.00	0.12	1.01	0.96	0.07	2.00	0.15	2.06	3.03	0.03	0.06	2.08	1.09	0.05	0.08	0.98	0.14	0.99	0.17	0.96	0.16	0.14	2.03	1.06	2.03	1.00	2.03	0.15	1.03	0.08	0.99	0.11	0.11	0.97	2.01	0.11	0.11	2.02	1.05	0.13	0.96	0.04	0.11	1.00	0.14	0.13	2.05	0.18	0.95	1.99	0.13	0.17	0.96	0.15	0.14	1.95	0.16	1.89	1.04	1.02	0.04	0.05	0.99	0.11	0.10	1.01	0.04	0.06	1.03	0.08	3.80	0.15	0.16	0.97	0.11	0.10	0.98	0.11	1.03	1.03	0.11	0.91	0.09	0.11	1.01	1.94	1.99	1.04	0.15	2.85	0.18	0.13	1.95	0.04	1.06	0.08	2.85	2.01	0.11	0.06	0.94	1.98	0.11	0.07	0.98	0.04	0.97	0.06	0.15	3.08	2.02	0.07	0.10	1.01	0.07	0.10	0.98	0.07	0.17	0.96	0.16	0.09	1.04	1.91	1.04	0.10	0.97	0.11	1.05	0.14	0.18	0.99	0.07	1.03	0.12	0.08	2.07	1.01	0.11	0.08	0.97	1.02	2.05	0.11	0.18	0.97	0.18	0.04	1.03	0.15	1.07	2.98	1.09	0.98	0.08	0.03	2.08	2.02	0.11	0.10	0.92	0.05	0.98	0.10	2.91	0.16	0.96	0.16	2.87	0.03	0.97	1.02	0.13	2.84	0.18	1.85	0.16	0.11	1.04	0.06	1.07	1.00	0.02	0.09	1.03	0.13	3.95	1.08	0.07	0.89	0.08	0.99	2.08	0.94	0.17	0.00	1.01	0.10	1.00	2.09	0.18	0.14	1.08	0.03	0.07	1.03	0.06	1.97	2.14	0.10	0.14	1.90	0.06	0.90	0.18	0.16	1.07	0.94	1.11	1.06	0.98	0.14	0.15	0.96	2.01	1.04	1.06	0.12	1.00	0.12	0.90	0.08	0.10	0.96	2.97	1.04	1.07	0.02	0.12	0.94	0.07	0.04	1.05	1.00	0.12	0.01	2.02	2.00	1.06	0.03	0.83	0.97	0.08	0.10	2.01	0.15	0.23	0.95	0.22	0.15",
    "	0.97	0.00	1.00	0.03	0.02	1.04	0.02	0.96	0.97	1.15	0.09	0.04	1.93	0.05	0.92	1.04	0.98	0.01	0.05	0.93	0.97	0.02	1.01	0.08	0.05	1.01	0.06	0.96	0.00	0.03	1.00	0.05	0.03	1.10	0.03	0.03	0.98	0.00	0.02	1.01	0.04	0.00	0.98	0.05	1.01	0.00	0.05	1.01	0.06	0.00	1.95	0.08	0.99	0.00	2.95	0.98	0.99	1.06	0.12	2.05	0.15	1.03	0.03	0.99	3.05	0.05	0.07	2.03	0.14	1.05	2.04	0.97	1.01	0.01	0.11	0.99	1.00	0.00	0.98	0.54	0.53	0.00	0.98	0.11	0.10	1.02	0.11	1.03	2.05	0.07	2.04	0.10	0.14	2.11	0.13	0.10	1.00	0.04	0.04	1.04	0.98	0.03	0.12	4.84	0.16	1.00	2.04	0.08	2.01	0.05	1.95	0.09	0.96	0.02	1.01	0.08	1.02	0.02	0.96	0.13	0.11	0.98	0.05	1.04	0.09	2.08	3.86	0.16	1.04	1.00	0.10	0.08	0.95	0.12	1.96	0.06	0.17	1.06	0.13	0.12	1.00	0.08	0.99	1.02	0.08	2.01	0.16	1.88	2.81	0.07	0.13	1.87	1.05	0.08	0.05	0.95	0.16	0.95	0.12	0.97	0.17	0.14	1.83	1.07	1.91	1.02	1.97	0.12	1.03	0.08	1.10	0.06	0.11	0.92	1.92	0.07	0.08	2.01	1.02	0.12	0.95	0.09	0.17	0.96	0.15	0.12	1.94	0.13	0.98	1.93	0.14	0.12	1.00	0.08	0.10	1.93	0.13	1.92	0.97	1.08	0.05	0.08	1.02	0.15	0.09	0.91	0.08	0.13	0.96	0.14	3.68	0.10	0.15	0.95	0.08	0.13	1.00	0.12	1.23	1.02	0.16	0.97	0.04	0.10	0.94	1.81	1.98	1.00	0.09	2.81	0.12	0.09	1.86	0.00	1.02	0.09	2.87	2.02	0.07	0.10	0.90	2.13	0.13	0.08	1.02	0.07	0.99	0.06	0.15	3.23	1.92	0.04	0.12	0.91	0.18	0.11	1.05	0.18	0.12	0.95	0.14	0.14	0.99	1.98	1.02	0.10	0.98	0.15	1.10	0.14	0.23	1.03	0.02	1.05	0.11	0.12	2.05	0.96	0.12	0.10	0.97	0.93	2.15	0.23	0.21	1.08	0.14	0.02	0.99	0.15	1.05	3.00	1.15	1.01	0.08	0.03	2.02	1.97	0.13	0.15	0.95	0.08	0.90	0.18	2.77	0.15	1.00	0.16	2.93	0.07	0.86	1.00	0.19	2.93	0.17	1.89	0.13	0.02	1.08	0.04	1.15	1.05	0.00	0.10	0.99	0.13	4.00	1.15	0.14	0.86	0.13	0.86	2.08	0.99	0.25	0.04	0.99	0.02	1.07	2.08	0.17	0.14	1.06	0.02	0.00	1.03	0.09	2.02	2.06	0.17	0.14	2.05	0.12	0.93	0.14	0.09	1.05	0.95	1.12	1.12	0.96	0.17	0.12	1.00	1.93	1.02	0.96	0.07	0.97	0.12	0.90	0.10	0.20	0.97	2.99	1.10	1.15	0.02	0.16	0.90	0.17	0.05	1.00	0.95	0.10	0.03	2.07	1.88	1.06	0.03	0.85	0.99	0.05	0.06	2.06	0.17	0.17	0.91	0.25	0.20",
    "	0.98	0.00	0.96	0.03	0.00	1.05	0.04	0.91	1.00	1.11	0.09	0.02	1.94	0.06	1.00	1.01	0.97	0.06	0.10	0.99	0.98	0.00	0.96	0.08	0.05	1.03	0.08	0.98	0.09	0.04	1.01	0.05	0.02	1.04	0.07	0.03	0.98	0.01	0.05	1.00	0.05	0.00	0.98	0.07	0.97	0.00	0.10	1.00	0.05	0.00	1.98	0.10	1.02	0.04	2.87	1.04	1.00	1.06	0.15	1.99	0.16	0.97	0.06	1.04	2.98	0.08	0.13	1.99	0.16	0.97	1.89	0.97	1.01	0.03	0.13	1.01	0.99	0.00	1.01	0.09	1.00	0.03	0.99	0.09	0.14	0.96	0.17	1.01	1.97	0.13	1.93	0.09	0.16	1.95	0.18	0.09	1.00	0.13	0.06	1.14	0.98	0.05	0.16	4.80	0.17	0.97	1.98	0.08	1.96	0.08	1.91	0.10	0.95	0.00	0.97	0.08	0.97	0.07	1.01	0.13	0.09	1.03	0.14	1.04	0.12	1.97	3.89	0.14	1.02	0.95	0.19	0.05	0.99	0.10	1.94	0.08	0.15	0.97	0.14	0.10	1.04	0.13	1.03	1.03	0.08	1.89	0.19	2.07	3.10	0.00	0.07	2.06	0.95	0.07	0.07	1.01	0.13	1.04	0.15	0.99	0.13	0.16	2.27	1.06	2.12	1.02	1.89	0.11	0.92	0.10	0.96	0.09	0.12	1.01	2.01	0.12	0.11	1.92	0.99	0.14	1.03	0.01	0.19	1.05	0.08	0.11	2.09	0.08	1.00	2.04	0.07	0.14	0.94	0.09	0.15	2.05	0.14	1.94	1.07	0.90	0.06	0.01	1.03	0.15	0.10	0.98	0.01	0.06	1.08	0.04	3.86	0.13	0.16	0.96	0.04	0.11	0.98	0.11	0.98	1.00	0.09	0.94	0.16	0.15	1.02	1.89	1.98	1.09	0.16	2.69	0.10	0.14	1.92	0.11	1.03	0.06	2.86	1.99	0.12	0.06	0.85	1.86	0.61	0.09	1.04	2.38	1.06	0.45	0.06	0.96	1.01	0.13	0.61	0.43	0.03	0.63	0.54	0.12	0.62	1.54	2.15	0.67	0.54	0.88	1.57	0.63	0.54	0.17	1.00	0.55	1.05	0.61	0.18	1.04	0.09	0.12	1.55	0.47	0.63	1.53	0.94	1.00	0.97	0.00	1.01	1.38	0.18	0.10	1.03	0.10	0.98	1.32	1.90	0.55	0.43	0.04	2.13	0.82	1.33	0.54	1.16	1.19	0.37	0.92	1.22	0.06	0.92	0.43	0.49	1.63	0.18	0.86	0.47	0.07	2.22	0.57	0.62	0.99	0.10	0.52	1.50	0.57	1.81	0.60	0.44	0.42	0.58	1.55	0.99	0.51	0.57	0.06	0.60	0.45	0.62	2.02	1.10	0.17	0.45	1.04	0.16	0.98	0.16	0.90	1.50	0.59	0.47	1.54	0.68	0.38	0.16	0.49	1.56	0.99	1.10	0.54	0.96	0.09	0.65	0.53	1.02	0.94	1.97	0.57	0.98	0.07	0.47	0.60	0.08	0.37	2.05	1.08	0.53	0.14	1.02	1.44	0.63	0.03	1.02	0.92	0.04	0.05	1.92	0.99	0.55	0.55	0.53	0.46	0.51	0.15	1.41	0.10	0.11	1.00	0.61	0.14	1.56	0.42	0.59	0.16",
    "	1.00	0.00	0.99	0.02	0.01	1.00	0.04	0.98	0.97	1.13	0.00	0.02	2.05	0.00	1.00	0.97	1.04	0.00	0.00	1.01	1.00	0.00	0.98	0.07	0.01	0.99	0.03	1.01	0.00	0.00	0.97	0.04	0.00	1.04	0.05	0.04	1.00	0.00	0.03	0.97	0.06	0.00	0.88	0.03	0.97	0.00	0.12	0.96	0.06	0.00	2.12	0.06	1.01	0.01	2.97	1.08	0.99	1.10	0.14	1.93	0.16	1.00	0.00	1.00	3.02	0.06	0.09	2.00	0.12	1.00	1.97	0.98	1.02	0.01	0.09	1.03	1.02	0.00	1.04	0.04	0.96	0.02	0.91	0.09	0.14	0.96	0.15	0.97	2.06	0.11	2.08	0.04	0.10	2.10	0.12	0.03	1.05	0.09	0.01	1.09	0.94	0.04	0.10	4.81	0.11	0.97	1.97	0.10	2.03	0.06	1.99	0.09	0.94	0.00	1.00	0.06	1.07	0.02	1.04	0.07	0.03	1.01	0.14	0.96	0.10	2.02	4.00	0.15	1.01	0.98	0.14	0.00	1.04	0.12	1.98	0.05	0.16	0.99	0.16	0.11	0.99	0.15	1.04	0.99	0.09	1.99	0.14	1.98	0.08	0.14	0.95	0.17	1.94	0.05	0.12	1.96	1.02	0.06	0.09	0.97	0.12	0.97	0.16	1.02	0.10	0.10	2.04	1.06	2.08	1.04	2.04	0.17	0.98	0.11	0.98	0.09	0.12	0.99	2.04	0.08	0.12	1.97	1.02	0.13	0.96	0.14	0.14	0.98	0.11	0.12	2.03	0.15	0.99	1.98	0.10	0.15	0.97	0.05	0.11	1.92	0.11	1.94	1.03	0.99	0.09	0.07	1.06	0.11	0.10	0.98	0.13	0.06	1.00	0.09	3.83	0.12	0.12	0.95	0.15	0.13	0.97	0.15	1.05	1.03	0.13	0.97	0.13	0.13	0.99	2.07	2.05	1.00	0.09	2.83	0.16	0.11	2.00	0.15	1.04	0.04	2.99	2.02	0.15	0.04	0.98	2.08	1.07	0.08	0.09	2.04	0.06	0.94	0.08	1.01	1.97	0.13	0.13	0.96	0.11	0.12	1.04	0.12	0.18	0.99	0.15	0.07	1.05	2.02	1.04	0.09	1.00	0.15	0.93	0.10	0.18	1.01	0.02	1.06	0.11	0.13	2.05	0.99	0.09	0.07	1.00	0.95	2.03	0.10	0.15	1.03	0.14	0.03	1.08	0.07	1.03	2.86	1.02	1.01	0.12	0.03	2.01	1.95	0.12	0.10	1.02	0.04	1.00	0.09	2.88	0.16	0.94	0.13	2.85	0.08	0.95	0.99	0.20	2.80	0.14	1.88	0.18	0.11	1.01	0.06	0.99	1.02	0.09	0.16	1.02	0.16	3.81	1.06	0.11	0.92	0.08	1.02	1.95	0.95	0.12	0.07	1.01	0.03	1.05	1.98	0.14	0.12	1.04	0.00	0.04	1.03	0.09	1.97	2.09	0.12	0.12	1.87	0.18	0.91	0.18	0.08	1.04	0.98	1.04	1.04	1.05	0.07	0.14	0.97	1.90	1.02	1.02	0.10	1.00	0.08	0.96	0.09	0.11	0.92	2.96	1.06	1.04	0.00	0.14	0.92	0.10	0.00	0.98	1.00	0.19	0.01	1.87	1.88	1.02	0.04	0.92	0.94	0.05	0.09	2.02	0.13",
    "	1.03	0.00	0.99	0.00	0.00	0.98	0.05	0.96	0.96	1.08	0.01	0.00	2.04	0.01	1.04	1.01	0.98	0.00	0.00	0.96	1.04	0.00	0.99	0.01	0.00	1.07	0.05	1.00	0.00	0.00	0.94	0.05	0.00	1.00	0.10	0.00	0.96	0.00	0.07	1.02	0.05	0.00	1.00	0.07	0.97	0.00	0.15	1.02	0.07	0.00	1.86	0.10	0.95	0.00	2.88	1.00	1.05	1.01	0.19	2.02	0.12	0.95	0.09	1.07	3.10	0.07	0.08	2.10	0.11	1.02	2.00	0.90	1.06	0.00	0.10	1.09	0.91	0.00	1.06	0.06	0.98	0.00	0.94	0.08	0.09	0.96	0.18	1.04	2.15	0.00	2.06	0.02	0.08	2.08	0.09	0.07	1.03	0.00	0.16	0.96	0.98	0.06	0.15	5.25	0.10	1.01	1.98	0.00	2.05	0.04	1.88	0.06	0.93	0.00	0.97	0.07	1.09	0.00	1.01	0.05	0.03	0.98	0.03	1.05	0.05	1.97	3.96	0.13	1.01	0.94	0.18	0.00	0.98	0.09	1.99	0.01	0.08	1.03	0.16	0.07	0.98	0.09	1.04	0.99	0.00	2.17	0.16	2.09	3.10	0.02	0.12	2.03	1.01	0.02	0.09	0.93	0.12	1.00	0.12	1.05	0.11	0.06	2.07	1.08	2.27	1.04	2.04	0.08	1.01	0.05	0.93	0.05	0.07	0.94	2.07	0.01	0.03	2.07	1.01	0.06	1.06	0.10	0.10	0.99	0.12	0.06	2.04	0.07	1.00	1.99	0.11	0.11	0.97	0.08	0.12	2.06	0.10	1.40	0.59	0.96	0.00	0.00	1.01	0.49	0.06	0.49	0.53	0.13	0.54	0.41	2.02	0.11	0.36	0.47	1.93	0.10	0.44	0.41	0.54	0.61	0.45	0.41	0.55	0.48	0.44	1.44	1.07	0.46	0.43	2.37	0.91	0.50	0.98	1.19	0.60	0.06	2.37	0.91	0.51	0.00	1.87	2.08	0.55	0.08	0.36	1.82	0.54	0.60	0.09	1.33	0.96	0.42	0.11	0.90	0.96	0.12	0.48	0.37	0.11	0.48	0.55	0.05	0.58	1.52	0.66	0.00	1.13	0.92	1.05	0.01	0.45	0.51	0.52	0.52	0.14	0.43	1.05	0.93	0.15	0.06	1.59	0.98	1.12	0.03	0.39	0.96	0.93	0.05	0.54	0.45	0.57	1.80	1.05	0.57	0.60	1.51	1.70	1.61	0.22	0.00	1.54	0.92	0.68	0.07	2.16	0.03	0.99	0.16	3.05	0.06	1.05	0.52	1.61	1.56	0.47	1.49	0.10	1.41	0.55	1.00	0.61	0.35	0.58	0.05	1.01	0.53	2.06	0.61	0.59	0.43	1.68	1.00	1.00	0.94	0.12	0.43	1.50	0.36	0.62	0.98	0.44	0.02	1.09	0.99	0.11	0.45	0.50	1.01	1.03	0.55	0.20	1.94	0.98	0.36	0.13	0.90	0.56	0.90	0.68	0.56	1.05	0.41	0.61	1.03	1.57	0.46	0.54	0.39	1.52	0.38	1.00	0.11	0.30	0.41	1.78	0.60	0.50	0.34	1.19	0.94	0.48	0.04	0.43	0.91	0.12	0.00	1.26	1.18	0.50	0.00	1.10	1.11	0.35	0.00	1.13	0.32	0.05	0.40	0.78	0.00",
    "	1.00	0.00	1.01	0.04	0.01	0.99	0.03	0.98	0.99	1.09	0.05	0.02	1.91	0.01	0.98	1.00	0.97	0.00	0.04	0.99	1.04	0.00	0.93	0.05	0.04	1.02	0.05	0.99	0.01	0.00	0.99	0.04	0.01	1.07	0.06	0.04	0.99	0.00	0.03	0.97	0.05	0.00	0.98	0.07	0.96	0.00	0.08	0.98	0.06	0.00	1.98	0.10	1.00	0.01	2.95	1.05	0.99	1.07	0.13	1.99	0.15	1.00	0.04	1.01	2.98	0.07	0.08	1.97	0.17	1.05	1.98	0.99	1.01	0.02	0.10	0.99	1.00	0.00	0.99	0.07	0.95	0.00	0.97	0.09	0.13	1.01	0.15	1.00	2.04	0.10	2.03	0.10	0.14	2.06	0.13	0.08	1.00	0.10	0.03	1.01	1.00	0.02	0.12	4.91	0.17	1.01	2.02	0.11	1.99	0.06	2.01	0.07	1.01	0.00	0.96	0.05	1.01	0.02	1.01	0.10	0.09	0.96	0.05	1.04	0.09	2.00	3.95	0.18	1.00	1.03	0.15	0.04	1.00	0.11	2.04	0.06	0.09	1.02	0.10	0.07	0.98	0.09	1.04	0.97	0.03	2.00	0.15	1.91	2.98	0.02	0.09	1.94	1.05	0.06	0.05	0.96	0.14	0.98	0.17	1.02	0.09	0.11	2.11	1.05	2.05	1.06	2.02	0.15	0.95	0.16	0.98	0.07	0.13	0.98	1.99	0.09	0.09	2.07	1.08	0.11	0.96	0.03	0.18	0.97	0.13	0.13	2.03	0.18	1.01	2.05	0.08	0.18	0.93	0.11	0.12	2.03	0.15	2.03	1.00	0.97	0.04	0.00	0.98	0.12	0.07	1.00	0.05	0.05	1.07	0.16	3.81	0.15	0.16	1.00	0.12	0.09	1.02	0.09	1.05	1.01	0.18	0.90	0.06	0.17	0.99	1.88	1.99	1.02	0.16	2.91	0.17	0.13	1.86	0.18	1.02	0.06	2.83	2.09	0.14	0.04	1.01	2.09	0.61	0.05	0.52	0.95	0.49	0.51	0.16	2.02	2.08	0.13	0.15	1.04	0.08	0.09	1.04	0.10	0.16	0.91	0.17	0.08	1.04	1.90	0.99	0.05	1.00	0.15	1.04	0.11	0.20	1.04	0.04	0.98	0.11	0.12	2.05	1.03	0.12	0.07	1.06	0.99	2.02	0.10	0.11	0.92	0.16	0.00	1.10	0.13	1.05	3.05	0.99	1.00	0.21	0.02	2.01	2.10	0.19	0.13	0.88	0.06	1.03	0.11	2.84	0.16	0.97	0.13	3.01	0.07	1.02	1.02	0.18	2.95	0.14	1.86	0.12	0.11	0.98	0.08	1.06	0.98	0.04	0.17	1.03	0.13	4.02	1.11	0.07	0.90	0.07	0.96	2.01	0.98	0.13	0.06	1.01	0.05	1.10	2.01	0.14	0.14	0.98	0.06	0.02	1.03	0.11	1.90	2.10	0.19	0.14	1.95	0.17	0.92	0.14	0.15	1.09	0.94	1.12	1.03	1.01	0.13	0.18	1.01	1.93	1.02	1.10	0.12	0.90	0.08	0.90	0.15	0.11	0.98	3.03	1.01	1.10	0.15	0.17	0.92	0.05	0.03	1.02	0.95	0.11	0.00	2.06	2.02	1.10	0.04	0.82	0.98	0.08	0.07	2.02	0.17	0.19	0.97	0.15	0.15",
    "	1.00	0.00	1.01	0.04	0.03	1.00	0.04	0.96	0.96	1.13	0.01	0.03	2.01	0.00	1.02	0.99	0.99	0.00	0.02	0.97	1.05	0.00	0.99	0.05	0.02	1.05	0.03	1.00	0.03	0.00	0.95	0.08	0.04	1.05	0.07	0.03	0.96	0.00	0.04	0.98	0.08	0.00	0.97	0.08	0.98	0.00	0.11	0.98	0.11	0.00	1.96	0.13	1.00	0.03	2.94	1.01	1.03	1.03	0.17	1.99	0.17	0.99	0.06	1.03	2.91	0.09	0.10	1.92	0.14	1.03	1.98	0.97	1.01	0.02	0.11	0.98	1.02	0.00	0.99	0.09	0.96	0.04	0.98	0.11	0.16	0.98	0.19	1.00	2.03	0.11	1.97	0.11	0.15	1.97	0.15	0.10	0.95	0.09	0.06	1.03	0.99	0.05	0.17	4.76	0.11	1.01	1.96	0.09	1.95	0.06	1.94	0.07	1.01	0.00	0.94	0.07	1.01	0.03	1.01	0.15	0.07	1.00	0.11	1.01	0.10	1.98	3.98	0.16	0.98	1.01	0.11	0.00	1.01	0.12	1.99	0.06	0.14	1.00	0.12	0.08	1.01	0.02	0.98	1.01	0.09	1.98	0.14	1.98	2.91	0.07	0.06	2.00	1.02	0.09	0.05	0.99	0.15	0.98	0.18	0.99	0.15	0.12	2.10	1.04	1.97	1.05	2.10	0.09	1.01	0.14	0.99	0.07	0.13	0.98	1.96	0.07	0.09	2.04	0.97	0.12	0.98	0.10	0.06	0.98	0.12	0.13	2.02	0.15	1.03	1.96	0.08	0.06	0.93	0.00	0.15	1.96	0.14	1.88	1.01	0.98	0.03	0.06	1.07	0.11	0.10	0.95	0.08	0.05	1.03	0.13	3.91	0.11	0.15	0.98	0.10	0.10	0.98	0.08	1.06	1.00	0.14	0.91	0.03	0.15	1.05	1.96	1.99	0.99	0.01	2.92	0.17	0.13	1.92	0.04	1.02	0.06	2.93	2.02	0.12	0.03	1.00	2.06	1.02	0.12	0.09	1.99	0.10	0.97	0.07	1.05	2.05	0.15	0.14	1.01	0.11	0.15	1.00	0.07	0.09	0.97	0.17	0.05	0.96	2.01	1.00	0.08	1.03	0.15	1.06	0.14	0.19	1.02	0.04	1.00	0.13	0.14	2.04	0.97	0.07	0.08	1.01	1.00	2.07	0.12	0.15	1.04	0.19	0.00	1.08	0.03	1.05	2.99	1.02	0.95	0.11	0.04	1.98	2.04	0.09	0.11	0.91	0.04	1.04	0.09	2.96	0.11	0.94	0.13	3.09	0.03	1.03	1.00	0.15	3.03	0.12	1.80	0.16	0.09	1.03	0.08	0.95	0.98	0.00	0.18	1.08	0.15	3.96	1.17	0.08	0.90	0.11	0.97	1.95	0.98	0.10	0.01	0.96	0.02	1.03	2.04	0.10	0.10	1.06	0.01	0.07	1.02	0.08	1.95	2.06	0.16	0.08	2.00	0.20	0.94	0.15	0.17	1.02	0.98	1.10	0.99	0.98	0.13	0.17	1.04	1.94	0.98	1.08	0.10	1.00	0.10	0.94	0.10	0.10	0.93	2.96	1.05	1.05	0.13	0.17	0.95	0.14	0.02	1.05	0.89	0.06	0.01	1.93	1.97	1.06	0.00	0.89	1.00	0.10	0.08	2.00	0.14	0.17	0.92	0.24	0.13",
    "	0.97	0.00	1.00	0.00	0.01	1.03	0.00	1.01	1.03	1.08	0.02	0.07	1.85	0.02	0.96	1.04	0.98	0.03	0.07	0.94	0.98	0.02	0.94	0.08	0.06	1.04	0.11	0.95	0.05	0.05	0.95	0.03	0.10	1.12	0.03	0.07	0.99	0.00	0.05	0.97	0.12	0.00	0.94	0.14	0.97	0.00	0.18	0.95	0.13	0.00	2.06	0.16	1.00	0.06	2.92	1.12	0.98	1.09	0.15	2.03	0.13	1.01	0.02	1.04	2.87	0.08	0.11	2.00	0.16	0.97	1.90	1.08	0.99	0.00	0.17	0.97	1.01	0.00	0.95	0.09	0.99	0.02	0.97	0.09	0.18	1.02	0.11	0.99	1.95	0.07	1.96	0.10	0.14	2.00	0.16	0.11	0.99	0.09	0.02	1.06	1.00	0.03	0.13	4.87	0.15	0.94	1.93	0.21	1.89	0.07	1.98	0.09	1.00	0.00	1.01	0.05	1.03	0.05	1.08	0.10	0.14	0.95	0.09	1.00	0.15	2.02	3.83	0.17	0.93	1.02	0.14	0.07	1.02	0.11	2.01	0.08	0.20	1.03	0.11	0.08	1.00	0.04	0.98	0.97	0.03	2.04	0.14	1.91	2.87	0.09	0.07	1.95	1.04	0.08	0.08	0.99	0.16	0.91	0.03	1.02	0.12	0.12	1.94	1.16	1.92	0.97	1.94	0.18	0.97	0.15	1.09	0.11	0.17	0.92	1.99	0.09	0.11	2.05	1.08	0.11	0.97	0.14	0.17	0.95	0.17	0.01	1.93	0.11	0.97	2.05	0.13	0.04	0.94	0.02	0.09	1.87	0.16	1.96	0.99	0.98	0.06	0.07	1.09	0.13	0.09	0.97	0.05	0.12	0.97	0.12	3.75	0.15	0.17	0.98	0.07	0.12	1.00	0.06	1.12	0.99	0.18	0.94	0.06	0.14	1.01	1.95	2.00	0.98	0.08	2.88	0.10	0.14	1.85	0.08	1.07	0.10	2.71	1.95	0.17	0.05	0.93	1.99	0.09	0.04	1.04	0.00	0.92	0.04	0.13	3.01	1.99	0.08	0.18	1.09	0.17	0.13	1.05	0.14	0.18	0.88	0.18	0.12	0.95	1.96	0.98	0.13	1.02	0.19	1.07	0.15	0.09	1.08	0.00	0.99	0.16	0.11	2.02	0.94	0.01	0.08	1.02	1.02	2.04	0.15	0.06	1.04	0.20	0.00	1.06	0.18	1.02	3.02	1.12	0.96	0.13	0.04	2.03	2.12	0.18	0.15	0.91	0.06	0.98	0.14	2.81	0.17	0.91	0.08	3.11	0.00	1.03	0.95	0.22	2.98	0.11	1.84	0.12	0.11	0.99	0.07	1.02	1.05	0.01	0.20	1.09	0.13	3.96	1.10	0.11	0.95	0.04	0.88	1.95	1.04	0.10	0.00	1.02	0.03	0.96	2.05	0.22	0.15	1.05	0.00	0.06	1.10	0.16	1.97	2.08	0.15	0.15	1.91	0.04	0.88	0.18	0.04	1.06	0.98	1.11	1.07	0.89	0.13	0.17	1.11	1.80	1.07	1.01	0.12	0.97	0.14	0.80	0.11	0.04	1.00	3.05	1.07	1.10	0.02	0.00	0.90	0.16	0.07	1.01	0.92	0.09	0.01	1.90	1.85	1.15	0.00	0.84	0.96	0.13	0.06	2.09	0.16	0.25	0.91	0.34	0.17",
    "	0.96	0.00	0.99	0.01	0.00	1.05	0.00	1.12	0.96	1.11	0.10	0.06	2.01	0.01	1.03	0.97	0.95	0.03	0.08	0.95	0.98	0.05	0.94	0.05	0.07	1.03	0.11	0.99	0.00	0.10	0.96	0.01	0.10	1.04	0.01	0.12	0.99	0.00	0.12	0.96	0.13	0.00	0.95	0.14	0.95	0.00	0.18	0.95	0.13	0.00	2.23	0.16	1.08	0.09	2.79	1.17	0.95	1.07	0.07	1.96	0.27	0.89	0.03	1.07	2.91	0.13	0.16	1.96	0.10	0.97	1.90	1.03	1.08	0.00	0.15	0.89	1.05	0.00	1.06	0.04	0.99	0.09	0.96	0.14	0.15	0.94	0.15	1.00	2.03	0.18	1.94	0.07	0.09	1.93	0.15	0.17	1.07	0.08	0.01	1.49	0.99	0.08	0.19	4.84	0.21	0.98	2.21	0.00	2.13	0.04	2.01	0.07	1.04	0.00	1.01	0.05	0.97	0.11	0.95	0.17	0.09	1.12	0.39	0.97	0.14	2.04	4.11	0.13	1.03	0.94	0.14	0.05	0.99	0.14	1.98	0.08	0.19	0.97	0.14	0.14	1.02	0.16	1.07	1.07	0.18	1.85	0.12	2.10	3.07	0.02	0.05	2.07	0.93	0.08	0.16	0.97	0.11	1.04	0.33	1.04	0.10	0.13	2.19	0.98	2.08	1.16	2.06	0.17	0.91	0.13	0.94	0.12	0.10	1.00	2.05	0.06	0.18	1.99	0.91	0.13	1.07	0.00	0.10	1.15	0.09	0.14	2.07	0.19	1.01	1.95	0.08	0.08	0.93	0.05	0.20	1.98	0.17	1.96	1.34	0.91	0.03	0.05	0.98	0.14	0.14	0.94	0.28	0.06	1.14	0.12	3.83	0.10	0.04	1.00	0.08	0.13	0.98	0.10	1.05	0.95	0.06	1.03	0.07	0.09	1.01	1.82	1.93	1.00	0.19	2.68	0.14	0.11	2.25	0.01	1.01	0.09	2.94	2.04	0.16	0.06	0.84	1.72	0.15	0.03	2.02	2.65	2.18	0.06	0.04	0.94	0.00	0.07	1.15	0.01	0.13	0.99	0.00	0.04	1.13	2.01	3.90	1.08	0.07	0.00	1.75	0.90	0.07	0.08	0.92	1.06	1.95	0.06	0.17	0.96	0.14	0.19	1.02	0.06	1.11	2.79	1.01	0.93	0.14	0.03	2.00	1.87	0.20	0.17	0.96	0.18	1.04	0.09	2.93	0.15	1.01	0.13	2.97	0.00	0.95	0.98	0.13	2.76	0.16	1.83	0.22	0.07	1.00	0.07	1.02	1.09	0.00	0.19	1.05	0.12	3.82	1.07	0.04	0.89	0.10	1.01	2.05	0.93	0.16	0.00	0.99	0.19	0.98	2.20	0.14	0.17	1.00	0.12	0.02	1.05	0.04	2.09	2.22	0.10	0.13	1.91	0.07	0.92	0.14	0.18	1.13	0.91	1.14	1.08	0.91	0.10	0.17	0.96	2.05	1.07	1.03	0.06	0.92	0.04	0.88	0.06	0.16	0.98	3.07	1.08	1.11	0.13	0.03	0.89	0.05	0.07	1.09	0.97	0.09	0.01	2.06	1.90	1.10	0.01	0.85	0.89	0.00	0.03	1.93	0.13	0.17	0.99	0.09	0.15	1.12	0.00	1.12	0.11	0.11	1.07	1.05	0.09	2.06	0.88	0.07	0.08",
    "	0.95	0.00	0.99	0.03	0.00	1.07	0.00	1.05	1.01	1.06	0.04	0.00	1.71	0.04	1.18	0.90	0.99	0.01	0.00	0.97	1.03	0.00	0.95	0.11	0.03	1.09	0.09	0.94	0.00	0.08	0.87	0.05	0.06	1.06	0.00	0.12	0.97	0.00	0.05	0.97	0.11	0.00	0.95	0.14	0.96	0.00	0.15	0.93	0.15	0.00	2.10	0.17	1.09	0.09	2.85	1.12	0.96	1.16	0.07	1.94	0.10	0.94	0.04	0.96	2.94	0.05	0.05	2.10	0.15	1.00	1.97	1.02	1.05	0.00	0.05	1.05	1.01	0.00	0.91	0.08	1.00	0.00	0.94	0.11	0.14	1.00	0.10	1.01	2.04	0.11	2.08	0.10	0.11	2.10	0.09	0.11	1.03	0.07	0.00	1.05	1.02	0.02	0.08	4.79	0.08	0.93	1.98	0.21	2.05	0.05	1.89	0.09	0.91	0.04	1.11	0.07	1.05	0.05	1.05	0.13	0.15	0.95	0.11	1.00	0.10	2.09	3.80	0.23	0.94	0.99	0.17	0.15	0.98	0.10	1.98	0.10	0.12	1.05	0.12	0.12	1.02	0.03	0.95	1.05	0.01	2.02	0.12	1.84	3.00	0.05	0.06	1.94	1.06	0.06	0.05	0.99	0.09	0.95	0.12	0.91	0.17	0.11	1.91	1.19	2.01	1.04	1.87	0.24	0.95	0.06	1.06	0.13	0.14	0.92	1.98	0.09	0.12	1.98	1.00	0.18	0.96	0.01	0.13	0.98	0.16	0.00	1.99	0.14	0.96	1.99	0.18	0.16	0.97	0.11	0.12	1.92	0.09	1.96	0.98	0.99	0.02	0.06	0.97	0.11	0.09	1.06	0.00	0.05	1.02	0.07	3.81	0.14	0.10	1.01	0.10	0.17	0.95	0.04	1.02	1.14	0.14	0.91	0.08	0.10	0.97	1.95	1.93	1.00	0.00	2.89	0.11	0.16	2.03	0.00	1.04	0.06	3.03	2.05	0.15	0.06	0.92	2.13	0.61	0.19	0.66	1.03	0.44	0.63	0.26	2.12	2.15	0.16	0.14	1.05	0.09	0.09	0.99	0.09	0.07	0.92	0.15	0.13	0.96	2.08	0.98	0.10	1.10	0.16	1.08	0.07	0.03	1.03	0.01	0.95	0.07	0.11	2.08	1.00	0.09	0.14	1.05	0.99	2.26	0.14	0.19	1.01	0.08	0.03	1.01	0.16	1.10	3.09	1.05	0.92	0.06	0.03	2.18	2.22	0.17	0.09	0.95	0.05	0.95	0.14	3.00	0.08	0.98	0.15	3.22	0.05	0.98	0.91	0.24	3.09	0.05	1.99	0.09	0.11	1.00	0.08	1.02	1.04	0.00	0.19	1.12	0.13	4.09	1.10	0.00	0.91	0.09	0.93	2.24	1.01	0.08	0.00	1.02	0.05	0.89	2.15	0.15	0.09	1.07	0.17	0.00	1.05	0.05	1.01	1.63	0.10	0.11	1.36	0.97	0.32	0.30	0.95	0.57	1.11	0.45	0.54	0.92	0.38	0.35	1.16	1.59	0.71	0.52	0.54	1.48	0.48	0.96	0.15	0.53	0.51	1.94	0.66	0.80	0.49	1.94	1.05	0.55	0.13	0.49	1.09	0.11	0.03	1.71	1.66	0.87	0.03	1.50	1.64	0.60	0.04	1.63	0.71	0.19	0.57	1.46	0.21",
    "	1.00	0.00	1.03	0.04	0.02	0.97	0.04	1.00	1.00	1.09	0.00	0.00	1.98	0.00	1.01	0.98	1.03	0.00	0.02	1.00	1.00	0.00	0.96	0.05	0.00	1.00	0.04	1.00	0.02	0.00	0.98	0.05	0.03	1.03	0.05	0.03	0.98	0.00	0.03	0.96	0.05	0.00	0.97	0.06	1.01	0.00	0.09	0.97	0.09	0.00	1.96	0.11	1.00	0.00	2.96	1.02	1.01	1.04	0.12	1.97	0.18	0.98	0.03	1.02	3.06	0.09	0.10	2.01	0.16	1.05	1.98	0.99	1.01	0.00	0.09	0.98	1.00	0.00	1.00	0.06	0.98	0.02	0.95	0.11	0.13	0.98	0.16	1.00	2.01	0.09	2.01	0.08	0.13	2.03	0.12	0.10	1.00	0.06	0.04	1.03	0.97	0.05	0.14	4.82	0.11	1.00	2.01	0.11	1.96	0.07	1.99	0.08	0.95	0.00	1.00	0.08	1.01	0.05	1.04	0.13	0.08	0.99	0.09	1.01	0.10	2.01	4.00	0.16	0.98	1.03	0.20	0.04	1.01	0.11	2.05	0.09	0.15	0.98	0.14	0.07	1.00	0.13	0.98	1.02	0.08	2.02	0.14	2.02	2.94	0.08	0.07	1.97	1.02	0.08	0.07	1.00	0.16	1.00	0.19	0.97	0.08	0.13	2.09	1.06	2.01	1.03	1.98	0.16	0.97	0.10	1.01	0.07	0.12	1.00	2.00	0.07	0.14	2.01	1.06	0.14	0.91	0.02	0.07	0.99	0.15	0.03	1.96	0.17	0.98	1.91	0.07	0.06	1.00	0.15	0.15	1.98	0.16	1.97	1.03	0.96	0.06	0.05	1.02	0.19	0.11	0.95	0.11	0.08	1.00	0.10	3.89	0.14	0.13	0.95	0.11	0.12	1.01	0.15	1.05	1.02	0.14	1.00	0.12	0.18	0.99	1.93	1.98	1.03	0.08	2.80	0.14	0.13	1.89	0.09	0.96	0.08	2.83	2.07	0.10	0.08	1.02	1.86	1.08	0.13	0.08	1.98	0.05	0.97	0.10	0.96	2.02	0.09	0.10	1.06	0.10	0.15	1.06	0.10	0.21	0.96	0.16	0.11	0.97	2.03	0.98	0.10	0.95	0.17	1.04	0.15	0.17	1.05	0.04	1.00	0.16	0.17	2.08	1.02	0.07	0.06	1.00	1.01	2.07	0.12	0.21	1.01	0.19	0.00	1.06	0.04	1.08	3.03	1.04	0.98	0.20	0.04	2.06	2.06	0.13	0.10	0.91	0.06	0.96	0.06	2.91	0.16	0.96	0.12	2.98	0.02	1.03	1.00	0.12	2.96	0.11	1.84	0.15	0.12	1.01	0.08	1.07	1.00	0.00	0.17	1.02	0.13	3.94	1.12	0.05	0.89	0.07	0.96	1.99	1.00	0.13	0.00	0.99	0.03	1.01	2.07	0.18	0.12	1.06	0.00	0.00	1.01	0.06	1.96	2.13	0.17	0.12	1.97	0.21	0.93	0.18	0.18	1.00	1.00	1.12	1.06	0.98	0.16	0.14	1.03	1.96	1.08	1.04	0.14	0.96	0.12	0.88	0.11	0.09	0.96	3.08	1.00	1.07	0.03	0.22	0.93	0.13	0.01	1.00	0.96	0.21	0.00	1.98	1.88	1.07	0.04	0.89	0.99	0.09	0.05	2.02	0.13	0.20	0.97	0.28	0.15",
    "	0.99	0.00	1.01	0.01	0.02	1.00	0.02	0.98	1.01	1.06	0.04	0.02	1.99	0.02	0.97	0.98	0.99	0.00	0.01	1.03	1.00	0.03	0.98	0.04	0.01	1.00	0.02	0.99	0.02	0.00	0.97	0.02	0.03	1.01	0.03	0.02	1.01	0.01	0.02	0.99	0.07	0.00	0.98	0.06	1.00	0.00	0.11	1.00	0.09	0.00	1.98	0.09	0.99	0.03	2.92	1.03	1.04	1.00	0.14	2.02	0.18	0.97	0.03	1.01	3.05	0.08	0.09	2.04	0.18	1.01	2.01	0.99	1.03	0.03	0.11	0.98	0.98	0.00	1.02	0.06	0.97	0.04	0.98	0.08	0.15	0.98	0.15	1.04	2.06	0.15	2.00	0.08	0.16	1.99	0.11	0.09	1.01	0.08	0.04	1.03	0.96	0.07	0.15	4.95	0.09	1.01	2.01	0.00	2.03	0.07	2.00	0.08	0.98	0.02	0.95	0.04	1.01	0.06	0.98	0.11	0.09	1.00	0.08	1.07	0.11	1.95	3.96	0.12	1.03	0.94	0.15	0.01	0.97	0.14	2.01	0.02	0.16	1.02	0.10	0.09	1.02	0.13	1.05	1.03	0.03	2.00	0.15	2.10	3.03	0.03	0.08	2.01	0.95	0.11	0.07	0.91	0.17	1.04	0.08	1.00	0.09	0.16	2.13	1.11	2.12	1.08	2.05	0.13	0.97	0.15	0.95	0.05	0.14	0.93	2.06	0.06	0.13	2.00	0.94	0.12	1.05	0.00	0.17	1.05	0.12	0.01	2.05	0.14	1.01	1.91	0.03	0.02	0.97	0.11	0.15	2.03	0.17	1.89	1.06	0.92	0.06	0.04	0.98	0.11	0.13	0.93	0.00	0.07	1.09	0.08	3.91	0.10	0.07	0.93	0.00	0.10	0.99	0.07	1.09	1.07	0.08	0.90	0.00	0.06	1.02	1.83	2.00	1.02	0.00	2.70	0.11	0.13	1.88	0.16	1.07	0.04	2.91	2.02	0.13	0.04	0.84	1.81	0.58	0.08	1.05	2.38	1.00	0.57	0.10	0.96	1.04	0.14	0.60	0.49	0.08	0.52	0.53	0.13	0.54	1.50	2.18	0.61	0.60	0.92	1.53	0.54	0.53	0.03	1.05	0.52	1.06	0.59	0.03	0.98	0.06	0.15	1.66	0.49	0.61	1.59	0.94	0.92	1.01	0.00	1.03	1.42	0.10	0.01	0.98	0.05	1.08	1.55	1.98	0.57	0.37	0.01	2.20	0.87	0.51	0.46	0.43	1.24	0.48	0.89	1.44	0.05	0.89	0.07	1.87	0.52	0.44	0.49	0.41	1.31	2.00	1.57	0.11	0.39	0.34	0.33	1.74	1.00	0.05	0.00	0.91	0.05	2.96	1.88	0.08	0.55	0.37	0.32	1.26	1.00	0.15	1.05	1.64	0.08	0.87	2.44	0.10	0.56	0.58	0.12	0.76	1.03	0.49	1.77	1.74	0.16	0.16	1.57	0.93	1.07	0.61	0.00	1.02	0.42	1.06	0.65	0.48	0.43	1.47	0.97	1.56	0.41	0.44	0.46	0.35	0.16	1.03	0.56	0.18	0.48	2.50	1.55	1.11	0.09	0.23	0.79	0.00	0.06	1.54	0.48	0.19	0.44	0.91	1.07	1.18	0.12	0.88	0.40	0.14	0.43	1.48	0.05	1.04	0.92	0.01	0.15",
    "	1.00	0.00	1.03	0.03	0.00	0.98	0.03	1.07	1.01	1.03	0.06	0.07	1.87	0.04	0.98	0.97	1.02	0.00	0.03	0.96	0.98	0.03	0.97	0.07	0.03	1.02	0.08	0.99	0.02	0.00	1.00	0.04	0.05	1.07	0.07	0.02	0.97	0.01	0.03	0.99	0.07	0.00	0.98	0.08	1.01	0.00	0.13	1.00	0.07	0.00	1.98	0.14	0.97	0.03	3.01	0.99	1.01	1.01	0.19	2.02	0.06	1.05	0.06	1.07	2.94	0.12	0.14	2.01	0.13	1.04	2.02	1.04	0.89	0.04	0.20	0.98	1.02	0.00	0.95	0.11	0.99	0.03	1.01	0.12	0.17	1.03	0.19	0.99	1.99	0.15	1.97	0.13	0.13	2.06	0.12	0.11	0.95	0.13	0.05	0.99	1.03	0.06	0.16	4.89	0.12	1.01	1.98	0.15	1.91	0.06	1.95	0.13	0.97	0.00	0.98	0.09	1.04	0.04	1.00	0.13	0.12	0.99	0.07	1.04	0.12	2.04	3.95	0.17	0.95	1.05	0.12	0.03	1.02	0.13	2.01	0.07	0.16	1.04	0.11	0.10	1.00	0.00	0.95	0.96	0.03	2.06	0.14	1.94	2.82	0.09	0.12	1.89	1.09	0.12	0.07	1.01	0.13	0.97	0.04	1.01	0.01	0.09	2.01	1.05	1.96	1.02	2.09	0.17	0.99	0.12	0.99	0.08	0.13	0.94	1.85	0.11	0.14	2.03	1.05	0.15	1.02	0.06	0.07	0.99	0.19	0.00	1.96	0.12	0.95	2.04	0.16	0.05	0.95	0.00	0.19	1.98	0.16	2.00	1.01	0.97	0.06	0.11	1.04	0.09	0.16	0.98	0.13	0.07	1.02	0.12	3.85	0.17	0.15	0.99	0.08	0.13	1.02	0.01	1.08	1.03	0.15	0.87	0.09	0.19	0.98	2.06	2.04	1.05	0.06	2.99	0.10	0.16	1.90	0.01	1.07	0.05	2.82	2.04	0.17	0.07	0.98	2.06	0.19	0.08	0.94	0.21	0.89	0.17	0.10	2.98	2.04	0.14	0.18	0.98	0.11	0.17	1.06	0.09	0.08	0.93	0.18	0.13	0.99	1.97	1.06	0.09	1.05	0.16	1.09	0.08	0.12	1.02	0.04	0.88	0.15	0.10	1.93	0.99	0.07	0.08	1.05	1.02	2.09	0.13	0.05	1.07	0.21	0.00	1.02	0.00	1.06	3.05	1.08	0.95	0.15	0.03	2.04	2.17	0.11	0.11	0.92	0.05	0.95	0.11	2.87	0.17	1.00	0.11	2.99	0.00	1.08	0.92	0.19	3.09	0.11	1.94	0.12	0.11	0.98	0.10	0.98	1.10	0.00	0.06	1.05	0.17	4.08	1.08	0.03	0.87	0.09	1.02	1.97	1.05	0.10	0.00	0.97	0.08	1.02	2.10	0.16	0.18	1.00	0.00	0.00	1.10	0.11	1.94	2.13	0.01	0.14	2.03	0.20	0.89	0.07	0.20	1.01	1.02	1.10	1.13	0.95	0.16	0.20	1.07	1.91	1.07	1.03	0.10	1.00	0.11	0.79	0.09	0.03	0.98	3.10	1.06	1.12	0.16	0.20	0.87	0.12	0.04	1.01	0.96	0.04	0.03	2.06	1.97	1.12	0.02	0.75	1.00	0.08	0.10	2.18	0.13	0.26	1.06	0.21	0.17",
    "	0.99	0.00	0.98	0.06	0.03	1.03	0.05	0.92	0.97	1.19	0.00	0.00	2.07	0.00	1.01	0.99	1.05	0.00	0.02	0.99	0.99	0.00	0.96	0.07	0.04	0.96	0.03	1.02	0.02	0.00	0.98	0.06	0.02	1.05	0.07	0.01	0.99	0.01	0.02	0.99	0.05	0.00	0.98	0.05	1.00	0.00	0.05	1.00	0.06	0.00	1.85	0.10	0.99	0.00	3.01	0.94	1.05	1.03	0.14	1.95	0.12	1.00	0.06	1.00	3.02	0.08	0.06	1.95	0.16	1.03	2.01	0.96	1.00	0.06	0.09	1.01	0.96	0.00	1.02	0.09	0.97	0.01	1.01	0.10	0.12	0.98	0.12	1.01	2.06	0.08	1.97	0.11	0.13	1.98	0.14	0.09	0.98	0.08	0.05	0.98	1.01	0.03	0.11	4.95	0.11	1.03	2.01	0.06	2.07	0.09	2.02	0.10	0.99	0.00	0.92	0.09	1.03	0.05	0.99	0.13	0.05	0.98	0.09	1.02	0.08	2.02	4.08	0.11	1.00	1.03	0.15	0.00	0.99	0.12	1.98	0.05	0.09	1.00	0.13	0.06	0.99	0.13	1.01	0.99	0.06	2.04	0.14	2.09	3.04	0.08	0.06	2.03	1.02	0.11	0.03	0.98	0.13	1.03	0.15	0.99	0.11	0.14	2.21	0.94	2.13	1.06	2.06	0.12	1.03	0.11	0.95	0.06	0.09	0.99	1.96	0.11	0.09	1.96	1.01	0.16	1.00	0.04	0.16	1.00	0.12	0.09	1.99	0.18	0.99	1.89	0.11	0.18	0.95	0.12	0.14	2.03	0.17	1.90	1.03	1.00	0.06	0.03	1.02	0.12	0.13	0.95	0.07	0.03	1.03	0.09	3.92	0.10	0.12	0.94	0.13	0.09	1.02	0.12	0.99	1.04	0.13	0.97	0.10	0.13	0.98	1.96	1.99	1.03	0.07	2.85	0.17	0.16	1.92	0.05	1.00	0.07	2.94	2.01	0.12	0.01	1.01	1.96	1.10	0.11	0.16	2.00	0.13	0.93	0.09	0.96	1.95	0.11	0.10	0.98	0.10	0.15	0.99	0.07	0.16	1.01	0.08	0.05	1.03	2.08	0.52	0.04	1.58	0.11	1.00	0.14	0.06	1.05	0.08	0.99	0.15	0.14	2.08	1.00	0.11	0.04	0.97	1.01	1.99	0.10	0.16	0.99	0.12	0.01	1.03	0.06	1.04	2.88	1.08	0.96	0.10	0.04	1.93	1.95	0.15	0.11	0.96	0.07	0.99	0.07	2.85	0.13	0.98	0.15	2.91	0.07	1.01	1.00	0.15	2.84	0.15	1.90	0.15	0.12	0.98	0.07	1.01	1.01	0.09	0.15	0.99	0.11	4.01	1.18	0.07	0.94	0.14	0.98	2.05	0.96	0.16	0.02	0.92	0.08	0.96	2.10	0.14	0.15	1.07	0.08	0.02	1.04	0.09	2.01	2.03	0.07	0.05	2.04	0.19	0.90	0.12	0.13	1.04	0.97	1.14	1.07	0.98	0.11	0.16	0.97	2.00	1.00	1.05	0.12	0.96	0.09	0.92	0.12	0.12	0.91	3.01	1.09	1.05	0.00	0.16	0.94	0.08	0.03	0.97	1.04	0.10	0.00	1.99	2.02	1.01	0.03	0.87	0.97	0.08	0.07	1.95	0.17	0.16	0.98	0.15	0.12",
    "	1.01	0.00	1.00	0.05	0.04	1.00	0.04	0.99	0.96	1.09	0.00	0.01	2.00	0.00	0.99	0.99	1.02	0.00	0.01	1.00	1.01	0.00	1.02	0.02	0.02	1.06	0.06	0.99	0.01	0.01	0.93	0.04	0.06	1.03	0.05	0.06	0.97	0.00	0.07	0.97	0.09	0.00	0.99	0.09	0.98	0.00	0.11	0.97	0.09	0.00	2.03	0.11	1.00	0.03	2.96	1.03	1.00	1.06	0.11	2.06	0.18	0.95	0.06	1.02	3.01	0.09	0.11	2.02	0.16	1.05	2.01	0.94	1.04	0.01	0.10	0.97	1.00	0.00	1.00	0.05	0.98	0.04	0.97	0.13	0.17	1.01	0.16	1.00	2.03	0.14	1.94	0.09	0.16	1.99	0.06	0.12	1.00	0.02	0.05	1.08	0.96	0.07	0.17	4.78	0.11	1.01	2.02	0.02	1.99	0.06	1.94	0.06	0.99	0.01	0.96	0.09	1.00	0.04	0.97	0.14	0.09	1.01	0.13	1.02	0.11	2.04	4.05	0.16	1.04	1.01	0.19	0.00	0.99	0.13	1.96	0.02	0.14	1.05	0.12	0.10	0.99	0.13	0.99	1.00	0.10	2.03	0.15	2.02	3.03	0.03	0.07	1.99	1.01	0.10	0.06	0.95	0.16	1.03	0.09	0.97	0.09	0.08	2.10	1.04	1.99	1.10	2.05	0.10	0.97	0.08	1.00	0.04	0.13	0.97	1.98	0.07	0.10	2.02	0.96	0.14	1.00	0.01	0.18	1.03	0.14	0.12	2.00	0.16	0.97	1.92	0.10	0.16	0.95	0.02	0.12	1.97	0.17	1.92	1.11	0.99	0.06	0.05	0.98	0.17	0.12	0.95	0.09	0.05	1.00	0.09	3.98	0.08	0.13	0.98	0.19	0.13	1.00	0.05	1.01	1.05	0.19	0.95	0.09	0.11	0.99	1.98	2.02	1.03	0.05	2.95	0.18	0.13	1.90	0.05	1.02	0.06	2.85	1.99	0.10	0.05	0.96	1.97	1.07	0.12	0.16	2.02	0.05	1.00	0.09	0.96	2.01	0.11	0.13	0.99	0.09	0.14	1.04	0.12	0.18	0.97	0.19	0.07	1.00	2.02	1.05	0.07	1.01	0.17	1.01	0.12	0.09	1.00	0.06	0.99	0.12	0.17	2.11	0.97	0.09	0.08	1.02	1.05	2.04	0.10	0.20	0.97	0.19	0.02	1.00	0.16	1.09	2.95	1.11	0.96	0.13	0.04	2.03	2.11	0.10	0.12	0.94	0.06	0.97	0.11	2.94	0.16	0.93	0.15	3.02	0.04	1.03	0.99	0.16	2.98	0.09	1.82	0.16	0.05	1.04	0.12	0.99	1.00	0.03	0.10	1.09	0.11	4.02	1.04	0.08	0.90	0.11	1.00	2.03	0.97	0.11	0.00	0.97	0.05	1.05	2.11	0.19	0.13	1.08	0.04	0.04	1.03	0.10	1.95	2.13	0.17	0.14	1.93	0.21	0.93	0.10	0.07	1.02	0.97	1.05	1.07	0.98	0.13	0.15	1.01	1.97	1.00	1.09	0.10	0.97	0.10	0.89	0.10	0.08	0.97	3.02	1.04	1.07	0.14	0.18	0.92	0.11	0.02	1.07	0.95	0.11	0.00	1.93	1.88	1.05	0.03	0.87	0.99	0.07	0.08	1.94	0.13	0.22	0.96	0.20	0.13",
    "	0.98	0.00	0.99	0.01	0.00	1.03	0.02	0.99	0.99	1.09	0.02	0.01	1.96	0.04	0.97	1.00	0.99	0.00	0.06	1.00	0.99	0.02	0.94	0.06	0.00	1.03	0.04	0.98	0.00	0.03	0.98	0.02	0.00	1.07	0.04	0.03	0.98	0.00	0.04	0.99	0.04	0.00	0.97	0.08	0.95	0.00	0.13	0.98	0.06	0.00	1.94	0.10	1.00	0.03	2.99	0.98	1.03	1.03	0.18	1.96	0.15	1.01	0.04	1.06	2.99	0.08	0.08	1.99	0.08	1.00	1.92	0.99	0.99	0.03	0.13	0.99	0.97	0.00	0.99	0.08	1.01	0.02	1.00	0.06	0.13	0.99	0.17	0.98	2.05	0.14	1.93	0.09	0.16	2.01	0.16	0.08	1.02	0.13	0.02	1.04	0.98	0.03	0.12	4.85	0.15	0.99	2.04	0.21	2.01	0.07	1.99	0.07	1.00	0.00	0.96	0.03	1.02	0.03	0.98	0.09	0.10	1.00	0.10	1.04	0.05	2.05	3.95	0.16	0.98	1.01	0.16	0.02	1.01	0.09	2.02	0.04	0.10	1.02	0.11	0.08	1.01	0.12	0.98	1.01	0.16	1.92	0.17	1.90	2.98	0.03	0.08	1.93	1.01	0.06	0.03	1.01	0.12	0.98	0.17	1.02	0.15	0.14	2.12	1.05	2.08	1.01	2.07	0.16	0.92	0.10	1.01	0.10	0.08	1.02	2.04	0.08	0.08	2.03	0.99	0.11	1.02	0.00	0.08	0.98	0.10	0.04	1.96	0.16	0.99	1.95	0.06	0.09	0.93	0.12	0.17	1.99	0.12	1.93	1.08	0.96	0.02	0.07	1.01	0.17	0.07	0.93	0.14	0.06	1.01	0.10	3.82	0.12	0.11	0.94	0.16	0.10	1.04	0.12	1.11	0.99	0.08	0.97	0.04	0.16	1.02	1.88	1.96	0.95	0.03	2.73	0.18	0.16	1.91	0.10	1.00	0.06	2.93	1.92	0.11	0.04	1.01	1.88	1.03	0.11	0.05	1.87	0.06	1.00	0.14	0.96	1.98	0.06	0.10	1.05	0.15	0.09	1.03	0.13	0.17	0.92	0.14	0.08	0.97	1.96	1.04	0.11	1.02	0.17	1.04	0.11	0.05	1.04	0.00	0.99	0.14	0.18	2.04	0.98	0.10	0.08	0.95	1.03	2.04	0.10	0.14	1.00	0.17	0.01	1.06	0.05	1.06	2.98	1.03	0.99	0.07	0.03	2.11	2.05	0.11	0.09	0.96	0.02	1.02	0.05	2.91	0.13	0.94	0.16	2.92	0.06	0.95	1.02	0.13	2.87	0.19	1.78	0.05	0.09	1.05	0.08	0.98	0.99	0.00	0.16	1.08	0.13	4.01	1.05	0.07	0.93	0.03	1.02	2.03	1.01	0.10	0.02	0.96	0.03	1.01	2.05	0.06	0.15	1.03	0.02	0.08	1.03	0.12	2.04	2.10	0.17	0.13	1.89	0.16	0.91	0.16	0.18	1.07	0.96	1.02	1.04	1.01	0.12	0.16	1.01	1.98	0.99	1.08	0.12	0.97	0.07	1.00	0.18	0.16	0.93	2.93	1.05	1.01	0.01	0.14	0.92	0.04	0.03	1.03	0.94	0.15	0.02	2.02	1.93	1.13	0.01	0.92	0.88	0.01	0.10	2.01	0.09	0.15	0.99	0.16	0.14",
    "	0.98	0.00	0.99	0.04	0.03	1.03	0.02	0.99	0.94	1.12	0.04	0.01	2.04	0.02	1.00	0.97	1.02	0.00	0.01	0.98	1.01	0.00	1.01	0.08	0.04	1.00	0.05	0.98	0.03	0.01	0.96	0.04	0.02	1.05	0.05	0.05	0.98	0.00	0.06	0.96	0.07	0.00	0.98	0.08	1.01	0.00	0.06	1.01	0.09	0.00	2.04	0.07	1.01	0.04	2.90	1.02	0.99	1.03	0.09	1.95	0.17	0.99	0.05	0.99	2.99	0.09	0.08	2.02	0.18	1.01	1.96	1.03	0.98	0.02	0.16	0.96	0.99	0.00	1.02	0.07	1.01	0.00	0.98	0.09	0.14	0.97	0.15	1.02	1.97	0.11	1.97	0.09	0.16	2.00	0.14	0.10	0.99	0.09	0.00	0.99	0.99	0.03	0.12	4.89	0.19	1.05	1.97	0.09	2.03	0.05	1.96	0.10	1.02	0.00	0.97	0.09	1.00	0.07	0.96	0.15	0.09	0.95	0.10	1.02	0.12	2.09	3.98	0.16	1.00	1.04	0.09	0.06	1.03	0.12	1.95	0.06	0.14	1.04	0.15	0.07	1.00	0.15	0.98	0.97	0.05	1.98	0.16	2.02	3.07	0.06	0.10	2.04	1.04	0.09	0.05	0.99	0.14	1.01	0.18	0.92	0.07	0.14	2.03	1.09	1.99	1.08	2.05	0.15	0.95	0.11	1.00	0.04	0.12	0.95	1.91	0.08	0.10	2.02	1.00	0.14	0.98	0.03	0.07	1.02	0.15	0.03	1.91	0.16	1.00	1.88	0.10	0.09	0.96	0.00	0.17	1.92	0.15	1.95	1.04	0.99	0.04	0.02	1.02	0.17	0.09	0.96	0.06	0.06	0.99	0.10	3.85	0.10	0.12	1.03	0.16	0.12	0.99	0.12	1.06	1.01	0.15	0.93	0.04	0.12	0.97	1.91	1.99	1.07	0.01	2.96	0.18	0.16	1.97	0.07	1.02	0.08	2.79	1.92	0.12	0.03	0.95	2.09	0.18	0.03	1.00	0.02	1.00	0.04	0.17	3.06	2.02	0.08	0.12	0.97	0.08	0.13	0.99	0.11	0.17	0.95	0.15	0.12	0.99	2.04	1.04	0.12	1.02	0.14	1.06	0.10	0.09	1.00	0.04	0.96	0.12	0.15	2.08	0.99	0.08	0.07	1.04	1.00	1.98	0.14	0.20	1.03	0.20	0.00	0.99	0.16	1.06	2.95	1.09	0.98	0.06	0.02	1.98	2.02	0.12	0.09	0.95	0.05	0.94	0.09	2.94	0.15	0.98	0.16	2.87	0.02	0.99	0.98	0.19	2.90	0.08	1.78	0.14	0.13	1.02	0.13	1.03	1.01	0.00	0.18	1.05	0.16	3.95	1.14	0.07	0.88	0.10	0.95	2.00	0.98	0.06	0.00	0.99	0.06	0.99	2.08	0.17	0.12	1.05	0.00	0.00	1.05	0.08	1.91	2.14	0.16	0.11	1.98	0.05	0.93	0.12	0.19	1.03	0.97	1.13	1.03	0.97	0.16	0.18	1.00	1.96	0.98	1.01	0.06	1.04	0.13	0.89	0.04	0.19	0.96	2.96	1.05	1.06	0.01	0.12	0.98	0.13	0.03	0.98	0.96	0.08	0.01	1.93	1.89	1.08	0.03	0.89	0.96	0.04	0.07	1.97	0.15	0.22	0.95	0.17	0.14",
    "	1.00	0.00	0.98	0.00	0.00	1.02	0.00	1.01	1.02	1.06	0.04	0.07	1.89	0.05	0.93	0.96	1.02	0.06	0.06	0.98	1.01	0.00	0.94	0.11	0.08	1.03	0.10	0.97	0.05	0.02	0.97	0.04	0.03	1.13	0.03	0.03	0.99	0.00	0.02	0.97	0.05	0.00	0.95	0.08	0.98	0.00	0.11	0.99	0.07	0.00	1.95	0.10	1.02	0.01	2.96	1.02	1.00	1.04	0.13	2.01	0.08	1.06	0.02	1.05	3.02	0.06	0.08	1.95	0.10	0.98	1.90	1.05	0.97	0.00	0.15	1.00	1.01	0.00	0.95	0.08	1.02	0.03	0.96	0.09	0.16	0.97	0.17	1.01	2.02	0.12	1.93	0.09	0.17	2.06	0.11	0.09	1.01	0.08	0.03	1.01	1.01	0.02	0.12	4.75	0.08	0.97	1.88	0.20	1.91	0.07	1.92	0.10	1.02	0.00	0.98	0.05	1.00	0.05	1.06	0.10	0.12	0.97	0.05	1.01	0.13	2.02	3.81	0.13	0.96	1.08	0.09	0.09	1.02	0.10	2.00	0.10	0.15	1.03	0.15	0.10	1.01	0.01	0.97	0.96	0.02	1.98	0.13	1.93	2.91	0.10	0.08	1.89	1.02	0.07	0.07	1.04	0.13	0.92	0.08	1.01	0.09	0.12	1.97	1.08	2.01	1.03	1.97	0.14	0.97	0.09	1.03	0.12	0.14	0.94	1.85	0.10	0.14	2.04	1.10	0.11	1.03	0.07	0.04	0.93	0.16	0.03	2.00	0.14	0.94	2.00	0.12	0.04	0.97	0.00	0.07	1.93	0.09	2.05	0.97	0.96	0.08	0.12	1.08	0.10	0.09	0.99	0.02	0.08	1.01	0.11	3.77	0.14	0.13	0.97	0.08	0.12	0.99	0.04	1.06	1.07	0.15	0.93	0.07	0.11	1.02	1.88	1.97	0.99	0.06	2.88	0.13	0.15	1.83	0.09	1.00	0.08	2.92	2.09	0.10	0.02	0.76	1.67	1.05	0.08	0.10	1.99	0.22	1.04	0.12	0.97	1.97	0.21	0.13	1.02	0.10	0.12	1.04	0.10	0.16	0.95	0.15	0.09	1.00	1.99	1.03	0.07	1.01	0.14	1.02	0.13	0.19	1.02	0.23	1.00	0.12	0.18	1.98	0.99	0.04	0.11	1.05	1.02	2.07	0.13	0.21	1.01	0.17	0.00	1.04	0.21	1.10	3.05	1.04	0.94	0.13	0.02	2.10	2.13	0.22	0.12	0.95	0.08	0.96	0.11	2.98	0.18	0.92	0.19	2.98	0.00	0.97	0.96	0.21	3.01	0.20	1.83	0.13	0.14	1.08	0.11	1.03	1.03	0.00	0.19	1.06	0.12	3.80	1.04	0.04	0.89	0.10	0.98	1.99	1.06	0.07	0.00	1.02	0.03	0.99	2.04	0.19	0.14	1.02	0.00	0.00	1.04	0.14	1.89	2.15	0.14	0.11	1.96	0.04	0.94	0.15	0.05	1.01	1.00	1.18	1.05	0.96	0.14	0.19	1.08	1.91	1.06	0.99	0.16	1.00	0.12	0.75	0.10	0.21	0.97	3.01	1.02	1.16	0.04	0.16	0.92	0.14	0.07	1.00	0.91	0.07	0.04	1.94	1.94	1.13	0.04	0.82	1.02	0.11	0.09	2.10	0.14	0.22	1.06	0.25	0.17"]

large_example_seqs = [
    'TATTCGTGTCAGCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGGTTTGGTGAGCCGTTACCTCACCAACTGCCTAATGGAACGCATCCCCATCGATAACCGAAATTCTTTAATAATAACACTGCGGTTACGTTATACATTCGGTACTATCGGTTATTAAGTCTTAACCGGTTATCCCGGTAGTTCGGATAGGTTGGTACTCGCCGTTCGCCCGGTGCCGCGTCCCTA',
    'TATTCGTGTCAGCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCCATCCATCGAAGGTTTGGTGAGCCGTTACCTCACCAACTGCCTAAGACGACTACCACCTCGTAACGAACTATTATAATAACCATAGCCGTTATTAGTACCGTATCGGTATTAATCCTTCTTCGGTAAGTACTCTTCCCGTAGTAAGTCGTACTACCGGTAGTATCGGTGTACGTCGACCGTGTCGTCCGTCGCCGTTCGCCGTTC',
    'TATTCGTGTCAGCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGGTTTGGTGAGCCGTTACCTCACCAACTGCCTAATGGAACGCATCCCCATCGATAACCGAAATTCTTTAATAACTAACACTGTCGGTTACGTTACTACGATTCGGTACTATCGTTATTACGTAGTCAAGCCGTAGTTACCGCGGTCTCGGTGAGTTAGGTTTACGTCGTACCGTGCCGTCGTCGCCGTACCACTTC',
    'TATTCGTGTCAGCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGGTTTGGTGAGCCGTTACCTCACCAACTGCCTAATGGACGCCTCCCCCGCACAACTACTTATTAACAACGACACTCGGTCGTTATATTACACTCGGTACGGTTAATTACTTCTTTCGTTAAGGACGTCTACCGTCCGTATTCGCGGTAGGTGTACGTCGTTCTTCCCGCGGCGCGCC',
    'TATTCGTGTCAGCATGCTGCCTCCCGTAGGAGTTTTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCTACTCACTCGAGGTTAGGTTGGAGCCGGTTACCCGTTCCACCAAACCTAGCCGTAACTTAGGAACGGACCATCCCCACCTCGACTAACCGAAAACTTACTTTTAATTAATACAAACCGAACTGCGGTCCGTTGAGTTATTACCACTCGGGTTACGGTTTAAATTACTTTCTTTTTCCGTTAAAGGACGTATTCCCCGTACCGGTTAATTCGGCCGAGGTTTGGGTAGTAACCGTTGGTTTACTTCCACCCCGTCGCGCCCGGTCCGGCCAC',

    'TATTCGTGTCAGCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGGTTTGGTGAGCCGTTACCTCACCAACTGCCTAATGGAACGCATCCCCATCGATAACCGAAATTCTTTAATAATTAAACCATGCGGTTTTATTATACCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTTATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCATCACTTA',
    'TATTCGTGTCAGCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGGTTTGGTGAGCCGTTACCTCACCAACTGCCTAATGGAACGCATCCCCATCGATAACCGAAATTCTTTAATAATCAAACCATGCGGTCTGATTATACCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTTATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCA',
    'TATTCGTGTCAGCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGGTTTGGTGAGCCGTTACCTCACCAACTGCCTAATGGAACGCATCCCCATCGATAACCGAAATTCTTTAATAACAAGACCATGCGGTCTGATTATACCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTTATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCA',
    'TATTCGTGTCAGCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGGTTTGGTGAGCCGTTACCTCACCAACTGCCTAATGGAACGCATCCCCATCGATAACCGAAATTCTTTAATAACTAACACAGTCGGTTACGTTACTACGATTCGGTACTATCGGTTATTTACTACGTTCTTACAACCGGCTTATCCCGGAGTAGTGGCTAGGTTGGTACTCGTCGTTCCGCCCGGTGCGCGTCCCA']

fasting_subset_sff = """Common Header:
  Magic Number:  0x2E736666
  Version:       0001
  Index Offset:  647488
  Index Length:  9251
  # of Reads:    392
  Header Length: 440
  Key Length:    4
  # of Flows:    400
  Flowgram Code: 1
  Flow Chars:    TACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACGTACG
  Key Sequence:  TCAG

>FLP3FBN01ELBSX
  Run Prefix:   R_2008_12_09_13_51_01_
  Region #:     1
  XY Location:  1766_0111

  Run Name:       R_2008_12_09_13_51_01_FLX04080350_Administrator_PC_Calrestricted
  Analysis Name:  D_2009_06_10_10_15_54_node176_fullProcessingAmplicons
  Full Path:      /srv/seq/454/2008_12_09/R_2008_12_09_13_51_01_FLX04080350_Administrator_PC_Calrestricted/D_2009_06_10_10_15_54_node176_fullProcessingAmplicons/

  Read Header Len:  32
  Name Length:      14
  # of Bases:       254
  Clip Qual Left:   5
  Clip Qual Right:  254
  Clip Adap Left:   0
  Clip Adap Right:  0

Flowgram:	1.04	0.13	1.07	0.19	0.21	0.86	0.20	0.97	0.20	1.01	1.06	0.13	0.11	0.95	0.15	1.05	0.15	1.03	0.15	1.04	1.08	0.06	1.01	1.89	0.14	0.05	1.04	0.15	1.00	0.10	0.98	0.15	0.16	1.02	0.18	0.15	1.00	0.13	0.19	0.95	0.18	0.14	1.03	0.16	1.04	0.11	0.17	1.00	0.14	0.11	1.81	0.14	1.05	0.13	2.79	0.94	1.01	1.02	0.19	1.87	0.13	0.99	0.18	1.01	1.03	0.13	1.06	0.10	1.01	0.10	0.08	2.89	0.12	0.09	1.95	0.87	0.94	0.10	0.10	1.05	1.01	0.08	1.04	0.13	0.98	0.10	1.06	0.15	0.08	1.03	0.11	1.03	0.97	0.11	2.84	0.13	0.08	1.95	0.13	0.13	1.01	0.01	0.18	0.83	1.02	0.10	0.09	1.74	0.11	0.09	1.80	1.07	2.75	1.11	2.72	0.13	0.94	0.12	0.84	0.15	1.05	0.13	1.02	0.16	0.12	1.03	0.06	1.88	0.20	0.06	1.85	1.88	0.10	0.08	0.87	0.05	1.09	1.07	0.91	1.06	0.10	0.07	0.99	0.11	0.06	1.00	0.11	0.11	1.07	0.06	1.05	0.09	0.03	1.09	0.07	0.03	0.93	0.06	1.04	0.95	0.02	0.11	2.09	0.00	1.91	0.09	0.03	1.94	0.92	0.06	0.00	2.78	0.08	0.13	2.14	1.04	2.08	1.16	1.95	0.14	1.04	0.06	0.81	0.19	0.11	0.93	2.02	0.13	0.12	1.83	1.08	0.04	1.12	0.83	0.08	1.04	0.08	0.00	1.06	0.16	1.01	1.90	0.06	0.10	1.06	0.00	0.09	0.98	0.13	0.11	0.97	0.96	0.17	0.09	1.91	1.02	0.10	0.12	0.95	0.00	0.18	1.07	0.06	2.20	1.08	0.09	2.03	0.10	0.09	0.96	0.00	0.17	1.05	0.08	2.01	0.16	0.01	1.04	0.08	0.09	1.00	0.03	0.11	1.06	1.85	0.03	1.10	0.00	0.08	0.92	0.78	1.12	0.01	0.04	2.01	0.01	1.93	0.00	0.04	1.04	0.10	1.04	0.16	0.00	1.10	0.00	0.04	3.00	0.12	0.10	1.13	0.82	0.13	0.05	0.99	0.09	2.88	1.87	0.16	0.13	0.76	0.61	0.12	0.17	0.92	0.99	1.18	0.06	1.04	0.00	0.09	1.03	0.10	1.03	0.07	1.02	0.14	0.08	1.10	0.10	0.04	1.03	0.13	0.01	1.02	0.03	0.09	1.08	0.06	0.07	0.96	0.97	0.07	0.09	1.00	0.10	0.98	0.11	1.05	0.11	0.93	0.10	0.15	1.21	1.06	0.99	0.03	0.06	1.04	0.92	2.32	0.06	1.04	0.99	0.01	0.15	0.78	0.00	2.28	2.11	4.18	1.11	0.00	0.90	0.13	0.13	1.14	0.04	0.85	1.05	1.87	1.15	2.85	0.00	1.79	0.00	0.11	1.01	0.02	1.07	0.09	0.12	1.14	0.00	0.11	0.98	0.08	1.20	2.03	0.97	0.16	0.00	1.02	0.00	3.20	2.20	0.14	0.89	0.72	0.07	0.17	0.93	1.20	0.08	0.02	0.98	0.17	0.12	1.21	0.12	0.09	3.20	0.05	0.00	1.14	0.00	0.64	1.33	0.35	2.36
Flow Indexes:	1	3	6	8	10	11	14	16	18	20	21	23	24	24	27	29	31	34	37	40	43	45	48	51	51	53	55	55	55	56	57	58	60	60	62	64	65	67	69	72	72	72	75	75	76	77	80	81	83	85	87	90	92	93	95	95	95	98	98	101	104	105	108	108	111	111	112	113	113	113	114	115	115	115	117	119	121	123	126	128	128	131	131	132	132	135	137	138	139	140	143	146	149	151	154	157	159	160	163	163	165	165	168	168	169	172	172	172	175	175	176	177	177	178	179	179	181	183	186	187	187	190	190	191	193	194	196	199	201	202	202	205	208	211	212	215	215	216	219	222	224	224	225	227	227	230	233	235	235	238	241	244	245	245	247	250	251	252	255	255	257	257	260	262	265	268	268	268	271	272	275	277	277	277	278	278	281	282	285	286	287	289	292	294	296	299	302	305	308	311	312	315	317	319	321	324	325	326	329	330	331	331	333	334	337	339	339	340	340	341	341	341	341	342	344	347	349	350	351	351	352	353	353	353	355	355	358	360	363	366	368	369	369	370	373	375	375	375	376	376	378	379	382	383	386	389	392	392	392	395	397	398	400	400
Bases:	tcagACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTTACCCTCTCAGGCCGGCTACGCATCATCGCCTTGGTGGGCCGTTACCTCACCAACTAGCTAATGCGCCGCAGGTCCATCCATGTTCACGCCTTGATGGGCGCTTTAATATACTGAGCATGCGCTCTGTATACCTATCCGGTTTTAGCTACCGTTTCCAGCAGTTATCCCGGACACATGGGCTAGG
Quality Scores:	37	36	36	36	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	36	36	33	33	33	36	37	37	37	37	37	37	40	40	40	39	39	38	40	40	40	40	40	40	40	37	37	37	37	37	35	35	35	37	37	37	37	37	35	35	35	31	31	23	23	23	31	21	21	21	35	35	37	37	37	36	36	36	36	36	36	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	28	28	28	36	36	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	36	36	36	37	37	37	37	37	37	37	37	37	37	37	37	36	36	36	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	35	32	32	32	32	35	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	36	32	32	32	36	37	35	32	32	32	32	32	32	32	32	36	37	37	37	37	36	36	31	31	32	32	36	36	36	36	36	36	36	36	36	36	36	28	27	27	27	26	26	26	30	29	30	29	24	24	24	21	15	15	13	13

>FLP3FBN01EG8AX
  Run Prefix:   R_2008_12_09_13_51_01_
  Region #:     1
  XY Location:  1719_1463

  Run Name:       R_2008_12_09_13_51_01_FLX04080350_Administrator_PC_Calrestricted
  Analysis Name:  D_2009_06_10_10_15_54_node176_fullProcessingAmplicons
  Full Path:      /srv/seq/454/2008_12_09/R_2008_12_09_13_51_01_FLX04080350_Administrator_PC_Calrestricted/D_2009_06_10_10_15_54_node176_fullProcessingAmplicons/

  Read Header Len:  32
  Name Length:      14
  # of Bases:       280
  Clip Qual Left:   5
  Clip Qual Right:  280
  Clip Adap Left:   0
  Clip Adap Right:  0

Flowgram:	1.05	0.13	1.03	0.18	0.21	0.93	0.20	0.94	0.17	1.04	1.04	0.10	0.13	1.05	0.18	1.01	0.14	1.04	0.14	1.03	1.08	0.06	1.01	2.02	0.14	0.04	0.95	0.12	1.04	0.08	0.98	0.15	0.14	1.08	0.19	0.12	1.04	0.12	0.15	0.96	0.13	0.11	1.01	0.12	0.93	0.10	0.16	1.02	0.13	0.12	2.15	0.13	1.03	0.14	2.57	1.08	1.01	1.04	0.18	1.83	0.20	0.91	0.17	0.96	2.68	0.11	0.19	1.92	0.14	1.07	1.95	0.99	1.05	0.14	0.14	0.98	1.07	0.11	1.04	0.11	0.97	0.15	0.95	0.15	0.14	0.96	0.12	1.06	2.02	0.08	1.93	0.13	0.14	1.96	0.12	0.12	1.03	0.04	0.17	1.26	0.93	0.08	0.19	5.41	0.12	0.10	2.00	0.00	1.93	0.08	1.83	0.13	1.01	0.08	1.00	0.15	1.04	0.09	0.83	0.19	0.13	1.07	0.18	1.09	0.11	2.03	3.90	0.11	1.08	0.82	0.16	0.10	1.02	0.11	1.88	0.09	0.07	1.04	0.13	0.13	1.04	0.05	1.09	1.02	0.11	1.93	0.14	1.88	0.11	0.03	1.01	0.04	1.85	0.09	0.13	2.12	0.93	0.10	0.11	2.96	0.10	0.13	2.16	0.83	2.13	1.17	3.97	1.01	0.07	0.02	1.72	0.08	0.13	2.15	0.92	0.12	0.13	1.89	1.88	0.09	1.00	1.81	0.06	0.16	1.06	0.14	0.12	2.05	0.11	1.89	1.05	1.01	0.13	0.05	0.96	0.10	0.07	0.88	0.21	0.11	1.10	0.14	3.70	0.17	0.06	0.97	0.13	0.06	1.09	0.13	1.05	1.04	0.06	1.03	0.15	0.00	1.03	0.03	0.06	1.04	0.13	0.91	1.96	0.94	0.09	2.04	0.16	1.09	2.00	0.00	1.09	0.13	2.80	2.05	0.11	0.06	0.87	0.91	0.11	1.11	2.00	0.14	1.02	0.07	0.90	1.07	1.91	0.16	0.03	0.95	0.12	0.04	1.06	0.05	0.11	1.04	0.09	0.01	1.10	1.98	0.17	1.96	0.09	0.96	0.08	1.79	1.07	0.10	1.00	0.79	0.09	0.06	1.03	0.07	0.15	1.07	0.09	0.14	2.06	0.11	0.04	1.09	0.01	0.11	1.01	0.00	1.11	2.95	0.90	1.08	0.16	0.00	1.89	2.04	0.08	0.17	0.96	0.01	1.03	0.13	2.77	0.06	0.92	0.04	2.77	0.08	1.00	0.95	0.04	2.95	0.11	1.95	0.06	0.07	1.02	0.08	1.08	1.10	0.12	0.05	0.92	0.12	3.68	0.93	0.02	0.98	0.15	1.11	1.03	0.14	1.04	0.09	0.01	1.02	0.04	0.11	1.07	0.03	0.98	1.94	0.11	0.04	1.11	0.00	0.09	0.94	0.11	1.95	2.10	0.00	0.16	1.87	0.11	0.88	0.03	0.00	1.07	1.11	1.16	1.05	0.88	0.11	0.00	0.97	1.84	1.04	1.10	0.14	0.98	0.15	0.85	0.00	0.00	1.05	2.79	1.07	1.17	0.02	0.00	0.74	0.08	0.10	1.01	1.02	0.10	0.10	1.78	2.02	1.12	0.10	0.83	0.71	0.15	0.12	1.96	0.16	0.25	1.09	0.16	0.19
Flow Indexes:	1	3	6	8	10	11	14	16	18	20	21	23	24	24	27	29	31	34	37	40	43	45	48	51	51	53	55	55	55	56	57	58	60	60	62	64	65	65	65	68	68	70	71	71	72	73	76	77	79	81	83	86	88	89	89	91	91	94	94	97	100	101	104	104	104	104	104	107	107	109	109	111	111	113	115	117	119	122	124	126	126	127	127	127	127	129	130	133	135	135	138	141	143	144	146	146	148	148	151	153	153	156	156	157	160	160	160	163	163	164	165	165	166	167	167	167	167	168	171	171	174	174	175	178	178	179	179	181	182	182	185	188	188	190	190	191	192	195	198	201	203	203	203	203	206	209	211	212	214	217	220	222	223	223	224	226	226	228	229	229	231	233	233	233	234	234	237	238	240	241	241	243	245	246	247	247	250	253	256	259	260	260	262	262	264	266	266	267	269	270	273	276	279	279	282	285	287	288	288	288	289	290	293	293	294	294	297	299	301	301	301	303	305	305	305	307	308	310	310	310	312	312	315	317	318	321	323	323	323	323	324	326	328	329	331	334	337	339	340	340	343	346	348	348	349	349	352	352	354	357	358	359	360	361	364	365	365	366	367	369	371	374	375	375	375	376	377	380	383	384	387	387	388	388	389	391	392	395	395	398
Bases:	tcagACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGGGGGCCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGGGCCGTTACCCCGCCAACAACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGAAGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCA
Quality Scores:	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	38	37	33	33	21	21	21	26	33	37	36	36	40	33	24	24	29	33	33	39	39	39	40	39	39	39	40	37	37	37	37	37	37	37	37	37	37	37	32	32	20	20	20	20	20	35	35	37	37	37	37	37	37	37	36	36	36	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	36	36	36	36	36	36	37	37	37	37	37	36	36	36	36	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	36	33	28	28	28	28	36	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	36	33	33	33	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	36	36	36	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	28	28	28	37	28	28	28	37	37	37	37	37	36	36	36	36	36	28	26	26	26	26	28	36	36	36	36	36	36	36	37	38	38	38	38	38	37	37	37	37	37	31	31	31	31	31	31	31	31	31	31	31	31	30	22	22	22	25	25	31	31	31	31	31	31	31	25	25	25	25	25	28

>FLP3FBN01EEWKD
  Run Prefix:   R_2008_12_09_13_51_01_
  Region #:     1
  XY Location:  1692_3531

  Run Name:       R_2008_12_09_13_51_01_FLX04080350_Administrator_PC_Calrestricted
  Analysis Name:  D_2009_06_10_10_15_54_node176_fullProcessingAmplicons
  Full Path:      /srv/seq/454/2008_12_09/R_2008_12_09_13_51_01_FLX04080350_Administrator_PC_Calrestricted/D_2009_06_10_10_15_54_node176_fullProcessingAmplicons/

  Read Header Len:  32
  Name Length:      14
  # of Bases:       249
  Clip Qual Left:   5
  Clip Qual Right:  248
  Clip Adap Left:   0
  Clip Adap Right:  0

Flowgram:	1.04	0.16	1.06	0.23	0.23	0.88	0.23	0.95	0.22	0.88	0.14	1.01	0.16	0.07	1.12	0.15	0.14	1.04	1.07	1.06	0.15	1.01	0.15	1.03	0.16	0.09	1.96	0.15	1.05	1.07	0.94	0.19	0.18	0.96	0.20	0.17	0.98	0.16	0.17	0.96	0.15	0.15	1.09	0.15	1.03	0.16	0.13	1.01	0.14	0.15	1.71	0.17	1.03	0.14	2.65	1.00	1.11	0.95	0.17	1.92	0.10	0.94	0.20	0.92	2.80	0.13	0.13	2.83	0.11	0.11	1.98	0.92	1.07	0.13	0.10	1.03	0.92	0.13	1.04	0.11	0.92	0.14	0.89	0.18	0.13	1.06	0.10	1.04	1.07	0.12	2.78	0.13	0.12	1.92	0.08	0.17	1.02	0.01	0.16	0.94	0.92	0.10	0.14	1.72	0.10	0.12	1.98	0.85	0.10	1.13	0.05	0.11	1.04	0.08	0.85	0.15	0.12	1.06	0.08	1.10	1.01	0.11	1.02	0.14	1.02	0.10	0.90	0.15	2.01	2.06	1.00	0.09	0.83	0.00	0.89	2.06	0.11	0.09	1.00	0.05	0.97	1.09	0.08	0.10	0.97	0.04	0.12	1.02	0.08	0.11	1.01	0.09	0.12	1.17	0.13	0.09	1.02	0.10	1.01	0.06	0.09	1.04	0.03	0.11	1.98	0.05	0.03	1.06	0.10	0.10	1.94	0.02	2.05	0.11	0.00	2.18	1.05	2.29	0.06	0.87	0.01	0.00	2.08	1.14	2.23	1.00	2.03	0.16	1.97	0.96	1.88	0.17	0.10	1.81	0.92	0.11	1.12	0.75	0.13	1.06	0.10	0.09	1.05	0.13	1.08	2.08	0.06	0.10	1.00	0.00	0.06	1.01	0.13	0.10	0.91	0.06	0.15	1.12	1.79	1.01	0.14	0.01	0.90	0.00	0.13	1.05	0.03	2.13	1.10	0.02	1.88	0.06	0.09	1.02	0.13	0.12	1.07	0.05	2.09	0.02	0.08	2.07	0.04	1.06	0.08	0.68	0.17	0.89	1.04	0.08	0.12	1.02	0.07	0.96	0.20	0.04	1.01	0.99	0.06	1.10	0.03	0.05	1.15	0.12	0.12	0.86	0.00	1.12	0.20	1.97	2.18	0.09	0.13	0.81	0.07	0.12	1.02	0.07	1.12	0.14	3.15	0.00	1.02	0.11	0.03	3.24	0.85	0.25	0.97	0.00	1.04	0.05	1.08	0.93	0.14	1.17	0.06	0.82	1.14	0.07	0.04	0.97	0.11	0.02	1.03	0.03	0.07	1.04	0.11	0.12	1.07	0.90	1.02	0.12	1.10	0.07	0.85	1.28	0.08	0.89	0.85	0.00	0.13	0.96	1.95	0.04	0.15	1.10	1.85	1.12	0.06	0.00	1.02	0.10	2.27	2.18	0.85	0.93	0.00	0.10	2.05	1.01	0.03	1.07	0.00	0.07	1.16	0.05	0.01	1.11	0.02	0.16	1.06	0.00	1.07	0.04	0.73	0.10	0.00	1.10	2.94	0.16	2.27	0.00	0.05	1.10	0.00	2.15	1.09	0.05	0.12	0.73	2.08	1.13	0.02	0.07	0.93	0.07	3.33	0.12	0.15	1.00	0.02	1.14	1.05	0.04	0.92	0.14	1.01	0.10	1.05	0.03	2.00	0.11	0.14	3.02	0.01	0.48	1.03	0.13
Flow Indexes:	1	3	6	8	10	12	15	18	19	20	22	24	27	27	29	30	31	34	37	40	43	45	48	51	51	53	55	55	55	56	57	58	60	60	62	64	65	65	65	68	68	68	71	71	72	73	76	77	79	81	83	86	88	89	91	91	91	94	94	97	100	101	104	104	107	107	108	110	113	115	118	120	121	123	125	127	129	129	130	130	131	133	135	136	136	139	141	142	145	148	151	154	157	159	162	165	165	168	171	171	173	173	176	176	177	178	178	180	183	183	184	185	185	186	187	187	189	189	190	191	191	194	194	195	197	198	200	203	205	206	206	209	212	215	218	219	219	220	223	226	228	228	229	231	231	234	237	239	239	242	242	244	246	248	249	252	254	257	258	260	263	266	268	270	270	271	271	274	277	279	281	281	281	283	286	286	286	287	289	291	293	294	296	298	299	302	305	308	311	312	313	315	317	318	320	321	324	325	325	328	329	329	330	333	335	335	336	336	337	338	341	341	342	344	347	350	353	355	357	360	361	361	361	363	363	366	368	368	369	372	373	373	374	377	379	379	379	382	384	385	387	389	391	393	393	396	396	396	399
Bases:	tcagAGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAGTCTCTTAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGACATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCTTGGGc
Quality Scores:	36	35	35	35	36	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	36	36	33	33	26	21	21	31	36	37	37	37	36	36	33	34	34	36	37	36	37	37	37	37	37	37	37	37	37	37	37	36	28	28	28	36	36	37	37	35	35	35	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	32	32	32	35	35	37	35	32	32	32	37	37	37	37	37	37	36	36	36	36	36	36	36	36	37	37	37	37	37	37	37	35	35	35	37	37	37	37	37	37	37	37	37	37	37	37	37	37	36	35	35	35	37	37	37	37	37	37	37	37	37	37	36	36	36	37	36	35	35	35	37	28	28	28	32	35	37	37	37	36	36	36	37	37	37	37	37	37	35	35	35	35	35	37	37	37	37	36	36	36	37	28	28	28	28	35	36	37	37	37	37	37	37	37	37	37	37	36	33	33	32	31	36	36	33	33	27	27	27	36	31	25	25	25	32	36	36	36	36	36	36	36	36	36	36	36

>FLP3FBN01DEHK3
  Run Prefix:   R_2008_12_09_13_51_01_
  Region #:     1
  XY Location:  1278_0245

  Run Name:       R_2008_12_09_13_51_01_FLX04080350_Administrator_PC_Calrestricted
  Analysis Name:  D_2009_06_10_10_15_54_node176_fullProcessingAmplicons
  Full Path:      /srv/seq/454/2008_12_09/R_2008_12_09_13_51_01_FLX04080350_Administrator_PC_Calrestricted/D_2009_06_10_10_15_54_node176_fullProcessingAmplicons/

  Read Header Len:  32
  Name Length:      14
  # of Bases:       256
  Clip Qual Left:   5
  Clip Qual Right:  256
  Clip Adap Left:   0
  Clip Adap Right:  0

Flowgram:	1.01	0.02	0.93	0.03	0.02	1.07	0.05	1.06	0.04	1.05	1.74	0.11	0.07	1.02	0.10	0.92	0.06	0.13	0.83	0.90	0.08	1.02	1.01	0.11	0.97	1.12	0.14	0.90	0.12	0.09	1.00	0.07	0.05	1.05	0.07	0.06	1.02	0.08	0.05	1.01	0.12	0.01	0.99	0.12	0.96	0.02	0.16	1.01	0.11	0.02	2.10	0.12	1.05	0.05	2.93	1.07	1.00	1.02	0.13	1.92	0.15	0.89	0.08	1.08	1.14	0.07	1.00	0.14	0.97	0.03	0.11	2.97	0.15	0.02	1.88	1.01	1.06	0.01	0.12	1.03	0.93	0.03	1.00	0.09	0.98	0.08	1.01	0.10	0.16	1.03	0.10	1.03	1.06	0.11	2.82	0.11	0.14	1.82	0.06	0.14	0.98	0.13	0.08	1.17	0.95	0.08	0.11	2.00	0.09	0.04	2.00	1.01	1.92	0.12	1.02	0.09	0.12	1.84	1.97	0.07	0.99	0.05	1.01	0.05	1.01	0.13	1.07	0.14	0.13	1.01	0.11	1.03	0.98	0.10	1.90	1.87	0.12	0.09	1.01	0.08	0.98	1.07	1.01	0.14	1.03	0.09	0.14	1.04	0.13	0.88	0.12	0.10	1.05	0.08	1.01	1.04	1.02	0.07	1.00	1.08	0.06	0.98	1.07	0.12	1.98	0.09	0.13	2.14	1.03	0.11	0.21	0.92	0.09	1.07	0.14	1.02	0.05	0.14	2.15	0.96	2.11	1.12	2.04	0.15	0.91	0.16	0.84	0.11	0.08	1.14	1.95	0.09	0.17	1.87	0.89	0.09	1.03	1.02	0.14	0.09	1.03	0.15	1.14	0.09	0.91	1.90	0.11	0.09	0.97	0.09	1.11	0.13	0.16	0.89	0.17	1.08	0.12	0.95	1.10	0.93	0.11	0.13	1.08	0.96	0.11	1.06	0.08	0.78	0.13	0.13	2.95	0.07	0.13	1.07	0.11	0.12	1.01	0.10	1.02	0.11	3.02	0.10	0.93	0.15	0.05	1.01	0.17	1.08	0.12	0.13	1.03	1.88	0.08	1.01	0.09	0.12	2.13	0.07	0.11	1.06	0.12	0.09	0.95	0.08	1.00	0.10	2.11	0.08	3.00	0.15	0.06	2.19	0.82	1.21	0.05	0.15	2.03	0.06	1.96	2.07	0.08	0.16	0.93	0.90	0.09	0.97	0.09	0.15	1.14	0.04	0.00	1.06	0.09	0.13	1.96	1.00	0.18	0.00	1.80	0.05	0.06	3.75	0.08	0.14	1.23	0.49	1.05	0.12	0.00	0.71	0.08	0.12	2.07	0.91	0.14	0.15	1.02	0.07	0.12	1.11	0.18	0.06	1.10	1.95	1.07	0.92	0.11	0.13	2.01	0.89	0.07	0.99	0.08	0.12	1.08	0.04	0.05	1.01	0.14	1.02	1.05	0.06	0.96	1.07	3.08	0.00	1.98	0.07	0.12	2.15	1.02	0.04	0.97	0.00	0.13	1.08	2.10	0.15	0.00	1.10	0.81	0.12	5.20	0.15	0.95	0.14	0.51	0.06	1.07	0.04	0.18	1.06	0.07	3.03	0.09	2.04	0.16	0.02	1.10	0.14	0.00	0.97	0.18	1.85	2.07	0.03	0.06	0.78	0.08	0.11	1.00	0.05	1.10	0.14	0.83	0.04	0.16	1.10	1.07	1.13
Flow Indexes:	1	3	6	8	10	11	11	14	16	19	20	22	23	25	26	28	31	34	37	40	43	45	48	51	51	53	55	55	55	56	57	58	60	60	62	64	65	67	69	72	72	72	75	75	76	77	80	81	83	85	87	90	92	93	95	95	95	98	98	101	104	105	108	108	111	111	112	113	113	115	118	118	119	119	121	123	125	127	130	132	133	135	135	136	136	139	141	142	143	145	148	150	153	155	156	157	159	160	162	163	165	165	168	168	169	172	174	176	179	179	180	181	181	182	183	183	185	187	190	191	191	194	194	195	197	198	201	203	205	206	206	209	211	214	216	218	219	220	223	224	226	228	231	231	231	234	237	239	241	241	241	243	246	248	251	252	252	254	257	257	260	263	265	267	267	269	269	269	272	272	273	274	277	277	279	279	280	280	283	284	286	289	292	295	295	296	299	299	302	302	302	302	305	307	310	313	313	314	317	320	323	324	324	325	326	329	329	330	332	335	338	340	341	343	344	345	345	345	347	347	350	350	351	353	356	357	357	360	361	363	363	363	363	363	365	367	369	372	374	374	374	376	376	379	382	384	384	385	385	388	391	393	395	398	399	400
Bases:	tcagACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGACTTGGTGAGCCGTTACCTCACCAACTATCTAATCAGACGCGAGCCCATCTTTCAGCGGATTGCTCCTTTGGTATTCCGGCGATGCCGCCAAAATCATTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGAAAGGCAGGTTGCTCACG
Quality Scores:	37	37	37	37	36	36	36	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	40	40	40	40	40	40	40	40	40	40	38	38	39	40	40	40	40	40	40	40	40	40	40	40	40	40	40	37	37	37	37	37	33	33	33	36	36	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	33	33	33	33	37	37	37	37	37	37	37	37	37	37	37	37	37	37	36	36	28	28	28	28	33	33	33	36	36	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	36	36	36	36	36	36	31	31	27	27	28	28	28	27	21	31	31	36	36	36	36	36	36	36	36	36	36	36	31	31	31	31	31	31	31

>FLP3FBN01DGFYQ
  Run Prefix:   R_2008_12_09_13_51_01_
  Region #:     1
  XY Location:  1300_1344

  Run Name:       R_2008_12_09_13_51_01_FLX04080350_Administrator_PC_Calrestricted
  Analysis Name:  D_2009_06_10_10_15_54_node176_fullProcessingAmplicons
  Full Path:      /srv/seq/454/2008_12_09/R_2008_12_09_13_51_01_FLX04080350_Administrator_PC_Calrestricted/D_2009_06_10_10_15_54_node176_fullProcessingAmplicons/

  Read Header Len:  32
  Name Length:      14
  # of Bases:       273
  Clip Qual Left:   5
  Clip Qual Right:  273
  Clip Adap Left:   0
  Clip Adap Right:  0

Flowgram:	1.01	0.00	1.01	0.00	0.00	1.02	0.00	1.09	0.00	1.12	0.79	0.15	0.14	1.04	0.03	1.11	0.00	0.95	0.14	0.84	0.70	0.19	1.05	1.87	0.03	0.33	1.00	0.28	1.01	0.13	0.92	0.10	0.08	1.10	0.14	0.07	1.02	0.13	0.06	1.00	0.17	0.00	0.98	0.15	0.94	0.00	0.21	0.82	0.15	0.00	2.13	0.14	1.03	0.05	3.29	1.11	1.17	1.04	0.31	2.11	0.25	1.01	0.01	1.22	2.73	0.12	0.08	2.73	0.19	0.03	1.74	1.05	0.99	0.00	0.11	1.15	1.00	0.00	1.05	0.12	1.04	0.03	0.83	0.13	0.17	0.86	0.14	0.81	1.11	0.08	2.81	0.06	0.10	2.00	0.13	0.11	1.07	0.08	0.17	1.02	0.90	0.12	0.13	2.15	0.03	0.07	1.70	1.07	2.08	0.01	1.01	0.14	0.11	0.88	2.91	0.07	1.09	0.07	1.10	0.01	0.85	0.14	1.03	0.10	0.12	1.01	0.29	1.72	0.14	0.11	1.74	1.82	0.10	0.08	1.04	0.12	0.98	1.02	1.09	0.13	1.01	0.11	0.13	0.94	0.09	1.15	0.09	0.14	0.85	0.11	0.83	0.83	1.19	0.00	0.89	1.16	0.07	0.00	1.01	0.11	2.98	0.07	0.11	2.12	1.04	0.93	0.10	2.10	0.12	0.05	2.07	1.16	2.20	1.02	3.12	0.08	1.04	0.15	0.32	0.77	0.07	0.23	2.08	0.11	0.08	2.09	1.12	0.07	0.98	0.18	0.13	1.91	0.11	0.15	0.92	0.09	1.06	2.09	0.11	0.15	0.82	0.08	1.01	0.06	0.02	1.01	0.07	0.95	0.08	0.84	1.17	1.05	0.10	0.03	1.13	2.70	1.06	0.10	1.82	0.00	0.16	1.01	0.15	0.13	0.97	0.07	1.06	0.07	1.00	0.08	1.04	0.10	0.18	1.08	0.94	0.10	0.11	1.15	1.85	1.01	0.17	0.76	0.04	0.14	2.24	2.13	0.20	0.11	0.73	0.20	1.08	0.06	4.64	0.19	1.98	0.22	0.00	2.06	2.09	0.15	0.21	1.10	0.16	0.90	0.24	0.95	0.17	1.00	0.17	0.16	1.09	0.09	0.15	1.05	0.15	0.07	1.33	0.08	0.08	0.96	0.10	0.23	0.84	0.98	0.12	0.11	3.50	0.09	1.01	0.11	0.01	1.07	2.19	0.20	0.12	1.88	0.02	0.05	1.05	0.01	2.09	1.01	0.11	0.12	1.18	0.06	0.01	0.88	0.39	0.05	1.11	1.93	0.92	0.98	0.07	0.13	2.04	0.98	0.08	0.92	0.06	0.09	0.96	1.62	1.20	0.15	1.01	1.05	2.95	0.01	1.68	0.15	0.15	2.03	0.93	0.15	1.04	0.09	0.21	1.04	2.03	1.06	0.00	0.19	1.08	0.02	4.70	0.14	1.00	0.10	0.08	1.03	0.88	0.10	0.19	0.79	1.11	0.14	0.12	1.12	0.17	1.10	0.13	1.93	0.14	0.13	1.21	0.00	0.07	1.06	0.10	2.14	2.23	0.76	2.99	0.03	0.01	0.70	0.78	1.00	0.09	0.13	1.22	1.18	2.00	0.96	1.12	0.07	0.88	0.10	1.09	0.05	0.08	0.98	2.85	1.05	1.06	0.07	2.26	1.04
Flow Indexes:	1	3	6	8	10	11	14	16	18	20	21	23	24	24	27	29	31	34	37	40	43	45	48	51	51	53	55	55	55	56	57	58	60	60	62	64	65	65	65	68	68	68	71	71	72	73	76	77	79	81	83	86	88	89	91	91	91	94	94	97	100	101	104	104	107	107	108	109	109	111	114	115	115	115	117	119	121	123	126	128	128	131	131	132	132	135	137	138	139	141	144	146	149	151	152	153	155	156	159	161	161	161	164	164	165	166	168	168	171	171	172	173	173	174	175	175	175	177	180	183	183	186	186	187	189	192	192	195	197	198	198	201	203	206	208	210	211	212	215	216	216	216	217	219	219	222	225	227	229	231	234	235	238	239	239	240	242	245	245	246	246	249	251	253	253	253	253	253	255	255	258	258	259	259	262	264	266	268	271	274	277	280	283	284	287	287	287	287	289	292	293	293	296	296	299	301	301	302	305	308	311	312	312	313	314	317	317	318	320	323	324	324	325	327	328	329	329	329	331	331	334	334	335	337	340	341	341	342	345	347	347	347	347	347	349	352	353	356	357	360	362	364	364	367	370	372	372	373	373	374	375	375	375	378	379	380	383	384	385	385	386	387	389	391	394	395	395	395	396	397	399	399	400
Bases:	tcagACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGTCCATCTCACACCGATTAATCTTTTTCCAACCAGAGCATGCGCCCCTGTTGGCTTATGCGGTATTAGCGGTCGTTTCCAACTGTTATCCCCCTGTGTGAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
Quality Scores:	35	35	35	35	35	35	35	35	33	31	31	31	33	35	35	35	35	35	35	35	35	35	35	35	35	35	23	20	20	31	31	33	33	33	35	23	17	17	21	20	20	20	31	31	33	35	35	35	35	35	33	33	33	35	31	31	31	35	35	35	35	35	35	35	31	31	31	33	35	35	35	35	35	35	35	35	35	35	31	31	31	26	26	26	26	35	35	35	35	35	35	35	33	31	31	31	35	35	35	35	35	35	35	35	35	35	35	35	35	35	31	31	31	35	35	35	33	33	33	33	33	35	35	35	35	35	35	35	35	35	35	35	35	35	35	35	35	35	30	26	26	26	30	33	35	35	35	35	35	35	35	35	33	33	33	35	33	27	27	25	25	25	27	14	14	14	14	14	25	25	34	34	35	35	35	32	33	33	32	35	35	32	25	25	15	20	20	20	28	35	33	33	33	33	35	35	35	35	35	35	35	35	35	35	35	35	35	35	35	29	24	24	24	29	35	35	35	35	33	33	31	31	34	34	34	34	34	34	31	20	20	20	20	20	31	34	31	31	31	31	32	31	31	33	34	25	25	20	20	18	25	28	28	22	20	22	28	28	28	30	30	29	29	29	30	25	25	25	29	29	26	26	25

>FLP3FBN01A55WZ
  Run Prefix:   R_2008_12_09_13_51_01_
  Region #:     1
  XY Location:  0363_0481

  Run Name:       R_2008_12_09_13_51_01_FLX04080350_Administrator_PC_Calrestricted
  Analysis Name:  D_2009_06_10_10_15_54_node176_fullProcessingAmplicons
  Full Path:      /srv/seq/454/2008_12_09/R_2008_12_09_13_51_01_FLX04080350_Administrator_PC_Calrestricted/D_2009_06_10_10_15_54_node176_fullProcessingAmplicons/

  Read Header Len:  32
  Name Length:      14
  # of Bases:       276
  Clip Qual Left:   5
  Clip Qual Right:  276
  Clip Adap Left:   0
  Clip Adap Right:  0

Flowgram:	1.00	0.00	1.01	0.02	0.01	1.04	0.04	1.05	0.01	1.16	0.79	1.82	0.90	0.11	0.09	0.98	0.11	1.02	0.09	0.97	0.97	0.07	0.13	0.96	1.05	0.14	1.92	0.05	0.12	1.12	0.17	0.03	1.00	0.16	0.06	0.93	0.15	0.03	0.92	0.11	1.00	0.01	0.16	0.94	0.16	0.00	1.97	0.14	0.94	0.02	3.19	0.97	1.07	1.01	0.36	2.12	0.15	1.10	0.12	1.14	2.76	0.12	0.08	2.04	0.22	0.78	1.74	1.14	0.93	0.06	0.16	1.09	0.99	0.01	1.01	0.15	1.05	0.03	1.01	0.11	0.14	0.91	0.17	0.90	1.94	0.12	1.97	0.13	0.12	2.04	0.22	0.07	0.97	0.16	0.16	1.01	1.06	0.14	0.11	4.82	0.10	1.01	1.77	0.25	1.95	0.09	1.85	0.16	0.95	0.02	1.05	0.12	1.03	0.07	1.15	0.08	0.08	1.01	0.20	0.89	0.15	1.92	3.80	0.12	1.05	0.97	0.12	0.10	0.92	0.09	2.04	0.18	0.08	1.02	0.14	0.10	1.07	0.12	1.09	1.04	0.10	2.00	0.14	0.99	0.14	0.93	0.95	0.12	1.06	0.92	0.11	2.07	1.07	0.12	0.10	3.06	0.12	0.08	2.07	0.93	2.17	1.04	3.75	1.12	0.12	0.06	1.83	0.18	1.00	1.02	1.08	0.14	1.01	1.03	0.20	0.14	0.94	0.16	1.33	0.06	1.05	2.20	0.21	0.09	0.86	0.08	0.16	2.10	0.06	2.06	1.14	0.78	0.11	0.13	0.97	0.02	0.12	1.12	0.04	0.07	1.07	0.10	3.77	0.12	0.07	1.08	0.08	0.10	0.98	0.12	1.17	0.11	2.06	1.04	0.14	0.16	0.76	1.04	2.14	2.18	0.95	2.92	2.05	0.06	2.92	1.94	0.12	0.13	0.77	0.09	1.11	0.09	0.14	1.11	0.13	0.10	1.08	0.08	0.18	1.08	0.14	1.04	0.16	0.80	0.12	3.60	0.21	0.06	1.06	0.00	0.12	1.00	0.09	0.14	1.04	0.03	1.09	0.09	1.00	0.12	0.16	0.99	0.88	0.12	1.03	0.16	0.85	0.21	0.15	1.07	0.08	0.11	1.12	0.16	0.12	1.21	0.13	0.90	0.13	0.02	1.08	0.93	1.96	0.13	0.06	0.87	0.13	0.15	1.12	0.02	1.03	0.11	2.06	0.11	0.12	1.08	0.87	1.01	0.13	0.05	2.01	1.99	0.14	0.12	1.12	0.00	1.05	0.17	1.02	0.11	2.97	0.13	3.08	0.15	0.63	0.17	0.08	0.89	0.06	1.03	0.17	1.80	0.15	1.79	0.15	0.09	1.07	0.00	1.07	1.00	0.00	0.16	1.09	0.02	2.04	0.10	0.09	1.86	0.05	0.99	0.20	0.60	0.14	0.99	1.12	0.75	0.17	0.00	0.99	1.92	0.18	1.98	0.14	0.00	1.09	0.00	0.15	1.11	0.13	1.89	2.10	0.13	0.18	1.79	0.11	1.02	0.19	0.13	1.11	0.75	1.14	0.91	0.04	0.12	0.82	1.01	2.03	1.01	1.12	0.06	0.87	0.14	1.07	0.15	0.11	1.06	2.79	1.08	1.11	0.03	0.14	0.94	0.13	0.11	1.04	0.79	0.23	0.14	2.21	2.05
Flow Indexes:	1	3	6	8	10	11	12	12	13	16	18	20	21	24	25	27	27	30	33	36	39	41	44	47	47	49	51	51	51	52	53	54	56	56	58	60	61	61	61	64	64	66	67	67	68	69	72	73	75	77	79	82	84	85	85	87	87	90	90	93	96	97	100	100	100	100	100	102	103	103	105	105	107	107	109	111	113	115	118	120	122	122	123	123	123	123	125	126	129	131	131	134	137	139	140	142	142	144	146	147	149	150	152	152	153	156	156	156	159	159	160	161	161	162	163	163	163	163	164	167	167	169	170	171	173	174	177	179	181	182	182	185	188	188	190	190	191	192	195	198	201	203	203	203	203	206	209	211	213	213	214	217	218	219	219	220	220	221	222	222	222	223	223	225	225	225	226	226	229	231	234	237	240	242	244	246	246	246	246	249	252	255	257	259	262	263	265	267	270	273	276	278	281	282	283	283	286	289	291	293	293	296	297	298	301	301	302	302	305	307	309	311	311	311	313	313	313	315	318	320	322	322	324	324	327	329	330	333	335	335	338	338	340	342	344	345	346	349	350	350	352	352	355	358	360	360	361	361	364	364	366	369	370	371	372	375	376	377	377	378	379	381	383	386	387	387	387	388	389	392	395	396	399	399	400	400
Bases:	tcagACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGAGAAAATGCTCACTCATGATACCATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCAGGTTGGATACGCGTTACTCACCCGTGCGCCGG
Quality Scores:	35	35	35	35	32	32	32	32	35	35	35	35	35	35	35	35	35	35	35	35	35	35	38	38	39	39	32	32	32	35	35	35	35	35	34	31	21	21	25	35	32	25	25	25	32	35	35	37	39	35	35	35	35	35	35	35	35	35	35	35	35	35	32	32	32	32	32	35	32	32	32	32	35	35	35	35	35	35	35	35	35	35	32	32	32	32	35	35	35	35	35	35	35	35	35	35	35	35	35	35	35	35	35	35	35	35	35	35	35	35	35	35	34	34	26	26	26	26	32	35	35	35	35	35	35	34	34	34	34	34	34	34	35	35	35	35	35	35	35	35	35	26	26	26	26	35	35	35	35	35	35	35	35	34	34	34	34	35	35	35	35	35	35	35	35	35	35	35	35	35	35	35	35	30	22	24	21	21	21	30	35	35	35	35	35	35	35	35	35	35	35	35	35	35	35	35	35	35	35	35	35	35	35	35	35	35	35	35	35	35	35	35	35	35	35	29	29	26	34	35	27	27	27	27	35	35	35	35	35	35	35	32	32	32	32	32	35	35	35	35	35	35	35	35	35	35	31	32	32	25	28	25	25	25	25	30	30	30	30	30	30	30	30	28	22	22	22	28	28	30	25	22	22	22	30	30

>FLP3FBN01D7O1S
  Run Prefix:   R_2008_12_09_13_51_01_
  Region #:     1
  XY Location:  1610_3070

  Run Name:       R_2008_12_09_13_51_01_FLX04080350_Administrator_PC_Calrestricted
  Analysis Name:  D_2009_06_10_10_15_54_node176_fullProcessingAmplicons
  Full Path:      /srv/seq/454/2008_12_09/R_2008_12_09_13_51_01_FLX04080350_Administrator_PC_Calrestricted/D_2009_06_10_10_15_54_node176_fullProcessingAmplicons/

  Read Header Len:  32
  Name Length:      14
  # of Bases:       287
  Clip Qual Left:   5
  Clip Qual Right:  283
  Clip Adap Left:   0
  Clip Adap Right:  0

Flowgram:	1.06	0.11	1.06	0.15	0.19	0.84	0.20	1.04	0.16	0.89	1.02	0.13	0.13	1.01	0.17	1.05	0.13	0.97	0.16	1.02	1.07	0.05	1.05	2.03	0.14	0.03	1.01	0.13	1.10	0.08	0.88	0.13	0.14	0.94	0.17	0.14	0.94	0.12	0.17	1.01	0.14	0.12	1.02	0.13	1.00	0.10	0.18	1.04	0.13	0.10	1.83	0.12	1.02	0.10	2.62	1.01	1.06	1.07	0.18	2.06	0.08	0.96	0.19	0.91	3.07	0.14	0.17	2.07	0.10	1.08	2.03	0.86	0.97	0.14	0.15	1.05	1.03	0.10	0.99	0.08	1.02	0.10	1.03	0.12	0.14	1.05	0.17	1.05	2.09	0.12	1.95	0.09	0.08	1.92	0.12	0.11	1.00	0.05	0.16	0.97	1.03	0.12	0.15	4.98	0.08	1.09	2.04	0.00	2.12	0.14	1.78	0.14	0.82	0.11	0.92	0.13	1.02	0.10	0.85	0.09	0.06	1.09	0.06	1.19	0.10	1.97	3.89	0.10	1.04	0.94	0.11	0.06	0.96	0.12	1.90	0.08	0.08	1.06	0.13	0.14	0.96	0.12	1.08	1.05	0.10	2.11	0.11	1.03	0.07	0.90	1.07	0.02	0.90	1.07	0.13	2.01	1.04	0.08	0.11	2.96	0.05	0.13	1.89	0.81	2.07	1.19	3.63	1.09	0.10	0.09	1.74	0.06	1.06	1.03	0.98	0.14	0.95	0.96	0.11	0.12	0.85	0.09	1.08	0.11	1.01	2.00	0.12	0.16	1.09	0.09	0.16	2.03	0.11	1.91	0.92	0.98	0.14	0.04	1.05	0.16	0.06	0.98	0.01	0.14	1.08	0.14	3.78	0.15	0.08	0.99	0.10	0.08	1.07	0.05	0.99	1.03	0.11	0.96	0.04	0.08	1.05	1.85	1.89	1.04	0.04	2.95	0.12	0.13	1.84	0.00	0.98	0.14	2.72	1.87	0.16	0.06	0.98	1.02	0.17	1.01	1.06	0.00	0.08	1.06	0.07	2.77	1.87	0.14	0.08	1.03	0.23	0.03	1.02	0.10	0.10	0.99	0.14	0.00	1.02	2.09	0.07	3.01	0.13	0.11	1.00	1.04	1.08	0.16	1.04	1.02	0.16	0.10	1.09	0.97	1.03	0.12	0.85	1.06	0.09	0.15	1.01	0.09	1.15	3.06	1.06	0.88	0.16	0.12	1.85	2.20	0.14	0.14	0.99	0.09	0.91	0.15	2.73	0.13	1.06	0.08	2.94	0.00	0.97	0.93	0.19	2.82	0.20	1.96	0.05	0.06	1.07	0.17	0.89	1.08	0.13	0.06	1.14	0.04	3.51	1.01	0.04	0.85	0.26	1.01	1.76	1.05	0.07	0.03	1.10	0.11	0.98	2.06	0.15	0.14	1.10	0.05	0.00	1.07	0.17	1.80	2.00	0.13	0.16	2.24	0.11	0.95	0.12	0.04	1.03	1.01	1.14	0.92	0.93	0.04	0.16	1.19	2.10	1.05	0.87	0.17	1.05	0.17	1.02	0.09	0.07	0.89	2.79	0.89	1.11	0.15	0.22	1.05	0.17	0.12	0.86	1.03	0.07	0.08	1.83	1.88	1.05	0.08	0.92	1.08	0.15	0.17	1.85	0.17	0.07	0.89	0.07	0.08	1.09	0.06	0.90	0.16	0.29	3.32	1.12	0.22
Flow Indexes:	1	3	6	8	10	11	14	16	18	20	21	23	24	24	27	29	31	34	37	40	43	45	48	51	51	53	55	55	55	56	57	58	60	60	62	64	65	65	65	68	68	70	71	71	72	73	76	77	79	81	83	86	88	89	89	91	91	94	94	97	100	101	104	104	104	104	104	106	107	107	109	109	111	111	113	115	117	119	122	124	126	126	127	127	127	127	129	130	133	135	135	138	141	143	144	146	146	148	150	151	153	154	156	156	157	160	160	160	163	163	164	165	165	166	167	167	167	167	168	171	171	173	174	175	177	178	181	183	185	186	186	189	192	192	194	194	195	196	199	202	205	207	207	207	207	210	213	215	216	218	221	222	222	223	223	224	226	226	226	229	229	231	233	233	233	234	234	237	238	240	241	244	246	246	246	247	247	250	253	256	259	260	260	262	262	262	265	266	267	269	270	273	274	275	277	278	281	283	284	284	284	285	286	289	289	290	290	293	295	297	297	297	299	301	301	301	303	304	306	306	306	308	308	311	313	314	317	319	319	319	319	320	322	324	325	325	326	329	331	332	332	335	338	340	340	341	341	344	344	346	349	350	351	352	353	356	357	357	358	359	361	363	366	367	367	367	368	369	372	375	376	379	379	380	380	381	383	384	387	387	390	393	395	398	398	398	399
Bases:	tcagACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGATAACCGAAATTCTTTAATAGTGAAACCATGCGGAAATACTATACTATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTTATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCATCaaac
Quality Scores:	37	36	36	36	36	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	38	37	36	40	28	28	28	33	40	38	37	37	39	37	40	37	38	38	38	40	40	40	40	40	40	40	40	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	36	36	33	33	33	36	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	36	35	35	37	31	24	21	21	21	21	23	32	36	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	36	28	28	28	28	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	36	28	28	28	33	33	37	37	37	37	37	28	28	28	33	33	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	36	33	33	33	33	37	36	28	28	26	36	37	37	37	37	36	31	31	26	31	31	36	36	30	17	16	16	19	19	19	26	30	30	32	33	36	36	36	36	36	33	33	33	36	36	33	33	28	31	31	31	31	31	28	28	28	31	31	31	31	25	22	22	22	25	28	28	28	31	28	28	28	28	31	29	28	28	28	29	25	16	16	16	16	15

>FLP3FBN01CV5ZU
  Run Prefix:   R_2008_12_09_13_51_01_
  Region #:     1
  XY Location:  1069_1480

  Run Name:       R_2008_12_09_13_51_01_FLX04080350_Administrator_PC_Calrestricted
  Analysis Name:  D_2009_06_10_10_15_54_node176_fullProcessingAmplicons
  Full Path:      /srv/seq/454/2008_12_09/R_2008_12_09_13_51_01_FLX04080350_Administrator_PC_Calrestricted/D_2009_06_10_10_15_54_node176_fullProcessingAmplicons/

  Read Header Len:  32
  Name Length:      14
  # of Bases:       278
  Clip Qual Left:   5
  Clip Qual Right:  278
  Clip Adap Left:   0
  Clip Adap Right:  0

Flowgram:	1.00	0.04	1.04	0.07	0.08	1.01	0.10	1.05	0.07	1.07	1.03	0.06	0.04	1.04	0.08	1.00	0.06	1.07	0.07	1.00	1.02	0.02	0.99	2.06	0.07	0.02	1.01	0.10	1.00	0.03	1.01	0.08	0.09	1.03	0.10	0.07	0.94	0.06	0.08	1.01	0.09	0.04	1.04	0.10	1.01	0.04	0.14	0.99	0.12	0.03	2.02	0.12	1.00	0.06	3.03	1.02	1.04	1.03	0.16	2.04	0.15	0.99	0.09	1.01	3.03	0.10	0.10	2.02	1.01	0.03	2.07	1.00	1.02	0.04	0.13	1.02	0.99	0.03	1.01	0.10	0.99	0.06	1.00	0.13	0.11	1.02	0.14	1.01	1.03	1.05	2.02	0.10	0.13	1.99	0.14	0.13	1.01	0.10	0.10	1.07	0.99	0.14	0.15	6.14	2.04	2.11	2.02	0.07	0.98	0.13	0.95	0.09	0.90	0.09	0.87	0.13	0.12	1.00	0.10	1.08	1.15	0.15	4.79	0.15	1.00	1.04	0.07	0.10	1.03	0.12	0.08	0.79	0.97	1.03	0.08	0.12	1.01	0.08	1.01	1.06	0.96	0.03	1.04	1.04	0.17	0.04	2.07	0.07	2.01	0.15	0.08	2.06	0.91	2.16	0.09	0.96	0.09	0.11	2.14	1.01	2.19	1.12	2.07	0.12	1.98	1.10	2.03	0.05	0.06	2.07	2.00	0.12	0.09	0.94	0.08	1.06	0.12	0.17	0.93	0.13	1.05	2.02	0.15	0.10	1.03	0.98	1.02	0.13	0.15	1.94	1.03	1.08	0.09	0.08	1.05	0.08	0.11	0.95	0.11	0.10	0.98	0.07	0.06	1.00	0.16	0.09	2.91	0.13	0.14	1.01	0.16	0.07	1.07	0.08	1.02	0.11	0.93	0.02	0.07	1.04	1.02	2.00	1.89	0.98	0.08	0.01	1.91	0.97	0.06	2.97	1.02	0.09	2.92	0.05	1.05	0.10	0.04	2.03	2.00	0.08	0.10	0.97	1.02	0.13	0.06	2.01	0.13	1.06	0.06	0.97	0.16	1.07	0.12	0.99	0.14	0.15	1.03	0.07	0.10	0.92	0.15	0.07	2.06	0.10	1.06	0.05	1.07	0.09	0.97	0.12	1.02	0.09	0.14	1.02	0.11	0.14	0.96	1.04	0.14	1.09	1.04	0.15	0.10	0.94	2.01	0.95	0.10	0.17	1.05	0.06	0.17	1.03	0.19	0.18	1.12	2.05	1.11	1.00	0.11	0.17	1.92	0.95	0.06	0.96	1.03	1.04	2.04	1.08	0.06	0.87	0.11	0.03	3.05	0.10	1.02	0.11	1.06	0.15	0.99	0.12	0.20	1.05	0.13	0.97	1.08	0.10	0.17	1.01	1.97	1.02	0.21	0.09	1.03	0.06	4.99	0.12	1.03	0.18	0.14	1.05	2.05	0.97	1.10	0.05	0.06	0.80	0.12	2.06	1.14	1.00	0.14	1.90	2.00	0.07	0.16	1.02	0.09	0.17	1.12	0.10	0.19	0.85	0.11	0.10	1.15	0.97	0.93	1.00	0.04	0.15	1.05	1.01	2.10	1.06	1.00	1.01	0.10	0.13	1.01	0.05	0.10	0.95	3.02	1.08	1.02	0.03	0.15	0.97	0.10	0.07	1.01	0.99	0.10	0.05	1.99	1.92	0.99	0.05	1.21	0.93
Flow Indexes:	1	3	6	8	10	11	14	16	18	20	21	23	24	24	27	29	31	34	37	40	43	45	48	51	51	53	55	55	55	56	57	58	60	60	62	64	65	65	65	68	68	69	71	71	72	73	76	77	79	81	83	86	88	89	90	91	91	94	94	97	100	101	104	104	104	104	104	104	105	105	106	106	107	107	109	111	113	115	118	120	121	123	123	123	123	123	125	126	129	132	133	134	137	139	140	141	143	144	147	147	149	149	152	152	153	154	154	156	159	159	160	161	161	162	163	163	165	165	166	167	167	170	170	171	171	174	176	179	181	182	182	185	186	187	190	190	191	192	195	198	201	204	207	207	207	210	213	215	217	220	221	222	222	223	223	224	227	227	228	230	230	230	231	233	233	233	235	238	238	239	239	242	243	246	246	248	250	252	254	257	260	263	263	265	267	269	271	274	277	278	280	281	284	285	285	286	289	292	295	296	296	297	298	301	301	302	304	305	306	307	307	308	310	313	313	313	315	317	319	322	324	325	328	329	329	330	333	335	335	335	335	335	337	340	341	341	342	343	346	348	348	349	350	352	352	353	353	356	359	362	365	366	367	368	371	372	373	373	374	375	376	379	382	383	383	383	384	385	388	391	392	395	395	396	396	397	399	400
Bases:	tcagACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAACCAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCTCTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCG
Quality Scores:	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	37	37	37	37	37	37	37	37	37	37	37	37	37	31	31	31	31	31	31	37	37	37	37	37	37	37	37	37	37	37	38	38	33	33	33	33	33	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	38	38	38	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	38	40	40	40	40	40	40	40	40	40	40	38	38	38	38	40	38	38	38	38	38	38	38	38	38	38	38	38	38	30	30	31	32	32	32	32	32	32	32	32	32	32	32	32	32	32	32	32	32	32	32	32	32	32	32	32	28	28

>FLP3FBN01DQ783
  Run Prefix:   R_2008_12_09_13_51_01_
  Region #:     1
  XY Location:  1423_0757

  Run Name:       R_2008_12_09_13_51_01_FLX04080350_Administrator_PC_Calrestricted
  Analysis Name:  D_2009_06_10_10_15_54_node176_fullProcessingAmplicons
  Full Path:      /srv/seq/454/2008_12_09/R_2008_12_09_13_51_01_FLX04080350_Administrator_PC_Calrestricted/D_2009_06_10_10_15_54_node176_fullProcessingAmplicons/

  Read Header Len:  32
  Name Length:      14
  # of Bases:       268
  Clip Qual Left:   5
  Clip Qual Right:  268
  Clip Adap Left:   0
  Clip Adap Right:  0

Flowgram:	1.00	0.03	1.01	0.08	0.06	1.04	0.07	0.93	0.06	1.07	1.01	0.07	0.03	1.02	0.06	0.97	0.06	1.05	0.06	1.02	1.00	0.02	0.93	1.96	0.08	0.02	1.03	0.09	1.01	0.03	1.00	0.07	0.07	1.09	0.09	0.07	1.00	0.04	0.06	1.01	0.08	0.02	1.02	0.11	0.97	0.03	0.12	1.01	0.08	0.02	1.96	0.11	1.03	0.04	2.97	0.98	1.02	1.07	0.14	1.97	0.15	1.01	0.07	1.03	3.01	0.09	0.09	1.94	0.10	1.03	2.00	0.99	0.95	0.06	0.16	0.98	1.02	0.02	1.05	0.11	1.01	0.04	1.03	0.09	0.14	1.03	0.14	1.01	2.03	0.12	1.97	0.11	0.15	2.05	0.13	0.10	1.03	0.12	0.07	1.00	1.01	0.05	0.12	4.89	0.13	1.04	2.07	0.06	2.05	0.08	1.94	0.09	1.02	0.02	0.98	0.08	1.00	0.07	1.02	0.11	0.10	1.01	0.09	1.04	0.10	2.02	4.02	0.11	1.04	1.01	0.16	0.06	1.00	0.07	2.01	0.09	0.13	1.03	0.11	0.10	1.04	0.11	1.03	1.01	0.12	1.96	0.12	0.97	0.11	0.97	1.00	0.14	1.08	1.04	0.15	1.99	1.02	0.12	0.16	2.92	0.14	0.07	2.13	1.01	2.05	1.06	3.84	1.00	0.09	0.07	1.92	0.11	0.98	1.05	0.99	0.10	0.99	1.03	0.13	0.09	1.03	0.13	1.04	0.08	1.02	2.06	0.09	0.15	1.03	0.06	0.12	1.99	0.07	1.85	1.02	1.00	0.07	0.09	1.02	0.09	0.10	0.97	0.06	0.09	1.08	0.11	3.94	0.09	0.11	1.02	0.12	0.11	1.04	0.03	0.97	1.05	0.98	0.11	1.04	0.07	1.07	1.01	2.04	1.99	0.14	2.00	0.07	0.09	1.02	0.92	2.03	0.13	2.99	2.05	0.19	0.08	1.01	0.11	1.02	0.08	0.17	1.03	0.11	0.13	0.99	0.07	0.10	0.98	1.05	0.02	0.13	1.00	0.10	1.91	1.10	0.12	0.14	0.93	0.10	0.15	1.03	0.06	0.16	1.05	0.07	0.09	0.97	1.94	0.14	1.06	0.94	0.10	1.08	0.11	1.04	0.12	0.14	0.99	0.09	0.14	0.95	0.09	0.12	1.12	0.14	0.95	0.10	0.17	1.03	0.06	0.06	1.01	0.09	0.13	2.05	0.09	0.17	1.06	0.17	0.12	0.94	0.05	1.02	0.14	2.14	0.14	0.11	1.16	1.01	0.92	0.14	0.06	2.17	2.13	0.12	0.13	0.97	0.06	1.01	0.10	2.03	0.11	1.98	0.09	3.08	0.08	1.07	0.16	0.10	0.84	0.09	1.05	0.07	1.99	0.17	1.98	0.07	0.07	0.99	0.09	1.09	0.13	0.06	1.04	1.01	0.03	2.06	0.16	0.10	1.98	0.12	1.04	0.05	1.01	0.16	0.94	1.10	0.92	0.18	1.00	0.06	0.92	1.17	1.97	0.14	0.15	0.98	0.04	0.04	0.98	0.07	1.95	2.19	0.16	0.14	2.02	0.15	0.87	0.17	0.07	1.04	0.98	1.25	0.98	1.04	0.11	0.14	1.04	1.87	1.05	1.09	0.10	1.03	0.15	0.81	0.07	0.09	0.97	3.06	1.25
Flow Indexes:	1	3	6	8	10	11	14	16	18	20	21	23	24	24	27	29	31	34	37	40	43	45	48	51	51	53	55	55	55	56	57	58	60	60	62	64	65	65	65	68	68	70	71	71	72	73	76	77	79	81	83	86	88	89	89	91	91	94	94	97	100	101	104	104	104	104	104	106	107	107	109	109	111	111	113	115	117	119	122	124	126	126	127	127	127	127	129	130	133	135	135	138	141	143	144	146	146	148	150	151	153	154	156	156	157	160	160	160	163	163	164	165	165	166	167	167	167	167	168	171	171	173	174	175	177	178	181	183	185	186	186	189	192	192	194	194	195	196	199	202	205	207	207	207	207	210	213	215	216	217	219	221	222	223	223	224	224	226	226	229	230	231	231	233	233	233	234	234	237	239	242	245	248	249	252	254	254	255	258	261	264	267	268	268	270	271	273	275	278	281	284	286	289	292	295	295	298	301	303	305	305	308	309	310	313	313	314	314	317	319	321	321	323	323	325	325	325	327	330	332	334	334	336	336	339	341	344	345	347	347	350	350	352	354	356	357	358	360	362	363	364	364	367	370	372	372	373	373	376	376	378	381	382	383	384	385	388	389	389	390	391	393	395	398	399	399	399	400
Bases:	tcagACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
Quality Scores:	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	38	38	38	38	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	38	38	38	37	37	38	38	38	38	37	37	37	37	37	37	37	37	37	37	37	37	40	40	40	40	40	40	40	40	38	38	38	38	40	38	38	38	38	38	38	38	38	38	36	38	37	37	36	36	36	30	31	31	31	31	31	31	31	31	31	32	31	31	31	31	31	25	25	25

>FLP3FBN01DDPFF
  Run Prefix:   R_2008_12_09_13_51_01_
  Region #:     1
  XY Location:  1269_0617

  Run Name:       R_2008_12_09_13_51_01_FLX04080350_Administrator_PC_Calrestricted
  Analysis Name:  D_2009_06_10_10_15_54_node176_fullProcessingAmplicons
  Full Path:      /srv/seq/454/2008_12_09/R_2008_12_09_13_51_01_FLX04080350_Administrator_PC_Calrestricted/D_2009_06_10_10_15_54_node176_fullProcessingAmplicons/

  Read Header Len:  32
  Name Length:      14
  # of Bases:       270
  Clip Qual Left:   5
  Clip Qual Right:  263
  Clip Adap Left:   0
  Clip Adap Right:  0

Flowgram:	1.02	0.02	1.03	0.08	0.05	1.00	0.06	1.03	0.05	1.05	1.00	0.05	0.02	1.03	0.06	1.00	0.04	1.07	0.03	1.01	1.00	0.00	0.92	2.04	0.04	0.01	0.95	0.08	1.06	0.00	0.97	0.08	0.05	1.06	0.08	0.07	0.98	0.03	0.07	0.99	0.07	0.01	1.04	0.11	0.99	0.01	0.12	0.99	0.10	0.02	1.98	0.13	1.03	0.04	2.96	1.01	1.02	1.05	0.16	2.02	0.15	1.02	0.06	1.03	3.10	0.10	0.11	2.04	0.09	1.01	2.03	0.95	1.03	0.04	0.11	1.01	1.04	0.01	1.07	0.10	1.02	0.05	0.96	0.11	0.13	0.99	0.13	0.99	2.03	0.09	1.97	0.10	0.13	1.97	0.10	0.12	1.06	0.09	0.08	1.06	0.98	0.07	0.14	4.80	0.14	1.00	2.05	0.05	2.03	0.05	1.86	0.10	1.02	0.02	1.01	0.07	1.02	0.07	0.99	0.14	0.10	1.06	0.10	1.03	0.09	2.05	4.03	0.12	1.06	0.92	0.12	0.06	0.97	0.10	2.06	0.04	0.12	1.08	0.12	0.12	1.01	0.10	1.06	1.04	0.08	2.10	0.14	0.94	0.10	1.01	1.03	0.16	1.01	1.04	0.14	2.03	0.92	0.11	0.11	2.88	0.17	0.09	2.10	1.06	1.97	1.05	3.84	1.04	0.08	0.13	1.99	0.15	1.06	1.03	1.07	0.09	0.98	1.00	0.16	0.07	0.86	0.13	1.00	0.08	1.07	2.07	0.15	0.12	0.99	0.13	0.15	2.04	0.17	2.02	1.04	0.95	0.09	0.12	1.03	0.19	0.08	0.98	0.07	0.07	1.02	0.09	4.00	0.12	0.12	1.02	0.16	0.13	1.03	0.15	1.02	1.05	1.05	0.10	1.01	0.20	1.00	1.03	2.12	2.02	0.17	2.06	0.07	0.04	1.02	0.99	2.06	0.13	3.02	2.02	0.18	0.07	1.02	0.10	1.04	0.09	0.13	1.02	0.07	0.12	1.05	0.04	0.11	0.99	1.02	0.15	0.12	0.97	0.14	1.95	1.09	0.07	0.13	0.89	0.08	0.15	1.04	0.05	0.14	1.04	0.06	0.12	0.99	1.98	0.12	1.08	1.00	0.11	1.01	0.10	1.06	0.13	0.13	0.96	0.05	0.15	0.99	0.11	0.11	1.08	0.15	0.93	0.08	0.17	1.00	0.16	0.07	1.05	0.06	0.12	2.08	0.07	0.14	1.05	0.14	0.13	1.11	0.05	0.95	0.16	2.05	0.17	0.19	1.03	0.94	0.94	0.12	0.09	1.94	2.07	0.17	0.07	1.06	0.05	1.09	0.05	1.01	0.09	2.90	0.10	2.76	0.14	0.94	0.15	0.08	0.99	0.04	1.03	0.15	1.91	0.12	1.84	0.16	0.14	1.08	0.07	1.14	0.97	0.03	0.15	0.93	0.02	2.03	0.14	0.05	1.99	0.08	0.91	0.16	1.12	0.11	1.04	1.15	0.93	0.09	0.04	0.92	2.11	0.15	2.00	0.10	0.14	1.00	0.05	0.03	0.95	0.09	1.89	1.88	0.14	0.17	2.74	1.11	1.03	1.11	1.01	0.02	0.10	0.92	1.05	1.96	1.10	1.01	0.11	0.98	0.13	0.86	0.22	0.16	0.95	2.88	0.92	1.02	0.00	0.23	1.26
Flow Indexes:	1	3	6	8	10	11	14	16	18	20	21	23	24	24	27	29	31	34	37	40	43	45	48	51	51	53	55	55	55	56	57	58	60	60	62	64	65	65	65	68	68	70	71	71	72	73	76	77	79	81	83	86	88	89	89	91	91	94	94	97	100	101	104	104	104	104	104	106	107	107	109	109	111	111	113	115	117	119	122	124	126	126	127	127	127	127	129	130	133	135	135	138	141	143	144	146	146	148	150	151	153	154	156	156	157	160	160	160	163	163	164	165	165	166	167	167	167	167	168	171	171	173	174	175	177	178	181	183	185	186	186	189	192	192	194	194	195	196	199	202	205	207	207	207	207	210	213	215	216	217	219	221	222	223	223	224	224	226	226	229	230	231	231	233	233	233	234	234	237	239	242	245	248	249	252	254	254	255	258	261	264	267	268	268	270	271	273	275	278	281	284	286	289	292	295	295	298	301	303	305	305	308	309	310	313	313	314	314	317	319	321	323	323	323	325	325	325	327	330	332	334	334	336	336	339	341	342	345	347	347	350	350	352	354	356	357	358	361	362	362	364	364	367	370	372	372	373	373	376	376	376	377	378	379	380	383	384	385	385	386	387	389	391	394	395	395	395	396	397	400
Bases:	tcagACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCAGGTTGGGTACGCGTTACTCacccgtg
Quality Scores:	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	37	37	37	37	37	37	37	37	37	37	37	37	37	34	33	33	33	33	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	38	38	38	38	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	38	38	31	31	31	38	37	37	36	36	33	33	33	36	37	36	38	38	38	37	38	38	38	38	38	38	36	36	36	36	38	36	36	36	28	28	29	21	21	22	28	32	31	31	31	31	31	32	31	31	28	28	28	31	31	31	30	30

>FLP3FBN01CPD70
  Run Prefix:   R_2008_12_09_13_51_01_
  Region #:     1
  XY Location:  0992_0942

  Run Name:       R_2008_12_09_13_51_01_FLX04080350_Administrator_PC_Calrestricted
  Analysis Name:  D_2009_06_10_10_15_54_node176_fullProcessingAmplicons
  Full Path:      /srv/seq/454/2008_12_09/R_2008_12_09_13_51_01_FLX04080350_Administrator_PC_Calrestricted/D_2009_06_10_10_15_54_node176_fullProcessingAmplicons/

  Read Header Len:  32
  Name Length:      14
  # of Bases:       280
  Clip Qual Left:   5
  Clip Qual Right:  278
  Clip Adap Left:   0
  Clip Adap Right:  0

Flowgram:	1.02	0.01	1.01	0.07	0.04	1.03	0.09	1.04	0.05	1.10	0.92	0.08	0.06	1.02	0.08	0.99	0.04	1.05	0.07	0.93	1.01	0.04	1.02	1.98	0.09	0.05	1.02	0.07	1.01	0.04	0.99	0.06	0.07	1.07	0.09	0.07	0.99	0.06	0.09	0.99	0.09	0.02	1.00	0.10	0.91	0.02	0.16	1.00	0.10	0.02	2.06	0.12	1.09	0.07	2.96	1.06	1.01	1.08	0.14	2.10	0.12	1.01	0.10	0.98	0.91	0.08	1.01	0.14	1.01	0.05	0.16	2.03	0.13	1.02	2.04	1.06	1.03	0.06	0.17	0.96	1.01	0.03	1.03	0.12	1.03	0.03	1.00	0.09	0.14	1.04	0.15	1.01	2.08	0.10	1.95	0.13	0.14	1.93	0.12	0.10	1.01	0.05	0.10	1.07	1.01	0.04	0.14	5.81	0.08	0.02	1.98	0.07	2.02	0.03	2.02	0.12	0.94	0.04	0.97	0.08	1.05	0.07	1.02	0.10	0.12	1.08	0.15	1.02	0.09	1.96	3.98	0.14	1.06	0.91	0.12	0.08	0.97	0.10	2.04	0.09	0.14	0.98	0.11	0.13	1.06	0.12	1.03	1.05	0.12	1.93	0.12	2.04	0.13	0.02	1.02	0.12	2.07	0.04	0.17	2.11	1.03	0.04	0.10	3.00	0.13	0.06	2.08	0.90	2.13	1.02	3.61	1.04	0.03	0.13	1.93	0.10	0.07	2.01	1.06	0.13	0.10	1.89	2.06	0.11	1.03	2.00	0.14	0.13	0.97	0.05	0.14	1.99	0.06	1.94	1.03	0.97	0.08	0.14	0.96	0.11	0.10	1.01	0.09	0.10	1.08	0.14	3.87	0.11	0.10	1.00	0.14	0.14	1.03	0.12	1.01	1.13	0.14	1.01	0.14	0.11	1.02	0.11	0.11	0.94	0.07	1.00	1.99	1.03	0.08	2.10	0.15	0.98	2.12	0.02	1.04	0.08	3.01	2.07	0.15	0.15	0.88	1.02	0.19	1.13	2.06	0.12	1.00	0.09	1.08	1.03	1.98	0.11	0.13	1.00	0.11	0.14	1.00	0.08	0.16	0.96	0.06	0.05	1.03	2.03	0.12	2.02	0.13	1.02	0.14	1.84	1.06	0.05	1.11	0.88	0.13	0.17	0.92	0.06	0.06	1.08	0.08	0.14	2.12	0.09	0.10	1.01	0.12	0.09	0.97	0.07	1.06	3.06	1.07	0.97	0.10	0.05	2.06	2.04	0.15	0.09	0.95	0.05	1.00	0.16	2.99	0.14	1.06	0.15	2.91	0.12	0.99	1.00	0.11	2.96	0.19	1.93	0.11	0.13	1.06	0.06	0.98	1.00	0.03	0.07	1.04	0.11	4.04	1.05	0.15	0.95	0.08	0.90	1.11	0.11	1.01	0.11	0.14	1.01	0.20	0.04	1.09	0.05	1.07	2.00	0.11	0.14	0.99	0.01	0.05	1.03	0.10	1.99	2.10	0.13	0.17	1.87	0.12	0.95	0.17	0.13	1.06	1.02	1.08	1.09	1.01	0.09	0.10	0.86	2.06	1.06	1.17	0.08	0.88	0.08	0.80	0.07	0.12	1.07	3.08	1.10	1.05	0.02	0.08	1.01	0.06	0.09	1.02	0.91	0.05	0.06	2.13	2.01	1.17	0.07	0.95	0.86	0.04	0.12	1.93	0.06
Flow Indexes:	1	3	6	8	10	11	14	16	18	20	21	23	24	24	27	29	31	34	37	40	43	45	48	51	51	53	55	55	55	56	57	58	60	60	62	64	65	67	69	72	72	74	75	75	76	77	80	81	83	85	87	90	92	93	93	95	95	98	98	101	104	105	108	108	108	108	108	108	111	111	113	113	115	115	117	119	121	123	126	128	130	130	131	131	131	131	133	134	137	139	139	142	145	147	148	150	150	152	152	155	157	157	160	160	161	164	164	164	167	167	168	169	169	170	171	171	171	171	172	175	175	178	178	179	182	182	183	183	185	186	186	189	192	192	194	194	195	196	199	202	205	207	207	207	207	210	213	215	216	218	221	224	226	227	227	228	230	230	232	233	233	235	237	237	237	238	238	241	242	244	245	245	247	249	250	251	251	254	257	260	263	264	264	266	266	268	270	270	271	273	274	277	280	283	283	286	289	291	292	292	292	293	294	297	297	298	298	301	303	305	305	305	307	309	309	309	311	312	314	314	314	316	316	319	321	322	325	327	327	327	327	328	330	332	333	335	338	341	343	344	344	347	350	352	352	353	353	356	356	358	361	362	363	364	365	368	369	369	370	371	373	375	378	379	379	379	380	381	384	387	388	391	391	392	392	393	395	396	399	399
Bases:	tcagACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAATGTGGGGGGCCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGGGCCGTTACCCCGCCAACAACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGAAGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCGcc
Quality Scores:	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	37	37	37	37	37	37	37	37	37	37	37	37	37	32	32	32	32	32	32	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	34	34	28	28	28	26	33	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	38	38	38	38	40	40	40	40	40	40	40	38	38	38	38	38	38	38	40	40	40	40	40	40	40	38	38	38	38	31	31	32	32	32	31	31	31	31	31	30	30	30	31	31	32	32	32	32	32	32	32	32	32	32	32	31	31	31	31	31

>FLP3FBN01BBAE6
  Run Prefix:   R_2008_12_09_13_51_01_
  Region #:     1
  XY Location:  0421_2032

  Run Name:       R_2008_12_09_13_51_01_FLX04080350_Administrator_PC_Calrestricted
  Analysis Name:  D_2009_06_10_10_15_54_node176_fullProcessingAmplicons
  Full Path:      /srv/seq/454/2008_12_09/R_2008_12_09_13_51_01_FLX04080350_Administrator_PC_Calrestricted/D_2009_06_10_10_15_54_node176_fullProcessingAmplicons/

  Read Header Len:  32
  Name Length:      14
  # of Bases:       273
  Clip Qual Left:   5
  Clip Qual Right:  273
  Clip Adap Left:   0
  Clip Adap Right:  0

Flowgram:	1.02	0.03	1.02	0.05	0.04	1.00	0.08	1.03	0.05	1.13	0.20	1.29	0.17	0.02	0.86	0.15	0.05	1.15	0.06	1.09	0.16	0.04	0.73	0.19	0.18	0.94	1.33	0.04	1.82	0.19	0.04	0.81	1.17	0.00	0.89	0.19	0.05	0.90	0.22	0.05	1.08	0.04	0.08	1.09	0.06	0.05	1.35	0.07	1.18	0.04	0.46	0.88	0.16	0.01	2.11	0.18	0.91	0.00	3.19	1.00	1.05	1.04	0.14	2.12	0.04	1.03	0.11	0.98	1.05	0.06	0.92	0.07	0.92	0.05	0.14	2.11	0.99	0.06	2.02	1.05	0.97	0.04	0.12	1.02	1.06	0.01	1.02	0.08	0.98	0.08	0.90	0.10	0.11	0.99	0.18	1.01	1.04	1.06	2.08	0.13	0.08	1.07	0.16	0.96	1.04	0.12	0.07	0.97	1.05	0.10	0.10	5.03	0.15	1.03	1.95	0.08	1.92	0.08	1.83	0.11	1.02	0.06	1.03	0.07	1.03	0.08	1.05	0.10	0.13	1.04	0.09	0.94	0.12	2.02	4.08	0.13	0.93	0.96	1.03	1.01	0.10	0.16	1.07	0.13	0.10	1.03	0.18	0.11	1.02	0.11	0.95	1.04	0.93	0.05	1.04	0.87	0.10	0.06	1.97	0.06	0.99	0.06	1.07	1.98	0.94	0.06	0.15	2.73	0.08	0.09	1.84	1.03	1.92	1.01	3.62	1.01	0.08	0.13	1.83	1.07	0.13	1.00	1.04	0.10	1.04	0.88	0.19	1.10	0.16	0.14	1.02	0.09	0.88	1.86	0.13	0.15	0.95	0.04	0.13	1.02	0.09	0.05	1.02	1.00	0.11	0.05	1.93	0.98	0.12	0.12	0.91	0.12	0.15	0.99	0.15	0.10	1.07	0.13	0.09	1.03	0.14	0.09	2.91	0.15	0.15	1.02	0.06	0.07	1.05	0.12	1.86	1.12	0.11	0.11	1.93	0.04	0.15	0.91	1.97	2.01	1.11	1.83	0.09	0.11	0.77	0.04	3.01	0.11	3.04	0.10	0.20	1.86	0.08	0.09	1.12	2.08	0.17	0.14	1.08	0.10	0.09	1.03	2.03	2.99	0.13	0.85	0.20	0.10	1.02	0.08	0.13	1.03	0.12	0.12	4.02	1.11	0.17	1.02	0.16	0.17	0.93	0.14	0.15	1.03	0.09	0.08	2.07	0.97	0.09	0.06	0.92	0.96	1.03	0.14	1.05	0.05	0.06	1.02	1.04	1.02	0.06	0.12	1.14	1.93	0.92	0.97	0.18	0.17	1.81	0.92	0.14	0.94	0.10	0.71	1.15	1.91	0.06	1.85	0.09	0.07	2.72	0.04	0.96	0.09	1.75	0.12	1.95	1.10	0.00	0.15	1.05	0.08	2.08	1.11	0.15	0.13	0.75	0.17	4.94	0.16	0.94	0.09	0.13	1.02	0.97	0.08	0.17	1.86	0.09	0.07	1.05	2.84	0.12	0.14	0.98	0.01	0.07	1.08	0.23	1.90	2.02	0.07	0.17	0.90	0.04	0.13	1.16	0.08	0.17	1.01	0.12	0.11	1.07	0.92	1.12	0.99	0.83	0.12	0.19	1.05	1.96	1.04	1.09	0.08	0.97	0.10	1.04	0.09	0.14	0.98	2.90	0.99	1.04	0.15	0.14	0.89	0.04	0.08	1.39	1.37
Flow Indexes:	1	3	6	8	10	12	15	18	20	23	26	27	29	29	32	33	35	38	41	44	47	49	52	55	55	57	59	59	59	60	61	62	64	64	66	68	69	71	73	76	76	77	79	79	80	81	84	85	87	89	91	94	96	97	98	99	99	102	104	105	108	109	112	112	112	112	112	114	115	115	117	117	119	119	121	123	125	127	130	132	134	134	135	135	135	135	137	138	139	140	143	146	149	151	152	153	155	156	159	159	161	163	164	164	165	168	168	168	171	171	172	173	173	174	175	175	175	175	176	179	179	180	182	183	185	186	188	191	193	194	194	197	200	203	204	207	207	208	211	214	217	220	223	223	223	226	229	231	231	232	235	235	238	239	239	240	240	241	242	242	245	247	247	247	249	249	249	252	252	255	256	256	259	262	263	263	264	264	264	266	269	272	275	275	275	275	276	278	281	284	287	287	288	291	292	293	295	298	299	300	303	304	304	305	306	309	309	310	312	314	315	316	316	318	318	321	321	321	323	325	325	327	327	328	331	333	333	334	337	339	339	339	339	339	341	344	345	348	348	351	352	352	352	355	358	360	360	361	361	364	367	370	373	374	375	376	377	380	381	381	382	383	385	387	390	391	391	391	392	393	396	399	400
Bases:	tcagAGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCGGCACCGGGATGCCCCGATGCCGCGTCACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTGGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCG
Quality Scores:	37	37	37	36	35	35	35	36	36	32	32	32	34	34	35	35	37	37	37	36	34	32	34	36	37	37	33	33	33	37	37	38	38	38	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	36	28	26	26	35	35	37	37	35	31	21	21	21	26	32	35	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	36	35	35	35	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	36	36	36	36	35	32	32	32	37	36	36	26	26	26	36	36	33	33	33	36	36	36	33	33	33	33	33	36	36	36	36	36	36	36	36	36	33	33	33	36	33	33	33	36	36	31	31	31	31	31	31	31	31	31	31	31	31	31	31	31	31	31	31	31	31	28	22	20	19

>FLP3FBN01AWYZD
  Run Prefix:   R_2008_12_09_13_51_01_
  Region #:     1
  XY Location:  0258_1671

  Run Name:       R_2008_12_09_13_51_01_FLX04080350_Administrator_PC_Calrestricted
  Analysis Name:  D_2009_06_10_10_15_54_node176_fullProcessingAmplicons
  Full Path:      /srv/seq/454/2008_12_09/R_2008_12_09_13_51_01_FLX04080350_Administrator_PC_Calrestricted/D_2009_06_10_10_15_54_node176_fullProcessingAmplicons/

  Read Header Len:  32
  Name Length:      14
  # of Bases:       248
  Clip Qual Left:   5
  Clip Qual Right:  248
  Clip Adap Left:   0
  Clip Adap Right:  0

Flowgram:	1.04	0.05	1.03	0.09	0.08	0.96	0.09	0.99	0.08	2.02	1.05	0.07	1.03	0.03	1.02	1.01	1.02	0.01	1.03	0.97	0.08	1.03	0.07	0.08	0.98	0.02	0.07	1.02	0.07	0.03	0.95	0.08	0.08	1.05	0.07	0.08	1.01	0.04	0.09	0.99	0.12	0.05	1.01	0.10	1.01	0.05	0.11	1.00	0.10	0.04	1.99	0.11	1.02	0.07	2.93	1.01	1.03	1.08	0.13	2.06	0.15	0.99	0.09	1.01	3.06	0.07	0.10	3.05	0.12	0.02	2.01	1.04	1.02	0.04	0.13	0.94	1.04	0.06	0.98	0.12	1.00	0.07	0.96	0.08	0.10	1.03	0.17	1.02	1.05	0.12	2.97	0.13	0.11	2.01	0.09	0.09	0.97	0.10	0.05	0.95	1.03	0.01	0.14	2.05	0.10	0.06	2.08	1.00	0.08	1.07	0.10	0.10	1.03	0.12	0.88	0.11	0.13	1.01	0.10	1.06	1.03	0.07	0.98	0.13	1.01	0.03	1.00	0.07	1.01	0.10	1.01	0.06	0.12	2.01	1.03	0.10	1.04	0.05	1.01	2.01	0.11	0.07	0.99	0.09	1.02	1.01	0.14	0.11	0.94	0.10	0.12	0.97	0.12	0.07	1.05	0.10	0.08	1.07	0.16	0.06	1.04	0.11	1.02	0.04	0.06	1.01	0.12	0.09	1.97	0.11	0.08	0.96	0.08	0.05	1.93	0.06	2.01	0.05	0.06	2.02	1.09	2.00	0.07	1.03	0.05	0.10	2.00	1.04	2.01	1.04	1.87	0.11	1.97	0.93	1.93	0.11	0.13	2.02	1.10	0.12	1.03	1.03	0.14	0.98	0.11	0.15	0.97	0.16	0.97	2.06	0.13	0.13	1.04	0.09	0.14	1.01	0.10	0.10	1.03	0.14	0.09	0.95	2.02	1.06	0.08	0.12	1.03	0.08	0.12	1.02	0.07	2.07	1.06	0.11	2.05	0.12	0.13	1.01	0.10	0.11	1.00	0.04	1.96	0.16	0.07	2.02	0.12	1.09	0.15	1.02	0.11	1.01	1.03	0.13	0.07	1.04	0.13	0.97	0.08	0.14	1.01	1.01	0.14	1.06	0.08	0.10	1.09	0.17	0.06	0.95	0.07	1.04	0.07	2.01	2.12	0.07	0.13	1.04	0.12	0.10	0.97	0.11	1.01	0.06	3.01	0.03	1.00	0.09	0.14	3.08	1.09	0.10	1.02	0.08	1.03	0.13	1.02	0.99	0.20	1.03	0.06	0.92	1.04	0.07	0.13	1.01	0.11	0.15	1.02	0.09	0.17	1.03	0.18	0.08	1.06	1.02	1.01	0.14	0.99	0.17	1.01	1.01	0.10	0.97	0.95	0.05	0.13	1.04	1.94	0.14	0.12	1.09	2.05	1.13	0.11	0.07	0.94	0.15	2.02	1.96	1.03	0.99	0.14	0.17	2.00	0.99	0.18	1.05	0.09	0.13	1.00	0.08	0.17	1.02	0.20	0.14	1.08	0.05	1.01	0.15	0.98	0.08	0.05	1.06	2.91	0.10	1.99	0.09	0.07	0.90	0.09	1.97	1.15	0.14	0.17	0.98	1.89	1.06	0.15	0.10	1.06	0.13	2.99	0.11	0.08	0.91	0.10	1.04	1.03	0.06	0.87	0.16	1.08	0.09	1.00	0.06	1.89	0.08	0.16	3.11
Flow Indexes:	1	3	6	8	10	10	11	13	15	16	17	19	20	22	25	28	31	34	37	40	43	45	48	51	51	53	55	55	55	56	57	58	60	60	62	64	65	65	65	68	68	68	71	71	72	73	76	77	79	81	83	86	88	89	91	91	91	94	94	97	100	101	104	104	107	107	108	110	113	115	118	120	121	123	125	127	129	131	134	134	135	137	139	140	140	143	145	146	149	152	155	158	161	163	166	169	169	172	175	175	177	177	180	180	181	182	182	184	187	187	188	189	189	190	191	191	193	193	194	195	195	198	198	199	201	202	204	207	209	210	210	213	216	219	222	223	223	224	227	230	232	232	233	235	235	238	241	243	243	246	246	248	250	252	253	256	258	261	262	264	267	270	272	274	274	275	275	278	281	283	285	285	285	287	290	290	290	291	293	295	297	298	300	302	303	306	309	312	315	316	317	319	321	322	324	325	328	329	329	332	333	333	334	337	339	339	340	340	341	342	345	345	346	348	351	354	357	359	361	364	365	365	365	367	367	370	372	372	373	376	377	377	378	381	383	383	383	386	388	389	391	393	395	397	397	400	400	400
Bases:	tcagAACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGACATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCTTGGG
Quality Scores:	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	38	38	38	38	38	40	38	38	38	38	38	38	38	38	38	38	38	38	38	38	38	38	38	38	38	38	38	37

>FLP3FBN01AM0P3
  Run Prefix:   R_2008_12_09_13_51_01_
  Region #:     1
  XY Location:  0145_0217

  Run Name:       R_2008_12_09_13_51_01_FLX04080350_Administrator_PC_Calrestricted
  Analysis Name:  D_2009_06_10_10_15_54_node176_fullProcessingAmplicons
  Full Path:      /srv/seq/454/2008_12_09/R_2008_12_09_13_51_01_FLX04080350_Administrator_PC_Calrestricted/D_2009_06_10_10_15_54_node176_fullProcessingAmplicons/

  Read Header Len:  32
  Name Length:      14
  # of Bases:       268
  Clip Qual Left:   5
  Clip Qual Right:  268
  Clip Adap Left:   0
  Clip Adap Right:  0

Flowgram:	1.02	0.01	1.03	0.00	0.00	1.00	0.04	1.02	0.00	1.14	0.86	0.08	0.07	0.87	0.05	1.01	0.09	1.06	0.05	1.02	0.89	0.04	1.02	1.76	0.13	0.11	0.95	0.13	1.04	0.04	1.01	0.04	0.09	1.03	0.08	0.02	1.01	0.07	0.05	1.01	0.07	0.00	0.99	0.10	0.97	0.00	0.12	0.89	0.12	0.01	2.03	0.13	1.02	0.05	3.03	1.09	1.02	1.03	0.17	2.13	0.11	0.99	0.07	1.13	1.05	0.07	0.74	0.19	1.04	0.02	0.11	2.93	0.12	0.00	1.77	1.18	0.97	0.00	0.14	0.97	1.00	0.02	1.00	0.08	0.93	0.09	1.01	0.08	0.11	1.04	0.19	0.85	1.04	0.12	2.71	0.12	0.11	1.78	0.11	0.09	1.00	0.10	0.06	1.23	1.00	0.04	0.12	2.20	0.16	0.04	2.07	1.73	0.21	0.04	1.75	0.72	0.11	0.05	2.78	0.09	0.85	0.08	1.08	0.09	0.94	0.12	1.21	0.06	0.11	1.03	0.24	1.82	0.07	0.15	1.95	2.02	0.10	0.10	1.02	0.13	0.98	0.95	1.08	0.08	0.98	0.15	0.11	1.03	0.13	0.79	0.10	0.10	1.06	0.08	1.00	1.02	1.06	0.02	0.92	1.16	0.15	0.01	1.95	0.21	1.90	0.05	0.20	1.89	1.02	0.07	0.20	2.79	0.04	0.14	2.02	0.96	1.83	1.13	3.79	0.97	0.11	0.11	1.97	0.08	0.10	1.91	1.92	0.12	0.07	1.10	0.09	0.96	0.10	0.05	1.08	0.08	0.93	2.08	0.11	0.10	0.95	0.12	1.09	0.10	0.05	0.92	0.16	0.95	0.05	1.04	1.03	1.02	0.04	0.10	1.06	2.97	0.07	0.14	2.82	0.17	0.13	1.05	0.22	0.10	0.96	0.16	2.90	1.06	1.06	1.05	2.01	0.03	0.17	0.96	1.95	1.98	0.15	1.04	0.11	0.96	4.77	0.15	0.99	0.06	0.05	1.03	1.05	0.11	0.18	1.01	0.97	0.12	1.13	0.11	0.10	0.99	0.09	0.11	1.03	0.12	1.97	0.17	0.97	0.12	0.10	1.01	0.13	0.11	1.01	0.13	0.13	1.01	0.06	0.06	1.01	0.95	0.12	1.87	0.08	1.00	0.07	0.08	1.01	0.12	1.11	0.16	0.10	1.01	1.00	0.15	0.08	1.01	0.19	0.13	0.91	0.87	0.17	0.15	1.06	0.15	2.10	1.06	0.14	0.11	1.08	0.11	0.09	1.04	0.16	0.08	1.04	2.00	1.07	0.84	0.15	0.11	2.15	1.01	0.16	1.06	0.16	0.17	0.99	0.06	0.15	1.01	2.16	0.21	1.10	0.92	0.08	0.18	3.00	0.06	1.07	0.09	0.85	1.96	0.18	1.09	1.09	0.14	0.20	0.94	2.03	1.08	0.16	0.16	1.11	0.12	4.70	0.11	0.05	0.93	0.04	1.06	1.12	0.86	0.17	0.02	1.02	0.98	1.09	2.21	0.12	0.16	1.02	0.00	0.04	1.04	0.11	2.07	2.11	0.08	1.07	0.01	0.73	0.08	2.01	0.18	0.16	0.92	1.06	1.07	0.14	0.15	1.00	1.07	1.86	0.89	1.08	0.13	0.95	0.14	1.06	0.08	0.18	0.82	3.25	1.25
Flow Indexes:	1	3	6	8	10	11	14	16	18	20	21	23	24	24	27	29	31	34	37	40	43	45	48	51	51	53	55	55	55	56	57	58	60	60	62	64	65	67	69	72	72	72	75	75	76	77	80	81	83	85	87	90	92	93	95	95	95	98	98	101	104	105	108	108	111	111	112	112	115	115	116	119	119	119	121	123	125	127	130	132	132	135	135	136	136	139	141	142	143	145	148	150	153	155	156	157	159	160	163	163	165	165	168	168	169	172	172	172	175	175	176	177	177	178	179	179	179	179	180	183	183	186	186	187	187	190	192	195	197	198	198	201	203	206	208	210	211	212	215	216	216	216	219	219	219	222	225	227	227	227	228	229	230	231	231	234	235	235	236	236	238	240	241	241	241	241	241	243	246	247	250	251	253	256	259	261	261	263	266	269	272	275	276	278	278	280	283	285	288	289	292	295	296	299	301	301	302	305	308	311	312	312	313	314	317	317	318	320	323	326	327	327	329	330	333	333	333	335	337	338	338	340	341	344	345	345	346	349	351	351	351	351	351	354	356	357	358	361	362	363	364	364	367	370	372	372	373	373	375	377	379	379	382	383	384	387	388	389	389	390	391	393	395	398	399	399	399	400
Bases:	tcagACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGGCCGGCCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATCAGACGCGGGCCCATCCCGTACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTTACTCACCCG
Quality Scores:	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	40	40	40	40	40	40	40	40	40	40	40	40	39	38	38	39	39	38	38	38	33	33	33	36	39	39	39	37	37	37	36	36	29	29	29	33	33	37	37	37	37	37	35	35	31	31	25	25	25	21	21	21	36	37	37	36	36	36	36	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	33	33	33	37	37	37	36	36	36	33	33	33	33	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	33	33	33	38	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	36	28	28	28	28	28	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	36	36	36	37	37	37	37	36	36	36	36	37	37	37	37	37	37	36	36	36	36	36	36	30	26	22	22	22	22	22	33	36	36	36	36	36	31	31	32	36	36	36	36	36	36	36	30	30	30	31	31	31	31	31	30	30	30	31	31	25	22	16	16	16	16

>FLP3FBN01BO4IE
  Run Prefix:   R_2008_12_09_13_51_01_
  Region #:     1
  XY Location:  0579_0388

  Run Name:       R_2008_12_09_13_51_01_FLX04080350_Administrator_PC_Calrestricted
  Analysis Name:  D_2009_06_10_10_15_54_node176_fullProcessingAmplicons
  Full Path:      /srv/seq/454/2008_12_09/R_2008_12_09_13_51_01_FLX04080350_Administrator_PC_Calrestricted/D_2009_06_10_10_15_54_node176_fullProcessingAmplicons/

  Read Header Len:  32
  Name Length:      14
  # of Bases:       245
  Clip Qual Left:   5
  Clip Qual Right:  244
  Clip Adap Left:   0
  Clip Adap Right:  0

Flowgram:	1.04	0.03	0.97	0.07	0.07	1.02	0.06	1.01	0.06	1.05	2.05	0.97	0.06	0.01	0.99	0.09	0.01	1.07	0.04	0.91	0.06	1.06	0.04	0.96	1.01	0.03	1.00	0.08	0.08	1.07	0.95	0.06	0.08	1.05	0.07	0.07	1.00	0.05	0.06	1.00	0.08	0.03	1.03	0.09	0.98	0.02	0.11	1.00	0.08	0.03	2.04	0.11	1.04	0.05	3.09	0.99	1.02	1.06	0.14	2.03	0.11	1.01	0.07	0.96	1.04	0.06	1.02	0.09	1.05	0.03	0.14	3.18	0.10	0.05	2.20	0.98	0.96	1.03	0.13	0.10	0.94	0.11	0.93	0.08	1.05	0.07	1.06	0.11	0.13	1.03	0.08	0.99	1.01	0.08	3.01	0.14	0.13	1.99	0.21	0.11	0.91	0.14	0.08	1.01	1.03	0.04	0.12	1.97	0.08	0.03	1.91	1.09	1.94	0.11	0.95	0.11	0.14	1.92	1.96	0.10	0.81	0.13	1.09	0.08	1.04	0.08	1.01	0.10	0.13	1.02	0.08	0.93	0.96	0.07	1.90	1.97	0.14	0.08	1.07	0.15	0.98	1.05	1.04	0.12	1.01	0.07	0.11	1.01	0.14	1.02	0.09	0.14	0.97	0.05	0.98	0.97	1.05	0.08	0.91	1.07	0.14	0.83	1.23	0.14	1.91	0.09	0.05	1.91	1.12	0.91	0.13	1.93	0.14	0.08	2.05	1.00	2.01	1.03	3.90	0.10	0.11	0.95	1.97	0.14	0.12	1.99	1.10	0.09	1.01	1.10	0.10	0.14	0.91	0.07	1.01	0.10	0.96	2.00	0.13	0.08	1.05	0.16	0.96	0.14	0.07	0.95	0.14	1.06	0.11	1.03	0.96	1.02	0.07	0.05	1.07	0.13	0.06	2.01	0.11	0.94	0.07	0.14	3.02	0.13	0.11	1.02	0.09	0.12	1.09	0.16	0.90	0.09	1.09	1.03	0.08	0.07	1.01	0.10	0.94	0.12	0.10	1.01	0.06	1.08	0.11	0.07	1.06	1.92	0.08	0.96	0.05	0.11	2.11	0.05	0.04	1.00	0.14	0.06	0.99	0.11	1.03	0.05	2.05	0.14	3.33	0.12	2.65	0.11	0.10	1.13	0.10	0.16	2.68	0.85	0.22	0.08	1.14	1.07	0.15	0.06	1.07	0.13	1.02	0.07	0.13	0.90	0.21	0.14	0.89	0.18	0.12	1.05	1.10	0.16	0.08	1.10	0.07	0.74	0.26	0.18	0.89	1.17	0.16	0.13	2.06	0.08	1.06	0.10	0.15	1.05	0.24	0.09	1.04	0.15	0.10	1.84	0.14	0.12	1.07	0.12	0.32	1.03	0.12	0.15	1.10	0.85	0.11	0.21	0.95	0.03	0.12	1.20	0.20	0.13	0.86	1.84	1.20	0.95	0.11	0.11	1.80	1.09	0.08	0.94	0.12	0.18	0.95	0.15	0.17	0.86	0.17	1.16	1.00	0.14	0.97	0.05	0.44	0.98	0.14	0.15	2.87	0.00	1.01	0.14	0.86	2.02	1.22	0.14	1.04	0.02	0.09	1.08	2.15	0.19	0.11	0.79	2.07	0.11	4.32	0.13	0.90	0.08	0.83	0.08	1.04	0.12	0.00	0.98	0.02	1.31	0.10	0.00	1.16	0.98	0.04	2.06	0.09	0.15	1.06	0.30
Flow Indexes:	1	3	6	8	10	11	11	12	15	18	20	22	24	25	27	30	31	34	37	40	43	45	48	51	51	53	55	55	55	56	57	58	60	60	62	64	65	67	69	72	72	72	75	75	76	77	78	81	83	85	87	90	92	93	95	95	95	98	98	101	104	105	108	108	111	111	112	113	113	115	118	118	119	119	121	123	125	127	130	132	133	135	135	136	136	139	141	142	143	145	148	150	153	155	156	157	159	160	162	163	165	165	168	168	169	170	172	172	175	175	176	177	177	178	179	179	179	179	182	183	183	186	186	187	189	190	193	195	197	198	198	201	203	206	208	210	211	212	215	218	218	220	223	223	223	226	229	231	233	234	237	239	242	244	247	248	248	250	253	253	256	259	261	263	263	265	265	265	267	267	267	270	273	273	273	274	277	278	281	283	286	289	292	293	296	298	301	302	305	305	307	310	313	316	316	319	322	325	326	329	332	335	336	336	337	338	341	341	342	344	347	350	352	353	355	358	361	361	361	363	365	366	366	367	369	372	373	373	376	377	377	379	379	379	379	381	383	385	388	390	393	394	396	396	399
Bases:	tcagACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGGCCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGACTTGGTAGGCCGTTACCCCACCAACTATCTAATCAGACGCAAGCCCATCTATCAGCGGATTGCTCCTTTCCCATTTATATCATGTGATATTCATGGCATATGCGGTATTAGCAGTCATTTCTAACTGTTGTTCCCCTCTGATAGGc
Quality Scores:	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	33	33	33	38	38	40	40	40	40	40	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	36	36	36	36	36	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	33	33	23	20	20	20	20	20	30	20	20	20	22	28	37	37	37	37	37	37	37	36	33	33	33	37	37	37	37	37	37	36	36	36	37	37	37	36	36	33	33	33	36	36	36	36	37	36	36	36	37	36	36	36	37	37	37	36	36	36	36	36	36	30	30	30	30	30	22	22	22	22	26	30	36	36	36	36	36	36	36	36

>FLP3FBN01BPX14
  Run Prefix:   R_2008_12_09_13_51_01_
  Region #:     1
  XY Location:  0588_1818

  Run Name:       R_2008_12_09_13_51_01_FLX04080350_Administrator_PC_Calrestricted
  Analysis Name:  D_2009_06_10_10_15_54_node176_fullProcessingAmplicons
  Full Path:      /srv/seq/454/2008_12_09/R_2008_12_09_13_51_01_FLX04080350_Administrator_PC_Calrestricted/D_2009_06_10_10_15_54_node176_fullProcessingAmplicons/

  Read Header Len:  32
  Name Length:      14
  # of Bases:       258
  Clip Qual Left:   5
  Clip Qual Right:  258
  Clip Adap Left:   0
  Clip Adap Right:  0

Flowgram:	1.01	0.00	1.00	0.03	0.03	1.03	0.04	1.01	0.04	1.12	1.85	0.63	0.18	0.02	0.60	0.22	0.12	1.02	0.14	1.01	0.11	1.20	0.15	1.02	1.00	0.10	0.91	0.12	0.24	0.77	1.05	0.02	0.02	1.14	0.05	0.05	1.04	0.03	0.02	1.00	0.05	0.00	1.01	0.06	0.97	0.01	0.07	0.99	0.06	0.02	2.09	0.06	1.02	0.03	3.04	1.01	1.06	1.05	0.11	2.04	0.10	1.07	0.03	1.08	1.29	0.14	0.67	0.24	0.84	0.03	0.14	2.12	0.87	0.00	1.91	1.08	1.07	0.00	0.13	0.95	1.06	0.02	1.12	0.09	0.98	0.10	0.92	0.11	0.13	0.96	0.24	0.89	1.02	1.07	2.14	0.13	0.10	1.02	0.12	1.01	1.19	0.06	0.05	1.25	0.91	0.01	0.17	5.57	0.15	0.01	2.05	0.11	1.84	0.05	2.11	0.07	1.00	0.01	0.97	0.05	0.96	0.05	1.00	0.10	0.10	1.06	0.11	1.09	0.11	1.88	3.90	0.16	1.01	1.09	1.21	0.96	0.12	0.16	0.99	0.12	0.16	0.84	0.14	0.20	0.71	0.11	1.06	0.93	0.99	0.13	1.10	1.10	0.17	0.05	2.33	0.10	0.27	0.89	0.90	2.26	1.31	0.07	0.08	3.31	0.14	0.09	2.35	1.35	2.24	1.14	4.47	1.08	0.13	0.16	2.49	0.94	0.81	0.25	1.32	0.11	0.08	2.28	0.11	1.02	0.15	0.13	1.00	0.16	1.00	2.13	0.14	0.07	0.91	0.18	0.12	1.15	0.87	0.15	1.07	0.12	0.12	0.93	1.01	1.06	0.04	0.14	1.05	1.02	0.04	0.86	0.24	1.03	0.12	0.22	1.93	0.06	1.18	0.91	0.31	0.15	0.94	0.09	1.96	0.18	1.00	0.11	1.29	0.18	0.06	1.00	0.33	0.11	1.19	0.28	1.79	0.72	0.49	0.93	1.21	1.63	0.14	1.22	0.17	0.19	1.01	0.15	1.02	0.12	3.26	0.80	0.23	0.97	0.13	1.02	0.22	0.09	1.07	0.08	0.26	1.89	0.13	2.04	1.86	0.08	0.13	0.98	0.12	1.14	0.15	0.87	0.28	0.22	0.97	0.15	0.10	0.95	0.08	0.23	2.07	0.09	1.05	0.22	0.12	1.00	0.07	1.10	0.02	0.08	2.14	0.12	2.25	0.15	0.10	1.03	0.11	0.04	1.21	0.09	0.87	1.43	0.18	0.09	1.77	0.11	0.24	1.03	0.16	0.13	1.07	0.07	0.16	3.95	0.25	0.04	1.08	0.05	0.05	0.98	0.12	0.13	2.39	0.73	0.35	1.00	0.14	1.04	1.02	1.10	0.29	0.14	1.67	1.16	2.74	0.14	2.11	0.07	0.14	0.88	0.15	1.25	0.14	0.15	1.14	0.92	0.00	1.06	0.09	0.30	2.11	0.94	0.12	0.09	1.64	0.23	3.53	0.06	1.21	0.13	0.13	1.08	0.20	0.86	0.22	0.10	1.15	0.13	0.58	0.64	0.22	0.88	0.15	2.64	0.17	0.09	1.12	0.12	0.02	1.62	0.20	1.03	1.99	0.19	0.16	1.06	0.08	0.13	1.01	0.15	0.01	0.87	0.31	0.11	1.31	0.83	1.10	0.88	0.21	0.10	1.40	1.18
Flow Indexes:	1	3	6	8	10	11	11	12	15	18	20	22	24	25	27	30	31	34	37	40	43	45	48	51	51	53	55	55	55	56	57	58	60	60	62	64	65	67	69	72	72	73	75	75	76	77	80	81	83	85	87	90	92	93	94	95	95	98	100	101	104	105	108	108	108	108	108	108	111	111	113	113	115	115	117	119	121	123	126	128	130	130	131	131	131	131	133	134	135	136	139	142	145	147	148	149	151	152	155	155	158	159	160	160	161	164	164	164	167	167	168	169	169	170	171	171	171	171	172	175	175	176	177	179	182	182	184	187	189	190	190	193	196	197	199	202	203	204	207	208	210	212	215	215	217	218	221	223	223	225	227	230	233	235	235	236	238	239	240	240	242	245	247	249	249	249	250	252	254	257	260	260	262	262	263	263	266	268	270	273	276	279	279	281	284	286	289	289	291	291	294	297	299	300	303	303	306	309	312	312	312	312	315	318	321	321	322	324	326	327	328	331	331	332	333	333	333	335	335	338	340	343	344	346	349	349	350	353	353	355	355	355	355	357	360	362	365	367	368	370	372	372	372	375	378	378	380	381	381	384	387	390	393	394	395	396	399	400
Bases:	tcagACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGGGGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGTCACGCGAGCCTATCCTCATCCGACGGATCTTTAGATGGAACCAGATGCCTGATTCCATCGCCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATCGAGGGCAAGTTGCATACGCG
Quality Scores:	34	34	34	34	32	22	22	22	22	27	31	34	34	34	32	32	32	34	34	34	34	34	40	40	40	40	40	40	40	40	39	40	39	39	36	28	23	23	22	33	36	36	36	36	36	36	36	36	37	34	34	34	34	34	34	33	32	32	32	32	27	27	14	14	14	14	14	14	29	29	32	32	32	32	33	34	34	34	34	34	34	34	34	34	34	34	34	34	29	34	30	31	31	31	34	33	30	30	24	24	25	24	18	18	18	15	15	15	18	18	14	17	17	17	14	14	14	14	16	12	12	16	17	22	25	25	29	33	34	34	34	32	32	32	34	34	34	34	34	34	32	32	32	34	31	31	30	32	32	32	25	22	16	16	16	16	16	15	16	16	16	22	29	14	14	14	18	24	32	34	34	34	34	34	34	34	34	30	30	30	32	34	34	34	34	34	29	29	29	29	32	25	25	18	18	18	29	32	29	34	29	29	32	29	17	17	17	17	29	24	21	14	14	14	14	14	17	29	24	24	24	24	24	24	24	24	24	14	14	14	14	14	14	14	19	19	14	14	14	14	14	14	14	19	19	19	19	24	26	22	22	19	20	20	20	20	16	16

>FLP3FBN01DB5I7
  Run Prefix:   R_2008_12_09_13_51_01_
  Region #:     1
  XY Location:  1251_1905

  Run Name:       R_2008_12_09_13_51_01_FLX04080350_Administrator_PC_Calrestricted
  Analysis Name:  D_2009_06_10_10_15_54_node176_fullProcessingAmplicons
  Full Path:      /srv/seq/454/2008_12_09/R_2008_12_09_13_51_01_FLX04080350_Administrator_PC_Calrestricted/D_2009_06_10_10_15_54_node176_fullProcessingAmplicons/

  Read Header Len:  32
  Name Length:      14
  # of Bases:       272
  Clip Qual Left:   5
  Clip Qual Right:  272
  Clip Adap Left:   0
  Clip Adap Right:  0

Flowgram:	1.00	0.00	1.03	0.03	0.02	1.03	0.03	1.05	0.01	1.15	0.94	0.11	0.07	0.98	0.09	0.94	0.08	1.04	1.83	0.11	0.08	1.07	0.92	0.09	1.06	0.16	0.96	0.12	0.10	1.00	1.03	0.03	0.04	1.07	0.07	0.05	1.02	0.09	0.06	0.98	0.10	0.00	0.98	0.12	0.91	0.01	0.15	0.91	0.12	0.01	2.10	0.12	1.03	0.05	2.94	1.07	1.04	1.07	0.15	1.99	0.10	1.00	0.06	1.09	2.74	0.09	0.12	2.83	0.13	0.04	1.72	1.14	0.92	0.02	0.17	1.00	1.04	0.01	0.91	0.10	1.01	0.04	1.02	0.07	0.10	1.00	0.14	0.90	1.00	0.14	2.77	0.13	0.09	2.05	0.19	0.07	0.93	0.10	0.08	1.02	1.02	0.06	0.12	2.15	0.10	0.03	1.87	1.14	1.02	0.06	1.86	0.11	0.11	0.90	2.73	0.06	0.91	0.12	1.10	0.05	0.95	0.10	1.02	0.10	0.10	1.03	0.23	1.87	0.07	0.09	1.84	1.81	0.10	0.09	1.09	0.16	0.93	1.03	1.09	0.11	0.93	0.10	0.14	1.00	0.10	1.00	0.14	0.12	0.95	0.11	0.92	1.01	1.06	0.06	0.97	1.08	0.10	0.01	1.92	0.19	1.99	0.04	0.19	2.01	1.04	0.06	0.16	2.82	0.10	0.07	1.91	0.14	2.84	1.07	3.75	1.00	0.13	0.09	1.99	0.09	0.12	1.97	2.08	0.13	0.09	1.08	0.12	0.95	0.11	0.10	1.02	0.11	0.96	2.06	0.15	0.11	1.01	0.11	1.03	0.12	0.04	1.02	0.20	1.00	0.06	0.97	1.03	0.96	0.10	0.08	1.03	2.78	1.11	0.13	1.96	0.14	0.09	1.05	0.18	0.11	0.96	0.12	1.95	0.11	1.06	0.14	0.17	0.95	0.11	0.17	1.09	0.07	0.17	1.03	1.95	0.92	0.14	0.12	1.90	1.87	0.17	1.02	0.15	1.02	0.14	0.10	1.02	0.14	2.15	0.13	4.55	1.08	0.04	0.14	1.71	0.94	0.17	0.13	3.71	0.13	0.13	0.93	0.09	0.09	1.00	0.09	0.16	0.95	0.11	0.12	1.04	3.68	0.12	0.07	1.05	0.07	1.03	0.09	0.09	1.01	0.13	0.09	1.00	2.85	0.14	0.10	0.99	0.04	0.10	1.02	0.14	0.17	1.02	0.92	0.17	0.09	1.10	0.05	0.13	1.02	0.09	0.14	1.03	1.89	1.15	0.95	0.13	0.09	2.04	0.91	0.12	1.09	0.09	0.10	1.05	0.04	0.17	0.96	0.05	1.07	0.13	0.04	1.96	0.89	2.95	0.14	1.84	1.89	0.13	0.11	0.92	0.13	1.08	0.21	0.15	1.08	2.00	0.10	0.18	0.88	1.04	0.10	4.53	0.13	0.09	0.90	0.03	1.04	1.14	0.06	0.20	0.91	0.10	0.07	1.04	0.08	0.18	1.01	0.12	2.80	0.20	0.09	1.16	0.00	0.06	1.01	0.12	2.00	2.17	0.06	0.18	0.89	0.00	0.08	2.84	0.08	0.18	0.97	0.89	1.15	0.10	0.11	1.10	1.01	2.02	0.98	1.02	0.11	0.83	0.14	1.04	0.11	0.03	0.94	2.86	1.07	1.15	0.05	2.06	0.98
Flow Indexes:	1	3	6	8	10	11	14	16	18	19	19	22	23	25	27	30	31	34	37	40	43	45	48	51	51	53	55	55	55	56	57	58	60	60	62	64	65	65	65	68	68	68	71	71	72	73	76	77	79	81	83	86	88	89	91	91	91	94	94	97	100	101	104	104	107	107	108	109	111	111	114	115	115	115	117	119	121	123	126	128	128	131	131	132	132	135	137	138	139	141	144	146	149	151	152	153	155	156	159	159	161	161	164	164	165	168	168	168	171	171	173	173	173	174	175	175	175	175	176	179	179	182	182	183	183	186	188	191	193	194	194	197	199	202	204	206	207	208	211	212	212	212	213	215	215	218	221	223	223	225	228	231	234	235	235	236	239	239	240	240	242	244	247	249	249	251	251	251	251	251	252	255	255	256	259	259	259	259	262	265	268	271	272	272	272	272	275	277	280	283	284	284	284	287	290	293	294	297	300	303	304	304	305	306	309	309	310	312	315	318	320	323	323	324	325	325	325	327	327	328	328	331	333	336	337	337	340	341	343	343	343	343	343	346	348	349	352	355	358	360	360	360	363	366	368	368	369	369	372	375	375	375	378	379	380	383	384	385	385	386	387	389	391	394	395	395	395	396	397	399	399	400
Bases:	tcagACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCCTGCACCGCCGGAGCTTCCCCCGCCGCCCCATGCGGGGCTGCGGGCATATGCGGTATTAGCAGCCGTTTCCGGCTGTTGTCCCCCAGTGCAGGGCAGGTTGCCCACGCGTTACTCACCCGTCCG
Quality Scores:	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	40	40	40	40	40	40	40	39	39	39	39	39	37	36	30	30	30	38	38	38	36	36	33	39	39	39	39	37	37	37	37	37	31	31	31	37	37	37	37	37	37	37	38	38	38	37	37	38	34	30	28	28	38	37	37	37	37	37	37	34	34	33	33	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	33	33	33	37	37	38	38	38	38	28	28	28	28	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	31	31	31	38	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	22	22	15	15	15	15	15	22	23	23	28	17	17	17	17	32	37	37	34	21	21	21	21	38	37	37	37	38	38	38	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	38	38	38	33	36	33	33	32	32	28	28	15	15	15	15	15	28	32	35	33	35	35	26	26	26	31	36	36	36	36	36	36	25	25	25	31	31	31	31	31	31	31	32	31	31	31	32	28	28	28	30	30	32	31	31

>FLP3FBN01AK9OO
  Run Prefix:   R_2008_12_09_13_51_01_
  Region #:     1
  XY Location:  0125_0438

  Run Name:       R_2008_12_09_13_51_01_FLX04080350_Administrator_PC_Calrestricted
  Analysis Name:  D_2009_06_10_10_15_54_node176_fullProcessingAmplicons
  Full Path:      /srv/seq/454/2008_12_09/R_2008_12_09_13_51_01_FLX04080350_Administrator_PC_Calrestricted/D_2009_06_10_10_15_54_node176_fullProcessingAmplicons/

  Read Header Len:  32
  Name Length:      14
  # of Bases:       257
  Clip Qual Left:   5
  Clip Qual Right:  253
  Clip Adap Left:   0
  Clip Adap Right:  0

Flowgram:	0.94	0.02	1.09	0.05	0.02	0.96	0.03	1.06	0.04	1.14	0.57	0.21	0.06	1.08	0.04	1.08	0.12	1.08	0.06	0.98	0.83	0.07	0.94	1.71	0.19	0.26	0.93	0.21	0.67	0.13	1.03	0.06	0.09	1.09	0.07	0.08	1.05	0.06	0.07	1.06	0.08	0.02	1.04	0.11	0.94	0.01	0.15	1.01	0.08	0.03	2.06	0.09	1.01	0.07	3.04	1.07	1.02	1.07	0.14	2.10	0.12	0.92	0.05	1.05	0.94	0.08	1.08	0.13	1.06	0.01	0.11	3.21	0.11	0.01	2.19	0.95	1.04	0.08	0.09	0.92	0.94	0.02	0.93	0.09	1.08	0.03	1.04	0.07	0.08	1.07	0.12	1.03	1.09	0.11	3.07	0.12	0.09	2.00	0.05	0.09	0.93	0.12	0.07	0.99	1.11	0.03	0.14	2.14	0.09	0.03	2.22	1.18	0.16	0.86	0.24	0.11	0.77	0.28	1.16	0.06	0.14	0.97	2.95	0.06	1.08	0.10	1.05	0.06	1.01	0.12	0.77	0.13	0.09	0.71	0.36	2.19	1.01	0.10	1.14	1.89	0.15	0.17	1.20	0.08	0.96	1.11	0.73	1.01	0.14	0.11	1.00	0.15	0.15	0.81	0.14	0.17	0.86	0.08	1.12	1.01	1.36	0.03	0.94	1.20	0.11	0.03	1.98	0.40	1.91	0.05	0.20	2.30	1.23	0.13	0.60	0.95	0.22	1.09	0.11	0.75	0.03	0.11	2.21	1.03	2.08	1.17	2.01	0.16	0.93	0.11	1.01	0.14	0.05	1.08	1.84	0.10	0.13	1.97	0.99	0.14	1.07	0.84	0.13	1.07	0.04	0.10	1.13	0.16	0.93	1.87	0.19	0.09	1.02	0.10	0.12	1.00	0.08	0.14	1.09	0.97	0.13	0.14	1.79	0.86	0.24	0.25	1.06	2.68	0.16	0.12	2.61	0.11	0.05	1.11	0.14	0.22	0.93	0.16	0.99	0.10	1.10	0.14	0.98	0.12	0.14	1.08	0.22	0.13	1.08	0.89	0.18	0.96	0.40	0.06	1.20	1.99	0.33	0.74	0.05	0.00	1.99	1.09	1.02	0.10	0.51	0.01	2.03	0.14	2.82	2.00	0.20	0.13	1.92	0.07	0.07	1.17	0.02	0.16	0.98	0.19	0.77	1.00	0.80	0.22	2.00	0.07	1.11	0.21	0.00	1.18	0.10	0.12	1.11	0.13	0.09	0.89	0.27	0.00	0.94	0.96	0.24	1.86	0.12	1.12	0.04	0.13	1.02	0.23	1.13	0.70	1.03	0.15	0.09	1.93	0.12	0.13	1.01	0.13	1.06	0.24	1.70	0.88	0.12	0.12	1.19	0.11	0.03	0.82	0.33	0.00	1.13	1.84	0.65	0.98	0.19	0.03	1.91	1.79	0.08	0.12	1.17	0.07	1.03	0.13	1.05	0.14	3.08	0.00	3.12	0.25	0.81	1.96	0.13	2.00	0.04	2.03	0.04	0.04	1.13	0.21	0.88	1.00	0.12	0.19	1.93	0.15	3.69	0.27	0.98	0.03	1.15	0.16	1.22	1.14	0.21	0.03	0.81	0.17	0.26	0.90	0.11	1.02	0.07	2.26	0.77	0.03	1.49	0.20	0.00	1.13	0.12	2.21	2.27	0.14	0.26	1.19	0.29	0.19	3.94	0.26
Flow Indexes:	1	3	6	8	10	11	14	16	18	20	21	23	24	24	27	29	31	34	37	40	43	45	48	51	51	53	55	55	55	56	57	58	60	60	62	64	65	67	69	72	72	72	75	75	76	77	80	81	83	85	87	90	92	93	95	95	95	98	98	101	104	105	108	108	111	111	112	114	117	119	122	123	123	123	125	127	129	131	134	136	136	137	139	140	140	143	145	146	147	148	151	154	157	159	160	161	163	164	167	167	169	169	172	172	173	175	176	178	180	183	183	184	185	185	186	187	187	189	191	194	195	195	198	198	199	201	202	204	207	209	210	210	213	216	219	220	223	223	224	227	228	228	228	231	231	231	234	237	239	241	243	246	249	250	252	255	256	256	258	261	261	262	263	265	267	267	269	269	269	270	270	273	273	276	279	281	282	283	285	285	287	290	293	296	299	300	302	302	304	307	309	310	311	314	314	317	319	321	321	322	325	328	331	332	332	333	334	337	337	338	338	341	343	345	347	347	347	349	349	349	351	352	352	354	354	356	356	359	361	362	365	365	367	367	367	367	369	371	373	374	377	380	382	384	384	385	387	390	392	392	393	393	396	399	399	399	399
Bases:	tcagACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTCGAGCCGTTACCTCACCAACTAGCTAATGCGCCGCGGGCCCATCTCATAGCGGATTACTCCTTTAATTGCTACTTCATGCGAAGCTACAATCTTATGCGGTATTAATCTCCCTTTCGGAAGGCTATTCCCCTCTATGAGGTCAGGTTGcccc
Quality Scores:	34	34	34	31	27	27	27	31	34	34	32	27	27	22	22	23	30	34	34	34	34	34	37	40	40	40	40	40	40	40	40	39	38	38	38	39	37	36	36	30	30	30	36	36	36	36	37	37	37	34	34	34	34	34	34	34	34	34	34	34	34	34	34	34	27	27	26	27	27	27	31	34	34	34	34	32	32	27	21	21	22	34	34	30	30	30	32	32	32	34	32	32	32	31	31	30	30	30	30	30	31	31	19	19	18	19	24	26	30	31	31	34	34	34	34	34	34	34	34	34	34	34	34	34	34	34	34	34	34	34	34	34	34	34	33	32	26	22	24	24	18	18	18	18	14	14	27	27	32	30	32	32	32	30	31	31	25	25	25	34	34	25	25	25	25	25	25	25	25	34	34	34	32	34	34	32	32	32	32	32	34	34	34	34	34	32	32	32	34	30	29	29	29	34	32	32	24	24	24	24	29	29	29	17	17	17	24	29	29	29	29	32	34	34	34	34	34	34	24	24	18	28	34	34	34	34	34	34	34	34	23	23	18	18	16	16	19	12	24	24	24	24	24	14	14	14	14	19	14	12	12	12	15	14	14	20	20

>FLP3FBN01ANGF2
  Run Prefix:   R_2008_12_09_13_51_01_
  Region #:     1
  XY Location:  0150_0112

  Run Name:       R_2008_12_09_13_51_01_FLX04080350_Administrator_PC_Calrestricted
  Analysis Name:  D_2009_06_10_10_15_54_node176_fullProcessingAmplicons
  Full Path:      /srv/seq/454/2008_12_09/R_2008_12_09_13_51_01_FLX04080350_Administrator_PC_Calrestricted/D_2009_06_10_10_15_54_node176_fullProcessingAmplicons/

  Read Header Len:  32
  Name Length:      14
  # of Bases:       257
  Clip Qual Left:   5
  Clip Qual Right:  257
  Clip Adap Left:   0
  Clip Adap Right:  0

Flowgram:	0.93	0.06	1.05	0.06	0.07	1.03	0.08	1.02	0.06	1.09	1.00	0.08	0.08	1.06	0.07	0.90	0.07	1.01	0.05	1.02	0.96	0.09	0.89	1.96	0.09	0.01	1.08	0.11	1.03	0.03	1.05	0.07	0.09	1.07	0.08	0.08	0.98	0.07	0.05	0.93	0.11	0.04	0.98	0.10	0.97	0.03	0.13	0.97	0.13	0.04	2.05	0.14	1.03	0.05	3.00	1.01	1.01	1.05	0.12	1.95	0.13	1.02	0.10	1.02	1.02	0.09	0.97	0.14	1.01	0.07	0.15	1.87	0.13	1.01	1.95	1.05	1.03	0.07	0.15	0.98	1.04	0.03	1.03	0.09	1.01	0.07	0.92	0.11	0.14	1.01	0.12	1.01	1.94	0.14	1.90	0.12	0.13	0.98	0.14	1.04	1.00	0.08	0.07	1.12	0.90	0.08	0.13	4.70	0.16	1.00	2.11	0.09	1.94	0.08	1.94	0.08	1.01	0.02	1.11	0.07	0.95	0.11	0.95	0.12	0.10	1.07	0.18	1.04	0.11	2.06	3.98	0.12	1.04	1.01	0.15	0.89	0.13	1.02	0.96	0.10	0.13	1.00	0.14	0.14	1.07	0.11	0.99	1.11	0.95	0.04	1.02	0.99	0.09	0.07	2.00	0.12	0.11	1.02	1.07	2.09	1.01	0.12	0.10	2.96	0.12	0.09	2.15	0.91	2.02	1.03	4.13	1.04	0.10	0.11	2.07	0.94	0.96	0.10	1.00	0.13	0.16	1.97	0.16	1.00	0.12	0.06	1.05	0.07	1.06	2.00	0.06	0.15	1.00	0.00	0.11	1.02	0.96	0.13	1.09	0.14	0.12	0.98	1.11	0.94	0.09	0.10	1.02	0.95	0.11	0.96	0.07	0.94	0.10	0.09	2.15	0.09	1.10	1.07	0.16	0.08	0.98	0.14	2.17	0.07	0.99	0.12	0.93	0.08	0.09	1.07	0.15	0.14	1.06	0.09	1.96	1.06	0.06	1.02	1.02	1.99	0.15	1.01	0.15	0.01	1.09	0.10	0.98	0.13	2.87	1.01	0.07	1.05	0.12	0.98	0.19	0.17	1.05	0.06	0.03	2.01	0.07	1.97	2.09	0.18	0.11	0.92	0.19	1.08	0.14	0.91	0.16	0.16	1.05	0.09	0.12	1.03	0.16	0.11	2.04	0.16	1.04	0.12	0.11	1.00	0.08	1.03	0.05	0.12	1.97	0.15	2.04	0.14	0.09	1.01	0.18	0.15	0.98	0.04	0.95	1.11	0.05	0.14	1.97	0.10	0.08	1.03	0.14	0.00	1.07	0.05	0.10	3.99	0.12	0.15	1.04	0.01	0.05	1.04	0.20	0.07	2.09	0.91	0.18	1.02	0.06	1.00	1.04	1.04	0.05	0.04	2.10	1.08	3.00	0.17	2.02	0.07	0.05	1.02	0.13	0.92	0.09	0.11	1.08	1.00	0.01	1.06	0.17	0.05	2.13	0.96	0.07	0.12	1.94	0.09	4.20	0.16	0.99	0.16	0.10	0.94	0.18	1.10	0.05	0.05	0.99	0.14	0.15	1.07	0.12	0.99	0.13	2.82	0.16	0.17	1.01	0.13	0.05	2.02	0.16	1.05	2.04	0.07	0.06	1.01	0.14	0.13	0.95	0.12	1.01	0.12	0.94	0.08	0.11	0.96	0.91	1.00	0.28	0.15	1.11	1.13
Flow Indexes:	1	3	6	8	10	11	14	16	18	20	21	23	24	24	27	29	31	34	37	40	43	45	48	51	51	53	55	55	55	56	57	58	60	60	62	64	65	67	69	72	72	74	75	75	76	77	80	81	83	85	87	90	92	93	93	95	95	98	100	101	104	105	108	108	108	108	108	110	111	111	113	113	115	115	117	119	121	123	126	128	130	130	131	131	131	131	133	134	136	138	139	142	145	147	148	149	151	152	155	155	158	159	160	160	161	164	164	164	167	167	168	169	169	170	171	171	171	171	172	175	175	176	177	179	182	182	184	187	189	190	190	193	196	197	199	202	203	204	207	208	210	212	215	215	217	218	221	223	223	225	227	230	233	235	235	236	238	239	240	240	242	245	247	249	249	249	250	252	254	257	260	260	262	262	263	263	266	268	270	273	276	279	279	281	284	286	289	289	291	291	294	297	299	300	303	303	306	309	312	312	312	312	315	318	321	321	322	324	326	327	328	331	331	332	333	333	333	335	335	338	340	343	344	346	349	349	350	353	353	355	355	355	355	357	360	362	365	368	370	372	372	372	375	378	378	380	381	381	384	387	389	391	394	395	396	399	400
Bases:	tcagACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGTCACGCGAGCCTATCCTCATCCGACGGATCTTTAGATGGAACCAGATGCCTGATTCCATCGCCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCAAGTTGCTCACGCG
Quality Scores:	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	37	37	37	37	37	37	37	37	37	37	37	38	38	28	28	28	28	28	38	38	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	38	38	38	38	37	37	36	36	36	29	28	28	28	36	36	36	36	36	36	28	28	29	36	38	38	38	38	38	36	32	32	32	32	30	28	28	28

>FLP3FBN01AF994
  Run Prefix:   R_2008_12_09_13_51_01_
  Region #:     1
  XY Location:  0068_1402

  Run Name:       R_2008_12_09_13_51_01_FLX04080350_Administrator_PC_Calrestricted
  Analysis Name:  D_2009_06_10_10_15_54_node176_fullProcessingAmplicons
  Full Path:      /srv/seq/454/2008_12_09/R_2008_12_09_13_51_01_FLX04080350_Administrator_PC_Calrestricted/D_2009_06_10_10_15_54_node176_fullProcessingAmplicons/

  Read Header Len:  32
  Name Length:      14
  # of Bases:       276
  Clip Qual Left:   5
  Clip Qual Right:  276
  Clip Adap Left:   0
  Clip Adap Right:  0

Flowgram:	1.02	0.06	1.01	0.06	0.07	1.02	0.08	1.01	0.07	1.06	1.00	0.05	0.03	1.02	0.07	0.99	0.04	1.06	0.03	1.01	1.03	0.01	1.03	2.09	0.08	0.04	0.99	0.10	1.04	0.03	1.00	0.05	0.06	1.04	0.08	0.06	0.99	0.05	0.07	0.93	0.09	0.05	1.08	0.10	0.99	0.05	0.09	0.98	0.07	0.03	1.97	0.11	1.01	0.05	3.09	1.01	1.06	1.03	0.14	2.11	0.14	1.02	0.07	0.98	3.05	0.07	0.08	2.07	1.00	0.05	2.09	1.01	1.00	0.05	0.12	0.98	1.05	0.04	1.00	0.11	1.02	0.05	1.03	0.09	0.09	0.98	0.14	1.01	1.02	1.07	2.06	0.10	0.10	2.00	0.12	0.09	1.01	0.12	0.07	0.99	0.98	0.06	0.11	4.99	0.14	1.05	2.06	0.12	2.02	0.10	2.04	0.11	1.06	0.01	0.92	0.07	1.00	0.07	1.01	0.11	0.10	1.01	0.07	1.07	0.10	2.02	4.17	0.10	1.05	1.05	0.13	0.03	0.99	0.16	2.04	0.05	0.13	1.01	0.10	0.11	0.96	0.09	0.96	1.03	1.05	0.03	1.04	2.02	3.00	0.06	0.11	1.96	1.05	0.05	0.14	3.03	0.09	0.08	2.08	0.99	2.09	1.08	4.05	0.94	0.04	0.16	2.05	0.13	0.09	2.09	0.94	0.09	0.95	0.06	0.13	1.08	0.16	0.09	1.96	0.15	0.97	1.98	0.07	0.16	1.06	0.10	0.10	2.10	0.14	1.90	1.06	1.03	0.07	0.08	0.95	0.17	0.09	0.95	0.14	0.11	0.99	0.12	0.14	1.06	0.11	0.07	2.21	0.16	1.03	1.02	0.13	0.01	0.99	0.12	1.02	0.00	1.05	0.97	0.11	0.04	1.07	0.17	1.01	0.11	0.15	1.02	0.18	1.04	0.05	0.10	1.07	1.05	0.09	0.91	0.19	0.15	1.06	0.07	0.07	0.96	0.15	2.00	0.14	0.15	1.01	0.10	1.04	0.15	3.05	1.05	0.13	1.06	0.10	0.02	0.92	0.09	0.16	3.09	0.12	0.10	1.03	1.03	0.08	0.04	0.93	0.17	4.07	0.08	0.11	1.04	0.17	0.16	1.03	0.16	0.20	1.13	0.10	0.04	1.01	3.81	0.11	0.06	3.83	0.13	0.96	0.05	0.18	0.94	0.06	0.07	1.05	0.13	1.96	0.09	0.93	0.11	0.10	1.03	0.13	0.19	0.97	0.17	0.18	1.09	0.19	0.12	1.17	2.00	1.04	0.98	0.14	0.10	1.94	0.94	0.06	1.00	1.05	0.15	1.99	1.07	0.08	0.93	1.05	0.14	2.79	0.08	1.01	0.94	0.23	0.14	2.05	2.87	1.67	0.96	0.33	0.32	1.12	0.13	4.48	0.11	1.01	0.10	1.57	0.20	1.10	0.13	0.23	1.04	0.18	0.99	0.18	0.14	1.03	0.91	0.10	1.77	0.87	1.75	0.16	1.15	1.78	0.36	0.24	1.08	0.44	0.22	0.74	0.30	0.19	0.72	0.23	0.20	0.96	1.16	0.86	0.97	0.27	0.24	1.23	1.17	1.62	1.00	1.10	0.29	1.57	0.21	1.35	0.23	0.34	0.91	2.89	0.83	0.91	0.19	0.76	1.15	0.33	0.14	1.32	1.58
Flow Indexes:	1	3	6	8	10	11	14	16	18	20	21	23	24	24	27	29	31	34	37	40	43	45	48	51	51	53	55	55	55	56	57	58	60	60	62	64	65	65	65	68	68	69	71	71	72	73	76	77	79	81	83	86	88	89	90	91	91	94	94	97	100	101	104	104	104	104	104	106	107	107	109	109	111	111	113	115	117	119	122	124	126	126	127	127	127	127	129	130	133	135	135	138	141	143	144	145	147	148	148	149	149	149	152	152	153	156	156	156	159	159	160	161	161	162	163	163	163	163	164	167	167	170	170	171	173	176	179	179	181	182	182	185	188	188	190	190	191	192	195	198	201	204	207	207	209	210	213	215	217	218	221	223	226	228	231	232	234	237	240	242	242	245	247	249	249	249	250	252	255	258	258	258	261	262	265	267	267	267	267	270	273	276	279	280	280	280	280	283	283	283	283	285	288	291	293	293	295	298	301	304	307	308	308	309	310	313	313	314	316	317	319	319	320	322	323	325	325	325	327	328	331	331	332	332	332	333	333	334	337	339	339	339	339	341	343	343	345	348	350	353	354	356	356	357	358	358	360	361	361	364	367	370	373	374	375	376	379	380	381	381	382	383	385	385	387	390	391	391	391	392	393	395	396	399	400	400
Bases:	tcagACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGTCGGTTTGGTGGGCCGTTACCCCGCCAACTGCCTAATGGAACGCATGCCTATCTATCAGCGATGAATCTTTAGCAAATATCCCCATGCGGGGCCCCTGCTTCATGCGGTATTAGTCCGACTTTCGCCGGGTTATCCCCTCCTGATAGGTAAGTTGCATACGCGTTACTTCACCCGTCGCGG
Quality Scores:	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	37	37	37	37	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	31	31	31	31	35	35	35	35	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	36	31	31	31	36	36	35	30	28	16	16	16	16	16	16	16	16	16	16	16	14	14	16	24	35	35	30	24	24	24	24	24	24	16	16	14	14	14	21	21	21	21	14	14	13	13	21	14	13	13	14	14	13	24	24	21	21	13	14	14	13	13

>FLP3FBN01AHXU8
  Run Prefix:   R_2008_12_09_13_51_01_
  Region #:     1
  XY Location:  0087_0802

  Run Name:       R_2008_12_09_13_51_01_FLX04080350_Administrator_PC_Calrestricted
  Analysis Name:  D_2009_06_10_10_15_54_node176_fullProcessingAmplicons
  Full Path:      /srv/seq/454/2008_12_09/R_2008_12_09_13_51_01_FLX04080350_Administrator_PC_Calrestricted/D_2009_06_10_10_15_54_node176_fullProcessingAmplicons/

  Read Header Len:  32
  Name Length:      14
  # of Bases:       289
  Clip Qual Left:   5
  Clip Qual Right:  289
  Clip Adap Left:   0
  Clip Adap Right:  0

Flowgram:	1.03	0.03	0.99	0.06	0.05	1.03	0.07	1.02	0.06	1.07	1.00	0.08	0.06	1.00	0.06	0.95	0.07	1.04	0.10	0.98	1.01	0.07	0.88	1.98	0.10	0.04	1.04	0.09	1.03	0.05	1.00	0.06	0.08	1.06	0.08	0.06	1.00	0.05	0.07	1.00	0.09	0.03	1.01	0.08	0.98	0.02	0.14	1.01	0.11	0.01	2.02	0.10	1.02	0.06	3.00	1.01	1.05	1.05	0.12	2.06	0.15	1.01	0.08	1.04	3.07	0.08	0.10	2.02	0.10	0.97	1.96	1.12	0.94	0.06	0.16	0.92	1.01	0.04	0.99	0.11	1.05	0.03	1.01	0.09	0.12	1.03	0.15	1.02	2.05	0.13	2.04	0.10	0.08	2.03	0.11	0.09	1.05	0.14	0.07	1.01	0.99	0.06	0.11	4.99	0.15	1.00	2.03	0.14	2.04	0.08	2.03	0.12	1.04	0.06	1.04	0.07	1.01	0.07	1.01	0.11	0.15	0.98	0.08	1.00	0.09	1.98	3.88	0.14	1.03	1.04	1.04	1.00	0.12	0.09	1.01	0.10	0.09	0.96	0.15	0.11	1.04	0.12	0.99	1.02	1.00	0.10	0.93	1.96	2.04	1.07	0.10	1.93	1.02	0.11	0.15	2.90	0.08	0.07	2.00	1.03	2.03	1.02	3.76	1.04	0.15	0.13	1.89	0.12	0.92	1.01	1.06	0.15	1.02	0.14	0.11	1.03	0.12	0.14	2.00	0.08	1.02	2.11	0.15	0.11	1.00	0.08	0.16	1.02	0.07	0.10	1.00	0.96	0.09	0.06	1.88	1.00	0.09	0.09	0.95	0.15	0.11	1.04	0.11	0.12	1.06	0.13	0.02	1.05	0.14	0.10	3.00	0.10	0.12	1.08	0.07	0.11	0.99	0.05	1.89	0.09	1.02	0.12	1.90	0.09	0.13	1.04	2.04	2.03	1.08	2.09	0.04	0.09	0.95	0.02	2.08	0.14	2.96	0.11	1.78	0.05	1.00	0.12	4.56	3.87	0.09	1.02	0.15	0.15	1.08	0.05	0.04	1.05	0.12	0.04	4.97	0.06	0.12	2.08	0.12	2.95	0.11	0.70	0.10	0.06	0.98	1.01	0.15	0.07	1.10	1.03	1.18	1.03	0.09	0.10	1.05	2.98	0.05	1.03	0.13	0.12	2.13	0.91	0.07	1.01	0.13	0.05	2.02	0.15	0.98	0.09	2.80	0.09	2.96	0.09	1.03	1.98	0.03	1.94	0.18	1.86	2.03	0.04	0.07	1.01	1.02	0.12	4.66	0.08	1.01	0.09	0.03	1.04	1.01	0.13	0.10	1.81	0.08	1.05	0.09	2.82	0.18	0.08	1.00	0.01	0.07	1.01	0.13	1.94	2.06	0.07	0.19	1.01	0.05	0.15	1.05	0.09	0.09	1.07	0.19	0.17	1.04	1.01	1.07	0.96	0.87	0.11	0.13	1.07	2.05	1.03	1.09	0.09	0.95	0.09	0.77	0.18	0.13	1.06	3.05	1.13	1.09	0.02	0.12	0.93	0.06	0.08	0.89	1.05	0.17	0.11	2.01	0.15	0.17	0.96	0.17	1.09	1.08	0.01	1.03	0.84	0.08	0.09	1.97	2.06	0.13	0.12	0.96	0.10	0.09	1.12	0.08	1.09	0.12	0.70	0.16	0.94	0.14	0.96	0.26	1.14
Flow Indexes:	1	3	6	8	10	11	14	16	18	20	21	23	24	24	27	29	31	34	37	40	43	45	48	51	51	53	55	55	55	56	57	58	60	60	62	64	65	65	65	68	68	70	71	71	72	73	76	77	79	81	83	86	88	89	89	91	91	94	94	97	100	101	104	104	104	104	104	106	107	107	109	109	111	111	113	115	117	119	122	124	126	126	127	127	127	127	129	130	131	132	135	138	141	143	144	145	147	148	148	149	149	150	152	152	153	156	156	156	159	159	160	161	161	162	163	163	163	163	164	167	167	169	170	171	173	176	179	179	181	182	182	185	188	191	192	195	195	196	199	202	205	208	211	211	211	214	217	219	219	221	223	223	226	227	227	228	228	229	230	230	233	235	235	237	237	237	239	239	241	243	243	243	243	243	244	244	244	244	246	249	252	255	255	255	255	255	258	258	260	260	260	262	265	266	269	270	271	272	275	276	276	276	278	281	281	282	284	287	287	289	291	291	291	293	293	293	295	296	296	298	298	300	300	301	301	304	305	307	307	307	307	307	309	312	313	316	316	318	320	320	320	323	326	328	328	329	329	332	335	338	341	342	343	344	345	348	349	349	350	351	353	355	358	359	359	359	360	361	364	367	368	371	371	374	376	377	379	380	383	383	384	384	387	390	392	394	396	398	400
Bases:	tcagACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTACTGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCGGGGATGCCCCCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGGCAGAGAG
Quality Scores:	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	40	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	32	32	32	32	38	38	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	32	28	15	15	15	15	15	22	22	37	37	37	37	37	37	37	37	37	37	37	37	37	38	38	38	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	34	34	34	37	37	37	37	37	37	37	36	36	36	36	36	36	31	24	22	22	22	22	33	36	36	36	36	36	26	27	28	37	38	38	38	37	38	38	38	32	32	31	32	32	31	31	31	32	32	31	30	30	30	30	32	32	32	32	32	32	32	31	31	31	31	31	32	31	31	31	31	31	31	28	28	28	28	28	28

>FLP3FBN01APRWO
  Run Prefix:   R_2008_12_09_13_51_01_
  Region #:     1
  XY Location:  0176_1782

  Run Name:       R_2008_12_09_13_51_01_FLX04080350_Administrator_PC_Calrestricted
  Analysis Name:  D_2009_06_10_10_15_54_node176_fullProcessingAmplicons
  Full Path:      /srv/seq/454/2008_12_09/R_2008_12_09_13_51_01_FLX04080350_Administrator_PC_Calrestricted/D_2009_06_10_10_15_54_node176_fullProcessingAmplicons/

  Read Header Len:  32
  Name Length:      14
  # of Bases:       276
  Clip Qual Left:   5
  Clip Qual Right:  276
  Clip Adap Left:   0
  Clip Adap Right:  0

Flowgram:	1.06	0.03	1.02	0.05	0.04	0.93	0.05	1.01	0.04	1.07	0.03	1.06	0.06	0.02	0.92	0.13	0.01	0.99	0.06	0.93	0.04	0.05	1.02	0.05	0.10	1.09	1.02	0.07	2.07	0.05	0.05	0.97	1.03	0.06	0.93	0.09	0.06	1.04	0.07	0.06	1.02	0.05	0.03	1.00	0.08	0.02	1.02	0.08	0.97	0.03	0.15	0.97	0.09	0.04	1.90	0.14	1.00	0.08	2.86	1.07	1.03	1.04	0.13	1.96	0.11	0.92	0.04	1.10	2.82	0.06	0.11	2.71	0.13	0.01	1.77	1.06	1.02	0.04	0.15	0.92	1.04	0.03	0.98	0.12	1.00	0.08	0.98	0.10	0.12	1.01	0.15	0.94	1.03	1.14	1.92	0.14	0.13	0.95	0.08	1.04	1.04	0.09	0.07	1.29	1.03	0.09	0.15	4.81	0.19	1.00	1.98	0.16	1.95	0.08	2.02	0.09	1.04	0.03	1.07	0.06	1.01	0.09	0.97	0.12	0.12	1.07	0.25	1.03	0.11	2.01	3.88	0.16	0.91	1.04	1.15	0.88	0.13	0.13	0.97	0.13	0.12	0.88	0.13	0.12	0.99	0.11	1.00	0.92	1.03	0.03	1.08	1.02	0.11	0.02	1.97	0.17	0.95	0.03	0.95	2.05	1.09	0.03	0.13	2.85	0.14	0.09	2.18	1.15	2.01	1.02	4.07	1.05	0.10	0.13	2.27	1.05	0.11	0.98	0.89	0.14	0.98	1.00	0.15	0.99	0.16	0.14	0.96	0.09	1.03	1.93	0.09	0.17	0.98	0.06	0.14	1.00	0.14	0.04	1.10	1.00	0.03	0.03	2.08	1.01	0.04	0.07	0.98	0.10	0.11	0.96	0.10	0.09	1.03	0.14	0.11	2.00	0.09	0.10	1.98	0.13	0.14	1.04	0.12	0.15	1.01	0.13	1.85	1.08	0.05	0.10	1.03	0.09	0.04	0.97	0.09	0.94	0.07	0.13	1.88	1.03	0.13	1.12	0.09	0.07	1.10	2.92	0.10	0.07	1.04	0.12	0.99	0.12	3.01	3.17	1.09	0.10	0.17	0.95	0.18	0.13	0.89	0.16	1.02	3.05	0.12	1.00	0.08	1.06	0.12	0.95	0.13	0.17	1.05	0.17	0.09	1.03	0.09	0.06	2.22	0.09	1.02	0.07	2.96	0.11	0.10	1.89	1.06	1.03	2.04	0.16	1.03	0.10	2.09	1.03	0.94	1.01	0.07	0.19	1.02	1.96	1.09	0.96	0.10	0.08	2.12	1.04	0.18	0.97	0.11	0.95	1.05	1.96	0.23	1.96	0.11	0.06	2.95	0.11	1.05	0.16	2.07	0.11	1.96	1.01	0.15	0.14	1.03	0.07	2.10	1.04	0.23	0.11	0.92	0.04	4.86	0.14	1.00	0.07	0.08	0.92	0.17	0.01	1.08	0.10	1.05	0.03	0.06	0.98	0.04	0.03	1.08	3.02	0.14	0.12	0.86	0.01	0.10	1.08	0.06	2.01	2.16	0.18	2.25	0.11	0.15	0.98	0.15	0.19	1.05	1.00	1.10	1.04	1.04	0.17	0.19	0.90	1.96	0.97	1.08	0.10	0.95	0.13	1.05	0.19	0.07	1.04	3.01	1.15	1.07	0.01	0.09	0.94	0.09	0.09	0.92	0.88	0.20	0.04	2.28	2.28
Flow Indexes:	1	3	6	8	10	12	15	18	20	23	26	27	29	29	32	33	35	38	41	44	47	49	52	55	55	57	59	59	59	60	61	62	64	64	66	68	69	69	69	72	72	72	75	75	76	77	80	81	83	85	87	90	92	93	94	95	95	98	100	101	104	105	108	108	108	108	108	110	111	111	113	113	115	115	117	119	121	123	126	128	130	130	131	131	131	131	133	134	135	136	139	142	145	147	148	149	151	152	155	155	157	159	160	160	161	164	164	164	167	167	168	169	169	170	171	171	171	171	172	175	175	176	178	179	181	182	184	187	189	190	190	193	196	199	200	203	203	204	207	210	213	216	216	219	219	222	225	227	227	228	231	234	236	239	239	240	242	245	246	246	246	249	251	253	253	253	254	254	254	255	258	261	263	264	264	264	266	268	270	273	276	279	279	281	283	283	283	286	286	287	288	289	289	291	293	293	294	295	296	299	300	300	301	302	305	305	306	308	310	311	312	312	314	314	317	317	317	319	321	321	323	323	324	327	329	329	330	333	335	335	335	335	335	337	340	343	345	348	351	352	352	352	355	358	360	360	361	361	363	363	366	369	370	371	372	373	376	377	377	378	379	381	383	386	387	387	387	388	389	392	395	396	399	399	400	400
Bases:	tcagAGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTCTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGG
Quality Scores:	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	40	40	40	40	38	38	38	39	39	39	39	39	39	39	34	34	36	30	30	30	39	39	39	39	39	39	39	37	37	37	37	37	37	37	37	37	37	37	38	38	33	33	33	33	33	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	38	38	38	38	38	38	37	37	37	37	37	37	37	38	38	38	38	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	38	38	38	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	38	36	37	38	38	37	37	38	38	38	38	38	38	38	38	38	38	38	38	36	36	36	38	38	38	38	29	29	29	23	28	30	31	32	31	31	31	31	32	32	32	32	32	32	31	31	31	32	32	31	28	25	19	18	18	16

>FLP3FBN01APRWO2
  Run Prefix:   R_2008_12_09_13_51_01_
  Region #:     1
  XY Location:  0176_1782

  Run Name:       R_2008_12_09_13_51_01_FLX04080350_Administrator_PC_Calrestricted
  Analysis Name:  D_2009_06_10_10_15_54_node176_fullProcessingAmplicons
  Full Path:      /srv/seq/454/2008_12_09/R_2008_12_09_13_51_01_FLX04080350_Administrator_PC_Calrestricted/D_2009_06_10_10_15_54_node176_fullProcessingAmplicons/

  Read Header Len:  32
  Name Length:      14
  # of Bases:       276
  Clip Qual Left:   5
  Clip Qual Right:  276
  Clip Adap Left:   0
  Clip Adap Right:  0

Flowgram:	1.06	0.03	1.02	0.05	0.04	0.93	0.05	1.01	0.04	1.07	0.03	1.06	0.06	0.02	0.92	0.13	0.01	0.99	0.06	0.93	0.04	0.05	1.02	0.05	0.10	1.09	1.02	0.07	2.07	0.05	0.05	0.97	1.03	0.06	0.93	0.09	0.06	1.04	0.07	0.06	1.02	0.05	0.03	1.00	0.08	0.02	1.02	0.08	0.97	0.03	0.15	0.97	0.09	0.04	1.90	0.14	1.00	0.08	2.86	1.07	1.03	1.04	0.13	1.96	0.11	0.92	0.04	1.10	2.82	0.06	0.11	2.71	0.13	0.01	1.77	1.06	1.02	0.04	0.15	0.92	1.04	0.03	0.98	0.12	1.00	0.08	0.98	0.10	0.12	1.01	0.15	0.94	1.03	1.14	1.92	0.14	0.13	0.95	0.08	1.04	1.04	0.09	0.07	1.29	1.03	0.09	0.15	4.81	0.19	1.00	1.98	0.16	1.95	0.08	2.02	0.09	1.04	0.03	1.07	0.06	1.01	0.09	0.97	0.12	0.12	1.07	0.25	1.03	0.11	2.01	3.88	0.16	0.91	1.04	1.15	0.88	0.13	0.13	0.97	0.13	0.12	0.88	0.13	0.12	0.99	0.11	1.00	0.92	1.03	0.03	1.08	1.02	0.11	0.02	1.97	0.17	0.95	0.03	0.95	2.05	1.09	0.03	0.13	2.85	0.14	0.09	2.18	1.15	2.01	1.02	4.07	1.05	0.10	0.13	2.27	1.05	0.11	0.98	0.89	0.14	0.98	1.00	0.15	0.99	0.16	0.14	0.96	0.09	1.03	1.93	0.09	0.17	0.98	0.06	0.14	1.00	0.14	0.04	1.10	1.00	0.03	0.03	2.08	1.01	0.04	0.07	0.98	0.10	0.11	0.96	0.10	0.09	1.03	0.14	0.11	2.00	0.09	0.10	1.98	0.13	0.14	1.04	0.12	0.15	1.01	0.13	1.85	1.08	0.05	0.10	1.03	0.09	0.04	0.97	0.09	0.94	0.07	0.13	1.88	1.03	0.13	1.12	0.09	0.07	1.10	2.92	0.10	0.07	1.04	0.12	0.99	0.12	3.01	3.17	1.09	0.10	0.17	0.95	0.18	0.13	0.89	0.16	1.02	3.05	0.12	1.00	0.08	1.06	0.12	0.95	0.13	0.17	1.05	0.17	0.09	1.03	0.09	0.06	2.22	0.09	1.02	0.07	2.96	0.11	0.10	1.89	1.06	1.03	2.04	0.16	1.03	0.10	2.09	1.03	0.94	1.01	0.07	0.19	1.02	1.96	1.09	0.96	0.10	0.08	2.12	1.04	0.18	0.97	0.11	0.95	1.05	1.96	0.23	1.96	0.11	0.06	2.95	0.11	1.05	0.16	2.07	0.11	1.96	1.01	0.15	0.14	1.03	0.07	2.10	1.04	0.23	0.11	0.92	0.04	4.86	0.14	1.00	0.07	0.08	0.92	0.17	0.01	1.08	0.10	1.05	0.03	0.06	0.98	0.04	0.03	1.08	3.02	0.14	0.12	0.86	0.01	0.10	1.08	0.06	2.01	2.16	0.18	2.25	0.11	0.15	0.98	0.15	0.19	1.05	1.00	1.10	1.04	1.04	0.17	0.19	0.90	1.96	0.97	1.08	0.10	0.95	0.13	1.05	0.19	0.07	1.04	3.01	1.15	1.07	0.01	0.09	0.94	0.09	0.09	0.92	0.88	0.20	0.04	2.28	2.28
Flow Indexes:	1	3	6	8	10	12	15	18	20	23	26	27	29	29	32	33	35	38	41	44	47	49	52	55	55	57	59	59	59	60	61	62	64	64	66	68	69	69	69	72	72	72	75	75	76	77	80	81	83	85	87	90	92	93	94	95	95	98	100	101	104	105	108	108	108	108	108	110	111	111	113	113	115	115	117	119	121	123	126	128	130	130	131	131	131	131	133	134	135	136	139	142	145	147	148	149	151	152	155	155	157	159	160	160	161	164	164	164	167	167	168	169	169	170	171	171	171	171	172	175	175	176	178	179	181	182	184	187	189	190	190	193	196	199	200	203	203	204	207	210	213	216	216	219	219	222	225	227	227	228	231	234	236	239	239	240	242	245	246	246	246	249	251	253	253	253	254	254	254	255	258	261	263	264	264	264	266	268	270	273	276	279	279	281	283	283	283	286	286	287	288	289	289	291	293	293	294	295	296	299	300	300	301	302	305	305	306	308	310	311	312	312	314	314	317	317	317	319	321	321	323	323	324	327	329	329	330	333	335	335	335	335	335	337	340	343	345	348	351	352	352	352	355	358	360	360	361	361	363	363	366	369	370	371	372	373	376	377	377	378	379	381	383	386	387	387	387	388	389	392	395	396	399	399	400	400
Bases:	tcagAGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTCTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGG
Quality Scores:	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	40	40	40	40	38	38	38	39	39	39	39	39	39	39	34	34	36	30	30	30	39	39	39	39	39	39	39	37	37	37	37	37	37	37	37	37	37	37	38	38	33	33	33	33	33	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	38	38	38	38	38	38	37	37	37	37	37	37	37	38	38	38	38	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	38	38	38	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	38	36	37	38	38	37	37	38	38	38	38	38	38	38	38	38	38	38	38	36	36	36	38	38	38	38	29	29	29	23	28	30	31	32	31	31	31	31	32	32	32	32	32	32	31	31	31	32	32	31	28	25	19	18	18	16

>FLP3FBN01APRWO3
  Run Prefix:   R_2008_12_09_13_51_01_
  Region #:     1
  XY Location:  0176_1782

  Run Name:       R_2008_12_09_13_51_01_FLX04080350_Administrator_PC_Calrestricted
  Analysis Name:  D_2009_06_10_10_15_54_node176_fullProcessingAmplicons
  Full Path:      /srv/seq/454/2008_12_09/R_2008_12_09_13_51_01_FLX04080350_Administrator_PC_Calrestricted/D_2009_06_10_10_15_54_node176_fullProcessingAmplicons/

  Read Header Len:  32
  Name Length:      14
  # of Bases:       276
  Clip Qual Left:   5
  Clip Qual Right:  276
  Clip Adap Left:   0
  Clip Adap Right:  0

Flowgram:	1.06	0.03	1.02	0.05	0.04	0.93	0.05	1.01	0.04	1.07	0.03	1.06	0.06	0.02	0.92	0.13	0.01	0.99	0.06	0.93	0.04	0.05	1.02	0.05	0.10	1.09	1.02	0.07	2.07	0.05	0.05	0.97	1.03	0.06	0.93	0.09	0.06	1.04	0.07	0.06	1.02	0.05	0.03	1.00	0.08	0.02	1.02	0.08	0.97	0.03	0.15	0.97	0.09	0.04	1.90	0.14	1.00	0.08	2.86	1.07	1.03	1.04	0.13	1.96	0.11	0.92	0.04	1.10	2.82	0.06	0.11	2.71	0.13	0.01	1.77	1.06	1.02	0.04	0.15	0.92	1.04	0.03	0.98	0.12	1.00	0.08	0.98	0.10	0.12	1.01	0.15	0.94	1.03	1.14	1.92	0.14	0.13	0.95	0.08	1.04	1.04	0.09	0.07	1.29	1.03	0.09	0.15	4.81	0.19	1.00	1.98	0.16	1.95	0.08	2.02	0.09	1.04	0.03	1.07	0.06	1.01	0.09	0.97	0.12	0.12	1.07	0.25	1.03	0.11	2.01	3.88	0.16	0.91	1.04	1.15	0.88	0.13	0.13	0.97	0.13	0.12	0.88	0.13	0.12	0.99	0.11	1.00	0.92	1.03	0.03	1.08	1.02	0.11	0.02	1.97	0.17	0.95	0.03	0.95	2.05	1.09	0.03	0.13	2.85	0.14	0.09	2.18	1.15	2.01	1.02	4.07	1.05	0.10	0.13	2.27	1.05	0.11	0.98	0.89	0.14	0.98	1.00	0.15	0.99	0.16	0.14	0.96	0.09	1.03	1.93	0.09	0.17	0.98	0.06	0.14	1.00	0.14	0.04	1.10	1.00	0.03	0.03	2.08	1.01	0.04	0.07	0.98	0.10	0.11	0.96	0.10	0.09	1.03	0.14	0.11	2.00	0.09	0.10	1.98	0.13	0.14	1.04	0.12	0.15	1.01	0.13	1.85	1.08	0.05	0.10	1.03	0.09	0.04	0.97	0.09	0.94	0.07	0.13	1.88	1.03	0.13	1.12	0.09	0.07	1.10	2.92	0.10	0.07	1.04	0.12	0.99	0.12	3.01	3.17	1.09	0.10	0.17	0.95	0.18	0.13	0.89	0.16	1.02	3.05	0.12	1.00	0.08	1.06	0.12	0.95	0.13	0.17	1.05	0.17	0.09	1.03	0.09	0.06	2.22	0.09	1.02	0.07	2.96	0.11	0.10	1.89	1.06	1.03	2.04	0.16	1.03	0.10	2.09	1.03	0.94	1.01	0.07	0.19	1.02	1.96	1.09	0.96	0.10	0.08	2.12	1.04	0.18	0.97	0.11	0.95	1.05	1.96	0.23	1.96	0.11	0.06	2.95	0.11	1.05	0.16	2.07	0.11	1.96	1.01	0.15	0.14	1.03	0.07	2.10	1.04	0.23	0.11	0.92	0.04	4.86	0.14	1.00	0.07	0.08	0.92	0.17	0.01	1.08	0.10	1.05	0.03	0.06	0.98	0.04	0.03	1.08	3.02	0.14	0.12	0.86	0.01	0.10	1.08	0.06	2.01	2.16	0.18	2.25	0.11	0.15	0.98	0.15	0.19	1.05	1.00	1.10	1.04	1.04	0.17	0.19	0.90	1.96	0.97	1.08	0.10	0.95	0.13	1.05	0.19	0.07	1.04	3.01	1.15	1.07	0.01	0.09	0.94	0.09	0.09	0.92	0.88	0.20	0.04	2.28	2.28
Flow Indexes:	1	3	6	8	10	12	15	18	20	23	26	27	29	29	32	33	35	38	41	44	47	49	52	55	55	57	59	59	59	60	61	62	64	64	66	68	69	69	69	72	72	72	75	75	76	77	80	81	83	85	87	90	92	93	94	95	95	98	100	101	104	105	108	108	108	108	108	110	111	111	113	113	115	115	117	119	121	123	126	128	130	130	131	131	131	131	133	134	135	136	139	142	145	147	148	149	151	152	155	155	157	159	160	160	161	164	164	164	167	167	168	169	169	170	171	171	171	171	172	175	175	176	178	179	181	182	184	187	189	190	190	193	196	199	200	203	203	204	207	210	213	216	216	219	219	222	225	227	227	228	231	234	236	239	239	240	242	245	246	246	246	249	251	253	253	253	254	254	254	255	258	261	263	264	264	264	266	268	270	273	276	279	279	281	283	283	283	286	286	287	288	289	289	291	293	293	294	295	296	299	300	300	301	302	305	305	306	308	310	311	312	312	314	314	317	317	317	319	321	321	323	323	324	327	329	329	330	333	335	335	335	335	335	337	340	343	345	348	351	352	352	352	355	358	360	360	361	361	363	363	366	369	370	371	372	373	376	377	377	378	379	381	383	386	387	387	387	388	389	392	395	396	399	399	400	400
Bases:	tcagAGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTCTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGG
Quality Scores:	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	40	40	40	40	38	38	38	39	39	39	39	39	39	39	34	34	36	30	30	30	39	39	39	39	39	39	39	37	37	37	37	37	37	37	37	37	37	37	38	38	33	33	33	33	33	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	38	38	38	38	38	38	37	37	37	37	37	37	37	38	38	38	38	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	38	38	38	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	37	38	36	37	38	38	37	37	38	38	38	38	38	38	38	38	38	38	38	38	36	36	36	38	38	38	38	29	29	29	23	28	30	31	32	31	31	31	31	32	32	32	32	32	32	31	31	31	32	32	31	28	25	19	18	18	16


"""

fasting_seqs_subset = """>PC.634_1 FLP3FBN01ELBSX orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTTACCCTCTCAGGCCGGCTACGCATCATCGCCTTGGTGGGCCGTTACCTCACCAACTAGCTAATGCGCCGCAGGTCCATCCATGTTCACGCCTTGATGGGCGCTTTAATATACTGAGCATGCGCTCTGTATACCTATCCGGTTTTAGCTACCGTTTCCAGCAGTTATCCCGGACACATGGGCTAGG
>PC.634_2 FLP3FBN01EG8AX orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGCCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGGGCCGTTACCCCGCCAACAACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGAAGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCA
>PC.354_3 FLP3FBN01EEWKD orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAGTCTCTTAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGACATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCTTGGG
>PC.481_4 FLP3FBN01DEHK3 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGACTTGGTGAGCCGTTACCTCACCAACTATCTAATCAGACGCGAGCCCATCTTTCAGCGGATTGCTCCTTTGGTATTCCGGCGATGCCGCCAAAATCATTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGAAAGGCAGGTTGCTCACG
>PC.634_5 FLP3FBN01DGFYQ orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGTCCATCTCACACCGATTAATCTTTTTCCAACCAGAGCATGCGCCCCTGTTGGCTTATGCGGTATTAGCGGTCGTTTCCAACTGTTATCCCCCTGTGTGAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.636_6 FLP3FBN01A55WZ orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGAGAAAATGCTCACTCATGATACCATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCAGGTTGGATACGCGTTACTCACCCGTGCGCCGG
>PC.634_7 FLP3FBN01D7O1S orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGATAACCGAAATTCTTTAATAGTGAAACCATGCGGAAATACTATACTATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTTATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCATC
>PC.634_8 FLP3FBN01CV5ZU orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAACCAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCTCTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCG
>PC.634_9 FLP3FBN01DQ783 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.634_10 FLP3FBN01DDPFF orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCAGGTTGGGTACGCGTTACTC
>PC.634_11 FLP3FBN01CPD70 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGACCGTGTCTCAGTTCCAATGTGGGGGGCCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGGGCCGTTACCCCGCCAACAACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGAAGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.593_12 FLP3FBN01BBAE6 orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCGGCACCGGGATGCCCCGATGCCGCGTCACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTGGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCG
>PC.355_13 FLP3FBN01AWYZD orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGACATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCTTGGG
>PC.634_14 FLP3FBN01AM0P3 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGCCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATCAGACGCGGGCCCATCCCGTACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTTACTCACCCG
>PC.635_15 FLP3FBN01BO4IE orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGACTTGGTAGGCCGTTACCCCACCAACTATCTAATCAGACGCAAGCCCATCTATCAGCGGATTGCTCCTTTCCCATTTATATCATGTGATATTCATGGCATATGCGGTATTAGCAGTCATTTCTAACTGTTGTTCCCCTCTGATAGG
>PC.635_16 FLP3FBN01BPX14 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGTCACGCGAGCCTATCCTCATCCGACGGATCTTTAGATGGAACCAGATGCCTGATTCCATCGCCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATCGAGGGCAAGTTGCATACGCG
>PC.356_17 FLP3FBN01DB5I7 orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCCTGCACCGCCGGAGCTTCCCCCGCCGCCCCATGCGGGGCTGCGGGCATATGCGGTATTAGCAGCCGTTTCCGGCTGTTGTCCCCCAGTGCAGGGCAGGTTGCCCACGCGTTACTCACCCGTCCG
>PC.634_18 FLP3FBN01AK9OO orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTCGAGCCGTTACCTCACCAACTAGCTAATGCGCCGCGGGCCCATCTCATAGCGGATTACTCCTTTAATTGCTACTTCATGCGAAGCTACAATCTTATGCGGTATTAATCTCCCTTTCGGAAGGCTATTCCCCTCTATGAGGTCAGGTTG
>PC.634_19 FLP3FBN01ANGF2 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGTCACGCGAGCCTATCCTCATCCGACGGATCTTTAGATGGAACCAGATGCCTGATTCCATCGCCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCAAGTTGCTCACGCG
>PC.634_20 FLP3FBN01AF994 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGTCGGTTTGGTGGGCCGTTACCCCGCCAACTGCCTAATGGAACGCATGCCTATCTATCAGCGATGAATCTTTAGCAAATATCCCCATGCGGGGCCCCTGCTTCATGCGGTATTAGTCCGACTTTCGCCGGGTTATCCCCTCCTGATAGGTAAGTTGCATACGCGTTACTTCACCCGTCGCGG
>PC.634_21 FLP3FBN01AHXU8 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTACTGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCGGGGATGCCCCCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGGCAGAGAG
>PC.593_22 FLP3FBN01APRWO orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGGCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTCTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGG
>PC.593_23 FLP3FBN01APRWO2 orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGGCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTCTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGG
>PC.593_24 FLP3FBN01APRWO3 orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGGCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTCTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGG
"""

# run unit tests if run from command-line
if __name__ == '__main__':
    main()
