#!/usr/bin/env python
# File created on 13 Jul 2012
from __future__ import division

__author__ = "Greg Caporaso"
__copyright__ = "Copyright 2011, The QIIME project"
__credits__ = ["Greg Caporaso", "Jai Ram Rideout"]
__license__ = "GPL"
__version__ = "1.9.1"
__maintainer__ = "Greg Caporaso"
__email__ = "gregcaporaso@gmail.com"


from glob import glob
from shutil import rmtree
from os import close
from os.path import exists, join
from tempfile import mkstemp, mkdtemp

from skbio.util import remove_files
from unittest import TestCase, main
from numpy.testing import assert_almost_equal
from biom.parse import parse_biom_table
from qiime.parse import parse_distmat
from qiime.util import get_qiime_temp_dir
from qiime.test import initiate_timeout, disable_timeout
from qiime.parallel.beta_diversity import (ParallelBetaDiversitySingle,
                                           ParallelBetaDiversityMultiple)


class ParallelBetaDiversityTests(TestCase):

    def setUp(self):

        self.files_to_remove = []
        self.dirs_to_remove = []

        # Create example output directory
        tmp_dir = get_qiime_temp_dir()
        self.test_out = mkdtemp(dir=tmp_dir,
                                prefix='qiime_parallel_tests_',
                                suffix='')
        self.dirs_to_remove.append(self.test_out)

        fd, self.input1_fp = mkstemp(dir=self.test_out,
                                    prefix='qiime_inseqs',
                                    suffix='.biom')
        close(fd)
        input1_f = open(self.input1_fp, 'w')
        input1_f.write(input1)
        input1_f.close()
        self.files_to_remove.append(self.input1_fp)

        fd, self.input2_fp = mkstemp(dir=self.test_out,
                                    prefix='qiime_inseqs',
                                    suffix='.biom')
        close(fd)
        input2_f = open(self.input2_fp, 'w')
        input2_f.write(input2)
        input2_f.close()
        self.files_to_remove.append(self.input2_fp)

        self.input1_fps = [self.input1_fp, self.input2_fp]

        fd, self.tree_fp = mkstemp(dir=self.test_out,
                                  prefix='qiime',
                                  suffix='.tre')
        close(fd)
        tree_f = open(self.tree_fp, 'w')
        tree_f.write(tree)
        tree_f.close()
        self.files_to_remove.append(self.tree_fp)

        # Define number of seconds a test can run for before timing out
        # and failing
        initiate_timeout(60)

    def tearDown(self):

        disable_timeout()
        remove_files(self.files_to_remove)
        # remove directories last, so we don't get errors
        # trying to remove files which may be in the directories
        for d in self.dirs_to_remove:
            if exists(d):
                rmtree(d)


class ParallelBetaDiversitySingleTests(ParallelBetaDiversityTests):

    def test_parallel_beta_diversity(self):
        """ parallel beta diveristy functions in single file mode """
        params = {'metrics': 'weighted_unifrac,unweighted_unifrac',
                  'tree_path': self.tree_fp,
                  'jobs_to_start': 3,
                  'full_tree': False
                  }
        app = ParallelBetaDiversitySingle()
        r = app(self.input1_fp,
                self.test_out,
                params,
                job_prefix='BTEST',
                poll_directly=True,
                suppress_submit_jobs=False)
        input_sample_ids = parse_biom_table(
            open(self.input1_fp, 'U')).ids()
        dm_fps = glob(join(self.test_out, '*weighted_unifrac*'))
        for dm_fp in dm_fps:
            dm_sample_ids = parse_distmat(open(dm_fp))[0]
            self.assertItemsEqual(dm_sample_ids, input_sample_ids)

    def test_parallel_beta_diversity_wo_tree(self):
        """ parallel beta diveristy functions in single file mode """
        params = {'metrics': 'bray_curtis',
                  'tree_path': None,
                  'jobs_to_start': 3,
                  'full_tree': False
                  }
        app = ParallelBetaDiversitySingle()
        r = app(self.input1_fp,
                self.test_out,
                params,
                job_prefix='BTEST',
                poll_directly=True,
                suppress_submit_jobs=False)
        input_sample_ids = parse_biom_table(
            open(self.input1_fp, 'U')).ids()
        dm_fps = glob(join(self.test_out, 'bray_curtis*'))
        for dm_fp in dm_fps:
            dm_sample_ids = parse_distmat(open(dm_fp))[0]
            self.assertItemsEqual(dm_sample_ids, input_sample_ids)


class ParallelBetaDiversityMultipleTests(ParallelBetaDiversityTests):

    def test_parallel_beta_diversity(self):
        """ parallel beta diveristy functions in multiple file mode """
        params = {'metrics': 'weighted_unifrac',
                  'tree_path': self.tree_fp,
                  'jobs_to_start': 3,
                  'full_tree': False
                  }
        app = ParallelBetaDiversityMultiple()
        r = app(self.input1_fps,
                self.test_out,
                params,
                job_prefix='BTEST',
                poll_directly=True,
                suppress_submit_jobs=False)
        input_sample_ids_l = []
        dm_sample_ids_l = []
        for input_fp, dm_fp in zip(self.input1_fps,
                                   glob(join(self.test_out, '*weighted_unifrac*'))):
            input_sample_ids_l.append(
                list(parse_biom_table(open(input_fp, 'U')).ids()))
            dm_sample_ids_l.append(parse_distmat(open(dm_fp))[0])
        dm_sample_ids_l.sort()
        input_sample_ids_l.sort()
        self.assertEqual(dm_sample_ids_l, input_sample_ids_l)
        # just confirming that there are different sample ids in
        # the two sets of results
        self.assertNotEqual(dm_sample_ids_l[0], dm_sample_ids_l[1])

    def test_parallel_beta_diversity_wo_tree(self):
        """ parallel beta diveristy functions in multiple file mode without tree
        """
        params = {'metrics': 'bray_curtis',
                  'tree_path': None,
                  'jobs_to_start': 3,
                  'full_tree': False
                  }
        app = ParallelBetaDiversityMultiple()
        r = app(self.input1_fps,
                self.test_out,
                params,
                job_prefix='BTEST',
                poll_directly=True,
                suppress_submit_jobs=False)
        input_sample_ids_l = []
        dm_sample_ids_l = []
        for input_fp, dm_fp in zip(self.input1_fps,
                                   glob(join(self.test_out, '*bray_curtis*'))):
            input_sample_ids_l.append(
                list(parse_biom_table(open(input_fp, 'U')).ids()))
            dm_sample_ids_l.append(parse_distmat(open(dm_fp))[0])
        dm_sample_ids_l.sort()
        input_sample_ids_l.sort()
        self.assertEqual(dm_sample_ids_l, input_sample_ids_l)
        # just confirming that there are different sample ids in
        # the two sets of results
        self.assertNotEqual(dm_sample_ids_l[0], dm_sample_ids_l[1])


input1 = """{"rows": [{"id": "0", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "1", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Bacilli", "o__Bacillales", "f__Staphylococcaceae"]}}, {"id": "2", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "3", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "4", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "5", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Bacilli", "o__Lactobacillales", "f__Lactobacillaceae"]}}, {"id": "6", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "7", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "8", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "9", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Tenericutes", "c__Erysipelotrichi", "o__Erysipelotrichales", "f__Erysipelotrichaceae"]}}, {"id": "10", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "11", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "12", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "13", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Bacilli", "o__Lactobacillales", "f__Streptococcaceae"]}}, {"id": "14", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Actinobacteria", "c__Actinobacteria", "o__Coriobacteriales", "f__Coriobacteriaceae"]}}, {"id": "15", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "16", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Actinobacteria", "c__Actinobacteria", "o__Coriobacteriales", "f__Coriobacteriaceae"]}}, {"id": "17", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Porphyromonadaceae"]}}, {"id": "18", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "19", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "20", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "21", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "22", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Proteobacteria", "c__Deltaproteobacteria", "o__Desulfovibrionales", "f__Desulfovibrionaceae"]}}, {"id": "23", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "24", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "25", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "26", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "27", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__TM7", "c__TM7-3", "o__CW040", "f__F16"]}}, {"id": "28", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "29", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "30", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "31", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "32", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "33", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Actinobacteria", "c__Actinobacteria", "o__Coriobacteriales", "f__Coriobacteriaceae"]}}, {"id": "34", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Actinobacteria", "c__Actinobacteria", "o__Coriobacteriales", "f__Coriobacteriaceae"]}}, {"id": "35", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "36", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Tenericutes", "c__Erysipelotrichi", "o__Erysipelotrichales", "f__Erysipelotrichaceae"]}}, {"id": "37", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "38", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "39", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "40", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "41", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "42", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "43", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "44", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "45", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Deferribacteres", "c__Deferribacteres", "o__Deferribacterales", "f__Deferribacteraceae"]}}, {"id": "46", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "47", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "48", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "49", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "50", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "51", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "52", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "53", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "54", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "55", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Tenericutes", "c__Erysipelotrichi", "o__Erysipelotrichales", "f__Erysipelotrichaceae"]}}, {"id": "56", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "57", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "58", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "59", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales"]}}, {"id": "60", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "61", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "62", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "63", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "64", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "65", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "66", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "67", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "68", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "69", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "70", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "71", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "72", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "73", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Tenericutes", "c__Erysipelotrichi", "o__Erysipelotrichales", "f__Erysipelotrichaceae"]}}, {"id": "74", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "75", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "76", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "77", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "78", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "79", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "80", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "81", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Actinobacteria", "c__Actinobacteria", "o__Coriobacteriales", "f__Coriobacteriaceae"]}}, {"id": "82", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Porphyromonadaceae"]}}, {"id": "83", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "84", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "85", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "86", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "87", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Tenericutes", "c__Mollicutes", "o__RF39", "f__"]}}, {"id": "88", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "89", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "90", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Porphyromonadaceae"]}}, {"id": "91", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "92", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "93", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "94", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Porphyromonadaceae"]}}, {"id": "95", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "96", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "97", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "98", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "99", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "100", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "101", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "102", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Bacilli", "o__Lactobacillales", "f__Lactobacillaceae"]}}, {"id": "103", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "104", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "105", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Rikenellaceae"]}}, {"id": "106", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "107", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "108", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Deferribacteres", "c__Deferribacteres", "o__Deferribacterales", "f__Deferribacteraceae"]}}, {"id": "109", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "110", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "111", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "112", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "113", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "114", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "115", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "116", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "117", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__"]}}, {"id": "118", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "119", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales"]}}, {"id": "120", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales"]}}, {"id": "121", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "122", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "123", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Tenericutes", "c__Mollicutes", "o__RF39", "f__"]}}, {"id": "124", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "125", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "126", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "127", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "128", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "129", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "130", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Peptococcaceae"]}}, {"id": "131", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "132", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "133", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "134", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "135", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "136", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "137", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "138", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "139", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "140", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "141", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "142", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "143", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "144", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "145", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "146", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "147", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "148", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Rikenellaceae"]}}, {"id": "149", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Rikenellaceae"]}}, {"id": "150", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales"]}}, {"id": "151", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "152", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "153", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Rikenellaceae"]}}, {"id": "154", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "155", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "156", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "157", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "158", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "159", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "160", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "161", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "162", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "163", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales"]}}, {"id": "164", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "165", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "166", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "167", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "168", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "169", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "170", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "171", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales"]}}, {"id": "172", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "173", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "174", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "175", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "176", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Bacilli", "o__Lactobacillales", "f__Lactobacillaceae"]}}, {"id": "177", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "178", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Tenericutes", "c__Erysipelotrichi", "o__Erysipelotrichales", "f__Erysipelotrichaceae"]}}, {"id": "179", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "180", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "181", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "182", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "183", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "184", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__"]}}, {"id": "185", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "186", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "187", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "188", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "189", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "190", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales"]}}, {"id": "191", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "192", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "193", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "194", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "195", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "196", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "197", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "198", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "199", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Peptococcaceae"]}}, {"id": "200", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Catabacteriaceae"]}}, {"id": "201", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "202", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "203", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "204", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "205", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "206", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Catabacteriaceae"]}}, {"id": "207", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "208", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "209", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "210", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "211", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Rikenellaceae"]}}, {"id": "212", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Catabacteriaceae"]}}, {"id": "213", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "214", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes"]}}, {"id": "215", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "216", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "217", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "218", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales"]}}, {"id": "219", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "220", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "221", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "222", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "223", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "224", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "225", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Actinobacteria", "c__Actinobacteria", "o__Coriobacteriales", "f__Coriobacteriaceae"]}}, {"id": "226", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "227", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "228", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Peptococcaceae"]}}, {"id": "229", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "230", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "231", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Tenericutes", "c__Erysipelotrichi", "o__Erysipelotrichales", "f__Erysipelotrichaceae"]}}, {"id": "232", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "233", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "234", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Rikenellaceae"]}}, {"id": "235", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "236", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "237", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "238", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "239", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "240", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Bacilli", "o__Lactobacillales", "f__Lactobacillaceae"]}}, {"id": "241", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "242", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "243", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "244", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__"]}}, {"id": "245", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "246", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "247", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "248", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Rikenellaceae"]}}, {"id": "249", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "250", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "251", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "252", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "253", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "254", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "255", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "256", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "257", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "258", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Prevotellaceae"]}}, {"id": "259", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "260", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "261", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "262", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "263", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "264", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "265", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "266", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Rikenellaceae"]}}, {"id": "267", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "268", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "269", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "270", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "271", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Actinobacteria", "c__Actinobacteria", "o__Coriobacteriales", "f__Coriobacteriaceae"]}}, {"id": "272", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "273", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "274", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "275", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "276", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "277", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "278", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "279", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "280", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "281", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales"]}}, {"id": "282", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Proteobacteria", "c__Deltaproteobacteria", "o__Desulfovibrionales", "f__Desulfovibrionaceae"]}}, {"id": "283", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Porphyromonadaceae"]}}, {"id": "284", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales"]}}, {"id": "285", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "286", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Rikenellaceae"]}}, {"id": "287", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "288", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "289", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Porphyromonadaceae"]}}, {"id": "290", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "291", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Bacilli", "o__Bacillales", "f__Staphylococcaceae"]}}, {"id": "292", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "293", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "294", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "295", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "296", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "297", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "298", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "299", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "300", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "301", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "302", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "303", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "304", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "305", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "306", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "307", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "308", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Proteobacteria", "c__Epsilonproteobacteria", "o__Campylobacterales", "f__Helicobacteraceae"]}}, {"id": "309", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "310", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "311", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "312", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "313", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "314", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "315", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "316", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "317", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "318", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "319", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "320", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "321", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "322", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "323", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Catabacteriaceae"]}}, {"id": "324", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "325", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "326", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "327", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "328", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Tenericutes", "c__Erysipelotrichi", "o__Erysipelotrichales", "f__Erysipelotrichaceae"]}}, {"id": "329", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "330", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Actinobacteria", "c__Actinobacteria", "o__Coriobacteriales", "f__Coriobacteriaceae"]}}, {"id": "331", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "332", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Catabacteriaceae"]}}, {"id": "333", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "334", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "335", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "336", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "337", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "338", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Rikenellaceae"]}}, {"id": "339", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "340", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "341", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes"]}}, {"id": "342", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "343", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "344", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "345", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "346", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "347", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes"]}}, {"id": "348", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "349", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "350", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "351", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "352", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales"]}}, {"id": "353", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "354", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Clostridiales Family XIII. Incertae Sedis"]}}, {"id": "355", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "356", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "357", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "358", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__"]}}, {"id": "359", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "360", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Clostridiaceae"]}}, {"id": "361", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "362", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "363", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "364", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "365", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Tenericutes", "c__Erysipelotrichi", "o__Erysipelotrichales", "f__Erysipelotrichaceae"]}}, {"id": "366", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "367", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "368", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__"]}}, {"id": "369", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "370", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "371", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "372", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Rikenellaceae"]}}, {"id": "373", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes"]}}, {"id": "374", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "375", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "376", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "377", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "378", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Rikenellaceae"]}}, {"id": "379", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "380", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "381", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "382", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "383", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "384", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "385", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "386", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "387", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "388", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "389", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "390", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "391", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "392", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "393", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "394", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "395", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Proteobacteria", "c__Deltaproteobacteria", "o__Desulfovibrionales", "f__Desulfovibrionaceae"]}}, {"id": "396", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__"]}}, {"id": "397", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "398", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Porphyromonadaceae"]}}, {"id": "399", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "400", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "401", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "402", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "403", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "404", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "405", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "406", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "407", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Tenericutes", "c__Erysipelotrichi", "o__Erysipelotrichales", "f__Erysipelotrichaceae"]}}, {"id": "408", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "409", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Bacilli", "o__Turicibacterales", "f__Turicibacteraceae"]}}, {"id": "410", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "411", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "412", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "413", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "414", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "415", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "416", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "417", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}], "format": "Biological Observation Matrix 0.9.1-dev", "data": [[0, 0, 1.0], [1, 0, 1.0], [2, 1, 1.0], [3, 2, 1.0], [4, 1, 1.0], [4, 3, 1.0], [5, 4, 1.0], [6, 0, 1.0], [6, 2, 1.0], [6, 3, 1.0], [6, 5, 1.0], [7, 1, 1.0], [8, 4, 1.0], [9, 4, 1.0], [10, 2, 2.0], [10, 6, 1.0], [11, 3, 1.0], [11, 5, 1.0], [12, 3, 1.0], [13, 7, 1.0], [14, 8, 1.0], [15, 5, 1.0], [16, 8, 1.0], [17, 8, 1.0], [18, 1, 1.0], [18, 2, 1.0], [19, 7, 1.0], [20, 7, 1.0], [21, 3, 2.0], [22, 3, 1.0], [23, 1, 1.0], [24, 1, 1.0], [24, 3, 1.0], [25, 7, 2.0], [26, 8, 1.0], [27, 8, 2.0], [28, 1, 1.0], [28, 3, 1.0], [29, 5, 2.0], [30, 3, 1.0], [30, 5, 1.0], [31, 3, 3.0], [31, 7, 6.0], [32, 1, 3.0], [33, 3, 1.0], [34, 7, 2.0], [35, 7, 1.0], [36, 7, 1.0], [37, 2, 1.0], [37, 5, 1.0], [38, 7, 5.0], [39, 7, 1.0], [40, 1, 1.0], [40, 7, 2.0], [41, 7, 2.0], [42, 2, 16.0], [42, 4, 12.0], [43, 4, 1.0], [44, 1, 1.0], [45, 0, 6.0], [45, 3, 2.0], [45, 7, 3.0], [45, 8, 5.0], [46, 4, 1.0], [47, 1, 1.0], [48, 4, 1.0], [49, 2, 1.0], [50, 8, 1.0], [51, 7, 1.0], [52, 7, 1.0], [53, 0, 37.0], [53, 1, 1.0], [53, 3, 10.0], [53, 8, 4.0], [54, 6, 1.0], [55, 2, 1.0], [56, 0, 5.0], [56, 1, 2.0], [56, 2, 1.0], [56, 3, 4.0], [56, 4, 1.0], [56, 5, 1.0], [56, 6, 3.0], [56, 7, 9.0], [56, 8, 2.0], [57, 7, 1.0], [58, 0, 1.0], [59, 7, 1.0], [60, 0, 1.0], [60, 1, 2.0], [60, 2, 2.0], [60, 3, 1.0], [60, 4, 24.0], [60, 5, 10.0], [60, 6, 1.0], [61, 6, 1.0], [62, 1, 1.0], [63, 1, 1.0], [64, 6, 1.0], [65, 7, 1.0], [66, 7, 1.0], [67, 4, 1.0], [68, 0, 2.0], [68, 3, 3.0], [68, 8, 2.0], [69, 0, 1.0], [69, 3, 4.0], [69, 4, 2.0], [69, 7, 2.0], [69, 8, 1.0], [70, 6, 1.0], [71, 0, 2.0], [71, 3, 2.0], [72, 2, 1.0], [73, 3, 1.0], [74, 5, 1.0], [75, 5, 1.0], [76, 5, 1.0], [77, 6, 1.0], [78, 0, 19.0], [78, 1, 1.0], [78, 3, 2.0], [78, 8, 1.0], [79, 1, 1.0], [80, 1, 1.0], [81, 1, 1.0], [82, 3, 3.0], [82, 8, 1.0], [83, 3, 1.0], [84, 3, 1.0], [85, 4, 1.0], [86, 0, 1.0], [86, 3, 2.0], [86, 4, 1.0], [86, 5, 4.0], [86, 6, 4.0], [86, 7, 2.0], [87, 5, 1.0], [88, 1, 6.0], [88, 2, 1.0], [88, 5, 2.0], [88, 6, 4.0], [89, 2, 2.0], [90, 0, 1.0], [91, 5, 1.0], [92, 2, 1.0], [92, 5, 2.0], [93, 8, 1.0], [94, 1, 1.0], [94, 3, 1.0], [94, 8, 3.0], [95, 3, 1.0], [96, 1, 1.0], [97, 7, 1.0], [98, 7, 1.0], [99, 0, 1.0], [100, 6, 1.0], [101, 6, 1.0], [102, 1, 2.0], [102, 2, 1.0], [102, 4, 4.0], [102, 6, 1.0], [103, 8, 1.0], [104, 7, 1.0], [105, 8, 5.0], [106, 7, 1.0], [107, 4, 1.0], [108, 0, 1.0], [109, 2, 1.0], [110, 1, 1.0], [110, 2, 1.0], [110, 7, 1.0], [111, 1, 1.0], [111, 5, 1.0], [112, 6, 1.0], [113, 5, 1.0], [114, 3, 1.0], [114, 8, 2.0], [115, 2, 1.0], [116, 0, 1.0], [116, 6, 1.0], [117, 6, 1.0], [118, 2, 1.0], [118, 5, 1.0], [118, 6, 3.0], [119, 3, 1.0], [120, 3, 1.0], [120, 5, 1.0], [121, 2, 1.0], [122, 2, 1.0], [123, 7, 1.0], [124, 2, 2.0], [124, 5, 2.0], [124, 6, 2.0], [125, 7, 1.0], [126, 1, 1.0], [127, 4, 1.0], [128, 4, 1.0], [128, 5, 1.0], [128, 6, 1.0], [129, 2, 1.0], [130, 2, 1.0], [130, 3, 1.0], [131, 2, 1.0], [132, 8, 1.0], [133, 8, 1.0], [134, 1, 1.0], [134, 2, 1.0], [134, 6, 4.0], [135, 2, 1.0], [136, 2, 1.0], [137, 7, 1.0], [138, 7, 1.0], [139, 3, 1.0], [139, 5, 2.0], [140, 3, 1.0], [141, 5, 1.0], [142, 1, 6.0], [142, 3, 1.0], [143, 7, 1.0], [144, 0, 1.0], [145, 1, 2.0], [145, 2, 4.0], [145, 3, 2.0], [145, 6, 9.0], [146, 1, 1.0], [147, 3, 1.0], [148, 0, 1.0], [149, 0, 2.0], [149, 1, 1.0], [149, 3, 5.0], [149, 8, 1.0], [150, 0, 1.0], [151, 8, 1.0], [152, 8, 1.0], [153, 3, 1.0], [154, 7, 1.0], [155, 1, 1.0], [155, 2, 1.0], [156, 6, 1.0], [157, 6, 2.0], [158, 0, 1.0], [158, 3, 1.0], [159, 0, 1.0], [159, 3, 1.0], [160, 5, 1.0], [161, 2, 6.0], [161, 4, 2.0], [161, 5, 4.0], [162, 2, 1.0], [162, 5, 1.0], [163, 1, 1.0], [163, 2, 1.0], [163, 5, 1.0], [164, 2, 2.0], [164, 4, 1.0], [164, 5, 8.0], [164, 6, 3.0], [165, 3, 1.0], [166, 1, 1.0], [167, 3, 1.0], [168, 4, 1.0], [169, 0, 1.0], [170, 0, 1.0], [170, 8, 1.0], [171, 2, 1.0], [171, 4, 10.0], [172, 8, 2.0], [173, 3, 1.0], [173, 5, 1.0], [173, 7, 2.0], [174, 7, 1.0], [175, 4, 9.0], [175, 7, 1.0], [176, 1, 1.0], [176, 2, 14.0], [176, 5, 14.0], [176, 6, 1.0], [177, 1, 1.0], [178, 0, 2.0], [178, 4, 4.0], [179, 1, 1.0], [180, 1, 1.0], [181, 7, 2.0], [182, 3, 1.0], [183, 5, 2.0], [184, 6, 1.0], [185, 0, 1.0], [185, 1, 1.0], [185, 3, 1.0], [185, 7, 1.0], [186, 3, 1.0], [187, 2, 1.0], [187, 4, 1.0], [188, 5, 1.0], [189, 5, 1.0], [190, 5, 1.0], [191, 5, 1.0], [192, 5, 1.0], [193, 8, 1.0], [194, 5, 1.0], [195, 4, 1.0], [196, 1, 1.0], [197, 3, 1.0], [198, 5, 1.0], [198, 6, 2.0], [199, 7, 1.0], [200, 7, 1.0], [201, 1, 5.0], [201, 2, 2.0], [201, 6, 1.0], [201, 7, 1.0], [202, 2, 29.0], [202, 5, 10.0], [202, 6, 1.0], [203, 3, 1.0], [204, 5, 1.0], [205, 7, 1.0], [206, 0, 1.0], [207, 0, 1.0], [208, 5, 2.0], [209, 1, 1.0], [210, 8, 13.0], [211, 1, 2.0], [212, 7, 1.0], [213, 5, 1.0], [214, 3, 1.0], [214, 7, 2.0], [215, 5, 1.0], [216, 3, 5.0], [216, 4, 2.0], [217, 1, 1.0], [217, 4, 4.0], [218, 2, 1.0], [218, 7, 4.0], [219, 1, 1.0], [219, 2, 2.0], [219, 5, 2.0], [220, 1, 4.0], [221, 5, 1.0], [222, 1, 1.0], [223, 0, 1.0], [224, 2, 1.0], [225, 8, 1.0], [226, 1, 1.0], [226, 2, 2.0], [226, 6, 2.0], [227, 2, 1.0], [228, 4, 1.0], [229, 2, 1.0], [229, 5, 1.0], [230, 7, 2.0], [231, 1, 5.0], [231, 2, 1.0], [231, 4, 17.0], [231, 7, 20.0], [232, 7, 1.0], [233, 3, 1.0], [234, 3, 1.0], [235, 0, 1.0], [236, 3, 1.0], [237, 8, 1.0], [238, 3, 1.0], [239, 6, 1.0], [240, 1, 1.0], [240, 2, 1.0], [241, 2, 1.0], [242, 8, 1.0], [243, 3, 1.0], [244, 4, 1.0], [245, 7, 1.0], [246, 0, 2.0], [246, 3, 2.0], [246, 7, 7.0], [247, 1, 1.0], [248, 3, 1.0], [248, 7, 1.0], [249, 7, 1.0], [250, 7, 1.0], [250, 8, 1.0], [251, 4, 1.0], [252, 2, 1.0], [252, 5, 1.0], [253, 1, 1.0], [254, 1, 1.0], [254, 7, 1.0], [255, 1, 2.0], [256, 3, 1.0], [257, 5, 1.0], [258, 3, 1.0], [259, 1, 1.0], [260, 2, 1.0], [261, 5, 1.0], [262, 7, 1.0], [263, 0, 1.0], [263, 8, 1.0], [264, 5, 1.0], [265, 4, 1.0], [265, 6, 1.0], [266, 5, 1.0], [267, 0, 1.0], [268, 2, 1.0], [268, 3, 2.0], [269, 1, 1.0], [270, 0, 1.0], [270, 1, 2.0], [270, 2, 1.0], [270, 5, 3.0], [270, 6, 5.0], [271, 0, 1.0], [272, 5, 2.0], [273, 6, 1.0], [274, 7, 1.0], [275, 4, 1.0], [276, 1, 1.0], [277, 1, 2.0], [277, 3, 1.0], [277, 6, 2.0], [278, 7, 1.0], [279, 3, 1.0], [280, 6, 1.0], [281, 4, 1.0], [282, 0, 1.0], [282, 8, 2.0], [283, 8, 5.0], [284, 4, 1.0], [285, 6, 1.0], [285, 8, 1.0], [286, 8, 1.0], [287, 1, 1.0], [288, 0, 1.0], [289, 0, 1.0], [289, 6, 1.0], [290, 3, 1.0], [291, 0, 2.0], [292, 3, 1.0], [293, 7, 1.0], [294, 1, 1.0], [295, 6, 1.0], [296, 3, 1.0], [297, 3, 1.0], [297, 7, 1.0], [298, 6, 1.0], [299, 5, 1.0], [300, 5, 2.0], [301, 7, 1.0], [302, 3, 1.0], [303, 7, 1.0], [304, 2, 1.0], [304, 5, 2.0], [304, 7, 6.0], [305, 3, 1.0], [306, 0, 1.0], [307, 3, 1.0], [308, 4, 5.0], [308, 7, 2.0], [309, 1, 13.0], [309, 2, 11.0], [309, 4, 2.0], [309, 5, 5.0], [309, 6, 12.0], [310, 0, 1.0], [310, 3, 1.0], [310, 8, 1.0], [311, 8, 1.0], [312, 8, 1.0], [313, 1, 1.0], [314, 8, 1.0], [315, 1, 2.0], [316, 5, 1.0], [317, 0, 1.0], [317, 6, 1.0], [318, 4, 1.0], [319, 0, 1.0], [320, 2, 1.0], [321, 0, 1.0], [322, 6, 1.0], [323, 7, 1.0], [324, 5, 1.0], [324, 6, 2.0], [325, 6, 1.0], [326, 2, 1.0], [326, 5, 1.0], [327, 5, 1.0], [328, 7, 1.0], [329, 2, 3.0], [330, 7, 1.0], [331, 3, 1.0], [332, 7, 1.0], [333, 3, 1.0], [334, 4, 2.0], [335, 0, 1.0], [336, 8, 1.0], [337, 0, 2.0], [337, 3, 1.0], [337, 5, 2.0], [338, 3, 1.0], [339, 0, 3.0], [339, 2, 1.0], [339, 4, 2.0], [339, 8, 2.0], [340, 0, 1.0], [341, 7, 1.0], [342, 3, 1.0], [342, 8, 2.0], [343, 0, 3.0], [343, 1, 1.0], [343, 4, 1.0], [343, 6, 1.0], [344, 5, 1.0], [345, 4, 3.0], [346, 6, 1.0], [347, 1, 1.0], [348, 1, 7.0], [348, 3, 2.0], [348, 6, 2.0], [349, 1, 1.0], [350, 5, 1.0], [351, 5, 1.0], [352, 8, 1.0], [353, 5, 1.0], [354, 7, 1.0], [354, 8, 1.0], [355, 3, 1.0], [356, 2, 1.0], [356, 6, 1.0], [357, 4, 2.0], [357, 8, 1.0], [358, 3, 1.0], [359, 3, 1.0], [360, 6, 1.0], [360, 8, 3.0], [361, 7, 1.0], [362, 8, 1.0], [363, 8, 1.0], [364, 6, 1.0], [365, 7, 1.0], [366, 6, 1.0], [367, 0, 1.0], [367, 3, 1.0], [368, 7, 1.0], [369, 5, 1.0], [370, 3, 1.0], [371, 1, 2.0], [372, 1, 4.0], [372, 3, 5.0], [372, 5, 2.0], [372, 6, 2.0], [372, 7, 5.0], [372, 8, 1.0], [373, 3, 1.0], [374, 3, 1.0], [375, 8, 1.0], [376, 0, 1.0], [377, 4, 1.0], [378, 0, 3.0], [378, 3, 5.0], [378, 7, 5.0], [378, 8, 9.0], [379, 0, 4.0], [379, 1, 1.0], [379, 2, 2.0], [379, 3, 4.0], [379, 6, 18.0], [379, 8, 21.0], [380, 1, 1.0], [381, 6, 2.0], [381, 7, 1.0], [382, 5, 1.0], [383, 2, 1.0], [383, 5, 1.0], [384, 5, 1.0], [385, 3, 1.0], [385, 5, 1.0], [386, 0, 1.0], [386, 5, 1.0], [387, 8, 1.0], [388, 4, 2.0], [388, 5, 2.0], [388, 6, 19.0], [388, 8, 3.0], [389, 3, 1.0], [390, 5, 1.0], [391, 5, 1.0], [392, 7, 1.0], [393, 5, 1.0], [394, 1, 1.0], [395, 7, 1.0], [396, 2, 1.0], [396, 5, 1.0], [396, 6, 1.0], [397, 0, 3.0], [397, 3, 1.0], [397, 4, 9.0], [397, 5, 2.0], [397, 6, 1.0], [397, 7, 1.0], [397, 8, 1.0], [398, 8, 2.0], [399, 0, 4.0], [399, 1, 1.0], [399, 4, 4.0], [399, 5, 3.0], [399, 6, 2.0], [399, 8, 5.0], [400, 8, 1.0], [401, 0, 1.0], [402, 6, 1.0], [403, 6, 1.0], [404, 0, 1.0], [404, 1, 2.0], [404, 3, 1.0], [404, 5, 1.0], [404, 6, 2.0], [404, 8, 1.0], [405, 1, 1.0], [406, 8, 1.0], [407, 8, 4.0], [408, 0, 1.0], [409, 1, 9.0], [409, 8, 3.0], [410, 4, 1.0], [411, 0, 1.0], [412, 2, 1.0], [413, 0, 2.0], [413, 3, 10.0], [413, 8, 8.0], [414, 1, 3.0], [414, 5, 1.0], [415, 6, 1.0], [416, 1, 1.0], [417, 1, 1.0]], "columns": [{"id": "PC.636", "metadata": null}, {"id": "PC.481", "metadata": null}, {"id": "PC.354", "metadata": null}, {"id": "PC.635", "metadata": null}, {"id": "PC.593", "metadata": null}, {"id": "PC.356", "metadata": null}, {"id": "PC.355", "metadata": null}, {"id": "PC.607", "metadata": null}, {"id": "PC.634", "metadata": null}], "generated_by": "QIIME 1.4.0-dev, svn revision 2835", "matrix_type": "sparse", "shape": [418, 9], "format_url": "http://biom-format.org", "date": "2012-03-14T20:48:29.112457", "type": "OTU table", "id": null, "matrix_element_type": "int"}"""

input2 = """{"rows": [{"id": "0", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "1", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Bacilli", "o__Bacillales", "f__Staphylococcaceae"]}}, {"id": "2", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "3", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "4", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "5", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Bacilli", "o__Lactobacillales", "f__Lactobacillaceae"]}}, {"id": "6", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "7", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "8", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "9", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Tenericutes", "c__Erysipelotrichi", "o__Erysipelotrichales", "f__Erysipelotrichaceae"]}}, {"id": "10", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "11", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "12", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "13", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Bacilli", "o__Lactobacillales", "f__Streptococcaceae"]}}, {"id": "14", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Actinobacteria", "c__Actinobacteria", "o__Coriobacteriales", "f__Coriobacteriaceae"]}}, {"id": "15", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "16", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Actinobacteria", "c__Actinobacteria", "o__Coriobacteriales", "f__Coriobacteriaceae"]}}, {"id": "17", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Porphyromonadaceae"]}}, {"id": "18", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "19", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "20", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "21", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "22", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Proteobacteria", "c__Deltaproteobacteria", "o__Desulfovibrionales", "f__Desulfovibrionaceae"]}}, {"id": "23", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "24", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "25", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "26", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "27", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__TM7", "c__TM7-3", "o__CW040", "f__F16"]}}, {"id": "28", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "29", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "30", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "31", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "32", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "33", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Actinobacteria", "c__Actinobacteria", "o__Coriobacteriales", "f__Coriobacteriaceae"]}}, {"id": "34", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Actinobacteria", "c__Actinobacteria", "o__Coriobacteriales", "f__Coriobacteriaceae"]}}, {"id": "35", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "36", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Tenericutes", "c__Erysipelotrichi", "o__Erysipelotrichales", "f__Erysipelotrichaceae"]}}, {"id": "37", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "38", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "39", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "40", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "41", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "42", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "43", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "44", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "45", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Deferribacteres", "c__Deferribacteres", "o__Deferribacterales", "f__Deferribacteraceae"]}}, {"id": "46", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "47", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "48", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "49", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "50", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "51", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "52", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "53", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "54", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "55", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Tenericutes", "c__Erysipelotrichi", "o__Erysipelotrichales", "f__Erysipelotrichaceae"]}}, {"id": "56", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "57", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "58", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "59", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales"]}}, {"id": "60", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "61", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "62", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "63", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "64", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "65", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "66", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "67", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "68", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "69", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "70", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "71", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "72", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "73", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Tenericutes", "c__Erysipelotrichi", "o__Erysipelotrichales", "f__Erysipelotrichaceae"]}}, {"id": "74", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "75", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "76", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "77", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "78", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "79", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "80", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "81", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Actinobacteria", "c__Actinobacteria", "o__Coriobacteriales", "f__Coriobacteriaceae"]}}, {"id": "82", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Porphyromonadaceae"]}}, {"id": "83", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "84", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "85", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "86", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "87", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Tenericutes", "c__Mollicutes", "o__RF39", "f__"]}}, {"id": "88", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "89", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "90", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Porphyromonadaceae"]}}, {"id": "91", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "92", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "93", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "94", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Porphyromonadaceae"]}}, {"id": "95", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "96", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "97", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "98", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "99", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "100", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "101", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "102", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Bacilli", "o__Lactobacillales", "f__Lactobacillaceae"]}}, {"id": "103", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "104", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "105", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Rikenellaceae"]}}, {"id": "106", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "107", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "108", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Deferribacteres", "c__Deferribacteres", "o__Deferribacterales", "f__Deferribacteraceae"]}}, {"id": "109", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "110", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "111", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "112", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "113", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "114", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "115", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "116", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "117", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__"]}}, {"id": "118", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "119", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales"]}}, {"id": "120", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales"]}}, {"id": "121", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "122", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "123", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Tenericutes", "c__Mollicutes", "o__RF39", "f__"]}}, {"id": "124", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "125", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "126", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "127", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "128", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "129", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "130", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Peptococcaceae"]}}, {"id": "131", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "132", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "133", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "134", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "135", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "136", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "137", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "138", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "139", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "140", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "141", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "142", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "143", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "144", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "145", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "146", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "147", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "148", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Rikenellaceae"]}}, {"id": "149", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Rikenellaceae"]}}, {"id": "150", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales"]}}, {"id": "151", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "152", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "153", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Rikenellaceae"]}}, {"id": "154", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "155", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "156", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "157", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "158", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "159", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "160", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "161", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "162", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "163", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales"]}}, {"id": "164", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "165", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "166", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "167", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "168", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "169", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "170", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "171", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales"]}}, {"id": "172", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "173", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "174", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "175", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "176", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Bacilli", "o__Lactobacillales", "f__Lactobacillaceae"]}}, {"id": "177", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "178", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Tenericutes", "c__Erysipelotrichi", "o__Erysipelotrichales", "f__Erysipelotrichaceae"]}}, {"id": "179", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "180", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "181", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "182", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "183", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "184", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__"]}}, {"id": "185", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "186", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "187", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "188", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "189", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "190", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales"]}}, {"id": "191", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "192", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "193", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "194", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "195", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "196", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "197", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "198", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "199", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Peptococcaceae"]}}, {"id": "200", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Catabacteriaceae"]}}, {"id": "201", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "202", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "203", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "204", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "205", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "206", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Catabacteriaceae"]}}, {"id": "207", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "208", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "209", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "210", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "211", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Rikenellaceae"]}}, {"id": "212", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Catabacteriaceae"]}}, {"id": "213", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "214", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes"]}}, {"id": "215", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "216", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "217", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "218", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales"]}}, {"id": "219", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "220", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "221", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "222", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "223", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "224", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "225", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Actinobacteria", "c__Actinobacteria", "o__Coriobacteriales", "f__Coriobacteriaceae"]}}, {"id": "226", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "227", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "228", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Peptococcaceae"]}}, {"id": "229", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "230", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "231", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Tenericutes", "c__Erysipelotrichi", "o__Erysipelotrichales", "f__Erysipelotrichaceae"]}}, {"id": "232", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "233", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "234", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Rikenellaceae"]}}, {"id": "235", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "236", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "237", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "238", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "239", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "240", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Bacilli", "o__Lactobacillales", "f__Lactobacillaceae"]}}, {"id": "241", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "242", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "243", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "244", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__"]}}, {"id": "245", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "246", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "247", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "248", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Rikenellaceae"]}}, {"id": "249", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "250", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "251", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "252", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "253", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "254", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "255", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "256", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "257", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "258", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Prevotellaceae"]}}, {"id": "259", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "260", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "261", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "262", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "263", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "264", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "265", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "266", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Rikenellaceae"]}}, {"id": "267", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "268", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "269", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "270", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "271", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Actinobacteria", "c__Actinobacteria", "o__Coriobacteriales", "f__Coriobacteriaceae"]}}, {"id": "272", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "273", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "274", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "275", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "276", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "277", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "278", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "279", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "280", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "281", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales"]}}, {"id": "282", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Proteobacteria", "c__Deltaproteobacteria", "o__Desulfovibrionales", "f__Desulfovibrionaceae"]}}, {"id": "283", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Porphyromonadaceae"]}}, {"id": "284", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales"]}}, {"id": "285", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "286", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Rikenellaceae"]}}, {"id": "287", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "288", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "289", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Porphyromonadaceae"]}}, {"id": "290", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "291", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Bacilli", "o__Bacillales", "f__Staphylococcaceae"]}}, {"id": "292", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "293", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "294", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "295", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "296", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "297", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "298", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "299", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "300", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "301", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "302", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "303", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "304", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "305", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "306", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "307", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "308", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Proteobacteria", "c__Epsilonproteobacteria", "o__Campylobacterales", "f__Helicobacteraceae"]}}, {"id": "309", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "310", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "311", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "312", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "313", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "314", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "315", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "316", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "317", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "318", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "319", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "320", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "321", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "322", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "323", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Catabacteriaceae"]}}, {"id": "324", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "325", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "326", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "327", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "328", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Tenericutes", "c__Erysipelotrichi", "o__Erysipelotrichales", "f__Erysipelotrichaceae"]}}, {"id": "329", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "330", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Actinobacteria", "c__Actinobacteria", "o__Coriobacteriales", "f__Coriobacteriaceae"]}}, {"id": "331", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "332", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Catabacteriaceae"]}}, {"id": "333", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "334", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "335", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "336", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "337", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "338", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Rikenellaceae"]}}, {"id": "339", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "340", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "341", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes"]}}, {"id": "342", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "343", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "344", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "345", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "346", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "347", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes"]}}, {"id": "348", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "349", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "350", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "351", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "352", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales"]}}, {"id": "353", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "354", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Clostridiales Family XIII. Incertae Sedis"]}}, {"id": "355", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "356", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "357", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "358", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__"]}}, {"id": "359", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "360", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Clostridiaceae"]}}, {"id": "361", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "362", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "363", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "364", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "365", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Tenericutes", "c__Erysipelotrichi", "o__Erysipelotrichales", "f__Erysipelotrichaceae"]}}, {"id": "366", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "367", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "368", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__"]}}, {"id": "369", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "370", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "371", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "372", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Rikenellaceae"]}}, {"id": "373", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes"]}}, {"id": "374", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "375", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "376", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "377", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "378", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Rikenellaceae"]}}, {"id": "379", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Bacteroidaceae"]}}, {"id": "380", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "381", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "382", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "383", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "384", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "385", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "386", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "387", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "388", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "389", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "390", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "391", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "392", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Ruminococcaceae"]}}, {"id": "393", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "394", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "395", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Proteobacteria", "c__Deltaproteobacteria", "o__Desulfovibrionales", "f__Desulfovibrionaceae"]}}, {"id": "396", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__"]}}, {"id": "397", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "398", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__Porphyromonadaceae"]}}, {"id": "399", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "400", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "401", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "402", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "403", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "404", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "405", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "406", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "407", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Tenericutes", "c__Erysipelotrichi", "o__Erysipelotrichales", "f__Erysipelotrichaceae"]}}, {"id": "408", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "409", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Bacilli", "o__Turicibacterales", "f__Turicibacteraceae"]}}, {"id": "410", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "411", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "412", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "413", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Bacteroidetes", "c__Bacteroidia", "o__Bacteroidales", "f__"]}}, {"id": "414", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "415", "metadata": {"taxonomy": ["Root", "k__Bacteria", "p__Firmicutes", "c__Clostridia", "o__Clostridiales", "f__Lachnospiraceae"]}}, {"id": "416", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}, {"id": "417", "metadata": {"taxonomy": ["Root", "k__Bacteria"]}}], "format": "Biological Observation Matrix 0.9.1-dev", "data": [[0, 0, 1.0], [1, 0, 1.0], [2, 1, 1.0], [3, 2, 1.0], [4, 1, 1.0], [4, 3, 1.0], [5, 4, 1.0], [6, 0, 1.0], [6, 2, 1.0], [6, 3, 1.0], [6, 5, 1.0], [7, 1, 1.0], [8, 4, 1.0], [9, 4, 1.0], [10, 2, 2.0], [10, 6, 1.0], [11, 3, 1.0], [11, 5, 1.0], [12, 3, 1.0], [13, 7, 1.0], [14, 8, 1.0], [15, 5, 1.0], [16, 8, 1.0], [17, 8, 1.0], [18, 1, 1.0], [18, 2, 1.0], [19, 7, 1.0], [20, 7, 1.0], [21, 3, 2.0], [22, 3, 1.0], [23, 1, 1.0], [24, 1, 1.0], [24, 3, 1.0], [25, 7, 2.0], [26, 8, 1.0], [27, 8, 2.0], [28, 1, 1.0], [28, 3, 1.0], [29, 5, 2.0], [30, 3, 1.0], [30, 5, 1.0], [31, 3, 3.0], [31, 7, 6.0], [32, 1, 3.0], [33, 3, 1.0], [34, 7, 2.0], [35, 7, 1.0], [36, 7, 1.0], [37, 2, 1.0], [37, 5, 1.0], [38, 7, 5.0], [39, 7, 1.0], [40, 1, 1.0], [40, 7, 2.0], [41, 7, 2.0], [42, 2, 16.0], [42, 4, 12.0], [43, 4, 1.0], [44, 1, 1.0], [45, 0, 6.0], [45, 3, 2.0], [45, 7, 3.0], [45, 8, 5.0], [46, 4, 1.0], [47, 1, 1.0], [48, 4, 1.0], [49, 2, 1.0], [50, 8, 1.0], [51, 7, 1.0], [52, 7, 1.0], [53, 0, 37.0], [53, 1, 1.0], [53, 3, 10.0], [53, 8, 4.0], [54, 6, 1.0], [55, 2, 1.0], [56, 0, 5.0], [56, 1, 2.0], [56, 2, 1.0], [56, 3, 4.0], [56, 4, 1.0], [56, 5, 1.0], [56, 6, 3.0], [56, 7, 9.0], [56, 8, 2.0], [57, 7, 1.0], [58, 0, 1.0], [59, 7, 1.0], [60, 0, 1.0], [60, 1, 2.0], [60, 2, 2.0], [60, 3, 1.0], [60, 4, 24.0], [60, 5, 10.0], [60, 6, 1.0], [61, 6, 1.0], [62, 1, 1.0], [63, 1, 1.0], [64, 6, 1.0], [65, 7, 1.0], [66, 7, 1.0], [67, 4, 1.0], [68, 0, 2.0], [68, 3, 3.0], [68, 8, 2.0], [69, 0, 1.0], [69, 3, 4.0], [69, 4, 2.0], [69, 7, 2.0], [69, 8, 1.0], [70, 6, 1.0], [71, 0, 2.0], [71, 3, 2.0], [72, 2, 1.0], [73, 3, 1.0], [74, 5, 1.0], [75, 5, 1.0], [76, 5, 1.0], [77, 6, 1.0], [78, 0, 19.0], [78, 1, 1.0], [78, 3, 2.0], [78, 8, 1.0], [79, 1, 1.0], [80, 1, 1.0], [81, 1, 1.0], [82, 3, 3.0], [82, 8, 1.0], [83, 3, 1.0], [84, 3, 1.0], [85, 4, 1.0], [86, 0, 1.0], [86, 3, 2.0], [86, 4, 1.0], [86, 5, 4.0], [86, 6, 4.0], [86, 7, 2.0], [87, 5, 1.0], [88, 1, 6.0], [88, 2, 1.0], [88, 5, 2.0], [88, 6, 4.0], [89, 2, 2.0], [90, 0, 1.0], [91, 5, 1.0], [92, 2, 1.0], [92, 5, 2.0], [93, 8, 1.0], [94, 1, 1.0], [94, 3, 1.0], [94, 8, 3.0], [95, 3, 1.0], [96, 1, 1.0], [97, 7, 1.0], [98, 7, 1.0], [99, 0, 1.0], [100, 6, 1.0], [101, 6, 1.0], [102, 1, 2.0], [102, 2, 1.0], [102, 4, 4.0], [102, 6, 1.0], [103, 8, 1.0], [104, 7, 1.0], [105, 8, 5.0], [106, 7, 1.0], [107, 4, 1.0], [108, 0, 1.0], [109, 2, 1.0], [110, 1, 1.0], [110, 2, 1.0], [110, 7, 1.0], [111, 1, 1.0], [111, 5, 1.0], [112, 6, 1.0], [113, 5, 1.0], [114, 3, 1.0], [114, 8, 2.0], [115, 2, 1.0], [116, 0, 1.0], [116, 6, 1.0], [117, 6, 1.0], [118, 2, 1.0], [118, 5, 1.0], [118, 6, 3.0], [119, 3, 1.0], [120, 3, 1.0], [120, 5, 1.0], [121, 2, 1.0], [122, 2, 1.0], [123, 7, 1.0], [124, 2, 2.0], [124, 5, 2.0], [124, 6, 2.0], [125, 7, 1.0], [126, 1, 1.0], [127, 4, 1.0], [128, 4, 1.0], [128, 5, 1.0], [128, 6, 1.0], [129, 2, 1.0], [130, 2, 1.0], [130, 3, 1.0], [131, 2, 1.0], [132, 8, 1.0], [133, 8, 1.0], [134, 1, 1.0], [134, 2, 1.0], [134, 6, 4.0], [135, 2, 1.0], [136, 2, 1.0], [137, 7, 1.0], [138, 7, 1.0], [139, 3, 1.0], [139, 5, 2.0], [140, 3, 1.0], [141, 5, 1.0], [142, 1, 6.0], [142, 3, 1.0], [143, 7, 1.0], [144, 0, 1.0], [145, 1, 2.0], [145, 2, 4.0], [145, 3, 2.0], [145, 6, 9.0], [146, 1, 1.0], [147, 3, 1.0], [148, 0, 1.0], [149, 0, 2.0], [149, 1, 1.0], [149, 3, 5.0], [149, 8, 1.0], [150, 0, 1.0], [151, 8, 1.0], [152, 8, 1.0], [153, 3, 1.0], [154, 7, 1.0], [155, 1, 1.0], [155, 2, 1.0], [156, 6, 1.0], [157, 6, 2.0], [158, 0, 1.0], [158, 3, 1.0], [159, 0, 1.0], [159, 3, 1.0], [160, 5, 1.0], [161, 2, 6.0], [161, 4, 2.0], [161, 5, 4.0], [162, 2, 1.0], [162, 5, 1.0], [163, 1, 1.0], [163, 2, 1.0], [163, 5, 1.0], [164, 2, 2.0], [164, 4, 1.0], [164, 5, 8.0], [164, 6, 3.0], [165, 3, 1.0], [166, 1, 1.0], [167, 3, 1.0], [168, 4, 1.0], [169, 0, 1.0], [170, 0, 1.0], [170, 8, 1.0], [171, 2, 1.0], [171, 4, 10.0], [172, 8, 2.0], [173, 3, 1.0], [173, 5, 1.0], [173, 7, 2.0], [174, 7, 1.0], [175, 4, 9.0], [175, 7, 1.0], [176, 1, 1.0], [176, 2, 14.0], [176, 5, 14.0], [176, 6, 1.0], [177, 1, 1.0], [178, 0, 2.0], [178, 4, 4.0], [179, 1, 1.0], [180, 1, 1.0], [181, 7, 2.0], [182, 3, 1.0], [183, 5, 2.0], [184, 6, 1.0], [185, 0, 1.0], [185, 1, 1.0], [185, 3, 1.0], [185, 7, 1.0], [186, 3, 1.0], [187, 2, 1.0], [187, 4, 1.0], [188, 5, 1.0], [189, 5, 1.0], [190, 5, 1.0], [191, 5, 1.0], [192, 5, 1.0], [193, 8, 1.0], [194, 5, 1.0], [195, 4, 1.0], [196, 1, 1.0], [197, 3, 1.0], [198, 5, 1.0], [198, 6, 2.0], [199, 7, 1.0], [200, 7, 1.0], [201, 1, 5.0], [201, 2, 2.0], [201, 6, 1.0], [201, 7, 1.0], [202, 2, 29.0], [202, 5, 10.0], [202, 6, 1.0], [203, 3, 1.0], [204, 5, 1.0], [205, 7, 1.0], [206, 0, 1.0], [207, 0, 1.0], [208, 5, 2.0], [209, 1, 1.0], [210, 8, 13.0], [211, 1, 2.0], [212, 7, 1.0], [213, 5, 1.0], [214, 3, 1.0], [214, 7, 2.0], [215, 5, 1.0], [216, 3, 5.0], [216, 4, 2.0], [217, 1, 1.0], [217, 4, 4.0], [218, 2, 1.0], [218, 7, 4.0], [219, 1, 1.0], [219, 2, 2.0], [219, 5, 2.0], [220, 1, 4.0], [221, 5, 1.0], [222, 1, 1.0], [223, 0, 1.0], [224, 2, 1.0], [225, 8, 1.0], [226, 1, 1.0], [226, 2, 2.0], [226, 6, 2.0], [227, 2, 1.0], [228, 4, 1.0], [229, 2, 1.0], [229, 5, 1.0], [230, 7, 2.0], [231, 1, 5.0], [231, 2, 1.0], [231, 4, 17.0], [231, 7, 20.0], [232, 7, 1.0], [233, 3, 1.0], [234, 3, 1.0], [235, 0, 1.0], [236, 3, 1.0], [237, 8, 1.0], [238, 3, 1.0], [239, 6, 1.0], [240, 1, 1.0], [240, 2, 1.0], [241, 2, 1.0], [242, 8, 1.0], [243, 3, 1.0], [244, 4, 1.0], [245, 7, 1.0], [246, 0, 2.0], [246, 3, 2.0], [246, 7, 7.0], [247, 1, 1.0], [248, 3, 1.0], [248, 7, 1.0], [249, 7, 1.0], [250, 7, 1.0], [250, 8, 1.0], [251, 4, 1.0], [252, 2, 1.0], [252, 5, 1.0], [253, 1, 1.0], [254, 1, 1.0], [254, 7, 1.0], [255, 1, 2.0], [256, 3, 1.0], [257, 5, 1.0], [258, 3, 1.0], [259, 1, 1.0], [260, 2, 1.0], [261, 5, 1.0], [262, 7, 1.0], [263, 0, 1.0], [263, 8, 1.0], [264, 5, 1.0], [265, 4, 1.0], [265, 6, 1.0], [266, 5, 1.0], [267, 0, 1.0], [268, 2, 1.0], [268, 3, 2.0], [269, 1, 1.0], [270, 0, 1.0], [270, 1, 2.0], [270, 2, 1.0], [270, 5, 3.0], [270, 6, 5.0], [271, 0, 1.0], [272, 5, 2.0], [273, 6, 1.0], [274, 7, 1.0], [275, 4, 1.0], [276, 1, 1.0], [277, 1, 2.0], [277, 3, 1.0], [277, 6, 2.0], [278, 7, 1.0], [279, 3, 1.0], [280, 6, 1.0], [281, 4, 1.0], [282, 0, 1.0], [282, 8, 2.0], [283, 8, 5.0], [284, 4, 1.0], [285, 6, 1.0], [285, 8, 1.0], [286, 8, 1.0], [287, 1, 1.0], [288, 0, 1.0], [289, 0, 1.0], [289, 6, 1.0], [290, 3, 1.0], [291, 0, 2.0], [292, 3, 1.0], [293, 7, 1.0], [294, 1, 1.0], [295, 6, 1.0], [296, 3, 1.0], [297, 3, 1.0], [297, 7, 1.0], [298, 6, 1.0], [299, 5, 1.0], [300, 5, 2.0], [301, 7, 1.0], [302, 3, 1.0], [303, 7, 1.0], [304, 2, 1.0], [304, 5, 2.0], [304, 7, 6.0], [305, 3, 1.0], [306, 0, 1.0], [307, 3, 1.0], [308, 4, 5.0], [308, 7, 2.0], [309, 1, 13.0], [309, 2, 11.0], [309, 4, 2.0], [309, 5, 5.0], [309, 6, 12.0], [310, 0, 1.0], [310, 3, 1.0], [310, 8, 1.0], [311, 8, 1.0], [312, 8, 1.0], [313, 1, 1.0], [314, 8, 1.0], [315, 1, 2.0], [316, 5, 1.0], [317, 0, 1.0], [317, 6, 1.0], [318, 4, 1.0], [319, 0, 1.0], [320, 2, 1.0], [321, 0, 1.0], [322, 6, 1.0], [323, 7, 1.0], [324, 5, 1.0], [324, 6, 2.0], [325, 6, 1.0], [326, 2, 1.0], [326, 5, 1.0], [327, 5, 1.0], [328, 7, 1.0], [329, 2, 3.0], [330, 7, 1.0], [331, 3, 1.0], [332, 7, 1.0], [333, 3, 1.0], [334, 4, 2.0], [335, 0, 1.0], [336, 8, 1.0], [337, 0, 2.0], [337, 3, 1.0], [337, 5, 2.0], [338, 3, 1.0], [339, 0, 3.0], [339, 2, 1.0], [339, 4, 2.0], [339, 8, 2.0], [340, 0, 1.0], [341, 7, 1.0], [342, 3, 1.0], [342, 8, 2.0], [343, 0, 3.0], [343, 1, 1.0], [343, 4, 1.0], [343, 6, 1.0], [344, 5, 1.0], [345, 4, 3.0], [346, 6, 1.0], [347, 1, 1.0], [348, 1, 7.0], [348, 3, 2.0], [348, 6, 2.0], [349, 1, 1.0], [350, 5, 1.0], [351, 5, 1.0], [352, 8, 1.0], [353, 5, 1.0], [354, 7, 1.0], [354, 8, 1.0], [355, 3, 1.0], [356, 2, 1.0], [356, 6, 1.0], [357, 4, 2.0], [357, 8, 1.0], [358, 3, 1.0], [359, 3, 1.0], [360, 6, 1.0], [360, 8, 3.0], [361, 7, 1.0], [362, 8, 1.0], [363, 8, 1.0], [364, 6, 1.0], [365, 7, 1.0], [366, 6, 1.0], [367, 0, 1.0], [367, 3, 1.0], [368, 7, 1.0], [369, 5, 1.0], [370, 3, 1.0], [371, 1, 2.0], [372, 1, 4.0], [372, 3, 5.0], [372, 5, 2.0], [372, 6, 2.0], [372, 7, 5.0], [372, 8, 1.0], [373, 3, 1.0], [374, 3, 1.0], [375, 8, 1.0], [376, 0, 1.0], [377, 4, 1.0], [378, 0, 3.0], [378, 3, 5.0], [378, 7, 5.0], [378, 8, 9.0], [379, 0, 4.0], [379, 1, 1.0], [379, 2, 2.0], [379, 3, 4.0], [379, 6, 18.0], [379, 8, 21.0], [380, 1, 1.0], [381, 6, 2.0], [381, 7, 1.0], [382, 5, 1.0], [383, 2, 1.0], [383, 5, 1.0], [384, 5, 1.0], [385, 3, 1.0], [385, 5, 1.0], [386, 0, 1.0], [386, 5, 1.0], [387, 8, 1.0], [388, 4, 2.0], [388, 5, 2.0], [388, 6, 19.0], [388, 8, 3.0], [389, 3, 1.0], [390, 5, 1.0], [391, 5, 1.0], [392, 7, 1.0], [393, 5, 1.0], [394, 1, 1.0], [395, 7, 1.0], [396, 2, 1.0], [396, 5, 1.0], [396, 6, 1.0], [397, 0, 3.0], [397, 3, 1.0], [397, 4, 9.0], [397, 5, 2.0], [397, 6, 1.0], [397, 7, 1.0], [397, 8, 1.0], [398, 8, 2.0], [399, 0, 4.0], [399, 1, 1.0], [399, 4, 4.0], [399, 5, 3.0], [399, 6, 2.0], [399, 8, 5.0], [400, 8, 1.0], [401, 0, 1.0], [402, 6, 1.0], [403, 6, 1.0], [404, 0, 1.0], [404, 1, 2.0], [404, 3, 1.0], [404, 5, 1.0], [404, 6, 2.0], [404, 8, 1.0], [405, 1, 1.0], [406, 8, 1.0], [407, 8, 4.0], [408, 0, 1.0], [409, 1, 9.0], [409, 8, 3.0], [410, 4, 1.0], [411, 0, 1.0], [412, 2, 1.0], [413, 0, 2.0], [413, 3, 10.0], [413, 8, 8.0], [414, 1, 3.0], [414, 5, 1.0], [415, 6, 1.0], [416, 1, 1.0], [417, 1, 1.0]], "columns": [{"id": "PC.636", "metadata": null}, {"id": "fake.sample", "metadata": null}, {"id": "PC.354", "metadata": null}, {"id": "PC.635", "metadata": null}, {"id": "PC.593", "metadata": null}, {"id": "PC.356", "metadata": null}, {"id": "PC.355", "metadata": null}, {"id": "PC.607", "metadata": null}, {"id": "PC.634", "metadata": null}], "generated_by": "QIIME 1.4.0-dev, svn revision 2835", "matrix_type": "sparse", "shape": [418, 9], "format_url": "http://biom-format.org", "date": "2012-03-14T20:48:29.112457", "type": "OTU table", "id": null, "matrix_element_type": "int"}"""

tree = """((((117:0.01623,(196:0.02706,306:0.01045)0.430:0.00015)0.914:0.00014,(314:0.02521,104:0.00015)0.465:0.01855)0.885:0.02154,407:0.02223)0.848:0.01046,(259:0.01763,185:0.02203)0.762:0.00511,((353:0.04271,(335:0.01062,(394:0.04443,169:0.00014)0.910:0.01046)0.671:0.00521)0.239:0.00014,((245:0.05025,((229:0.01061,95:0.00523)0.825:0.02163,(45:0.00543,(380:0.02839,(366:0.02249,((100:0.00528,411:0.00534)0.801:0.00586,((80:0.02709,62:0.00014)0.787:0.00524,4:0.01624)0.743:0.00448)0.385:0.00528)0.821:0.01601)0.407:0.00986)0.749:0.0047)1.000:0.00014)0.884:0.01045,(((404:0.02473,(321:0.05743,345:0.04116)0.770:0.01042)0.871:0.0175,(159:0.00512,(((((130:0.00524,223:0.02109)0.768:0.00506,53:0.00505)0.700:0.00015,((417:0.00455,(((309:0.04229,(332:0.00014,(152:0.03495,49:0.01132)0.757:0.00501)0.726:0.00689)0.484:0.01637,93:0.00014)0.782:0.00501,(128:0.01468,(254:0.00886,367:0.00882)0.813:0.00016)0.838:0.01501)0.773:0.00525)0.893:0.01078,(350:0.00518,(203:0.01589,7:0.03797)0.303:0.00015)0.778:0.00522)0.803:0.00528)0.893:0.01013,(388:0.03663,(110:0.02502,144:0.00823)0.883:0.02317)0.392:0.01286)0.340:0.00015,((251:0.03298,129:0.00595)0.743:0.02131,((187:0.01683,(192:0.04094,333:0.02254)0.819:0.01106)0.075:0.00016,(291:0.03997,374:0.02052)0.773:0.01301)0.706:0.00748)0.960:0.03217)0.827:0.00518)0.783:0.00016)0.968:0.01009,((((((331:0.0213,60:0.01357)0.425:0.01773,((109:0.02075,(317:0.07537,264:0.01319)0.577:0.02234)0.840:0.02594,(312:0.13538,173:0.0804)0.863:0.04059)0.906:0.04474)0.732:0.00594,(250:0.02106,253:0.01877)0.861:0.01467)1.000:0.08412,((37:0.02019,(284:0.01874,301:0.05561)0.553:0.01707)0.821:0.01394,(((391:0.01339,94:0.03006)0.863:0.01092,((141:0.02689,(90:0.01568,166:0.01544)0.845:0.01038)0.773:0.00831,(216:0.02563,288:0.04234)0.553:0.02164)0.778:0.00999)0.751:0.00694,((174:0.04134,(19:0.01184,((11:0.01042,(279:0.0051,(97:0.03227,172:0.00015)0.826:0.00518)0.773:0.00507)0.865:0.01734,(202:0.02027,(198:0.0444,(16:0.02344,297:0.01386)0.707:0.01593)0.900:0.01926)0.904:0.01639)0.717:0.00775)0.879:0.01642)0.783:0.01157,((((124:0.06649,((405:0.00158,89:0.0157)0.911:0.02924,(81:0.05513,((32:0.01302,(111:0.01924,418:0.01395)0.079:0.00502)0.876:0.01356,(67:0.01066,140:0.00015)0.894:0.01716)0.882:0.02725)0.951:0.03825)0.231:0.01554)0.841:0.01263,(((((41:0.0103,171:0.0168)0.841:0.01081,(278:0.01609,305:0.01125)0.772:0.00474)0.784:0.00577,((84:0.0344,(186:0.04377,142:0.03554)0.843:0.01608)0.912:0.02736,(372:0.02034,(287:0.03183,409:0.01693)0.856:0.01073)0.745:0.00582)0.571:0.00506)0.753:0.00517,((268:0.03304,(213:0.01051,382:0.01052)0.811:0.00998)0.952:0.02716,(9:0.08093,(52:0.03741,(359:0.02766,58:0.07021)0.869:0.02854)0.679:0.01388)0.836:0.01208)0.521:0.00015)0.899:0.01075,(((364:0.0264,66:0.01994)0.913:0.0207,(222:0.00015,10:0.02739)0.845:0.01067)0.860:0.01237,(61:0.03289,(((330:0.04019,351:0.02244)0.698:0.0276,(323:0.12995,(289:0.00535,(413:0.02178,139:0.01068)0.762:0.00528)0.942:0.00015)0.819:0.0102)0.938:0.00015,303:0.01596)0.858:0.01157)0.209:0.00476)0.954:0.02615)0.496:0.00014)0.832:0.00517,(((214:0.05163,230:0.07569)0.927:0.03951,((280:0.03044,(194:0.01081,(65:0.01809,(316:0.04535,(((387:0.00014,(30:0.02953,145:0.00015)0.831:0.00568)0.806:0.00587,(204:0.01043,346:0.00438)0.793:0.00431)0.960:0.03307,((343:0.01764,393:0.01827)0.727:0.00451,(127:0.02335,83:0.0179)0.759:0.00672)0.588:0.00574)0.643:0.00015)0.950:0.0228)0.748:0.00571)0.961:0.04287)0.925:0.03217,(241:0.01939,43:0.08019)0.857:0.02177)0.849:0.01507)0.829:0.01924,((((132:0.03373,(221:0.02342,352:0.04396)0.946:0.03287)0.888:0.01823,((373:0.10091,177:0.13147)0.751:0.00015,(((293:0.0161,86:0.00014)0.968:0.03626,102:0.04302)0.693:0.00897,(((123:0.01071,(209:0.00014,(376:0.0,156:0.0):0.00014)0.842:0.01038)0.945:0.0265,(210:0.00016,146:0.01595)0.549:0.01437)0.654:0.00218,((12:0.0484,8:0.02104)0.969:0.0424,(272:0.02828,(22:0.01045,29:0.02189)0.760:0.00525)0.749:0.00487)0.701:0.01121)0.821:0.01228)0.883:0.01772)0.724:0.00651)0.902:0.02042,(((((408:0.00014,(38:0.0526,78:0.00074)0.996:0.08306)0.998:0.10223,((76:0.08846,182:0.04382)0.958:0.0539,(410:0.0971,15:0.10716)0.678:0.01328)0.691:0.0071)0.897:0.03797,(((0:0.00016,292:0.02665)1.000:0.17598,(((236:0.08389,165:0.0064)0.425:0.03841,(328:0.07297,91:0.05669)0.981:0.09592)0.924:0.06437,(355:0.00016,55:0.0732)0.675:0.01105)0.763:0.07187)0.636:0.01026,((319:0.09501,(((200:0.02705,98:0.02199)0.983:0.0685,(24:0.06833,398:0.03572)0.511:0.02027)0.798:0.02205,((322:0.01463,39:0.05175)0.967:0.0826,((274:0.04802,18:0.02754)0.792:0.0172,((176:0.03312,(35:0.03377,17:0.0252)0.746:0.01061)0.939:0.03209,300:0.04129)0.818:0.01326)0.988:0.09877)0.924:0.06455)0.891:0.03516)0.560:0.03086,362:0.07093)0.915:0.04272)0.692:0.01378)0.979:0.07885,(((304:0.26243,(267:0.00968,118:0.08099)0.885:0.05434)0.524:0.05109,(371:0.07182,(73:0.0922,31:0.21665)0.855:0.03897)0.740:0.02336)0.929:0.05844,((220:0.0938,(21:0.01109,320:0.00014)0.968:0.07128)0.872:0.03879,(207:0.00418,327:0.03108)0.983:0.07498)0.770:0.01237)0.473:0.02492)0.802:0.01473,((((136:0.00661,153:0.00473)0.997:0.09292,((71:0.02198,(271:0.00016,(((168:0.03029,369:0.01489)0.765:0.01385,122:0.01842)0.806:0.01117,361:0.02448)0.835:0.01151)0.936:0.02566)0.911:0.028,(199:0.00914,239:0.08898)0.899:0.02838)0.827:0.02548)0.927:0.02861,(((((266:0.00016,(((383:0.02613,(226:0.02876,201:0.14721)0.798:0.02147)0.884:0.02148,((((360:0.00015,(337:0.02264,(414:0.0107,72:0.01132)0.862:0.01097)0.621:0.01705)0.998:0.19154,(131:0.00205,(334:0.01891,(232:0.00015,143:0.02891)0.846:0.01454)0.999:0.09463)0.978:0.1571)0.878:0.07516,(324:0.0258,(219:0.00015,224:0.03191)0.981:0.04417)0.807:0.0125)0.867:0.03328,(((164:0.05516,(235:0.05657,27:0.0195)0.806:0.01096)0.678:0.00115,((315:0.00529,113:0.05507)0.889:0.02149,(348:0.09547,((311:0.00508,163:0.00014)0.965:0.00015,(56:0.00015,(205:0.04314,101:0.0156)0.748:0.00014)0.902:0.01018)0.166:0.01624)0.845:0.02166)1.000:0.07732)0.801:0.01079,(5:0.0081,((((400:0.0051,((63:0.016,(396:0.04485,64:0.00015)0.879:0.01096)0.971:0.02743,181:0.03152)0.722:0.00016)0.907:0.01479,(((215:0.02059,54:0.00014)0.202:0.00016,(2:0.01021,103:0.02082)0.886:0.01027)0.994:0.03906,(3:0.04134,(((339:0.00014,(170:0.00507,59:0.00519)0.815:0.00508)0.335:0.00016,397:0.01549)0.690:0.00014,(269:0.01064,310:0.0052)0.866:0.01042)0.861:0.01459)0.870:0.01673)0.925:0.00014)0.880:0.01752,(225:0.02073,((249:0.05068,(246:0.02824,138:0.00448)0.961:0.02374)0.177:0.00015,75:0.05899)0.734:0.01674)0.832:0.01477)0.654:0.00602,(74:0.00488,(28:0.03622,((273:0.03289,390:0.00016)0.990:0.03262,363:0.01567)0.796:0.00459)0.869:0.00642)0.981:0.03561)0.824:0.01317)0.887:0.01496)0.779:0.00739)0.883:0.02152)0.702:0.01128,(((283:0.00014,399:0.0678)0.963:0.03671,((121:0.0375,96:0.00525)0.874:0.0219,154:0.07411)0.824:0.02578)0.927:0.03762,(260:0.10849,(((((120:0.02607,(155:0.06721,137:0.00495)0.974:0.02947)0.535:0.0036,242:0.04238)0.861:0.02023,(349:0.04521,(((((82:0.00014,341:0.06325)0.993:0.03755,379:0.00016)0.786:0.00014,234:0.06992)0.997:0.06111,69:0.00014)0.994:0.04534,211:0.00332)0.381:0.00588)0.481:0.00823)0.938:0.03703,157:0.07989)0.717:0.00692,57:0.00882)0.961:0.07989)0.916:0.05076)0.798:0.02675)0.998:0.096)0.929:0.03959,((285:0.00015,(108:0.00014,(149:0.07407,((344:0.01413,151:0.01634)0.736:0.00657,(134:0.02781,(212:0.02652,(378:0.00015,(238:0.04745,247:0.04301)0.703:0.00014)0.966:0.02148)0.015:0.00015)0.466:0.01024)0.756:0.00588)0.883:0.01526)0.969:0.03759)0.988:0.03167,370:0.00014)0.867:0.02372)0.998:0.08133,158:0.00015)0.922:0.03621,((358:0.01281,(20:0.02969,99:0.02372)0.985:0.06249)0.998:0.11815,(190:0.06903,(357:0.06998,356:0.00015)0.829:0.02015)0.858:0.03439)0.919:0.05706)0.807:0.02571,193:0.01711)0.882:0.02352)0.729:0.01709,(248:0.0579,((244:0.03253,(206:0.05079,(((262:0.01325,255:0.03775)0.835:0.01673,88:0.02516)0.336:0.00015,40:0.02313)0.090:0.01515)0.865:0.01872)0.925:0.03028,(263:0.0345,(36:0.0456,183:0.04272)0.933:0.04591)0.868:0.02359)0.930:0.04794)0.957:0.06027)0.972:0.06398)0.617:0.00519)0.724:0.00431,((277:0.01408,(227:0.04382,233:0.00014)1.000:0.09125)0.682:0.0064,((((((((14:0.00015,188:0.00525)0.779:0.00517,(257:0.01105,375:0.04818)0.740:0.00015)0.775:0.01082,(119:0.00016,(313:0.01642,347:0.00015)0.843:0.01076)0.674:0.0108)0.000:0.00016,(298:0.0345,((325:0.05824,(184:0.00016,(178:0.06962,(340:0.06024,(((175:0.08004,(270:0.02469,282:0.06081)0.944:0.04574)0.357:0.01243,(258:0.06066,((365:0.00016,((26:0.00534,70:0.03876)0.939:0.01609,(195:0.03389,(44:0.01056,(180:0.0222,386:0.02757)0.969:0.00014)0.868:0.02803)0.745:0.00533)0.343:0.01063)0.802:0.00638,125:0.0208)0.354:0.01817)0.372:0.00767)0.820:0.0178,(218:0.03567,147:0.00245)0.981:0.05918)0.768:0.0159)0.847:0.01215)0.768:0.00569)0.978:0.02296)0.000:0.00015,(237:0.03489,296:0.01051)0.925:0.01719)0.896:0.01736)0.515:0.0105)0.741:0.01842,((126:0.01118,179:0.02795)0.939:0.02231,(160:0.01105,(329:0.01155,(336:0.00015,34:0.01644)0.840:0.02223)0.949:0.02512)0.759:0.00599)0.000:0.00334)0.952:0.0329,384:0.01969)0.777:0.00942,133:0.04451)0.888:0.02052,(208:0.0403,(23:0.01161,((115:0.00837,46:0.08317)0.999:0.11513,(276:0.02789,162:0.02676)0.797:0.03037)0.884:0.03986)0.910:0.0309)0.796:0.0182)0.951:0.03291)0.958:0.03269)0.545:0.00537)0.583:0.00014)0.727:0.00654,(((((295:0.02551,(368:0.02676,47:0.03946)0.602:0.00136)0.826:0.00588,((((228:0.00015,((290:0.01639,354:0.01649)0.568:0.01074,197:0.0162)0.879:0.00016)0.924:0.00529,256:0.00519)0.880:0.00015,(231:0.01531,403:0.02714)0.818:0.00041)0.733:0.00505,(191:0.0051,(33:0.02234,377:0.03943)0.879:0.01609)0.882:0.01086)0.858:0.01046)0.759:0.00506,(77:0.02734,(114:0.02206,50:0.03351)0.705:0.00412)0.855:0.01095)0.861:0.01075,((48:0.01954,(389:0.00015,13:0.01579)0.776:0.00512)0.800:0.00644,(((107:0.00513,116:0.01096)0.872:0.01062,(150:0.01086,(401:0.00014,189:0.05119)0.390:0.01048)0.911:0.01611)0.887:0.01052,(307:0.00016,((395:0.01672,402:0.01628)0.732:0.00525,(((((243:0.01546,(265:0.03394,412:0.01126)0.542:0.01041)0.781:0.00609,(326:0.01055,167:0.0162)0.760:0.0052)0.814:0.0052,(294:0.05971,(308:0.01435,381:0.0377)0.751:0.00763)0.895:0.0161)0.987:0.00014,(112:0.03413,25:0.01612)0.905:0.00014)0.876:0.00015,(252:0.00015,342:0.01556)0.980:0.02642)0.502:0.00526)0.893:0.01027)0.907:0.01031)0.851:0.00014)0.868:0.01018)0.889:0.01626,(((106:0.00831,(302:0.03114,85:0.01229)0.901:0.02187)0.886:0.01264,((((275:0.01585,(217:0.00016,338:0.03721)0.892:0.03778)0.770:0.00488,(392:0.00516,(1:0.00527,(240:0.01873,(51:0.02977,281:0.0106)0.492:0.01443)0.777:0.00831)0.772:0.00526)0.778:0.00519)0.833:0.00503,(406:0.01023,416:0.00515)0.865:0.00015)0.799:0.01576,68:0.01626)0.749:0.00507)0.757:0.00518,(87:0.02794,161:0.00014)0.233:0.00511)0.904:0.0159)0.896:0.02029)0.683:0.02168)0.833:0.01463)0.781:0.00532)0.723:0.00556)0.820:0.01376,((92:0.03867,((318:0.01724,(415:0.02669,148:0.00014)0.736:0.00422)0.837:0.01019,(261:0.00516,385:0.00502)0.967:0.02157)0.000:0.00015)0.895:0.01312,(6:0.04864,(((135:0.00901,(286:0.06254,105:0.01165)0.753:0.01766)0.580:0.0171,79:0.01655)0.632:0.00016,42:0.00015)0.829:0.00928)0.759:0.00636)0.758:0.01044)0.646:0.00165,299:0.00508)0.922:0.00016)0.860:0.01021)0.765:0.00517)0.776:0.00522);"""


if __name__ == "__main__":
    main()
