#!/usr/bin/env python
# File created on 20 Feb 2013
from __future__ import division

__author__ = "Greg Caporaso"
__copyright__ = "Copyright 2011, The QIIME project"
__credits__ = ["Greg Caporaso", "Kyle Bittinger", "Jai Ram Rideout"]
__license__ = "GPL"
__version__ = "1.9.1"
__maintainer__ = "Greg Caporaso"
__email__ = "gregcaporaso@gmail.com"

import os
from os import close
from shutil import rmtree
from glob import glob
from os.path import join, exists, getsize, dirname
from tempfile import mkstemp, mkdtemp
from unittest import TestCase, main

from skbio.util import remove_files
from qiime_default_reference import get_template_alignment_column_mask

from qiime.util import (load_qiime_config,
                        count_seqs,
                        get_qiime_temp_dir)
from qiime.parse import (parse_qiime_parameters)
from qiime.test import initiate_timeout, disable_timeout
from qiime.workflow.util import (call_commands_serially,
                                 no_status_updates)
from qiime.workflow.ampliconnoise import run_ampliconnoise


class AmpliconNoiseWorkflowTests(TestCase):

    def setUp(self):
        """ """
        self.qiime_config = load_qiime_config()
        self.dirs_to_remove = []
        self.files_to_remove = []

        self.tmp_dir = get_qiime_temp_dir()

        self.wf_out = mkdtemp(dir=self.tmp_dir,
                              prefix='qiime_wf_out', suffix='')
        self.dirs_to_remove.append(self.wf_out)

        fd, self.fasting_mapping_fp = mkstemp(dir=self.tmp_dir,
                                             prefix='qiime_wf_mapping', suffix='.txt')
        close(fd)
        fasting_mapping_f = open(self.fasting_mapping_fp, 'w')
        fasting_mapping_f.write(fasting_map)
        fasting_mapping_f.close()
        self.files_to_remove.append(self.fasting_mapping_fp)

        fd, self.fasting_seqs_fp = mkstemp(dir=self.tmp_dir,
                                          prefix='qiime_wf_seqs', suffix='.fasta')
        close(fd)
        fasting_seqs_f = open(self.fasting_seqs_fp, 'w')
        fasting_seqs_f.write(fasting_seqs_subset)
        fasting_seqs_f.close()
        self.files_to_remove.append(self.fasting_seqs_fp)

        fd, self.fasting_fna_fp = mkstemp(dir=self.tmp_dir,
                                         prefix='qiime_wf_fna', suffix='.fasta')
        close(fd)
        fasting_seqs_f = open(self.fasting_fna_fp, 'w')
        fasting_seqs_f.write(fasting_tutorial_fna)
        fasting_seqs_f.close()
        self.files_to_remove.append(self.fasting_fna_fp)

        fd, self.fasting_qual_fp = mkstemp(dir=self.tmp_dir,
                                          prefix='qiime_wf_qual', suffix='.qual')
        close(fd)
        fasting_seqs_f = open(self.fasting_qual_fp, 'w')
        fasting_seqs_f.write(fasting_tutorial_qual)
        fasting_seqs_f.close()
        self.files_to_remove.append(self.fasting_qual_fp)

        fd, self.fasting_seqs_denoiser_fp = mkstemp(dir=self.tmp_dir,
                                                   prefix='qiime_wf_denoiser_seqs', suffix='.fasta')
        close(fd)
        fasting_seqs_f = open(self.fasting_seqs_denoiser_fp, 'w')
        fasting_seqs_f.write('\n'.join(fasting_seqs_subset.split('\n')[:44]))
        fasting_seqs_f.close()
        self.files_to_remove.append(self.fasting_seqs_denoiser_fp)

        fd, self.fasting_otu_table_fp = mkstemp(dir=self.tmp_dir,
                                               prefix='qiime_wf_otu_table', suffix='.biom')
        close(fd)
        fasting_otu_table_f = open(self.fasting_otu_table_fp, 'w')
        fasting_otu_table_f.write(fasting_subset_otu_table)
        fasting_otu_table_f.close()
        self.files_to_remove.append(self.fasting_otu_table_fp)

        fd, self.fasting_tree_fp = mkstemp(dir=self.tmp_dir,
                                          prefix='qiime_wf_tree', suffix='.tre')
        close(fd)
        fasting_tree_f = open(self.fasting_tree_fp, 'w')
        fasting_tree_f.write(fasting_subset_tree)
        fasting_tree_f.close()
        self.files_to_remove.append(self.fasting_tree_fp)

        fd, self.template_aln_fp = mkstemp(dir=self.tmp_dir,
                                          prefix='wf_template', suffix='.fasta')
        close(fd)
        template_aln_f = open(self.template_aln_fp, 'w')
        template_aln_f.write(template_alignment_subset)
        template_aln_f.close()
        self.files_to_remove.append(self.template_aln_fp)

        fd, self.lanemask_fp = mkstemp(dir=self.tmp_dir,
                                      prefix='wf_lanemask', suffix='.txt')
        close(fd)
        lanemask_f = open(self.lanemask_fp, 'w')
        lanemask_f.write(get_template_alignment_column_mask())
        lanemask_f.close()
        self.files_to_remove.append(self.lanemask_fp)

        fd, self.fasting_subset_fna = mkstemp(dir=self.tmp_dir,
                                             prefix='wf_denoise_fna', suffix='.fasta')
        sff_f = open(self.fasting_subset_fna, 'w')
        sff_f.write(fasting_subset_fna)
        sff_f.close()
        self.files_to_remove.append(self.fasting_subset_fna)

        fd, self.fasting_subset_qual = mkstemp(dir=self.tmp_dir,
                                              prefix='wf_denoise_qual', suffix='.qual')
        close(fd)
        sff_f = open(self.fasting_subset_qual, 'w')
        sff_f.write(fasting_subset_qual)
        sff_f.close()
        self.files_to_remove.append(self.fasting_subset_qual)

        jobs_dir = join(self.tmp_dir, 'jobs')
        if not exists(jobs_dir):
            # only clean up the jobs dir if it doesn't already exist
            self.dirs_to_remove.append(jobs_dir)
        self.params = parse_qiime_parameters(qiime_parameters_f)
        self.params['align_seqs']['template_fp'] = self.template_aln_fp
        self.params['filter_alignment']['lane_mask_fp'] = self.lanemask_fp

        fd, self.pick_ref_otus_seqs1 = mkstemp(
            dir=self.tmp_dir, prefix='ref_otus_wf', suffix='.fna')
        close(fd)
        f = open(self.pick_ref_otus_seqs1, 'w')
        f.write(pick_ref_otus_seqs1)
        f.close()
        self.files_to_remove.append(self.pick_ref_otus_seqs1)

        fd, self.pick_ref_otus_refseqs1 = mkstemp(
            dir=self.tmp_dir, prefix='ref_otus_wf', suffix='.fna')
        close(fd)
        f = open(self.pick_ref_otus_refseqs1, 'w')
        f.write(pick_ref_otus_refseqs1)
        f.close()
        self.files_to_remove.append(self.pick_ref_otus_refseqs1)

        fd, self.pick_ref_otus_tax1 = mkstemp(
            dir=self.tmp_dir, prefix='ref_otus_wf', suffix='.fna')
        close(fd)
        f = open(self.pick_ref_otus_tax1, 'w')
        f.write(pick_ref_otus_tax1)
        f.close()
        self.files_to_remove.append(self.pick_ref_otus_tax1)

        self.pick_ref_otus_params1 =\
            parse_qiime_parameters(pick_ref_otus_params1.split('\n'))

        self.run_core_qiime_analyses_params1 =\
            parse_qiime_parameters(run_core_qiime_analyses_params1.split('\n'))

        initiate_timeout(240)

    def tearDown(self):
        """ """
        disable_timeout()
        remove_files(self.files_to_remove)
        # remove directories last, so we don't get errors
        # trying to remove files which may be in the directories
        for d in self.dirs_to_remove:
            if exists(d):
                rmtree(d)

    def test_run_ampliconnoise(self):
        """ ampliconnoise workflow functions as expected """
        test_dir = dirname(os.path.abspath(__file__))
        sff_txt_fp = join(test_dir, '..',
                          'test_support_files',
                          'Fasting_Example.sff.txt')
        output_fp = join(self.wf_out, 'ampliconnoise_out.fna')
        output_dir = join(self.wf_out, 'ampliconnoise_out.fna_dir')

        self.files_to_remove.append(output_fp)
        self.dirs_to_remove.append(output_dir)

        run_ampliconnoise(mapping_fp=self.fasting_mapping_fp,
                          output_dir=output_dir,
                          command_handler=call_commands_serially,
                          params=parse_qiime_parameters([]),
                          qiime_config=self.qiime_config,
                          logger=None,
                          status_update_callback=no_status_updates,
                          chimera_alpha=-3.8228,
                          chimera_beta=0.6200,
                          sff_txt_fp=sff_txt_fp,
                          numnodes=2,
                          suppress_perseus=True,
                          output_filepath=output_fp,
                          platform='flx',
                          seqnoise_resolution=None,
                          truncate_len=None)

        # Check that the log file is created and has size > 0
        log_fp = glob(join(output_dir, 'log*.txt'))[0]
        self.assertTrue(getsize(log_fp) > 0)

        # Check that a reasonable number of sequences were written
        # to the output file
        seq_count, a, b = count_seqs(output_fp)
        self.assertTrue(seq_count > 500,
                        ("Sanity check of sequence count failed - "
                         "fewer than 1000 sequences in output file."))

qiime_parameters_f = """# qiime_parameters.txt
# WARNING: DO NOT EDIT OR DELETE Qiime/qiime_parameters.txt. Users should copy this file and edit copies of it.

# OTU picker parameters
pick_otus:otu_picking_method	uclust
pick_otus:clustering_algorithm	furthest
pick_otus:max_cdhit_memory	400
pick_otus:refseqs_fp
pick_otus:blast_db
pick_otus:similarity	0.97
pick_otus:max_e_value	1e-10
pick_otus:prefix_prefilter_length
pick_otus:trie_prefilter
pick_otus:prefix_length
pick_otus:suffix_length
pick_otus:optimal_uclust
pick_otus:exact_uclust
pick_otus:user_sort
pick_otus:suppress_presort_by_abundance_uclust
pick_otus:suppress_new_clusters
pick_otus:denovo_otu_id_prefix  workflowOTU

# Parallel options
parallel:jobs_to_start	2
parallel:retain_temp_files	False
parallel:seconds_to_sleep	1

# Representative set picker parameters
pick_rep_set:rep_set_picking_method	most_abundant
pick_rep_set:sort_by	otu

# Multiple sequence alignment parameters
align_seqs:template_fp
align_seqs:alignment_method	pynast
align_seqs:pairwise_alignment_method	uclust
align_seqs:blast_db
align_seqs:min_length	150
align_seqs:min_percent_id	75.0

# Alignment filtering (prior to tree-building) parameters
filter_alignment:lane_mask_fp
filter_alignment:allowed_gap_frac	 0.999999
filter_alignment:remove_outliers	False
filter_alignment:threshold	3.0

# Taxonomy assignment parameters
assign_taxonomy:id_to_taxonomy_fp
assign_taxonomy:reference_seqs_fp
assign_taxonomy:assignment_method	rdp
assign_taxonomy:confidence	0.8

# Phylogenetic tree building parameters
make_phylogeny:tree_method	fasttree
make_phylogeny:root_method	tree_method_default

# Summarize taxonomy parameters
summarize_taxa:level 3
summarize_taxa:absolute_abundance True

# Summarize taxonomy parameters
plot_taxa_summary:include_html_legend True
plot_taxa_summary:include_html_counts True
plot_taxa_summary:background black
plot_taxa_summary:labels Class

# Beta diversity parameters
beta_diversity:metrics	weighted_unifrac,unweighted_unifrac

# Make 3D plot parameters
make_3d_plots:custom_axes

# Rarefaction parameters
multiple_rarefactions:num_reps	1
multiple_rarefactions:depth
multiple_rarefactions:lineages_included	False

# Even-depth rarefaction parameters
multiple_rarefactions_even_depth:num_reps	5

# Alpha diversity parameters
alpha_diversity:metrics	PD_whole_tree

# Make rarefaction plots parameters
make_rarefaction_plots:imagetype	png
make_rarefaction_plots:resolution	75

# Collate alpha
collate_alpha:example_path

""".split('\n')

fasting_map = """#SampleID	BarcodeSequence	LinkerPrimerSequence	Treatment	DOB	Description
#Example mapping file for the QIIME analysis package.  These 9 samples are from a study of the effects of exercise and diet on mouse cardiac physiology (Crawford, et al, PNAS, 2009).
PC.354	AGCACGAGCCTA	YATGCTGCCTCCCGTAGGAGT	Control	20061218	Control_mouse__I.D._354
PC.355	AACTCGTCGATG	YATGCTGCCTCCCGTAGGAGT	Control	20061218	Control_mouse__I.D._355
PC.356	ACAGACCACTCA	YATGCTGCCTCCCGTAGGAGT	Control	20061126	Control_mouse__I.D._356
PC.481	ACCAGCGACTAG	YATGCTGCCTCCCGTAGGAGT	Control	20070314	Control_mouse__I.D._481
PC.593	AGCAGCACTTGT	YATGCTGCCTCCCGTAGGAGT	Control	20071210	Control_mouse__I.D._593
PC.607	AACTGTGCGTAC	YATGCTGCCTCCCGTAGGAGT	Fast	20071112	Fasting_mouse__I.D._607
PC.634	ACAGAGTCGGCT	YATGCTGCCTCCCGTAGGAGT	Fast	20080116	Fasting_mouse__I.D._634
PC.635	ACCGCAGAGTCA	YATGCTGCCTCCCGTAGGAGT	Fast	20080116	Fasting_mouse__I.D._635
PC.636	ACGGTGAGTGTC	YATGCTGCCTCCCGTAGGAGT	Fast	20080116	Fasting_mouse__I.D._636
"""

fasting_seqs_subset = """>PC.634_1 FLP3FBN01ELBSX orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTTACCCTCTCAGGCCGGCTACGCATCATCGCCTTGGTGGGCCGTTACCTCACCAACTAGCTAATGCGCCGCAGGTCCATCCATGTTCACGCCTTGATGGGCGCTTTAATATACTGAGCATGCGCTCTGTATACCTATCCGGTTTTAGCTACCGTTTCCAGCAGTTATCCCGGACACATGGGCTAGG
>PC.634_2 FLP3FBN01EG8AX orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGCCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGGGCCGTTACCCCGCCAACAACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGAAGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCA
>PC.354_3 FLP3FBN01EEWKD orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAGTCTCTTAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGACATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCTTGGG
>PC.481_4 FLP3FBN01DEHK3 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGACTTGGTGAGCCGTTACCTCACCAACTATCTAATCAGACGCGAGCCCATCTTTCAGCGGATTGCTCCTTTGGTATTCCGGCGATGCCGCCAAAATCATTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGAAAGGCAGGTTGCTCACG
>PC.634_5 FLP3FBN01DGFYQ orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGTCCATCTCACACCGATTAATCTTTTTCCAACCAGAGCATGCGCCCCTGTTGGCTTATGCGGTATTAGCGGTCGTTTCCAACTGTTATCCCCCTGTGTGAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.636_6 FLP3FBN01A55WZ orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGAGAAAATGCTCACTCATGATACCATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCAGGTTGGATACGCGTTACTCACCCGTGCGCCGG
>PC.634_7 FLP3FBN01D7O1S orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGATAACCGAAATTCTTTAATAGTGAAACCATGCGGAAATACTATACTATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTTATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCATC
>PC.634_8 FLP3FBN01CV5ZU orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAACCAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCTCTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCG
>PC.634_9 FLP3FBN01DQ783 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.634_10 FLP3FBN01DDPFF orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCAGGTTGGGTACGCGTTACTC
>PC.634_11 FLP3FBN01CPD70 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGACCGTGTCTCAGTTCCAATGTGGGGGGCCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGGGCCGTTACCCCGCCAACAACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGAAGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.593_12 FLP3FBN01BBAE6 orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCGGCACCGGGATGCCCCGATGCCGCGTCACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTGGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCG
>PC.355_13 FLP3FBN01AWYZD orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGACATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCTTGGG
>PC.634_14 FLP3FBN01AM0P3 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGCCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATCAGACGCGGGCCCATCCCGTACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTTACTCACCCG
>PC.635_15 FLP3FBN01BO4IE orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGACTTGGTAGGCCGTTACCCCACCAACTATCTAATCAGACGCAAGCCCATCTATCAGCGGATTGCTCCTTTCCCATTTATATCATGTGATATTCATGGCATATGCGGTATTAGCAGTCATTTCTAACTGTTGTTCCCCTCTGATAGG
>PC.635_16 FLP3FBN01BPX14 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGTCACGCGAGCCTATCCTCATCCGACGGATCTTTAGATGGAACCAGATGCCTGATTCCATCGCCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATCGAGGGCAAGTTGCATACGCG
>PC.356_17 FLP3FBN01DB5I7 orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCCTGCACCGCCGGAGCTTCCCCCGCCGCCCCATGCGGGGCTGCGGGCATATGCGGTATTAGCAGCCGTTTCCGGCTGTTGTCCCCCAGTGCAGGGCAGGTTGCCCACGCGTTACTCACCCGTCCG
>PC.634_18 FLP3FBN01AK9OO orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTCGAGCCGTTACCTCACCAACTAGCTAATGCGCCGCGGGCCCATCTCATAGCGGATTACTCCTTTAATTGCTACTTCATGCGAAGCTACAATCTTATGCGGTATTAATCTCCCTTTCGGAAGGCTATTCCCCTCTATGAGGTCAGGTTG
>PC.634_19 FLP3FBN01ANGF2 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGTCACGCGAGCCTATCCTCATCCGACGGATCTTTAGATGGAACCAGATGCCTGATTCCATCGCCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCAAGTTGCTCACGCG
>PC.634_20 FLP3FBN01AF994 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGTCGGTTTGGTGGGCCGTTACCCCGCCAACTGCCTAATGGAACGCATGCCTATCTATCAGCGATGAATCTTTAGCAAATATCCCCATGCGGGGCCCCTGCTTCATGCGGTATTAGTCCGACTTTCGCCGGGTTATCCCCTCCTGATAGGTAAGTTGCATACGCGTTACTTCACCCGTCGCGG
>PC.634_21 FLP3FBN01AHXU8 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTACTGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCGGGGATGCCCCCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGGCAGAGAG
>PC.593_22 FLP3FBN01APRWO orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGGCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTCTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGG
>PC.634_23 FLP3FBN01C645E orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCAACAACCTAATGGAACGCATCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGAAGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAGGTTGGATACGCGTTACTCACCCGTGCGCCAGTCG
>PC.634_24 FLP3FBN01A2K0Q orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTGGTCTTGGTGAGCCGTTACCCCACCAACTAACTAATACAACGCATGCCCATCCATTACCACCGGAGTTTTCAACCCAAGAAGATGCCTCCCTGGATGTTATGGGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTAATGGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCAT
>PC.634_25 FLP3FBN01DJY30 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTGCCAACTAGCTAATCAGACGCGGGCCCATCTCATACCACCGGAGTTTTTGCCGCTGCACCATGCGGTGCTGTGGCCTTATGCGGTATTAGCAGCCATTTCTGGCTGTTATCCCCCTGTATGAGGCAGGTTGCCCACG
>PC.355_26 FLP3FBN01EB63O orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCCCGCCAACTACCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACCGGACCATGCGGTCCTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTGTCCCCCTGTACAGGGCAGGTTACCCACGCGTTACTCACCCGTCCGCCACT
>PC.634_27 FLP3FBN01DUWHJ orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCGGAATCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.634_28 FLP3FBN01ESJ5R orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGCCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGGGCCGTTACCCCGCCAACAACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGAAGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.635_29 FLP3FBN01BJH6S orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCGACCTCTCAGTCCGGCTACCGATCGTCGGCTTGGTGAGCCGTTACCTCACCAACTACCTAATCGGACGCGAGCCCATCTCCGAGCGATATAATCTTTGATACCAAAGGGATGTCCCTCCAGTATGTTATGCGGTATTAGCGACCGTTTCCAGCCGTTATTCCCCTCTCGAAGGTAGGTTGCTCACGTGTTACTCACCCGTCCG
>PC.634_30 FLP3FBN01BF988 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGCGAGAATGCTCACTCATGATACTATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCAGGTTGGATACGCGTTACTC
>PC.634_31 FLP3FBN01DRASA orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.634_32 FLP3FBN01EMWGJ orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTATCCAAGAGTATAAGGCAGGTTGGATACGCGTTACTCACCCG
>PC.634_33 FLP3FBN01BMERW orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGCGTTACTCACCCGTGCGCCGGTCG
>PC.634_34 FLP3FBN01A724Q orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGGTTCCGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>PC.634_35 FLP3FBN01AKXJC orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGCCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCTTTTATCTCACCAACTAGCTAATCAGACGCGGATCCATCCCATACCACCGGAGTTTTTCACACAGGGCCATGCAGCCTCGTGCGCTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCGGTATGGGGCAGGTTATCCACGCGTT
>PC.634_36 FLP3FBN01EC04T orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGCTGCCCCGCCAACAAGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAGGGAAAAGCCATGCGGCTTCCCCCGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGCCAGGTTGCCCACGCGTTACTCACCCGTCCGCC
>PC.634_37 FLP3FBN01AX7NV orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.607_38 FLP3FBN01CS9UE orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGTCGGCTACCGATCGTCGCCTTGGTGGGCTGTTACCCCTCCAACTAGCTAATCGGACGCGGATCCATCTCTCACCGATTTCTCTTTCCCTCTCTCCTCATGCGACAAAAGAGGCTCATGCGGTATTAGCAACCGTTTCCAGCTGTTATCCCCCTGTGAAAGGCAGGTT
>PC.634_39 FLP3FBN01A38B0 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGGCCACCCTCTCAGGCCGGCTACCCATCGTCGCCTTGGTAGGCCATTACCCTACCAACTAGCTAATGGGACGCGAGTCCATCTTTCAGCGTCAGGTCTCCCCAACTTTTCCTATATAAACATGCACTTATATAACTCATCCGGCATTAGCTCACCTTTCGGTAAGTTGTTCCAGTCTAAAAGGCAGGTCACTCACGTGTT
>PC.634_40 FLP3FBN01CB1XF orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.634_41 FLP3FBN01ARU00 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGTTTGGTGGGCCGTTACCCCGCCAACTGCCTAATCAGACGCGGGGCCATCTCATACCACCGGAGTTTTTCACACCGTGTCATGCGACACTGTGCGCTTATGCGGCATTAGCAGTCATTTCTAACTGTTATTCCCCTGTATGAGGCAGGTTCCCCACGCGTTACT
>PC.634_42 FLP3FBN01A24G3 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGTCACGCGAGCCTATCCTCATCCGACGGATCTTTAGATGGAACCAGATGCCTGATTCCATCGCCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCAAGTTGCTCACGCGTTACG
>PC.593_43 FLP3FBN01AYGTR orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGGCCGTATTTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAACTAGCTAATCAGACGCGAGGCCATCTTTCAGCGATAAATCTTTGACATAAATGCCATGCGACACCTATGTGTTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGAAAGGCAGGTTCCT
>PC.634_44 FLP3FBN01DPZMM orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTCTGGATGATCATCCTCTCAAACCATCTAACGATCGTCGACTTGGTGAGCCTTTACCTCACCAACTATCTAATCGTACGCAGGCCATTCCTAAAGCGCATAAATGCTTTAATCCGAAGATCATATGCGGTATTAGCCACTCTTTCGAGTAGTTATCCCTCACTTTAGGGTATGTTCCCACGCGTTACTCAGCCGTCCG
>PC.634_45 FLP3FBN01AJ7M7 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCCCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGGTTCTGGGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>PC.634_46 FLP3FBN01DV3TK orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGTCCATCATATACCACCGGAGTTTTTCACACAGAAACATGCGTCCCCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATAAGGCAGATTACCCACGTGTTACTCACCCG
>PC.634_47 FLP3FBN01AL42P orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGTCGGTTTGGTGGGCCGTTACCCCGCCAACTGCCTAATGGAACGCATGCCTATCTATCAGCGATGAATCTTTAGCAAATATCCCCATGCGGGGCCCCTGCTTCATGCGGTATTAGTCCGACTTTCGCCGGGTTATCCCCCTCTGATAGGCAAGTTGCATACGCGTTACTC
>PC.634_48 FLP3FBN01BYPDD orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.481_49 FLP3FBN01CTMCP orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGTCCATCATATACCACCGGAGTTTTTCACACCGGGGCATGCGCCCCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATATGGCAGGTTACCCACGCGTTACTCACCCG
>PC.607_50 FLP3FBN01BAJ58 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCGACCTCTCAGTCCGGCTACCGATCGTCGGCTTGGTGAGCCGTTACCTCACCAACTACCTAATCGGACGCGAGCCCATCTTCGAGCGATAAAATCTTTGATACCAGCAGGATGTCCTCCCGGTATGTTATGCGGTATTAGCGACCGTTTCCAGCCGTTATTCCCCTCTCGAAGGCAGGTTGCTCACGTGTTACTCACCCGTCCG
>PC.634_51 FLP3FBN01A4K2C orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.634_52 FLP3FBN01DRBCR orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTTGCCTTGGTGGGCCGTTACCCCGCCAACAAACTAATGGAACGCATCCCCATCGATTATCGAAATTCTTTAATAACAAGAAGATGCCTTCTCGTTATGCTATCCAGTATTAATCTTTCTTTCGAAAGGCTATCCCGGAATAATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCAT
>PC.634_53 FLP3FBN01BWH0G orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGGTGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTC
>PC.634_54 FLP3FBN01D0JVY orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCCGCCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCGCTGTTACCATGCGGTACTGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCACGGTTGCCCACGCGTTACTCACCCGTCCGCCGCT
>PC.634_55 FLP3FBN01EDR9T orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGGACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.634_56 FLP3FBN01CDTMI orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.634_57 FLP3FBN01AFERM orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACAACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGAAGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGG
>PC.634_58 FLP3FBN01BFQFB orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTGAGCCGTTACCTCACCAACCAGCTAATACAACGCATGCCCATCATCAACCACCGGAGTTTTCAACCCCATGCGATGCCGCATGAGATATTATGGGGTATTAGTACCAATTTCTCAGTGTTATCCCCCTGTTGATGGTAGGTTGCATACGCGTTACGCACCCG
>PC.635_59 FLP3FBN01D3JOZ orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAACCAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCTCTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGCTACAGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGG
>PC.634_60 FLP3FBN01DVIVE orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATGCGCCGCATGCCCATCCATAGCCGCCTCAACTTTCCTCCCCAAGGGATGCCCCTCAGGGAGTGCACGCGGATTTAGACCGGATTTCTCCGGATTATTCCCCTGACAAGGGTAGGTTGCATACGTGTTACTCACCCGTGCGCCGG
>PC.636_61 FLP3FBN01EQVOU orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGAGCCATTACCTCACCAACTAACTAATACGCCGCGGGATCATCTAAGAGCGACAGCAAAGCCGTCTTTCATTCATGAACCATGCGGTTCGCGAATCTATGCGGTATTAGCATCCGTTTCCGAATGTTATCCCCCACTCCTAGGCAGATTCCCC
>PC.634_62 FLP3FBN01DH4LG orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGGCCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGGGCCGTTACCCCGCCAACAACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGAAGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCA
>PC.354_63 FLP3FBN01EEU42 orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAGGACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.634_64 FLP3FBN01AE4G6 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.634_65 FLP3FBN01A559N orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGGGGGACCTTCCTCTCAGAACCCTTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGTCACGCGAGCCTATCCTCATCCGACGGATCTTTAGATGGAACCAGATGCCTGATTCCATCGCCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCAAGTTGCTCACGCG
>PC.355_66 FLP3FBN01BYAA8 orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCTCACCAACCAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTCACACCGAACCATGCGGTTCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.356_67 FLP3FBN01CSQSY orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACTAGCTAATCAGACGCGAGTCCATCTCAGAGCGATAAATCTTTGGCAGTCAGAGCCATGCGACCCAACTGCATTATGCGGTATTAGCAGCTGTTTCCAGCTGTTATTCCCCACTCCAAGGCTAGG
>PC.634_68 FLP3FBN01DTXOJ orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTCCCAATGTGGCCGGTCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCGCCAACCAGCTAATCAGACGCGGGCCCCTCCCATACCGCCGGAGCTTTCCCCACAAAGGCATGCGCCTCCCTGGTTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTATGGGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCC
>PC.481_69 FLP3FBN01EJD3A orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGCTGCCCCGCCAACCAGCTAATCAGACGCGGGCCCATCCCACACCGCCCGAAGGCTTTTCACACCGCTCCATGCGAAGCTGTGCGCTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTGTGGGGCAGGTTGCCCACGCGTT
>PC.634_70 FLP3FBN01CIW8K orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGACCGTCTCTCAGTTCCAATGTGGCCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGACTTGGTGAGCCGTTACCTCACCAACTATCTAATCAGACATGAGCCCATCTTTCAGCGGATTGCTCCTTTGATAACAGGATCATGCGATCCCGTCATTTCATGCGGTATTAGCACACCTTTCGATGTGTTATCCCCCTCTGAAAGG
>PC.481_71 FLP3FBN01BYVGX orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGCTGCCCCGCCAACAAGCTAATCAGACGCGGGCCCATCGCATACCACCGGAGTTTTTCACACCAAGCCATGCGGCTCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGGTATGCGGCAGGTTGCCCACGCG
>PC.635_72 FLP3FBN01BKL5B orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGTCACGCGAGCCTATCCTCATCCGACGAATCTTTAGATGGATTCAGATGCCTGATTCCATCACCATGGGGCGATTAGACGCCGTTTCCTAGCGATTATTCCCCTCGATGAGGGCAAGTTGCTCACG
>PC.634_73 FLP3FBN01DX9GS orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGTGGCCGGGATTGCTCCCTTTGGCGGCCCGGGGATGCCCCAAGGCCGCGTTACGCGGTATTAGACGGGGTTTCCCCCGCTTATCCCCCTGCCACGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCGG
>PC.593_74 FLP3FBN01BVHTK orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGAGCCGTTACCTCACCAACAACCTAATGGAACGCATCCCCATCCTTTACCGGAATCCTTTAATAATGAAACCATGCGGAATCATTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTAAAGGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCAG
>PC.607_75 FLP3FBN01A9PL2 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACTAGCTAATGTCACGCATGCCCATCCGTCTCCGGAGTCGCCTCCTTTGGCCGAAAGAAGATGCCTCCCCTCGGCCACATGCGGTATTAGGCCGGATTTCTCCGGGTTATCCCGCTGAAACGGGCAGGTTGCATACGCGTTACTCACCCGTGCGCCGG
>PC.634_76 FLP3FBN01C3PFP orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACTAGCTAATCAGACGCGAGTCCATCTCAGAGCGATAAATCTTTGGCAGTCAGAGCCATGCGACTCAACTGCATCATGCGGTATTAGCAGCTGTTTCCAACTGTTATTCCCCACTCCAAGGCAGGTTACT
>PC.634_77 FLP3FBN01A17GS orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATCAGACGCGAGTCCATCTCAGAGCGATAAATCTTTGATATCCAGAGCCATGCGACCCAGATATATTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTATGGGGCAGGTTGCCCACGCG
>PC.634_78 FLP3FBN01DM0XL orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGCCACCCTCTCAGGTCGGCTACCCGTCATCGCCTTGGTGAGCCGTTACCTCACCAACTAACTAATAGGCCGCGAGCCCATCCCCAAGCGCATTGCTGCTTTAATCTTTCGATACTATGCGGTATTAATCCCAGTTTCCCAGGGCTATCCCGCTCTCGGGGGCAGGTTACTCACGTGTTACTCACCCGTGCGCC
>PC.634_79 FLP3FBN01DFQEI orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGTCGGTTTGGTGGGCCATTACCCCGCCAACTGCCTAATGGAACGCATGCCTATCTATCAGCGATGAATCTTTAACAAATATTCCCATGCGGGACCCCTGTTTTATGGGGTATTAATCCGACTTTCGCCGGGCTATCCCCTCTGATAGGTAAGTTGCATACGCGTTACTCACCCGTGCGCCGGTCCG
>PC.607_80 FLP3FBN01DN5MU orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCGACCTCTCAGTCCGGCTACCCATCGTTGCCTTGGTGGGCCGTTACCTCACCAACTAGCTAATGGGTCGCGAGCCCATCCTATACCGATAAATCTTTTACCCCTGTATCATGCGATACTGTGGTCTTATACCGTATTAGCACAAATTTCTCTGTGTTATCCCGTTGTATAGGGCAGGTTGCTCACGTGTTACTCACCCGTTCG
>PC.607_81 FLP3FBN01B1L15 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCTTTACCCCACCAACTAGCTAATCAGACGCGGGTCGCTCTATCAGCGATAGCCTCTCTCGAGTCCGCCACCTTTCCTTCTGCCATCATGCGATGACTGAACCTTATGCGGTATTAGCACTGATTTCTCATTGTTATTCCCCT
>PC.634_82 FLP3FBN01AW01I orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACAAGCTAATCAGACGCGGAACCATCGTATACCACCGGAGTTTTTCACACAGGGCCATGCGGCCCCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATAGGGCAGGTTCCCCACGCGTTACTCACCCGTCCGCCG
>PC.634_83 FLP3FBN01BI38I orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCGGCACCGGGATGCCCCGACGCCGCGTCACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTGGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.593_84 FLP3FBN01DSC1C orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGGTCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTGAGCCGTTACCTCACCAACCAGCTAATGCGCCATAAGTCCATCCTCTACCAGTGCCTTGCAGCACTTTTAATACGGTCACCATGCAGTGTCCCTACCTATGCGGTCTTAGCTGCCGTTTCCAGCAGTTATCCCCCTGTAAAGGCCAGG
>PC.355_85 FLP3FBN01AIELL orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.634_86 FLP3FBN01D1T0J orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.607_87 FLP3FBN01EOHM5 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCTCGCCAACCAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCTCACAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCT
>PC.634_88 FLP3FBN01BMLLI orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGTCGGTTTGGTGGGCCGTTACCCCGCCAACTGCCTAATGGAACGCATGCCTATCTATCAGCGATGAATCTTTAACAAATATTCCCATGCGGGACCCCTGTTTTATGGGGTATTAATCCGACTTTCGCCGGGCTATCCCCTCCTGATAGGTAAGTTGCATACGCGTTACTCACCCGTGCGCCGGTCG
>PC.634_89 FLP3FBN01D74QJ orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCGG
>PC.634_90 FLP3FBN01EAAV0 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAACCAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCTCTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCG
>PC.634_91 FLP3FBN01BC3GH orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTACCTATCATTGCCTTGGTGGGCCGTTACCCCCCAACTAGCTAATAGGACGCATGCCCATCTGATACCTCGAATGATTTAATTATTAAAAGATGCCTTCAAATAATATTATGGGGTGTTAATCCACGTTTCCATGGGCTATCCCCCTGTATCAGCCAGGTTGCATACGCGTTACTCACCCGTGCGCCGG
>PC.634_92 FLP3FBN01D9A6M orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.481_93 FLP3FBN01A8V4Q orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAACTAGCTAATGTCACGCATGCCCATCCGTTGCCGGAATCGCTCCCTTTGGCCGCAGGGCCATGCAGCCCCGCGGCATTACGCGGTATTAGACGGGATTTCTCCCGATTATCCCCCTGCAACGGGCAGGTCGCATACGCGTTACT
>PC.634_94 FLP3FBN01DCU7E orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGGCCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGGGCCGTTACCCCGCCAACAACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGAAGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCA
>PC.634_95 FLP3FBN01AM688 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.607_96 FLP3FBN01DMRPA orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGATCAACCTCTCAGTCCGGCTACCGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACTAGCTAATCAGACGCGGGTCCATCTCATACCACCGGAGTTTTTCACACCAGGACATGCATCCCTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATGAGGCAGGTTACCCACGCGTTACT
>PC.607_97 FLP3FBN01BAZV6 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGGCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTACTGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCAAGGATGCCCCCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGAGGGCAGGTTGCATACGCGTTACTCACCCGTGCGCCGGTCGCCGGCGGAGCAAAG
>PC.634_98 FLP3FBN01DEBKQ orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGCAAGCCGTTACCTTACCAACCAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCTCTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGG
>PC.634_99 FLP3FBN01BAWOL orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGGCCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCCCGTACCACCGGAGTTTTCAAGAAAGGAACATGCGTCCCCTTCTGTTATGCGGTATTAGCACCTGTTTCCAGGTGTTATCCCCCGGTACGGGGCAGGTTCCCCACGCGTTACTCACCCGTTCGCCACTCGGGCAC
>PC.636_100 FLP3FBN01D7AVV orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAGTGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.634_101 FLP3FBN01B5T3T orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGGTGGGGGACCTTCCTCTCAGAACCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCGGCACCGGGATGCCCCGACGCCGCGTCACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCTGTGGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGTCG
>PC.635_102 FLP3FBN01B02WO orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.635_103 FLP3FBN01B2DRT orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCACTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGCCGCATGCCCATCCTTTACCGGATCGCTCCTTTGACATACCGATCATGCGGTCGGTATGTATTATGAGGTATTAGACGGAATTTCTCCCGATTATCCCTCTGTAAAGGGCAGGTCGCATACGTGTTACTCACCCG
>PC.634_104 FLP3FBN01A8KH9 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.634_105 FLP3FBN01AU6ML orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.634_106 FLP3FBN01AJ4R9 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTTGCCTTGGTGAGCCGTTACCTCACCAACTAGCTAATACAACGCATGCCCATCTTTAACCACCGGAGTTTTTAACCCAAGAAGATGCCTTCTCGAATGTTATGGGGAATTAGTACCAATTTCTCAGTGTTATGCCCCTGTTAAAGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCACCAAAG
>PC.481_107 FLP3FBN01B32LN orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTT
>PC.634_108 FLP3FBN01COYKA orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGGCCACCCTCTCAGGCCGGCTACCCATCGTCGCCTTGGTGGGCCATTACCCTACCAACTAGCTAATGGGACGCGAGTCCATCTTTCAGCGTCAGGTCTCCCCAACTTTTCCTATATAAACATGCACTTATATAACTCATCCGGCATTAGCTCACCTTTCGGTAAGTTGTTCCAGTCTAAAAGGCAGGTCACTCACGTGTT
>PC.593_109 FLP3FBN01C4BE9 orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGGAACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTATCCACGCGTTACTCACCCG
>PC.634_110 FLP3FBN01BU1CU orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGCCGTTCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCCACCAACTAACTAATCAGACGCAGGCCCATCCCTTACCGATAAATCTTTCACATCCCTGATATGCATCAGACTTGCATTATGAGGTTTTAATCCCGGTTTCCCGAGGCTATCCCTCTGTAAGGGGCAGGTTGCCTACGCGTTACTCACCCGTCCGCCG
>PC.634_111 FLP3FBN01AYTJU orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGGCCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGGGCCGTTACCCCGCCAACAACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGAAGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCA
>PC.634_112 FLP3FBN01DXJGO orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.636_113 FLP3FBN01BOOHE orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGGCCGATCACCCTCTCAGGTCGGCTACGTATCGTTGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATACGGCGCGGGTCCATCTATAAGTGACAGCCGAAACCGTCTTTCAACATTGAACCATGCGGTTCAATATATTATCCGGTATTAGCCCCGGTTTCCCGGAGTTATCCCAGTCTTATAGGTAGGTTACCCACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.634_114 FLP3FBN01BJL13 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGACCGTGTTCCAGTTCCAGTGTGACTGATCATCCTCTCAGACCAGTTAACCATCGTCGCCTTGGTAGGCCTTTACCCCACCAACCAGCTAATGGTACGCGGACTCATCAAAAAGCAACAGCATATGCAGAGGCCATCTTTCCCACATAAGATAGATATGCAGCGTATCCGGTATTAGCAGCCGTTTCCAGCTGTTATCCCAAACTTCTTGG
>PC.634_115 FLP3FBN01CTRWB orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCCTTACCCCACCAACTAGCTAATCAGACGCGGGTCCATCTCATACCACCGGAGTTTTTCACACTGGACCATGCAGTCCCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATGGGGCAGGTTGCCCACGCGTTACT
>PC.636_116 FLP3FBN01BJCG1 orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTC
>PC.481_117 FLP3FBN01CORWL orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGTCCATCCCATACCACCGGAGTTTTTCACACAGGAGCATGCGCTCCCGTGCGCTTATGCGGTATTAGCACCTGTTTCCAGGTGGTATCCCCCGGTATGGGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.634_118 FLP3FBN01C68BF orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGGCCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGAGCCGTTACCCCGCCAACAACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGAAGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.635_119 FLP3FBN01B449W orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCCCGTACCACCGGAGTTTTCAAGAAAGAAACATGCGTCCCCTTCTGTTATGCGGTATTAGCACCTGTTTCCAGGTGTTATCCCCCGGTACGGGGCAGGTTCCCCACGCGTTACTCACCCGTCCGCCACTAACTCATACAT
>PC.607_120 FLP3FBN01EZ4SI orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGTCCATCCCATACCACCGGAGTTTTTCACACAGGAGCATGCGCTTCCGTGCGCTTATGCGGTATTAGCACCTGTTTCCAGGTGGTATCCCCCCGGTGTGGGGCAGGTTGCCCACGCGTTACTCACCCGTCCG
>PC.634_121 FLP3FBN01DD5NY orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTTGCCTTGGTGAGCCGTTACCTCACCAACTAGCTAATACAACGCATGCCCATCTTTAACCACCGGAGTTTTTAACCCAAGAAGATGCCTTCTCGAATGTTATGGGGAATTAGTACCAATTTCTCAGTGTTATGCCCCTGTTAAAGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCC
>PC.355_122 FLP3FBN01BHKSP orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAACCAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGTACCATGCGGTACTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCAGGTTACCCACGCGTTACTCACCCGTCCGCCACTAAG
>PC.634_123 FLP3FBN01DHYS5 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAACTAGCTAATCAGACGCGAGGCCATCTTTCAGCGATAAATCTTTGACATAAATGCCATGCGACACCTATGTGTTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGAAAGGCAGGTTCCT
>PC.634_124 FLP3FBN01D837F orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGGCCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGGGCCGTTACCCCGCCAACAACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGAAGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCA
>PC.634_125 FLP3FBN01DC9HR orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGTCACGCGAGCCTATCCTCATCCGACGGATCTTTAGATGGAACCAGATGCCTGATTCCATCGCCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCAAGTTGCTCACGCG
>PC.634_126 FLP3FBN01DLDHU orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTACCTATCATTGCCTTGGTGGGCCGTTACCCCCCAACTAGCTAATAGGACGCATGCCCATCTGATACCTCGAACGATTTAATTATTATAAGATGCCTTACAATAATATTATGGGGTGTTAATCCACGTTTCCATGGGCTATCCCCCTGTATCAGCCAGGTTTGCATACGCGTTACTCACCCGTGCGCCGG
>PC.634_127 FLP3FBN01D6Q98 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.634_128 FLP3FBN01CNKON orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAACCAGCCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCTCTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTTACAGGTAGGTTGCCATACGCGTTACGCACCCGTGCGCCGGTCG
>PC.634_129 FLP3FBN01BCWX5 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTACTGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCAAGGATGCCCCCAAGGGATATACGCGGGATTAGCCCTCCCTTTCGGAAGGTTCGTCCCCCTGTGGAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGGCAG
>PC.635_130 FLP3FBN01EA91G orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAACCAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGTACCATGCGGTACTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCAGGTTACCCACGCGTTACTCACCCGTCCGCCACTAAG
>PC.635_131 FLP3FBN01B06QS orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTGAGCCGTTACCTCACCAACCAGCTAATACAACGCATGCCCATCTTCCACCACAAAAAGCTTTCAACCCAGAGAGATGCCTCTCCGAATTATATGGGGTATTAGTACCAATTTCTCAGTGTTATCCCCCTGTGAAAGGTAGGTTGCATACGCGTTACGCACCCGTCCGCCGGTCG
>PC.634_132 FLP3FBN01D0SFK orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAACTAGCTAATCAGACGCGAGGCCATCTTTCAGCGATAAATCTTTGACATAAATGCCATGCGACACCTATGTGTTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGAAAGGCAGGTTCCT
>PC.634_133 FLP3FBN01DVTEG orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCAACCTCTCAGTCCGGCTATGGATCGTTGACTTGGTAGGCCGTTACCCCACCAACTATCTAATCCAACGCGAGCCCATCCTTCGGCACCTCAGTTTTGGTATTTCTATCATGCGGTAAAAATACGTTATGCGGTATTACTGTCCGTTTCCAGACACTATCCCCCTCCGAAGGGCAGGTTGCTCACGCGTTACTCACCCGTCCGCC
>PC.634_134 FLP3FBN01CECHO orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGATCGACCTCTCAGTCCGGCTACCCGTCGTCGGCTAGGTGAGCTGCTACCTCACCTACTACCTGATGGGCCGCGACCCCATCCCAGACCGCAAGAGCTTTCATCCCTCCGCCATGCGGTGGAAGGATAGTATCCGGTATTAGCTGCCGTTTCCGGCAGTTATCCCGATGTCTGGGGCAGGTTGGTCACGTGTT
>PC.634_135 FLP3FBN01ES2BT orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGTCGGTTTGGTGGGCCGTTACCCCGCCAACTGCCTAATGGAACGCATGCCTATCTATCAGCGATGAATCTTTAGCAAATATCCCCATGCGGGGCCCCTGCTTCATGCGGTATTAGTCCGACTTTCGCCGGGTTAGTCCCCTCCTGATAGGTAAGTTGCATACGCGTTACTCACCCGTCGCG
>PC.636_136 FLP3FBN01BJO3F orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACCCCGCCAACTAGCTAATGCGCCGCATGGCCATCCGTAGCCGGTGTTACCCTTTAAACCCCAAGAGATGCCTCTCGGAGTTATTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTTCATACGTGTTACTCACCCGTGCGCCGGTCGCCA
>PC.635_137 FLP3FBN01AMOUT orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTGAGCCGTTACCTCACCAACCAGCTAATACAACGCATGCCCATCATCAACCACCGGAGTTTTCAACCCCATGCGATGCCGCATGAGATATTATGGGGTATTAGTACCAATTTCTCAGTGTTATCCCCCTGTTGATGGTAGGTTGCATACGCGTTACGCACCCG
>PC.356_138 FLP3FBN01CNT1Q orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGAGTCCATCTCAGAGCGATAAATCTTTGATATCCAGAGCCATGCGACCCAGATATATTATGCGGTATTAGCAGCTGTTTCCAGCTGTTATTCCCCATCCAAGGCAGGTT
>PC.634_139 FLP3FBN01DR739 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGACCGTCTCTCAGTTCCAATGTGGCCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGACTTGGTGAGCCGTTACCTCACCAACTATCTAATCAGACATGAGCCCATCTTTCAGCGGATTGCTCCTTTGATAACAGGATCATGCGATCCCGTTCATTTCATTGCGGTATTAGCACACCTTTCGATGTGTTATCCCCCTCTGAAAGG
>PC.634_140 FLP3FBN01BBDL9 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCTTTTACTCCACCAACTAGCTAATCAGACGCGGAACCATCGTATACCACCTCAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATAAGGCAGGTTACCCACGCGTTACT
>PC.634_141 FLP3FBN01CLP3T orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
ATGGACCGTGTCTCAGTTCCATTGTGGCCGTTCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTAGGCCTTTACCCCACCAACTAGCTAATGCGCCGCAGGCTCATCCATCAGTGATGCCAGGAGCATCTTTAAACTTTCGTCCTATCCGGTATTAGCGATCGTTTCCAATCGTTGTCCCCGTCTGATGGGCAGATTACCTACGTGTTACTCACCCGTTCG
>PC.355_142 FLP3FBN01C1O95 orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCGCCAACCAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTCACACCGAACCATGCGGTTCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGTAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.634_143 FLP3FBN01DHLQN orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTTACCCTCTCAGGCCGGCTACGCATCATCGCCTTGGTGGGCCGTTACCTCACCAACTAGCTAATGCGCCGCAGGTCCATCCATGTTCACGCCTTGATGGGCGCTTTAATATACTGAGCATGCGCTCTGTATACCTATCCGGTTTTAGCTACCGTTTCCAGCAGTTATCCCGGACACATGGGCAGGTT
>PC.634_144 FLP3FBN01BA44D orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTTCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTAGGCCTTTACCCCACCAACTAGCTAATGCGCCGCAGGCTCATCCATCAGTGATGCCAGGAGCATCTTTAAAACTTTCGTCCTATCCGGTATTAGCGATCGTTTCCAATCGTTGTCCCCGTCTGATGGGCAGATCACCTACGTGTTACTCACCCG
>PC.634_145 FLP3FBN01EHG6O orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTG
>PC.356_146 FLP3FBN01EE1C3 orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACCGGGCCATGCGGCCCTGTGCGCTTATGCGGTATTAGCACCTGTTTCCAGGTGTTATCCCCCTGTGTAAGGCAGGTTACCCACGCGTTACTCACCCGTCCGCC
>PC.634_147 FLP3FBN01AER1M orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTGAGCCGTTACCTCACCAACTAGCTAATGCGCCGCGGGCCCATCTCATAGCGGATTGCTCCTTTAATGCTACTTCATGCGAAGCTACAATCTTATGCGGTATTAATCTCCCTTTCGGAAGGCTATTCCCCTCTATGAGGCAGGTTCG
>PC.636_148 FLP3FBN01A1RXA orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTACTGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCAAGGATGCCCCCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGG
>PC.634_149 FLP3FBN01B5Z5E orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTTCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTAGGCCTTTACCCCACCAACTAGCTAATGCGCCGCAGGCTCATCCATCAGTGATGCCAGGAGCATCTTTAAACTTTCGTCCTATCCGGTATTAGCGATCGTTTCCAATCGTTGTCCCCGTCTGATGGGCAGATCACCTACGTGTTACTCACCCG
>PC.634_150 FLP3FBN01EB7G2 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGACCGTGTTCCAGTTCCAGTGTGACTGATCATCCTCTCAGACCAGTTAACCATCGTCGCCTTGGTAGGCCTTTACCCCACCAACCAGCTAATGGTACGCGGACTCATCAAAAAGCAACAGCATATGCAGAGGCCATCTTTCCCACATAAGATAGATATGCAGCGTATTCGGTATTAGCAGCCGTTTCCAGCTGTTATCCCAAACTT
>PC.607_151 FLP3FBN01AYGA0 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTACTGCCTAATGCGCCGCATGCCCATCCCTGTCCGGCCGAAGCCTTTCCTGCCTCCGGGATGCCCCGGTGGCATGTACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGG
>PC.481_152 FLP3FBN01AL8JX orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAACCAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTCACACCGGGGCATGCGCCCCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATATGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.355_153 FLP3FBN01B7E8J orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAACCAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGTACCATGCGGTACTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCAGGTTACCCACGCGTTACTCACCCGTCCGCCACTAAG
>PC.634_154 FLP3FBN01D6BHF orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGTCACGCGAGCCTATCCTCATCCGACGAATCTTTAGATGGATTCAGATGCCTGATTCCATCACCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCAAGTTGCTCACG
>PC.634_155 FLP3FBN01BLTKB orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTACCTATCATTGCCTTGGTGGGCCGTTACCCCCCAACTAGCTAATAGGACGCATGCCCATCTGATACCTCGAATGATTTAATTATTAAAAGATGCCTTCAAATAATATTATGGGGTGTTAATCCACGTTTCCATGGGCTATCCCCCTGTATCAGCCAGGTTGCATACGCGTTACTCACCCGTGCGCCGG
>PC.636_156 FLP3FBN01EK2JX orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTACTGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCGGGGATGCCCCCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGGCAGAGAG
>PC.634_157 FLP3FBN01EF15V orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGTCGGTTTGGTGGGCCGTTACCCCGCCAACTGCCTAATGGAACGCATGCCTATCTATCAGCGATGAATCTTTAGCAAATATCCCCATGCGGGGCCCCTGCTTCATGCGGTATTAGTCCGACTTTCGCCGGGTTATCCCCCTCTGATAGGCAAGTTGCATACGCGTTACTC
>PC.634_158 FLP3FBN01BB8KH orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTGGTCTTGGTGAGCCGTTACCCCACCAACTAACTAATACAACGCATGCCCATCCATTACCACCGGAGTTTTCAACCCAAGAAGATGCCTTCTCGAATGTTATGGGGAATTAGTACCAATTTCTCAGTGTTATGCCCCTGTTAAAGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCACTAA
>PC.481_159 FLP3FBN01AMFGP orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTCATACCACCGGAGTTTTTCACACCGGAACATGTGTCCCTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATGAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.481_160 FLP3FBN01A6LEJ orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTT
>PC.634_161 FLP3FBN01BUCNS orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAACCAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCTCTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.634_162 FLP3FBN01ET0T4 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGGCCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGGGCCGTTACCCCGCCAACAACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGAAGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCA
>PC.634_163 FLP3FBN01B0AX7 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATGCGCCGCATGCCCATCCTTTGCCGGAATTGCTTCCTTTGACTCCCAAACCATGTGGTAAGGGAGTGTTATGCAGTATTAGTCGGAATTTCTCCCGGTTATCCTCCTGCAAAGGGCAGGTTGCATACGTGTTACTCACCCGTGCG
>PC.355_164 FLP3FBN01BM4QE orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACTAACTAATGCGCCGCATGCCCATCCATGACCGGATCGCTCCTTTGACTCCCGAGAGATGTCTCCCGGGGGTGTTATGCGGTATTAGACGGAATTTCTCCCGCTTATCCCCCTGTCATGGGCAAGTTGCATACGTGTTACTC
>PC.593_165 FLP3FBN01EPSX1 orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCCACCAACCAGCTAATCAGACGCGGGCTCATCTTATACTACCGGAGTTTTTCACACAGAAACATGCGTCCCCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATAAGGCAGATTACCCACGTGTTACT
>PC.634_166 FLP3FBN01DZ97Y orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGTAGTCTTGGTGGGCCGTTACCCCCCAACAAACTAATGGAACGCATCCCCATCTCATACCGAATTTCTTTAATATAAAACCCATGCGGGAAATATATGCTATCGGATATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTATGAGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCATCGG
>PC.634_167 FLP3FBN01BFDT5 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGGCCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGGGCCGTTACCCCGCCAACAACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGAAGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCA
>PC.634_168 FLP3FBN01BKTRL orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGGCTTGGTGGGCCGTTACCCCGCCAACTACCTAATGTCGCGCGTGCCCGTCCCGTACCACCGGAATTTTAAATCGAGAGCCATGCGGCTCTCGAGTATCATGGGATGTTAGTCCACGTTTCCGCGGGTTATCTCCCGGTACGGGGTTGGTTGCACACGTGTTACTCACCCGTGCGCCGGTCGCCGGCAGG
>PC.634_169 FLP3FBN01C9NT9 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGTCACGCGAGCCTATCCTCATCCGACGGATCTTTAGATGGAACCAGATGCCTGATTCCATCGCCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCAAGTTGCTCACGCGTTACG
>PC.634_170 FLP3FBN01BE65F orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGCTGCCCCGCCAACAAGCTAATCAGACGCGGGCCCCTCCCATACCGCCGGAGCTTTCCCCACAAAGGCATGCGCCTCCCTGGTTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTATGAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>PC.607_171 FLP3FBN01AY2QI orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGCTGCCCCGCCAACAAGCTAATCAGACGCGGGCCCCTCCCATACCGCCGGAACTTTCCCCAGAAAGGCATGCGCCTCCCTGGTTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTATGGGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCC
>PC.636_172 FLP3FBN01BYYFZ orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGCGAGAATGCTCACTCACGATACTATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCAGGTTGGATACGCGTTACTCACCCGTGCGCCGG
>PC.634_173 FLP3FBN01AFUF1 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTTGCCTTGGTGAGCCGTTACCTCACCAACTAGCTAATACAACGCATGCCCATCTTTAACCACCGGAGTTTTTAACCCAAGAAGATGCCTTCTCGAATGTTATGGGGAATTAGTACCAATTTCTCAGTGTTATGCCCCTGTTAAAGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCACCAAAG
>PC.634_174 FLP3FBN01B7I8E orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTTGCCTTGGTGAGCCGTTACCTCACCAACTAGCTAATACAACGCATGCCCATCTTTAACCACCGGAGTTTTTAACCCAAGAAGATGCCTTCTCGAATGTTATGGGGAATTAGTACCAATTTCTCAGTGTTATGCCCCTGTTAAAGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCACCAAAG
>PC.356_175 FLP3FBN01EIDVP orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGACATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCTTGGG
>PC.634_176 FLP3FBN01DOR0M orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGTTGACTTGGTGGGCCGTTACCCCGCCAACTATCTAATGGAACGCATCCCCATCGATAACCGAAATTCTTTAATAGTGAAACCATGCGGAAATACTATACTATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTTATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCATC
>PC.634_177 FLP3FBN01DO8MU orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAACCAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCTCTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCG
>PC.634_178 FLP3FBN01ATR31 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAACCAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCTCTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCG
>PC.634_179 FLP3FBN01CO4O9 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAACCAGCTAATACAACGCGTGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCTCTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCG
>PC.356_180 FLP3FBN01CZUXO orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTCTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCACGGTTCCATACGTGTTACTCACCCGTGCGCCGG
>PC.634_181 FLP3FBN01A9EX0 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCGGCACCGGGATGCCCCGATGCCGCGTCACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTGGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.634_182 FLP3FBN01BYSJT orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.634_183 FLP3FBN01AV5YV orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGGCCACCCTCTCAGGCCGGCTACCCATCGTCGCCTTGGTAGGCCATTACCCTACCAACTAGCTAATGGGACGCGAGTCCATCTTTCAGCGTCAGGTCTCCCCAACTTTTCCTATATAAACATGCACTTATATAACTCATCCGGCATTAGCTCACCTTTCGGTAAGTTGTTCCAGTCTAAAAGGCAGGTCACTCACGTGTT
>PC.634_184 FLP3FBN01DU3OR orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGGTTCTGGGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTACGGGCAGGTTGCCA
>PC.636_185 FLP3FBN01EWW35 orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAACCAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCTCTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCG
>PC.634_186 FLP3FBN01A3LZ0 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGTCGGTTTGGTGGGCCGTTACCCCGCCAACTGCCTAATGGAACGCATGCCTATCTATCAGCGATGAATCTTTAGCAAATATCCCCATGCGGGGCCCCTGCTTCATGCGGTATTAGTCCGACTTTCGCCGGGTTATCCCCTCTGATAGGTAAGTTGCATACGCGTTACTCACCCGTGCG
>PC.634_187 FLP3FBN01D55K5 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACAAGCTAATCAGACGCGAGCTCATCTCAGAGCGATAAGTCTTTGACAGCCAACCCCATGCGGGATAGCTGTATCATGCGGTATTAGCGGCTGTTTCCAGCCGTTATTCCCCACTCCAAGGCAGATTG
>PC.636_188 FLP3FBN01CHNXI orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCAACCTCTCAGTCCGGCTATGGATCGTTGACTTGGTAGGCCGTTACCCCACCAACTATCTAATCCAACGCGAGCCCATCCTTCGGCACCTCAGTTTTGGTATTTCTATCATGCGGTAAAAATACGTTATGCGGTATTACTGTCCGTTTCCAGACACTATCCCCCTCCGAAGGGCAGGTTCCCCACGCGTTACTCACCCGTCCGCCG
>PC.634_189 FLP3FBN01EETLW orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATGCGCCGCATGCCCATCCATAGCCGCCTCAACTTTCCTCCCCAAGGGATGCCCCTCAGGGAGTGTACGCGGATTTAGACCGGATTTCTCCGGATTATTCCCCTGCTATGGGCAGGTTGCATACGTGTTACTCACCCGTGCG
>PC.634_190 FLP3FBN01B4E5A orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGTCACGCGAGCCTATCCTCATCCGACGAATCTTTAGATGGATTCAGATGCCTGATTCCATCACCATGGGGCGATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCAAGTTGCTCACG
>PC.356_191 FLP3FBN01ET99E orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCCCGCCAACCACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCACCGGACCATGCGGTCCTGCGGTCTTATGCGGTATTAGCAGTCATTTCTGACTGTTGTCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCC
>PC.634_192 FLP3FBN01BS7JZ orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGGCCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGGGCCGTTACCCCGCCAACAACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGAAGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCC
>PC.634_193 FLP3FBN01DW7MZ orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAACCAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCTCTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCG
>PC.634_194 FLP3FBN01BKWR9 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTGAGCCGTTACCTCACCAACTAACTAATGCGCCGCGGGCCCATCTCATAGCGGATTGCTCCTTTAATTGCTACTTCATGCGAAGCTGCAATCTTATGCGGTATTAATCTCCCTTTCGGAAGGCTATTCCCCTCTATGAGGCAGGTT
>PC.636_195 FLP3FBN01DLUYP orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGAGAAAATGCTCACTCATGATACCATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCAGGTTGGATACGCGTTACTCACCCGTGCGCCGG
>PC.634_196 FLP3FBN01DBVV5 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTTACCCTCTCAGGCCGGCTACGCATCATCGCCTTGGTGGGCCGTTACCTCACCAACTAGCTAATGCGCCGCAGGTCCATCCATGTTCACGCCTTGATGGGCGCTTTAATATACTGAGCATGCGCTCTGTATACCTATCCGGTTTTAGCTACCGTTTCCAGCAGTTATCCCGGACACATGGGCAGGTT
>PC.607_197 FLP3FBN01B5DKJ orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCCTATACCACCGGAGTTTTTCACACCGGAGCATGCGCTCCTGTGCGCTTATGCGGTATTAACAGTCGTTTCCAACTGTTATCCCCCTGTATAGGGCAGGTTACCCACGCGTTACTCACCCGTCCGCCACT
>PC.634_198 FLP3FBN01ASQES orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGTCACGCGAGCCTATCCTCATCCGACGGATCTTTAGATGGAACCAGATGCCTGATTCCATCGCCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCAAGTTGCTCACGCG
>PC.634_199 FLP3FBN01DPNWR orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACTAGCTAATGCGCCGCATGGCCATCCGTAGCCGGTGTTACCCTTTAAACCCCAGGAGATGCCTCTCGGAGTTATTACGCGATATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTACGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.634_200 FLP3FBN01BLIY9 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCATTGCCTTGGTGGGCCGTTACCCCCCAACTAGCTAATAGGACGCATGCCCATCTGATACCTCGAACGATTTAATTATTATAAGATGCCTTACAATAATATTATGGGGTGTTAATCCACGTTTCCATGGGCTATCCCCCTGTATCAGCCAGGTTGCATACGCGTTACTCACCCGTGCGCCGG
>PC.634_201 FLP3FBN01DQGLV orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGATCGACCTCTCAGTCCGGCTACCCGTCGTCGGCTAGGTGGGCCACTGCCCCGCCTACTACCTGATGGGCCGCGACCCCATCCTGCACCGCCGGAGCTTTCATCCGCTCCCCATGCGGGGTGCGGATAGTATCCGGTATTAGCTGCCGTTTCCGGCAGTTATCCCGATGTGCAGGGCAGGTTGGTCACGTGTTACT
>PC.354_202 FLP3FBN01CUQNU orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTATCCACGCGTT
>PC.634_203 FLP3FBN01B0QGX orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGGTGGCCGATCGCCCTCCTCAGGCCGGCTACCCATCGCAGGCTAGGTGGGCCGTTGCCCCGCCTACTACCTAATGGGCCGCGACCCCATCCCGCACCGTCGGAGCTTTCCTCCGTGGCGCATGCGCGCCTCGGAGAGTATCCGGTATTAGCCGCCGTTTCCGGCGGTTGTCCCGGGGTGCGGGGCAGGTTGGTCACGTGTTACTCACCCGTTCGCC
>PC.634_204 FLP3FBN01A3PGP orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGACCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCGGCACCGGGATGCCCCGATGCCGCGTCACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTGGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGG
>PC.634_205 FLP3FBN01ALY2B orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGTCACGCGAGCCTATCCTCATCCGACGAATCTTTAGATGGATTCAGATGCCTGATTCCATCACCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCAAGTTGCTCACG
>PC.634_206 FLP3FBN01BT8QN orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTCTGGATGATCATCCTCTCAAACCATCTAACGATCGTCGACTTGGTGAGCCTTTACCTCACCAACTATCTAATCGTACGCAGGCCATTCCTAAAGCGCATAAATGCTTTAATCCGAAGATCATATGCGGTATTAGCCACTCTTTCGAGTAGTTATCCCTCACTTTAGGGTATGTTCCCACGCGTTACTCAGCCGTCCG
>PC.634_207 FLP3FBN01DATE7 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGCGGGGGGTTAACCTCTCAGTCCCCTATGTATCGTTGCCTTGGTGAGCCGTTACCTCACCAACTAGCTAATACAACGCATGCCCATCTTTAACCACCGGAGTTTTTAACCCAAGAAGATGCCTTCTCGAATGTTATGGGGAATTAGTACCAATTTCTCAGTGTTATGCCCCTGTTAAAGGTAGTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCACCAAAG
>PC.635_208 FLP3FBN01A5XYK orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCGCCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTAGGCTTCTACCCCACCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCTCAGTTTTTACCACTGTACCATGCAGTACCGTGGTCTTATGCGGTATTAGCAATCATTTCTAATTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>PC.607_209 FLP3FBN01BAT5U orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGCTGCCCCGCCAACAAGCTAATCAGACGCGGGCCCCTCCCATACCGCCGGAACTTTCCCCAGAAAGGCATGCGCCTCCCTGGTTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTATGGGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCC
>PC.634_210 FLP3FBN01ECTYY orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGCGCCGCATGCCCATCCTCCCCCGATGAATCTTTCCTCCTCCGGAGATGCCTCCGGAGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGGGAAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.634_211 FLP3FBN01A2T1M orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTTACCCTCTCAGGCCGGCTACGCATCATCGCCTTGGTGGGCCGTTACCTCACCAACTAGCTAATGCGCCGCAGGTCCATCCATGTTCACGCCTTGATGGGCGCTTTAATATACTGAGCATGCGCTCTGTATACCTATCCGGTTTTAGCTACCGTTTCCAGCAGTTATCCCGGACACATGGGCAGGTT
>PC.634_212 FLP3FBN01CBDAE orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGTCACGCGAGCCTATCCTCATCCGACGGATCTTTAGATGGAACCAGATGCCTGATTCCATCGCCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCAAGTTGCTCACGCG
>PC.634_213 FLP3FBN01D8Y20 orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGACCTTCCTCTCAGAACCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCGCCAACCAGCTAATGCGCCGCATGCCCATCCTTTACCGGATCGCTCCTTTGACATACCGATCATGCGGTCGGTATGTGTTATGAGGTATTAGACGGAATTTTCTCCCGATTATCCCTCTGTAAAGGGCAGGTCGCATACGTGTTACTCACCCG
>PC.634_214 FLP3FBN01A6MXL orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGATGTCTTGGTGGGCCGTTACCCCGCCAACAAACTAATGGAACGCATCCCCATCGATTATCGAAATTCTTTAATAACAAGAAGATGCCTTCTCGTTATGCTATCCAGTATTAATCTTTCTTTCGAAAGGCTATCCCGGAATAATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.635_215 FLP3FBN01ED1E9 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGTCACGCGAGCCTATCCTCATCCGACGGATCTTTAGATTGGAACCAGATGCCTGATTCCATCGCATGGGGCATTAGACGCCGTTTCCAGCCGATTATTCCCCTGATGAGGGCAAGTTGCTCACGCG
>PC.356_216 FLP3FBN01AYIUY orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAGACTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTGAGGCAGGTTATCCACG
>PC.634_217 FLP3FBN01A0UCW orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGGCCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGGGCCGTTACCCCGCCAACAACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGAAGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAGGTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCC
>PC.636_218 FLP3FBN01CDJ70 orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCTTTGGTGGGCCTCTACCCCGCCAACTGGCTAATCAGACGCGGGCCCCTCCCATACCACTATTGTTTTTCACACAGAACCATGCGGTCCCGTGCGCTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTATGGGGCAGGTTGCCCACGCGTTACTCACCCGTCCG
>PC.481_219 FLP3FBN01EFY7W orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACAAGCTAATCAGACGCGGATCCATCGTATACCACCAAAAGCTTTAGCTTTTTGTTTTCCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCT
>PC.634_220 FLP3FBN01AGI0O orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGGCCACCCTCTCAGGCCGGCTACCCATCGTCGCCTTGGTAGGCCATTACCCTACCAACTAGCTAATGGGACGCGAGTCCATCTTTCAGCGTCAGGTCTCCCCAACTTTTCCTATATAAACATGCACTTATATAACTCATCCGGCATTAGCTCACCTTTCGGTAAGTTGTTCCAGTCTAAAAGGCAGGTCACTCACGTGTT
>PC.356_221 FLP3FBN01CUN6D orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACTAGCTAATGCGCCGCATGGCCATCCGTAGCCGGTGTTACCCTTTAAACCCCAAGAGATGCCTCTCGGAGTTATTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTATGGGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCC
>PC.636_222 FLP3FBN01EFPI1 orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.634_223 FLP3FBN01A8V8L orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGCCGATCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCCACCAACTAGCTAATCAGACGCAGGCCCATCCTTTGCCGATAAATCTTTGACCAAACAGCCATGTGACTATTTAGTATTATGAGGTTTTAATCATCGTTTCCAATGGCTATCCCTCTGCAAAGGGCAGGTTGCCTACGCGTTACGT
>PC.354_224 FLP3FBN01BZOEE orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCG
>PC.593_225 FLP3FBN01EGX4W orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATGCGCCGCATGGCCATCCGTAGCCGGTGTTACCCTTTAAACCCCAAGAGATGCCTCTCGGAGTTATTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTACGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.635_226 FLP3FBN01D2AYU orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCCCGCCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCACCGGACCATGCGGTCCTGTGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCGCT
>PC.634_227 FLP3FBN01BHVDZ orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCGGCACCGGGATGCCCCGACGCCGCGTCACGCGGTATTAGACGTAATTTCTTCCGCTTATCCCCCTGTGGCGGGCAGGTTGCATACGCGTTACTCACCCGTGCGCCGGTCG
>PC.481_228 FLP3FBN01ED5UR orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTT
>PC.634_229 FLP3FBN01CU37V orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCAACCTCTCAGTCCGGCTACCGATCGTCGACTTGGTGGGCCGTTACCCCGCCAACTATCTAATCGGACGCGAGCCCATCTTTCAGCGGATTGCTCCTTTGATCCCCAAGGGATGTCCCTCAAGGATGTCATGCGGTATTAGCGTTCGTTTCCAAACGTTATCCCCCTCTGAAAGGCAGGTTGCTCACGTGTTACTCACCCGTCCG
>PC.354_230 FLP3FBN01B86XL orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCACCAACTGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGGAACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTATCCACGCGTTACTCACCCGTCCG
>PC.355_231 FLP3FBN01DJ1VW orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.634_232 FLP3FBN01AKD0L orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACTCTCTCAAGCCGGCTACTGATCGTTGTCTTGGTAGGCCGTTACCCTGCCAACTAACTAATCAGACGCGGGCCCATCCTGTACCACCGTGGTTTTCCCTGCTGTGCCATGCGGCACAGCAGGCTTATGCGGTATTAGCAGCCATTTCTGGCTGTTGTCCCCCGGTACAGGGCAGGTTGCCCACGCG
>PC.356_233 FLP3FBN01D6H8C orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAGGGAAAAGCCATGCGGCTTCCCCCGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGCCAGGTTGCCCACGCGTTACTCACCCGTCCGCCACT
>PC.635_234 FLP3FBN01DQFJG orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGACTTGGTGGGCCTCTACCCCGCCAACTATCTAATCAGCCGCGAGTCCATCTCCCATCGGATTTCTCCTTTGACTATCCAAGGATGCCCTCAAATAGTATTATGCGGTATTAGCGTCCGTTTCCAGACGTTATCCCCCTTTGAAAGGTAGGTTACTCACGTGTTACTCACCCGTCCG
>PC.635_235 FLP3FBN01AGC0G orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.634_236 FLP3FBN01D0FVD orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGGCCACCCTCTCAGGCCGGCTACCCATCGTCGCCTTGGTAGGCCATTACCCCACCAACTAGCTAATGGGACGCGAGTCCATCTTTCAGCGTCAGGTCTCCCCAACTTTTCCTATATAAACATGCACTTATATAACTCATCCGGCATTAGCTCACCTTTCGGTAAGTTGTTCCAGTCTAAAAGGCAGGTCACTCACGTGTT
>PC.634_237 FLP3FBN01BG1KW orig_bc=ACAGAGTCGGCT new_bc=ACAGAGTCGGCT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.356_238 FLP3FBN01BW4TT orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGACCGTATCTCAGTTCCAATGTGGCCGATCAGCCTCTCAGCTCGGCTATGCATCGTTGCCTTGGTAGGCCATTGCCCCACCAACTAGCTAATACACCGCAAGCTCATCCTAAGGTGAAGCAAACGCTTCTTTTAACATATCTAGTTATCTAGTTATGTATCATCCGGTATTAGCGTTCGTTTCCAAACGTTATCCCAGTCCCTAGGGTAGATTACCC
>PC.635_239 FLP3FBN01BOFGD orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGGCCGTGTCCCAGTCCCAATGTGGCCGGTCGCCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTAGGCTTCTACCCCACCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCTCAGTTTTTACCTCTGTACCATGCGGTACTGGGGTCTTATGCGGTATTAGCAATCATTTCTAATTGTTATCCCCCTGTGTAAGGCAGGTTGCTCACGCGTTACTCACCCGTCCGCC
>PC.355_240 FLP3FBN01CWLZU orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAACCAGCTAATCAGACGCGGGCCCATCCCGCACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTT
>PC.593_241 FLP3FBN01AMXMK orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATCAGACGCGGGTCCATCTTATACCACCGGAGTTTTTTCACACTGTACCATGCAGTACTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATAAGGCAGGTTACCCACGCGTTACTCACCCG
>PC.354_242 FLP3FBN01BHNXZ orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.356_243 FLP3FBN01DRT2O orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTTACT
>PC.607_244 FLP3FBN01ATU7A orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCACCAACTACCTAATGCGCCGCATGCCCATCCTTTACCGGATCGCTCCTTTGACATACCGATCATGCGGTCGGTATGTGTTATGAGGTATTAGACGGAATTTCTCCCGATTATCCCTCTGTAAAGGGCAGGTCGCATACGTGTTACTC
>PC.607_245 FLP3FBN01DQMNE orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTGCCAACAAGCTAATCAGACGCGGGTCCATCTCGCACCACCGGAGTTTTCAGGGCAGGGGCATGCGCCCCCTCCCGTTATGCGGTGTTAGCACCTATTTCTGGGTGTTATCCCCCAGTGTAAGGCAGGTTACCCACGCGTTACTCACCCGTCCGCCACTAAG
>PC.635_246 FLP3FBN01BGO68 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.355_247 FLP3FBN01EC56I orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGAGCTCTTACCTCACCAACTAGCTAATCAGACGCGGGCCCATCTTACACCACCTCAGTTTTTTCCACAAGGTCATGCGACCCTGTGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACT
>PC.636_248 FLP3FBN01D9EK7 orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCATGTGGGGGCCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTCACTCACGTGTTACT
>PC.356_249 FLP3FBN01BXX60 orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTCCACCCTCTCAGGCCGGCTGCTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATCGGACGCGGATCCATCGTATGCCGATAAATCTTTTCACACCAGACCATGCGATCCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTACCCACGCGTTACT
>PC.636_250 FLP3FBN01A208L orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.635_251 FLP3FBN01EBSVN orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAACCAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCTCTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAGGTTGCATACGCGTTACGCACCCGTCCGCCGCT
>PC.636_252 FLP3FBN01BDPN5 orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGCGAGAATGCTCACTCATGATACTATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCAGGTTGGATACGCGTTACTC
>PC.635_253 FLP3FBN01DF6NE orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTTAGACCGGCAGGTTGGATACGTGTTACTCACCCG
>PC.355_254 FLP3FBN01BSM84 orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTT
>PC.354_255 FLP3FBN01BJIH7 orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAACTAGCTAATGTCACGCATGCCCATCCGTTGCCGGAATCGCTTCCTTTGGCCGCAGGGCCATGCAGCCCCGCGGCATTACGCGGTATTAGACGGGATTTCTCCCGATTATCCCCCTGCAACGGGCAGGTCGCATACGCGTTACT
>PC.354_256 FLP3FBN01D8FO0 orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGGAACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTATCCACGCGTTACTCACCCGTCCG
>PC.354_257 FLP3FBN01AGYWU orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTT
>PC.481_258 FLP3FBN01D6UF4 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAACTAGCTAATGCGCCGCATGACCATCCGCAGCCGGATCGCTCCTTTGAACCAACAGGGATGCCCCCGTCGGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTTCATACGTGTTACTCACCCGTGCG
>PC.635_259 FLP3FBN01AEB9M orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAACCAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCTCTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCG
>PC.355_260 FLP3FBN01CIT14 orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGCCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACTAGCTAATGCGCCGCATGGCCATCCGTAGCCGGTGTTACCCTTTAAACCCCAGGAGATGCCTCTCGGAGTTATTACGCGATATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTACGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.636_261 FLP3FBN01DZG2M orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.635_262 FLP3FBN01CN7JX orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTCGGTGGGCCGTTACCCCGCCGACTGCCTAATGCGCCGCATGCCCATCCTCCACCACCGGAGTTTTCCTCCCACGGAGATGCCTCCGCGGGATTTACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCACTAACTCAAGG
>PC.636_263 FLP3FBN01B74TP orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGGCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTCCGCCACTAGGGCG
>PC.636_264 FLP3FBN01BXZ1T orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGGCCGATCACCCTCTCAGGTCGGCTACGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATACGGCGCGGGTCCATCTATAAGTGATAGCAAAACCATCTTTCACTTTAGAACCATGCGGTTCTAAATGTTATCCGGTATTAGCTCCGGTTTCCCGAAGTTATCCCAGTCTTATAGGTAGGTTACCCACGTGTTACTCACCCGTCCGCCGCTAAG
>PC.355_265 FLP3FBN01C6LRW orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.355_266 FLP3FBN01AOR4O orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCACCAACTAGCTAATCAGACGCGGGTCCATCTCATACCACCGGAGTTTTTCACACCGGATCATGCAATCCTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATAGGGCAGGTTACCCACGCGTTACTCACCCG
>PC.607_267 FLP3FBN01CSZZD orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACAAGCTAATGCGCCGCATGCCCATCCTCCGCCGGAATCGCTTCCTTTTAACACCCGTGGGATGCCCCACAGATGTACCACGCGGTATTAGTCCGGATTTCTCCGGGTTATCCCCCTGCGGAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.636_268 FLP3FBN01DNS9D orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGAGAAAATGCTCACTCATGATACCATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCAGGTTGGATACGCGTTACTCACCCGTGCGCCGG
>PC.356_269 FLP3FBN01EGW4M orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTCTACCCGCCAACAAGCTAATCAGACGCGGGTCCATCGTATACCACCGGAGTTTTTCACACCGGACCATGCGATCCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTCGTTATCCCCCTGTATACGGCAGGTTACCCACGCGTTACTCACCCGTCCGCC
>PC.636_270 FLP3FBN01C9TUA orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTATGGATCGTCGGTTTGGTGGGCCGTTACCCCGCCAACTGCCTAATCCAACGCGGACCCATCCTATGCCGCCTCAGCTTTTCACACCGTACCATGCGGTACTGTGCGCTTATGCGGTATTAGCACCCGTTTCCGGATGTTATCCCCCTGCATAGGGCAGGTTGTCCACGCGTTACTCACCCGTCCGCCG
>PC.635_271 FLP3FBN01DZ0UC orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGTGGCCGGGATTGCTCCCTTTGGCGGCCCGGGGATGCCCCAAGGCCGCGTTACGCGGTATTAGACGGGGTTTCCCCCGCTTATCCCCCTGCCACGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.607_272 FLP3FBN01D9E65 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGTCTTGGTGGTCCGTTACACCGCCAACTAACTAATGCGACGCATGCCCATCCTTCACCGAAATTCTTTCCCCCTCGGAAGATGCCTCCCAAGGAGTATATGCGGTATTAGGCGAAATTTCTTCCGGTTATCCCGCTGTAAAGGGTGCTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.636_273 FLP3FBN01ECWT9 orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCTCACCAACTACCTAATCAGACGCGGGTCCCTCCTATACCACTATCGTTTTTCACACAGGGCCATGCGGCCCCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATAGGGCAGGTTCCCCACGCGTTACTCACCCGTCCGCCGCT
>PC.607_274 FLP3FBN01AHH1K orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATCGGACGCGGATCCATCGTATGCCGATAAATCTTTTCACACTATACCATGCGGTACTGTGCGCTTATGCGGTATTAGCAACTGTTTCCAGTTGGTATCCCCCTGCATACGGCAGGTTATCCACGCGTTACTCACCCG
>PC.636_275 FLP3FBN01DQLWU orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTC
>PC.636_276 FLP3FBN01ALCMH orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAACTAGCTAATGTCACGCATGCCCATCCGTTGCCGGAATCGCTTCCTTTGGCCGCAGGGCCATGCAGCCCCGCGGCATTACGCGGTATTAGACGGGATTTCTCCCGATTATCCCCCTGCAACGGGCAGGTCGCATACGCGTTACTCACCCG
>PC.635_277 FLP3FBN01C4N17 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAACCAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCTCTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCG
>PC.607_278 FLP3FBN01B8F4K orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCCACCAACTAGCTAATCAGACGCGGAACCATCGTATACCACCAGAGTTTTTCACACCGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTCTGAAAGGCAGGTTGCTCACGTGTTACTCACCCG
>PC.481_279 FLP3FBN01CATW7 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGG
>PC.593_280 FLP3FBN01B97H8 orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGCCACCGGATTGCTCCTTTGACCGCCCCGGGATGTCCCGGAATGGTGTTACGCGGAATTAGTCGGAATTTCTTCCGGTTATTCCCCTGTGACGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.355_281 FLP3FBN01BA934 orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGTCTTGGTGGTCCGTTACACCGCCAACTAACTAATGCGACGCATGCCCATCCGCCACCGGAATCAACCTTTGGCACCAACAGGATGTCCCATAGGTGCATTATGCCGTATTAGACGGAATTTCTCCCGATTATCCGGCTGTGGCAGGCAGGTTGCATACGTGTT
>PC.636_282 FLP3FBN01A3YNV orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACAAGCTAATCAGACGCGGACCCATCCCGCACCGCATGCGCTTTCCGCGCGGCCCCATGCGGGGCCGTGCGCATATGCGGTATTAGCAGCCGTTTCCAGCTGTTGTCCCCCAGTGCGGGGTAGGTTGTCCACGCGTTACTCACCCG
>PC.481_283 FLP3FBN01BJOS4 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAACCAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTCACACCGAACCATGCGGTTCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.607_284 FLP3FBN01EP59G orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAACAAACTAATGCACCGCAGGTCCATCCGCACCCCATCCCCTAAAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCTACGTGTTACTCACCCG
>PC.354_285 FLP3FBN01CTU4A orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCGGGACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTATCCACGCGTTACTCACCCGTCCG
>PC.607_286 FLP3FBN01DD28G orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAACAAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCTACGTGTTACTCACCCG
>PC.635_287 FLP3FBN01BW0YB orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGGGGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGCGCCGCATGCCCAGCCGCCACCGGATTACTCCTTTCGCCCGGACCGGATGCCCGGTCCGGGCGGCATGGGGTATTAGGCCGGGTTTCCCCGGGTTATCCCCCTGTGGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCAGGTG
>PC.355_288 FLP3FBN01AZT44 orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTTCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.355_289 FLP3FBN01AGMG5 orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAACCAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGTACCATGCGGTACTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCAGGTTACCCACGCGTTACTCACCCGTCCGCC
>PC.635_290 FLP3FBN01ARUVC orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACAAGCTAATCAGACGCGGGTCCATCTTACACCACTAATGTTTTTCACTCTGTCCCATGCGGGACTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTGTAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.355_291 FLP3FBN01D86I8 orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAACTGTTATCCCCTGTATGAGGCAGGTTACCCACGCGTT
>PC.481_292 FLP3FBN01COOOS orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTACTGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCAAGGATGCCCCCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGGCAG
>PC.355_293 FLP3FBN01D8J4C orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCACCAACAGGATGTCCCATAGGTGCATTATGCCGTATTAGACGGAATTTCTCCCGATTATCCGGCTGTGGCAGGCAGGTTGCATACGTGTTACT
>PC.636_294 FLP3FBN01BR61D orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCGGCACCGGGATGCCCCGACGCCGCGTCACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTGGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTTTAATC
>PC.356_295 FLP3FBN01BF0UO orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGGGGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGCGCCGCATGCCCAGCCGCCACCGGATTCCTCCTTTCGCCCGGTCCGGATGCCCGGTCCGGGCGGCATGGGGTATTAGGCCGGGTTTCCCCGGGTTATCCCCCTCGTGGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCAGGTG
>PC.354_296 FLP3FBN01DB7BE orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTTGCCTTGGTGGGCCGTTACCCCTCCAACTAGCTAATCAGACGCGGGTCCATCTCATACCGTCTCGGCTTTTCACCCCGAACCATGCGGTTCTGTGTGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATGAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.607_297 FLP3FBN01AZSCT orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAACCAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCTACGTGTTACTCACCCG
>PC.636_298 FLP3FBN01D3EWI orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.354_299 FLP3FBN01A55LW orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAACTAGCTAATGTCACGCATGCCCATCCCGCACCGGATCGCTCCTTTGACCGCTCCCCCATGCAGAGGAACGGTGTCATGCCGTATTAGTCCGGATTTCTCCGGGTTATCCGGCTGTGGCAGGCAGGTTGCATACGTGTT
>PC.355_300 FLP3FBN01DZ6P8 orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACAAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTAAGGAAAAGACATGCATCTTCTCCTGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTGTAAGGCAGGTTACCCACGCGTTACTCACCCGTCCGCC
>PC.355_301 FLP3FBN01AEB2N orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTACCTCAGAACCATGCGGTTCCGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>PC.607_302 FLP3FBN01B9U1E orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGTCTTGGTGGTCCGTTACACCGCCAACTAACTAATGCGACGCATGCCCATCCTTCACCGAAATTCTTTCCCCCTCGGAAGATGCCTCCCAAGGAGTATATGCGGTATTAGGCGAAATTTCTTCCGGTTATCCCGCTGTAAAGGGTAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCAG
>PC.354_303 FLP3FBN01A77ZB orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAGGACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.481_304 FLP3FBN01BCQ7B orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTTCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTAGGCCTTTACCCCACCAACTAGCTAATGCGCCGCAGGCTCATCCATCAGTGATGCCAGGAGCATCTTTAAACTTTCGTCCTATCCGGTATTAGCGATCGTTTCCAATCGTTGTCCCCGTCTGATGGGCAGATCACCTACGTGTTACTCACCCG
>PC.635_305 FLP3FBN01D5TAZ orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCGCCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTAGGCTTCTACCCCACCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCTCAGTTTTTACCTCTGTACCATGCGGTACTGGGGTCTTATGCGGTATTAGCAATCATTTCTAATTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>PC.635_306 FLP3FBN01BX26A orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTACTGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCAAGGATGCCCCCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAGGTTGCATACGTGTTACTCACCTGTGCGCCAGTCGCCGG
>PC.481_307 FLP3FBN01DTX5C orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTT
>PC.354_308 FLP3FBN01BCTWA orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCACCAACTGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGGAACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTATCCACGCGTTACTCACCCGTCCG
>PC.636_309 FLP3FBN01BA22V orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGAGAAAATGCTCACTCATGATACCATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCAGGTTGGATACGCGTTACTCACCCGTGCGCCGG
>PC.636_310 FLP3FBN01A4UET orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.636_311 FLP3FBN01EII4M orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTCCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTCAAGCTAATGTCACGCGAGCCTATCCTCATCCGACGAATCTTTAGATGGATTCAGATGCCTGATTCCATCACCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCAAGTTGCTCACGCG
>PC.607_312 FLP3FBN01DPG83 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGGTTAACCTCTCAGTCCCCCTATGTATCGTGGTCTTGGTGAGCCGTTACCCCACCAACTAACTAATACAACGCATGCCCATCCATTACCACCGGAGTTTTCAACCCAAGAAGATGCCTCCCTGGATGTTATGGGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTAATGGGTAGGTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCGACAAT
>PC.593_313 FLP3FBN01AYGIQ orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAACAAACTAATGCACCGCAGGTCCATCCGCACCCCATCCCCTAAAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCTACGTGTT
>PC.593_314 FLP3FBN01AGF7L orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGCCGATCACCCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTAAGCCGTTACCCTACCAACTAGCTAATCAGACGCGGGTCCATCCTGTACCGCAAAAGCTTTGATACTTCTACCATGCGATAAAGTATTTTATCTCGTATTAGCATACCTTTCGGTATGTTATCCGTGTGTACAGGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.635_315 FLP3FBN01A4NV1 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCCCGTACCACCGGAGTTTTCAAGGAGTCCCCATGCAGGGTCCCCTGTTATGCGGTATTAGCACCTGTTTCCAGGTGTTATCCCCCGGTACGGGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCACTAAAACAGTCCGGGG
>PC.635_316 FLP3FBN01AZ963 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGCTGCCCCGCCAACAAGCTAATCAGACGCGGGCCCCTCCCATACCGCCGGAACTTTCCCTAGAAAGGCATGCGCCTCCCTGGTTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTATGGGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCACT
>PC.481_317 FLP3FBN01DMBQY orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTAGGCCTTTACCCCACCAACTAGCTAATCAGACGCGGGTCCATCTCATACCACCGGAGTTTTTCACACCGAATCATGCGATTCTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATGAGGCAGGTTACCCACGCGTT
>PC.636_318 FLP3FBN01A6SLO orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGCGAGAATGCTCACTCATGATACTATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCAGGTTGGATACGCGTTACTCACCCGTGCG
>PC.635_319 FLP3FBN01CQHPF orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGGCCACCCTCTCAGGCCGGCTACCCATCGTCGCCTTGGTAGGCCATTACCCTACCAACTAGCTAATGGGACGCGAGTCCATCTTTCAGCGTCAGGTCTCCCCAACTTTTCCTATATAAACATGCACTTATATAACTCATCCGGCATTAGCTCACCTTTCGGTAAGTTGTTCCAGTCTAAAAGGCAGGTCACTCACGTGTT
>PC.635_320 FLP3FBN01CSQG7 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTTGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGTCCATCGTATACCTCCGGAAATTTTCACACTCTGTCATGCGACAGTGTGCGCTTATGCGGTATTAGCAGTTGTTTCCAACTGTTATCCCCCTGTATACGGCAGGTTACCCACGCGTTACTCACCCG
>PC.481_321 FLP3FBN01AMENY orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCCCGTACCACCGGAGTTTTCAAGAAAGGAACATGCGTCCCCTTCTGTTATGCGGTATTAGCACCTGTTTCCAGGTGTTATCCCCCGGTACGGGGCAGGTTCCCCACGCGTTACTCACCCGTCCGCCACTAAGGCCCGCGCCTTCCGGGT
>PC.481_322 FLP3FBN01D1PPR orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGTATCATGCGATACTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCAGGTTCCCCACGCGTTACTCACCCGTCCG
>PC.607_323 FLP3FBN01CPKVQ orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAGGGAAAAGCCATGCGGCTTCCCCCGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGCCAGGTTGCCCACGCGTTACTCACCCGTCCGCC
>PC.355_324 FLP3FBN01BZK6L orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCTCCAACCAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGTAGTTTTTCACACTGTACCATGCGGTACTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTCAGGCAGGTTACCCACGCGTTACTCACCCGTCCGCCACTAAG
>PC.356_325 FLP3FBN01DF6XB orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCAACCTCTCAGTCCGGCTACCGATCGTCGCCTTGGTGGGCCTCTACCCCGCCAACAAGCTAATCAGACGCGGGTCCATCGTATACCACCGGAGTTTTTCACACCGGACCATGCGATCCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.481_326 FLP3FBN01CF2XK orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGAGAAAATGCTCACTCATGATACCATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCAGGTTGGATACGCGTTACTCACCCGTGCG
>PC.355_327 FLP3FBN01CXD3D orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACAAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTAAGGAAAAGACATGCATCTTTCCCTGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCAGGTTACCCACGCGTTACTCACCCGTCCGCCACTAAGAT
>PC.355_328 FLP3FBN01CQGY3 orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTAGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAGGGAAAAGCCATGCGGCTTCCCCCGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGCCAGGTTGCCCACGCGTTACTCACCCGTCCGCCACTCA
>PC.481_329 FLP3FBN01B52I9 orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAACCTCTCAGTTCGGCTACGCATCATTGCCTTGGTAAGCCTTTACCCCACCAACTAGCTAATGCGCCGCGGGCCCATCCAAAAGCGGTAGCATAGCCACCTTTTACATAGTTACCATGCGGTAACTATGGTTATGCGGTATTAGCACCTGTTTCCAAGTGTTATCCCCCTCTTTTGGGCAGGTTGCCCACGTGTTACTCACCCG
>PC.607_330 FLP3FBN01AS2O8 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCTTTACCCCACCAACTAGCTAATCAGACGCGGGTCCATCATATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTATATGGCAGGTTACCCACGCGTTACTCACCCG
>PC.636_331 FLP3FBN01D7MLX orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGACCGTGTCTCAGTTCCAGTGTGGGGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTACTGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCAAGGATGCCCCCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGG
>PC.607_332 FLP3FBN01EUKQQ orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAATCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAACCAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCTACGTGTTACTCACCCG
>PC.635_333 FLP3FBN01AYCPW orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGGTCAACCTCTCAGTCCGGCTACTGATCGTCGCCTAGGTGGGCCGTTACCCCGCCTACCAGCTAATCAGACGCGAGGCCATCTTCCAGCGATAAATCTTTGGTGTCTCGATGATGCCATCGAAACACATCATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGGAAGGCAGGTTCCTCACG
>PC.354_334 FLP3FBN01BDSSU orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGGAACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTATCCACGCGTTACTCACCCGTCCG
>PC.636_335 FLP3FBN01ELLAQ orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGACCATCCGCAGCCGGATCGCTCCTTTGAATCACCAGAGATGCCTCCGGTGATTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTTCATACGTGTTACTCACCCG
>PC.356_336 FLP3FBN01CEUYF orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGCCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTCTACCCCGCCAACAAGCTAATCAGACGCGGGTCCATCGTATACCACCGGAGTTTTTCACACCAGACCATGCGATCCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGCCAGGTTGCCCACGCGTTACTCACCCG
>PC.636_337 FLP3FBN01CI8Z2 orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCGGCACCGGGATGCCCCGATGCCGCGTCACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTGGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.481_338 FLP3FBN01ENO6X orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCACCAAACCATGCGGTTTTGTGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>PC.636_339 FLP3FBN01B33P1 orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTACTGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCAAGGATGCCCCCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGG
>PC.636_340 FLP3FBN01ALS1C orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCATCGGCTTGGTGGGCCGTTACCTCACCAACTACCTAATCAGACGCGGGTCCCTCCTATACCACTATCGTTTTTCACACAGGGCCATGCGGCCCCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATAGGGCAGGTTCCCCACGCGTTACTCACCCGTCCGCC
>PC.356_341 FLP3FBN01AY1NE orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGTCCATCTCACACCGATAAATCTTTTCCGTCAGTACCATGCGGTACCAGCGGGTTATGCGGTATTAGCGGTCGTTTCCAACTGTTATCCCCCTGTGTGAGGCAGGTTACCCACGCGTTACTCACCCGTCCGCC
>PC.593_342 FLP3FBN01D9HWD orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTGAGCCGTTACCTCACCAACCAGCTAATGCGCCATAAGTCCATCCTCTACCAGTGCCTTGCAGCACTTTTAATACGGTCACCATGCAGTGTCCCTACCTATGCGGTCTTAGCTGCCGTTTCCAGCAGTTATCCCCCTGTAAAGGCCAGGTT
>PC.593_343 FLP3FBN01B0EIF orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTCCCAATGTGGCCGGCCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCTTTTATCTCACCAACTAGCTAATCAGACGCAGATCCATCCCATACCACCGGAGTTTTTCACACAGGGCCATGCAGCCTCGTGCGCTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCGGTATGGGGCAGGTTATCCACG
>PC.354_344 FLP3FBN01BPI1O orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAACCAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGTACCATGCGGTACTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCGCC
>PC.635_345 FLP3FBN01BBDRA orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGCTGCCCCGCCAACAAGCTAATCAGACGCGGGCCCCTCCCATACCGCCGGAACTTTCCCCAGAAAGGCATGCGCCTCCCTGGTTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTATGGGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCC
>PC.354_346 FLP3FBN01C3FYG orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAACTAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGACATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCTTGGG
>PC.356_347 FLP3FBN01DCLBL orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGACCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGTGGCCGGGATTGCTCCCTTTGGCGGCCTTGGGATGTCCCTAGGCCGCGTTACGCGGTATTAGACGGGGTTTCCCCCGCTTATCCCCCTGCCACGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>PC.607_348 FLP3FBN01D7ZOI orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTACTGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCAAGGATGCCCCCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGG
>PC.636_349 FLP3FBN01C0173 orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGCGAGAATGCTCACTCATGATACTATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCAGGTTGGATACGCGTTACTC
>PC.635_350 FLP3FBN01DPEUG orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACTAGCTAATGCGCCGCATGACCATCCGCAGCCGGATCGCTCCTTTGAATCTCCGGAGATGCCTCCGGAGATTATTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTTCATACGTGTT
>PC.636_351 FLP3FBN01B7KTL orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGGCCGTATCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAACTAGCTAATCAGACGCGAGGCCATCTTTCAGCGTCAGGTCTCCCCAACTTTTCCTATATAAACATGCACTTATATAACTCATCCGGCATTAGCTCACCTTTCGGTAAGTTGTTCCAGTCTAAAAGGCAGGTCACTC
>PC.607_352 FLP3FBN01CXEBD orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAACTGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAGGGAAAAGCCATGCGGCTTCCCCCGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGCCAGGTTGCCCACGCGTTACTCACCCGTCCGCC
>PC.355_353 FLP3FBN01BVDVK orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTT
>PC.636_354 FLP3FBN01CMYHR orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGCGAGAATGCTCACTCATGATACTATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCAGGTTGGATACGCGTTACTCACCCGTGCG
>PC.356_355 FLP3FBN01D6N8J orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACCAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAGGACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.607_356 FLP3FBN01COUSC orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCCTTGGTAGGCCATTACCCCACCAACTAGCTAATCAGACGCGGAACCATCGTATACCACCAGAGTTTTTCACACCGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTTTCCACGCGTTACTCACCCGTCCG
>PC.481_357 FLP3FBN01AZPBJ orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCGCCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCCCGCCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCGCTGTACCATGCGGTACTGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCGCT
>PC.354_358 FLP3FBN01A8PO2 orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGGAACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTATCCACGCGTTACTCACCCGTCCG
>PC.354_359 FLP3FBN01CVZXE orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTACTGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCAAGGATGCCCCCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGGCAG
>PC.607_360 FLP3FBN01D1IC0 orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCACCAACCAGCTAATCAGACGCGGGCCCATCTTGCACCACCGGAATCAACCTTTGGCACCAACAGGATGTCCCGTCGATGCATTATGCCGTATTAGACGGAATTTCTCCCGATTATCCCTCTGTAAAGGGCAGGTCGCATACGTGTTACT
>PC.355_361 FLP3FBN01AF0HW orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACTCTCTCAAGCCGGCTACTGATCGTTGCTTTGGTAGGCCATTACCCTGCCAACTGGCTAATCAGACGCGGGGCCATCGTATGCCGATAACTCTTTTCACACCATGCCATGCAGCATTGTGTGCTTATGCGGTATTAGCACCTATTTCTAACTGTTATCCCCCTGTGTAAGGCAGG
>PC.593_362 FLP3FBN01DS5AU orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGCCCATCCGTCACCGGATTGCTCCTTTGACCGCTCCGGGATGCCCCGGAATGGTGTTACGCGGAATTAGTCGGAATTTCTTCCGGTTATTCCCCTGTGACGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGG
>PC.356_363 FLP3FBN01DVSSP orig_bc=ACAGACCACTCA new_bc=ACAGACCACTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGTCTTGGTGGTCCGTTACACCGCCAACTAACTAATGCGACGCATGCCCATCCGCCACCGGAATCAACCTTTGGCACCAACAGGATGTCCCATAGGTGCATTATGCCGTATTAGACGGAATTTCTCCCGATTATCCGGCTGTGGCAGGCAGGTTGCATACGTGTT
>PC.635_364 FLP3FBN01AYYJN orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAGTGTGGCCGTCCGCCCTCTCAGGCCGGCTACTGATCGTGGGCTTGGTGGGCCGTTACCCCGCCAACTACCTAATCAGACGCGGACCCATCGTGTACCGTACTAGATAAGATCTAGGCTTTCCGCCCTGTGCCATGCGGCACTGTGCGCATATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTACACGGCAGGTTG
>PC.607_365 FLP3FBN01BQL2S orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
TTGGTCCGTGTCTCAGTACCAATGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATGCGCCGCATGCCCATCCTTTGCCGGAATTGCTTCCTTTGACTCCCAAACCATGTGGTAAGGGAGTGTTATGCAGTATTAGTCGGAATTTCTCCCGGTTATCCTCCTGCAAAGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGG
>PC.636_366 FLP3FBN01C6OIE orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACTCTCTCAAGCCGGCTACTGATCGTTGCTTTGGTAGGCCATTACCCTGCCAACTGGCTAATCAGACGCGGGGCCATCGTATGCCGATAACTCTTTTCACACCATGCCATGCAGCATTGTGTGCTTATGCGGTATTAGCAGTCATTTCTGACTGTTGTCCCCCTGCATACGG
>PC.635_367 FLP3FBN01D8PY9 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAACTAGCTAATCAGACGCGAGCTCATCTCAGAGCGATAAATCTTTGGCGTCCAGAGAGATGCCTCCCAGACGCATCATGCGGTATTAGCGGCTGTTTCCAACCGTTATTCCCCACTCCAAGG
>PC.635_368 FLP3FBN01BRKZB orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTACCTATCATTGCCTTGGTGGGCCGTTACCCCCCAACTAGCTAATAGGACGCATGCCCATCTGATACCTCGAATGATTTAATTATTAAAAGATGCCTTCAAATAATATTATGGGGTGTTAATCCACGTTTCCATGGGCTATCCCCCTGTATCAGCCAGGTTGCATACGCGTTACTCACCCGTGCGCCGG
>PC.593_369 FLP3FBN01EHLMD orig_bc=AGCAGCACTTGT new_bc=AGCAGCACTTGT bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTCCCAACGTTCTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAGGTTCCATACGTGTTACTCACCCGTGCG
>PC.354_370 FLP3FBN01BOOJJ orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCTCCAACTAGCTAATCGGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGGAACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAGGTTATCCACGCGTTACTCACCCGTCCG
>PC.481_371 FLP3FBN01CO1SB orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACTAACTAATGCGCCGCATGCCCATCCATGACCGGATCGCTCCTTTGACTCCCGAGAGATGTCTCCCGGGGGTGTTATGCGGTATTAGACGGAATTTCTCCCGCTTATCCCCCTGTCATGGGCAGGTTGCATACGTGTTACTC
>PC.355_372 FLP3FBN01D9QTP orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAACCAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGTACCATGCGGTACTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTTACTCACCCGTCCG
>PC.636_373 FLP3FBN01CRC8T orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGAGAAAATGCTCACTCATGATACCATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCAGGTTGGATACGCGTTACTCACCCGTGCGCCGG
>PC.355_374 FLP3FBN01DMQWE orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTT
>PC.355_375 FLP3FBN01EU78N orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGACCGTGTCTCAGTTCCAATGTGGGGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGGTTGGATACGTGTTACTCACCCG
>PC.636_376 FLP3FBN01AN2IX orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>PC.481_377 FLP3FBN01AMBQN orig_bc=ACCAGCGACTAG new_bc=ACCAGCGACTAG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACCAGCTAATCAGACGCGGGCCCATCCTGTACCACCGTGGTTTTCCCTGCTGTGCCATGCGGCACCGCAGGCTTATGCGGTATTAGCAGCCATTTCTGGCTGTTGTCCCCCGGTACAGGGCAGGTTGCCCACGCGTTACTCACCCG
>PC.354_378 FLP3FBN01B8NID orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACTGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGGAATCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.636_379 FLP3FBN01A2TMU orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACTAGCTAATGCGCCGCATGGCCATCCGTAGCCGGTGTTACCCTTTAAACCCCAAGAGATGCCTCTCGGAGTTATTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTACGGGCAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCAG
>PC.607_380 FLP3FBN01AELDH orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAACCAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGAAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCGCGGGCAGGTTCCCTACGTGTTACTCACCCG
>PC.635_381 FLP3FBN01ED2F6 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTCGGTGGGCCGTTACCCCGCCGACTGCCTAATGCGCCGCATGCCCATCCTCCACCACCGGAGTTTTCCTCCCAAGGAGATGCCTCCATGGGATTTACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAGGTTGCATACGTGTTACTCCCCCGTGCGCCAGTCGCCG
>PC.355_382 FLP3FBN01EABDR orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCACCAACTAGCTAATCAGACGCGGGTCCATCTCATACCACCGGAGTTTTTCACACCGGATCATGCAATCCTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATGAGGCAGGTTACCCACGCGTTACT
>PC.635_383 FLP3FBN01DIYZ3 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTGCCAACAAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGTACCATGCGGTACTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCAGGTTACCCACGCGTTACTCACCCGTCCG
>PC.355_384 FLP3FBN01APLDI orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCCACCAACCAGCTAATCAGACGCGGGCTCATCTTATACTACCGGAGTTTTTCACACAGAAACATGCGTCCCCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATAAGGCAGATTACCCACGTGTTACTCACCCGTCCG
>PC.355_385 FLP3FBN01D0DDK orig_bc=AACTCGTCGATG new_bc=AACTCGTCGATG bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAACCAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGTACCATGCGGTACTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCAGGTTACCCACGCGTTACTCACCCGTCCGCCACTAAG
>PC.635_386 FLP3FBN01CDWQ4 orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGTCCGTGTCTCAGTACCAGTGTGGGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACTAGCTAATGCGCCGCATGCCCATCCTTGTCCGGATAAATCCTTTGATCGAATTCTCATGCGAGAACCCGATGTCACGCGGTATTAGACCGGATTTCTCCGGCTTATCCCCCTGACAAGGGTAGGTTGCATACGTGTTACTC
>PC.354_387 FLP3FBN01AGMY0 orig_bc=AGCACGAGCCTA new_bc=AGCACGAGCCTA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAACCAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAGGTTCTCCACGCGTT
>PC.635_388 FLP3FBN01DQSWF orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGGTCGCCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTAGGCTTCTACCCCACCAACTACCTAATCAGACGCGGGCCCATCTTACACCACCTCAGTTTTTACCTCTGTACCATGCGGTACTGGGGTCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTGTAAGGCAGGTTCTCCACGCGTTACTCACCCGTCCGCCACTAAG
>PC.636_389 FLP3FBN01CXUVC orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
TTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCCTTGGTGGGCCGTTACCCCGCCAACAAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTCAAGTAAAAGACATGCGTCTCCTACTGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTAAAGGCCAGGTTACTTATGTATTACTCACCCGTTCGCCACTCGGGC
>PC.635_390 FLP3FBN01EFNNG orig_bc=ACCGCAGAGTCA new_bc=ACCGCAGAGTCA bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGTTCTACCTCTCAGTACGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAACCAGCTAATCAGACGCGAGCCCATCTTTAAGCGATAAATCTTTGATACACAAACCATGCGATTCATGTATATTATGCGGTATTAGCGGTCGTTTCCGACCGTTATCCCACTCTTAAAGGCAGGTTGCTC
>PC.607_391 FLP3FBN01EAC1O orig_bc=AACTGTGCGTAC new_bc=AACTGTGCGTAC bc_diffs=0
CTGGGCCGTGTCTCAGTCCCAATGTGGCCGTTCGACCTCTCAGTCCGGCTACCGATCGTCGGCTTGGTGAGCCGTTACCTCACCAACTACCTAATCGGACGCGAGCCCATCTCCGAGCGATAAATCTTTGATACCAAAGGGATGTCCCTCCAGTATGTTATGCGGTATTAGCGACCGTTTCCAGCCGTTATTCCCCTCTCGAAGGCAGGTTGCTCACGTGTTACTCACCCGTCCG
>PC.636_392 FLP3FBN01DO7JM orig_bc=ACGGTGAGTGTC new_bc=ACGGTGAGTGTC bc_diffs=0
CTGGGCCGTATCTCAGTCCCAATGTGGCCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAACTAGCTAATCAGACGCGAGGCCATCTTTCAGCGATAAATCTTTGACATAAATGCCATGCGACACCTATGTGTTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGAAAGGCAGGTTCCTCACG"""

fasting_subset_otu_table = """{"rows": [{"id": "0", "metadata": {"taxonomy": ["Root", "Bacteria", "Actinobacteria", "Actinobacteria", "Coriobacteridae", "Coriobacteriales", "Coriobacterineae", "Coriobacteriaceae"]}}, {"id": "1", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "2", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "3", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "4", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes"]}}, {"id": "5", "metadata": {"taxonomy": ["Root"]}}, {"id": "6", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "7", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes"]}}, {"id": "8", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes"]}}, {"id": "9", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes"]}}, {"id": "10", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes"]}}, {"id": "11", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Bacilli", "Bacillales", "Staphylococcaceae", "Staphylococcus"]}}, {"id": "12", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes"]}}, {"id": "13", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "14", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes"]}}, {"id": "15", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "16", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes", "Bacteroidetes", "Bacteroidales", "Rikenellaceae", "Alistipes"]}}, {"id": "17", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes"]}}, {"id": "18", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "19", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes", "Bacteroidetes", "Bacteroidales", "Rikenellaceae", "Alistipes"]}}, {"id": "20", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes"]}}, {"id": "21", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "22", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes"]}}, {"id": "23", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Bacilli", "Bacillales", "Staphylococcaceae", "Staphylococcus"]}}, {"id": "24", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes"]}}, {"id": "25", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes"]}}, {"id": "26", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes"]}}, {"id": "27", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes"]}}, {"id": "28", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "29", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "30", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes"]}}, {"id": "31", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "32", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes", "Bacteroidetes", "Bacteroidales", "Bacteroidaceae", "Bacteroides"]}}, {"id": "33", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes", "Bacteroidetes", "Bacteroidales"]}}, {"id": "34", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes", "Bacteroidetes", "Bacteroidales", "Bacteroidaceae", "Bacteroides"]}}, {"id": "35", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes", "Bacteroidetes", "Bacteroidales", "Rikenellaceae", "Alistipes"]}}, {"id": "36", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes"]}}, {"id": "37", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes", "Bacteroidetes", "Bacteroidales", "Porphyromonadaceae", "Parabacteroides"]}}, {"id": "38", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes"]}}, {"id": "39", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes", "Bacteroidetes", "Bacteroidales", "Prevotellaceae"]}}, {"id": "40", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "41", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "42", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "43", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "44", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes"]}}, {"id": "45", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes"]}}, {"id": "46", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes", "Bacteroidetes", "Bacteroidales", "Rikenellaceae", "Alistipes"]}}, {"id": "47", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "48", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia"]}}, {"id": "49", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes", "Bacteroidetes", "Bacteroidales", "Bacteroidaceae", "Bacteroides"]}}, {"id": "50", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes", "Bacteroidetes", "Bacteroidales", "Bacteroidaceae", "Bacteroides"]}}, {"id": "51", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes", "Bacteroidetes", "Bacteroidales", "Rikenellaceae", "Alistipes"]}}, {"id": "52", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "53", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes", "Bacteroidetes", "Bacteroidales", "Bacteroidaceae", "Bacteroides"]}}, {"id": "54", "metadata": {"taxonomy": ["Root", "Bacteria", "Actinobacteria", "Actinobacteria", "Coriobacteridae", "Coriobacteriales", "Coriobacterineae", "Coriobacteriaceae", "Olsenella"]}}, {"id": "55", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "56", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "57", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes"]}}, {"id": "58", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes", "Bacteroidetes", "Bacteroidales", "Rikenellaceae"]}}, {"id": "59", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "60", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "61", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "62", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "63", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes", "Bacteroidetes", "Bacteroidales", "Porphyromonadaceae", "Parabacteroides"]}}, {"id": "64", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "65", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "66", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "67", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes", "Bacteroidetes", "Bacteroidales"]}}, {"id": "68", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "69", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes"]}}, {"id": "70", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae", "Butyrivibrio"]}}, {"id": "71", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes"]}}, {"id": "72", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "73", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "74", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes", "Bacteroidetes", "Bacteroidales", "Rikenellaceae", "Alistipes"]}}, {"id": "75", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "76", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "77", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "78", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "79", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes", "Bacteroidetes", "Bacteroidales", "Porphyromonadaceae", "Parabacteroides"]}}, {"id": "80", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "81", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "82", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "83", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes", "Bacteroidetes", "Bacteroidales", "Bacteroidaceae", "Bacteroides"]}}, {"id": "84", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes", "Bacteroidetes", "Bacteroidales", "Rikenellaceae", "Alistipes"]}}, {"id": "85", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "86", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes", "Bacteroidetes", "Bacteroidales", "Bacteroidaceae", "Bacteroides"]}}, {"id": "87", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "88", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "89", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "90", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes", "Bacteroidetes", "Bacteroidales", "Bacteroidaceae", "Bacteroides"]}}, {"id": "91", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "92", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "93", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes", "Bacteroidetes", "Bacteroidales", "Bacteroidaceae", "Bacteroides"]}}, {"id": "94", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "95", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes"]}}, {"id": "96", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes"]}}, {"id": "97", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes"]}}, {"id": "98", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "99", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "100", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "101", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "102", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "103", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "104", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes"]}}, {"id": "105", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "106", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes"]}}, {"id": "107", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae", "Lachnospiraceae Incertae Sedis"]}}, {"id": "108", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "109", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes", "Bacteroidetes", "Bacteroidales", "Porphyromonadaceae", "Parabacteroides"]}}, {"id": "110", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "111", "metadata": {"taxonomy": ["Root", "Bacteria", "Actinobacteria", "Actinobacteria", "Coriobacteridae", "Coriobacteriales", "Coriobacterineae", "Coriobacteriaceae"]}}, {"id": "112", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "113", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "114", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "115", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes"]}}, {"id": "116", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes", "Bacteroidetes", "Bacteroidales"]}}, {"id": "117", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "118", "metadata": {"taxonomy": ["Root", "Bacteria", "Deferribacteres", "Deferribacteres", "Deferribacterales", "Deferribacteraceae", "Mucispirillum"]}}, {"id": "119", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "120", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae", "Lachnospiraceae Incertae Sedis"]}}, {"id": "121", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "122", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "123", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "124", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "125", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "126", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "127", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae", "Bryantella"]}}, {"id": "128", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "129", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "130", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "131", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "132", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "133", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "134", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Bacilli", "Lactobacillales", "Lactobacillaceae", "Lactobacillus"]}}, {"id": "135", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "136", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes", "Bacteroidetes", "Bacteroidales", "Bacteroidaceae", "Bacteroides"]}}, {"id": "137", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Incertae Sedis XIII", "Anaerovorax"]}}, {"id": "138", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "139", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes"]}}, {"id": "140", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "141", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae", "Lachnospiraceae Incertae Sedis"]}}, {"id": "142", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes"]}}, {"id": "143", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae", "Lachnospiraceae Incertae Sedis"]}}, {"id": "144", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes"]}}, {"id": "145", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "146", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "147", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "148", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae", "Lachnospiraceae Incertae Sedis"]}}, {"id": "149", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "150", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Bacilli", "Lactobacillales", "Lactobacillaceae", "Lactobacillus"]}}, {"id": "151", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "152", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "153", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "154", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae", "Lachnospiraceae Incertae Sedis"]}}, {"id": "155", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "156", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "157", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "158", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "159", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia"]}}, {"id": "160", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "161", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "162", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "163", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "164", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "165", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Ruminococcaceae"]}}, {"id": "166", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "167", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "168", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "169", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "170", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "171", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "172", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "173", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Ruminococcaceae"]}}, {"id": "174", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "175", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "176", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Ruminococcaceae"]}}, {"id": "177", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "178", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "179", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "180", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "181", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes"]}}, {"id": "182", "metadata": {"taxonomy": ["Root", "Bacteria", "Actinobacteria", "Actinobacteria", "Coriobacteridae", "Coriobacteriales", "Coriobacterineae", "Coriobacteriaceae"]}}, {"id": "183", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "184", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "185", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "186", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "187", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "188", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Peptostreptococcaceae", "Peptostreptococcaceae Incertae Sedis"]}}, {"id": "189", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "190", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "191", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "192", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes"]}}, {"id": "193", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "194", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "195", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "196", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "197", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "198", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "199", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "200", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "201", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "202", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "203", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "204", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "205", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "206", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "207", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "208", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "209", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "210", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "211", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes"]}}, {"id": "212", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "213", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "214", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "215", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "216", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "217", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "218", "metadata": {"taxonomy": ["Root", "Bacteria", "Proteobacteria", "Deltaproteobacteria"]}}, {"id": "219", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "220", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "221", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "222", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes"]}}, {"id": "223", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "224", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "225", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Ruminococcaceae"]}}, {"id": "226", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Ruminococcaceae"]}}, {"id": "227", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes", "Bacteroidetes", "Bacteroidales", "Bacteroidaceae", "Bacteroides"]}}, {"id": "228", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "229", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "230", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "231", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "232", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "233", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Ruminococcaceae"]}}, {"id": "234", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "235", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "236", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "237", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "238", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "239", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "240", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "241", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "242", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "243", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "244", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "245", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "246", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "247", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Ruminococcaceae"]}}, {"id": "248", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "249", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "250", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "251", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae", "Lachnospiraceae Incertae Sedis"]}}, {"id": "252", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae", "Lachnospiraceae Incertae Sedis"]}}, {"id": "253", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "254", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "255", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "256", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "257", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Incertae Sedis XIII"]}}, {"id": "258", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "259", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "260", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "261", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "262", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "263", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae", "Lachnospiraceae Incertae Sedis"]}}, {"id": "264", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "265", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "266", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "267", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Bacilli", "Lactobacillales", "Lactobacillaceae", "Lactobacillus"]}}, {"id": "268", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "269", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "270", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "271", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "272", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae", "Butyrivibrio"]}}, {"id": "273", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "274", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes", "Bacteroidetes", "Bacteroidales", "Rikenellaceae", "Alistipes"]}}, {"id": "275", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes", "Bacteroidetes", "Bacteroidales", "Rikenellaceae", "Alistipes"]}}, {"id": "276", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes"]}}, {"id": "277", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "278", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "279", "metadata": {"taxonomy": ["Root", "Bacteria", "Deferribacteres", "Deferribacteres", "Deferribacterales", "Deferribacteraceae", "Mucispirillum"]}}, {"id": "280", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes"]}}, {"id": "281", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Ruminococcaceae"]}}, {"id": "282", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "283", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "284", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Ruminococcaceae"]}}, {"id": "285", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes"]}}, {"id": "286", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes"]}}, {"id": "287", "metadata": {"taxonomy": ["Root", "Bacteria", "Proteobacteria", "Epsilonproteobacteria", "Campylobacterales", "Helicobacteraceae", "Helicobacter"]}}, {"id": "288", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "289", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "290", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes"]}}, {"id": "291", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "292", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "293", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "294", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "295", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "296", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "297", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "298", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "299", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "300", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "301", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "302", "metadata": {"taxonomy": ["Root", "Bacteria", "Actinobacteria", "Actinobacteria", "Coriobacteridae", "Coriobacteriales", "Coriobacterineae", "Coriobacteriaceae"]}}, {"id": "303", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "304", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "305", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "306", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "307", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Erysipelotrichi", "Erysipelotrichales", "Erysipelotrichaceae", "Turicibacter"]}}, {"id": "308", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Erysipelotrichi", "Erysipelotrichales", "Erysipelotrichaceae", "Turicibacter"]}}, {"id": "309", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "310", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "311", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "312", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Erysipelotrichi", "Erysipelotrichales", "Erysipelotrichaceae", "Turicibacter"]}}, {"id": "313", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "314", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "315", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "316", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "317", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "318", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "319", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae", "Butyrivibrio"]}}, {"id": "320", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "321", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "322", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "323", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia"]}}, {"id": "324", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes"]}}, {"id": "325", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes"]}}, {"id": "326", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Ruminococcaceae"]}}, {"id": "327", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "328", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "329", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Ruminococcaceae", "Acetanaerobacterium"]}}, {"id": "330", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "331", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Erysipelotrichi", "Erysipelotrichales", "Erysipelotrichaceae", "Erysipelotrichaceae Incertae Sedis"]}}, {"id": "332", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "333", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes", "Bacteroidetes", "Bacteroidales", "Bacteroidaceae", "Bacteroides"]}}, {"id": "334", "metadata": {"taxonomy": ["Root", "Bacteria", "Verrucomicrobia", "Verrucomicrobiae", "Verrucomicrobiales", "Verrucomicrobiaceae", "Akkermansia"]}}, {"id": "335", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "336", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "337", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "338", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes"]}}, {"id": "339", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "340", "metadata": {"taxonomy": ["Root", "Bacteria", "Actinobacteria", "Actinobacteria", "Coriobacteridae", "Coriobacteriales", "Coriobacterineae", "Coriobacteriaceae"]}}, {"id": "341", "metadata": {"taxonomy": ["Root", "Bacteria", "Deferribacteres", "Deferribacteres", "Deferribacterales", "Deferribacteraceae", "Mucispirillum"]}}, {"id": "342", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "343", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "344", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "345", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "346", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "347", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "348", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "349", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "350", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "351", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "352", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Ruminococcaceae"]}}, {"id": "353", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Ruminococcaceae"]}}, {"id": "354", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Bacilli", "Lactobacillales", "Carnobacteriaceae", "Carnobacteriaceae 1", "Atopostipes"]}}, {"id": "355", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes"]}}, {"id": "356", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Erysipelotrichi", "Erysipelotrichales", "Erysipelotrichaceae", "Coprobacillus"]}}, {"id": "357", "metadata": {"taxonomy": ["Root", "Bacteria", "Bacteroidetes"]}}, {"id": "358", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "359", "metadata": {"taxonomy": ["Root", "Bacteria", "TM7", "TM7_genera_incertae_sedis"]}}, {"id": "360", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "361", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "362", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "363", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Ruminococcaceae"]}}, {"id": "364", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae", "Lachnospiraceae Incertae Sedis"]}}, {"id": "365", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Ruminococcaceae"]}}, {"id": "366", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "367", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "368", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "369", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae", "Butyrivibrio"]}}, {"id": "370", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "371", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "372", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "373", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes"]}}, {"id": "374", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Ruminococcaceae"]}}, {"id": "375", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "376", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "377", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "378", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Erysipelotrichi", "Erysipelotrichales", "Erysipelotrichaceae", "Erysipelotrichaceae Incertae Sedis"]}}, {"id": "379", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Ruminococcaceae"]}}, {"id": "380", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "381", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "382", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Ruminococcaceae"]}}, {"id": "383", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "384", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "385", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Ruminococcaceae"]}}, {"id": "386", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Ruminococcaceae"]}}, {"id": "387", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "388", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "389", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Ruminococcaceae"]}}, {"id": "390", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "391", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Bacilli", "Lactobacillales", "Lactobacillaceae", "Lactobacillus"]}}, {"id": "392", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Clostridiaceae", "Clostridiaceae 1", "Clostridium"]}}, {"id": "393", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "394", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "395", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "396", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "397", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "398", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "399", "metadata": {"taxonomy": ["Root", "Bacteria", "Actinobacteria", "Actinobacteria"]}}, {"id": "400", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "401", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes"]}}, {"id": "402", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Ruminococcaceae"]}}, {"id": "403", "metadata": {"taxonomy": ["Root", "Bacteria", "Proteobacteria", "Deltaproteobacteria"]}}, {"id": "404", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Ruminococcaceae"]}}, {"id": "405", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Bacilli", "Lactobacillales", "Streptococcaceae", "Streptococcus"]}}, {"id": "406", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "407", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Ruminococcaceae"]}}, {"id": "408", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "409", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "410", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Ruminococcaceae"]}}, {"id": "411", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "412", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes"]}}, {"id": "413", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Ruminococcaceae"]}}, {"id": "414", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "415", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes"]}}, {"id": "416", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "417", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Bacilli", "Lactobacillales", "Lactobacillaceae", "Lactobacillus"]}}, {"id": "418", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "419", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Ruminococcaceae", "Ruminococcus"]}}, {"id": "420", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "421", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "422", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "423", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Ruminococcaceae", "Ruminococcaceae Incertae Sedis"]}}, {"id": "424", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "425", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Ruminococcaceae"]}}, {"id": "426", "metadata": {"taxonomy": ["Root", "Bacteria"]}}, {"id": "427", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "428", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "429", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "430", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "431", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "432", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "433", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}, {"id": "434", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes"]}}, {"id": "435", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales", "Lachnospiraceae"]}}, {"id": "436", "metadata": {"taxonomy": ["Root", "Bacteria", "Firmicutes", "Clostridia", "Clostridiales"]}}], "format": "Biological Observation Matrix v0.9", "data": [[0, 8, 1.0], [1, 3, 1.0], [2, 4, 2.0], [2, 5, 2.0], [2, 6, 1.0], [2, 7, 5.0], [2, 8, 1.0], [3, 8, 1.0], [4, 8, 1.0], [5, 2, 2.0], [5, 7, 1.0], [6, 7, 1.0], [7, 1, 1.0], [7, 2, 2.0], [7, 4, 9.0], [7, 5, 1.0], [7, 6, 1.0], [7, 7, 1.0], [7, 8, 3.0], [8, 8, 1.0], [9, 6, 2.0], [9, 7, 7.0], [9, 8, 22.0], [10, 0, 1.0], [10, 1, 2.0], [10, 3, 2.0], [10, 4, 1.0], [10, 5, 6.0], [10, 7, 2.0], [10, 8, 4.0], [11, 8, 2.0], [12, 1, 1.0], [12, 5, 3.0], [12, 6, 1.0], [12, 7, 1.0], [12, 8, 1.0], [13, 3, 1.0], [13, 7, 1.0], [14, 2, 1.0], [14, 6, 1.0], [14, 7, 1.0], [15, 6, 1.0], [15, 7, 3.0], [16, 6, 5.0], [17, 5, 5.0], [18, 7, 1.0], [19, 1, 2.0], [19, 2, 2.0], [19, 3, 4.0], [19, 5, 5.0], [19, 6, 1.0], [19, 7, 5.0], [20, 6, 1.0], [20, 7, 1.0], [20, 8, 1.0], [21, 7, 1.0], [22, 6, 1.0], [22, 7, 2.0], [22, 8, 6.0], [23, 8, 1.0], [24, 7, 1.0], [25, 8, 1.0], [26, 7, 1.0], [26, 8, 1.0], [27, 3, 1.0], [27, 6, 1.0], [27, 7, 1.0], [27, 8, 9.0], [28, 5, 1.0], [29, 1, 1.0], [30, 1, 1.0], [30, 2, 1.0], [30, 3, 1.0], [30, 4, 1.0], [30, 6, 3.0], [30, 8, 4.0], [31, 4, 9.0], [31, 5, 1.0], [32, 6, 2.0], [32, 7, 1.0], [33, 5, 2.0], [34, 6, 1.0], [35, 6, 1.0], [36, 5, 1.0], [37, 8, 1.0], [38, 8, 1.0], [39, 7, 1.0], [40, 2, 2.0], [41, 0, 6.0], [41, 2, 5.0], [41, 4, 1.0], [42, 2, 1.0], [43, 0, 1.0], [43, 1, 1.0], [43, 2, 1.0], [44, 4, 1.0], [45, 4, 2.0], [46, 5, 5.0], [46, 6, 9.0], [46, 7, 5.0], [46, 8, 3.0], [47, 1, 1.0], [48, 6, 1.0], [49, 6, 1.0], [50, 4, 3.0], [51, 8, 1.0], [52, 2, 1.0], [53, 6, 13.0], [54, 5, 2.0], [55, 1, 1.0], [56, 2, 1.0], [57, 7, 1.0], [58, 2, 1.0], [59, 4, 1.0], [60, 8, 1.0], [61, 4, 2.0], [61, 7, 4.0], [62, 1, 2.0], [62, 3, 1.0], [62, 7, 1.0], [63, 6, 2.0], [64, 1, 1.0], [65, 0, 5.0], [65, 1, 9.0], [65, 3, 3.0], [65, 7, 2.0], [66, 7, 2.0], [67, 1, 2.0], [67, 5, 1.0], [68, 1, 1.0], [69, 1, 1.0], [69, 2, 2.0], [69, 4, 3.0], [69, 6, 1.0], [70, 5, 1.0], [71, 5, 1.0], [71, 6, 1.0], [72, 5, 1.0], [73, 3, 1.0], [73, 7, 1.0], [74, 7, 1.0], [75, 1, 11.0], [75, 4, 1.0], [75, 6, 1.0], [76, 0, 2.0], [76, 1, 1.0], [76, 2, 10.0], [76, 3, 2.0], [76, 4, 24.0], [76, 7, 1.0], [76, 8, 1.0], [77, 2, 1.0], [77, 5, 2.0], [77, 7, 1.0], [78, 2, 1.0], [78, 4, 1.0], [79, 1, 1.0], [79, 8, 2.0], [80, 1, 4.0], [80, 2, 3.0], [80, 4, 1.0], [80, 5, 2.0], [80, 7, 2.0], [80, 8, 1.0], [81, 3, 1.0], [82, 5, 1.0], [83, 3, 1.0], [83, 6, 1.0], [83, 7, 2.0], [83, 8, 19.0], [84, 3, 2.0], [85, 1, 1.0], [85, 2, 1.0], [85, 3, 1.0], [85, 6, 1.0], [85, 7, 1.0], [86, 1, 1.0], [87, 2, 1.0], [88, 4, 1.0], [89, 8, 1.0], [90, 6, 1.0], [91, 3, 1.0], [92, 1, 7.0], [92, 2, 1.0], [92, 6, 1.0], [93, 4, 1.0], [94, 2, 1.0], [95, 3, 1.0], [95, 6, 1.0], [96, 7, 1.0], [96, 8, 1.0], [97, 7, 1.0], [98, 2, 2.0], [99, 7, 2.0], [99, 8, 2.0], [100, 6, 1.0], [101, 5, 1.0], [102, 5, 1.0], [103, 7, 1.0], [104, 1, 1.0], [104, 2, 1.0], [104, 8, 1.0], [105, 3, 1.0], [106, 6, 2.0], [106, 7, 1.0], [107, 1, 1.0], [107, 2, 1.0], [107, 3, 4.0], [107, 8, 1.0], [108, 1, 1.0], [109, 6, 5.0], [110, 2, 1.0], [111, 6, 1.0], [112, 3, 2.0], [113, 2, 1.0], [114, 5, 6.0], [114, 7, 3.0], [115, 6, 2.0], [115, 7, 1.0], [116, 5, 1.0], [116, 7, 1.0], [117, 0, 1.0], [117, 3, 5.0], [117, 4, 17.0], [117, 5, 20.0], [118, 5, 3.0], [118, 6, 5.0], [118, 7, 2.0], [118, 8, 5.0], [119, 1, 1.0], [119, 3, 1.0], [119, 5, 2.0], [120, 0, 1.0], [121, 6, 1.0], [122, 0, 1.0], [122, 1, 3.0], [122, 2, 1.0], [123, 1, 1.0], [123, 4, 1.0], [124, 0, 2.0], [124, 1, 1.0], [124, 3, 5.0], [124, 5, 1.0], [125, 1, 1.0], [125, 3, 3.0], [125, 7, 1.0], [126, 3, 1.0], [127, 1, 1.0], [127, 8, 1.0], [128, 1, 1.0], [128, 3, 1.0], [128, 4, 1.0], [128, 8, 3.0], [129, 0, 1.0], [129, 2, 3.0], [129, 5, 4.0], [130, 6, 1.0], [130, 8, 1.0], [131, 0, 1.0], [132, 1, 2.0], [133, 1, 1.0], [134, 4, 1.0], [135, 3, 1.0], [136, 6, 1.0], [137, 6, 1.0], [138, 5, 1.0], [139, 6, 1.0], [140, 3, 1.0], [141, 2, 1.0], [142, 5, 1.0], [143, 0, 1.0], [144, 3, 2.0], [145, 1, 1.0], [146, 0, 2.0], [146, 1, 3.0], [146, 2, 8.0], [146, 4, 1.0], [147, 5, 1.0], [148, 0, 1.0], [149, 0, 1.0], [149, 2, 1.0], [150, 0, 1.0], [150, 3, 2.0], [150, 4, 4.0], [151, 1, 2.0], [151, 2, 1.0], [152, 0, 1.0], [152, 5, 4.0], [153, 8, 1.0], [154, 0, 1.0], [155, 7, 1.0], [156, 3, 1.0], [157, 0, 1.0], [158, 8, 1.0], [159, 0, 1.0], [159, 2, 1.0], [159, 3, 1.0], [160, 1, 1.0], [160, 2, 1.0], [160, 4, 1.0], [161, 7, 1.0], [162, 3, 1.0], [163, 7, 1.0], [164, 6, 1.0], [165, 6, 1.0], [166, 6, 1.0], [167, 0, 8.0], [167, 4, 3.0], [168, 5, 1.0], [169, 3, 1.0], [170, 3, 1.0], [171, 0, 1.0], [171, 4, 1.0], [172, 0, 1.0], [172, 1, 4.0], [172, 2, 2.0], [172, 3, 6.0], [173, 5, 1.0], [174, 6, 1.0], [175, 6, 1.0], [176, 5, 2.0], [176, 7, 1.0], [177, 0, 3.0], [177, 4, 2.0], [178, 0, 29.0], [178, 1, 1.0], [178, 2, 10.0], [179, 4, 1.0], [180, 2, 1.0], [181, 7, 1.0], [181, 8, 1.0], [182, 3, 1.0], [183, 0, 1.0], [183, 2, 1.0], [184, 1, 1.0], [185, 0, 2.0], [185, 2, 2.0], [185, 3, 1.0], [186, 0, 1.0], [187, 3, 4.0], [188, 0, 1.0], [188, 4, 10.0], [189, 4, 1.0], [190, 2, 1.0], [191, 3, 1.0], [191, 7, 1.0], [192, 2, 2.0], [193, 2, 1.0], [194, 3, 1.0], [195, 3, 1.0], [196, 0, 9.0], [196, 4, 5.0], [197, 2, 1.0], [198, 2, 1.0], [199, 5, 1.0], [200, 7, 1.0], [201, 5, 1.0], [202, 5, 1.0], [203, 1, 1.0], [204, 1, 1.0], [204, 3, 3.0], [204, 7, 1.0], [205, 7, 1.0], [206, 2, 1.0], [207, 7, 1.0], [208, 5, 1.0], [209, 0, 1.0], [210, 1, 1.0], [211, 5, 1.0], [212, 8, 1.0], [213, 5, 1.0], [214, 5, 2.0], [215, 3, 1.0], [216, 0, 1.0], [216, 2, 2.0], [217, 3, 1.0], [218, 5, 1.0], [219, 2, 1.0], [220, 3, 1.0], [221, 7, 1.0], [222, 5, 7.0], [222, 7, 2.0], [222, 8, 2.0], [223, 7, 2.0], [224, 8, 1.0], [225, 5, 1.0], [226, 7, 1.0], [227, 0, 2.0], [227, 1, 18.0], [227, 3, 1.0], [227, 6, 21.0], [227, 7, 4.0], [227, 8, 4.0], [228, 4, 1.0], [229, 0, 1.0], [229, 4, 1.0], [230, 7, 1.0], [231, 4, 1.0], [232, 4, 1.0], [233, 3, 1.0], [233, 5, 2.0], [233, 7, 1.0], [233, 8, 1.0], [234, 2, 2.0], [235, 0, 1.0], [236, 8, 1.0], [237, 7, 1.0], [238, 2, 1.0], [239, 5, 2.0], [240, 3, 1.0], [241, 7, 1.0], [242, 7, 1.0], [243, 0, 1.0], [244, 3, 1.0], [245, 3, 4.0], [245, 7, 1.0], [246, 6, 1.0], [247, 0, 1.0], [248, 7, 1.0], [249, 4, 1.0], [250, 0, 1.0], [251, 3, 1.0], [252, 1, 2.0], [253, 4, 1.0], [254, 0, 1.0], [255, 0, 1.0], [255, 1, 3.0], [256, 5, 1.0], [257, 5, 1.0], [257, 6, 1.0], [258, 7, 1.0], [259, 2, 1.0], [260, 3, 1.0], [261, 2, 1.0], [262, 1, 1.0], [262, 6, 1.0], [263, 0, 1.0], [264, 8, 1.0], [265, 0, 1.0], [265, 1, 1.0], [266, 3, 1.0], [267, 1, 1.0], [268, 6, 1.0], [269, 5, 1.0], [270, 5, 1.0], [271, 0, 1.0], [272, 1, 1.0], [272, 8, 1.0], [273, 0, 1.0], [273, 3, 1.0], [273, 5, 1.0], [274, 3, 1.0], [274, 6, 1.0], [274, 7, 5.0], [274, 8, 2.0], [275, 7, 1.0], [276, 0, 1.0], [276, 1, 4.0], [276, 2, 3.0], [276, 3, 2.0], [277, 2, 1.0], [278, 5, 1.0], [279, 8, 1.0], [280, 0, 2.0], [280, 1, 2.0], [280, 3, 1.0], [281, 4, 1.0], [282, 3, 1.0], [282, 4, 4.0], [283, 3, 1.0], [283, 5, 2.0], [284, 0, 1.0], [285, 0, 1.0], [285, 2, 1.0], [286, 8, 1.0], [287, 4, 5.0], [287, 5, 2.0], [288, 5, 1.0], [289, 2, 1.0], [290, 1, 1.0], [291, 3, 2.0], [292, 0, 9.0], [292, 1, 12.0], [292, 2, 5.0], [292, 3, 13.0], [292, 4, 2.0], [293, 1, 1.0], [294, 3, 1.0], [295, 0, 4.0], [295, 1, 2.0], [295, 2, 1.0], [296, 3, 1.0], [297, 1, 1.0], [298, 2, 1.0], [299, 3, 1.0], [300, 7, 2.0], [301, 6, 8.0], [301, 7, 8.0], [301, 8, 3.0], [302, 6, 1.0], [303, 3, 3.0], [304, 3, 1.0], [305, 0, 1.0], [306, 6, 1.0], [307, 6, 2.0], [308, 3, 1.0], [309, 6, 1.0], [310, 3, 1.0], [311, 2, 1.0], [312, 3, 8.0], [312, 6, 1.0], [313, 4, 1.0], [314, 3, 1.0], [315, 5, 1.0], [316, 6, 1.0], [317, 0, 1.0], [317, 2, 1.0], [318, 4, 1.0], [319, 3, 1.0], [320, 0, 1.0], [320, 7, 1.0], [321, 4, 2.0], [322, 4, 2.0], [322, 6, 1.0], [323, 7, 1.0], [324, 1, 2.0], [324, 2, 1.0], [325, 7, 1.0], [326, 2, 1.0], [327, 3, 1.0], [328, 0, 1.0], [329, 2, 2.0], [330, 5, 1.0], [331, 4, 4.0], [331, 8, 2.0], [332, 5, 1.0], [333, 8, 1.0], [334, 6, 1.0], [335, 7, 1.0], [336, 4, 1.0], [337, 8, 1.0], [338, 3, 1.0], [339, 2, 1.0], [340, 6, 1.0], [341, 8, 1.0], [342, 3, 1.0], [343, 3, 1.0], [344, 7, 1.0], [345, 2, 1.0], [346, 5, 1.0], [347, 4, 1.0], [348, 7, 1.0], [349, 3, 1.0], [350, 1, 1.0], [351, 7, 1.0], [352, 5, 1.0], [353, 5, 2.0], [353, 7, 1.0], [354, 8, 1.0], [355, 7, 2.0], [356, 0, 1.0], [357, 6, 2.0], [357, 7, 3.0], [357, 8, 1.0], [358, 6, 1.0], [359, 6, 2.0], [360, 5, 1.0], [361, 1, 1.0], [362, 6, 1.0], [362, 8, 1.0], [363, 7, 1.0], [364, 0, 1.0], [365, 0, 1.0], [365, 4, 2.0], [365, 6, 1.0], [365, 8, 2.0], [366, 5, 1.0], [367, 2, 1.0], [368, 2, 1.0], [369, 1, 1.0], [370, 2, 1.0], [371, 2, 1.0], [372, 6, 1.0], [373, 1, 1.0], [374, 3, 1.0], [375, 5, 1.0], [376, 7, 1.0], [377, 4, 1.0], [378, 6, 4.0], [379, 2, 1.0], [380, 2, 1.0], [380, 7, 1.0], [381, 2, 1.0], [381, 3, 1.0], [382, 2, 1.0], [382, 7, 1.0], [383, 3, 1.0], [384, 3, 1.0], [385, 3, 1.0], [385, 5, 1.0], [386, 3, 1.0], [387, 5, 1.0], [388, 5, 1.0], [389, 6, 1.0], [389, 8, 1.0], [390, 8, 1.0], [391, 0, 1.0], [391, 3, 1.0], [392, 1, 1.0], [392, 6, 3.0], [393, 6, 1.0], [394, 2, 1.0], [395, 7, 1.0], [396, 8, 1.0], [397, 2, 1.0], [398, 6, 1.0], [399, 7, 1.0], [400, 5, 1.0], [401, 5, 1.0], [402, 3, 1.0], [403, 6, 2.0], [403, 8, 1.0], [404, 2, 1.0], [404, 7, 1.0], [405, 5, 1.0], [406, 4, 1.0], [407, 2, 1.0], [408, 5, 1.0], [409, 6, 1.0], [410, 0, 1.0], [410, 3, 1.0], [411, 5, 1.0], [412, 8, 1.0], [413, 5, 1.0], [414, 2, 3.0], [414, 7, 1.0], [414, 8, 3.0], [415, 5, 1.0], [416, 5, 1.0], [417, 0, 14.0], [417, 1, 1.0], [417, 2, 14.0], [417, 3, 1.0], [418, 2, 1.0], [419, 0, 1.0], [419, 7, 2.0], [420, 2, 1.0], [421, 1, 1.0], [422, 5, 1.0], [423, 1, 1.0], [424, 5, 1.0], [425, 6, 2.0], [426, 7, 1.0], [427, 5, 1.0], [428, 1, 1.0], [429, 2, 1.0], [429, 7, 1.0], [430, 0, 1.0], [430, 1, 1.0], [430, 2, 1.0], [431, 7, 1.0], [432, 4, 1.0], [433, 0, 1.0], [434, 7, 1.0], [435, 0, 1.0], [435, 2, 1.0], [435, 7, 1.0], [435, 8, 1.0], [436, 4, 1.0]], "columns": [{"id": "PC.354", "metadata": null}, {"id": "PC.355", "metadata": null}, {"id": "PC.356", "metadata": null}, {"id": "PC.481", "metadata": null}, {"id": "PC.593", "metadata": null}, {"id": "PC.607", "metadata": null}, {"id": "PC.634", "metadata": null}, {"id": "PC.635", "metadata": null}, {"id": "PC.636", "metadata": null}], "generated_by": "QIIME 1.4.0-dev, svn revision 2614", "matrix_type": "sparse", "shape": [437, 9], "format_url": "http://www.qiime.org/svn_documentation/documentation/biom_format.html", "date": "2011-12-22T13:02:39.349314", "type": "OTU table", "id": null, "matrix_element_type": "float"}"""

fasting_subset_tree = """(91:0.03138,((56:0.01554,(171:0.00015,41:0.00014)0.868:0.01019)0.786:0.00551,(80:0.01059,(((260:0.00581,244:0.02632)0.871:0.00571,((319:0.01626,70:0.00524)0.764:0.00509,(298:0.03941,369:0.01625)0.858:0.01096)0.925:0.00016)0.907:0.0156,(272:0.00015,219:0.02761)0.371:0.00527)0.886:0.01577)0.904:0.01538)0.213:0.00506,(((((124:0.02331,193:0.01557)0.550:0.02651,390:0.02213)0.922:0.02481,((376:0.02606,(420:0.01061,(375:0.01758,(398:0.05303,(((429:0.00014,(427:0.00569,409:0.00793)0.175:0.00015)0.715:0.00016,(434:0.00016,426:0.02959)0.962:0.01738)0.942:0.02633,(414:0.01138,(413:0.01803,424:0.02362)0.839:0.01221)0.715:0.00567)0.706:0.00547)0.860:0.01803)0.748:0.00612)0.961:0.04689)0.973:0.0476,(((152:0.06102,400:0.06529)0.948:0.04668,((((188:0.14301,207:0.09317)0.630:0.00562,((((318:0.00503,320:0.00632)0.963:0.06151,((421:0.02097,(430:0.00014,(((432:0.03044,366:0.01302)0.756:0.01196,361:0.02203)0.807:0.01147,431:0.02457)0.835:0.0118)0.928:0.02756)0.899:0.03039,(401:0.00901,433:0.09069)0.881:0.02632)0.941:0.06451)0.909:0.04762,(170:0.04503,187:0.02247)0.806:0.01706)0.835:0.0181,347:0.03498)0.900:0.02275)0.843:0.01899,(((((21:0.06701,(137:0.00015,257:0.07002)0.343:0.01903)0.900:0.04272,((248:0.00016,(139:0.03037,(115:0.00014,95:0.03129)0.822:0.02203)0.998:0.07935)0.999:0.11876,(82:0.00014,(59:0.0964,(117:0.00015,332:0.02889)0.836:0.01403)0.926:0.01929)1.000:0.23864)0.050:0.00109)0.759:0.03556,((87:0.01453,(57:0.00252,(58:0.04076,(((((((116:0.04298,46:0.00016)0.942:0.02413,84:0.02412)0.751:0.00986,((274:0.01098,275:0.0279)0.767:0.00508,74:0.01398)0.758:0.00703)0.746:0.00561,51:0.07406)0.864:0.01536,16:0.00015)0.955:0.03768,35:0.00015)0.997:0.03175,19:0.00015)0.861:0.02236)0.996:0.07959)0.974:0.06082)0.901:0.04353,(278:0.06061,(((28:0.02168,(88:0.00612,(20:0.00544,((((10:0.0,12:0.0,14:0.0):0.00014,(6:0.04332,36:0.0156)0.911:0.00016)0.947:0.01021,26:0.00014)0.722:0.00016,24:0.00508)0.995:0.06124)0.842:0.0489)0.997:0.0834)0.999:0.0992,((131:0.02646,((30:0.00014,(100:0.01017,69:0.00014)0.928:0.00015)0.808:0.0051,(((((((((45:0.00506,((224:0.00014,144:0.01568)0.399:0.01048,76:0.00016)0.787:0.00519)0.815:0.00508,44:0.00016)0.860:0.01021,7:0.00526)0.754:0.00768,113:0.04271)0.692:0.01687,((2:0.01311,4:0.03731)0.748:0.01016,(104:0.00015,276:0.03287)0.973:0.03782)0.892:0.02397)0.854:0.01508,((((((285:0.03762,(15:0.10332,((358:0.04675,(((158:0.06378,354:0.1522)0.803:0.03759,(194:0.00973,((189:0.01519,249:0.02483)0.954:0.03416,68:0.01853)0.789:0.01036)0.934:0.05047)0.819:0.02898,(164:0.03164,277:0.02089)0.136:0.01345)0.829:0.0189)0.942:0.06408,((301:0.01074,(355:0.01673,300:0.02274)0.867:0.01124)0.785:0.00582,357:0.00539)1.000:0.17151)0.830:0.03922)0.939:0.05382)0.701:0.01345,67:0.02688)0.823:0.017,(((109:0.04428,63:0.03926)0.979:0.05805,(37:0.08014,79:0.0163)0.220:0.02124)0.923:0.05077,(39:0.10195,(93:0.01059,((((53:0.00352,(((227:0.00016,(83:0.00014,286:0.06319)0.991:0.03771)0.788:0.00014,333:0.06987)0.998:0.06107,86:0.00015)0.994:0.04517)0.349:0.00596,50:0.04512)0.503:0.0083,(34:0.0426,(32:0.02608,(49:0.06727,90:0.00496)0.974:0.02959)0.523:0.00358)0.863:0.02028)0.936:0.03486,136:0.07971)0.734:0.00702)0.961:0.08516)0.910:0.05397)0.866:0.0375)0.894:0.02444,((8:0.02694,(3:0.03192,5:0.00016)0.973:0.04398)0.978:0.04594,(33:0.02159,(106:0.03443,96:0.03596)0.864:0.0251)0.831:0.02373)0.304:0.00014)0.978:0.02591,(71:0.051,(222:0.02838,246:0.00436)0.937:0.0244)0.284:0.01734)0.429:0.00645,(((((27:0.0,9:0.0):0.00014,(25:0.01023,38:0.02084)0.885:0.01026)0.133:0.00014,22:0.01547)0.827:0.0208,(324:0.03204,17:0.0441)0.123:0.01)0.947:0.03497,280:0.02253)0.718:0.00493)0.842:0.00611)0.000:0.00016,181:0.02181)0.925:0.01592,31:0.02127)0.477:0.00997,(289:0.01603,290:0.01098)0.959:0.02745)0.924:0.00014)0.998:0.07977)0.749:0.0055,(((209:0.01801,325:0.03269)0.853:0.01642,(42:0.05299,29:0.01425)0.827:0.014)0.417:0.00015,232:0.02092)0.982:0.03398)0.649:0.00014)0.953:0.02633,(89:0.0342,(138:0.0459,327:0.04257)0.940:0.04586)0.883:0.02242)0.907:0.0455)0.948:0.06336)0.574:0.04246)0.813:0.02689,(((102:0.01072,54:0.05569)0.961:0.07824,((0:0.0468,111:0.02902)0.786:0.01542,((340:0.03302,(302:0.02503,399:0.03369)0.781:0.01109)0.945:0.03337,182:0.0403)0.832:0.01488)0.992:0.10706)0.904:0.05948,(((211:0.02581,213:0.02321)0.969:0.06007,(142:0.04144,223:0.05483)0.800:0.02186)0.955:0.05472,((((267:0.03308,(134:0.07285,150:0.00015)0.813:0.00015)0.366:0.00872,((391:0.08217,417:0.00818)0.524:0.03815,(330:0.07239,370:0.05715)0.980:0.09661)0.925:0.06443)0.974:0.0939,((11:0.0269,23:0.00015)1.000:0.15357,((378:0.00014,(168:0.05313,97:0.00066)0.999:0.08303)0.999:0.11217,((331:0.04314,356:0.08848)0.965:0.05433,(405:0.1024,(312:0.01087,(308:0.03192,307:0.0212)0.910:0.00015)0.995:0.08955)0.688:0.01126)0.228:0.00016)0.806:0.02968)0.833:0.03579)0.642:0.00917,(334:0.12276,392:0.05097)0.847:0.02572)0.877:0.02757)0.426:0.02497)0.933:0.04732)0.871:0.02613,(((287:0.26862,(217:0.08165,218:0.00929)0.884:0.05245)0.705:0.05092,((403:0.09144,359:0.22127)0.852:0.04039,351:0.07049)0.676:0.02494)0.959:0.07101,((350:0.10412,(402:0.00014,410:0.00551)0.945:0.06389)0.892:0.04421,(412:0.00448,415:0.03109)0.992:0.06861)0.462:0.0102)0.914:0.03206)0.981:0.00016)0.831:0.01613,(((((((((((395:0.0607,((((228:0.06268,419:0.02151)0.928:0.04763,((173:0.00015,176:0.01044)0.978:0.05847,(130:0.10381,(((165:0.00015,((225:0.03875,226:0.00536)0.937:0.01609,((281:0.01056,(386:0.02218,326:0.02751)0.987:0.00015)0.867:0.02806,329:0.0339)0.744:0.00533)0.338:0.01064)0.794:0.00618,284:0.021)0.594:0.01533,163:0.06412)0.670:0.00331)0.833:0.01554)0.827:0.01685)0.796:0.01647,425:0.06687)0.769:0.00951,374:0.00443)0.714:0.01154)0.714:0.00872,(363:0.01064,387:0.03462)0.889:0.01546)0.000:0.00502,396:0.06189)0.480:0.01186,(360:0.04307,(416:0.05007,237:0.02283)0.740:0.00734)0.835:0.01155)0.921:0.01801,((365:0.00014,389:0.00014)0.846:0.00538,(407:0.00015,(385:0.01085,(233:0.00728,404:0.00015)0.166:0.00014)0.963:0.02238)0.828:0.00729)0.942:0.00015)0.151:0.01071,((352:0.00013,380:0.01643)0.954:0.02126,(247:0.01614,243:0.04517)0.430:0.00014)0.987:0.03338)0.758:0.00496,(353:0.00014,382:0.00014)0.999:0.00014)0.871:0.0346,379:0.00364)0.970:0.03438,(337:0.04417,(422:0.00664,((341:0.0223,(118:0.03295,279:0.00014)0.888:0.02922)1.000:0.11353,(411:0.02454,423:0.0304)0.693:0.01914)0.931:0.05373)0.904:0.03125)0.719:0.00734)0.909:0.02779,(159:0.04393,221:0.00015)0.999:0.07613)0.313:0.02026,259:0.01429)0.933:0.02439)0.521:0.00282)0.846:0.00221,((((127:0.04567,(((191:0.00016,(236:0.01619,408:0.01671)0.757:0.00515)0.542:0.00016,(256:0.00014,342:0.01065)0.997:0.03284)0.721:0.02916,(394:0.05783,(151:0.04317,(103:0.01259,161:0.02538)0.818:0.01669)0.468:0.02681)0.415:0.01576)0.916:0.03996)0.922:0.04147,(323:0.01447,338:0.00321)0.853:0.02723)0.389:0.01609,((406:0.08533,(162:0.04584,(153:0.06999,322:0.02039)0.809:0.02113)0.098:0.01333)0.844:0.01478,((((123:0.02634,(169:0.01045,174:0.01039)0.949:0.00015)0.693:0.01633,269:0.03999)0.838:0.01016,(((201:0.02365,283:0.00356)0.852:0.01226,(114:0.01093,346:0.01663)0.691:0.00343)0.884:0.01235,(112:0.0158,(195:0.02712,212:0.02152)0.872:0.01073)0.750:0.0053)0.753:0.00522)0.753:0.00535,(180:0.04103,(155:0.03597,242:0.04333)0.767:0.00883)0.919:0.0276)0.323:0.00014)0.958:0.02654)0.123:0.00016,((192:0.03384,(((110:0.13051,(373:0.00534,(254:0.0107,264:0.0218)0.763:0.00528)0.953:0.00015)0.830:0.01036,156:0.05032)0.141:0.00015,206:0.01589)0.844:0.01058)0.738:0.00516,((335:0.01999,372:0.02628)0.912:0.02056,(266:0.00015,294:0.02745)0.853:0.01082)0.856:0.01148)0.902:0.01565)0.816:0.00518)0.736:0.00672)0.886:0.01398)0.396:0.00015,((13:0.00015,(1:0.0051,(48:0.03225,18:0.00014)0.610:0.00519)0.912:0.01029)0.908:0.01953,(179:0.0515,(52:0.01038,149:0.06621)0.812:0.0109)0.739:0.00921)0.783:0.00867)0.952:0.01558,((261:0.05074,262:0.02236)0.919:0.03024,(((238:0.03629,166:0.00767)0.097:0.00537,(303:0.01324,383:0.03403)0.887:0.02784)0.870:0.01529,(((72:0.00982,(47:0.00014,(146:0.0082,239:0.0327)0.763:0.00803)0.841:0.01299)0.138:0.00826,((62:0.00015,((((105:0.02651,186:0.00015)0.775:0.00513,(215:0.00874,(203:0.00016,305:0.03337)0.824:0.00879)0.766:0.00877)0.933:0.01583,(94:0.0389,(65:0.00014,(184:0.0267,185:0.02183)0.938:0.00015)0.883:0.01609)0.804:0.01015)0.799:0.00516,((((((((((((200:0.01046,(98:0.01556,(73:0.01564,(122:0.02128,175:0.00015)0.889:0.01398)0.762:0.00707)0.970:0.03203)0.938:0.00014,349:0.02174)0.532:0.00015,(((((((299:0.01635,306:0.01635)0.288:0.00542,135:0.03053)0.776:0.007,(345:0.01083,(119:0.00512,310:0.00014)0.789:0.00517)0.810:0.00514)0.458:0.00014,255:0.00015)0.916:0.00016,343:0.01618)0.804:0.0102,(388:0.06006,(190:0.03767,315:0.01441)0.751:0.00755)0.878:0.01614)0.867:0.00016,(((183:0.01056,258:0.0162)0.763:0.00521,(241:0.01533,(202:0.01143,362:0.0273)0.551:0.01038)0.782:0.00615)0.801:0.00519,(304:0.01611,364:0.03412)0.898:0.00015)0.763:0.00015)0.857:0.00514)0.782:0.00513,252:0.01635)0.754:0.00514,(((268:0.0139,(147:0.00998,265:0.03126)0.228:0.01262)0.793:0.00914,((296:0.00014,(245:0.00528,291:0.01087)0.884:0.01072)0.564:0.02251,((348:0.00837,384:0.00827)0.951:0.04585,(428:0.02172,435:0.02712)0.838:0.02582)0.743:0.03345)0.023:0.00016)0.926:0.016,(229:0.00015,((309:0.02156,(157:0.01646,(235:0.0739,288:0.02349)0.935:0.04288)0.888:0.02448)0.806:0.01075,((143:0.01222,99:0.02577)0.065:0.0049,(120:0.02212,148:0.03357)0.732:0.00443)0.943:0.02146)0.851:0.01089)0.855:0.01021)0.889:0.01046)1.000:0.00014,((101:0.01909,(141:0.00014,154:0.01582)0.763:0.00487)0.875:0.01291,((128:0.01036,253:0.00519)0.734:0.00539,(((240:0.02482,(126:0.01979,(140:0.0054,199:0.03843)0.738:0.01722)0.550:0.00726)0.066:0.00598,(40:0.01546,((107:0.00014,((263:0.02739,((230:0.00525,251:0.02704)0.668:0.00015,(((231:0.02647,297:0.01435)0.843:0.01391,(393:0.01089,(75:0.00014,160:0.00014)0.397:0.00015)0.843:0.00513)0.635:0.00014,92:0.00016)0.838:0.00511)0.957:0.01032)0.321:0.00014,85:0.0051)0.840:0.00508)0.877:0.01022,(61:0.00016,66:0.03726)0.983:0.03719)0.018:0.00015)0.797:0.00886)0.893:0.01188,132:0.00015)0.974:0.02665)0.931:0.01579)0.788:0.00514)0.871:0.01027,((133:0.00464,381:0.02942)0.932:0.01625,(121:0.01046,344:0.01099)0.921:0.01599)0.908:0.00015)0.964:0.00015,(313:0.03321,314:0.04605)0.895:0.01872)0.862:0.02644,(((177:0.00015,(((418:0.01756,((129:0.0,214:0.0):0.01191,(273:0.0172,316:0.02655)0.704:0.00543)0.747:0.00614)0.880:0.01298,282:0.01152)0.761:0.00625,(234:0.02239,436:0.03053)0.762:0.00988)0.898:0.00015)0.838:0.00517,((167:0.0,196:0.0):0.00016,321:0.01066)0.928:0.01041)0.975:0.00015,(295:0.00015,(292:0.00016,(((108:0.00824,(210:0.00509,336:0.06592)0.913:0.02095)0.752:0.00816,328:0.00552)0.249:0.01037,271:0.00015)0.959:0.02084)0.580:0.0103)0.920:0.01027)0.865:0.01052)0.751:0.00597,((293:0.01657,270:0.04174)0.741:0.00513,((311:0.05063,371:0.01364)0.418:0.01153,(205:0.00787,367:0.01549)0.887:0.01792)0.700:0.00769)0.935:0.0179)0.871:0.01345,(250:0.01588,339:0.01855)0.940:0.02734)0.837:0.0123,(((((178:0.01106,((125:0.0,204:0.0):0.00531,220:0.0385)0.740:0.00464)0.672:0.00524,(197:0.04008,81:0.00398)0.917:0.01677)0.527:0.00015,(((64:0.04233,(317:0.00014,(368:0.03503,377:0.01132)0.757:0.00501)0.725:0.00686)0.876:0.01641,172:0.00015)0.785:0.00554,145:0.00934)0.874:0.01052)0.674:0.00549,78:0.00469)0.829:0.00531,(198:0.03931,(216:0.05763,77:0.0047)0.807:0.0104)0.553:0.0107)0.802:0.00014)0.903:0.01049)0.741:0.00014)0.859:0.00015,(55:0.0211,(397:0.0598,60:0.03851)0.760:0.01044)0.752:0.00488)0.762:0.00647)0.821:0.01394,(208:0.04985,43:0.00598)0.937:0.02237)0.829:0.02204)0.606:0.00014)0.335:0.00016)0.788:0.00544);"""

template_alignment_subset = """>114239
------------------------------------------------------------------------------------------------------------aGAGTTT-GA--T-CC-T-G-GCTC-AG-AT-TGAA-C-GC--TGG-C--G-GT-A-TG--C----T-T--AACACA-T-GC-A-AGT-CGA-A-CG----------G-TAA-CA-G-----------------------------CAG-A-AG----------------------------------------------------CTT-G----------------------------------------------------------------------------------CTT-CT------------------GGCT--G--AC--G--AG-T-GG-C-GG-A--C-------------GGG-TGAGT-A--AC-GC-G-T-A-GG---A-A--T-CT-G--C-CTTA---CA-G------------------------------------------------------------------T-GG----GGG-AT-AG-CCC-------------------------G-G-T-----------------------GAA-A---ACC-GGA-TTAA-TA---CC-G--C-AT-A----------C--------------------G-------------------------------------CC-C-----------------------------------------------------------------------------------------------------------------------T-AC-G--------------------------------------------------------------------------------------------------------------------------------------G-G-G---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------GAAA--G-CTG--G-----G--GA-T--C--------------------------------------------------------------------------------------------------------------------TTC-G----------------------------------------------------------------------------------------------------------------------G-A--CC-TG--G---C-A--------------C----T-G---T-TA-G---AT---G-A-----G-CCT-GCG--T-GAG--A------TT--A--G-CT-T----G---TTGG-T-G-GG-G-T----AAT-GG---C-CTACCA--A-GG-C-A--A-CG-A------------TCT-C-T------AG-CT-G-G-TCT-G-AG----A--GG-AC--G-AT-C-AG-CCAC-A-CTGGG--A-C-TG-A-GA-C-AC-G-GCCCAG-A-CTCC-TAC-G--G-G-A-G-GC-A-GC-A-G-TG---GG-G-A-ATA-TTGCA-C-AA-T-GG--GC-GA-G----A-G-CC-T-GA-TG-CA-GCAA-TACC-G-CG-T---G-T-G--T--GA-A-G--A--A-G-G--C----CTG-AG---------G-G-T-T-G-T--A---AA-G-CAC--------TT-TC-A-A--T--TG---TGA-A--G---AAAAGCT---T-TT-GG----T--T--AA-T---A----------AC-C-TTGAGTC-TT-GA-CA-TTAA-C-A--A-TA-C---------AA-----------GAAGC-ACC-GG-C-TAA---C--T-CCGT--GCCA--G-C---A--GCCG---C-GG--TA-AT--AC---GG-AG-GGT-GCA-A-G-CG-TTAA-T-CGG-AA-TT-A--C-T--GGGC-GTA----AA-GAGT-AC--G-TA-G-G-T-G------------G--T-TC-G-T-T-AA----GTC---A---G-ATG-TG-A-AA-GC--CC-CGG-G--------------------------------------------------------------------CT-C-AA-------------------------------------------------------------------------CC-T-G-GG-AA-C----T-G-C-A-T-T--------T---GAA-A-C-T-G-GCA--A-A-C---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------T-T-G-A-G-T-A-----T-GG--TA-G-A------------G-GC-G-GG-T----AG--AA-T-TCA-TAGT--GT-A-GCG-GTGAAA-TG-CGT-AGAT-ATTA-T-G-A--GG-A-AT-A-CC-AG--T--G--GC-GAA-G--G-C---G----G--C-C-CGCTG------G-AC-CA--------------------------------------------------------------AT-A-C-C--GA--CA-----CT-GA-GG--T-A-CGA--AA-G-C--------------G-TGGG-GAG-C-G-AACA--GG-ATTA-G-ATA-C-----CC-T-G-GTA-G-T----C-CA--C-G-CCG-T-AAA--C-GATG-TC--AA-CT---------A-GC--C--G-T-TG-G-GA-C-----------------------------------------------------------------------------------------CTT-GA--------------------------------------------------------------------------------------------------------------------------------------------------G-G-T-CT--C-A-G-T-GG-T------GC--A----GC-TAA--CG-C---G--T-GAA-GT--T----G-ACC-GCC-T-G-GG-GAG-TA---CGG-----C-C--G-C-A-A-GGC-T--AAA-ACTC-AAA---------TGAA-TTG-ACGAG-G-G-CCCG----C-A--C-A-A-GCG-GT-G--G--AG-CA-T--GT-GGT-TT-AATT-C-G-ATG-CAAC-G-CG-A-AG-A-A-CC-TT-A-CC-ATCCC-TT-G-AC-ATC-A---------------TC-A-G-------------A-A--CTT-G--TT--A-GA-G-A-T--A-A-C----T-CGG--T-G-----CC-------------------------------------T--TC-G------------------------------------------GG------------AA--CTGA-AT--GA---------------------------------------------------C-A-G-G-T-GCTG-CA-TGG-CT--GTC-GTC-A-GC-TC---G-TG-TC-G--TGA-GA-TGT-T-GG-G-TT-AA-GT-CCCGC-AA--------C-GAG-CGC-A-ACC-C-T-TA--TC--C-TTAG--T-T-G-C-C---AG-C-A----CG------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------TAAT------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------GG---T----G-G-----------------G---A-A--CT---------------C-T-A-A-G-GA-G--AC-T-G-CCG--G-T------------------------------------G-A---TAA----------------------------------A-C-C-G--G-A-GG-A--AGG-T--GGGG-A-TGAC-GTC--AAGT-C---ATC-A-T-G-G-C-C-CTT----AT-G--GG-A-T-GG-GC-TA-CAC-ACGTG-C--TA--CAATG---G-T-CA-GTA--C-AAA-GG-GC--------------------------------------------------------------------------------------------------A-G-C-C-A--A-CTCG-C--G---------------------------------------A-GA-G-T-----------G--C-G-CA---A----------A--TCC-C------A-T-AAAGCTGA---T-C-G-TAG-TCC--------GGA-T-TGGAG-TC--T-GCAA-CT-C-------------------------------------------------------------------------------------------------G-ACTCC-A-T-G-AA-G-TT-GGAAT-CG-C-TA--G-TA-AT-C-G-T----GAA-TC-A-G--A------AT--GTC-AC-G-GT-G-AAT-ACGT-T-CCCGGGCCT-TGCA----CACACCG-CCC-GTC-----a---ca--cca-tg-gg-a--g---tgg-g-ct-gc-aaa--a-gaa------g--t-agg-ta-g-t-t-t-aa-c-c--------------------------------------------------------------ttc-g------------------------------------------------------------------------------------------------------gg-a--ga-a--c---gc-tta--cc--act-t----t-gtg-gt-tca------------------------tg--act-gggg-tg-aag-tcgtaacaa-ggtag-ccct-aggggaa-cctg-gggc-tggatcacctcctt-----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
>200959
------------------------------------------------------------------------------------------------------------agagttt-ga--t-cc-t-g-gctc-ag-at-tgaa-c-gc--tgg-c--g-gc-a-gg--c----c-t--aacaca-t-gc-a-agt-cga-a-cg----------g-taa-ca-g----------------------------gaag-c-ag----------------------------------------------------ctt-g----------------------------------------------------------------------------------ctg-cttt----------------g-ct--g--ac--g--ag-t-gg-c-gg-a--c-------------ggg-tgagt-a--at-gt-c-t-g-GA---TAA--T-CT-A--C-C-CTT--GA-G------------------------------------------------------------------T-CA----AGG-AT-AA-GCA-------------------------C-G-G-----------------------GAA-A---CTG-TGG-CTAA-CA---CT-T--G-AT-A---------------------------------G--G-G-A--A--T-----------------TA-GGCC-T-----------------------------------------------------------------------------------------------------------------------G-GA-A--------------------------------------------------------------------------------------------------------------------------------------A-G-GC-A-T---------------T--T-T-C-C-T-------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------GAAAC-TCGT---------------------------------------------------------------------------------------------------------------------------------------GCT-A----------------------------------------------------------------------------------------------------------------------------------C-------------------------------A-AG-G---AT---G-A-----G-TCT-GCG--G-ATG--A------TT--A--G-CT-T----G---TTGG-T-G-GG-G-T----AAC-TG-C-C-T-ACCA--A-GG-C-G--A-TG-A------------TCA-T-T------AC-GG-G-C-TCT-T-AG----C--GG-AG--T-TG-C-CC-GGAG-A-AGGAA--T-C-TG-A-GA-C-AC-T-A-TTCCTA-GCTC-TAC-G--G-A-G-T-GC-A-GC-A-G-TC---GC-G-G-AAC-ATTTA-C-AA-T-GC--AC-GA-A----A-G-TG-T-GA-TA-AT-GCAA-GCCA-A-AG-T---G-C-T--T--A----------T-C-A--------TT-T--------------A-G-A--------T--A-AGC--------TT-TT-T-C--T-TAG----T------------------------------G--T--AA-A---A----A------------------------------T-C-TA-G-GA-G---------AA-----------TAAGGATCTGGG-A--AA---G-AC-TGGT--GCCA--G-C---C--GCCG---C-GG--TA-AT--AC---CA-GC-AGT-TCA-A-G-TT-GCAT-C-CAG-TT-TT-A--T-T--GGGT-CTA----AA-ACAT-CC--G-TA-G-C-T-T------------G--T-TT-A-C-T-AA----G-T-C-T---C-CTG-TG-A-AA-TC--CT-ACG-T--------------------------------------------------------------------CT-C-AA-------------------------------------------------------------------------GC-G-T-GG-GG-CT---T-G-C-A-G-G--------A--GA-T-A-C-T-G-GTA--A-G-C---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------T-A-G-A-G-A-T-----C-GG--TA-G-A------------C-GT-A-AG-G----AG--TACG-CCG-G-GG--GT-A-ACG-GTTAAA-TG-TGT-TAAT-C-CT-C-GGT--GG-A-CT-A-AC-AA--T--G-AGC-GAA-G--G-C---A----C--C-T-TACGA------G-GA-C---------------------------------------------------------------GAAT-C-T--GA--CA-----GT-GA-GG--G-A-TGA--AG-G-C--------------C-AGGG-GCG-C-A-AAAC--GG-ATTA-G-ATA-C-----CC-G-T-GTA-G-T----C-CT--G-G-CAG-T-AAA--C-GCTG-CT--CA-CT---------A-AA--C--A-T-CG-G-GC-C--C--TC----------------------------------------------------------------------------------TTC-GG---------------------------------------------------------------------------------------------------------------------------------------------GA---G-G-A-TT--C-G-G-T-GC-T------GA--A----GC-GAA--GG-C-G-A--T--AA-GT--G----A-GCT-ACC-T-G-GG-AAG-TA---TAG-----T-C--G-C-A-A-GAT-C--GAA-ACTT-AAA---------GGAA-TTG-GC-GG-G-G-AGAC----ATA--C-A-A-CGA-GT-G--A--CG-CG-T--GC-GGT-TC-RATT-A-G-ATT-TTAC-A-CC-G-TG-A-A-CC-TC-A-CC-AGGAG-------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------CGACAGC--AGGATGAAGGTCAGTCTGAAGGGCTTACC-TGA-CACGCTGAG-C-G-G-A-GATG-CA-TGG-CC--GAC-GTC-A-GC-CT---G-TG-CT-G--TGA-AG-TGA-C-CC-G-TT-AA-AT-CAGGT-AA--------C-AGG-CGA-G-ACC-C-G-TG--TC--G-TTAA--T-T-A-C-T---AC-A-G--A--AA-------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------T-G------------G----G---T-A--CC---------------T-T-A-A-C-GA-G--AC-T-G-CC---T-T------------------------------------G-G---TAA----------------------------------C-A-A-G--G-A-GG-A--AGG-T--GCGG-G-CTAC-GTT--AGGT-G---AGT-A-C-G-T-C-C-CTA----AT-C--CC-C-T-GG-GC-TA-CAC-GCGCG-C--GG--CAATG---G-TATG-T-T--C-AAA-TG-GA--------------------------------------------------------------------------------------------------T-G-C-A-A--C-TCCG-A--A---------------------------------------A-GG-A-G-----------A--A-G-CT---A----------A--TCC-C------C-T-AACGC-AT-A-T-C-T-CAG-TCC--------AGA-T-TGAGG-GT--T-GCAA-CC-C-------------------------------------------------------------------------------------------------A-CCCTC-A-T-G-AC-G-AT-GGAAT-TC-G-TA--G-TA-AT-C-G-G----GTA-TT-A-G--C------AC--TGC-CC-G-GT-G-AAT-AAGT-C-CCTGTCTCT-TGCA----CACACCG-CCC-GTC-----A---AA--CCA-AC-CG-A--G---TTA-T-GT-AG-GGG--C-GA-------A--G-CCC-T--T-A-T-T-aa-c-c-------------------------------------------------------------t-tc-g------------------------------------------------------------------------------------------------------gg-a--gg-g--c---gc-tta--cc--act-t----t-gtg-at-tca------------------------tg--act-gggg-tg-aag-tcgtaacaa-ggtaa-ccgt-aggggaa-cctg-cggc-tggatcacctcctt-----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
>83155
------------------------------------------------------------------------------------------------------------agagttt-ga--t-cc-t-g-gctc-ag-aa-cgaa-c-gc--tgg-c--g-gc-g-cg--t----c-t--taagca-t-gc-a-agt-cga-a-cgg---------c-aa-gag-g------------------------------aa---ag----------------------------------------------------ctt-g----------------------------------------------------------------------------------ct--tt------------------tctC--C--TA--G--AG-T-GG-C-GG-A--C-------------TGG-TGAGT-A--AC-AC-G-T-G-GG---TGA--T-AT-A--C-C-TTT--TG-G------------------------------------------------------------------A-CG----GGG-AT-AG-CCC-------------------------C-T-A-----------------------GAA-A---TAG-GGG-GTAA-TA---CC-G--G-AT-A----------A--------------------G-G--C-C-G--T--G----------------CG----GG-T-----------------------------------------------------------------------------------------------------------------------T-GG-A--------------------------------------------------------------------------------------------------------------------------------------G-C-C----G---------------C--A-C-G-G-G-------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------GAAA--G-G-C-GC-----------------------------------------------------------------------------------------------------------------------------------TAT-G----------------------------------------------------------------------------------------------------------------------------G-CG--C---C-G--------------C----C-G---A-AA-G---AA---T-A-----G-CCT-GCG--A-CCT--A------TC--A--G-CT-T----G---TTGG-T-G-AG-G-T----AAA-GG-C-C-C-ACCA--A-GG-C-G--A-TG-A------------CAG-G-T------AT-CC-G-G-CCT-G-AG----A--GG-GT--G-AA-C-GG-ACAC-A-TTGGG--A-C-TG-A-GA-T-AC-G-G-CCCAGA-CTCC-TAC-G--G-G-A-G-GC-A-GC-A-G-CT---AA-G-A-ATA-TTCCG-C-AA-T-GG--AG-GA-A----A-C-TC-T-GA-CG-GA-GCAA-CGCC-G-CG-T---G-G-A--C--GA-T-G--A--A-G-G-CC-----GG-AA---------G-G-T-T-G-T--A---AA-G-TCC--------TT-TT-A-T--A-ATT----GA-G--G---AATAAGC---GGGA-CA----G--G--GA-A--------T-----GG-T-TCC-GCG-GT-GA-CT-GT-A-G-AT-TAT--G---------AA-----------TAAGC-ACC-GG-C-TAA---T--T-ACGT--GCCA--G-C---A--GCCG---C-GG--TA-AC--AC---GT-AA-GGT-GCG-A-G-CG-TTAT-T-CGG-AA-TT-A--T-T--GGGC-GTA----AA-GGGC-AC--G-CA-G-G-C-G------------G--C-TT-T-G-C-AA----G-C-T-T---G-GTG-TG-A-AA-TC--TC-AGG-G--------------------------------------------------------------------CT-C-AA-------------------------------------------------------------------------CC-C-T-GA-AA-C----T-G-C-A-T-T--------G--AG-A-A-C-T-G-CAT--G-G-C---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------T-A-G-A-G-T-T-----A-GT--GA-G-G------------G-GA-A-AC-C----GG--AATT-CCA-G-GT--GT-A-GGG-GTGAAA-TC-TGT-AGAT-A-TC-T-GGA--AG-A-AC-A-CC-AA--T--G--GC-GAA-G--G-C---A----G--G-T-TTCCA------G-CA-CA--------------------------------------------------------------TA-A-C-T--GA--CG-----CT-CA-GG--T-G-CGA--AG-G-T--------------G-CGGG-GAT-C-A-AACA--GG-ATTA-G-ATA-C-----CC-T-G-GTA-G-T----C-CG--C-A-CAG-T-AAA--C-GATG-TG--CA-CT---------A-GG--T--G-T-CT-G-GG-C------------------------------------------------------------------------------------------AT-AA----------------------------------------------------------------------------------------------------------------------------------------------------G-C-CC--G-G-G-T-GC-C------AA--A----GC-AAA--CG-C-G-A--T--AA-GT--G----C-ACC-GCC-T-G-GG-GAG-TA---TGC-----C-C--G-C-A-A-GGG-T--GAA-ACTC-AAA---------GGAA-TTG-ACGGG-G-G-CCCG----C-A--C-A-A-GCG-GT-G--G--AG-CA-T--GT-GGT-TT-AATT-C-G-ATG-GTAC-G-CG-A-GG-A-A-CC-TT-A-CC-TGGGT-TT-G-AC-A-T-A------------C-ATA-C-C-------------G-AT-T-A-T--TT--A-GA-G-A-T--A-A-G--T-A-A-G--C-G-----TA-------------------------------------G--CA-A------------------------------------------TA----C----G---GG-TAT---G--AA---------------------------------------------------C-A-G-G-T-GCTG-CA-TGG-CT--GTC-GTC-A-GC-TC---G-TG-CC-G--TGA-GG-TGT-T-GG-G-TT-AA-GT-CCCGC-AA--------C-GAG-CGC-A-ACC-C-C-TA--CT--G-CCAG--T-T-A-C-T---AA-C-A--G--G-------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------TGA------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------AG---C----T-G------------A----G---G-A--CT---------------C-T-G-G-C-GG-A--AC-T-G-CCG--G-C------------------------------------G-A---CAA----------------------------------G-C-C-G--G-A-GG-A--AGG-C--GGGG-A-TGAC-GTC--AAGT-C---ATC-A-T-G-G-C-C-CTT----AT-G--TC-C-A-GG-GC-TA-CAC-ACGTG-C--TA--CAATG---G-TCGG-T-A--C-AGA-GT-GA--------------------------------------------------------------------------------------------------T-G-C-G-A--G-GCAG-T--G---------------------------------------A-TG-C-G-----------G--A-G-CA---A----------A--ACG-C------A-G-AAAAC-CG-A-T-C-G-TAG-TCC--------GGA-T-TGGAG-TC--T-GAAA-CC-C-------------------------------------------------------------------------------------------------G-ACTCC-A-T-G-AA-G-TT-GGAAT-CG-C-TA--G-TA-AT-C-G-C----ACA-TC-A-G-C-------AC--GGT-GC-G-GT-G-AAT-ACGT-T-CCCGGGCCT-TGTA----CACACCG-CCC-GTC-----A---CA--CCA-TC-CG-A--G---TCG-A-GG-AT-ACC--C-AAA------G--T-CGG-TA-G-G-C-T-AA-C-C---------------------------------------------------------------GC-A---------------------------------------------------------------------------------------------------A--GG-G--GG-C--C---GC-TGC--CC--AAG-G----T-ATG-CT-TGG------------------------TA--AGG-GGGG-TG-AAG-TCGTAACAA-GGTAG-CCGT-ACTGGAA-AGTG-Cggc-tggatcacctcctt-----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
>105109
------------------------------------------------------------------------------------------------------------AGAGTTT-GA--T-CC-T-G-GCTC-AG-GA-TAAA-C-CT--TGG-A--A-GGAG-CA--C----A-T--AAGACA-T-GC-A-AGT-CGA-A-CGA---------A-C---AG-A-------------A--------------GCTA-T-TC----------------------------------------------------TTA-T----------------------------------------------------------------------------------AGA-TTAGT-----------------GG--A-AGT--T--AG-T-GG-C-GG-A--C-------------TGG-TGAGT-A--AT-GT-A-T-A-AG---TAA--C-CT-G--C-C-TAT--CA-G------------------------------------------------------------------A-GG----GGA-AC-AA-CAG-------------------------T-T-G-----------------------GAA-A---CGA-CTG-CTAA-TA---CC-G--C-AT-A----------T--------------------G-C--C-A-T-AA--G-----------------GT---TC-G-------------------------------------------------------------------------------------------------------------------------CA-T--------------------------------------------------------------------------------------------------------------------------------------G-G-A--C-C---------------A--A-GT--G-G-------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------GAAA--G-G-A--------------------------------------------------------------------------------------------------------------------------------------GCA-A-------------------------------------------------------------------------------------------------------------------------------T--C---C-G--------------C----T-G---A-TA-G---AT---G-G-----A-CTT-ATA--T-CTG--A------TT--A--G-CT-A----G---TAGG-T-AGAG-G-T----AAT-GG-C-T-C-ACCT--A-GG-C-G--A-CG-A------------TCA-G-T------AG-CC-G-G-ACT-G-A-G---A--GG-TT--A-AA-C-GG-CCAC-A-TTGGT--A-C-TG-A-GA-T-AC-G-G-CCCAGA-CTCC-TAC-G--G-G-A-G-GC-A-GC-A-G-TC---GG-G-A-ATA-TTGCG-C-AA-T-GG--AG-GA-A----A-C-TC-T-GA-CG-CA-GTGA-CGCC-G-CG-T---G-C-A--G--GA-A-G--A--A-G-G-TT-----TT-CG---------G-A-T-T-G-T--A---AA-C-TGC--------TTTAG-A-C--A-GGG----AA-G--A---A-----------------------G--AA------------------------------GTGA-CA-GT-A-C-CT-G-TA-G---------AA-----------TAAGC-TCC-GG-C-TAA---C--T-ACGT--GCCA--G-C---A--GCCG---C-GG--TA-AT--AC---GT-AG-GGA-GCG-A-G-CG-TTAT-C-CGG-AT-TT-A--T-T--GGGT-GTA----AA-GGGT-GC--G-TA-G-A-C-G------------G--G-AA-G-T-T-AA----G-T-T-A---G-TTG-TG-A-AA-TC--CC-TCG-G--------------------------------------------------------------------CT-C-AA-------------------------------------------------------------------------CT-G-A-GG-AA-C----T-G-C-A-A-C--------T--AA-A-A-C-T-G-ATT--T-T-C---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------T-T-G-A-G-T-A-----C-TG--GA-G-A------------G-GA-A-AG-T----GG--AATT-CCT-A-GT--GT-A-GCG-GTGAAA-TG-CGT-AGAT-A-TT-A-GGA--GG-A-AC-A-CC-AG--T--G--GCG--AAG--G-C---G----A--C-T-TTCTG------G-AC-AG--------------------------------------------------------------AA-A-C-T--GA--CG-----TT-GA-GG--C-A-CGA--AA-G-T--------------G-TGGG-GAG-C-A-AACA--GG-ATTA-G-ATA-C-----CC-C-T-GGT-A-G----TCCA--C-A-CCG-T-AAA--C-GATG-GA--TA-CT---------A-GG--T--G-T-AG-G-GC-A----------------------------------------------------------------------------------------T-GA-AT-AA-------------------------------------------------------------------------------------------------------------------------------------------------T-G-CT--C-T-G-T-GC-C------GT--C----GC-AAA--CG-C-A-A--T--AA-GT--A----T-CCC-ACC-T-G-GG-GAG-TA---CGG-----C-C--G-C-A-A-GGT-T--GAA-ACTC-AAA---------GGAA-TTG-ACGGG-G-G-CCCG----C-A--C-A-A-GCA-GT-G--G--AG-TA-T--GT-GGT-TT-AATT-C-G-AAG-CAAC-G-CG-A-AG-A-A-CC-TT-A-CC-AGGGC-TT-G-AC-A-T-A-------------TAAG-T-G-------------A-AT-A-A-C--TA--A-GA-G-A-T----T-A--G-T-T-A--G-------TT-------------------------------------C--TT-C------------------------------------------GG---------A---AC-ACT-T-A--TA---------------------------------------------------C-A-G-G-T-GGTG-CA-TGG-TT--GTC-GTC-A-GC-TC---G-TG-TC-G--TGA-GA-TGT-T-AG-G-TT-AA-GT-CCTGC-AA--------C-GAG-CGC-A-ACC-C-C-TG--TT--C-TTAG--T-T-G-C-C---AG-C-ATGT----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------AAAG----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------A----T-G------------G----G---A-A--CT---------------C-T-A-G-G-AA-G--AC-T-G-CCG--G-T------------------------------------G-A---CAA----------------------------------A-T-C-G--G-A-GG-A--AGG-T--GGGG-A-CGAC-GTC--AAAT-C---ATC-A-T-G-C-C-C-TTT----AT-G--TC-C-T-GG-GC-TA-CAC-ACGTA-C--TA--CAATG---G-CTAT-A-A--C-AAA-GG-GA--------------------------------------------------------------------------------------------------A-G-C-G-A--A-GTAG-T--G---------------------------------------A-TA-T-G-----------G--A-G-CA---A----------A--ACC-C------A---AAAAAGTA-G-T-C-T-CAG-TTC--------GGA-T-TGAAG-GC--T-GAAA-TT-C-------------------------------------------------------------------------------------------------G-CCTTC-A-T-G-AA-G-CT-GGAAT-TG-C-TA--G-TA-AT-G-G-C----AGG-TC-A-G-C-------AT--ACT-GC-C-GT-G-AAT-ACGT-T-CCCGGGCCT-TGTA----CACACCG-CCC-GTC-----A---CA--CCA-TG-AG-A--G---TTG-G-GA-AT-ACC--C-GAA------G--C-CTG-TG-A-G-C-C-AA-C-CG------------------------------------------------------------T-AA-G------------------------------------------------------------------------------------------------------GG-G--GC-A------GC-AGT--CG--AAG-G----T-AGA-AT-CAA------------------------TG--ATT-GGGG-TG-AAG-TCGTAACAA-GGTAG-CCGT-ATCGGAA-GGTG-CGGC-TGGATCACCTCCTTTCT--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
"""

sra_ref_collection = """>114239
AGAGTTTGATCCTGGCTCAGATTGAACGCTGGCGGTATGCTTAACACATGCAAGTCGAACGGTAACAGCAGAAGCTTGCTTCTGGCTGACGAGTGGCGGACGGGTGAGTAACGCGTAGGAATCTGCCTTACAGTGGGGGATAGCCCGGTGAAAACCGGATTAATACCGCATACGCCCTACGGGGGAAAGCTGGGGATCTTCGGACCTGGCACTGTTAGATGAGCCTGCGTGAGATTAGCTTGTTGGTGGGGTAATGGCCTACCAAGGCAACGATCTCTAGCTGGTCTGAGAGGACGATCAGCCACACTGGGACTGAGACACGGCCCAGACTCCTACGGGAGGCAGCAGTGGGGAATATTGCACAATGGGCGAGAGCCTGATGCAGCAATACCGCGTGTGTGAAGAAGGCCTGAGGGTTGTAAAGCACTTTCAATTGTGAAGAAAAGCTTTTGGTTAATAACCTTGAGTCTTGACATTAACAATACAAGAAGCACCGGCTAACTCCGTGCCAGCAGCCGCGGTAATACGGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGAGTACGTAGGTGGTTCGTTAAGTCAGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTTGAAACTGGCAAACTTGAGTATGGTAGAGGCGGGTAGAATTCATAGTGTAGCGGTGAAATGCGTAGATATTATGAGGAATACCAGTGGCGAAGGCGGCCCGCTGGACCAATACCGACACTGAGGTACGAAAGCGTGGGGAGCGAACAGGATTAGATACCCTGGTAGTCCACGCCGTAAACGATGTCAACTAGCCGTTGGGACCTTGAGGTCTCAGTGGTGCAGCTAACGCGTGAAGTTGACCGCCTGGGGAGTACGGCCGCAAGGCTAAAACTCAAATGAATTGACGAGGGCCCGCACAAGCGGTGGAGCATGTGGTTTAATTCGATGCAACGCGAAGAACCTTACCATCCCTTGACATCATCAGAACTTGTTAGAGATAACTCGGTGCCTTCGGGAACTGAATGACAGGTGCTGCATGGCTGTCGTCAGCTCGTGTCGTGAGATGTTGGGTTAAGTCCCGCAACGAGCGCAACCCTTATCCTTAGTTGCCAGCACGTAATGGTGGGAACTCTAAGGAGACTGCCGGTGATAAACCGGAGGAAGGTGGGGATGACGTCAAGTCATCATGGCCCTTATGGGATGGGCTACACACGTGCTACAATGGTCAGTACAAAGGGCAGCCAACTCGCGAGAGTGCGCAAATCCCATAAAGCTGATCGTAGTCCGGATTGGAGTCTGCAACTCGACTCCATGAAGTTGGAATCGCTAGTAATCGTGAATCAGAATGTCACGGTGAATACGTTCCCGGGCCTTGCACACACCGCCCGTCACACCATGGGAGTGGGCTGCAAAAGAAGTAGGTAGTTTAACCTTCGGGAGAACGCTTACCACTTTGTGGTTCATGACTGGGGTGAAGTCGTAACAAGGTAGCCCTAGGGGAACCTGGGGCTGGATCACCTCCTT
>200959
AGAGTTTGATCCTGGCTCAGATTGAACGCTGGCGGCAGGCCTAACACATGCAAGTCGAACGGTAACAGGAAGCAGCTTGCTGCTTTGCTGACGAGTGGCGGACGGGTGAGTAATGTCTGGATAATCTACCCTTGAGTCAAGGATAAGCACGGGAAACTGTGGCTAACACTTGATAGGGAATTAGGCCTGGAAAGGCATTTTCCTGAAACTCGTGCTACAAGGATGAGTCTGCGGATGATTAGCTTGTTGGTGGGGTAACTGCCTACCAAGGCGATGATCATTACGGGCTCTTAGCGGAGTTGCCCGGAGAAGGAATCTGAGACACTATTCCTAGCTCTACGGAGTGCAGCAGTCGCGGAACATTTACAATGCACGAAAGTGTGATAATGCAAGCCAAAGTGCTTATCATTTAGATAAGCTTTTTCTTAGTGTAAAAATCTAGGAGAATAAGGATCTGGGAAAGACTGGTGCCAGCCGCCGCGGTAATACCAGCAGTTCAAGTTGCATCCAGTTTTATTGGGTCTAAAACATCCGTAGCTTGTTTACTAAGTCTCCTGTGAAATCCTACGTCTCAAGCGTGGGGCTTGCAGGAGATACTGGTAAGCTAGAGATCGGTAGACGTAAGGAGTACGCCGGGGGTAACGGTTAAATGTGTTAATCCTCGGTGGACTAACAATGAGCGAAGGCACCTTACGAGGACGAATCTGACAGTGAGGGATGAAGGCCAGGGGCGCAAAACGGATTAGATACCCGTGTAGTCCTGGCAGTAAACGCTGCTCACTAAACATCGGGCCCTCTTCGGGAGGATTCGGTGCTGAAGCGAAGGCGATAAGTGAGCTACCTGGGAAGTATAGTCGCAAGATCGAAACTTAAAGGAATTGGCGGGGAGACATACAACGAGTGACGCGTGCGGTTCRATTAGATTTTACACCGTGAACCTCACCAGGAGCGACAGCAGGATGAAGGTCAGTCTGAAGGGCTTACCTGACACGCTGAGCGGAGATGCATGGCCGACGTCAGCCTGTGCTGTGAAGTGACCCGTTAAATCAGGTAACAGGCGAGACCCGTGTCGTTAATTACTACAGAAATGGGTACCTTAACGAGACTGCCTTGGTAACAAGGAGGAAGGTGCGGGCTACGTTAGGTGAGTACGTCCCTAATCCCCTGGGCTACACGCGCGCGGCAATGGTATGTTCAAATGGATGCAACTCCGAAAGGAGAAGCTAATCCCCTAACGCATATCTCAGTCCAGATTGAGGGTTGCAACCCACCCTCATGACGATGGAATTCGTAGTAATCGGGTATTAGCACTGCCCGGTGAATAAGTCCCTGTCTCTTGCACACACCGCCCGTCAAACCAACCGAGTTATGTAGGGGCGAAGCCCTTATTAACCTTCGGGAGGGCGCTTACCACTTTGTGATTCATGACTGGGGTGAAGTCGTAACAAGGTAACCGTAGGGGAACCTGCGGCTGGATCACCTCCTT
>83155
AGAGTTTGATCCTGGCTCAGAACGAACGCTGGCGGCGCGTCTTAAGCATGCAAGTCGAACGGCAAGAGGAAAGCTTGCTTTTCTCCTAGAGTGGCGGACTGGTGAGTAACACGTGGGTGATATACCTTTTGGACGGGGATAGCCCCTAGAAATAGGGGGTAATACCGGATAAGGCCGTGCGGGTTGGAGCCGCACGGGGAAAGGCGCTATGGCGCCGCCGAAAGAATAGCCTGCGACCTATCAGCTTGTTGGTGAGGTAAAGGCCCACCAAGGCGATGACAGGTATCCGGCCTGAGAGGGTGAACGGACACATTGGGACTGAGATACGGCCCAGACTCCTACGGGAGGCAGCAGCTAAGAATATTCCGCAATGGAGGAAACTCTGACGGAGCAACGCCGCGTGGACGATGAAGGCCGGAAGGTTGTAAAGTCCTTTTATAATTGAGGAATAAGCGGGACAGGGAATGGTTCCGCGGTGACTGTAGATTATGAATAAGCACCGGCTAATTACGTGCCAGCAGCCGCGGTAACACGTAAGGTGCGAGCGTTATTCGGAATTATTGGGCGTAAAGGGCACGCAGGCGGCTTTGCAAGCTTGGTGTGAAATCTCAGGGCTCAACCCTGAAACTGCATTGAGAACTGCATGGCTAGAGTTAGTGAGGGGAAACCGGAATTCCAGGTGTAGGGGTGAAATCTGTAGATATCTGGAAGAACACCAATGGCGAAGGCAGGTTTCCAGCACATAACTGACGCTCAGGTGCGAAGGTGCGGGGATCAAACAGGATTAGATACCCTGGTAGTCCGCACAGTAAACGATGTGCACTAGGTGTCTGGGCATAAGCCCGGGTGCCAAAGCAAACGCGATAAGTGCACCGCCTGGGGAGTATGCCCGCAAGGGTGAAACTCAAAGGAATTGACGGGGGCCCGCACAAGCGGTGGAGCATGTGGTTTAATTCGATGGTACGCGAGGAACCTTACCTGGGTTTGACATACATACCGATTATTTAGAGATAAGTAAGCGTAGCAATACGGGTATGAACAGGTGCTGCATGGCTGTCGTCAGCTCGTGCCGTGAGGTGTTGGGTTAAGTCCCGCAACGAGCGCAACCCCTACTGCCAGTTACTAACAGGTGAAGCTGAGGACTCTGGCGGAACTGCCGGCGACAAGCCGGAGGAAGGCGGGGATGACGTCAAGTCATCATGGCCCTTATGTCCAGGGCTACACACGTGCTACAATGGTCGGTACAGAGTGATGCGAGGCAGTGATGCGGAGCAAAACGCAGAAAACCGATCGTAGTCCGGATTGGAGTCTGAAACCCGACTCCATGAAGTTGGAATCGCTAGTAATCGCACATCAGCACGGTGCGGTGAATACGTTCCCGGGCCTTGTACACACCGCCCGTCACACCATCCGAGTCGAGGATACCCAAAGTCGGTAGGCTAACCGCAAGGGGGCCGCTGCCCAAGGTATGCTTGGTAAGGGGGGTGAAGTCGTAACAAGGTAGCCGTACTGGAAAGTGCGGCTGGATCACCTCCTT
>105109
AGAGTTTGATCCTGGCTCAGGATAAACCTTGGAAGGAGCACATAAGACATGCAAGTCGAACGAACAGAAGCTATTCTTATAGATTAGTGGAAGTTAGTGGCGGACTGGTGAGTAATGTATAAGTAACCTGCCTATCAGAGGGGAACAACAGTTGGAAACGACTGCTAATACCGCATATGCCATAAGGTTCGCATGGACCAAGTGGGAAAGGAGCAATCCGCTGATAGATGGACTTATATCTGATTAGCTAGTAGGTAGAGGTAATGGCTCACCTAGGCGACGATCAGTAGCCGGACTGAGAGGTTAAACGGCCACATTGGTACTGAGATACGGCCCAGACTCCTACGGGAGGCAGCAGTCGGGAATATTGCGCAATGGAGGAAACTCTGACGCAGTGACGCCGCGTGCAGGAAGAAGGTTTTCGGATTGTAAACTGCTTTAGACAGGGAAGAAGAAGTGACAGTACCTGTAGAATAAGCTCCGGCTAACTACGTGCCAGCAGCCGCGGTAATACGTAGGGAGCGAGCGTTATCCGGATTTATTGGGTGTAAAGGGTGCGTAGACGGGAAGTTAAGTTAGTTGTGAAATCCCTCGGCTCAACTGAGGAACTGCAACTAAAACTGATTTTCTTGAGTACTGGAGAGGAAAGTGGAATTCCTAGTGTAGCGGTGAAATGCGTAGATATTAGGAGGAACACCAGTGGCGAAGGCGACTTTCTGGACAGAAACTGACGTTGAGGCACGAAAGTGTGGGGAGCAAACAGGATTAGATACCCCTGGTAGTCCACACCGTAAACGATGGATACTAGGTGTAGGGCATGAATAATGCTCTGTGCCGTCGCAAACGCAATAAGTATCCCACCTGGGGAGTACGGCCGCAAGGTTGAAACTCAAAGGAATTGACGGGGGCCCGCACAAGCAGTGGAGTATGTGGTTTAATTCGAAGCAACGCGAAGAACCTTACCAGGGCTTGACATATAAGTGAATAACTAAGAGATTAGTTAGTTCTTCGGAACACTTATACAGGTGGTGCATGGTTGTCGTCAGCTCGTGTCGTGAGATGTTAGGTTAAGTCCTGCAACGAGCGCAACCCCTGTTCTTAGTTGCCAGCATGTAAAGATGGGAACTCTAGGAAGACTGCCGGTGACAAATCGGAGGAAGGTGGGGACGACGTCAAATCATCATGCCCTTTATGTCCTGGGCTACACACGTACTACAATGGCTATAACAAAGGGAAGCGAAGTAGTGATATGGAGCAAAACCCAAAAAAGTAGTCTCAGTTCGGATTGAAGGCTGAAATTCGCCTTCATGAAGCTGGAATTGCTAGTAATGGCAGGTCAGCATACTGCCGTGAATACGTTCCCGGGCCTTGTACACACCGCCCGTCACACCATGAGAGTTGGGAATACCCGAAGCCTGTGAGCCAACCGTAAGGGGGCAGCAGTCGAAGGTAGAATCAATGATTGGGGTGAAGTCGTAACAAGGTAGCCGTATCGGAAGGTGCGGCTGGATCACCTCCTTTCT
"""

fasting_tutorial_fna = """>FLP3FBN01ELBSX length=250 xy=1766_0111 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTTACCCTCTCAGGCCGGCTACGCATCATCGCCTTGGTGGGCCGTTACCTCACCAAC
TAGCTAATGCGCCGCAGGTCCATCCATGTTCACGCCTTGATGGGCGCTTTAATATACTGA
GCATGCGCTCTGTATACCTATCCGGTTTTAGCTACCGTTTCCAGCAGTTATCCCGGACAC
ATGGGCTAGG
>FLP3FBN01EG8AX length=276 xy=1719_1463 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGCCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGGGCCGTTACCCCGCCAACA
ACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGAA
GAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAGG
TTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCA
>FLP3FBN01EEWKD length=244 xy=1692_3531 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGATCAGTCTCTTAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAAC
TAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGA
CATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCT
TGGG
>FLP3FBN01DEHK3 length=252 xy=1278_0245 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGACTTGGTGAGCCGTTACCTCACCAAC
TATCTAATCAGACGCGAGCCCATCTTTCAGCGGATTGCTCCTTTGGTATTCCGGCGATGC
CGCCAAAATCATTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGAAAGGC
AGGTTGCTCACG
>FLP3FBN01DGFYQ length=269 xy=1300_1344 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGGTCCATCTCACACCGATTAATCTTTTTCCAACCAGAGCATGCG
CCCCTGTTGGCTTATGCGGTATTAGCGGTCGTTTCCAACTGTTATCCCCCTGTGTGAGGC
AGGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01A55WZ length=272 xy=0363_0481 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGAGAAAATGCTC
ACTCATGATACCATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCA
GGTTGGATACGCGTTACTCACCCGTGCGCCGG
>FLP3FBN01D7O1S length=279 xy=1610_3070 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCGATAACCGAAATTCTTTAATAGTGAAACCATGCGGA
AATACTATACTATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTTATCGGCAG
GTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCATC
>FLP3FBN01CV5ZU length=274 xy=1069_1480 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAAC
CAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCT
CTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAG
GTTGCATACGCGTTACGCACCCGTGCGCCGGTCG
>FLP3FBN01DQ783 length=264 xy=1423_0757 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGG
ACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCA
GGTTGGATACGTGTTACTCACCCG
>FLP3FBN01DDPFF length=259 xy=1269_0617 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGG
ACTCATGATGCCATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCA
GGTTGGGTACGCGTTACTC
>FLP3FBN01CPD70 length=274 xy=0992_0942 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAATGTGG
GGGGCCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGGGCCGTTACCCCGCCAAC
AACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGA
AGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAG
GTTGGATACGTGTTACTCACCCGTGCGCCGGTCG
>FLP3FBN01BBAE6 length=269 xy=0421_2032 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCGGCACCGGGATGCC
CCGATGCCGCGTCACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTGGCGGGC
AGGTTGCATACGTGTTACTCACCCGTGCG
>FLP3FBN01AWYZD length=244 xy=0258_1671 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAAC
TAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGA
CATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCT
TGGG
>FLP3FBN01AM0P3 length=264 xy=0145_0217 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGCCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
CAGCTAATCAGACGCGGGCCCATCCCGTACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTCTCCACGCGTTACTCACCCG
>FLP3FBN01BO4IE length=240 xy=0579_0388 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGACTTGGTAGGCCGTTACCCCACCAAC
TATCTAATCAGACGCAAGCCCATCTATCAGCGGATTGCTCCTTTCCCATTTATATCATGT
GATATTCATGGCATATGCGGTATTAGCAGTCATTTCTAACTGTTGTTCCCCTCTGATAGG
>FLP3FBN01BPX14 length=254 xy=0588_1818 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGCCACGGTGGGCCGTTACCCCGCCGTC
AAGCTAATGTCACGCGAGCCTATCCTCATCCGACGGATCTTTAGATGGAACCAGATGCCT
GATTCCATCGCCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATCGAGGGC
AAGTTGCATACGCG
>FLP3FBN01DB5I7 length=268 xy=1251_1905 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCCTGCACCGCCGGAGCTTCCCCCGCCGCCCCATGCGG
GGCTGCGGGCATATGCGGTATTAGCAGCCGTTTCCGGCTGTTGTCCCCCAGTGCAGGGCA
GGTTGCCCACGCGTTACTCACCCGTCCG
>FLP3FBN01AK9OO length=249 xy=0125_0438 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGATCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTCGAGCCGTTACCTCACCAA
CTAGCTAATGCGCCGCGGGCCCATCTCATAGCGGATTACTCCTTTAATTGCTACTTCATG
CGAAGCTACAATCTTATGCGGTATTAATCTCCCTTTCGGAAGGCTATTCCCCTCTATGAG
GTCAGGTTG
>FLP3FBN01ANGF2 length=253 xy=0150_0112 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTC
AAGCTAATGTCACGCGAGCCTATCCTCATCCGACGGATCTTTAGATGGAACCAGATGCCT
GATTCCATCGCCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCA
AGTTGCTCACGCG
>FLP3FBN01AF994 length=272 xy=0068_1402 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGTCGGTTTGGTGGGCCGTTACCCCGCCAAC
TGCCTAATGGAACGCATGCCTATCTATCAGCGATGAATCTTTAGCAAATATCCCCATGCG
GGGCCCCTGCTTCATGCGGTATTAGTCCGACTTTCGCCGGGTTATCCCCTCCTGATAGGT
AAGTTGCATACGCGTTACTTCACCCGTCGCGG
>FLP3FBN01AHXU8 length=285 xy=0087_0802 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTAC
TGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCGGGGATGCCC
CCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAG
GTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGGCAGAGAG
>FLP3FBN01APRWO length=272 xy=0176_1782 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCT
CCCAACGTTCTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCA
GGTTCCATACGTGTTACTCACCCGTGCGCCGG
>FLP3FBN01C645E length=273 xy=1194_1600 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCAAC
AACCTAATGGAACGCATCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGAA
GAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAGG
TTGGATACGCGTTACTCACCCGTGCGCCAGTCG
>FLP3FBN01A2K0Q length=278 xy=0322_1368 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTGGTCTTGGTGAGCCGTTACCCCACCAAC
TAACTAATACAACGCATGCCCATCCATTACCACCGGAGTTTTCAACCCAAGAAGATGCCT
CCCTGGATGTTATGGGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTAATGGGTAG
GTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCAT
>FLP3FBN01DJY30 length=251 xy=1340_2286 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTGCCAAC
TAGCTAATCAGACGCGGGCCCATCTCATACCACCGGAGTTTTTGCCGCTGCACCATGCGG
TGCTGTGGCCTTATGCGGTATTAGCAGCCATTTCTGGCTGTTATCCCCCTGTATGAGGCA
GGTTGCCCACG
>FLP3FBN01EB63O length=273 xy=1662_0098 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCCCGCCAAC
TACCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACCGGACCATGCGG
TCCTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTGTCCCCCTGTACAGGGCA
GGTTACCCACGCGTTACTCACCCGTCCGCCACT
>FLP3FBN01DUWHJ length=279 xy=1465_0101 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCGGAATCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGCA
GGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01ESJ5R length=277 xy=1848_1661 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGCCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGGGCCGTTACCCCGCCAACA
ACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGAA
GAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAGG
TTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01BJH6S length=269 xy=0515_0322 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCGACCTCTCAGTCCGGCTACCGATCGTCGGCTTGGTGAGCCGTTACCTCACCAAC
TACCTAATCGGACGCGAGCCCATCTCCGAGCGATATAATCTTTGATACCAAAGGGATGTC
CCTCCAGTATGTTATGCGGTATTAGCGACCGTTTCCAGCCGTTATTCCCCTCTCGAAGGT
AGGTTGCTCACGTGTTACTCACCCGTCCG
>FLP3FBN01BF988 length=259 xy=0478_1626 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGCGAGAATGCTC
ACTCATGATACTATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCA
GGTTGGATACGCGTTACTC
>FLP3FBN01DRASA length=264 xy=1423_4040 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGG
ACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCA
GGTTGGATACGTGTTACTCACCCG
>FLP3FBN01EMWGJ length=264 xy=1783_3905 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGG
ACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTATCCAAGAGTATAAGGCA
GGTTGGATACGCGTTACTCACCCG
>FLP3FBN01BMERW length=277 xy=0548_0698 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGG
CAGGTTGCATACGCGTTACTCACCCGTGCGCCGGTCG
>FLP3FBN01A724Q length=271 xy=0385_0072 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAAC
TACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGG
TTCCGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCA
GGTTGCCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01AKXJC length=255 xy=0121_1078 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGCCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCTTTTATCTCACCAAC
TAGCTAATCAGACGCGGATCCATCCCATACCACCGGAGTTTTTCACACAGGGCCATGCAG
CCTCGTGCGCTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCGGTATGGGGCA
GGTTATCCACGCGTT
>FLP3FBN01EC04T length=269 xy=1671_2155 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGCTGCCCCGCCAAC
AAGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAGGGAAAAGCCATGCGG
CTTCCCCCGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGCCAG
GTTGCCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01AX7NV length=264 xy=0272_2233 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGG
ACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCA
GGTTGGATACGTGTTACTCACCCG
>FLP3FBN01CS9UE length=244 xy=1036_1668 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGTCGGCTACCGATCGTCGCCTTGGTGGGCTGTTACCCCTCCAAC
TAGCTAATCGGACGCGGATCCATCTCTCACCGATTTCTCTTTCCCTCTCTCCTCATGCGA
CAAAAGAGGCTCATGCGGTATTAGCAACCGTTTCCAGCTGTTATCCCCCTGTGAAAGGCA
GGTT
>FLP3FBN01A38B0 length=262 xy=0341_0414 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGGCCACCCTCTCAGGCCGGCTACCCATCGTCGCCTTGGTAGGCCATTACCCTACCAAC
TAGCTAATGGGACGCGAGTCCATCTTTCAGCGTCAGGTCTCCCCAACTTTTCCTATATAA
ACATGCACTTATATAACTCATCCGGCATTAGCTCACCTTTCGGTAAGTTGTTCCAGTCTA
AAAGGCAGGTCACTCACGTGTT
>FLP3FBN01CB1XF length=264 xy=0840_1073 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGG
ACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCA
GGTTGGATACGTGTTACTCACCCG
>FLP3FBN01ARU00 length=258 xy=0200_0834 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGTTTGGTGGGCCGTTACCCCGCCAAC
TGCCTAATCAGACGCGGGGCCATCTCATACCACCGGAGTTTTTCACACCGTGTCATGCGA
CACTGTGCGCTTATGCGGCATTAGCAGTCATTTCTAACTGTTATTCCCCTGTATGAGGCA
GGTTCCCCACGCGTTACT
>FLP3FBN01A24G3 length=258 xy=0328_2005 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTC
AAGCTAATGTCACGCGAGCCTATCCTCATCCGACGGATCTTTAGATGGAACCAGATGCCT
GATTCCATCGCCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCA
AGTTGCTCACGCGTTACG
>FLP3FBN01AYGTR length=247 xy=0275_1821 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTATTTCAGTCCCAATGTGG
CCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
TAGCTAATCAGACGCGAGGCCATCTTTCAGCGATAAATCTTTGACATAAATGCCATGCGA
CACCTATGTGTTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGAAAGGCA
GGTTCCT
>FLP3FBN01DPZMM length=253 xy=1409_0268 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTCTGG
ATGATCATCCTCTCAAACCATCTAACGATCGTCGACTTGGTGAGCCTTTACCTCACCAAC
TATCTAATCGTACGCAGGCCATTCCTAAAGCGCATAAATGCTTTAATCCGAAGATCATAT
GCGGTATTAGCCACTCTTTCGAGTAGTTATCCCTCACTTTAGGGTATGTTCCCACGCGTT
ACTCAGCCGTCCG
>FLP3FBN01AJ7M7 length=271 xy=0113_0289 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCCCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAAC
TACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGG
TTCTGGGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCA
GGTTGCCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01DV3TK length=264 xy=1478_3014 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGGTCCATCATATACCACCGGAGTTTTTCACACAGAAACATGCGT
CCCCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATAAGGCA
GATTACCCACGTGTTACTCACCCG
>FLP3FBN01AL42P length=260 xy=0135_0159 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGTCGGTTTGGTGGGCCGTTACCCCGCCAAC
TGCCTAATGGAACGCATGCCTATCTATCAGCGATGAATCTTTAGCAAATATCCCCATGCG
GGGCCCCTGCTTCATGCGGTATTAGTCCGACTTTCGCCGGGTTATCCCCCTCTGATAGGC
AAGTTGCATACGCGTTACTC
>FLP3FBN01BYPDD length=264 xy=0688_0863 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGG
ACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCA
GGTTGGATACGTGTTACTCACCCG
>FLP3FBN01CTMCP length=264 xy=1040_1495 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGGTCCATCATATACCACCGGAGTTTTTCACACCGGGGCATGCGC
CCCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATATGGCA
GGTTACCCACGCGTTACTCACCCG
>FLP3FBN01BAJ58 length=269 xy=0413_0782 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCGACCTCTCAGTCCGGCTACCGATCGTCGGCTTGGTGAGCCGTTACCTCACCAAC
TACCTAATCGGACGCGAGCCCATCTTCGAGCGATAAAATCTTTGATACCAGCAGGATGTC
CTCCCGGTATGTTATGCGGTATTAGCGACCGTTTCCAGCCGTTATTCCCCTCTCGAAGGC
AGGTTGCTCACGTGTTACTCACCCGTCCG
>FLP3FBN01A4K2C length=277 xy=0345_0530 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>FLP3FBN01DRBCR length=278 xy=1424_0681 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTTGCCTTGGTGGGCCGTTACCCCGCCAAC
AAACTAATGGAACGCATCCCCATCGATTATCGAAATTCTTTAATAACAAGAAGATGCCTT
CTCGTTATGCTATCCAGTATTAATCTTTCTTTCGAAAGGCTATCCCGGAATAATCGGCAG
GTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCAT
>FLP3FBN01BWH0G length=259 xy=0663_0414 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGG
ACTCATGGTGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCA
GGTTGGATACGTGTTACTC
>FLP3FBN01D0JVY length=274 xy=1529_1564 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCCGCCAACT
ACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCGCTGTTACCATGCGG
TACTGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCA
CGGTTGCCCACGCGTTACTCACCCGTCCGCCGCT
>FLP3FBN01EDR9T length=264 xy=1680_0463 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGGACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGG
ACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCA
GGTTGGATACGTGTTACTCACCCG
>FLP3FBN01CDTMI length=264 xy=0860_1704 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGG
ACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCA
GGTTGGATACGTGTTACTCACCCG
>FLP3FBN01AFERM length=271 xy=0058_1520 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
AACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGA
AGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAG
GTTGGATACGTGTTACTCACCCGTGCGCCGG
>FLP3FBN01BFQFB length=263 xy=0472_0501 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
CAGCTAATACAACGCATGCCCATCATCAACCACCGGAGTTTTCAACCCCATGCGATGCCG
CATGAGATATTATGGGGTATTAGTACCAATTTCTCAGTGTTATCCCCCTGTTGATGGTAG
GTTGCATACGCGTTACGCACCCG
>FLP3FBN01D3JOZ length=270 xy=1563_2017 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAACC
AGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCTC
TCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGCTACAGGTAGG
TTGCATACGCGTTACGCACCCGTGCGCCGG
>FLP3FBN01DVIVE length=272 xy=1472_0440 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
CAGCTAATGCGCCGCATGCCCATCCATAGCCGCCTCAACTTTCCTCCCCAAGGGATGCCC
CTCAGGGAGTGCACGCGGATTTAGACCGGATTTCTCCGGATTATTCCCCTGACAAGGGTA
GGTTGCATACGTGTTACTCACCCGTGCGCCGG
>FLP3FBN01EQVOU length=254 xy=1829_1116 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGAGCCATTACCTCACCAAC
TAACTAATACGCCGCGGGATCATCTAAGAGCGACAGCAAAGCCGTCTTTCATTCATGAAC
CATGCGGTTCGCGAATCTATGCGGTATTAGCATCCGTTTCCGAATGTTATCCCCCACTCC
TAGGCAGATTCCCC
>FLP3FBN01DH4LG length=277 xy=1319_2098 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGGCCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGGGCCGTTACCCCGCCAAC
AACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGA
AGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAG
GTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCA
>FLP3FBN01EEU42 length=268 xy=1692_1684 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAG
GACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01AE4G6 length=264 xy=0055_0472 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGG
ACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCA
GGTTGGATACGTGTTACTCACCCG
>FLP3FBN01A559N length=253 xy=0363_0937 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAGTGTGG
GGGACCTTCCTCTCAGAACCCTTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTC
AAGCTAATGTCACGCGAGCCTATCCTCATCCGACGGATCTTTAGATGGAACCAGATGCCT
GATTCCATCGCCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCA
AGTTGCTCACGCG
>FLP3FBN01BYAA8 length=268 xy=0683_1826 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCTCACCAAC
CAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTCACACCGAACCATGCGG
TTCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01CSQSY length=244 xy=1030_1568 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
TAGCTAATCAGACGCGAGTCCATCTCAGAGCGATAAATCTTTGGCAGTCAGAGCCATGCG
ACCCAACTGCATTATGCGGTATTAGCAGCTGTTTCCAGCTGTTATTCCCCACTCCAAGGC
TAGG
>FLP3FBN01DTXOJ length=269 xy=1454_0049 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTCCCAATGTGG
CCGGTCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCGCCAAC
CAGCTAATCAGACGCGGGCCCCTCCCATACCGCCGGAGCTTTCCCCACAAAGGCATGCGC
CTCCCTGGTTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTATGGGGCAG
GTTGCCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01EJD3A length=257 xy=1743_3972 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGCTGCCCCGCCAAC
CAGCTAATCAGACGCGGGCCCATCCCACACCGCCCGAAGGCTTTTCACACCGCTCCATGC
GAAGCTGTGCGCTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTGTGGGG
CAGGTTGCCCACGCGTT
>FLP3FBN01CIW8K length=240 xy=0918_2098 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGACCGTCTCTCAGTTCCAATGTGG
CCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGACTTGGTGAGCCGTTACCTCACCAAC
TATCTAATCAGACATGAGCCCATCTTTCAGCGGATTGCTCCTTTGATAACAGGATCATGC
GATCCCGTCATTTCATGCGGTATTAGCACACCTTTCGATGTGTTATCCCCCTCTGAAAGG
>FLP3FBN01BYVGX length=253 xy=0690_0575 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGCTGCCCCGCCAAC
AAGCTAATCAGACGCGGGCCCATCGCATACCACCGGAGTTTTTCACACCAAGCCATGCGG
CTCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGGTATGCGGCA
GGTTGCCCACGCG
>FLP3FBN01BKL5B length=254 xy=0527_2957 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTC
AAGCTAATGTCACGCGAGCCTATCCTCATCCGACGAATCTTTAGATGGATTCAGATGCCT
GATTCCATCACCATGGGGCGATTAGACGCCGTTTCCTAGCGATTATTCCCCTCGATGAGG
GCAAGTTGCTCACG
>FLP3FBN01DX9GS length=286 xy=1503_1242 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGCCCATCCGTGGCCGGGATTGCTCCCTTTGGCGGCCCGGGGAT
GCCCCAAGGCCGCGTTACGCGGTATTAGACGGGGTTTCCCCCGCTTATCCCCCTGCCACG
GGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCGG
>FLP3FBN01BVHTK length=278 xy=0651_2662 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGAGCCGTTACCTCACCAAC
AACCTAATGGAACGCATCCCCATCCTTTACCGGAATCCTTTAATAATGAAACCATGCGGA
ATCATTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTAAAGGGCAG
GTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCAG
>FLP3FBN01A9PL2 length=276 xy=0403_2136 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
TAGCTAATGTCACGCATGCCCATCCGTCTCCGGAGTCGCCTCCTTTGGCCGAAAGAAGAT
GCCTCCCCTCGGCCACATGCGGTATTAGGCCGGATTTCTCCGGGTTATCCCGCTGAAACG
GGCAGGTTGCATACGCGTTACTCACCCGTGCGCCGG
>FLP3FBN01C3PFP length=248 xy=1155_1011 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
TAGCTAATCAGACGCGAGTCCATCTCAGAGCGATAAATCTTTGGCAGTCAGAGCCATGCG
ACTCAACTGCATCATGCGGTATTAGCAGCTGTTTCCAACTGTTATTCCCCACTCCAAGGC
AGGTTACT
>FLP3FBN01A17GS length=254 xy=0318_0186 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
CAGCTAATCAGACGCGAGTCCATCTCAGAGCGATAAATCTTTGATATCCAGAGCCATGCG
ACCCAGATATATTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTATGGGGC
AGGTTGCCCACGCG
>FLP3FBN01DM0XL length=256 xy=1375_1255 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGCCACCCTCTCAGGTCGGCTACCCGTCATCGCCTTGGTGAGCCGTTACCTCACCAAC
TAACTAATAGGCCGCGAGCCCATCCCCAAGCGCATTGCTGCTTTAATCTTTCGATACTAT
GCGGTATTAATCCCAGTTTCCCAGGGCTATCCCGCTCTCGGGGGCAGGTTACTCACGTGT
TACTCACCCGTGCGCC
>FLP3FBN01DFQEI length=276 xy=1292_0984 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGTCGGTTTGGTGGGCCATTACCCCGCCAAC
TGCCTAATGGAACGCATGCCTATCTATCAGCGATGAATCTTTAACAAATATTCCCATGCG
GGACCCCTGTTTTATGGGGTATTAATCCGACTTTCGCCGGGCTATCCCCTCTGATAGGTA
AGTTGCATACGCGTTACTCACCCGTGCGCCGGTCCG
>FLP3FBN01DN5MU length=268 xy=1388_0756 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCGACCTCTCAGTCCGGCTACCCATCGTTGCCTTGGTGGGCCGTTACCTCACCAAC
TAGCTAATGGGTCGCGAGCCCATCCTATACCGATAAATCTTTTACCCCTGTATCATGCGA
TACTGTGGTCTTATACCGTATTAGCACAAATTTCTCTGTGTTATCCCGTTGTATAGGGCA
GGTTGCTCACGTGTTACTCACCCGTTCG
>FLP3FBN01B1L15 length=247 xy=0721_1371 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCTTTACCCCACCAAC
TAGCTAATCAGACGCGGGTCGCTCTATCAGCGATAGCCTCTCTCGAGTCCGCCACCTTTC
CTTCTGCCATCATGCGATGACTGAACCTTATGCGGTATTAGCACTGATTTCTCATTGTTA
TTCCCCT
>FLP3FBN01AW01I length=271 xy=0259_0244 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
AAGCTAATCAGACGCGGAACCATCGTATACCACCGGAGTTTTTCACACAGGGCCATGCGG
CCCCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATAGGGCA
GGTTCCCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01BI38I length=276 xy=0510_2720 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCGGCACCGGGATGCC
CCGACGCCGCGTCACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTGGCGGGC
AGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>FLP3FBN01DSC1C length=249 xy=1436_0366 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGGTCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
CAGCTAATGCGCCATAAGTCCATCCTCTACCAGTGCCTTGCAGCACTTTTAATACGGTCA
CCATGCAGTGTCCCTACCTATGCGGTCTTAGCTGCCGTTTCCAGCAGTTATCCCCCTGTA
AAGGCCAGG
>FLP3FBN01AIELL length=264 xy=0092_2007 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGG
ACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCA
GGTTGGATACGTGTTACTCACCCG
>FLP3FBN01D1T0J length=264 xy=1543_4001 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGG
ACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCA
GGTTGGATACGTGTTACTCACCCG
>FLP3FBN01EOHM5 length=256 xy=1802_0191 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCTCGCCAA
CCAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCTCACAGGATGTTTCACAGAAAG
AAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTG
CGCGGGCAGGTTCCCT
>FLP3FBN01BMLLI length=276 xy=0550_1348 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGTCGGTTTGGTGGGCCGTTACCCCGCCAAC
TGCCTAATGGAACGCATGCCTATCTATCAGCGATGAATCTTTAACAAATATTCCCATGCG
GGACCCCTGTTTTATGGGGTATTAATCCGACTTTCGCCGGGCTATCCCCTCCTGATAGGT
AAGTTGCATACGCGTTACTCACCCGTGCGCCGGTCG
>FLP3FBN01D74QJ length=265 xy=1615_2921 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGG
ACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCA
GGTTGGATACGTGTTACTCACCCGG
>FLP3FBN01EAAV0 length=274 xy=1640_1806 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAAC
CAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCT
CTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAG
GTTGCATACGCGTTACGCACCCGTGCGCCGGTCG
>FLP3FBN01BC3GH length=271 xy=0442_0303 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTACCTATCATTGCCTTGGTGGGCCGTTACCCCCCAACT
AGCTAATAGGACGCATGCCCATCTGATACCTCGAATGATTTAATTATTAAAAGATGCCTT
CAAATAATATTATGGGGTGTTAATCCACGTTTCCATGGGCTATCCCCCTGTATCAGCCAG
GTTGCATACGCGTTACTCACCCGTGCGCCGG
>FLP3FBN01D9A6M length=264 xy=1629_0588 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGG
ACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCA
GGTTGGATACGTGTTACTCACCCG
>FLP3FBN01A8V4Q length=262 xy=0394_0792 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAAC
TAGCTAATGTCACGCATGCCCATCCGTTGCCGGAATCGCTCCCTTTGGCCGCAGGGCCAT
GCAGCCCCGCGGCATTACGCGGTATTAGACGGGATTTCTCCCGATTATCCCCCTGCAACG
GGCAGGTCGCATACGCGTTACT
>FLP3FBN01DCU7E length=277 xy=1259_2408 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGGCCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGGGCCGTTACCCCGCCAAC
AACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGA
AGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAG
GTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCA
>FLP3FBN01AM688 length=264 xy=0147_0490 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGG
ACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCA
GGTTGGATACGTGTTACTCACCCG
>FLP3FBN01DMRPA length=258 xy=1372_1580 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGATCAACCTCTCAGTCCGGCTACCGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
TAGCTAATCAGACGCGGGTCCATCTCATACCACCGGAGTTTTTCACACCAGGACATGCAT
CCCTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATGAGGCA
GGTTACCCACGCGTTACT
>FLP3FBN01BAZV6 length=287 xy=0418_0676 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTAC
TGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCAAGGATGCCC
CCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGAGGGCAGG
TTGCATACGCGTTACTCACCCGTGCGCCGGTCGCCGGCGGAGCAAAG
>FLP3FBN01DEBKQ length=271 xy=1276_0648 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGCAAGCCGTTACCTTACCAAC
CAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCT
CTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAG
GTTGCATACGCGTTACGCACCCGTGCGCCGG
>FLP3FBN01BAWOL length=279 xy=0417_0611 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGGCCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCCCGTACCACCGGAGTTTTCAAGAAAGGAACATGCGT
CCCCTTCTGTTATGCGGTATTAGCACCTGTTTCCAGGTGTTATCCCCCGGTACGGGGCAG
GTTCCCCACGCGTTACTCACCCGTTCGCCACTCGGGCAC
>FLP3FBN01D7AVV length=281 xy=1606_1097 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAGTGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01B5T3T length=273 xy=0769_1815 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGGTG
GGGGACCTTCCTCTCAGAACCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCGCCGACT
AGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCGGCACCGGGATGCCC
CGACGCCGCGTCACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCTGTGGCGGGCAG
GTTGCATACGTGTTACTCACCCGTGCGCCGTCG
>FLP3FBN01B02WO length=277 xy=0715_1126 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>FLP3FBN01B2DRT length=266 xy=0730_0423 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCACTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGCCGCATGCCCATCCTTTACCGGATCGCTCCTTTGACATACCGATCATGC
GGTCGGTATGTATTATGAGGTATTAGACGGAATTTCTCCCGATTATCCCTCTGTAAAGGG
CAGGTCGCATACGTGTTACTCACCCG
>FLP3FBN01A8KH9 length=264 xy=0390_2111 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGG
ACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCA
GGTTGGATACGTGTTACTCACCCG
>FLP3FBN01AU6ML length=264 xy=0238_0187 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGG
ACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCA
GGTTGGATACGTGTTACTCACCCG
>FLP3FBN01AJ4R9 length=282 xy=0112_0679 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGCCGTGTCTCAGTACCAATGTGGG
GGGTTAACCTCTCAGTCCCCCTATGTATCGTTGCCTTGGTGAGCCGTTACCTCACCAACT
AGCTAATACAACGCATGCCCATCTTTAACCACCGGAGTTTTTAACCCAAGAAGATGCCTT
CTCGAATGTTATGGGGAATTAGTACCAATTTCTCAGTGTTATGCCCCTGTTAAAGGTAGG
TTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCACCAAAG
>FLP3FBN01B32LN length=255 xy=0749_1433 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTCTCCACGCGTT
>FLP3FBN01COYKA length=262 xy=0987_1128 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGGCCACCCTCTCAGGCCGGCTACCCATCGTCGCCTTGGTGGGCCATTACCCTACCAAC
TAGCTAATGGGACGCGAGTCCATCTTTCAGCGTCAGGTCTCCCCAACTTTTCCTATATAA
ACATGCACTTATATAACTCATCCGGCATTAGCTCACCTTTCGGTAAGTTGTTCCAGTCTA
AAAGGCAGGTCACTCACGTGTT
>FLP3FBN01C4BE9 length=264 xy=1162_0835 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAAC
TGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGG
AACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCA
GGTTATCCACGCGTTACTCACCCG
>FLP3FBN01BU1CU length=271 xy=0646_1804 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
CCGTTCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCCACCAAC
TAACTAATCAGACGCAGGCCCATCCCTTACCGATAAATCTTTCACATCCCTGATATGCAT
CAGACTTGCATTATGAGGTTTTAATCCCGGTTTCCCGAGGCTATCCCTCTGTAAGGGGCA
GGTTGCCTACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01AYTJU length=277 xy=0279_1928 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGGCCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGGGCCGTTACCCCGCCAAC
AACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGA
AGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAG
GTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCA
>FLP3FBN01DXJGO length=264 xy=1495_0310 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGG
ACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCA
GGTTGGATACGTGTTACTCACCCG
>FLP3FBN01BOOHE length=286 xy=0574_0096 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAGTGTGG
CCGATCACCCTCTCAGGTCGGCTACGTATCGTTGCCTTGGTAAGCCGTTACCTTACCAAC
TAGCTAATACGGCGCGGGTCCATCTATAAGTGACAGCCGAAACCGTCTTTCAACATTGAA
CCATGCGGTTCAATATATTATCCGGTATTAGCCCCGGTTTCCCGGAGTTATCCCAGTCTT
ATAGGTAGGTTACCCACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01BJL13 length=247 xy=0516_1241 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGACCGTGTTCCAGTTCCAGTGTGA
CTGATCATCCTCTCAGACCAGTTAACCATCGTCGCCTTGGTAGGCCTTTACCCCACCAAC
CAGCTAATGGTACGCGGACTCATCAAAAAGCAACAGCATATGCAGAGGCCATCTTTCCCA
CATAAGATAGATATGCAGCGTATCCGGTATTAGCAGCCGTTTCCAGCTGTTATCCCAAAC
TTCTTGG
>FLP3FBN01CTRWB length=258 xy=1042_0489 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCCTTACCCCACCAAC
TAGCTAATCAGACGCGGGTCCATCTCATACCACCGGAGTTTTTCACACTGGACCATGCAG
TCCCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATGGGGCA
GGTTGCCCACGCGTTACT
>FLP3FBN01BJCG1 length=259 xy=0513_1107 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGG
ACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCA
GGTTGGATACGTGTTACTC
>FLP3FBN01CORWL length=268 xy=0985_0691 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGGTCCATCCCATACCACCGGAGTTTTTCACACAGGAGCATGCGC
TCCCGTGCGCTTATGCGGTATTAGCACCTGTTTCCAGGTGGTATCCCCCGGTATGGGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01C68BF length=274 xy=1195_1609 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGGCCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGAGCCGTTACCCCGCCAAC
AACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGA
AGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAG
GTTGGATACGTGTTACTCACCCGTGCGCCGGTCG
>FLP3FBN01B449W length=283 xy=0761_2402 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCCCGTACCACCGGAGTTTTCAAGAAAGAAACATGCGT
CCCCTTCTGTTATGCGGTATTAGCACCTGTTTCCAGGTGTTATCCCCCGGTACGGGGCAG
GTTCCCCACGCGTTACTCACCCGTCCGCCACTAACTCATACAT
>FLP3FBN01EZ4SI length=269 xy=1934_2736 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGGTCCATCCCATACCACCGGAGTTTTTCACACAGGAGCATGCGC
TTCCGTGCGCTTATGCGGTATTAGCACCTGTTTCCAGGTGGTATCCCCCCGGTGTGGGGC
AGGTTGCCCACGCGTTACTCACCCGTCCG
>FLP3FBN01DD5NY length=276 xy=1274_1180 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTTGCCTTGGTGAGCCGTTACCTCACCAAC
TAGCTAATACAACGCATGCCCATCTTTAACCACCGGAGTTTTTAACCCAAGAAGATGCCT
TCTCGAATGTTATGGGGAATTAGTACCAATTTCTCAGTGTTATGCCCCTGTTAAAGGTAG
GTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCC
>FLP3FBN01BHKSP length=276 xy=0493_0503 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAAC
CAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGTACCATGCGG
TACTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCA
GGTTACCCACGCGTTACTCACCCGTCCGCCACTAAG
>FLP3FBN01DHYS5 length=247 xy=1317_2791 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
TAGCTAATCAGACGCGAGGCCATCTTTCAGCGATAAATCTTTGACATAAATGCCATGCGA
CACCTATGTGTTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGAAAGGCA
GGTTCCT
>FLP3FBN01D837F length=277 xy=1626_3833 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGGCCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGGGCCGTTACCCCGCCAAC
AACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGA
AGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAG
GTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCA
>FLP3FBN01DC9HR length=253 xy=1264_0445 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTC
AAGCTAATGTCACGCGAGCCTATCCTCATCCGACGGATCTTTAGATGGAACCAGATGCCT
GATTCCATCGCCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCA
AGTTGCTCACGCG
>FLP3FBN01DLDHU length=272 xy=1356_2048 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTACCTATCATTGCCTTGGTGGGCCGTTACCCCCCAACT
AGCTAATAGGACGCATGCCCATCTGATACCTCGAACGATTTAATTATTATAAGATGCCTT
ACAATAATATTATGGGGTGTTAATCCACGTTTCCATGGGCTATCCCCCTGTATCAGCCAG
GTTTGCATACGCGTTACTCACCCGTGCGCCGG
>FLP3FBN01D6Q98 length=264 xy=1600_0270 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGG
ACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCA
GGTTGGATACGTGTTACTCACCCG
>FLP3FBN01CNKON length=276 xy=0971_2021 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAACC
AGCCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCT
CTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTTACAGGTA
GGTTGCCATACGCGTTACGCACCCGTGCGCCGGTCG
>FLP3FBN01BCWX5 length=283 xy=0440_0059 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTAC
TGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCAAGGATGCCC
CCAAGGGATATACGCGGGATTAGCCCTCCCTTTCGGAAGGTTCGTCCCCCTGTGGAGGGC
AGGTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGGCAG
>FLP3FBN01EA91G length=276 xy=1651_2306 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAAC
CAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGTACCATGCGG
TACTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCA
GGTTACCCACGCGTTACTCACCCGTCCGCCACTAAG
>FLP3FBN01B06QS length=275 xy=0716_2002 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
CAGCTAATACAACGCATGCCCATCTTCCACCACAAAAAGCTTTCAACCCAGAGAGATGCC
TCTCCGAATTATATGGGGTATTAGTACCAATTTCTCAGTGTTATCCCCCTGTGAAAGGTA
GGTTGCATACGCGTTACGCACCCGTCCGCCGGTCG
>FLP3FBN01D0SFK length=247 xy=1532_0350 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
TAGCTAATCAGACGCGAGGCCATCTTTCAGCGATAAATCTTTGACATAAATGCCATGCGA
CACCTATGTGTTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGAAAGGCA
GGTTCCT
>FLP3FBN01DVTEG length=270 xy=1475_1798 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCAACCTCTCAGTCCGGCTATGGATCGTTGACTTGGTAGGCCGTTACCCCACCAAC
TATCTAATCCAACGCGAGCCCATCCTTCGGCACCTCAGTTTTGGTATTTCTATCATGCGG
TAAAAATACGTTATGCGGTATTACTGTCCGTTTCCAGACACTATCCCCCTCCGAAGGGCA
GGTTGCTCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01CECHO length=255 xy=0866_1578 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGATCGACCTCTCAGTCCGGCTACCCGTCGTCGGCTAGGTGAGCTGCTACCTCACCTAC
TACCTGATGGGCCGCGACCCCATCCCAGACCGCAAGAGCTTTCATCCCTCCGCCATGCGG
TGGAAGGATAGTATCCGGTATTAGCTGCCGTTTCCGGCAGTTATCCCGATGTCTGGGGCA
GGTTGGTCACGTGTT
>FLP3FBN01ES2BT length=271 xy=1854_0631 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGTCGGTTTGGTGGGCCGTTACCCCGCCAAC
TGCCTAATGGAACGCATGCCTATCTATCAGCGATGAATCTTTAGCAAATATCCCCATGCG
GGGCCCCTGCTTCATGCGGTATTAGTCCGACTTTCGCCGGGTTAGTCCCCTCCTGATAGG
TAAGTTGCATACGCGTTACTCACCCGTCGCG
>FLP3FBN01BJO3F length=280 xy=0517_1081 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACCCCGCCAAC
TAGCTAATGCGCCGCATGGCCATCCGTAGCCGGTGTTACCCTTTAAACCCCAAGAGATGC
CTCTCGGAGTTATTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGG
CAGGTTTCATACGTGTTACTCACCCGTGCGCCGGTCGCCA
>FLP3FBN01AMOUT length=263 xy=0141_1219 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
CAGCTAATACAACGCATGCCCATCATCAACCACCGGAGTTTTCAACCCCATGCGATGCCG
CATGAGATATTATGGGGTATTAGTACCAATTTCTCAGTGTTATCCCCCTGTTGATGGTAG
GTTGCATACGCGTTACGCACCCG
>FLP3FBN01CNT1Q length=244 xy=0974_1868 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
CAGCTAATCAGACGCGAGTCCATCTCAGAGCGATAAATCTTTGATATCCAGAGCCATGCG
ACCCAGATATATTATGCGGTATTAGCAGCTGTTTCCAGCTGTTATTCCCCATCCAAGGCA
GGTT
>FLP3FBN01DR739 length=242 xy=1434_2183 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGACCGTCTCTCAGTTCCAATGTGG
CCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGACTTGGTGAGCCGTTACCTCACCAAC
TATCTAATCAGACATGAGCCCATCTTTCAGCGGATTGCTCCTTTGATAACAGGATCATGC
GATCCCGTTCATTTCATTGCGGTATTAGCACACCTTTCGATGTGTTATCCCCCTCTGAAA
GG
>FLP3FBN01BBDL9 length=258 xy=0422_2079 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCTTTTACTCCACCAAC
TAGCTAATCAGACGCGGAACCATCGTATACCACCTCAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATAAGGCA
GGTTACCCACGCGTTACT
>FLP3FBN01CLP3T length=258 xy=0950_1751 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTATGGACCGTGTCTCAGTTCCATTGTGG
CCGTTCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTAGGCCTTTACCCCACCAAC
TAGCTAATGCGCCGCAGGCTCATCCATCAGTGATGCCAGGAGCATCTTTAAACTTTCGTC
CTATCCGGTATTAGCGATCGTTTCCAATCGTTGTCCCCGTCTGATGGGCAGATTACCTAC
GTGTTACTCACCCGTTCG
>FLP3FBN01C1O95 length=268 xy=1132_1707 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTCACACCGAACCATGCGG
TTCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGTAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01DHLQN length=251 xy=1313_2237 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTTACCCTCTCAGGCCGGCTACGCATCATCGCCTTGGTGGGCCGTTACCTCACCAAC
TAGCTAATGCGCCGCAGGTCCATCCATGTTCACGCCTTGATGGGCGCTTTAATATACTGA
GCATGCGCTCTGTATACCTATCCGGTTTTAGCTACCGTTTCCAGCAGTTATCCCGGACAC
ATGGGCAGGTT
>FLP3FBN01BA44D length=255 xy=0419_3355 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGTTCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTAGGCCTTTACCCCACCAAC
TAGCTAATGCGCCGCAGGCTCATCCATCAGTGATGCCAGGAGCATCTTTAAAACTTTCGT
CCTATCCGGTATTAGCGATCGTTTCCAATCGTTGTCCCCGTCTGATGGGCAGATCACCTA
CGTGTTACTCACCCG
>FLP3FBN01EHG6O length=245 xy=1722_0686 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGG
ACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCA
GGTTG
>FLP3FBN01EE1C3 length=270 xy=1694_1557 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACCGGGCCATGCGG
CCCTGTGCGCTTATGCGGTATTAGCACCTGTTTCCAGGTGTTATCCCCCTGTGTAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01AER1M length=247 xy=0051_0744 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGATCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
TAGCTAATGCGCCGCGGGCCCATCTCATAGCGGATTGCTCCTTTAATGCTACTTCATGCG
AAGCTACAATCTTATGCGGTATTAATCTCCCTTTCGGAAGGCTATTCCCCTCTATGAGGC
AGGTTCG
>FLP3FBN01A1RXA length=278 xy=0313_0524 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTAC
TGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCAAGGATGCCC
CCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAG
GTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGG
>FLP3FBN01B5Z5E length=254 xy=0771_1456 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGTTCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTAGGCCTTTACCCCACCAAC
TAGCTAATGCGCCGCAGGCTCATCCATCAGTGATGCCAGGAGCATCTTTAAACTTTCGTC
CTATCCGGTATTAGCGATCGTTTCCAATCGTTGTCCCCGTCTGATGGGCAGATCACCTAC
GTGTTACTCACCCG
>FLP3FBN01EB7G2 length=242 xy=1662_0580 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGACCGTGTTCCAGTTCCAGTGTGA
CTGATCATCCTCTCAGACCAGTTAACCATCGTCGCCTTGGTAGGCCTTTACCCCACCAAC
CAGCTAATGGTACGCGGACTCATCAAAAAGCAACAGCATATGCAGAGGCCATCTTTCCCA
CATAAGATAGATATGCAGCGTATTCGGTATTAGCAGCCGTTTCCAGCTGTTATCCCAAAC
TT
>FLP3FBN01AYGA0 length=280 xy=0275_1146 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTAC
TGCCTAATGCGCCGCATGCCCATCCCTGTCCGGCCGAAGCCTTTCCTGCCTCCGGGATGC
CCCGGTGGCATGTACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGC
AGGTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGG
>FLP3FBN01AL8JX length=268 xy=0136_0571 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
CAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTCACACCGGGGCATGCGC
CCCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATATGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01B7E8J length=276 xy=0787_2129 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAAC
CAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGTACCATGCGG
TACTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCA
GGTTACCCACGCGTTACTCACCCGTCCGCCACTAAG
>FLP3FBN01D6BHF length=251 xy=1595_0273 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTC
AAGCTAATGTCACGCGAGCCTATCCTCATCCGACGAATCTTTAGATGGATTCAGATGCCT
GATTCCATCACCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCA
AGTTGCTCACG
>FLP3FBN01BLTKB length=271 xy=0541_1881 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTACCTATCATTGCCTTGGTGGGCCGTTACCCCCCAACT
AGCTAATAGGACGCATGCCCATCTGATACCTCGAATGATTTAATTATTAAAAGATGCCTT
CAAATAATATTATGGGGTGTTAATCCACGTTTCCATGGGCTATCCCCCTGTATCAGCCAG
GTTGCATACGCGTTACTCACCCGTGCGCCGG
>FLP3FBN01EK2JX length=285 xy=1763_0411 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTAC
TGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCGGGGATGCCC
CCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAG
GTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGGCAGAGAG
>FLP3FBN01EF15V length=260 xy=1706_0097 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGTCGGTTTGGTGGGCCGTTACCCCGCCAAC
TGCCTAATGGAACGCATGCCTATCTATCAGCGATGAATCTTTAGCAAATATCCCCATGCG
GGGCCCCTGCTTCATGCGGTATTAGTCCGACTTTCGCCGGGTTATCCCCCTCTGATAGGC
AAGTTGCATACGCGTTACTC
>FLP3FBN01BB8KH length=280 xy=0432_1231 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGTTAACCTCTCAGTCCCCCTATGTATCGTGGTCTTGGTGAGCCGTTACCCCACCAACT
AACTAATACAACGCATGCCCATCCATTACCACCGGAGTTTTCAACCCAAGAAGATGCCTT
CTCGAATGTTATGGGGAATTAGTACCAATTTCTCAGTGTTATGCCCCTGTTAAAGGTAGG
TTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCACTAA
>FLP3FBN01AMFGP length=268 xy=0138_1335 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTCATACCACCGGAGTTTTTCACACCGGAACATGTGT
CCCTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATGAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01A6LEJ length=255 xy=0368_0073 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTCTCCACGCGTT
>FLP3FBN01BUCNS length=274 xy=0638_2566 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAAC
CAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCT
CTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAG
GTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>FLP3FBN01ET0T4 length=277 xy=1865_0298 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGGCCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGGGCCGTTACCCCGCCAAC
AACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGA
AGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAG
GTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCA
>FLP3FBN01B0AX7 length=272 xy=0706_1757 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCTTTGGTGGGCCGTTACCCCGCCAAC
TGGCTAATGCGCCGCATGCCCATCCTTTGCCGGAATTGCTTCCTTTGACTCCCAAACCAT
GTGGTAAGGGAGTGTTATGCAGTATTAGTCGGAATTTCTCCCGGTTATCCTCCTGCAAAG
GGCAGGTTGCATACGTGTTACTCACCCGTGCG
>FLP3FBN01BM4QE length=261 xy=0556_1572 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
TAACTAATGCGCCGCATGCCCATCCATGACCGGATCGCTCCTTTGACTCCCGAGAGATGT
CTCCCGGGGGTGTTATGCGGTATTAGACGGAATTTCTCCCGCTTATCCCCCTGTCATGGG
CAAGTTGCATACGTGTTACTC
>FLP3FBN01EPSX1 length=258 xy=1817_0055 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCCACCAAC
CAGCTAATCAGACGCGGGCTCATCTTATACTACCGGAGTTTTTCACACAGAAACATGCGT
CCCCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATAAGGCA
GATTACCCACGTGTTACT
>FLP3FBN01DZ97Y length=280 xy=1526_1324 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGTAGTCTTGGTGGGCCGTTACCCCCCAACA
AACTAATGGAACGCATCCCCATCTCATACCGAATTTCTTTAATATAAAACCCATGCGGGA
AATATATGCTATCGGATATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTATGAGGCAGG
TTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCATCGG
>FLP3FBN01BFDT5 length=277 xy=0468_0571 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGGCCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGGGCCGTTACCCCGCCAAC
AACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGA
AGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAG
GTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCA
>FLP3FBN01BKTRL length=283 xy=0530_0543 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTAGACATCGTCGGCTTGGTGGGCCGTTACCCCGCCAAC
TACCTAATGTCGCGCGTGCCCGTCCCGTACCACCGGAATTTTAAATCGAGAGCCATGCGG
CTCTCGAGTATCATGGGATGTTAGTCCACGTTTCCGCGGGTTATCTCCCGGTACGGGGTT
GGTTGCACACGTGTTACTCACCCGTGCGCCGGTCGCCGGCAGG
>FLP3FBN01C9NT9 length=258 xy=1223_0351 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTC
AAGCTAATGTCACGCGAGCCTATCCTCATCCGACGGATCTTTAGATGGAACCAGATGCCT
GATTCCATCGCCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCA
AGTTGCTCACGCGTTACG
>FLP3FBN01BE65F length=270 xy=0466_0097 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGCTGCCCCGCCAAC
AAGCTAATCAGACGCGGGCCCCTCCCATACCGCCGGAGCTTTCCCCACAAAGGCATGCGC
CTCCCTGGTTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTATGAGGCAG
GTTGCCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01EJ91Q length=252 xy=1754_0332 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTGTCTTGGTGAGCCGTTACCCCACCAACT
AACTAATACAACGCATGCCCATCCATTACCACCGGAGTTTTCAACCCAAGAAGATGCCTC
CCTGGATGTTATGGGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTAATGGGTAGG
TTGCANCGTCGT
>FLP3FBN01AY2QI length=269 xy=0282_1544 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGCTGCCCCGCCAAC
AAGCTAATCAGACGCGGGCCCCTCCCATACCGCCGGAACTTTCCCCAGAAAGGCATGCGC
CTCCCTGGTTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTATGGGGCAG
GTTGCCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01BYYFZ length=272 xy=0691_0333 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGCGAGAATGCTC
ACTCACGATACTATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCA
GGTTGGATACGCGTTACTCACCCGTGCGCCGG
>FLP3FBN01AFUF1 length=283 xy=0063_1359 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTTGCCTTGGTGAGCCGTTACCTCACCAAC
TAGCTAATACAACGCATGCCCATCTTTAACCACCGGAGTTTTTAACCCAAGAAGATGCCT
TCTCGAATGTTATGGGGAATTAGTACCAATTTCTCAGTGTTATGCCCCTGTTAAAGGTAG
GTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCACCAAAG
>FLP3FBN01B7I8E length=283 xy=0788_3212 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTTGCCTTGGTGAGCCGTTACCTCACCAAC
TAGCTAATACAACGCATGCCCATCTTTAACCACCGGAGTTTTTAACCCAAGAAGATGCCT
TCTCGAATGTTATGGGGAATTAGTACCAATTTCTCAGTGTTATGCCCCTGTTAAAGGTAG
GTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCACCAAAG
>FLP3FBN01EIDVP length=244 xy=1732_2099 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAAC
TAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGA
CATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCT
TGGG
>FLP3FBN01DOR0M length=279 xy=1395_1092 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGTTGACTTGGTGGGCCGTTACCCCGCCAAC
TATCTAATGGAACGCATCCCCATCGATAACCGAAATTCTTTAATAGTGAAACCATGCGGA
AATACTATACTATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTTATCGGCAG
GTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCATC
>FLP3FBN01DO8MU length=274 xy=1400_2148 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAAC
CAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCT
CTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAG
GTTGCATACGCGTTACGCACCCGTGCGCCGGTCG
>FLP3FBN01ATR31 length=274 xy=0222_0255 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAAC
CAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCT
CTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAG
GTTGCATACGCGTTACGCACCCGTGCGCCGGTCG
>FLP3FBN01CO4O9 length=274 xy=0989_0891 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAAC
CAGCTAATACAACGCGTGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCT
CTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAG
GTTGCATACGCGTTACGCACCCGTGCGCCGGTCG
>FLP3FBN01CZUXO length=273 xy=1111_1738 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCT
CCCAACGTTCTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCA
CGGTTCCATACGTGTTACTCACCCGTGCGCCGG
>FLP3FBN01A9EX0 length=276 xy=0400_0598 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCGGCACCGGGATGCC
CCGATGCCGCGTCACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTGGCGGGC
AGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>FLP3FBN01BYSJT length=264 xy=0689_0887 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGG
ACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCA
GGTTGGATACGTGTTACTCACCCG
>FLP3FBN01AV5YV length=262 xy=0249_0933 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGGCCACCCTCTCAGGCCGGCTACCCATCGTCGCCTTGGTAGGCCATTACCCTACCAAC
TAGCTAATGGGACGCGAGTCCATCTTTCAGCGTCAGGTCTCCCCAACTTTTCCTATATAA
ACATGCACTTATATAACTCATCCGGCATTAGCTCACCTTTCGGTAAGTTGTTCCAGTCTA
AAAGGCAGGTCACTCACGTGTT
>FLP3FBN01DU3OR length=249 xy=1467_1241 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAAC
TACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGG
TTCTGGGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTACGGGC
AGGTTGCCA
>FLP3FBN01EWW35 length=274 xy=1898_0275 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAAC
CAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCT
CTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAG
GTTGCATACGCGTTACGCACCCGTGCGCCGGTCG
>FLP3FBN01A3LZ0 length=268 xy=0334_0142 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGTCGGTTTGGTGGGCCGTTACCCCGCCAAC
TGCCTAATGGAACGCATGCCTATCTATCAGCGATGAATCTTTAGCAAATATCCCCATGCG
GGGCCCCTGCTTCATGCGGTATTAGTCCGACTTTCGCCGGGTTATCCCCTCTGATAGGTA
AGTTGCATACGCGTTACTCACCCGTGCG
>FLP3FBN01D55K5 length=246 xy=1593_0823 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
AAGCTAATCAGACGCGAGCTCATCTCAGAGCGATAAGTCTTTGACAGCCAACCCCATGCG
GGATAGCTGTATCATGCGGTATTAGCGGCTGTTTCCAGCCGTTATTCCCCACTCCAAGGC
AGATTG
>FLP3FBN01CHNXI length=271 xy=0904_0724 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCAACCTCTCAGTCCGGCTATGGATCGTTGACTTGGTAGGCCGTTACCCCACCAAC
TATCTAATCCAACGCGAGCCCATCCTTCGGCACCTCAGTTTTGGTATTTCTATCATGCGG
TAAAAATACGTTATGCGGTATTACTGTCCGTTTCCAGACACTATCCCCCTCCGAAGGGCA
GGTTCCCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01EETLW length=268 xy=1691_3794 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
CAGCTAATGCGCCGCATGCCCATCCATAGCCGCCTCAACTTTCCTCCCCAAGGGATGCCC
CTCAGGGAGTGTACGCGGATTTAGACCGGATTTCTCCGGATTATTCCCCTGCTATGGGCA
GGTTGCATACGTGTTACTCACCCGTGCG
>FLP3FBN01B4E5A length=252 xy=0753_1308 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTC
AAGCTAATGTCACGCGAGCCTATCCTCATCCGACGAATCTTTAGATGGATTCAGATGCCT
GATTCCATCACCATGGGGCGATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGC
AAGTTGCTCACG
>FLP3FBN01ET99E length=270 xy=1868_0224 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCCCGCCAAC
CACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCACCGGACCATGCGG
TCCTGCGGTCTTATGCGGTATTAGCAGTCATTTCTGACTGTTGTCCCCCTGTGTAAGGCA
GGTTGCCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01BS7JZ length=276 xy=0625_2541 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGGCCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGGGCCGTTACCCCGCCAAC
AACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGA
AGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAG
GTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCC
>FLP3FBN01DW7MZ length=274 xy=1491_1369 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAAC
CAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCT
CTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAG
GTTGCATACGCGTTACGCACCCGTGCGCCGGTCG
>FLP3FBN01BKWR9 length=246 xy=0531_0359 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGATCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
TAACTAATGCGCCGCGGGCCCATCTCATAGCGGATTGCTCCTTTAATTGCTACTTCATGC
GAAGCTGCAATCTTATGCGGTATTAATCTCCCTTTCGGAAGGCTATTCCCCTCTATGAGG
CAGGTT
>FLP3FBN01DLUYP length=272 xy=1362_0111 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGAGAAAATGCTC
ACTCATGATACCATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCA
GGTTGGATACGCGTTACTCACCCGTGCGCCGG
>FLP3FBN01DBVV5 length=251 xy=1248_1699 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTTACCCTCTCAGGCCGGCTACGCATCATCGCCTTGGTGGGCCGTTACCTCACCAAC
TAGCTAATGCGCCGCAGGTCCATCCATGTTCACGCCTTGATGGGCGCTTTAATATACTGA
GCATGCGCTCTGTATACCTATCCGGTTTTAGCTACCGTTTCCAGCAGTTATCCCGGACAC
ATGGGCAGGTT
>FLP3FBN01B5DKJ length=273 xy=0764_0865 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCCTATACCACCGGAGTTTTTCACACCGGAGCATGCGC
TCCTGTGCGCTTATGCGGTATTAACAGTCGTTTCCAACTGTTATCCCCCTGTATAGGGCA
GGTTACCCACGCGTTACTCACCCGTCCGCCACT
>FLP3FBN01ASQES length=254 xy=0210_0546 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAGTGGTG
GGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGT
CAAGCTAATGTCACGCGAGCCTATCCTCATCCGACGGATCTTTAGATGGAACCAGATGCC
TGATTCCATCGCCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGC
AAGTTGCTCACGCG
>FLP3FBN01DPNWR length=281 xy=1405_1465 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
TAGCTAATGCGCCGCATGGCCATCCGTAGCCGGTGTTACCCTTTAAACCCCAGGAGATGC
CTCTCGGAGTTATTACGCGATATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTACGGG
CAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01DY5C2 length=264 xy=1513_1620 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCTCTCAGGTCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAACT
GGCTAATCAGACGCGGGTCCATCTTATACCACCGGAGTTTTTCACACTTGCACCATGCGG
TGCTGTGCGCTTATGCGGTATTAGCAGTCCATTTCTAACTGTTACTCCCCTGTATAAGGC
AGGTTACCACGCGTTACTCACCCG
>FLP3FBN01BLIY9 length=271 xy=0538_0451 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCATTGCCTTGGTGGGCCGTTACCCCCCAACT
AGCTAATAGGACGCATGCCCATCTGATACCTCGAACGATTTAATTATTATAAGATGCCTT
ACAATAATATTATGGGGTGTTAATCCACGTTTCCATGGGCTATCCCCCTGTATCAGCCAG
GTTGCATACGCGTTACTCACCCGTGCGCCGG
>FLP3FBN01DQGLV length=258 xy=1414_1793 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGATCGACCTCTCAGTCCGGCTACCCGTCGTCGGCTAGGTGGGCCACTGCCCCGCCTAC
TACCTGATGGGCCGCGACCCCATCCTGCACCGCCGGAGCTTTCATCCGCTCCCCATGCGG
GGTGCGGATAGTATCCGGTATTAGCTGCCGTTTCCGGCAGTTATCCCGATGTGCAGGGCA
GGTTGGTCACGTGTTACT
>FLP3FBN01CUQNU length=255 xy=1053_0488 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCA
GGTTATCCACGCGTT
>FLP3FBN01B0QGX length=272 xy=0711_1391 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGGTG
GCCGATCGCCCTCCTCAGGCCGGCTACCCATCGCAGGCTAGGTGGGCCGTTGCCCCGCCT
ACTACCTAATGGGCCGCGACCCCATCCCGCACCGTCGGAGCTTTCCTCCGTGGCGCATGC
GCGCCTCGGAGAGTATCCGGTATTAGCCGCCGTTTCCGGCGGTTGTCCCGGGGTGCGGGG
CAGGTTGGTCACGTGTTACTCACCCGTTCGCC
>FLP3FBN01A3PGP length=273 xy=0335_0535 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCGGCACCGGGATGCC
CCGATGCCGCGTCACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTGGCGGGC
AGGTTGCATACGTGTTACTCACCCGTGCGCCGG
>FLP3FBN01ALY2B length=251 xy=0133_0561 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTC
AAGCTAATGTCACGCGAGCCTATCCTCATCCGACGAATCTTTAGATGGATTCAGATGCCT
GATTCCATCACCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCA
AGTTGCTCACG
>FLP3FBN01BT8QN length=253 xy=0637_1581 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTCTGG
ATGATCATCCTCTCAAACCATCTAACGATCGTCGACTTGGTGAGCCTTTACCTCACCAAC
TATCTAATCGTACGCAGGCCATTCCTAAAGCGCATAAATGCTTTAATCCGAAGATCATAT
GCGGTATTAGCCACTCTTTCGAGTAGTTATCCCTCACTTTAGGGTATGTTCCCACGCGTT
ACTCAGCCGTCCG
>FLP3FBN01DATE7 length=281 xy=1236_0993 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGCGG
GGGGTTAACCTCTCAGTCCCCTATGTATCGTTGCCTTGGTGAGCCGTTACCTCACCAACT
AGCTAATACAACGCATGCCCATCTTTAACCACCGGAGTTTTTAACCCAAGAAGATGCCTT
CTCGAATGTTATGGGGAATTAGTACCAATTTCTCAGTGTTATGCCCCTGTTAAAGGTAGT
TGCATACGCGTTACGCACCCGTGCGCCGGTCGCCACCAAAG
>FLP3FBN01A5XYK length=271 xy=0360_2458 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCGCCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTAGGCTTCTACCCCACCAAC
TACCTAATCAGACGCGGGCCCATCTTACACCACCTCAGTTTTTACCACTGTACCATGCAG
TACCGTGGTCTTATGCGGTATTAGCAATCATTTCTAATTGTTATCCCCCTGTGTAAGGCA
GGTTGCCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01BAT5U length=269 xy=0416_1440 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGCTGCCCCGCCAAC
AAGCTAATCAGACGCGGGCCCCTCCCATACCGCCGGAACTTTCCCCAGAAAGGCATGCGC
CTCCCTGGTTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTATGGGGCAG
GTTGCCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01ECTYY length=278 xy=1669_1064 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCACGGTGGGCCGTTACCCCGCCGTC
AAGCTAATGCGCCGCATGCCCATCCTCCCCCGATGAATCTTTCCTCCTCCGGAGATGCCT
CCGGAGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGGGAAGGGCAG
GTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01A2T1M length=251 xy=0325_0776 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTTACCCTCTCAGGCCGGCTACGCATCATCGCCTTGGTGGGCCGTTACCTCACCAAC
TAGCTAATGCGCCGCAGGTCCATCCATGTTCACGCCTTGATGGGCGCTTTAATATACTGA
GCATGCGCTCTGTATACCTATCCGGTTTTAGCTACCGTTTCCAGCAGTTATCCCGGACAC
ATGGGCAGGTT
>FLP3FBN01CBDAE length=253 xy=0832_1908 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTC
AAGCTAATGTCACGCGAGCCTATCCTCATCCGACGGATCTTTAGATGGAACCAGATGCCT
GATTCCATCGCCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCA
AGTTGCTCACGCG
>FLP3FBN01D8Y20 length=264 xy=1625_1290 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGACCTTCCTCTCAGAACCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCGCCAACCAG
CTAATGCGCCGCATGCCCATCCTTTACCGGATCGCTCCTTTGACATACCGATCATGCGGT
CGGTATGTGTTATGAGGTATTAGACGGAATTTTCTCCCGATTATCCCTCTGTAAAGGGCA
GGTCGCATACGTGTTACTCACCCG
>FLP3FBN01A6MXL length=274 xy=0368_2055 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGATGTCTTGGTGGGCCGTTACCCCGCCAAC
AAACTAATGGAACGCATCCCCATCGATTATCGAAATTCTTTAATAACAAGAAGATGCCTT
CTCGTTATGCTATCCAGTATTAATCTTTCTTTCGAAAGGCTATCCCGGAATAATCGGCAG
GTTGGATACGTGTTACTCACCCGTGCGCCGGTCG
>FLP3FBN01ED1E9 length=255 xy=1683_0035 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAGTGGTG
GGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGT
CAAGCTAATGTCACGCGAGCCTATCCTCATCCGACGGATCTTTAGATTGGAACCAGATGC
CTGATTCCATCGCATGGGGCATTAGACGCCGTTTCCAGCCGATTATTCCCCTGATGAGGG
CAAGTTGCTCACGCG
>FLP3FBN01AYIUY length=251 xy=0276_0360 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
GACTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTGAGGCA
GGTTATCCACG
>FLP3FBN01A0UCW length=276 xy=0302_2078 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGGCCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGGGCCGTTACCCCGCCAAC
AACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGA
AGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAG
GTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCC
>FLP3FBN01CDJ70 length=268 xy=0857_1806 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCTTTGGTGGGCCTCTACCCCGCCAAC
TGGCTAATCAGACGCGGGCCCCTCCCATACCACTATTGTTTTTCACACAGAACCATGCGG
TCCCGTGCGCTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTATGGGGCA
GGTTGCCCACGCGTTACTCACCCGTCCG
>FLP3FBN01EFY7W length=260 xy=1705_0378 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
AAGCTAATCAGACGCGGATCCATCGTATACCACCAAAAGCTTTAGCTTTTTGTTTTCCAC
ACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCC
CCCAGTATACGGCAGGTTCT
>FLP3FBN01AGI0O length=262 xy=0071_0438 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGGCCACCCTCTCAGGCCGGCTACCCATCGTCGCCTTGGTAGGCCATTACCCTACCAAC
TAGCTAATGGGACGCGAGTCCATCTTTCAGCGTCAGGTCTCCCCAACTTTTCCTATATAA
ACATGCACTTATATAACTCATCCGGCATTAGCTCACCTTTCGGTAAGTTGTTCCAGTCTA
AAAGGCAGGTCACTCACGTGTT
>FLP3FBN01CUN6D length=272 xy=1052_1363 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
TAGCTAATGCGCCGCATGGCCATCCGTAGCCGGTGTTACCCTTTAAACCCCAAGAGATGC
CTCTCGGAGTTATTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTATGGGG
CAGGTTGCCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01EFPI1 length=281 xy=1702_0107 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01A8V8L length=259 xy=0394_0931 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
CCGATCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCCACCAAC
TAGCTAATCAGACGCAGGCCCATCCTTTGCCGATAAATCTTTGACCAAACAGCCATGTGA
CTATTTAGTATTATGAGGTTTTAATCATCGTTTCCAATGGCTATCCCTCTGCAAAGGGCA
GGTTGCCTACGCGTTACGT
>FLP3FBN01BZOEE length=268 xy=0699_1204 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCT
CCCAACGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCA
GGTTCCATACGTGTTACTCACCCGTGCG
>FLP3FBN01EGX4W length=281 xy=1716_0574 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
CAGCTAATGCGCCGCATGGCCATCCGTAGCCGGTGTTACCCTTTAAACCCCAAGAGATGC
CTCTCGGAGTTATTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTACGGG
CAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01D2AYU length=273 xy=1549_1396 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCCCGCCAAC
TACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCACCGGACCATGCGG
TCCTGTGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCA
GGTTGCCCACGCGTTACTCACCCGTCCGCCGCT
>FLP3FBN01BHVDZ length=276 xy=0496_1941 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCGGCACCGGGATGCC
CCGACGCCGCGTCACGCGGTATTAGACGTAATTTCTTCCGCTTATCCCCCTGTGGCGGGC
AGGTTGCATACGCGTTACTCACCCGTGCGCCGGTCG
>FLP3FBN01ED5UR length=255 xy=1684_1681 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTCTCCACGCGTT
>FLP3FBN01CU37V length=270 xy=1057_1673 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCAACCTCTCAGTCCGGCTACCGATCGTCGACTTGGTGGGCCGTTACCCCGCCAAC
TATCTAATCGGACGCGAGCCCATCTTTCAGCGGATTGCTCCTTTGATCCCCAAGGGATGT
CCCTCAAGGATGTCATGCGGTATTAGCGTTCGTTTCCAAACGTTATCCCCCTCTGAAAGG
CAGGTTGCTCACGTGTTACTCACCCGTCCG
>FLP3FBN01B86XL length=268 xy=0807_2759 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCACCAAC
TGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGG
AACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCA
GGTTATCCACGCGTTACTCACCCGTCCG
>FLP3FBN01DJ1VW length=264 xy=1341_1786 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGG
ACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCA
GGTTGGATACGTGTTACTCACCCG
>FLP3FBN01AKD0L length=253 xy=0115_0355 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACTCTCTCAAGCCGGCTACTGATCGTTGTCTTGGTAGGCCGTTACCCTGCCAAC
TAACTAATCAGACGCGGGCCCATCCTGTACCACCGTGGTTTTCCCTGCTGTGCCATGCGG
CACAGCAGGCTTATGCGGTATTAGCAGCCATTTCTGGCTGTTGTCCCCCGGTACAGGGCA
GGTTGCCCACGCG
>FLP3FBN01D6H8C length=272 xy=1597_0826 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAGGGAAAAGCCATGCGG
CTTCCCCCGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGCCAG
GTTGCCCACGCGTTACTCACCCGTCCGCCACT
>FLP3FBN01DQFJG length=270 xy=1414_0410 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGACTTGGTGGGCCTCTACCCCGCCAAC
TATCTAATCAGCCGCGAGTCCATCTCCCATCGGATTTCTCCTTTGACTATCCAAGGATGC
CCTCAAATAGTATTATGCGGTATTAGCGTCCGTTTCCAGACGTTATCCCCCTTTGAAAGG
TAGGTTACTCACGTGTTACTCACCCGTCCG
>FLP3FBN01AGC0G length=277 xy=0069_0846 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>FLP3FBN01D0FVD length=262 xy=1528_0455 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGGCCACCCTCTCAGGCCGGCTACCCATCGTCGCCTTGGTAGGCCATTACCCCACCAAC
TAGCTAATGGGACGCGAGTCCATCTTTCAGCGTCAGGTCTCCCCAACTTTTCCTATATAA
ACATGCACTTATATAACTCATCCGGCATTAGCTCACCTTTCGGTAAGTTGTTCCAGTCTA
AAAGGCAGGTCACTCACGTGTT
>FLP3FBN01BG1KW length=277 xy=0487_0174 region=1 run=R_2008_12_09_13_51_01_
ACAGAGTCGGCTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>FLP3FBN01BW4TT length=253 xy=0670_1311 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGACCGTATCTCAGTTCCAATGTGG
CCGATCAGCCTCTCAGCTCGGCTATGCATCGTTGCCTTGGTAGGCCATTGCCCCACCAAC
TAGCTAATACACCGCAAGCTCATCCTAAGGTGAAGCAAACGCTTCTTTTAACATATCTAG
TTATCTAGTTATGTATCATCCGGTATTAGCGTTCGTTTCCAAACGTTATCCCAGTCCCTA
GGGTAGATTACCC
>FLP3FBN01BOFGD length=270 xy=0571_0683 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCCCAGTCCCAATGTGG
CCGGTCGCCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTAGGCTTCTACCCCACCAAC
TACCTAATCAGACGCGGGCCCATCTTACACCACCTCAGTTTTTACCTCTGTACCATGCGG
TACTGGGGTCTTATGCGGTATTAGCAATCATTTCTAATTGTTATCCCCCTGTGTAAGGCA
GGTTGCTCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01CWLZU length=255 xy=1074_1736 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAAC
CAGCTAATCAGACGCGGGCCCATCCCGCACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTCTCCACGCGTT
>FLP3FBN01AMXMK length=265 xy=0144_0298 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAAC
TGGCTAATCAGACGCGGGTCCATCTTATACCACCGGAGTTTTTTCACACTGTACCATGCA
GTACTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATAAGGC
AGGTTACCCACGCGTTACTCACCCG
>FLP3FBN01BHNXZ length=264 xy=0494_0485 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGG
ACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCA
GGTTGGATACGTGTTACTCACCCG
>FLP3FBN01DRT2O length=258 xy=1430_0366 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTCTCCACGCGTTACT
>FLP3FBN01ATU7A length=261 xy=0223_0164 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCACCAAC
TACCTAATGCGCCGCATGCCCATCCTTTACCGGATCGCTCCTTTGACATACCGATCATGC
GGTCGGTATGTGTTATGAGGTATTAGACGGAATTTCTCCCGATTATCCCTCTGTAAAGGG
CAGGTCGCATACGTGTTACTC
>FLP3FBN01DQMNE length=275 xy=1416_1432 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTGCCAAC
AAGCTAATCAGACGCGGGTCCATCTCGCACCACCGGAGTTTTCAGGGCAGGGGCATGCGC
CCCCTCCCGTTATGCGGTGTTAGCACCTATTTCTGGGTGTTATCCCCCAGTGTAAGGCAG
GTTACCCACGCGTTACTCACCCGTCCGCCACTAAG
>FLP3FBN01BGO68 length=277 xy=0483_0514 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>FLP3FBN01EC56I length=258 xy=1673_0504 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGAGCTCTTACCTCACCAAC
TAGCTAATCAGACGCGGGCCCATCTTACACCACCTCAGTTTTTTCCACAAGGTCATGCGA
CCCTGTGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCA
GGTTGCCCACGCGTTACT
>FLP3FBN01D9EK7 length=256 xy=1630_0905 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCATGTGGG
GGCCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACTA
TCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGAC
TCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAGG
TCACTCACGTGTTACT
>FLP3FBN01BXX60 length=258 xy=0679_2506 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGTCCACCCTCTCAGGCCGGCTGCTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAAC
TGGCTAATCGGACGCGGATCCATCGTATGCCGATAAATCTTTTCACACCAGACCATGCGA
TCCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCA
GGTTACCCACGCGTTACT
>FLP3FBN01A208L length=281 xy=0327_1907 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01EBSVN length=272 xy=1657_2145 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAAC
CAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCT
CTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAG
GTTGCATACGCGTTACGCACCCGTCCGCCGCT
>FLP3FBN01BDPN5 length=259 xy=0449_0419 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGCGAGAATGCTC
ACTCATGATACTATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCA
GGTTGGATACGCGTTACTC
>FLP3FBN01DF6NE length=265 xy=1297_1560 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCGCCTACT
ATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGGA
CTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTTAGACCGGC
AGGTTGGATACGTGTTACTCACCCG
>FLP3FBN01D87QS length=255 xy=1628_0226 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAAC
TACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGG
TTCCGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCA
GGTTCNCCNCGTCTC
>FLP3FBN01BSM84 length=255 xy=0619_0806 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTCTCCACGCGTT
>FLP3FBN01BJIH7 length=262 xy=0515_0733 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAAC
TAGCTAATGTCACGCATGCCCATCCGTTGCCGGAATCGCTTCCTTTGGCCGCAGGGCCAT
GCAGCCCCGCGGCATTACGCGGTATTAGACGGGATTTCTCCCGATTATCCCCCTGCAACG
GGCAGGTCGCATACGCGTTACT
>FLP3FBN01D8FO0 length=268 xy=1619_0738 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGG
AACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCA
GGTTATCCACGCGTTACTCACCCGTCCG
>FLP3FBN01AGYWU length=255 xy=0076_0556 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTT
>FLP3FBN01D6UF4 length=270 xy=1601_0274 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAAC
TAGCTAATGCGCCGCATGACCATCCGCAGCCGGATCGCTCCTTTGAACCAACAGGGATGC
CCCCGTCGGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGG
CAGGTTTCATACGTGTTACTCACCCGTGCG
>FLP3FBN01AEB9M length=274 xy=0046_0776 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAAC
CAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCT
CTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAG
GTTGCATACGCGTTACGCACCCGTGCGCCGGTCG
>FLP3FBN01CIT14 length=281 xy=0917_2074 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGCCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
TAGCTAATGCGCCGCATGGCCATCCGTAGCCGGTGTTACCCTTTAAACCCCAGGAGATGC
CTCTCGGAGTTATTACGCGATATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTACGGG
CAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01DZG2M length=281 xy=1517_0412 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCCAA
CTACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATG
TCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCTGTAGCGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01CN7JX length=281 xy=0978_2987 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTCGGTGGGCCGTTACCCCGCCGAC
TGCCTAATGCGCCGCATGCCCATCCTCCACCACCGGAGTTTTCCTCCCACGGAGATGCCT
CCGCGGGATTTACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAG
GTTGCATACGTGTTACTCACCCGTGCGCCACTAACTCAAGG
>FLP3FBN01B74TP length=281 xy=0795_2523 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGG
CAGGTTGCATACGTGTTACTCACCCGTCCGCCACTAGGGCG
>FLP3FBN01BXZ1T length=282 xy=0680_0815 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAGTGTGG
CCGATCACCCTCTCAGGTCGGCTACGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAAC
TAGCTAATACGGCGCGGGTCCATCTATAAGTGATAGCAAAACCATCTTTCACTTTAGAAC
CATGCGGTTCTAAATGTTATCCGGTATTAGCTCCGGTTTCCCGAAGTTATCCCAGTCTTA
TAGGTAGGTTACCCACGTGTTACTCACCCGTCCGCCGCTAAG
>FLP3FBN01C6LRW length=264 xy=1188_1066 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGG
ACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCA
GGTTGGATACGTGTTACTCACCCG
>FLP3FBN01AOR4O length=264 xy=0165_0470 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCACCAAC
TAGCTAATCAGACGCGGGTCCATCTCATACCACCGGAGTTTTTCACACCGGATCATGCAA
TCCTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATAGGGCA
GGTTACCCACGCGTTACTCACCCG
>FLP3FBN01CSZZD length=280 xy=1033_1175 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
AAGCTAATGCGCCGCATGCCCATCCTCCGCCGGAATCGCTTCCTTTTAACACCCGTGGGA
TGCCCCACAGATGTACCACGCGGTATTAGTCCGGATTTCTCCGGGTTATCCCCCTGCGGA
GGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>FLP3FBN01DNS9D length=272 xy=1384_1103 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGAGAAAATGCTC
ACTCATGATACCATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCA
GGTTGGATACGCGTTACTCACCCGTGCGCCGG
>FLP3FBN01EGW4M length=270 xy=1715_3364 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTCTACCCGCCAACA
AGCTAATCAGACGCGGGTCCATCGTATACCACCGGAGTTTTTCACACCGGACCATGCGAT
CCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTCGTTATCCCCCTGTATACGGCA
GGTTACCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01C9TUA length=271 xy=1224_4032 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTATGGATCGTCGGTTTGGTGGGCCGTTACCCCGCCAAC
TGCCTAATCCAACGCGGACCCATCCTATGCCGCCTCAGCTTTTCACACCGTACCATGCGG
TACTGTGCGCTTATGCGGTATTAGCACCCGTTTCCGGATGTTATCCCCCTGCATAGGGCA
GGTTGTCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01DZ0UC length=282 xy=1523_1458 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGCCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACT
AGCTAATGCGCCGCATGCCCATCCGTGGCCGGGATTGCTCCCTTTGGCGGCCCGGGGATG
CCCCAAGGCCGCGTTACGCGGTATTAGACGGGGTTTCCCCCGCTTATCCCCCTGCCACGG
GCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01D9E65 length=276 xy=1630_1695 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGTCTTGGTGGTCCGTTACACCGCCAAC
TAACTAATGCGACGCATGCCCATCCTTCACCGAAATTCTTTCCCCCTCGGAAGATGCCTC
CCAAGGAGTATATGCGGTATTAGGCGAAATTTCTTCCGGTTATCCCGCTGTAAAGGGTGC
TGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01ECWT9 length=273 xy=1670_0687 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCTCACCAAC
TACCTAATCAGACGCGGGTCCCTCCTATACCACTATCGTTTTTCACACAGGGCCATGCGG
CCCCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATAGGGCA
GGTTCCCCACGCGTTACTCACCCGTCCGCCGCT
>FLP3FBN01AHH1K length=264 xy=0082_0774 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
CAGCTAATCGGACGCGGATCCATCGTATGCCGATAAATCTTTTCACACTATACCATGCGG
TACTGTGCGCTTATGCGGTATTAGCAACTGTTTCCAGTTGGTATCCCCCTGCATACGGCA
GGTTATCCACGCGTTACTCACCCG
>FLP3FBN01DQLWU length=259 xy=1416_0476 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGG
ACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCA
GGTTGGATACGTGTTACTC
>FLP3FBN01ALCMH length=268 xy=0126_0151 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAAC
TAGCTAATGTCACGCATGCCCATCCGTTGCCGGAATCGCTTCCTTTGGCCGCAGGGCCAT
GCAGCCCCGCGGCATTACGCGGTATTAGACGGGATTTCTCCCGATTATCCCCCTGCAACG
GGCAGGTCGCATACGCGTTACTCACCCG
>FLP3FBN01C4N17 length=274 xy=1166_0829 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAAC
CAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCT
CTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAG
GTTGCATACGCGTTACGCACCCGTGCGCCGGTCG
>FLP3FBN01B8F4K length=264 xy=0799_0786 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCCACCAAC
TAGCTAATCAGACGCGGAACCATCGTATACCACCAGAGTTTTTCACACCGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTCTGAAAGGCA
GGTTGCTCACGTGTTACTCACCCG
>FLP3FBN01CATW7 length=272 xy=0826_1385 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCT
CCCAACGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCA
GGTTCCATACGTGTTACTCACCCGTGCGCCGG
>FLP3FBN01B97H8 length=277 xy=0819_1006 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGCCCATCCGCCACCGGATTGCTCCTTTGACCGCCCCGGGATGT
CCCGGAATGGTGTTACGCGGAATTAGTCGGAATTTCTTCCGGTTATTCCCCTGTGACGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>FLP3FBN01BA934 length=257 xy=0421_1634 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGTCTTGGTGGTCCGTTACACCGCCAAC
TAACTAATGCGACGCATGCCCATCCGCCACCGGAATCAACCTTTGGCACCAACAGGATGT
CCCATAGGTGCATTATGCCGTATTAGACGGAATTTCTCCCGATTATCCGGCTGTGGCAGG
CAGGTTGCATACGTGTT
>FLP3FBN01A3YNV length=264 xy=0338_0169 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
AAGCTAATCAGACGCGGACCCATCCCGCACCGCATGCGCTTTCCGCGCGGCCCCATGCGG
GGCCGTGCGCATATGCGGTATTAGCAGCCGTTTCCAGCTGTTGTCCCCCAGTGCGGGGTA
GGTTGTCCACGCGTTACTCACCCG
>FLP3FBN01BJOS4 length=268 xy=0517_0710 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
CAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTCACACCGAACCATGCGG
TTCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01EP59G length=271 xy=1821_0930 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAAC
AAACTAATGCACCGCAGGTCCATCCGCACCCCATCCCCTAAAGGATGTTTCACAGAAAGA
AGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGC
GCGGGCAGGTTCCCTACGTGTTACTCACCCG
>FLP3FBN01CTU4A length=268 xy=1043_0568 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCGG
GACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCA
GGTTATCCACGCGTTACTCACCCGTCCG
>FLP3FBN01DD28G length=271 xy=1273_2126 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAAC
AAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGA
AGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGC
GCGGGCAGGTTCCCTACGTGTTACTCACCCG
>FLP3FBN01BW0YB length=287 xy=0669_0385 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAGTGTGG
GGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGCCACGGTGGGCCGTTACCCCGCCGTC
AAGCTAATGCGCCGCATGCCCAGCCGCCACCGGATTACTCCTTTCGCCCGGACCGGATGC
CCGGTCCGGGCGGCATGGGGTATTAGGCCGGGTTTCCCCGGGTTATCCCCCTGTGGCGGG
CAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCAGGTG
>FLP3FBN01AZT44 length=264 xy=0291_0198 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGG
ACTCATGATGCCATCTTGTATTAATCTTTCTTTCAGAAGGCTGTCCAAGAGTAGACGGCA
GGTTGGATACGTGTTACTCACCCG
>FLP3FBN01AGMG5 length=270 xy=0072_0823 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAAC
CAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGTACCATGCGG
TACTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCA
GGTTACCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01ARUVC length=268 xy=0200_0630 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
AAGCTAATCAGACGCGGGTCCATCTTACACCACTAATGTTTTTCACTCTGTCCCATGCGG
GACTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTGTAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01D86I8 length=254 xy=1627_2754 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAACTGTTATCCCCTGTATGAGGCAG
GTTACCCACGCGTT
>FLP3FBN01COOOS length=281 xy=0984_0618 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTAC
TGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCAAGGATGCCC
CCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAG
GTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGGCAG
>FLP3FBN01D8J4C length=259 xy=1620_2378 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCACCAACAGGATGTC
CCATAGGTGCATTATGCCGTATTAGACGGAATTTCTCCCGATTATCCGGCTGTGGCAGGC
AGGTTGCATACGTGTTACT
>FLP3FBN01BR61D length=280 xy=0614_0271 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCGGCACCGGGATGCC
CCGACGCCGCGTCACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTGGCGGGC
AGGTTGCATACGTGTTACTCACCCGTGCGCCGGTTTAATC
>FLP3FBN01BF0UO length=288 xy=0475_1726 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAGTGTGG
GGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGCCACGGTGGGCCGTTACCCCGCCGTC
AAGCTAATGCGCCGCATGCCCAGCCGCCACCGGATTCCTCCTTTCGCCCGGTCCGGATGC
CCGGTCCGGGCGGCATGGGGTATTAGGCCGGGTTTCCCCGGGTTATCCCCCTCGTGGCGG
GCAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCAGGTG
>FLP3FBN01DB7BE length=268 xy=1252_0120 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTTGCCTTGGTGGGCCGTTACCCCTCCAAC
TAGCTAATCAGACGCGGGTCCATCTCATACCGTCTCGGCTTTTCACCCCGAACCATGCGG
TTCTGTGTGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATGAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01AZSCT length=271 xy=0290_1979 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAAC
CAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGA
AGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGC
GCGGGCAGGTTCCCTACGTGTTACTCACCCG
>FLP3FBN01D3EWI length=281 xy=1561_4000 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01A55LW length=257 xy=0363_0082 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAAC
TAGCTAATGTCACGCATGCCCATCCCGCACCGGATCGCTCCTTTGACCGCTCCCCCATGC
AGAGGAACGGTGTCATGCCGTATTAGTCCGGATTTCTCCGGGTTATCCGGCTGTGGCAGG
CAGGTTGCATACGTGTT
>FLP3FBN01DZ6P8 length=269 xy=1525_0894 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
AAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTAAGGAAAAGACATGCAT
CTTCTCCTGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTGTAAGGCAG
GTTACCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01AEB2N length=271 xy=0046_0525 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTACCTCAGAACCATGCGG
TTCCGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCA
GGTTGCCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01B9U1E length=281 xy=0815_1232 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGTCTTGGTGGTCCGTTACACCGCCAAC
TAACTAATGCGACGCATGCCCATCCTTCACCGAAATTCTTTCCCCCTCGGAAGATGCCTC
CCAAGGAGTATATGCGGTATTAGGCGAAATTTCTTCCGGTTATCCCGCTGTAAAGGGTAG
GTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCAG
>FLP3FBN01A77ZB length=268 xy=0386_2261 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAG
GACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01BCQ7B length=254 xy=0438_0805 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGTTCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTAGGCCTTTACCCCACCAAC
TAGCTAATGCGCCGCAGGCTCATCCATCAGTGATGCCAGGAGCATCTTTAAACTTTCGTC
CTATCCGGTATTAGCGATCGTTTCCAATCGTTGTCCCCGTCTGATGGGCAGATCACCTAC
GTGTTACTCACCCG
>FLP3FBN01D5TAZ length=271 xy=1589_1289 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCGCCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTAGGCTTCTACCCCACCAAC
TACCTAATCAGACGCGGGCCCATCTTACACCACCTCAGTTTTTACCTCTGTACCATGCGG
TACTGGGGTCTTATGCGGTATTAGCAATCATTTCTAATTGTTATCCCCCTGTGTAAGGCA
GGTTGCCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01BX26A length=278 xy=0681_0768 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTAC
TGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCAAGGATGCCC
CCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAG
GTTGCATACGTGTTACTCACCTGTGCGCCAGTCGCCGG
>FLP3FBN01DTX5C length=255 xy=1454_0654 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTCTCCACGCGTT
>FLP3FBN01BCTWA length=268 xy=0439_0200 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCACCAAC
TGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGG
AACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCA
GGTTATCCACGCGTTACTCACCCGTCCG
>FLP3FBN01BA22V length=272 xy=0419_0709 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGAGAAAATGCTC
ACTCATGATACCATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCA
GGTTGGATACGCGTTACTCACCCGTGCGCCGG
>FLP3FBN01A4UET length=281 xy=0348_0355 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01EII4M length=253 xy=1734_0708 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTCCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTC
AAGCTAATGTCACGCGAGCCTATCCTCATCCGACGAATCTTTAGATGGATTCAGATGCCT
GATTCCATCACCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCA
AGTTGCTCACGCG
>FLP3FBN01DPG83 length=282 xy=1403_1029 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTGGTCTTGGTGAGCCGTTACCCCACCAAC
TAACTAATACAACGCATGCCCATCCATTACCACCGGAGTTTTCAACCCAAGAAGATGCCT
CCCTGGATGTTATGGGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTAATGGGTAG
GTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCGACAAT
>FLP3FBN01AYGIQ length=262 xy=0275_1424 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAAC
AAACTAATGCACCGCAGGTCCATCCGCACCCCATCCCCTAAAGGATGTTTCACAGAAAGA
AGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGC
GCGGGCAGGTTCCCTACGTGTT
>FLP3FBN01AGF7L length=268 xy=0070_0895 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
CCGATCACCCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTAAGCCGTTACCCTACCAA
CTAGCTAATCAGACGCGGGTCCATCCTGTACCGCAAAAGCTTTGATACTTCTACCATGCG
ATAAAGTATTTTATCTCGTATTAGCATACCTTTCGGTATGTTATCCGTGTGTACAGGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01A4NV1 length=286 xy=0346_0095 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCCCGTACCACCGGAGTTTTCAAGGAGTCCCCATGCAG
GGTCCCCTGTTATGCGGTATTAGCACCTGTTTCCAGGTGTTATCCCCCGGTACGGGGCAG
GTTGCCCACGCGTTACTCACCCGTCCGCCACTAAAACAGTCCGGGG
>FLP3FBN01AZ963 length=272 xy=0296_0525 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGCTGCCCCGCCAAC
AAGCTAATCAGACGCGGGCCCCTCCCATACCGCCGGAACTTTCCCTAGAAAGGCATGCGC
CTCCCTGGTTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTATGGGGCAG
GTTGCCCACGCGTTACTCACCCGTCCGCCACT
>FLP3FBN01DMBQY length=255 xy=1367_1384 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGATCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTAGGCCTTTACCCCACCAAC
TAGCTAATCAGACGCGGGTCCATCTCATACCACCGGAGTTTTTCACACCGAATCATGCGA
TTCTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATGAGGCA
GGTTACCCACGCGTT
>FLP3FBN01A6SLO length=268 xy=0370_1210 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGCGAGAATGCTC
ACTCATGATACTATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCA
GGTTGGATACGCGTTACTCACCCGTGCG
>FLP3FBN01CQHPF length=262 xy=1004_2961 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGGCCACCCTCTCAGGCCGGCTACCCATCGTCGCCTTGGTAGGCCATTACCCTACCAAC
TAGCTAATGGGACGCGAGTCCATCTTTCAGCGTCAGGTCTCCCCAACTTTTCCTATATAA
ACATGCACTTATATAACTCATCCGGCATTAGCTCACCTTTCGGTAAGTTGTTCCAGTCTA
AAAGGCAGGTCACTCACGTGTT
>FLP3FBN01CSQG7 length=264 xy=1030_1145 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTTGCTTTGGTAGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGGTCCATCGTATACCTCCGGAAATTTTCACACTCTGTCATGCGA
CAGTGTGCGCTTATGCGGTATTAGCAGTTGTTTCCAACTGTTATCCCCCTGTATACGGCA
GGTTACCCACGCGTTACTCACCCG
>FLP3FBN01AMENY length=292 xy=0138_0300 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCCCGTACCACCGGAGTTTTCAAGAAAGGAACATGCGT
CCCCTTCTGTTATGCGGTATTAGCACCTGTTTCCAGGTGTTATCCCCCGGTACGGGGCAG
GTTCCCCACGCGTTACTCACCCGTCCGCCACTAAGGCCCGCGCCTTCCGGGT
>FLP3FBN01D1PPR length=268 xy=1542_2525 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGTATCATGCGA
TACTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCA
GGTTCCCCACGCGTTACTCACCCGTCCG
>FLP3FBN01CPKVQ length=269 xy=0994_1380 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAGGGAAAAGCCATGCGG
CTTCCCCCGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGCCAG
GTTGCCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01BZK6L length=276 xy=0698_1131 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGGTG
GGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCTCCAA
CCAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGTAGTTTTTCACACTGTACCATGC
GGTACTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTCAGGCA
GGTTACCCACGCGTTACTCACCCGTCCGCCACTAAG
>FLP3FBN01DF6XB length=268 xy=1297_1917 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCAACCTCTCAGTCCGGCTACCGATCGTCGCCTTGGTGGGCCTCTACCCCGCCAAC
AAGCTAATCAGACGCGGGTCCATCGTATACCACCGGAGTTTTTCACACCGGACCATGCGA
TCCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01CF2XK length=268 xy=0886_0582 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGAGAAAATGCTC
ACTCATGATACCATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCA
GGTTGGATACGCGTTACTCACCCGTGCG
>FLP3FBN01CXD3D length=277 xy=1083_1287 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
AAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTAAGGAAAAGACATGCAT
CTTTCCCTGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCAG
GTTACCCACGCGTTACTCACCCGTCCGCCACTAAGAT
>FLP3FBN01CQGY3 length=274 xy=1004_2013 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAAC
TAGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAGGGAAAAGCCATGCGG
CTTCCCCCGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGCCAG
GTTGCCCACGCGTTACTCACCCGTCCGCCACTCA
>FLP3FBN01B52I9 length=270 xy=0772_0451 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGATCAACCTCTCAGTTCGGCTACGCATCATTGCCTTGGTAAGCCTTTACCCCACCAAC
TAGCTAATGCGCCGCGGGCCCATCCAAAAGCGGTAGCATAGCCACCTTTTACATAGTTAC
CATGCGGTAACTATGGTTATGCGGTATTAGCACCTGTTTCCAAGTGTTATCCCCCTCTTT
TGGGCAGGTTGCCCACGTGTTACTCACCCG
>FLP3FBN01AS2O8 length=264 xy=0214_0090 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCTTTACCCCACCAAC
TAGCTAATCAGACGCGGGTCCATCATATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTATATGGCA
GGTTACCCACGCGTTACTCACCCG
>FLP3FBN01D7MLX length=278 xy=1609_4003 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAGTGTGG
GGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTAC
TGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCAAGGATGCCC
CCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAG
GTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGG
>FLP3FBN01EUKQQ length=271 xy=1871_1520 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAATCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAAC
CAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGA
AGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGC
GCGGGCAGGTTCCCTACGTGTTACTCACCCG
>FLP3FBN01AYCPW length=251 xy=0274_0594 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGGTCAACCTCTCAGTCCGGCTACTGATCGTCGCCTAGGTGGGCCGTTACCCCGCCTAC
CAGCTAATCAGACGCGAGGCCATCTTCCAGCGATAAATCTTTGGTGTCTCGATGATGCCA
TCGAAACACATCATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGGAAGGCA
GGTTCCTCACG
>FLP3FBN01BDSSU length=268 xy=0450_0380 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAAC
TGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGG
AACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCA
GGTTATCCACGCGTTACTCACCCGTCCG
>FLP3FBN01ELLAQ length=266 xy=1769_0128 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGACCATCCGCAGCCGGATCGCTCCTTTGAATCACCAGAGATGC
CTCCGGTGATTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGG
CAGGTTTCATACGTGTTACTCACCCG
>FLP3FBN01CEUYF length=264 xy=0872_0933 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGCCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTCTACCCCGCCAAC
AAGCTAATCAGACGCGGGTCCATCGTATACCACCGGAGTTTTTCACACCAGACCATGCGA
TCCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGCCA
GGTTGCCCACGCGTTACTCACCCG
>FLP3FBN01CI8Z2 length=276 xy=0922_0960 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCGGCACCGGGATGCC
CCGATGCCGCGTCACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTGGCGGGC
AGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>FLP3FBN01ENO6X length=271 xy=1793_0183 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAAC
TACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCACCAAACCATGCGG
TTTTGTGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCA
GGTTGCCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01B33P1 length=278 xy=0749_2887 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTAC
TGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCAAGGATGCCC
CCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAG
GTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGG
>FLP3FBN01ALS1C length=270 xy=0131_0942 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCATCGGCTTGGTGGGCCGTTACCTCACCAAC
TACCTAATCAGACGCGGGTCCCTCCTATACCACTATCGTTTTTCACACAGGGCCATGCGG
CCCCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATAGGGCA
GGTTCCCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01AY1NE length=270 xy=0282_0136 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGGTCCATCTCACACCGATAAATCTTTTCCGTCAGTACCATGCGG
TACCAGCGGGTTATGCGGTATTAGCGGTCGTTTCCAACTGTTATCCCCCTGTGTGAGGCA
GGTTACCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01D9HWD length=251 xy=1631_1099 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
CAGCTAATGCGCCATAAGTCCATCCTCTACCAGTGCCTTGCAGCACTTTTAATACGGTCA
CCATGCAGTGTCCCTACCTATGCGGTCTTAGCTGCCGTTTCCAGCAGTTATCCCCCTGTA
AAGGCCAGGTT
>FLP3FBN01B0EIF length=251 xy=0707_2277 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTCCCAATGTGG
CCGGCCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCTTTTATCTCACCAAC
TAGCTAATCAGACGCAGATCCATCCCATACCACCGGAGTTTTTCACACAGGGCCATGCAG
CCTCGTGCGCTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCGGTATGGGGCA
GGTTATCCACG
>FLP3FBN01BPI1O length=270 xy=0583_2842 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAAC
CAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGTACCATGCGG
TACTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01BBDRA length=269 xy=0422_2260 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGCTGCCCCGCCAAC
AAGCTAATCAGACGCGGGCCCCTCCCATACCGCCGGAACTTTCCCCAGAAAGGCATGCGC
CTCCCTGGTTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTATGGGGCAG
GTTGCCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01C3FYG length=244 xy=1152_1014 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAAC
TAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGA
CATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCT
TGGG
>FLP3FBN01DCLBL length=279 xy=1256_1887 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGCCCATCCGTGGCCGGGATTGCTCCCTTTGGCGGCCTTGGGAT
GTCCCTAGGCCGCGTTACGCGGTATTAGACGGGGTTTCCCCCGCTTATCCCCCTGCCACG
GGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>FLP3FBN01D7ZOI length=278 xy=1614_0464 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTAC
TGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCAAGGATGCCC
CCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAG
GTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGG
>FLP3FBN01C0173 length=259 xy=1125_0497 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGCGAGAATGCTC
ACTCATGATACTATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCA
GGTTGGATACGCGTTACTC
>FLP3FBN01DPEUG length=257 xy=1402_2006 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
TAGCTAATGCGCCGCATGACCATCCGCAGCCGGATCGCTCCTTTGAATCTCCGGAGATGC
CTCCGGAGATTATTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGG
CAGGTTTCATACGTGTT
>FLP3FBN01B7KTL length=255 xy=0789_1175 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTTTGGGCCGTATCTCAGTCCCAATGTGG
CCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
TAGCTAATCAGACGCGAGGCCATCTTTCAGCGTCAGGTCTCCCCAACTTTTCCTATATAA
ACATGCACTTATATAACTCATCCGGCATTAGCTCACCTTTCGGTAAGTTGTTCCAGTCTA
AAAGGCAGGTCACTC
>FLP3FBN01CXEBD length=269 xy=1083_1575 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAGGGAAAAGCCATGCGG
CTTCCCCCGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGCCAG
GTTGCCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01BVDVK length=255 xy=0650_1646 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTCTCCACGCGTT
>FLP3FBN01CMYHR length=268 xy=0964_1933 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGCGAGAATGCTC
ACTCATGATACTATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCA
GGTTGGATACGCGTTACTCACCCGTGCG
>FLP3FBN01D6N8J length=268 xy=1599_0417 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAG
GACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01COUSC length=268 xy=0986_0330 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCCTTGGTAGGCCATTACCCCACCAAC
TAGCTAATCAGACGCGGAACCATCGTATACCACCAGAGTTTTTCACACCGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTTTCCACGCGTTACTCACCCGTCCG
>FLP3FBN01AZPBJ length=273 xy=0289_2141 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCGCCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCCCGCCAAC
TACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCGCTGTACCATGCGG
TACTGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCA
GGTTGCCCACGCGTTACTCACCCGTCCGCCGCT
>FLP3FBN01A8PO2 length=268 xy=0392_0644 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAAC
TGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGG
AACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCA
GGTTATCCACGCGTTACTCACCCGTCCG
>FLP3FBN01CVZXE length=281 xy=1067_1808 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTAC
TGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCAAGGATGCCC
CCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAG
GTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGGCAG
>FLP3FBN01D1IC0 length=263 xy=1540_1186 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCACCAAC
CAGCTAATCAGACGCGGGCCCATCTTGCACCACCGGAATCAACCTTTGGCACCAACAGGA
TGTCCCGTCGATGCATTATGCCGTATTAGACGGAATTTCTCCCGATTATCCCTCTGTAAA
GGGCAGGTCGCATACGTGTTACT
>FLP3FBN01AF0HW length=242 xy=0065_1010 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACTCTCTCAAGCCGGCTACTGATCGTTGCTTTGGTAGGCCATTACCCTGCCAAC
TGGCTAATCAGACGCGGGGCCATCGTATGCCGATAACTCTTTTCACACCATGCCATGCAG
CATTGTGTGCTTATGCGGTATTAGCACCTATTTCTAACTGTTATCCCCCTGTGTAAGGCA
GG
>FLP3FBN01DS5AU length=274 xy=1445_0132 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGCCCATCCGTCACCGGATTGCTCCTTTGACCGCTCCGGGATGC
CCCGGAATGGTGTTACGCGGAATTAGTCGGAATTTCTTCCGGTTATTCCCCTGTGACGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGG
>FLP3FBN01DVSSP length=257 xy=1475_1015 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGTCTTGGTGGTCCGTTACACCGCCAAC
TAACTAATGCGACGCATGCCCATCCGCCACCGGAATCAACCTTTGGCACCAACAGGATGT
CCCATAGGTGCATTATGCCGTATTAGACGGAATTTCTCCCGATTATCCGGCTGTGGCAGG
CAGGTTGCATACGTGTT
>FLP3FBN01AYYJN length=257 xy=0281_0209 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGTCCGCCCTCTCAGGCCGGCTACTGATCGTGGGCTTGGTGGGCCGTTACCCCGCCAAC
TACCTAATCAGACGCGGACCCATCGTGTACCGTACTAGATAAGATCTAGGCTTTCCGCCC
TGTGCCATGCGGCACTGTGCGCATATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCC
CTGTACACGGCAGGTTG
>FLP3FBN01BQL2S length=276 xy=0596_0178 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCTTTGGTGGGCCGTTACCCCGCCAAC
TGGCTAATGCGCCGCATGCCCATCCTTTGCCGGAATTGCTTCCTTTGACTCCCAAACCAT
GTGGTAAGGGAGTGTTATGCAGTATTAGTCGGAATTTCTCCCGGTTATCCTCCTGCAAAG
GGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGG
>FLP3FBN01C6OIE length=238 xy=1189_0516 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACTCTCTCAAGCCGGCTACTGATCGTTGCTTTGGTAGGCCATTACCCTGCCAAC
TGGCTAATCAGACGCGGGGCCATCGTATGCCGATAACTCTTTTCACACCATGCCATGCAG
CATTGTGTGCTTATGCGGTATTAGCAGTCATTTCTGACTGTTGTCCCCCTGCATACGG
>FLP3FBN01D8PY9 length=239 xy=1622_1779 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAAC
TAGCTAATCAGACGCGAGCTCATCTCAGAGCGATAAATCTTTGGCGTCCAGAGAGATGCC
TCCCAGACGCATCATGCGGTATTAGCGGCTGTTTCCAACCGTTATTCCCCACTCCAAGG
>FLP3FBN01BRKZB length=271 xy=0607_0357 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTACCTATCATTGCCTTGGTGGGCCGTTACCCCCCAACT
AGCTAATAGGACGCATGCCCATCTGATACCTCGAATGATTTAATTATTAAAAGATGCCTT
CAAATAATATTATGGGGTGTTAATCCACGTTTCCATGGGCTATCCCCCTGTATCAGCCAG
GTTGCATACGCGTTACTCACCCGTGCGCCGG
>FLP3FBN01EHLMD length=268 xy=1723_2339 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCT
CCCAACGTTCTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCA
GGTTCCATACGTGTTACTCACCCGTGCG
>FLP3FBN01BOOJJ length=268 xy=0574_0173 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCTCCAAC
TAGCTAATCGGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGG
AACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCA
GGTTATCCACGCGTTACTCACCCGTCCG
>FLP3FBN01CO1SB length=261 xy=0988_1209 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
TAACTAATGCGCCGCATGCCCATCCATGACCGGATCGCTCCTTTGACTCCCGAGAGATGT
CTCCCGGGGGTGTTATGCGGTATTAGACGGAATTTCTCCCGCTTATCCCCCTGTCATGGG
CAGGTTGCATACGTGTTACTC
>FLP3FBN01D9QTP length=268 xy=1634_0379 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAAC
CAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGTACCATGCGG
TACTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTCTCCACGCGTTACTCACCCGTCCG
>FLP3FBN01CRC8T length=272 xy=1014_2875 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGAGAAAATGCTC
ACTCATGATACCATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCA
GGTTGGATACGCGTTACTCACCCGTGCGCCGG
>FLP3FBN01DMQWE length=255 xy=1372_0540 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTCTCCACGCGTT
>FLP3FBN01EU78N length=264 xy=1878_3301 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGG
ACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCA
GGTTGGATACGTGTTACTCACCCG
>FLP3FBN01AN2IX length=281 xy=0157_0055 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01AMBQN length=264 xy=0137_0605 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
CAGCTAATCAGACGCGGGCCCATCCTGTACCACCGTGGTTTTCCCTGCTGTGCCATGCGG
CACCGCAGGCTTATGCGGTATTAGCAGCCATTTCTGGCTGTTGTCCCCCGGTACAGGGCA
GGTTGCCCACGCGTTACTCACCCG
>FLP3FBN01B8NID length=268 xy=0801_2163 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAAC
TGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGG
AATCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01A2TMU length=284 xy=0325_0244 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
TAGCTAATGCGCCGCATGGCCATCCGTAGCCGGTGTTACCCTTTAAACCCCAAGAGATGC
CTCTCGGAGTTATTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTACGGG
CAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCAG
>FLP3FBN01AELDH length=271 xy=0049_0291 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAAC
CAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGA
AGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGC
GCGGGCAGGTTCCCTACGTGTTACTCACCCG
>FLP3FBN01ED2F6 length=277 xy=1683_1364 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTCGGTGGGCCGTTACCCCGCCGAC
TGCCTAATGCGCCGCATGCCCATCCTCCACCACCGGAGTTTTCCTCCCAAGGAGATGCCT
CCATGGGATTTACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAG
GTTGCATACGTGTTACTCCCCCGTGCGCCAGTCGCCG
>FLP3FBN01EABDR length=258 xy=1640_2445 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCACCAAC
TAGCTAATCAGACGCGGGTCCATCTCATACCACCGGAGTTTTTCACACCGGATCATGCAA
TCCTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATGAGGCA
GGTTACCCACGCGTTACT
>FLP3FBN01DIYZ3 length=268 xy=1329_0545 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTGCCAAC
AAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGTACCATGCGG
TACTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01APLDI length=268 xy=0174_1508 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCCACCAAC
CAGCTAATCAGACGCGGGCTCATCTTATACTACCGGAGTTTTTCACACAGAAACATGCGT
CCCCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATAAGGCA
GATTACCCACGTGTTACTCACCCGTCCG
>FLP3FBN01D0DDK length=276 xy=1527_1318 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAAC
CAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGTACCATGCGG
TACTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCA
GGTTACCCACGCGTTACTCACCCGTCCGCCACTAAG
>FLP3FBN01CDWQ4 length=261 xy=0861_1662 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
TAGCTAATGCGCCGCATGCCCATCCTTGTCCGGATAAATCCTTTGATCGAATTCTCATGC
GAGAACCCGATGTCACGCGGTATTAGACCGGATTTCTCCGGCTTATCCCCCTGACAAGGG
TAGGTTGCATACGTGTTACTC
>FLP3FBN01AGMY0 length=255 xy=0072_1466 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTCTCCACGCGTT
>FLP3FBN01DQSWF length=276 xy=1418_1341 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCGCCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTAGGCTTCTACCCCACCAAC
TACCTAATCAGACGCGGGCCCATCTTACACCACCTCAGTTTTTACCTCTGTACCATGCGG
TACTGGGGTCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTGTAAGGCA
GGTTCTCCACGCGTTACTCACCCGTCCGCCACTAAG
>FLP3FBN01CXUVC length=277 xy=1088_2550 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCCTTGGTGGGCCGTTACCCCGCCAAC
AAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTCAAGTAAAAGACATGCGT
CTCCTACTGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTAAAGGCCAG
GTTACTTATGTATTACTCACCCGTTCGCCACTCGGGC
>FLP3FBN01EFNNG length=248 xy=1701_1770 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGTTCTACCTCTCAGTACGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
CAGCTAATCAGACGCGAGCCCATCTTTAAGCGATAAATCTTTGATACACAAACCATGCGA
TTCATGTATATTATGCGGTATTAGCGGTCGTTTCCGACCGTTATCCCACTCTTAAAGGCA
GGTTGCTC
>FLP3FBN01EAC1O length=268 xy=1641_0506 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCGACCTCTCAGTCCGGCTACCGATCGTCGGCTTGGTGAGCCGTTACCTCACCAAC
TACCTAATCGGACGCGAGCCCATCTCCGAGCGATAAATCTTTGATACCAAAGGGATGTCC
CTCCAGTATGTTATGCGGTATTAGCGACCGTTTCCAGCCGTTATTCCCCTCTCGAAGGCA
GGTTGCTCACGTGTTACTCACCCGTCCG
>FLP3FBN01DO7JM length=251 xy=1400_0736 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
TAGCTAATCAGACGCGAGGCCATCTTTCAGCGATAAATCTTTGACATAAATGCCATGCGA
CACCTATGTGTTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGAAAGGCA
GGTTCCTCACG
>FLP3FBN01B5KK8 length=277 xy=0766_1770 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
TAGCTAATGCGCCGCATGGCCATCCGTAGCCGGTGTTACCCTTTAAACCCCAGGAGATGC
CTCTCGGAGTTATTACGCGATATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTACGGG
CAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCG
>FLP3FBN01BDDUQ length=258 xy=0445_1488 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTCGGTGGGCCGTTACCCCGCCGTC
AAGCTAATGTCACGCGAGCCTATCCTCATCCGACGGATCTTTAGATGGAACCAGATGCCT
GATTCCATCGCCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCA
AGTTGCTCACGCGTTACG
>FLP3FBN01ENY6F length=264 xy=1796_0837 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTCATACCACCGGAGTTTTTCACACCAGGACATGCAT
CCCTGTGCGCTTATGCGGTATTAGCAGCCGTTTCCAGCTGTTGTCCCCCAGTGTGGGGCA
GGTTGCCCACGCGTTACTCACCCG
>FLP3FBN01B8DUH length=271 xy=0798_1927 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCCACCAAC
CAGCTAATCAGACGCGGGCTCATCTTATACTACCGGAGTTTTTCACACAGAAACATGCGT
CCCCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATAAGGCA
GATTACCCACGTGTTACTCACCCGTCCGCCG
>FLP3FBN01AX3YR length=268 xy=0271_1537 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCCCGCCAAC
TACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCACTGCACCATGCAG
TACTGTGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCA
GGTTGCCCACGCGTTACTCACCCGTCCG
>FLP3FBN01BI664 length=287 xy=0511_2462 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGCCCATCCGTGGCCGGGATTGCTCCCTTTGGCGGCCCGGGGAT
GCCCCAAGGCCGCGTTACGCGGTATTAGACGGGGTTTCCCCCGCTTATCCCCCCTGCCAC
GGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCGG
>FLP3FBN01A3VZS length=270 xy=0337_0806 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTACCGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAAC
TGGCTAATCGGGCGCGGGCCCATCCCGTGCCGCCGGAGCTTTCCGCATACACCCATGCGG
CTGTATGCGCTTATGCGGTATTAGCGCCTGTTTCCAGGCGGTATCCCCCGGCACGGGGCA
GGTTGCCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01C65QI length=247 xy=1194_2360 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGTGCTTTTACCACTCCAAC
TAGCTAATCAGACGCGGGTCCATCTCATACCAGATTTCTCCTTTTCACACGAATACATGC
GTCTTCGTGCGCTTATGCGGTTTTACCACCAGTTTCCCGATGCTATCCCCCTGTATGAGG
CAGGTTG
>FLP3FBN01AMRRA length=251 xy=0142_0884 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTACCAAC
TAGCTAATCAGACGCGAGGCCATCTCTCAGCGATAAATCTTTGATATCCATACCATGCGA
TACAGATATATTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGAGAGGTA
GGTTCCTCACG
>FLP3FBN01CSPWG length=265 xy=1030_0398 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGGCCACCCTCTCAGGCCGGCTACCCATCGTCGCCTTGGTAGGCCATTACCCTACCAAC
TAGCTAATGGGACGCGAGTCCATCTTTCAGCGTCAGGTCTCCCCAACTTTTCCTATATAA
ACATGCACTTATATAACTCATCCGGCATTAGCTCACCTTTCGGTAAGTTGTTCCAGTCTA
AAAGGCAGGTCACTCACGTGTTACT
>FLP3FBN01CMVBH length=262 xy=0963_1915 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAAC
TAGCTAATGTCACGCATGCCCATCCGTTGCCGGAATCGCTTCCTTTGGCCGCAGGGCCAT
GCAGCCCCGCGGCATTACGCGGTATTAGACGGGATTTCTCCCGATTATCCCCCTGCAACG
GGCAGGTCGCATACGCGTTACT
>FLP3FBN01C8VGW length=262 xy=1214_0446 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGGCCACCCTCTCAGGCCGGCTACCCATCGTCGCCTTGGTAGGCCATTACCCTACCAAC
TAGCTAATGGGACGCGAGTCCATCTTTCAGCGTCAGGTCTCCCCAACTTTTCCTATATAA
ACATGCACTTATATAACTCATCCGGCATTAGCTCACCTTTCGGTAAGTTGTTCCAGTCTA
AAAGGCAGGTCACTCACGTGTT
>FLP3FBN01CMRIY length=261 xy=0962_1096 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGCCGCATGCCCATCCTTTACCGGATCGCTCCTTTGACATACCGATCATGC
GGTCGGTATGTGTTATGAGGTATTAGACGGAATTTCTCCCGATTATCCCTCTGTAAAGGG
CAGGTCGCATACGTGTTACTC
>FLP3FBN01DHESU length=263 xy=1311_1436 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCTTGGTGGGCCGTTGCCCCACCAAC
TAGCTAATGTCACGCATGCCCATCCGTTGCCGGAATCGCTTCCTTTGGCCGCAGGGCCAT
GCAGCCCCGCGGCATTACGCGGTATTAGACGGGATTTTCTCCCGATTATCCCCCTGCAAC
GGGCAGGTCGCATACGCGTTACT
>FLP3FBN01AMVJR length=287 xy=0143_1701 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGTTATGCGGTATTAGACGGGGTTTCCCCCGCTTATCCCCCTGCCACGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCAACGG
>FLP3FBN01D9BZ0 length=253 xy=1629_1646 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
TAGCTAATCAGACGCGGATCCATCGTATACCACCAGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTCTCCACGCG
>FLP3FBN01CZIJ2 length=272 xy=1107_2080 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGAGAAAATGCTC
ACTCATGATACCATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCA
GGTTGGATACGCGTTACTCACCCGTGCGCCGG
>FLP3FBN01A8P22 length=242 xy=0392_1148 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAAC
TAGCTAATCAGACGCGAGCTCATCTCAGAGCGATAAATCTTTGACAGTCAGAGAGATGCC
TCCCAACTGTATCATGCGGTATTAGCGGCTGTTTCCAGCCGTTATTCCCCACTCCAAGGC
AG
>FLP3FBN01DVDLD length=268 xy=1470_1791 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAAC
TGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGG
AATCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCA
GGTTATCCACGCGTTACTCACCCGTCCG
>FLP3FBN01DYU35 length=244 xy=1510_0627 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAAC
TAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGA
CATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCT
TGGG
>FLP3FBN01AJSD6 length=276 xy=0108_1004 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCGCCCTCTCAGGCCGGCTACCGATCGTCGCCTTGGTATGCCGTTACCATACCAAC
TAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTAAGGAAAAGACATGCAT
CTTTCCCTGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGTAAGGCAG
GTTACCCACGCGTTACTCACCCGTTCGCCACTAAGA
>FLP3FBN01B5GN0 length=255 xy=0765_0782 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACTCTCTCAAGCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCTTACCAAC
TAGCTAATCAGACGCGGGTCCATCCCATAGCGATAAATCTTTGAAGAAAAAGACATGCGT
CTCTTTCCTGCTATGCGGTATTAGCAGCCGTTTCCAGCTGTTGTCCCCCTCTATGGGGCA
GGTTACCCACGCGTT
>FLP3FBN01CIJW1 length=281 xy=0914_1219 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01D44U7 length=268 xy=1581_2385 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCCTTGGTAGGCCTTTACCCCACCAAC
TAGCTAATCAGACGCGGAACCATCGTATACCACCAGAGTTTTTCACACCGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTTTCCACGCGTTACTCACCCGTCCG
>FLP3FBN01DTYVK length=274 xy=1454_1598 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
TAGCTAATGCGCCGCATGCCCATCCTTGTCCGGATAAATCCTTTGATCGAATTCTCATGC
GAGAACCCGATGTCACGCGGTATTAGACCGGATTTCTCCGGCTTATCCCCCTGACAAGGG
TAGGTTGCATACGTGTTACTCACCCGTGCGCCGG
>FLP3FBN01DKJQE length=278 xy=1347_0340 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTGGTCTTGGTGAGCCGTTACCCACCAACT
AACTAATACAACGCATGCCCATCCATTACCACCGGAGTTTTCAACCCAAGAAGATGCCTC
CCTGGATGTTATGGGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTAATGGGTAGG
TTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCGAC
>FLP3FBN01D6990 length=255 xy=1606_0310 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGCTGCCCCGCCAAC
AAGCTAATCAGACGCGGGCCCATCCCATACCGCGAAAGCTTTCCATGCAGGGGCATGCGC
CCCCGCATGTGCATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTATGGGGCA
GGTTACCCACGCGTT
>FLP3FBN01A5MND length=258 xy=0357_0087 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAAC
TGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACCGCTCCATGCGG
AGCTGTGCGCTCATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGGTATACGGCA
GGTTATCCACGCGTTACT
>FLP3FBN01AFDPD length=268 xy=0058_0143 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGCGAGAATGCTC
ACTCATGATACTATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCA
GGTTGGATACGCGTTACTCACCCGTGCG
>FLP3FBN01C9021 length=281 xy=1227_1131 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGTCTTGGTGGTCCGTTACACCGCCAAC
TAACTAATGCGACGCATGCCCATCCTTCACCGAAATTCTTTCCCCCTCGGAAGATGCCTC
CCAAGGAGTATATGCGGTATTAGGCGAAATTTCTTCCGGTTATCCCGCTGTAAAGGGTAG
GTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCAG
>FLP3FBN01D6MDJ length=268 xy=1598_2101 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGAGAAAATGCTC
ACTCATGATACCATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCA
GGTTGGATACGCGTTACTCACCCGTGCG
>FLP3FBN01BD6JX length=293 xy=0454_1819 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGGTCGACCTCTCAGTCCGGCTACCCATCGCGGGCACGGTGGGCCGTTACCCCGCCGTC
TACCTAATGGGCCGCGACCCCATCCCGCGCCGTCTCGGCTTTCCCGGCAGGGCCATGGGG
CCCTCCCGGAGTATCCGGTATTAGCCGCCGTTTCCAGCGGTTGTCCCGGAGCGCGGGGCA
GGTTGGTCACGTGTTACTCACCCGTTCGCCACTCTATGTCCGCCCGAGGGCGG
>FLP3FBN01CKTYU length=258 xy=0940_1060 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTTGCTTTGGTAGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGGTCCATCGTATACCTCCGGAAATTTTCACACTCTGTCATGCGA
CAGTGTGCGCTTATGCGGTATTAGCAGTTGTTTCCAACTGTTATCCCCCTGTATACGGCA
GGTTACCCACGCGTTACT
>FLP3FBN01EIOYS length=262 xy=1736_0082 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAAC
CAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGA
AGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGC
GCGGGCAGGTTCCCTACGTGTT
>FLP3FBN01EMEL5 length=242 xy=1778_1259 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGACCGTGTTCCAGTTCCAGTGTGA
CTGATCATCCTCTCAGACCAGTTAACCATCGTCGCCTTGGTAGGCCTTTACCCCACCAAC
CAGCTAATGGTACGCGGACTCATCAAAAAGCAACAGCATATGCAGAGGCCATCTTTCCCA
CATAAGATAGATATGCAGCGTATTCGGTATTAGCAGCCGTTTCCAGCTGTTATCCCAAAC
TT
>FLP3FBN01ACUB7 length=268 xy=0029_0517 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCGG
GACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01BGEG7 length=277 xy=0479_3001 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
CCGTTCACCCTCTCAGGTCGGCTACCGATCGTCGCCTTGGTGGGCCGTTACCCCACCAAC
TAACTAATACAACGCATGCCCATCCATTACCACCGGAGTTTTCAACCCAAGAAGATGCCT
CCCTGGATGTTATGGGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTAATGGGTAG
GTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCA
>FLP3FBN01DLFJ6 length=281 xy=1357_0628 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01DDMH6 length=264 xy=1268_0924 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGCTGCCTTGGTGGGCCGTTACCCCGCCAAC
TAGCTAATCAGACGCGGAACCATCATATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGGTATACGGCA
GGTTTTCCACGCGTTACTCACCCG
>FLP3FBN01C7M7W length=267 xy=1200_0442 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCGTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTCAAGGAAAAGACATGCAT
CTTTCCCTGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGTAAGGCAG
GTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01EFF35 length=271 xy=1699_0195 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAAC
TACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGG
TTCTGGGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCA
GGTTGCCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01D8F9Q length=237 xy=1619_1484 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGGTCAACCTCTCAGTCCGGCTACTGATCGTCGCCTAGGTGGGCCTTTACCCCGCCAAC
TAGCTAATCAGACGCGGATCCATCGTATACCACCAGAGTTTTTCACACTGTTCCATGCGG
AACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACG
>FLP3FBN01APXMT length=255 xy=0178_1011 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGCCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCTCACCAAC
CAGCTAATCAGACGCGGGTCCATCTCATACCACCGGAGTTTTTCACACTTGACCATGCAG
TCCTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATGAGGCA
GGTTACCCACGCGTT
>FLP3FBN01DQIRW length=271 xy=1415_0506 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTAGGCCGTTACCCCACCAAC
TACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCACCAAACCATGCGG
TTTTGTGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCA
GGTTGCCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01D1PYD length=276 xy=1542_2835 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAAC
CAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGTACCATGCGG
TACTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCA
GGTTACCCACGCGTTACTCACCCGTCCGCCACTAAG
>FLP3FBN01A6R98 length=277 xy=0370_0798 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGG
CAGGTTGCATACGTGCTACTCACCCGTGCGCCGGTCG
>FLP3FBN01DEJQB length=271 xy=1278_3025 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCGCCCTCTCAGGTCGGCTACTGATCGTTGGCTTGGTAGGCTTCTACCCCACCAAC
TACCTAATCAGACGCGGGCCCATCTTACACCACCTCAGTTTTTACCTCTGTACCATGCGG
TACTGGGGTCTTATGCGGTATTAGCAATCATTTCTAATTGTTATCCCCCTGTGTAAGGCA
GGTTGCCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01CUA96 length=271 xy=1048_1036 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAAC
TACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGG
TTCCGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCA
GGTTGCCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01DFTRG length=244 xy=1293_1242 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAAC
TAGCTAATGCACCGCAGGTCTATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGA
CATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCT
TGGG
>FLP3FBN01CZE2X length=244 xy=1106_1671 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAAC
TAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGA
CAAGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCT
TGGG
>FLP3FBN01B1OGH length=254 xy=0722_0383 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGTTCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTAGGCCTTTACCCCACCAAC
TAGCTAATGCGCCGCAGGCTCATCCATCAGTGATGCCAGGAGCATCTTTAAACTTTCGTC
CTATTCGGTATTAGCGATCGTTTCCAATCGTTGTCCCCGTCTGATGGGCAGATCACCTAC
GTGTTACTCACCCG
>FLP3FBN01A00UU length=276 xy=0304_2308 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGCCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAACT
ACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGTC
CCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGGC
AGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>FLP3FBN01DL2J4 length=268 xy=1364_1762 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTCTACCCCGCCAAC
AAGCTAATCAGACGCGGGTCCATCGTATACCACCGGAGTTTTTCACACCAGACCATGCGA
TCCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01D4O1Y length=283 xy=1576_2372 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGTCTTGGTGGTCCGTTACACCGCCAAC
TAACTAATGCGACGCATGCCCATCCTTCACCGAAATTCTTTCCCCCTCGGAAGATGCCTC
CCAAGGAGTATATGCGGTATTAGGCGAAATTTCTTCCGGTTATCCCGCTGTAAAGGGTAG
GTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCAAAG
>FLP3FBN01EOUWL length=271 xy=1806_0995 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCCCGCCAAC
TACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCGCTGTACCATGCGG
TACTGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCA
GGTTGCCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01EEO6R length=272 xy=1690_2161 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGAGAAAATGCTC
ACTCATGATACCATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCA
GGTTGGATACGCGTTACTCACCCGTGCGCCGG
>FLP3FBN01BV60P length=275 xy=0659_2551 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAAAGGATTACCATGCGG
TATTCCCTATTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGGTACAGGGCAG
GTTGCCCACGCGTTACTCACCCGTCCGCCACTAAG
>FLP3FBN01EGU6E length=268 xy=1715_0836 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
CCGATCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTAAGCCGTTACCTTACCAAC
TAGCTAATCAGACGCGGGTCCATCCTGTACCGCAAAAGCTTTGATACTTCTACCATGCGA
TAAAAGTATTTTATCTCGTATTAGCATACCTTTCGGTATGTTATCCGTGTGTACAGGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01AF9TE length=264 xy=0068_0800 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGCTGTCTTGGTAGGCCGTTACCCTGCCAAC
TAGCTAATCAGACGCGGGTCCATCCCGTACCACCGTGGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGGTATACGGCA
GGTTTTCCACGCGTTACTCACCCG
>FLP3FBN01BI23K length=259 xy=0510_1246 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAGTGTGT
CCGATCACCCTCTCAGGCCGGATACCCGTCATAGCCTTGGTGAGCCATTACCTCACCAAC
AAGCTGATAGGACATAGGCTGATCCTTTAGCGAAAAACTTTCCCCCGTAGGGAGTATCCA
GTATTAATCACCGTTTCCAGTGGCTATCCCAGACTAAAGGGCACATAACCTATGCATTAC
TCACCCGTGCGCCACTAAT
>FLP3FBN01BZKF3 length=258 xy=0698_0177 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCG
GGTTCTCCACGCGTTACT
>FLP3FBN01AX4MJ length=259 xy=0271_2393 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCT
CCCAACGTTGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCA
GGTTCCATACGTGTTACTC
>FLP3FBN01EENPI length=255 xy=1690_0244 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGACTTGGTGGGCCGTTACCCCGCCAAC
TATCTAATCAGACGCGGACCCATCGTGTACCACTATCGTTTTCCGCCCTGTGCCATGCGG
CACTGTGCGCATATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTACACGGCA
GGTTGTCCACGCGTT
>FLP3FBN01A2E42 length=258 xy=0320_1940 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGCCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAAC
CAGCTAATCAGACGCGGGCCCATCCTGCACCGCCGGAGCTTCCCCCGCCGCCCCATGCGG
GGCTGCGGGCATATGCGGTATTAGCAGCCGTTTCCGGCTGTTGTCCCCCAGTGCAGGGCA
GGTTGCCCACGCGTTACT
>FLP3FBN01A0QD0 length=271 xy=0301_1030 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAAC
CAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGA
AGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGC
GCGGGCAGGTTCCCTACGTGTTACTCACCCG
>FLP3FBN01B3PY0 length=264 xy=0745_1450 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGCTACCCCGCCAAC
TGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGG
AATCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCG
>FLP3FBN01B533J length=274 xy=0772_2477 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGGCCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGGGCCGTTACCCCGCCAAC
AACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGA
AGAACTATGCCATCGGGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCAG
GTTGGATACGTGTTACTCACCCGTGCGCCGGTCG
>FLP3FBN01EZQR8 length=271 xy=1930_0966 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGGTG
GAGGACCTTCCTCTCCAGAACCCCTAGACATCGTCGCCTTGGTCGGGCCGTTACCCCGCC
AACTAGCTAATGTCACCGCATGCCCATCCGTCTCCGGAGTCGCCTCCTTTGGCCGAAGAA
GATGCCTCCCCTCGGCCACATGCGGTATTAGGCCGGATTTCTCCGGGTTATCCCGCTGAA
ACGGGCAGATTGCATACGCGTTACTCACCCG
>FLP3FBN01A2G7F length=251 xy=0321_0521 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGGTCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGAGCCTTTACCTCACCAAC
TAGCTAATCAGACGCGAGGCCATCTTTCAGCGATAAATCTTTGACATTCAAACCATGCGA
TTTAAATGTATCATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGAAAGGCA
GGTTCCTCACG
>FLP3FBN01CAN8M length=273 xy=0824_2212 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCCCGCCAAC
TACCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACCGGACCATGCGG
TCCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTCTCCACGCGTTACTCACCCGTCCGCCACT
>FLP3FBN01EDLIV length=268 xy=1677_4005 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCT
CCCAACGTTCTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCA
GGTTCCATACGTGTTACTCACCCGTGCG
>FLP3FBN01AN9E6 length=262 xy=0159_0800 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAAC
AAACTAATGCACCGCAGGTCCATCCGCACCCCATCCCCTAAAGGATGTTTCACAGAAAGA
AGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGC
GCGGGCAGGTTCCCTACGTGTT
>FLP3FBN01AIA4D length=255 xy=0091_1595 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTCTCCACGCGTT
>FLP3FBN01DSMJS length=267 xy=1439_0406 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTACT
ATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGCGAGAATGCTCA
CTCATGATACTATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCAG
GTTGGATACGCGTTACTCGCCCGTGCG
>FLP3FBN01DHCC5 length=281 xy=1310_2375 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGACGCATGCCCATTCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01CLZRI length=268 xy=0953_1980 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTAGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAAC
CAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGTCCCATGCAG
GACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01CEU1Z length=243 xy=0872_1061 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGCCAACCTCTCAGTCCGGCTACTGATCGTTGCCTTGGTGGGCCGTTGCCCCGCCAAC
TAGCTAATCAGACGCGAGTCCATCTCAGAGCGATAAATCTTTGGCAGTCAGAGCCATGCG
ACCCAACTGCATTATGCGGTATTAGCAGCTGTTTCCAGCTGTTATTCCCCACTCCAAGGC
AGG
>FLP3FBN01DRJEK length=271 xy=1426_2922 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCCCGCCAAC
TACCTAATCAGACGCGGATCCATCCTGTACCACCGGAGTTTTTCACACCGGACCATGCGG
TCCTGTGCGCTTATGCGGTATTAGCAATCATTTCTAATTGTTATCCCCCTGTGTAAGGCA
GGTTGCCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01A7RFW length=259 xy=0381_1306 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGG
ACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCA
GGTTGGATACGTGTTACTC
>FLP3FBN01DYJU9 length=254 xy=1506_2435 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCGCCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCAGTATACGGCAG
GTTCTCCACGCGTT
>FLP3FBN01BLL5V length=274 xy=0539_0481 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCACGGTGGGCCGTTACCCCGCCGTC
CAGCTAATGCGCCGCATGCCCATCCTTCCCCGATGAATCTTTCCTCCTCCGGAGATGCCT
CCGGAGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGGGAAGGGCAG
GTTGCATACGTGTTACTCACCCGTGCGCCAGTCG
>FLP3FBN01DUUQ8 length=288 xy=1464_1954 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTAC
TGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCGGGGATGCCC
CCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGTAAGGCAG
GTTGCCCACGCGTTACTCACCCGTCCGCCGCTCAGTCACAAAAATCTT
>FLP3FBN01EABFG length=263 xy=1640_2506 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
CAGCTAATACAACGCATGCCCATCATCAACCACCGGAGTTTTCAACCCCATGCGATGCCG
CATGAGATATTATGGGGTATTAGTACCAATTTCTCAGTGTTATCCCCCTGTTGATGGTAG
GTTGCATACGCGTTACGCACCCG
>FLP3FBN01CZPD2 length=265 xy=1109_2744 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAAC
CAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGA
AGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGC
GCGGGCAGGTTCCCTACGTGTTACT
>FLP3FBN01BU66X length=271 xy=0648_1175 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACCCATCGTCGCCTTGGTAGGCCATTACCCTACCAAC
TAGCTAATGGGACGCGAGTCCATCTTTCAGCGTCAGGTCTCCCCAACTTTTCCTATATAA
ACATGCACTTATATAACTCATCCGGCATTAGCTCACCTTTCGGTAAGTTGTTCCAGTCTA
AAAGGCAGGTCACTCACGTGTTACTCACCCG
>FLP3FBN01EDUZ7 length=282 xy=1680_4005 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTA
CTGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCAAGGATGCC
CCCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCA
GGTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGGCAG
>FLP3FBN01A5RCL length=248 xy=0358_2083 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGTTCATCCTCTCAGACCGGCTACTGATCGCAGCCTTGGTGGGCCGTTACCCCGCCAAC
TAGCTAATCAGACATGAGCCCATCCTTCAGCGATAAATCTTTGATTTCAGGAAGATGCCT
TCCCAAAATGTTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGAAGGGCA
GGTTGCTC
>FLP3FBN01AULX0 length=257 xy=0231_2054 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
CAGCTAATACAACGCATGCCCATCATCAACCACCGGAGTTTTCAACCCCATGCGATGCCG
CATGAGATATTATGGGGTATTAGTACCAATTTCTCAGTGTTATCCCCCTGTTGATGGTAG
GATGCATACGCGTTACG
>FLP3FBN01BULUD length=258 xy=0641_2179 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGCCACCCTCTCAGGTCGGCTACTGATCGCTGCCTTGGTGGGCCGTTACCCCGCCAAC
TAGCTAATCAGACGCGGATCCATCCCATACCACCGGAGTTTTTCACACTGCGCCATGCGG
CGCCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCGGTATACGGCA
GGTTCTCCACGCGTTACT
>FLP3FBN01EZAZQ length=281 xy=1925_0980 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCAAC
CAGCTAATGCGCCGCATGGCCATCCGTAGCCGGTGTTACCCTTTAAACCCCAGGAGATGC
CTCTCGGAGTTATTACGCGATATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTACGGG
CAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01EQG8B length=264 xy=1824_2857 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGG
ACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCA
GGTTGGATACGTGTTACTCACCCG
>FLP3FBN01D7715 length=268 xy=1616_3131 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAAC
TACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGG
TTCCGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCA
GGTTGCCCACGCGTTACTCACCCGTCCG
>FLP3FBN01EYHU5 length=274 xy=1916_0095 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAAC
TACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTTCAGAACCATGCG
GTTCCGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTCGTGTAAGG
CAGGTTGCCCACGCGTTACTCACCCGTCCCGCCG
>FLP3FBN01BTVUO length=255 xy=0633_1262 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTCTCCACGCGTT
>FLP3FBN01B8NKS length=268 xy=0801_2250 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAG
GACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01BLVU7 length=264 xy=0542_0769 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGTCGGCTACCGATCGTCGCCTTGGTGGGCTGTTACCCCTCCAAC
TAGCTAATCGGACGCGGATCCATCTCTCACCGATTTCTCTTTCCCTCTCTCCTCATGCGA
CAAAAGAGGCTCATGCGGTATTAGCAACCGTTTCCAGCTGTTATCCCCCTGTGAAAGGCA
GGTTATCCACGCGTTACTCACCCG
>FLP3FBN01AVPQ2 length=258 xy=0244_0396 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTCATACCACCGGAGTTTTTCACACCAGGACATGCAT
CCCTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATGAGGCA
GGTTACCCACGCGTTACT
>FLP3FBN01EAFRV length=261 xy=1641_4041 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
TAACTAATGCGCCGCATGCCCATCCATGACCGGATCGCTCCTTTGACTCCCGAGAGATGT
CTCCCGGGGGTGTTATGCGGTATTAGACGGAATTTCTCCCGCTTATCCCCCTGTCATGGG
CAGGTTGCATACGTGTTACTC
>FLP3FBN01DQ4XV length=268 xy=1422_0561 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAG
GACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01EVPJE length=281 xy=1884_1144 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01D1105 length=265 xy=1546_2103 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGCCAACCTCTCAGGCCGGCTACCCATCGTCGCCTTGGTAGGCCATTACCCTACCAAC
TAGCTAATGGGACGCGAGTCCATCTTTCAGCGTCAGGTCTCCCCAACTTTTCCTATATAA
ACATGCACTTATATAACTCATCCGGCATTAGCTCACCTTTCGGTAAGTTGTTCCAGTCTA
AAAGGCAGGTCACTCACGTGTTACT
>FLP3FBN01CZHBB length=265 xy=1107_0469 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGGCCACCCTCTCAGGCCGGCTACCCATCGTCGCCTTGGTAGGCCATTACCCTACCAAC
TAGCTAATGGGACGCGAGTCCATCTTTCAGCGTCAGGTCTCCCCAACTTTTCCTATATAA
ACATGCACTTATATAACTCATCCGGCATTAGCTCACCTTTCGGTAAGTTGTTCCAGTCTA
AAAGGCAGGTCACTCACGTGTTACT
>FLP3FBN01DWQWF length=268 xy=1486_0157 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
CCGATCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTAAGCCGTTACCTTACCAAC
TAGCTAATCAGACGCGGGTCCATCCTGTACCGCAAAAGCTTTGATACTTCTACCATGCGA
TAAAAGTATTTTATCTCGTATTAGCATACCTTTCGGTATGTTATCCGTGTGTACAGGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01CHR1X length=258 xy=0905_1971 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGG
ACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCA
GGTTGGATACGTGTTACT
>FLP3FBN01EHASU length=281 xy=1720_0604 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGTCTTGGTGGTCCGTTACACCGCCAAC
TAACTAATGCGACGCATGCCCATCCTTCACCGAAATTCTTTCCCCCTCGGAAGATGCCTC
CCAAGGAGTATATGCGGTATTAGGCGAAATTTCTTCCGGTTATCCCGCTGTAAAGGGTAG
GTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCAG
>FLP3FBN01ARODF length=265 xy=0198_0401 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTACCAAC
TAGCTAATCAGACGCGAGGCCATCTCTCAGCGATAAATCTTTGGTACCAGTACCATGCGA
TACCAGTACGTTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCACTGAAAGGCA
GGTTCCTCACGCGTTACTCACCCCG
>FLP3FBN01DM0L1 length=254 xy=1375_0839 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGTTCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTAGGCCTTTACCCCACCAAC
TAGCTAATGCGCCGCAGGCTCATCCATCAGTGATGCCAGGAGCATCTTTAAACTTTCGTC
CTATCCGGTATTAGCGATCGTTTCCAATCGTTGTCCCCGTCTGATGGGCAGATCACCTAC
GTGTTACTCACCCG
>FLP3FBN01AMXVM length=277 xy=0144_0624 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAGTGGTG
GGGGACCTTCCTCTCAGAACCCCTAGACATCGTCGGCTTGGTGGGCCGTTACCCCGCCAA
CTACCTAATGTCGCGCGTGCCCGTCCCGTACCACCGGAATTTTAAATCGAGAGCCATGCG
GCTCTCGAGTATCATGGGATGTTAGTCCACGTTTCCGCGGGTTATCTCCCGGTACGGGTT
GGTTGCACACGTGTTACTCACCCGTGCGCCGGTCGCC
>FLP3FBN01EN0R5 length=256 xy=1796_2915 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGATCACCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCTCGCCAA
CTAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCTGTAAAGGATGCTTCACAGAAAG
AAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTG
CGCGGGCAGGTTCCCT
>FLP3FBN01ANMU2 length=269 xy=0152_0236 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGTCCACCCTCTCAGGCCGGCTGCTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAAC
TGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAAAGGATTACCATGCGG
TATTCCCTATTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAG
GTTATCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01DH6AB length=272 xy=1320_0193 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCT
CCCAACGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCA
GGTTCCATACGTGTTACTCACCCGTGCGCCGG
>FLP3FBN01AMFQA length=282 xy=0138_1680 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCACGGTGGGCCGTTACCCCGCCGTC
CAGCTAATGCGCCGCATGCCCATCCTTCCCCGATGAATCTTTCTTCCTCCGGTACGATGC
CTCCGGAGGATATACGCGGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGGGAAGGG
CAGGTTGCCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01ANJSA length=275 xy=0151_0344 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTGCTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAAC
TGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAAAGGATTACCATGCGG
TATTCCCTATTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGGTACAGGGCAG
GTTGCCCACGCGTTACTCACCCGTCCGCCACTAAG
>FLP3FBN01BAA4G length=281 xy=0410_1342 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01DZYQE length=281 xy=1522_2820 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01EAKES length=251 xy=1643_1858 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCTCTCTCAAGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
AAGCTAATCAGACGCGGGACCATCCTGTACCACCGGAGTTTTCCCTGCGCTGCCATGCGG
CAGCGCAGGATTATGCGGTATTAGCAGCTGTTTCCAGCTGTTATTCCCCACTCCAAGGCA
GGTTACTCACG
>FLP3FBN01B495L length=268 xy=0763_0535 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
CCGATCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTAAGCCGTTACCTTACCAAC
TAGCTAATCAGACGCGGGTCCATCCTGTACCGCAAAAGCTTTGATACTTCTACCATGCGA
TAAAAGTATTTTATCTCGTATTAGCATACCTTTCGGTATGTTATCCGTGTGTACAGGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01BS5XB length=271 xy=0625_0429 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAGTGTGG
GGGGCCTTCCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAAC
CAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCT
CTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAG
GTTGCATACGCGTTACGCACCCGTGCGCCGG
>FLP3FBN01DRO6W length=244 xy=1428_2230 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAAC
TAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGA
CATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCT
TGGG
>FLP3FBN01B08NI length=277 xy=0717_0380 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>FLP3FBN01BYZHF length=262 xy=0691_1681 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAAC
AAACTAATGCACCGCAGGTCCATCCGCACCCCATCCCCTAAAGGATGTTTCACAGAAAGA
AGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGC
GCGGGCAGGTTCCCTACGTGTT
>FLP3FBN01AVTON length=233 xy=0245_1397 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACTCTCTCAAGCCGGCTACTGATCGTTGCTTTGGTAGGCCATTACCCTGCCAAC
TGGCTAATCAGACGCGGGGCCATCGTATGCCGATAACTCTTTTCACACCATGCCATGCAG
CATTGTGTGCTTATGCGGTATTAGCAGTCATTTCTGACTGTTGTCCCCTTGCA
>FLP3FBN01BR8CU length=256 xy=0614_1980 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGGTCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
CAGCTAATGCGCCATAAGTCCATCCTCTACCAGTGCCTTGCAGCACTTTTAATACGGTCA
CCATGCAGTGTCCATACCTATGCGGTCTTAGCTACCGTTTCCAGCAGTTATCCCCCTGTA
AAGGCCAGGTTACTTA
>FLP3FBN01C2EP9 length=268 xy=1140_1919 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAG
GACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01EMTB4 length=278 xy=1782_3954 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTAC
TGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCGGGGATGCCC
CCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAG
GTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGG
>FLP3FBN01EYX93 length=264 xy=1921_0889 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCAACCTCTCAGTCCGGCTACTGATCGTCGCCTAGGTGGGCCGTTACCCCGCCTAC
CAGCTAATCAGACGCGAGGCCATCTTCCAGCGATAAATCTTTGGTATCAGGGTCATGCGG
CCCCAATACTTTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGGAAGGCA
GGTTCCTCACGCGTTACTCACCAG
>FLP3FBN01AE4FR length=263 xy=0055_0421 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCACTCTCTCAAGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAACA
AGCTAATCAGACGCGGGTCCATCTTACACCACTAATGTTTTTCACTCTGTCCCATGCGGG
ACTGTGCGCTTATGCGGTATTAGCACCTATTTCTAACTGTTATCCCCCTGTGTAAGGCAG
GTTACCCACGCGTTACTCACCCG
>FLP3FBN01DX6WZ length=271 xy=1502_2033 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAAC
CAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGA
AGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGC
GCGGGCAGGTTCCCTACGTGTTACTCACCCG
>FLP3FBN01CHYG5 length=264 xy=0907_2103 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTTGGCTTGGTGGGCCGTTACCCCGCCAAC
TAGCTAATAGGACGCATGCCCATCTGATACCTCGAATGATTTAATTATTAAAAGATGCCT
TCAAATAATATTATGGGGTGTTAATCCACGTTTCCATGGGCTATCCCCCTGTATCAGCCA
GGTTGCATACGCGTTACTCACCCG
>FLP3FBN01D75ZT length=279 xy=1616_0455 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGCCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
CAGCTAATCAGACGCGGGCCCATCTTGTACCACCGGAGTTTTCAAAGGATTACCATGCGG
TATTCCCTATTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGGTACAGGGCAG
GTTGCCCACGCGTTACTCACCCGTCCGCCACTAAGACAG
>FLP3FBN01B67MI length=268 xy=0785_0456 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAAC
TACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGG
TTCCGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCACTCCGAGGCA
GGTTCCTCACGCGTTACTCACCCGTCCG
>FLP3FBN01C1PP4 length=267 xy=1132_2282 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTGCCAAC
AAGCTAATCAGACGCGGGTCCATCTCGCACCACCGGAGTTTTCAGGGCAGGGGCATGCGC
CCCCTCCCGTTATGCGGTGTTAGCACCTATTTCTGGGTGTTATCCCCCAGTGCGAGGCAG
GTTGCCCACGCGTTACTCACCCGTCCG
>FLP3FBN01CCNA1 length=270 xy=0847_0107 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGGTCCATCCCATACCACCGGAGTTTTTCACACAGGGGCATGCAC
TCCCGTGCGCTTATGCGGTATTAGCACCTGTTTCCAGGTGGTATCCCCCGGTATGGGGCA
GGTTACCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01EEO1S length=268 xy=1690_1982 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCT
CCCAACGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCA
GGTTCCATACGTGTTACTCACCCGTGCG
>FLP3FBN01B39O5 length=256 xy=0751_2439 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAAC
AAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGA
AGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGC
GCGGGCAGGTTCCCTG
>FLP3FBN01DQGNZ length=255 xy=1414_1869 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCTCTCTCAAGCCGGCTACTGATCGTTGCCTTGGTGGGCCGTTACCCCGCCAAC
AAGCTAATCAGACGCGGGACCATCCTGTACCACCGGAGTTTTCCCTGCGCTGCCATGCGG
CAGCGCAGGATTATGCGGTATTAGCAGCCATTTCTGGCTGTTGTCCCCCGGTACAGGGCA
GGTTCCCCACGCGTT
>FLP3FBN01DD2LQ length=243 xy=1273_1308 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCATTACCTCACCAAC
TAGCTAATCAGACGCGAGGCCATCTTTCGGCGATAAATCTTTGATGTCCAGTGCCATGCG
ACACAGACATATTATGCGGTATTAGCAGCTGTTTCCAGCTGTTATTCCCCACTCCAAGGC
AGG
>FLP3FBN01CKK5R length=270 xy=0937_1933 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGGTCCCTCCTATACCACTATCGTTTTTCACACAGGGCCATGCGG
CCCCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATAGGGCA
GGTTCCCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01C1EUH length=287 xy=1129_0471 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
TAGCTAATGCGCCGCATGGCCATCCGTAGCCGGTGTTACCCTTTAAACCCCAAGAGATGC
CTCTCGGAGTTATTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTACGGG
CAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCGGAAG
>FLP3FBN01AXLO6 length=281 xy=0265_2440 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01BAHPU length=244 xy=0412_1696 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAAC
TAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGA
CATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCT
TGGG
>FLP3FBN01B85QO length=261 xy=0807_1214 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAAC
TAGCTAATGTCACGCATGCCCATCCGTTGCCGGAATCGCTCCCTTTGGCCGCAGGGCCAT
GCAGCCCGCGGCATTACGCGGTATTAGACGGGATTTCTCCCGATTATCCCCCTGCAACGG
GCAGGTCGCATACGCGTTACT
>FLP3FBN01DOCE5 length=250 xy=1390_1359 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGCCGTATCTCAGTCCCAATGTGGC
CGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAACT
AGCTAATCAGACGCGAGGCCATCTTTCAGCGATAAATCTTTGACATAAATGCCATGCGAC
ACCTATGTGTTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGAAAGGCAG
GTTCCTCACG
>FLP3FBN01B8169 length=277 xy=0806_0723 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
TAGCTAATGCGCCGCATGGCCATCCGTAGCCGGTGTTACCCTTTAAACCCCAGGAGATGC
CTCTCGGAGTTATTACGCGATATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTACGGG
CAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCG
>FLP3FBN01D5N1L length=271 xy=1587_2663 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAAC
TACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGG
TTCCGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCA
GGTTGCCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01BKT0E length=268 xy=0530_0860 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCACCAAC
TGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGG
AACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGGTATACGGCA
GGTTATCCACGCGTTACTCACCCGTCCG
>FLP3FBN01D0BE4 length=268 xy=1526_2878 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCACCAAC
TGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGG
AACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCA
GGTTCTCCACGCGTTACTCACCCGTCCG
>FLP3FBN01BSMR0 length=244 xy=0619_0190 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCTTTACCCCACCAAC
TAGCTAATCAGACGCGGGTCCATCGTATACCACCTCAGTTTTTCACACTGCGCCATGCGC
CGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTT
>FLP3FBN01BYGK2 length=247 xy=0685_1764 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGTTCACCCTCTCAGGTCGGCTATGTATCGTCGCCTAGGTGAGCCTTTACCTCACCTAC
TAGCTAATACAACGCAGGTCCATCTGGTAGTGATGCAGTTGCATCTTTCAAACTTCAATC
ATGCAATCAAAGATATTATGCGGTATTAGCTATCGTTTCCAATAGTTATCCCCCGCTACC
AGGCAGG
>FLP3FBN01D868W length=268 xy=1627_3678 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAG
GACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01EAMCO length=233 xy=1644_0278 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACTCTCTCAAGCCGGCTACTGATCGTTGCTTTGGTAGGCCATTACCCTGCCAAC
TGGCTAATCAGACGCGGGGCCATCGTATGCCGATAACTCTTTTCACACCATGCCATGCAG
CATTGTGTGCTTATGCGGTATTAGCAGTCATTTCTGACTGTTGTCCCCCTGCA
>FLP3FBN01B0H89 length=270 xy=0708_3035 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCTTTGGTAGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGGTCCATCTCATACCGATAAATCTTTTCCGTCCGTATCATGCGA
TACTAGCGGGTTATGCGGTATTAGCGGTCGTTTCCAACTGTTATCCCCCTGTATGAGGTA
GGTTACCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01CPQSL length=253 xy=0996_0851 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCAACCTCTCAGTCCGGCTACTGATCGTTGACTTGGTAGGCCATTACCCCACCAAC
TATCTAATCAGACGCAAGCCCATCTATTAGCGGATTTCTCCTTTCCCACTAGAACCATGT
GGTTCCTATGGCATATGCGGTATTAGCAATGATTTCTCACTGTTATTCCCCTCTAATAGG
CAGGTTGCTTACG
>FLP3FBN01DZX6B length=270 xy=1522_2097 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGTGGACTTGGTGAGCCGTTACCTCACCAAC
TATCTAATGGAACGCATGCCCATCTATCAGCGATAATTCTTTAACAAATATTTCCATGTG
GAACCCCTGTTTTATGCGGTATTAGTCCGACTTTCGCCGGGTTATTCCCTCTGATAGGTA
GGTTGCATACGCGTTACTTCACCCGTCGCG
>FLP3FBN01ARBPM length=279 xy=0194_0376 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
TAGCTAATGCGCCGCATGGCCATCCGCAGCCGGTAAACCTTTAAACCACCGGAGATGCCT
CCGACGGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCA
GGTTCCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01B1VVL length=275 xy=0724_1807 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGAAAAAATGCTC
ACTCATGATACCATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCA
GGTTGGATACGCGTTACTCACCCGTGCGCCGGTCG
>FLP3FBN01BAWPK length=253 xy=0417_0646 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTGCCAAC
TAGCTAATCAGACGCGGGCCCATCTCATACCACCGGAGTTTTTGCCGCTGCACCATGCGG
TGCTGTGGCCTTATGCGGTATTAGCAGCCATTTCTGGCTGTTATCCCCCTGTATGAGGCA
GGTTGCCCACGCG
>FLP3FBN01EUT65 length=264 xy=1874_1487 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCATTACCCCACCAAC
TAACTAATCAGACGCGGGTCCATCTCATACCACCGGAGTTTTTACCACTGCACCATGCGG
CGCTGTGGTCTTATGCGGTATTAGCAGCCATTTCTAACTGTTATCCCCCTGTATGAGGCA
GGTTACCCACGCGTTACTCACCCG
>FLP3FBN01EPDN4 length=268 xy=1812_0738 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
CCGATCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTAAGCCGTTACCTTACCAAC
TAGCTAATCAGACGCGGGTCCATCCTGTACCGCAAAAGCTTTGATACTTCTACCATGCGA
TAAAAGTATTTTATCTCGTATTAGCATACCTTTCGGTATGTTATCCGTGTGTACAGGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01BSTU3 length=257 xy=0621_1181 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTGCCCCGCCAAC
TACCTAATCGGACGCGAGCCCATCTTTCAGCGGATTGCTCCTTTGATTATCTCACCATGC
GGCAAAATAATGTCATGCGGTATTAGCGTTCGTTTCCAAACGTTATCCCCCTCTGAAAGG
CAGGTTGCTCACGCGTT
>FLP3FBN01DP84W length=275 xy=1412_0302 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
CAGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAAAGGATTACCATGCGG
TATTCCCTATTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGGTACAGGGCAG
GTTGCCCACGCGTTACTCACCCGTCCGCCACTAAG
>FLP3FBN01A9328 length=251 xy=0408_0418 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTC
AAGCTAATGTCACGCGAGCCTATCCTCATCCGACGAATCTTTAGATGGATTCAGATGCCT
GATTCCATCACCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCA
AGTTGCTCACG
>FLP3FBN01A6SQ3 length=251 xy=0370_1405 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCTCACCAAC
TAGCTAATCAGACGCGAGGCCATCTTCCAGCGATAAATCTTTGACGTCGGAGTCATGCGG
CTCCAACGCATCATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCAGTGCGGGGCA
GGTTGCCCACG
>FLP3FBN01C6I0F length=281 xy=1187_1581 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTGGTCTTGGTGAGCCGTTACCCCACCAAC
TAACTAATACAACGCATGCCCATCCATTACCACCGGAGTTTTCAACCCAAGAAGATGCCT
CCCTGGATGTTATGGGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTAATGGGTAG
GTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCGACAA
>FLP3FBN01D1YSR length=270 xy=1545_2009 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCGCCGACT
AGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTC
CCAACGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCTGCTGCGGGCAGG
TTTCATACGTGTTACTCACCCGTGCGCCGG
>FLP3FBN01DAS79 length=289 xy=1236_0743 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGCGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGCCCATCCGTGGCCGGGATTGCTCCCTTTGGCGGCCCGGGGAT
GCCCCAAGGCCGCGTTACGCGGTATTAGACGGGGTTTCCCCCGCTTATCCCCCTGCCACG
GGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCAACGG
>FLP3FBN01CR7MF length=251 xy=1024_1285 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTAGGTGGGCCGTTACCCCGCCTAC
CAGCTAATCAGACGCGAGGCCATCTTCCAGCGATAAATCTTTGGTGTCTCGATGATGCCA
TCGAAACACATCATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGGAAGGCA
GGTTCCTCACG
>FLP3FBN01D7F5H length=264 xy=1607_3827 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
CAGCTAATCAGACGCGAGGCCATCTCTCAGCGATAAATCTTTGGTACCAGTACCATGCGA
TACCAGTACGTTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCACTGAAAGGCA
GGTTCCTCACGCGTTACTCACCCG
>FLP3FBN01CSG40 length=281 xy=1027_1330 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01BBORZ length=236 xy=0426_0157 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGAGCCGTGTCTCAGTCCCAATGTGG
CCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAAC
TAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGA
CATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGT
>FLP3FBN01BY17V length=272 xy=0692_1129 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCT
CCCAACGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCA
GGTTCCATACGTGTTACTCACCCGTGCGCCGG
>FLP3FBN01AQTYD length=268 xy=0188_1939 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCCTTGGTAGGCCATTACCCCACCAAC
TAGCTAATCAGACGCGGAACCATCGTATACCACCAGAGTTTTTCACACCGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTTTCCACGCGTTACTCACCCGTCCG
>FLP3FBN01BGT75 length=264 xy=0484_2931 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTTGCTTTGGTAGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGGTCCATCGTATACCTCCGGAAATTTTCACACTCTGTCATGCGA
CAGTGTGCGCTTATGCGGTATTAGCAGTTGTTTCCAACTGTTATCCCCCTGTATACGGCA
GGTTACCCACGCGTTACTCACCCG
>FLP3FBN01AJFDA length=253 xy=0104_0508 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGATCAACCTCTCAGTCCGGCTACCGATCGTCGCCTTGGTGGGCCGTTACCCCACCAAC
TAGCTAATCGGCCGCGAGCCCATCTTTCAGCGGATTGCTCCTTTGGCTGCATCGGGATGC
CCCATCGCAGCGTTATGCGGTATTAGCAGTCGTTTCCAACTGTTATCCCCCTCTGAAAGG
CAGGTTGCTCACG
>FLP3FBN01D5SZS length=258 xy=1589_0886 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTC
AAGCTAATGTCACGCGAGCCTATCCTCATCCGACGGATCTTTAGATGGAACCAGATGCCT
GATTCCATCGCCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCA
AGTTGCTCACGCGTTACG
>FLP3FBN01EB4D8 length=253 xy=1661_0686 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCTCACCAAC
TAGCTAATCAGACGCGGGTCCATCTCTATCCGTCAAGGCTTTTCACACCGTGCCATGCGG
CACTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTGTCCCCCTGATAGAGGCA
GGTTACCCACGCG
>FLP3FBN01A7HFD length=270 xy=0378_0615 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAAC
AACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGAA
GATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCG
CGGGCAGGTTCCCTACGTGTTACTCACCCG
>FLP3FBN01BZTX6 length=247 xy=0701_0204 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTACCAAC
TAGCTAATCAGACGCGAGGCCATCTCTCAGCGATAAATCTTTGATATATCTGCCATGCGA
CAAACATATATTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGAAAGGCA
GGTTCCT
>FLP3FBN01AQFZV length=264 xy=0184_0233 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCTTTGGTAGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGGCCCATCCTATGGCGATAACTCTTTCTTATTATGACCATGCAA
TCTTAATAAGGTATGCGGTATTAGCAGCGGTTTCCCACTGTTGTCCCCCTCCATAGGGCA
GGTTACCCACGCGTTACTCACCCG
>FLP3FBN01ESUQV length=282 xy=1851_3093 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCGACT
AGCTAATGCGCCGCATGCCCATCCGTGGCCGGGATTGCTCCCTTTGGCGGCCCGGGGATG
CCCCAAGGCCGCGTTACGCGGTATTAGACGGGGTTTCCCCCGCTTATCCCCCTGCCACGG
GCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01C1PAK length=266 xy=1132_1722 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTCCCAATGTGG
CCGGTTGCCCTCTCAGGCCGGCTACTGATCGTCGGTTTGGTGGGCCGTTACCCCGCCAAC
TGCCTAATCGGACGCGAGCCCATCTTTCAGCGGATTGCTCCTTTGATATCAGTGCCATGC
GACTCCGATATTTTATGCGGTATTAGCGTTCGTTTCCAAACGTTATCCCCCTCTGAAAGG
CAGGTTGCTCACGTGTTACTCACCCG
>FLP3FBN01DVCKT length=270 xy=1470_0475 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTACCGATCGTCGCTTTGGTGGGCCTCTACCCCGCCAAC
TGGCTAATCGGACGCGGGTCCATCCCATACCACCGGAGTTTTTCACACTGTATCATGCGT
TACTGTGCGCTTATGCGGTATTAGCAGCCGTTTCCGGCTGGTATCCCCCGGTATGGGGCA
GGTTGCCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01DR7VU length=268 xy=1434_1880 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGCGAGAATGCTC
ACTCATGATACTATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCA
GGTTGGATACGCGTTACTCACCCGTGCG
>FLP3FBN01CZ007 length=284 xy=1113_1449 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTAGACATCGTCGGCTTGGTGGGCCGTTACCCCGCCAAC
TACCTAATGTCGCGCGTGCCCGTCCCGTACCACCGGAATTTTAAATCGAGAGCCATGCGG
CTCTCGAGTATCATGGGATGTTAGTCCACGTTTCCGCGGGTTATCTCCCGGTACGGGGTT
GGTTGCACACGTGTTACTCACCCGTACGCCGGTCGCCACCAGTG
>FLP3FBN01C903U length=266 xy=1227_1160 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGCCGCATGCCCATCCTTTACCGGATCGCTCCTTTGACATACCGATCATGC
GGTCGGTATGTGTTATGAGGTATTAGACGGAATTTCTCCCGATTATCCCTCTGTAAAGGG
CAGGTCGCATACGTGTTACTCACCCG
>FLP3FBN01C68GY length=268 xy=1195_1808 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAAC
CAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGCTTCATGCAA
AGCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCA
GGTTGCCCACGCGTTACTCACCCGTCCG
>FLP3FBN01A0XKK length=277 xy=0303_2146 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTGCTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAAC
TGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAGGGAAAAGCCATGCGG
CTTCCCCCGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGCCAG
GTTGCCCACGCGTTACTCACCCGTCCGCCACTAACAG
>FLP3FBN01D0C70 length=264 xy=1527_1118 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGG
ACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCA
GGTTGGATACGTGTTACTCACCCG
>FLP3FBN01DXGDW length=255 xy=1494_0418 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTCCCAGTGTGG
CCGGTCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
CAGCTAATGCGCCATAAGTCCATCCTCTACCAGTGCCTTGCAGCACTTTTAATACGGTCA
CCATGCAGTGTCCGTACCTATGCGGTCTTAGCTACCGTTTCCAGCAGTTATCCCCCTGTA
AAGGCCAGGTTACTT
>FLP3FBN01D37W3 length=265 xy=1571_0645 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGGCCACCCTCTCAGGCCGGCTACCCATCGTCGCCTTGGTAGGCCATTACCCTACCAAC
TAGCTAATGGGACGCGAGTCCATCTTTCAGCGTCAGGTCTCCCCAACTTTTCCTATATAA
ACATGCACTTATATAACTCATCCGGCATTAGCTCACCTTTCGGTAAGTTGTTCCAGTCTA
AAAGGCAGGTCACTCACGTGTTACT
>FLP3FBN01BPYDP length=255 xy=0588_2235 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
TAGCTAATCAGACGCGAGGCCATCTTTCAGCGATAAATCTTTGACATAAATGCCATGCGA
CACCTATGTGTTATGCGGTATTAGCACCTGTTTCCAGGTGGTATCCCCCGGTATGGGGCA
GGTTACCCACGCGTT
>FLP3FBN01COSQ8 length=245 xy=0985_1794 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCTCACCAAC
AAGCTAATCAGACGCGAGCTCATCCATATCCGATAAATCTTTGATCAAACGAAGATGCCT
TCAAATGATATTATGCGGTATTATCAGTCGTTTCCAACTGTTATCCCCCAGATATGGGCA
GATTG
>FLP3FBN01DSGMM length=268 xy=1437_0924 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCT
CCCAACGTTCTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCA
GGTTCCATACGTGTTACTCACCCGTGCG
>FLP3FBN01CDLEW length=281 xy=0857_3350 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01AHK73 length=281 xy=0083_0801 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01DX2PW length=285 xy=1501_0690 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTAC
TGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCGGGGATGCCC
CCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAG
GTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGGCAGAGAG
>FLP3FBN01BE4YH length=278 xy=0465_1351 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCTCGGTGGGCCGTTACCCCGCCTAC
TGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCGGGGATGCCC
CCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAG
GTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGG
>FLP3FBN01A38MX length=258 xy=0341_0807 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGGTCCATCATATACCACCGGAGTTTTTCACACCAGATCATGCGA
TCCTGTGCGCTTATGCGGTATTAGCAGTTGTTTCCAACTGTTATCCCCCTGTATACGGCA
GGTTACCCACGCGTTACT
>FLP3FBN01ALQ01 length=259 xy=0130_2435 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGG
ACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCA
GGTTGGATACGTGTTACTC
>FLP3FBN01CE5R8 length=272 xy=0875_2678 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGCGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGTGGACTTGGTGAGCCGTTACCTCACCAAC
TATCTAATGGAACGCATGCCCATCTATCAGCGATAATTCTTTAACAAATATTTCCATGTG
GAACCCCTGTTTTATGCGGTATTAGTCCGACTTTCGCCGGGTTATTCCCTCTGATAGGTA
GGTTGCATACGCGTTACTCACCCGTGCGCCGG
>FLP3FBN01CDDGT length=244 xy=0855_1243 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAAC
TAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGA
CATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCT
TGGG
>FLP3FBN01EMNUG length=244 xy=1781_0934 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAAC
TAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGA
CATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCT
TGGG
>FLP3FBN01BT0AE length=264 xy=0634_2916 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
CCGATCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCCACCAAC
TAGCTAATCAGACGCAGGCCCATCCTTTGCCGATAAATCTTTGACCAAACAGCCATGTGA
CTATTTAGTATTATGAGGTTTTAATCATCGTTTCCAATGGCTATCCCTCTGCAAAGGGCA
GGTTGCCTACGCGTTACTCACCCG
>FLP3FBN01D7CTS length=275 xy=1606_3614 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTCAAGGAAAAGACATGCAT
CTTTCCCTGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGTAAGGCAG
GTTACCCACGCGTTACTCACCCGTCCGCCACTAAG
>FLP3FBN01CXKKK length=264 xy=1085_1490 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTTGCTTTGGTAGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGGTCCATCGTATACCTCCGGAAATTTTCACACTCTGTCATGCGA
CAGTGTGCGCTTATGCGGTATTAGCAGTTGTTTCCAACTGTTATCCCCCTGTATACGGCA
GGTTACCCACGCGTTACTCACCCG
>FLP3FBN01CXUT9 length=279 xy=1088_2511 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCAAC
TAACTAATGCGCCGCATGGCCATCCGCAGCCGGTAAACCTTTAAACCTTCAGAGATGCCT
CCAAAAGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCA
GGTTCCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01AR4YL length=264 xy=0203_1419 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGG
ACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCA
GGTTGGATACGTGTTACTCACCCG
>FLP3FBN01A7NE9 length=260 xy=0380_0195 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCAACCTCTCAGTCCGGCTACCGATCGTTGACTTGGTGGGCCGTTACCCCGCCAAC
TATCTAATCGGACGCGAGCCCATCTCCCAGCGGATTGCTCCTTTGATATATCTACCATGT
GATAAATATATGTTATGCGGTATTAGCGTTCGTTTCCAAACGTTATCCCCCTCTGGAAGG
CAGGTTGCTCACGTGTTACT
>FLP3FBN01ELYL8 length=268 xy=1773_1006 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGCCGGCTACTGATCGTCGGTTTGGTGGGCCGCTGCCCCGCCAAC
TGCCTAATCAGACGCGGGCCCATCCCATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01BKQF8 length=274 xy=0529_0342 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCCTCCAAC
CAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTCAGGGAAAAGCCATGCGG
CTTCCCCCGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGCCAG
GTTGCCCACGCGTTACTCACCCGTCCGCCACTCA
>FLP3FBN01D4OI7 length=273 xy=1576_1697 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTACCGATCGTCGCTTTGGTGGGCCGCTGCCCCGCCAAC
TGGCTAATCGGACGCGGGCCCATCCCGTACCACCGGAGTTTTTCACACTTACCCATGCGG
GTATGTGCGCTTATGCGGTATTAGCAGCTGTTTCCAGCTGGTATCCCCCGGTACGGGGCA
GGTTGCCCACGCGTTACTCACCCGTCCGCCACT
>FLP3FBN01AUBJF length=270 xy=0228_0857 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
CAGCTAATTGGACGCGAGCCCATCTCTCAGCGGATTTCTCCTTTGGTCCAGACAGGATGT
CCCATCCGGACGTTATGCGGTATTAGCGATCGTTTCCAATCGTTATCCCCCTCTGAAAGG
CAGGTTGCTCACGCGTTACTCACCCGTCCG
>FLP3FBN01AZSFJ length=267 xy=0290_2077 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCTTGGTAAGCCGTTACCTTACCAAC
CAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCT
CTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAG
GTTGCATACGCGTTACGCACCCGTGCG
>FLP3FBN01C847I length=261 xy=1217_0780 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGATCAACCTCTCAGTTCGGCTACGCATCATTGCCTTGGTAAGCCTTTACCCCACCAAC
TAGCTAATGCGCCGCGGGCCCATCCAAAAGCGGTAGCATAGCCACCTTTTACATAGCTAC
CATGCGGTAACTATGGTTATGCGGTATTAGCACCTGTTTCCAAGTGTTATCCCCCTCTTT
TGGGCAGGTTGCCCACGTGTT
>FLP3FBN01DU20K length=247 xy=1467_0370 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGATCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
TAGCTAATGCGCCGCGGGCCCATCTCATAGCGGATTACTCCTTTAATTGCTACTTCATGC
GAAGCTACAATCTTATGCGGTATTAATCTCCCTTTCGGAAGGCTATTCCCCTCTATGAGG
CAGGTTG
>FLP3FBN01B5XY2 length=251 xy=0770_2732 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCTCACCAAC
TAGCTAATCAGACGCGAGGCCATCTTCCAGCGATAAATCTTTGACGTCGGAGTCATGCGG
CTCCAACGCATCATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGGAAGGCA
GGTTCCTCACG
>FLP3FBN01C354R length=243 xy=1160_2169 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGCTGCCCCGCCAAC
AAGCTAATCAGACGCGGGCCCCTCCCATACCGCCGGAACTTTCCCCAGAAAGGCATGCGC
CTCCCTGGTTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTATGGGGCAG
GTT
>FLP3FBN01C667Q length=258 xy=1195_0180 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCTCACCAAC
CAGCTAATCAGACGCGGGTCCATCTCTATCCGTCTCGGCTTTTCACACCGTGCCATGCGG
CACTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCA
GGTTATCCACGCGTTACT
>FLP3FBN01DJCPD length=271 xy=1333_1919 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGTTAACCTCTCAGTCCCCTATGTATCGTCGCCTTGGTGAGCCGTTACCTCACCAACCA
GCTAATACAACGCATGCCCATCCATAACCACCGGAGTTTTCAATCAAAAGGGATGCCCCT
CTTGATGTTATGGGATATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTATGGGTAGTT
GCATACGCGTTACGCACCCGTGCGCCGGTCG
>FLP3FBN01C1ONI length=271 xy=1132_0892 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTTACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAAC
TACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGG
TTCCGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCA
GGTTGCCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01ARRYC length=280 xy=0199_0946 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGTCCTGTCTCAGTACCAATGTGGG
GGGTTAACCTCTCAGTCCCCCTATGTATCGTGGTCTTGGTGAGCCGTTACCCCACCAACT
AACTAATACAACGCATGCCCATCCATTACCACCGGAGTTTTCAACCCAAGAAGATGCCTC
CCTGGATGTTATGGGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTAATGGGTAGG
TTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCGACAA
>FLP3FBN01AKJ6W length=268 xy=0117_0166 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAG
GACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01DD2QI length=282 xy=1273_1480 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
CCGTTCACCCTCTCAGGTCGGCTACCGATCGTCGCCTTGGTGGGCCGTCACCCCGCCAAC
TAACTAATCGGACGCAGGCCCATCCCTCGCCGTTCCCTTTGGCCCCTCGAGCATGCGCTC
CAAAGGCTTCATGGGGTCTTAATCCCGGTTTCCCGAGGCTGTCCCCCAGCGAGGGGCAGG
TTGCCTACGCGTTACTCACCCGTCCGCCGCTTTCCGCCATCG
>FLP3FBN01AOLSN length=263 xy=0163_0453 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
CAGCTAATACAACGCATGCCCATCATCAACCACCGGAGTTTTCAACCCCATGCGATGCCG
CATGAGATATTATGGGGTATTAGTACCAATTTCTCAGTGTTATCCCCCTGTTGATGGTAG
GTTGCATACGCGTTACGCACCCG
>FLP3FBN01AMKWI length=258 xy=0140_0192 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
CAGCTAATACAACGCATGCCCATCATCAACCACCGGAGTTTTCAACCCCATGCGATGCCG
CATGAGATATTATGGGGTATTAGTACCAATTTCTCAGTGTTATCCCCCTGTTGATGGTAG
GTTGCATACGCGTTACGC
>FLP3FBN01AFTMX length=281 xy=0063_0311 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01DQAD5 length=244 xy=1412_1931 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAAC
TAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGA
CATGCGTCTAGTGTTGTTATCCGGTATTGGCATCTGTTTCCAGGTGTTATCCCAGTCTCT
TGGG
>FLP3FBN01CHB2R length=261 xy=0900_1745 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGCCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCACCAAC
TACCTAATGCGCCGCATGCCCATCCTTTACCGGATCGCTCCTTTGACATACCGATCATGC
GGTCGGTATGTGTTATGAGGTATTAGACGGAATTTCTCCCGATTATCCCTCTGTAAAGGG
CAGGTCGCATACGTGTTACTC
>FLP3FBN01EV5MZ length=273 xy=1889_1529 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
GCCGTCTACCCTCTCAGGCCGGCTACGTATCATCGCTTTGGTGGGCCGTTACCCGCCAAC
TGGCTAATACGCCGCAGGTCCATCCATGTCCACACCCGAAGGTCCTTTAATATAGTCAAG
ATGCCTTGTCTATACCTATCCGGTTTTAGCAGACGTTTCCATCTGTTATCCCGGGTACAT
GGGCAGGTTCCCTACGTGTTACTCACCCGTTCG
>FLP3FBN01CQDN5 length=271 xy=1003_1827 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCTCACCAAC
TACCTAATCAGACGCGGGTCCCTCCTATACCACTATCGTTTTTCACACAGGGCCATGCGG
CCCCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATAGGGCA
GGTTCCCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01EBIVZ length=243 xy=1654_1485 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAAC
TAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGA
CATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCT
TGG
>FLP3FBN01D06GF length=255 xy=1536_2141 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCACCAAC
TAGCTAATCAGACGCGGGTCCATCTCATACCACCGGAGTTTTTCACACTGGATCATGCAA
TCCTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTATATGAGGCA
GGTTACCCACGCGTT
>FLP3FBN01DNQQE length=274 xy=1383_1924 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGCCCATCCGTCACCGGATTGCTCCTTTGACCGCTCCGGGATGC
CCCGGAATGGTGTTACGCGGAATTAGTCGGAATTTCTTCCGGTTATTCCCCTGTGACGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGG
>FLP3FBN01CJ7ZY length=247 xy=0933_1260 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTACCAAC
TAGCTAATCAGACGCGAGGCCATCTCTCAGCGATAAATCTTTGATATCCATACCATGCGA
TACAGATATATTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGAGAGGCA
GGTTCCT
>FLP3FBN01BW1QP length=258 xy=0669_1407 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGTTCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTAGGCCTTTACCCCACCAAC
TAGCTAATGCGCCGCAGGCTCATCCATCAGTGATGCCAGAAGCATCTTTAAACTTTCGTC
CTATCCGGTATTAGCGATCGTTTCCAATCGTTGTCCCCGTCTGATGGGCAGATCACCTAC
GTTTTACTCACCCGTTCG
>FLP3FBN01D9D5Q length=271 xy=1630_0348 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAAC
AAACTAATGCACCGCAGGTCCATCCGCACCCCATCCCCTAAAGGATGTTTCACAGAAAGA
AGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGC
GCGGGCAGGTTCCCTACGTGTTACTCACCCG
>FLP3FBN01D0IW2 length=271 xy=1529_0308 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
CAACTAATCAGACGCGGGTCCATCTCATACCACCGGAGTTTTTCACACCGGACCATGCGG
TCCTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATGAGGCA
GGTTACCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01B6DNW length=269 xy=0775_2586 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
CAGCTAATCAGACGCGGGCCCATCCCATACCACCGGAGTTTTTCACACCAGGCCATGCGG
CCCTTGTGCGCTTATGCGGTTTTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTATGGGGC
AGGTTGCCCACGCGTTACTCACCCGTCCG
>FLP3FBN01B54IG length=245 xy=0772_3014 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCTCACCAAC
TAGCTAATCAGACGCGAGCCCATCTCAGAGCGATAAATCTTTGGCAGTCAGGGAGATGTC
TCCCAACTGCTTCATGCGGTATTAGCAGATGTTTCCATCTGTTATTCCCCACTCCAAGGC
AGGTT
>FLP3FBN01DECHE length=279 xy=1276_1824 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTTGTCTTGGTGAGCCGTTACCCCACCAAC
TAACTAATACAACGCATGCCCATCCATTACCACCGGAGTTTTCAACCCAAGAAGATGCCT
CCCTGGATGTTATGGGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTAATGGGTAG
GTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCGAC
>FLP3FBN01BDNJS length=278 xy=0448_1766 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGAGCCGTTACCTCACCAAC
AACCTAATGGAACGCATCCCCATCCTTTACCGGAATCCTTTAATAATGAAACCATGCGGA
ATCATTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTAAAGGGCAG
GTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCAG
>FLP3FBN01DVZQC length=268 xy=1477_1810 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACCGATCGTCGCCTTGGTATGCCGTTACCATACCAAC
TAGCTAATCGGACGCAGGGCCATCCCTTACCACCTTAGTTTTTCACTCTGCCTCATGCGA
GGCTGGGCGCTTATGCGGTATTAGCAGTTGTTTCCATCTGTTATCCCCCTGTAAGGGCCA
GGTTCCCTACGCGTTACTCACCCGTCCG
>FLP3FBN01EI5JL length=270 xy=1741_1087 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
CCGGCCACCCTCTCAGGTCGGCTACCGATCGTCGCCTTGGTGGGCCTCTCCCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCCCATACCACCGGAGTTTTTCACACAGGAGCATGCGC
TCCCGTGCGCTTATGCGGTATTAGCACCCGTTTCCAGGTGGTATCCCCCGGTATGGGGCA
GGTTACCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01EPSYJ length=256 xy=1817_0073 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGT
CCGATCAGCCTCTCAGCTCGGATATGCATCGTTGCCTTGGTAGGCCATTACCCCACCAAC
TAGCTAATACAACGCAAGCCCATCTCAAAGTGAAGCAAATGCTCCTTTTAACGTATCTAG
TTATCTAATTACGTATCATCTGGTATTAGCGTTCATTTCTAAACGTTATCCCAGTCTCTG
AGGTAGGTTACCCACG
>FLP3FBN01EOGU6 length=244 xy=1801_3280 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAAC
TAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGA
CATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCT
TGGG
>FLP3FBN01CL1LE length=278 xy=0954_0256 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTAC
TGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCAAGGATGCCC
CCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAG
GTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCTGG
>FLP3FBN01DQSSH length=277 xy=1418_1199 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTGGTCTTGGTGAGCCGTTACCCCACCAAC
TAACTAATACAACGCATGCCCATCCATTACCACCGGAGTTTTCAACCCAAGAAGATGCCT
CCCTGGATGTTATGGGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTAATGGGTAG
GTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCG
>FLP3FBN01CHYQT length=279 xy=0907_2451 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
AAGCTAATCAGACGCGGGTCCATCCTATACCACCTCAGTTTTTCACACCGAACCATGCGG
TTTTAGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGC
AGGTTACCCACGCGTTACTCACCCGTCCGCCACTAACAG
>FLP3FBN01DI11Q length=281 xy=1330_0396 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCGCCGACT
AGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCGGCTACCGGGATGCC
CCGACGCCGCGTCACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTGGCGGGC
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01A3YSL length=281 xy=0338_0339 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
TAGCTAATGCGCCGCATGGCCATCCGTAGCCGGTGTTACCCTTTAAACCCCAAGAGATGC
CTCTCGGAATTATTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTACGGG
CAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01CUX8S length=271 xy=1055_2122 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCGCCAAC
CAGCTAATCAGACGCGGGTCCATCCCATACCGTCTCGGCTTTCCCCTCCCGGCCATGCGG
CCAAGAGGAATCATGCGGTATTAGCGGCCGTTTCCGGCCGTTGTCCCCCTGTATGGGGCA
GGTTGCCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01A4SET length=274 xy=0347_1859 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGTC
CAGCTAATGCGCCGCATGCCCATCCTTCCCCGATGAATCTTTCCTCCTCCGGAGATGCCT
CCGGAGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGGGAAGGGCAG
GTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>FLP3FBN01BVAH3 length=248 xy=0649_1369 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGATCGACCTCTCAGTCCGGCTACCCGTCGTCGGCTAGGTAGGCCACTGCCCCACCTAC
TACCTGATGGGCCGCGACCCCATCTCACGCCGCGAAAGCTTTCATCCGCTCCTCATGCGA
GGTGCGGATAATATCCGGTATTAGCTGCCGTTTCCGGCAGTTATCCCGATGCGTGAGGCA
GGTTGGTC
>FLP3FBN01AHKPL length=244 xy=0083_0135 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAAC
TAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGA
CATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCT
TGGG
>FLP3FBN01ET7S1 length=239 xy=1867_1139 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCCCACCAAC
CAGCTAATCAGACGCGAGTCCATCTCAGAGCGATAAATCTTTGGCAGTCAGAGCCATGCG
GCCCAACTGCATCATGCGGTATTAGCAGCTGTTTCCAGCTGTTATTCCCCACTCCAAGG
>FLP3FBN01DBO5X length=255 xy=1246_1171 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCACGTGCAAGGCA
GGTTACCCACGCGTT
>FLP3FBN01EPNQQ length=249 xy=1815_1504 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGGTCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
CAGCTAATGCGCCATAAGTCCATCCTCTACCAGTGCCTTGCAGCACTTTTAATACGGTCA
CCATGCAGTGTCCCTACCTATGCGGTCTTAGCTGCCGTTTCCAGCAGTTATCCCCCTGTA
AAGGCCAGG
>FLP3FBN01BXFJH length=272 xy=0673_2907 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCT
CCCAACGTTCTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCA
GGTTCCATACGTGTTACTCACCCGTGCGCCGG
>FLP3FBN01CP0TU length=271 xy=0999_1568 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAAC
TACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGG
TTCCGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCA
GGTTGCCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01D4230 length=272 xy=1581_0110 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAAC
CAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCT
CTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAG
GTTGCATACGCGTCGCNCCGTCGCGCGGCCGG
>FLP3FBN01CTP94 length=250 xy=1041_2490 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCTACCTCTCAGTACGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCCACCAAC
TAGCTAATCAGACGCGGATCCATCTCATACCGCCTACAGTCTTTTCACACCGGAAACATG
CATCTCTGTGCGCTTATGCGGTATTAGCAGTCATTTCCTAACTTGTTATCCCCTGTATGA
GGCAGGTTAT
>FLP3FBN01AMBXT length=255 xy=0137_0863 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAACTGTTATCCCCCTGTGTAAGGCA
GGTTGCCCACGCGTT
>FLP3FBN01DRS0G length=264 xy=1429_3086 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTACCAAC
TAGCTAATCAGACGCGAGGCCATCTCTCAGCGATAAATCTTTGGTACCAGTACCATGCGA
TACCAGTACGTTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCACTGAAAGGCA
GGTTCCTCACGCGTTACTCACCCG
>FLP3FBN01AXM4C length=284 xy=0266_0186 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAGTGTGG
GGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGCCACGGTGGGCCGTTACCCCGCCAAC
TAGCTAATGCGCCGCATGGCCATCCGTAGCCGGTGTTACCCTTTAAACCCCAAGAGATGC
CTCTCGGAGTTATTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTACGGG
CAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCGG
>FLP3FBN01D82DP length=264 xy=1626_1467 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGGCCACCCTCTCAGGCCGGCTACCCATCGTCGCCTTGGTAGGCCATTACCCTACCAAC
TAGCTAATGGGACGCGAGTCCATCTTTCAGCGTCAGGTCTCCCAACTTTTCCTATATAAA
CATGCACTTATATAACTCATCCGGCATTAGCTCACCTTTCGGTAAGTTGTTCCAGTCTAA
AAGGCAGGTCACTCACGTGTTACT
>FLP3FBN01ED2R2 length=263 xy=1683_1792 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
CAGCTAATACAACGCATGCCCATCATCAACCACCGGAGTTTTCAACCCCATGCGATGCCG
CATGAGATATTATGGGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAG
GTTGCATACGCGTTACGCACCCG
>FLP3FBN01B82L6 length=282 xy=0806_1260 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTGGTCTTGGTGAGCCGTTACCCCACCAAC
TAACTAATACAACGCATGCCCATCCATTACCACCGGAGTTTTCAACCCAAGAAGATGCCT
CCCTGGATGTTATGGGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTAATGGGTAG
GTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCGACAAT
>FLP3FBN01BW42C length=253 xy=0670_1618 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGACTTGGTAGGCCGTTACCCCACCAAC
TATCTAATCAGACGCAAGTCCATCTGTTAGCGAATTGCTCCTTTCCCACTATTATCATGT
GATAATTATGGCATATGCGGTATTAGCAATGATTTCTCATTGTTATTCCCCTCTAACAGG
CAGGTTACTTACG
>FLP3FBN01A86UW length=255 xy=0397_2406 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCA
GGTTATCCACGCGTT
>FLP3FBN01AJ5LD length=261 xy=0112_1727 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTAGCATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAACT
AGCTAATGTCACGCATGCCCATCCGTTGCCGGAATCGCTTCCTTTGGCCGCAGGGCCATG
CAGCCCCGCGGCATTACGCGGTATTAGACGGGATTTCTCCCGATTATCCCCCTGCAACGG
GCAGGTCGCATACGCGTTACT
>FLP3FBN01DLI9X length=262 xy=1358_1347 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAAC
AAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGA
AGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGC
GCGGGCAGGTTCCCTACGTGTT
>FLP3FBN01BIKSR length=243 xy=0504_2105 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGACTTGGTGAGCCGTTACCTCACCAAC
TATCTAATCAGACGCGAGCCCATCTCTGAGCGAATTTCTTTGATATCCAAATCATGCGAT
TTAAATATGTTATGAGGTATTACCATCCGTTTCCAGAAGCTATCCCTCTCTCGAGAGCAG
GTT
>FLP3FBN01DUZ3A length=272 xy=1466_0676 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCT
CCCAACGTTCTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCA
GGTTCCATACGTGTTACTCACCCGTGCGCCGG
>FLP3FBN01BCBOG length=238 xy=0433_1166 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAAGCCGGCTACTGATCGTTGCTTTGGTAGGCCATTACCCTGCCAAC
TGGCTAATCAGACGCGGGGCCATCGTATGCCGATAACTCTTTTCACACCATGCCATGCAG
CATTGTGTGCTTATGCGGTATTAGCAGTCATTTCTGACTGTTGTCCCCCTGCATACGG
>FLP3FBN01AEVVE length=247 xy=0052_1608 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAAC
TGGCTAATCAGACGCGAGCTCATCTTTGGACAATAAATCTTTGGCAGCAGGACCATGCGA
TCCCGTTGCATTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCATCCAAAGGCA
GATTGCT
>FLP3FBN01D5ZFA length=249 xy=1591_1028 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGGTCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
CAGCTAATGCGCCATAAGTCCATCCTCTACCAGTGCCTTGCAGCACTTTTAATACGGTCA
CCATGCAGTGTCCCTACCTATGCGGTCTTAGCTGCCGTTTCCAGCAGTTATCCCCCTGTA
AAGGCCAGG
>FLP3FBN01BPHY3 length=268 xy=0583_1453 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTACTGATCGTTGCCTTGGTGGGCCGTTACCCCTCCAAC
TAGCTAATCAGACGCGGGTCCATCTCATACCGTCTCGGCTTTTCACCCCGAACCATGCGG
TTCTGTGTGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATGAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01B4FWO length=258 xy=0753_2294 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTTTTACCTCACCAAC
TAGCTAATCAGACGCGGAACCATCGTATACCACCGGAGTTTTTCACATTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTTTCCACGCGTTACT
>FLP3FBN01AXQ2Z length=253 xy=0267_1225 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTACCAAC
TAGCTAATCAGACGCGAGGCCATCTTTCAGCGATAAATCTTTGACATAAATGCCATGCGA
CACCTATATGTTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGAAAGGCA
GGTTCCTCACGCG
>FLP3FBN01ECMN2 length=268 xy=1666_3888 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAG
GACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01EDX65 length=272 xy=1681_4047 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAGGGAAAAGCCATGCGG
CTTCCCCCGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGCCAG
GTTGCCCACGCGTTACTCACCCGTCCGCCACT
>FLP3FBN01AP0N3 length=289 xy=0179_0849 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGCCCATCCGTGGCCGGGATTGCTCCCTTTGGCGGCCCGGGGAT
GCCCCAAGGCCGCGTTACGCGGTATTAGACGGGGTTTCCCCCGCTTATCCCCCTGCCACG
GGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCAACGG
>FLP3FBN01EADWQ length=255 xy=1641_1624 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTCTCCACGCGTT
>FLP3FBN01CEH6T length=281 xy=0868_0771 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTGGTCTTGGTGAGCCGTTACCCCACCAAC
TAACTAATACAACGCATGCCCATCCATTACCACCGGAGTTTTCAACCCAAGAAGATGCCT
CCCTGGATGTTATGGGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTAATGGGTAG
GTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCGACAA
>FLP3FBN01D28RJ length=270 xy=1560_0141 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGATCAACCTCTCAGTTCGGCTACGCATCATTGCCTTGGTAAGCCTTTACCCCACCAAC
TAGCTAATGCGTCGCGGGCCCATCCAAAAGCGGTAGCATAGCCACCTTTTACATAGTTAC
CATGCGGTAACTATGGTTATGCGGTATTAGCACCTGTTTCCAAGTGTTATCCCCCTCTTT
TGGGCAGGTTGCCCACGTGTTACTCACCCG
>FLP3FBN01AWE2B length=244 xy=0252_0433 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAAC
TAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGA
CATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCT
TGGG
>FLP3FBN01B9JHP length=264 xy=0811_2651 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCTCTACCCCGCCAAC
AAGCTAATCAGACGCGGGTCCATCGTATACCACCGGAGTTTTTCACACCAGACCATGCGA
TCCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTCTCCACGCGTTACTCACCCG
>FLP3FBN01DE1KU length=257 xy=1284_1580 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACCGATCGTCGGCTTGGTGGGCCGTTACCTCACCAAC
TACCTAATCGGACGCGAGCCCATCTTTCAGCGGATTGCTCCTTTGATTATCAAGCCATGC
AGCTCAATAATGTCATGCGGTATTAGCGTTCGTTTCCAAACGTTATCCCCCTCTGAAAGG
CAGGTTGCTCACGCGTT
>FLP3FBN01A9EJ1 length=264 xy=0400_0095 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCCACCAAC
CAGCTAATCAGACGCGGGCTCATCTTATACTACCGGAGTTTTTCACACCGGACCATGCGG
TCCTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATGAGGCA
GGTTACCCACGCGTTACTCACCCG
>FLP3FBN01A8MZH length=281 xy=0391_1227 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTGGTCTTGGTGAGCCGTTACCCCACCAAC
TAACTAATACAACGCATGCCCATCCATTACCACCGGAGTTTTCAACCCAAGAAGATGCCT
CCCTGGATGTTATGGGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTAATGGGTAG
GTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCGACAA
>FLP3FBN01ALP0V length=255 xy=0130_1133 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGCTGCCTTGGTGGGCCGTTACCCCGCCAAC
TAGCTAATCAGACGCGGGTCCATCTCATACCACCGGAGTTTTTCACACAGAACCATGCGG
TTCCATGCGCTTATGCGGTATTAGCAGTCATTTCTGGCTGTTATCCCCCTGTATGAGGCA
GGTTACCCACGCGTT
>FLP3FBN01DMENX length=270 xy=1368_1067 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTACCGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAAC
TGGCTAATCGGGCGCGGGCCCATCCCGTGCCGCCGGAGCTTTCCGCATACACCCATGCGG
CTGTATGCGCTTATGCGGTATTAGCGCCTGTTTCCAGGCGGTATCCCCCGGCACGGGGCA
GGTTGCCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01B4MLO length=254 xy=0755_2778 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTCACCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTCTCCACGTCG
>FLP3FBN01AQ9FC length=281 xy=0193_1510 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01DVF2L length=264 xy=1471_0907 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGTCCACCCTCTCAGGCCGGCTGCTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAAC
TAACTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTTCCTGCTGTACCATGCGG
CACTGTAGGCTTATGCGGTATTAGCAGCCGTTTCCAGCTGTTGTCCCCCTCTGAAAGGCA
GGTTCCTCACGCGTTACTCACCCG
>FLP3FBN01CWQIO length=264 xy=1075_3502 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCTTTACCCCGCCAAC
TGGCTAATCGGACGCGGATCCATCGTATGCCGATAAATCTTTTCACACTGTACCATGCGG
TACCGTGCGCTTATGCGGTATTAGCAACTGTTTCCAGTTGGTATCCCCCTGCATACGGCA
GGTTATCCACGCGTTACTCACCCG
>FLP3FBN01BC6PA length=259 xy=0443_0412 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
CAGCTAATGCGCCGCATGCCCATCCATAGCCGCCTCAACTTTCCTCCCCAAGGGATGCCC
CTCAGGGAGTGCACGCGGATTTAGACCGGATTTCTCCGGATTATTCCCCTGCTATGGGCA
GGTTGCATACGTGTTACTC
>FLP3FBN01CHIXK length=269 xy=0902_2438 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGTCCACCCTCTCAGGCCGGCTGCTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAAC
TGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAAAGGATTACCATGCGG
TATTCCCTATTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGGTACAGGGCAG
GTTGCCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01BSQTU length=268 xy=0620_1344 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAAC
CAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGCTTCATGCAA
AGCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCA
GGTTGCCCACGCGTTACTCACCCGTCCG
>FLP3FBN01D15YG length=275 xy=1547_3094 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAGC
AAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTAAGGAAAAGACATGCAT
CTTTCCCTGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGTAAGGCAG
GTTACCCACGCGTTACTCACCCGTCCGCCACTAAG
>FLP3FBN01BTIK9 length=268 xy=0629_0459 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
CAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTCACACTGTCCCATGCAG
GACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTGTCCCCCTGTACAGGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01AHV2W length=268 xy=0086_2582 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCTTTGGTAGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGGCCCATCCTATGGCGATAACTCTTTCTTATTATGACCATGCAA
TCTTAATAAGGTATGCGGTATTAGCAGCGGTTTCCCACTGTTGTCCCCCTCCATAGGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01EGJ5Q length=255 xy=1711_2940 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGCTGCCTTGGTGGGCCGTTACCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTCTCCACGCGTT
>FLP3FBN01A2YVL length=255 xy=0326_2943 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGG
ACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACAGCA
GGTTGGATACGTGTT
>FLP3FBN01A5KXW length=258 xy=0356_1970 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTGTTTCCAGGTGTTATCCCCCTGTGTAAGGCA
GGTTACCCACGCGTTACT
>FLP3FBN01D0KPW length=268 xy=1529_2642 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
CCGATCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTAAGCCGTTACCTTACCAAC
TAGCTAATCAGACGCGGGTCCATCCTGTACCGCAAAAGCTTTGATACTTCTACCATGCGA
TAAAAGTATTTTATCTCGTATTAGCATACCTTTCGGTATGTTATCCGTGTGTACAGGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01EF8RN length=266 xy=1708_0465 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCAACCTCTCAGTCCGGCTACCGATTGTCGGTTTGGTGGGCCGTTACCCCGCCAAC
TGCCTAATCGGACGCGAGCCCATCTTTCAGCGGATTGCTCCTTTGATATCAGTGCCATGC
GACTCCGATATTTTATGCGGTATTAGCGTTCGTTTCCAAACGTTATCCCCCTCTGAAAGG
CAGGTTGCTCACGTGTTACTCACCCG
>FLP3FBN01A0OWM length=274 xy=0300_3204 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
CAGCTAATACAACGCATGCCCATCCATAACCACCGGAGTTTTCAATCAAAAGGGATGCCC
CTCTTGATGTTATGGGATATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTATGGGTAG
GTTGCATACGCGTTACGCACCCGTGCGCCGGTCG
>FLP3FBN01DUQQ0 length=255 xy=1463_0858 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCAACCTCTCAGTCCGGCTACCGATCGTCGGCTTGGTGGGCCGTTACCTCACCAAC
TACCTAATCGGACGCGAGCCCATCTTTCAGCGGATTGCTCCTTTGATTATCAAGCCATGC
AGCTCAATAATGTCATGCGGTATTAGCGTTCGTTTCCAAACGTTATCCCCCTCTGAAAGG
CAGGTTGCTCACGCG
>FLP3FBN01CIWNC length=255 xy=0918_1334 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATCTGG
CCGGTCGGTCTCTCAACCCGGCTACCCATAGAAGCCTTGGTGGGCCGTTGCCCCGCCAAC
TAGCTAATCAGACGCGAGCTCATCCAAAGGCAATAAATCTTTGGCAGCTTGGCCATGCGA
CCACGCTGCGTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCACCTAAGGGCA
GATTGCTCACGCGTT
>FLP3FBN01D07UJ length=281 xy=1536_3945 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTAC
TGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCAAGGATGCCC
CCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAG
GTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGGCAG
>FLP3FBN01B27CZ length=268 xy=0739_1905 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAG
GACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01DQI3A length=268 xy=1415_0916 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGCTTTGGTGGGCCGCTGCCCCGCCAAC
TGGCTAATCAGACGCGGGTCCATCCCATACCACCGGAGTTTTTCACACAGGAGCATGCGC
TCCCGTGCGCTTATGCGGTATTAGCACCTGTTTCCAGGTGGTATCCCCCGGTATGGGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01AT5GO length=273 xy=0226_1174 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAAC
CAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGTACCATGCGG
TACTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCA
GGTTACCCACGCGTTACTCACCCGTCCGCCACT
>FLP3FBN01CP6X7 length=272 xy=1001_1309 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCCCGCCAAC
TACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCGCTGTACCATGCGG
TACTGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCCTGTGTAAGGC
AGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01DJ7Y0 length=272 xy=1343_1482 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCT
CCCAACGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCA
GGTTCCATACGTGTTACTCACCCGTGCGCCGG
>FLP3FBN01AXR34 length=290 xy=0267_2562 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGCCCATCCGTGGCCGGGATTGCTCCCTTTGGCGGCCCGGGGAT
GCCCCAAGGCCGCGTTACGCGGTATTAGACGGGGTTTTCCCCCGCTTATCCCCCTGCCAC
GGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCGGCGG
>FLP3FBN01C9YGU length=278 xy=1226_1836 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTAC
TGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCAAGGATGCCC
CCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAG
GTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGG
>FLP3FBN01AVACO length=268 xy=0239_0918 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAG
GACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01DTNE9 length=261 xy=1450_3139 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCACCAAC
TACCTAATGCGCCGCATGCCCATCCTTTACCGGATCGCTCCTTTGACATACCGATCATGC
GGTCGGTATGTGTTATGAGGTATTAGACGGAATTTCTCCCGATTATCCCTCTGTAAAGGG
CAGGTCGCATACGTGTTACTC
>FLP3FBN01DLTYU length=274 xy=1361_2916 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAAC
CAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCT
CTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAG
GTTGCATACGCGTTACGCACCCGTGCGCCGGTCG
>FLP3FBN01DIYX9 length=271 xy=1329_0479 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAAC
CAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCT
CTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAG
GTTGCATACGCGTTACGCACCCGTGCGCCGG
>FLP3FBN01CMQ1H length=272 xy=0962_0467 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
CAGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAGGGGAAAGCCATGCGG
CTCCCCCCGTTATGCGGTATTAGCACCTATTTCTAAGTGTTGTCCCCCTGTACAGGGCAG
GTTGCCCACGCGTTACTCACCCGTCCGCCACT
>FLP3FBN01ECGSC length=281 xy=1665_0362 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTGGTCTTGGTGAGCCGTTACCCCACCAAC
TAACTAATACAACGCATGCCCATCCATTACCACCGGAGTTTTCAACCCAAGAAGATGCCT
CCCTGGATGTTATGGGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTAATGGGTAG
GTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCGACAA
>FLP3FBN01AP7UV length=275 xy=0181_1973 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTCAAGGAAAAGACATGCAT
CTTTCCCTGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGTAAGGCAG
GTTACCCACGCGTTACTCACCCGTCCGCCACTAAG
>FLP3FBN01COR6A length=268 xy=0985_1040 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTCTACCCCGCCAAC
AAGCTAATCAGACGCGGGTCCATCGTATACCACCGGAGTTTTTCACACCAGACCATGCGA
TCCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAACTGTTATCCCCCTGTATAGGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01D7CGI length=268 xy=1606_3136 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
TGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGG
AACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCA
GGTTATCCACGCGTTACTCACCCGTCCG
>FLP3FBN01BHLXV length=281 xy=0493_1985 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01CXMVM length=258 xy=1086_0384 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
AAGCTAATCAGACGCGGATCCATCGTATACCACCAAAAGCTTTAGCTTTTTGTTTTCCAC
ACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCC
CCCAGTATACGGCAGGTT
>FLP3FBN01A7TP4 length=268 xy=0382_0170 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTACCGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAAC
TGGCTAATCGGACGCGGATCCATCGTATGCCGTTAAACCTTTTCACACTGTGTCATGCGA
CACCGTGCGCTTATGCGGTATTAGCAACTGTTTCCAGTTGGTATCCCCCTGCATACGGCA
GGTTATCCACGCGTTACTCACCCGTCCG
>FLP3FBN01CWOPF length=268 xy=1075_1153 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCGG
GACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01C1B5E length=255 xy=1128_1072 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTCTCCACGCGTT
>FLP3FBN01EGXD4 length=245 xy=1715_3706 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
CCGATCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
AAGCTAATCAGACGCGAGCTCATCCATATCCGATAAATCTTTGATGGAAAGAAGATGCCT
TCAAACCATGTTATGCGGTATTATCAGTCGTTTCCAACTGTTATCCCCCAGATATGGGCA
GATTG
>FLP3FBN01APPTP length=261 xy=0175_3179 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAAC
AACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGAA
GATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGCG
CGGGCAGGTTCCCTACGTGTT
>FLP3FBN01C4BE2 length=268 xy=1162_0828 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCT
CCCAACGTTCTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCA
GGTTCCATACGTGTTACTCACCCGTGCG
>FLP3FBN01ASJ6H length=281 xy=0208_0663 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGCCCATCCGTCACCGGATTGCTCCTTTGACCGCTCCGGGATGC
CCCGGAATGGTGTTACGCGGAATTAGTCGGAATTTCTTCCGGTTATTCCCCTGTGACGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01DI8XZ length=284 xy=1332_1141 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTAC
TGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCAAGGATGCCC
CCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAG
GTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGGCGGAAG
>FLP3FBN01DXZGT length=275 xy=1500_0571 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGTCCACCCTCTCAGGCCGGCTGCTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAAC
TGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAAAGGATTACCATGCGG
TATTCCCTATTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGGTACAGGGCAG
GTTGCCCACGCGTTACTCACCCGTCCGCCACTAAG
>FLP3FBN01CVPVO length=264 xy=1064_1074 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
CCGATCACCCTCTCAGTTCGGCTACGCATCATTGCCTTGGTAAGCCTTTACCCCACCAAC
TAGCTAATGCGTCGCGGGCCCATCCAAAAGCGGTAGCATAGCCACCTTTTACATAGTTAC
CATGCGGTAACTATGGTTATGCGGTATTAGCACCTGTTTCCAAGTGTTATCCCCCTCTTT
TGGGCAGGTTGCCCACGTGTTACT
>FLP3FBN01C6P40 length=282 xy=1189_2626 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCGGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGGCCAA
CTACCTAACTGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGAT
GTCCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCTGTAGCGG
GCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01EFG5X length=247 xy=1699_1555 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCGCCCTCTCAGGCCGGCTACTGATCGTTGCCTTGGTGGGCTCTTACCCCTCCAAC
TAGCTAATCAGACGCGGACTCATCCTGCACCGAATTTCTTTTCCTACTCCCTCATGCGAG
GCTGTAGGCTTATGCGGTATTAGCAGCCATTTCTAGCTGTTGTCCCCCTGTGCAGGGCAG
ATTATCC
>FLP3FBN01B7UNT length=271 xy=0792_1639 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGTCTTGGTGGGCCGCTACCCCGCCAAC
TAACTAATCAGACGCGGGTCCATCTCATACCACCGGAGTTTTTCACACCGGACCATGCGG
TCCTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATGAGGCA
GGTTACCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01CSORT length=272 xy=1029_3031 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGTAGTTTTCTAGGGAAAAGCCATGC
GGCTTCCCCGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCTGTACAGGCCAG
GTTGCCCACGCGTTACTCACCCGTCCGCCACT
>FLP3FBN01DY4CI length=263 xy=1513_0304 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCTCGCCAA
CCAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCTTACAGGATGTTTCACAGAAAG
AAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTG
CGCGGGCAGGTTCCCTACGTGTT
>FLP3FBN01BE4FD length=265 xy=0465_0663 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGGCCACCCTCTCAGGCCGGCTACCCATCGTCGCCTTGGTAGGCCATTACCCTACCAAC
TAGCTAATGGGACGCGAGTCCATCTTTCAGCGTCAGGTCTCCCCAACTTTTCCTATATAA
ACATGCACTTATATAACTCATCCGGCATTAGCTCACCTTTCGGTAAGTTGTTCCAGTCTA
AAAGGCAGGTCACTCACGTGTTACT
>FLP3FBN01B4V0D length=268 xy=0758_2683 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACCGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAG
GACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01BZ1F6 length=268 xy=0703_1732 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGATCGACCTCTCAGTCCGGCTACCCATCGCAGGCTCGGTGGGCCGTTACCCCGCCGAC
TACCTAATGGGCCGCGACCCCATCTCACACCGCGAGAGCTTTCATCCGGCTAACATGTGT
TGGCCGGATAGTATCCGGTATTAGCTACCGTTTCCAGTAGTTATCCCGGTGTGTGAGGCA
GGTTGGTCACGTGTTACTCACCCGTTCG
>FLP3FBN01AZK2X length=271 xy=0288_0743 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAGGGAAAAGCCATGCGG
CTTCCCCGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGCCAGG
TTGCCCACGCGTTACTCACCCGTCCGCCACT
>FLP3FBN01BSEFO length=259 xy=0616_1666 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATTGTCGCTTTGGTAGGCCGTTACCCCTGCCAA
CTGGCTAATCAGACGCGGGTCCATCCCATACCACCGGAGTTTTTCACACAGCAACATGCG
TTGCCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATGGGGC
AGGTTGCCCACGCGTTACT
>FLP3FBN01BCXZU length=264 xy=0440_1416 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGG
ACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCA
GGTTGGATACGTGTTACTCACCCG
>FLP3FBN01ALCR0 length=268 xy=0126_0350 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCTCACCAAC
TACCTAATCAGACGCGGGTCCCTCCTATACCACTATCGTTTTTCACTCAGCGCCATGCGG
CTCCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATAGGGCA
GGTTCCCCACGCGTTACTCACCCGTCCG
>FLP3FBN01AO474 length=275 xy=0169_1058 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCT
CCCAACGTTCTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTACGGGCA
GGTTCCATACGTGTTACTCACCCGTGCGCCGGTCG
>FLP3FBN01EJKEX length=253 xy=1745_3975 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTAGCATCGTCGTCACGGTGGGCCGTTACCCCGCCGTCA
AGCTAATGTCACGCGAGCCTATCCTCATCCGACGGATCTTTAGATGGAACCAGATGCCTG
ATTCCATCGCCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTCGATGAGGGCA
AGTTGCTCACGCG
>FLP3FBN01CPIOY length=264 xy=0993_2640 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCCACCAAC
TGGCTAATCAGACGCGGGTCCATCTTATACCACCGGAGTTTTTCACACTGCACCATGCGG
TGCTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATAAGGCA
GGTTACCCACGCGTTACTCACCCG
>FLP3FBN01BUOD1 length=281 xy=0642_1383 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAGTGTGG
GGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01AWUMH length=268 xy=0257_0119 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAG
GACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01BI3RV length=272 xy=0510_2121 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGAGAAAATGCTC
ACTCATGATACCATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCA
GGTTGGATACGCGTTACTCACCCGTGCGCCGG
>FLP3FBN01DQWOP length=253 xy=1419_2151 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTCTCCACGCG
>FLP3FBN01BSFE9 length=264 xy=0616_2947 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTACCAAC
TAGCTAATCAGACGCGAGGCCATCTCTCAGCGATAAATCTTTGGTACCAGTACCATGCGA
TACCCGTACGTTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCACTGAAAGGCA
GGTTCCTCACGCGTTACTCACCCG
>FLP3FBN01DA3NU length=253 xy=1239_1976 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTC
AAGCTAATGTCACGCGAGCCTATCCTCATCCGACGAATCTTTAGATGGATTCAGATGCCT
GATTCCATCACCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCA
AGTTGCTCACGCG
>FLP3FBN01EU5A3 length=260 xy=1877_3597 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGCTGCCCCGCCAAC
CAGCTAATCAGACGCGGGCCCATCCCACACCGCCCGAAGGCTTTTCACACCGCTTCATGC
GAAGCTGTGCGCTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTGTGGGG
CAGGTTGCCCACGCGTTACT
>FLP3FBN01EON7J length=261 xy=1804_0509 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
TAACTAATGCGCCGCATGCCCATCCATGACCGGATCGCTCCTTTGACTCCCGAGAGATGT
CTCCCGGGGGTGTTATGCGGTATTAGACGGAATTTCTCCCGCTTATCCCCCTGTCATGGG
CAGGTTGCATACGTGTTACTC
>FLP3FBN01CIA5C length=258 xy=0911_2142 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCACCAAC
TGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGG
AACCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCAGTATACGGCA
GGTTCTCCACGCGTTACT
>FLP3FBN01BE5VX length=268 xy=0465_2555 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCT
CCCAACGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCA
GGTTCCATACGTGTTACTCACCCGTGCG
>FLP3FBN01DKS5K length=271 xy=1350_0262 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
AAGCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGG
TTCTGGGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCA
GGTTGCCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01AIWWA length=262 xy=0098_1144 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAAC
TAGCTAATGTCACGCATGCCCATCCGTTGCCGGAATCGCTTCCTTTGGCCGCAGGGCCAT
GCAGCCCCGCGGCATTACGCGGTATTAGACGGGATTTCTCCCGATTATCCCCCTGCAACG
GGCAGGTCGCATACGCGTTACT
>FLP3FBN01ES3TI length=268 xy=1854_2564 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCACCAAC
TGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGG
AACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCA
GGTTATCCACGCGTTACTCACCCGTCCG
>FLP3FBN01DVTZC length=254 xy=1475_2550 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCTCACCAAC
TAGCTAATCAGACGCGAGGCCATCTTCCAGCGATAAAATCTTTGACATTCAAACCATGCG
ATTCAAATGTATCATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGGAAGGC
AGGTTCCTCACGCG
>FLP3FBN01CAD0I length=277 xy=0821_1248 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>FLP3FBN01C82RZ length=258 xy=1216_1725 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
CAGCTAATACAACGCATGCCCATCATCAACCACCGGAGTTTTCAACCCCATGCGATGCCG
CATGAGATATTATGGGGTATTAGTACCAATTTCTCAGTGTTATCCCCCTGTTGATGGTAG
GTTGCATACGCGTTACGC
>FLP3FBN01DQ1LW length=268 xy=1421_0338 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAG
GACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01C3YCV length=290 xy=1158_0285 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAGTGTGG
GGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGCCACGGTGGGCCGTTACCCCGCCGTC
AAGCTAATGCGCCGCATGCCCAGCCGCCACCGGATTACTCCTTTCGCCCGGACCGGATGC
CCGGTCCGGGCGGCATGGGGTATTAGGCCGGGTTTCCCCGGGTTATCCCCCTGACAAGGG
TAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCAGGCAGTA
>FLP3FBN01ETH11 length=264 xy=1859_0535 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
CAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTCACACTGAACCATGCGG
TTCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCG
>FLP3FBN01A449C length=271 xy=0351_2126 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCCACTGATCGTCGCCTTGGTAGGCCGTTACCCCACCAAC
CAGCTAATCAGACGCGGGCTCATCTTATACTACCGGAGTTTTTCACACAGAAACATGCGT
CCCCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATAAGGCA
GATTACCCACGTGTTACTCACCCGTCCGCCG
>FLP3FBN01CK4EX length=268 xy=0943_2311 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGCCGCCCTCTCAGGCCGGCTACTGATCGTCGGTTTGGTGGGCCGCTGCCCCGCCAAC
TGCCTAATCAGACGCGGGCCCATCCCATACCACCGGAGTTTTCCGCCCCATGCCATGCGG
CATTTGGCGCTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCGGTATGGGGCA
GGTTGCCCACGCGTTACTCACCCGTCCG
>FLP3FBN01ASO4F length=284 xy=0209_2973 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
TAGCTAATGCGCCGCATGGCCATCCGTAGCCGGTGTTACCCTTTAAACCCCAAGAGATGC
CTCTCGGAGTTATTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTACGGG
CAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCGG
>FLP3FBN01ET6LV length=265 xy=1866_3681 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCTCACCAAC
TACCTAATCAGACGCGGGTCCATCTTACACCGATGAAATCTTTTCCGTCAGAGTCATGCG
ACTCCAGCGGGTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGC
AGGTTGCCCACGCGTTACTCACCCG
>FLP3FBN01DR37T length=257 xy=1433_1223 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGCCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCTTTTATCTCACCAAC
TAGCTAATCAGACGCGGATCCATTCCCATACCACCGGAGTTTTTCACACAGGGCCATGCA
GTCCCGTGCGCTTATGCGGTATTAGCACGCCGTTTCCGGCTGTTATCCCCCGGTATGGGG
CAGGTTATCCACGCGTT
>FLP3FBN01BS9WF length=271 xy=0626_1485 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCCCGCCAAC
TACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCACTGTACCATGCAG
TACCGTGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCA
GGTTGCCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01DH65L length=286 xy=1320_1319 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTAC
TGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCGGGGATGCCC
CCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAG
GTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCAGTAGTG
>FLP3FBN01DQKWR length=267 xy=1415_3273 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGACCGTGTTTCAGTTCCAGTGTGG
CCGGTCATCCTCTTAGACCGGCTACCCATCGTTGCCTTGGTGGGCCGTTACCTCACCAAC
TAGCTAATGGGACGCGGACTCATCCCCTCGCGGAAGCTTTCATGAAGAGGCCTCCTTTCC
TCCACCATACAATTGGTGGAGCTTATCCGGTATTAGCTGCCGTTTCCAGCAGTTATCCCG
AACAAGAGGGCAGATTATCCACGCGTT
>FLP3FBN01DVCUY length=268 xy=1470_0840 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAAC
TGGCTAATCAGACGCGGGTCCATCTTATACCACCGGAGTTTTTCACACACTACCATGCGG
TACTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01A6VU8 length=270 xy=0371_1346 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTCTACCCCGCCAAC
AAGCTAATCAGACGCGGGTCCATCGTATACCACCGGAGTTTTTCACACCGGACCATGCGA
TCCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTGTAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01B9O3L length=245 xy=0813_1727 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAAC
TGGCTAATCAGACGCGAGCTCATCTTTGGACGATAAATCTTTGGCAGCAAAATCATGCGG
TCTCGCTGCATCATGCGGTATTAGCAGTCGTTTCCGGCTGTTATCCCCCATCCAAAGGCA
GATTG
>FLP3FBN01AJ7P4 length=277 xy=0113_0394 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>FLP3FBN01CTE7A length=262 xy=1038_0420 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTTCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAAC
CAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGA
AGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGC
GCGGGCAGGTTCCCTACGTGTT
>FLP3FBN01EC99X length=281 xy=1674_1715 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAGTGTGG
GGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01DQR2G length=281 xy=1418_0262 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGTTGACTTGGTGGGCCGTTACCCCGCCAAC
TATCTAATGGAACGCATCCCCATCGATAACCGAAATTCTTTAATAGTGAAACCATGCGGA
AATACTATACTATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTTATCGGCAG
GTTGGATACGTGTTACTCACCCGTGCGCCGGTCGCCATCAA
>FLP3FBN01CFBG4 length=268 xy=0877_1862 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGAGAAAATGCTC
ACTCATGATACCATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCA
GGTTGGATACGCGTTACTCACCCGTGCG
>FLP3FBN01B5NT1 length=268 xy=0767_1879 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
AAGCTAATCAGACGCGGGTCCATCTTACACCACTAATGTTTTTCACTCTGTCCCATGCGG
GACTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTGTAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01EZ9L3 length=277 xy=1936_0793 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGCCCATCCGCCACCGGATTGCTCCTTTGACCGCCCCGGGATGT
CCCGGAATGGTGTTACGCGGAATTAGTCGGAATTTCTTCCGGTTATTCCCCTGTGACGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>FLP3FBN01CRN6U length=259 xy=1018_0676 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGCGAGAATGCTC
ACTCATGATACTATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCA
GGTTGGATACGCGTTACTC
>FLP3FBN01APRO8 length=251 xy=0176_1514 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTC
AAGCTAATGTCACGCGAGCCTATCCTCATCCGACGAATCTTTAGATAGATTCAGATGCCT
GATTCCATCACCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCA
AGTTGCTCACG
>FLP3FBN01BUMCM length=277 xy=0641_2836 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCGCCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTAGGCTTCTACCCCACCAAC
TACCTAATCAGACGCGGGCCCATCTTACACCACCTCAGTTTTTACCCCTGTACCATGCGG
TACCGGGGTCTTATGCGGTATTAGCAATCATTTCTAATTGTTATCCCCCTGTGTAAGGCA
GGTTGCCCACGCGTTACTCACCCGTCCGCCGCTAAGT
>FLP3FBN01EEB9U length=281 xy=1686_1808 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01A0HNZ length=264 xy=0298_2013 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCACCAAC
TAACTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTTCCTGCTGTGCCATGCGG
CACCGCAGGCTTATGCGGTATTAGCAGCCGTTTCCAGCTGTTGTCCCCCTGTACAGGGCA
GGTTGCCCACACGTTACTCACCCG
>FLP3FBN01AXZN7 length=266 xy=0270_0069 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTATCTCACCAAC
TAGCTAATCAGATGCGGGCCCATCTTTTACCGAATATTTCCTTTCCTTCTCGGAAGATGC
CTTCCAAGAATATTATGCGGTATTAGTCACCGTTTCCAGTGATTATTCCCCAGTAAAAGG
CAGGTTGCCCACACGTTACTCACCCG
>FLP3FBN01CN9ZN length=271 xy=0979_2049 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAAC
TACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGG
TTCCGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCA
GGTTGCCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01CM7ZF length=273 xy=0967_1945 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCGGCACCGGGATGCC
CCGACGCCGCGTCACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTGGCGGGC
AGGTTGCATACGTGTTACTCACCCGTGCGCCGG
>FLP3FBN01A8C9C length=262 xy=0388_0910 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTAGGCCATTACCCTACCAAC
TAGCTAATGGGACGCGAGTCCATCTTTCAGCGTCAGGTCTCCCCAACTTTTCCTATATAA
ACATGCACTTATATAACTCATCCGGCATTAGCTCACCTTTCGGTAAGTTGTTCCAGTCTA
AAAGGCAGGTCACTCACGTGTT
>FLP3FBN01DYE6O length=239 xy=1505_0462 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCCCACCAAC
CAGCTAATCAGACGCGAGTCCATCTCAGAGCGATAAATCTTTGGCAGTCAGAGCCATGCG
GCCCAACTGCATCATGCGGTATTAGCAGCTGTTTCCAGCTGTTATTCCCCACTCCAAGG
>FLP3FBN01B03Q3 length=254 xy=0715_2221 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCCTTGGTAGGCCGTCACCCTTCCAAC
TAGCTAATCAGACGCGGGTCCATCCTTTTGCGATAGCATCTTCAGAGGCCATCTTTCCTT
AAAGAATCATGCGATCCTCTATTATTATGCGGTATTAGCTCCGATTTCTCGAAGTTGTTC
CCCACAAAAGGGCA
>FLP3FBN01D79PA length=279 xy=1617_1164 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTAGGCCGTTACCCCACCAAC
TACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCACCAAACCATGCGG
TTTTGTGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCA
GGTTTTCCACGCGTTACTCACCCGTCCGCCACTAACTCA
>FLP3FBN01AR65G length=263 xy=0204_0162 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGCCCATCCGTCACCGGAATCGCTTCCTTTGATCTCCCGGAGAT
GCCTCCGGAAGATGTCACGCGGTATTAGACGGGATTTCTCCCGATTATCCCCCTGTGACG
GGCAGGTTGCATACGTGTTACTC
>FLP3FBN01B8KLV length=264 xy=0800_2497 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTACCCATCGTCGCCTTGGTGGGCCGTTACCCCACCAAC
CAGCTAATCAGACGCGGGTCCATCTCATACCACCGGAGTTTTTCACACTTTATCATGCGA
TCTCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATGAGGCA
GGTTACCCACGCGTTACTCACCCG
>FLP3FBN01DGRID length=270 xy=1303_4019 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCTAATGTGG
CCGGCCGCCCTCTCAGGCCGGCTGCTGATCGCGGTCTTGGTGGGCCGTTACCCCGCCAAC
TAACTAATCAGACGCGGGCCCATCCCGTACCACCGGAGTTTTTCACGCAGAGCCATGCGG
CTCCGCGCGCTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCGGTACGGGGCA
GGTTGCCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01D7KO6 length=265 xy=1609_1528 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAAC
AAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGA
AGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGC
GCGGGCAGGTTCCCTACGTGTTACT
>FLP3FBN01DI2NJ length=272 xy=1330_1181 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCT
CCCAACGTTCTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCA
GGTTCCATACGTGTTACTCACCCGTGCGCCGG
>FLP3FBN01COFU8 length=265 xy=0981_1474 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAAC
CAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCGAAAGGATGTTTCACAGAAAGA
AGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGC
GCGGGCAGGTTCCCTACGTGTTACT
>FLP3FBN01BTCQ8 length=268 xy=0627_1090 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTTGTACCACCGGAGTTTTTCACACTGTCCCATGCAG
GACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01BN228 length=271 xy=0567_1042 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCCCGCCAAC
TACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCACTGTACCATGCAG
TACCGTGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCA
GGTTGCCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01BD9IY length=261 xy=0455_1576 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGCCGCATGCCCATCCTTTACCGGATCGCTCCTTTGACATACCGATCATGC
GGTCGGTATGTATTATGAGGTATTAGACGGAATTTCTCCCGATTATCCCTCTGTAAAGGG
CAGGTCGCATACGTGTTACTC
>FLP3FBN01AQPXC length=266 xy=0187_0814 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAGTGTGG
CCGTCCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
TAGCTAATGCGCCGCGAGCCCATCCTATACCGGATTTCTCCTTTTACCCCTCAGCCATGC
GACTGCGTGGTCTTATACCGTATTAACTACAGTTTCCCGTAGGTATCCCGTTGTATAGGG
CAGGTTGCTCACGTGTTACTCACCCG
>FLP3FBN01EIL6N length=268 xy=1735_0573 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGCCACCCTCTCAGGTCGGCTACTGATCGTTGCTTTGGTAGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGGCCCATCCTATGGCGATAACTCTTTCTTATTATGACCATGCAA
TCTTAATAAGGTATGCGGTATTAGCAGCGGTTTCCCACTGTTGTCCCCCTCCATAGGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01DAXF4 length=255 xy=1237_2114 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTCTCCACGCGTT
>FLP3FBN01DF870 length=274 xy=1298_0798 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAAC
CAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCT
CTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAG
GTTGCATACGCGTTACGCACCCGTGCGCCGGTCG
>FLP3FBN01B65YI length=274 xy=0784_2392 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCGTTCCTCTCAGAACCCCTACCTATCATTGCCTTGGTGGGCCGTTACCCCCCAAC
CTAGCTAATAGGACGCATGCCCATCTGATACCTCGAACGATTTAATTATTATAAGATGCC
TTACAATAATATTATGGGTGGTTAATCCACGTTTCCATGGGCTATCCCCCTCGTATCAGC
CAGGTTGCATACGCGTTACTCACCCGTGCGCCGG
>FLP3FBN01DY7BY length=263 xy=1514_0076 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCTCGCCAA
CCAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCTTACAGGATGTTTCACAGAAAG
AAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTG
CGCGGGCAGGTTCCCTACGTGTT
>FLP3FBN01AUK9B length=273 xy=0231_1165 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCGGCACCGGGATGCC
CCGACGCCGCGTCACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTGGCGGGC
AGGTTGCATACGTGTTACTCACCCGTGCGCCGG
>FLP3FBN01DWFUS length=268 xy=1482_2226 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAAC
TGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGG
AACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCA
GGTTATCCACGCGTTACTCACCCGTCCG
>FLP3FBN01B4PVG length=253 xy=0756_2922 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTC
AAGCTAATGTCACGCGAGCCTATCCTCATCCGACGGATCTTTAGATGGAACCAGATGCCT
GATTCCATCGCCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCA
AGTTGCTCACGCG
>FLP3FBN01CEUV1 length=267 xy=0872_0847 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAAC
AGGCTAATCAGACGCGGGTCCATCCATACCACCGGAGTTTTTCACACCAGACCATGCGAT
CCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCAG
GTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01BYVKN length=277 xy=0690_0709 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>FLP3FBN01CHCL3 length=261 xy=0900_2441 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCACCAAC
TACCTAATGCGCCGCATGCCCATCCTTTACCGGATCGCTCCTTTGACATACCGATCATGC
GGTCGGTATGTGTTATGAGGTATTAGACGGAATTTCTCCCGATTATCCCTCTGTAAAGGG
CAGGTCGCATACGTGTTACTC
>FLP3FBN01AWPFQ length=271 xy=0255_1588 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGAGCCGTTACCTCGCCAAC
CAGCTAATCAGACGCGGGTCCATCCCATACCGTCTCGGCTTTCCCCTCCCGGCCATGCGG
CCAAGAGGAATCATGCGGTATTAGCGGCCGTTTCCGGCCGTTGTCCCCCTGTATGGGGCA
GGTTGCCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01CAKFA length=240 xy=0823_1364 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCCACCAAC
TAGCTAATCAGACGCAAGCCCATCTATCAGCGGATTGCTCCTTTTCTAGCTATATCATGC
GATACTACTAGCTTATGCGGTATTAGCAATGATTTCTCACTGTTATTCCCCTCTGATAGG
>FLP3FBN01CKTVO length=254 xy=0940_0946 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCATCCTCTCAGACCAGCTACCCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
AAGCTAATGGGACGCGGACTCATCGAAAAGCGGAAGCATATGCAGAGGCCCCCTTTCCTG
CATAGACATATATGCAGCGTATGCGGTATTAGCAGCCGTTTCCAACTGTTATTCCCCACT
CCCCGGTAGATTAT
>FLP3FBN01CSJ87 length=264 xy=1028_1273 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGG
ACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCA
GGTTGGATACGTGTTACTCACCCG
>FLP3FBN01A6FDX length=258 xy=0366_0467 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCACCAAC
TAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTCTCCACGCGTTACT
>FLP3FBN01EECRG length=281 xy=1686_2442 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01B4KSX length=275 xy=0755_0447 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTCCCAATGTGG
CCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCGCCAACT
GGCTAATCAGACGCGGGTCCATCTCACACCGATAAATCTTTTCCGGCCGGATCGTGCGAT
CCTTCCGGCTTATGCGGTATTAGCGGTCGTTTCCAACTGTTATCCCCCTGTGTGAGGCAG
GTTACCCACGCGTTACTCACCCGTCCGCCACTAAG
>FLP3FBN01DJ1EH length=244 xy=1341_1159 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGACTTGGTGAGCCGTTACCTCACCAAC
TATCTAATCAGACGCGAGCCCATCTCTGAGCGAATTTCTTTGATATCCAAATCATGCGAT
TTAAATATGTTATGAGGTATTACCATCCGTTTCCAGAAGCTATCCCTCTCTCAGAGGCAG
GTTG
>FLP3FBN01DT1QK length=281 xy=1455_1210 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTCGGTGGGCCGTTACCCCGCCGAC
TGCCTAATGCGCCGCATGCCCATCCTCCACCACCGGAGTTTTCCTCCCACGGAGATGCCT
CCGCGGGATTTACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAG
GTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGGCAG
>FLP3FBN01EM8RT length=281 xy=1787_3479 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01DRFRX length=268 xy=1425_2315 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
AAGCTAATCAGACGCGGAACCATCGTATACCACCGGAGTTTTTCACACTGTCCCATGCGG
GCCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCGAGGCA
GGTTGCCCACGCGTTACTCACCCGTCCG
>FLP3FBN01CG2MW length=255 xy=0897_1798 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTCTCCACGCGTT
>FLP3FBN01CQO5F length=287 xy=1007_0321 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAGTGTGG
GGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGCCACGGTGGGCCGTTACCCCGCCGTC
AAGCTAATGCGCCGCATGCCCAGCCGCCACCGGATTCCTCCTTTCGCCCGGTCCGGATGC
CCGGTCCGGGCGGCATGGGGTATTAGGCCGGGTTTCCCCGGGTTATCCCCCTGTGGCGGG
CAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCAGGTG
>FLP3FBN01D8YZ5 length=279 xy=1625_1187 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAAC
TGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAAAGGATTACCATGCGG
TATTCCCTATTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGGTACAGGGCAG
GTTGCCCACGCGTTACTCACCCGTCCGCCACTAAGACAG
>FLP3FBN01D32U0 length=271 xy=1569_2282 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGGTCCATCCCATACCACCGGAGTTTTTCACACAGGAGCATGCGC
TCCCGTGCGCTTATGCGGTATTAGCACCTGTTTCCAGGTGGTATCCCCCGGTATGGGGCA
GGTTACCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01EBLMN length=264 xy=1655_0941 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGCTTTTCACACTGTCCCATGCAG
GACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCG
>FLP3FBN01A6BZY length=264 xy=0365_0172 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGTTTTGGTAGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGGTCCATCCCATACCACCGGAGTTTTTCACACAGCAACATGCGT
TGCCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATGGGGCA
GGTTGCCCACGCGTTACTCACCCG
>FLP3FBN01CT7NN length=268 xy=1047_0433 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAAC
TGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGG
AACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCA
GGTTATCCACGCGTTACTCACCCGTCCG
>FLP3FBN01CKH8P length=274 xy=0936_2247 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAAC
CAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCT
CTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAG
GTTGCATACGCGTTACGCACCCGTGCGCCGGTCG
>FLP3FBN01CYE9Z length=253 xy=1095_0325 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTC
AAGCTAATGTCACGCGAGCCTATCCTCATCCGACGGATCTTTAGATGGAACCAGATGCCT
GATTCCATCGCCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCA
AGTTGCTCACGCG
>FLP3FBN01EF1R5 length=276 xy=1705_3699 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAAC
CAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGTACCATGCGG
TACTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCA
GGTTACCCACGCGTTACTCACCCGTCCGCCACTAAG
>FLP3FBN01C4K5P length=265 xy=1165_1163 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGGCCACCCTCTCAGGCCGGCTACCCATCGTCGCCTTGGTAGGCCATTACCCTACCAAC
TAGCTAATGGGACGCGAGTCCATCTTTCAGCGTCAGGTCTCCCCAACTTTTCCTATATAA
ACATGCACTTATATAACTCATCCGGCATTAGCTCACCTTTCGGTAAGTTGTTCCAGTCTA
AAAGGCAGGTCACTCACGTGTTACT
>FLP3FBN01C2MYD length=264 xy=1143_0291 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGTATCATGCGA
TACTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTCTCCACGCGTTACTCACCCG
>FLP3FBN01EOUN6 length=272 xy=1806_0692 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAGGGAAAAGCCATGCGG
CTTCCCCCGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGCCAG
GTTGCCCACGCGTTACTCACCCGTCCGCCACT
>FLP3FBN01EKS2G length=262 xy=1760_0406 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAAC
CAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGA
AGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGC
GCGGGCAGGTTCCCTACGTGTT
>FLP3FBN01C77VC length=282 xy=1206_2630 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTGGTCTTGGTGAGCCGTTACCCCACCAAC
TAACTAATACAACGCATGCCCATCCATTACCACCGGAGTTTTCAACCCAAGAAGATGCCT
CCCTGGATGTTATGGGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTAATGGGTAG
GTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCGACAAT
>FLP3FBN01A18XL length=271 xy=0318_2087 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTTACACCACTAATGTTTTTCACTCTGTACCATGCGG
TACTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCA
GGTTACCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01EVR1U length=271 xy=1885_0304 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAAC
TACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGG
TTCTGGGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCA
GGTTGCCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01D9H69 length=263 xy=1631_1491 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCTCGCCAA
CTAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCTTACAGGATGTTTCACAGAAAG
AAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTG
CGCGGGCAGGTTCCCTACGTGTT
>FLP3FBN01DPA1A length=275 xy=1401_1164 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGTCCACCCTCTCAGGCCGGCTGCTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAAC
TGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAAAGGATTACCATGCGG
TATTCCCTATTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGGTACAGGGCAG
GTTGCCCACGCGTTACTCACCCGTCCGCCACTAAG
>FLP3FBN01D0Z6N length=275 xy=1534_2205 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCT
CCCAACGTTCTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCA
GGTTCCATACGCGTTACTCACCCGTGCGCCGGTCG
>FLP3FBN01BZ72V length=271 xy=0705_2133 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAAC
TACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGG
TTCCGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCA
GGTTGCCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01D3QET length=249 xy=1565_2531 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGGTCCATCCTGTACCGGGATCGCTCCCTTTCCGCATCTGCCCAT
GCGCGCTGATGCGGTTATGCGGTATTAGCATCTGTTTCCAGATGGTATCCCCCTGTACAG
GGCAGGTTG
>FLP3FBN01AZ92N length=275 xy=0296_0365 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAGGGAAAAGCCATGCGG
CTTCCCCCGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGCCAG
GTTGCCCACGCGTTACTCACCCGTCCGCCACTAAG
>FLP3FBN01DUG1R length=280 xy=1460_0573 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGG
CAGGTTGTCCACGCGTTACTCACCCGTCCGCCGCTAAGCC
>FLP3FBN01CUD3O length=270 xy=1049_0594 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCCCGCCAAC
CACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCACCGGACCATGCGG
TCCTGCGGTCTTATGCGGTATTAGCAGTCGTTTCCGGCTGTTGTCCCCCAGTGCAGGGCA
GGTTGCTCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01CTZRH length=268 xy=1044_2491 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTTACACCACTAATGTTTTTCACTCTGTCCCATGCGG
GACTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTGTAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01D7ZG7 length=269 xy=1614_0201 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCTCACCAAC
TACCTAATCAGACGCGGGCCCATCCTATGGCGATAACTCTTTCTTATTATGACCATGCAA
TCTTAATAAGGTATGCGGTATTAGCAGCGGTTTCCCACTGTTGTCCCCCCTCCATAGGGC
AGGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01BA9G8 length=281 xy=0421_0810 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01B0SNX length=270 xy=0712_0139 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGTTTGGTGGGCCGTTACCCCGCCAAC
TGCCTAATCAGACGCGGGGCCATCTCATACCACCGGAGTTTTTCCTGCTGTACCATGCGG
TACTGTAGGCTTATGCGGTATTAGCAGCCGTTTCCAGCTGTTGTCCCCCTGTACAGGGCA
GGTTACCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01AYO4X length=270 xy=0278_0303 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCT
CCCAACGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCA
GGTTCCATACGTGTTACTCACCCGTGCCGG
>FLP3FBN01DSQSF length=251 xy=1440_1805 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTC
AAGCTAATGTCACGCGAGCCTATCCTCATCCGACGAATCTTTAGATGGATTCAGATGCCT
GATTCCATCACCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCA
AGTTGCTCACG
>FLP3FBN01D6CH7 length=281 xy=1595_1597 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01DSRTP length=255 xy=1440_3147 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATGCCGTTAAACCTTTTCACACTGTGTCATGCGA
CACCGTGCGCTTATGCGGTATTAGCAACTGTTTCCAGTTGGTATCCCCCTGCATACGGCA
GGTTATCCACGCGTT
>FLP3FBN01ENV4O length=264 xy=1795_0982 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCCCACCAAC
TACCTAATCAGACGCGGGTCCATCTTACACCACCTCAGTTTTTACCACTGTACCATGCAG
TACTGTGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCA
GGTTGCCCACGCGTTACTCACCCG
>FLP3FBN01DZOKU length=263 xy=1519_1948 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCTCGCCAA
CCAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCTTACAGGATGTTTCACAGAAAG
AAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTG
CGCGGGCAGGTTCCCTACGTGTT
>FLP3FBN01C5M5W length=240 xy=1177_1266 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCCACCAAC
TAGCTAATCAGACGCAAGCCCATCTATCAGCGGATTGCTCCTTTTCTAGCTATATCATGC
GATACTACTAGCTTATGCGGTATTAGCAATGATTTCTCACTGTTATTCCCCTCTGATAGG
>FLP3FBN01CLMSS length=258 xy=0949_1562 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGCTGCCTTGGTGGGCCGTTACCCCGCCAAC
TAGCTAATCAGACGCGGAACCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCAGTCTCTTGGGCA
GGTTACCCACGTGTTACT
>FLP3FBN01DY2AX length=268 xy=1512_1751 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTCTACCCCGCCAAC
AAGCTAATCAGACGCGGGTCCATCGTATACCACCGGAGTTTTTCACACCGGACCATGCGA
TCCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01CFM8N length=268 xy=0881_0725 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAAC
TGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGG
AACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCA
GGTTATCCACGCGTTACTCACCCGTCCG
>FLP3FBN01DYIFI length=287 xy=1506_0572 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
TAGCTAATGCGCCGCATGGCCATCCGTAGCCGGTGTTACCCTTTAAACCCCAAGAGATGC
CTCTCGGAGTTATTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTACGGG
CAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCGGAAG
>FLP3FBN01A3EIE length=253 xy=0331_2724 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTC
AAGCTAATGTCACGCGAGCCTATCCTCATCCGACGGATCTTTAGATGGAACCAGATGCCT
GATTCCATCGCCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCA
AGTTGCTCACGCG
>FLP3FBN01D02CB length=272 xy=1535_0905 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCT
CCCAACGTTCTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCA
GGTTCCATACGTGTTACTCACCCGTGCGCCGG
>FLP3FBN01EY83L length=258 xy=1924_2623 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCACCAAC
TAGCTAATCAGACGCGGGTCCATCTCATACCACCGGAGTTTTTCACACCGGATCATGCAA
TCCTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATGAGGCA
GGTTACCCACGCGTTACT
>FLP3FBN01CS0ML length=264 xy=1033_2011 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCGCCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
CAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTCACACCGAACCATGCGG
TTCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCG
>FLP3FBN01EDF0K length=271 xy=1676_0962 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGCTGCCTTGGTGGGCCGTTACCCCGCCAAC
TAGCTAATCAGACGCGGAACCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTTATTTCTAAGTGTTACTCCCCCGGTATACGG
CTAGGTTTTCCACGCGTTACTCACCCGTCCG
>FLP3FBN01AO1DG length=257 xy=0168_0162 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGTCCGTTACACCGCCAAC
TAACTAATGCGACGCATGCCCATCCGCCACCGGAATCAACCTTTGGCACCAACAGGATGT
CCCGTCGATGCATTATGCCGTATTAGACGGAATTTCTCCCGATTATCCCCCTGTACAGGG
CAGGTTCCCCACGCGTT
>FLP3FBN01CLZGK length=244 xy=0953_1586 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAAC
TAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGA
CATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCT
TGGG
>FLP3FBN01A2AJL length=255 xy=0319_0079 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGGCCCATCGTATACCGTCTCGGCTTTTCACACTCCAGTATGCAC
TGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATATGGCA
GGTTACCCACGCGTT
>FLP3FBN01AW4PZ length=247 xy=0260_0917 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAG
GACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACC
>FLP3FBN01DYX26 length=250 xy=1511_0384 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCAACCTCTCAGTCCGGCTACCGATTGTCGGTTTGGTGGGCCGTTACCCCGCCAAC
TGCCTAATCGGACGCGAGCCCATCTCTCAGCGGATTGCTCCTTTGATATATCTACCATGC
GATACATATATTTCATGCGGTATTAGCGTTCGTTTCCAAACGTTATCCCCCTCTGAGAGG
CAGGTTGCTC
>FLP3FBN01DUDZY length=264 xy=1459_0716 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGG
ACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCA
GGTTGGATACGTGTTACTCACCCG
>FLP3FBN01A44JU length=280 xy=0351_1208 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCGGCACCGGGATGCC
CCGACGCCGCGTCACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTGGCGGGC
AGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01C0GMZ length=266 xy=1118_1193 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTTGCCTTGGTGGGCTGTTATCTCACCAAC
TAGCTAATCAGATGCGGGCCCATCTTTTACCGAATTTCTCCTTTCCTTCTCAGAAGATGC
CTCCTAAGAATATTATGCGGTATTAGTCACCGTTTCCAGTGATTATTCCCCAGTAAAAGG
CAGGTTGCCCACACGTTACTCACCCG
>FLP3FBN01ARHYF length=268 xy=0196_0277 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAAC
TGGCTAATCAGACACGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGG
AATCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01DNVVZ length=258 xy=1385_0413 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTC
AAGCTAATGTCACGCGAGCCTATCCTCATCCGACGGATCTTTAGATGGAACCAGATGCCT
GATTCCATCGCCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCA
AGTTGATCACGCGTTACG
>FLP3FBN01BAT9T length=275 xy=0416_1583 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCT
CCCAACGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCA
GGTTCCATACGTGTTACTCACCCGTGCGCCGGTCG
>FLP3FBN01BAL72 length=285 xy=0413_3440 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTAC
TGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCGGGGATGCCC
CCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAG
GTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGGCAGAGAG
>FLP3FBN01CIG7X length=255 xy=0913_1819 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTCTCCACGCGTT
>FLP3FBN01DVJS5 length=269 xy=1472_1655 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGCTGCCCCGCCAAC
AAGCTAATCAGACGCGGGCCCCTCCCATACCGCCGGAACTTTCCCCAGAAAGGCATGCGC
CTCCCTGGTTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTATGGGGCAG
GTTGCCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01ARTG1 length=288 xy=0199_2915 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCGCCGACT
AGCTAATGCGCCGCATGCCCATCCGTGGCCGGGATTGCTCCCTTTGGCGCCCGGGGATGC
CCCAAGGCCGCGTTACGCGGTATTAGACGGGGTTTTCCCCCGCTTATCCCCCTGCCACGG
GCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCGGCGG
>FLP3FBN01BV86P length=268 xy=0660_1263 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAG
GACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01EFIS6 length=272 xy=1699_3688 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGCCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCCTCTCCCCCGCCAAC
TAGCTAATCAGACGCGGGCCCATCCCATACCACCGGAGTTTTTCACACCAGGCCATGCGG
CCCTTGTGCGCTTATGCGGTTTTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTATGGGGC
AGGTTACCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01BJO96 length=252 xy=0517_1324 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
CAGCTAATCAGACGCGAGCCCATCTCAGAGCGATAAATCTTTGGCAGTCAGGGAGATGTC
TCCCAACTGCTTCATGCGGTATTAGCAGATGTTTCCATCTGTTATTCCCCACTCCAAGGC
AGGTTACTCACG
>FLP3FBN01EMI0C length=268 xy=1779_2858 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAAC
TGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGG
AACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCA
GGTTATCCACGCGTTACTCACCCGTCCG
>FLP3FBN01AOU7Q length=276 xy=0166_0372 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGGTCCATCGTATACCACCTCGGTTTTTCACACTGTCCCATGCGG
AACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCA
GGTTACCCACGCGTTACTCACCCGTCCGCCACTAAG
>FLP3FBN01DWLJX length=259 xy=1484_1419 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAGTGTGT
CCGATCACCCTCTCAGGCCGGATACCCGTCATAGCCTTGGTGAGCCATTACCTCACCAAC
AAGCTGATAGGACATAGGCTGATCCTTTAGCGAAAAACTTTCCCCCGTAGGGAGTATCCA
GTATTAATCACCGTTTCCAGTGGCTATCCCAGACTAAAGGGCACATAACCTATGCATTAC
TCACCCGTGCGCCACTAAT
>FLP3FBN01CKW1J length=264 xy=0941_0949 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGG
ACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCA
GGTTGGATACGTGTTACTCACCCG
>FLP3FBN01D7325 length=281 xy=1615_2079 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01BR4U6 length=272 xy=0613_1552 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGAGAAAATGCTC
ACTCATGATACCATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCA
GGTTGGATACGCGTTACTCACCCGTGCGCCGG
>FLP3FBN01A2KBL length=271 xy=0322_0463 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAAC
TACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGG
TTCTGGGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCA
GGTTGCCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01AT6OD length=268 xy=0226_2747 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGTATCATGCGA
TACTGTGCGCTTATGCGGTATTAGCACCTATTTCTAACTGTTATCCCCTCGTGTAAGGCA
GGTTGCCCACGCGTTACTCACCCGTCCG
>FLP3FBN01BD8E7 length=257 xy=0455_0145 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGTCTTGGTGGTCCGTTACACCGCCAAC
TAACTAATGCGACGCATGCCCATCCGCCACCGGAATCAACCTTTGGCACCAACAGGATGT
CCCGTCGATGCATTATGCCGTATTAGACGGAATTTCTCCCGATTATCCGGCTGTGGCAGG
CAGGTTGCATACGTGTT
>FLP3FBN01EKI61 length=280 xy=1756_3995 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACCGATCGATGCCTTGGTGGGCCGTTACCCCGCCAAC
AAGCTAATCGGACGCATCCCCATCCTTTACCGATAAATCTTTGCTCCGACACAGATGTCC
GGGACGGAGAACATGGGGGATTAATCCGCCTTTCGACGGGCTGAACCCCACGTAAAGGGA
AAGGGTTGGATACGCGTTACTCACCCGTGCGCCGGTCGCC
>FLP3FBN01EDR54 length=255 xy=1680_0330 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTCTCCACGCGTT
>FLP3FBN01CODQH length=265 xy=0980_2807 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAAC
AAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGA
AGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGC
GCGGGCAGGTTCCCTACGTGTTACT
>FLP3FBN01BVQJR length=264 xy=0654_1685 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCACCAAC
TAGCTAATCAGACGCGGGTCCATCTCATACCACCGGAGTTTTTCACACTGTCCCATGCAG
GACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCGAGGCA
GGTTGCCCACGCGTTACTCACCCG
>FLP3FBN01EV1JR length=251 xy=1888_0325 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTC
AAGCTAATGTCACGCGAGCCTATCCTCATCCGACGAATCTTTAGATGGATTCAGATGCCT
GATTCCATCACCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCA
AGTTGCTCACG
>FLP3FBN01BQAKZ length=245 xy=0592_1665 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
TAGCTAATCAGACGCGAGTCCATCTCAGAGCGATAAATCTTTGGCAGTCAGAGCCATGCG
ACCCAACTGCATTATGCGGTATTAGCAGCTGTTTCCAGCTGTTATTCCCCACTCCAAGGC
AGGTT
>FLP3FBN01CFTC2 length=279 xy=0883_0468 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGG
CAGGTTGCATACGTGTTCACTCCNCCGNGCGTCCGCCGG
>FLP3FBN01BDWJA length=244 xy=0451_1124 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAAC
TAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGA
CATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCT
TGGG
>FLP3FBN01CNXDO length=268 xy=0975_2090 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
CCGATCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTAAGCCGTTACCTTACCAAC
TAGCTAATCAGACGCGGGTCCATCCTGTACCGCAAAAGCTTTGATACTTCTACCATGCGA
TAAAAGTATTTTATCTCGTATTAGCATACCTTTCGGTATGTTATCCGTGTGTACAGGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01ASXZX length=281 xy=0212_2187 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGGCCATCCGTAGCCGGTGTTACCCTTTAAACCCCAAGAGATGC
CTCTCGGAGTTATTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTACGGG
CAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01B9E1U length=268 xy=0810_0992 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCCTGGTAGGCCTTTACCCCACCAAC
TAGCTAATCAGACGCGGGCCCATCTCATACCACCGGAGTTTTTACCGCTGTGCCATGCGG
CACCGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATGAGGCA
GGTTGCCCACGCGTTACTCACCCGTCCG
>FLP3FBN01AKJ8H length=258 xy=0117_0223 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
AAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAG
GACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACT
>FLP3FBN01BD3KF length=253 xy=0453_2045 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAGTGTGT
CCGATCACCCTCTCAGGCCGGATACCCGTCATAGCCTTGGTGAGCCATTACCTCACCAAC
AAGCTGATAGGACATAGGCTGATCCTTTAGCGAAAAACTTTCCCCCGTAGGGAGTATCCA
GTATTAATCACCGTTTCCAGTGGCTATCCCAGACTAAAGGGCACATAACCTATGCATTAC
TCACCCGTGCGCC
>FLP3FBN01DBIH6 length=268 xy=1244_0732 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAAC
CAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGCTTCATGCAA
AGCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCA
GGTTGCCCACGCGTTACTCACCCGTCCG
>FLP3FBN01C61FL length=281 xy=1193_0879 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01ENY5A length=269 xy=1796_0796 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAGGGAAAAGCCATGCGG
CTTCCCCCGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGCCAG
GTTGCCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01ESGKH length=281 xy=1847_1103 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01DMOT4 length=270 xy=1371_1962 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTCTACCCCGCCAAC
AAGCTAATCAGACGCGGGTCCATCGTATACCACCGGAGTTTTTCACACCGGACCATGCGA
TCCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCA
GGTTACCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01BEC0A length=242 xy=0456_1992 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAAC
TAGCTAATCAGACGCGAGCTCATCTCAGAGCGATAAATCTTTGGCAATCAGAGAGATGCC
TCCCAATTGCATCATGCGGTATTAGCGGCTGTTTCCAACCGTTATTCCCCACTCCAAGGC
AG
>FLP3FBN01C1F87 length=270 xy=1129_2297 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTTGCCTTGGTGGGCTGTTATCTCACCAAC
TAGCTAATCAGATGCGGGCCCATCTTTTACCGAATTTCTCCTTTCCTTCTCAGAAGATGC
CTCCTAAGAATATTATGCGGTATTAGTCACCGTTTCCAGTGATTATTCCCCAGTAAAAGG
CAGGTTGCCCACACGTTACTTACCCGTCCG
>FLP3FBN01AR785 length=260 xy=0204_1591 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAAC
TAGCTAATGTCACGCATGCCCATCCCGCACCGGATCGCTCCTTTGACCGCTCCCCCATGC
AGAGGAACGGTGTCATGCCGTATTAGTCCGGATTTCTCCGGGTTATCCGTCTGTGCGGGG
CAGGTCGCATACGCGTTACT
>FLP3FBN01DT66J length=244 xy=1457_0073 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGACTTGGTGAGCCGTTACCTCACCAAC
TATCTAATCAGACGCGAGCCCATCTCTGAGCGAATTTCTTTGATATCCAAATCATGCGAT
TTAAATATGTTATGAGGTATTACCATCCGTTTCCAGAAGCTATCCCTCTCTCAGAGGCAG
GTTG
>FLP3FBN01ER6TY length=264 xy=1844_0772 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTTGCTTTGGTAGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGGTCCATCGTATACCTCCGGAAATTTTCACACTCTGTCATGCGA
CAGTGTGCGCTTATGCGGTATTAGCAGTTGTTTCCAACTGTTATCCCCCTGTATACGGCA
GGTTACCCACGCGTTACTCACCCG
>FLP3FBN01BWI9J length=254 xy=0663_2037 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGGTCAACCTCTCAGTCCGGCTACTGATCGTCGCCTAGGTGGGCCGTTACCCCGCCTAC
CAGCTAATCAGACGCGAGGCCATCTTCCAGCGATAAATCTTTGACATGTCCGTGATGCCA
CGGTCATGTGTCATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGGAAGGCA
GGTTCCTCACGTCG
>FLP3FBN01BR0YS length=255 xy=0612_0594 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTCTCCACGCGTT
>FLP3FBN01BTOX2 length=264 xy=0631_0504 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTACCGATCGTTGCTTTGGTGGGCCGTTGCCCCGCCAAC
TGGCTAATCGGACGCGGATCCATCGTATGCCGTTAAACCTTTTCACACTGTGTCATGCGA
CACCGTGCGCTTATGCGGTATTAGCAACTGTTTCCAGTTGGTATCCCCCTGCATACGGCA
GGTTATCCACGCGTTACTCACCCG
>FLP3FBN01BR1S4 length=268 xy=0612_1686 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
CCGATCACCCTCTCAGGTCGGCTACTGATCGCCGCCTTGGTAAGCCGTTACCTTACCAAC
TAGCTAATCAGACGCGGGTCCATCCTGTACCGCAAAAGCTTTGATACTTCTACCATGCGA
TAAAAGTATTTTATCTCGTATTAGCATACCTTTCGGTATGTTATCCGTGTGTACAGGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01BI8Q8 length=268 xy=0512_0386 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTTACACCACTAATGTTTTTCACTCTGTCCCATGCGG
GACTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTGTAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01C73G0 length=254 xy=1205_1026 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGTTCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTAGGCCTTTACCCCACCAAC
TAGCTAATGCGCCGCAGGCTCATCCATCAGTGATGCCAGGAGCATCTTTAAACTTTCGTC
CTATCCGGTATTAGCGATCGTTTCCAATCGTTGTCCCCGTCTGATGGGCAGATCACCTAC
GTGTTACTCACCCG
>FLP3FBN01EIDZ1 length=268 xy=1732_2255 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGCTGCCCCGCCAAC
TAGCTAATCAGACGCGGGCCCATCCCACACCGCCGGAGCTTTCCGCACCGCCCCATGCGG
GGCTGTGCGCTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTGTCCCCCTGTGTGGGGTA
GGTTGCCCACGCGTTACTCACCCGTCCG
>FLP3FBN01AE1JZ length=280 xy=0054_0781 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCGGCACCGGGATGCC
CCGGTGCCGCGTCACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTGGCGGGC
AGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01EEXTN length=244 xy=1693_1065 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAAC
TAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCACCTTTCAAACTCTAGA
CATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCT
TGGG
>FLP3FBN01BVGJF length=253 xy=0651_1001 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGCTGCCTTGGTGGGCCGTTACCCCGCCAAC
TAGCTAATCAGACGCGGAACCATCATATGCCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCG
>FLP3FBN01BWF4P length=255 xy=0662_2071 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTGCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
AAGCTAATCAGACGCGGAACCATCGTATACCACCGGAGTTTTTCACACTTTATCATGCGA
TCTCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATGAGGCA
GGTTACCCACGCGTT
>FLP3FBN01BU6SJ length=265 xy=0648_0657 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTACGCATCGTCGCCGTCGGTGGGCCGTTACCCCGCCGA
CTAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCC
TCCCAACGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGC
AGGTTCCATACGTGTTACTCACCCG
>FLP3FBN01DY8W5 length=240 xy=1514_2135 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCCACCAAC
TAGCTAATCAGACGCAAGCCCATCTATCAGCGGATTGCTCCTTTTCTAGCTATATCATGC
GATACTACTAGCTTATGCGGTATTAGCAATGATTTCTCACTGTTATTCCCCTCTGATAGG
>FLP3FBN01CXZHC length=253 xy=1090_0334 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTC
AAGCTAATGTCACGCGAGCCTATCCTCATCCGACGGATCTTTAGATGGAACCAGATGCCT
GATTCCATCGCCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCA
AGTTGCTCACGCG
>FLP3FBN01EDY3K length=281 xy=1682_1118 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTTGTCTTGGTGAGCCGTTACCCCACCAAC
TAACTAATACAACGCATGCCCATCCATTACCACCGGAGTTTTCAACCCAAGAAGATGCCT
CCCTGGATGTTATGGGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTAATGGGTAG
GTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCGACAA
>FLP3FBN01AF3UI length=264 xy=0066_1256 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAG
GACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCG
>FLP3FBN01CNBHT length=268 xy=0968_2399 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
CAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTCACACCGAACCATGCGG
TTCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01BW3YV length=269 xy=0670_0197 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGTCCACCCTCTCAGGCCGGCTGCTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAAC
TGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAAAGGATTACCATGCGG
TATTCCCTATTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGGTACAGGGCAG
GTTGCCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01DZVEP length=268 xy=1521_2607 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCCCATACCACCGGAGTTTTTCACACTGTTCCATGCGA
AACCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATGAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01EK704 length=274 xy=1764_3414 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGG
>FLP3FBN01DQ5NS length=271 xy=1422_1494 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAAC
TACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGG
TTCCGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCA
GGTTGCCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01CX6KS length=268 xy=1092_1338 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTTGCCGCTGTACCATGCGG
TACTGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCA
GGTTGCCCACGCGTTACTCACCCGTCCG
>FLP3FBN01BDSR0 length=254 xy=0450_0350 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGTTCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTAGGCCTTTACCCCACCAAC
TAGCTAATGCGCCGCAGGCTCATCCATCAGTGATGCCAGGAGCATCTTTAAACTTTCGTC
CTATCCGGTATTAGCGATCGTTTCCAATCGTTGTCCCCGTCTGATGGGCAGATCACCTAC
GTGTTACTCACCCG
>FLP3FBN01D5PH5 length=268 xy=1588_0459 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCT
CCCAACGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCA
GGTTCCATACGTGTTACTCACCCGTGCG
>FLP3FBN01AMPHD length=271 xy=0141_2031 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAAC
AAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGA
AGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGC
GCGGGCAGGTTCCCTACGTGTTACTCACCCG
>FLP3FBN01D88A5 length=269 xy=1628_0959 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
TAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGG
AACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCCTGTATACGGC
AGGTTATCCACGCGTTACTCACCCGTCCG
>FLP3FBN01DTZ2Y length=274 xy=1454_3160 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAAC
CAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCT
CTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAG
GTTGCATACGCGTTACGCACCCGTGCGCCGGTCG
>FLP3FBN01EAT6O length=268 xy=1646_2238 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAG
GACCGTGCGCCTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01AYJGP length=281 xy=0276_1143 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCTAC
TGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCGGGGATGCCC
CCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAG
GTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCAG
>FLP3FBN01BDJRH length=272 xy=0447_0955 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCTTATACCGGTAAACCTTTAATCATGAGAAAATGCTC
ACTCATGATACCATCTTGTATTAATCTCCCTTTCAGAAGGCTATCCAAGAGTATAAGGCA
GGTTGGATACGCGTTACTCACCCGTGCGCCGG
>FLP3FBN01B2CCU length=260 xy=0729_2684 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGGCTGTATCTCAGTCCCAGTGTGG
CCGTCCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
TAGCTAATGCGCCGCGAGCCCATCCTATACCGGATTTCTCCTTTTACCCCTCAGCCATGC
GACTGCGTGGTCTTATACCGTATTAACTACAGTTTCCCGTAGGTATCCCGTTGTATAGGG
CAGGTTGCTCACGTGTTACT
>FLP3FBN01BSTZH length=284 xy=0621_1339 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAGTGTGG
CCGATCACCCTCTCAGGTCGGCTACGTATCGTTGCCTTGGTAAGCCGTTACCTTACCAAC
TAGCTAATACGGCGCGGGTCCATCTATAAGTGACAGCCGAAACCGTCTTTCAACATTGAA
CCATGCGGTTCAATATATTATCCGGTATTAGCCCCGGTTTCCCGGAGTTATCCCAGTCTT
ATAGGTAGGTTACCCACGTGTTACTCACCCGTCCGCCGCTAACA
>FLP3FBN01EG5XE length=251 xy=1718_2480 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
TAGCTAATCAGACGCGAGGCCATCTTTCAGCGATAAATCTTTGACATAAATGCCATGCGA
CACCTATGTGTTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGAAAGGCA
GGTTCCTCACG
>FLP3FBN01DHSDB length=268 xy=1315_2637 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCT
CCCAACGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCA
GGTTCCATACGTGTTACTCACCCGTGCG
>FLP3FBN01BG5PZ length=280 xy=0488_1445 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCTCACCAAC
TATCTAATGGAACGCATGCCCATCTATCAGCGATAATTCTTTAACAAATATTTCCATGTG
GAACCCCTGTTTTATGCGGTATTAGTCCGACTTTCGCCGGGTTAGTTCCCCTCGTACAGG
CCAGGTTGGCCCACCGCGTTACTCACCCGTCCGCCACTCA
>FLP3FBN01ERW7M length=250 xy=1841_0592 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCCTTGGTAGGCCATTACCCCACCAAC
TAGCTAATCGGCCGCGAGCCCATCTTTCAGCGGATTGCTCCTTTGGCTGCATCGGGATGC
CCCATCGCAGCGTTATGCGGTATTAGCAGTCGTTTCCAACTGTTATCCCCCTCTGAAAGG
CAGGTTGCTC
>FLP3FBN01A3D61 length=244 xy=0331_2315 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAAC
TAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGA
CATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCT
TGGG
>FLP3FBN01AOYAO length=253 xy=0167_0270 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTC
AAGCTAATGTCACGCGAGCCTATCCTCATCCGACGGATCTTTAGATGGAACCAGATGCCT
GATTCCATCGCCATGGGGCATTAGACGCCGTTTCCAGCGATTATTCCCCTGATGAGGGCA
AGTTGCTCACGCG
>FLP3FBN01A5KJQ length=281 xy=0356_1460 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01BOMSP length=272 xy=0573_2007 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCCTTGGTGGGCCGTTACCCCGCCAAC
AAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAATTTTCAAGTAAAAGACATGCGT
CTCCTACTGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAG
GTTCTCCACGCGTTACTCACCCGTCCGCCACT
>FLP3FBN01D9NXA length=255 xy=1633_0716 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTT
>FLP3FBN01D0A3H length=277 xy=1526_2459 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAATGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>FLP3FBN01DKHEO length=268 xy=1346_1422 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGCCACCCTCTCAGGCCGGCTACTGATCGTCGGTTTGGTGGGCCTTTACCCCGCCAAC
TGCCTAATCAGACGCGGGCCCATCCCATACCGCAAAAGCTTTCCATGCCAGGGCATGCGC
CCCCGCATGTGCATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTATGGGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01EJF8M length=262 xy=1744_2660 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAAC
CAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGA
AGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGC
GCGGGCAGGTTCCCTACGTGTT
>FLP3FBN01DBC7G length=275 xy=1242_2058 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTGCCAAC
AAGCTAATCAGACGCGGGTCCATCTCGCACCACCGGAGTTTTCAGGGCGGGGGCATGCGC
CCCCTCCCGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAG
GTTACCCACGCGTTACTCACCCGTCCGCCACTAAG
>FLP3FBN01CKKP0 length=286 xy=0937_1366 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGCCCATCCGTGGCCGGGATTGCTCCCTTTGGCGGCCCGGGGAT
GCCCCAAGGCCGCGTTACGCGGTATTAGACGGGGTTTCCCCCGCTTATCCCCCTGCCACG
GGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCGG
>FLP3FBN01A38C0 length=268 xy=0341_0450 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
CAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTCACACCGAACCATGCGG
TTCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01DR8GQ length=281 xy=1434_2632 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTAGACATCGTCGGCTTGGTGGGCCGTTGCCCCGCCAAC
TACCTAATGTCGCGCGTGCCCGTCCCGTACCACCGGAATTTTAAATCGAGAGCCATGCGG
CTCTCGAGTATCATGGGATGTTAGTCCACGTTTCCGCGGGTTATCTCCCGGTACGGGGTT
GGTTGCACACGTGTTACTCACCCGTGCGCCGGTCGCCGGCG
>FLP3FBN01C170V length=261 xy=1138_1421 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGCCGCATGCCCATCCTTTACCGGATCGCTCCTTTGACATACCGATCATGC
GGTCGGTATGTGTTATGAGGTATTAGACGGAATTTCTCCCGATTATCCCTCTGTAAAGGG
CAGGTCGCATACGTGTTACTC
>FLP3FBN01DOBTJ length=281 xy=1390_0581 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01DI8DL length=243 xy=1332_0407 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
CAGCTAATCAGACGCGAGTCCATCTCAGAGCGATAAATCTTTGGCAGTCAGAGCCATGCG
GCCCAACTGCATTATGCGGTATTAGCACATCTTTCGATGTGTTATTCCCCACTCCAAGGC
AGG
>FLP3FBN01BW7BY length=287 xy=0671_0460 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTCACACCGCCAAC
TACCTAATGCGCCGCATGCCCAGCCGCCACCGGATTACTCCTTTCGCCCGGACCGGATGC
CCGGTCCGGGCGGCATGGGGTATTAGGCCGGGTTTCCCCGGGTTATCCCCCTGTGGCGGG
CAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCAGGTG
>FLP3FBN01BWI6J length=282 xy=0663_1929 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTGGTCTTGGTGAGCCGTTACCCCACCAAC
TAACTAATACAACGCATGCCCATCCATTACCACCGGAGTTTTCAACCCAAGAAGATGCCT
CCCTGGATGTTATGGGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTAATGGGTAG
GTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCGACAAT
>FLP3FBN01CN8VK length=273 xy=0979_0606 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCGGCACCGGGATGCC
CCGACGCCGCGTCACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTGGCGGGC
AGGTTGCATACGTGTTACTCACCCGTGCGCCAG
>FLP3FBN01B12IL length=244 xy=0726_2219 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
TAGCTAATCAGACGCGAGGCCATCTTTCAGCGATAAATCTTTGACATAAATGCCATGCGA
CACCTATGTGTTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGAAAGGCA
GGTT
>FLP3FBN01DTCGT length=268 xy=1447_1227 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCACCAAC
TGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGG
AACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCA
GGTTATCCACGCGTTACTCACCCGTCCG
>FLP3FBN01C7P3U length=277 xy=1201_0088 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>FLP3FBN01BUFD0 length=277 xy=0639_2006 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGCTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>FLP3FBN01BI3Q6 length=264 xy=0510_2096 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTACCAAC
TAGCTAATCAGACGCGAGGCCATCTCTCAGCGATAAATCTTTGGTACAGATACCATGCGA
TACCCGTACGTTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCACTGAAAGGCA
GGTTCCTCACGCGTTACTCACCCG
>FLP3FBN01CKM98 length=275 xy=0938_0590 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGTCCACCCTCTCAGGCCGGCTGCTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAAC
TGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAAAGGATTACCATGCGG
TATTCCCTATTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGGTACAGGGCAG
GTTGCCCACGCGTTACTCACCCGTCCGCCACTAAG
>FLP3FBN01C4VM7 length=258 xy=1168_2465 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
CAGCTAATCAGACGCGGGCCCATCGCATACCACCGGAGTTTTTCACACCGAGCCATGCGG
CTCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGGTATGCGGCA
GGTTGCCCACGCGTTACT
>FLP3FBN01CVDF5 length=255 xy=1060_1347 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTCTCCACGCGTT
>FLP3FBN01CM73M length=253 xy=0967_2096 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCGAGGCA
GGTTGCCCACGCG
>FLP3FBN01CLMTS length=258 xy=0949_1598 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCCTTGGTAGGCCGTCACCCTTCCAAC
TAGCTAATCAGACGCGGGTCCATCCTTTTGCGATAGCATCTTCAGAGGCCATCTTTCCTT
AAAGAATCATGCGATCCTCTATTATTATGCGGTATTAGCTCCGATTTCTCGAAGTTGTTC
CCCACAAAAGGGCAGGTT
>FLP3FBN01ANKI1 length=264 xy=0151_1307 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTTTTACCTCACCAAC
TAGCTAATCAGACGCGGAACCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTTTCCACGCGTTACTCACCCG
>FLP3FBN01CEL0P length=279 xy=0869_1639 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAAC
CAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCT
CTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCCTGTTACAGGTA
GGTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01DQMD6 length=271 xy=1416_1100 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCGGCACCGGGATGCC
CCGATGCCGCGTCACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTGGCGGGC
AGGTTGCATACGTGTTACTCACCCGTGCCGG
>FLP3FBN01AIT2F length=281 xy=0097_1573 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGACTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01ENHOZ length=268 xy=1790_2753 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCGG
GACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01A5RUT length=244 xy=0358_2739 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAAC
TAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGA
CATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCT
TGGG
>FLP3FBN01AY6CG length=272 xy=0283_2126 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTTCCAGTGTGG
CCGTCCACCCTCTCAGGCCGGCTGCTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAAC
TGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAAAGGATTACCATGCGG
TATTCCCTATTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGGTACAGGGCAG
GTTGCCCACGCGTTACTCACCCGTCCGCCACT
>FLP3FBN01EQJ0R length=247 xy=1825_2377 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTACCGATCGTCGCTTTGGTGGGCTTCTACCCCGCCAAC
TGGCTAATCGGACGCGGATCCATCGTATGCCGATAAATCTTTTCACACTATACCATGCGG
TACTGTGCGCTTATGCGGTATTAGCAACTGTTTCCAGTTGGTATCCCCCTGCATACGGCA
GGTTACC
>FLP3FBN01DNNDF length=269 xy=1382_1665 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGCTGCCCCGCCAAC
AAGCTAATCAGACGCGGGCCCCTCCCATACCGCCGGAGCTTTCCCCACAAAGGCATGCGC
CTCCCTGGTTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTATGGGGCAG
GTTGCCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01BBWRU length=270 xy=0428_2328 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACCGTATCATGCGA
TACTGTGCGCTTATGCGGTTTTAGCAGCCGTTTCCGGCTGTTATCCCCCGGTACGGGGCA
GGTTCCCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01CJZCS length=266 xy=0930_2346 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCTCGCCAA
CCAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAG
AAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTG
CGCGGGCAGGTTCCCTACGTGTTACT
>FLP3FBN01EJZ09 length=258 xy=1750_3739 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
CAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAG
GACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACT
>FLP3FBN01BGWL1 length=268 xy=0485_1927 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTCTACCCCGCCAAC
AAGCTAATCAGACGCGGGTCCATCGTATACCACCGGAGTTTTTCACACCAGACCATGCGA
TCCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01AYFQO length=270 xy=0275_0414 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCTTTGGTAGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGGTCCATCTCATACCGATAAATCTTTTCCGTCCGTATCATGCGA
TACTAGCGGGTTATGCGGTATTAGCGGTCGTTTCCAACTGTTATCCCCCTGTATGAGGTA
GGTTACCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01C9PO9 length=252 xy=1223_2763 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCACGGTGGGCCGTTACCCCGCCGTC
AAGCTAATGTCACGCGAGCCTATCCTCATCCGACGAATCTTTAGATGGATTCAGATGCCT
GATTCCATCACCATGGGGCGATTAGACGCCGGTTCCAGCGATTATTCCCCTGATGAGGGC
AAGTTGCTCACG
>FLP3FBN01DR99Q length=268 xy=1435_0876 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCT
CCCAACGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCA
GGTTCCATACGTGTTACTCACCCGTTCG
>FLP3FBN01BHOAC length=278 xy=0494_0930 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTGGTCTTGGTGAGCCGTTACCCCACCAAC
TAACTAATACAACGCATGCCCATCCATTACCACCGGAGTTTTCAACCCAAGAAGATGCCT
CCCTGGATGTTATGGGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAG
GTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCAT
>FLP3FBN01CGYKR length=269 xy=0896_0633 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGATCAGTCTCTCAACTCGGCTACGTATCATCGCTTTGGTGGGCCGTTACCCCGCCAAC
TGGCTAATACGCCGCAGGTCCATCCATGTCCACACCCGAAGGTCCTTTAATATAGTCAAG
ATGCCTTGTCTATACCTATCCGGTTTTAGCAGACGTTTCCATCTGTTATCTCGGGTACAT
GGGCAGGTTCCCTACGTGTTACTCACCCG
>FLP3FBN01BJE5P length=277 xy=0514_0491 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATCTGG
CCGGTCGGTCTCTCAACCCGGCTACCCATAGAAGCCTTGGTGGGCCGTTACCCCGCCAAC
TAGCTAACAGGCCGCGGGCCCATCCCTCTCCGCCAGAGCTTTCCCGACGCTTCCATGCGA
AAGCGTCGGAGTATCCGGTATTATCCCCGGTTTCCCGAGGCTATCCCGATGAGAGGGGCA
GGTTGCCCACGTGTTACTCAGCCGTTCGCCACTATCC
>FLP3FBN01DI95G length=271 xy=1332_2706 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGTCCGTTACACCGCCAAC
TAACTAATGCGACGCATGCCCATCCTTCACCGAAATTCTTTCCCCCTCGGAAGATGCCTC
CCAAGGAGTATATGCGGTATTAGGCGAAATTTCTTCCGGTTATCCCGCTGTAAAGGGTAG
GTTGCATACGTGTTACTCACCCGTGCGCCGG
>FLP3FBN01DHLY4 length=271 xy=1313_2542 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCGCCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTAGGCTTCTACCCCACCAAC
TACCTAATCAGACGCGGGCCCATCTTACACCACCTCAGTTTTTACCTCTGTACCATGCGG
TACCGGGGTCTTATGCGGTATTAGCAATCATTTCTAATTGTTATCCCCCTGTGTAAGGCA
GGTTGCCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01BHLW4 length=258 xy=0493_1958 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCCTTGGTGGGCCTTTACCCCGCCAAC
AAGCTAATCAGACGCGGATCCATCGTATACCACCAAAAGCTTTAGCTTTTTGTTTTCCAC
ACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCC
CCCAGTATACGGCAGGTT
>FLP3FBN01B2YIQ length=263 xy=0736_2736 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
CAGCTAATACAACGCATGCCCATCATCAACCACCGGAGTTTTCAACCCCATGCGATGCCG
CATGAGATATTATGGGGTATTAGTACCAATTTCTCAGTGTTATCCCCCTGTTGATGGTAG
GTTGCATACGCGTTACGCACCCG
>FLP3FBN01A1YKO length=240 xy=0315_0950 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCCTTGGTGGGCTTTTATCTCACCAAC
TAGCTAATCAGACGCGGGTCGCTCTCCCAGCGGCAGCTGCTCTGCTCTGCCACCTTTCTT
TCTTCCATCATGCGATGGTTGAACCTCATGCGGTATTAGCTGTGATTTCTCACGGTTATT
>FLP3FBN01A44HD length=255 xy=0351_1119 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCACCAAC
TAGCTAATCAGACGCGGGTCCATCTCATACCACCGGAGTTTTTCACACTGGATCATGCAA
TCCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTGTAAGGCA
GGTTCTCCACGCGTT
>FLP3FBN01ADKEK length=249 xy=0037_1530 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTATGGACCGTGTCTCAGTTCCATTGTGG
CCGATCAGTCTCTCAACTCGGCTATGCATCATCGCCTTGGTAAGCCGTTACCTTACCAAC
TAGCTAATGCACCGCAGGTCCATCCCAGAGTGATAGCCAAAGCCATCTTTCAAACAAAAG
CCATGTGGCTTTTGTTGTTATGCGGTATTAGCATCTGTTTCCAAATGTTATCCCCCGCTC
CGGGGCAGG
>FLP3FBN01A4HWP length=281 xy=0344_0535 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTGGTCTTGGTGAGCCGTTACCCCACCAAC
TAACTAATACAACGCATGCCCATCCATTACCACCGGAGTTTTCAACCCAAGAAGATGCCT
CCCTGGATGTTATGGGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTAATGGGTAG
GTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCGACAA
>FLP3FBN01A7BP1 length=274 xy=0376_1415 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGCCCATCCGTCACCGGATTGCTCCTTTGACCGCTCCGGGATGC
CCCGGAATGGTGTTACGCGGAATTAGTCGGAATTTCTTCCGGTTATTCCCCTGTGACGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGG
>FLP3FBN01AS674 length=281 xy=0215_1858 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
TAGCTAATGCGCCGCATGGCCATCCGTAGCCGGTGTTACCCTTTAAACCCCAAGAGATGC
CTCTCGGAGTTATTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTACGGG
CAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01A1FYD length=270 xy=0309_1395 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCCCGCCAAC
TACCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGGACCATGCGG
TCCTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTGTCCCCCTGTACAGGGCA
GGTTACCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01BJH0F length=261 xy=0515_0093 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCACCAAC
TACCTAATGCGCCGCATGCCCATCCTTTACCGGATCGCTCCTTTGACATACCGATCATGC
GGTCGGTATGTGTTATGAGGTATTAGACGGAATTTCTCCCGATTATCCCTCTGTAAAGGG
CAGGTCGCATACGTGTTACTC
>FLP3FBN01EEXQK length=268 xy=1693_0954 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
TAGCTAATGCGCCGCATGCCCATCCGTCACCGGAATCGCTTCCTTTGATCTCCCGGAGAT
GCCTCCGGAAGATGTCACGCGGTATTAGACGGGATTTCTCCCGATTATCCCCCTGTGACG
GGCAGGTTGCATACGTGTTACTCACCCG
>FLP3FBN01ADKU8 length=268 xy=0037_2130 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAAC
TGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGG
AACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCA
GGTTATCCACGCGTTACTCACCCGTCCG
>FLP3FBN01D4CIQ length=271 xy=1572_2512 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTACCAAC
TACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGG
TTCCGCGGTCTTATGCGGTATTAGCAGTCATTTCTAGCTGTTATCCCCCTGTGTAAGGCA
GGTTGCCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01BN9BB length=244 xy=0569_0917 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGTCCGCCCTCTCAGGTCAGCTACTGATCGTCGCCTTGGTAGGCCATTACCCTACCAAC
TAGCTAATCAGACGCGAGGCCATCTCTCAGCGATAAATCTTTGATATATCTGCCATGCGA
CAAACATATATTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGAAAGGCA
GGTT
>FLP3FBN01AEJLU length=262 xy=0048_2096 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAAC
AAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGA
AGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGC
GCGGGCAGGTTCCCTACGTGTT
>FLP3FBN01CHVM4 length=259 xy=0906_2526 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAACCATGTGAACATGCGG
ACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCA
GGTTGGATACGTGTTACTC
>FLP3FBN01EBPW0 length=251 xy=1656_2402 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACTCTCTCAAGCCGGCTACTGATCGTCGCTTTGGTAGGCCATTACCCTGCCAAC
TGGCTAATCAGACGCGGGCCCATCCTGTACCTATAAATATTTGATAGCAATATCATGCGG
TATCGCTATGTTATGCGGTGTTAGCAGCCGTTTCCAGCTGTTATCCCCCTGTACAGGGCA
GGTTACCCACG
>FLP3FBN01BGO9L length=277 xy=0483_0599 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGGCTTGGTGGTCCGTTACACCGCCAAC
TACCTAATGCGACGCATGCCCATCCGCTACCGGATCGCTCCTTTGGAATCCCGGGGATGT
CCCCGGAACTCGTTATGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTAGCGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>FLP3FBN01EDVUS length=275 xy=1681_1010 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGTGGACTTGGTGAGCCGTTACCTCACCAAC
TATCTAATGGAACGCATGCCCATCTATCAGCGATAATTCTTTAACAAATATTTCCATGTG
GAACCCCTGTTTTATGCGGTATTAGTCCGACTTTCGCCGGGTTATTCCCCTCCTGATAGG
TAGGTTGCATACGCGTTACTCACCCGTCGCGCCGG
>FLP3FBN01BOAPJ length=251 xy=0569_2725 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGCTGCCCCGCCAAC
AAGCTAATCAGACGCGGGCCCATCGCATACCACCGGAGTTTTTCACACCAAGCCATGCGG
CTCTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCA
GGTTGCCCACG
>FLP3FBN01B9SW2 length=248 xy=0814_2580 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTCTACCCCGCCAAC
AAGCTAATCAGACGCGGGTCCATCGTATACCACCGGAGTTTTTCACACTGTCCCATGCAG
GACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGATGCAGGTCC
AGTCCTAC
>FLP3FBN01AST80 length=255 xy=0211_1426 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTCTCCACGCGTT
>FLP3FBN01DOAO5 length=263 xy=1389_3223 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTGGCCTTGGTGAGCCGTTACCTCACCAAC
CAGCTAATACAACGCATGCCCATCATCAACCACCGGAGTTTTCAACCCCATGCGATGCCG
CATGAGATATTATGGGGTATTAGTACCAATTTCTCAGTGTTATCCCCCTGTTGCTGGTAG
GTTGCATACGCGTTACTCACCCG
>FLP3FBN01CSLHO length=271 xy=1028_2874 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCCACCAAC
CAGCTAATCAGACGCGGGCTCATCTTATACTACCGGAGTTTTTCACACAGAAACATGCGT
CCCCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATAAGGCA
GATTACCCACGTGTTACTCACCCGTCCGCCG
>FLP3FBN01CSRAJ length=272 xy=1030_2201 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGCTGACTTGGTAGGCCTCTACCCCACCAAC
TATCTAATCAGACGCGGGCCCATCCCGCTACCGCCGCAGCTTTCCGCCGGGGGCCATGCG
GCCCCCGCCGTGTATGCGGTATTAGCGGCCGTTTCCGGCCGTTGTCCCCCAGTGCCGGGG
CAGGTTGCCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01DVOX8 length=272 xy=1474_0126 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCT
CCCAACGTTCTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCA
GGTTCCATACGTGTTACTCACCCGTGCGCCGG
>FLP3FBN01ETLIN length=278 xy=1860_0925 region=1 run=R_2008_12_09_13_51_01_
ACGGTGAGTGTCCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGGAAAACCTCTCAGTCCCCCTATGTATCGTAGCCTTGGTGAGCCGTTACCTCACCAAC
TAGCTAATACAACGCATGCCCATCTCTAACCATCGGAATTTTCAACCATCTAAGATGCCT
TAAATGATATTATGGGGTATTAGACCAAATTTCTTCGGATTATCCCCCTGTTAAAGGTAG
GTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01EIT5M length=284 xy=1737_2712 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
TAGCTAATGCGCCGCATGGCCATCCGTAGCCGGTGTTACCCTTTAAACCCCAAGAGATGC
CTCTCGGAGTTATTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTACGGG
CAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCGG
>FLP3FBN01DAUO4 length=247 xy=1236_2646 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGTCCACCCTCTCAGGCCGGCTGCTGATCGTCGCCTTGGTGGGCCGTTACCTCACCAAC
TAGCTAATCAGACGCGAGGCCATCTTCCAGCGATAAATCTTTGACGTCGGAGTCATGCGG
CTCCAACGCATCATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGGAAGGCA
GGTTCCT
>FLP3FBN01DH3EM length=274 xy=1319_0556 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGCCCATCCGTCACCGGATTGCTCCTTTGACCGCTCCGGGATGC
CCCGGAATGGTGTTACGCGGAATTAGTCGGAATTTCTTCCGGTTATTCCCCTGTGACGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGG
>FLP3FBN01A6YVR length=270 xy=0372_1157 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGATCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTGGGCCGTTACCTCACCAAC
TAGCTAATGCGACGCGAGCCCATCTCATACCGGATTTCTCCTTTTACCCCTCTACCATGC
GGTAGTGTGGTCTTATACCGTATTAACCGAAGTTTCCCTCGGGTATCCCGTTGTATGAGG
CAGGTTGCTCACGCGTTACTCACCCGTTCG
>FLP3FBN01ATZTQ length=254 xy=0224_2060 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGTCCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTAGGCCGTTACCCTACCAAC
TAGCTAATGCGCCATAAGTCCATCCTTCAGCTATCCCGTGGGATATTTAATATGTCGATC
ATGCGATCTTCATACCTATGCGGTCTTAGCTATCATTTCTAATAGTTATCCCCCTCTTAA
GGCCAGGTTACTTA
>FLP3FBN01EV4Z5 length=282 xy=1889_0707 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAACGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTGGTCTTGGTGAGCCGTTACCCCACCAAC
TAACTAATACAACGCATGCCCATCCATTACCACCGGAGTTTTCAACCCAAGAAGATGCCT
CCCTGGATGTTATGGGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTAATGGGTAG
GTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCGACAAT
>FLP3FBN01ELE68 length=264 xy=1767_0418 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGCCGCCCTCTCAGGCCGGCTGCTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
GAGCTAATCAGACGCGGGCCCATCCCGTACCACCGGAGTTTTTCACACCAGGGCATGCGC
CCCTGTGCGCTTATGCGGTATTACCAGCCGTTTCCAGCTGCTATCCCCCTGTATGGGCCA
GGTTATCCACGCGTTACTCACCCG
>FLP3FBN01DNL4E length=268 xy=1382_0044 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAAC
TGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGG
AATCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCA
GGTTATCCACGCGTTACTCACCCGTCCG
>FLP3FBN01COXAR length=268 xy=0986_3585 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAG
GACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01B6YON length=267 xy=0782_1157 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGCCAACCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTGCCAAC
AAGCTAATCAGACGCGGGTCCATCTCGCACCACCGGAGTTTTCAGGGCGGGGGCATGCGC
CCCCTCCCGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCGAGGCAG
GTTGCCCACGCGTTACTCACCCGTCCG
>FLP3FBN01C8AH6 length=242 xy=1207_1948 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
CAGCTAATCAGACGCGAGCTCATCTCAGAGCGATAAATCTTTGACAGCCAGGGAGATGCC
TCCCAGCTGTATCATGCGGTATTAGCGGCTGTTTCCAGCCGTTATTCCCCACTCCAAGGC
AG
>FLP3FBN01B2ALM length=257 xy=0729_0408 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAACC
AGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGCTTCATGCGAA
GCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCAG
GTTCTCCACGCGTTACT
>FLP3FBN01EXM1D length=244 xy=1906_1103 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAGGCCGTTACCTTACCAAC
TAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGA
CATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCT
TGGG
>FLP3FBN01C6ISY length=268 xy=1187_1312 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCGCTCTCTCAAGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAAC
CAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACCAGACCATGCGA
TCCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01C44O7 length=277 xy=1171_1913 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATCTGG
CCGGTCGGTCTCTCAACCCGGCTACCCATAGAAGCCTTGGTGGGCCGTTACCCCGCCAAC
TAGCTAACAGGCCGCGGGCCCATCCCTCTCCGCCAGAGCTTTCCCAACGCTTCCATGCGA
AAGCGTCGGAGTATCCGGTATTATCCCCGGTTTCCCGAGGCTATCCCGATGAGAGGGGCA
GGTTGCCCACGTGTTACTCAGCCGTTCGCCACTATCC
>FLP3FBN01C5Q76 length=255 xy=1178_2436 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTCTCCACGCGTT
>FLP3FBN01DOZ1S length=268 xy=1397_3310 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTTACACCACTAATGTTTTTCACTCTGTCCCATGCGG
GACTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTGTAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01AZ93K length=270 xy=0296_0398 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTAAGACGAAGTCATGCGA
CTCCCTCTGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGTAAGGCAG
GTTACCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01CK0M3 length=255 xy=0942_1517 region=1 run=R_2008_12_09_13_51_01_
ACCGCAGAGTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACTCTCTCAAGCCGGCTACTGATCGTCGCTTTGGTAGGCCATTACCCTGCCAAC
TGGCTAATCAGACGCGAGCCCATCCTGTACCTATAAATATTTGATGGAAATATCATGCGG
TATCTCCATGTTATGCGGTTTTAGCAGCCGTTTCCAACTGTTATCCCCCTGTACAGGGCA
GGTTACCCACGCGTT
>FLP3FBN01ENPIT length=268 xy=1793_0611 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTAGGCTTGGTAGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGGTCCATCCCATACCACCGGAGTTTTTCACACAGGAGCATGCGC
TCCCGTGCGCTTATGCGGTATTAGCACCTGTTTCCAGGTGGTATCCCCCGGTATGGGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01B3DCW length=260 xy=0741_1486 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCAACCTCTCAGTCCGGCTACCGATCGTCGGCTTGGTGGGCCGTTACCCCACCAAC
TACCTAATCGGACGCGAGCCCATCTTTCAGCGGATTGCTCCTTTGATTATCCTACCATGC
GATAAGATAATGTCATGCGGTATTAGCGTTCGTTTCCAAACGTTATCCCCCTCTGAAAGG
CAGGTTGCTCACGCGTTACT
>FLP3FBN01BMNXJ length=244 xy=0551_0277 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAAC
TAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGA
CATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCT
TGGG
>FLP3FBN01DAQ2Q length=268 xy=1235_2048 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCCTTGGTAGGCCATTACCCCACCAAC
TAGCTAATCAGACGCGGAACCATCGTATACCACCAGAGTTTTTCACACCGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTTTCCACGCGTTACTCACCCGTCCG
>FLP3FBN01B4Q1P length=270 xy=0757_0347 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTCTACCCCGCCAAC
AAGCTAATCAGACGCGGGTCCATCGTATACCACCGGAGTTTTTCACACCGGACCATGCGA
TCCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCA
GGTTACCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01A7RZC length=260 xy=0381_2006 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGCTGCCCCGCCAAC
CAGCTAATCAGACGCGGGCCCATCCCACACCGCCCGAAGGCTTTTCACACCGCTTCATGC
GAAGCTGTGCGCTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTGTGGGG
CAGGTTGCCCACGCGTTACT
>FLP3FBN01CFA5S length=274 xy=0877_1454 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGCCAACCTCTCAGTCCGGCCACTGATCGTCGCCTTGGTGGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAGGGGAAAGCCATGCGG
CTCCCCCGTTATGCGGTATTAGCACCTATTTCTAAGTGTTGTCCCCCTGTACAGGGCAGG
TTGCCCACGCGTTACTCACCCGTCCGCCACTGGG
>FLP3FBN01DPJ0D length=269 xy=1404_0507 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGCTGCCCCGCCAAC
AAGCTAATCAGACGCGGGCCCCTCCCATACCGCCGGAGCTTTCCCCACAAAGGCATGCGC
CTCCCTGGTTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTATGGGGCAG
GTTGCCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01DOYEX length=264 xy=1397_1191 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACTCTCTCAAGCCGGCTACTGATCGTTGTCTTGGTAGGCCGTTACCCTGCCAAC
TAACTAATCAGACGCGGGTCCATCGTATACCTCCGGAAATTTTCACACTCTGTCATGCGA
CAGTGTGCGCTTATGCGGTATTAGCAGTTGTTTCCAACTGTTATCCCCCTGTATACGGCA
GGTTACCCACGCGTTACTCACCCG
>FLP3FBN01A13CQ length=268 xy=0316_3048 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCCTTGGTAGGCCATTACCCCACCAAC
TAGCTAATCAGACGCGGAACCATCGTATACCACCAGAGTTTTTCACACCGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTTTCCACGCGTTACTCACCCGTCCG
>FLP3FBN01CMHTY length=253 xy=0959_0820 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAGTGTGT
CCGATCACCCTCTCAGGCCGGATACCCGTCATAGCCTTGGTGAGCCATTACCTCACCAAC
AAGCTGATAGGACATAGGCTGATCCTTTAGCGAAAAACTTTCCCCCGTAGGGAGTATCCA
GTATTAATCACCGTTTCCAGTGGCTATCCCAGACTAAAGGGCACATAACCTATGCATTAC
TCACCCGTGCGCC
>FLP3FBN01DD801 length=258 xy=1275_1443 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGTTCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTAGGCCTTTACCCCACCAAC
TAGCTAATGCGCCGCAGGCTCATCCATCAGTGATGCCAGGAGCATCTTTAAACTTTCGTC
CTATCCGGTATTAGCGATCGTTTCCAATCGTTGTCCCCGTCTGATGGGCAGATTACCTAC
GTGTTACTCACCCGTTCG
>FLP3FBN01B1MJU length=264 xy=0721_2008 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGTCGGCTACCGATCGTCGCCTTGGTGGGCTGTTACCCCTCCAAC
TAGCTAATCGGACGCGGATCCATCTCTCACCGATTTCTCTTTCCCTCTCTCCTCATGCGA
TAAAAGAGGCTCATGCGGTATTAGCAGCCGTTTCCAGCTGTTATCCCCCTGTGAAAGGCA
GGTTATCCACGCGTTACTCACCCG
>FLP3FBN01D8L02 length=258 xy=1621_0756 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCCTTGGTGGGCCTTTACCCCGCCAAC
AAGCTAATCAGACGCGGATCCATCGTATACCACCAAAAGCTTTAGCTTTTTGTTTTCCAC
ACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCC
CCCAGTATACGGCAGGTT
>FLP3FBN01DWXHY length=270 xy=1488_0516 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCTTTGGTAGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGGTCCATCTCATACCGATAAATCTTTTCCGTCCGTATCATGCGA
TACTAGCGGGTTATGCGGTATTAGCGGTCGTTTCCAACTGTTATCCCCCTGTATGAGGTA
GGTTACCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01DO0PN length=269 xy=1398_0073 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTGCCAAC
AAGCTAATCAGACGCGGGTCCATCTCGCACCACCGGAGTTTTCAGGGCGGGGGCATGCGC
CCCCTCCCGTTATGCGGTGTTAGCACCTATTTCTGGGTGTTATCCCCCAGTGCGAGGCAG
GTTGCCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01A2RID length=268 xy=0324_1587 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCCCAGTCCCAGTGTGG
CCGATCACCCTCTCAGGTCGGCTACCCATCGTTGCCTTGGTGGGCCGTTACCTCACCAAC
TAGCTAATGGGTCGCGAGCCCATCCTATACCGATAAATCTTTTACCCCTGTATCATGCGA
TACTGTGGTCTTATACCGTATTAGCACAAATTTCTCTGTGTTATCCCGTTGTATAGGGCA
GGTTGCTCACGTGTTACTCACCCGTTCG
>FLP3FBN01AJ4LN length=258 xy=0112_0441 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCACCAAC
TAGCTAATCAGACGCGGGTCCATCTCATACCACCGGAGTTTTTCACACAAGACCATGCAG
TCCCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTGAGGCA
GGTTATCCACGCGTTACT
>FLP3FBN01DJSHD length=253 xy=1338_1887 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGCTGCCCCGCCAAC
AAGCTAATCAGACGCGGGCCCATCGCATACCACCGGAGTTTTTCACACCAAGCCATGCGG
CTCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGGTATGCGGCA
GGTTGCCCACGCG
>FLP3FBN01ES2CK length=267 xy=1854_0658 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTGCCAAC
AAGCTAATCAGACGCGGGTCCATCTCGCACCACCGGAGTTTTCAGGGCGGGGGCATGCGC
CCCCTCCCGTTATGCGGTGTTAGCACCTATTTCTGGGTGTTATCCCCCAGTGCGAGGCAG
GTTGCCCACGCGTTACTCACCCGTCCG
>FLP3FBN01D5PDA length=268 xy=1588_0284 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGGTCCATCTCACACCGATAAATCTTTTCCGGCCGGATCATGCGA
TCCTTCCGGCTTATGCGGTATTAGCGGTCGTTTCCAACTGTTATCCCCCTGTGTGAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01CPOTU length=269 xy=0995_2400 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCCGTCGCTTTGGTGGGCCGTTACCCCGCCAA
CTGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCG
GAACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGC
AGGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01DDGAY length=273 xy=1266_1080 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCGGCACCGGGATGCC
CCGATGCCGCGTCACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGC
AGGTTCCATACGTGTTACTCACCCGTGCGCCGG
>FLP3FBN01DDAHB length=255 xy=1264_1725 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTCTCCACGCGTT
>FLP3FBN01A8N1I length=270 xy=0391_2596 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTCTACCCCGCCAAC
AAGCTAATCAGACGCGGGTCCATCGTATACCACCGGAGTTTTTCACACCGGACCATGCGA
TCCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCA
GGTTACCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01ELV1R length=268 xy=1772_1773 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCT
CCCAACGTTCTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTGGCGGGCA
GGTTGCATACGTGTTACTCACCCGTGCG
>FLP3FBN01DZNIH length=264 xy=1519_0567 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCCTCTACCCCGCCAAC
TAGCTAATCAGACGCGGGTCCATCGTATACCGTTTACGCTTTTCACACTGCTCCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCA
GGTTACCCACGCGTTACTCACCCG
>FLP3FBN01DWOHH length=253 xy=1485_1123 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAGTGTGT
CCGATCACCCTCTCAGGCCGGATACCCGTCATAGCCTTGGTGAGCCATTACCTCACCAAC
AAGCTGATAGGACATAGGCTGATCCTTTAGCGAAAAACTTTCCCCCGTAGGGAGTATCCA
GTATTAATCACCGTTTCCAGTGGCTATCCCAGACTAAAGGGCACATAACCTATGCATTAC
TCACCCGTGCGCC
>FLP3FBN01EKZQ1 length=243 xy=1762_0875 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
TAGCTAATCAGACGCGAGTCCATCTCAGAGCGATAAATCTTTGGCAGTCAGAGCCATGCG
ACCCAACTGCATTATGCGGTATTAGCAGCTGTTTCCAGCTGTTATTCCCCACTCCAAGGC
AGG
>FLP3FBN01EVVYX length=268 xy=1886_1287 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTGCCAAC
AAGCTAATCAGACGCGGGTCCATCCCATACCACCGGAGTTTTTCACACAGGAGCATGCGC
TCCCGTGCGCTTATGCGGTATTAGCACCTGTTTCCAGGTGGTATCCCCCGGTATGGGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01BX9OT length=269 xy=0683_1019 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTTACACCACTAATGTTTTTCACTCTGTCCCATAGCG
GGACTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTGTAGGGC
AGGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01C11Y6 length=272 xy=1136_1776 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGCCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCCTCTCCCCCGCCAAC
TAGCTAATCAGACGCGGGCCCATCCCATACCACCGGAGTTTTTCACACCAGGCCATGCGG
CCCTTGTGCGCTTATGCGGTTTTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTATGGGGC
AGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01BORQ9 length=278 xy=0575_0243 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTAC
TGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCAAGGATGCCC
CCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAG
GTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGG
>FLP3FBN01CORTK length=264 xy=0985_0582 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCTTTACCCCACCAAC
TAGCTAATCAGACGCGGGTCCATCATATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTTTCCACGCGTTACTCACCCG
>FLP3FBN01DDXJ9 length=269 xy=1271_2967 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGCTGCCCCGCCAAC
AAGCTAATCAGACGCGGGCCCCTCCCATACCGCCGGAACTTTCCCCAGAAAGGCATGCGC
CTCCCTGGTTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTATGGGGCAG
GTTGCCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01D3MIZ length=270 xy=1564_1593 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAAC
CAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACTCTGTCCCATGCGG
GACTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTGTAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01COS84 length=258 xy=0985_2438 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGCCGGCTACTGATCGTTGCTTTGGTAGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGGTCCATCGTATACCTCCGGAAATTTTCACACTCTGTCATGCGA
CAGTGTGCGCTTATGCGGTATTAGCAGTTGTTTCCAACTGTTATCCCCCTGTATACGGCA
GGTTACCCACGCGTTACT
>FLP3FBN01CJOJP length=242 xy=0927_0627 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
CAGCTAATCAGACGCGAGCTCATCTCAGAGCGATAAATCTTTGACAGCCAGAGCCATGCG
GCTCAGCTGTATCATGCGGTATTAGCGGCTGTTTCCAGCCGTTATTCCCCACTCCAAGGC
AG
>FLP3FBN01AH4QL length=270 xy=0089_1515 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTCTACCCCGCCAAC
AAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTCACACCGGGCCATGCGG
CCCTGTGCGCTTATGCGGTATTAGCACCTGTTTCCAGGTGTTATCCCCCTGTGTAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01ARXOA length=264 xy=0201_0168 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTTGCCTTGGTGGGCCGTTACCCCGCCAAC
CAGCTAATCAGACGCGGGCCCATCCCGCACCACCGGAGTTTTTCACACCGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCAGCCGTTTCCAGCTGTTGTCCCCCAGTGCGGGGCA
GGTTGCCCACGCGTTACTCACCCG
>FLP3FBN01BOZTS length=247 xy=0577_2510 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTACCAAC
TAGCTAATCAGACGCGAGGCCATCTCTCAGCGATAAATCTTTGATATATCTGCCATGCGA
CAAACATATATTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGAAAGGCA
GGTTCCT
>FLP3FBN01CP1E3 length=262 xy=0999_2333 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAAC
TAGCTAATGTCACGCATGCCCATCCGTTGCCGGAATCGCTTCCTTTGGCCGCAGGGCCAT
GCAGCCCCGCGGCATTACGCGGTATTAGACGGGATTTCTCCCGATTATCCCCCTGCAACG
GGCAGGTCGCATACGCGTTACT
>FLP3FBN01AYTXJ length=268 xy=0279_2421 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAG
GACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01ESMLF length=278 xy=1849_0721 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTAC
TGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCAAGGATGCCC
CCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAG
GTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGG
>FLP3FBN01CAOAX length=253 xy=0824_2295 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAAC
CAGCTAATCAGACGCGGGCCCATCCCGCACCACCGGAGTTTTCCACACTGCTTCATGCAA
AGCTGTGCGCATATGCGGTATTAGCAGCCGTTTCCGGCTGTTGTCCCCCAGTGCGGGGCA
GGTTGCCCACGCG
>FLP3FBN01D5KU1 length=264 xy=1586_2635 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCCCGCCAAC
TACCTAATCAGACGCGGGTCCATCTCACACCACCGGAGTTTTTCACACCAGACCATGCGG
TCCTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTGAGGCA
GGTTATCCACGCGTTACTCACCCG
>FLP3FBN01DR6KE length=272 xy=1434_0172 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGTCCACCCTCTCAGGCCGGCTGCTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAAC
TGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAAAGGATTACCATGCGG
TATTCCCTATTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGGTACAGGGCAG
GTTGCCCACGCGTTACTCACCCGTCCGCCACT
>FLP3FBN01BWBS4 length=276 xy=0661_0566 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCGGCACCGGGATGCC
CCGACGCCGCGTCACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTGGCGGGC
AGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>FLP3FBN01B9R0O length=271 xy=0814_1414 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCCCGCCAAC
TACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCACCGGACCATGCGG
TCCTGCAGTCTTATGCGGTATTAGCAGTCATTTCTGACTGTTATCCCCCTGTGTAAGGCA
GGTTGCCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01EGUWG length=268 xy=1715_0478 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAG
GACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01CMT8N length=268 xy=0963_0517 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAG
GACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01DAJ0V length=268 xy=1233_1101 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCCTTGGTGGGCCGTTACCCCGCCAAC
AAGCTAATCAGACGCGGGTCCATCTTACACCACTAATGTTTTTCACTCTGTCCCATGCGG
GACTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01EDYEP length=268 xy=1682_0223 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTCTACCCCGCCAAC
AAGCTAATCAGACGCGGGTCCATCGTATACCACCGGAGTTTTTCACACCGGACCATGCGA
TCCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGCCA
GGTTGCCCACGCGTTACTCACCCGTCCG
>FLP3FBN01DAF3E length=268 xy=1232_0104 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAG
GACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01BVUDC length=267 xy=0655_2542 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGACCGTGTTTCAGTTCCAGTGTGG
CCGGTCATCCTCTTAGACCGGCTACCCATCGTTGCCTTGGTGGGCCGTTACCTCACCAAC
TAGCTAATGGGACGCGGACTCATTCCCTCGCGGAAGCTTTCATGAAGAGGCCTCCTTTCC
TCCACCATACAATTGGTGGAGCTTATCCGGTATTAGCTGCCGTTTCCAGTAGTTATCCCG
GTGTGTGAGGCAGGTTGGTCACGTGTT
>FLP3FBN01D7PW9 length=265 xy=1611_0107 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAAC
CAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGA
AGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGC
GCGGGCAGGTTCCCTACGTGTTACT
>FLP3FBN01C62R4 length=268 xy=1193_2626 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCT
CCCAACGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCA
GGTTCCATACGTGTTACTCGCCCGTGCG
>FLP3FBN01CV26L length=264 xy=1068_1931 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCTTTGGTAGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGGCCCATCCTATGGCGATAACTCTTTCTTATTATGACCATGCAA
TCTTAATAAGGTATGCGGTATTAGCAGCGGTTTCCCACTGTTGTCCCCCTCCATAGGGCA
GGTTACCCACGCGTTACTCACCCG
>FLP3FBN01DGJIL length=264 xy=1301_1851 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGG
ACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCA
GGTTGGATACGTGTTACTCACCCG
>FLP3FBN01ENBFP length=258 xy=1788_2835 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCCTTGGTGGGCTTTTACCTCACCAAC
TAGCTAATCAGACGCGGAACCATCGTATACCACCGGAGTTTTTCGCACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTTTCCACGCGTTACT
>FLP3FBN01D5Y47 length=251 xy=1591_0665 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCTCAGTTTTTCACACTGTCCCATGCGA
GACTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTGAGGCA
GGTTATCCACG
>FLP3FBN01D6TCI length=274 xy=1600_2944 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAAC
CAGCTAATCAGACGCGGGCCCATCCATTACCACCGGAGTTTTCAACCCAAGAAGATGCCT
CCCTGGATGTTATGGGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTAATGGGTAG
ATTGCATACGCGTTACTCACCCGTGCGCCGGTCG
>FLP3FBN01DBSZG length=263 xy=1247_2026 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCTCGCCAA
CCAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCTTACAGGATGTTTCACAGAAAG
AAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTG
CGCGGGCAGGTTCCCTACGTGTT
>FLP3FBN01A6KUH length=264 xy=0367_3447 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCCTTGGTGGGCCGTTACCCCGCCAAC
AAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTCACACCGGATCATGCAA
TCCTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATGAGGCA
GGTTACCCACGCGTTACTCACCCG
>FLP3FBN01CFKFW length=266 xy=0880_1194 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGCACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGTCTTGGTGGTCCGTTACACCACCAAC
TACCTAATGCGCCGCATGCCCATCCTTTACCGGATCGCTCCTTTGACATACCGATCATGC
GGTCGGTATGTGTTATGAGGTATTAGACGGAATTTCTCCCGATTATCCCTCTGTAAAGGG
CAGGTCGCATACGTGTTACTCACCCG
>FLP3FBN01CWV71 length=281 xy=1077_2703 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTAC
TGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCAAGGATGCCC
CCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAG
GTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGGCAG
>FLP3FBN01BUKY6 length=243 xy=0641_1056 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCTCACCAAC
TAGCTAATCAGACGCGAGCCCATCTCAGAGCGATAGATCTTTGATGTCCAGTACCATGTG
ATACAGACATATCATGCGGTATTAGCAGCTGTTTCCAGCTGTTATTCCCCACTCCAAGGT
AGG
>FLP3FBN01APUP8 length=258 xy=0177_1342 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
CAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTCACACTGAACCATGCGG
TTCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACT
>FLP3FBN01DVAA4 length=268 xy=1469_1630 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAAC
CAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGCTTCATGCAA
AGCTGTGCGCTTATGCGGTTTTAGCACCTATCTCTAAGTGTTATCCCCCGGTATACGGCA
GGTTTTCCACGCGTTACTCACCCGTCCG
>FLP3FBN01AYSZA length=260 xy=0279_1188 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGTCCACCCTCTCAGGTCGGCTACTGATCGTTGCCTTGGTGGGCTGTTATCTCACCAAC
TAGCTAATCAGATGCGGGCCCATCTTTTACCGAATTTCTCCTTTCCTTCTCAGAAGATGC
CTCCTAAGAATATTATGCGGTATTAGTCACCGTTTCCAGTGATTATTCCCCAGTAAAAGG
CAGGTTGCCCACACGTTACT
>FLP3FBN01E11F6 length=271 xy=1956_1604 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAAC
AAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGA
AGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGC
GCGGGCAGGTTCCCTACGTGTTACTCACCCG
>FLP3FBN01D2E5F length=268 xy=1550_2721 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCCCTCCAAC
CAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGTACCATGCGG
TACTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01AY6JF length=275 xy=0283_2377 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
CAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCTTACAGGATGTTTCACAGAAAGA
AGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGC
GCGGGCAGGTTCCCTACGTGTTACTCACCCGTTCG
>FLP3FBN01AOSD0 length=282 xy=0165_0806 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTGGTCTTGGTGAGCCGTTACCCCACCAAC
TAACTAATACAACGCATGCCCATCCATTACCACCGGAGTTTTCAACCCAAGAAGATGCCT
CCCTGGATGTTATGGGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTAATGGGTAG
GTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCGACAAT
>FLP3FBN01DORQN length=266 xy=1395_0733 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTACGGAGTTTGGGCCGTGTCTCAGTCCCAATGTG
GCCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCCCGCCAA
CCAGCTAATCAGACGCGGGTCCATCTCATACCACCTCGGTTTTTCACACTGGACCATGCA
GTCCCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATGAGGC
AGGTTGCCCACGCGTTACTCACCCGG
>FLP3FBN01DHOF4 length=258 xy=1314_1650 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTTGCCTTGGTGAGCCGTTACCTCACCAAC
TAGCTAATCAGACGCGGGCCCATCTCATACCACCGGAGTTTTTACCACTGTACCATGCAG
TACCGTGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATGAGGCA
GGTTGCCCACGCGTTACT
>FLP3FBN01DP53S length=276 xy=1411_0470 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCGGCACCGGGATGCC
CCGGTGCCGCGTCACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTGGCGGGC
AGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>FLP3FBN01DPT2H length=271 xy=1407_1255 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAAC
TACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGG
TTCTGGGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCA
GGTTGCCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01A9OKC length=271 xy=0403_0778 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCCCGCCAAC
TACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCACCAGACCATGCGG
TCCTGTGGTCTTATGCGGTATTAGCAGTCATTTCTGACTGTTATCCCCCTGTGTAAGGCA
GGTTGCCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01C45DS length=264 xy=1171_2798 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTCTACCCCGCCAAC
AAGCTAATCAGACGCGGGTCCATCGTATACCACCGGAGTTTTTCACACCAGACCATGCGA
TCCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAACTGTTATCCCCCTGTGTAAGGCA
GGTTGCCCACGCGTTACTCACCCG
>FLP3FBN01BTP39 length=278 xy=0631_2023 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
TAGCTAATCAGACGCGGGCCCATCCCATACCACCGGAGTTTTTCACACAATCCCATGCGG
GATTGGGCGCTTATGCGGTTTTAGCACCTGTTTCCAGGTGTTATCCCCCGGTATGGGGCA
GGTTGCCCACGCGTTACTCACCCGTCCGCCACTAGGTT
>FLP3FBN01D8UA3 length=263 xy=1623_3293 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCTCGCCAA
CCAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCTTACAGGATGTTTCACAGAAAG
AAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTG
CGCGGGCAGGTTCCCTACGTGTT
>FLP3FBN01BTBC0 length=245 xy=0626_3378 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTAGGCCATTACCCTACCAAC
TAGCTAATCAGACGCGAGTCCATCTCAGAGCGATAAATCTTTGGCAATCAGTACCATGCG
ATACAATTGCATTATGCGGTATTAGCACAGCTTTCGTTGTGTTATTCCCCACTCCAAGGC
AGGTT
>FLP3FBN01BZF87 length=258 xy=0696_2937 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCA
GGTTCCCCACGCGTTACT
>FLP3FBN01ANQ3B length=258 xy=0153_1621 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGCCGGCTACTGATCGTTGCCTTGGTAGGCCGTTACCCTTCCAAC
TAGCTAATCAGACGCGGGTCCATCCTTTTGCGATAGCATTCTCAGAGGCCATCTTTCCTT
AGAAAATCATGCGATTCTCTATTATTATGCGGTATTAGCTCCGATTTCTCGAAGTTGTTC
CCCACAAAAGGGCAGGTT
>FLP3FBN01ATVZT length=263 xy=0223_1191 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCTCGCCAA
CCAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCTTACAGGATGTTTCACAGAAAG
AAGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTG
CGCGGGCAGGTTCCCTACGTGTT
>FLP3FBN01BY1GK length=255 xy=0692_0146 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTCTCCACGCGTT
>FLP3FBN01B8WRO length=254 xy=0804_1874 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGTTCACCCTCTCAGGTCGGCTACGCATCGTCGCCTTGGTAGGCCTTTACCCCACCAAC
TAGCTAATGCGCCGCAGGCTCATCCATCAGTGATGCCAGGAGCATCTTTAAACTTTCGTC
CTATCCGGTATTAGCGATCGTTTCCAATCGTTGTCCCCGTCTGATGGGCAGATCACCTAC
GTGTTACTCACCCG
>FLP3FBN01A68PA length=268 xy=0375_1596 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCGG
GACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01B6F7P length=268 xy=0776_1795 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGGTCCATCCCATACCACCGGAGTTTTTCACACAGGAGCATGCGC
TCCCGTGCGCTTATGCGGTATTAGCACCTGTTTCCAGGTGGTATCCCCCGGTATGGGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01CURUA length=263 xy=1053_2016 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAAC
AAGCTAATCAGACGCGAGCCCATCCTGCACCGTATTACTTTCCACACCGCCCCATGCAGG
GCTGTGCGCTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTATGGGGCAG
GTTGCCCACGCGTTACTCACCCG
>FLP3FBN01DEVFX length=275 xy=1282_1819 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
AAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTCAAGTAAAAGACATGCGT
CTCCTACTGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAG
GTTACCCACGCGTTACTCACCCGTCCGCCGCTAAG
>FLP3FBN01DJVP5 length=269 xy=1339_1995 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAAC
TACCTAATCGGACGCGAGCCCATCTCCGAGCGATAAAATCTTTGATACCAAAGGGATGTC
CCTCCAGTATGTTATGCGGTATTAGCGACCGTTTCCAGCCGTTATTCCCCTCTCGAAGGT
AGGTTGCTCACGTGTTACTCACCCGTCCG
>FLP3FBN01AVA3U length=268 xy=0239_1896 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCTCACCAAC
TAGCTAATCAGACGCGGGTCCATCTCATACCGATAAATCTTTTCCGTCTGTATCATGCGA
TACTAGCGGGTTATGCGGTATTAGCGGTCGTTTCCAACTGTTATCCCCCTGTATGAGGTA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01A1JXR length=272 xy=0310_2461 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
AAGCTAATGTCACGCATGCCCATCCGTCTCCGGAGTCGCCTCCTTTGGCCGAAAGAAGAT
GCCTTCCCTCGGCCACATGCGGTATTAGGCCGGATTTCTCCGGGTTATCCCGCTGAAACG
GGCAGGTTGCATACGCGTTACTCACCCGTGCG
>FLP3FBN01CU9QX length=273 xy=1059_0647 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTCACACCGATAAATCTTTTCCGTCAGTACCATGCGG
TACCAGCGGGTTATGCGGTATTAGCGGTCGTTTCCAACTGTTATCCCCCTGTGTGAGGCA
GGTTACCCACGCGTTACTCACCCGTCCGCCACT
>FLP3FBN01EFEUM length=281 xy=1698_2652 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
TAGCTAATGCGCCGCATGGCCATCCGTAGCCGGTGTTACCCTTTAAACCCCAAGAGATGC
CTCTCGGAGTTATTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTACGGG
CAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01DR3RW length=253 xy=1433_0650 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGGTCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
AAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGCTTCATGCAA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCG
>FLP3FBN01BTNQ7 length=274 xy=0630_3057 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAAC
CAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCT
CTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAG
GTTGCATACGCGTTACGCACCCGTGCGCCGGTCG
>FLP3FBN01DG439 length=271 xy=1308_1175 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCCACCAAC
CAGCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGG
TTCTGGGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCA
GGTTGCCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01BJV71 length=258 xy=0519_2127 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAAC
TGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACCGCTCCATGCGG
AGCTGTGCGCTCATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGGTATACGGCA
GGTTATCCACGCGTTACT
>FLP3FBN01BBGPO length=268 xy=0423_1994 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAAC
TGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGG
AACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCA
GGTTATCCACGCGTTACTCACCCGTCCG
>FLP3FBN01CPEDQ length=273 xy=0992_1148 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
TAGCTAACAGGCCGCGGGCCCATCCCTCTCCGCCAGAGCTTTCCCAACGCTTCCATGCGA
AAGCGTCGGAGTATCCGGTATTATCCCCGGTTTCCCGAGGCTATCCCGATGAGAGGGGCA
GGTTGCCCACGTGTTACTCAGCCGTTCGCCACT
>FLP3FBN01BJETT length=264 xy=0514_0063 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGG
ACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCA
GGTTGGATACGTGTTACTCACCCG
>FLP3FBN01D81I4 length=264 xy=1626_0366 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAG
GACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCG
>FLP3FBN01CXUH7 length=261 xy=1088_2077 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
TAACTAATGCGCCGCATGCCCATCCATGACCGGATCGCTCCTTTGACTCCCGAGAGATGT
CTCCCGGGGGTGTTATGCGGTATTAGACGGAATTTCTCCCGCTTATCCCCCTGTCATGGG
CAGGTTGCATACGTGTTACTC
>FLP3FBN01D66OZ length=286 xy=1604_3857 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGCCCATCCGTGGCCGGGATTGCTCCCTTTGGCGGCCCGGGGAT
GCCCCAAGGCCGCGTTACGCGGTATTAGACGGGGTTTCCCCCGCTTATCCCCCTGCCACG
GGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGGCAG
>FLP3FBN01EP18M length=244 xy=1819_3908 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAAC
TAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGA
CATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCT
TGGG
>FLP3FBN01AYADU length=264 xy=0273_1664 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGGTCAACCTCTCAGTCCGGCTACTGATCGTCGCCTAGGTGGGCCGTTACCCCGCCTAC
CAGCTAATCAGACGCGAGGCCATCTTCCAGCGATAAATCTTTGGTATCAGGGTCATGCGG
CCCCAATACTTTATGCGGTATTAGCAGTCATTTCTAACTGTTGTCCCCCTGTACAGGGCA
GGTTACCCACGCGTTACTCACCCG
>FLP3FBN01DZ6HR length=255 xy=1525_0589 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTCTCCACGCGTT
>FLP3FBN01DQS40 length=249 xy=1418_1650 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTACCGATCGTCGCCTTGGTGGGCCGCTGCCCCGCCAAC
AAGCTAATCGGACGCGGACCCATCGCATGCCGGGATCGCTCCCTTTCCGCACTGCGCCAT
GCGGCGCCGTGCGCATATGCGGTATTAGCGGTTGTTTCCAGCCGGTATCCCCCTGCATGC
GGCAGGTTG
>FLP3FBN01ETR1Z length=258 xy=1862_1205 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGTACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTCTCCACGCGTTACT
>FLP3FBN01BYEMY length=270 xy=0684_3336 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAAC
TACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGG
TTCCGCGGTCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01DWFB9 length=274 xy=1482_1559 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGCCCATCCGTCACCGGATTGCTCCTTTGACCGCTCCGGGATGC
CCCGGAATGGTGTTACGCGGAATTAGTCGGAATTTCTTCCGGTTATTCCCCTGTGACGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGG
>FLP3FBN01B350B length=281 xy=0750_1753 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTAC
TGCCTAATGCGCCGCATGCCCATCCTACACCGGTAATCCTTTCCTCCCCCAAGGATGCCC
CCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAG
GTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGGCAG
>FLP3FBN01D8VJ6 length=271 xy=1624_0820 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAAC
CAGCTAATCAGACGCGGGTCCATCCTATACCACCTCAGTTTTTCACACCGAACCATGCGG
TTTTAGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATAGGGC
AGGTTACCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01AWH2Z length=264 xy=0253_0249 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTTGCCTTGGTGGGCTGTTATCTCACCAAC
TACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCGCTGTACCATGCGG
TACTGCGGTCTTATGCGGTATTAGCAGTCATTTCTAGCTGTTATCCCCCTGTGTAAGGCA
GGTTGCCCACGCGTTACTCACCCG
>FLP3FBN01A0B3E length=272 xy=0296_2984 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAAC
TAGCTAATGTCACGCATGCCCATCCGTTGCCGGAATCGCTTCCTTTGGCCCCAGGGCCAT
GCAGCCCCGCGGCATTACGCGGTATTAGACGGGGTTTCCCCCGCTTATCCCCCTGCCACG
GGCAGGTTGCATACGTGTTACTCACCCGTGCG
>FLP3FBN01B6CNT length=264 xy=0775_1287 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGG
ACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCA
GGTTGGATACGTGTTACTCACCCG
>FLP3FBN01AJVO0 length=268 xy=0109_1186 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCCACCAAC
CAGCTAATCAGACGCGGGCTCATCTTATACTACCGGAGTTTTTCACACAGAAACATGCGT
CCCCGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTATAAGGCA
GATTACCCACGTGTTACTCACCCGTCCG
>FLP3FBN01A5TI5 length=270 xy=0359_0815 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAAC
CAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACCGGACCATGCGG
TCCTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTGTCCCCCTGTACAGGGCA
GGTTACCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01CSGAL length=270 xy=1027_0235 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
AAGCTAATCAGACGCGGGTCCATCGTATACCACCGGAGTTTTTCACACCAGACCATGCGA
TCCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCA
GGTTACCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01A26LS length=268 xy=0329_0670 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAG
GACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01D1OTA length=274 xy=1542_1356 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGCCCATCCGTCACCGGATTGCTCCTTTGACCGCTCCGGGATGC
CCCGGAATGGTGTTACGCGGAATTAGTCGGAATTTCTTCCGGTTATTCCCCTGTGACGGG
CAGGTTGCATACGTGTTACTCACCCGTGCGCCGG
>FLP3FBN01EOJO5 length=239 xy=1802_2855 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGCCAACCTCTCAGTCCGGCTACTGATCGTTGCCTTGGTGGGCCGTTGCCCCACCAAC
TAGCTAATCAGACGCGAGTCCATCTCAGAGCGATAAATCTTTGGCAGTCAGAGCCATGCG
ACTCAACTGCATTATGCGGTATTAGCAGCTGTTTCCAGCTGTTATTCCCCACTCCAAGG
>FLP3FBN01EEA7Q length=259 xy=1686_0436 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTTGCCCTCTCAGGCCGGCTACTGATCGTCGGTTTGGTGGGCCGTTACCCCGCCAAC
TGCCTAATCAGACGCGGGCCCATCCCATACCGCGAAAGCTTTCCATGCAGGGGCATGCGC
CCCCGCATGTGCATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGGAAGGCA
GGTTCCTCACGCGTTACTC
>FLP3FBN01C7IA4 length=264 xy=1198_2270 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGTCGGCTACCGATCGTCGCCTTGGTGGGCTGTTACCCCTCCAAC
TAGCTAATCGGACGCGGATCCATCTCTCACCGATTTCTCTTTCCCTCTCTCCTCATGCGA
TAAAAGAGGCTCATGCGGTATTAGCAGCCGTTTCCAGCTGTTATCCCCCTGTGAAAGGCA
GGTTATCCACGCGTTACTCACCCG
>FLP3FBN01DIPTV length=276 xy=1326_0945 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCGGCACCGGGATGCC
CCGACGCCGCGTCACGCGGTATTAGACGGAATTTCTTCCGTTTATCCCCCTGTGGCGGGC
AGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>FLP3FBN01A95YV length=279 xy=0408_2853 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCCCGCCAAC
TACCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACCGGACCATGCGG
TACTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCCTGTACAGGGC
AGGTTACCCACGCGTTACTCACCCGTCCGCCACTAAGAT
>FLP3FBN01C17R1 length=272 xy=1138_1103 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAAC
TACCTAATCAGACGCGGGCCCATCTTACACCACCGGTAGTTTTTACCTCAGAACCATGCG
GTTCTGGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTACTCCCCCTGTGTAAGGC
AGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01ENDTE length=258 xy=1789_1824 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTCTCCACGCGTTACT
>FLP3FBN01CGNQ7 length=272 xy=0892_2993 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGACCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCT
CCCAACGTTCTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCA
GGTTCCATACGTGTTACTCACCCGTGCGCCGG
>FLP3FBN01CPBZJ length=262 xy=0991_2141 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAAC
TAGCTAATGTCACGCATGCCCATCCGTTGCCGGAATCGCTTCCTTTGGCCGCAGGGCCAT
GCAGCCCCGCGGCATTACGCGGTATTAGACGGGATTTCTCCCGATTATCCCCCTGCAACG
GGCAGGTCGCATACGCGTTACT
>FLP3FBN01C94TW length=272 xy=1228_1890 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCT
CCCAACGTTCTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCA
GGTTCCATACGTGTTACTCACCCGTGCGCCGG
>FLP3FBN01AJ8QN length=255 xy=0113_1709 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTCTCCACGCGTT
>FLP3FBN01ERY9Y length=253 xy=1841_3268 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTGCCAAC
TAGCTAATCAGACGCGAGGCCATCTTCCAGCGATAAATCTTTGACATTCAAACCATGCGA
TTCAAATGTGTCATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGGAAGGCA
GGTTCCTCACGCG
>FLP3FBN01D1576 length=271 xy=1547_3444 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAAC
AAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGA
AGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGC
GCGGGCAGGTTCCCTACGTGTTACTCACCCG
>FLP3FBN01D9ANJ length=270 xy=1628_3997 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGTCGGCTACCGATCGTCGCCTTGGTGGGCTGTTACCCCTCCAAC
TAGCTAATCGGACGCGGATCCATCTCTCACCGATTTCTCTTTCCCTCTCTCCTCATGCGA
CAAAAGAGGCTCATGCGGTATTAGCAACCGTTTCCAGCTGTTATCCCCCTCGTGAAAGGC
AGGTTATCCACGCGTTACTCACCCGTCCCG
>FLP3FBN01DE016 length=276 xy=1284_0908 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
AAGCTAATCAGACGCGGGTCCATCTTATACCACCGGAGTTTTTCACACTGGGCCATGCAG
CCCCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCGCCACTAAG
>FLP3FBN01BEAYH length=268 xy=0455_3431 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCCTTGGTAGGCCATTACCCCACCAAC
TAGCTAATCAGACGCGGAACCATCGTATACCACCAGAGTTTTTCACACCGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTTTCCACGCGTTACTCACCCGTCCG
>FLP3FBN01B4KRR length=249 xy=0755_0405 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTATGGACCGTGTCTCAGTTCCATTGTGG
CCGATCAGTCTCTCAACTCGGCTATGCATCATCGCCTTGGTAAGCCGTTACCTTACCAAC
TAGCTAATGCACCGCAGGTCCATCCCAGAGTGATAGCCAAAGCCATCTTTCAAACAAAAG
CCATGTGGCTTTTGTTGTTATGCGGTATTAGCATCTGTTTCCAAATGTTATCCCCCGCTC
CGGGGCAGG
>FLP3FBN01EDWJK length=274 xy=1681_1902 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGGCTTGGTGAGCCGTTACCTCACCAAC
AACCTAATGGAACGCATCCCCATCCTTTATCGGAATCCTTTAATAATGAAACCATGCGGA
ATCATTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTAAAGGGCAG
GTTGGATACGTGTTACTCACCCGTGCGCCGGTCG
>FLP3FBN01B2A3Z length=258 xy=0729_1069 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGATCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCTTACCAAC
CAGCTAATCAGACGCGGGCCCATCCTGTACCACCGTGGTTTTCCCTGCTGTTCCATGCGG
CACAGCAGGCTTATGCGGTATTAGCAGCCATTTCTGGCTGTTGTCCCCCGGTACAGGGCA
GGTTGCCCACGCGTTACT
>FLP3FBN01EEKR5 length=272 xy=1689_0547 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGGTG
GCCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAA
CCAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGTACCATGCG
GTACTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGC
AGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01CQ5AB length=274 xy=1012_0753 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTCGCCTTGGTAAGCCGTTACCTTACCAAC
CAGCTAATACAACGCATGCCCATCTGTAACCGCCGAAACTTTCAACCACAAGAGATGCCT
CTCATAGTGTTATGCGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTTACAGGTAG
GTTGCATACGCGTTACGCACCCGTGCGCCGGTCG
>FLP3FBN01DXG0S length=268 xy=1494_1242 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCTGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTGCCAAC
AAGCTAATCAGACGCGGGTCCATCTCGCACCACCGGAGTTTTCAGGGCGGGGGCATGCGC
CCCCCTCCCGTTATGCGGTGTTAGCACCTATTTCTGGGTGTTATCCCCCTGTACAGGCCA
GGTTGCCCACGCGTTACTCACCCGTCCG
>FLP3FBN01B1B5Q length=268 xy=0718_0828 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCGG
GACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCA
GGTTATCCACGCGTTACTCACCCGTCCG
>FLP3FBN01CQKDB length=274 xy=1005_2317 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGCCCATCCGTCACCGGATTGCTCCTTTGACCGCTCCGGGATGC
CCCGGAATGGTGTTACGCGGAATTAGTCGGAATTTCTTCCGGTTATTCCCCTGTGACGGG
CAGGTTGCATACGTGGTACTCACCCGTGCGCCGG
>FLP3FBN01CB8GB length=270 xy=0842_1337 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAAC
TACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGG
TTCCGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCA
GGTTGCCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01C03VW length=267 xy=1125_2650 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCATCCCAATGTGGC
CGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTCTACCCCGCCAACA
AGCTAATCAGACGCGGGTCCATCGTATACCACCGGAGTTTTTCACACCAGACCATGCGAT
CCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCAG
GTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01ETEMV length=268 xy=1858_0197 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGTACCATGCGG
TACTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01DYMXO length=268 xy=1507_2314 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAG
GACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTCCCCACGCGTTACTCACCCGTCCG
>FLP3FBN01D0XGC length=268 xy=1533_2762 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAAC
TACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGG
TTCCGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCA
GGTTGCCCACGCGTTACTCACCCGTCCG
>FLP3FBN01D069Y length=275 xy=1536_3204 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAAC
TACCTAATCAGACGCGGGCCCATCTTACACCACCCGGAGTTTTTACCTCAGAACCATGCG
GTTCCGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACCTGTTTATCCCCCTGTGTTAA
GGCAGGTTGCCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01C8P52 length=268 xy=1212_1768 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCATTACCCCGCCAAC
TAGCTAATCAGACGCGGAACCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGGTATACGGCA
GGTTTTCCACGCGTTACTCACCCGTCCG
>FLP3FBN01C1Q79 length=283 xy=1133_0135 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTAC
TGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCAAGGATGCCC
CCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAG
GTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGGCGGAG
>FLP3FBN01BJ2ZS length=270 xy=0521_2710 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAAC
TAGCTAATGCGCCGCATGACCATCCGCAGCCGGATCGCTCCTTTGAACCAACAGGGATGC
CCCCGTCGGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGG
CAGGTTTCATACGTGTTACTCACCCGTGCG
>FLP3FBN01AK0GJ length=268 xy=0122_0769 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
AAGCTAATCAGACGCGGGTCCATCTTACACCACTAATGTTTTTCACTCTGTCCCATGCGG
GACTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTGTAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01D8L81 length=245 xy=1621_1043 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAAC
TGGCTAATCAGACGCGAGCTCATCTTTGGACGATAAATCTTTGGCAGCAAAATCATGCGG
TCCTGCTGCATTATGCGGTATTAGCAGTCGTTTCCGGCTGTTATCCCCCATCCAAAGGCA
GATTG
>FLP3FBN01EHQW9 length=270 xy=1725_1019 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCCCGCCAAC
TACCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACCGGACCATGCGG
TCCTGTGCGCTTATGCGGTATTAGCAGCCGTTTCCGGCTGTTATCCCCCTGTATGAGGCA
GGTTGCCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01APYG5 length=268 xy=0178_2103 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCT
CCCAACGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCA
GGTTCCATACGTGTTACTCACCCGTGCG
>FLP3FBN01CXOXZ length=283 xy=1086_3061 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGGCCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGCCCATCCGTGGCCGGGATTGCTCCCTTTGGCGGCCCGGGGAT
GCCCCAAGGCCGCGTTACGCGGTATTAGACGGGGTTTCCCCCGCTTATCCCCCTGCCACG
GGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01C6H6I length=253 xy=1187_0504 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAAC
TACCTAATCAGACGCGAGGCCATCTTCCAGCGATAAATCTTTGGCGTCGGAGTCATGCGG
CTCCAACGCATCATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGGAAGGCA
GGTTCCTCACGCG
>FLP3FBN01BB5R9 length=286 xy=0431_1719 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCCCGTACCACCGGAGTTTTCAAGGGGTCCCCATGCAG
GGTCCCCTGTTATGCGGTATTAGCACCTGTTTCCAAGTGTTATCCCCCGGTACGGGGCAG
GTTGCCCACGCGTTACTCACCCGTCCGCCACTAAAACAGTCCGGGG
>FLP3FBN01BF51P length=271 xy=0477_0267 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAAC
TACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGG
TTCCGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCA
GGTTGCCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01DN1Z3 length=281 xy=1387_0145 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCTAC
TGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCGGGGATGCCC
CCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAG
GTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGGCAG
>FLP3FBN01CJJE7 length=268 xy=0925_2177 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
CAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTCACACCGAACCATGCGG
TTCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01AK9WE length=258 xy=0125_0716 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
TAGCTAATCAGACGCGGAACCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTGTAAGGCA
GGTTACCCACGCGTTACT
>FLP3FBN01D2BCC length=247 xy=1549_1882 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCTCACCAAC
TAGCTAATCAGACGCGAGGCCATCTCTCAGCGATAAATCTTTGGCACAGATACCATGCGA
TACCCATGCGTTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCTCTGAAAGGCA
GGTTCCT
>FLP3FBN01D2DVT length=268 xy=1550_1079 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGG
AATCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01CC1O1 length=280 xy=0851_2371 region=1 run=R_2008_12_09_13_51_01_
AACTGTGCGTACCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
AAGCTAATGCGCCGCATGCCCATCCTCCGCCGGAATCGCTTCCTTTTAACACCCGTGGGA
TGCCCCACAGATGTACCACGCGGTATTAGTCCGGATTTCTCCGGGTTATCCCCCTGCGGA
GGGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCG
>FLP3FBN01BBP7L length=268 xy=0426_2015 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAAC
CAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGCTTCATGCAA
AGCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCA
GGTTGCCCACGCGTTACTCACCCGTCCG
>FLP3FBN01DPA2Y length=264 xy=1401_1224 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGGTCCATCTTACACCACTAATGTTTTTCACTCTGTCCCATGCGG
GACTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTGTAAGGCA
GGTTACCCACGCGTTACTCACCCG
>FLP3FBN01D9LCY length=268 xy=1632_1488 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGGTCCATCCCATACCACCGGAGTTTTTCACACAGGAGCATGCGC
TCCCGTGCGCTTATGCGGTATTAGCACCTGTTTCCAGGTGGTATCCCCCGGTATGGGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01C0XC7 length=271 xy=1123_2393 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCGCCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
TAGCTAATCAGACGCGGGCCCATCCCATACCACCGGAGTTTTTACCGCTGTGCCATGCGG
CACCGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCCGTATGGGGCA
GGTTGCCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01EKL55 length=264 xy=1757_3755 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGATCAACCTCTCAGTTCGGCTACGCATCATTGCCTTGGTAAGCCTTTACCCCACCAAC
TAGCTAATGCGCCGCGGGCCCATCCAAAAGCGGTAGCATAGCCACCTTTTACATAGTTAC
CATGCGGTAACTATGGTTATGCGGTATTAGCACCTGTTTCCAAGTGTTATCCCCCTGTAC
AGGCCAGGTTGCCCACGCGTTACT
>FLP3FBN01CG3KS length=244 xy=0897_3018 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGCCACCCTCTCAGGTCGGCTACTGATCGTCACCTTGGTAGGCCGTTACCCCACCAAC
TAGCTAATCAGACGCAAGCCCATCTATCAGCGGATTGCTCCTTTTCTAGCTATATCATGC
GATACTACTAGCTTATGCGGTATTAGCAATGATTTCTCACTGTTATTCCCCTCTGATAGG
CAGG
>FLP3FBN01D5RTH length=272 xy=1588_3459 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAGGGAAAAGCCATGCGG
CTTCCCCCGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGCCAG
GTTGCCCACGCGTTACTCACCCGTCCGCCACT
>FLP3FBN01D97XE length=268 xy=1639_2064 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
CAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTCACACCGAACCATGCGG
TTCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01ASUWG length=275 xy=0211_2270 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGTCCACCCTCTCAGGCCGGCTGCTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAAC
TGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAAAGGATTACCATGCGG
TATTCCCTATTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGGTACAGGGCAG
GTTGCCCACGCGTTACTCACCCGTCCGCCACTAAG
>FLP3FBN01BTIOW length=276 xy=0629_0590 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCTCCAAC
CAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGTACCATGCGG
TACTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCA
GGTTACCCACGCGTTACTCACCCGTCCGCCACTAAG
>FLP3FBN01EM86A length=255 xy=1787_4000 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
TAGCTAATCAGACGCGGGTCCATCTCATACCACCGGAGTTTTTCACACTGGATCATGCAA
TCCTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTATATGAGGCA
GGTTACCCACGCGTT
>FLP3FBN01CHFB7 length=268 xy=0901_1877 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGCTGCCTTGGTGGGCCGTTACCCCGCCAAC
TAGCTAATCAGACGCGGAACCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGGTATACGGCA
GGTTTTCCACGCGTTACTCACCCGTCCG
>FLP3FBN01CNAWV length=244 xy=0968_1645 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAAC
TAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGA
CATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCT
TGGG
>FLP3FBN01AKUQT length=268 xy=0120_1555 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAG
GACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGTAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01AP4SS length=264 xy=0180_2106 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGATCAACCTCTCAGTTCGGCTACGCATCATTGCCTTGGTAAGCCTTTACCCCACCAAC
TAGCTAATGCGCCGCGGGCCCATCCAAAAGCGGTAGCATAGCCACCTTTTACATAGTTAC
CATGCGGTAACTATGGTTATGCGGTATTAGCACCTGTTTCCAAGTGTTATCCCCCTCTTT
TGGGCAGGTTGCCCACGTGTTACT
>FLP3FBN01B770T length=268 xy=0796_2571 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGTATCATGCGA
TACTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCA
GGTTCCCCACGCGTTACTCACCCGTCCG
>FLP3FBN01BV9MF length=253 xy=0660_1829 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTCTCCACGCG
>FLP3FBN01EEO8Q length=268 xy=1690_2232 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
CAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTCACACCAGACCATGCGA
TCCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01BC5FW length=264 xy=0442_2874 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGATCAACCTCTCAGTTCGGCTACGCATCATTGCCTTGGTAAGCCTTTACCCCACCAAC
TAGCTAATGCGCCGCGGGCCCATCCAAAAGCGGTAGCATAGCCACCTTTTACATAGCTAC
CATGCGGTAACTATGGTTATGCGGTATTAGCACCTGTTTCCAAGTGTTATCCCCCTCTTT
TGGGCAGGTTGCCCACGTGTTACT
>FLP3FBN01BOTE5 length=268 xy=0575_2399 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGATCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
TAACTAATCAGACGCGGGTCCATCTCATACCACCGGAGTTTTTCACACCGGACCATGCGG
TCCTGTGCGCTTATGCGGTATTAGCAGTCATTTCTAACTGTTGTCCCCCAGTATACGGCA
GGTTGCCCACGCGTTACTCACCCGTCCG
>FLP3FBN01BUMJK length=265 xy=0641_3086 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAAC
AAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGA
AGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGC
GCGGGCAGGTTCCCTACGTGTTACT
>FLP3FBN01EGB8E length=273 xy=1709_0860 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAAC
TACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGG
TTCCGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCGCCGCT
>FLP3FBN01AUN6V length=255 xy=0232_0869 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTCTCCACGCGTT
>FLP3FBN01CBJWL length=264 xy=0834_2291 region=1 run=R_2008_12_09_13_51_01_
AGCACGAGCCTACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCCTTGGTGGGCTTTTACCTCACCAAC
TAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAG
GACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCG
>FLP3FBN01DH3NR length=257 xy=1319_0885 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAAC
TACCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAG
GACCGTGCGCTTATGCGGTATTGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAG
GTTACCCACGCGTTACT
>FLP3FBN01EM0GU length=271 xy=1785_0908 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAAC
TACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGG
TTCCGCGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCA
GGTTGCCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01D69SQ length=271 xy=1605_3784 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGAGCCGTTACCTCACCAAC
TACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCTCAGAACCATGCGG
TTCTGGGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCA
GGTTGCCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01BFKOF length=268 xy=0470_1245 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGGTCCATCGTATACCGTCTCGGCTTTTCACTCGGTATCATGCGA
TACCGGGCGCTTATGCGGTATTAGCACCTGTTTCCAGGTGTTATCCCCCTGTATACGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01D0CQF length=264 xy=1527_0485 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGGCTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGG
ACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCA
GGTTGGATACGTGTTACTCACCCG
>FLP3FBN01CWRA1 length=276 xy=1076_0427 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGGTTTGGTGGGCCGTTACCCCTCCAAC
CAGCTAATCAGACGCGGGTCCATCCTGTACCACCGGAGTTTTTCACACTGTACCATGCGG
TACTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCA
GGTTACCCACGCGTTACTCACCCGTCCGCCACTAAG
>FLP3FBN01A85QC length=268 xy=0397_0946 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTACCAAC
AAGCTAATCAGACGCGGAGCCATCTTACACCACCTCAGTTTTTCACACCGGACCATGCGG
TCCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTGTAAGGCA
GGTCCTCCACGCGTTACTCACCCGTCCG
>FLP3FBN01CBFWS length=255 xy=0833_1210 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCTGTTACCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTCTCCACGCGTT
>FLP3FBN01CSXJW length=258 xy=1032_2122 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACCGATCGTCGCCTTGGTAGGCCTTTACCCCACCAAC
TAGCTAATGCGCCGCAGGCTCATCCATCAGTGATGCCAGGAGCATCTTTAAACTTTCGTC
CTATCCGGTATTAGCGATCGTTTCCAATCGTTGTCCCCGTCTGATGGGCAGATCACCTAC
GTGTTACTCACCCGTTCG
>FLP3FBN01CABHI length=264 xy=0820_2068 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGGCCAACCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
CAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTCACACCGAACCATGCGG
TTCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCG
>FLP3FBN01BIK18 length=259 xy=0504_2446 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCT
CCCAACGTTCTTACGCGGTATTAGACGGAATTTCTCCCGATTATCCGGCTGTGGCAGGCA
GGTTGCATACGTGTTACTC
>FLP3FBN01D26CI length=270 xy=1559_1104 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTCTACCCCGCCAAC
AAGCTAATCAGACGCGGGTCCATCGTATACCACCGGAGTTTTTCACACCGGACCATGCGA
TCCTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCA
GGTTACCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01B5LFD length=268 xy=0766_2855 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCT
CCCAACGTTCTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCA
GGTTCCATACGTGTTACTCACCCGTGCG
>FLP3FBN01ADQU7 length=244 xy=0039_1713 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAAC
TAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGA
CATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCT
TGGG
>FLP3FBN01D5SXD length=253 xy=1589_0799 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTCTCCACGCG
>FLP3FBN01BKOLX length=268 xy=0528_2051 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGCCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAAC
CAGCTAATCAGACGCGGGCCCATCCTGCACCGCCGGAGCTTCCCCCGCCGCCCCATGCGG
GGCTGCGGGCATATGCGGTATTAGCAGCCGTTTCCGGCTGTTGTCCCCCGGTATACGGCA
GGTTTTCCACGCGTTACTCACCCGTCCG
>FLP3FBN01CEGMP length=267 xy=0867_2847 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGCCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGACT
AGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCTC
CCAACGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCAG
GTTCCATACGTGTTACTCACCCGTGCG
>FLP3FBN01E2P4M length=258 xy=1964_0820 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGG
ACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCA
GGTTGGATACGTGTTACT
>FLP3FBN01C64QW length=270 xy=1194_1078 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTACCGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAAC
TGGCTAATCGGGCGCGGGCCCATCCCGTGCCGCCGGAGCTTTCCGCATACACCCATGCGG
CTGTATGCGCTTATGCGGTATTAGCGCCTGTTTCCAGGCGGTATCCCCCGGCACGGGGCA
GGTTGCCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01DGS0V length=264 xy=1304_1885 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGG
ACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCA
GGTTGGATACGTGTTACTCACCCG
>FLP3FBN01C2VAB length=244 xy=1145_2897 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAAC
TAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGA
CATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCT
TGGG
>FLP3FBN01DYU4B length=268 xy=1510_0633 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCTTTGGTAGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGGTCCATCTCATACCGATAAATCTTTTCCGTCCGTATCATGCGA
TACTAGCGGGTTATGCGGTATTAGCGGTCGTTTCCAACTGTTATCCCCCTGTATGAGGTA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01AIIWJ length=274 xy=0093_3489 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCAAC
AACCTAATGGAACGCATCCCCATCGATGACCGAAGTTCTTTAATAGTTCTACCATGCGGA
AGAACTATGCCATCGGGTATTAATCTTTCTTTCGAAAGGCTATCCCCGAGTCATCGGCAG
GTTGGATACGTGTTACTCACCCGTGCGCCGGTCG
>FLP3FBN01BJV1A length=264 xy=0519_1884 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGG
ACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCA
GGTTGGATACGTGTTACTCACCCG
>FLP3FBN01ANBIL length=276 xy=0148_1915 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCGCCGACT
AGCTAATGCGCCGCATGCCCATCCGCCACCGGTAATCCCTTTGGCGGCACCGGGATGCCC
CGATGCCGCGTCACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGTCGGCGGGC
AGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCG
>FLP3FBN01AJ8QF length=268 xy=0113_1701 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAAC
TAGCTAATGTCACGCATGCCCATCCGTTGCCGGAATCGCTTCCTTTGGCCCCAGGGCCAT
GCAGCCCCGCGGCATTACGCGGTATTAGACGGGATTTCTCCCGATTATCCCCCTGCAACG
GGCAGGTCGCATACGCGTTACTCACCCG
>FLP3FBN01C4QQE length=272 xy=1167_0196 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTGGGCCGTTACCCCGCCAAC
TACCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTACCACCGGACCATGCAG
TCCTGTGGTCTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGCA
GGTTGCCCAACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01BS0JY length=268 xy=0623_1660 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAAC
TGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGG
AACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCA
GGTTATCCACGCGTTACTCACCCGTCCG
>FLP3FBN01DFM2G length=265 xy=1291_0758 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTCTACCCCGCCAAC
TAGCTAATCAGACGCGGGTCCATCGTATACCGTTTACGCTTTTCACACTGCTCCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCA
GGTTACCCACGCGTTACTCACCCCG
>FLP3FBN01AIAA5 length=277 xy=0091_0543 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTGGTCTTGGTGAGCCGTTACCCCACCAAC
TAACTAATACAACGCATGCCCATCCATTACCACCGGAGTCTTCAACCCAAGAAGATGCCT
CCCTGGATGTTATGGGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTAATGGGTAG
GTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCA
>FLP3FBN01CAKPN length=244 xy=0823_1737 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAAC
TAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGA
CATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCT
TGGG
>FLP3FBN01A6R51 length=281 xy=0370_0647 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGTCCGTGTCTCAGTACCAATGTGG
GGGGTTAACCTCTCAGTCCCCCTATGTATCGTTGTCTTGGTGAGCCGTTACCCCACCAAC
TAACTAATACAACGCATGCCCATCCATTACCACCGGAGTTTTCAACCCAAGAAGATGCCT
CCCTGGATGTTATGGGGTATTAGTACCGATTTCTCAGTGTTATCCCCCTGTAATGGGTAG
GTTGCATACGCGTTACGCACCCGTGCGCCGGTCGCCGACAA
>FLP3FBN01EH9DF length=258 xy=1731_0353 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGGCCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTAGGCCGTTACCCTACCAAC
TAGCTAATCAGACGCGAGGCCATCTCTCAGCGATAAATCTTTGGTACAGATACCATGCGA
TACCCGTACGTTATGCGGTATTAGCAGTCGTTTCCAACTGTTGTCCCCCACTGAAAGGCA
GGTTCCTCACGCGTTACT
>FLP3FBN01C1ALJ length=240 xy=1127_3157 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTATCTCAGTCCCAATGTGG
CCGTTCAACCTCTCAGTCCGGCTACTGATCGTCGCCTTGGTAGGCCTTTACCCCACCAAC
TAGCTAATCAGACGCAAGCCCATCTGTAAGTGGATTGCTCCTTTCCTCACTCTAACATGT
GTCAGTGTGAGCATATGCGGTATTAGCAATGATTTCTCACTGTTATTCCCCTCTTACAGG
>FLP3FBN01B0TAP length=258 xy=0712_0959 region=1 run=R_2008_12_09_13_51_01_
ACCAGCGACTAGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGGTCCATCGTATGCCACCGGAGTTTTTCACACAGGAGCATGCGC
TCCCGTGCGCTTATGCGGTGTTAGCACCTATTTCTAAGTGTTATCCCCCTGCATACGGCA
GGTTACCCACGCGTTACT
>FLP3FBN01C2UWD length=264 xy=1145_2395 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
CAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTCACACTGAACCATGCGG
TTCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGTGTTACTCACCCG
>FLP3FBN01CKQDF length=271 xy=0939_0497 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGGTCCATCTCATACCGATAAATCTTTTCCGTCCGTATCATGCGA
TACTAGCGGGTTATGCGGTATTAGCGGTCGTTTCCAACTGTTATCCCCCTGTGTAAGGCA
GGTTGCCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01A1GYR length=268 xy=0309_2705 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAAC
TGGCTAATCAGACGCGGGTCCATCTTACACCACTAATGTTTTTCACTCTGTCCCATGCGG
GACTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTCTCCACGCGTTACTCACCCGTCCG
>FLP3FBN01CD97W length=268 xy=0865_2730 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCGG
GACCGTGCGCTTATGCAGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01DIARK length=264 xy=1321_1902 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGGCCCATCGTATACCGTCTCGGCTTTTCACACTCCAGTATGCAC
TACTGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCA
GGTTACCCACGCGTTACTCACCCG
>FLP3FBN01B438A length=274 xy=0761_1048 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCCTGCACCACCGGAGTTTTCAAAGGATTACCATGCGG
TATTCCCTATTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGGTACAGGGCAG
GTTGCCCACGCGTTACTCACCCGTGCGCCAGTCG
>FLP3FBN01D0219 length=268 xy=1535_1839 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
CAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGTCCCATGCAG
GACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01EA4TG length=268 xy=1649_3730 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGGCCATTCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCT
CCCAACGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCA
GGTTCCATACGTGTTACTCACCCGTGCG
>FLP3FBN01BGW2P length=244 xy=0485_2527 region=1 run=R_2008_12_09_13_51_01_
ACAGACCACTCACATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGATCAGTCTCTCAACTCGGCTATGCATCATTGCCTTGGTAAGCCGTTACCTTACCAAC
TAGCTAATGCACCGCAGGTCCATCCAAGAGTGATAGCAGAACCATCTTTCAAACTCTAGA
CATGCGTCTAGTGTTGTTATCCGGTATTAGCATCTGTTTCCAGGTGTTATCCCAGTCTCT
TGGG
>FLP3FBN01CLS7R length=251 xy=0951_1685 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCGCCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCCCACCAAC
AAGCTAATCAGACGCGGGTCCATCTTGCACCACCGGAGTTTTTCACACTGAGCCATGCAG
CTCCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACG
>FLP3FBN01D8ZMU length=264 xy=1625_2004 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGATCAACCTCTCAGTTCGGCTACGCATCATTGCCTTGGTAAGCCTTTACCCCACCAAC
TAGCTAATGCGCCGCGGGCCCATCCAAAAGCGGTAGCATAGCCACCTTTTACATAGTTAC
CATGCGGTAACTATGGTTATGCGGTATTAGCACCTGTTTCCAAGTGTTATCCCCCTCTTT
TGGGCAGGTTGCCCACGTGTTACT
>FLP3FBN01COFBD length=265 xy=0981_0759 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGATCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
CAGCTAATCAGACGCAGGTCCATCTTACACCGATAAAATCTTTTCCGTCCGGGCCATGCG
GCCCTAGCGGGTTATGCGGTATTAGCAGTCATTTCTAACTGTTATCCCCCTGTGTAAGGC
AGGTTACCCACGCGTTACTCACCCG
>FLP3FBN01A88RD length=255 xy=0398_0775 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGGTCGTCGCCTTGGTGGGCTGTTACCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGTGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTCTCCACGCGTT
>FLP3FBN01AIDK4 length=277 xy=0092_0694 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTTGCCTTGGTGGGCCGTTACCCCGCCAAC
AAGCTAATCAGACGCGGGCCCATCTTACACCACCGGAGTTTTTCACACTGTCCCATGCAG
GACCGTGCGCTTATGCGGTTTTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGGCG
AGGTTACCCACGCGTTACTCACCCGTCCGCCACTAAG
>FLP3FBN01DCKQ6 length=258 xy=1256_1152 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAATGTGG
GGGACCTTCCTCTCAGAACCCCTATCCATCGAAGACTAGGTGGGCCGTTACCCCGCCTAC
TATCTAATGGAACGCATCCCCATCGTCTACCGGAATACCTTTAATCATGTGAACATGCGG
ACTCATGATGCCATCTTGTATTAATCTTCCTTTCAGAAGGCTGTCCAAGAGTAGACGGCA
GGTTGGATACGTGTTACT
>FLP3FBN01BZNMV length=272 xy=0699_0213 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCT
CCCAACGTTGTTACGCGGTATTAGACGGAATTTCTTCCGGTTATTCCCCTGTGACGGGCA
GGTTGCATACGTGTTACTCACCCGTGCGCCGG
>FLP3FBN01BLVR2 length=262 xy=0542_0656 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTAGACATCGTCGCCTTGGTGGGCCGTTGCCCCGCCAAC
TAGCTAATGTCACGCATGCCCATCCGTTGCCGGAATCGCTTCCTTTGGCCGCAGGGCCAT
GCAGCCCCGCGGCATTACGCGGTATTAGACGGGATTTCTCCCGATTATCCCCCTGCAACG
GGCAGGTCGCATACGCGTTACT
>FLP3FBN01D9KO8 length=269 xy=1632_0634 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGAGCCGTTACCTCACCAAC
CAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTTCACACCGAACCATGCGG
TTCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCA
GGTTACCCACGCGTTACTCACCCGTCCCG
>FLP3FBN01EWA03 length=277 xy=1891_0325 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
TAGCTAATGCGCCGCATGGCCATCCGTAGCCGGTGTTACCCTTTAAACCCCAGGAGATGC
CTCTCGGAGTTATTACGCGATATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTACGGG
CAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCG
>FLP3FBN01CUHT2 length=268 xy=1050_1336 region=1 run=R_2008_12_09_13_51_01_
AACTCGTCGATGCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCACCCTCTCAGGCCGGCTATGGATCGTCGGTTTGGTGGGCCGTTACCCCGCCAAC
TGCCTAATCCAACGCGGACCCATCCTATGCCGCCTCAGCTTTTCACACCGTACCATGCGG
TACTGTGCGCTTATGCGGTATTAGCACCCGTTTCCGGATGTTATCCCCCTGCATAGGGCA
GGTTGTCCACGCGTTACTCACCCGTCCG
>FLP3FBN01DZILX length=262 xy=1517_2403 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAAC
CAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGA
AGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGC
GCGGGCAGGTTCCCTACGTGTT
>FLP3FBN01EMNLG length=268 xy=1781_0610 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGACCGTGTCTCAGTTCCAATGTGG
CCGATCACCCTCTCAGGTCGGCTACTGATCGCCGCCTTGGTAAGCCGTTACCTTACCAAC
TAGCTAATCAGACGCGGGTCCATCCTGTACCGCAAAAGCTTTGATACTTCTACCATGCGA
TAAAAGTATTTTATCTCGTATTAGCATACCTTTCGGTATGTTATCCGTGTGTACAGGGCA
GGTTACCCACGCGTTACTCACCCGTCCG
>FLP3FBN01CJ0R3 length=253 xy=0931_0097 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAGTGTGT
CCGATCACCCTCTCAGGCCGGATACCCGTCATAGCCTTGGTGAGCCATTACCTCACCAAC
AAGCTGATAGGACATAGGCTGATCCTTTAGCGAAAAACTTTCCCCCGTAGGGAGTATCCA
GTATTAATCACCGTTTCCAGTGGCTATCCCAGACTAAAGGGCACATAACCTATGCATTAC
TCACCCGTGCGCC
>FLP3FBN01EPZM5 length=272 xy=1819_0543 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCT
CCCAACGTTGTTACGCGGTATTAGACGGAATTTCTTCCGGTTATTCCCCTGTGACGGGCA
GGTTGCATACGTGTTACTCACCCGTGCGCCGG
>FLP3FBN01DKURH length=268 xy=1350_2347 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAAC
TAGCTAATCAGACGCGGGCCCATCTCATACCACCGGAGTTTTTCACACTGTTCCATGCGG
AACCGTGCGCTTATGCGGTATTAGCACCTGTTTCCAAGTGTTATCCCCCTCTTTTGGGCA
GGTTGCCCACGTGTTACTCACCCGTTCG
>FLP3FBN01A52PB length=283 xy=0362_0413 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGCCCATCCGTGGCCGGGATTGCTCCCTTTGGCGGCCCGGGGAT
GCCCCAAGGCCGCGTTACGCGGTATTAGACGGGGTTTCCCCCGCTTATCCCCCTGCCACG
GGCAGGTTGCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
>FLP3FBN01DKNLI length=255 xy=1348_1252 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
CAGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGCTTCATGCGA
AGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTATACGGCA
GGTTCCTCACGCGTT
>FLP3FBN01BIX4J length=268 xy=0508_2993 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCT
CCCAACGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCA
GGTTCCATACGTGTTACTCACCCGTGCG
>FLP3FBN01CCNMN length=264 xy=0847_0525 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTTCCAATGTGG
CCGATCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTAAGCCGTTACCTTACCAAC
TAGCTAATCAGACGCGGGTCCATCCTGTACCGCAAAAGCTTTGATACTTCTACCATGCGA
TAAAAGTATTTTATCTCGTATTAGCATACCTTTCGGTATGTTATCCGTGTGTACAGGGCA
GGTTCCCTACGTGTTACTCACCCG
>FLP3FBN01EO993 length=264 xy=1811_0441 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTTCCAATGTGG
CCGATCACCCTCTCAGGTCGGCTACTGATCGTCGCCTTGGTAAGCCGTTACCTTACCAAC
TAGCTAATCAGACGCGGGTCCATCCTGTACCGCAAAAGCTTTGATACTTCTACCATGCGA
TAAAAGTATTTTATCTCGTATTAGCATACCTTTCGGTATGTTATCCGTGTGTACAGGGCA
GGTTCCCTACGTGTTACTCACCCG
>FLP3FBN01A1THR length=268 xy=0313_2557 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCT
CCCAACGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCA
GGTTCCATACGTGTTACTCACCCGTGCG
>FLP3FBN01A1UPS length=265 xy=0314_0046 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCAACCTCTCAGTCCGGCTACCGATCGTCGACTTGGTGGGCCGTTACCTCACCAAC
TATCTAATCGGGTGCGAGCCCATCTCTAAGCGGATTTCTCCTTTGCCGGTTGTACAATGC
TATACTGCCGGTTTATGTGGTATTACCACCCGTTTCCAGGTGCTATCCCTCTCTTAAAGG
TAGGTTGCTCACATGTTACTCACCC
>FLP3FBN01D1GVB length=268 xy=1539_3349 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCACT
AGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCCGGGAGATGCCT
CCCAACGTTCTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCA
GGTTCCATACGTGTTACTCACCCGTGCG
>FLP3FBN01DCLZP length=255 xy=1256_2755 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCACCCTCTCAGGTCGGCTACTGATCGTCGGTTTGGTGGGCCGTTACCCCACCAAC
TAGCTAATCAGACGCAAGCCCATCTGTAAGTGGATTGCTCCTTTCCTCACTCTAACATGT
GTCAGTGTGAGCATATGCGGTATTAGCACCCTATTTCTAACGTGTTATCCCCCAGTATAC
GGCAGGTTCTCCACG
>FLP3FBN01CYQG0 length=268 xy=1098_2546 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAAC
TGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGG
AACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCA
GGTTATCCACGCGTTACTCACCCGTCCG
>FLP3FBN01BLZYI length=268 xy=0543_1976 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAAC
TGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGG
AACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTATACGGCA
GGTTATCCACGCGTTACTCACCCGTCCG
>FLP3FBN01CYH88 length=271 xy=1096_0090 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGGTCGCCCTCTCAGGTCGGCTACTGATCGTCGGCTTGGTAGGCTTCTACCCCACCAAC
TACCTAATCAGACGCGGGCCCATCTTACACCACCTCAGTTTTTACCTCTGTACCATGCGG
TACCGGGGTCTTATGCGGTATTAGCAATCATTTCTAATTGTTATCCCCCTGTGTAAGGCA
GGTTGCCCACGCGTTACTCACCCGTCCGCCG
>FLP3FBN01C5F5F length=277 xy=1175_0369 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTTGGTGGGCCGTTACCCCGCCAAC
TAGCTAATGCGCCGCATGGCCATCCGTAGCCGGTGTTACCCTTTAAACCCCAGGAGATGC
CTCTCGGAGTTATTACGCGATATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTACGGG
CAGGTTCCATACGTGTTACTCACCCGTGCGCCGGTCG
>FLP3FBN01BHIQC length=278 xy=0492_1922 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCGTTACCCCGCCTAC
TGCCTAATGCGCCGCATGCCCATCCTCCACCGGTAATCCTTTCCTCCCCCAAGGATGCCC
CCAAGGGATATACGCGGGATTAGCCTCCCTTTCGGAAGGTTGTCCCCCTGTGGAGGGCAG
GTTGCATACGTGTTACTCACCCGTGCGCCAGTCGCCGG
>FLP3FBN01B03J2 length=271 xy=0715_1968 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGATCAACCTCTCAGTTCGGCTACGCATCATTGCCTTGGTAAGCCCTTTACCCCACCAA
CTAGCTAATGCGCCGCGGGCCCATCCAAAAGCGGTAGCATAGCCACCTTTTACATAGTTA
CCATGCGGTAACTATGGTTATGCGGTATTAGCACCTGTTTCCAAGTGTTACTCCCCCTTC
TTTTGGGCAGGGTTCGCCCCACGTTGTTACT
>FLP3FBN01CKVKR length=262 xy=0940_3145 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAAC
CAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGA
AGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGC
GCGGGCAGGTTCCCTACGTGTT
>FLP3FBN01DBF9Z length=253 xy=1243_1941 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGACCGTGTCTCAGTTCCAGTGTGT
CCGATCACCCTCTCAGGCCGGATACCCGTCATAGCCTTGGTGAGCCATTACCTCACCAAC
AAGCTGATAGGACATAGGCTGATCCTTTAGCGAAAAACTTTCCCCCGTAGGGAGTATCCA
GTATTAATCACCGTTTCCAGTGGCTATCCCAGACTAAAGGGCACATAACCTATGCATTAC
TCACCCGTGCGCC
>FLP3FBN01APHXO length=265 xy=0173_1146 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAAC
CAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGA
AGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGC
GCGGGCAGGTTCCCTACGTGTTACT
>FLP3FBN01C0GRI length=268 xy=1118_1356 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTGGGCCGTTACCCCGCCAAC
TGGCTAATCAGACGCGGATCCATCGTATACCACCGGAGTTTTTCACACTGTTCCATGCGG
AACCGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCGGTATACGGCA
GGTTATCCACGCGTTACTCACCCGTCCG
>FLP3FBN01CEZZE length=262 xy=0873_3352 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAAC
AAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGA
AGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGC
GCGGGCAGGTTCCCTACGTGTT
>FLP3FBN01AGO5G length=277 xy=0073_0194 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGGTTAGGTGGGCCTTTACCCCGCCAAC
AAACTAATGCACCGCAGGTCCATCCGCGCCCCATCCCCTAAAGGATGTTTCACAGAAAGA
AGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGC
GCGGGCAGGTTCCCTACGTGTTACTCACCCGTTCGCC
>FLP3FBN01A9WKI length=262 xy=0405_2960 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTCCGCCCTCTCAGGCCGGCTATGCATCATCGTCTTGGTGGGCCTTTACCCCGCCAAC
AAACTAATGCACCGCAGGTCCATCCGCACCCCATCCCCTAAAGGATGTTTCACAGAAAGA
AGATGCCTCCTTCCTGTACATCGGGATTTGTTCTCCGTTTCCAGAGCGTATTCCCGGTGC
GCGGGCAGGTTCCCTACGTGTT
>FLP3FBN01AMKUE length=268 xy=0140_0116 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCT
CCCAACGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCA
GGTTCCATACGTGTTACTCACCCGTGCG
>FLP3FBN01CD50B length=269 xy=0864_1369 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTTGCCTTGGTGGGCCGTTACCCCGCCAAC
AAGCTAATCAGACGCGGGTCCATCTTACACCACCGGAGTTTTCAAGTAAAAGACATGCGT
CTCCTACTGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCAGTGCAAGGCAG
GTTACCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01CXEQE length=258 xy=1083_2116 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTTTGGGCCGTGTCTCAGTCCCAATGTGG
CCGTTCACCCTCTCAGGCCGGCTACTGATCGTCGCCTTGGTGGGCCTTTACCCCGCCAAC
AAGCTAATCAGACGCGGATCCATCGTATACCACCAAAAGCTTTAGCTTTTTGTTTTCCAC
ACTGCTTCATGCGAAGCTGTGCGCTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCC
CCCAGTATACGGCAGGTT
>FLP3FBN01C6MIF length=269 xy=1188_2021 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGGCCGTGTCTCAGTCCCAGTGTGG
CCGTCCGCCCTCTCAGGCCGGCTACTGATCGTCGCTTTGGTAGGCCGTTACCCTGCCAAC
TGGCTAATCAGACGCGGGCCCATCCTGTACCACCGGAGTTTTCAGGGAAAAGCCATGCGG
CTTCCCCCGTTATGCGGTATTAGCACCTATTTCTAAGTGTTATCCCCCTGTACAGGCCAG
GTTGCCCACGCGTTACTCACCCGTCCGCC
>FLP3FBN01CKQYM length=268 xy=0939_1260 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGGCCATCCGCAGCCGATAAATCTTTAAACATCGGGAGATGCCT
CCCAACGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCA
GGTTCCATACGTGTTACTCACCCGTGCG
>FLP3FBN01A68AG length=279 xy=0375_1062 region=1 run=R_2008_12_09_13_51_01_
AGCAGCACTTGTCATGCTGCCTCCCGTAGGAGTCTGGTCCGTGTCTCAGTACCAGTGTGG
GGGACCTTCCTCTCAGAACCCCTACGCATCGTCGCCTCGGTGGGCCGTTACCCCGCCGAC
TAGCTAATGCGCCGCATGGCCATCCGCAGCCGGTAAACCTTTAAACCACCGGAGATGCCT
CCAACGGTTGTTACGCGGTATTAGACGGAATTTCTTCCGCTTATCCCCCTGCTGCGGGCA
GGTTCCATACGTGTTACTCACCCGTGCGCCGGTCGCCGG
"""

fasting_tutorial_qual = """>FLP3FBN01ELBSX length=250 xy=1766_0111 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 33 33 33 36 37 37 37 37 37 37 40 40 40 39 39 38 40 40 40 40 40 40 40 37 37 37 37 37 35 35 35 37 37 37 37 37 35 35
35 31 31 23 23 23 31 21 21 21 35 35 37 37 37 36 36 36 36 36 36 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 28 28 28 36 36 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 36 37 37 37 37 37 37 37 37 37 37 37 37 36 36 36 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 35 32 32 32 32 35 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 32 32 32 36 37 35 32 32 32 32 32 32 32 32 36 37 37 37 37 36 36 31 31 32 32 36 36 36 36 36 36 36 36 36 36 36 28 27 27 27 26 26 26 30 29
30 29 24 24 24 21 15 15 13 13
>FLP3FBN01EG8AX length=276 xy=1719_1463 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 37 33 33 21 21 21 26 33 37 36 36 40 33 24 24 29 33 33 39 39 39 40 39 39 39 40 37 37 37 37 37 37 37 37 37 37 37 32 32 20 20
20 20 20 35 35 37 37 37 37 37 37 37 36 36 36 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 36 36 36 36 37 37 37 37 37 36 36 36 36 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 33 28 28 28 28 36 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 33 33 33 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 36 36 36 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 28 28 28 37 28 28 28 37 37 37 37 37 36 36 36 36 36 28 26 26 26 26 28 36 36 36 36 36 36 36 37 38 38 38 38 38
37 37 37 37 37 31 31 31 31 31 31 31 31 31 31 31 31 30 22 22 22 25 25 31 31 31 31 31 31 31 25 25 25 25 25 28
>FLP3FBN01EEWKD length=244 xy=1692_3531 region=1 run=R_2008_12_09_13_51_01_
36 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 33 33 26 21 21 31 36 37 37 37 36 36 33 34 34 36 37 36 37 37 37 37 37 37 37 37 37 37 37 36 28 28 28 36 36 37 37 35 35 35
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 32 32 32 35 35 37 35 32 32 32 37 37 37 37 37 37 36 36 36 36 36
36 36 36 37 37 37 37 37 37 37 35 35 35 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 35 35 35 37 37 37 37 37 37 37 37 37 37 36 36 36 37 36 35 35 35 37 28 28 28 32 35 37 37 37 36 36
36 37 37 37 37 37 37 35 35 35 35 35 37 37 37 37 36 36 36 37 28 28 28 28 35 36 37 37 37 37 37 37 37 37 37 37 36 33 33 32 31 36 36 33 33 27 27 27 36 31 25 25 25 32 36 36 36 36 36 36
36 36 36 36
>FLP3FBN01DEHK3 length=252 xy=1278_0245 region=1 run=R_2008_12_09_13_51_01_
36 36 36 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 38 38 39 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 33 33 33 36 36 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 33 33 33 33 37 37 37 37 37 37 37 37 37 37 37 37
37 37 36 36 28 28 28 28 33 33 33 36 36 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 36 36 36 36 31 31 27 27 28 28 28 27 21 31 31 36 36 36 36 36 36
36 36 36 36 36 31 31 31 31 31 31 31
>FLP3FBN01DGFYQ length=269 xy=1300_1344 region=1 run=R_2008_12_09_13_51_01_
35 35 35 35 33 31 31 31 33 35 35 35 35 35 35 35 35 35 35 35 35 35 23 20 20 31 31 33 33 33 35 23 17 17 21 20 20 20 31 31 33 35 35 35 35 35 33 33 33 35 31 31 31 35 35 35 35 35 35 35
31 31 31 33 35 35 35 35 35 35 35 35 35 35 31 31 31 26 26 26 26 35 35 35 35 35 35 35 33 31 31 31 35 35 35 35 35 35 35 35 35 35 35 35 35 35 31 31 31 35 35 35 33 33 33 33 33 35 35 35
35 35 35 35 35 35 35 35 35 35 35 35 35 35 30 26 26 26 30 33 35 35 35 35 35 35 35 35 33 33 33 35 33 27 27 25 25 25 27 14 14 14 14 14 25 25 34 34 35 35 35 32 33 33 32 35 35 32 25 25
15 20 20 20 28 35 33 33 33 33 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 29 24 24 24 29 35 35 35 35 33 33 31 31 34 34 34 34 34 34 31 20 20 20 20 20 31 34 31 31 31 31 32 31 31 33
34 25 25 20 20 18 25 28 28 22 20 22 28 28 28 30 30 29 29 29 30 25 25 25 29 29 26 26 25
>FLP3FBN01A55WZ length=272 xy=0363_0481 region=1 run=R_2008_12_09_13_51_01_
32 32 32 32 35 35 35 35 35 35 35 35 35 35 35 35 35 35 38 38 39 39 32 32 32 35 35 35 35 35 34 31 21 21 25 35 32 25 25 25 32 35 35 37 39 35 35 35 35 35 35 35 35 35 35 35 35 35 32 32
32 32 32 35 32 32 32 32 35 35 35 35 35 35 35 35 35 35 32 32 32 32 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 34 34 26 26 26 26 32 35 35 35 35 35
35 34 34 34 34 34 34 34 35 35 35 35 35 35 35 35 35 26 26 26 26 35 35 35 35 35 35 35 35 34 34 34 34 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 30 22 24 21 21 21 30 35 35 35 35
35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 29 29 26 34 35 27 27 27 27 35 35 35 35 35 35 35 32 32 32 32 32 35 35 35 35 35 35 35 35
35 35 31 32 32 25 28 25 25 25 25 30 30 30 30 30 30 30 30 28 22 22 22 28 28 30 25 22 22 22 30 30
>FLP3FBN01D7O1S length=279 xy=1610_3070 region=1 run=R_2008_12_09_13_51_01_
36 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 37 36 40 28 28 28 33 40 38 37 37 39 37 40 37 38 38 38 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 36 36 33 33 33 36 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 35 35 37 31 24 21 21 21 21 23 32 36 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 28 28 28 28 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 28 28 28 33 33 37 37 37 37 37 28 28 28 33 33 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 33 33 33 33 37 36 28 28 26 36 37 37 37 37 36 31 31 26 31 31 36 36 30 17 16 16 19 19 19 26 30 30 32 33 36 36 36 36 36 33 33
33 36 36 33 33 28 31 31 31 31 31 28 28 28 31 31 31 31 25 22 22 22 25 28 28 28 31 28 28 28 28 31 29 28 28 28 29 25 16
>FLP3FBN01CV5ZU length=274 xy=1069_1480 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 31 31
31 31 31 31 37 37 37 37 37 37 37 37 37 37 37 38 38 33 33 33 33 33 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 40 40 40 40 40 40 40 40 40 40 38 38 38 38 40 38 38 38 38 38 38 38 38
38 38 38 38 38 30 30 31 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 28 28
>FLP3FBN01DQ783 length=264 xy=1423_0757 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 38 38 38 38 40 38 38 38 38 38 38 38 38 38 36 38
37 37 36 36 36 30 31 31 31 31 31 31 31 31 31 32 31 31 31 31 31 25 25 25
>FLP3FBN01DDPFF length=259 xy=1269_0617 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 34 33
33 33 33 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 31 31 31 38 37 37 36 36 33 33 33 36 37 36 38 38 38 37 38 38 38 38 38 38 36 36 36 36 38 36
36 36 28 28 29 21 21 22 28 32 31 31 31 31 31 32 31 31 28
>FLP3FBN01CPD70 length=274 xy=0992_0942 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 32 32
32 32 32 32 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 34 28 28 28 26 33 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 40 40 40 40 40 40 40 38 38 38 38 38 38 38 40 40 40 40 40 40
40 38 38 38 38 31 31 32 32 32 31 31 31 31 31 30 30 30 31 31 32 32 32 32 32 32 32 32 32 32 32 31 31 31
>FLP3FBN01BBAE6 length=269 xy=0421_2032 region=1 run=R_2008_12_09_13_51_01_
35 35 35 36 36 32 32 32 34 34 35 35 37 37 37 36 34 32 34 36 37 37 33 33 33 37 37 38 38 38 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 28 26 26 35 35 37 37 35 31 21 21 21 26 32 35 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 35 35 35 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 36 36 35 32 32 32 37 36 36 26 26 26 36 36 33 33 33 36 36 36 33 33 33 33 33 36 36 36 36 36 36 36 36 36 33 33 33 36
33 33 33 36 36 31 31 31 31 31 31 31 31 31 31 31 31 31 31 31 31 31 31 31 31 28 22 20 19
>FLP3FBN01AWYZD length=244 xy=0258_1671 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 40 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38
38 38 38 37
>FLP3FBN01AM0P3 length=264 xy=0145_0217 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 39 38 38 39 39 38 38 38 33 33 33 36 39 39 39 37 37 37 36 36 29 29 29 33 33 37 37 37 37 37
35 35 31 31 25 25 25 21 21 21 36 37 37 36 36 36 36 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 33 33 33 37 37 37 36 36 36 33 33 33 33 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 33 33 33 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 28 28 28 28 28 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 36 37 37 37 37 36 36 36 36 37 37 37 37 37 37 36 36 36 36 36 36 30 26 22 22 22 22 22 33 36 36 36 36 36 31 31 32 36 36
36 36 36 36 36 30 30 30 31 31 31 31 31 30 30 30 31 31 25 22 16 16 16 16
>FLP3FBN01BO4IE length=240 xy=0579_0388 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 33 33 33 38 38 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 36 36 36 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 33 33 23 20 20 20 20 20 30 20 20 20 22 28 37 37 37 37 37 37 37
36 33 33 33 37 37 37 37 37 37 36 36 36 37 37 37 36 36 33 33 33 36 36 36 36 37 36 36 36 37 36 36 36 37 37 37 36 36 36 36 36 36 30 30 30 30 30 22 22 22 22 26 30 36 36 36 36 36 36 36
>FLP3FBN01BPX14 length=254 xy=0588_1818 region=1 run=R_2008_12_09_13_51_01_
32 22 22 22 22 27 31 34 34 34 32 32 32 34 34 34 34 34 40 40 40 40 40 40 40 40 39 40 39 39 36 28 23 23 22 33 36 36 36 36 36 36 36 36 37 34 34 34 34 34 34 33 32 32 32 32 27 27 14 14
14 14 14 14 29 29 32 32 32 32 33 34 34 34 34 34 34 34 34 34 34 34 34 34 29 34 30 31 31 31 34 33 30 30 24 24 25 24 18 18 18 15 15 15 18 18 14 17 17 17 14 14 14 14 16 12 12 16 17 22
25 25 29 33 34 34 34 32 32 32 34 34 34 34 34 34 32 32 32 34 31 31 30 32 32 32 25 22 16 16 16 16 16 15 16 16 16 22 29 14 14 14 18 24 32 34 34 34 34 34 34 34 34 30 30 30 32 34 34 34
34 34 29 29 29 29 32 25 25 18 18 18 29 32 29 34 29 29 32 29 17 17 17 17 29 24 21 14 14 14 14 14 17 29 24 24 24 24 24 24 24 24 24 14 14 14 14 14 14 14 19 19 14 14 14 14 14 14 14 19
19 19 19 24 26 22 22 19 20 20 20 20 16 16
>FLP3FBN01DB5I7 length=268 xy=1251_1905 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 39 39 39 39 39 37 36 30 30 30 38 38 38 36 36 33 39 39 39 39 37 37 37 37 37 31 31 31 37 37 37 37 37 37 37
38 38 38 37 37 38 34 30 28 28 38 37 37 37 37 37 37 34 34 33 33 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 33 33 33 37 37 38 38 38 38 28 28 28 28 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 31 31 31 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 22 22 15 15 15 15 15 22 23 23 28 17 17 17 17 32 37 37 34 21 21
21 21 38 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 33 36 33 33 32 32 28 28 15 15 15 15 15 28 32 35 33 35 35 26 26 26 31 36
36 36 36 36 36 25 25 25 31 31 31 31 31 31 31 32 31 31 31 32 28 28 28 30 30 32 31 31
>FLP3FBN01AK9OO length=249 xy=0125_0438 region=1 run=R_2008_12_09_13_51_01_
27 27 27 31 34 34 32 27 27 22 22 23 30 34 34 34 34 34 37 40 40 40 40 40 40 40 40 39 38 38 38 39 37 36 36 30 30 30 36 36 36 36 37 37 37 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34
27 27 26 27 27 27 31 34 34 34 34 32 32 27 21 21 22 34 34 30 30 30 32 32 32 34 32 32 32 31 31 30 30 30 30 30 31 31 19 19 18 19 24 26 30 31 31 34 34 34 34 34 34 34 34 34 34 34 34 34
34 34 34 34 34 34 34 34 34 34 33 32 26 22 24 24 18 18 18 18 14 14 27 27 32 30 32 32 32 30 31 31 25 25 25 34 34 25 25 25 25 25 25 25 25 34 34 34 32 34 34 32 32 32 32 32 34 34 34 34
34 32 32 32 34 30 29 29 29 34 32 32 24 24 24 24 29 29 29 17 17 17 24 29 29 29 29 32 34 34 34 34 34 34 24 24 18 28 34 34 34 34 34 34 34 34 23 23 18 18 16 16 19 12 24 24 24 24 24 14
14 14 14 19 14 12 12 12 15
>FLP3FBN01ANGF2 length=253 xy=0150_0112 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 38 38 28 28
28 28 28 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 36 36 36 29 28 28 28 36 36 36 36 36 36 28 28 29 36 38
38 38 38 38 36 32 32 32 32 30 28 28 28
>FLP3FBN01AF994 length=272 xy=0068_1402 region=1 run=R_2008_12_09_13_51_01_
36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36
36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 37 37 37 37 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36
36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 31
31 31 31 35 35 35 35 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 31 31 31 36 36 35 30 28 16 16 16 16 16 16 16 16 16 16 16 14 14 16 24 35 35 30 24 24 24
24 24 24 16 16 14 14 14 21 21 21 21 14 14 13 13 21 14 13 13 14 14 13 24 24 21 21 13 14 14 13 13
>FLP3FBN01AHXU8 length=285 xy=0087_0802 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 32 32 32 32 38 38 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 32 28 15 15 15 15 15 22 22 37 37 37 37 37 37 37 37
37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 34 34 37 37 37 37 37 37 37 36 36 36 36 36 36 31 24 22 22 22 22 33 36 36 36 36 36 26 27 28 37 38 38
38 37 38 38 38 32 32 31 32 32 31 31 31 32 32 31 30 30 30 30 32 32 32 32 32 32 32 31 31 31 31 31 32 31 31 31 31 31 31 28 28 28 28 28 28
>FLP3FBN01APRWO length=272 xy=0176_1782 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 38 38 38 39 39 39 39 39 39 39 34 34 36 30 30 30 39 39 39 39 39 39 39 37 37 37 37 37 37 37 37 37 37 37 38 38 33 33
33 33 33 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 38 37 37 37 37 37 37 37 38 38 38 38 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 36 37 38 38 37 37 38 38 38 38 38 38 38 38 38 38 38 38 36 36 36 38 38
38 38 29 29 29 23 28 30 31 32 31 31 31 31 32 32 32 32 32 32 31 31 31 32 32 31 28 25 19 18 18 16
>FLP3FBN01C645E length=273 xy=1194_1600 region=1 run=R_2008_12_09_13_51_01_
35 35 35 35 33 33 33 35 35 35 35 35 35 35 35 35 35 35 40 40 40 40 29 29 29 36 35 35 35 35 35 33 33 33 35 31 23 23 21 21 23 30 35 35 35 35 35 33 33 30 30 30 30 33 33 35 35 35 35 35
35 35 35 33 30 30 30 30 35 35 35 35 35 35 33 33 33 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 29 29 33 33 33
29 29 29 28 33 28 28 28 32 33 27 27 29 28 33 27 20 20 20 20 27 35 35 35 33 33 33 35 35 35 35 35 35 35 35 35 33 24 24 24 35 35 33 33 33 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35
35 29 29 29 35 33 33 33 33 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 33 33 28 23 23 35 35 35 35 35 35 24 24 24 29 31 31 20 20 20 20 20 33 34 34 34 34 34 34 34 34 34 34 26 26
20 20 20 20 34 28 28 28 28 28 28 28 21 21 21 21 21 27 27 29 29 29 29 29 29 29 29 29 29 27 25 18 18
>FLP3FBN01A2K0Q length=278 xy=0322_1368 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 35 35
35 35 35 35 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 34 34 37 37 37 37 38 37 36 38 38 38 38 38 38 38 38 38 38 38 38 38 37 36 33 33 33 38 38 38
38 38 38 38 38 32 32 32 32 32 32 32 32 32 32 32 32 32 32 25 25 25 32 32 32 32 32 32 32 32 31 31 31 32 32 31 31 31
>FLP3FBN01DJY30 length=251 xy=1340_2286 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 26 26 26 26 26 26 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 31 31 31 38 37 37 38 38 38 38 36 36 36 31 24 24 24 24 22 26 36 36 36 36 36 36 36 36 36 36
36 36 36 36 36 31 30 22 18 18 18
>FLP3FBN01EB63O length=273 xy=1662_0098 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 36 36 36 36 36 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 38 38 38 40 40 38 36 37 36 36 36 36
28 28 28 28 28 22 25 25 31 31 32 31 31 32 32 32 32 31 31 32 32 31 31 32 32 32 32 32 32 32 32 32 32
>FLP3FBN01DUWHJ length=279 xy=1465_0101 region=1 run=R_2008_12_09_13_51_01_
35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 38 38 38 38 38 37 37 39 38 36 35 35 35 36 29 29 29 36 36 37 36 36 38 38 38 39 40 35 35 35 35 35 35 35 35 35 35 35 35 35 32 32
32 32 32 35 35 35 32 32 32 32 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 32 26 26 26 32 32 35 35 35 35
35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 34 34 34 35 35 35 35 35 35 35 35 35 35 35 35 35 22 22 21 21 18 26 26 26 26 35 34 18 18 16 16 16 16 26 32 26 26
18 18 18 21 21 26 26 26 32 31 31 33 35 35 35 35 35 35 35 35 35 35 35 35 34 34 34 34 34 34 35 35 35 35 35 35 27 27 27 27 27 31 35 35 35 35 35 35 35 35 35 35 35 35 31 23 22 22 26 26
31 35 30 30 30 25 28 22 22 22 22 25 30 30 30 30 28 25 25 25 25 28 28 28 30 30 28 28 22 22 16 16 16 25 25 22 22 22 21
>FLP3FBN01ESJ5R length=277 xy=1848_1661 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 40 29 29 29 33 36 39 39 39 39 40 38 38 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 32 32 15 15
15 15 15 33 33 35 35 32 32 34 37 37 37 37 37 36 36 29 29 29 29 36 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 31 24 24 21 21 21 17 32 28 28 31 31 36 35
36 36 36 37 37 37 37 37 37 37 37 37 37 37 36 33 26 26 26 26 33 36 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 36 36 36 36 36 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
36 36 36 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 33 33 33 37 37 37 37 37 36 36 36 36 33 22 24 22 22 22 22 33 36 36 36 36 36 36 36 36 36 36 33 33
28 28 28 28 36 31 31 31 31 31 31 31 31 31 31 31 31 22 18 18 18 22 25 28 29 28 16 16 16 16 28 25 22 22 22 29 29
>FLP3FBN01BJH6S length=269 xy=0515_0322 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 40 40 40 40 40 40 40 40 40 40 40 40 40 40 38 38 38 38 38 38
38 38 38 38 38 31 31 31 31 32 32 32 32 32 32 31 32 32 31 31 28 23 22 22 31 30 30 30 30
>FLP3FBN01BF988 length=259 xy=0478_1626 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 31 31 31 31 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 38 36 36 36 36 36 36 37 36 36 36 36 36 38 36 36 37 33 28 28 29 33 33 37 38
36 36 36 36 36 31 32 32 32 32 32 32 32 32 32 32 31 31 31
>FLP3FBN01DRASA length=264 xy=1423_4040 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 39 39 39 39 37 38 31 30 30 29 34 34 38 38 37 37 37 36 36
36 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 31 31 31 31 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 38 38 38 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 31 31 31 31 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 33 33 33 33 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 38 38 37 37 37 36 36 36 36 36 36 36 36 32 32 33 33 33 36 36 36 36 36 36 38 36 36 36 36
28 28 28 28 28 22 28 28 31 31 31 31 31 31 31 31 30 30 28 28 22 18 18 16
>FLP3FBN01EMWGJ length=264 xy=1783_3905 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 30 30 31 31 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 34 34 33 38 38 38 28 28 28 28 38 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 34 26 26 26 26 24 24 38 37 37 37 30 30 30 31 36 36 36 36 36 36 36 33 33 33 33 36 36 36 36 36 36 36 37 37
37 38 37 38 38 30 30 30 30 30 31 31 31 31 31 32 31 30 30 30 30 25 25 25
>FLP3FBN01BMERW length=277 xy=0548_0698 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 36 36 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38
38 38 38 38 38 32 31 31 31 31 31 32 32 32 32 32 32 32 32 32 32 32 32 31 31 31 31 31 31 31 30 30 30 30 30 25 25
>FLP3FBN01A724Q length=271 xy=0385_0072 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 38 38 38 40 40 40 40 38 38 38 38 38 40 40 38 38 38 40 38 38 38 38 38
38 38 38 38 38 23 23 23 28 31 32 32 32 32 32 32 32 32 32 32 31 31 32 32 31 26 26 25 25 28 28
>FLP3FBN01AKXJC length=255 xy=0121_1078 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 37 37 36 36 37 36 33 33 33 33 33 37 37 36 38 36 38 38 37 38 38 29
29 29 33 33 38 31 31 31 32 31 32 31 28 28 22
>FLP3FBN01EC04T length=269 xy=1671_2155 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 38 38 38 38 38 40 40 40 40 38 38 38 38 38 38 38 38 36 33 33 33 33 37 37
37 36 36 36 37 31 31 32 31 31 31 30 30 30 32 31 31 31 31 25 25 25 31 31 31 31 31 31 31
>FLP3FBN01AX7NV length=264 xy=0272_2233 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40
40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 40 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 33 33 33 37 37 37 38 38 38 38 38 38 40 40 38 38 38 38 38 38 38 38 33 33 36 37 36 36 36 37
36 36 36 38 38 31 31 31 31 32 32 32 32 32 32 32 31 31 31 31 31 28 28 28
>FLP3FBN01CS9UE length=244 xy=1036_1668 region=1 run=R_2008_12_09_13_51_01_
35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35
35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 33 33 33 33 35 35 35 35 35 35
35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35
33 24 24 24 24 29 20 29 29 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 34 34 34 35 35 35 35 35 35 35 35 35 34 34 34 34 32 32 30
30 30 15 15
>FLP3FBN01A38B0 length=262 xy=0341_0414 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 38 38 38 38 38 38 36 33
33 33 33 36 36 31 30 30 30 31 31 32 31 31 31 31 32 32 32 32 32 31
>FLP3FBN01CB1XF length=264 xy=0840_1073 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 33 33 33 33 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 33 33 33 33 37 37 37 37 37 37 33 33 33 38 38 37 36 36 36 36 37 37 38 40 38 38 38 38 40 38 38 38 40 40 40 40 38 38 38 38
40 40 40 40 40 31 30 30 30 32 32 31 31 31 32 32 31 31 31 31 31 28 28 28
>FLP3FBN01ARU00 length=258 xy=0200_0834 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 40 38 38 38 38 38 38 38 38 38 38 36 36 36 36 36 36 36 36
36 36 31 31 22 19 19 19 22 26 31 31 31 31 31 31 31 28
>FLP3FBN01A24G3 length=258 xy=0328_2005 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 34 34 38 38 37 36 38 36 38 38 38 36 36 36 27 27 25 25 36 36 36 36 36 36 28 28 28 36 36
37 38 38 38 36 31 31 31 31 31 32 32 31 31 30 30 28 28
>FLP3FBN01AYGTR length=247 xy=0275_1821 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 38 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 40 40 40 40 40 40 40 38 38 38 38 38 38 38 40 40 40 40 40 38 38 38 38
40 40 40 40 38 31 25
>FLP3FBN01DPZMM length=253 xy=1409_0268 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 34 34 34 38 38 37 36 38 38 36 36 29 29 29 38 38 38 38 38 38 38 40 40 40 40 38 38 38 40 40
40 38 36 36 37 31 31 31 31 28 21 20 19
>FLP3FBN01AJ7M7 length=271 xy=0113_0289 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 40 38 38 38 38 38 40
40 40 40 40 38 32 32 32 32 32 32 32 31 31 31 32 31 31 25 25 21 20 20 25 31 31 31 30 28 28 28
>FLP3FBN01DV3TK length=264 xy=1478_3014 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 38 38 36 36 37 37 37 37 37 37 37 37 40 40 40 40 40 40 38 39 39 39 39 39 36 26 24 24 24 23 23 23 33 33 33 33 36 37 39 37 37 37 37 37 33 33 33 36 36 37 37 37 37 37
37 37 37 37 36 36 36 28 26 26 33 36 36 37 37 37 37 36 36 36 36 37 37 37 37 37 37 37 37 37 37 37 37 37 36 29 29 29 36 36 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 36 36 37 37 36 36 36 36 37 37 37 37
33 33 33 33 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 33 33 33 37 37 37 36 36 36 36 36 36 36 33 26 26 26 26 25 36 36 36 36 36 36 36 33 33 33 37
37 37 36 36 36 31 31 31 31 31 31 31 31 31 30 30 30 31 29 25 16 16 13 15
>FLP3FBN01AL42P length=260 xy=0135_0159 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 33 33 34 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 31 31 31 31 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 33 33 36 36 32 32 31 28 28 15 15 15 15 15 17 17 30 31 35 35 33 36 36 36
36 36 37 37 37 30 30 30 31 31 32 32 31 31 31 31 31 31 31 31
>FLP3FBN01BYPDD length=264 xy=0688_0863 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40
40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 32 32 32 32 38 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 38 38 38 40 40 40 40 38 38 38 38 38 38 40 40 38 38 38 40
38 38 38 38 38 31 32 31 31 32 32 32 32 32 32 32 31 31 31 32 32 28 28 28
>FLP3FBN01CTMCP length=264 xy=1040_1495 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 38 38 38 38 38 38 38 29 29 29 29 29 36 38 36 38 36 36 36 37 38 38 38
40 40 40 38 38 30 30 30 32 32 32 31 31 31 31 31 31 31 25 19 18 18 18 16
>FLP3FBN01BAJ58 length=269 xy=0413_0782 region=1 run=R_2008_12_09_13_51_01_
35 35 35 35 35 35 35 35 35 34 32 32 32 35 35 35 35 35 39 37 37 39 35 35 35 35 35 33 32 32 35 35 34 34 35 25 25 25 35 35 35 35 35 35 39 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35
32 26 26 26 32 32 32 34 35 35 35 35 35 35 35 32 32 32 32 34 34 35 35 35 35 35 35 35 35 35 34 34 34 35 35 35 35 35 34 34 34 35 35 35 32 32 32 32 32 34 35 35 35 35 35 35 35 35 35 35
35 35 35 35 35 35 35 35 35 32 32 32 32 32 35 35 35 35 35 35 35 35 35 35 35 35 35 35 34 30 30 30 35 23 21 21 21 21 30 33 21 24 24 35 35 35 35 35 35 35 35 35 35 35 34 34 34 35 35 35
35 35 30 30 30 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 34 29 29 29 29 34 35 35 35 35 35 35 27 27 27 27 35 35 35 35 31 30 30 26 27 19 31 33 35 35 35 35 35 35
33 31 32 32 32 30 30 30 30 30 30 30 30 30 30 25 22 22 22 22 30 28 28 28 30 30 28 28 28
>FLP3FBN01A4K2C length=277 xy=0345_0530 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 33 33
33 33 33 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 38 38 37 36 36 36 36 36 36 38 38 38 38 38 38 40 40 40 40 40 40 40 40 40 40 38 38
38 38 40 40 38 32 32 32 32 32 32 32 32 31 31 31 31 31 31 31 31 23 23 23 25 31 32 31 31 31 31 31 31 25 22 20 20
>FLP3FBN01DRBCR length=278 xy=1424_0681 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 38 38 38 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 35 35
40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 38 31 31 31 31 31 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38
33 33 33 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 34 34 37 37 34 28 28 26 26 26 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 34 31 31 31 38 37 37 37 37 37 34 34 29 38 38 40 40 38 38 38 38 38 40 40 38 38 38 38 38 40 40 40 40 40 38 38
38 38 38 38 38 32 32 32 32 32 32 32 32 32 32 32 32 32 32 30 28 28 28 28 30 30 30 31 31 25 25 25 28 31 31 31 22 18
>FLP3FBN01BWH0G length=259 xy=0663_0414 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 34 34 34 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 36 36 36 36 36 37 37 37 37 38 38 38 38 38 38 40 40 40 38 38
38 38 38 38 38 30 30 25 25 25 25 28 31 31 31 32 32 31 31
>FLP3FBN01D0JVY length=274 xy=1529_1564 region=1 run=R_2008_12_09_13_51_01_
29 30 29 35 35 33 33 30 29 29 35 35 35 35 35 35 35 35 40 40 40 40 40 40 40 40 40 39 39 39 39 40 40 40 40 36 36 36 40 40 40 40 40 40 40 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35
35 35 33 33 38 35 35 35 35 35 35 35 35 35 35 33 33 33 35 35 35 35 35 35 35 35 35 35 35 35 35 35 33 33 33 33 33 33 35 35 35 20 20 20 21 21 20 28 27 36 33 35 28 28 20 20 20 23 22 29
33 35 35 35 31 31 31 32 32 32 31 27 19 19 16 16 16 16 16 16 27 27 27 35 35 35 35 35 35 35 35 35 35 35 35 27 20 20 20 20 20 20 19 19 25 33 32 32 32 35 35 35 35 35 33 33 33 35 35 35
35 35 33 33 33 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 33 33 33 35 35 35 35 35 35 35 32 32 28 26 26 26 26 22 14 14 14 14 14 15 22 22 28 28 30 30 28 31 28 24
20 20 20 34 29 25 18 16 16 25 22 28 27 25 25 25 26 26 27 28 25 18 18 18 27 26 18 18 21 21 25 27 25 21
>FLP3FBN01EDR9T length=264 xy=1680_0463 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 38 38 38 38 38 38 38 38 38 38 38 36 37 36 36 36 38
36 37 37 37 36 30 30 30 30 30 30 30 31 31 31 32 32 31 31 32 31 22 22 22
>FLP3FBN01CDTMI length=264 xy=0860_1704 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 34 34 33 38 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 38 38 38 38 38 40 38 38 38 38 38 38 38 38 38 38 38 38 38
38 38 38 38 40 32 32 32 32 32 32 32 32 32 32 32 32 31 31 31 31 25 25 25
>FLP3FBN01AFERM length=271 xy=0058_1520 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 40 40 40 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 38 38 38 38 38 38 38 38 38 38 37 37 38 38 38 38 38 38 38 38 38 38 38
38 38 38 38 38 32 32 32 32 32 31 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 31 31 31 31 31
>FLP3FBN01BFQFB length=263 xy=0472_0501 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40
40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 36 36 38 38 38 38 38 38 38
38 38 38 38 38 32 31 31 31 32 32 32 32 32 32 32 32 32 32 32 23 23 23
>FLP3FBN01D3JOZ length=270 xy=1563_2017 region=1 run=R_2008_12_09_13_51_01_
26 26 26 31 35 35 35 35 35 35 35 35 35 35 35 35 35 35 40 40 40 40 40 40 40 40 39 39 37 37 37 37 39 39 39 28 28 26 36 38 37 37 37 37 38 35 35 35 35 33 31 31 29 29 29 37 34 29 14 14
14 14 14 20 20 20 20 32 32 35 33 33 33 33 32 33 20 20 20 20 20 33 35 33 33 33 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 31 26 26 26 26 31 33 35 35 35 35 35 35 35 35 33 33
33 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 33 33 33 33 35 35 35 35 35 35 35 35 35 35 35 35 35 33 33 33 35 35 35 35 35 35
35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 34 34 34 34 33 30 20 20 19 19 19 31 32 30 28 26 26 15 15 19 25 19 20 20
20 20 34 34 34 30 29 30 30 31 31 30 30 30 29 29 29 30 29 29 29 30 30 25 18 18 18 18 18 18
>FLP3FBN01DVIVE length=272 xy=1472_0440 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 26 26 26
26 34 38 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 38 38 38 38 38 37 26 26 29 38 38 33 27 28 28 33 36 36 36 36 36 36 36 38 36 36 31 31 31 27 27 24 22 22 31 30
31 31 33 33 33 28 31 31 31 31 31 30 30 30 31 31 31 31 31 32 31 31 31 32 32 31 25 22 21 21 21 20
>FLP3FBN01EQVOU length=254 xy=1829_1116 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40
40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 38 38 38 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 38 38 38 38 40 38 38 38 38 40 38 38 38 38 33 33 33
36 38 38 37 33 28 28 30 31 31 31 31 31 31
>FLP3FBN01DH4LG length=277 xy=1319_2098 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 39 39 39 39 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 32 32
32 32 32 32 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 40 40 40 40 40 40 40 38 38 38 38 38 40 40 38 38 36 37 36
37 37 37 36 36 30 28 28 28 28 31 31 31 31 31 31 31 32 32 32 32 32 32 32 32 32 32 28 28 28 28 28 31 31 31 31 31
>FLP3FBN01EEU42 length=268 xy=1692_1684 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 32 32 32 32 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 36 36 36 36 36 33 25 25 25 25 25 31 36 38 36 38 36 36 36 36 36 36
36 36 36 33 33 21 21 21 25 26 32 31 31 31 31 32 31 32 32 32 32 32 32 31 31 26 26 25
>FLP3FBN01AE4G6 length=264 xy=0055_0472 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 36 36 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38
38 38 38 38 38 31 31 31 31 31 32 32 32 32 32 32 32 31 31 31 32 23 23 23
>FLP3FBN01A559N length=253 xy=0363_0937 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 33 33
33 33 33 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 34 34 34 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 40 40 40 38 38 38 38 38 38 37 36 36 36 36 36 33 29 27 27 25 33 36
36 36 38 36 36 31 30 30 30 28 28 28 28
>FLP3FBN01BYAA8 length=268 xy=0683_1826 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 40 38 38 38 38 40 38 38 38 38 38 38 38 40 40 38 38 38 40 40 38 38 38 40
38 38 38 38 40 32 32 32 32 32 32 32 32 32 32 32 32 31 31 31 32 32 32 32 30 30 30 30
>FLP3FBN01CSQSY length=244 xy=1030_1568 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 39 39 39 40 39 39 39 39 37 37 37 37 38 34 34 34 38 38 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 33 33 33 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 33 33 33 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 31 31 30 34 34 36 36 36 36 36 36 26 26 22 22 22 22 22 22 36 36 36 32 32 32 32 32 31 31
20 20 16 16
>FLP3FBN01DTXOJ length=269 xy=1454_0049 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 40 38 38 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 38 38 38 38 38 38 38 38 38 40 40 40 40 40 40 40 40 40 40 40 40 40 38 38
38 38 38 38 33 25 25 28 30 32 31 31 31 31 32 31 31 31 32 31 31 31 31 31 26 26 25 31 31
>FLP3FBN01EJD3A length=257 xy=1743_3972 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 38 38 38 38 38 38 38 38 38 40 40 38 38 38 40 40 40 38 38 38 38 38 38
38 40 40 40 40 32 32 32 32 32 32 32 32 32 32 32 32
>FLP3FBN01CIW8K length=240 xy=0918_2098 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 37 36 36 36 36 36 36 36 22 22 22 22 20 30 25 36 31 16 16 16 30 30
>FLP3FBN01BYVGX length=253 xy=0690_0575 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 38
38 37 38 37 36 22 22 22 25 26 25 25 30
>FLP3FBN01BKL5B length=254 xy=0527_2957 region=1 run=R_2008_12_09_13_51_01_
35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 35 35 35 35 35 35 35 35 35 35 35 35 35 26 26
26 26 26 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35
35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 32 32 30 30 30 30 34 35 35 35 35 35 34 34 34 35 35 35 35 35 35
35 35 35 35 35 35 35 35 35 34 29 29 29 29 35 35 35 34 34 34 35 35 35 29 29 29 33 33 20 21 21 29 29 29 21 21 19 19 25 35 31 30 30 30 35 30 30 30 30 26 26 22 14 22 30 35 35 35 35 31
30 25 25 25 30 25 25 25 25 25 22 22 22 28
>FLP3FBN01DX9GS length=286 xy=1503_1242 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 31 31
31 31 31 31 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 34 31 31 29 33 33 38 38 38 38 36 38 38 38 38 38 38 40 40 40 40 40 40 40 40 40 40
40 40 40 40 40 32 31 31 31 30 30 30 31 32 32 32 32 32 32 32 32 31 31 32 31 32 32 32 32 31 31 31 25 25 25 25 30 31 31 22 22 22 22 31 28 28
>FLP3FBN01BVHTK length=278 xy=0651_2662 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 37 37 36 28 28 28 34 37 39 37 37 37 37 38 38 38 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 32 32 31 35 37 37 37 37 23 23 22 22 22 22 24 31 36 36 36 37 37 37 37 36 36 36 36 37 37 37 37 36 36 36 36 37 32 32 32 35 35 37 37 37 37 37 37 37 37 37 37 35
32 32 32 32 37 37 37 37 37 37 37 37 37 37 37 36 32 26 26 22 22 35 36 35 35 23 23 23 33 32 35 35 35 37 37 37 37 37 37 36 36 36 36 37 37 37 37 37 37 37 37 36 36 36 37 37 37 35 35 35
35 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 35 28 28 28 36 37 32 32 28 36 36 36 36 36 36 36 36 27 27 28 31 25 25 25 24 31 33 33 36 36 36 36
36 36 36 36 36 30 30 30 31 31 31 31 30 30 30 31 31 31 22 19 19 18 22 25 31 28 22 22 22 25 25 25 22 22 22 22 25 25
>FLP3FBN01A9PL2 length=276 xy=0403_2136 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 38 36 33 25 27 27 27 27 32 32 36 36 36 38 38 38 38 38 40 40 40 40 40 40
38 38 38 40 40 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 31 31 31 32 32 31 31 32 32 31 30 30 22 22 22 22
>FLP3FBN01C3PFP length=248 xy=1155_1011 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 36 36 15 15
15 15 15 15 28 28 35 35 35 35 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 38 38 38 37 37 37 37 34 34 21 21 21 21 34 38 38 38 38 38
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 33 33 33 37 37 31 31 31 33 36 36 30 26 26 26 26 26 31 26 26 23 26 26 36 31 31 31 31 36
36 36 36 38 36 30 30 30
>FLP3FBN01A17GS length=254 xy=0318_0186 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
34 31 31 31 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 38 40 40 40 40 40 40 40 38 38 38 38 38 38 38 38 38 38 37
37 37 38 38 36 31 31 31 31 31 26 25 22 22
>FLP3FBN01DM0XL length=256 xy=1375_1255 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 36 31 31 31 31 31 36 36 36 36 36 36 36 36 36 36 36 36 37 36 36 36
36 36 36 33 33 28 22 22 22 28 31 31 30 30 30 31
>FLP3FBN01DFQEI length=276 xy=1292_0984 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 33 33 33 33 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 33 33 33 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 36 36 36 36 37 36 26 26 21 21 35 37 35 35 35 35 36 36 36 36 36 36 36 36 36 37 37 37 31 31 31 22 22 19 19 19 19 26 26 26 30 36 36 36 36 30 26 23 26 27 36 33 27 27 26 30 33
33 33 36 33 31 25 25 25 25 25 25 25 25 31 31 31 28 25 25 25 28 28 31 31 31 31 29 26 13 13 13 13 22 13 13 15
>FLP3FBN01DN5MU length=268 xy=1388_0756 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 38 38 38 38 33 33 37 36 36 36 36 36 33 31 31 32 33 33 36
32 32 32 31 33 30 32 31 32 31 28 28 25 25 25 32 32 31 31 25 19 19 19 25 25 31 30 30
>FLP3FBN01B1L15 length=247 xy=0721_1371 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 34 34 34 34 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 36 36 37 37 36 36 36 36 38 38 38 38 37 36 36 37 37 38 38 36 36 33
33 33 37 37 38 31 31
>FLP3FBN01AW01I length=271 xy=0259_0244 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 36 33 33 33 33 37 36 33 33 33 33 33 36 38 38 38 38 38 38 38 38 38 38
40 40 38 38 38 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 31 31 31 31 31 31 31
>FLP3FBN01BI38I length=276 xy=0510_2720 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 40 40 40 38 38 38 38 40 40 40 40 40 38 38 38 38 38 38 38 38 38
38 38 38 38 38 28 31 31 32 32 32 32 32 32 32 32 32 32 32 32 32 31 31 31 32 32 32 31 31 31 31 31 31 31 25 25
>FLP3FBN01DSC1C length=249 xy=1436_0366 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 38 38 38 38 38 38 38 38 38 40 40 40 40 40 40 40 40 40 40 40 40 40
38 38 38 38 38 28 28 30 31
>FLP3FBN01AIELL length=264 xy=0092_2007 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 38 40 40 40 40 40 38 40 40 40 40 40 38 38 38 38 36 36 37
36 36 36 36 36 30 31 31 31 31 32 31 31 31 31 32 32 31 31 31 20 19 19 18
>FLP3FBN01D1T0J length=264 xy=1543_4001 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 38 38 38 38 38 38 38 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 34 34 38 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 33 33 28 28 28 38 37 37 36 36 36 36 38 36 33 31 32 32 36 36 36 36 36 36 36 36 36 36 36 36 36 36
22 22 22 20 26 22 30 22 22 22 22 28 25 25 25 33 31 31 25 22 16 16 16 16
>FLP3FBN01EOHM5 length=256 xy=1802_0191 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 34 34 34 37 37 37 37 37 37 36 36 36 36 36 36 36 37 38 38 38 38 38 38 38 38 38 40 40 40 40 38 38 38
38 40 40 40 38 31 31 32 32 32 32 32 32 32 32 32
>FLP3FBN01BMLLI length=276 xy=0550_1348 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38
38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 30 30 16 16 16 28 28 28 28 16 16 16 16 17 15 15 17 28 31 31 30 31 31 33
36 36 36 36 33 28 28 28 30 31 31 31 31 31 31 30 28 28 25 25 25 25 28 31 31 31 31 31 31 31 31 31 25 25 25 25
>FLP3FBN01D74QJ length=265 xy=1615_2921 region=1 run=R_2008_12_09_13_51_01_
32 32 34 34 35 35 31 23 22 22 23 23 35 35 35 35 35 35 35 35 35 35 26 26 26 32 35 35 35 35 35 35 35 35 36 38 38 38 38 38 38 40 40 40 40 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35
35 35 35 35 35 35 34 34 26 26 26 34 35 35 35 35 35 35 26 26 26 26 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 34 34 34 35 35 35 35 34 34 32 32 32 32 32 34 34 35 26 26 32 34 35
35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 34 25 25 25 25 34 35 35 35 35 35 35 35 26 26 26 26 35 35 35 35 35 35 35 35 35 30 30 32 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35
35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 29 29 29 35 35 35 35 35 35 35 35 35 35 34 27 27 27 27 35 35 35 35 35 35 35 35 33 27 27 27 27 33 35 35 35 35 35 35
35 35 33 33 33 28 30 28 28 28 28 30 29 22 13 14 14 13 21 24 21 13 13 13 13
>FLP3FBN01EAAV0 length=274 xy=1640_1806 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 31 31
31 31 31 31 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 34 34 37 34 34 33 33 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 38 38 40 40 40 40 38 38 38 40 40 38 38 38 38 40 40 38 38
38 40 38 38 38 31 32 32 32 32 32 32 32 32 32 31 30 28 28 25 25 25 32 32 32 32 32 32 32 32 31 32 25 25
>FLP3FBN01BC3GH length=271 xy=0442_0303 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 39 39 39 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 22 22 15 15 15 15 15 15 32 32 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 33 33 33 33 38 37 37 37 37 37 37 37
37 38 38 38 37 37 37 37 37 37 37 37 37 34 34 34 34 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 40 40 38 38 38 38 40 40 38 38 38 38 36 37 36 36 36 36 36 36 36 33
33 33 33 36 38 31 31 31 32 32 32 32 32 32 32 31 31 31 31 31 32 31 32 32 31 31 30 25 25 22 22
>FLP3FBN01D9A6M length=264 xy=1629_0588 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 33 33 33 38 38 38 36 36 36 36 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38
38 38 38 38 38 31 31 31 31 32 32 32 32 32 32 32 31 31 31 32 32 28 28 28
>FLP3FBN01A8V4Q length=262 xy=0394_0792 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 33 33 33 33 33 33 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 37 36 36 36 36 36 36 37 36 36 36 33 33 33 33 33 36 37 37 37 37 38 38
38 38 38 38 38 32 31 31 31 31 31 31 31 32 32 31 31 31 31 31 22 19
>FLP3FBN01DCU7E length=277 xy=1259_2408 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 31 31
31 31 31 31 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 36 36 37 36 36 29 29 29 29 33 36 36 37 37 37 38 38 37 38 38 40 40
40 38 38 38 38 31 31 32 32 32 32 32 32 31 32 30 30 30 32 31 32 32 32 32 32 32 32 32 32 32 32 32 32 32 31 31 31
>FLP3FBN01AM688 length=264 xy=0147_0490 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 40 40 40 38 38 38 40 40
38 38 38 38 38 30 30 31 31 32 32 31 31 31 31 31 31 31 31 31 31 22 22 22
>FLP3FBN01DMRPA length=258 xy=1372_1580 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 38 38 38 38 38 38 36 29 29 29 29 29 36 36 36 36 36 36 36 36 36 38 38
38 38 38 38 38 32 32 31 31 31 32 31 30 30 30 32 28 25
>FLP3FBN01BAZV6 length=287 xy=0418_0676 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 30 30 30 28 28 37 37 37 37 37 37 37 36 36 36
36 36 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 38 38 36 36 36 29 29 29 29 28 36 36 36 36 38 38
36 36 36 38 38 30 31 32 32 32 32 32 32 32 32 31 31 31 31 31 31 31 32 32 32 32 32 32 32 32 32 32 32 32 32 32 31 31 31 32 32 32 32 32 32 31 30
>FLP3FBN01DEBKQ length=271 xy=1276_0648 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 31 31
31 31 31 31 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 36 36 36 36 36 36 36 38 38 38 38 38 38 38 37 38 38 38 38
38 38 38 38 38 28 30 31 32 32 32 32 32 32 32 32 31 31 31 31 31 31 31 31 31 28 25 19 19 18 18
>FLP3FBN01BAWOL length=279 xy=0417_0611 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 36 33 29 29 29 38 38 36 36 36 36 36 38 38 33 28 24 24 22 26 26 30 32 31
31 26 26 20 20 18 18 28 34 31 31 31 31 31 32 31 32 32 32 30 30 30 32 31 31 30 30 30 30 30 31 32 31 31 31 31 31 28 28
>FLP3FBN01D7AVV length=281 xy=1606_1097 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 38 38 38 40 40 40 40 38 38 38
38 40 40 40 38 31 31 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32
>FLP3FBN01B5T3T length=273 xy=0769_1815 region=1 run=R_2008_12_09_13_51_01_
34 34 34 34 34 34 33 33 27 27 27 33 34 34 34 34 34 34 37 37 38 37 28 28 26 34 34 32 31 31 27 18 21 21 18 19 28 27 31 31 30 31 33 34 34 32 27 27 27 31 34 33 33 32 32 31 27 27 27 14
14 14 14 14 21 18 18 30 30 29 29 30 30 32 29 27 27 18 18 19 19 26 26 24 20 26 32 32 32 32 34 34 34 32 26 26 26 34 34 34 34 30 30 29 29 32 20 26 19 17 14 14 14 16 19 26 27 27 27 27
16 17 19 26 26 26 26 26 31 26 26 26 32 32 32 32 32 18 18 18 31 31 31 31 32 32 34 31 30 30 31 31 34 34 34 34 32 32 32 34 30 30 25 25 25 25 25 25 29 32 29 14 14 14 29 27 27 18 18 18
18 24 29 21 29 29 32 32 32 34 34 34 34 34 34 34 32 32 32 34 24 24 24 30 30 34 30 30 30 30 34 34 34 34 22 20 16 16 16 29 30 29 24 24 14 14 14 14 19 14 14 14 14 14 14 14 14 19 24 26
26 24 24 24 27 25 25 20 16 16 16 25 25 25 25 25 25 25 25 16 16 16 19 19 17 19 17 12 12 12 12 11 12
>FLP3FBN01B02WO length=277 xy=0715_1126 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 38 38 31 31 31 38 33 36 36 36 38 36 38 37 38 38 38 38 38 38 38 38 38 38 37 36 37 36 36 36
36 37 38 38 37 31 31 31 31 31 31 32 31 31 31 31 32 32 32 32 32 32 32 31 31 31 31 31 31 31 25 25 25 25 31 25 25
>FLP3FBN01B2DRT length=266 xy=0730_0423 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 39 39 39 36 36 36 39 39 39 38 38 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 34 34
33 33 33 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 40 40 40 40 40 38 38 38 38 40 38 38 38 38 33 36 38 37 36 36 36 36
37 37 37 37 37 30 30 30 31 32 31 32 32 31 31 25 25 25 28 31 31 31 25 18 18 18
>FLP3FBN01A8KH9 length=264 xy=0390_2111 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37 37 38 33 33 34 34 36 36 36 36 36 37 37 36 38 40 38 38 38 38 38 40 40 40 40 40 38 38 38 38
40 38 38 38 38 30 30 30 30 31 31 31 31 31 31 31 30 30 30 31 31 28 28 28
>FLP3FBN01AU6ML length=264 xy=0238_0187 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 40 40 40 40 38 38 38 38
40 40 40 40 40 32 32 32 32 32 32 31 31 31 32 32 32 31 31 32 32 32 32 32
>FLP3FBN01AJ4R9 length=282 xy=0112_0679 region=1 run=R_2008_12_09_13_51_01_
35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 40 40 40 40 40 40 40 40 40 40 40 38 38 39 40 38 38 29 29 29 29 32 32 38 38 39 37 35 35 35 35 35 35 35 33 33 39 34 32 14 14 14
14 14 14 22 22 22 22 32 32 35 35 35 33 39 29 29 14 14 14 14 14 23 32 29 29 29 34 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 25 25 24 25 23 34 31 31 31 35 35 35 35 32 32 32 35
34 34 34 35 35 35 35 35 35 35 35 35 35 35 35 35 35 34 34 34 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 30 30 30 35 35 35 35 35 35
35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 34 34 34 34 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 33 33 33 33 33 35 35 33 33 31 31 35 35 35 26 23 26 26
26 26 30 30 30 25 28 28 30 30 30 30 28 28 28 30 30 30 25 25 25 30 30 30 30 30 25 25 19 19 19 19 22 30 30 30 30 30 30 30 30 30
>FLP3FBN01B32LN length=255 xy=0749_1433 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 34 34 34 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 38 38 38 38 38 38 38 38 40 40 40 40 40 40 40 38 38 38 38
40 40 40 40 40 31 31 31 31 32 32 32 31 31 25
>FLP3FBN01COYKA length=262 xy=0987_1128 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 34 34 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 38 38 37 33 29 29 29 33 37 38 38 38 36 36 36 36 36 36 36 38 36 31 25
25 25 25 31 31 31 30 30 30 31 31 31 31 31 31 31 31 32 31 31 31 28
>FLP3FBN01C4BE9 length=264 xy=1162_0835 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 39 39 39 37 37 40 38 34 34 34 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 34 34 37 37 37 37 37 37 37 37 37 37 37 37 33 33 33 33 37 33 33 34 34 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38
40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 40 40 40 40 40 40 40 38 38 38 38 40 40 40 40 38 38 38 38
38 38 38 38 38 32 32 31 32 31 31 31 28 28 28 31 31 31 30 30 22 22 22 22
>FLP3FBN01BU1CU length=271 xy=0646_1804 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 40 40 40 40 38 38 36 36 36 38 36 33 33 29 29 27 27 33 36
36 36 38 38 36 31 31 32 32 32 32 32 32 32 32 32 31 31 31 32 28 28 28 25 22 20 20 19 22 31 31
>FLP3FBN01AYTJU length=277 xy=0279_1928 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 32 32
32 32 32 32 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 38 33 38 38 38 38 40 40 40 40 40 40 40 40 40 40 40 40 40 38 38 37 36 36 36
36 33 33 33 33 25 28 31 31 32 31 31 31 31 32 31 31 31 32 31 31 31 32 32 32 32 31 31 31 31 31 30 30 30 30 31 31
>FLP3FBN01DXJGO length=264 xy=1495_0310 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 33 33 33 33 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 37 37 37 36 36 36 36 36 36 36 36 36 36 36 36 36 33 38 38 38 38 36 38 36 36 38 38
40 40 38 38 38 30 31 31 32 32 32 32 32 32 32 32 32 31 31 31 32 31 31 31
>FLP3FBN01BOOHE length=286 xy=0574_0096 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 33 33 33 38 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 34 34 34 34 37 38 38 38 37 37 37 37 37 37 37 38 40 38 38 29 29 29 38 38 38 38 37 38 36 38 38 38 38 38 38 38 38 38 40
40 40 40 40 40 32 32 31 31 31 31 31 32 32 31 31 31 32 31 31 31 32 32 32 31 31 31 30 30 30 32 32 32 32 31 31 31 31 31 31 31 31 22 22 22 22
>FLP3FBN01BJL13 length=247 xy=0516_1241 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37
37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 38 38 38 38 40 40 38 38 38 38 40 40 40 40 38 38 38 38 40 40
40 38 38 38 38 23 23
>FLP3FBN01CTRWB length=258 xy=1042_0489 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 38 38 38 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 38 38 38 38 40 40 40 40 40 40 40 40 40 40 40 38 38 38 38 38 38 38 38
38 38 40 40 38 32 32 32 32 31 32 31 31 31 31 32 28 22
>FLP3FBN01BJCG1 length=259 xy=0513_1107 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 38 38 36 36 37 37 38 36 38 38 37 37 37 38 38 33 33 33 36 36 36 36 36 38
33 33 33 33 36 31 31 25 25 25 30 31 31 31 31 31 31 31 31
>FLP3FBN01CORWL length=268 xy=0985_0691 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 37 37 36 36 36 38 36 29 29 29 29 28 33 33 36 38 36 36 36 33 33 33 36
33 33 33 33 36 30 30 30 31 32 32 31 31 31 31 32 31 31 31 28 22 22 22 31 31 31 30 30
>FLP3FBN01C68BF length=274 xy=1195_1609 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 30 30
30 30 30 30 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 34 34 34 37 37 37 37 36 36 36 36 37 36 36 33 33 33 33 37 37 36 38 38 38 38 37 36 36 36 37 33
33 33 33 36 36 31 31 31 31 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 31 31 31 31
>FLP3FBN01B449W length=283 xy=0761_2402 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 32 32 32 38 38 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 36 36 36 37 36 29 29 29 29 29 33 33 36 38 36 36 36 36 36 36 38 33
33 33 33 29 29 22 22 31 31 32 31 31 31 31 32 32 32 32 32 32 32 32 31 31 30 30 30 31 32 32 32 32 32 32 32 32 32 32 32 32 32 31 32
>FLP3FBN01EZ4SI length=269 xy=1934_2736 region=1 run=R_2008_12_09_13_51_01_
32 35 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 38 38 37 37 37 37 37 37 33 33 33 37 36 36 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 36 36 36 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 35 35 35 37 36 32 28 28 28 36 37 37 37 37 37 36
36 36 37 37 37 37 37 37 37 37 37 37 37 37 37 37 35 35 35 37 37 37 37 37 37 37 37 37 37 37 36 36 36 36 36 36 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 26 26 24 29 30 37 30 24 15 15 15 15 15 15 16 16 30 30 30 27 36 36 36 36
36 37 38 37 37 31 31 30 30 30 31 31 31 31 31 31 31 31 31 30 28 22 22 22 22 31 26 26 25
>FLP3FBN01DD5NY length=276 xy=1274_1180 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 35 35
35 35 35 35 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 38 38 38 40 40 40 40 40 38 38 38 40 38 38 38 36 36 36 38
38 38 38 36 33 25 25 28 32 32 32 32 32 32 32 32 32 31 32 31 32 32 32 32 32 32 32 31 31 31 31 32 31 31 22 22
>FLP3FBN01BHKSP length=276 xy=0493_0503 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 34 34 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 33 38 38 38 38 38 38 38 38 38 40 38 38 38 38 38 40 40 38 38 38 38 37 37
37 37 37 37 36 31 28 28 28 30 32 32 31 32 32 32 32 31 31 31 32 32 32 32 32 32 31 30 30 30 31 28 22 22 22 30
>FLP3FBN01DHYS5 length=247 xy=1317_2791 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 38 31 31 31 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 33 33 33 33 38 33 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 37 33 33 33 38 36 36 37
38 38 38 38 38 32 31
>FLP3FBN01D837F length=277 xy=1626_3833 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 30 30
30 30 30 30 37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 34 34 34 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 33 33 33 33 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 30 30 30 38 38 37 37 36 36 36 36 36 36 33 29 29 29 29 36 36 36 37 37 36 37 37 38 38 38 38 38
38 38 38 38 38 31 31 31 31 31 32 31 31 31 32 32 32 32 32 31 31 31 32 32 31 31 31 30 30 28 28 28 31 31 31 31 31
>FLP3FBN01DC9HR length=253 xy=1264_0445 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 33 33
33 33 33 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 31 31 31 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 38 38 33 33 33 33 33 32 32 32 36 36 38 38 36 36 29 29 29 38 38
38 38 38 38 36 32 32 32 32 31 28 28 28
>FLP3FBN01DLDHU length=272 xy=1356_2048 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40
40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 31 31 31 31 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 32 32 34 32 31 22 15 15 15 15 15 15 28 28 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 38 38 26 26 26 26 26 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 36 36 36 36 36 33 33 24 24 24 24 24 36 36 36 36 33 35 35 35 32 32 31 16
16 16 16 20 20 16 25 25 28 28 33 31 31 31 31 32 31 31 31 32 25 25 25 31 32 31 22 22 18 18 18 18
>FLP3FBN01D6Q98 length=264 xy=1600_0270 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 31 31 31 31 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 37 37 37 37 38 38 38 38 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 36 36 33 33 33 33 33 33 37 37 37 36 38 38 38 40 40 40 40 40 40 40
40 40 38 38 38 31 32 32 32 31 31 31 31 31 32 32 32 31 31 32 32 32 32 32
>FLP3FBN01CNKON length=276 xy=0971_2021 region=1 run=R_2008_12_09_13_51_01_
34 34 34 34 34 34 33 27 27 27 27 31 34 34 34 34 34 34 40 40 40 40 40 40 39 40 39 38 38 36 36 34 22 22 22 35 34 32 21 22 22 22 31 34 36 34 34 34 34 34 34 34 34 34 32 33 32 32 14 14
14 14 14 14 18 18 18 18 27 27 29 24 27 27 27 21 21 14 14 14 14 24 25 24 25 30 31 30 31 34 34 34 34 34 34 34 34 34 31 24 26 26 24 26 26 24 25 25 25 27 27 22 22 16 15 15 16 16 15 15
22 25 17 19 19 31 32 32 34 34 34 34 34 34 34 34 34 34 34 34 34 32 20 20 20 25 34 32 32 18 16 16 29 27 22 18 14 14 15 14 14 18 28 28 28 25 25 25 32 33 33 32 32 29 30 25 25 25 30 30
30 34 34 34 34 34 34 34 34 34 34 34 34 30 24 18 18 18 24 30 34 34 34 34 34 34 34 34 34 34 34 34 34 32 34 34 34 34 34 27 27 24 14 14 14 14 14 14 19 14 14 14 14 14 14 14 14 14 14 24
14 14 14 16 22 16 16 16 20 26 26 26 26 26 26 26 26 26 26 26 26 25 25 20 20 20 20 20 17 12 12 12 12 12 15 19
>FLP3FBN01BCWX5 length=283 xy=0440_0059 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 33 33
33 33 33 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 28 28 28 28 28 36 36 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 33 33 33 37 37 37 37 32 20 20 20 22 20 20 20 20 20 20 20 22 31 16 16 16 16 26 26 30 30 26 15 15 15 15 19 30 30 30 30 30 30 31 36 30 27
27 27 36 36 36 31 31 31 31 31 30 30 30 30 31 31 31 31 25 22 19 19 19 19 22 31 31 30 29 29 29 31 32 32 29 29 16 16 13 13 15 22 24
>FLP3FBN01EA91G length=276 xy=1651_2306 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 34 34 34 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 38 38 38 38 37 37 37 38 38 38 38 38 38 38 38 38 38 38 38 38 40 38 38 36 36 36 37 38 38 37 37
29 29 29 29 38 28 28 28 31 31 31 31 31 32 32 32 31 31 31 32 32 32 32 31 30 30 30 30 31 31 32 31 22 22 22 25
>FLP3FBN01B06QS length=275 xy=0716_2002 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40
40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 40 36 40 40 40 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 34 34 34 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 38 38 38 38 38 38 38 38 38 38 36 36 36 36 38 36 36 29 28 28 33 33 33 33
33 33 33 33 33 31 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 30 28 23 23 22 25 30 31 31 30 25 25
>FLP3FBN01D0SFK length=247 xy=1532_0350 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 38 38 36 36 36 36 28 29 29 29 29 36 36 36 38 38 38 38 40 40 40 40
40 40 40 40 40 31 30
>FLP3FBN01DVTEG length=270 xy=1475_1798 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 36 36 36 36 31 27 27 26 26 26 33 33 36 36 33 33 24 24 25 33 36
36 37 38 37 36 31 31 31 32 32 32 31 31 31 31 32 31 31 31 25 22 22 22 30 31 26 26 25 28 30
>FLP3FBN01CECHO length=255 xy=0866_1578 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 38 38 38 38 38 38 38 40 40 39 39 39 40 40 40 40 40 40 40 40 40 40 40 40 38 40 37 37 37 37 36 32 32 32 35 35 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 32 28 28 28 35 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 36 36 36 36 37 37 35 35 35 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 33 26 26 26 26 21 35 37 37 37 37 37 37 37 37 37 37 37 37 36 36 36 37 37 37 37 37 37 37 37 37 37 35 32 32 32 32 32 36 36 37 37 37 37 37
36 32 32 32 32 28 28 28 32 32 35 36 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 36 36 37 36 36 36 33 32 32 32 36 36 36 36 36 36 36 36 36 36 33 24 22 22 22 26 36
36 36 36 36 22 18 18 22 28 31 31 31 31 31 25
>FLP3FBN01ES2BT length=271 xy=1854_0631 region=1 run=R_2008_12_09_13_51_01_
36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 40 40 40 40 40 40 40 40 40 40 40 40 40 40 39 39 39 40 40 40 40 40 40 40 40 40 40 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36
36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 32 32 32 32 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 35 28 28 28 28 35 36 36 36 36 36
36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 35 35 35 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36
36 36 36 35 35 35 35 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 35 31 31 27 27 16 16 16 16 16 16 16 16 16 16 16 16 16 16 16 19 21 28 29 29 24 16 16 19
25 31 35 35 31 22 14 14 14 22 22 22 21 14 14 13 13 21 21 21 21 21 13 13 13 21 21 13 14 14 14
>FLP3FBN01BJO3F length=280 xy=0517_1081 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 38 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 33 32 31 30 36 36 38 36 33 30 22 20 20 20 20 26 32 36 38 36 31 22 24 25
28 36 38 38 38 31 31 31 31 31 31 32 32 31 31 31 32 32 32 31 31 31 32 32 32 32 32 32 32 32 31 31 28 28 28 31 31 31 31 32
>FLP3FBN01AMOUT length=263 xy=0141_1219 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40
40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 31 31 31 31 34 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 38 38 38 38 38 38 38 38 40 38 38 38 38 38 38 38 36 37 33
33 33 33 38 38 30 30 30 31 32 32 32 32 32 32 32 31 31 32 31 25 25 25
>FLP3FBN01CNT1Q length=244 xy=0974_1868 region=1 run=R_2008_12_09_13_51_01_
37 37 37 40 34 34 34 38 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 39 39 39 39 39 38 29 29 29 26 26 26 28 28 28 34 39 39 39 37 37 37 37 37 40 40 40 37 37 37 37 37 37 37
38 38 38 37 38 38 34 24 24 24 38 37 37 38 34 34 34 28 28 28 28 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 26 26 26 34 26 26 26 26 38 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 36 31 30 30 31 20 20 20 20 16 16 28 26 31 36 36 36 36 36 36 36
36 38 38 28
>FLP3FBN01DR739 length=242 xy=1434_2183 region=1 run=R_2008_12_09_13_51_01_
34 34 34 34 34 34 34 34 28 28 28 28 28 34 34 34 34 34 37 37 38 39 29 29 29 40 37 36 36 35 34 34 36 34 34 25 25 25 24 24 31 33 33 34 34 34 34 34 34 33 33 33 33 34 34 34 34 34 34 34
34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 32 33 33 32 34 34 34 34 34 34 34 34 34 33 33 28 28 28 33 28 28 23 23 22 22 34 34 34 34 34 34 34 34 34 34 34 34
34 34 34 34 34 28 28 28 32 34 34 33 33 33 33 34 33 33 19 21 21 27 31 33 19 19 19 31 31 33 34 33 27 27 27 27 31 34 35 33 33 31 31 31 31 31 27 19 19 19 27 31 31 33 33 31 31 31 27 27
27 35 35 31 31 27 27 27 27 20 30 23 23 23 29 23 16 19 19 19 19 19 19 25 33 32 32 33 33 33 33 33 20 20 20 20 19 34 34 34 34 34 34 34 34 24 24 24 24 24 32 29 29 31 16 27 27 31 31 31
17 17
>FLP3FBN01BBDL9 length=258 xy=0422_2079 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 40 40 40 40 40 40 38 38 38 38 38 38 38 38 38 38 38 38 38
40 40 40 40 38 30 30 30 31 31 32 31 31 28 28 28 22 16
>FLP3FBN01CLP3T length=258 xy=0950_1751 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 38 38 38 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 34 30 30 30 38 38 38 38 38 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 32 32 32 38 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 36 36 36 36 37 37 37 29 29 29 33 37 37 36 37 37 36 38 38 38 38 40
38 38 38 38 38 32 31 31 31 30 28 28 28 31 22 22 18 18
>FLP3FBN01C1O95 length=268 xy=1132_1707 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 33 33 33 36 36 36 36 36 36 36 37 38 36 36 33 33 33 33 38 38
40 40 40 40 40 32 32 32 32 32 32 32 32 32 32 32 32 31 30 30 30 31 31 32 32 32 32 32
>FLP3FBN01DHLQN length=251 xy=1313_2237 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 38 38 38 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 34 34 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 38 38 38 36 37 37 37 37 37 37 38 36 36 33 33 33 38 38 38 38 38 38
38 38 38 37 37 31 32 31 31 30 25
>FLP3FBN01BA44D length=255 xy=0419_3355 region=1 run=R_2008_12_09_13_51_01_
35 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 36 33 33 26 26 26 33 36 37 37 37 37 37 37 37 37 38 38 38 38 38 38 38 38 38 40 37 37 37 37 37 32 32 35 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 35 32 32 32 37 37 35 35 28 27 27 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 35 35 35 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 32 32 28 19 19 17 17 17 17 21 16 16 16 28 30 32
32 32 36 36 37 36 36 32 32 32 36 37 37 37 37 37 37 37 37 37 37 33 33 33 37 37 37 37 37 37 35 35 35 37 37 32 28 28 28 36 36 36 36 36 36 36 28 28 28 36 36 36 36 36 36 36 36 36 36 36
36 36 36 30 26 22 22 25 28 28 28 13 13 13 15
>FLP3FBN01EHG6O length=245 xy=1722_0686 region=1 run=R_2008_12_09_13_51_01_
35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 35 35 35 35 35 35 35 35 35 35 35 35 35 34 35
35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35
35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35
35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 29 29 29 29 35 35 35 35 35 35 34 34 34 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35
35 30 23 15 15
>FLP3FBN01EE1C3 length=270 xy=1694_1557 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 40 40 37 38 38 38 38 38 40 40 40 40 40 40 40 40 40 40 38 38 38 38 38 29 29 29 38
37 37 37 37 36 31 31 32 32 31 31 30 30 30 32 32 32 31 31 31 31 31 31 32 31 31 30 30 30 25
>FLP3FBN01AER1M length=247 xy=0051_0744 region=1 run=R_2008_12_09_13_51_01_
35 35 35 35 35 33 34 34 35 35 35 35 35 35 35 35 35 35 35 33 31 31 23 23 20 23 31 35 35 35 35 33 33 33 33 31 31 31 34 34 33 35 35 35 35 35 35 35 35 35 31 31 31 31 31 31 31 35 35 35
35 35 35 35 35 35 35 35 35 35 35 35 35 35 34 34 34 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 31 31 26 26 26 31 35 35 35 35 35 35 35 35 35 35 35 35
35 35 35 35 35 35 35 35 35 35 35 31 31 31 21 20 20 20 20 20 20 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 23 23 23 23 23 21 21 28 35 35 35 35 35 35 35 35 35 35 35 30 26
26 26 35 35 35 35 35 35 35 35 33 33 33 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 33 20 23 20 19 19 19 19 23 23 26 26 28 15 19 15 19 25 18 25 25 31 31 30 26 30 30
30 25 25 25 25 16 16
>FLP3FBN01A1RXA length=278 xy=0313_0524 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 38 38 38 38 38 37 37 37 37 38 38 38 38 38 38 38 40 40 40 40 40 40 40 40 40 40 40 38 38 38 38 37 37
37 33 38 38 38 31 31 31 31 31 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 31 31 31 32 32 32 32 32 32 32 30 30
>FLP3FBN01B5Z5E length=254 xy=0771_1456 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 38 38 38 38 40 40 38 38 38 39 39 39 39 37 37 37 37 37 33 33 33 37 37 37 37 37 37 37
37 37 37 37 37 37 36 29 29 29 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 36 36 36 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 28 28 28 36 36 37 37 37 37 37 33 33 33 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 28 26 19 19 30 36 36 36 32 30 26 19 19 22 33 33 36 36 36 36 33 33 33 36 36 36
36 36 36 28 28 22 25 28 28 28 28 25 25 25
>FLP3FBN01EB7G2 length=242 xy=1662_0580 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 36 36 37 37 38 38 38 33 33 33 38 38 38 38 38 38 38 38
38 38
>FLP3FBN01AYGA0 length=280 xy=0275_1146 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 38 38 38 38 37 37 36 38 38 36 38 38 38 38 38 38 38 36 38 36 36 36 38
38 38 38 38 38 31 31 31 31 32 32 32 32 32 32 32 32 32 32 31 31 28 28 28 31 32 32 32 32 31 31 31 31 32 32 31 31 31 32 32
>FLP3FBN01AL8JX length=268 xy=0136_0571 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 33 34 33 34 34 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38
38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 37 36 36 38 38 38 38 33 33 33 33 33 36 37 36 37 36 36 36 31 32 32 36
36 38 38 36 32 21 21 21 28 28 32 31 31 31 31 31 31 31 31 32 32 32 32 32 31 31 31 31
>FLP3FBN01B7E8J length=276 xy=0787_2129 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 38 38 38 38 40 40 38 38 36 36 36 36
28 28 28 27 27 19 19 19 28 28 31 31 31 31 31 31 32 32 32 32 32 32 32 32 32 32 32 32 31 31 32 32 31 31 31 31
>FLP3FBN01D6BHF length=251 xy=1595_0273 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40
40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 38 38 38 37 37 38 38 38 38 38 34 34 34 34 38 38 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 33 36 36 36 36 36 36 36 36 36 36 32 28 28 28 32 36 36 36 36 36 36 36 33
33 33 36 36 33 28 28 30 22 18 18
>FLP3FBN01BLTKB length=271 xy=0541_1881 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 34 34 34 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 24 20 20 20 20 20 20 38 38 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 34 34 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 37 36 29 29 29 29 29 36 36 36 36 36 36 36 37 37 38 40 38
38 38 38 38 38 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 23 23 23 23 32 32
>FLP3FBN01EK2JX length=285 xy=1763_0411 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 32 32 32 32 38 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 30 30 30 30 30 38 38 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 36 33 29 29 29 29 29 36 36 36 36 36 38 36 36 38 38 38 40
40 40 38 38 38 32 32 32 32 31 31 31 31 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 31 31 31 31 31 30 28 25 25 25 28
>FLP3FBN01EF15V length=260 xy=1706_0097 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 34 34 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 33 32 36 32 32 32 28 17 17 15 15 15 15 15 28 18 28 28 35 35 33 36 36 36
36 36 36 28 28 22 25 28 30 31 31 31 31 30 30 30 28 25 25 25
>FLP3FBN01BB8KH length=280 xy=0432_1231 region=1 run=R_2008_12_09_13_51_01_
35 35 35 35 35 35 35 32 29 29 29 33 35 35 35 35 35 35 38 40 40 40 40 40 39 40 39 40 37 37 37 36 27 27 27 36 36 35 34 34 29 36 35 35 40 35 35 35 35 35 35 35 35 35 33 33 32 31 14 14
14 14 14 20 20 20 20 20 20 31 33 33 32 32 29 29 14 14 14 14 14 29 32 33 33 31 31 33 33 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 37 31 19 15 15 15 15 19 13 13 19 19 25 27
19 19 19 28 28 35 35 35 35 35 35 35 35 35 35 35 35 24 24 24 35 35 35 35 35 35 35 35 35 35 33 33 33 35 35 35 24 23 23 23 20 27 35 35 35 27 27 27 20 21 21 24 24 29 35 35 35 35 35 35
35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 33 33 33 33 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 34 34 34 34 34 26 26 26 26 26 24 24 28 21 24 24
32 31 24 24 24 16 16 16 27 28 25 21 21 21 29 29 29 29 29 29 29 29 29 29 29 29 29 29 27 27 27 28 28 27 21 16 16 16 16 16
>FLP3FBN01AMFGP length=268 xy=0138_1335 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 38 38 38 38 38 38 38 38
40 40 40 40 40 32 32 32 31 32 32 32 32 32 32 32 32 32 31 32 31 31 31 32 31 31 31 31
>FLP3FBN01A6LEJ length=255 xy=0368_0073 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 40 40 40 40 40 40 38 38 38 38 40 40 40 40 40 38 38 38 40
40 40 40 40 40 32 31 31 31 31 32 32 32 32 32
>FLP3FBN01BUCNS length=274 xy=0638_2566 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40
40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 38 38 38 38 38 38 38 38 38 38 38 36 37 36 36 37 36 33 33 33 36 36 36
36 36 36 36 37 31 31 32 32 32 31 31 31 32 32 32 31 31 32 32 32 32 32 31 30 30 30 30 31 31 31 31 22 22
>FLP3FBN01ET0T4 length=277 xy=1865_0298 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 32 32
32 32 32 32 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 38 38 38 38 40 40 40 40 40 40 40 40 40 38 38 38 38 40 38 38 38 38 38
40 38 38 38 38 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 31 31 31 32 32 32 32 32 32 32 32 31 31 31 30 30 30
>FLP3FBN01B0AX7 length=272 xy=0706_1757 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 31 31 29 38 38 40 40 38 38 38 38 40 38 38 38 38 38 38 38 38 38 38 38 38 38 38
38 38 38 38 40 32 32 32 31 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 31 31 31 30 25 21 21
>FLP3FBN01BM4QE length=261 xy=0556_1572 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37 38 38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 29 28 28 36 36 38 38 36 36 28 28 28 29 29 36 36 36 37 36 36 36 36 38
38 38 38 38 38 32 32 32 32 32 32 32 32 31 31 31 32 32 32 31 31
>FLP3FBN01EPSX1 length=258 xy=1817_0055 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 36 36 36 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 36 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 36 36 36 36 36 36 37 37 37 37 37 37 37 37 37 37 37 37 28 28 28 37 37 37 36 33 33 33 36 36 36 33 31 31 31 31 31 33 33 36 36 36 33 33 33 33 36 36
36 36 36 36 36 31 30 22 22 22 28 28 28 28 28 28 16 16
>FLP3FBN01DZ97Y length=280 xy=1526_1324 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 24 20 20 20 20 20 20 38 32 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 34 29 29 38 38 38 38 36 36 38 38 38 40 40 38 38 38 38 38 38 38 38 38 40 40
38 38 38 38 40 32 32 32 32 32 32 32 32 32 32 31 31 31 31 31 31 31 31 31 31 32 32 32 31 32 32 32 32 32 32 31 30 23 23 23
>FLP3FBN01BFDT5 length=277 xy=0468_0571 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40
40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 34 34 34 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 33 36 36 38 38 37 36 37 37 37 38 38 40 40 40 38 38 38 38 38 38 38 40 40
40 40 38 38 38 31 31 31 28 28 28 30 30 31 32 32 31 32 32 31 31 31 31 32 32 32 32 32 32 32 31 31 31 31 31 31 31
>FLP3FBN01BKTRL length=283 xy=0530_0543 region=1 run=R_2008_12_09_13_51_01_
36 36 36 36 36 36 36 32 32 32 36 36 36 36 36 36 36 36 38 38 38 38 32 32 32 36 36 36 36 36 36 36 36 36 38 34 34 34 38 38 40 40 40 40 40 36 36 36 36 36 36 36 36 36 36 36 35 35 26 26
26 26 26 32 32 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 35 35 35 36 36 36 36 36 36 35 35 35 36 36 36 36 36 32 28 28 28 28 28 36 36 36 36 36
36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 28 28 28 32 36 36 35 35 35 36 32 37 37 32 35 21 21 21 21 21 21 21 35 32 37 37 37 37 36 36 36 36 36 31 28 28 28
28 35 36 36 36 36 36 36 36 36 36 36 36 36 35 35 35 36 36 36 36 36 36 36 36 36 36 31 28 28 28 28 28 36 36 36 35 33 33 33 35 35 35 35 35 35 35 35 35 35 35 35 32 23 19 19 19 19 25 25
22 22 25 25 35 30 31 30 25 25 25 28 31 31 31 31 31 31 31 31 31 30 30 30 31 31 28 28 25 25 25 25 31 31 28 28 28 31 31 31 30 25 25
>FLP3FBN01C9NT9 length=258 xy=1223_0351 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 36 36 36 36 36 26 26 22 20 20 20 20 26 32 36 36 36 36 36 36 36 36
37 38 38 38 36 31 32 31 32 31 32 32 31 31 31 30 25 25
>FLP3FBN01BE65F length=270 xy=0466_0097 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 33 33 33 33 37 32 32 32 37 37 37 37 37 37 37 37 37 37 37 37 37 38 34 34 21 21 21 21 38 37 24 24 24 34 34 37 37 37 37 37 37 37
37 37 38 38 38 38 38 38 30 30 30 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 36 36 36 36 36 36 36 27 27 28 28 28 36 36 36 36 36 36 36 36 36 36 36 37
38 38 36 36 33 25 25 28 28 31 31 31 31 31 32 31 31 31 31 25 25 25 31 30 30 30 30 31 30 25
>FLP3FBN01EJ91Q length=252 xy=1754_0332 region=1 run=R_2008_12_09_13_51_01_
34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 34 34 34 34 34 34 34 34 34 34 34 34 34 25 25
30 30 30 30 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 31 23 22 22 28 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 33 33 34 34 34 34
34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34
34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 30 30 30 34 38 34 34 34 34 34 34 34 34 34 33 33 33 33 33 34 34 34 34 34 34 34 34 34 34 30 27 27
27 18 16 16 16 0 12 12 22 22 22 17
>FLP3FBN01AY2QI length=269 xy=0282_1544 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 40 40 40 40 40 40 40 40 38 38 38 40 38 38 38 40 38 38 38 40 40
40 40 40 40 40 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32
>FLP3FBN01BYYFZ length=272 xy=0691_0333 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 38 40 40 40 40 40 40 39 39 38 39 39 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 34 33 37 37 37 37 37 37 37 37 38 38 38 38 38 38 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 33 33 36 37 36 36 36 36 36 36 36 36 33 33 33 32 32 36 36 36 36
36 36 36 36 36 28 28 28 30 31 31 31 32 32 32 32 32 31 31 31 31 31 31 32 32 31 31 31 19 19 19 19
>FLP3FBN01AFUF1 length=283 xy=0063_1359 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 31 31
31 31 31 31 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 38 38 38 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 36 36 36 36 36 38 36 36 28 28 28 28 36 36 36 36 28 28 28 36 36 36 36 36
36 36 38 38 38 28 28 31 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 31 31 31 32 32 32 32 32 32 32 32 31 31 30 30 30 30
>FLP3FBN01B7I8E length=283 xy=0788_3212 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 38 30 28
28 28 28 28 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 38 40 38 38 38 38 38 38 38 36 36 36 36 36 36 33 33 33 33 28 25 25
25 28 28 36 36 30 30 31 32 32 31 31 31 32 32 32 32 31 28 23 23 23 31 32 31 32 31 31 31 31 31 30 30 30 31 31 31 28 28 19 19 19 25
>FLP3FBN01EIDVP length=244 xy=1732_2099 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 38 38 38 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 38 38 38 38 38 38 38 38 40 38 38 38 38 38 38 38 40 40 38
38 38 38 38
>FLP3FBN01DOR0M length=279 xy=1395_1092 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 33 33
33 33 33 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 33 33 33 37 37 37 33 33 33 37 37 37 37 37 37 37 34 34 34 38 38 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 38 38 38 38 38 40 40 38 38 38 38 40 40 38 38 38 38 38 38 40
40 38 38 38 38 32 32 31 32 32 32 32 32 32 32 32 31 31 31 32 32 32 32 32 32 32 32 32 32 32 32 32 31 31 31 31 31 30 22
>FLP3FBN01DO8MU length=274 xy=1400_2148 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40
40 40 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 37 38 36 36 36
38 38 38 38 38 31 31 31 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 31 31 31 32 31 31 31 31
>FLP3FBN01ATR31 length=274 xy=0222_0255 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 35 35
40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 38 38 40 40 40 40 40 40 40 40 40 38 38 38 38 38 38 38 38
38 38 38 38 38 31 31 31 32 32 31 31 31 31 32 31 31 28 28 23 21 21 30 31 31 32 31 31 31 30 28 28 25 25
>FLP3FBN01CO4O9 length=274 xy=0989_0891 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 32 32
32 32 32 32 38 38 37 37 37 37 37 37 37 37 37 37 37 34 33 33 33 33 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 36 37 36 36 36 36 36 36 36 37 36 36 37 37 36 37 37 38 38 38
38 38 38 38 38 31 32 32 32 32 32 31 32 32 32 32 32 32 32 32 32 32 32 32 32 32 31 31 31 31 31 32 32 32
>FLP3FBN01CZUXO length=273 xy=1111_1738 region=1 run=R_2008_12_09_13_51_01_
35 35 35 35 38 38 38 35 35 35 35 35 35 35 35 35 35 35 36 36 36 37 36 36 36 35 33 34 33 36 35 35 36 36 37 37 39 37 40 40 40 40 40 40 40 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35
35 31 31 33 33 33 35 35 35 33 33 31 32 32 32 32 20 20 16 16 16 16 23 23 30 35 35 35 35 35 35 35 35 35 35 35 35 35 34 33 33 35 35 35 35 32 31 31 19 19 15 15 15 15 19 14 28 31 32 32
31 31 33 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 33 33 33 35 35 35 35 33 33 31 31 27 17 17 17 19 27 17 17 17 17 17 17 25 31 33 35 35 35 35 35 35 35 35 35 35 35 35
35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 29 29 29 29 29 35 35 35 34 34 34 32 34 31 31 30 20 19 19 19 19 31 28 28 28 28 30 25 25 25 25 25
22 22 22 34 34 28 28 28 29 29 30 30 29 18 18 18 22 22 28 25 22 18 18 18 28 28 28 28 25 25 25 28 28
>FLP3FBN01A9EX0 length=276 xy=0400_0598 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 31 31
31 31 31 31 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40
40 40 40 38 38 31 31 31 31 32 31 31 30 30 30 32 32 31 31 31 31 31 32 32 32 32 32 32 32 32 32 32 32 25 23 23
>FLP3FBN01BYSJT length=264 xy=0689_0887 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 36 37 38 38 38 38
38 38 38 38 38 30 31 31 31 32 32 32 31 31 31 32 31 31 31 31 31 21 21 21
>FLP3FBN01AV5YV length=262 xy=0249_0933 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 38 38 38 40 40 40 38 38 38 38 36 37 37 37 38 38 36 33 27
27 25 25 32 32 32 32 31 28 28 28 28 31 32 31 32 31 31 31 31 31 28
>FLP3FBN01DU3OR length=249 xy=1467_1241 region=1 run=R_2008_12_09_13_51_01_
35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35
35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35
35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 33 33 33 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35
35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 34 34 35 34 34 34 31 29 25 18 18 31 26 15 15 15 15 15 15 15 15 23
30 30 30 30 30 24 25 21 14
>FLP3FBN01EWW35 length=274 xy=1898_0275 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 31 31
31 31 31 31 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 34 34 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 36 36 36 36 36 28 28 28 28 28 33 33 36 36 36 36 36 36 38 38 36 36
36 38 38 38 38 30 30 31 31 31 31 31 32 32 32 32 32 32 32 32 32 32 32 31 31 31 31 31 31 30 30 30 31 31
>FLP3FBN01A3LZ0 length=268 xy=0334_0142 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 30 30 29 37 38 30 30 32 32 32 32 28 17 17 16 16 16 16 20 20 26 26 36 36 36 36 36 36 36
36 36 32 32 28 22 22 25 30 30 30 28 25 25 25 28 31 31 31 31 31 31 31 31 25 18 18 18
>FLP3FBN01D55K5 length=246 xy=1593_0823 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 40 38 38 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 33 33 33 33 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 36 32 32 31 33 33 36 36 36 36 31 31 31 31 36 36 36 36 36 36 36 36 36 36
37 36 36 36 36 31
>FLP3FBN01CHNXI length=271 xy=0904_0724 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 38 38 38 38 38 38 38 38 36 36 36 38 36 36 28 28 28 28 36
36 38 33 33 27 21 21 23 30 34 32 32 32 31 32 32 32 32 31 32 31 32 32 32 32 32 32 32 32 32 32
>FLP3FBN01EETLW length=268 xy=1691_3794 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 32 32 32 32 38 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 34 34 33 33 38 33 33 34 34 34 34 34 34 34 37 37 37 22 22 22
22 37 38 38 29 29 29 33 38 37 37 37 37 37 37 37 37 38 38 30 29 29 33 37 37 34 34 34 34 38 38 38 38 34 25 26 24 28 28 36 36 36 36 36 36 38 38 38 38 38 38 38 38 38 38 38 37 37 37 36
38 38 38 38 38 32 32 31 32 32 32 31 31 31 31 32 31 31 28 22 21 20 20 31 31 28 22 22
>FLP3FBN01B4E5A length=252 xy=0753_1308 region=1 run=R_2008_12_09_13_51_01_
36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 36 36 36 36 36 36 36 36 36 36 36 36 31 20 20
20 20 20 31 35 36 36 36 36 36 36 36 36 36 36 36 36 36 28 28 28 28 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36
36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 31 26 26 26 26 22 22 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36
36 36 36 36 36 36 36 36 36 36 35 35 35 35 35 36 36 31 31 22 31 31 35 36 36 36 36 36 36 36 36 36 36 36 31 31 26 26 26 25 31 35 35 35 25 25 25 25 25 25 24 24 29 35 24 24 16 16 19 25
25 25 31 29 29 24 24 26 28 22 16 16
>FLP3FBN01ET99E length=270 xy=1868_0224 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 34 34 34 40 40 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 36 38 36 36 36 36 36 36 28 28 28 28 28 31 36 36 36 36 36 36 36 36 37 38
38 38 38 38 38 25 25 25 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 31 31 31 31 23 23
>FLP3FBN01BS7JZ length=276 xy=0625_2541 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 38 38 38 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40
40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 34 34 37 37 37 37 37 37 37 37 40 38 38 38 38 38 38 38 38 40 40 38 38 38 38 38 38 38 40 38 38 38 40 40
40 38 38 38 38 32 32 32 32 32 31 31 32 32 32 31 31 31 32 32 32 32 32 31 31 31 31 32 32 32 31 32 31 31 30 30
>FLP3FBN01DW7MZ length=274 xy=1491_1369 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40
40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 36 36 36 36 36 28 28 28 28 28 33 36 36 36 36 36 33 28 28 28 36 36
36 36 36 36 36 30 30 30 31 31 32 32 32 32 32 32 31 31 32 32 32 32 32 32 32 32 32 32 32 32 30 28 23 22
>FLP3FBN01BKWR9 length=246 xy=0531_0359 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 34 34 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 38 38 38 38 38 38 38 38 40 38 38 38 38 38 38 38 38 38 38
38 40 40 40 38 30
>FLP3FBN01DLUYP length=272 xy=1362_0111 region=1 run=R_2008_12_09_13_51_01_
37 36 36 36 36 36 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 36 37 37 37 37 37 37 37 37 37 36 34 34 34 36 36 36 37 37 37 37 37 37 38 37 37 37 37 37 37 37 37 37 37 37 37 36 32 32
32 32 32 37 37 37 37 37 37 37 37 37 37 37 37 37 26 26 26 26 26 26 36 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 35 35 35 35 37 37 37 37 37 37
37 37 37 37 37 37 37 36 36 36 37 37 37 37 37 37 36 32 32 32 32 37 37 37 37 37 37 37 37 35 35 35 35 37 37 37 37 37 37 28 28 28 32 32 37 37 37 37 36 35 32 28 28 28 28 37 37 37 37 37
36 36 36 37 37 37 37 37 37 37 37 37 37 37 36 36 36 37 37 37 37 37 37 37 37 31 22 22 22 22 22 22 22 27 37 37 31 33 33 33 36 36 37 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36
36 36 36 36 36 30 30 30 30 30 28 22 22 22 30 30 30 30 30 25 22 22 22 22 25 31 25 22 16 16 16 16
>FLP3FBN01DBVV5 length=251 xy=1248_1699 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 38 37 37 37 37 37 37 37 40 40 40 40 38 38 38 37 37 37 36 36 36 36 31 28 28 28 33 33 36 33 33 33
36 36 36 36 36 31 31 31 31 22 22
>FLP3FBN01B5DKJ length=273 xy=0764_0865 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 40 40 40 40 40 40 38 38 38 38 38 38 38 38 38 38 36 36 38 38 38 38 38
38 38 38 38 40 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 31 30 30 30 31 32 33 22 16
>FLP3FBN01ASQES length=254 xy=0210_0546 region=1 run=R_2008_12_09_13_51_01_
35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 33 33 32 32 26 26 26 35 35 35 35 38 36 39 36 36 38 38 38 38 38 38 40 39 38 38 35 35 35 35 35 35 35 35 35 35 34 24 24 24 24
24 24 24 24 34 34 35 35 35 35 35 35 35 35 32 32 32 32 35 26 26 25 25 32 34 35 35 35 35 35 35 35 35 35 35 35 34 31 23 23 23 34 33 21 24 25 25 24 32 35 35 35 35 35 35 35 35 35 35 35
35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 37 37 37 35 35 35 32 32 32 34 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35
35 35 35 35 35 35 35 35 35 35 35 35 35 34 34 34 34 34 34 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 23 23 23 23 33 35 35 35 35 35 35 35 35 35
35 35 35 35 35 30 30 30 30 30 22 19 19 19
>FLP3FBN01DPNWR length=281 xy=1405_1465 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 34 34 34 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 34 34 34 33 33 33 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 38 38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 38 38 38 38 40 40 40 40 40 40 40 40 40 40 40 40 40
40 40 40 40 38 32 32 31 31 30 30 30 31 31 31 31 31 32 32 31 31 31 31 31 31 32 32 31 32 31 31 31 31 31 31 30 30 18 18 16 16
>FLP3FBN01DY5C2 length=264 xy=1513_1620 region=1 run=R_2008_12_09_13_51_01_
30 27 19 19 19 21 24 30 30 28 27 27 29 30 33 33 33 30 30 25 25 25 23 17 16 18 22 28 25 25 27 23 15 15 18 17 17 17 18 18 13 18 27 27 33 28 28 28 27 24 19 19 19 24 24 29 30 29 25 25
26 26 28 28 28 22 17 17 17 24 22 29 25 17 17 17 17 24 23 23 30 30 31 30 30 30 31 31 31 31 30 30 30 23 18 17 17 22 22 19 23 23 23 15 15 11 15 13 13 13 13 13 13 18 14 14 13 13 13 21
21 29 31 30 30 32 32 32 32 32 30 30 30 32 30 30 30 26 26 26 26 22 17 17 17 28 20 15 13 13 18 22 22 22 22 13 13 13 13 13 13 13 19 19 13 13 13 21 21 21 21 21 25 21 15 15 15 21 15 15
15 24 29 29 29 28 26 26 26 26 32 32 32 32 32 32 32 32 32 32 29 29 26 26 26 26 25 18 12 12 12 12 12 12 12 9 17 20 17 17 19 19 23 26 26 26 26 21 21 21 24 19 19 19 15 11 11 11 11 17
20 20 20 11 11 11 11 11 15 15 16 18 15 11 11 11 18 19 22 22 22 22 22 22
>FLP3FBN01BLIY9 length=271 xy=0538_0451 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40
40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 38 38 31 31 32 32 31 31 15 15 15 15 20 20 38 38 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 31 31 30 30 38 38 38 38 37 37 37 37 37 37 38 38 30 31 31 37 37 37 37 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 33 33 33 38 38
38 36 37 36 38 28 28 30 32 32 31 31 32 32 32 32 31 31 32 31 32 32 32 32 32 31 30 25 25 25 25
>FLP3FBN01DQGLV length=258 xy=1414_1793 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 31 31 31 31 38 37 37 37 38 38
38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 38 38 38 38 38 38 38 38 40 38 38 40 40 40 40 38 38 38 38 40 38 38 38 38 38 38 38
40 40 40 40 40 32 32 32 32 32 32 32 32 32 32 32 30 23
>FLP3FBN01CUQNU length=255 xy=1053_0488 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 38 38 38 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 40 40 40 40 38 38 38 38 40 40 40 40 40 40 40 38 38 38 40 38 38 38
38 38 40 40 40 31 32 32 32 32 32 32 32 32 31
>FLP3FBN01B0QGX length=272 xy=0711_1391 region=1 run=R_2008_12_09_13_51_01_
35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 40 40 40 40 29 29 29 36 36 36 35 35 35 29 24 24 24 31 35 35 31 31 27 35 35 35 36 35 33 33 33 35 35 35 35 35 35 35 35 35 35 33
33 33 33 33 35 35 35 35 35 35 29 24 22 22 29 29 35 35 35 35 35 35 35 35 35 35 35 35 35 35 30 30 30 33 30 24 23 23 23 23 33 35 35 33 33 33 35 35 35 35 35 35 35 35 24 24 24 22 22 33
35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 33 33 33 35 35 35 35 35 35 35 35 35 29 29 28 29 29 35 35 35 35 35 35 35 35 35 35 35
35 35 35 33 33 33 35 35 35 35 33 21 21 21 33 35 33 33 27 27 27 33 35 35 35 35 35 35 35 27 27 27 20 20 20 27 26 26 22 22 15 15 15 15 31 28 24 24 24 28 29 20 20 20 29 34 34 34 34 34
34 34 34 34 34 29 29 29 29 29 29 27 27 25 18 18 18 21 27 28 28 18 18 18 25 25 28 28 27 28 28 28
>FLP3FBN01A3PGP length=273 xy=0335_0535 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 40 40 40 40 40 38 38 38 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40
40 40 40 40 40 32 32 32 32 32 32 32 32 32 32 32 32 32 31 31 32 31 31 31 31 31 31 32 31 25 25 25 25
>FLP3FBN01ALY2B length=251 xy=0133_0561 region=1 run=R_2008_12_09_13_51_01_
34 34 34 32 27 27 27 27 27 30 30 31 34 34 34 34 34 34 40 38 38 38 40 40 40 40 40 40 39 39 38 33 33 33 33 36 36 36 35 35 34 34 34 34 34 27 27 27 27 27 20 21 21 32 31 31 20 20 20 25
29 28 28 32 32 34 32 32 32 32 34 34 34 30 31 31 30 34 30 30 30 30 34 34 34 34 34 34 34 34 34 34 32 32 22 20 21 26 32 34 34 34 34 34 31 31 31 34 34 34 34 34 34 34 32 25 25 34 34 34
32 32 32 26 20 20 20 26 32 34 32 26 26 26 32 34 34 34 34 34 34 34 34 34 34 34 34 34 32 32 32 34 32 32 32 32 32 31 18 18 18 18 18 18 29 32 32 32 29 29 29 32 32 32 32 32 34 34 29 29
29 32 34 34 25 25 25 25 25 24 24 24 24 29 25 25 29 25 29 29 29 24 24 24 24 24 29 32 32 34 34 34 34 34 34 34 34 23 23 23 23 23 34 23 22 16 16 16 16 27 27 27 27 29 29 27 27 27 23 18
16 16 16 16 29 25 25 25 25 16 16
>FLP3FBN01BT8QN length=253 xy=0637_1581 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 31 31 31 38 38 37 36 36 36 37 36 33 33 33 38 38 38 40 38 38 38 38 38 38 40 40 40 40 40 40
38 36 36 36 36 30 31 31 28 25 19 18 18
>FLP3FBN01DATE7 length=281 xy=1236_0993 region=1 run=R_2008_12_09_13_51_01_
27 27 27 32 34 34 34 31 30 30 30 31 34 34 34 34 34 34 40 38 40 40 40 40 40 40 40 40 40 37 36 36 35 35 35 35 35 35 22 23 22 22 31 35 35 34 34 34 34 34 34 34 34 34 34 32 32 22 21 20
21 21 21 21 32 32 32 32 32 32 32 24 27 27 27 27 27 14 19 19 19 32 32 26 26 26 31 30 32 34 34 34 34 34 34 34 34 34 34 34 32 31 30 26 19 18 18 17 17 15 15 15 15 25 22 22 16 18 18 32
30 31 31 32 34 34 34 34 34 34 34 34 34 34 34 34 34 32 32 32 34 34 34 32 32 32 34 34 34 34 34 34 34 34 34 34 34 31 31 31 29 29 25 25 19 19 19 34 34 32 32 32 29 25 25 25 30 34 34 32
32 32 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 18 18 18 24 24 32 32 32 32 32 32 32 34 34 32 34 34 34 34 34 34 34 34 34 23 18 18 18 16 29 26 19 19 19 19 19 19 19 14 11 14 14
14 26 24 22 16 16 16 20 25 25 25 25 25 25 25 25 25 16 16 16 16 16 22 21 17 12 12 12 12 12 15 18 22 22 22 15 15 17 20 20 25
>FLP3FBN01A5XYK length=271 xy=0360_2458 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 38 33 31 31 31 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 38 38 38 38 38 38 38 38 40 40 40 38 38 38 38 40 38 38 38 38 38 38 38 38
38 38 38 38 38 30 30 32 32 31 31 31 31 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32
>FLP3FBN01BAT5U length=269 xy=0416_1440 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 40 40 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38
38 38 38 38 38 32 32 32 31 32 32 32 32 32 32 32 32 32 32 32 32 32 31 31 26 26 25 31 32
>FLP3FBN01ECTYY length=278 xy=1669_1064 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 33 33
33 33 33 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 38 36 29 29 29 29 29 33 37 37 37 38 38 36 38 38 38 40 40
40 40 38 38 38 32 32 32 32 31 31 31 31 32 32 32 31 31 31 31 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32
>FLP3FBN01A2T1M length=251 xy=0325_0776 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 38 38 38 40 40 40 40 37 39 37 37 37 37 34 34 34 28 28 28 34 34 33 36 37 37 37 37 37 37 36 36 28 28 28 36 36 37 37 37 37 37
37 35 32 28 28 28 28 32 37 37 37 37 37 37 35 35 35 32 32 32 32 37 37 37 37 37 37 37 37 37 37 37 37 37 32 32 32 32 31 31 35 26 26 26 35 35 35 35 35 35 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 32 32 32 32 35 37 37 37 37 37 35 35 35 36 36 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 32 32 32 32 35 37 37 37 37 35 35 26 26 28 28 37 37 37 37 37 37 36 33 31 31 32 36 36 36 33 32 32 32 33 36 36 36 36 36 36 36 36 36 36 36 31
31 31 26 26 26 22 31 25 25 25 25
>FLP3FBN01CBDAE length=253 xy=0832_1908 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40
40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 34 34 34 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 37 37 37 36 36 36 33 33 33 36 37 36 37 37 36 36 38 38 36 36 29 29 29 36 36
36 36 36 36 36 31 32 31 31 28 28 28 31
>FLP3FBN01D8Y20 length=264 xy=1625_1290 region=1 run=R_2008_12_09_13_51_01_
34 34 34 32 22 22 23 23 22 22 22 23 31 34 34 34 34 34 40 40 40 40 40 40 40 40 38 38 38 37 36 36 31 31 30 31 35 36 34 34 34 35 37 37 36 34 34 34 34 33 33 33 33 35 36 36 27 27 14 14
14 14 21 19 27 31 31 31 31 33 33 33 33 33 32 19 19 16 16 16 21 20 24 36 33 32 31 32 32 33 34 34 34 34 33 33 33 31 31 15 15 15 19 19 15 19 19 18 14 14 14 18 15 24 31 31 31 24 24 26
32 32 32 32 31 32 32 31 31 31 34 33 32 32 31 19 19 19 34 27 27 27 21 21 19 31 33 35 35 35 33 31 27 20 21 21 21 19 27 27 27 33 33 33 33 27 27 27 27 33 34 34 34 34 34 34 34 34 34 34
34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 30 30 30 30 30 26 26 26 26 34 33 34 27 27 29 31 29 29 29 33 19 18 17 24 27 31 34 31 31 24 24 17 17 18 29 31
24 24 24 24 24 21 25 27 27 27 27 27 27 27 27 27 21 21 21 21 21 25 25 25
>FLP3FBN01A6MXL length=274 xy=0368_2055 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 34 34 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37
33 33 33 37 38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 36 36 38 38 38 38 33 33 33 33 33 37 37 37 36 36 36 36 36 36 36 36 36
36 33 33 33 33 31 22 22 22 22 25 30 31 31 32 32 31 31 32 32 32 32 32 32 32 32 32 32 32 32 32 32 31 31
>FLP3FBN01ED1E9 length=255 xy=1683_0035 region=1 run=R_2008_12_09_13_51_01_
30 31 31 34 32 33 33 33 33 33 33 34 34 34 34 34 34 34 36 36 36 36 22 22 21 28 32 28 27 27 27 24 21 21 18 19 28 28 28 28 24 31 33 33 34 34 32 32 32 32 32 32 32 34 34 34 27 27 27 25
25 29 26 26 27 31 31 31 31 30 30 33 29 30 30 27 27 18 18 19 19 19 19 26 24 33 32 32 32 34 34 34 34 34 32 26 26 28 30 32 32 32 34 32 32 32 32 30 34 26 19 18 18 18 18 26 16 16 32 32
27 17 19 19 26 31 32 32 34 34 34 34 32 32 32 34 34 34 34 34 34 25 25 25 25 31 34 34 32 32 32 32 32 32 25 25 25 32 32 32 32 34 32 32 29 24 19 19 18 18 25 25 24 24 29 32 29 29 29 29
32 33 32 24 18 18 16 16 27 28 24 24 21 21 14 14 14 14 14 14 14 21 21 28 24 24 29 29 34 34 34 34 34 34 30 30 23 18 18 16 16 27 26 26 24 19 19 19 19 14 14 14 14 19 19 19 19 19 19 27
27 30 27 23 23 20 26 25 25 25 26 26 25 25 25
>FLP3FBN01AYIUY length=251 xy=0276_0360 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 38 38 38 38 38 38 36 36 38 38 38 38 38 38 38 38 38 38 38
40 40 40 40 38 32 32 32 32 28 28
>FLP3FBN01A0UCW length=276 xy=0302_2078 region=1 run=R_2008_12_09_13_51_01_
34 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 34 33 33 25 25 24 24 24 33 35 37 35 35 35 35 37 36 37 38 38 38 36 36 36 35 35 35 35 35 35 35 35 34 34 34 34 35 35 35 35 35 27 27
27 27 27 27 35 35 35 35 35 34 34 34 35 35 35 35 35 35 34 34 34 34 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 32 32 26 26 26 26 26 26 35 35 35 35 35 34 34 34
35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 26 26 26 26 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35
35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 34 24 24 24 34 35 35 35 35 35 35 35 35 35 35 31 31 31 26 22 22 22 22 22 31 35 35 35 35 35 35 35 35 35 35 33
33 33 33 33 33 30 30 30 30 30 30 30 30 30 30 30 30 30 30 30 30 30 30 30 30 30 30 30 30 30 25 25 25 25 16 16
>FLP3FBN01CDJ70 length=268 xy=0857_1806 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 33 30 28 28 37 37 37 37 38 27 27 27 34 37 37 37 37 37 37 37 37 37 38 34 26 26 26 26 26 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 38 38 38 38 40 40 38 38 38 38 40 38 38 38 38 38 38 38 38
40 38 38 38 38 32 32 32 32 31 32 31 31 32 32 32 32 30 30 30 31 32 32 32 32 31 31 31
>FLP3FBN01EFY7W length=260 xy=1705_0378 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 29 29 29 34 28 28 28 28 38 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 37 36 36 36
36 36 36 36 37 31 31 31 32 30 30 30 31 32 32 32 32 32 32 30
>FLP3FBN01AGI0O length=262 xy=0071_0438 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 34 34 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 38 38 38 40 38 38 38 38 37 36 36 36 36 38 36 28
28 28 28 33 33 28 28 28 31 32 32 32 31 31 31 31 31 32 32 32 32 31
>FLP3FBN01CUN6D length=272 xy=1052_1363 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 33 33 33 33 38 38 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 34 34 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 40 40 40 38 38 38 38 38 38 38 38 38 37 38 38 38 29 29 29 29
33 37 38 38 38 31 31 31 31 31 31 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 31 31 31 31 31
>FLP3FBN01EFPI1 length=281 xy=1702_0107 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37
38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 38 38 38
38 40 40 40 38 31 31 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 31 31 31 32 32 31 31 31 31 31 31 31 31 31 31 31
>FLP3FBN01A8V8L length=259 xy=0394_0931 region=1 run=R_2008_12_09_13_51_01_
34 34 34 34 33 28 25 25 25 23 28 34 34 34 34 34 34 34 36 36 36 36 36 36 36 36 35 35 34 33 33 33 33 33 33 31 31 22 22 22 21 21 31 34 36 34 35 34 32 32 32 34 34 32 32 32 33 34 35 35
32 28 28 28 28 28 21 19 19 19 28 33 35 35 33 32 32 32 28 28 28 34 34 32 31 31 33 33 34 33 33 33 34 34 31 31 32 32 34 34 34 33 31 31 19 19 20 26 31 18 14 14 14 14 31 24 33 35 35 33
32 31 32 32 34 34 34 34 34 34 33 31 32 31 33 31 19 19 16 16 16 24 24 24 24 16 16 16 24 27 27 31 35 31 19 19 19 31 33 31 31 31 31 23 17 17 16 16 16 17 23 27 27 27 31 33 35 35 34 34
34 34 31 31 31 34 34 34 33 34 33 33 33 25 25 25 25 25 19 19 19 14 23 23 21 33 33 33 23 16 19 19 19 19 19 30 27 28 28 28 28 31 31 31 14 14 14 14 13 28 21 14 14 14 14 14 14 14 21 21
14 14 14 14 28 13 16 16 21 25 27 25 25 25 25 21 16 16 16
>FLP3FBN01BZOEE length=268 xy=0699_1204 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 38 36 38 36 36 36 25 27 27 25 25 33 36 36 36 36 38 36 36 36 36 38
38 38 38 38 38 31 31 31 31 32 32 31 31 31 31 31 31 31 31 31 31 31 31 32 30 22 21 21
>FLP3FBN01EGX4W length=281 xy=1716_0574 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 36 36 36 36 36 38 37 36 36 28 28 28 28 29 36 36 36 36 36 36 38 36 36
36 38 37 37 33 25 25 25 31 31 31 32 32 31 31 31 32 32 32 31 31 31 28 28 28 30 32 32 32 31 31 31 31 31 31 31 31 20 20 19 18
>FLP3FBN01D2AYU length=273 xy=1549_1396 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 33 34 34 37 37 37 38 38 38 38 38 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 38 38 38 40 38 38 38 40 40 40 40 40 40 40 38 38 38 40 40 38 38 38 38
38 38 38 38 40 32 32 32 32 31 32 31 32 32 32 32 32 32 32 31 25 25 25 31 30 30 30 30 30 31 31 31 32
>FLP3FBN01BHVDZ length=276 xy=0496_1941 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40
40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 40 40 40 38 38 38 38 38 38 38 38 38 40 40 40 40 40 40 38 38 38
40 40 40 40 40 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 31 31 23 23
>FLP3FBN01ED5UR length=255 xy=1684_1681 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 38 38 38 38 40 40 40 40 40 40 40 40 40 38 38 38 38 40 40 38 38 38 40
40 40 40 40 40 32 32 31 32 31 32 30 25 25 22
>FLP3FBN01CU37V length=270 xy=1057_1673 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38
31 31 31 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 36 38 36 36 31 31 24 22 22 22 22 26 21 33 33 36 36 36 36 36
36 37 38 38 38 31 31 31 32 31 32 31 32 31 28 28 25 25 25 28 30 31 31 31 31 32 32 30 30 30
>FLP3FBN01B86XL length=268 xy=0807_2759 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 39 36 36 36 39 39 38 39 39 39 39 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38
38 38 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 30 30 31 38 38 38 37 37 37 37 37 37 37 37 38 33 33 33 33 26 29 29 33 33 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 40 40 40 40 40 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38
38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 38 38 38 38 38 38 38 38 40 40 40 38 38 38 38 38 38 38 40 40 38 38 38
38 38 40 38 38 32 32 32 32 31 31 31 32 32 32 31 30 30 28 28 22 22 22 31 31 26 22 22
>FLP3FBN01DJ1VW length=264 xy=1341_1786 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 34 34 34 34 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 36 36 37 37 38 38 38 38 38 38 38 38 38 38 40 40 40 40 38 38 38 38
40 40 38 38 38 32 32 32 32 32 32 31 31 31 31 31 31 31 30 25 22 19 18 18
>FLP3FBN01AKD0L length=253 xy=0115_0355 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 31 31 31 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 33 33 33 33 38 38 38 38 37 37 37 37 37 37 38 38 38 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 36 36 36 36 36 36 31 31 31 27 27 27 27 36 36 36 36 36 33 33 33 33
33 33 36 36 36 30 28 28 28 22 22 22 22
>FLP3FBN01D6H8C length=272 xy=1597_0826 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 34 34 34 38 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37
37 38 38 28 28 28 28 28 34 33 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 40 40 40 40 40 40 40 40 40 38 38 38 38 36 36 36 36 28 28 28 28 36 31
31 31 31 32 33 31 31 31 31 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 30 30 30 31 31 32 32 31
>FLP3FBN01DQFJG length=270 xy=1414_0410 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 32 32 32 36 36 36 36 36 36 37 37 37 37 37 32 32 32 32 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 32 32 32 37 37 37 37 35 23 21 21 26 26 32 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 28 28 28 32 37 37 37 37 37 37 37 37 37 37 37 37 36 36 35 35 35 35 37 37 36 23
26 26 32 35 26 26 26 32 36 37 37 37 37 37 36 36 36 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 37 36 36 33 32 32 33 38 38 38 38 33 33 33 36 36 36 36 36 36
36 36 36 36 33 28 22 22 22 22 28 28 28 28 28 25 25 24 24 24 21 21 13 13 13 15 21 18 18 21
>FLP3FBN01AGC0G length=277 xy=0069_0846 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 34 34 34 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 38 38 38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 38 38 40 40 40 40 40 40 40 40 40 40 40 40 38 38 38 38 38 38 38 38 40 38
38 40 38 38 38 31 31 31 31 31 31 32 32 32 32 32 32 32 32 31 31 31 31 32 32 32 32 32 32 31 31 31 32 32 31 23 23
>FLP3FBN01D0FVD length=262 xy=1528_0455 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 40 40 40 40 40 40 40 40 40 40 40 40 40 38 38 38 38 40 40 40 40 40 40
40 40 40 38 38 32 32 32 32 32 32 32 32 32 31 32 32 32 32 32 32 32
>FLP3FBN01BG1KW length=277 xy=0487_0174 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 38 38 36 38 38 38 38 29 29 29 29 29 38 38 38 38 38 38 38 38 38
38 40 40 40 40 32 31 31 31 32 31 32 32 31 31 32 32 32 31 31 31 32 31 32 32 32 32 31 32 31 31 31 31 31 22 21 21
>FLP3FBN01BW4TT length=253 xy=0670_1311 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 34 34 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 38 38 38 38 38 38 36 38 38 38 38 38 38 38 38 38 37 36 37
38 38 36 36 36 31 31 31 30 30 22 22 22
>FLP3FBN01BOFGD length=270 xy=0571_0683 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 40 40 40 40 40 40 40 40 40 40 40 40 40 38 38 38 38 40 40
40 40 40 38 38 32 31 31 31 31 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32
>FLP3FBN01CWLZU length=255 xy=1074_1736 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 38 38 38 38 38 38 38 40 40 40 40 40 40 40 40 40 40 40 38 38 38 38
40 38 38 40 38 31 31 31 31 31 31 31 31 31 30
>FLP3FBN01AMXMK length=265 xy=0144_0298 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 38 32 32 32 32 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 36 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 36 33 33 33 33 36 36 36 33 27 27 27 27 27 33 37 36 38 36 36 36 38 38 38
38 38 38 38 38 32 32 31 31 31 32 32 31 31 31 31 31 31 31 31 25 18 16 16 16
>FLP3FBN01BHNXZ length=264 xy=0494_0485 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 35 35 35 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 33 33
33 33 33 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 34 34 34 34 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 33 33 34 38 38 37 37 34 24 21 21 21 21 24 38 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 34 34 34 37 37 38 38 38 38 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 38 38 38 38 38 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 38 38 28 28 28 38 38 37 36 36 36 36 36 33 33 33 37 37 38 38 38 36 36 38 38 38 38 38 38 38 38 38
38 38 37 37 36 31 31 31 31 31 31 28 25 25 25 28 33 30 25 25 13 13 13 17
>FLP3FBN01DRT2O length=258 xy=1430_0366 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 34 34 34 38 33 33 33 33 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 37 36 36 36 36 36 36 29 29 29 29 29 37 38 38 40 40 40 40 40 40 40 40
40 40 38 38 38 32 32 32 32 32 32 32 32 32 32 32 28 23
>FLP3FBN01ATU7A length=261 xy=0223_0164 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 38 38 31 31
31 31 31 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 38 36 36 36 36 38 36 36 36 33 24 24 24 31 33 36 36 36 28 28 27 31 32 36
36 38 38 38 36 31 31 31 31 31 31 32 32 31 31 31 31 32 32 30 30
>FLP3FBN01DQMNE length=275 xy=1416_1432 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 37 33 38 38 38 38
38 38 38 38 38 31 31 32 32 32 32 32 31 31 31 32 32 32 32 32 32 32 31 30 30 30 31 31 32 32 31 28 28 28 30
>FLP3FBN01BGO68 length=277 xy=0483_0514 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 40 40 40 38 38 38 38 40 40 40 38 38 38 38 38 38 38 40 38 38
38 38 38 40 40 32 31 31 31 31 31 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 31 25 25
>FLP3FBN01EC56I length=258 xy=1673_0504 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40
40 40 40 40 40 32 31 30 30 30 31 31 31 31 31 31 30 25
>FLP3FBN01D9EK7 length=256 xy=1630_0905 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 34 34 34 34 38 38 26 26 26
26 26 34 34 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 38 38 38 38 38 38 38 38 33 33 33 33 38 36 36 38 36 36 36 38 38 38
38 38 38 38 38 31 32 31 32 32 31 31 31 31 28 22
>FLP3FBN01BXX60 length=258 xy=0679_2506 region=1 run=R_2008_12_09_13_51_01_
35 35 35 31 26 26 26 26 31 35 35 35 35 35 35 35 35 35 40 38 38 38 40 40 40 40 40 40 40 38 39 38 39 39 39 40 40 39 36 36 36 40 40 40 40 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35
35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 33 33 33 33 35 35 35 35 35 35 35 35 35 35 35 35 35 31 21 21 21 33 31 28 20 20 20 31 31 20 21 21 35 35 35 35 35 35 35 35 35 35 35
35 35 35 35 35 35 35 35 35 35 35 35 35 35 30 30 30 35 35 35 35 35 35 35 35 35 35 35 33 30 21 21 21 33 20 17 17 19 27 25 28 28 33 33 33 33 33 29 29 29 33 35 35 35 33 33 33 35 35 35
35 35 35 35 35 35 33 33 33 35 35 35 35 35 35 35 33 33 23 23 23 20 20 35 35 34 35 35 33 27 20 20 23 29 20 20 19 32 32 30 30 30 32 34 25 26 26 26 26 34 33 33 31 31 34 34 34 34 34 31
26 26 27 27 33 29 30 29 29 29 29 29 29 30 30 29 22 18
>FLP3FBN01A208L length=281 xy=0327_1907 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 38 40 40 40 40 38 38 36 36 36 36 37 36 37 36 32 27 27 27
33 36 36 36 36 30 31 31 31 31 32 32 32 32 32 32 32 32 31 31 31 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 31 31 31 31 31
>FLP3FBN01EBSVN length=272 xy=1657_2145 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40
40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 38 38 38 38 40 38 38 38 38 38 40 38 38 38 38 38 40 38 38
38 38 38 38 38 32 32 32 32 32 32 32 32 32 32 32 31 31 32 31 31 32 32 32 32 32 32 32 32 32 32 32
>FLP3FBN01BDPN5 length=259 xy=0449_0419 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 33 33 33 37 37 37 36 36 36 36 36 37 38 36 37 37 36 36 36 36 36 37 36 36 36 36 36 36 37 38
38 38 38 37 36 31 32 32 32 32 32 31 25 25 25 30 32 31 31
>FLP3FBN01DF6NE length=265 xy=1297_1560 region=1 run=R_2008_12_09_13_51_01_
34 31 31 30 30 33 34 34 34 33 27 27 27 34 34 34 34 34 40 40 40 40 40 40 40 40 40 38 38 38 38 36 36 36 36 36 36 36 23 23 22 30 33 35 36 34 34 34 32 32 32 34 34 34 34 34 34 34 34 34
34 34 34 34 32 32 32 32 34 34 34 30 30 27 27 27 34 34 34 34 34 34 31 26 26 28 31 31 32 34 34 32 32 32 34 34 31 28 28 28 34 32 32 32 34 34 30 34 26 19 18 14 14 16 13 13 25 27 29 28
22 22 17 18 19 20 30 31 32 34 34 33 32 32 32 24 20 20 20 20 31 31 24 24 24 30 32 34 34 34 25 25 25 25 32 32 32 34 34 34 34 25 25 22 30 30 30 30 30 30 30 30 34 34 34 34 34 34 34 34
34 34 34 34 34 34 34 34 25 24 24 30 30 34 32 32 32 32 32 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 22 22 16 14 14 16 16 16 16 16 16 16 16 27 34 34 34 27 27 27
34 34 34 34 34 26 26 26 26 26 26 26 26 26 25 25 25 25 25 25 25 25 16 16 16
>FLP3FBN01D87QS length=255 xy=1628_0226 region=1 run=R_2008_12_09_13_51_01_
36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36
36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 35 35 35 36 35 35 35 35 35 36 36 36 36 36 36 36 36 36 36 36 36 36
36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 28 28 28 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 32 32 32 33 33 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36
36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 36 35 35 35 36 36 36 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35
35 35 19 19 19 0 13 13 0 14 24 14 14 14 21
>FLP3FBN01BSM84 length=255 xy=0619_0806 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 36 36 36 36 36 36 31 31 25 25 25 25 25 31 33 38 36 36 36 38 37 38 38 38
38 40 40 38 38 31 31 31 31 28 28 28 30 31 22
>FLP3FBN01BJIH7 length=262 xy=0515_0733 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 33 28 27 26 33 36 36 36 36 36 22 24 24 25 25 33 36 36 36 36 36 36
36 36 36 38 38 28 28 32 31 31 31 31 31 32 32 32 32 32 32 32 28 23
>FLP3FBN01D8FO0 length=268 xy=1619_0738 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 38 38 38 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 32 32 32 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 36 36 36 38 38 38 33 33 33 33 33 38 38 38 38 38 40 40 38 38 38 38
38 38 40 40 40 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 31 31 30 28 28
>FLP3FBN01AGYWU length=255 xy=0076_0556 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 36 36 36 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 40 40 38 38 38 38 38 40
40 40 40 40 38 32 31 31 31 32 32 32 32 31 25
>FLP3FBN01D6UF4 length=270 xy=1601_0274 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 38 38 38 38 38 40 40 40 38 38 38 40 40 40 40 38 38
38 38 38 38 38 31 31 31 32 32 31 32 32 32 32 32 32 32 32 31 31 32 31 31 31 32 28 21 21 20
>FLP3FBN01AEB9M length=274 xy=0046_0776 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38
38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 38 37 36 36 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38
38 36 36 36 37 31 31 31 31 31 31 32 32 32 32 31 31 31 32 32 32 32 32 31 31 31 31 30 30 30 30 31 31 31
>FLP3FBN01CIT14 length=281 xy=0917_2074 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 34 34 34 34 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 33 33 33 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 36 36 36 36 36 38 36 36 36 32 33 33 33 33 36 36 38 38 36 36 36 36 36
36 38 33 33 33 25 25 25 31 32 32 32 32 31 31 31 32 32 32 32 32 32 32 32 32 32 32 32 32 32 31 31 31 31 32 30 28 21 21 21 20
>FLP3FBN01DZG2M length=281 xy=1517_0412 region=1 run=R_2008_12_09_13_51_01_
34 34 34 34 34 32 22 22 21 27 30 31 31 32 34 34 34 34 34 34 34 34 20 21 21 28 27 28 27 27 32 32 34 34 34 28 30 27 34 33 33 33 35 35 36 34 34 34 34 32 27 27 27 30 31 31 32 32 18 18
14 14 14 21 27 27 30 30 29 18 18 18 18 18 27 27 27 27 14 14 14 14 25 30 16 26 26 31 31 34 34 30 30 30 30 34 34 34 34 34 34 34 34 34 21 21 18 18 16 25 22 22 22 16 17 14 14 14 17 17
14 17 15 16 16 22 22 30 30 31 31 30 32 32 34 34 32 32 32 32 32 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 32 31 30 31 32 34 19 19 19 25 25 34 34 34 34 33 33 24 24 25 25 29 29 25
25 25 25 25 24 32 34 34 34 29 34 30 18 18 18 30 30 34 34 34 34 34 34 34 34 24 24 24 24 24 32 24 24 17 17 17 27 13 11 14 14 24 24 26 26 24 19 14 14 14 14 14 24 26 26 19 14 14 14 14
14 16 16 16 27 25 25 25 25 26 26 26 26 26 26 26 25 25 25 25 25 26 26 26 26 26 26 25 25 25 20 20 20 20 25 20 20 20 20 20 20
>FLP3FBN01CN7JX length=281 xy=0978_2987 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 30 30 31 31 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 40 40 40 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38
38 40 38 38 38 30 30 32 32 32 32 32 32 32 32 32 32 31 30 25 25 25 30 30 32 31 31 31 31 32 31 32 31 31 31 31 30 22 22 22 22
>FLP3FBN01B74TP length=281 xy=0795_2523 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 38 38 38 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38
38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 40 40 40 40 40 40 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38
38 38 38 38 40 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 31 31 31 31 31 31 31 32 31 28 22 22 22 22 28 28
>FLP3FBN01BXZ1T length=282 xy=0680_0815 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 38 38 38 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 38 38 38 37 37 37 37 37 37 37 33 33 29 29 24 24 24 36 36 36 36 36 36 36 36 36 36 33 33 32 32 33 33 36 36 36 36 38
36 36 37 37 38 32 32 32 32 31 31 31 31 31 31 32 31 31 31 32 32 32 32 31 31 32 32 31 31 32 32 31 31 31 31 31 31 31 31 31 31 32
>FLP3FBN01C6LRW length=264 xy=1188_1066 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 36 36 36 36 37 37 37 36 37 38 38 38 38 38 38 38 40 40 40 40 38 38 38 38
38 38 38 38 38 31 31 31 31 30 30 30 30 30 31 31 30 30 30 30 28 18 18 18
>FLP3FBN01AOR4O length=264 xy=0165_0470 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 38 38 38 38 38 38 38 40 40 40 40 40 40 40 38 38 38 38 38 38 38 38 38
38 38 38 38 38 31 31 31 32 32 32 31 31 31 31 31 31 31 28 22 18 18 18 16
>FLP3FBN01CSZZD length=280 xy=1033_1175 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 33 33 33 33 38 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 36 36 28 28 28 28 31 36 36 36 36 36 36 36 36 36 38 38 38 38 38 38
38 38 38 40 40 32 32 32 31 30 30 30 31 32 32 32 32 32 32 32 32 31 31 31 31 31 32 32 32 32 31 32 31 31 31 31 28 22 21 21
>FLP3FBN01DNS9D length=272 xy=1384_1103 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 38 38 38 37 37 37 37 37 37 38 40 40 40 40 40 40 40 38 38 37 37 37 36 36 38 36 33 29 29 29 29 33 37
38 38 38 38 36 32 32 32 32 32 32 31 32 32 32 32 32 32 32 32 31 31 31 32 32 31 28 25 19 19 18 18
>FLP3FBN01EGW4M length=270 xy=1715_3364 region=1 run=R_2008_12_09_13_51_01_
34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 36 33 33 33 30 30 30 34 34 33 32 32 32 18 19 19 19 27 27 27 19 21 16 28 34 34 36 34 34 34 34 34 34 34 34 34 34 33 33 31 31 34
34 34 34 34 34 34 34 34 34 34 34 30 31 31 31 32 34 34 34 34 34 34 34 34 32 32 32 32 31 31 31 30 30 30 30 30 32 32 32 32 30 24 24 24 29 25 25 22 16 17 15 15 15 19 20 27 33 33 32 32
32 33 33 34 26 26 26 26 26 26 26 26 30 26 18 18 18 31 30 30 31 34 34 31 30 31 30 34 34 34 34 34 34 25 20 20 20 25 28 31 30 25 34 32 32 25 19 19 19 19 19 18 18 29 32 34 32 32 29 30
30 34 34 34 34 34 34 34 34 30 30 30 34 34 34 34 34 34 34 34 34 34 34 24 18 18 18 24 24 32 32 34 34 28 32 32 30 27 23 23 23 34 30 27 25 25 25 22 22 22 27 27 27 29 27 18 18 18 18 23
30 30 34 34 34 26 26 26 26 26 26 26 26 26 26 26 20 20 16 16 16 25 20 20 20 11 11 17 15 15
>FLP3FBN01C9TUA length=271 xy=1224_4032 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 38 38 38 40 40 39 39 39 39 40 39 39 38 30 29 29 29 29 33 39 39 39 39 37 37 37 37 37 40 40 40 37 37 37 37 37 37 37
37 37 37 37 37 37 38 33 34 34 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 33 29 28 30 30 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 33 36 31 26 26 26 26 26 20 20 20 20 20 30 32 36 36 36 36 33 33 33 36 37
38 38 38 38 38 32 31 31 31 31 32 32 32 32 32 32 32 31 30 30 30 30 32 32 32 32 32 32 32 32 32
>FLP3FBN01DZ0UC length=282 xy=1523_1458 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 38 38 26 26
26 26 20 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 33 33 33 34 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 38 37 37 37
38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 33 33 29 30 29 31 31 34 34 34 36 36 38 37 37 36 38 40 40 40 38 38 38 38 38 38 40 38 38
38 36 37 38 38 31 31 31 31 31 31 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 31 31 31 31 31 30 30 30 30
>FLP3FBN01D9E65 length=276 xy=1630_1695 region=1 run=R_2008_12_09_13_51_01_
35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 36 36 36 36 29 29 29 35 33 31 29 29 35 35 33 34 33 31 31 29 35 36 36 36 37 37 40 35 35 35 35 35 35 38 38 38 35 35 35 35 35 35
35 35 35 33 33 33 35 35 35 35 35 35 35 35 35 35 29 29 23 23 23 23 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35
35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 33 33 24 24 24 24 21 35 35 33 33 33 33 27 27 27 33 33 35 33 33
33 33 35 35 29 29 29 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 27 23 23 27 26 24 26 26 26 26 34 34 34 32 32 32 30 30 32 28 18 18 18 15 15 15 13 22
26 30 30 30 22 21 21 25 25 27 27 29 25 25 25 29 29 29 29 29 29 29 29 28 28 27 27 21 21 21 16 16 16 16 13 13
>FLP3FBN01ECWT9 length=273 xy=1670_0687 region=1 run=R_2008_12_09_13_51_01_
35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 40 40 40 40 37 36 36 40 36 37 31 29 35 35 34 34 34 35 35 35 35 35 35 35 37 35 36 35 35 35 35 35 35 35 35 35 35 35 35 33 29 29
30 30 35 35 35 29 30 20 20 20 30 35 35 33 21 21 21 21 30 29 29 29 30 33 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 28 28 20 20 20
29 28 28 28 28 28 28 33 35 35 35 35 35 35 35 35 35 35 35 33 33 33 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 33 20 20 14 14 14 31 33 32 32 32 25 16 16 16 19 19 33 33 32 25 19 19
15 15 15 15 19 19 32 32 32 33 33 35 35 35 35 35 35 35 35 35 35 35 35 35 35 27 27 27 35 35 35 35 35 33 23 27 24 32 32 32 32 32 31 22 14 14 14 14 14 22 22 22 22 28 22 22 22 28 28 28
30 30 24 24 18 16 16 16 21 14 21 21 27 25 27 26 26 26 27 26 16 18 18 27 26 13 13 16 21 21 25 27 25
>FLP3FBN01AHH1K length=264 xy=0082_0774 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 36 36 36 38 38 38 39 39 38 39 39 39 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 31 31 31 31 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 37 37 36 36 36 37 36 36 36 36 36 36 36 37 38 38 38 40 40 38 38 38 40
40 40 40 40 40 32 32 32 32 32 32 32 32 31 31 32 31 31 28 22 20 19 19 18
>FLP3FBN01DQLWU length=259 xy=1416_0476 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 33 33
33 33 33 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 30 28 28 28 28 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 38 38 38 38 40 40 40 38 38 38 38 38 38 38 38 38 38 38 38 38
38 40 40 40 40 32 32 32 32 32 32 32 32 32 32 32 31 31 31
>FLP3FBN01ALCMH length=268 xy=0126_0151 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 38 38 38 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 34 34 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 38 38 33 36 33 27 27 29 28 33 37 37 36 36 36 36 38 38 38 38 38 38 38 40 40 38
38 38 38 38 38 31 31 31 31 31 31 31 31 31 32 32 32 31 31 31 31 31 31 32 30 25 25 25
>FLP3FBN01C4N17 length=274 xy=1166_0829 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38
40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 34 33 34 38 38 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 40 40 40 40 40 40 40 38 38 38 38 38 38 38 38 38 38 38 38 38
38 38 38 38 38 32 32 32 32 32 32 31 31 31 32 32 32 32 32 32 32 32 31 31 31 31 32 30 30 30 30 30 28 28
>FLP3FBN01B8F4K length=264 xy=0799_0786 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 34 34 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 40 37 37 37 37 37 37 37 36 36 36 33 31 31 31 33 26 26 26 26 26 30 25 33 31 22 22 22 26 26 38 38
38 38 36 36 36 30 30 30 30 31 31 25 19 18 18 25 30 31 31 30 18 16 16 22
>FLP3FBN01CATW7 length=272 xy=0826_1385 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 38 38 38 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40
40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 38 38 38 38 38 38 38 38 40 40 38 38 38 38 38 38 40 40 40 40 40 40
40 40 38 38 38 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 31 31 32 32 31 30 28 20 19 19 18
>FLP3FBN01B97H8 length=277 xy=0819_1006 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36
36 36 36 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 38 34 26 28 26 28 33 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 31 31 31 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 40 40 40 40 40 40 40 38 38 38 38 38 38 38 38 38 38 38 38 38
38 38 38 38 38 28 31 31 31 31 31 32 32 32 32 31 31 31 32 31 32 32 32 32 32 32 32 31 32 31 31 31 31 28 22 19 19
>FLP3FBN01BA934 length=257 xy=0421_1634 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40
40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
34 34 34 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 40 40 40 40 40 40 40 40 40 38 38 38 40 40 38 38 38 38 38 38 38
40 40 38 38 38 32 31 32 32 32 32 32 32 32 32 32 31
>FLP3FBN01A3YNV length=264 xy=0338_0169 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 40 40 40 40 38 38 36 36 36 37 38 36 36 36 25 25 25 25 28 33
33 33 33 33 36 31 31 32 32 32 32 31 31 31 31 30 28 28 28 22 19 18 18 16
>FLP3FBN01BJOS4 length=268 xy=0517_0710 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 38 38 38 38 38 40
40 40 40 40 40 31 31 31 31 31 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 30 30 30
>FLP3FBN01EP59G length=271 xy=1821_0930 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 34 34 37 37 37 37 37 37 37 37 37 37 37
34 34 34 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 38 38 38 38 38 38 38 38 40 40 40 40 40 40 40 40 40 40 40
38 38 36 36 36 31 31 23 23 23 23 25 25 28 31 31 31 32 31 31 31 31 31 31 31 28 25 18 18 18 16
>FLP3FBN01CTU4A length=268 xy=1043_0568 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 40 40 40 40 38 38 38 38
38 38 38 38 38 31 31 31 31 31 32 31 30 30 30 30 30 31 30 30 25 25 25 25 25 22 22 22
>FLP3FBN01DD28G length=271 xy=1273_2126 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 30 30 30 34 34 38 38 38 37 37 37 37 37 37 37 37 38 38 38
33 33 33 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 33 34 34 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 38 38 38 38 38 38 38 36 36 36 36 38 38 33 33 29 29 27 28 29 36 36 37
36 36 36 36 36 31 32 32 32 30 30 23 23 23 30 32 32 32 31 31 31 31 32 32 31 30 28 20 19 19 18
>FLP3FBN01BW0YB length=287 xy=0669_0385 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40
40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 33 33 33 33 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 33 33 34 34 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 31 31 31 31 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 38 38 38 38 38 38 38 38 38 38 38
38 38 40 40 40 32 31 31 31 32 32 32 32 32 32 32 32 32 32 31 31 32 32 32 32 32 32 32 31 31 31 31 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32
>FLP3FBN01AZT44 length=264 xy=0291_0198 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 40 38 38 38 36 36 36 37
38 38 33 33 33 25 25 25 30 30 30 30 31 31 31 32 32 31 31 31 25 21 21 20
>FLP3FBN01AGMG5 length=270 xy=0072_0823 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 31 31 31 34 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 37 37 36 37 36 36 36 36
33 33 33 33 36 25 25 25 31 31 31 31 31 32 32 32 32 31 31 32 32 31 31 31 31 31 31 31 21 21
>FLP3FBN01ARUVC length=268 xy=0200_0630 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 40 40 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 36 36 37 37
37 38 38 37 36 23 23 23 30 31 32 32 32 32 32 32 32 32 32 32 32 32 32 31 30 30 30 30
>FLP3FBN01D86I8 length=254 xy=1627_2754 region=1 run=R_2008_12_09_13_51_01_
35 35 35 35 35 37 35 35 35 35 35 35 35 35 35 35 35 35 40 40 40 40 40 40 40 40 40 39 39 39 40 40 40 40 40 40 40 40 36 36 36 38 40 40 40 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35
35 32 32 32 32 35 32 26 26 26 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 34 34 34 34 35 35 35 35 35 33 32 32 32 33 28 21 16 16 16 16 21 15 24 32 32 33
33 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 34 34 34 34 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35
35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 29 29 29 35 35 35 35 35 35 35 35 35 24 19 19 25 25 35 35 23 15 15 15 15 15 15 29 26 23 23 23 23 28 31 35 31 22
19 19 19 30 31 28 28 28 21 22 21 22 25 22
>FLP3FBN01COOOS length=281 xy=0984_0618 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 34 34 34 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 33 33 33 37 37 37 36 36 36 36 36 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 38 38 38 40 40 40 40 40 40 40 40 38 38 38
38 38 38 38 38 32 31 31 31 32 32 32 32 32 32 31 31 31 31 31 31 32 32 32 32 32 32 32 32 32 32 32 31 31 31 31 28 28 25 25 25
>FLP3FBN01D8J4C length=259 xy=1620_2378 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 40 40 38 38 38 40 40 40 40 38 38 38 38 38 38 38 38 38 38 38 38 38
38 38 38 38 38 31 31 31 31 31 32 32 31 31 31 31 32 25 21
>FLP3FBN01BR61D length=280 xy=0614_0271 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 38 38 38 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 29 29 29 33 33 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 36 38 36 38 36 31 26 22 22 22 22 22 26 31 36 36 36 38 36 36 36 38
38 38 38 38 38 31 32 32 32 32 32 32 32 32 32 32 32 32 28 28 28 30 32 32 31 31 31 31 32 32 32 32 32 32 32 32 32 30 28 25
>FLP3FBN01BF0UO length=288 xy=0475_1726 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 31 31
31 31 31 31 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 34 34 34 34 37 37 37 31 29 29 29 22 22 20 20 20 20 20 20 17 16 16 16 16 16 30 30 30 31 17 15 15 15 15 15 17 17 28 28 32 36 36 32 32
30 31 31 31 36 31 31 22 22 22 25 30 32 31 30 30 30 31 31 30 25 25 25 25 25 31 32 31 31 31 31 25 25 25 25 30 31 31 25 25 25 25 25 22 22 22 22 30
>FLP3FBN01DB7BE length=268 xy=1252_0120 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 38 38 38 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
40 40 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 38 38 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 38 38 38 38 40 40 40 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38
40 40 40 40 40 32 32 32 32 32 32 32 32 32 32 32 31 31 30 30 30 30 30 30 25 18 18 18
>FLP3FBN01AZSCT length=271 xy=0290_1979 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 33 33 33 37 37 29 26 26 34 33 33 33 33 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 31 31 31 37 37 37 37 37 37 37 38 36 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38
36 36 38 38 38 28 28 28 28 31 31 31 31 31 31 32 31 32 31 31 31 30 30 30 31 30 30 22 22 22 22
>FLP3FBN01D3EWI length=281 xy=1561_4000 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 31 31 31 31 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 33 33 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 38 38 38 38 38 33 33 33 38 38 40 40 38 38 38 38 38 38 38 38 38 38 38
38 38 38 38 38 31 32 31 31 31 31 32 31 31 31 31 32 32 32 32 31 31 25 25 25 32 32 31 32 31 22 22 19 18 19 22 28 22 22 22 22
>FLP3FBN01A55LW length=257 xy=0363_0082 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 38 38 38 38 38 40 40 40 40 38 38 38
40 40 40 40 40 32 32 31 31 31 31 32 32 32 32 32 30
>FLP3FBN01DZ6P8 length=269 xy=1525_0894 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 36 33 33 33 33 37 37 33 33 33 33 38 38 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 38 38 38 40 40 38 38 38 40 38 38 38 38 37 36 36 36 36 36
36 36 28 25 25 20 20 22 26 32 31 31 31 31 32 31 32 32 32 32 32 32 32 32 32 32 31 31 31
>FLP3FBN01AEB2N length=271 xy=0046_0525 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 40 40 40 40 40 40 40 38 38 38 40 40 40 40 38 38 38 38 38 38
38 38 38 38 38 31 31 31 31 31 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 31 31 31 31 31
>FLP3FBN01B9U1E length=281 xy=0815_1232 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 34 15 15 15 15 15 28 23 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 40 40 40 40 40 40 40 40 40 38 38 38 40 38 38 38 38 38 38 38 38 38 38
38 38 38 38 38 32 32 32 32 31 31 31 31 32 32 31 31 31 31 28 28 28 32 32 32 32 31 31 31 31 31 31 31 31 28 28 28 28 31 31 31
>FLP3FBN01A77ZB length=268 xy=0386_2261 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 36 36 36 36 36 28 28 28 28 28 36 36 36 36 36 38 38 38 38 38 40
38 38 38 38 40 32 31 31 31 31 32 32 32 32 32 32 32 31 31 31 31 31 31 31 30 26 26 25
>FLP3FBN01BCQ7B length=254 xy=0438_0805 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 33 32 32 32 38 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38
38 36 36 36 36 32 32 31 30 25 22 18 18 18
>FLP3FBN01D5TAZ length=271 xy=1589_1289 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 31 31 31 31 31 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 38 38 38 38 38 38 38 38 40 40 38 38 38 37 36 36 36 38 37 33 33 33 36
36 36 37 37 38 31 31 31 32 32 32 32 32 32 32 32 32 32 31 25 25 25 25 32 32 31 31 31 31 22 22
>FLP3FBN01BX26A length=278 xy=0681_0768 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 34 34
34 34 34 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 33 30 30 30 28 28 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 38 40 40 40 40 40 40 40 38 38 38 38 38 38 38 38 38 38 38
38 38 38 38 38 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 31 31 31 31 31 32 32 32 32 28 28
>FLP3FBN01DTX5C length=255 xy=1454_0654 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 36 36 36 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 38 38 38 38 38 40 38 38 38 40 40 40 40 40 40 38 38 38 38
40 40 38 38 38 32 31 31 32 32 32 32 32 32 32
>FLP3FBN01BCTWA length=268 xy=0439_0200 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 38 38 38 38 38 38 38 40 40 38 38 38 38 40 40 40 40 40 38 38 38 38
38 38 40 40 40 31 31 31 32 31 32 32 32 32 32 32 32 31 31 32 32 32 32 32 31 31 31 31
>FLP3FBN01BA22V length=272 xy=0419_0709 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 34 34 37 37 37 37 37 37 37 37 38 33 33 33 33 38 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 33 33 33 38 37 37 36 36 36 36 38 38 38 38 38 38 38 38 38 38 38 38 40 40 38 38 38 38 40 40
40 40 40 40 38 31 31 32 32 32 32 31 31 32 32 32 31 31 32 32 31 31 31 32 32 31 28 28 21 21 20 19
>FLP3FBN01A4UET length=281 xy=0348_0355 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 30 30
30 30 30 30 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 38 38 40 40 40 40 38 38 38 38 38 40 40 38 38 38 38 38 40 40 40 40 38 38
38 40 40 40 40 32 32 31 31 31 31 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32
>FLP3FBN01EII4M length=253 xy=1734_0708 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36
36 36 36 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 29 29 29 29 36 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 36 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 36 36 37 37 37 37 37 37 37 37 36 36 36 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 36 37 37 37 37 37 37 37 37 37 37 37 33 33 28 26 26 26 32 31 30 30 30 33 30 30 19 19 19 19 19 19 30 36 36 32 31 31 33 36 36
36 36 36 36 36 30 31 31 31 25 22 22 22
>FLP3FBN01DPG83 length=282 xy=1403_1029 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 38 38 38 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 32 32
32 32 32 32 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 34 34 34 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 36 36 36 36 37 38 38 38 38 38 38 38 40 40 38 38 38 38 38 38 40 38 38
38 38 38 38 38 31 31 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 31 28 28 25 25 25 31 31 31 31
>FLP3FBN01AYGIQ length=262 xy=0275_1424 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 34 34 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 34 34 34 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 37 37 37 38 38 38
38 38 33 33 33 31 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32
>FLP3FBN01AGF7L length=268 xy=0070_0895 region=1 run=R_2008_12_09_13_51_01_
33 33 32 34 34 34 34 34 34 34 34 34 34 34 34 34 34 33 34 34 34 34 31 28 28 31 31 34 34 34 34 35 36 35 36 36 36 37 37 37 36 38 38 38 37 34 34 34 34 34 34 34 34 34 34 32 23 22 22 28
33 32 31 31 31 31 21 14 14 14 14 21 19 31 31 33 33 35 35 34 33 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 32 28 28 23 23 22 23 34 34 33 33 33 34 34 33 34 34 34
31 32 31 34 34 34 34 34 34 34 33 23 22 23 21 27 33 33 34 34 34 34 34 34 34 34 34 27 27 21 21 21 34 33 27 27 27 27 27 33 33 33 33 31 19 19 19 21 27 31 33 33 33 33 33 35 35 35 35 35
33 17 16 16 16 17 17 15 19 19 19 14 24 18 37 33 34 34 34 34 33 31 31 30 34 34 19 19 19 19 19 19 33 30 30 30 27 24 31 24 24 24 24 24 24 24 24 27 27 27 24 28 24 21 21 21 14 14 14 14
14 14 14 14 21 13 13 13 16 13 16 21 25 27 27 27 27 27 27 27 27 27 27 27 27 27 27 27
>FLP3FBN01A4NV1 length=286 xy=0346_0095 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 34 34 34 34 37 37 37 38 38 38 38 38 29 30 30 30 37 37 37 37 37 37
37 37 37 34 34 34 34 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 36 36 37 36 36 36 36 33 33 29 29 28 33 36 36 36 36 36 36 36 28
28 28 29 36 36 30 28 28 28 30 30 31 32 32 32 32 32 32 32 32 32 32 32 31 31 31 31 31 31 31 31 31 25 25 25 25 25 30 31 31 31 31 31 31 28 28
>FLP3FBN01AZ963 length=272 xy=0296_0525 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 40 40 40 40 38 38 38 38 38 38 38 40 40 40 40 40 38 38 38 40 40
40 40 40 38 29 23 23 28 30 32 32 32 32 32 32 32 32 32 32 32 32 32 32 32 31 31 31 31 32 32 31 31
>FLP3FBN01DMBQY length=255 xy=1367_1384 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 38 38 38 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38
40 40 40 40 40 32 32 32 32 32 32 32 32 32 25
>FLP3FBN01A6SLO length=268 xy=0370_1210 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 28 30 30 38 38 37 31 31 31 31 36 36 36 36 36 36 38 38 38 38 40 40 40 40 40 40 38 38 38 40
38 38 38 38 40 32 32 32 31 32 32 32 32 32 32 32 31 31 31 32 31 31 31 32 31 22 20 20
>FLP3FBN01CQHPF length=262 xy=1004_2961 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 34 34 34 34 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 38 38 36 33 33 33 33 33 38 33 33 33 33 37 37 38 38 40 40
40 38 38 38 38 32 32 32 32 32 32 32 31 31 31 32 32 32 32 32 32 28
>FLP3FBN01CSQG7 length=264 xy=1030_1145 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 33 33 34 34 37 37 37 38 38 38 38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 38 33 36 36 36 36 36 31 26 22 22 22 20 20 33 32 36 36 36 36 31 31 31 36 36
36 36 36 36 36 25 25 25 30 31 32 31 31 31 31 31 30 25 25 25 22 22 22 25
>FLP3FBN01AMENY length=292 xy=0138_0300 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 33 33 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 36 36 36 36 38 38
38 33 33 29 29 21 21 31 31 32 31 31 31 31 32 31 31 31 31 31 31 31 31 28 26 26 25 31 32 32 32 31 31 31 32 32 32 32 32 31 31 31 31 31 32 32 32 32 32 32 32 32
>FLP3FBN01D1PPR length=268 xy=1542_2525 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 38 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 40 38 40 40 40 40 40 40 40 40 40 40 40 40 38 38 38 40 38 38 38 37 37 36 37
36 36 33 33 33 25 30 30 32 31 31 31 32 32 32 32 32 32 32 32 32 31 31 31 31 31 31 31
>FLP3FBN01CPKVQ length=269 xy=0994_1380 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 38 38 28 28 28 28 30 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 38 38 38 40 40 40 40 40 40 40 40 40 40 38 38 38 38 38 38 38 38 38 38
38 37 38 38 38 32 32 32 31 31 31 32 32 32 32 32 31 31 32 32 32 32 32 32 32 32 32 28 28
>FLP3FBN01BZK6L length=276 xy=0698_1131 region=1 run=R_2008_12_09_13_51_01_
34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 40 40 40 40 29 29 29 38 39 38 38 38 38 38 38 38 38 39 38 39 38 38 38 37 37 37 38 34 34 34 34 34 31 32 32 34 34 33 22 22 22 26
27 28 28 26 28 28 28 34 34 34 35 35 33 32 31 28 28 19 19 20 15 15 15 20 26 26 32 32 32 31 26 28 28 33 33 28 28 28 28 33 32 28 18 15 15 26 26 24 31 31 26 14 14 18 18 28 19 19 32 32
34 34 34 34 34 34 34 34 34 34 34 34 28 28 21 21 21 21 31 34 35 33 33 31 31 31 35 35 31 20 19 19 31 31 31 17 19 20 19 27 19 19 19 19 19 31 35 34 34 34 34 34 34 34 34 34 34 34 34 34
34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 34 30 30 20 20 19 19 31 30 30 33 34 33 33 25 19 19 19 33 27 28 24 21 14 14 14 14 14 21 14 14 14 14 14 14 24 14 14 14 21 28 28 28
27 27 27 24 17 16 16 16 27 17 21 21 25 27 27 27 27 27 27 27 27 27 21 21 21 17 17 25 25 25 13 13 13 13 13 13
>FLP3FBN01DF6XB length=268 xy=1297_1917 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 40 40 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 33 33 33 33 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 38 37 36 36 38 38 36 36 37 38 38 36 38 38 38 36 36 36 37
36 36 33 33 33 22 22 22 30 30 31 31 31 31 31 32 31 31 31 31 31 31 31 31 30 26 26 25
>FLP3FBN01CF2XK length=268 xy=0886_0582 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 33 33 33 38 38 38 38 38 37 37 37 37 37 37 37 37 33 26 26 26 28 34 38 37 37 37
37 37 37 37 37 37 37 37 37 37 38 38 38 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 40 40 40 38 38 33 33 33 38
38 38 38 38 38 31 31 31 31 31 31 31 32 32 32 32 32 31 31 31 31 32 32 32 25 21 20 19
>FLP3FBN01CXD3D length=277 xy=1083_1287 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 38 38 38 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 33 33 33 33 38 37 37 37 37 37 37 37 37
37 38 38 38 31 31 31 38 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 38 38 38 38 38 38 38 38 38 38 40 40 38 38 37 36 36 38 37
37 37 36 33 29 23 23 28 30 32 31 31 32 32 32 32 31 31 31 31 32 32 32 32 32 32 32 32 32 32 32 32 32 32 31 30 30
>FLP3FBN01CQGY3 length=274 xy=1004_2013 region=1 run=R_2008_12_09_13_51_01_
35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 40 39 39 39 40 40 40 40 40 40 40 38 38 39 40 40 40 38 38 38 36 36 36 38 40 38 39 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35
33 33 33 35 35 33 23 20 20 17 29 31 33 29 23 20 20 29 31 31 31 31 31 31 35 35 35 35 35 35 35 35 35 31 23 21 21 21 33 29 28 29 29 29 31 29 29 31 31 33 33 33 33 33 33 33 35 35 35 35
35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 35 30 30 30 30 28 29 29 35 32 34 34 33 33
33 25 25 14 14 14 14 14 19 18 27 30 33 34 34 35 35 35 35 35 35 35 35 35 35 35 35 35 35 23 23 23 35 35 35 30 26 26 26 23 23 23 15 14 14 14 14 14 23 30 28 28 28 26 26 30 34 34 34 26
26 26 26 33 34 30 29 29 28 28 28 29 30 30 28 28 28 28 22 16 16 16 22 16 13 13 13 13 19 21 19 25 22 23
>FLP3FBN01B52I9 length=270 xy=0772_0451 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 37 33 33 33 33 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 33 33 33 37 37 37 37 34 34 34 34 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 38 38 38 38 38 38 38 38 40 40 40 38 38 38 36 36 36 36 36 36 27 33 33 36
36 37 37 37 37 31 32 32 32 32 32 32 32 32 32 32 32 32 31 31 31 32 31 25 22 22 19 19 18 22
>FLP3FBN01AS2O8 length=264 xy=0214_0090 region=1 run=R_2008_12_09_13_51_01_
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 40 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37
37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 37 