/**
 * Copyright (C) 2007-2013 Lawrence Murray
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * @author Lawrence Murray <lawrence@indii.org>
 * $Rev$
 * $Date$
 */
#ifndef INDII_TINT_MODEL_CLUSTERMODEL_HPP
#define INDII_TINT_MODEL_CLUSTERMODEL_HPP

#include "ClusterModelObservable.hpp"
#include "../../model/Model.hpp"
#include "../../image/ImageManipulation.hpp"
#include "../../cluster/KMeansClusterer.hpp"
#include "../../cluster/PearsonDistance.hpp"

#include "wx/colour.h"

#include "boost/numeric/ublas/vector.hpp"

#include <vector>

namespace indii {

class ClusterModelObserver;

/**
 * Cluster model.
 */
class ClusterModel: public Model, public ClusterModelObservable {
public:
  using Model::calcFg;

  /**
   * Value type for clusterer.
   */
  typedef boost::numeric::ublas::vector<float,
      boost::numeric::ublas::bounded_array<float, 3> > value_type;

  /**
   * Distance type for clusterer.
   */
  typedef PearsonDistance<value_type, float> distance_type;

  /**
   * Clusterer type.
   */
  typedef KMeansClusterer<distance_type> clusterer_type;

  /**
   * Vector type.
   */
  typedef boost::numeric::ublas::vector<float> vector_type;

  /**
   * Integer vector type.
   */
  typedef boost::numeric::ublas::vector<int> int_vector_type;

  /**
   * Constructor.
   *
   * @param res Image resource.
   */
  ClusterModel(ImageResource* res);

  /**
   * Destructor.
   */
  virtual ~ClusterModel();

  /**
   * Get number of clusters.
   *
   * @return Number of clusters.
   */
  unsigned getNumClusters();

  /**
   * Set number of clusters.
   *
   * @param k Number of clusters.
   */
  void setNumClusters(const unsigned k);

  /**
   * Is cluster assignment hard?
   *
   * @return True if clustering is hard, false otherwise.
   */
  bool isHard();

  /**
   * Get cluster hardness.
   *
   * @return Cluster hardness.
   */
  float getHard();

  /**
   * Set cluster hardness.
   *
   * @param hard Cluster hardness, between 0 and 1.
   */
  void setHard(const float hard = 1.0f);

  /**
   * Get number of repetitions.
   *
   * @return Number of repetitions.
   */
  unsigned getNumRepetitions();

  /**
   * Set number of clusters.
   *
   * @param reps Number of repetitions.
   */
  void setNumRepetitions(const unsigned reps);

  /**
   * Get saturation threshold.
   *
   * @return Saturation threshold.
   */
  float getSaturationThreshold();

  /**
   * Set saturation threshold.
   *
   * @param x Saturation threshold.
   */
  void setSaturationThreshold(const float x);

  /**
   * Get maximum pixels for clustering.
   *
   * @return Maximum pixels for clustering.
   */
  unsigned getMaxPixels();

  /**
   * Set maximum pixels for clustering.
   *
   * @param n Maximum pixels for clustering.
   */
  void setMaxPixels(const unsigned n);

  /**
   * Get saturation decay.
   *
   * @return Saturation decay.
   */
  float getSaturationDecay();

  /**
   * Set saturation decay.
   *
   * @param x Saturation decay.
   */
  void setSaturationDecay(const float x);

  /**
   * Get centroid distance decay.
   *
   * @return Centroid distance decay.
   */
  float getCentroidDecay();

  /**
   * Set centroid distance decay.
   *
   * @param x Centroid distance decay.
   */
  void setCentroidDecay(float x);

  /**
   * Get saturation softness.
   *
   * @return Saturation softness.
   */
  float getSaturationSoftness();

  /**
   * Set saturation softness.
   *
   * @param x Saturation softness.
   */
  void setSaturationSoftness(const float x);

  /**
   * Get centroid distance softness.
   *
   * @return Centroid distance softness.
   */
  float getCentroidSoftness();

  /**
   * Set centroid distance softness.
   *
   * @param x Centroid distance softness.
   */
  void setCentroidSoftness(const float x);

  /**
   * Is cluster shown in colour?
   *
   * @param i Cluster number.
   */
  bool isShown(const unsigned i);

  /**
   * Show (or hide) cluster.
   *
   * @param i Cluster number.
   * @param on True to show cluster in colour, false for b&w.
   */
  void show(const unsigned i, const bool on = true);

  /**
   * Show (or hide) all clusters.
   *
   * @param on True to show all clusters in colour, false for b&w.
   */
  void showAll(const bool on = true);

  /**
   * Get cluster colour.
   *
   * @param i Cluster number.
   *
   * @return Colour representing given cluster.
   */
  const wxColour& getColour(const unsigned i);

  /**
   * Get cluster colour.
   *
   * @param i Cluster number.
   * @param r Red value.
   * @param g Green value.
   * @param b Blue value.
   */
  void getColour(const unsigned i, float* r, float* g, float* b);
  
  /**
   * Set cluster colour.
   *
   * @param i Cluster number.
   * @param col Cluster colour.
   */
  void setColour(const unsigned i, const wxColour& col);

  /**
   * Set cluster colour.
   *
   * @param i Cluster number.
   * @param r Red value.
   * @param g Green value.
   * @param b Blue value.
   */
  void setColour(const unsigned i, const float r, const float g, const float b);

  /**
   * Get cluster hue rotation.
   *
   * @param i Cluster number.
   *
   * @return Hue rotation of given cluster.
   */
  float getHue(const unsigned i);

  /**
   * Set cluster hue rotation.
   *
   * @param i Cluster number.
   * @param x Hue rotation to set.
   */
  void setHue(const unsigned i, const float x);

  /**
   * Get cluster saturation adjustment.
   *
   * @param i Cluster number.
   *
   * @return Saturation adjustment of given cluster.
   */
  float getSat(const unsigned i);

  /**
   * Set cluster saturation adjustment.
   *
   * @param i Cluster number.
   * @param x Saturation adjustment to set.
   */
  void setSat(const unsigned i, const float x);

  /**
   * Does cluster have any saturation?
   *
   * @param i Cluster number.
   * 
   * @return True if saturation of cluster @p i is significant (i.e. does not
   * round to grey), false otherwise.
   */
  bool hasSat(const unsigned i);

  /**
   * Get cluster lightness adjustment.
   *
   * @param i Cluster number.
   *
   * @return Lightness adjustment of given cluster.
   */
  float getLight(const unsigned i);

  /**
   * Set cluster lightness adjustment.
   *
   * @param i Cluster number.
   * @param x Lightness adjustment to set.
   */
  void setLight(const unsigned i, const float x);

  /**
   * Does cluster have any light?
   *
   * @param i Cluster number.
   * 
   * @return True if lightness of cluster @p i is significant (i.e. does not
   * round to black), false otherwise.
   */
  bool hasLight(const unsigned i);

  /**
   * Get cluster alpha adjustment.
   *
   * @param i Cluster number.
   *
   * @return Alpha adjustment of given cluster.
   */
  float getAlpha(const unsigned i);

  /**
   * Set cluster alpha adjustment.
   *
   * @param i Cluster number.
   * @param x Alpha adjustment to set.
   */
  void setAlpha(const unsigned i, const float x);

  /**
   * Calculate alpha channel of subimage for particular cluster.
   *
   * @param i Cluster number.
   * @param rect Rectangular region of interest, relative to scaled image.
   * @param scale Inverse scale of image.
   * @param[out] c Output channel.
   */
  void calcAlpha(const unsigned i, const wxRect& rect, const float scale,
      channel& c);

  /**
   * Calculate mask for particular cluster.
   *
   * @param i Cluster number.
   * @param rect Rectangular region of interest, relative to scaled image.
   * @param width Scaled width of image.
   * @param height Scaled height of image.
   *
   * @return Mask for the given cluster and region.
   */
  sparse_mask calcMask(const unsigned i, const wxRect& rect, const float scale);

  /**
   * Prepare data set for clustering.
   */
  void prepare();

  /**
   * Cluster image.
   */
  void cluster();

  /**
   * Lock model. Ignore any calls to prepare() or cluster() until unlocked.
   * This is basically a hack to stop them being overcalled, lock(), change
   * a bunch of settings, unlock(), then prepare() and cluster() to keep the
   * model consistent.
   */
  void lock();

  /**
   * Unlock model.
   */
  void unlock();

  virtual void calcFg(const wxRect& rect, const float scale, Image& o);
  virtual void setDefaults();
  virtual void setForDialog();

private:
  /**
   * Number of clusters.
   */
  unsigned k;

  /**
   * Cluster hardness.
   */
  float hard;

  /**
   * Number of repetitions.
   */
  unsigned reps;

  /**
   * Saturation threshold.
   */
  float saturationThreshold;

  /**
   * Maximum number of pixels for clustering.
   */
  unsigned maxPixels;

  /**
   * Saturation decay.
   */
  float saturationDecay;

  /**
   * Centroid decay.
   */
  float centroidDecay;

  /**
   * Saturation softness.
   */
  float saturationSoftness;

  /**
   * Centroid softness.
   */
  float centroidSoftness;

  /**
   * Cluster hue rotation.
   */
  std::vector<float> hues;

  /**
   * Cluster saturation adjustment.
   */
  std::vector<float> sats;

  /**
   * Cluster light adjustment.
   */
  std::vector<float> lights;

  /**
   * Cluster alpha adjustment.
   */
  std::vector<float> alphas;

  /**
   * Cluster colours.
   */
  std::vector<wxColour> colours;

  /**
   * Cluster means.
   */
  std::vector<value_type> mus;

  /**
   * Clustering data set.
   */
  std::vector<value_type> data;

  /**
   * Clusterer.
   */
  clusterer_type clusterer;

  /**
   * Is model locked?
   */
  bool locked;
};
}

inline unsigned indii::ClusterModel::getNumClusters() {
  return k;
}

inline bool indii::ClusterModel::isHard() {
  return hard >= 1.0f;
}

inline float indii::ClusterModel::getHard() {
  return hard;
}

inline unsigned indii::ClusterModel::getNumRepetitions() {
  return reps;
}

inline float indii::ClusterModel::getSaturationThreshold() {
  return saturationThreshold;
}

inline unsigned indii::ClusterModel::getMaxPixels() {
  return maxPixels;
}

inline float indii::ClusterModel::getSaturationDecay() {
  return saturationDecay;
}

inline float indii::ClusterModel::getCentroidDecay() {
  return centroidDecay;
}

inline float indii::ClusterModel::getSaturationSoftness() {
  return saturationSoftness;
}

inline float indii::ClusterModel::getCentroidSoftness() {
  return centroidSoftness;
}

inline bool indii::ClusterModel::isShown(const unsigned i) {
  return hasSat(i);
}

inline const wxColour& indii::ClusterModel::getColour(const unsigned i) {
  /* pre-condition */
  assert(i < k);

  return colours[i];
}

inline void indii::ClusterModel::getColour(const unsigned i, float* r, float* g, float* b) {
  /* pre-condition */
  assert(i < k);
  
  *r = mus[i][0];
  *g = mus[i][1];
  *b = mus[i][2];
}

inline float indii::ClusterModel::getHue(const unsigned i) {
  /* pre-condition */
  assert(i < k);

  return hues[i];
}

inline float indii::ClusterModel::getSat(const unsigned i) {
  /* pre-condition */
  assert(i < k);

  return sats[i];
}

inline float indii::ClusterModel::getLight(const unsigned i) {
  /* pre-condition */
  assert(i < k);

  return lights[i];
}

inline float indii::ClusterModel::getAlpha(const unsigned i) {
  /* pre-condition */
  assert(i < k);

  return alphas[i];
}

inline bool indii::ClusterModel::hasLight(const unsigned i) {
  /* pre-condition */
  assert(i < k);

  return lights[i] > -1.0f;
}

inline bool indii::ClusterModel::hasSat(const unsigned i) {
  /* pre-condition */
  assert(i < k);

  return sats[i] > -1.0f;
}

inline void indii::ClusterModel::lock() {
  locked = true;
}

inline void indii::ClusterModel::unlock() {
  locked = false;
}

#endif

